/*******************************************************************************
* FILE NAME: imenu.cpp                                                         *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in imenu.cpp                                                               *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1995       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

/*******************************************************************************
|                                                                              |
|  IMPORTANT NOTE:                                                             |
|  Any call within this compiliation unit to an IMenu virtual function such    |
|  as setText, setItem, etc. should be qualified as such:                      |
|     IMenu::setText, IMenu::setItem                                           |
|  This must be done because the corresponding versions of these virtual       |
|  functions in ISubmenu, will call their IMenu counterparts to change a menu  |
|  item, then save the change made in a list so that it can be undone.         |
|  If we don't qualify the calls here with IMenu we can get ISubmenu::setText  |
|  rather than IMenu::setText and we'll needlessly save that change in the list|
|  This can cause unwarranted exceptions at undo time.                         |
|                                                                              |
*******************************************************************************/
// Priority INT_MIN (-2147483647 - 1) + 1024 + 512
#pragma priority( -2147482112 )

extern "C" {
  #define INCL_WINFRAMEMGR
  #define INCL_WINWINDOWMGR
  #define INCL_WINMENUS
  #define INCL_GPIPRIMITIVES
  #define INCL_WINSYS
  #define INCL_GPIBITMAPS
  #include <iwindefs.h>
}

#ifndef _IMENU_
 #include <imenu.hpp>
#endif
#ifndef _IMNITEM_
 #include <imnitem.hpp>
#endif
#ifndef _IWINDOW_
 #include <iwindow.hpp>
#endif
#ifndef _IRECT_
 #include <irect.hpp>
#endif
#ifndef _ICOLOR_
 #include <icolor.hpp>
#endif
#ifndef _ICCONST_
 #include <icconst.h>
#endif
#ifndef _IEXCEPT_
 #include <iexcept.hpp>
#endif
#ifndef _ITRACE_
 #include <itrace.hpp>
#endif
#ifndef _IRESLIB_
 #include <ireslib.hpp>
#endif
#ifndef _INOTIFEV_
  #include <inotifev.hpp>
#endif
#include <iframe.hpp>
#include <imenuprv.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _IMENU_CPP_
  #include <ipagetun.h>
#endif

#pragma data_seg(ICLStaticConst)
const IMenu::Style
  IMenu::noStyle            = 0,
  IMenu::classDefaultStyle  = 0;
#pragma data_seg()

#pragma data_seg(ICLNonConst)
  IMenu::Style IMenu::currentDefaultStyle = 0;
#pragma data_seg()

// maintain list of bitmaps so that they can be freed
typedef struct _BITMAPLIST {

 IBitmapHandle* pBitmap;
 _BITMAPLIST*   pNext;
} BITMAPLIST;

#ifdef IC_PM
/*------------------------------------------------------------------------------
| queryItem                                                                    |
|  Local function used to query menu-item. Also used to check for valid        |
|  menu item and throw exception.                                              |
------------------------------------------------------------------------------*/
MENUITEM queryItem( const IWindowHandle& menuHandle,
                    unsigned long        itemId )
{
  IMODTRACE_DEVELOP("IMenu::queryItem");
  MENUITEM miItem;
  if(!(Boolean)menuHandle.sendEvent(MM_QUERYITEM,
                                    IEventParameter1((unsigned short)itemId,
                                                     true),
                                    IEventParameter2(&miItem))) {
     ITHROWLIBRARYERROR(IC_INVALID_MENUITEM,
                        IErrorInfo::invalidRequest,
                        IException::recoverable);
  } /* endif */
  return miItem;
}
#endif
#ifdef IC_WIN
/*------------------------------------------------------------------------------
| queryItemState                                                               |
|  Local function used to query menu-item. Also used to check for valid        |
|  menu item and throw exception.                                              |
------------------------------------------------------------------------------*/
UINT queryItemState( const IMenuHandle& menuHandle,
                     unsigned long      itemId )
{
  IMODTRACE_DEVELOP("IMenu::queryItem");
  unsigned long menuState = GetMenuState( menuHandle, itemId, MF_BYCOMMAND );
  if ( menuState == 0xFFFFFFFF )
    ITHROWLIBRARYERROR( IC_INVALID_MENUITEM,
                        IErrorInfo::invalidRequest,
                        IException::recoverable );
  return menuState;
}
#endif

/*------------------------------------------------------------------------------
| IMenu::IMenu                                                                 |
------------------------------------------------------------------------------*/
IMenu::IMenu ( )
  : pBitmapList(0), fMenuData(0)
#ifdef IC_WIN
    , fpSubmenuList(0)
#endif
{ }

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
  IMenu constructor for an existing menu handle
------------------------------------------------------------------------------*/
IMenu::IMenu ( const IMenuHandle& menuHandle )
  : pBitmapList(0), fMenuData(0)
#ifdef IC_WIN
    , fpSubmenuList(0)
#endif
{
  IASSERTPARM(menuHandle!=0);
  setAutoDestroyWindow(false);
#ifdef IC_PM
  startHandlingEventsFor((IWindowHandle)menuHandle);
#endif
  // IC_NOTYET - Currently not implemented on windows
}
#endif

/*------------------------------------------------------------------------------
| IMenu::~IMenu                                                                |
| Definition of pure virtual destructor so linker won't complain               |
------------------------------------------------------------------------------*/
IMenu :: ~IMenu()
{
   IMODTRACE_DEVELOP("IMenu::dtor");
   if (pBitmapList)
   {
      BITMAPLIST* pList = (BITMAPLIST*)pBitmapList;
      BITMAPLIST* pNextRec;
      pNextRec = pList->pNext;
      while (pNextRec) {
        BITMAPLIST* pTempRec = pNextRec;
        delete pNextRec->pBitmap;
        pNextRec = pNextRec->pNext;
        delete pTempRec;
      } /* endwhile */
      delete pList->pBitmap;
      delete pList;
   } /* endif */
#ifdef IC_WIN
   if (fpSubmenuList)
   {
     FMTABLE* pFmTable = (FMTABLE*)fpSubmenuList;
     FMTABLE* pNext;
     do {
       pNext = pFmTable->pfmTable;
       delete pFmTable;
       pFmTable = pNext;
     } while (pFmTable);
   } /* endif */
#endif
}


/*------------------------------------------------------------------------------
| IMenu::addItem                                                               |
| Add new menu item from IMenuItem structure.                                  |
------------------------------------------------------------------------------*/
IMenu& IMenu::addItem( IMenuItem&          newItem,
                       unsigned long       intoSubmenuId)
{
   IMODTRACE_DEVELOP("IMenu::addItem (1)");
#ifdef IC_WIN
   // Depending on menu type, place identity into menu
   UINT  menu1 = newItem.id();
   if (newItem.isSubmenu())
     menu1 = (UINT)(newItem.submenuHandle().asUnsigned());

   // Depending on menu type, place information into menu
   LPSTR menu2 = (LPSTR)(char*)newItem.text();
   if (newItem.isBitmap())
     menu2 = (LPSTR)(void*)newItem.bitmap();

   // IC_NOTYET - Need to resolve placing menuitems into submenuId, currently
   //             we ignore the intoSubmenuId parameter
   HMENU hMenu = (intoSubmenuId) ?
                     getHMenuFromLookup(intoSubmenuId) : (HMENU)menuHandle();

   InsertMenu( hMenu,
               newItem.index(),
               newItem.style() | MF_BYPOSITION,
               menu1, menu2 );
   DrawMenuBar( fowner );

   // Ref count the bitmap if one passed in
   if (newItem.isBitmap() && newItem.bitmap() )
     refCountBitmap( menu2 );
#else
   MENUITEM mn;
   mn.iPosition    = (unsigned short)(newItem.index());
   mn.afStyle      = (unsigned short)(newItem.style());
   mn.afAttribute  = (unsigned short)(newItem.attribute());
   mn.id           = (unsigned short)(newItem.id());
   mn.hItem        = newItem.bitmap();
   mn.hwndSubMenu  = newItem.submenuHandle();

   IMenuHandle hwnd = (intoSubmenuId) ?
                         submenuHandle(intoSubmenuId) : menuHandle();

   long reply = hwnd.sendEvent( MM_INSERTITEM,
                                IEventData(&mn),
                                (newItem.text().length()) ?
                                  IEventData((char*)newItem.text()) :
                                  IEventData());

   if (reply == MIT_MEMERROR || reply == MIT_ERROR)
     ITHROWGUIERROR("MM_INSERTITEM");
   else
     newItem.setIndex(reply);

   if (mn.afStyle & MIS_BITMAP) {   //ref. count bitmap.
      if (mn.hItem) {
         if(!(Boolean)hwnd.sendEvent( MM_SETITEMHANDLE,
                                      IEventData(mn.id),
                                      IEventData(mn.hItem))) {
            ITHROWLIBRARYERROR(IC_SETMENUITEM_FAIL,
                               IErrorInfo::invalidRequest,
                               IException::recoverable);
         } /* endif */
         refCountBitmap(mn.hItem);
      } else {
         ITHROWLIBRARYERROR(IC_NO_BITMAP_HANDLE,
                            IErrorInfo::invalidRequest,
                            IException::recoverable);
      } /* endif */
   } else {
      if (mn.afStyle & MIS_TEXT)
         // In case we were called by ISubmenu
         IMenu::setText(mn.id, newItem.text());
   } /* endif */
#endif
   return *this;
}

/*------------------------------------------------------------------------------
| IMenu::addSubmenu                                                            |
| Create an empty menu and add it to the menu-item as submenu                  |
------------------------------------------------------------------------------*/
IMenu& IMenu::addSubmenu( unsigned long itemId )
{
   IMODTRACE_DEVELOP("IMenu::addSubmenu(1)");
   IASSERTSTATE(itemId != FID_MENU); //menu item cannot be menu-bar ID

   IMenuItem mn = menuItem(itemId);
   if (mn.submenuHandle()) {
      ITHROWLIBRARYERROR(IC_ADD_SUBMENU_FAIL,
                         IErrorInfo::invalidRequest,
                         IException::recoverable);
   } /* endif */
#ifdef IC_WIN
   HMENU hmenuMenu = CreatePopupMenu();
   if (hmenuMenu)
   {
     mn.setSubmenuHandle(hmenuMenu);
     addToLookup( hmenuMenu, itemId );
   }
#else
   HWND hwndMenu =   this -> create( mn.id(),
                      0,
                      0,
                      WC_MENU,
                      // defect 9732 - menu items detach from frame when
                      // frame is minimized and then restored.  The menu
                      // items usually appear in the lower left corner,
                      // detached from the frame.  Changed HWND_DESKTOP to
                      // HWND_OBJECT
                      HWND_OBJECT,
                      this->handle(),
                      IRectangle(),
                      0,
                      0,
                      IWindow::onTopOfSiblings);
   if (hwndMenu) {
       mn.setSubmenuHandle(hwndMenu);

       unsigned long afStyle = WinQueryWindowULong(mn.submenuHandle(),
                                                   QWL_STYLE);
       afStyle &= (unsigned long)
                    ~(MS_CONDITIONALCASCADE |  //Menu is create as menu-bar
                      MS_VERTICALFLIP |        //Change it.
                      MS_ACTIONBAR);
       WinSetWindowULong(mn.submenuHandle(), QWL_STYLE, afStyle);
   }
#endif
   IMenu::setItem(mn);  //set it.

   return *this;
}

/*------------------------------------------------------------------------------
| IMenu::addSubmenu                                                            |
| Load a menu from resource fileand add it to the menu-item as submenu         |
------------------------------------------------------------------------------*/
IMenu& IMenu::addSubmenu( unsigned long      itemId,
                          const IResourceId& submenuResId )
{
   IMODTRACE_DEVELOP("IMenu::addSubmenu(2)");
   IASSERTPARM(itemId != FID_MENU); //menu item cannot be menu-bar ID

   IMenuItem mn = menuItem(itemId);
   if (mn.submenuHandle()) {        //cannot add if already has submenu.
      ITHROWLIBRARYERROR(IC_ADD_SUBMENU_FAIL,
                         IErrorInfo::invalidRequest,
                         IException::recoverable);
   } /* endif */

   mn.setSubmenuHandle(submenuResId.resourceLibrary().loadMenu
                                             (submenuResId, owner()));
#ifdef IC_WIN
   // Add handle to lookup table to create mapping to id for Windows
   addToLookup( (void*)mn.submenuHandle().asUnsigned(), itemId );
#endif
#ifndef IC_WIN
   // defect 9732 - menu items detach from frame when
   // frame is minimized and then restored.  The menu
   // items usually appear in the lower left corner,
   // detached from the frame.  Changed HWND_DESKTOP to
   // HWND_OBJECT
   WinSetParent(mn.submenuHandle(), HWND_OBJECT, false);  //KKL ? is needed?

   unsigned long afStyle = WinQueryWindowULong(mn.submenuHandle(), QWL_STYLE);
   WinSetWindowUShort(mn.submenuHandle(),  //Menu window ID is create with
                      QWS_ID,              //FID_MENU. Change it.
                      (unsigned short)itemId);
   afStyle &= (unsigned long)
                ~(MS_CONDITIONALCASCADE |  //Menu is create as menu-bar
                  MS_VERTICALFLIP |        //Change it.
                  MS_ACTIONBAR);
   WinSetWindowULong(mn.submenuHandle(), QWL_STYLE, afStyle);
#endif
   IMenu::setItem(mn);  //set it.
   return *this;
}

/*------------------------------------------------------------------------------
| IMenu::addSeparator                                                          |
------------------------------------------------------------------------------*/
IMenu& IMenu::addSeparator( unsigned long intoSubmenuId )
{
   return IMenu::addSeparator( 0, intoSubmenuId );
}

/*------------------------------------------------------------------------------
| IMenu::setSubmenu                                                            |
------------------------------------------------------------------------------*/
IMenu& IMenu::setSubmenu( unsigned long        itemId,
                          const IResourceId&   submenuResId )
{
   IMODTRACE_DEVELOP("IMenu::setSubmenu(2)");
   IASSERTPARM(itemId != FID_MENU); //menu item cannot be menu-bar ID

   IMenuItem mn = menuItem(itemId);

#ifndef IC_WIN   // ModifyMenu will automatically destroy old pop-up
                 // and free memory associated with it
   if (mn.submenuHandle()) {
      WinDestroyWindow(mn.submenuHandle());
   } /* endif */
#endif

   mn.setSubmenuHandle( submenuResId.resourceLibrary().loadMenu
                                               (submenuResId, owner()));

#ifndef IC_WIN
   unsigned long afStyle = WinQueryWindowULong(mn.submenuHandle(), QWL_STYLE);

   WinSetWindowUShort(mn.submenuHandle(),  //Menu window ID is create with
                      QWS_ID,          //FID_MENU. Change it.
                      (unsigned short)itemId);
   afStyle &= (unsigned long)
                ~(MS_CONDITIONALCASCADE |  //Menu is create as menu-bar
                  MS_VERTICALFLIP |        //Change it.
                  MS_ACTIONBAR);
   WinSetWindowULong(mn.submenuHandle(), QWL_STYLE, afStyle);
#endif
   IMenu::setItem(mn);  //set it.
   return *this;
}

/*------------------------------------------------------------------------------
| IMenu::setItem                                                               |
| Replace an existing menu-item data with IMenuItem                            |
------------------------------------------------------------------------------*/
IMenu& IMenu::setItem( const IMenuItem&  menuItem )
{
  IMODTRACE_DEVELOP("IMenu::setItem(2)");
#ifdef IC_WIN
   // Depending on menu type, place identity into menu
   UINT  menu1 = menuItem.id();
   if (menuItem.isSubmenu())
     menu1 = (UINT)(menuItem.submenuHandle().asUnsigned());

   // Depending on menu type, place information into menu
   LPSTR menu2 = (LPSTR)(char*)menuItem.text();
   if (menuItem.isBitmap())
     menu2 = (LPSTR)(void*)menuItem.bitmap();

  if (!(Boolean) ModifyMenu( menuHandle(),
                             menuItem.id(),
                             menuItem.style() | MF_BYCOMMAND,
                             menu1, menu2 ));
  DrawMenuBar( fowner );

  // Ref count the bitmap if one passed in
  if (menuItem.isBitmap() && menuItem.bitmap() )
    refCountBitmap( menu2 );

#else
   MENUITEM mn;
   mn.iPosition    = (unsigned short)(menuItem.index());
   mn.afStyle      = (unsigned short)(menuItem.style());
   mn.afAttribute  = (unsigned short)(menuItem.attribute());
   mn.id           = (unsigned short)(menuItem.id());
   mn.hwndSubMenu  = menuItem.submenuHandle();
   mn.hItem        = menuItem.bitmap();

   if (!(Boolean)sendEvent(MM_SETITEM,
                           IEventData(0, true),
                           IEventData(&mn))) {
      ITHROWLIBRARYERROR(IC_SETMENUITEM_FAIL,
                         IErrorInfo::invalidRequest,
                         IException::recoverable);
   } /* endif */

   if (mn.afStyle & MIS_BITMAP) {
      if (mn.hItem) {
         if(!(Boolean)sendEvent( MM_SETITEMHANDLE,
                                 IEventData(mn.id),
                                 IEventData(mn.hItem))) {
            ITHROWLIBRARYERROR(IC_SETMENUITEM_FAIL,
                               IErrorInfo::invalidRequest,
                               IException::recoverable);
         } /* endif */
         refCountBitmap(mn.hItem);
      } else {
            ITHROWLIBRARYERROR(IC_NO_BITMAP_HANDLE,
                               IErrorInfo::invalidRequest,
                               IException::recoverable);
      } /* endif */
   } else {
      if (mn.afStyle & MIS_TEXT)
         IMenu::setText(mn.id, menuItem.text());
   } /* endif */
#endif
   return *this;
}

/*------------------------------------------------------------------------------
| IMenu::setText                                                               |
------------------------------------------------------------------------------*/
IMenu& IMenu::setText( unsigned long  menuItemId,
                       const char*    newText )
{
   IMODTRACE_DEVELOP("IMenu::setText");
#ifdef IC_WIN
   // Get the menuitem for the id passed in
   IMenuItem mn = menuItem(menuItemId);

   // Check and make sure that the item is a text item
   if ( mn.isText() )
   {
     // If so, change the item text and update the menu
     mn.setText( newText );
     setItem( mn );
   }
   else
   {
     ITHROWLIBRARYERROR( IC_SETMENUITEM_FAIL,
                         IErrorInfo::invalidRequest,
                         IException::recoverable);
   }
#else
   //No need to ensure menu item has textItem style.
   //WinSendMsg will fail, if style is not textItem.

   if(!(Boolean)sendEvent( MM_SETITEMTEXT,
                           IEventData((short)menuItemId),
                           IEventData((char*)newText))) {
      ITHROWLIBRARYERROR(IC_SETMENUITEM_FAIL,
                         IErrorInfo::invalidRequest,
                         IException::recoverable);
   } /* endif */
#endif
   return *this;
}

/*------------------------------------------------------------------------------
| IMenu::setText                                                               |
------------------------------------------------------------------------------*/
IMenu& IMenu::setText( unsigned long        menuItemId,
                       const IResourceId&   newTextResId )
{
  return IMenu::setText(menuItemId,
                        newTextResId.resourceLibrary().loadString(newTextResId));
}

/*------------------------------------------------------------------------------
| IMenu::setBitmap                                                             |
------------------------------------------------------------------------------*/
IMenu& IMenu::setBitmap( unsigned long        menuItemId,
                         const IBitmapHandle& bitmapHandle )
{
   //No need to ensure menu item has textItem style.
   //WinSendMsg will fail, if style is not textItem.
   IMenuItem mItem = menuItem(menuItemId);
   mItem.setBitmap(bitmapHandle);
   return IMenu::setItem(mItem);
}

/*------------------------------------------------------------------------------
| IMenu::setBitmap                                                             |
------------------------------------------------------------------------------*/
IMenu& IMenu::setBitmap( unsigned long        menuItemId,
                         const IResourceId&   newBitmapResId )
{

   return IMenu::setBitmap(menuItemId,
                   newBitmapResId.resourceLibrary().loadBitmap(newBitmapResId));
}

/*------------------------------------------------------------------------------
| IMenu::setBitmap                                                             |
| This version needed to resolved ambiguity                                    |
------------------------------------------------------------------------------*/
IMenu& IMenu::setBitmap( unsigned long        menuItemId,
                         unsigned long        newBitmapResId )
{
   return IMenu::setBitmap(menuItemId, IResourceId(newBitmapResId));
}

#ifndef IC_WIN
/*------------------------------------------------------------------------------
| IMenu::setConditionalCascade                                                 |
------------------------------------------------------------------------------*/
IMenu& IMenu::setConditionalCascade( unsigned long itemId,
                                     unsigned long defaultItemId )
{
  queryItem(handle(), itemId);   //Check is valid ID to cause exception.
  queryItem(handle(), defaultItemId);   //Check is valid ID to cause exception.
  IMenuItem mn = menuItem(itemId);
  if (mn.submenuHandle() == 0) {
      ITHROWLIBRARYERROR(IC_NO_SUBMENU,
                         IErrorInfo::invalidRequest,
                         IException::recoverable);
  } else {
     WinSetWindowBits(mn.submenuHandle(), QWL_STYLE,
                      MS_CONDITIONALCASCADE,
                      MS_CONDITIONALCASCADE);
     mn.submenuHandle()
       .sendEvent(MM_SETDEFAULTITEMID,
                  IEventData((unsigned short)defaultItemId,0), 0);
  } /* endif */
  return *this;
}

/*------------------------------------------------------------------------------
| IMenu::removeConditionalCascade                                              |
------------------------------------------------------------------------------*/
IMenu& IMenu::removeConditionalCascade( unsigned long itemId )
{
  queryItem(handle(), itemId);   //Check is valid ID to cause exception.
  IMenuItem mn = menuItem(itemId);
  if (mn.submenuHandle() == 0) {
      ITHROWLIBRARYERROR(IC_NO_SUBMENU,
                         IErrorInfo::invalidRequest,
                         IException::recoverable);
  } else {
     // determine the default item and remove its checkmark
     long defaultItemId =
            mn.submenuHandle().sendEvent( MM_QUERYDEFAULTITEMID, 0, 0);
     if (defaultItemId != MIT_ERROR &&
         defaultItemId != MIT_NONE &&
         defaultItemId != 0)
       uncheckItem(defaultItemId);

     // remove the conditional cascade style
     WinSetWindowBits(mn.submenuHandle(), QWL_STYLE,
                      (unsigned long)~MS_CONDITIONALCASCADE,
                      MS_CONDITIONALCASCADE);
  } /* endif */
  return *this;
}
#endif

/*------------------------------------------------------------------------------
| IMenu::deleteItem                                                            |
------------------------------------------------------------------------------*/
IMenu& IMenu::deleteItem( unsigned long ulItemId )
{
#ifdef IC_WIN
  queryItemState( menuHandle(), ulItemId );
  // IC_NOTYET - Should check be added for submenu here - need to look
  //             at OS/2 for consistency
  RemoveMenu( menuHandle(), ulItemId, MF_BYCOMMAND );
#else
  queryItem(handle(), ulItemId);   //Check is valid ID to cause exception.
  sendEvent(MM_DELETEITEM, IEventData((unsigned short)ulItemId, true), 0);
#endif
  return *this;
}

/*------------------------------------------------------------------------------
| IMenu::removeSubmenu                                                         |
------------------------------------------------------------------------------*/
IMenu& IMenu::removeSubmenu( unsigned long itemWithSubmenuId )
{
#ifndef IC_WIN
   // IC_NOTYET - Will use DeleteMenu for popups, but need to resolve
   //             id usage here since ids are not valid for popups
   IMenuItem mn  = menuItem(itemWithSubmenuId);
   if ( mn.submenuHandle() )
   {
      mn.setSubmenuHandle(0);
      unsigned long afStyle = mn.style();
      afStyle &= (unsigned long)~MIS_SUBMENU;
      mn.setStyle(afStyle);
      IMenu::setItem(mn);
   } /* endif */
#endif
   return *this;
}

/*------------------------------------------------------------------------------
| IMenu::checkItem                                                             |
------------------------------------------------------------------------------*/
IMenu& IMenu::checkItem( unsigned long ulItemId, Boolean check )
{
#ifdef IC_WIN
  UINT fuflags = MF_BYCOMMAND | MF_UNCHECKED;
  unsigned long menuState = queryItemState( menuHandle(), ulItemId );
#else
  MENUITEM miItem = queryItem(handle(), ulItemId);
  unsigned long menuState = miItem.afAttribute;
#endif
  Boolean bYes = false;

  if (check) {
     if (!(menuState & MIA_CHECKED)) {
        menuState |= MIA_CHECKED;
        bYes = true;
#ifdef IC_WIN
        fuflags |= MF_CHECKED;
#endif
     } /* endif */
  } else {
     if (menuState & MIA_CHECKED) {
        menuState &= (unsigned long)~MIA_CHECKED;
        bYes = true;
     } /* endif */
  } /* endif */

  if (bYes) {
#ifdef IC_WIN
     if( CheckMenuItem( menuHandle(), ulItemId, fuflags ) == 0xFFFFFFFF )
#else
     if(!(Boolean)sendEvent(MM_SETITEMATTR,
                            IEventData((unsigned short)ulItemId, true),
                            IEventData(MIA_CHECKED,
                                       (unsigned short)menuState )))
#endif
     {
       ITHROWLIBRARYERROR(IC_SETMENUITEM_FAIL,
                          IErrorInfo::invalidRequest,
                          IException::recoverable);
     }
  }

  return *this;
}

/*------------------------------------------------------------------------------
| IMenu::uncheckItem                                                           |
------------------------------------------------------------------------------*/
IMenu& IMenu::uncheckItem(unsigned long ulItemId)
{
  return checkItem(ulItemId, false);
}

/*------------------------------------------------------------------------------
| IMenu::isItemChecked                                                         |
------------------------------------------------------------------------------*/
Boolean IMenu::isItemChecked(unsigned long ulItemId) const
{
#ifdef IC_WIN
  unsigned long menuState = queryItemState( menuHandle(), ulItemId );
  return (Boolean)(menuState & MF_CHECKED);
#else
   //no exception required. PM returns false if ID is not found.
   return (Boolean) sendEvent(MM_QUERYITEMATTR,
                              IEventData((unsigned short)ulItemId, true),
                              IEventData(MIA_CHECKED));
#endif
}

/*------------------------------------------------------------------------------
| IMenu::enableItem                                                            |
------------------------------------------------------------------------------*/
IMenu& IMenu::enableItem(unsigned long ulItemId, Boolean enable)
{
#ifdef IC_WIN
  UINT fuflags = MF_BYCOMMAND | MF_UNCHECKED;
  unsigned long menuState = queryItemState( menuHandle(), ulItemId );
#else
  MENUITEM miItem = queryItem(handle(), ulItemId);
  unsigned long menuState = miItem.afAttribute;
#endif
  Boolean bYes = false;

  if (enable)
  {
#ifdef IC_WIN
     if (menuState & (MF_DISABLED | MF_GRAYED) )
        {
        fuflags |= MF_ENABLED;
        bYes = true;
        } /* endif */
#else
     if (menuState & MIA_DISABLED)
        {
        menuState &= (unsigned long)~MIA_DISABLED;
        bYes = true;
        } /* endif */
#endif
     }
  else
     {
#ifdef IC_WIN
     if ( !(menuState & (MF_DISABLED | MF_GRAYED)) )
        {
        fuflags |= MF_GRAYED;    // not MF_DISABLED to get visual too
        bYes = true;
        }
#else
     if (!(menuState & MIA_DISABLED))
        {
        menuState |= MIA_DISABLED;
        bYes = true;
        } /* endif */
#endif
  } /* endif */

  if (bYes) {
#ifdef IC_WIN
     if( EnableMenuItem( menuHandle(), ulItemId, fuflags ) == 0xFFFFFFFF )
#else
     if (!(Boolean)sendEvent(MM_SETITEMATTR,
                             IEventData((unsigned short)ulItemId, true),
                             IEventData(MIA_DISABLED,
                                        (unsigned short)menuState)))
#endif
     {
       ITHROWLIBRARYERROR(IC_SETMENUITEM_FAIL,
                          IErrorInfo::invalidRequest,
                          IException::recoverable);
     } /* endif */
  } /* endif */
  return *this;
}

/*------------------------------------------------------------------------------
| IMenu::disableItem                                                           |
------------------------------------------------------------------------------*/
IMenu& IMenu::disableItem(unsigned long ulItemId)
{
  return enableItem(ulItemId, false);
}

/*------------------------------------------------------------------------------
| IMenu::isItemEnabled                                                         |
------------------------------------------------------------------------------*/
Boolean IMenu::isItemEnabled(unsigned long ulItemId) const
{
#ifdef IC_WIN
  unsigned long menuState = queryItemState( menuHandle(), ulItemId );
  return (Boolean)( !(menuState & (MF_DISABLED | MF_GRAYED)) );
#else
  //no exception required. PM returns false if ID is not found.
  return (!((Boolean)sendEvent(MM_QUERYITEMATTR,
                               IEventData((unsigned short)ulItemId, true),
                               IEventData(MIA_DISABLED))));
#endif
}

/*------------------------------------------------------------------------------
| IMenu::selectItem                                                            |
------------------------------------------------------------------------------*/
IMenu& IMenu::selectItem( unsigned long itemId )
{
#ifdef IC_WIN
  queryItemState( menuHandle(), itemId );
  HiliteMenuItem( fowner, menuHandle(), itemId, MF_BYCOMMAND | MF_HILITE );
#else
  MENUITEM miItem = queryItem(handle(), itemId);
  sendEvent(MM_SELECTITEM,
            IEventData((unsigned short)itemId, true),
            IEventData(0, (unsigned short)
                            (!(miItem.afAttribute & MIA_NODISMISS))));
#endif
  return *this;
}

/*------------------------------------------------------------------------------
| IMenu::menuItem                                                              |
|  Query menu-item and return as IMenuItem object.                             |
------------------------------------------------------------------------------*/
IMenuItem IMenu::menuItem(unsigned long ulItemId) const
{
#ifdef IC_WIN
  unsigned long menuState = queryItemState( menuHandle(), ulItemId );
#else
  MENUITEM miItem = queryItem(handle(), ulItemId);
#endif
  IMenuItem mn(ulItemId);

#ifdef IC_WIN
  mn.setStyle( menuState & IMS_STYLEMASK );
  mn.setAttribute( menuState & IMS_ATTRMASK );

  if (!(menuState & (MF_BITMAP | MF_OWNERDRAW | MF_SEPARATOR)))
  {
    mn.setExtendedStyle( IMIS_TEXT );
    char* pszItemText = new char[100];
    UINT byFlag = MF_BYCOMMAND;
    GetMenuString( menuHandle(), ulItemId, pszItemText, 100, byFlag );
    mn.setText(pszItemText);
  }
#else
  mn.setIndex(miItem.iPosition);
  mn.setStyle(miItem.afStyle);
  mn.setAttribute(miItem.afAttribute);
  mn.setSubmenuHandle(miItem.hwndSubMenu);

  if (miItem.afStyle & MIS_TEXT) {

      /* Shoudn't need to throw exceptions here */
      unsigned long ulLen = sendEvent(MM_QUERYITEMTEXTLENGTH,
                                      IEventData(ulItemId),
                                      0);
      ++ulLen;
      char* pszItemText = new char[ulLen];
      sendEvent(MM_QUERYITEMTEXT,
                IEventData((unsigned short)ulItemId,
                           (unsigned short)ulLen),
                IEventData(pszItemText));

      mn.setText(pszItemText);
      delete pszItemText;
   } else {
      if (miItem.afStyle & MIS_BITMAP)
         mn.hBitmapHandle = miItem.hItem;
   } /* endif */
#endif

   return mn;
}

/*------------------------------------------------------------------------------
| IMenu::numberOfItems                                                         |
------------------------------------------------------------------------------*/
unsigned long IMenu::numberOfItems ( unsigned long forSubmenuId) const
{
#ifdef IC_WIN
   // IC_NOTYET - Still need to add support for submenus
   return (unsigned long)GetMenuItemCount( menuHandle() );
#else
   IMenuHandle hwnd = (forSubmenuId == 0) ?
                         menuHandle() : submenuHandle(forSubmenuId);
   return (unsigned long) hwnd.sendEvent(MM_QUERYITEMCOUNT, 0, 0);
#endif
}

/*------------------------------------------------------------------------------
| IMenu::itemRect                                                              |
|  Returns the bounding rectangle of a menu item                               |
------------------------------------------------------------------------------*/
IRectangle IMenu::itemRect( unsigned long itemId ) const
{
#ifdef IC_WIN
   //IC_NOTYET - Don't know if this can be supported in Windows
   return IRectangle();
#else
   PRECTL prectl = new RECTL;
   sendEvent(MM_QUERYITEMRECT,
             IEventData((unsigned short)itemId, true),
             IEventData(prectl));

   return IRectangle(prectl->xLeft, prectl->yBottom,
                     prectl->xRight, prectl->yTop);
#endif
}

/*------------------------------------------------------------------------------
| IMenu::menuItemId                                                            |
|  Return menu-item ID of item at cursor position                              |
------------------------------------------------------------------------------*/
unsigned long IMenu::menuItemId ( const Cursor& cursor ) const
{
   IASSERTSTATE( cursor.isValid());
#ifdef IC_WIN
   long id = GetMenuItemID( menuHandle(), (int)cursor.lCurrent );
   if ( id == 0xFFFFFFFF ) {
     if ((cursor.lCurrent < 0) ||
         (cursor.lCurrent >= GetMenuItemCount( menuHandle() )))
#else
   IMenuHandle hwnd = (cursor.ulSubmenu == 0) ?
                          cursor.pMenu->menuHandle() :
                          cursor.pMenu->submenuHandle(cursor.ulSubmenu);
   long id = (long)hwnd.sendEvent(MM_ITEMIDFROMPOSITION,
                                  (int)cursor.lCurrent,
                                  0);
   if (id == MIT_ERROR) {
     /*-------------------------------------------------------------------------
     | If menu item number is bad, then throw exception.  Otherwise the item   |
     | must be a MENUITEM SEPARATOR and the dummy value is valid               |
     -------------------------------------------------------------------------*/
     if ((cursor.lCurrent < 0) ||
         (cursor.lCurrent >= hwnd.sendEvent(MM_QUERYITEMCOUNT,0,0)))
#endif
     {
        ITHROWLIBRARYERROR(IC_INVALID_MENUITEM,
                           IErrorInfo::invalidRequest,
                           IException::recoverable);
     }
   } /* endif */
   return id;
}

#ifndef IC_WIN
/*------------------------------------------------------------------------------
| IMenu::foregroundColor                                                       |
|                                                                              |
| Returns the forground color of the menu.                                     |
------------------------------------------------------------------------------*/
IColor IMenu::foregroundColor () const
{
  return (IWindow::color(PP_MENUFOREGROUNDCOLOR,
                         IGUIColor(IGUIColor::menuText)));
}

/*------------------------------------------------------------------------------
| IMenu::backgroundColor                                                       |
|                                                                              |
| Returns the backround color of the menu.                                     |
------------------------------------------------------------------------------*/
IColor IMenu::backgroundColor () const
{
  return (IWindow::color(PP_MENUBACKGROUNDCOLOR,
                         IGUIColor(IGUIColor::menuBgnd)));
}

/*------------------------------------------------------------------------------
| IMenu::disabledForegroundColor                                               |
|                                                                              |
| Returns the disabled foreground color of the menu.                           |
------------------------------------------------------------------------------*/
IColor IMenu::disabledForegroundColor () const
{
  return (IWindow::color(PP_MENUDISABLEDFGNDCOLOR,
                         IGUIColor(IGUIColor::disableMenuText)));
}

/*------------------------------------------------------------------------------
| IMenu::disabledBackgroundColor                                               |
|                                                                              |
| Returns the disabled background color of the menu.                           |
------------------------------------------------------------------------------*/
IColor IMenu::disabledBackgroundColor () const
{
  return (IWindow::color(PP_MENUDISABLEDBGNDCOLOR,
                         IGUIColor(IGUIColor::menuBgnd)));
}

/*------------------------------------------------------------------------------
| IMenu::hiliteForegroundColor                                                 |
|                                                                              |
| Returns the hilite foreground color of the menu.                             |
------------------------------------------------------------------------------*/
IColor IMenu::hiliteForegroundColor () const
{
  return (IWindow::color(PP_MENUHILITEFGNDCOLOR,
                         IGUIColor(IGUIColor::menuHiliteText)));
}

/*------------------------------------------------------------------------------
| IMenu::hiliteBackgroundColor                                                 |
|                                                                              |
| Returns the hilite background color of the menu.                             |
------------------------------------------------------------------------------*/
IColor IMenu::hiliteBackgroundColor () const
{
  return (IWindow::color(PP_MENUHILITEBGNDCOLOR,
                         IGUIColor(IGUIColor::menuHiliteBgnd)));
}

/*------------------------------------------------------------------------------
| IMenu::setForegroundColor                                                    |
|                                                                              |
| Sets the foreground color of the menu.                                       |
------------------------------------------------------------------------------*/
IMenu& IMenu::setForegroundColor (const IColor &color)
{
  IWindow::setColor(PP_MENUFOREGROUNDCOLOR, color);
  return *this;
}

/*------------------------------------------------------------------------------
| IMenu::setBackgroundColor                                                    |
|                                                                              |
| Sets the background color of the menu.                                       |
------------------------------------------------------------------------------*/
IMenu& IMenu::setBackgroundColor (const IColor &color)
{
  IWindow::setColor(PP_MENUBACKGROUNDCOLOR, color);
  return *this;
}

/*------------------------------------------------------------------------------
| IMenu::setDisabledForegroundColor                                            |
|                                                                              |
| Sets the disabled foreground color of the menu.                              |
------------------------------------------------------------------------------*/
IMenu& IMenu::setDisabledForegroundColor (const IColor &color)
{
  IWindow::setColor(PP_MENUDISABLEDFGNDCOLOR, color);
  return *this;
}

/*------------------------------------------------------------------------------
| IMenu::setDisabledBackgroundColor                                            |
|                                                                              |
| Sets the disabled background color of the menu.                              |
------------------------------------------------------------------------------*/
IMenu& IMenu::setDisabledBackgroundColor (const IColor &color)
{
  IWindow::setColor(PP_MENUDISABLEDBGNDCOLOR, color);
  return *this;
}

/*------------------------------------------------------------------------------
| IMenu::setHiliteForegroundColor                                              |
|                                                                              |
| Sets the hilite foreground color of the menu.                                |
------------------------------------------------------------------------------*/
IMenu& IMenu::setHiliteForegroundColor (const IColor &color)
{
  IWindow::setColor(PP_MENUHILITEFGNDCOLOR, color);
  return *this;
}

/*------------------------------------------------------------------------------
| IMenu::setHiliteBackgroundColor                                              |
|                                                                              |
| Sets the hilite background color of the menu.                                |
------------------------------------------------------------------------------*/
IMenu& IMenu::setHiliteBackgroundColor (const IColor &color)
{
  IWindow::setColor(PP_MENUHILITEBGNDCOLOR, color);
  return *this;
}

/*------------------------------------------------------------------------------
| IMenu::resetForegroundColor                                                  |
|                                                                              |
| Resets the foreground color by undoing a previous set.                       |
------------------------------------------------------------------------------*/
IMenu& IMenu::resetForegroundColor ()
{
  resetColor(PP_MENUFOREGROUNDCOLOR);
  return *this;
}

/*------------------------------------------------------------------------------
| IMenu::resetBackgroundColor                                                  |
|                                                                              |
| Resets the background color by undoing a previous set.                       |
------------------------------------------------------------------------------*/
IMenu& IMenu::resetBackgroundColor ()
{
  resetColor(PP_MENUBACKGROUNDCOLOR);
  return *this;
}

/*------------------------------------------------------------------------------
| IMenu::resetDisabledForegroundColor                                          |
|                                                                              |
| Resets the disabled foreground color by undoing a previous set.              |
------------------------------------------------------------------------------*/
IMenu& IMenu::resetDisabledForegroundColor ()
{
  resetColor(PP_MENUDISABLEDFGNDCOLOR);
  return *this;
}

/*------------------------------------------------------------------------------
| IMenu::resetDisabledBackgroundColor                                          |
|                                                                              |
| Resets the disabled background color by undoing a previous set.              |
------------------------------------------------------------------------------*/
IMenu& IMenu::resetDisabledBackgroundColor ()
{
  resetColor(PP_MENUDISABLEDBGNDCOLOR);
  return *this;
}

/*------------------------------------------------------------------------------
| IMenu::resetHiliteForegroundColor                                            |
|                                                                              |
| Resets the hilite foreground color by undoing a previous set.                |
------------------------------------------------------------------------------*/
IMenu& IMenu::resetHiliteForegroundColor ()
{
  resetColor(PP_MENUHILITEFGNDCOLOR);
  return *this;
}

/*------------------------------------------------------------------------------
| IMenu::resetHiliteBackgroundColor                                            |
|                                                                              |
| Resets the hilite background color by undoing a previous set.                |
------------------------------------------------------------------------------*/
IMenu& IMenu::resetHiliteBackgroundColor ()
{
  resetColor(PP_MENUHILITEBGNDCOLOR);
  return *this;
}
#endif

/*------------------------------------------------------------------------------
| IMenu::Style  IMenu:: defaultStyle                                           |
------------------------------------------------------------------------------*/
IMenu::Style  IMenu:: defaultStyle()
{
  return currentDefaultStyle;
}

/*------------------------------------------------------------------------------
| IMenu::convertToGUIStyle                                                     |
|                                                                              |
| Returns base style for the control by default, or extended style if          |
| extended flag (bExtOnly) is set.                                             |
------------------------------------------------------------------------------*/
unsigned long IMenu::convertToGUIStyle(const IBitFlag& guiStyle,
                                       Boolean bExtOnly) const
{
  // Obtain the style from the class (IWindow) that we inherit from
  unsigned long ulStyle = Inherited::convertToGUIStyle( guiStyle, bExtOnly );

  if (bExtOnly)
  {
    // Use mask to only return extended styles in the user defined range
    ulStyle |= extendedStyle() & IS_EXTMASK;
  }
  else
  {
    // MS_ styles have a one-to-one correspondence to our style bits, and
    // inhabit the lower word of the base GUI style.  Therefore, obtain
    // that portion asis, masking out the upper word.
    ulStyle |= guiStyle.asUnsignedLong() & IMS_MASK;
  }

  return( ulStyle );
}


/*------------------------------------------------------------------------------
| IMenu::setItemHelpId                                                         |
|                                                                              |
------------------------------------------------------------------------------*/
IMenu& IMenu::setItemHelpId( unsigned long  menuItemId,
                             unsigned long  helpTopicId )
{
#ifdef IC_PM
  // IC_NOTYET  help stuff needs testing
  IFrameWindow* fmenuFrame( (IFrameWindow*)this->owner() );
  if ( fmenuFrame && !fmenuFrame->isFrameWindow() )
     fmenuFrame = 0;
#endif
  IMenuPrivate::setMenuItemHelpId( fmenuFrame, menuItemId, helpTopicId );
  return *this;
}


/*------------------------------------------------------------------------------
| IMenu::itemHelpId                                                            |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IMenu::itemHelpId( unsigned long  menuItemId ) const
{
#ifdef IC_PM
  // IC_NOTYET  help stuff needs testing
  IFrameWindow* fmenuFrame( (IFrameWindow*)this->owner() );
  if ( fmenuFrame && !fmenuFrame->isFrameWindow() )
     fmenuFrame = 0;
#endif
  return IMenuPrivate::queryMenuItemHelpId( fmenuFrame, menuItemId );
}


/*------------------------------------------------------------------------------
| IMenu::Cursor::Cursor                                                        |
| Cursor ctor.                                                                 |
------------------------------------------------------------------------------*/
IMenu::Cursor::Cursor(const IMenu&  menu, unsigned long forSubmenuId )
  : lCurrent(0), pMenu((IMenu*)&menu), ulSubmenu(forSubmenuId), fCursorData(0)
{
  IASSERTSTATE(menu.isValid());
  // If we have a forSubmenuId ensure it exists in menu
  // queryItem will throw if forSubmenuId not found in menu
#ifndef IC_WIN
  if (forSubmenuId)
    MENUITEM tempItem = queryItem(menu.handle(), forSubmenuId);
#endif
}

/*------------------------------------------------------------------------------
| IMenu::Cursor::Cursor                                                        |
| Copy ctor.                                                                   |
------------------------------------------------------------------------------*/
IMenu::Cursor::Cursor(const IMenu::Cursor& cursor )
  : lCurrent(cursor.lCurrent), pMenu(cursor.pMenu), ulSubmenu(cursor.ulSubmenu),
    fCursorData(0)
{
}

/*------------------------------------------------------------------------------
| IMenu::Cursor :: ~Cursor                                                     |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
IMenu::Cursor :: ~Cursor() { }

/*------------------------------------------------------------------------------
| IMenu::Cursor IMenu::cursor                                                  |
------------------------------------------------------------------------------*/
IMenu::Cursor IMenu::cursor ( unsigned long itemId,
                              unsigned long inSubmenuId ) const
{
   IMenuHandle hwnd = (inSubmenuId == 0) ?
                         menuHandle() : submenuHandle(inSubmenuId);
#ifdef IC_WIN
   long id;
   if (!locateMenuItem( menuHandle(), itemId, &id )) {
#else
   long id = hwnd.sendEvent(MM_ITEMPOSITIONFROMID,
                            IEventData((unsigned short)itemId, false),
                            0);
   if (id == MIT_NONE) {
#endif
       ITHROWLIBRARYERROR(IC_INVALID_MENUITEM,
                          IErrorInfo::invalidRequest,
                          IException::recoverable);
   } /* endif */
   IMenu::Cursor cursor(*this, inSubmenuId);
   cursor.lCurrent = id;
   return cursor;
}

/*------------------------------------------------------------------------------
| IMenu::Cursor :: setToLast                                                   |
------------------------------------------------------------------------------*/
Boolean IMenu::Cursor::setToLast()
{
  try {
     lCurrent = pMenu->numberOfItems(ulSubmenu) - 1;
  }
  // Want to return false like the other cursor functions if
  // ulSubmenu doesn't exist
  catch(...) {
     lCurrent = -1;
  }
  return (lCurrent < 0) ? false : true;
}

/*------------------------------------------------------------------------------
| IMenu::refCountBitmap                                                        |
| Add reference count on bitmap handle.                                        |
------------------------------------------------------------------------------*/
void IMenu::refCountBitmap(const IBitmapHandle& bitmapHandle)
{
   IBitmapHandle* pBmp = new IBitmapHandle(bitmapHandle);

   if (pBitmapList) {
      BITMAPLIST* pNew = new BITMAPLIST;
      pNew->pBitmap = pBmp;
      pNew->pNext   = (BITMAPLIST*)pBitmapList;
      pBitmapList   = pNew;
   } else {
      BITMAPLIST* pList = new BITMAPLIST;          //Create new record.
      pList->pBitmap = pBmp;
      pList->pNext   = 0;
      pBitmapList    = (void*)pList;
   } /* endif */
}

/*------------------------------------------------------------------------------
| IMenu::submenuHandle                                                         |
------------------------------------------------------------------------------*/
IMenuHandle IMenu::submenuHandle ( unsigned long ulInSubmenuId ) const
{
#ifdef IC_WIN
  // IC_NOTYET - Because Windows does not allow submenus to have
  //             ids (all done by position), we will have to emulate
  //             this under the covers - Potentially using a linked list
  //             for a menu to map ids to the submenus inserted at
  //             certain positions.
  return IMenuHandle( 0 );
#else
  MENUITEM miItem = queryItem(handle(), ulInSubmenuId);
  return IMenuHandle( miItem.hwndSubMenu );
#endif
}

/*------------------------------------------------------------------------------
| IMenu::addText                                                               |
------------------------------------------------------------------------------*/
IMenu& IMenu::addText( unsigned long      newItemId,
                       const char*        itemText,
                       unsigned long      intoSubmenuId )
{
   IMenuItem mn(newItemId);
   mn.setText(itemText);
   return addItem(mn, intoSubmenuId);
}


/*------------------------------------------------------------------------------
| IMenu::addText                                                               |
------------------------------------------------------------------------------*/
IMenu& IMenu::addText( unsigned long      newItemId,
                       const IResourceId& textResId,
                       unsigned long      intoSubmenuId )
{
   IMenuItem mn(newItemId);
   mn.setText(textResId);
   return addItem(mn, intoSubmenuId);
}

/*------------------------------------------------------------------------------
| IMenu::addBitmap                                                             |
------------------------------------------------------------------------------*/
IMenu& IMenu::addBitmap( unsigned long        newItemId,
                         const IResourceId&   bitmapResId,
                         unsigned long        intoSubmenuId )
{
   IMenuItem mn(newItemId);
   mn.setBitmap(bitmapResId);
   return addItem(mn, intoSubmenuId);
}

/*------------------------------------------------------------------------------
| IMenu::addBitmap                                                             |
------------------------------------------------------------------------------*/
IMenu& IMenu::addBitmap( unsigned long        newItemId,
                         unsigned long        bitmapResId,
                         unsigned long        intoSubmenuId )
{
   IMenuItem mn(newItemId);
   mn.setBitmap(IResourceId(bitmapResId));
   return addItem(mn, intoSubmenuId);
}

/*------------------------------------------------------------------------------
| IMenu::addBitmap                                                             |
------------------------------------------------------------------------------*/
IMenu& IMenu::addBitmap( unsigned long        newItemId,
                         const IBitmapHandle& bitmapHandle,
                         unsigned long        intoSubmenuId )
{
   IMenuItem mn(newItemId);
   mn.setBitmap(bitmapHandle);
   return addItem(mn, intoSubmenuId);
}


/*------------------------------------------------------------------------------
| IMenu::addSeparator                                                          |
------------------------------------------------------------------------------*/
IMenu& IMenu::addSeparator( unsigned long  newItemId,
                            unsigned long  intoSubmenuId )
{
   IMenuItem mn(newItemId, IMenuItem::separator);
   return addItem(mn, intoSubmenuId);
}

/*------------------------------------------------------------------------------
| IBase::Boolean IMenu::Cursor :: setToNext                                    |
------------------------------------------------------------------------------*/
IBase::Boolean IMenu::Cursor::setToNext()
{
  if (lCurrent >= 0 && lCurrent < pMenu->numberOfItems(ulSubmenu) - 1 ) {
     ++lCurrent;
     return true;
  } else {
     lCurrent = -1L;
     return false;
  } /* endif */
}

/*------------------------------------------------------------------------------
| IMenu::deleteAt                                                              |
------------------------------------------------------------------------------*/
IMenu& IMenu::deleteAt( Cursor& cursor )
{
   cursor.pMenu->deleteItem(cursor.pMenu->menuItemId(cursor));
   if (cursor.lCurrent > 0)
     --cursor.lCurrent;
   //------------------------------------------------------------------
   // Defect 9796/9797: Check if we removed the only item in the menu.
   //                   If so, then invalidate the cursor.
   //------------------------------------------------------------------
   else if ( this->numberOfItems( cursor.ulSubmenu ) == 0 )
     cursor.invalidate();
   return *this;
}

/*------------------------------------------------------------------------------
| IMenu::add                                                                   |
------------------------------------------------------------------------------*/
IMenu& IMenu::add( IMenuItem& menuItem, Cursor& cursor )
{
   menuItem.setIndex(cursor.lCurrent);
   return cursor.pMenu->addItem(menuItem, cursor.ulSubmenu);
}

/*------------------------------------------------------------------------------
| IMenu::addAsNext                                                             |
------------------------------------------------------------------------------*/
IMenu& IMenu::addAsNext( IMenuItem& menuItem, Cursor& cursor )
{
   menuItem.setIndex(cursor.lCurrent + 1);
   cursor.pMenu->addItem(menuItem, cursor.ulSubmenu);
   cursor.setToNext();
   return *this;
}

/*------------------------------------------------------------------------------
  Have the cursor point to the first element
------------------------------------------------------------------------------*/
IBase::Boolean IMenu::Cursor::setToFirst()
{
  lCurrent = 0;
  return true;
}

/*------------------------------------------------------------------------------
  Have the cursor point to the previous element
------------------------------------------------------------------------------*/
IBase::Boolean IMenu::Cursor::setToPrevious()
{
  --lCurrent;
  return (lCurrent < 0) ? false : true;
}

/*------------------------------------------------------------------------------
  Invalidate the cursor
------------------------------------------------------------------------------*/
IBase::Boolean IMenu::Cursor::isValid() const
{
  return (lCurrent < 0) ? false : true;
}

/*------------------------------------------------------------------------------
  Invalidate the cursor
------------------------------------------------------------------------------*/
void IMenu::Cursor::invalidate()
{
  lCurrent = -1L;
}

/*------------------------------------------------------------------------------
  Get menu item at cursor position.
------------------------------------------------------------------------------*/
IMenuItem IMenu::elementAt( const Cursor& cursor ) const
{
   return cursor.pMenu->menuItem(cursor.pMenu->menuItemId(cursor));
}

/*------------------------------------------------------------------------------
  Remove submenu at cursor position.
------------------------------------------------------------------------------*/
IMenu& IMenu::removeSubmenuAt( Cursor& cursor )
{
   return cursor.pMenu->removeSubmenu(menuItemId(cursor));
}

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
  Function to query menu handle for owner window
------------------------------------------------------------------------------*/
IMenuHandle IMenu::menuHandle() const
{
#ifdef IC_WIN
   return fhMenu;
#endif
#ifdef IC_PM
   return (IMenuHandle) handle();
#endif
}
#endif
