/*
 * This file is part of the portable Forth environment written in ANSI C.
 * Copyright (C) 1993  Dirk Uwe Zoller
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This file is version 0.9.5 of 15-May-94
 * Check for the latest version of this package via anonymous ftp at
 *	roxi.rz.fht-mannheim.de
 *	/pub/unix/languages/pfe-VERSION.tar.gz
 * Please direct any comments via internet to
 *	duz@roxi.rz.fht-mannheim.de.
 * Thank You.
 */
/*
 * support.c ---	Subroutines for the Forth-System
 * (duz 09Jul93)
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>
#include <setjmp.h>

#include "config.h"
#include "term.h"
#include "lined.h"
#include "forth.h"
#include "support.h"
#include "compiler.h"
#include "dblsub.h"


Cell
aligned (Cell n)
{
  while (!ALIGNED (n)) n++;
  return n;
}

/*****************************************************************************/
/* miscellaneous execution semantics and runtimes			     */
/*****************************************************************************/

void
sysvar_runtime (void)
{
  *--sp = PFA [0];
}

void
dictvar_runtime (void)
{
  *--sp = (Cell)((char *)sys.dict + PFA [0]);
}

void
sysconst_runtime (void)
{
  *--sp = **(Cell **)PFA;
}

void				/* compiles the execution semantics */
compile1 (void)			/* of a state smart word */
{
  question_comp_();
  COMMA (&((Semant **)w) [-1]->exec [0]);
}

void				/* compiles the alternative exec.sem. */
compile2 (void)			/* of an even smarter word (e.g. TO) */
{
  question_comp_();
  COMMA (&((Seman2 **)w) [-1]->exec [1]);
}


/*****************************************************************************/
/* strings								     */
/*****************************************************************************/

void
to_lower (char *p, int n)	/* tolower() applied to counted string */
{
  while (--n >= 0)
    *p = tolower (*p), p++;
}

void
to_upper (char *p, int n)	/* toupper() applied to counted string */
{
  while (--n >= 0)
    *p = toupper (*p), p++;
}

#ifndef HAVE_STPCPY
#define stpcpy(DST,SRC) (strcpy(DST,SRC) + strlen(SRC))
#endif

char *
expand_filename (const char *nm, const char *path,
		 const char *ext, char *fn)
/*
 * nm	file name input, short
 * path	search path for files
 * ext	default file extensions
 * fn	full file name, output
 */
{
  char buf [256], *p, *q;
  const char *e;
  char *home = getenv ("HOME");

  if (path == NULL)
    path = ".";
  while (*path)
    {
      p = buf;
      while (*path == PATH_DELIMITER) path++;
      while (*path && *path != PATH_DELIMITER)
	if (*path == '~' && home)
	  p = stpcpy (p, home), path++;
	else
	  *p++ = *path++;
      *p++ = DIR_DELIMITER;
      p = stpcpy (p, nm);
      if (file_access (buf) >= 0)
	{
	  strcpy (fn, buf);
	  return fn;
	}
      for (e = ext; *e; )
	{
	  q = p;
	  while (*e == PATH_DELIMITER) e++;
	  while (*e && *e != PATH_DELIMITER)
	    *q++ = *e++;
	  *q = '\0';
	  if (file_access (buf) >= 0)
	    {
	      strcpy (fn, buf);
	      return fn;
	    }
	}
    }
  strcpy (fn, nm);
  return fn;
}

char *
search (char *p1, int u1, const char *p2, int u2)
{
  if (u2 == 0)
    return p1;
  if (u2 > u1)
    return NULL;
  u1 -= u2;
  for (;;)
    {
      char *p = (char *)memchr (p1, *(Byte *)p2, u1 + 1);
      if (p == NULL)
	return NULL;
      if (memcmp (p, p2, u2) == 0)
	return p;
      u1 -= p - p1;
      if (u1 == 0)
	return NULL;
      p1 = p + 1;
      u1--;
    }
}


/*****************************************************************************/
/* unsigned and floored divide and number i/o conversion		     */
/*****************************************************************************/

udiv_t
udiv (uCell num, uCell denom)	/* unsigned divide procedure, single prec */
{
  udiv_t res;

  res.quot = num / denom;
  res.rem = num % denom;
  return res;
}

fdiv_t
fdiv (Cell num, Cell denom)	/* floored divide procedure, single prec */
{
  fdiv_t res;

  res.quot = num / denom;
  res.rem = num % denom;
  if (res.rem && (num ^ denom) < 0)
    res.quot--,
    res.rem += denom;
  return res;
}

uCell
u_d_div (udCell *ud, uCell denom)
/* Divides *ud by denom, leaves result in *ud, returns remainder. */
/* For number output conversion: dividing by BASE. */
{
  udCell nom = *ud;
  udiv_t h;

  h = udiv (D0 (nom), denom);
  D0 (*ud) = h.quot;
  D0 (nom) = h.rem;
  h = udiv (nom.hi, denom);
  D1 (*ud) = h.quot;
  D1 (nom) = h.rem;
  h = udiv (CELL (D1 (nom), D2 (nom)), denom);
  D2 (*ud) = h.quot;
  D2 (nom) = h.rem;
  h = udiv (nom.lo, denom);
  D3 (*ud) = h.quot;
  return h.rem;
}

void
u_d_mul (udCell *ud, uCell w, uCell c)
/* Computes *ud * w + c, where w is actually only half of a Cell in size. */
/* Leaves result in *ud. */
/* For number input conversion: multiply by BASE and add digit. */
{
  c += D3 (*ud) * w; D3 (*ud) = W1 (c); c >>= HALFCELL;
  c += D2 (*ud) * w; D2 (*ud) = W1 (c); c >>= HALFCELL;
  c += D1 (*ud) * w; D1 (*ud) = W1 (c); c >>= HALFCELL;
  D0 (*ud) = D0 (*ud) * w + c;
}

int
dig2num (Byte c, uCell *n, uCell base)
/* get value of digit c into *n, return flag: valid digit */
{
  if (c < '0')
    return FALSE;
  if (c <= '9')
    c -= '0';
  else
    {
      if (LOWER_CASE)
	c = toupper (c);
      if (c < 'A')
	return FALSE;
      if (c <= 'Z')
	c -= 'A' - ('9' - '0' + 1);
      else
	{
	  if (LOWER_CASE || c < 'a')
	    return FALSE;
	  c -= 'a' - ('9' - '0' + 1) - ('Z' - 'A' + 1);
	}
    }
  if (c >= base)
    return FALSE;
  *n = c;
  return TRUE;
}

char
num2dig (uCell n)		/* make digit */
{
  return (char)(n <= 9 ? n + '0' : n + 'A' - 10);
}

void
hold (char c)			/* insert into pictured numeric output string*/
{
  if (HLD <= (char *)DP)
    tHrow (THROW_PICNUM_OVER);
  *--HLD = c;
}

const char *
to_number (const char *p, uCell *n, udCell *d, uCell base)
{
  for ( ; *n > 0; p++, --*n)
    {
      uCell c;
      if (!dig2num (*p, &c, base))
	break;
      u_d_mul (d, base, c);
      if (DPL >= 0)
	DPL++;
    }
  return p;
}

int
number_question (const char *p, uCell n, dCell *d)
{
  uCell base = 0;
  int sign = 0;

  for ( ; n; p++, n--)
    {
      switch (*p)
	{
	default:
	  break;
	case '-':
	  if (sign) return 0;
	  sign = 1;
	  continue;
#if PREFIX_HEX
	case PREFIX_HEX:
	  if (base) return 0;
	  base = 16;
	  continue;
#endif
#if PREFIX_BINARY
	case PREFIX_BINARY:
	  if (base) return 0;
	  base = 2;
	  continue;
#endif
	}
      break;
    }
  if (base == 0)
    base = BASE;
  d->lo = d->hi = 0;
  DPL = -1;
  p = to_number (p, &n, (udCell *)d, base);
  if (n == 0) goto happy;
  if (*p != '.') return 0;
  DPL = 0; p++; n--;
  p = to_number (p, &n, (udCell *)d, base);
  if (n != 0) return 0;
 happy:
  if (sign) dnegate (d);
  return 1;
}

Flag
to_float (char *p, Cell n, double *f)
{
  char buf [80], *q;
  double res;

  while (n > 0 && isspace (p [n - 1]))
    n--;
  if (n == 0)
    {
      *f = 0;
      return TRUE;
    }
  store_asciiz (p, n, buf, sizeof buf);
  if (LOWER_CASE)
    to_upper (buf, n);
  q = strrchr (buf, 'D');
  if (q) *q = 'E';
  if (buf [n - 1] == 'E')
    {
      buf [n++] = '0';
      buf [n] = '\0';
    }
  res = strtod (buf, &p);
  if (p && p != buf + n)
    return FALSE;
  *f = res;
  return TRUE;
}

/* These are for internal use only (SEE and debugger), */
/* The real `UD.R' etc. words use HOLD and the memory area below PAD */

char *
str_ud_dot_r (udCell ud, char *p, int w, int base)
{
  *--p = '\0';
  do
    {
      *--p = num2dig (u_d_div (&ud, base));
      w--;
    }
  while (ud.lo || ud.hi);
  while (w > 0)
    *--p = ' ', w--;
  return p;
}

char *
str_d_dot_r (dCell d, char *p, int w, int base)
{
  int sign = 0;

  if (d.hi < 0)
    dnegate (&d), sign = 1;
  *--p = '\0';
  do
    {
      *--p = num2dig (u_d_div ((udCell *)&d, base));
      w--;
    }
  while (d.lo || d.hi);
  if (sign)
    *--p = '-', w--;
  while (w > 0)
    *--p = ' ', w--;
  return p;
}

char *
str_dot (Cell n, char *p, int base)
{
  dCell d;
  char *bl;

  *--p = '\0';
  bl = p - 1;
  d.lo = n;
  d.hi = n < 0 ? -1 : 0;
  p = str_d_dot_r (d, p, 0, base);
  *bl = ' ';
  return p;
}


/*****************************************************************************/
/* console i/o								     */
/*****************************************************************************/

/* output adjusting the OUT variable */

void
outc (char c)			/* emit single character */
{
  int x, y;

  cputc (c);
  wherexy (&x, &y);
  OUT = x;
}

void
outs (const char *s)		/* type a string */
{
  int x, y;

  cputs (s);
  wherexy (&x, &y);
  OUT = x;
}

int
outf (const char *s, ...)	/* type a string with formatting */
{
  char buf [0x100];
  va_list p;
  int r;

  va_start (p, s);
  r = vsprintf (buf, s, p);
  outs (buf);
  va_end (p);
  return r;
}

void
type (const char *s, Cell n)	/* TYPE counted string to terminal */
{
  int x, y;

  while (--n >= 0)
    cputc (*s++);
  fflush (stdout);
  wherexy (&x, &y);
  OUT = x;
}

void
spaces (int n)
{
  int x, y;

  while (--n >= 0)
    cputc (' ');
  fflush (stdout);
  wherexy (&x, &y);
  OUT = x;
}

void
tab (int n)
{
  spaces (n - OUT % n);
}

void
dot_line (Cell n, Cell l)
{
  char *p = block (n);
  type (p + l * 64, 64);
}


/* input */

int
expect (char *p, Cell n)	/* EXPECT counted string from terminal, */
{				/* simple editing facility with Backspace, */
  int i;			/* very traditional, use lined() instead! */
  char c;

  for (i = 0; i < n; )
    {
      switch (c = getkey ())
	{
	default:
	  p [i++] = c;
	  outc (c);
	  continue;
	case 27:
	  for ( ; i > 0; i--)
	    backspace_();
	  continue;
	case '\t':
	  while (i < n)
	    {
	      p [i++] = ' ';
	      space_();
	      if (OUT % 8 == 0)
		break;
	    }
	  continue;
	case '\r':
	case '\n':
	  space_();
	  goto fin;
	case 127:
	case '\b':
	  if (i <= 0)
	    {
	      bing ();
	      continue;
	    }
	  i--;
	  backspace_();
	  continue;
	}
    }
 fin: p [i] = 0;
  SPAN = i;
  return i;
}

int
accept (char *p, int n)		/* better input facility using lined() */
{
  extern struct lined accept_lined;

  accept_lined.string = p;
  accept_lined.max_length = n;
  lined (&accept_lined, NULL);
  space_();
  return accept_lined.length;
}

int
question_stop (void)		/* check for 'q' pressed */
{
  if (ekeypressed ())
    {
      if (tolower (getkey ()) == 'q')
	return 1;
      if (tolower (getkey ()) == 'q')
	return 1;
    }
  return 0;
}

int
question_cr (void)		/* like CR but stop after one screenful */
{				/* and return flag if 'q' pressed */
  cr_();
  if (sys.lines < sys.more)
    return 0;
  sys.lines = 0;
  for (;;)
    switch (tolower (getkey ()))
      {
      case 'q':
	return 1;
      case ' ':
	sys.more = rows - 2;
	return 0;
      case '\r':
      case '\n':
	sys.more = 1;
	return 0;
      default:
	bing ();
      }
}


/*****************************************************************************/
/* files								     */
/*****************************************************************************/

static char open_mode [][4] =	/* mode strings for fopen() */
{
  "r",  "w",  "r+",		/* R/O W/O R/W */
  "rb", "wb", "r+b"		/* after application of BIN */
};

void
file_error (void)
{
  tHrow (-256 - errno);
}

static File *
free_file_slot (void)
{
  File *f;
  for (f = membot.files; f < memtop.files; f++)
    if (f->f == NULL)
      {
	memset (f, 0, sizeof *f);
	return f;
      }
  return NULL;
}

File *
create_file (const char *name, int len, int mode)
{
  File *f = free_file_slot ();
  if (f == NULL)
    return NULL;
  store_asciiz (name, len, f->name, sizeof f->name);
  f->mode = mode;
  switch (mode)			/* force file existence */
    {				/*  if mode other than W/O (BIN) */
    case FMODE_RO:
    case FMODE_RW:
      fclose (fopen (f->name, "w"));
      break;
    case FMODE_ROB:
    case FMODE_RWB:
      fclose (fopen (f->name, "wb"));
      break;
    }
  strcpy (f->mdstr, open_mode [mode - FMODE_RO]);
  if ((f->f = fopen (f->name, f->mdstr)) == NULL)
    return NULL;
  f->size = 0;
  f->n = (unsigned)-1;
  return f;
}

File *
open_file (const char *name, int len, int mode)
{
  File *f = free_file_slot ();
  if (f == NULL)
    return NULL;
  store_asciiz (name, len, f->name, sizeof f->name);
  f->mode = mode;
  strcpy (f->mdstr, open_mode [mode - FMODE_RO]);
  if ((f->f = fopen (f->name, f->mdstr)) == NULL)
    return NULL;
  f->size = fsize (f->f) / BPBUF;
  f->n = (unsigned)-1;
  return f;
}

int
close_file (File *f)
{
  int res = 0;

  if (f->f)
    {
      res = fclose (f->f);
      f->f = NULL;
    }
  return res;
}

fpos_t
fsize (FILE *f)			/* Result: file length, -1 if error */
{
  fpos_t pos, len;

  pos = ftell (f);
  if (pos == -1)
    return -1;
  if (fseek (f, 0, SEEK_END) != 0)
    return -1;
  len = ftell (f);
  if (fseek (f, pos, SEEK_SET) != 0)
    return -1;
  return len;
}

int
reposition_file (fpos_t pos, File *f)
{
  return fseek (f->f, pos, SEEK_SET)
    ? errno : 0;
}

int
read_file (void *p, uCell *n, File *f)
{
  int m = fread (p, 1, *n, f->f);

  errno = 0;
  if (m != *n)
    {
      *n = m;
      return errno;
    }
  return 0;
}

int
write_file (void *p, uCell n, File *f)
{
  errno = 0;
  return fwrite (p, 1, n, f->f) != n
    ? errno : 0;
}

int
read_line (char *p, uCell *u, File *f, Cell *ior)
{
  int c, n;

  if (feof (f->f))
    {
      *u = 0;
      *ior = 0;
      return FALSE;
    }
  for (n = 0; n < *u; n++)
    switch (c = getc (f->f))
      {
      case EOF:
	if (!ferror (f->f))
	  goto happy;
	*u = n;
	*ior = errno;
	return FALSE;
      case '\r':
	c = getc (f->f);
	if (c != '\n')
	  ungetc (c, f->f);
      case '\n':
	goto happy;
      default:
	*p++ = c;
      }
 happy:	*u = n;
  *ior = 0;
  f->n++;
  return TRUE;
}

fpos_t
file_copy (const char *src, const char *dst, fpos_t limit)
/*
 * copies file, but maximum limit characters.
 * Returns destination file length if successful, -1 otherwise
 */
{
  FILE *f, *g;
  char buf [BUFSIZ];
  fpos_t n, m;

  if ((f = fopen (src, "rb")) == NULL)
    return 0;
  if ((g = fopen (dst, "wb")) == NULL)
    {
      fclose (f);
      return 0;
    }
  for (m = limit; m; m -= n)
    {
      n = BUFSIZ < m ? BUFSIZ : m;
      n = fread (buf, 1, n, f);
      if (n == 0 || n != fwrite (buf, 1, n, g))
	break;
    }
  n = ferror (f) || ferror (g);
  fclose (f);
  fclose (g);
  return n ? (fpos_t)-1 : limit - m;
}

int
file_move (const char *src, const char *dst)
{
  return rename (src, dst) == 0
    ? 1
    : file_copy (src, dst, LONG_MAX) != (fpos_t)-1 &&
      remove (src) == 0;
}

int
lengthen_file (const char *fn, fpos_t n)
{
  FILE *f = fopen (fn, "ab");

  if (f == NULL)
    return 0;
  while (n--)
    if (putc (' ', f) == EOF)
      {
	fclose (f);
	return 0;
      }
  fclose (f);
  return 1;
}

int
resize (const char *fn, fpos_t size)
/* returns 1 if successful, 0 otherwise */
{
  char tfn [L_tmpnam];
  fpos_t old, len;
  FILE *f;

  f = fopen (fn, "rb");
  if (f == NULL)
    return 0;
  old = fsize (f);
  fclose (f);
  if (old == (fpos_t)-1)
    return 0;
  if (old <= size)
    return lengthen_file (fn, size - old);
  tmpnam (tfn);
  len = file_copy (fn, tfn, size);
  if (len == size && remove (fn) == 0)
    return file_move (tfn, fn);
  remove (tfn);
  return 0;
}

int
resize_file (File *f, fpos_t size)
{
  int r;

  if (f == NULL || f->f == NULL)
    tHrow (THROW_FILE_NEX);
  fclose (f->f);
  r = resize (f->name, size);
  f->f = fopen (f->name, f->mdstr);
  return r;
}

int
systemf (const char *s, ...)	/* issue a system() call after formatting */
{
  char buf [0x100];
  va_list p;
  int r;

  va_start (p, s);
  vsprintf (buf, s, p);
  va_end (p);
  system_terminal ();
  swap_signals ();
  r = system (buf);
  swap_signals ();
  interactive_terminal ();
  normal ();
  return r;
}


/*****************************************************************************/
/* source input								     */
/*****************************************************************************/

/* 1. read from terminal */

code (query)
{
  SOURCE_ID = 0;
  BLK = 0;
  TO_IN = 0;
  TIB = membot.tib;
  NUMBER_TIB = accept (TIB, TIB_SIZE);
  SPAN = NUMBER_TIB;
}

/* 2. read from text-file */

int
next_line (void)
{
  Cell ior;

  SOURCE_FILE->len = sizeof SOURCE_FILE->buffer;
  if (!read_line (SOURCE_FILE->buffer, &SOURCE_FILE->len, SOURCE_FILE, &ior))
    return 0;
  TIB = SOURCE_FILE->buffer;
  NUMBER_TIB = SOURCE_FILE->len;
  BLK = 0;
  TO_IN = 0;
  return 1;
}

/* 3. read from block-file */

int
use_block_file (const char *name, int len)
{
  char nm [0x80], fn [0x100];
  int mode;
  File *f;

  store_asciiz (name, len, nm, sizeof nm);
  expand_filename (nm, option.blkpaths, option.blkext, fn);
  mode = file_access (fn);
  if (mode == -1)
    return FALSE;
  f = open_file (fn, strlen (fn), mode + FMODE_BIN);
  if (f == NULL)
    return FALSE;
  if (BLOCK_FILE)
    {
      save_buffers_();
      close_file (BLOCK_FILE);
    }
  BLOCK_FILE = f;
  return TRUE;
}

void
read_write (File *f, char *p, uCell n, Flag readflag)
/* very traditional block read/write primitive */
{
  fpos_t len;

  question_file_open (f);
  clearerr (f->f);
  if (n > f->size)
    tHrow (THROW_INVALID_BLOCK);
  if (readflag && n == f->size)
    {
      memset (p, ' ', BPBUF);
      return;
    }
  if (fseek (f->f, n * BPBUF, SEEK_SET) != 0)
    file_error ();
  if (readflag)
    {
      len = fread (p, 1, BPBUF, f->f);
      if (ferror (f->f))
	file_error ();
      memset (p + len, ' ', BPBUF - len);
    }
  else
    {
      len = fwrite (p, 1, BPBUF, f->f);
      if (len < BPBUF || ferror (f->f))
	file_error ();
      if (n == f->size)
	f->size++;
    }
  return;
}

char *
buffer (uCell n, int *reload)
{
  File *f;

  f = BLOCK_FILE;
  if (f->n != n)
    {
      if (f->updated)
	read_write (f, f->buffer, f->n, FALSE);
      f->n = n;
      *reload = 1;
    }
  else
    *reload = 0;
  return f->buffer;
}

char *
block (uCell n)
{
  char *p;
  int reload;

  p = buffer (n, &reload);
  if (reload)
    read_write (BLOCK_FILE, p, n, TRUE);
  return p;
}

/* dispatch input source */

void
source (char **p, int *n)
{
  switch (SOURCE_ID)
    {
    case -1:			/* string from EVALUATE */
      *p = TIB;
      *n = NUMBER_TIB;
      break;
    case 0:			/* string from QUERY or BLOCK */
      if (BLK)
	{
	  *p = block (BLK);
	  *n = BPBUF;
	}
      else
	{
	  *p = TIB;
	  *n = NUMBER_TIB;
	}
      break;
    default:			/* source line from text file */
      *p = SOURCE_FILE->buffer;
      *n = SOURCE_FILE->len;
    }
}

void *
save_input (void *p)
{
  DEC (p, Iframe);
  ((Iframe *)p)->magic = INPUT_MAGIC;
  ((Iframe *)p)->input = sys.input;
  ((Iframe *)p)->prev = sys.saved_input;
  sys.saved_input = (Iframe *)p;
  return p;
}

void *
restore_input (void *p)
{
  if (((Iframe *)p)->magic != INPUT_MAGIC)
    tHrow (THROW_ARG_TYPE);
  sys.input = ((Iframe *)p)->input;
  sys.saved_input = ((Iframe *)p)->prev;
  INC (p, Iframe);
  return p;
}

int
refill (void)
{
  switch (SOURCE_ID)
    {
    case -1:
      return 0;
    case 0:
      if (BLK)
	{
	  BLK++;
	  TO_IN = 0;
	}
      else
	query_();
      return 1;
    default:
      return next_line ();
    }
}

int
parse (char del, char **p, uCell *l)
{
  char *q;
  int i, n;

  source (&q, &n);
  q += TO_IN;
  n -= TO_IN;
  if (del == ' ')
    for (i = 0; i < n && !(isascii (q [i]) && isspace (q [i])); i++);
  else
    for (i = 0; i < n && q [i] != del; i++);
  *p = q;
  *l = i;
  TO_IN += i;
  if (i == n)
    return 0;
  TO_IN++;
  return 1;
}

char *
word (char del)			/* action of WORD callable from C functions */
{
  char *p, *q;
  int n, i;

  source (&q, &n);
  q += TO_IN;
  n -= TO_IN;
  if (del == ' ')
    for (i = 0; i < n && isascii (*q) && isspace (*q); i++)
      q++;
  else
    for (i = 0; i < n && *q == del; i++)
      q++;
  n -= i;
  TO_IN += i;
  p = (char *)DP + 1;
  if (del == ' ')
    for (i = 0; i < n && !(isascii (*q) && isspace (*q)); i++)
      *p++ = *q++;
  else
    for (i = 0; i < n && *q != del; i++)
      *p++ = *q++;
  TO_IN += i + (i < n);
  *p = '\0';
  *DP = i;
  return (char *)DP;
}


/*****************************************************************************/
/* inner and outer interpreter						     */
/*****************************************************************************/

Code (longjmp)			/* longjmp via *(jmp_buf*) following inline */
{				/* purpose: stop the inner interpreter */
  longjmp (**(jmp_buf **)ip, 1);
}

static pCode longjmp_p = longjmp_;

void
run_forth (Xt xt)		/* Run a forth word from within C-code. */
{				/* This is the inner interpreter. */
  Xt *saved_ip;
  Xt list [3];
  jmp_buf stop;

  list [0] = xt;
  list [1] = &longjmp_p;
  list [2] = (Xt)&stop;
  saved_ip = ip;
  ip = list;
  if (setjmp (stop))
    {
      ip = saved_ip;
      return;
    }
  for (;;)
    {
      w = *ip++;
      (**w) ();
    }
}

code (interpret)
{
  char *s, *p;
  int len;
  dCell d;
  double f;

  for (;;)
    {
      for (;;)
	{
	  p = word (' ');
	  if ((len = *(Byte *)p++) != 0)
	    break;
	  switch (SOURCE_ID)
	    {
	    default:
	      if (next_line ())
		continue;
	    case 0:
	    case -1:
	      return;
	    }
	}
      if (STATE)
	{
	  if (sys.locals && compile_local (p, len))
	    continue;
	  s = find (p, len);
	  if (s != NULL)
	    {
	      if (*s & IMMEDIATE)
		{
		  run_forth (name_from (s));
		  question_stack_();
		}
	      else
		COMMA (name_from (s));
	      continue;
	    }
	  if (number_question (p, len, &d))
	    {
	      if (DPL >= 0)
		{
		  COMPILE1 (two_literal);
		  COMMA (d.hi);
		}
	      else
		COMPILE1 (literal);
	      COMMA (d.lo);
	      continue;
	    }
	  if (BASE == 10 && to_float (p, len, &f))
	    {
#if DFLOAT_ALIGN > CELL_ALIGN
	      if (DFALIGNED (DP))
		COMPILE2 (f_literal);
#endif
	      COMPILE1 (f_literal);
	      FCOMMA (f);
	      continue;
	    }
	}
      else
	{
	  s = find (p, len);
	  if (s != NULL)
	    {
	      run_forth (name_from (s));
	      question_stack_();
	      continue;
	    }
	  if (number_question (p, len, &d))
	    {
	      *--sp = d.lo;
	      if (DPL >= 0)
		*--sp = d.hi;
	      continue;
	    }
	  if (BASE == 10 && to_float (p, len, &f))
	    {
	      *--fp = f;
	      continue;
	    }
	}
      tHrow (THROW_UNDEFINED);
    }
}

void
evaluate (char *p, int n)
{
  rp = (Xt **)save_input (rp);
  SOURCE_ID = -1;
  BLK = 0;
  TIB = p;
  NUMBER_TIB = n;
  TO_IN = 0;
  interpret_();
  rp = (Xt **)restore_input (rp);
}

void
load (uCell blk)
{
  if (blk == 0)
    tHrow (THROW_INVALID_BLOCK);
  rp = (Xt **)save_input (rp);
  SOURCE_ID = 0;
  BLK = blk;
  TO_IN = 0;
  interpret_();
  rp = (Xt **)restore_input (rp);
}

void
include_file (File *f)
{
  if (f == NULL || f->f == NULL)
    tHrow (THROW_FILE_NEX);
  rp = (Xt **)save_input (rp);
  SOURCE_ID = (Cell)f;
  BLK = 0;
  TO_IN = 0;
  interpret_();
  rp = (Xt **)restore_input (rp);
}

void
included (const char *name, int len)
{
  char nm [0x80], fn [0x100];
  File *f;

  store_asciiz (name, len, nm, sizeof nm);
  expand_filename (nm, option.incpaths, option.incext, fn);
  f = open_file (fn, strlen (fn), FMODE_RO);
  if (!f) tHrow (THROW_FILE_NEX);
  include_file (f);
  close_file (f);
}

static void
unnest_input (Iframe *p)
{
  while (sys.saved_input && sys.saved_input != p)
    {
      switch (SOURCE_ID)
	{
	case -1:
	case 0:
	  break;
	default:
	  close_file (SOURCE_FILE);
	}
      rp = (Xt **)restore_input (sys.saved_input);
    }
}


/*****************************************************************************/
/* QUIT, ABORT and exception handling					     */
/*****************************************************************************/

jmp_buf quit_dest;		/* QUIT and ABORT do a THROW which longjmp() */
jmp_buf abort_dest;		/* here thus C-stack gets cleaned up too */

static void
quit_initializations (void)	/* Things quit has to initialize: */
{
  rp = sys.r0;			/* return stack */
  lp = NULL;			/* including all local variables */
  STATE = FALSE;		/* interpreting now */
  sys.cAtch = NULL;		/* and no exceptions caught */
}

static void			/* normal interactive QUIT */
do_quit (void)			/* doing the QUERY-INTERPRET loop */
{
  setjmp (quit_dest);
  quit_initializations ();
  unnest_input (NULL);
  for (;;)
    {
      cr_();
      query_();
      interpret_();
      question_stack_();
      if (!STATE)
	cputs ("ok");
    }
}

static void
abort_initializations (void)	/* Things ABORT has to initialize: */
{
  sp = sys.s0;			/* stacks */
  fp = sys.f0;
  reset_order_();		/* reset search order */
  definitions_();		/* and vocabulary in extension */
  decimal_();			/* number i/o base */
  standard_io_();		/* disable i/o redirection */
}

void
do_abort (void)
{
  setjmp (abort_dest);
  abort_initializations ();
  do_quit ();
}


static void
show_error (char *fmt, ...)
{
  char buf [128];
  va_list p;
  int n;

  sys.input_err = sys.input;	/* save input specification of error */
  va_start (p, fmt);
  vsprintf (buf, fmt, p);
  va_end (p);
  outf ("\nError: \"%.*s\" %s\n", *DP, DP + 1, buf);
  switch (SOURCE_ID)
    {
    case 0:
      if (BLK && BLOCK_FILE && !ferror (BLOCK_FILE->f))
	{
	  outf ("Block %lu line %d:\n",
		(unsigned long)BLK, (int)TO_IN / 64);
	  dot_line (BLK, TO_IN / 64);
	  n = TO_IN % 64;
	  break;
	}
    case -1:
      type (TIB, NUMBER_TIB);
      n = TO_IN;
      break;
    default:
      outf ("File %s line %lu:\n",
	    SOURCE_FILE->name, (unsigned long)SOURCE_FILE->n);
      type (TIB, NUMBER_TIB);
      n = TO_IN;
    }
  outf ("\n%*s", n, "^");
  longjmp (abort_dest, 2);
}

static void
throw_msg (int id, char *msg)
{
  static char
    *throw_explanation [] =
      {
	/*  -1 */ NULL, /* ABORT */
	/*  -2 */ NULL, /* ABORT" */
	/*  -3 */ "stack overflow",
	/*  -4 */ "stack underflow",
	/*  -5 */ "return stack overflow",
	/*  -6 */ "return stack underflow",
	/*  -7 */ "do-loops nested too deeply during execution",
	/*  -8 */ "dictionary overflow",
	/*  -9 */ "invalid memory address",
	/* -10 */ "division by zero",
	/* -11 */ "result out of range",
	/* -12 */ "argument type mismatch",
	/* -13 */ "undefined word",
	/* -14 */ "interpreting a compile-only word",
	/* -15 */ "invalid FORGET",
	/* -16 */ "attempt to use a zero-length string as a name",
	/* -17 */ "pictured numeric output string overflow",
	/* -18 */ "parsed string overflow",
	/* -19 */ "word name too long",
	/* -20 */ "write to a read-only location",
	/* -21 */ "unsupported operation",
	/* -22 */ "control structure mismatch",
	/* -23 */ "address alignment exception",
	/* -24 */ "invalid numeric argument",
	/* -25 */ "return stack imbalance",
	/* -26 */ "loop parameters unavailable",
	/* -27 */ "invalid recursion",
	/* -28 */ "user interrupt",
	/* -29 */ "compiler nesting",
	/* -30 */ "obsolescent feature",
	/* -31 */ ">BODY used on non-CREATEDd definition",
	/* -32 */ "invalid name argument",
	/* -33 */ "block read exception",
	/* -34 */ "block write exception",
	/* -35 */ "invalid block number",
	/* -36 */ "invalid file position",
	/* -37 */ "file I/O exception",
	/* -38 */ "non-existent file",
	/* -39 */ "unexpected end of file",
	/* -40 */ "invalid BASE for floating-point conversion",
	/* -41 */ "loss of precision",
	/* -42 */ "floating-point divide by zero",
	/* -43 */ "floating-point result out of range",
	/* -44 */ "floating-point stack overflow",
	/* -45 */ "floating-point stack underflow",
	/* -46 */ "floating-point invalid argument",
	/* -47 */ "compilation word list deleted",
	/* -48 */ "invalid POSTPONE",
	/* -49 */ "search-order overflow",
	/* -50 */ "search-order underflow",
	/* -51 */ "compilation word list changed",
	/* -52 */ "control flow stack overflow",
	/* -53 */ "exception stack overflow",
	/* -54 */ "floating-point underflow",
	/* -55 */ "floating-point unidentified fault",
	/* -56 */ NULL, /* QUIT */
	/* -57 */ "error in sending or receiving a character",
	/* -58 */ "[IF], [ELSE] or [THEN] error"
      },
    *pfe_throw_explanation [] =
      {
	/* -2048 */ "no or not matching binary image",
	/* -2049 */ "binary image too big",
	/* -2050 */ "out of memory",
	/* -2051 */ "index out of range",
      };

  if (-1 - DIM (throw_explanation) < id && id <= -1)
    strcpy (msg, throw_explanation [-1 - id]);
  else if (-2048 < id && id <= -256)
    sprintf (msg, "[%d] %s", -256 - id, strerror (-256 - id));
  else if (-2048 - DIM (pfe_throw_explanation) < id && id <= -2048)
    strcpy (msg, pfe_throw_explanation [-2048 - id]);
  else
    sprintf (msg, "%d THROW unassigned", id);
}

int
cAtch (Xt xt)
{
  Except *x = DEC (rp, Except);
  int id;

  x->magic = EXCEPTION_MAGIC;
  x->ip = ip;
  x->sp = sp;
  x->iframe = sys.saved_input;
  x->prev = sys.cAtch;
  sys.cAtch = x;
  id = setjmp (x->jmp);
  if (!id)
    run_forth (xt);
  sys.cAtch = x->prev;
  rp = (Xt **)&x [1];
  return id;
}

void
tHrow (int id, ...)
{
  Except *x = sys.cAtch;
  char msg [80];

  if (x && x->magic == EXCEPTION_MAGIC)
    {
      ip = x->ip;
      sp = x->sp;
      unnest_input (x->iframe);
      longjmp (x->jmp, id);
    }
  switch (id)
    {
    case THROW_ABORT_QUOTE:
      {
	va_list p;
	char *addr;
	int len;

	va_start (p, id);
	addr = va_arg (p, char *);
	len = va_arg (p, int);
	va_end (p);
	show_error ("%.*s", len, addr);
      }
    case THROW_ABORT:
      longjmp (abort_dest, 1);
    case THROW_QUIT:
      longjmp (quit_dest, 1);
    }
  throw_msg (id, msg);
  show_error (msg);
}

void
abortq (const char *fmt, ...)
{
  char buf [128];
  va_list p;

  va_start (p, fmt);
  tHrow (-1, buf, vsprintf (buf, fmt, p));
  va_end (p);
}

void
question_pairs (Cell n)
{
  if (n != *sp++)
    tHrow (THROW_CONTROL_MISMATCH);
}

void
question_file_open (File *f)
{
  if (f == NULL || f->f == NULL)
    tHrow (THROW_FILE_NEX);
}


/*****************************************************************************/
/* Set up dictionary and system variables, include files from commandline.   */
/*****************************************************************************/

static void
init_dictionary (Dict *dict, uCell size)
{
  Dict *saved_sysdict;

  /* Temporarily activate this dictionary: */
  saved_sysdict = sys.dict;
  sys.dict = dict;

  if (option.load_dict)
    {
      if (!reload_dictionary (option.load_dict, sys.dict))
	fatal ("Couldn't reload dictionary file %s", option.load_dict);
    }
  else
    {
      /* Wipe the dictionary: */
      memset (dict, 0, size);
      preload_dictionary ();

      /* Define the following default search order:
	 ONLY EXTENSIONS ALSO FORTH ALSO */
      only_runtime ();
      CONTEXT [0] = FORTH;
      CONTEXT [1] = extensions_list.wid;
      also_();
      default_order_();

      /* Action of ABORT and QUIT, but don't enter the interactive QUIT */
      abort_initializations ();
      quit_initializations ();

      /* Include .pferc if it exists: */
      if (option.pferc_file && file_access (option.pferc_file) >= 0)
	included (option.pferc_file, strlen (option.pferc_file));

      /* Include file from command line: */
      if (option.include_file)
	included (option.include_file, strlen (option.include_file));
    }

  /* Switch back to the former dictionary: */
  sys.dict = saved_sysdict;
}

void				/* set up all system variables */
initialize_system (void)	/* and initialize the dictionary */
{
  memset (&sys, 0, sizeof sys);
  sys.dict = (Dict *)membot.dict;
  sys.s0 = memtop.stack;
  sys.f0 = memtop.fstack;
  sys.r0 = memtop.rstack;
  TIB = membot.tib;
  BASE = 10;
  DPL = -1;
  PRECISION = 6;
  LOWER_CASE = option.lower_case_on;
  REDEFINED_MSG = TRUE;
  sys.local = (char (*) [32])membot.stack;

  memset (memtop.files - 3, 0, sizeof (File) * 3);
  sys.stdIn = memtop.files - 3;
  sys.stdIn->f = stdin;
  strcpy (sys.stdIn->name, "<STDIN>");
  strcpy (sys.stdIn->mdstr, "r");
  sys.stdIn->mode = FMODE_RO;

  sys.stdOut = memtop.files - 2;
  sys.stdOut->f = stdout;
  strcpy (sys.stdOut->name, "<STDOUT>");
  strcpy (sys.stdOut->mdstr, "a");
  sys.stdOut->mode = FMODE_WO;

  sys.stdErr = memtop.files - 1;
  sys.stdErr->f = stderr;
  strcpy (sys.stdErr->name, "<STDERR>");
  strcpy (sys.stdErr->mdstr, "a");
  sys.stdErr->mode = FMODE_WO;

  if (option.block_file)
    if (file_access (option.block_file) >= 0)
      BLOCK_FILE = open_file (option.block_file, strlen (option.block_file),
			      FMODE_BIN + file_access (option.block_file));
    else
      if (strcmp (option.block_file, DEFAULT_BLKFILE))
	fatal ("Can't find block file %s", option.block_file);
      else
	BLOCK_FILE = NULL;

  init_dictionary (sys.dict, memsiz.dict);
  if (option.save_dict)
    {
      extern void *getmem (size_t n);
      Dict *dict2 = (Dict *)getmem ((size_t)memsiz.dict);
      long size;

      init_dictionary (dict2, memsiz.dict);
      size = save_dictionary (sys.dict, dict2, option.save_dict);
      if (size)
	outf ("\nSaved dictionary to %s, wrote %ld bytes.\n",
	      option.save_dict, size);
      else
	outs ("\nCouldn't create relocatable dictionary image.\n");
      free (dict2);
    }
}
