/*
 * This file is part of the portable Forth environment written in ANSI C.
 * Copyright (C) 1993  Dirk Uwe Zoller
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This file is version 0.9.5 of 15-May-94
 * Check for the latest version of this package via anonymous ftp at
 *	roxi.rz.fht-mannheim.de
 *	/pub/unix/languages/pfe-VERSION.tar.gz
 * Please direct any comments via internet to
 *	duz@roxi.rz.fht-mannheim.de.
 * Thank You.
 */
/*
 * term-ux.c ---	Terminal driver for UNIX-like systems
 *			using termcap or termcap emulation in terminfo/curses.
 *
 * Refer to this file as an example of how to write such a driver, though
 * other drivers for non-portable operating systems should be much simpler.
 * E.g. on an IBM-PC you could do everything here with a few INT-10h calls.
 *
 * (duz 24Feb94)
 */

#include <stdio.h>		/* putchar() */
#include <stdlib.h>		/* getenv() */
#include <unistd.h>		/* isatty(), read() and others */
#include <fcntl.h>		/* fcntl() */
#include <errno.h>		/* EINTR */

#include "config.h"

#if HAVE_TERMIOS_H
# include <termios.h>		/* a new system */
#elif HAVE_TERMIO_H
# include <sys/termio.h>	/* not quite such a new system */
#endif
#include <sys/ioctl.h>		/* else rely on ioctl.h alone */

#ifdef TIOCGWINSZ		/* defined in ioctl.h */
# include <signal.h>		/* SIGWINCH */
#endif

#ifndef HAVE_TERMCAP_H
  /* Systems emulating the termcap functions in the curses library
   * usually haven't declared them in a file <termcap.h>,
   * so here are these declarations: */

# ifdef _CPLUSPLUS_
  extern "C" {
# endif

  int tgetent (char *, char *);
  int tgetnum (char *);
  int tgetflag (char *);
  char *tgetstr (char *, char **);
  int tputs (char *, int, int (*)(int));
  char *tgoto (char *, int, int);

# ifdef _CPLUSPLUS_
}
# endif
#else
# include <termcap.h>
#endif

#include "term.h"

#define DIM(X)	(sizeof(X)/sizeof*(X))

/*
 * configurable options:
 */

#define	INTR_KEY	''	/* which key should send a SIGINT */
#define	QUIT_KEY	''	/* which key should send a SIGQUIT */
#define	SUSP_KEY	''	/* which key should send a SIGTSTP */

/*
 * exported variables
 */

unsigned char *rawkey_string [NO_OF_KEYS]; /* what function keys send */

/*****************************************************************************/
/* Initialization for usage of screen package.				     */
/*****************************************************************************/

/*
 * All this stuff isn't neccessary or very easy on non unix-like machines
 * where there are direct console i/o functions available. See term-emx.c.
 * It implements the three required functions:
 *
 *	- prepare_terminal()		called once at startup
 *	- interactive_terminal()	called to set up the terminal for pfe
 *	- system_terminal()		called to reset the terminal on exit
 *
 * The first problem is to get information about the connected terminal.
 * UNIX employs a database file called termcap or (in newer versions)
 * terminfo, where informations on the actually used terminal are stored.
 * Before we begin we must get these informations out of that data base.
 * Fortunately the newer curses/terminfo libraries emulate enough of
 * the termcap method that I could mostly ignore the difference.
 */


enum				/* These constants serve as indices */
{				/* to the vectors below. Same order! */
  cm, ho,
  bc, le, nd, up, dO,		/* Terminal features are encoded */
  cl, cd, ce, bl,		/* by such two-letter codes in termcap.	*/
  dc, dl,
  sf, sr,			/* I use the same codes as indices here */
  so, se,			/* to refer to the features. */
  md, mr, mb, me,		/* This is meaningless outside this file. */
  us, ue
};
enum
{
  k1, k2, k3, k4, k5,		/* indices to rawkey strings */
  k6, k7, k8, k9, k0,
  kl, kr, ku, kd,
  kh, kH, kN, kP,
  kb, kD, kM, kI,
  kA, kE, kL, kC
};

static char			/* Strings to query the termcap data base, */
tcctlcode [][3] =		/* control strings same oder as enum above! */
{
  "cm", "ho",
  "bc", "le", "nd", "up", "do",
  "cl", "cd", "ce", "bl",
  "dc", "dl",
  "sf", "sr",
  "so", "se",
  "md", "mr", "mb", "me",
  "us", "ue"
},
*control_string [DIM (tcctlcode)]; /* results are stored here */

static char			/* in case termcap doesn't work... */
*vt100ctl [] =			/* hardcoded vt100 sequences: */
{
  "\033[%i%d;%dH",		/* cm - cursor move */
  "\033[H",			/* ho - home position */
  "\b",				/* bc - like le */
  "\b",				/* le - cursor left */
  "\033[C",			/* nd - right one column */
  "\033[A",			/* up - up one column */
  "\n",				/* dO - down one column */
  "\033[;H\033[2J",		/* cl - clear screen and home */
  "\033[J",			/* cd - clear down */
  "\033[K",			/* ce - clear to end of line */
  "\a",				/* bl - bell */
  "\033[P",			/* dc - delete character in line */
  "\033[M",			/* dl - delete line from screen */
  "\033D",			/* sf - scroll screen up */
  "\033M",			/* sr - scroll screen down */
  "\033[7m",			/* so - enter standout mode */
  "\033[m",			/* se - leave standout mode */
  "\033[1m",			/* md - enter double bright mode */
  "\033[7m",			/* mr - enter reverse video mode */
  "\033[5m",			/* mb - enter blinking mode */
  "\033[m",			/* me - turn off all appearance modes */
  "\033[4m",			/* us - turn on underline mode */
  "\033[m",			/* ue - turn off underline mode */
},
*vt100key [] =			/* vt100 and xterm keys */
{
  "\033OP",			/* k1 - function keys 1 - 4 from vt100 */
  "\033OQ",			/* k2 */
  "\033OR",			/* k3 */
  "\033OS",			/* k4 */
  "\033[15~",			/* k5 - function keys 5 - 10 from xterm */
  "\033[17~",			/* k6 */
  "\033[18~",			/* k7 */
  "\033[19~",			/* k8 */
  "\033[20~",			/* k9 */
  "\033[21~",			/* k0 */

  "\033OD",			/* kl - arrow left */
  "\033OC",			/* kr - arrow right */
  "\033OA",			/* ku - arrow up */
  "\033OB",			/* kd - arrow down */

  "\033[1~",			/* kh - home key */
  "\033[4~",			/* kH - home down key (end key) */
  "\033[6~",			/* kN - next page */
  "\033[5~",			/* kP - previous page */

  "\b",				/* kb - backspace key */
  "\033[3~",			/* kD - delete character key */
  NULL,				/* kM - exit insert mode key */
  "\033[2~",			/* kI - insert character key */

  NULL,				/* kA - insert line key */
  NULL,				/* kE - clear end of line key */
  NULL,				/* kL - delete line key */
  NULL,				/* kC - clear screen key */

  "\r"				/* enter key, for uniformity with curses */
};

static char
tckeycode [][3] =		/* Strings to query the termcap data base, */
{
  "k1", "k2", "k3", "k4", "k5",	/* keys in same order as enum keycode */
  "k6", "k7", "k8", "k9", "k0",	/* from term.h */
  "kl", "kr", "ku", "kd",	/* result is just what has to be exported */
  "kh", "kH", "kN", "kP",	/* via variable rawkey_string */
  "kb", "kD", "kM", "kI",
  "kA", "kE", "kL", "kC"
};

static int
query_termcap (void)
/*
 * Get the strings needed out of termcap/terminfo.
 * Store them for later use in control_string[] and rawkey_string[].
 */
{
  int i;
  char	*ttype = getenv ("TERM"),
	*tcbuf, *tctop, *p;

#if BSD
  char tcent [2048];
  if (ttype == NULL || tgetent (tcent, ttype) <= 0)
    return 0;
  tctop = tcbuf = (char *)malloc (strlen (tcent));
#elif HAVE_TERMCAP
  char tcent [2048], *pc;
  if (ttype == NULL || tgetent (tcent, ttype) <= 0)
    return 0;
  tctop = tcbuf = (char *)malloc (strlen (tcent));
  pc = tgetstr ("pc", &tctop);
  if (pc != NULL)
    PC = *pc, tctop = tcbuf;
  else
    PC = 0;
#else
  if (ttype == NULL || tgetent (NULL, ttype) <= 0)
    return 0;
  tctop = tcbuf = (char *)malloc (2048);
#endif

  rows = tgetnum ("li");
  cols = tgetnum ("co");

  /* Read all termcap strings we need, */
  /* assume vt100 where termcap has nothing: */
  for (i = 0; i < DIM (tcctlcode); i++)
    {
      p = tgetstr (tcctlcode [i], &tctop);
      control_string [i] = p ? p : vt100ctl [i];
    }
  for (i = 0; i < NO_OF_KEYS; i++)
    {
      p = tgetstr (tckeycode [i], &tctop);
      rawkey_string [i] = (unsigned char *)(p ? p : vt100key [i]);
    }
/*rawkey_string [EKEY_enter - EKEY_k1] = "\r";
*/
#if HAVE_TERMCAP && !BSD
  /* these are defined inside the termcap-library for use by tgoto(): */
  BC = control_string [bc];
  UP = control_string [up];
#endif

  return 1;
}

void
show_control_strings (int (*outf) (const char *fmt, ...))
/* for your information why screen manipulation doesn't work :-) */
{
  int i;
  char *p;

  for (i = 0; i < DIM (tcctlcode); i++)
    {
      outf ("\n\"%s\"=", tcctlcode [i]);
      if ((p = control_string [i]))
	while (*p)
	  cputc_printable (*p++);
      else
	cputs ("undefined");
    }
}

void
show_rawkey_strings (int (*outf) (const char *fmt, ...))
/* for your information why function keys don't work :-) */
{
  int i;
  unsigned char *p;

  for (i = 0; i < DIM (tckeycode); i++)
    {
      outf ("\n\"%s\"=", tckeycode [i]);
      if ((p = rawkey_string [i]))
	while (*p)
	  cputc_printable (*p++);
      else
	cputs ("undefined");
    }
}

/*
 * The second problem is that console i/o is done via the file system.
 * It takes a little hacking to get an interactive flavour from this
 * arrangement. The kind of hacking changes while Unix evolves.
 * The hacking is addressed at two instances:
 *    - UNIX file system. fcntl() calls on the standard input are used
 *	to prevent it from waiting for a pressed key when we only want
 *	to detect if a key is pressed.
 *    - terminal device driver. It must be configured to pass keys
 *	immediately instead of assembling it to lines, to pass through
 *	all keys and not interpret some of them or interpret those keys
 *	we need in a way we want.
 * All what's changed must be carefully restored on exit otherwise
 * the terminal won't work any more.
 */

#if HAVE_TERMIOS_H

  typedef struct termios tmode;
# define set_tmode(F,M) tcsetattr (F, TCSAFLUSH, M)
# define get_tmode(F,M) tcgetattr (F, M)

#else

  typedef struct termio tmode;
# define set_tmode(F,M) ioctl (F, TCSETAW, M)
# define get_tmode(F,M) ioctl (F, TCGETA, M)

# define VINTR	0
# define VQUIT	1
# define VMIN	4
# define VTIME	5

#endif

#define	C_IFLAGS_OFF	(ICRNL | IGNBRK | IGNCR | INLCR | \
			 ISTRIP | IXOFF | IXON)
#define C_IFLAGS_ON	(BRKINT)
#define C_OFLAGS_OFF	(0)
#define C_OFLAGS_ON	(0)
#define C_LFLAGS_OFF	(ECHO | ICANON)
#define C_LFLAGS_ON	(ISIG)

static tmode system_tmode;	/* state prior to our changes */
static int saved_fcntl [3];

#ifdef O_NDELAY
#define NOWAIT O_NDELAY
#else
#define NOWAIT O_NONBLOCK
#endif

static void			/* set file associated with fd to */
waitchar (int fd)		/* "wait for character" */
{
  int flags;

  flags = fcntl (fd, F_GETFL, 0);
  fcntl (fd, F_SETFL, flags & ~NOWAIT);
}

static void			/* set file associated with fd to */
no_waitchar (int fd)		/* "don't wait for character" */
{
  int flags;

  flags = fcntl (fd, F_GETFL, 0);
  fcntl (fd, F_SETFL, flags | NOWAIT);
}

int
set_interrupt_key (char ch)
/*
 * If ch != 0 enables SIGINT on key ch, else disables SIGINT
 * returns old break key.
 * Porting this function might be hard. It isn't neccessary though.
 * This is needed only by the block editor: What's usually the break key
 * means exit to the block editor. If you have no break key or you have
 * it somewhere else than ^U, just #define this to nothing in term.h.
 */
{
  tmode t;
  char old;

  if (!isatty (0))
    return -1;
  get_tmode (0, &t);
  if (ch)
    t.c_iflag |= BRKINT;
  else
    t.c_iflag &= ~BRKINT;
  old = t.c_cc [VINTR];
  t.c_cc [VINTR] = ch;
  set_tmode (0, &t);
  return old;
}

void				/* set the terminal to interactive */
interactive_terminal (void)
{
  tmode t = system_tmode;

  if (!isatty (0))
    return;
  t.c_iflag &= ~C_IFLAGS_OFF; t.c_iflag |= C_IFLAGS_ON;
  t.c_oflag &= ~C_OFLAGS_OFF; t.c_oflag |= C_OFLAGS_ON;
  t.c_lflag &= ~C_LFLAGS_OFF; t.c_lflag |= C_LFLAGS_ON;
  t.c_cc [VMIN] = 1;
  t.c_cc [VTIME] = 0;
  t.c_cc [VINTR] = INTR_KEY;
  t.c_cc [VQUIT] = QUIT_KEY;
#if HAVE_TERMIOS_H
  t.c_cc [VSUSP] = SUSP_KEY;
  t.c_cc [VSTART] = '\xFF';
  t.c_cc [VSTOP] = '\xFF';
#endif
  set_tmode (0, &t);
  no_waitchar (0);
}

void				/* resets terminal state to */
system_terminal (void)		/* as it was before */
{
  int fd;

  fflush (stdout);
  if (!isatty (0))
    return;
  set_tmode (0, &system_tmode);
  for (fd = 0; fd < 3; fd++)
    fcntl (fd, F_SETFL, saved_fcntl [fd]);
}

#ifdef TIOCGWINSZ
void
query_winsize (void)
{
  struct winsize size;

  if (ioctl (1, TIOCGWINSZ, (char *)&size) >= 0)
    {
      rows = size.ws_row;
      cols = size.ws_col;
      xmax = size.ws_xpixel;
      ymax = size.ws_ypixel;
    }
}
#else
void query_winsize (void) {}
#endif

int				/* Prepares usage of all other functions. */
prepare_terminal (void)		/* Call only once at startup. */
{
  int fd, result;

  /* save state before all changes */
  for (fd = 0; fd < 3; fd++)
    saved_fcntl [fd] = fcntl (fd, F_GETFL, 0);
  get_tmode (0, &system_tmode);

  /* initialize termcap-stuff */
#if HAVE_TERMCAP && !BSD
  ospeed = system_tmode.c_cflag & CBAUD;
#endif
  if (!query_termcap ())
    {
      /* completely assume vt100 */
      memcpy (rawkey_string,  vt100key, sizeof rawkey_string);
      memcpy (control_string, vt100ctl, sizeof control_string);
      result = 0;
    }
  else
    result = 1;
  return result;
}


/*****************************************************************************/
/* Input from keyboard.							     */
/*****************************************************************************/

/*
 * Having hacked the terminal driver and the standard input in the appropiate
 * modes all we have to do is read one character from standard input to get
 * a key.
 * The only problem is: we can't detect if a key is available without reading
 * it. So we have to store it in case this happens.
 */

#define NOCH ((unsigned short)0xABCD)
				/* encodes 'no character available' */

static unsigned short		/* the next character when read by */
	nxch = NOCH;		/* keypressed() */

static int
nextch (void)
/*
 * Read a single character from standard input.
 */
{
  unsigned char c = 0;
  int err;

  if (nxch != NOCH)
    {
      c = (unsigned char)nxch;
      nxch = NOCH;
    }
  else
    for (;;)
      {
	waitchar (0);
	err = read (0, &c, 1);
	no_waitchar (0);
	if (err != -1)
	  break;
	/* Note, the EWOULDBLOCK test handles the case that we were
	 * suspended and restarted, in which case on_continue() goes
	 * and puts us in O_NDELAY mode... */
	if (errno != EWOULDBLOCK)
	  return -1;
      }
  return c;
}

int
keypressed (void)
/*
 * Checks if a character is available in the standard input. Saves it in nxch.
 * Returns: 1 if char available, 0 if not
 */
{
  unsigned char c;		/* place to read the character */

  fflush (stdout);
  if (nxch != NOCH)
    return 1;			/* char from previos keypressed() */
  if (read (0, &c, 1) != 1)
    return 0;
  nxch = c;
  return 1;
}

int
getkey (void)
{
  int c;

  fflush (stdout);		/* make sure screen is up to date */
  do c = nextch ();
  while (c == -1 && errno == EINTR);
  return c;
}


/*****************************************************************************/
/* Output to screen, control with termcap:				     */
/*****************************************************************************/

static int row, col;		/* position of curser as tracked */

static int			/* put control character to screen */
ct_putc (int c)			/* not updating cursor position */
{
  return putchar (c);		/* putchar() is a macro, we need a */
}				/* function for tputs() */

static void
ct_puts (int tcidx, int n)	/* issue termcap string to terminal */
{
  if (!control_string [tcidx])	/* no harm if feature not available */
    return;
  tputs (control_string [tcidx], n, ct_putc);
}

static void			/* output character and */
cputc_(int c)			/* trace the cursor position */
{
  putchar (c);
  switch (c)
    {
    case '\a':			/* bell, no change of cursor position */
      break;
    case '\b':			/* backspace, move cursor left */
      if (col > 0)
	col--;
      break;
    case '\r':			/* carriage return, ->column 0 */
      col = 0;
      break;
    default:			/* ordinary character: */
      if (col < cols - 1)	/* at right edge of screen? */
	{
	  col++;		/* no: increment column */
	  break;
	}			/* yes: like line feed */
    case '\n':			/* line feed */
      col = 0;
      if (row < rows - 1)	/* if not at bottom of screen: */
	row++;			/* increment row */
    }				/* otherwise terminal is supposed to scroll */
}

void
cputc (int c)
{
  cputc_(c);
  fflush (stdout);
}

void
cputs (const char *s)
{
  while (*s)
    cputc_(*s++);
  fflush (stdout);
}

void
gotoxy (int x, int y)
{
  col = x;
  row = y;
  tputs (tgoto (control_string [cm], x, y), 1, ct_putc);
}

void
wherexy (int *x, int *y)
{
  *x = col;
  *y = row;
}

void goleft (void)		{ ct_puts (le, 1); --col; }
void goright (void)		{ ct_puts (nd, 1); ++col; }
void goup (void)		{ ct_puts (up, 1); --row; }
void godown (void)		{ ct_puts (dO, 1); ++row; }

void home (void)		{ ct_puts (ho, 1); row = col = 0; }
void clrscr (void)		{ ct_puts (cl, ymax); home (); }
void clreol (void)		{ ct_puts (ce, 1); }
void clrdown (void)		{ ct_puts (cd, 1); }
void bing (void)		{ ct_puts (bl, 1); }

void standout_on (void)		{ ct_puts (so, 1); }
void standout_off (void)	{ ct_puts (se, 1); }
void bright (void)		{ ct_puts (md, 1); }
void reverse (void)		{ ct_puts (mr, 1); }
void blinking (void)		{ ct_puts (mb, 1); }
void normal (void)		{ ct_puts (me, 1); }
void underline_on (void)	{ ct_puts (us, 1); }
void underline_off (void)	{ ct_puts (ue, 1); }
