/*
================================================================================
  RPL/2 (R) version 4.1.4
  Copyright (C) 1989-2011 Dr. BERTRAND Jol

  This file is part of RPL/2.

  RPL/2 is free software; you can redistribute it and/or modify it
  under the terms of the CeCILL V2 License as published by the french
  CEA, CNRS and INRIA.
 
  RPL/2 is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the CeCILL V2 License
  for more details.
 
  You should have received a copy of the CeCILL License
  along with RPL/2. If not, write to info@cecill.info.
================================================================================
*/


#ifndef INCLUSION_RPL
#define INCLUSION_RPL

#ifndef __RPLCAS
#	define _GNU_SOURCE
#	define _XOPEN_SOURCE		700
#endif

#define _REENTRANT

#ifndef __RPLCAS
#	ifdef Darwin
#		define	_DARWIN_C_SOURCE
#	endif

#	ifdef OpenBSD
#		define _BSD_SOURCE
#	endif
#endif

#ifdef OS2
#	ifndef __RPLCAS
#		define _BSD_SOURCE
#		define _XOPEN_SOURCE	600
#	endif

#	include <types.h>
	enum { SHUT_RD = 0, SHUT_WR, SHUT_RDWR };
#	define SHUT_RD		SHUT_RD
#	define SHUT_WR		SHUT_WR
#	define SHUT_RDWR	SHUT_RDWR

#	include "getaddrinfo-conv.h"

#	define sched_yield(arg)
#endif


/*
================================================================================
  INCLUSIONS
================================================================================
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <strings.h>
#include <syslog.h>
#include <errno.h>
#include <stdarg.h>

#ifndef RPLARGS
#	include <sys/mman.h>
#	include <sys/time.h>
#	include <sys/resource.h>
#	include <sys/select.h>
#	include <sys/socket.h>
#	include <sys/stat.h>
#	include <sys/timeb.h>
#	include <sys/types.h>
#	include <sys/un.h>
#	include <sys/wait.h>

#	include <arpa/inet.h>
#	include <netinet/in.h>
#	include <netdb.h>

#	include <dlfcn.h>
#	include <fcntl.h>
#	include <pthread.h>

#	ifdef OS2
#		undef pthread_mutexattr_settype
#		define pthread_mutexattr_settype(a, b)
#	endif

#	include <pwd.h>
#	ifdef SHARED_MEMORY
#		include <sys/shm.h>
#	endif

#	ifndef IPCS_SYSV
#		include <semaphore.h>
#	else
#		include <sys/ipc.h>
#		include <sys/sem.h>

#		ifdef OS2
#			define INCL_DOSSEMAPHORES
#			define INCL_DOSMEMMGR
#			define INCL_DOSERRORS
#			include <os2.h>

			typedef struct _OS2SEM
			{
				HMTX	hmtx;
				HEV		hev;
				ULONG	shared;
				ULONG	*cnt;
				ULONG	*nopened;
				ULONG	allocated;
			} sem_t;
#		else // IPCS_SYSV
			typedef struct
		   	{
				int				sem;
				int				alloue;
				unsigned char	*path;
				pid_t			pid;
				pthread_t		tid;
			} sem_t;
#		endif

#		ifndef SEM_FAILED
#			define SEM_FAILED	NULL
#		endif

		sem_t *sem_open_SysV(const char *nom, int oflag, ...);
		int sem_init_SysV(sem_t *sem, int shared, unsigned int value);
		int sem_close_SysV(sem_t *sem);
		int sem_wait_SysV(sem_t *sem);
		int sem_trywait_SysV(sem_t *sem);
		int sem_post_SysV(sem_t *sem);
		int sem_getvalue_SysV(sem_t *sem, int *value);
		int sem_unlink_SysV(const char *nom);
		int sem_destroy_SysV(sem_t *sem);
#	endif

#	include <setjmp.h>
#	include <signal.h>
#	include <termios.h>
#	include <time.h>
#	include <unistd.h>
#	include <locale.h>

#	include "history.h"
#	include "readline.h"
#	include "termcap.h"
#	include "iconv.h"

#	include "openssl/evp.h"
#	include "sqlite3.h"
#	include "sigsegv.h"

#	define HAVE_INLINE
#	define GSL_RANGE_CHECK_OFF
#	include "gsl/gsl_cdf.h"
#	include "gsl/gsl_cblas.h"
#	include "gsl/gsl_errno.h"
#	include "gsl/gsl_fft_complex.h"
#	include "gsl/gsl_permutation.h"
#	include "gsl/gsl_randist.h"
#	include "gsl/gsl_rng.h"
#	include "gsl/gsl_sf.h"
#endif

#ifdef SunOS
#	include <ieeefp.h>
#endif

// Dfinition spcifique  Hurd
#ifndef PIPE_BUF
#	define fpathconf (".", _PC_PIPE_BUF)
#endif

#ifdef MYSQL_SUPPORT
#   include <mysql.h>
#   define mysqlclose(arg)		mysql_close(arg.mysql)
#else
#   define mysqlclose(arg)
#endif

#ifdef POSTGRESQL_SUPPORT
#   include <libpq-fe.h>
#	define postgresqlclose(arg)	PQfinish(arg.postgresql)
#else
#   define postgresqlclose(arg)
#endif

#include "librplprototypes.h"

#ifndef RPLARGS
#	ifndef UNIX_PATH_MAX
		struct sockaddr_un sizecheck;
#		define UNIX_PATH_MAX sizeof(sizecheck.sun_path)
#	endif
#endif

/*
================================================================================
  Bugs spcifiques
================================================================================
*/

enum signaux_rpl
{
	rpl_signull = 0,
	rpl_sigint,
	rpl_sigterm,
	rpl_sigstart,		// Signal envoy par un pre pour lancer son fils.
	rpl_sigcont,		// Signal de redmarrage d'un processus arrt par
						// SUSPEND
	rpl_sigstop,		// Signal d'arrt envoy par l'instruction STOP ou
						// FUSE. (FSTOP)
	rpl_sigabort,		// Signal envoy par l'instruction ABORT ( regrouper
						// avec FABORT)
	rpl_sigurg,			// Signal d'arrt urgent
	rpl_siginject,		// Signal indiquant la prsence d'une donne  lire
						// envoye depuis le pre
	rpl_sigalrm,		// Signal d'alarme (erreur systme) depuis un fils
	rpl_sighup,
	rpl_sigtstp,
	rpl_sigexcept,
	rpl_sigmax
};

#define	LONGUEUR_QUEUE_SIGNAUX			1024

// Une structure s_queue_signaux est cre par processus (et non par thread).
// Elle est projete dans un segment de mmoire partage pour qu'elle soit
// accessible  la fois du pre et des fils.

#ifndef RPLARGS
typedef struct queue_signaux
{
#	ifndef IPCS_SYSV
#		ifndef SEMAPHORES_NOMMES
			sem_t				semaphore;
#		endif
#	else
		sem_t					semaphore;
#	endif

	volatile int				pointeur_lecture;
	volatile int				pointeur_ecriture;

	volatile struct
	{
		pid_t					pid;
		enum signaux_rpl		signal;
	}							queue[LONGUEUR_QUEUE_SIGNAUX];
} struct_queue_signaux;

#	ifndef MAIN_RPL
		extern struct_queue_signaux			*s_queue_signaux;
		extern int							f_queue_signaux;
#		ifdef SEMAPHORES_NOMMES
			extern sem_t					*semaphore_queue_signaux;
#		endif
#	else
		struct_queue_signaux				*s_queue_signaux;
		int									f_queue_signaux;
#		ifdef SEMAPHORES_NOMMES
			sem_t							*semaphore_queue_signaux;
#		endif
#	endif
#endif

#ifdef OpenBSD
#	ifdef PTHREAD_SCOPE_SYSTEM
#		undef PTHREAD_SCOPE_SYSTEM
#	endif
#	define PTHREAD_SCOPE_SYSTEM	0
#endif

#ifdef OS2
#	define readline(s) readline_wrapper(s)
	unsigned char *readline_wrapper(unsigned char *s);
#endif

#if ! defined(UNION_SEMUN) && defined(IPCS_SYSV)
union semun
{
	int					val;
	struct semid_ds		*buf;
	unsigned short		*array;
	struct seminfo		*__buf;
};
#endif

#ifndef RTLD_LOCAL
#	define RTLD_LOCAL		0
#endif

#ifndef SIGSTKSZ
#	define SIGSTKSZ			65536
#endif

/*
================================================================================
  SMAPHORES
================================================================================
*/

#ifdef IPCS_SYSV
#	define sem_init(a, b, c)	sem_init_SysV(a, b, c)
#	define sem_destroy(a)		sem_destroy_SysV(a)
#	define sem_wait(a)			sem_wait_SysV(a)
#	define sem_trywait(a)		sem_trywait_SysV(a)
#	define sem_post(a)			sem_post_SysV(a)
#	define sem_getvalue(a, b)	sem_getvalue_SysV(a, b)
#	define sem_open(...)		sem_open_SysV(__VA_ARGS__)
#	define sem_close(a)			sem_close_SysV(a)
#	define sem_unlink(a)		sem_unlink_SysV(a)
#endif


/*
================================================================================
  SIGNAUX
================================================================================
*/

#ifndef RPLARGS

// Signaux utiliss par dfaut :
//	SIGINT
//	SIGTSTP
//	SIGCONT
//	SIGURG
//	SIGPIPE
//	SIGALRM

#	define SIGTEST					SIGUSR1

//	Nombre d'interruptions disponibles
#	define d_NOMBRE_INTERRUPTIONS	64
#endif

/*
================================================================================
  Granularit temporelle
================================================================================
*/

#ifndef RPLARGS
#	define GRANULARITE_us			10
#	define GRANULARITE_us_max		100000
#	define INCR_GRANULARITE(g) \
	do  g = 1000 * ((g >= (GRANULARITE_us_max * 1000)) \
			? GRANULARITE_us_max : ((g / 1000) + GRANULARITE_us)); while(0)
#endif

/*
================================================================================
  Options de debug
================================================================================
*/

#ifndef RPLARGS
#	define d_debug_pile_utilisateur			((integer8) 0x1)
#	define d_debug_pile_systeme				((integer8) 0x2)
#	define d_debug_appels_fonctions			((integer8) 0x4)
#	define d_debug_processus				((integer8) 0x8)
#	define d_debug_analyse					((integer8) 0x10)
#	define d_debug_fusible					((integer8) 0x20)
#	define d_debug_variables				((integer8) 0x40)
#	define d_debug_fonctions_intrinseques	((integer8) 0x80)
#	define d_debug_niveaux					((integer8) 0x100)
#	define d_conversion_RPN					((integer8) 0x200)
#	define d_traitement_interruption		((integer8) 0x400)
#	define d_debug_signaux					((integer8) 0x800)
#endif

/*
================================================================================
  TYPES
================================================================================
*/

typedef FILE					file;
typedef unsigned char			t_8_bits;

#ifndef RPLARGS
#	include "rpltypes-conv.h"
#else
#	include "rpltypes.h"
#endif

/*
================================================================================
  Variables globales
================================================================================
*/

#ifndef RPLARGS
#	ifdef SEMAPHORES_NOMMES
#		define LONGUEUR_NOM_SEMAPHORE	64
#	endif

#	ifndef MAIN_RPL
		extern jmp_buf				contexte_ecriture;
		extern jmp_buf				contexte_impression;
		extern jmp_buf				contexte_initial;
		extern jmp_buf				contexte_processus;
		extern jmp_buf				contexte_thread;

		extern int					signal_test;

		extern pid_t				pid_processus_pere;

#		ifdef SEMAPHORES_NOMMES
		extern pthread_mutex_t		mutex_sem;
#		endif

		extern pthread_mutex_t		mutex_liste_threads;
		extern pthread_mutex_t		mutex_gestionnaires_signaux_atomique;

		extern sem_t
#		ifdef SEMAPHORES_NOMMES
			*
#		endif
									semaphore_gestionnaires_signaux;

		extern volatile int			routine_recursive;
#	else
		jmp_buf						contexte_ecriture;
		jmp_buf						contexte_impression;
		jmp_buf						contexte_initial;
		jmp_buf						contexte_processus;
		jmp_buf						contexte_thread;

		int							signal_test;

		pid_t						pid_processus_pere;

#		ifdef SEMAPHORES_NOMMES
		pthread_mutex_t				mutex_sem = PTHREAD_MUTEX_INITIALIZER;
#		endif

		pthread_mutex_t				mutex_liste_threads =
											PTHREAD_MUTEX_INITIALIZER;
		pthread_mutex_t				mutex_gestionnaires_signaux =
											PTHREAD_MUTEX_INITIALIZER;
		pthread_mutex_t				mutex_gestionnaires_signaux_atomique =
											PTHREAD_MUTEX_INITIALIZER;
		sem_t
#		ifdef SEMAPHORES_NOMMES
			*
#		endif
									semaphore_gestionnaires_signaux;

		volatile int				routine_recursive;
#	endif
#endif

#ifdef SEMAPHORES_NOMMES
#define SEM_FORK			0
#define	SEM_QUEUE			1
#define	SEM_SIGNAUX			2
sem_t *sem_init2(unsigned int valeur, pid_t pid, int ordre);
sem_t *sem_init3(unsigned int valeur, pid_t pid, pthread_t tid, int ordre);
int sem_destroy2(sem_t *semaphore_p, pid_t pid, int ordre);
int sem_destroy3(sem_t *semaphore_p, pid_t pid, pthread_t tid, int ordre);
int sem_getvalue2(sem_t *semaphore, int *valeur);
sem_t *sem_open2(pid_t pid, int ordre);

// Le mutex est l uniquement pour pouvoir muler le comportement
// de sem_getvalue() sur un systme comme MacOS X qui ne possde pas
// cette fonction.

#define sem_getvalue(semaphore, value) sem_getvalue2(semaphore, value)
#define sem_post(semaphore) \
		({ int r; pthread_mutex_lock(&mutex_sem); \
		r = sem_post(semaphore), pthread_mutex_unlock(&mutex_sem); r; })
#define sem_wait(semaphore) \
		({ int r; pthread_mutex_lock(&mutex_sem); \
		r = sem_wait(semaphore), pthread_mutex_unlock(&mutex_sem); r; })
#define sem_trywait(semaphore) \
		({ int r; pthread_mutex_lock(&mutex_sem); \
		r = sem_trywait(semaphore), pthread_mutex_unlock(&mutex_sem); r; })
#endif

/*
================================================================================
  Macros
================================================================================
*/

// Par dfaut, tout est redirig vers stderr sauf indication explicite de
// stdout.

#define printf(...) transliterated_fprintf(s_etat_processus, \
			stderr, __VA_ARGS__)
#define fprintf(flux, ...) transliterated_fprintf(s_etat_processus, \
			flux, __VA_ARGS__)
#define uprintf(...) transliterated_fprintf(NULL, \
			stderr, __VA_ARGS__)
#define ufprintf(flux, ...) transliterated_fprintf(NULL, \
			flux, __VA_ARGS__)

#ifdef SunOS
#	define fopen(...) ({ FILE *desc; \
	        while((desc = fopen(__VA_ARGS__)) == NULL) \
	        { if ((errno != EINTR) && (errno != 0)) break; } desc; })
#	define freopen(...) ({ FILE *desc; \
	        while((desc = freopen(__VA_ARGS__)) == NULL) \
	        { if ((errno != EINTR) && (errno != 0)) break; } desc; })
#	define fclose(...) ({ int ios; \
	        while((ios = fclose(__VA_ARGS__)) != 0) \
	        { if ((errno != EINTR) && (errno != 0)) break; } ios; })
#	define fread(...) ({ int ios; \
	        while((ios = fread(__VA_ARGS__)) < 0) \
	        { if ((errno != EINTR) && (errno != 0)) break; } ios; })
#	define fflush(flux) ({ int ios; \
			while((ios = fflush((flux == stdout) ? stderr : flux)) != 0) \
			{ if ((errno != EINTR) && (errno != 0)) break; } ios; })
#	define sem_open(...) ({ sem_t *semaphore; \
			while((semaphore = sem_open(__VA_ARGS__)) == SEM_FAILED) \
	 		{ if (errno != EINTR) break; }	semaphore; })
#else // !SunOS
#	define fflush(flux) fflush((flux == stdout) ? stderr : flux)
#endif

// Ajout d'un timeout sur les fonctions pipe() pour viter une erreur
// systme dans le cas o l'on a atteint le nombre maximale de fichiers
// ouverts.

#define pipe(descripteurs) \
	({ \
		struct timespec temporisation; \
		int erreur, nombre_echecs; \
		nombre_echecs = 0; \
		do \
		{ \
			if ((erreur = pipe(descripteurs)) == 0) break; \
			nombre_echecs++; \
			temporisation.tv_sec = 0; \
			temporisation.tv_nsec = ((long) 1000000000) * rand(); \
			nanosleep(&temporisation, NULL); \
		} while(nombre_echecs < 100); \
		erreur; \
	 })

// Redfinition de abs pour un fonctionnement en entier de type long long int

#ifdef __GNUC__
#	define abs(i) ({ typeof(i) _i; _i = (i); (_i >= 0) ? _i : -_i; })
// typeof() est une extension de gcc, mais est prsent sur d'autres compilateurs
// comme Sun Studio. Dans le cas o typeof() n'existe pas, il est possible
// d'utiliser la macro suivante, mais elle possde un effet de bord d 
// l'valuation multiple de l'argument.
#else
#	define abs(i) (((i) >= 0) ? (i) : (-i))
#endif

#define NOTICE(m) \
	do \
	{ \
		ufprintf(stderr, "[%d-%llu] %s() from %s at line %d: %s\n", \
				getpid(), (unsigned long long) pthread_self(), \
			   	__FUNCTION__, __FILE__, __LINE__, m); \
		fflush(stderr); \
	} while(0)

#ifdef __GLIBC__
#include <execinfo.h>
#define __BACKTRACE
#define BACKTRACE(n) \
	do \
	{ \
		void    		*buffer[n]; \
		int     		nb, i; \
		char    		**appels; \
		pthread_mutex_t	mutex = PTHREAD_MUTEX_INITIALIZER; \
		nb = backtrace(buffer, n); \
		appels = backtrace_symbols(buffer, nb); \
		pthread_mutex_lock(&mutex); \
		uprintf("[%d-%llu] BACKTRACE <%s()> at line #%d\n", \
				getpid(), pthread_self(), __FUNCTION__, __LINE__); \
		if (appels != NULL) \
		{ \
			for(i = 0; i < nb; i++) \
				uprintf("    %s\n", appels[i]); \
			free(appels); \
		} \
		else \
			uprintf("Nullified backtrace\n"); \
		uprintf("[%d-%llu] END OF BACKTRACE\n", getpid(), pthread_self()); \
		fflush(stdout); \
		pthread_mutex_unlock(&mutex); \
	} while(0)
#else
#define BACKTRACE(n) NOTICE("BACKTRACE only defined in glibc")
#endif

#ifdef __BACKTRACE
#	define BUG(b, ...) \
			do { if (b) { uprintf("[%d] BUG! <%s()> at line #%d of %s\n", \
			(int) getpid(), __FUNCTION__, __LINE__, __FILE__); \
			__VA_ARGS__; fflush(stdout); \
			BACKTRACE(64); \
			pthread_kill(pthread_self(), SIGBUS); }} while(0)
#else
#	define BUG(b, ...) \
			do { if (b) { uprintf("[%d] BUG! <%s()> at line #%d of %s\n", \
			(int) getpid(), __FUNCTION__, __LINE__, __FILE__); \
			__VA_ARGS__; fflush(stdout); \
			pthread_kill(pthread_self(), SIGBUS); }} while(0)
#endif


/*
================================================================================
  Affichage des points de retour des fonctions
================================================================================
*/

#ifdef DEBUG_MEMOIRE
void analyse_post_mortem();
void *debug_memoire_ajout(size_t taille, const unsigned char *fonction,
		unsigned long ligne, const unsigned char *argument);
void debug_memoire_initialisation();
void *debug_memoire_modification(void *pointeur, size_t taille,
		const unsigned char *fonction, unsigned long ligne,
		const unsigned char *argument);
void debug_memoire_retrait(void *ptr);
void debug_memoire_verification();

pid_t debug_fork();

#define malloc(s) debug_memoire_ajout(s, __FUNCTION__, __LINE__, #s)
#define free(s) debug_memoire_retrait(s)
#define realloc(s, t) debug_memoire_modification(s, t, \
			__FUNCTION__, __LINE__, #t)
#define fork() debug_fork()
#endif

#ifdef DEBUG_RETURN
#define return uprintf("[%d] Return from <%s()> at line #%d " \
		"(%s [%d])\n", (int) getpid(), __FUNCTION__, \
		__LINE__, strerror(errno), errno); fflush(stdout); errno = 0; return
#endif

#ifdef DEBUG_MUTEX
#define pthread_mutex_lock(mutex) uprintf("[%d-%llu] Mutex %s (%p) " \
		"locking at %s() " \
		"line #%d\n", (int) getpid(), (unsigned long long) pthread_self(), \
		#mutex, mutex, __FUNCTION__, __LINE__), fflush(stdout), \
		pthread_mutex_lock(mutex)
#define pthread_mutex_unlock(mutex) uprintf("[%d-%llu] Mutex %s (%p) " \
		"unlocking at " \
		"%s() line #%d\n", (int) getpid(), (unsigned long long) \
		pthread_self(), #mutex, mutex, __FUNCTION__, __LINE__), \
		fflush(stdout), \
		pthread_mutex_unlock(mutex)
#endif

#ifdef DEBUG_SEMAPHORES
#define sem_wait(a) ({ int value; sem_getvalue(a, &value); \
		uprintf("[%d-%llu] Semaphore %s (%p) "\
		"waiting at %s() " \
		"line #%d\n", (int) getpid(), (unsigned long long) pthread_self(), \
		#a, a, __FUNCTION__, __LINE__), fflush(stdout); \
		if (value > 1) BUG(1, uprintf("Value %d\n", value)); sem_wait(a); })
#define sem_post(a) ({ int value; sem_getvalue(a, &value); \
		uprintf("[%d-%llu] Semaphore %s (%p) "\
		"posting at %s() " \
		"line #%d\n", (int) getpid(), (unsigned long long) pthread_self(), \
		#a, a, __FUNCTION__, __LINE__), fflush(stdout); \
		if (value > 0) BUG(1, uprintf("Value %d\n", value)); sem_post(a); })
#define sem_destroy(a) ({ int value; sem_getvalue(a, &value); \
		if (value == 0) BUG(1, uprintf("Value %d\n", value)); sem_destroy(a); })
#endif


/*
================================================================================
  DEFINIONS
================================================================================
*/

/*
--------------------------------------------------------------------------------
  Valeurs
--------------------------------------------------------------------------------
*/

#define d_faux					((logical1) 0L)
#define d_vrai					((logical1) -1L)

#define d_absence_erreur		((logical1) d_faux)
#define d_erreur				((logical1) d_vrai)

#define d_code_fin_chaine					'\0'
#define d_code_espace						' '
#define d_code_retour_chariot				'\n'
#define d_code_tabulation					'\t'
#define d_longueur_maximale_instruction		16

#define d_INT					"integer"
#define	d_REL					"real"
#define d_CPL					"complex"
#define d_VIN					"[ integer ]"
#define d_VRL					"[ real ]"
#define d_VCX					"[ complex ]"
#define d_MIN					"[[ integer ]]"
#define d_MRL					"[[ real ]]"
#define d_MCX					"[[ complex ]]"
#define d_BIN					"# binary[bodh]"
#define d_NOM					"'name'"
#define d_CHN					"\"string\""
#define d_LST					"{ list }"
#define d_ALG					"'algebraic'"
#define d_RPN					"<< RPN >>"
#define d_FCH					"File $n { format }"
#define d_SLB					"Library $n"
#define d_SQL					"Sql $n (type)"
#define d_PRC					"Process $n"
#define d_SCK					"Socket $n { format }"
#define d_SPH					"Semaphore $n 'name'"
#define d_TAB					"<[ table ]>"
#define d_MTX					"Mutex $n"

/*
--------------------------------------------------------------------------------
  Paramtres du systme
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
#	define ds_beep								""
#	define ds_fichier_historique				".RPL2_history"
#	define ds_imposition_separateur_decimal		d_vrai
#	define ds_longueur_historique				256
#	define ds_nombre_maximum_fichiers_ouverts	65536
#	define ds_preprocesseur						"rplpp " \
													"--nostdinc -C 2> /dev/null"
#	define ds_rplcore							"rpl-core"
#	define ds_rplprofile						"rpl-profile"
#	define ds_sdat								"SDAT"
#endif

/*
--------------------------------------------------------------------------------
  Erreurs systme
--------------------------------------------------------------------------------
*/

#ifdef DEBUG_ERREURS
#	ifdef MESSAGES
#		define __erreur(i)	i
#		define __erreur_(i)	i
#	else
#		undef _ERREURS_VARIABLES
#		ifdef _ERREURS_VARIABLES
#			define __erreur_(i)	__erreur(i)
#		else
#			define __erreur_(i)	i
#		endif
#		define __erreur(i)	({ if (strstr(__FUNCTION__, "recherche_variable") \
				== NULL) ufprintf(stderr, \
				"ERROR %d AT %s() FROM %s LINE %d\n", \
				i, __FUNCTION__, __FILE__, __LINE__); i; })
#	endif
#else
#	define __erreur(i)	i
#	define __erreur_(i)	i
#endif


// -1 : erreur provoque par une bibliothque externe
#ifndef RPLARGS
#	define d_es									2000
#	define d_es_allocation_memoire				__erreur(2001)
#	define d_es_variable_introuvable			__erreur_(2002)
#	define d_es_pile_vide						__erreur(2003)
#	define d_es_end_incoherent					__erreur(2004)
#	define d_es_peripherique_stdin				__erreur(2005)
#	define d_es_erreur_fichier					__erreur(2006)
#	define d_es_execution_bibliotheque			__erreur(2007)
#	define d_es_signal							__erreur(2008)
#	define d_es_processus						__erreur(2009)
#	define d_es_pile_operationnelle_vide		__erreur(2010)
#	define d_es_nombre_max_descripteurs			__erreur(2011)
#	define d_es_interruption_invalide			__erreur(2012)
#	define d_es_contexte						__erreur(2013)
#	define d_es_somme_controle					__erreur(2014)
#	define d_es_semaphore						__erreur(2015)
#endif

/*
--------------------------------------------------------------------------------
  Exceptions
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
#	define d_ep									1000
#	define d_ep_division_par_zero				__erreur(1001)
#	define d_ep_matrice_non_inversible			__erreur(1002)
#	define d_ep_resultat_indefini				__erreur(1003)
#	define d_ep_underflow						__erreur(1004)
#	define d_ep_overflow						__erreur(1005)
#	define d_ep_domaine_definition				__erreur(1006)
#	define d_ep_perte_precision					__erreur(1007)
#	define d_ep_decomposition_QR				__erreur(1008)
#	define d_ep_matrice_non_definie_positive	__erreur(1009)
#	define d_ep_decomposition_QZ				__erreur(1010)
#	define d_ep_decomposition_SVD				__erreur(1011)
#endif

/*
--------------------------------------------------------------------------------
  Erreurs de compilation
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
#	define d_ec									3000
#	define d_ec_niveau_definition_negatif		__erreur(3001)
#	define d_ec_nom_definition_invalide			__erreur(3002)
#	define d_ec_erreur_boucle_definie			__erreur(3003)
#	define d_ec_erreur_instruction_end			__erreur(3004)
#	define d_ec_erreur_instruction_until		__erreur(3005)
#	define d_ec_source_incoherent				__erreur(3006)
#	define d_ec_erreur_instruction_while		__erreur(3007)
#	define d_ec_erreur_instruction_then			__erreur(3008)
#	define d_ec_erreur_instruction_else			__erreur(3009)
#	define d_ec_erreur_instruction_elseif		__erreur(3010)
#	define d_ec_erreur_instruction_select		__erreur(3011)
#	define d_ec_erreur_instruction_case			__erreur(3012)
#endif

/*
--------------------------------------------------------------------------------
  Erreurs  l'excution
--------------------------------------------------------------------------------
*/

// -1 : erreur provoque par une bibliothque externe
#ifndef RPLARGS
#	define d_ex									0
#	define d_ex_pile_vide						__erreur(1)
#	define d_ex_manque_argument					__erreur(2)
#	define d_ex_argument_invalide				__erreur(3)
#	define d_ex_erreur_type_argument			__erreur(4)
#	define d_ex_nom_reserve						__erreur(5)
#	define d_ex_nombre_arguments				__erreur(6)
#	define d_ex_dimensions_invalides			__erreur(7)
#	define d_ex_expression_invalide				__erreur(8)
#	define d_ex_absence_graphique_courant		__erreur(9)
#	define d_ex_erreur_traitement_condition		__erreur(10)
#	define d_ex_erreur_traitement_boucle		__erreur(11)
#	define d_ex_variable_non_definie			__erreur_(12)
#	define d_ex_drapeau_inexistant				__erreur(13)
#	define d_ex_nom_invalide					__erreur(14)
#	define d_ex_element_inexistant				__erreur(15)
#	define d_ex_absence_variable				__erreur(16)
#	define d_ex_erreur_evaluation				__erreur(17)
#	define d_ex_semaphore						__erreur(18)
#	define d_ex_longueur_fft					__erreur(19)
#	define d_ex_queue_impression				__erreur(20)
#	define d_ex_type_trace_invalide				__erreur(21)
#	define d_ex_erreur_fichier					__erreur(22)
#	define d_ex_absence_equation				__erreur(23)
#	define d_ex_erreur_bibliotheque				__erreur(24)
#	define d_ex_execution_bibliotheque			__erreur(25)
#	define d_ex_stop							__erreur(26)
#	define d_ex_matrice_statistique_invalide	__erreur(27)
#	define d_ex_dimensions_matrice_statistique	__erreur(28)
#	define d_ex_absence_observations			__erreur(29)
#	define d_ex_statistiques_echantillon		__erreur(30)
#	define d_ex_observations_inexistantes		__erreur(31)
#	define d_ex_syntaxe							__erreur(32)
#	define d_ex_cycle_hors_boucle				__erreur(33)
#	define d_ex_conversion_unite				__erreur(34)
#	define d_ex_erreur_parametre_fichier		__erreur(35)
#	define d_ex_erreur_acces_fichier			__erreur(36)
#	define d_ex_erreur_requete_fichier			__erreur(37)
#	define d_ex_erreur_format_fichier			__erreur(38)
#	define d_ex_fichier_verrouille				__erreur(39)
#	define d_ex_verrou_indefini					__erreur(40)
#	define d_ex_fin_de_fichier_atteinte			__erreur(41)
#	define d_ex_debut_de_fichier_atteint		__erreur(42)
#	define d_ex_erreur_type_fichier				__erreur(43)
#	define d_ex_fichier_vide					__erreur(44)
#	define d_ex_dimensions_differentes			__erreur(45)
#	define d_ex_routines_mathematiques			__erreur(46)
#	define d_ex_exit_hors_boucle				__erreur(47)
#	define d_ex_longueur_dft					__erreur(48)
#	define d_ex_contexte						__erreur(49)
#	define d_ex_processus						__erreur(50)
#	define d_ex_image_processus					__erreur(51)
#	define d_ex_erreur_sql						__erreur(52)
#	define d_ex_variable_verrouillee			__erreur(53)
#	define d_ex_variable_volatile				__erreur(54)
#	define d_ex_erreur_processus				__erreur(55)
#	define d_ex_erreur_impression				__erreur(56)
#	define d_ex_nombre_arguments_fonction		__erreur(57)
#	define d_ex_fonction_indisponible			__erreur(58)
#	define d_ex_bibliotheque_chargee			__erreur(59)
#	define d_ex_aucun_symbole					__erreur(60)
#	define d_ex_definition_ambigue				__erreur(61)
#	define d_ex_fichier_hors_contexte			__erreur(62)
#	define d_ex_socket_en_ecoute				__erreur(63)
#	define d_ex_interruption_invalide			__erreur(64)
#	define d_ex_erreur_transcodage				__erreur(65)
#	define d_ex_absence_processus_pere			__erreur(66)
#	define d_ex_creation_variable				__erreur(67)
#	define d_ex_fusible							__erreur(68)
#	define d_ex_iswi_hors_interruption			__erreur(69)
#	define d_ex_daemon							__erreur(70)
#	define d_ex_mutex							__erreur(71)
#	define d_ex_variable_statique_partagee		__erreur(72)
#	define d_ex_variable_partagee				__erreur(73)
#	define d_ex_graphique_memorise				__erreur(74)
#	define d_ex_matrice_non_diagonale			__erreur(75)
#	define d_ex_locales							__erreur(76)
#	define d_ex_representation					__erreur(77)
#	define d_ex_erreur_profilage				__erreur(78)
#	define d_ex_enregistrement_inexistant		__erreur(79)
#	define d_ex_clef_inexistante				__erreur(80)
#	define d_ex_nom_implicite					__erreur(81)
#	define d_ex_version_bibliotheque			__erreur(82)
#	define d_ex_creation_variable_globale		__erreur(83)
#	define d_ex_erreur_interne_rplcas			__erreur(84)
#endif


/*
================================================================================
  STRUCTURES
================================================================================
*/

/*
--------------------------------------------------------------------------------
  Structure objet
--------------------------------------------------------------------------------
*/

#ifdef MIN
#	undef MIN
#endif

#ifdef MAX
#	undef MAX
#endif

#define ADR	__RPL_ADR
#define ALG	__RPL_ALG
#define BIN	__RPL_BIN
#define CHN	__RPL_CHN
#define CPL	__RPL_CPL
#define FCH	__RPL_FCH
#define FCT	__RPL_FCT
#define INT	__RPL_INT
#define LST	__RPL_LST
#define MCX	__RPL_MCX
#define MIN	__RPL_MIN
#define MRL	__RPL_MRL
#define MTX	__RPL_MTX
#define NOM	__RPL_NOM
#define NON	__RPL_NON
#define PRC	__RPL_PRC
#define REL	__RPL_REL
#define RPN	__RPL_RPN
#define SCK	__RPL_SCK
#define SLB	__RPL_SLB
#define SPH	__RPL_SPH
#define SQL	__RPL_SQL
#define TBL	__RPL_TBL
#define VCX	__RPL_VCX
#define VIN	__RPL_VIN
#define VRL	__RPL_VRL

enum t_rplcas_commandes		{ RPLCAS_INTEGRATION = 0, RPLCAS_LIMITE };

enum t_type		{ ADR = 0, ALG, BIN, CHN, CPL, FCH, FCT, INT, LST,
				MCX, MIN, MRL, MTX, NOM, NON, PRC, REL, RPN, SCK,
				SLB, SPH, SQL, TBL, VCX, VIN, VRL };

	/*
	ADR : adresse sur 32 bits (au moins) non signs (unsigned long)
			Adresse d'une dfinition  interprter.

	ALG : expression algbrique (struct_liste_chainee *)
			La liste chane contient la dfinition convertie en notation
			polonaise inverse. Ce type diffre du type 'RPN' au niveau
			du format de sortie (notation algbrique).

	BIN : entier binaire sur 64 bits (unsigned integer8 *)
			Sans objet.
			Type C/Fortran : unsigned integer8

	CHN : chaine de caracteres (character*(*), unsigned char *)
			Sans objet.

	CPL : complexe sur 2*64 bits (complex*16, struct_complexe16 *)
			Sans objet.
			Type C/Fortran : complex16

	FCH : descripteur de fichier (struct_fichier *).

	FCT : dclaration d'une fonction et de son nombre d'arguments
			(struct_fonction *). Ce type n'est ncessaire que dans le
			traitement des types 'ALG' et 'RPN'.

	INT : entier sur 64 bits (integer*8, integer8 *)
			Sans objet.
			Type C/Fortran : integer8

	LST : liste (struct_liste_chainee *)
			Les objets sont enregistrs sous forme de liste chane rcursive.
			Cela permet de traiter des tableaux de donnes htrognes de
			grandes dimensions. Le nombre de dimensions maximal est fix par
			la mmoire de la station.

	MCX : matrice de complexes (struct_matrice *)
			Sans objet.

	MIN : matrice d'entiers (struct_matrice *)
			Sans objet.

	MRL : matrice de reels (struct_matrice *)
			Sans objet.

	MTX : mutex (struct_mutex *)
			Sans objet.

	NOM : nom de variable (struct_nom *)
			Sans objet.

	PRC : processus (struct_processus_fils *)
			Sans objet.

	REL : reel sur 64 bits (real*8, real8 *)
			Sans objet.

	RPN : definition (struct_liste_chainee *)
			Ce type diffre du type 'ALG' au niveau du format de sortie
			(notation polonaise inverse).

	SCK : socket (struct_socket *)

	SLB : bibliothque dynamique partage (struct_bibliotheque *) 
			Sans objet.

	SPH : smaphore nomm (struct_semaphore *)

	SQL : connecteur sur une base de donnes SQL (struct_connecteur_sql *)

	TBL : tableau multidimensionnel d'objets (struct_tableau *).

	VCX : vecteur de complexes (struct_vecteur *)
			Sans objet.

	VIN : vecteur d'entiers (struct_vecteur *)
			Sans objet.

	VRL : vecteur de reels (struct_vecteur *)
			Sans objet.
	*/

typedef struct objet
{
	enum t_type			type;

	volatile long		nombre_occurrences;

	pthread_mutex_t		mutex;

	void				*objet;
} struct_objet;


/*
--------------------------------------------------------------------------------
  Structure fichier
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
typedef struct fichier
{
	unsigned long				descripteur;

	unsigned char				acces;			/* S/D/I */
	unsigned char				binaire;		/* Y/N/F */
	unsigned char				ouverture;
	unsigned char				protection;
	unsigned char				*nom;

	struct_objet				*format;

	pid_t						pid;
	pthread_t					tid;

	integer8					position_clef;
} struct_fichier;
#endif


/*
--------------------------------------------------------------------------------
  Structure mutex
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
typedef struct rpl_mutex
{
	pthread_mutex_t		mutex;
} struct_mutex;
#endif


/*
--------------------------------------------------------------------------------
  Structure smaphore
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
typedef struct semaphore
{
	unsigned char	*nom;
	sem_t			*semaphore;
} struct_semaphore;
#endif


/*
--------------------------------------------------------------------------------
  Structure marque
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
typedef struct marque
{
	unsigned char		*label;
	unsigned char		*position;

	struct marque		*suivant;
} struct_marque;
#endif


/*
--------------------------------------------------------------------------------
  Structure bibliothque
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
typedef struct bibliotheque
{
	void						*descripteur;

	pid_t						pid;
	pthread_t					tid;

	unsigned char				*nom;
} struct_bibliotheque;
#endif


/*
--------------------------------------------------------------------------------
  Structure descripteur fichier
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
typedef struct descripteur_fichier
{
	unsigned long				identifiant;

	unsigned char				effacement;
	unsigned char				type; // C (file *) ou S (sqlite *)
	unsigned char				*nom;

	pid_t						pid;
	pthread_t					tid;

	file						*descripteur_c;
	sqlite3						*descripteur_sqlite;
} struct_descripteur_fichier;
#endif


/*
--------------------------------------------------------------------------------
  Structure connecteur SQL
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
typedef struct connecteur_sql
{
	pid_t						pid;
	pthread_t					tid;

	/*
	 * "MYSQL"
	 * "POSTGRESQL"
	 */

	unsigned char				*type;
	unsigned char				*locale;

	union
	{
#		ifdef MYSQL_SUPPORT
			MYSQL				*mysql;
#		endif
#		ifdef POSTGRESQL_SUPPORT
			PGconn				*postgresql;
#		endif
	} descripteur;
} struct_connecteur_sql;
#endif

#define sqlclose(s_objet) \
	if (strcmp((*((struct_connecteur_sql *) (*s_objet).objet)).type, "MYSQL") \
			== 0) \
		mysqlclose((*((struct_connecteur_sql *) (*s_objet).objet)) \
				.descripteur); \
	else if (strcmp((*((struct_connecteur_sql *) (*s_objet).objet)).type, \
			"POSTGRESQL") == 0) \
		postgresqlclose((*((struct_connecteur_sql *) (*s_objet).objet)) \
				.descripteur); \
	else \
		BUG(1, uprintf("SQL type '%s' not allowed!", \
				(*((struct_connecteur_sql *) (*s_objet).objet)).type));

/*
--------------------------------------------------------------------------------
  Structure socket
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
typedef struct socket
{
	int							domaine;
	int							socket;

	pid_t						pid;
	pthread_t					tid;

	unsigned char				*adresse;
	unsigned char				*adresse_distante;
	unsigned char				binaire;
	unsigned char				socket_en_ecoute;
	logical1					socket_connectee;
	unsigned char				effacement;
	unsigned char				localisation;
	unsigned char				protection;
	unsigned char				type[19 + 1];

	struct_objet				*format;
} struct_socket;
#endif

/*
--------------------------------------------------------------------------------
  Structure liste chanee
--------------------------------------------------------------------------------
*/

typedef struct liste_chainee
{
	struct liste_chainee		*suivant;

	struct_objet				*donnee;
} struct_liste_chainee; 


/*
--------------------------------------------------------------------------------
  Structure liste profilage
--------------------------------------------------------------------------------
*/

typedef struct liste_profilage
{
	struct liste_profilage		*suivant;

	unsigned char				*fonction;

	struct timeval				dernier_appel;
	struct timeval				cumul;

	integer8					nombre_appels;
} struct_liste_profilage; 

typedef struct liste_profilage2
{
	struct liste_profilage2		*suivant;

	unsigned char				*fonction;
} struct_liste_profilage2; 

/*
--------------------------------------------------------------------------------
  Structure contenant les fichiers graphiques (liste chane)
--------------------------------------------------------------------------------
*/

#ifndef RPLARLS
typedef struct fichier_graphique
{
	struct fichier_graphique	*suivant;

	logical1					presence_axes;

	unsigned char				*nom;
	unsigned char				type[21];

	unsigned char				*legende;

	int							dimensions;

	integer8					systeme_axes;
} struct_fichier_graphique;
#endif

/*
--------------------------------------------------------------------------------
  Structure nom
--------------------------------------------------------------------------------
*/

typedef struct nom
{
	unsigned char				*nom;
	logical1					symbole;	/* d_vrai/d_faux */
	// symbole == d_vrai signifie que le nom a t introduit dans un
	// calcul sans les apostrophes.
} struct_nom;


/*
--------------------------------------------------------------------------------
  Structure processus fils
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
typedef struct descripteur_thread
{
	struct processus			*s_etat_processus;
	struct processus			*s_nouvel_etat_processus;

	struct_objet				*argument;

	pid_t						pid;
	volatile pthread_t			tid;

	pthread_t					thread_pere;

	pthread_mutex_t				mutex;
	pthread_mutex_t				mutex_nombre_references;

	volatile logical1			thread_actif;

	int							pipe_erreurs[2];
	int							pipe_objets[2];
	int							pipe_injections[2];
	int							pipe_nombre_injections[2];
	int							pipe_interruptions[2];
	int							pipe_nombre_objets_attente[2];
	int							pipe_nombre_interruptions_attente[2];
	int							pipe_acquittement[2];

	volatile integer8			nombre_objets_dans_pipe;
	volatile integer8			nombre_interruptions_dans_pipe;

	logical1					processus_detache;
	logical1					destruction_objet;

	volatile signed long		nombre_references;
} struct_descripteur_thread;

typedef struct processus_fils
{
	struct_descripteur_thread		*thread;
} struct_processus_fils;
#endif


/*
--------------------------------------------------------------------------------
  Structure liste pile systeme
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
typedef struct liste_pile_systeme
{
	struct liste_pile_systeme	*suivant;

	struct_liste_chainee		*pointeur_objet_retour;

	struct_objet				*indice_boucle;
	struct_objet				*limite_indice_boucle;
	struct_objet				*objet_de_test;

	logical1					arret_si_exception;
	logical1					creation_variables_statiques;
	logical1					creation_variables_partagees;
	logical1					evaluation_expression;

	unsigned char				clause;

	/*
	clause :
	I : IF
	R : IFERR
	X : IFERR indiquant la prsence d'une erreur.
	T : THEN
	E : ELSE
	Z : traitement en cours de ELSE
	D : DO
	U : UNTIL
	W : WHILE
	M : WHILE avec une clause fausse.
	S : SELECT
	K : aucun cas CASE n'a encore t trait.
	C : au moins un cas de SELECT...CASE a t trait.
	Q : traitement en cours d'un cas CASE.
	F : traitement du cas DEFAULT
	*/

	unsigned char				retour_definition;
	unsigned char				*nom_variable;
	unsigned char				origine_routine_evaluation;
	unsigned char				type_cloture;

	/*
	type clture :
	C     SELECT  : END termine un test SELECT/DEFAULT/END.
	D     DO      : END termine une boucle DO/UNTIL/END.
	I     IF      : END termine un test IF/THEN (ELSE)/END.
	J     IFERR   : END termine un test IFERR/THEN (ELSE)/END.
	K     CASE    : END termine un test CASE/THEN/END
	W     WHILE   : END termine une boucle WHILE/REPEAT/END.

	F     FOR     : NEXT ou STEP termine une boucle avec compteur.
	S     START   : NEXT ou STEP termine une boucle sans compteur.
	L     LOOP    : boucle utilis dans le traitement de l'instruction RETURN.
	        Elle correspond  une boucle FOR ou START mais sans son
			initialisation.
	*/

	unsigned long				adresse_retour;
	unsigned long				niveau_courant;
	struct_objet				*pointeur_adresse_retour;
} struct_liste_pile_systeme;
#endif


/*
--------------------------------------------------------------------------------
  Structure variable
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
typedef union position_variable
{
	unsigned long		adresse;
	struct_objet		*pointeur;
} union_position_variable;

typedef struct variable
{
	unsigned char			*nom;
	unsigned char			origine;

	unsigned long			niveau;

	union_position_variable	variable_statique;
	union_position_variable	variable_partagee;

	struct_objet			*objet;

	logical1				variable_verrouillee;
} struct_variable;

#endif

/*
--------------------------------------------------------------------------------
  Structure variable partage
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS

typedef struct variable_partage
{
	unsigned char				*nom;
	unsigned char				origine;

	unsigned long				niveau;

	union_position_variable		variable_partagee;

	struct_objet				*objet;
} struct_variable_partagee;

typedef struct table_variables_partagees
{
	pthread_mutex_t						mutex;

	volatile struct_variable_partagee	*table;

	volatile unsigned long				nombre_variables;
	volatile unsigned long				nombre_variables_allouees;

	unsigned long						position_variable;
} struct_table_variables_partagees;

#endif


/*
--------------------------------------------------------------------------------
  Structure variable statique
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
typedef struct variable_statique
{
	unsigned char			*nom;
	unsigned char			origine;

	// Position de cration de la variable statique dans le programme ou dans
	// l'expression. 'origine' vaut 'P' pour programme ou 'E' pour expression.

	unsigned long			niveau;

	union_position_variable	variable_statique;

	struct_objet			*objet;
} struct_variable_statique;
#endif


/*
--------------------------------------------------------------------------------
  Structure vecteur
--------------------------------------------------------------------------------
*/

typedef struct vecteur
{
	unsigned long		taille;

	unsigned char		type; /* C (complex*16), R (real*8), I (integer*8) */

	void				*tableau;
} struct_vecteur;


/*
--------------------------------------------------------------------------------
  Structure matrice
--------------------------------------------------------------------------------
*/

typedef struct matrice
{
	unsigned long		nombre_lignes;
	unsigned long		nombre_colonnes;

	unsigned char		type; /* C (complex*16), R (real*8), I (integer*8) */

	void				**tableau;
} struct_matrice;


/*
--------------------------------------------------------------------------------
  Structure tableau
--------------------------------------------------------------------------------
*/

typedef struct tableau
{
	unsigned long		nombre_elements;

	struct_objet		**elements;
} struct_tableau;


/*
--------------------------------------------------------------------------------
  Structure arbre
--------------------------------------------------------------------------------
*/

typedef struct arbre
{
	struct_objet		*objet;

	unsigned long		nombre_feuilles;
	struct arbre		**feuilles;
} struct_arbre;


/*
--------------------------------------------------------------------------------
  Structures de gestion des instructions externes
--------------------------------------------------------------------------------
*/

typedef struct instruction_externe
{
	unsigned char		*nom;
	unsigned char		*nom_bibliotheque;
	void				*descripteur_bibliotheque;
} struct_instruction_externe;


/*
--------------------------------------------------------------------------------
  Structure argument (pour les fonctions dynamiques)
--------------------------------------------------------------------------------
*/

typedef struct rpl_arguments
{
/* Piles operationnelles								*/

	struct_liste_chainee		*l_base_pile;
	struct_liste_chainee		*l_base_pile_last;

/* Etat du calculateur									*/

	t_8_bits					drapeaux_etat[8];

/* Traitement des erreurs								*/

	unsigned char				*message_erreur;
	unsigned char				type_erreur;			/* S = systme
														   E = excution */
	integer8					erreur;

/* Traitement des messages d'aide						*/

	logical1					aide;					/* Affichage d'un
														   message d'aide
														   par la routine */

/* Nombre d'arguments									*/

	unsigned char				affichage_arguments;
	unsigned char				constante_symbolique;
	unsigned char				instruction_valide;
	unsigned char				test_instruction;

	integer8					nombre_arguments;
	// Nombre d'arguments de la fonction, positif ou nul.

	void						*s_etat_processus;
} struct_rpl_arguments;

/*
--------------------------------------------------------------------------------
  Structure d'arbre des instructions intrinsques
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
typedef struct instruction
{
	struct instruction		**noeuds;
	void					(*feuille)(struct processus *);
} struct_instruction;
#endif

/*
--------------------------------------------------------------------------------
  Structure d'arbre des variables variable globales et locales
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
typedef struct arbre_variables
{
	unsigned int			noeuds_utilises;
	signed int				indice_tableau_pere;
	struct arbre_variables	*noeud_pere;
	struct arbre_variables	**noeuds;
	struct liste_variables	*feuille;
} struct_arbre_variables;

typedef struct tableau_variables
{
	unsigned char			origine;
	unsigned char			*nom;		// pointeur sur la struct_variable
										// relle et non copie de la chane
	unsigned long			niveau;

	struct_objet			*objet;		// pointeur sur l'objet et non copie
										// de l'objet.

	logical1				variable_verrouillee;
	union_position_variable	variable_statique;
	union_position_variable	variable_partagee;
} struct_tableau_variables;

typedef struct liste_variables
{
	union
	{
		// Utilisation dans la gestion des variables
		struct_variable					*variable;
		// Utilisation dans la pile systme (variables par niveau)
		struct_liste_chainee			*liste;
	};

	struct arbre_variables				*noeud_pere;
	struct arbre_variables				*noeud;
	struct liste_variables				*suivant;
	struct liste_variables				*precedent;
} struct_liste_variables;

#endif

/*
--------------------------------------------------------------------------------
  Structure processus
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
typedef struct processus
{
/* Pointeurs sur le programme							*/

	unsigned char				*definitions_chainees;
	struct_liste_chainee		*expression_courante;

	unsigned char				*instruction_courante;
	struct_objet				*objet_courant;
	unsigned char				evaluation_expression_compilee;		/*Y/N*/

	unsigned char				*nom_fichier_source;
	unsigned char				*nom_fichier_historique;

	unsigned char				autorisation_conversion_chaine;		/*Y/N*/
	unsigned char				autorisation_empilement_programme;	/*Y/N*/
	unsigned char				autorisation_evaluation_nom;		/*Y/N*/
	unsigned char				autorisation_nom_implicite;			/*Y/N*/
	unsigned char				evaluation_forcee;					/*Y/N*/
	unsigned char				instruction_valide;					/*Y/N*/
	unsigned char				instruction_intrinseque;			/*Y/I/N*/
	unsigned char				test_instruction;					/*Y/N*/
	unsigned char				affichage_arguments;				/*Y/N*/
	unsigned char				constante_symbolique;				/*Y/N*/
	unsigned char				traitement_symbolique;				/*Y/N*/
	unsigned char				mode_execution_programme;			/*Y/I/N*/
	unsigned char				mode_interactif;					/*Y/N*/
	unsigned char				mode_evaluation_expression;			/*Y/N*/
	unsigned char				traitement_cycle_exit;				/*N/E/C*/
	unsigned char				recherche_type;						/*Y/N*/

	unsigned long				position_courante;
	unsigned long				longueur_definitions_chainees;

/* Pointeurs sur les instructions						*/

	int							*pointeurs_caracteres;
	int							nombre_caracteres;
	struct_instruction			*arbre_instructions;

/* Requetes												*/

	unsigned char				requete_arret;						/*Y/N*/

/* Piles operationnelles								*/

	struct_liste_chainee		*l_base_pile;
	struct_liste_chainee		*l_base_pile_last;

	unsigned long				hauteur_pile_operationnelle;

/* Piles systemes										*/

	struct_liste_pile_systeme	*l_base_pile_systeme;

	unsigned long				hauteur_pile_systeme;

/* Gestion des processus                                */

	volatile
	struct_liste_chainee		*l_base_pile_processus;

	logical1					presence_pipes;

	int							pipe_acquittement;
	int							pipe_donnees;
	int							pipe_injections;
	int							pipe_nombre_injections;
	int							pipe_interruptions;
	int							pipe_nombre_objets_attente;
	int							pipe_nombre_interruptions_attente;
	int							nombre_objets_envoyes_non_lus;
	int							nombre_objets_injectes;

	real8						pourcentage_maximal_cpu;
	real8						temps_maximal_cpu;

	logical1					presence_fusible;
	pthread_t					thread_fusible;
	pthread_t					thread_surveille_par_fusible;

	volatile int				pointeur_signal_ecriture;
	volatile int				pointeur_signal_lecture;
	volatile enum signaux_rpl	signaux_en_queue[LONGUEUR_QUEUE_SIGNAUX];
	volatile logical1			demarrage_fils;
	volatile logical1			redemarrage_processus;

/* Contextes											*/

	struct_liste_chainee		*l_base_pile_contextes;
	struct_liste_chainee		*l_base_pile_taille_contextes;

/* Variables											*/

	// La liste des variables par niveau est doublement chane.
	//  tout moment, elle pointe sur le niveau le plus haut existant et
	// l_liste_variable_par_niveau->precedent renvoie la liste des
	// dfinitions. l_liste_variable_par_niveau->precedent->precedent pointe
	// sur la liste des variables globales.
	//
	//  l'initialisation :
	// l_liste_variables_par_niveau->suivant == l_liste_variables_par_niveau
	// l_liste_variables_par_niveau->precedent == l_liste_variables_par_niveau

	struct_arbre_variables		*s_arbre_variables;
	struct_liste_variables		*l_liste_variables_par_niveau;
	logical1					niveau_supprime;

	struct_variable				*pointeur_variable_courante;
	struct_liste_variables		*pointeur_feuille_courante;

	int							*pointeurs_caracteres_variables;
	int							nombre_caracteres_variables;

	struct_variable_statique	*s_liste_variables_statiques;
	unsigned long				nombre_variables_statiques;
	unsigned long				nombre_variables_statiques_allouees;

	struct_table_variables_partagees
								*s_liste_variables_partagees;

	unsigned long				niveau_courant;
	unsigned long				niveau_initial;
	unsigned long				position_variable_statique_courante;

	logical1					creation_variables_statiques;
	logical1					creation_variables_partagees;
	logical1					gel_liste_variables;

/* Fichiers												*/

	struct_liste_chainee		*s_fichiers;

/*	Sockets												*/

	struct_liste_chainee		*s_sockets;

/*	Connecteurs aux bases de donnes					*/

	struct_liste_chainee		*s_connecteurs_sql;

/* Bibliothques dynamiques								*/

	struct_liste_chainee		*s_bibliotheques;
	struct_instruction_externe	*s_instructions_externes;
	unsigned long				nombre_instructions_externes;

/* Fichier d'impression									*/

	unsigned char				*nom_fichier_impression;
	unsigned char				format_papier[16];

/* Parametres graphiques								*/

	struct_fichier_graphique	*fichiers_graphiques;	 /* NULL */

	unsigned char				type_trace_eq[21];

								/*
								 * 2D :
								 * FONCTION
								 * PARAMETRIQUE
								 * POLAIRE
								 * COURBES DE NIVEAU
								 *
								 * 3D :
								 * GRILLE 3D
								 * COURBE PARAMETRIQUE // A FAIRE
								 * SURFACE PARAMETRIQUE // A FAIRE
								 * CYNLINDRIQUE // A FAIRE
								 * SPHERIQUE // A FAIRE
								 */

	unsigned char				type_trace_sigma[21];

								/*
								 * POINTS
								 * HISTOGRAMME
								 * TABLE TRACANTE
								 */

								/*
								 * ECHELLES
								 */

	integer8					systeme_axes;

	real8						x_min;
	real8						x_max;
	real8						y_min;
	real8						y_max;
	real8						z_min;
	real8						z_max;

	real8						x_tics;
	real8						y_tics;
	real8						z_tics;
	
	integer8					mx_tics;
	integer8					my_tics;
	integer8					mz_tics;

	logical1					x_lines;
	logical1					y_lines;
	logical1					z_lines;

	logical1					mx_lines;
	logical1					my_lines;
	logical1					mz_lines;

								/*
								 * -1 : absent du graphique
								 *  0 : graduations automatiques
								 * >0 : valeur manuelle
								 */

	real8						x2_min;
	real8						x2_max;
	real8						y2_min;
	real8						y2_max;
	real8						z2_min;
	real8						z2_max;

	real8						x2_tics;
	real8						y2_tics;
	real8						z2_tics;

	integer8					mx2_tics;
	integer8					my2_tics;
	integer8					mz2_tics;

	logical1					x2_lines;
	logical1					y2_lines;
	logical1					z2_lines;

	logical1					mx2_lines;
	logical1					my2_lines;
	logical1					mz2_lines;

	real8						resolution;

	logical1					souris_active;

	logical1					echelle_automatique_x;
	logical1					echelle_automatique_y;
	logical1					echelle_automatique_z;

	logical1					echelle_automatique_x2;
	logical1					echelle_automatique_y2;
	logical1					echelle_automatique_z2;

	logical1					echelle_log_x;
	logical1					echelle_log_y;
	logical1					echelle_log_z;

	logical1					echelle_log_x2;
	logical1					echelle_log_y2;
	logical1					echelle_log_z2;

	real8						point_de_vue_theta;
	real8						point_de_vue_phi;
	real8						echelle_3D;

	unsigned char				*label_x;
	unsigned char				*label_y;
	unsigned char				*label_z;
	unsigned char				*titre;
	unsigned char				*legende;

	unsigned char				*nom_fichier_gnuplot;
	unsigned char				*type_fichier_gnuplot;

	struct_objet				*indep;
	struct_objet				*depend;
	struct_objet				*parametres_courbes_de_niveau;

	struct_marque				*s_marques;

	file						*entree_standard;

	logical1					requete_nouveau_plan;
	logical1					mise_a_jour_trace_requise;

/* Parametres statistiques								*/

	integer8					colonne_statistique_1;
	integer8					colonne_statistique_2;

/* Etat du calculateur									*/

	t_8_bits					drapeaux_etat[8];

/* Traitement des exceptions							*/

	logical1					arret_si_exception;
	unsigned int				exception;
	unsigned int				derniere_exception;
	unsigned int				exception_processus_fils;

/* Traitement des erreurs								*/

	unsigned int				erreur_compilation;
	unsigned int				erreur_execution;
	unsigned int				erreur_systeme;

	struct_objet				*s_objet_errone;
	struct_objet				*s_objet_erreur;

	logical1					erreur_scrutation;

	volatile unsigned int		erreur_systeme_processus_fils;
	unsigned int				erreur_execution_processus_fils;

	pid_t						pid_erreur_processus_fils;

	logical1					erreur_processus_fils;

	integer8					derniere_erreur_fonction_externe;

	unsigned int				derniere_erreur_systeme;
	unsigned int				derniere_erreur_execution;
	unsigned int				derniere_erreur_evaluation;

	unsigned char				*instruction_derniere_erreur;

	unsigned long				niveau_derniere_erreur;

	logical1					core;
	logical1					invalidation_message_erreur;

/* Debug												*/

	logical1					debug;
	integer8					type_debug;
	logical1					debug_programme;
	logical1					execution_pas_suivant;
	logical1					traitement_instruction_halt;

/* Interruptions logicielles							*/

	unsigned char				masque_interruptions[d_NOMBRE_INTERRUPTIONS];

	/*
	 * 'I' : interruption ignore
	 * 'Q' : interruption mise en queue pour un traitement ultrieur
	 * 'N' : traitement normal de l'interruption
	 */

	integer8					queue_interruptions[d_NOMBRE_INTERRUPTIONS];
	integer8					nombre_interruptions_en_queue;
	struct_liste_chainee		*pile_origine_interruptions
										[d_NOMBRE_INTERRUPTIONS];
	volatile integer8			nombre_interruptions_non_affectees;

	struct_objet				*corps_interruptions[d_NOMBRE_INTERRUPTIONS];

	unsigned char				traitement_interruption;  /* Y/N */
	unsigned char				traitement_interruptible; /* Y/N */
	unsigned char				traitement_at_poke;       /* Y/N */

	struct_objet				*at_exit;
	struct_objet				*at_poke;

/* Variables volatiles									*/

	volatile sig_atomic_t		var_volatile_alarme;
	// Traitement de ctrl+C et des arrts brutaux
	volatile sig_atomic_t		var_volatile_requete_arret;
	// Traitement de ctrl+Z
	volatile sig_atomic_t		var_volatile_requete_arret2;
	volatile sig_atomic_t		var_volatile_traitement_retarde_stop;
	volatile sig_atomic_t		var_volatile_traitement_sigint;

	volatile sig_atomic_t		var_volatile_processus_racine;
	volatile sig_atomic_t		var_volatile_processus_pere;
	volatile sig_atomic_t		var_volatile_recursivite;

	volatile sig_atomic_t		arret_depuis_abort;

	volatile int				var_volatile_exception_gsl;

/* Autres												*/

	char						*rpl_home;

	gsl_rng						*generateur_aleatoire;
	const gsl_rng_type			*type_generateur_aleatoire;

	void						*contexte_cas;

	integer8					nombre_arguments;

	/*
	 * Il s'agit du nombre d'arguments utilis pour l'valuation
	 * des expressions algbriques.
	 * -1 : la fonction ne peut tre dans un objet de type ALG.
	 * -2 : nombre d'arguments inconnu
	 *  0 : la fonction doit apparatre en notation infixe
	 * >0 : la fonction apparat comme une fonction normale
	 */

	logical1					lancement_interactif;

	unsigned char				*chemin_fichiers_temporaires;
	unsigned char				langue;
	unsigned char				retour_routine_evaluation;
	unsigned char				*localisation;

	unsigned long				niveau_recursivite;

	pid_t						pid_processus_pere;
	pthread_t					tid_processus_pere;

	logical1					processus_detache;

/* Profilage											*/

	logical1					profilage;
	int							niveau_profilage;
	struct_liste_profilage		*pile_profilage;
	struct_liste_profilage2		*pile_profilage_fonctions;
	struct timeval				horodatage_profilage;

	// Mutex spcifique au processus et donnant accs  la pile des processus
	pthread_mutex_t				mutex;
	pthread_mutex_t				mutex_allocation;

	// Mutex autorisant les fork()
	sem_t
#	ifdef SEMAPHORES_NOMMES
		*
#	endif
		 						semaphore_fork;

/* Mutexes											*/

	struct_liste_chainee		*liste_mutexes;
	pthread_mutex_t				protection_liste_mutexes;

/* Gestion optimise de la mmoire					*/

	double						estimation_taille_pile_tampon;
	unsigned long				taille_pile_tampon;
	struct_liste_chainee		*pile_tampon;

	double						estimation_taille_pile_systeme_tampon;
	unsigned long				taille_pile_systeme_tampon;
	struct_liste_pile_systeme	*pile_systeme_tampon;

	unsigned long				taille_pile_objets;
	struct_objet				*pile_objets;

#	define TAILLE_CACHE 16384

	unsigned long				*objets_adr[TAILLE_CACHE];
	int							pointeur_adr;

	logical8					*objets_bin[TAILLE_CACHE];
	int							pointeur_bin;

	complex16					*objets_cpl[TAILLE_CACHE];
	int							pointeur_cpl;

	struct fonction				*objets_fct[TAILLE_CACHE];
	int							pointeur_fct;

	integer8					*objets_int[TAILLE_CACHE];
	int							pointeur_int;

	struct_matrice				*objets_mat[TAILLE_CACHE];
	int							pointeur_mat;

	struct_nom					*objets_nom[TAILLE_CACHE];
	int							pointeur_nom;

	real8						*objets_rel[TAILLE_CACHE];
	int							pointeur_rel;

	struct_tableau				*objets_tab[TAILLE_CACHE];
	int							pointeur_tab;

	struct_vecteur				*objets_vec[TAILLE_CACHE];
	int							pointeur_vec;

	struct_liste_chainee		*maillons[TAILLE_CACHE];
	int							pointeur_maillons;

	struct_arbre_variables		*variables_noeud[TAILLE_CACHE];
	int							pointeur_variables_noeud;

	struct_liste_variables		*variables_feuille[TAILLE_CACHE];
	int							pointeur_variables_feuille;

	struct_variable				*variables_variable[TAILLE_CACHE];
	int							pointeur_variables_variable;

	struct_arbre_variables		**variables_tableau_noeuds[TAILLE_CACHE];
	int							pointeur_variables_tableau_noeuds;
} struct_processus;
#endif

/*
--------------------------------------------------------------------------------
  Structure fonction
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
typedef struct fonction
{
	unsigned char						*nom_fonction;
	unsigned long						nombre_arguments;

	void								(*fonction)(struct_processus *);

	volatile struct_liste_chainee		*prediction_saut;
	volatile logical1					prediction_execution;
} struct_fonction;
#endif


/*
================================================================================
  PROTOTYPES
================================================================================
*/

/*
--------------------------------------------------------------------------------
  Procdures d'excution des mots-clef du langage RPL/2
--------------------------------------------------------------------------------
*/

// BEGIN C PROTOTYPES
void instruction_sensible_e(struct_processus *s_etat_processus);
void instruction_sensible_i(struct_processus *s_etat_processus);
void instruction_sensible_infinity(struct_processus *s_etat_processus);

void instruction_abort(struct_processus *s_etat_processus);
void instruction_abs(struct_processus *s_etat_processus);
void instruction_acos(struct_processus *s_etat_processus);
void instruction_acosh(struct_processus *s_etat_processus);
void instruction_alarm(struct_processus *s_etat_processus);
void instruction_alog(struct_processus *s_etat_processus);
void instruction_and(struct_processus *s_etat_processus);
void instruction_append(struct_processus *s_etat_processus);
void instruction_array_fleche(struct_processus *s_etat_processus);
void instruction_arg(struct_processus *s_etat_processus);
void instruction_asin(struct_processus *s_etat_processus);
void instruction_asinh(struct_processus *s_etat_processus);
void instruction_asl(struct_processus *s_etat_processus);
void instruction_asr(struct_processus *s_etat_processus);
void instruction_atan(struct_processus *s_etat_processus);
void instruction_atanh(struct_processus *s_etat_processus);
void instruction_atexit(struct_processus *s_etat_processus);
void instruction_atpoke(struct_processus *s_etat_processus);
void instruction_autoscale(struct_processus *s_etat_processus);
void instruction_axes(struct_processus *s_etat_processus);

void instruction_b_vers_r(struct_processus *s_etat_processus);
void instruction_backspace(struct_processus *s_etat_processus);
void instruction_backtrace(struct_processus *s_etat_processus);
void instruction_beep(struct_processus *s_etat_processus);
void instruction_bessel(struct_processus *s_etat_processus);
void instruction_bin(struct_processus *s_etat_processus);

void instruction_c_vers_r(struct_processus *s_etat_processus);
void instruction_case(struct_processus *s_etat_processus);
void instruction_ceil(struct_processus *s_etat_processus);
void instruction_centr(struct_processus *s_etat_processus);
void instruction_cf(struct_processus *s_etat_processus);
void instruction_chr(struct_processus *s_etat_processus);
void instruction_clear(struct_processus *s_etat_processus);
void instruction_cllcd(struct_processus *s_etat_processus);
void instruction_clmf(struct_processus *s_etat_processus);
void instruction_close(struct_processus *s_etat_processus);
void instruction_clratexit(struct_processus *s_etat_processus);
void instruction_clratpoke(struct_processus *s_etat_processus);
void instruction_clrcntxt(struct_processus *s_etat_processus);
void instruction_clrerr(struct_processus *s_etat_processus);
void instruction_clrfuse(struct_processus *s_etat_processus);
void instruction_clrmtx(struct_processus *s_etat_processus);
void instruction_clrsmphr(struct_processus *s_etat_processus);
void instruction_clrswi(struct_processus *s_etat_processus);
void instruction_cls(struct_processus *s_etat_processus);
void instruction_clusr(struct_processus *s_etat_processus);
void instruction_cnrm(struct_processus *s_etat_processus);
void instruction_cols(struct_processus *s_etat_processus);
void instruction_col_fleche(struct_processus *s_etat_processus);
void instruction_col_moins(struct_processus *s_etat_processus);
void instruction_col_plus(struct_processus *s_etat_processus);
void instruction_comb(struct_processus *s_etat_processus);
void instruction_con(struct_processus *s_etat_processus);
void instruction_cond(struct_processus *s_etat_processus);
void instruction_cont(struct_processus *s_etat_processus);
void instruction_continue(struct_processus *s_etat_processus);
void instruction_conj(struct_processus *s_etat_processus);
void instruction_convert(struct_processus *s_etat_processus);
void instruction_copy(struct_processus *s_etat_processus);
void instruction_copyright(struct_processus *s_etat_processus);
void instruction_corr(struct_processus *s_etat_processus);
void instruction_cos(struct_processus *s_etat_processus);
void instruction_cosh(struct_processus *s_etat_processus);
void instruction_cov(struct_processus *s_etat_processus);
void instruction_cr(struct_processus *s_etat_processus);
void instruction_create(struct_processus *s_etat_processus);
void instruction_crmtx(struct_processus *s_etat_processus);
void instruction_cross(struct_processus *s_etat_processus);
void instruction_crsmphr(struct_processus *s_etat_processus);
void instruction_crtab(struct_processus *s_etat_processus);
void instruction_cstop(struct_processus *s_etat_processus);
void instruction_cswp(struct_processus *s_etat_processus);
void instruction_currenc(struct_processus *s_etat_processus);
void instruction_cycle(struct_processus *s_etat_processus);

void instruction_d_vers_r(struct_processus *s_etat_processus);
void instruction_daemonize(struct_processus *s_etat_processus);
void instruction_date(struct_processus *s_etat_processus);
void instruction_dec(struct_processus *s_etat_processus);
void instruction_decr(struct_processus *s_etat_processus);
void instruction_default(struct_processus *s_etat_processus);
void instruction_deg(struct_processus *s_etat_processus);
void instruction_delete(struct_processus *s_etat_processus);
void instruction_depnd(struct_processus *s_etat_processus);
void instruction_depth(struct_processus *s_etat_processus);
void instruction_der(struct_processus *s_etat_processus);
void instruction_det(struct_processus *s_etat_processus);
void instruction_detach(struct_processus *s_etat_processus);
void instruction_dft(struct_processus *s_etat_processus);
void instruction_dgtiz(struct_processus *s_etat_processus);
void instruction_diag_fleche(struct_processus *s_etat_processus);
void instruction_disp(struct_processus *s_etat_processus);
void instruction_division(struct_processus *s_etat_processus);
void instruction_do(struct_processus *s_etat_processus);
void instruction_dot(struct_processus *s_etat_processus);
void instruction_draw(struct_processus *s_etat_processus);
void instruction_drax(struct_processus *s_etat_processus);
void instruction_drop(struct_processus *s_etat_processus);
void instruction_drop2(struct_processus *s_etat_processus);
void instruction_dropcntxt(struct_processus *s_etat_processus);
void instruction_dropn(struct_processus *s_etat_processus);
void instruction_drws(struct_processus *s_etat_processus);
void instruction_dup(struct_processus *s_etat_processus);
void instruction_dup2(struct_processus *s_etat_processus);
void instruction_dupcntxt(struct_processus *s_etat_processus);
void instruction_dupn(struct_processus *s_etat_processus);

void instruction_e(struct_processus *s_etat_processus);
void instruction_edit(struct_processus *s_etat_processus);
void instruction_egalite(struct_processus *s_etat_processus);
void instruction_egv(struct_processus *s_etat_processus);
void instruction_egvl(struct_processus *s_etat_processus);
void instruction_else(struct_processus *s_etat_processus);
void instruction_elseif(struct_processus *s_etat_processus);
void instruction_end(struct_processus *s_etat_processus);
void instruction_eng(struct_processus *s_etat_processus);
void instruction_epsilon(struct_processus *s_etat_processus);
void instruction_erase(struct_processus *s_etat_processus);
void instruction_errm(struct_processus *s_etat_processus);
void instruction_errn(struct_processus *s_etat_processus);
void instruction_eval(struct_processus *s_etat_processus);
void instruction_exget(struct_processus *s_etat_processus);
void instruction_exit(struct_processus *s_etat_processus);
void instruction_exp(struct_processus *s_etat_processus);
void instruction_expm(struct_processus *s_etat_processus);
void instruction_exsub(struct_processus *s_etat_processus);
void instruction_externals(struct_processus *s_etat_processus);
void instruction_eyept(struct_processus *s_etat_processus);

void instruction_fact(struct_processus *s_etat_processus);
void instruction_false(struct_processus *s_etat_processus);
void instruction_fc_test(struct_processus *s_etat_processus);
void instruction_fc_test_c(struct_processus *s_etat_processus);
void instruction_fc_test_s(struct_processus *s_etat_processus);
void instruction_fft(struct_processus *s_etat_processus);
void instruction_fix(struct_processus *s_etat_processus);
void instruction_fleche(struct_processus *s_etat_processus);
void instruction_fleche_array(struct_processus *s_etat_processus);
void instruction_fleche_col(struct_processus *s_etat_processus);
void instruction_fleche_diag(struct_processus *s_etat_processus);
void instruction_fleche_hms(struct_processus *s_etat_processus);
void instruction_fleche_lcd(struct_processus *s_etat_processus);
void instruction_fleche_list(struct_processus *s_etat_processus);
void instruction_fleche_num(struct_processus *s_etat_processus);
void instruction_fleche_q(struct_processus *s_etat_processus);
void instruction_fleche_row(struct_processus *s_etat_processus);
void instruction_fleche_str(struct_processus *s_etat_processus);
void instruction_fleche_table(struct_processus *s_etat_processus);
void instruction_floor(struct_processus *s_etat_processus);
void instruction_for(struct_processus *s_etat_processus);
void instruction_format(struct_processus *s_etat_processus);
void instruction_fp(struct_processus *s_etat_processus);
void instruction_fs_test(struct_processus *s_etat_processus);
void instruction_fs_test_c(struct_processus *s_etat_processus);
void instruction_fs_test_s(struct_processus *s_etat_processus);
void instruction_function(struct_processus *s_etat_processus);
void instruction_fuse(struct_processus *s_etat_processus);

void instruction_gamma(struct_processus *s_etat_processus);
void instruction_ge(struct_processus *s_etat_processus);
void instruction_gegv(struct_processus *s_etat_processus);
void instruction_gegvl(struct_processus *s_etat_processus);
void instruction_get(struct_processus *s_etat_processus);
void instruction_getc(struct_processus *s_etat_processus);
void instruction_geti(struct_processus *s_etat_processus);
void instruction_getr(struct_processus *s_etat_processus);
void instruction_glegv(struct_processus *s_etat_processus);
void instruction_gregv(struct_processus *s_etat_processus);
void instruction_gt(struct_processus *s_etat_processus);

void instruction_halt(struct_processus *s_etat_processus);
void instruction_head(struct_processus *s_etat_processus);
void instruction_help(struct_processus *s_etat_processus);
void instruction_hex(struct_processus *s_etat_processus);
void instruction_histogram(struct_processus *s_etat_processus);
void instruction_hms_fleche(struct_processus *s_etat_processus);
void instruction_hms_moins(struct_processus *s_etat_processus);
void instruction_hms_plus(struct_processus *s_etat_processus);

void instruction_i(struct_processus *s_etat_processus);
void instruction_idft(struct_processus *s_etat_processus);
void instruction_idn(struct_processus *s_etat_processus);
void instruction_if(struct_processus *s_etat_processus);
void instruction_iferr(struct_processus *s_etat_processus);
void instruction_ifft(struct_processus *s_etat_processus);
void instruction_ift(struct_processus *s_etat_processus);
void instruction_ifte(struct_processus *s_etat_processus);
void instruction_im(struct_processus *s_etat_processus);
void instruction_implicit(struct_processus *s_etat_processus);
void instruction_in(struct_processus *s_etat_processus);
void instruction_incr(struct_processus *s_etat_processus);
void instruction_indep(struct_processus *s_etat_processus);
void instruction_input(struct_processus *s_etat_processus);
void instruction_infinity(struct_processus *s_etat_processus);
void instruction_inquire(struct_processus *s_etat_processus);
void instruction_int(struct_processus *s_etat_processus);
void instruction_interrupt(struct_processus *s_etat_processus);
void instruction_inv(struct_processus *s_etat_processus);
void instruction_ip(struct_processus *s_etat_processus);
void instruction_iswi(struct_processus *s_etat_processus);
void instruction_itrace(struct_processus *s_etat_processus);

void instruction_jdate(struct_processus *s_etat_processus);

void instruction_key(struct_processus *s_etat_processus);
void instruction_keylabel(struct_processus *s_etat_processus);
void instruction_keytitle(struct_processus *s_etat_processus);
void instruction_kill(struct_processus *s_etat_processus);
void instruction_kind(struct_processus *s_etat_processus);

void instruction_label(struct_processus *s_etat_processus);
void instruction_last(struct_processus *s_etat_processus);
void instruction_lcase(struct_processus *s_etat_processus);
void instruction_lcd_fleche(struct_processus *s_etat_processus);
void instruction_lchol(struct_processus *s_etat_processus);
void instruction_le(struct_processus *s_etat_processus);
void instruction_legv(struct_processus *s_etat_processus);
void instruction_limit(struct_processus *s_etat_processus);
void instruction_line(struct_processus *s_etat_processus);
void instruction_list_fleche(struct_processus *s_etat_processus);
void instruction_ln(struct_processus *s_etat_processus);
void instruction_lnp1(struct_processus *s_etat_processus);
void instruction_localization(struct_processus *s_etat_processus);
void instruction_lock(struct_processus *s_etat_processus);
void instruction_log(struct_processus *s_etat_processus);
void instruction_logger(struct_processus *s_etat_processus);
void instruction_logscale(struct_processus *s_etat_processus);
void instruction_lq(struct_processus *s_etat_processus);
void instruction_lsq(struct_processus *s_etat_processus);
void instruction_lt(struct_processus *s_etat_processus);
void instruction_lu(struct_processus *s_etat_processus);
void instruction_l_vers_t(struct_processus *s_etat_processus);

void instruction_mant(struct_processus *s_etat_processus);
void instruction_mark(struct_processus *s_etat_processus);
void instruction_max(struct_processus *s_etat_processus);
void instruction_maxs(struct_processus *s_etat_processus);
void instruction_mclrin(struct_processus *s_etat_processus);
void instruction_mean(struct_processus *s_etat_processus);
void instruction_mem(struct_processus *s_etat_processus);
void instruction_memlock(struct_processus *s_etat_processus);
void instruction_memunlock(struct_processus *s_etat_processus);
void instruction_min(struct_processus *s_etat_processus);
void instruction_mins(struct_processus *s_etat_processus);
void instruction_mod(struct_processus *s_etat_processus);
void instruction_moins(struct_processus *s_etat_processus);
void instruction_mtxlock(struct_processus *s_etat_processus);
void instruction_mtxstatus(struct_processus *s_etat_processus);
void instruction_mtxtrylock(struct_processus *s_etat_processus);
void instruction_mtxunlock(struct_processus *s_etat_processus);
void instruction_multiplication(struct_processus *s_etat_processus);

void instruction_ne(struct_processus *s_etat_processus);
void instruction_neg(struct_processus *s_etat_processus);
void instruction_next(struct_processus *s_etat_processus);
void instruction_newplane(struct_processus *s_etat_processus);
void instruction_not(struct_processus *s_etat_processus);
void instruction_nrand(struct_processus *s_etat_processus);
void instruction_nrproc(struct_processus *s_etat_processus);
void instruction_ns(struct_processus *s_etat_processus);
void instruction_num(struct_processus *s_etat_processus);

void instruction_obget(struct_processus *s_etat_processus);
void instruction_obsub(struct_processus *s_etat_processus);
void instruction_oct(struct_processus *s_etat_processus);
void instruction_open(struct_processus *s_etat_processus);
void instruction_or(struct_processus *s_etat_processus);
void instruction_over(struct_processus *s_etat_processus);

void instruction_p_vers_r(struct_processus *s_etat_processus);
void instruction_paper(struct_processus *s_etat_processus);
void instruction_parameter(struct_processus *s_etat_processus);
void instruction_parametric(struct_processus *s_etat_processus);
void instruction_pcov(struct_processus *s_etat_processus);
void instruction_peek(struct_processus *s_etat_processus);
void instruction_perm(struct_processus *s_etat_processus);
void instruction_persist(struct_processus *s_etat_processus);
void instruction_pi(struct_processus *s_etat_processus);
void instruction_pick(struct_processus *s_etat_processus);
void instruction_plot(struct_processus *s_etat_processus);
void instruction_plotter(struct_processus *s_etat_processus);
void instruction_plus(struct_processus *s_etat_processus);
void instruction_pmax(struct_processus *s_etat_processus);
void instruction_pmin(struct_processus *s_etat_processus);
void instruction_poke(struct_processus *s_etat_processus);
void instruction_polar(struct_processus *s_etat_processus);
void instruction_pos(struct_processus *s_etat_processus);
void instruction_pourcent(struct_processus *s_etat_processus);
void instruction_pourcent_ch(struct_processus *s_etat_processus);
void instruction_pourcent_t(struct_processus *s_etat_processus);
void instruction_ppar(struct_processus *s_etat_processus);
void instruction_pr1(struct_processus *s_etat_processus);
void instruction_print(struct_processus *s_etat_processus);
void instruction_private(struct_processus *s_etat_processus);
void instruction_prlcd(struct_processus *s_etat_processus);
void instruction_prmd(struct_processus *s_etat_processus);
void instruction_procid(struct_processus *s_etat_processus);
void instruction_prompt(struct_processus *s_etat_processus);
void instruction_protect(struct_processus *s_etat_processus);
void instruction_prst(struct_processus *s_etat_processus);
void instruction_prstc(struct_processus *s_etat_processus);
void instruction_prusr(struct_processus *s_etat_processus);
void instruction_prvar(struct_processus *s_etat_processus);
void instruction_psdev(struct_processus *s_etat_processus);
void instruction_puissance(struct_processus *s_etat_processus);
void instruction_purge(struct_processus *s_etat_processus);
void instruction_pshcntxt(struct_processus *s_etat_processus);
void instruction_pshprfl(struct_processus *s_etat_processus);
void instruction_pulcntxt(struct_processus *s_etat_processus);
void instruction_pulprfl(struct_processus *s_etat_processus);
void instruction_put(struct_processus *s_etat_processus);
void instruction_putc(struct_processus *s_etat_processus);
void instruction_puti(struct_processus *s_etat_processus);
void instruction_putr(struct_processus *s_etat_processus);
void instruction_pvar(struct_processus *s_etat_processus);

void instruction_qr(struct_processus *s_etat_processus);

void instruction_r_vers_b(struct_processus *s_etat_processus);
void instruction_r_vers_c(struct_processus *s_etat_processus);
void instruction_r_vers_d(struct_processus *s_etat_processus);
void instruction_r_vers_p(struct_processus *s_etat_processus);
void instruction_rad(struct_processus *s_etat_processus);
void instruction_rand(struct_processus *s_etat_processus);
void instruction_rank(struct_processus *s_etat_processus);
void instruction_rceq(struct_processus *s_etat_processus);
void instruction_rci(struct_processus *s_etat_processus);
void instruction_rcij(struct_processus *s_etat_processus);
void instruction_rcl(struct_processus *s_etat_processus);
void instruction_rclf(struct_processus *s_etat_processus);
void instruction_rcls(struct_processus *s_etat_processus);
void instruction_rclswi(struct_processus *s_etat_processus);
void instruction_rcws(struct_processus *s_etat_processus);
void instruction_rdate(struct_processus *s_etat_processus);
void instruction_rdm(struct_processus *s_etat_processus);
void instruction_rdgn(struct_processus *s_etat_processus);
void instruction_rdz(struct_processus *s_etat_processus);
void instruction_re(struct_processus *s_etat_processus);
void instruction_read(struct_processus *s_etat_processus);
void instruction_recode(struct_processus *s_etat_processus);
void instruction_recv(struct_processus *s_etat_processus);
void instruction_redraw(struct_processus *s_etat_processus);
void instruction_relax(struct_processus *s_etat_processus);
void instruction_remove(struct_processus *s_etat_processus);
void instruction_res(struct_processus *s_etat_processus);
void instruction_recall(struct_processus *s_etat_processus);
void instruction_regv(struct_processus *s_etat_processus);
void instruction_repeat(struct_processus *s_etat_processus);
void instruction_repl(struct_processus *s_etat_processus);
void instruction_return(struct_processus *s_etat_processus);
void instruction_revlist(struct_processus *s_etat_processus);
void instruction_rewind(struct_processus *s_etat_processus);
void instruction_rfuse(struct_processus *s_etat_processus);
void instruction_rl(struct_processus *s_etat_processus);
void instruction_rlb(struct_processus *s_etat_processus);
void instruction_rnd(struct_processus *s_etat_processus);
void instruction_rnrm(struct_processus *s_etat_processus);
void instruction_roll(struct_processus *s_etat_processus);
void instruction_rolld(struct_processus *s_etat_processus);
void instruction_rot(struct_processus *s_etat_processus);
void instruction_row_fleche(struct_processus *s_etat_processus);
void instruction_row_moins(struct_processus *s_etat_processus);
void instruction_row_plus(struct_processus *s_etat_processus);
void instruction_rr(struct_processus *s_etat_processus);
void instruction_rrb(struct_processus *s_etat_processus);
void instruction_rsd(struct_processus *s_etat_processus);
void instruction_rstop(struct_processus *s_etat_processus);
void instruction_rswp(struct_processus *s_etat_processus);

void instruction_s_moins(struct_processus *s_etat_processus);
void instruction_s_plus(struct_processus *s_etat_processus);
void instruction_same(struct_processus *s_etat_processus);
void instruction_save(struct_processus *s_etat_processus);
void instruction_scatter(struct_processus *s_etat_processus);
void instruction_scale(struct_processus *s_etat_processus);
void instruction_sched(struct_processus *s_etat_processus);
void instruction_schur(struct_processus *s_etat_processus);
void instruction_sci(struct_processus *s_etat_processus);
void instruction_scls(struct_processus *s_etat_processus);
void instruction_sconj(struct_processus *s_etat_processus);
void instruction_sdev(struct_processus *s_etat_processus);
void instruction_select(struct_processus *s_etat_processus);
void instruction_send(struct_processus *s_etat_processus);
void instruction_sf(struct_processus *s_etat_processus);
void instruction_shared(struct_processus *s_etat_processus);
void instruction_sign(struct_processus *s_etat_processus);
void instruction_sin(struct_processus *s_etat_processus);
void instruction_sinh(struct_processus *s_etat_processus);
void instruction_sinv(struct_processus *s_etat_processus);
void instruction_size(struct_processus *s_etat_processus);
void instruction_sl(struct_processus *s_etat_processus);
void instruction_slb(struct_processus *s_etat_processus);
void instruction_slice(struct_processus *s_etat_processus);
void instruction_slicescale(struct_processus *s_etat_processus);
void instruction_smphrdecr(struct_processus *s_etat_processus);
void instruction_smphrgetv(struct_processus *s_etat_processus);
void instruction_smphrincr(struct_processus *s_etat_processus);
void instruction_smphrtrydecr(struct_processus *s_etat_processus);
void instruction_sneg(struct_processus *s_etat_processus);
void instruction_sort(struct_processus *s_etat_processus);
void instruction_spar(struct_processus *s_etat_processus);
void instruction_spawn(struct_processus *s_etat_processus);
void instruction_splash(struct_processus *s_etat_processus);
void instruction_sq(struct_processus *s_etat_processus);
void instruction_sqlconnect(struct_processus *s_etat_processus);
void instruction_sqldisconnect(struct_processus *s_etat_processus);
void instruction_sqlquery(struct_processus *s_etat_processus);
void instruction_sqrt(struct_processus *s_etat_processus);
void instruction_sr(struct_processus *s_etat_processus);
void instruction_srb(struct_processus *s_etat_processus);
void instruction_srev(struct_processus *s_etat_processus);
void instruction_sst(struct_processus *s_etat_processus);
void instruction_star_d(struct_processus *s_etat_processus);
void instruction_star_h(struct_processus *s_etat_processus);
void instruction_star_s(struct_processus *s_etat_processus);
void instruction_star_w(struct_processus *s_etat_processus);
void instruction_start(struct_processus *s_etat_processus);
void instruction_static(struct_processus *s_etat_processus);
void instruction_std(struct_processus *s_etat_processus);
void instruction_step(struct_processus *s_etat_processus);
void instruction_steq(struct_processus *s_etat_processus);
void instruction_sto(struct_processus *s_etat_processus);
void instruction_sto_division(struct_processus *s_etat_processus);
void instruction_sto_fois(struct_processus *s_etat_processus);
void instruction_sto_moins(struct_processus *s_etat_processus);
void instruction_sto_plus(struct_processus *s_etat_processus);
void instruction_stof(struct_processus *s_etat_processus);
void instruction_stop(struct_processus *s_etat_processus);
void instruction_store(struct_processus *s_etat_processus);
void instruction_stos(struct_processus *s_etat_processus);
void instruction_stoswi(struct_processus *s_etat_processus);
void instruction_str_fleche(struct_processus *s_etat_processus);
void instruction_stws(struct_processus *s_etat_processus);
void instruction_sub(struct_processus *s_etat_processus);
void instruction_suspend(struct_processus *s_etat_processus);
void instruction_svd(struct_processus *s_etat_processus);
void instruction_svl(struct_processus *s_etat_processus);
void instruction_swap(struct_processus *s_etat_processus);
void instruction_swi(struct_processus *s_etat_processus);
void instruction_swilock(struct_processus *s_etat_processus);
void instruction_swiqueue(struct_processus *s_etat_processus);
void instruction_swistatus(struct_processus *s_etat_processus);
void instruction_swiunlock(struct_processus *s_etat_processus);
void instruction_swapcntxt(struct_processus *s_etat_processus);
void instruction_sx(struct_processus *s_etat_processus);
void instruction_sx2(struct_processus *s_etat_processus);
void instruction_sxy(struct_processus *s_etat_processus);
void instruction_sy(struct_processus *s_etat_processus);
void instruction_sy2(struct_processus *s_etat_processus);
void instruction_sync(struct_processus *s_etat_processus);
void instruction_syseval(struct_processus *s_etat_processus);

void instruction_table_fleche(struct_processus *s_etat_processus);
void instruction_tail(struct_processus *s_etat_processus);
void instruction_tan(struct_processus *s_etat_processus);
void instruction_tanh(struct_processus *s_etat_processus);
void instruction_target(struct_processus *s_etat_processus);
void instruction_taylr(struct_processus *s_etat_processus);
void instruction_then(struct_processus *s_etat_processus);
void instruction_time(struct_processus *s_etat_processus);
void instruction_title(struct_processus *s_etat_processus);
void instruction_tokenize(struct_processus *s_etat_processus);
void instruction_tot(struct_processus *s_etat_processus);
void instruction_trim(struct_processus *s_etat_processus);
void instruction_trn(struct_processus *s_etat_processus);
void instruction_trnc(struct_processus *s_etat_processus);
void instruction_true(struct_processus *s_etat_processus);
void instruction_type(struct_processus *s_etat_processus);
void instruction_t_vers_l(struct_processus *s_etat_processus);

void instruction_ucase(struct_processus *s_etat_processus);
void instruction_uchol(struct_processus *s_etat_processus);
void instruction_unlock(struct_processus *s_etat_processus);
void instruction_unprotect(struct_processus *s_etat_processus);
void instruction_until(struct_processus *s_etat_processus);
void instruction_use(struct_processus *s_etat_processus);
void instruction_utpc(struct_processus *s_etat_processus);
void instruction_utpf(struct_processus *s_etat_processus);
void instruction_utpn(struct_processus *s_etat_processus);
void instruction_utpt(struct_processus *s_etat_processus);

void instruction_var(struct_processus *s_etat_processus);
void instruction_variable(struct_processus *s_etat_processus);
void instruction_vars(struct_processus *s_etat_processus);
void instruction_verify(struct_processus *s_etat_processus);
void instruction_vers_niveau_inferieur(struct_processus *s_etat_processus);
void instruction_vers_niveau_superieur(struct_processus *s_etat_processus);
void instruction_version(struct_processus *s_etat_processus);
void instruction_visit(struct_processus *s_etat_processus);
void instruction_volatile(struct_processus *s_etat_processus);

void instruction_wait(struct_processus *s_etat_processus);
void instruction_warranty(struct_processus *s_etat_processus);
void instruction_wfdata(struct_processus *s_etat_processus);
void instruction_wfack(struct_processus *s_etat_processus);
void instruction_wflock(struct_processus *s_etat_processus);
void instruction_wfpoke(struct_processus *s_etat_processus);
void instruction_wfproc(struct_processus *s_etat_processus);
void instruction_wfsock(struct_processus *s_etat_processus);
void instruction_wfswi(struct_processus *s_etat_processus);
void instruction_while(struct_processus *s_etat_processus);
void instruction_wireframe(struct_processus *s_etat_processus);
void instruction_workdir(struct_processus *s_etat_processus);
void instruction_write(struct_processus *s_etat_processus);

void instruction_xcol(struct_processus *s_etat_processus);
void instruction_xor(struct_processus *s_etat_processus);
void instruction_xpon(struct_processus *s_etat_processus);
void instruction_xroot(struct_processus *s_etat_processus);

void instruction_ycol(struct_processus *s_etat_processus);
void instruction_yield(struct_processus *s_etat_processus);

/*
--------------------------------------------------------------------------------
  Procdures
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
void affectation_interruptions_logicielles(struct_processus *s_etat_processus);
void affichage_pile(struct_processus *s_etat_processus,
		struct_liste_chainee *l_element_courant, unsigned long
		niveau_courant);
#endif

void *allocation_maillon(struct_processus *s_etat_processus);

#ifndef RPLARGS
void analyse(struct_processus *s_etat_processus, void (*fonction)());
void *analyse_instruction(struct_processus *s_etat_processus,
		unsigned char *candidat);
void appel_gnuplot(struct_processus *s_etat_processus,
		unsigned char persistance);
void cf(struct_processus *s_etat_processus, unsigned char indice_drapeau);
void cond(struct_processus *s_etat_processus,
		struct_matrice *s_matrice, real8 *condition);
void conversion_decimal_vers_hms(real8 *angle);
void conversion_degres_vers_radians(real8 *angle);
void conversion_element_tex(unsigned char **element, unsigned char *fonction);
void conversion_format(struct_processus *s_etat_processus,
		unsigned char *chaine);
void conversion_hms_vers_decimal(real8 *angle);
void conversion_majuscule_limitee(unsigned char *chaine_entree, 
		unsigned char *chaine_sortie, unsigned long longueur);
void conversion_radians_vers_degres(real8 *angle);
void copie_arbre_variables(struct_processus *s_etat_processus,
		struct_processus *s_nouvel_etat_processus);
void correction_formateur_tex(struct_processus *s_etat_processus,
		unsigned char **ligne);
void depilement_pile_systeme(struct_processus *s_etat_processus);
void derivation(struct_processus *s_etat_processus,
		struct_objet **s_expression);
void determinant(struct_processus *s_etat_processus, struct_matrice *s_matrice,
		void *valeur);
void deverrouillage_threads_concurrents(struct_processus *s_etat_processus);
void ecriture_pile(struct_processus *s_etat_processus, file *flux,
		struct_liste_chainee *l_element_courant, unsigned long niveau_courant);
void ecriture_profil(struct_processus *s_etat_processus);
void effacement_pile_systeme(struct_processus *s_etat_processus);
void empilement_pile_systeme(struct_processus *s_etat_processus);
void encart(struct_processus *s_etat_processus, unsigned long duree);
void evaluation_romberg(struct_processus *s_etat_processus,
		struct_objet *s_expression, unsigned char *variable, real8 *point,
		real8 *valeur, logical1 *validite);
void factorisation_cholesky(struct_processus *s_etat_processus,
		struct_matrice *s_matrice, unsigned char mode);
void factorisation_lq(struct_processus *s_etat_processus,
		struct_matrice *s_matrice, void **tau);
void factorisation_lu(struct_processus *s_etat_processus,
		struct_matrice *s_matrice, struct_matrice **s_permutation);
void factorisation_qr(struct_processus *s_etat_processus,
		struct_matrice *s_matrice, void **tau);
void factorisation_schur(struct_processus *s_etat_processus,
		struct_matrice *s_matrice, struct_matrice **s_schur);
void *fusible(void *argument);
void formateur_tex(struct_processus *s_etat_processus, struct_objet *s_objet,
		unsigned char environnement);
void formateur_elementaire_tex(struct_processus *s_etat_processus,
		file *fichier, struct_objet *s_objet, unsigned char environnement);
void impression_pile(struct_processus *s_etat_processus,
			struct_liste_chainee *l_element_courant,
			unsigned char methode, unsigned long niveau_courant);
void impression_tex(struct_processus *s_etat_processus);
void informations(struct_processus *s_etat_processus);
void initialisation_allocateur(struct_processus *s_etat_processus);
void initialisation_completion(void);
void initialisation_contexte_cas(struct_processus *s_etat_processus);
void initialisation_drapeaux(struct_processus *s_etat_processus);
void initialisation_generateur_aleatoire(struct_processus *s_etat_processus,
		logical1 initialisation_automatique, unsigned long int racine);
void initialisation_instructions(struct_processus *s_etat_processus);
void initialisation_variables(struct_processus *s_etat_processus);
#endif

void initialisation_objet(struct_objet *s_objet);

#ifndef RPLARGS
void insertion_thread(struct_processus *s_etat_processus,
		logical1 thread_principal);
void insertion_thread_surveillance(struct_processus *s_etat_processus,
		struct_descripteur_thread *s_argument_thread);
void integrale_romberg(struct_processus *s_etat_processus,
		struct_objet *s_expression, unsigned char *variable,
		real8 a, real8 b, real8 precision);
void interface_cas(struct_processus *s_etat_processus,
		enum t_rplcas_commandes commande);
void interruption1(int signal);
void interruption2(int signal);
void interruption3(int signal);
void interruption4(int signal);
void interruption5(int signal);
void interruption_depassement_pile(int urgence, stackoverflow_context_t scp);
void inversion_matrice(struct_processus *s_etat_processus,
		struct_matrice *s_matrice);
void lancement_daemon(struct_processus *s_etat_processus);
void *lancement_thread(void *argument);
#endif

void liberation(struct_processus *s_etat_processus, struct_objet *s_objet);
void liberation_maillon(struct_processus *s_etat_processus,
		struct_liste_chainee *maillon);

#ifndef RPLARGS
void liberation_allocateur(struct_processus *s_etat_processus);
void liberation_arbre_instructions(struct_processus *s_etat_processus,
		struct_instruction *arbre);
void liberation_arbre_variables(struct_processus *s_etat_processus,
		struct_arbre_variables *arbre, logical1 definitions);
void liberation_contexte_cas(struct_processus *s_etat_processus);
void liberation_generateur_aleatoire(struct_processus *s_etat_processus);
void liberation_threads(struct_processus *s_etat_processus);
void liberation_profil(struct_processus *s_etat_processus);
void localisation_courante(struct_processus *s_etat_processus);
void modification_pid_thread_pere(struct_processus *s_etat_processus);
void moindres_carres(struct_processus *s_etat_processus,
		struct_matrice *s_matrice_a, struct_matrice *s_matrice_b,
		struct_matrice *s_matrice_x);
void profilage(struct_processus *s_etat_processus, unsigned char *fonction);
void rang(struct_processus *s_etat_processus, struct_matrice *s_matrice,
		integer8 *valeur);
void recherche_type(struct_processus *s_etat_processus);
void retrait_thread(struct_processus *s_etat_processus);
void retrait_thread_surveillance(struct_processus *s_etat_processus,
		struct_descripteur_thread *s_argument_thread);
void rplcore(struct_processus *s_etat_processus);
#endif

void scrutation_injection(struct_processus *s_etat_processus);

#ifndef RPLARGS
void scrutation_interruptions(struct_processus *s_etat_processus);
void sf(struct_processus *s_etat_processus, unsigned char indice_drapeau);
void *surveillance_processus(void *argument);
void swap(void *variable_1, void *variable_2, unsigned long taille_octets);
void trace(struct_processus *s_etat_processus, FILE *flux);
void traitement_asynchrone_exceptions_gsl(struct_processus *s_etat_processus);
void traitement_exceptions_gsl(const char *reason, const char *file,
		int line, int gsl_errno);
void traitement_interruptions_logicielles(struct_processus *s_etat_processus);
void tri_base_symboles_externes(struct_processus *s_etat_processus);
void tri_vecteur(real8 *vecteur, unsigned long taille);
void valeurs_propres(struct_processus *s_etat_processus,
		struct_matrice *s_matrice,
		struct_vecteur *s_valeurs_propres,
		struct_matrice *s_vecteurs_propres_gauches,
		struct_matrice *s_vecteur_propres_droits);
void valeurs_propres_generalisees(struct_processus *s_etat_processus,
		struct_matrice *s_matrice,
		struct_matrice *s_metrique,
		struct_vecteur *s_valeurs_propres,
		struct_matrice *s_vecteurs_propres_gauches,
		struct_matrice *s_vecteur_propres_droits);
void valeurs_singulieres(struct_processus *s_etat_processus,
		struct_matrice *s_matrice,
		struct_matrice *u, struct_vecteur *s, struct_matrice *vh);
void verrouillage_threads_concurrents(struct_processus *s_etat_processus);
#endif

/*
--------------------------------------------------------------------------------
  Fonctions de gestion des signaux
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
void creation_queue_signaux(struct_processus *s_etat_processus);
void liberation_queue_signaux(struct_processus *s_etat_processus);
void destruction_queue_signaux(struct_processus *s_etat_processus);
int envoi_signal_contexte(struct_processus *s_etat_processus,
		enum signaux_rpl signal);
int envoi_signal_processus(pid_t pid, enum signaux_rpl signal);
int envoi_signal_thread(pthread_t tid, enum signaux_rpl signal);
#endif

/*
--------------------------------------------------------------------------------
  Fonctions de sommation de vecteur dans perte de prcision
--------------------------------------------------------------------------------
*/

real8 sommation_vecteur_reel(real8 *vecteur, unsigned long *taille,
		logical1 *erreur_memoire);
complex16 sommation_vecteur_complexe(complex16 *vecteur, unsigned long *taille,
		logical1 *erreur_memoire);

/*
--------------------------------------------------------------------------------
  Routine principale
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
int rplinit(int argc, char *argv[], unsigned char ***resultats, char *rpl_home);
#endif

/*
--------------------------------------------------------------------------------
  Routines de statistiques
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
real8 correlation_statistique(struct_matrice *s_matrice,
		unsigned long colonne_1, unsigned long colonne_2, logical1 *erreur);
real8 covariance_statistique(struct_matrice *s_matrice,
		unsigned long colonne_1, unsigned long colonne_2,
		unsigned char type, logical1 *erreur);
struct_vecteur *ecart_type_statistique(struct_matrice *s_matrice,
		unsigned char type);
struct_vecteur *moyenne_statistique(struct_matrice *s_matrice);
struct_vecteur *sommation_colonne_statistique(struct_matrice *s_matrice,
		unsigned long colonne);
struct_vecteur *sommation_produits_colonnes_statistiques(struct_matrice
		*s_matrice, unsigned long colonne_1, unsigned long colonne_2);
struct_vecteur *sommation_statistique(struct_matrice *s_matrice);
struct_vecteur *variance_statistique(struct_matrice *s_matrice,
		unsigned char type);
#endif

/*
--------------------------------------------------------------------------------
  Gestion des bibliothques
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
void *chargement_bibliotheque(struct_processus *s_etat_processus,
		unsigned char *bibliotheque);
logical1 execution_fonction_de_bibliotheque(struct_processus *s_etat_processus,
		unsigned char *nom_fonction, unsigned char *bibliothque);
logical1 retrait_bibliotheque(struct_processus *s_etat_processus,
		struct_bibliotheque *descripteur);
#endif

struct_liste_chainee *depilement_pile_operationnelle(struct_rpl_arguments
		*s_rpl_arguments, struct_objet **s_objet);
struct_liste_chainee *empilement_pile_operationnelle(struct_rpl_arguments
		*s_rpl_arguments, struct_objet *s_objet);
struct_liste_chainee *sauvegarde_arguments(struct_rpl_arguments
		*s_rpl_arguments, unsigned long nombre_arguments);

/*
--------------------------------------------------------------------------------
  Fonctions retournant un CHARACTER*(*)
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
unsigned char *analyse_algebrique(struct_processus *s_etat_processus,
		unsigned char *chaine_algebrique, struct_liste_chainee **l_base_liste);
unsigned char *recherche_chemin_fichiers_temporaires(struct_processus
		*s_etat_processus);
unsigned char *compactage(unsigned char *chaine);
unsigned char *conversion_majuscule(unsigned char *chaine);
unsigned char *creation_nom_fichier(struct_processus *s_etat_processus,
		unsigned char *chemin);
unsigned char *extraction_chaine(unsigned char *chaine, unsigned long i,
		unsigned long j);
unsigned char *formateur(struct_processus *s_etat_processus, long offset,
		struct_objet *s_objet);
unsigned char *formateur_fichier(struct_processus *s_etat_processus,
		struct_objet *s_objet, struct_objet *s_format,
		long longueur, long longueur_champ, unsigned char format,
		unsigned char type, long *longueur_effective, long *recursivite);
unsigned char *formateur_flux(struct_processus *s_etat_processus,
		unsigned char *donnees, long *longueur);
unsigned char *formateur_fichier_nombre(struct_processus *s_etat_processus,
		void *valeur_numerique, unsigned char type,
		long longueur, long longueur_champ, unsigned char format);
unsigned char *formateur_fichier_reel(struct_processus *s_etat_processus,
		void *valeur_numerique, unsigned char type,
		long longueur, long longueur_champ, unsigned char format);
unsigned char *formateur_nombre(struct_processus *s_etat_processus,
		void *valeur_numerique, unsigned char type);
unsigned char *formateur_reel(struct_processus *s_etat_processus,
		void *valeur_numerique, unsigned char type);
unsigned char *messages(struct_processus *s_etat_processus);
unsigned char *purification_chaine(unsigned char *chaine);
unsigned char *reencodage(struct_processus *s_etat_processus,
		unsigned char *chaine_entree, unsigned char *codage_entree,
		unsigned char *codage_sortie);
unsigned char *transliteration(struct_processus *s_etat_processus,
		unsigned char *chaine_entree, unsigned char *codage_entree,
		unsigned char *codage_sortie);
#endif

/*
--------------------------------------------------------------------------------
  Fonctions retournant un INTEGER*1 (drapeau d'erreur)
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
logical1 analyse_syntaxique(struct_processus *s_etat_processus);
logical1 caracteristiques_fichier(struct_processus *s_etat_processus,
		unsigned char *nom, logical1 *existence, logical1 *ouverture,
		unsigned long *unite);
logical1 controle(struct_processus *s_etat_processus, unsigned char *fichier,
		unsigned char *type, unsigned char *somme_candidate);
logical1 creation_variable(struct_processus *s_etat_processus,
		struct_variable *s_variable,
		unsigned char autorisation_creation_variable_statique,
		unsigned char autorisation_creation_variable_partagee);
logical1 creation_variable_partagee(struct_processus *s_etat_processus,
		struct_variable_partagee *s_variable);
logical1 creation_variable_statique(struct_processus *s_etat_processus,
		struct_variable_statique *s_variable);
logical1 chainage(struct_processus *s_etat_processus);
logical1 compilation(struct_processus *s_etat_processus);
logical1 depilement(struct_processus *s_etat_processus,
		struct_liste_chainee **l_base_liste, struct_objet **s_objet);
logical1 destruction_fichier(unsigned char *nom_fichier);
logical1 empilement(struct_processus *s_etat_processus,
		struct_liste_chainee **l_base_liste, struct_objet *s_objet);
logical1 empilement_pile_last(struct_processus *s_etat_processus,
		unsigned long nombre_arguments);
logical1 evaluation(struct_processus *s_etat_processus, struct_objet *s_objet,
		unsigned char type_evaluation);
logical1 initialisation_fichier_acces_direct(struct_processus *s_etat_processus,
		sqlite3 *sqlite, logical1 binaire);
logical1 initialisation_fichier_acces_indexe(struct_processus *s_etat_processus,
		sqlite3 *sqlite, integer8 position_clef, logical1 binaire);
logical1 recherche_instruction_suivante(struct_processus *s_etat_processus);
logical1 retrait_variable(struct_processus *s_etat_processus,
		unsigned char *nom_variable, unsigned char type);
logical1 retrait_variable_par_niveau(struct_processus *s_etat_processus);
logical1 retrait_variable_partagee(struct_processus *s_etat_processus,
		unsigned char *nom_variable, union_position_variable position);
logical1 retrait_variable_statique(struct_processus *s_etat_processus,
		unsigned char *nom_variable, union_position_variable position);
logical1 sequenceur(struct_processus *s_etat_processus);
logical1 sequenceur_optimise(struct_processus *s_etat_processus);
#endif

/*
--------------------------------------------------------------------------------
  Fonctions retournant un INTEGER*1 (drapeau)
--------------------------------------------------------------------------------
*/

logical1 depassement_addition(integer8 *a, integer8 *b, integer8 *resultat);
logical1 depassement_multiplication(integer8 *a, integer8 *b,
		integer8 *resultat);
logical1 depassement_puissance(integer8 *a, integer8 *b, integer8 *resultat);

#ifndef RPLARGS
logical1 ecriture_pipe(struct_processus *s_etat_processus, int pipe,
		struct_objet *s_objet);
logical1 recherche_variable(struct_processus *s_etat_processus,
		unsigned char *nom_variable);
logical1 recherche_variable_globale(struct_processus *s_etat_processus,
		unsigned char *nom_variable);
logical1 recherche_variable_partagee(struct_processus *s_etat_processus,
		unsigned char *nom_variable, union_position_variable position,
		unsigned char origine);
logical1 recherche_variable_statique(struct_processus *s_etat_processus,
		unsigned char *nom_variable, union_position_variable position,
		unsigned char origine);
logical1 test_cfsf(struct_processus *s_etat_processus,
		unsigned char indice_drapeau);
logical1 test_expression_rpn(unsigned char *chaine);
logical1 test_fonction(unsigned char *chaine);
#endif

/*
--------------------------------------------------------------------------------
  Fonctions retournant un ssize_t
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
ssize_t read_atomic(struct_processus *s_etat_processus,
		int fd, void *buf, size_t count);
ssize_t write_atomic(struct_processus *s_etat_processus,
		int fd, void *buf, size_t count);
#endif

/*
--------------------------------------------------------------------------------
  Fonctions renvoyant un int
--------------------------------------------------------------------------------
*/

int alsprintf(unsigned char **strp, const char *fmt, ...);
int interruption_violation_access(void *adresse_fautive, int gravite);
#ifndef RPLARGS
int liste_variables(struct_processus *s_etat_processus,
		struct_tableau_variables *tableau, int position,
		struct_arbre_variables *l_element_courant);
int nombre_variables(struct_processus *s_etat_processus,
		struct_arbre_variables *l_element_courant);
#endif
int tex_fprintf(struct_processus *s_etat_processus,
		file *flux, const char *format, ...);
int transliterated_fprintf(struct_processus *s_etat_processus, file *flux,
		const char *format, ...);
int valsprintf(unsigned char **strp, const char *fmt, va_list ap);
int wrapper_instruction_intrinseque(void (*fonction)(),
		struct_rpl_arguments *rpl_arguments);

/*
--------------------------------------------------------------------------------
  Fonctions renvoyant un INTEGER*1 (longueur d'un entier binaire)
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
unsigned char longueur_entiers_binaires(struct_processus *s_etat_processus);
unsigned char recherche_longueur_optimale(integer8 parametre);
#endif

/*
--------------------------------------------------------------------------------
  Fonctions retournant un INTEGER*8 (valeur binaire non signe)
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
logical8 masque_entiers_binaires(struct_processus *s_etat_processus);
#endif

/*
--------------------------------------------------------------------------------
  Fonctions retournant un INTEGER*8 (entier standard RPL/2)
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
integer8 occupation_memoire(struct_objet *s_objet);
#endif

/*
--------------------------------------------------------------------------------
  Fonctions renvoyant une liste chane (pointeur sur *struct_liste_chainee)
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
struct_liste_chainee *analyse_rpn(struct_processus *s_etat_processus,
		unsigned char *chaine_rpn);
#endif

/*
--------------------------------------------------------------------------------
  Fonctions renvoyant une copie de la structure de description des processus
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
struct_processus *copie_etat_processus(struct_processus *s_etat_processus);
#endif

/*
--------------------------------------------------------------------------------
  Fonctions renvoyant un pointeur sur un fichier (file *) ou un descripteur
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
file *creation_fichier_tex(struct_processus *s_etat_processus);
struct_descripteur_fichier *descripteur_fichier(struct_processus
		*s_etat_processus, struct_fichier *s_fichier);
#endif

/*
--------------------------------------------------------------------------------
  Fonctions renvoyant un pointeur sur une *struct_objet
--------------------------------------------------------------------------------
*/

struct_objet *copie_objet(struct_processus *s_etat_processus,
		struct_objet *s_objet, unsigned char type);

struct_objet *allocation(struct_processus *s_etat_processus, enum t_type type);
#ifndef RPLARGS
struct_objet *formateur_date(struct_processus *s_etat_processus,
		struct timeval *temps);
struct_objet *lecture_pipe(struct_processus *s_etat_processus, int pipe);
struct_objet *parametres_sql(struct_processus *s_etat_processus,
		struct_objet *s_parametres);
struct_objet *simplification(struct_processus *s_etat_processus,
		struct_objet *s_objet);
#endif

/*
--------------------------------------------------------------------------------
  Fonctions grant les arbres de simplification
--------------------------------------------------------------------------------
*/

#ifndef RPLARGS
struct_arbre *creation_arbre(struct_processus *s_etat_processus,
		struct_objet **t_objets, unsigned long indice,
		unsigned long indice_maximal);
void liberation_arbre(struct_processus *s_etat_processus,
		struct_arbre *s_noeud);
void parcours_arbre(struct_processus *s_etat_processus, struct_arbre *s_noeud);
void simplification_arbre(struct_processus *s_etat_processus,
		struct_arbre *s_noeud);
#endif

/*
--------------------------------------------------------------------------------
  Fonctions grant la compltion automatique
--------------------------------------------------------------------------------
*/

char **completion_rpl();
char *generateur_commandes();
char **completion_matches();

// END C PROTOTYPES

/*
--------------------------------------------------------------------------------
  Arithmtique
--------------------------------------------------------------------------------
*/

void f77absc_(complex16 *c, real8 *resultat);

void f77additioncc_(complex16 *ca, complex16 *cb, complex16 *resultat);
void f77additionci_(complex16 *ca, integer8 *ib, complex16 *resultat);
void f77additioncr_(complex16 *ca, real8 *rb, complex16 *resultat);

void f77divisioncc_(complex16 *ca, complex16 *rb, complex16 *resultat);
void f77divisionci_(complex16 *ca, integer8 *ib, complex16 *resultat);
void f77divisioncr_(complex16 *ca, real8 *rb, complex16 *resultat);
void f77divisionic_(integer8 *ib, complex16 *cb, complex16 *resultat);
void f77divisionrc_(real8 *ra, complex16 *cb, complex16 *resultat);

void f77multiplicationcc_(complex16 *ca, complex16 *cb, complex16 *resultat);
void f77multiplicationci_(complex16 *ca, integer8 *ib, complex16 *resultat);
void f77multiplicationcr_(complex16 *ca, real8 *rb, complex16 *resultat);

void f77puissancecc_(complex16 *ca, complex16 *cb, complex16 *resultat);
void f77puissanceci_(complex16 *ca, integer8 *ib, complex16 *resultat,
		integer4 *troncature);
void f77puissancecr_(complex16 *ca, real8 *rb, complex16 *resultat);
void f77puissanceic_(integer8 *ia, complex16 *cb, complex16 *resultat);
void f77puissanceii_(integer8 *ia, integer8 *ib, integer8 *resultat);
void f77puissanceir_(integer8 *ia, real8 *rb, real8 *resultat);
void f77puissancerc_(real8 *ra, complex16 *cb, complex16 *resultat);
void f77puissanceri_(real8 *ra, integer8 *ib, real8 *resultat,
		integer4 *troncature);
void f77puissancerr_(real8 *ra, real8 *rb, real8 *resultat);

void f77racinecarreec_(complex16 *ca, complex16 *resultat);
void f77racinecarreein_(integer8 *ia, complex16 *resultat);
void f77racinecarreeip_(integer8 *ia, real8 *resultat);
void f77racinecarreern_(real8 *ia, complex16 *resultat);
void f77racinecarreerp_(real8 *ia, real8 *resultat);

void f77soustractioncc_(complex16 *ca, complex16 *cb, complex16 *resultat);
void f77soustractionci_(complex16 *ca, integer8 *ib, complex16 *resultat);
void f77soustractioncr_(complex16 *ca, real8 *rb, complex16 *resultat);
void f77soustractionic_(integer8 *ib, complex16 *cb, complex16 *resultat);
void f77soustractionrc_(real8 *ra, complex16 *cb, complex16 *resultat);

/*
--------------------------------------------------------------------------------
  Trigonomtrie complexe
--------------------------------------------------------------------------------
*/

void f77acos_(complex16 *argument, complex16 *resultat);
void f77asin_(complex16 *argument, complex16 *resultat);
void f77atan_(complex16 *argument, complex16 *resultat, integer4 *erreur);
void f77cos_(complex16 *argument, complex16 *resultat);
void f77sin_(complex16 *argument, complex16 *resultat);
void f77tan_(complex16 *argument, complex16 *resultat, integer4 *erreur);

/*
--------------------------------------------------------------------------------
  Fonctions hyperboliques
--------------------------------------------------------------------------------
*/

void f77acoshc_(complex16 *argument, complex16 *resultat);
void f77alogc_(complex16 *argument, complex16 *resultat);
void f77asinhc_(complex16 *argument, complex16 *resultat);
void f77atanhc_(complex16 *argument, complex16 *resultat);
void f77cosh_(complex16 *argument, complex16 *resultat);
void f77expc_(complex16 *argument, complex16 *resultat);
void f77lnc_(complex16 *argument, complex16 *resultat, integer4 *erreur);
void f77lnin_(integer8 *argument, complex16 *resultat, integer4 *erreur);
void f77lnrn_(real8 *argument, complex16 *resultat, integer4 *erreur);
void f77logc_(complex16 *argument, complex16 *resultat, integer4 *erreur);
void f77login_(integer8 *argument, complex16 *resultat, integer4 *erreur);
void f77logrn_(real8 *argument, complex16 *resultat, integer4 *erreur);
void f77sinh_(complex16 *argument, complex16 *resultat);
void f77tanh_(complex16 *argument, complex16 *resultat, integer4 *erreur);
void f77acoshi_(integer8 *argument, real8 *resultat);
void f77acoshr_(real8 *argument, real8 *resultat);
void f77asinhi_(integer8 *argument, real8 *resultat);
void f77asinhr_(real8 *argument, real8 *resultat);
void f77atanhi_(integer8 *argument, real8 *resultat);
void f77atanhr_(real8 *argument, real8 *resultat);
void f77lnip_(integer8 *argument, real8 *resultat, integer4 *erreur);
void f77lnrp_(real8 *argument, real8 *resultat, integer4 *erreur);
void f77logip_(integer8 *argument, real8 *resultat, integer4 *erreur);
void f77logrp_(real8 *argument, real8 *resultat, integer4 *erreur);

/*
--------------------------------------------------------------------------------
  Calculs divers
--------------------------------------------------------------------------------
*/

void f90arrangement(integer8 *n, integer8 *k, real8 *resultat);
void f90combinaison(integer8 *n, integer8 *k, real8 *resultat);
void f90fcd(real8 *x, integer8 *n1, integer8 *n2, real8 *resultat);
void f90gausscd(real8 *x, real8 *moyenne, real8 *variance, real8 *resultat);
void f90tcd(real8 *x, integer8 *n, real8 *resultat);
void f90x2cd(real8 *x, integer8 *n, real8 *resultat);

// BEGIN C PROTOTYPES
void dft(complex16 *dft, integer4 *nombre_lignes, integer4 *nombre_colonnes,
		integer4 *inverse, integer4 *erreur);
// END C PROTOTYPES

/*
--------------------------------------------------------------------------------
  Fonctions Lapack
--------------------------------------------------------------------------------
*/

	/*
	 * Calcul de la longueur d'un espace de travail
	 */

#ifndef RPLARGS
integer4 ilaenv_(integer4 *optimal, char *nom_fonction, char *option,
		integer4 *n1, integer4 *n2, integer4 *n3, integer4 *n4,
		integer4 longueur_nom_fonction, integer4 longueur_option);

	/*
	 * Inversion
	 */

void dgetri_(integer4 *ordre_a, real8 *a, integer4 *lda, integer4 *pivot,
		real8 *work, integer4 *dim_work, integer4 *ios);
void zgetri_(integer4 *ordre_a, complex16 *a, integer4 *lda, integer4 *pivot,
		complex16 *work, integer4 *dim_work, integer4 *ios);

	/*
	 * Valeurs propres
	 */

void zgeev_(unsigned char *calcul_vp_gauches, unsigned char *calcul_vp_droits,
		integer4 *dim_matrice1, complex16 *matrice_f77,
		integer4 *dim_matrice2, complex16 *valeurs_propres,
		complex16 *vpg_f77, integer4 *dim_matrice3,
		complex16 *vpd_f77, integer4 *dim_matrice4,
		complex16 *work, integer4 *lwork, real8 *rwork,
		integer4 *erreur, integer4 longueur_1, integer4 longueur_2);

	/*
	 * Valeurs propres gnralises
	 */

void zggev_(unsigned char *calcul_vp_gauches, unsigned char *calcul_vp_droits,
		integer4 *dim_matrice1, complex16 *matrice_f77,
		integer4 *dim_matrice2, complex16 *metrique_f77,
		integer4 *dim_matrice3, complex16 *alpha, complex16 *beta,
		complex16 *vpg_f77, integer4 *dim_matrice4,
		complex16 *vpd_f77, integer4 *dim_matrice5,
		complex16 *work, integer4 *lwork, real8 *rwork,
		integer4 *erreur, integer4 longueur_1, integer4 longueur_2);

	/*
	 * Moindres carrs
	 */

void dgelsd_(integer4 *nombre_lignes_a, integer4 *nombre_colonnes_a,
		integer4 *nombre_colonnes_b, real8 *matrice_a, integer4 *lda,
		real8 *matrice_b, integer4 *ldb, real8 *valeurs_singulieres,
		real8 *rcond, integer4 *rank, real8 *work, integer4 *lwork,
		integer4 *iwork, integer4 *info);
void zgelsd_(integer4 *nombre_lignes_a, integer4 *nombre_colonnes_a,
		integer4 *nombre_colonnes_b, complex16 *matrice_a, integer4 *lda,
		complex16 *matrice_b, integer4 *ldb, real8 *valeurs_singulieres,
		real8 *rcond, integer4 *rank, complex16 *work, integer4 *lwork,
		real8 *rwork, integer4 *iwork, integer4 *info);

	/*
	 * Moindres carrs gnralis
	 */

	/*
	 * Dcomposition en valeurs singulires
	 */

void dgesvd_(unsigned char *jobu, unsigned char *jobvh,
		integer4 *m, integer4 *n, real8 *a, integer4 *lda,
		real8 *s, real8 *u, integer4 *ldu, real8 *vh, integer4 *ldvh,
		real8 *work, integer4 *lwork, integer4 *info, integer4 longueur_jobu,
		integer4 longueur_jobvh);
void zgesvd_(unsigned char *jobu, unsigned char *jobvh,
		integer4 *m, integer4 *n, real8 *a, integer4 *lda,
		real8 *s, real8 *u, integer4 *ldu, real8 *vh, integer4 *ldvh,
		real8 *work, integer4 *lwork, real8 *rwork, integer4 *info,
		integer4 longueur_jobu, integer4 longueur_jobvh);

	/*
	 * Factorisations diverses
	 */

		/* LU de Crout */
void dgetrf_(integer4 *nb_lignes_a, integer4 *nb_colonnes_a, real8 *a,
		integer4 *lda, integer4 *pivot, integer4 *ios);
void zgetrf_(integer4 *nb_lignes_a, integer4 *nb_colonnes_a, complex16 *a,
		integer4 *lda, integer4 *pivot, integer4 *ios);

		/* LQ */
void dgelq2_(integer4 *nb_lignes_a, integer4 *nb_colonnes_a, real8 *a,
		integer4 *lda, real8 *tau, real8 *work, integer4 *ios);
void zgelq2_(integer4 *nb_lignes_a, integer4 *nb_colonnes_a, complex16 *a,
		integer4 *lda, complex16 *tau, complex16 *work, integer4 *ios);

		/* QR */
void dgeqp3_(integer4 *nb_lignes_a, integer4 *nb_colonnes_a, real8 *a,
		integer4 *lda, integer4 *pivot, real8 *tau, real8 *work,
		integer4 *lwork, integer4 *ios);
void zgeqp3_(integer4 *nb_lignes_a, integer4 *nb_colonnes_a, complex16 *a,
		integer4 *lda, integer4 *pivot, complex16 *tau, complex16 *work,
		integer4 *lwork, real8 *rwork, integer4 *ios);

		/* Cholesky */
void dpptrf_(unsigned char *mode, integer4 *n, complex16 *triangle,
		integer4 *ios, integer4 longueur);
void zpptrf_(unsigned char *mode, integer4 *n, complex16 *triangle,
		integer4 *ios, integer4 longueur);

		/* Schur */
void dgees_(unsigned char *calcul_vecteur, unsigned char *valeurs_triees,
		logical4 (*fonction_tri)(), integer4 *n, real8 *a,
		integer4 *lda, integer4 *sdim, real8 *wr, real8 *wi,
		real8 *vs, integer4 *ldvs, real8 *work, integer4 *lwork,
		logical4 *bwork, integer4 *info, integer4 l1, integer4 l2);
void zgees_(unsigned char *calcul_vecteur, unsigned char *valeurs_triees,
		logical4 (*fonction_tri)(), integer4 *n, complex16 *a,
		integer4 *lda, integer4 *sdim, complex16 *w,
		complex16 *vs, integer4 *ldvs, complex16 *work, integer4 *lwork,
		real8 *rwork, logical4 *bwork, integer4 *info,
		integer4 l1, integer4 l2);

		/* Dterminant et rang */
void dgecon_(unsigned char *norm, integer4 *n, real8 *a, integer4 *lda,
		real8 *anorm, real8 *rcond, real8 *work, integer4 *iwork,
		integer4 *info, integer4 l1);
void dgelsy_(integer4 *m, integer4 *n, integer4 *nrhs, real8 *a,
		integer4 *lda, real8 *b, integer4 *ldb, integer4 *jpvt, real8 *rcond,
		integer4 *rank, real8 *work, integer4 *lwork, integer4 *info);
real8 dlange_(unsigned char *norme, integer4 *n, integer4 *m, real8 *a,
		integer4 *lda, real8 *work, integer4 longueur);
real8 zlange_(unsigned char *norme, integer4 *n, integer4 *m, real8 *a,
		integer4 *lda, real8 *work, integer4 longueur);
void zgecon_(unsigned char *norm, integer4 *n, complex16 *a, integer4 *lda,
		real8 *anorm, real8 *rcond, complex16 *work, real8 *rwork,
		integer4 *info, integer4 l1);
void zgelsy_(integer4 *m, integer4 *n, integer4 *nrhs, complex16 *a,
		integer4 *lda, complex16 *b, integer4 *ldb, integer4 *jpvt,
		real8 *rcond, integer4 *rank, complex16 *work, integer4 *lwork,
		real8 *rwork, integer4 *info);
#endif

/*
 * Fonction testant la requte d'arrt (ncessaire pour tester le
 * drapeau depuis une bibliothque).
 */

int test_arret(struct_processus *s_etat_processus);

#endif

// vim: ts=4
