/*******************************************************************
 * driver2.c                                                       *
 *******************************************************************
 * A simple pdd driver sample driving the parallel port.           *
 * Also it implements a timer procedure for polling.               *
 *******************************************************************
 * Warpstock Europe 1999                                           *
 * PDD workshop                                                    *
 *******************************************************************
 * Frank Meilinger                                                 *
 * Hirtesemuehle 1                                                 *
 * 65529 Waldems-Esch                                              *
 * e-mail: meilinger@wiesbaden.netsurf.de                          *
 * web: http://rover.wiesbaden.netsurf.de/~meile                   *
 *******************************************************************/


/* header */
#define INCL_DOSMISC
#define INCL_NOPMAPI
#define INCL_ERRORS

#include <os2.h>              /* special types  includes os2defs.h */
#include <bseerr.h>           /* error definitions */

#include <devhdr.h>           /* device driver header definitions */
#include <devcmd.h>           /* device driver strategy commands */
#include <strat2.h>           /* strategy 2 definitions */
#include <reqpkt.h>           /* request packet structure definitions */
#include <dhcalls.h>          /* device helper calls */
#include <infoseg.h>          /* InfoSegGDT definition */


/* prototyping */
extern void STRATEGY();       /* defined in assembler header */
extern void TIMERPROC();      /* defined in assembler header */
void ticker();                /* C timer handler */
int Init(PRPINITIN);          /* init function */


/*************************************************************************
 * Begin of resident data segment - it MUST start with the device header.*
 * All varaibles need to be initialized, to put it in the data segment!!!*
 *************************************************************************/

/* device header */
struct SysDev3 device =
       {
        { 
         (ULONG)0xFFFFFFFF,
         (USHORT)(DEV_CHAR_DEV | DEV_30 | DEVLEV_3),
         (USHORT)STRATEGY,
         (USHORT)0,
         "DRIVER2$",
         (USHORT)0,
         (USHORT)0,
         (USHORT)0,
         (USHORT)0,
        },
        (ULONG)(DEV_IOCTL2 | DEV_PARALLEL)
       };

/* DevHlp entry point (the variable name is fix!) */
PFN Device_Help    = NULL;

/* device status */
unsigned short deviceOpen = 0;

/* PID of the process which opened the device */
unsigned short currentPID = 0;

ULONG blockedReadPending = 0L;

ULONG lockHandle = 0L;

BYTE inputByte = 0;

/* marks the end of the data segment */
BYTE EndMarker = 0;

/******************************************************************************
 * end of resident data segment - it MUST end with the EndMarker (see Init()) *
 ******************************************************************************/


/************************************************************************
 * begin of deletable data segment (it MUST locate after the EndMarker) *
 * only valid during init-time!                                         *
 ************************************************************************/
CHAR CrLf[] = "\r\n";
CHAR InitMessage1[] = "Warpstock Europe 1999 !!!\r\n";
CHAR InitMessage2[] = "driver sample: DRIVER2$\r\n";
/**********************************
 * end of deletable data segment  *
 **********************************/



/*************************************************************
 * int main(PRPH rp)                                         *
 *-----------------------------------------------------------*
 * description:                                              *
 *  This is the drivers STRATEGY routine. Called by the      *
 *  kernel via the assembler startup code.                   *
 * parameter:                                                *
 *  rp is a far pointer to the request packet.               *
 *  devno is the device number, but ignored in this sample.  *
 * return                                                    *
 *  The return values are filled in the status field of the  *
 *  rp. The return status is returend in ax (main's return   *
 *  value).                                                  *
 *************************************************************/
int main(PRPH rp, int devno)
{
 PVOID   ptemp = NULL;
 USHORT  ustemp = 0;
 USHORT  result;


 switch(rp->Cmd)
 {
  /*****************************************************************
   * init request:                                                 *
   * The init function MUST reside at the end of the code segment. *
   *****************************************************************/
  case CMDInit:
   return Init((PRPINITIN)rp);


  /****************
   * open request *
   ****************/
  case CMDOpen:
   /* check, if device is already open */
   if(deviceOpen)
   {
    /* Device is already open, return with the DONE, ERROR and BUSY flags */
    return (STDON | STERR | STBUI);
   }
   else
   {
    /* Device not open, set open flag */

    /* Get PID out of the local info segment.
       If the GetDosVar function fails, 
       the DONE, ERROR and UNKNOWN_COMMAND flags are set
    */
    if(DevHelp_GetDOSVar(DHGETDOSV_LOCINFOSEG,0,&ptemp))
     return (STDON | STERR | STATUS_ERR_UNKCMD);
    currentPID = ((struct InfoSegLDT FAR*)ptemp)->LIS_CurProcID;

    deviceOpen = 1;
   }
   return STDON;


  /*****************
   * close request *
   *****************/
  case CMDClose:
   /* check, if the device is open */
   if(!deviceOpen)
    return (STDON | STERR | STATUS_ERR_UNKCMD);

   /* get PID out of the local info segment.
      If the GetDosVar function fails, 
      the DONE, ERROR and UNKNOWN_COMMAND flags are set
   */
   if (DevHelp_GetDOSVar(DHGETDOSV_LOCINFOSEG,0,&ptemp))
    return (STDON | STERR | STATUS_ERR_UNKCMD);

   /* check, if the correct PID want to close the device */
   if(currentPID != ((struct InfoSegLDT FAR*)ptemp)->LIS_CurProcID)
    return (STDON | STERR | STATUS_ERR_UNKCMD);

   /* set device status close */
   deviceOpen = 0;

   return STDON;


  /*************************************************************
   * read request: reads one byte from port 0x379 without wait *
   *************************************************************/
  case CMDINPUT:
   /* check, if the transfer buffer has at least a size of 1 byte */
   if (((PRP_RWV)rp)->NumSectors < 1)
    return (STDON | STERR | ERROR_INSUFFICIENT_BUFFER);

   /* set read byte count in request packet to 0 (error handling) */
   ((PRP_RWV)rp)->NumSectors = 0;

   /* read byte from port to a temporary buffer */
   inputByte = inp(0x379);

   /* the transfer address for the read request packet is a physical address
      and need to be convert to a virtual address */
   if(DevHelp_PhysToVirt(((PRP_RWV)rp)->XferAddr,1,&ptemp,&ustemp))
    return (STDON | STERR | STATUS_ERR_UNKCMD);

   /* copy the temporary buffer to the virtual address */
   ((PBYTE)ptemp)[0] = inputByte;

   /* set read byte count in request packet to 1 */
   ((PRP_RWV)rp)->NumSectors = 1;

   return STDON;


  /************************************************
   * write request: writes one byte to port 0x378 *
   ************************************************/
  case CMDOUTPUT:
   /* check, if the transfer buffer has at least a size of 1 byte */
   if (((PRP_RWV)rp)->NumSectors < 1)
    return (STDON | STERR | ERROR_INSUFFICIENT_BUFFER);

   /* set write byte count in request packet to 0 (error handling) */
   ((PRP_RWV)rp)->NumSectors = 0;

   /* convert physical transfer address to virtual address */
   if(DevHelp_PhysToVirt(((PRP_RWV)rp)->XferAddr,
                         1,
                         &ptemp,
                         &ustemp))
    return (STDON | STERR | STATUS_ERR_UNKCMD);

   /* put byte to printer data port */
   outp(0x378,((PBYTE)ptemp)[0]);

   /* set read byte count in request packet to 1 */
   ((PRP_RWV)rp)->NumSectors = 1;

   return STDON;


  /******************
   * IOCTL requests *
   ******************/
  case CMDGenIOCTL:
   if( (((PRP_GENIOCTL)rp)->Category == 0x80) &&
       (((PRP_GENIOCTL)rp)->Function == 0x60) )
   {
    /* lock callers buffer until request ends */
    if(DevHelp_Lock(SELECTOROF(((PRP_GENIOCTL)rp)->DataPacket),
                    LOCKTYPE_LONG_ANYMEM,
                    0,
                    &lockHandle))
     return (STDON | STERR | STATUS_ERR_UNKCMD);

    /* verify callers access rights to the buffer */
    if(DevHelp_VerifyAccess(SELECTOROF(((PRP_GENIOCTL)rp)->DataPacket),
                            1,
                            OFFSETOF(((PRP_GENIOCTL)rp)->DataPacket),
                            VERIFY_READWRITE))
    {
     DevHelp_UnLock(lockHandle);
     return (STDON | STERR | STATUS_ERR_UNKCMD);
    }

    /* block the running thread. disable intrrupts (ProcBlock enables it) */
    _asm {cli}
    do
    {
     /* inform ticker function to poll */
     blockedReadPending = (ULONG)rp;

     result = DevHelp_ProcBlock(blockedReadPending,
                                -1L,
                                WAIT_IS_INTERRUPTABLE);
     _asm {cli}
     if(result == WAIT_INTERRUPTED)
     {
      /* inform ticker function not longer to poll */
      blockedReadPending = 0L;
      _asm {sti}
      /* unlock callers buffer */
      DevHelp_UnLock(lockHandle);
      return (STDON | STERR | ERROR_I24_CHAR_CALL_INTERRUPTED);
     }
    }
    while(result || blockedReadPending);
    _asm {sti}
    
    /* inform ticker function not longer to poll */
    blockedReadPending = 0L;

    /* copy the temporary buffer to the transfer buffer */
    ((PBYTE)(((PRP_GENIOCTL)rp)->DataPacket))[0] = inputByte;

    /* unlock callers buffer until request ends */
    if(DevHelp_UnLock(lockHandle))
     return (STDON | STERR | STATUS_ERR_UNKCMD);

    return STDON;
   }
   else
    return (STDON | STERR | STATUS_ERR_UNKCMD);


  /******************
   * other requests *
   ******************/
  default:
   return (STDON | STERR | STATUS_ERR_UNKCMD);
 }
} /* main */




/*************************************************************
 * int ticker()                                              *
 *-----------------------------------------------------------*
 * description:                                              *
 *  This routine is called from the time routine _TIMERPROC  *
 *  from the assembler stub. After this routine is installed *
 *  in the init section, it is called every 55 milliseconds. *
 *  It checks, if the bit 5 (paper empty) is 0 and then runs *
 *  the blocking process.                                    *
 * parameter:                                                *
 *  none                                                     *
 * return                                                    *
 *  none                                                     *
 *************************************************************/
void ticker()
{
 USHORT awakeCount;

 _asm {cli}
 if(blockedReadPending)
 {
  inputByte = inp(0x379);
  if( !(inputByte & 0x20))
  {
   DevHelp_ProcRun(blockedReadPending,&awakeCount);
   blockedReadPending = 0L;
  }
 }
 _asm {sti}
}



/*************************************************************
 * int Init(PRPINITIN rp)                                    *
 *-----------------------------------------------------------*
 * description:                                              *
 *  This routine is only called once at initialization time  *
 *  of the driver. This code get deleted after execution!    *
 *  This MUST be the last code in the CS !                   *
 * parameter:                                                *
 *  rp is a far pointer to the request packet.               *
 * return                                                    *
 *  The return value is filled in the status field of the    *
 *  rp. The STDON status is returned if successful.          *
 *************************************************************/
int Init(PRPINITIN rp)
{
 /* save device helper entry */
 Device_Help = rp->DevHlpEP;

 /* install timer handler for ticker function */
 if(DevHelp_SetTimer((NPFN)TIMERPROC))
  return (STDON | STERR | STATUS_ERR_UNKCMD);

 /* display init message */
 DosPutMessage(1,2,CrLf);
 DosPutMessage(1,strlen(InitMessage1),InitMessage1);
 DosPutMessage(1,strlen(InitMessage2),InitMessage2);

 /* set end of CS and DS */
 ((PRPINITOUT)rp)->CodeEnd = (USHORT)&Init;
 ((PRPINITOUT)rp)->DataEnd = (USHORT)&EndMarker;

 return STDON;
}

