//
// Class definition implementation file for dbfLIB Version 3
// (c) Copyright 1996-97, dSOFT Development Inc.
//
#include "dbfClass.h"
#include "io.h"

#ifndef min
#define min(a,b)  (((a) < (b)) ? (a) : (b))
#endif

//////////////////////////////////////////////////////////////
// 		Standard construction
//
CDbf::CDbf( )
{
	pDbf = (PDBF)NULL;
	pListHead = (PCFLD_ITEM)NULL;
	DefaultIndexType = NDX;
    SortReset();
}
//////////////////////////////////////////////////////////////
// 		Open database on construction
//
CDbf::CDbf(  PCHAR pFileName )
{
	pDbf = (PDBF)NULL;
	pListHead = (PCFLD_ITEM)NULL;
	DefaultIndexType = NDX;
    SortReset();
	Open( pFileName );
}
//////////////////////////////////////////////////////////////
// 		Open database and index on construction
//
CDbf::CDbf(  char* pFileName,  char* pIndexName )
{
	pDbf = (PDBF)NULL;
	pListHead = (PCFLD_ITEM)NULL;
	DefaultIndexType = NDX;
    SortReset();
	Open( pFileName, pIndexName );
}
//////////////////////////////////////////////////////////////
// 		Standard destruction
//
CDbf::~CDbf( )
{
	PCFLD_ITEM pItem = pListHead;
	PCFLD_ITEM pNextItem;
	//
	// Free all items in list
	//
	while (pItem){
		pNextItem = (PCFLD_ITEM)(pItem->Next);
		delete pItem;
		pItem = pNextItem;
		}
    //
    // Close database AFTER destroying CFld list since the
    // destruction code of CFld will need the pDbf intact.
	if (pDbf)
		DbfClose(pDbf);

}

//////////////////////////////////////////////////////////////
// 	AppendFrom operation
//	This version of AppendFrom looks at extension for filetype
//  and if not found assumes FILE_TYPE_SDF
//
DS_LONG CDbf::AppendFrom(  char* pFileName )
{
    unsigned long ftype = FILE_TYPE_SDF;
    char* pFile = new char[strlen(pFileName) + 1];
    strcpy(pFile, pFileName);
    strupr(pFile);
    if (strstr(pFile, ".DBF")) 
        ftype = FILE_TYPE_DBF;
    delete pFile;
	return ( DbfAppendFrom( pDbf, pFileName, ftype, (char*)NULL ));
}
//////////////////////////////////////////////////////////////
// 	AppendFrom operation (overload)
//	This version of AppendFrom requires filetype input from user
//
DS_LONG CDbf::AppendFrom(  char* pFileName, DS_ULONG ulType,  char* pDelim )
{
	return ( DbfAppendFrom( pDbf, pFileName, ulType, pDelim ));
}

//////////////////////////////////////////////////////////////
// Clone operation - create new database same structure
//                   as current database. 
//
DS_LONG CDbf::Clone(  char* pFileName )
{
	if (pDbf){
		PDBF pNewDbf = 
			DbfCreateFromHandle( pDbf, pFileName );
		if (pNewDbf){
			DbfClose(pNewDbf);
			return DBF_SUCCESS;
			}
		}
	return DBF_ERROR;
}

//////////////////////////////////////////////////////////////
// 		Close operation
//
DS_LONG CDbf::Close() 
{ 	
	DS_LONG rc = 0;
	if (pDbf)
		{
		PCFLD_ITEM pItem = pListHead;
		PCFLD_ITEM pNextItem;
		//
		// Free all field items in list
		//
		while (pItem){
			pNextItem = (PCFLD_ITEM)(pItem->Next);
			delete pItem;
			pItem = pNextItem;
			}
		pListHead = NULL;
        //
        // We must close the database AFTER destroying
        // all of the CFld's because pDbf must remain
        // intact for the life of a CFld object.
		rc = (DS_LONG)DbfClose(pDbf);
		pDbf = (PDBF)NULL;

		}
	return( rc); 
}

//////////////////////////////////////////////////////////////
// 	Create operation (overloaded function)
//     Uses DB3_TYPE as default   
//
DS_LONG CDbf::Create( char* pFileName )
{
	return Create( pFileName, DB3_TYPE );
}

//////////////////////////////////////////////////////////////
// 		Create operation
//
DS_LONG CDbf::Create( char* pFileName, DS_LONG lType )
{
	DS_LONG rc = DBF_ERROR;
	PCFLD_ITEM pItem;
	PDBFFLD pFlds, pCurFld;
	PDBF pNewDbf;
	DS_SHORT NumFlds;
	DS_SHORT rc_create =0;

	//
	// Create pFlds array from CFlds...
	//
	pItem = pListHead;
	NumFlds = 0;
	while (pItem){
		if ( !pItem->cfld.bRemove )
		    { // only count if bRemove is false
			NumFlds++;
			}
		pItem = (PCFLD_ITEM)(pItem->Next);
		}
	if ( NumFlds == 0 )
		return DBF_ERROR;

	pFlds = new DBFFLD[NumFlds];
	pCurFld = pFlds;
	pItem = pListHead;

	while (pItem){
		if ( !pItem->cfld.bRemove )
		    { // only copy if bRemove is false
			if (pItem->cfld.bModified){
				//
				// if the field has been modified, then
				// get the info from the "New" fields.
				//
				strcpy(pCurFld->Name, pItem->cfld.szNewFieldName);
				pCurFld->Type = (char)pItem->cfld.cNewFldType;
				pCurFld->Length = pItem->cfld.sNewFldLen;
				pCurFld->Dec = pItem->cfld.sNewFldDec;
				}
			else{
				strcpy(pCurFld->Name, pItem->cfld.szFieldName);
				pCurFld->Type = (char)pItem->cfld.cFldType;
				pCurFld->Length = pItem->cfld.sFldLen;
				pCurFld->Dec = pItem->cfld.sFldDec;
				}

			pCurFld++;
			}
		pItem = (PCFLD_ITEM)(pItem->Next);
		}
	//
	// Create new dbf
	//
	pNewDbf = DbfCreateFile(pFileName, 
						 (DS_SHORT)NumFlds,
						 pFlds,
						 (PDS_SHORT)&rc_create,
						 (DS_SHORT)lType,
						 USE_DEFAULT
						 );
	rc = rc_create;
	//
	// Free pFlds array memory
	delete []pFlds;

	// 
	// Call Close() to close cur dbf & free pListHead
	// if database is open...
	if ( pDbf ){
		Close();
		}
	//
	// Directly close new dbf with DbfClose()
	// then call Open() on new dbf
	if (rc == DBF_SUCCESS)
		{
		DbfClose(pNewDbf);
		rc = Open(pFileName);
		}
	return (rc);
		
} // Create()

//////////////////////////////////////////////////////////////
// 		Create index operation
//
DS_LONG CDbf::CreateIndex(  char* pIndexName,
						 char* pExpression )
{
	DS_LONG rc = DBF_ERROR;

	if (pDbf != NULL) { // check to see if valid handle
		rc = (DS_LONG)IndexCreate(pDbf, pIndexName, 
		              pExpression, DefaultIndexType);
		}
	return (rc);
} // CreateIndex()

//////////////////////////////////////////////////////////////
// 		Create index operation (overloaded)
//
DS_LONG CDbf::CreateIndex(  char* pIndexName,
						 char* pExpression,
						int iType )
{
	DS_LONG rc = DBF_ERROR;

	if (pDbf != NULL) { // check to see if valid handle
		rc = (DS_LONG)IndexCreate(pDbf, pIndexName, 
		              pExpression, iType);
		}
	return (rc);
} // CreateIndex()

//////////////////////////////////////////////////////////////
// 	CopyTo operation
//	This version of CopyTo looks at extension for filetype
//  and if not found assumes FILE_TYPE_SDF
//
DS_LONG CDbf::CopyTo(  char* pFileName )
{
    unsigned long ftype = FILE_TYPE_SDF;
    char* pFile = new char[strlen(pFileName) + 1];
    strcpy(pFile, pFileName);
    strupr(pFile);
    if (strstr(pFile, ".DBF")) 
        ftype = FILE_TYPE_DBF;
    delete pFile;
	return ( DbfCopyTo( pDbf, pFileName, ftype, (char*)NULL ));
}
//////////////////////////////////////////////////////////////
// 		CopyTo operation (overload)
//	This version of CopyTo requires filetype input from user
//
DS_LONG CDbf::CopyTo(  char* pFileName, DS_ULONG ulType,  char* pDelim )
{
	return ( DbfCopyTo( pDbf, pFileName, ulType, pDelim ));
}

//////////////////////////////////////////////////////////////
// 		Locate record based on field ref
//
DS_LONG CDbf::Locate( DS_LONG FieldRef,  char* Searchdata )
{
	return( (DS_LONG)DbfLocate(pDbf, (DS_SHORT)FieldRef, Searchdata) );
}

//////////////////////////////////////////////////////////////
// 		Locate record based on field name (overloaded function)
//
DS_LONG CDbf::Locate(  char* pFldName,  char* Searchdata )
{
	return( 
	  (DS_LONG)DbfLocate(pDbf, FldRef(pDbf, pFldName), Searchdata) );
}

DS_LONG CDbf::LockRecord(DS_LONG recno, DS_LONG locktype)
{
    switch( locktype ){
        case CLIPPER_LOCK:
            return DbfClipperLock(pDbf, recno);
            break;
        case DB4_LOCK:
            return DbfDB4Lock(pDbf, recno);
            break;
        case DB3_LOCK:
            return DbfDB3Lock(pDbf, recno);
            break;
        default:
            break;
    }

    return DbfLock(pDbf, recno);
}
//////////////////////////////////////////////////////////////
// 		Modify Structure operation
//
DS_LONG CDbf::ModifyStructure( )
{
	DS_LONG rc = DBF_ERROR;
	PCFLD_ITEM pItem;
	PDBF pOldDbf;
	int ChangeCount = 0;

	if ( pDbf->openflag != DBF_OPEN_EXCLUSIVE )
		return DBF_REQUIRES_EXCLUSIVE;
	 
	//
	// Check to see if any modifications actually made
	// 
	pItem = pListHead;
	while (pItem){
		if (pItem->cfld.bModified || pItem->cfld.bRemove)
			ChangeCount++;

		pItem = (PCFLD_ITEM)(pItem->Next);
		}
	//
	// Just return if no changes made
	//
	if ( !ChangeCount )
		return DBF_SUCCESS;
	//
	// Save the path of the current database
	// and come up with temp file name...
	//
	char* ptr;
	char* pszOrigName;
	char* pszTempName;
	DS_ULONG bufsize = GetDbfName(NULL, 0) + 1;

	pszOrigName = new char[bufsize];
	pszTempName = new char[bufsize];
	GetDbfName( pszOrigName, bufsize );

	strcpy(pszTempName, pszOrigName);

	ptr = pszTempName + strlen(pszTempName);
	while (ptr >= pszTempName){
		if (*ptr ==	'\\')
			{
			*(++ptr) = 0x0;
			break;
			}
		ptr--;
		}
	if ( ptr < pszTempName )
		ptr = pszTempName;
	
	// temp file name first character is dollar sign
	*ptr = '$';

	//
	// We must call Create to get a temporary
	// database with our new structure.
	// NOTE: Upon successful return from Create
	// our pDbf will now point to the temp database
	//
	if (Create( pszTempName, GetDatabaseType() ) == DBF_SUCCESS)
		{
		DS_SHORT result;
		DS_LONG  recs = 0;
		pOldDbf = DbfOpen(pszOrigName, DBF_OPEN_EXCLUSIVE, &result);
		if (pOldDbf)
			{
			recs = DbfCopyTo(pOldDbf, (char*)pDbf, FILE_TYPE_DBF_HANDLE, NULL);
			// Now rename the temp file to the original
			DbfClose(pOldDbf);
			Close();
			RenameDbf(pszTempName, pszOrigName);
			OpenExclusive( pszOrigName );
			rc = DBF_SUCCESS;
			}
		} // endif Create...

	delete pszTempName;
	delete pszOrigName;
	
return (rc);

} // ModifyStructure()

//////////////////////////////////////////////////////////////
// 		Move record pointer to next record 
//
DS_LONG CDbf::Next( )
{
	DS_LONG rc;
	rc = DbfSkip(pDbf, 1);
	if (rc != DBF_SUCCESS)
		{
		// We must be at the logical bottom, so execute
		// DbfBottom to fill in the record buffer correctly.
		//
		DbfBottom(pDbf);
		}
	return( rc );
}

//////////////////////////////////////////////////////////////
// 		Move record pointer to previous record 
//
DS_LONG CDbf::Prev( )
{
	DS_LONG rc;
	rc = DbfSkip(pDbf, -1);
	if (rc != DBF_SUCCESS)
		{
		// We must be at the logical top, so execute
		// DbfTop to fill in the record buffer correctly.
		//
		DbfTop(pDbf);
		}
	return( rc );
}

//////////////////////////////////////////////////////////////
// 	RenameDbf operation - renames dbf (handles memo file)
//
DS_LONG CDbf::RenameDbf( char* pOldName, char* pNewName )
{
	DS_LONG rc = DBF_SUCCESS;
	PDBF pOld;
	char* pOldBuf;
	char* pNewBuf;
	char* ptr;
	char  ext[5];

	pOld = DbfUse( pOldName );
	if ( pOld == NULL )
		return DBF_ERROR;

	pOldBuf = new char[strlen(pOldName)+5];
	pNewBuf = new char[strlen(pNewName)+5];
	//
	//	Set memo extension based on type of database
	//
	switch ( pOld->Version ){
		case DB3MEMO:
		case DB4MEMO:
			strcpy(ext, ".DBT");
			break;
		case FPMEMO:
			strcpy(ext, ".FPT");
			break;
		case DB3:
		default:
			ext[0] = 0x0;
			break;
		}
	DbfClose( pOld );

	//
	// if pNewName already exists, it will be backed up
	//
	if ( access( pNewName, 0 ) == 0 ){
		strcpy(pNewBuf, pNewName);
		ptr = strrchr(pNewBuf, '.');
		if (ptr)
			strcpy(ptr, ".BAK");
		else
			strcat(pNewBuf, ".BAK");
		//
		// remove the BAK file if it exists
		remove(pNewBuf);
		// rename the dbf to bak
		rename(pNewName, pNewBuf); 
		// rename the memo file to .TBK
		if (strlen(ext)){
			// get the memo file name into oldbuf
			strcpy(pOldBuf, pNewName);
			ptr = strrchr(pOldBuf, '.');
			if (ptr)
				strcpy(ptr, ext);
			else
				strcat(pOldBuf, ext);

			// get the backup name (TBK)
			strcpy(pNewBuf, pNewName);
			ptr = strrchr(pNewBuf, '.');
			if (ptr)
				strcpy(ptr, ".TBK");
			else
				strcat(pNewBuf, ".TBK");
			// remove the TBK file if it exists
			remove( pNewBuf );
			// rename the memo file
			rename( pOldBuf, pNewBuf );
			}
		}

	//
	// place extension if memo file existed.
	//
	if (strlen(ext)){
		strcpy(pOldBuf, pOldName);
		ptr = strrchr(pOldBuf, '.');
		if (ptr)
			strcpy(ptr, ext);
		else
			strcat(pOldBuf, ext);

		strcpy(pNewBuf, pNewName);
		ptr = strrchr(pNewBuf, '.');
		if (ptr)
			strcpy(ptr, ext);
		else
			strcat(pNewBuf, ext);

		// rename the memo file
		rename( pOldBuf, pNewBuf );
		}
	//
	// Finally, rename the dbf's
	//
	rename(pOldName, pNewName);

	delete pNewBuf;
	delete pOldBuf;
return rc;
} // RenameDbf()


//////////////////////////////////////////////////////////////
// 		Open operation
//
DS_LONG CDbf::Open( char* pFileName, DS_ULONG OpenMode )
{
	DS_LONG rc = DBF_ERROR;
	DS_SHORT result;

	if (pDbf == NULL) { // check to see if we already have a handle
		pDbf = DbfOpen( pFileName, (DS_SHORT)OpenMode, &result );
		if (pDbf != NULL)
			{
			//
			// if the cfld list is populated, we must
			// free all the items since we are opening
			// a database and must start clean
			// 
			if (pListHead){
				PCFLD_ITEM pNextItem;
				while (pListHead){
					pNextItem = (PCFLD_ITEM)(pListHead->Next);
					delete pListHead;
					pListHead = pNextItem;
					}
				}
			//
			//	Add all of the database fields to our linked list of CFlds
			//
			pListHead = new CFLD_ITEM;
			pListHead->Next = NULL;
			PCFLD_ITEM pCurItem = pListHead;
			PCFLD_ITEM pNewItem;
			int i = 0, irc;
			do  {
				char fname[16];
				FldName(pDbf, i, fname);
				irc = pCurItem->cfld.Init( pDbf, fname );
				if (irc == DBF_SUCCESS &&
					(i + 1) < pDbf->NumFlds){
					pNewItem = new CFLD_ITEM;
					pNewItem->Next = NULL;
					
					pCurItem->Next = pNewItem;
					pCurItem = pNewItem;
					}
				i++;
				} while( i < pDbf->NumFlds && irc == DBF_SUCCESS );

			rc = DBF_SUCCESS;
			}
		}
	return (rc);
} // Open()

//////////////////////////////////////////////////////////////
// 		Open operation (overloaded function)
//
DS_LONG CDbf::Open( char* pFileName )
{
	DS_ULONG curmode = GetDbfOpenMode();
	return( Open( pFileName, curmode ));
}

//////////////////////////////////////////////////////////////
// 		Open operation (overloaded function)
//
DS_LONG CDbf::Open(  char* pFileName,  char* pIndexName )
{
	DS_LONG rc = DBF_ERROR;

	if (Open( pFileName ) == DBF_SUCCESS)
		{
		if (OpenIndex( pIndexName ) >= 0)
			{
			rc = DBF_SUCCESS;
			}
		else{
			Close();
			}
		}
	return (rc);
} // Open() - database and index
 
//////////////////////////////////////////////////////////////
// 		Open exclusive operation
//
DS_LONG CDbf::OpenExclusive(  char* pFileName )
{
	return( Open( pFileName, DBF_OPEN_EXCLUSIVE ));
}

//////////////////////////////////////////////////////////////
// 		Open index operation
//
DS_LONG CDbf::OpenIndex(  char* pIndexName )
{
    return( OpenIndex( pIndexName, AUTO_INDEX ) );
} // OpenIndex()

//////////////////////////////////////////////////////////////
// 		Open index operation - overloaded
//
DS_LONG CDbf::OpenIndex(  char* pIndexName, int iType )
{
	DS_LONG rc = DBF_ERROR;

	if (pDbf != NULL) { // check to see if valid handle
		rc = (DS_LONG)IndexOpen(pDbf, pIndexName, iType);
		}
	return (rc);
} // OpenIndex()

//////////////////////////////////////////////////////////////
// 		Pack operation
//
DS_LONG CDbf::Pack()
{
	return( CDbf::Pack( FALSE ) );
}

//////////////////////////////////////////////////////////////
// 		Pack operation (overloaded function)
//
DS_LONG CDbf::Pack(DS_LONG PackMemo)
{
	DS_LONG rc = DbfPackAndReindex( &pDbf, (DS_SHORT)PackMemo );
	if (rc == DBF_SUCCESS)
		{
		// Since pack was successful, we must reload the
		// dbf handle back into our CFld list.
			PCFLD_ITEM pCurItem = pListHead;
			char fname[16];
			int i = 0;
            while (pCurItem && i < pDbf->NumFlds)
			{
				FldName(pDbf, i++, fname);
				pCurItem->cfld.Init( pDbf, fname );
                pCurItem = (PCFLD_ITEM)pCurItem->Next;
            } // endwhile 

		}  // endif
	return( rc );
}

//////////////////////////////////////////////////////////////
// 		Sort operation 
//
DS_LONG CDbf::Sort(  char* pFileOut )
{
    DS_LONG rc = DBF_ERROR;
    if (iFieldsToSort > 0)
    {
        rc = DbfSort(pDbf, pFileOut, &iSortRef[0], 
                     iFieldsToSort, &iDirection[0] );
    }
return (rc);
}

//////////////////////////////////////////////////////////////
// 		SortField operation - returns number of fields to sort
//
DS_LONG CDbf::SortField( DS_LONG fref, DS_LONG direction )
{
    if (iFieldsToSort < MAX_SORT_FIELDS)
    {
        iSortRef[iFieldsToSort] = (DS_SHORT)fref;
        iDirection[iFieldsToSort] = (DS_SHORT)direction;
        return (++iFieldsToSort);
    }

    return (DBF_ERROR);
}

//////////////////////////////////////////////////////////////
// 		SortField operation - returns number of fields to sort
//      overloaded function
DS_LONG CDbf::SortField( PDS_CHAR pFldname, DS_LONG direction )
{
    if (iFieldsToSort < MAX_SORT_FIELDS)
    {
        iSortRef[iFieldsToSort] = (DS_SHORT)FldRef(pDbf,pFldname);
        iDirection[iFieldsToSort] = (DS_SHORT)direction;
        return (++iFieldsToSort);
    }

    return (DBF_ERROR);
}

//////////////////////////////////////////////////////////////
// 		SortField operation - returns number of fields to sort
//      overloaded function
DS_LONG CDbf::SortField( CFld cf, DS_LONG direction )
{
    if (iFieldsToSort < MAX_SORT_FIELDS && cf.sFldRef >= 0)
    {
        iSortRef[iFieldsToSort] = cf.sFldRef;
        iDirection[iFieldsToSort] = (DS_SHORT)direction;
        return (++iFieldsToSort);
    }

    return (DBF_ERROR);
}

//////////////////////////////////////////////////////////////
// 		SortReset operation 
//
void CDbf::SortReset()
{
    // init sort fields and direction
    for( int i=0; i < MAX_SORT_FIELDS; i++)
    {
        iSortRef[i] = -1; iDirection[i] = SORT_ASCEND;
    }
    iFieldsToSort = 0;
}

//////////////////////////////////////////////////////////////
// 		Soundex operation 
//
char* CDbf::Soundex( char* pString,  char* pSCode)
{
return( DbfSoundex(pString, pSCode) );
}

//////////////////////////////////////////////////////////////
// 		Soundex operation (overload)
//
char* CDbf::Soundex( char* pString)
{
return( DbfSoundex(pString, TmpSoundexBuffer) );
}

//////////////////////////////////////////////////////////////
// 		Add Field operation - This operation will add to the
//      internal field list.  This will be used to define
//      all fields before calling Create.
//
DS_LONG CDbf::AddField(  char* pFldName, CHAR cType,
					 int Width, int Dec )
{
	//DS_LONG rc = DBF_ERROR;
	DS_LONG rc = DBF_SUCCESS;
	PCFLD_ITEM pItem = pListHead;
	PCFLD_ITEM pNewItem = new CFLD_ITEM;
	pNewItem->Next = NULL;
	//
	// Add new item to end of list
	//
	if (pItem){
		while (pItem->Next){
			pItem = (PCFLD_ITEM)(pItem->Next);
			}
		pItem->Next = pNewItem;
		}
	else { 
		// List head must be null, so lets start one...
		pListHead = pItem = pNewItem;
		}

	//
	// Make sure field name is upper case
	//
	strupr( pFldName );

	//
	// Set values that were passed in...
	//
	strcpy( pNewItem->cfld.szFieldName,
			pFldName);
	pNewItem->cfld.cFldType = cType;
	pNewItem->cfld.sFldLen  = Width;
	pNewItem->cfld.sFldDec  = Dec;

	strcpy( pNewItem->cfld.szNewFieldName,
			pFldName);
	pNewItem->cfld.cNewFldType = cType;
	pNewItem->cfld.sNewFldLen  = Width;
	pNewItem->cfld.sNewFldDec  = Dec;

	pNewItem->cfld.bModified = TRUE;

	return (rc);

} // AddField (singular)

//////////////////////////////////////////////////////////////
// 	Overloaded function: AddField 
//  
//  Takes pointer to array of DBFFLDs as argument
//
DS_LONG CDbf::AddField(  PDBFFLD pFldList, int iFldCnt )
{
	PDBFFLD pCur = pFldList;
	int i;
	for (i = 0; i < iFldCnt; i++)
		{
		if(	AddField( 	pFldList[i].Name,
						pFldList[i].Type,
						pFldList[i].Length,
						pFldList[i].Dec
					) != DBF_SUCCESS)
			{
			return DBF_ERROR;
			}
		}
return DBF_SUCCESS;
}
//////////////////////////////////////////////////////////////
// 	Overloaded function: AddField 
//  
//  Takes CFld as argument
//
DS_LONG CDbf::AddField(  CFld newFld )
{
	return( AddField( newFld.szNewFieldName,
					  (CHAR)newFld.cNewFldType,
					  newFld.sNewFldLen,
					  newFld.sNewFldDec
					 ));
}
//////////////////////////////////////////////////////////////
// 		Get CFld pointer for a field
//
CFld* CDbf::GetField(  char* pFieldName )
{
	PCFLD_ITEM pCur = pListHead;
	while (pCur)
		{
		if (strcmpi( pFieldName, pCur->cfld.szFieldName) == 0)
			{
			return( &pCur->cfld );
			}
		pCur = (PCFLD_ITEM)pCur->Next;
		}

	//
	// if not found, try looking at new field names
	//
	pCur = pListHead;
	while (pCur)
		{
		if (strcmpi( pFieldName, pCur->cfld.szNewFieldName) == 0)
			{
			return( &pCur->cfld );
			}
		pCur = (PCFLD_ITEM)pCur->Next;
		}

	return NULL;
}

//////////////////////////////////////////////////////////////
// 		Get CFld pointer for a field by ref (overloaded func)
//
CFld* CDbf::GetField( DS_LONG iFieldRef )
{
	PCFLD_ITEM pCur = pListHead;
	while (pCur)
		{
		if (iFieldRef == (DS_LONG)pCur->cfld.sFldRef) 
			{
			return( &pCur->cfld );
			}
		pCur = (PCFLD_ITEM)pCur->Next;
		}

	return NULL;
}

//////////////////////////////////////////////////////////////
//  
//  Open a database
//
DS_LONG CDbf::Use( char* pFilename )
{
	return( Open( pFilename ));
}

//////////////////////////////////////////////////////////////////////////
//
//  CLASS CFld 
//
//////////////////////////////////////////////////////////////////////////

CFld::CFld( )    // standard construction
{
	pFldPtr = NULL;
	pMemoBuf = NULL;
	sFldRef = sFldLen = sFldDec = -1;
    cFldType = -1;
	szFieldName[0] = 0x0;
	sNewFldLen = sNewFldDec = -1;
    cNewFldType = -1;
	szNewFieldName[0] = 0x0;
	bRemove = FALSE;
	bModified = FALSE;
    pDbf = NULL;
}

//////////////////////////////////////////////
//  initialized construction
//
CFld::CFld( PDBF pDbfHandle,  char* pFldName )
{
	Init( pDbfHandle, pFldName );
}

//////////////////////////////////////////////
//  enhanced construction - use these constructors
//  when setting up fields to be added to a
//  CDbf object via its AddField member.
//
CFld::CFld( PDBFFLD pFldDef )
{
	SetFieldName( pFldDef->Name );
    SetFieldType( pFldDef->Type );
    SetFieldLength( pFldDef->Length );
    SetFieldDecimals( pFldDef->Dec );
}

CFld::CFld( char* pFldName, char cType, int Length, int Dec )
{
	SetFieldName( pFldName );
    SetFieldType( cType );
    SetFieldLength( Length );
    SetFieldDecimals( Dec );
}

//////////////////////////////////////
// CFld destructor
//
CFld::~CFld( )
{
	if (pMemoBuf)
		{
		MemoFreeBuffer(pDbf, sFldRef);
		}
}

////////////////////////////////////////////////
// Initialization of CFld
DS_LONG CFld::Init( PDBF pDbfHandle,  char* pFldName )
{
	DS_LONG rc = DBF_ERROR;
	pDbf = pDbfHandle;

	sFldRef = FldRef(pDbf, pFldName);
	if (sFldRef == DBF_ERROR)
		{
		return DBF_ERROR;
		}

	pFldPtr = FldPtr( pDbf, sFldRef );
	sFldLen = FldWidth( pDbf, sFldRef );
	sFldDec = FldDecimals( pDbf, sFldRef );
	cFldType= FldType( pDbf, sFldRef );
	FldName( pDbf, sFldRef, szFieldName );

	sNewFldLen = sFldLen;
	sNewFldDec = sFldDec;
	cNewFldType   = cFldType;
	strcpy(szNewFieldName, szFieldName);

	pMemoBuf= NULL;
	if 	( cFldType == MEMO_FLD )
		{
		// allocate a memo buffer if memo field
		pMemoBuf = MemoAllocBuffer( pDbf, sFldRef, MEMO_BUF_SIZE);

		// force update to newly allocated buffer
		DbfTop(pDbf);
		}
	rc = DBF_SUCCESS;

	bRemove = FALSE;
	
	return (rc);
}

///////////////////////////////////////////
// Change the field's name (requires exclusive
// open of the database)
//
DS_LONG CFld::ChangeName(  char* pNewName )
{
	DS_LONG rc;
	rc =  FldChangeName( pDbf, sFldRef, pNewName );
	if ( rc == DBF_SUCCESS )
		{
		FldName( pDbf, sFldRef, szFieldName );		
		strcpy(szNewFieldName, szFieldName );
		}

	return( rc );
}

///////////////////////////////////////////
// Mark this field to be removed when the
// database is modified with CDbf::ModifyStructure()
//
void CFld::DeleteField( )
{
	bRemove = TRUE;
}

///////////////////////////////////////////////
// Retrieve the formatted date of a date field
//
 char* CFld::GetDate(  char* pDateBuf, int nBufsize )
{
	if (nBufsize > sFldLen)
		{
		FldCopyDate( pDbf, sFldRef, pDateBuf );
		}
	return (pDateBuf);
}

///////////////////////////////////////////////
// Retrieve the field's name into user buffer
//
char* CFld::FieldName(  char* pNameBuf, int nBufsize )
{
	if (nBufsize > 10)
		{
		strcpy(pNameBuf, szFieldName);
		}
	return (pNameBuf);
}

///////////////////////////////////////////////
// Retrieve the field's name
//
char* CFld::FieldName()
{
	return (szFieldName);
}

///////////////////////////////////////////////
// Retrieve the field's new name
//
 char* CFld::GetNewFieldName(  char* pNameBuf, int nBufsize )
{
	if (nBufsize > 10)
		{
		strcpy(pNameBuf, szNewFieldName);
		}
	return (pNameBuf);
}

///////////////////////////////////////////
// Retrieve the value of a num field
//
 char* CFld::GetString(  char* pStrBuf, int nBufsize )
{
	if (cFldType != MEMO_FLD)
		{
		if (nBufsize >= sFldLen)
			{
			FldCopy( pDbf, sFldRef, pStrBuf);
			}
		else // set to null if buffer not big enough
			*pStrBuf = 0x0;
		}
	else
		{ // must be memo type field, get the memo data
		memcpy(pStrBuf, pMemoBuf, 
		   min(nBufsize, MEMO_BUF_SIZE));
		}
	return (pStrBuf);
}

///////////////////////////////////////////
// Retrieve the value of a num field
//
double CFld::GetValue( )
{
	switch(cFldType)
    {
    case NUMERIC_FLD:
	    #ifdef BCCWIN
		    double dtmp;
		    SFldValue(pDbf, sFldRef, &dtmp);
		    return (dtmp);
	    #else
		    return( FldValue( pDbf, sFldRef) );
	    #endif
        break;
    case CHARACTER_FLD:
        return( CvtAsciiToDouble(pFldPtr,sFldLen) );
        break;
    case LOGICAL_FLD:
        return((FldTrue(pDbf,sFldRef) ? 1.0 : 0.0));
        break;
    default:
        break;
    }
return 0.0;
}

///////////////////////////////////////////
// Set a date field 
//
DS_LONG  CFld::SetDate( char* pDate)
{ 
    return((DS_LONG)FldReplaceDate(pDbf, sFldRef, pDate));
}

///////////////////////////////////////////
// Overloaded function.
// Set a date field with a specific format
//
DS_LONG  CFld::SetDate( char* pDate, char* pFormat)
{ 
    char dbDate[9];
    if (CvtCharToDate( pDate, pFormat, dbDate) == NULL)
        return DBF_ERROR;

    return((DS_LONG)FldReplace(pDbf, sFldRef, dbDate));
}

///////////////////////////////////////////
// Set a logical field to FALSE
//
void CFld::SetFalse( )
{
	DS_SHORT bool = FALSE;
	FldReplaceLog(pDbf, sFldRef, &bool);
	return;
}

///////////////////////////////////////////
// Set a logical field to TRUE
//
void CFld::SetTrue( )
{
	DS_SHORT bool = TRUE;
	FldReplaceLog(pDbf, sFldRef, &bool);
	return;
}


////////////////////////////////////////////////////////////
//   SetString - replaces the character field or date field
//				 or memo buffer
DS_LONG CFld::SetString(  char* pBuf )
{
	switch (cFldType)
		{
		case CHARACTER_FLD:
			FldReplace(pDbf, sFldRef, pBuf);
			break;
		case LOGICAL_FLD:
		case NUMERIC_FLD:
            SetValue( pBuf );
			break;
		case DATE_FLD:
			FldReplaceDate(pDbf, sFldRef, pBuf);
			break;
		case MEMO_FLD:
			if (pBuf != pMemoBuf && pMemoBuf != NULL)
			{
				memcpy(pMemoBuf, pBuf,
					min( strlen(pBuf), MEMO_BUF_SIZE ));
			}
			break;
		default:
			break;
		}
return DBF_SUCCESS;
}

///////////////////////////////////////////
// Set the value of a num field
// This is the same as using the equal operator
//
void CFld::SetValue( double dVal )
{
	switch (cFldType)
	{
	case NUMERIC_FLD:
		FldReplaceNum(pDbf, sFldRef, &dVal);
		break;
	case CHARACTER_FLD:
		{
		char* pBuf = new char[sFldLen + 1];
		CvtDoubleToAscii( dVal, sFldLen, sFldDec, pBuf);
		FldReplace(pDbf, sFldRef, pBuf);
		delete []pBuf;
		}
		break;
	case LOGICAL_FLD:
		FldReplaceLog(pDbf, sFldRef, (PDS_SHORT)&dVal);
		break;
	default:
		break;
	}
return;
} // SetValue()

///////////////////////////////////////////
// Set the value of a field (overloaded)
// This is the same as using the equal operator
//
void CFld::SetValue(  char* pBuf )
{
	switch (cFldType)
	{
	case CHARACTER_FLD:
		FldReplace(pDbf, sFldRef, pBuf);
		break;
	case DATE_FLD:
		FldReplaceDate(pDbf, sFldRef, pBuf);
		break;
	case NUMERIC_FLD:
		{
		DS_SHORT ilen = (DS_SHORT)min( sFldLen, (DS_SHORT)strlen(pBuf));
		FldReplaceDirect(pDbf, sFldRef, pBuf, ilen);
		}
		break;
	case LOGICAL_FLD:
		{
		DS_SHORT bool = 0;
		if (*pBuf == 'T' || *pBuf == 'Y' || *pBuf == '1' ||
			*pBuf == 't' || *pBuf == 'y')
			bool = 1;
		FldReplaceLog(pDbf, sFldRef, &bool);
		}
		break;
	case MEMO_FLD:
		if (pBuf != pMemoBuf && pMemoBuf != NULL)
		{
			memcpy(pMemoBuf, pBuf,
				min( strlen(pBuf), MEMO_BUF_SIZE ));
		}
		break;
	default:
		break;
	}
return;
} // SetValue( char* ) - overloaded

///////////////////////////////////////////////
// EQUAL operator
//
// This operator is multipurpose.  It will 
// replace the field data differently based
// on what the field type is.  
// Input is a pointer to a character buffer.
//
void CFld::operator= (  char* pBuf )
{
	switch (cFldType)
	{
	case CHARACTER_FLD:
		FldReplace(pDbf, sFldRef, pBuf);
		break;
	case DATE_FLD:
		FldReplaceDate(pDbf, sFldRef, pBuf);
		break;
	case NUMERIC_FLD:
		{
		DS_SHORT ilen = (DS_SHORT)min( sFldLen, (DS_SHORT)strlen(pBuf));
		FldReplaceDirect(pDbf, sFldRef, pBuf, ilen);
		}
		break;
	case LOGICAL_FLD:
		{
		DS_SHORT bool = 0;
		if (*pBuf == 'T' || *pBuf == 'Y' || *pBuf == '1' ||
			*pBuf == 't' || *pBuf == 'y')
			{bool = 1;}
		FldReplaceLog(pDbf, sFldRef, &bool);
		}
		break;
	case MEMO_FLD:
		if (pBuf != pMemoBuf && pMemoBuf != NULL)
		{
			memcpy(pMemoBuf, pBuf,
				min( strlen(pBuf), MEMO_BUF_SIZE ));
		}
		break;
	default:
		break;
	}
return;
}

///////////////////////////////////////////////
// EQUAL operator (overloaded) - numeric
//
// This operator is multipurpose.  It will 
// replace the field data differently based
// on what the field type is.
// Input is a double value.
//
void CFld::operator= ( double dVal )
{
	switch (cFldType)
	{
	case NUMERIC_FLD:
		FldReplaceNum(pDbf, sFldRef, &dVal);
		break;
	case CHARACTER_FLD:
		{
		char* pBuf = new char[sFldLen + 1];
		CvtDoubleToAscii( dVal, sFldLen, sFldDec, pBuf);
		FldReplace(pDbf, sFldRef, pBuf);
		delete []pBuf;
		}
		break;
	case LOGICAL_FLD:
		FldReplaceLog(pDbf, sFldRef, (PDS_SHORT)&dVal);
		break;
	default:
		break;
	}
return;
}

/////////////////////////////////////////////////////
// Set the new field name
//
// returns 0 for success and -1 for failure.
DS_LONG CFld::SetFieldName(  char* pFieldName )
{
	int bytesToCopy = min(10, strlen(pFieldName));
	memset(szNewFieldName, 0x0, sizeof(szNewFieldName));
	memcpy(szNewFieldName, pFieldName, bytesToCopy);
	//
	// copy to normal field name buffer if it is empty
	//
	if (strlen(szFieldName) == 0)
		{
		strcpy(szFieldName, szNewFieldName);
		}

bModified = TRUE;
return DBF_SUCCESS;
}
/////////////////////////////////////////////////////
// Set the new field Type
// This will also set the width and decimals to some
// default values.
//
// returns 0 for success and -1 for failure.
DS_LONG CFld::SetFieldType( DS_CHAR iFldType )
{
	switch(iFldType){
		case CHARACTER_FLD:
			cNewFldType = iFldType;
			sNewFldLen  = 20;
			sNewFldDec  = 0;
			break;
		case NUMERIC_FLD:
			cNewFldType = iFldType;
			sNewFldLen  = 8;
			sNewFldDec  = 2;
			break;
		case LOGICAL_FLD:
			cNewFldType = iFldType;
			sNewFldLen  = 1;
			sNewFldDec  = 0;
			break;
		case DATE_FLD:
			cNewFldType = iFldType;
			sNewFldLen  = 8;
			sNewFldDec  = 0;
			break;
		case MEMO_FLD:
			cNewFldType = iFldType;
			sNewFldLen  = 10;
			sNewFldDec  = 0;
			break;
		default:
			return DBF_ERROR;
		}
	//
	// set normal type and length if un-initialized
	//
	if (cFldType == -1)
		{
		cFldType = cNewFldType;
		sFldLen  = sNewFldLen;
		sFldDec  = sNewFldDec;
		}

bModified = TRUE;
return DBF_SUCCESS;
}
/////////////////////////////////////////////////////
// Set the field Width
//
// returns 0 for success and -1 for failure.
DS_LONG CFld::SetFieldLength(  int iWidth )
{
	switch(cNewFldType){
		case CHARACTER_FLD:
		case NUMERIC_FLD:
			sNewFldLen  = iWidth;
			break;
		case LOGICAL_FLD:
		case DATE_FLD:
		case MEMO_FLD:
		 	// cannot set width of these types
			break;
		default:
			return DBF_ERROR;
		}
	//
	// set normal length if un-initialized
	//
	if (sFldLen == -1)
		{
		sFldLen  = sNewFldLen;
		}

bModified = TRUE;
return DBF_SUCCESS;
}
/////////////////////////////////////////////////////
// Set the field decimals
//
// returns 0 for success and -1 for failure.
DS_LONG CFld::SetFieldDecimals(  int iDec )
{
	switch(cNewFldType){
		case CHARACTER_FLD:
		case NUMERIC_FLD:
			sNewFldDec  = iDec;
			break;
		case LOGICAL_FLD:
		case DATE_FLD:
		case MEMO_FLD:
		 	// cannot set decimals of these types
			break;
		default:
			return DBF_ERROR;
		}
	//
	// set normal decimals if un-initialized
	//
	if (sFldDec == -1)
		{
		sFldDec  = sNewFldDec;
		}

bModified = TRUE;
return DBF_SUCCESS;
}
//////////////////////////////////////////////////////////////////////////
//
//  CLASS CdbfLIB
//
//////////////////////////////////////////////////////////////////////////

CDbfLib::CDbfLib( )    // standard construction
{
	unsigned char maj, min, rev;
	unsigned short env;
	dbfLIBVersionAndRev(&maj, &min, &rev, &env);
	VerStr[0]=maj+0x30;
	VerStr[1]='.';
	VerStr[2]=min+0x30;
	VerStr[3]=rev+0x30;
	VerStr[4]=0x0;

}

//////////////////////////////////////
// CdbfLIB destructor
//
CDbfLib::~CDbfLib( )
{
}

////////////////////////////////////////////////
// Library version
char* CDbfLib::Version()
{
	return( &VerStr[0] );
}

////////////////////////////////////////////////
// Set operation
DS_ULONG CDbfLib::Set(DS_ULONG Setting, DS_ULONG SetValue)
{
	DS_ULONG rc = 0;
	switch (Setting){
		case ALLOW_NULLS:
			rc = ::SetAllowNulls(SetValue);
			break;
		case AUTO_PACK:
			rc = ::SetAutoPack(SetValue);
			break;
		case CENTURY:
			rc = ::SetCentury(SetValue);
			break;
		case DATE_FORMAT:
            break;
		case OPEN_MODE:
			rc = ::SetDbfOpenMode(SetValue);
			break;
		case BLOCK_SIZE:
			rc = ::SetDefaultBlockSize((DS_SHORT)SetValue);
			break;
		case LOCK_TYPE:
			rc = ::SetDefaultLockType((DS_SHORT)SetValue);
			break;
		case PADDING:	
			rc = ::SetPadding((DS_UCHAR)SetValue);
			break;
		case UNIQUE:	
			rc = ::SetUnique((DS_UCHAR)SetValue);
			break;
		case DELETED:
			rc = ::DbfSetDeleted((DS_SHORT)SetValue);
			break;
		}
return (rc);
}

////////////////////////////////////////////////
// Set operation - overloaded
// Only sets date format...
//
DS_ULONG CDbfLib::Set(DS_ULONG Setting, PDS_CHAR pDateFmt)
{
    if (Setting == DATE_FORMAT)
       return((DS_ULONG)::SetDbfDateFormat(pDateFmt));
return 0;
}

////////////////////////////////////////////////
// Get operation
DS_ULONG CDbfLib::Get(DS_ULONG Setting)
{
	DS_ULONG rc = 0;
	switch (Setting){
		case ALLOW_NULLS:
			rc = ::GetAllowNulls();
			break;
		case AUTO_PACK:
			rc = ::GetAutoPack();
			break;
		case CENTURY:
			rc = ::GetCentury();
			break;
		case DATE_FORMAT: // see overloaded function for date format
            break;
		case OPEN_MODE:
			rc = ::GetDbfOpenMode();
			break;
		case BLOCK_SIZE:
			rc = ::GetDefaultBlockSize();
			break;
		case LOCK_TYPE:
			rc = ::GetDefaultLockType();
			break;
		case PADDING:	
			rc = ::GetPadding();
			break;
		case UNIQUE:	
			rc = ::GetUnique();
			break;
		case DELETED:
			// rc = ::DbfGetDeleted();
			break;
		}
return (rc);
}
////////////////////////////////////////////////
// Get operation - overloaded function
// Only gets date format
// Returns strlen of date format.
DS_ULONG CDbfLib::Get( DS_ULONG Setting, PDS_CHAR pBuf )
{
    if (Setting == DATE_FORMAT)
    {
        ::GetDbfDateFormat(pBuf);
        return((DS_ULONG)strlen(pBuf));
    }
return 0;
}
/****** End of dbfClass.Cpp *********/

