/*****************************************************************************/
/*
 *  exceptq.c - exported entrypoints and supporting code
 *
 *  Parts of this file are
 *    Copyright (c) 2000-2010 Steven Levine and Associates, Inc.
 *    Copyright (c) 2010-2011 Richard L Walsh
 *
 */
/*****************************************************************************/
/*
 * DLL containing an exception handler for gathering trap information.
 * This DLL dumps all important debugging Data and is accessible
 * from both 16 bit and 32 bit programs.
 *
 * DLL is 32-bit code, but can handle exceptions from 16-bit code.
 *
 * Symbolic data can be extracted from
 *   Embedded debug data (VAC HLL or CodeView format)
 *   .dbg files (VAC HLL or CodeView format)
 *   .xqs files (generated by mapxqs)
 *   .sym files (generated by mapsym)
 *
 * fixme to have xlseek with error checking
 * fixme to have xread with error checking
 * fixme to use DosAllocMem rather than calloc/free -
 *          avoids death if heap is corrupted
 * fixme to merge CodeView debug reader into HLL debug reader - saves space
 *
 */
/*****************************************************************************/
/*
 * Version: 2.2             |   Marc Fiammante (marc.fiammante@free.fr)
 *                          |   La Gaude FRANCE
 * Version: 5.0             |   John Currier  (JCURRIER at CLTVM1)
 *                          |   Internet: currier@vnet.ibm.com
 * Version: 6.0             |   Kim Rasmussen (krasmus@ibm.net)
 *                          |   Denmark
 * Version: 6.1             |   Anthony Cruise (CRUISE at YKTVMH)
 *                          |   Watson Research
 * Version: 6.2             |   John Currier  (JCURRIER at CLTVM1)
 *                          |   Internet: currier@vnet.ibm.com
 * Version: 6.3             |   Kim Rasmussen (kr@belle.dk)
 *                          |   Denmark
 *                          |   Marc Fiammante (marc.fiammante@free.fr)
 *                          |   La Gaude FRANCE
 * Version: 6.4             |   Kim Rasmussen (kr@belle.dk)
 *                          |   Denmark - http://www.belle.dk/kr/
 * Version: 6.5             |   Steven Levine (steve53@earthlink.net)
 * Version: 6.6             |   Steven Levine (steve53@earthlink.net)
 * Version: 7.0             |   Rich Walsh    (rich@e-vertise.com)
 * Version: 7.1             |   Rich Walsh    (rich@e-vertise.com)
 */
/*****************************************************************************/
/*
 * History:
 * --------
 *
 * created: Marc Fiammante December 1992
 * changed: John Currier   August   1994
 * changed: Kim Rasmussen, May 1995
 *          Dump of auto-variables added (32-bit only)
 * changed: Anthony Cruise, May 1995
 *          Do not dump duplicate lines  (32-bit only)
 * changed: Marc Fiammante thanks to Bill Siddall
 *          Dump of auto-variables wrong values
 * changed: John Currier fix stack thunk on non fatal exceptions
 *          fixed DosQueryMem Size test on return
 * changed: Marc Fiammante find stack bottom from top to bottom
 *          avoid traps on disassemble
 * changed: Support for VisualAge C added (new debug format)
 * changed: Steven Levine Jul 2000
 *          Read32PmDebug: avoid autovar_def overflow trap
 *          Read32PmDebug: avoid locating wrong function vars
 *                         use symproc.length to bound lookup
 * changed: Steven Levine Aug 2005
 *          DosQueryModFromEIP Rework decl to match toolkit
 * changed: Steven Levine   Aug 2005
 *          Reformat with indent.exe
 *          Correct source filename decode
 * changed: Steven Levine   Jan 2006
 *          Enhance 32-bit detect
 * changed: Steven Levine   Jul 2007
 *          More 32-bit rework.  More 16-bit support
 * changed: Steven Levine   Aug 2007
 *          Generate hex trap file name
 *          Show PID in hex/dec
 *          Show exceptq build date
 * changed: Steven Levine   Oct 2007
 *          List labels on stack
 *          Show more in hex/dec
 * changed: Steven Levine   Dec 2007
 *          Show exceptq version
 * changed: Steven Levine   Feb 2008
 *          Use bsedos.h
 *          Use more omf.h
 * changed: Steven Levine   Apr 2008
 *          Version 6.6
 *          Show stack as dwords or words
 * changed: Steven Levine   May 2008
 *          Version 6.7
 *          Support oversized (>64KiB) symbol files
 *          Support stripped LX executables
 * changed: Steven Levine   May 2008
 *          Version 6.8
 *          Support Borland OA debug format - untested
 * changed: Steven Levine   Feb 2009
 *          Version 6.8
 *          Print blocks pointed to by registers
 * changed: Steven Levine   Dec 2009
 *          Version 6.8
 *          Correct EDI display value
 *          Beep while dumping memory
 * changed: Steven Levine   Apr 2010
 *          Version 6.9
 *          Rework recursion logic to try to report exceptq traps
 * changed: Steven Levine   Apr 2010
 *          Version 6.10
 *          Avoid buffer overflow with long C++ symbols
 * changed: Rich Walsh      May & June 2010
 *          Version 7.0
 *          Reformatted, restructured, or rewrote virtually everything.
 *          Reformatted exception report.
 *          Extended .sym file support to files with > 1mb symbol data
 *            and > 64k symbols.
 *          Fixed errors calculating accessible top of stack.
 *          Removed support for OS/2 versions < 2.1.
 *          Replaced the infernal beeping with less stress-inducing sounds.
 *          Split source into multiple files and headers.
 *          Added runtime configuration options.
 * changed: Rich Walsh      Jan & Feb 2011
 *          Version 7.1
 *          Added support for symbol files in XQS format.
 *          Added app-supplied build ID or descriptive text to report.
 *          Added EXCEPTQ_DEBUG_EXCEPTION support.
 *          Added memory & multi-processor info to report header.
 *          Reformatted exception report to remove box-drawing characters.
 */
/*****************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#define INCL_DOS
#include <os2.h>

#include "exq.h"

/*****************************************************************************/
/** 16-bit stuff */

APIRET16 APIENTRY16 DOSQPROCSTATUS(ULONG * _Seg16 pBuf, USHORT cbBuf);
APIRET16 APIENTRY16 DOS16ALLOCSEG(USHORT cbSize,        /* number of bytes requested */
                                  USHORT * _Seg16 pSel, /* sector allocated (returned) */
                                  USHORT fsAlloc);      /* attributes requested */

/*****************************************************************************/

/** exceptq options **/
BOOL    fReport = TRUE;
BOOL    fBeep = TRUE;
BOOL    fReportInfo = FALSE;
BOOL    fDebug = FALSE;
ULONG   verbosity = XQ_CONCISE;
char    szReportInfo[80] = "";

static BOOL   fInitOpts = FALSE;
static BOOL   fInForceExit = FALSE;

/*****************************************************************************/

void    InitReportInfo(const char* pszReportInfo);
void    WakeThreads(void);

/*****************************************************************************/
/*  The Exception Handler                                                    */
/*****************************************************************************/
/* MyHandler() - per-thread exception handler */

ULONG APIENTRY MYHANDLER(EXCEPTIONREPORTRECORD* pExRepRec,
                         EXCEPTIONREGISTRATIONRECORD* pExRegRec,
                         CONTEXTRECORD* pCtxRec,
                         void* p)
{
  if ((pExRepRec->ExceptionNum & XCPT_SEVERITY_CODE) != XCPT_FATAL_EXCEPTION) {
    if (pExRepRec->ExceptionNum == EXCEPTQ_DEBUG_EXCEPTION) {
      ReportException(pExRepRec, pExRegRec, pCtxRec, TRUE);
      if (!(pExRepRec->fHandlerFlags & EH_NONCONTINUABLE))
        return XCPT_CONTINUE_EXECUTION;
    }
    return XCPT_CONTINUE_SEARCH;
  }

  if (pExRepRec->ExceptionNum == XCPT_UNWIND ||
      pExRepRec->ExceptionNum == XCPT_SINGLE_STEP ||
      pExRepRec->ExceptionNum == XCPT_PROCESS_TERMINATE ||
      pExRepRec->ExceptionNum == XCPT_ASYNC_PROCESS_TERMINATE ||
      pExRepRec->ExceptionNum == XCPT_SIGNAL)
    return XCPT_CONTINUE_SEARCH;

  if (fInForceExit)
    return XCPT_CONTINUE_SEARCH;

  /* If not one of the above handle here */
  ReportException(pExRepRec, pExRegRec, pCtxRec, FALSE);

  return XCPT_CONTINUE_SEARCH;
}

/*****************************************************************************/
/*  Install Exceptq - 32-bit entrypoint                                      */
/*****************************************************************************/
/* Note:  UninstallExceptq(pExRegRec) is implemented as a macro
 *        that resolves to DosUnsetExceptionHandler(pExRegRec).
 */

APIRET APIENTRY InstallExceptq(EXCEPTIONREGISTRATIONRECORD* pExRegRec,
                               const char* pszOptions,
                               const char* pszReportInfo)
{
  BOOL   fInitReportInfo = FALSE;

  /* Init options & possibly report info if they haven't already been set.
   * Note:  to ensure compatibility with Exceptq v7.0 which didn't support
   * pszReportInfo, it is only examined if an 'I' appears in the supplied
   * options.  SetExceptqOptions() which was introduced in v7.1 doesn't
   * require that option to be set.
   */
  InitOptions(pszOptions, &fInitReportInfo);
  if (fInitReportInfo)
    InitReportInfo(pszReportInfo);

  /* Set the exception handler. */
  pExRegRec->prev_structure = 0;
  pExRegRec->ExceptionHandler = &MYHANDLER;

  return DosSetExceptionHandler(pExRegRec);
}

/*****************************************************************************/
/* Enables apps to set/reset options & report info after
 * InstallExceptq() has been called.
 */

void   APIENTRY SetExceptqOptions(const char* pszOptions,
                                  const char* pszReportInfo)
{
  /* Reset & reinit options if new ones were provided. */
  if (pszOptions && *pszOptions) {
    fInitOpts = FALSE;
    fReport = TRUE;
    fBeep = TRUE;
    fDebug = FALSE;
    verbosity = XQ_CONCISE;
    InitOptions(pszOptions, NULL);
  }

  /* Reinit report info if it was provided. */
  InitReportInfo(pszReportInfo);

  return;
}

/*****************************************************************************/
/* Init report options - called by InstallExceptq() & ReportFatalException() */

void    InitOptions(const char* pszOptions, BOOL* pfInitReportInfo)
{
  BOOL    fUserOpts = FALSE;
  char    last = 0;
  char *  ptr;

  if (fInitOpts)
    return;

  fInitOpts = TRUE;
  *szBuffer = 0;

  /* Start with options set by the app. */
  if (pszOptions)
    strcpy(szBuffer, pszOptions);

  /* Let them be overridden by the environment variable - users rule!
   * Note:  insert a separator between the strings to ensure that a
   * 'v' or 't' from each source doesn't become a 'vv' or 'tt'.
   */
  if (!DosScanEnv("EXCEPTQ", &ptr)) {
    strcat(szBuffer, "|");
    strcat(szBuffer, ptr);
  }

  /* Avoid a runaway loop. */
  szBuffer[32] = 0;

  /* Scan for options, ignore undefined characters. */
  for (ptr = szBuffer; *ptr; ptr++) {
    switch (*ptr) {

      /* verbose ('v') or very verbose ('vv') */
      case 'v':
      case 'V':
        verbosity = (last == 'v') ? XQ_VERYVERBOSE : XQ_VERBOSE;
        last = 'v';
        break;

      /* terse ('t') or terribly terse ('tt') */
      case 't':
      case 'T':
        verbosity = (last == 't') ? XQ_TERRIBLYTERSE : XQ_TERSE;
        last = 't';
        break;

      /* concise ('c') - default */
      case 'c':
      case 'C':
        verbosity = XQ_CONCISE;
        last = 'c';
        break;

      /* separator (internal use) */
      case '|':
        last = '|';
        fUserOpts = TRUE;
        break;

      /* quiet ('q') - no beeps */
      case 'q':
      case 'Q':
        fBeep = FALSE;
        break;

      /* beep ('b') - default */
      case 'b':
      case 'B':
        fBeep = TRUE;
        break;

      /* yes ('y') - create report - default */
      case 'y':
      case 'Y':
        fReport = TRUE;
        break;

      /* no ('n') - don't create report */
      case 'n':
      case 'N':
        fReport = FALSE;
        break;

      /* debug ('d') - report debug & fatal exceptions */
      case 'd':
      case 'D':
        fDebug = TRUE;
        break;

      /* fatal ('f') - report fatal exceptions only - default */
      case 'f':
      case 'F':
        fDebug = FALSE;
        break;

      /* info ('i') - app has provided an info string;
       * note:  this option is only available if InitOptions() was called
       * by InstallExceptq() and if we're examining app-supplied options
       */
      case 'i':
      case 'I':
        if (!fUserOpts && pfInitReportInfo)
          *pfInitReportInfo = TRUE;
        break;
    }
  }

  return;
}

/*****************************************************************************/
/* Set/reset app's report info string -
 * called by InstallExceptq() & SetExceptqOptions()
 */

void    InitReportInfo(const char* pszReportInfo)
{
  size_t  cnt;
  size_t  ctr;

  /* Exit without any changes if the ptr is null or the string is null. */
  if (!pszReportInfo || !*pszReportInfo)
    return;

  /* Turn off report info. */
  fReportInfo = FALSE;
  *szReportInfo = 0;

  /* If the string is too long, it may not be a string at all, so exit */
  for (cnt = 0; cnt < sizeof(szReportInfo); cnt++) {
    if (!pszReportInfo[cnt])
      break;
  }
  if (cnt >= sizeof(szReportInfo))
      return;

  /* A string with only whitespace will erase any existing report info. */
  ctr = strspn(pszReportInfo, " \t\n\r");
  if (ctr >= cnt)
    return;

  /* Copy the string, trimming leading & trailing whitespace. */
  strcpy(szReportInfo, &pszReportInfo[ctr]);
  cnt -= ctr;
  while (cnt) {
    cnt--;
    if (!strchr(" \t\n\r", szReportInfo[cnt]))
      break;
    szReportInfo[cnt] = 0;
  }
  fReportInfo = TRUE;

  return;
}

/*****************************************************************************/
/*  Install/Remove Exceptq - 16-bit entrypoints
/*****************************************************************************/

APIRET16 APIENTRY16 SETEXCEPT(EXCEPTIONREGISTRATIONRECORD* _Seg16 pExRegRec)
{
  DosError(FERR_DISABLEEXCEPTION | FERR_DISABLEHARDERR);
  pExRegRec->prev_structure = 0;
  pExRegRec->ExceptionHandler = &MYHANDLER;

  return DosSetExceptionHandler(pExRegRec);
}

/*****************************************************************************/

APIRET16 APIENTRY16 UNSETEXCEPT(EXCEPTIONREGISTRATIONRECORD* _Seg16 pExRegRec)
{
  return DosUnsetExceptionHandler(pExRegRec);
}

/*****************************************************************************/
/*  Forced Exit                                                              */
/*****************************************************************************/
/* Force application exit by forced trap - fixme to know if this works */

void APIENTRY FORCEEXIT(void)
{
  EXCEPTIONREPORTRECORD except;
  char *  Trap = NULL;

  fInForceExit = TRUE;
  fclose(stderr);      /* I don't want error messages since all is intentional */
  DosError(FERR_DISABLEEXCEPTION | FERR_DISABLEHARDERR);
  DosEnterCritSec();
  WakeThreads();
  printf("Exiting by exception\n");
  DosSetPriority(PRTYS_THREAD, PRTYC_FOREGROUNDSERVER, PRTYD_MAXIMUM, 0);
  DosExitCritSec();
  *Trap = 0;        /* Say bye bye */

  return;
}

/*****************************************************************************/
/* WakeThreads() Wake all waiting threads fixme */

void    WakeThreads(void)
{
  APIRET    rc;
  APIRET16  rc16;
  QSPTRREC* pRec;
  QSPREC *  pProc;        /* 06 Feb 08 SHL */
  QSTREC *  pThread;      /* 06 Feb 08 SHL */
  ULONG     ListedThreads = 0;
  ULONG *   pBuf;
  USHORT    Selector;
  PTIB      ptib;
  PPIB      ppib;

  DosGetInfoBlocks(&ptib, &ppib);

  rc16 = DOS16ALLOCSEG(0xFFFF, &Selector, 0);
  if (rc16) {
    printf("DosAllocSeg Failed %hd\n", rc16);
    return;
  }

  pBuf = MAKEP(Selector, 0);
  rc16 = DOSQPROCSTATUS(pBuf, 0xFFFF);
  if (rc16) {
    printf("DosQProcStatus Failed %hd\n", rc16);
    return;
  }

  pRec = (QSPTRREC*)pBuf;
  pProc = (QSPREC*)pRec->pProcRec;
  ListedThreads = 0;
  while (ListedThreads < pRec->pGlobalRec->cThrds) {
    INT Tid;

    if (pProc->pThrdRec == NULL)
        break;
    ListedThreads += pProc->cTCB;

    if (ppib->pib_ulpid == pProc->pid) {
      for (Tid = 0; Tid < pProc->cTCB; Tid++) {
        pThread = pProc->pThrdRec + Tid;
        /* if fixme? */
        if (pThread->state == 0x09) {
          printf("Resuming Thread %d\n", (TID) pThread->tid);
          DosResumeThread((TID) pThread->tid);
        }
      }
      break;
    }

    /* 06 Feb 08 SHL */
    pProc = (QSPREC *)(
              ((PUCHAR)pProc) + sizeof(QSPREC) +
              sizeof(USHORT) * (pProc->c16Sem + pProc->cLib + pProc->cShrMem) +
              pProc->cTCB * sizeof(QSTREC));

  } /* while */

  return;
}

/*****************************************************************************/

