/* PDumpCtl - ProcDump control front-end

   Dump named process/pid or run interactive
   OK to exit to shell and update options
   FIXME to configure multiple apps
   FIXME to allow full command line control

   Copyright (c) 2011-2023 Steven Levine and Associates, Inc.
   All rights reserved.

   This program is free software licensed under the terms of the GNU
   General Public License, Version 2.  The GPL Software License can be found
   in gnugpl2.txt or at http://www.gnu.org/licenses/licenses.html#GPL

   2011-02-22 SHL Baseline - clone from PDumpCtl4.cmd
   2015-04-07 SHL Honor config.sys DUMPPROCESS
   2015-04-07 SHL Track procdump on/off and complain when needed
   2015-11-07 SHL Drop unneeded setlocal
   2016-11-30 SHL Accept more valid process names
   2018-03-08 SHL Convert to Globals style
   2018-03-08 SHL Force normal style as needed
   2018-03-23 SHL Tune messages
   2019-04-29 SHL Show version with help
   2019-08-11 SHL Sync with templates
   2019-08-11 SHL Allow _ in module name
   2019-12-28 SHL Support multiple procnames
   2020-01-07 SHL Refactor duplicated code
   2020-01-07 SHL Support mixed pids and process names
   2020-01-31 SHL Support 0t prefix for decimal - more like pmdf, was 0n
   2021-04-21 SHL Show procdump error result in hex too
   2021-06-12 SHL Correct typo
   2023-06-08 SHL Correct extended - was missing sysldr
   2023-06-08 SHL Reword extended/shared - shared is now preferred for DLLs
*/

signal on Error
signal on Failure name Error
signal on Halt
signal on NotReady name Error
signal on NoValue name Error
signal on Syntax name Error

Globals = 'gBatch gCmdName gCmds gDumpDir gDumpStyle gEnv gErrCondition',
	  'gPid gProc gProliant gTmpDir gVersion'

gVersion = '0.18 2023-06-08'

call Initialize

Main:
  parse arg cmdLine
  call ScanArgs cmdLine
  drop cmdLine

  if gBatch & gCmds == '' then
    call ScanArgsUsage 'Batch mode requires one or more commands'

  call ChkRequired
  call FindDumpDir

  /* pdumpusr option summary - see \OS2\SYSTEM\RAS\PROCDUMP.DOC
     summ       Summary for dumped threads (default)
     syssumm	Summary for all threads
     idt	Interrupt descriptor table
     laddr	Linear address range(s)
     paddr(all)	Add physical memory
     sysldr	Loader data for all processes
     sysfs	File System data for all processes (default)
     sysvm	Virtual Memory data for all processes
     systk	Task Management related data for all processes
     private	Private code and data referenced by process
     shared	Shared code and data referenced by process
     idt	Interrupt descriptor table
     instance	Instance data referenced by the process.
     mvdm	MVDM instance data for process (default)
     sysmvdm	MVDM data for all VDM and the kernel resident heap
     sem	Semaphore data for all blocked threads in process (default)
     syssem	SEM data for all blocked threads in system
     krheaps	Kernel Resident Heaps
     ksheaps	Kernel Swappable Heaps
     smp	???
     syspg	Physical and Page Memory management records (PF, VP, PTE, PDE)
     sysio	IO subsystem structures (AIRQI, DIRQ, PDD eps, PDD chain)
     trace	System trace buffers
     strace	STRACE buffer

     pdumpsys kernel defaults are
     smp, syssumm, idt, sysfs, systk, sysvm, syssem, syspg, sysio, trace, strace

     pdumpusr kernel defaults are
     summ, sysfs, mvdm, sem

  */

  if \ gBatch then
    call ShowSettings

  call DoCmds

  if gBatch then
    exit

  call AskCommands

  call ShowSettings

  exit

/* end main */

/*=== AddExtended() add instance private shared and more ===*/

AddExtended: procedure expose (Globals)
  if gDumpStyle == 'All' then do
    say
    say 'Dump style is' gDumpStyle '- resetting to Normal'
    call SetNormal 'quiet'
  end

  if \ IsProcDumpOn() then
    call SetNormal 'quiet'

  if pos('Instance', gDumpStyle) = 0 then
    gDumpStyle = gDumpStyle 'Instance'
  if pos('Shared', gDumpStyle) = 0 then
    gDumpStyle = gDumpStyle 'Shared'
  if pos('Full', gDumpStyle) = 0 then
    gDumpStyle = gDumpStyle 'Full'
  if pos('Extended', gDumpStyle) = 0 then
    gDumpStyle = gDumpStyle 'Extended'

  say
  say 'Updating dump style to' gDumpStyle
  'pdumpusr syspg,private,mvdm,instance,sysio,shared,update'
  'pdumpusr query'
  return
  /* end AddExtended */

/*=== AddFull() add instance data, private data and more ===*/

AddFull: procedure expose (Globals)
  if gDumpStyle == 'All' then do
    say
    say 'Dump style is' gDumpStyle '- resetting to Normal'
    call SetNormal 'quiet'
  end

  if \ IsProcDumpOn() then
    call SetNormal 'quiet'

  if pos('Instance', gDumpStyle) = 0 then
    gDumpStyle = gDumpStyle 'Instance'

  if pos('Full', gDumpStyle) = 0 then
    gDumpStyle = gDumpStyle 'Full'

  say
  say 'Updating dump style to' gDumpStyle
  'pdumpusr private,instance,update'
  'pdumpusr query'
  return
  /* end AddFull */

/*=== AddInstance() add instance and more ===*/

AddInstance: procedure expose (Globals)
  if gDumpStyle == 'All' then do
    say
    say 'Dump style is' gDumpStyle '- resetting to Normal'
    call SetNormal 'quiet'
  end

  if \ IsProcDumpOn() then
    call SetNormal 'quiet'

  if pos('Instance', gDumpStyle) = 0 then
    gDumpStyle = gDumpStyle 'Instance'

  say
  say 'Updating dump style to' gDumpStyle
  'pdumpusr instance,update'
  'procdump query'
  return
  /* end AddInstance */

/*=== AddShared() Add private and shared code and data ===*/

AddShared: procedure expose (Globals)
  if gDumpStyle == 'All' then do
    say
    say 'Dump style is' gDumpStyle '- resetting to Normal'
    call SetNormal 'quiet'
  end

  if \ IsProcDumpOn() then
    call SetNormal 'quiet'

  if pos('Private', gDumpStyle) = 0 then
    gDumpStyle = gDumpStyle 'Private'
  if pos('Instance', gDumpStyle) = 0 then
    gDumpStyle = gDumpStyle 'Instance'
  if pos('Shared', gDumpStyle) = 0 then
    gDumpStyle = gDumpStyle 'Shared'

  say
  say 'Updating dump style to' gDumpStyle
  'pdumpusr private,instance,shared,update'
  'pdumpusr query'
  return

/* end AddShared */

/*=== AskCommands() Ask for commands ===*/

AskCommands: procedure expose (Globals)

  escKey = x2c('1b')
  enterKey = x2c('0d')

  do forever

    call ShowProcsAndPids

    keys = 'adfhinorsvqxADFHINORSVQX!?[Esc][Enter]'
    say
    key = InKey(keys, 'D)ump N)orm. I)nst. F)ull S)hare X)tend. A)ll R)eset O)ff V)iew H)elp Q)uit')
    key = ToLower(key)
    if key == 'q' | key == escKey then
      leave
    else if key == enterKey then do
      say
      say
      iterate
    end
    else if key == 'a' then
      call SetAll
    else if key == 'd' then
      call ForceDump
    else if key == 'f' then
      call AddFull
    else if key == 'i' then
      call AddInstance
    else if key == 'n' then
      call SetNormal
    else if key == 'o' then
      call SetOff
    else if key == 'r' then
      call Reset
    else if key == 's' then
      call AddShared
    else if key == 'v' then
      call ShowSettings
    else if key == 'x' then
      call AddExtended
    else if key == '!' then do
      /* Shell */
      say
      shell = value('COMSPEC',, gEnv)
      signal off Error
      shell
      signal on Error
    end
    else if key == 'h' | key == '?' then
      call DoHelp
    else do
      say
      say 'Unexpected key'
      'pause'
      exit 1
    end
  end
  return

/* end AskCommands */

/*=== ChkRequired() check for required executables ===*/

ChkRequired: procedure expose (Globals)
  e = 'PATH'
  x = 'procdump.exe'
  s = SysSearchPath(e, x)
  if s == '' then
    call ScanArgsUsage 'Can not locate' x 'in' e '- please check your installation'
  x = 'pdumpusr.exe'
  s = SysSearchPath(e, x)
  if s == '' then
    call ScanArgsUsage 'Can not locate' x 'in' e '- please check your installation'
  return
  /* end ChkRequired */

/*=== DoCmds() Execute queued commands ===*/

DoCmds: procedure expose (Globals)

  do ndx = 1 to length(gCmds)
    cmd = substr(gCmds, ndx, 1)
    select
    when cmd == 'a' then
      call SetAll
    when cmd == 'd' then
      call ForceDump
    when cmd == 'f' then
      call AddFull
    when cmd == 'i' then
      call AddInstance
    when cmd == 'n' then
      call SetNormal
    when cmd == 'o' then
      call SetOff
    when cmd == 'r' then
      call Reset
    when cmd == 's' then
      call AddShared
    when cmd == 'v' then
      call ShowSettings
    when cmd == 'x' then
      call AddExtended
    otherwise
      call ScanArgsUsage 'Command' cmd 'unexpected'
    end
  end
  return

/* end DoCmds */

/*=== DoHelp() Show help ===*/

DoHelp: procedure expose (Globals)
  say
  say gCmdName gVersion
  say
  say 'D - Force dump using current settings'
  say 'N - Reset to Normal - set to summ,sem,sysldr,sysfs,sysvm,syssem'
  say 'I - Instance - add instance data to current settings'
  say 'F - Full - add private/instance code and data to current settings'
  say 'S - Shared - add private/instance/shared code and data to current'
  say 'X - Extended - add private/shared code/data mdvm sysio and syspg to current'
  say 'A - Reset to All - set to dump all physical memory'
  say 'R - Reset to Default - set to system default settings'
  say 'O - Turn off dump facility'
  say 'V - View current settings'
  say 'H - Display this screen'
  say 'Q - Quit'
  say '? - Display this screen'
  say '! - Shell'
  say
  return

/* end DoHelp */

/*=== FindDumpDir() Default dump directory and normalize path for procdump ===*/

FindDumpDir: procedure expose (Globals)

  /* If already set, use existing value */
  if gDumpDir \== '' then
    return

  do 1
    /* Try config.sys setting */
    cfgsys = SysBootDrive() || '\config.sys'
    match = 'DUMPPROCESS'
    do while lines(cfgsys) \= 0
      line = linein(cfgsys)
      line = strip(line)
      /* DUMPPROCESS=D:\Dumps\Slat60-1 */
      parse var line key '=' dir
      if translate(key) \==  match then
	iterate
      if dir = '' then
	iterate				/* Silently ignore errors */
      dir = strip(dir)
      /* FIXME to warn */
      if \ IsDir(dir) then
	iterate				/* Silently ignore errors */
      gDumpDir = dir
      leave
    end
    call stream cfgsys, 'C', 'CLOSE'

    if gDumpDir \== '' then
      leave				/* Set from config.sys */

    /* Look for ?:\Dumps directory on local drives
       Optimized for me - sorry
       FIXME to check readonly?
       FIXME to check hostname?
    */
    lclDrives = SysDriveMap(, 'LOCAL')
    do ndx = words(lclDrives) to 1 by -1
      drv = word(lclDrives, ndx)
      info = SysDriveInfo(drv)
      if info \== '' then do
	/* Drive is accessible */
	fstype = SysFileSystemType(drv)
	if fstype \== 'CDFS' then do
	  dir = drv || '\Dumps'
	  dir = IsDir(dir, 'FULL')
	  if dir \== '' then
	    gDumpDir = dir
	end
      end
    end

    if gDumpDir \== '' then
      leave

    gDumpDir = gTmpDir || '\Dumps'
    gDumpDir = IsDir(gDumpDir, 'FULL')
    if gDumpDir \== '' then
      leave

    call Die 'Can not select dump directory - checked config.sys ?:\Dumps and %TMP\Dumps'

  end /* do */

  return

/* end FindDumpDir */

/*=== ForceDump() Force dump ===*/

ForceDump: procedure expose (Globals)

  /* Force dump now */
  say
  do 1
    if \ IsProcDumpOn() then do
      say 'Procdump is not turned on - can not force dump'
      say
      leave
    end
    if gDumpStyle == 'All' then do
      say 'Dumping all memory'
      /* procdump force /system */
      cmd = 'procdump force /pid:all'
    end
    else do
      if gPid == '' gProc == '' then do
	say 'Process/pid required for gDumpStyle dump style'
	leave
      end
      suffix = ''
      if gPid \== '' then do
	say 'Dumping PID(s)' gPid
	suffix = suffix '/pid:' || gPid
      end
      if gProc \== '' then do
	say 'Dumping process(es)' gProc
	suffix = suffix '/proc:' || gProc
      end
      cmd = 'procdump force' strip(suffix)
    end
    signal off Error
    cmd
    signal on Error
    if RC = 0 then
      leave
    /* 2013-10-24 SHL FIXME to know why apiret changes to 20548 */
    say
    say 'Procdump failed with error' RC '(' || d2x(RC) || ')'
    say 'Check process/pid running'
    say

  end /* do */

  return

  /* end ForceDump */

/*=== IsProcDumpOn() Guess DumpStyle, return true if procdump turned on ===*/

IsProcDumpOn: procedure expose (Globals)
  if gDumpStyle = 'Unknown' then do
    '@procdump query | rxqueue'
    do while queued() \= 0
      pull s
      if s == 'PROCDUMP OFF' then
	gDumpStyle = 'Off'
      else if s == 'PDUMPUSR PADDR(ALL)' then
	gDumpStyle = 'All'
    end
  end

  return gDumpStyle \== 'Off'

/* end IsProcDumpOn */

/*=== Reset() reset to system default settings ===*/

Reset: procedure expose (Globals)
  /* Reset to default */
  gDumpStyle = 'Default'
  say
  say 'Resetting dump style to' gDumpStyle
  'procdump reset /pid:all'
  'procdump on /l:' || gDumpDir
  'pdumpusr reset'
  'procdump query'
  return
  /* end Reset */

/*=== SetAll() reset to dump all memeory style ===*/

SetAll: procedure expose (Globals)
  /* All physical memory */
  gDumpStyle = 'All'
  say
  say 'Setting dump style to' gDumpStyle
  'pdumpusr reset'
  'pdumpusr paddr(all)'
  'pdumpusr query'
  return
  /* end SetAll */

/*=== SetNormal([quiet]) - reset to Normal style ===*/

SetNormal: procedure expose (Globals)
  parse arg s
  verbose = s \== '' & s \== 1 & s \== 'quiet'

  gDumpStyle = 'Normal'

  if verbose then do
    say
    say 'Resetting dump style to' gDumpStyle
  end

  /* Set to normal */
  'procdump reset /pid:all'
  'procdump on /l:' || gDumpDir
  if gProliant then
    'pdumpusr summ,sem,sysldr,sysfs,syssem'	/* Omit sysvm */
  else
    'pdumpusr summ,sem,sysldr,sysfs,sysvm,syssem'

  if verbose then
    'procdump query'
  return
  /* end SetNormal */

/*=== SetOff() turn off dump facility ===*/

SetOff: procedure expose (Globals)
  /* Off */
  gDumpStyle = 'Off'
  say
  say 'Turning off Dump Facility'
  'procdump off'
  say
  return
  /* end SetOff */

/*=== ShowProcsAndPids() ===*/

ShowProcsAndPids: procedure expose (Globals)
  if gDumpStyle \== 'All' then do
    if gPid == '' & gProc == '' then
      say 'No Process/PID selected'
    else do
      if gPid \== '' then do
	s = gPid
	do while s \== ''
	  parse var s pid';'s
	  s = strip(s)
	  say 'PID 0x' || pid '('x2d(pid)') selected'
	end
      end
      if gProc \== '' then do
	s = gProc
	do while s \== ''
	  parse var s proc';'s
	  s = strip(s)
	  say 'Process' proc 'selected'
	end
      end
    end
  end
  return
  /* end ShowProcsAndPids */

/*=== ShowSettings() ===*/

ShowSettings: procedure expose (Globals)
  say
  say 'Dump directory is' gDumpDir
  say 'Dump style is' gDumpStyle
  call ShowProcsAndPids
  'procdump query'
  return
  /* end ShowSettings */

/*=== Initialize() Initialize globals ===*/

Initialize: procedure expose (Globals)
  call SetCmdName
  call LoadRexxUtil
  gEnv = 'OS2ENVIRONMENT'
  call SetTmpDir
  return

/* end Initialize */

/*=== ScanArgsInit() ScanArgs initialization exit routine ===*/

ScanArgsInit: procedure expose (Globals) cmdTail swCtl keepQuoted

  if cmdTail == '' then
    call ScanArgsHelp

  /* Preset defaults */
  gBatch = 1				/* Run in batch mode */
  gProliant = 0				/* Run in Proliat hack mode - omit sysvm */
  gDumpDir = ''				/* Dump directory */
  gCmds = ''				/* Queued commands */
  gDumpStyle = 'Unknown'
  gProc = ''				/* Process name */
  gPid = ''				/* Hex pid */

  keepQuoted = 1			/* Set to 1 to keep arguments quoted */

  return

/* end ScanArgsInit */

/*=== ScanArgsSwitch() ScanArgs switch option exit routine ===*/

ScanArgsSwitch: procedure expose (Globals) curSw curSwArg
  select
  when curSw == 'i' then
    gBatch = 0
  when curSw == 'h' | curSw == '?' then
    call ScanArgsHelp
  when curSw == 'p' then
    gProliant = 1
  when curSw == 'v' then do
    say gCmdName gVersion
    exit
  end
  otherwise
    call ScanArgsUsage 'switch '''curSw''' unexpected'
  end /* select */

  return

/* end ScanArgsSwitch */

/*=== ScanArgsArg() ScanArgs argument option exit routine ===*/

ScanArgsArg: procedure expose (Globals) curArg
  dir = ''
  pid = ''
  proc = ''

  do 1
    s = ToLower(curArg)
    if length(s) == 1 & s >= 'a' & s <= 'z' then do
      /* Got command */
      if pos(s, 'adfinorsvx') > 0 then do
	gCmds = gCmds || s
	leave
      end
    end

    /* Strip quotes from quoted argument */
    quoted = left(curArg, 1) == '"'

    if quoted then
      s = strip(curArg, 'B', '"')
    else
      s = curArg

    /* Check if looks like directory name */
    if pos('\', s) > 0 | pos('/', s) > 0 | pos(':', s) > 0 then
      dir = s

    /* Check if looks like hex pid */
    if \ quoted & left(s, 2) == '0x' then do
      s = substr(curArg, 3)
      if \ datatype(s, 'X') then
	call ScanArgsUsage curArg 'is not a valid hex PID'
      else do
	pid = s
	if gProc == '' then
	  leave
      end
    end

    /* Check if looks like decimal pid */
    if \ quoted & left(s, 2) == '0t' then do
      s = substr(curArg,3)
      if \ datatype(s, 'W') then
	call ScanArgsUsage curArg 'is not a valid decimal PID'
      else do
	pid = d2x(s)
	if gProc == '' then
	  leave
      end
    end

    /* Check if looks like dump directory */
    if IsDir(s) then do
      dir = s
      if gDumpDir == '' then do
	gDumpDir = dir
	leave
      end
    end

    if \ quoted & datatype(s, 'X') then do
      /* hex number - assume pid */
      pid = s
      if gProc == '' then
	leave
    end

    if \ quoted & datatype(s, 'N') then do
      /* dec number - assume pid */
      pid = d2x(s)
      leave
    end

    /* Probably a module name */
    /* Allow underscores in names */
    if datatype(translate(s,'X', '_'), 'A') then do
      /* Got module name */
      proc = s
      leave
    end

    /* Guess what's wrong */
    if dir \= '' then do
      if \ IsDir(dir) then
	call ScanArgsUsage 'Cannot access' curArg 'directory'
      else if gDumpDir \== '' then
	call ScanArgsUsage 'Dump directory already set to' gDumpDir
    end

    call ScanArgsUsage curArg 'unexpected'

  end /* do */

  /* Accumulate PIDs */
  if pid \== '' then do
    if gPid == '' then
      gPid = pid
    else
      gPid = gPid || ';' || pid
  end

  /* Accumulate process names */
  if proc \== '' then do
    if gProc == '' then
      gProc = proc
    else
      gProc = gProc || ';' || proc
  end

  return

/* end ScanArgsArg */

/*=== ScanArgsTerm() ScanArgs scan end exit routine ===*/

ScanArgsTerm: procedure expose (Globals)
  return

/* end ScanArgsTerm */

/*=== ScanArgsHelp() Display ScanArgs usage help exit routine ===*/

ScanArgsHelp:
  say
  say gCmdName gVersion
  say 'Control Process Dump Facility in batch or interactive mode.'
  say
  say 'Usage:' gCmdName '[-h] [-i] [-p] [-v] [-?] [commands...] [procname|pid] [dirname]'
  say
  say ' -h -?     Display this message'
  say ' -i        Run interactive (default is batch mode)'
  say ' -p        Enable Proliant mode, disables sysvm to prevent system traps'
  say ' -v        Display version number and quit'
  say
  say ' pid       Select PID to dump, default radix is hex'
  say '           Prefix with 0x for hex or 0t for decimal if ambiguous'
  say ' procname  Select process to dump'
  say '           Quote if process name looks like a number'
  say ' dirname   Set Dump directory, quote if name looks like a number'
  say '           Defaults to ?:\Dumps or %TMP\Dumps'
  say ' commands  Batch mode commands'
  say '   a       Reset to All - set to dump all physical memory'
  say '   d       Force dump - requires PID or process name'
  say '   f       Full - add private/instance code and data to current settings'
  say '   i       Instance - add instance data to current settings'
  say '   n       Reset to Normal - set to summ sem sysldr sysfs sysvm syssem'
  say '   o       Turn off dump facility'
  say '   r       Reset to Default - set to system default settings'
  say '   s       Shared - add private/instance/shared code and data to current'
  say '   v       View dump settings'
  say '   x       Extended - add private/shared code/data mvdm sysio syspg to current'

  exit 255

/* end ScanArgsHelp */

/*=== ScanArgsUsage(message) Report Scanargs usage error exit routine ===*/

ScanArgsUsage:
  parse arg msg
  say
  if msg \== '' then
    say msg
  say 'Usage:' gCmdName '[-h] [-i] [-p] [-v] [-?] [commands...] [procname|pid] [dirname]'
  exit 255

/* end ScanArgsUsage */

/*==============================================================================*/
/*=== SkelRexxFunc standards - Delete unused - Move modified above this mark ===*/
/*==============================================================================*/

/*=== ChopDirSlash(directory) Chop trailing \ from directory name unless root ===*/

ChopDirSlash: procedure
  parse arg dir
  if right(dir, 1) == '\' & right(dir, 2) \== ':\' & dir \== '\' then
    dir = substr(dir, 1, length(dir) - 1)
  return dir

/* end ChopDirSlash */

/*=== InKey(Keys, Prompt) returns key code ===*/

InKey: procedure
  parse arg keys, msg
  /* Convert key names to characters */
  i = pos('[Enter]', keys)
  if i > 0 then
    keys = substr(keys, 1, i - 1) || x2c('0d') || substr(keys, i + 7)
  i = pos('[Esc]', keys)
  if i > 0 then
    keys = substr(keys, 1, i - 1) || x2c('1b') || substr(keys, i + 5)
  call charout 'STDERR', msg '? '
  do forever
    key = SysGetKey('NOECHO')
    i = pos(key, keys)
    if i > 0 then do
      i = pos(key, xrange('20'x, '7e'x))
      if i > 0 then
	call lineout 'STDERR', key
      leave
    end
  end /* forever */
  return key

/* end InKey */

/*=== IsDir(dirName[, full]) return true if directory is valid, accessible directory ===*/

IsDir: procedure
  /* If arg(2) not omitted, return full directory name or empty string */
  parse arg dir, full
  newdir = ''

  do 1
    if dir == '' then do
      cwd = ''				/* No restore needed */
      leave
    end
    dir = translate(dir, '\', '/')	/* Convert to OS/2 slashes */
    s = strip(dir, 'T', '\')		/* Chop trailing slashes unless root */
    if s \== '' & right(s, 1) \== ":" then
      dir = s				/* Chop */
    drv = filespec('D', dir)
    cwd = directory()			/* Remember */
    /* If have drive letter and requested directory on some other drive */
    if drv \== '' & translate(drv) \== translate(left(cwd, 2)) then do
      /* Avoid slow failures and unwanted directory changes */
      drvs = SysDriveMap('A:')
      if pos(translate(drv), drvs) = 0 then
	leave				/* Unknown drive */
      if SysDriveInfo(drv) == '' then
	leave				/* Drive not ready */
      cwd2 = directory(drv)		/* Remember current directory on other drive */
      newdir = directory(dir)		/* Try to change and get full path name */
      call directory cwd2		/* Restore current directory on other drive */
      leave
    end

    /* If no drive letter or same drive and not UNC name */
    if left(dir, 2) \== '\\' then do
      newdir = directory(dir)		/* Try to change and get full path name */
      leave
    end

    /* UNC name - hopefully server is accessible or this will be slow
       Accept
	 \\server
	 \\server\
	 \\server\dir\
	 \\server\dir
     */
    cwd = ''				/* No restore needed */
    wc = dir
    if right(wc, 1) \== '\' then
      wc = wc || '\'
    i = lastpos('\', wc)
    if substr(wc, 3, 1) == '\' then
      leave				/* Malformed UNC - no server name */
    if pos('*', wc) > 0 | pos('?', wc) > 0 then
      leave				/* No wildcards allowed */
    call SysFileTree wc, 'files', 'O'
    if files.0 > 0 then do
      s = files.1			/* Exists and is not empty */
      i = lastpos('\', s)
      newdir = left(s, i - 1)		/* Extract directory name from full path name */
      leave
    end
    /* Try wildcarded directory name */
    wc = strip(wc, 'T', '\')
    i = lastpos('\', wc)
    base = substr(wc, i + 1)
    if base == '' then
      leave				/* Should have matched above */
    wc = substr(wc, 1, i) || '*' || base || '*'
    call SysFileTree wc, 'files', 'DO'
    do fileNum = 1 to files.0
      /* Find directory name is list */
      s = files.fileNum
      i = lastpos('\', s)
      s2 = substr(s, i + 1)
      if translate(base) == translate(s2) then do
	newdir = left(s, i - 1)
	leave
      end
    end /* i */
  end /* 1 */

  if cwd \== '' then
    call directory cwd			/* Restore original directory and drive */

  if full \== '' then
    ret = newdir			/* Return full directory name or empty string */
  else
    ret = newdir \== ''			/* Return true if valid and accessible */
  return ret

/* end IsDir */

/*=== ToLower(s) Convert to lower case ===*/

ToLower: procedure
  parse arg s
  return translate(s, xrange('a', 'z'), xrange('A', 'Z'))

/* end ToLower */

/*==========================================================================*/
/*=== SkelRexx standards - Delete unused - Move modified above this mark ===*/
/*==========================================================================*/

/*=== Error() Set gErrCondition; report to STDOUT; trace and exit or return if called ===*/

Error:
  say
  parse source . . cmd
  gErrCondition = condition('C')
  say gErrCondition 'signaled at line' SIGL 'of' cmd || '.'
  if condition('D') \== '' then
    say 'REXX reason =' condition('D') || '.'
  if gErrCondition == 'SYNTAX' & symbol('RC') == 'VAR' then
    say 'REXX error =' RC '-' errortext(RC) || '.'
  else if symbol('RC') == 'VAR' then
    say 'RC =' RC || '.'
  say 'Source =' sourceline(SIGL)

  if condition('I') \== 'CALL' | gErrCondition == 'NOVALUE' | gErrCondition == 'SYNTAX' then do
    trace '?A'
    say 'Enter REXX commands to debug failure.  Press enter to exit script.'
    nop
    if symbol('RC') \== 'VAR' then
      RC = 255
    exit RC
  end

  return

/* end Error */

/*=== Die([message,...]) Write multi-line message to STDERR and die ===*/

Die:
  call lineout 'STDERR', ''
  do i = 1 to arg()
    call lineout 'STDERR', arg(i)
  end
  call lineout 'STDERR', gCmdName 'aborting at line' SIGL || '.'
  call beep 200, 300
  call SysSleep 2
  exit 254

/* end Die */

/*=== SetCmdName() Set gCmdName to short script name ===*/

SetCmdName: procedure expose (Globals)
  parse source . . cmd
  cmd = filespec('N', cmd)		/* Chop path */
  c = lastpos('.', cmd)
  if c > 1 then
    cmd = left(cmd, c - 1)		/* Chop extension */
  gCmdName = translate(cmd, xrange('a', 'z'), xrange('A', 'Z'))	/* Lowercase */
  return

/* end SetCmdName */

/*=== SetTmpDir() Set gTmpDir to %TMP with trailing backslash or empty string ===*/

SetTmpDir: procedure expose (Globals)
  s = value('TMP',,gEnv)
  if s \== '' & right(s, 1) \== ':' & right(s, 1) \== '\' then
    s = s'\'				/* Stuff backslash */
  gTmpDir = s
  return

/* end SetTmpDir */

/*=== Halt() Report HALT condition to STDOUT and exit ===*/

Halt:
  say
  parse source . . cmd
  say condition('C') 'signaled at' cmd 'line' SIGL || '.'
  say 'Source =' sourceline(SIGL)
  say 'Sleeping for 2 seconds...'
  call SysSleep 2
  exit 253

/* end Halt */

/*=== LoadRexxUtil() Load RexxUtil functions ===*/

LoadRexxUtil:
  if RxFuncQuery('SysLoadFuncs') then do
    call RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
    if RESULT then
      call Die 'Cannot load SysLoadFuncs.'
    call SysLoadFuncs
  end
  return

/* end LoadRexxUtil */

/*=== ScanArgs(cmdLine) Scan command line ===*/

ScanArgs: procedure expose (Globals)

  /* Calls user exits to process arguments and switches */

  parse arg cmdTail
  cmdTail = strip(cmdTail)

  call ScanArgsInit

  /* Ensure optional settings initialized */
  if symbol('SWCTL') \== 'VAR' then
    swCtl = ''				/* Switches that take args, append ? if optional */
  if symbol('KEEPQUOTED') \== 'VAR' then
    keepQuoted = 0			/* Set to 1 to keep arguments quoted */

  /* Scan */
  curArg = ''				/* Current arg string */
  curSwList = ''			/* Current switch list */
  /* curSwArg = '' */			/* Current switch argument, if needed */
  noMoreSw = 0				/* End of switches */

  do while cmdTail \== '' | curArg \== '' | curSwList \== ''

    /* If arg buffer empty, refill */
    if curArg == '' then do
      qChar = left(cmdTail, 1)		/* Remember quote */
      if \ verify(qChar,'''"', 'M') then
	parse var cmdTail curArg cmdTail	/* Not quoted */
      else do
	/* Arg is quoted */
	curArg = ''
	do forever
	  /* Parse dropping quotes */
	  parse var cmdTail (qChar)quotedPart(qChar) cmdTail
	  curArg = curArg || quotedPart
	  /* Check for escaped quote within quoted string (i.e. "" or '') */
	  if left(cmdTail, 1) \== qChar then do
	    cmdTail = strip(cmdTail)	/* Strip leading whitespace */
	    leave			/* Done with this quoted arg */
	  end
	  curArg = curArg || qChar	/* Append quote */
	  if keepQuoted then
	    curArg = curArg || qChar	/* Append escaped quote */
	  parse var cmdTail (qChar) cmdTail	/* Strip quote */
	end /* do forever */
	if keepQuoted then
	  curArg = qChar || curArg || qChar	/* requote */
      end /* if quoted */
    end /* if curArg empty */

    /* If switch buffer empty, refill */
    if curSwList == '' & \ noMoreSw then do
      if left(curArg, 1) == '-' & curArg \== '-' then do
	if curArg == '--' then
	  noMoreSw = 1
	else
	  curSwList = substr(curArg, 2)	/* Remember switch string */
	curArg = ''			/* Mark empty */
	iterate				/* Refill arg buffer */
      end /* if switch */
    end /* if curSwList empty */

    /* If switch in progress */
    if curSwList \== '' then do
      curSw = left(curSwList, 1)	/* Next switch */
      curSwList = substr(curSwList, 2)	/* Drop from pending */
      /* Check switch allows argument, avoid matching ? */
      if pos(curSw, translate(swCtl,,'?')) \= 0 then do
	if curSwList \== '' then do
	  curSwArg = curSwList		/* Use rest of switch string for switch argument */
	  curSwList = ''
	end
	else if curArg \== '' & left(curArg, 1) \== '-' then do
	  curSwArg = curArg		/* Arg string is switch argument */
	  curArg = ''			/* Mark arg string empty */
	end
	else if pos(curSw'?', swCtl) = 0 then
	  call ScanArgsUsage 'Switch "-' || curSw || '" requires an argument'
	else
	  curSwArg = ''			/* Optional arg omitted */
      end

      call ScanArgsSwitch		/* Passing curSw and curSwArg */
      drop curSwArg			/* Must be used by now */
    end /* if switch */

    /* If arg */
    else if curArg \== '' then do
      noMoreSw = 1
      call ScanArgsArg			/* Passing curArg */
      curArg = ''
    end

  end /* while not done */

  call ScanArgsTerm

  return

/* end ScanArgs */

/* The end */
