/*
  littlemore.c
  Author:  sava (lpproj)
  License: Public Domain
           (You may use/modify/redistribute it freely BUT NOT WARRANTY.)
*/
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <io.h>
#include <limits.h>
#include <malloc.h>
#include <process.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifndef _WIN32
#include <unistd.h>
#endif

#define APP_NAME "littlemore"
#define APP_MAJOR 0
#define APP_MINOR 1

#if defined(_WIN32)
# if defined(_WIN64) || defined(WIN64)
#  define APP_TARGET "Win64"
# else
#  define APP_TARGET "Win32"
# endif
#elif defined(__OS2__) || defined(__EMX__)
# define APP_TARGET "OS/2"
#endif

static int opt_h;
static int opt_v;
static int opt_n;
static int opt_err;
static int arg_files;

static
int
do_more(const char *fname, FILE *fo, int treat_crlf)
{
  FILE *fi;
  int rc;
  int c_next, c;

  if (!fname && !*fname) return -1;
  if (fname[0] == '-' && fname[1] == '\0') {
    fi = stdin;
#if defined O_BINARY
    if (!isatty(fileno(fi))) setmode(fileno(fi), O_BINARY);
#endif
  }
  else {
    fi = fopen(fname, "rb");
  }
  if (!fi) return -1;
  
  rc = 0; c = EOF;
  for(;;) {
    c_next = fgetc(fi);
    if (c != EOF) {
      if (treat_crlf && c_next == '\n') {
        if (c == '\r') {
          c = c_next;
          continue;
        }
      }
      rc = fputc(c, fo);
      if (c == '\n') fflush(fo);
      if (rc < 0) break;
    }
    c = c_next;
    if (c == EOF) {
      rc = ferror(fi) || ferror(fo) ? -1 : 0;
      break;
    }
  }
  return rc;
}

static
int
do_more_args(int do_invoke, int argc, char **argv, FILE *fo, int treat_crlf)
{
  int rc = 0;
  char *s;

  arg_files = 0;
  while(argc--) {
    s = *argv++;
    if (!s) break;
    if (*s == '-' && s[1] != '\0') {
      if (strcmp(s, "-h")==0 || strcmp(s, "-?")==0 || strcmp(s, "--help")==0) {
        opt_h = 1;
      }
      else if (strcmp(s, "--version")==0 || strcmp(s, "-v")==0) {
        opt_v = 1;
      }
      else if (strcmp(s, "-n")==0 || strcmp(s, "--lines")==0) {
        long l = 0;
        if (*argv) {
          l = strtol(*argv, NULL, 0);
          --argc;
          ++argv;
        }
        if (l <= 0 || l > INT_MAX) {
          opt_err = 1;
          break;
        }
        opt_n = (int)l;
      }
    }
    else {
      ++arg_files;
      if (do_invoke) {
        rc = do_more(s, fo, treat_crlf);
        if (rc < 0) break;
      }
    }
    if (opt_err) {
      rc = 1;
      break;
    }
  }
  if (!opt_err && do_invoke && arg_files == 0) {
    rc = do_more("-", fo, treat_crlf);
  }
  return rc;
}


static
void
version()
{
  printf("%s version %d.%d (" __DATE__ ")\n", APP_NAME, APP_MAJOR, APP_MINOR);
}

static
void
usage(const char *prog)
{
  printf("%s - a minimal \"more\" wrapper (for " APP_TARGET ").\n", prog);
  printf("Usage: littlemore [file...]\n");
}

int
main(int argc, char **argv)
{
  const char more_cmd[] = "more.com";
  int treat_crlf = 0;
  char *progname;
  struct stat st;
  int rc;
  int fd;

  FILE *fo;
  
  progname = argv[0];
  if (!progname) progname = "littlemore";
  fd = fileno(stdin);
  if (isatty(fd)) {
    setvbuf(stdin, NULL, _IONBF, 0);
  }
  fd = fileno(stdout);
  if (fstat(fd, &st)>=0 && !S_ISREG(st.st_mode)) {
    treat_crlf = S_ISCHR(st.st_mode);
    fo = popen(more_cmd, treat_crlf ? "wt" : "wb");
    if (!fo) {
      fprintf(stderr, "%s: error at popen(\"%s\"...) (%s)\n", progname, more_cmd, strerror(errno));
      exit(127);
    }
    setvbuf(fo, NULL, _IOLBF, 0);
  }
  else {
    fo = stdout;
#if defined O_BINARY
    setmode(fd, O_BINARY);
#endif
  }

  rc = do_more_args(0, argc-1, argv+1, fo, treat_crlf);
  if (opt_err || opt_h) {
    usage(progname);
    return opt_err ? 127 : 0;
  }
  if (opt_v) {
    version();
    return 0;
  }
  rc = do_more_args(!0, argc-1, argv+1, fo, treat_crlf);
  if (fo && fo != stdout && fo != stderr) {
    rc = pclose(fo);
  }

  return rc >= 0 ? 0 : 1;
}
