/*****************************************************************************
 * RMX Executable Header Functions
 ***************************************************************************** 
 *
 * Source:   $Source$
 *
 * Overview: This file contains the implementation of the following
 *           functions: 
 *
 *           Exported
 *               RmxExeRead32Header
 *               RmxExeRead32Modules
 *
 * $Log$
 *
 ***************************************************************************** 
 * 
 * Copyright (c) 1994 Johan Wikman
 *
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee, 
 * provided that the above copyright notice appear in all copies and 
 * that both that copyright notice and this permission notice appear in 
 * supporting documentation.
 *
 * THERE IS NO WARRANTY FOR THIS SOFTWARE, TO THE EXTENT PERMITTED BY
 * APPLICABLE LAW. THE SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY
 * KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE. THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE PROGRAM
 * IS WITH YOU.  SHOULD THE PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF
 * ALL NECESSARY SERVICING, REPAIR OR CORRECTION. 
 *
 *****************************************************************************/

#define INCL_NOCOMMON
#define INCL_DOSERRORS
#define INCL_DOSFILEMGR
#include "rmxpatch.h"
// The exe.h, newexe.h and exe386.h files seems to have been written
// by different people not aware of each other. Lots of conflicts,
// therefore the defines.
#define BYTE unsigned char
#include <exe.h>
#undef BYTE
#include <newexe.h>
#undef NSMOVE
#undef NSSHARED
#undef NSPRELOAD
#undef NSEXRD
#undef NSCONFORM
#undef NSDISCARD
typedef ULONG  DWORD;
typedef USHORT WORD;
#include <exe386.h>
#include <mem.h>
#include <stdlib.h>
#include <os2.h>


/*****************************************************************************
 * MODULE VARIABLES
 *****************************************************************************/

const ULONG SIZ_OLDEXEHEADER = 0x40; // Enough for our purposes. See e.g.
                                     // the IBM document lxexe.doc.
const ULONG IDX_NEWEXEHEADER = 0x3C; // Index of position containing
				     // the offset of the 32 bit header. Again
				     // see lxexe.doc.
const ULONG TAG_EXE32        = 0x40; // Value of the 'erelof' field in the old
				     // executable header if the exe is 32
				     // bit. Beats me where I got this
				     // information. Seems to be right though.


/*****************************************************************************
 * MODULE FUNCTIONS
 *****************************************************************************/

static ULONG ExeFind32Offset            (HFILE hFile, ULONG* pulExe32Offset);
static ULONG PreParseReference          (BYTE* pbRecord);
static ULONG ParseInternalReference     (BYTE* pbRecord);
static ULONG ParseOrdinalImportReference(BYTE* pbRecord, RMXMODULE* pModules);
static ULONG ParseNameImportReference   (BYTE* pbRecord, RMXMODULE* pModules);
static ULONG ParseInternalEntryReference(BYTE* pbRecord);
static void  UpdateFixupInfo            (BYTE  sourceType, RMXMODULE* pModule);


/*****************************************************************************
 * EXPORTED FUNCTIONS
 *****************************************************************************
 *
 * FUNCTION: ULONG RmxExeRead32Header(HFILE hFile, ULONG* pulExe32Offset,
 *                                    struct e32_exe* exe32);  
 *
 * INPUT:
 *    hFile:           Handle of the file.
 *
 * OUTPUT:
 *    *pulExe32Offset: Offset of 32-bit header.
 *    *exe32:          The 32-bit executable header.
 *
 * RETURN:
 *    NO_ERROR
 *    ERROR_BAD_EXE_FORMAT
 *
 *  plus all codes returned by
 *
 *    DosSetFilePtr
 *    DosRead
 *
 *****************************************************************************/

ULONG RMXENTRY RmxExeRead32Header(HFILE    hFile, 
				  ULONG*   poffExe32, 
				  e32_exe* pexe32)
{
  ULONG
    rc;

  rc = ExeFind32Offset(hFile, poffExe32);

  if (rc != NO_ERROR)
    return rc;
  
  ULONG
    ulPosition;
  
  // This should always succeed...
  
  rc = DosSetFilePtr(hFile, *poffExe32, FILE_BEGIN, &ulPosition);

  if (rc != NO_ERROR)
    return rc;
  
  ULONG
    cBytes;
  
  rc = DosRead(hFile, pexe32, sizeof(e32_exe), &cBytes);

  if (rc != NO_ERROR)
    return rc;
  
  // If we can't read the new header, the file can't possibly be a
  // 32-bit executable. 
  
  if (cBytes != sizeof(e32_exe))
    return ERROR_BAD_EXE_FORMAT;

  if ((E32_MAGIC1(*pexe32) != E32MAGIC1) ||
      (E32_MAGIC2(*pexe32) != E32MAGIC2))
    return ERROR_BAD_EXE_FORMAT;
  
  return NO_ERROR;
}


/*****************************************************************************
 *
 * FUNCTION: ULONG RmxExeRead32Modules(HFILE hFile, ULONG* pcModules,
 *                                     RMXMODULE** ppModules);
 *
 * INPUT:
 *    hFile:           Handle of the file.
 *
 * OUTPUT:
 *    *pcModule:       Count of modules.
 *    *ppModules:      Points to array of RMXMODULES.
 *
 * RETURN:
 *    NO_ERROR
 *    ERROR_NOT_ENOUGH_MEMORY
 *    ERROR_READ_FAULT
 *    
 *  plus all codes returned by
 *    
 *    RmxExeRead32Header
 *    DosRead
 *
 * NOTES:
 *    It is the callers responsibility to deallocate the array of
 *    RMXMODULEs.
 *
 *****************************************************************************/

ULONG RMXENTRY RmxExeRead32Modules(HFILE hFile, ULONG* pcModules,
				   RMXMODULE** ppModules)
{
  ULONG
    offExe32;
  struct e32_exe
   exe32;
  ULONG
    rc;
  
  rc = RmxExeRead32Header(hFile, &offExe32, &exe32);

  if (rc != NO_ERROR)
    return rc;

  // The first task is to read all fixup information. All offsets in
  // the 32-bit header are relative to the beginning of the 32-bit
  // header. To get an offset in the file, the offset of the header
  // itself must be added. 
  
  ULONG
    offFixups = offExe32 + exe32.e32_fpagetab;
  ULONG
    ulPosition;

  rc = DosSetFilePtr(hFile, offFixups, FILE_BEGIN, &ulPosition);

  if (rc != NO_ERROR)
    return rc;

  // Actually I wouldn't need to read all fixup data as I don't use
  // the procedure entries, but what the heck.
  
  ULONG
    cbFixups = exe32.e32_fixupsize,
    cBytes;
  BYTE
    *pbFixups = (BYTE*) malloc(cbFixups);
  
  if (!pbFixups)
    return ERROR_NOT_ENOUGH_MEMORY;

  rc = DosRead(hFile, pbFixups, cbFixups, &cBytes);
  
  if (rc != NO_ERROR)
    {
      free(pbFixups);
      return rc;
    }
  
  if (cBytes != cbFixups)
    {
      free(pbFixups);
      return ERROR_READ_FAULT;
    }
  
  // Next thing to do is to allocate memory for all modules.
  
  *pcModules = exe32.e32_impmodcnt;

  ULONG
    ulModuleSize = *pcModules * sizeof(RMXMODULE);

  *ppModules = (RMXMODULE*) malloc(ulModuleSize);

  if (!*ppModules)
    {
      free(pbFixups);
      return ERROR_NOT_ENOUGH_MEMORY;
    }

  memset(*ppModules, 0, ulModuleSize);

  // Then the array of modules is initialized. For that we need the
  // offset of the modules into the read chunk of fixup data. 
  
  ULONG
    offModules = exe32.e32_impmod - exe32.e32_fpagetab;
  BYTE
    *pbRawModules = pbFixups + offModules;
  int 
    i;
  
  for (i = 0; i < *pcModules; i++)
    {
      // The length of the module name is in the first byte.
      
      ULONG
	ulSize = *pbRawModules;
      RMXMODULE
	*pModule = &(*ppModules)[i];

      // And the actual name is then from the second byte forward.
      
      pbRawModules++;
      
      memcpy(pModule->achName, pbRawModules, ulSize);
      
      // The strings are not NULL terminated which is why the NULL
      // must be set explicitly.
      
      pModule->achName[ulSize] = 0;
      pModule->ulOffset = offFixups + (pbRawModules - pbFixups);
      
      pbRawModules += ulSize;
    }
  
  // The fixup page table provides a mapping of a logical page number
  // to an offset into the fixup record table for that page. The table
  // contains one additional entry that is an offset to the end of the
  // fixup record table. As the offsets are 4 bytes the number of
  // pages is. 

  ULONG
    cPages = (exe32.e32_frectab - exe32.e32_fpagetab) / 4 - 1;

  // As the page table is first in the fixup chunk we can simply cast
  // the fixups pointer into an unsigned long pointer and, voila, we
  // have the page table.

  ULONG
    *offPages = (ULONG*) pbFixups;
  BYTE
    *pbRecords = pbFixups + (exe32.e32_frectab - exe32.e32_fpagetab);

  for (i = 0; i < cPages; i++)
    {
      BYTE
	*pbCurrentRecord = pbRecords + offPages[i],
        *pbNextRecord    = pbRecords + offPages[i + 1];

      while (pbCurrentRecord < pbNextRecord)
	{
	  BYTE
	    flTarget = pbCurrentRecord[1]; // NOTE: 1 not 0
	  ULONG
	    ulSize = 0;

	  switch (flTarget & NRRTYP)
	    {
	    case NRRINT:
	      ulSize = ParseInternalReference(pbCurrentRecord);
	      break;
	      
	    case NRRORD:
	      ulSize = ParseOrdinalImportReference(pbCurrentRecord, 
						   *ppModules);
	      break;
	      
	    case NRRNAM:
	      ulSize = ParseNameImportReference(pbCurrentRecord, *ppModules);
	      break;
	      
	    case NRRENT:
	      ulSize = ParseInternalEntryReference(pbCurrentRecord);
	    }
	  
	  pbCurrentRecord += ulSize;
	}
    }
  
  return NO_ERROR;
}


/*****************************************************************************
 * MODULE FUNCTIONS (IMPLEMENTATION)
 *****************************************************************************
 *
 * FUNCTION: ULONG ExeFind32Offset(HFILE hFile, ULONG* poffExe32);
 * 
 * INPUT:
 *    hFile:           Handle of the file.
 *
 * OUTPUT:
 *    *poffExe32: Offset of 32-bit header.
 *
 * RETURN:
 *    NO_ERROR
 *    ERROR_BAD_EXE_FORMAT
 *    
 *  plus all codes returned by
 *
 *    DosRead
 *
 *****************************************************************************/

static ULONG ExeFind32Offset(HFILE hFile, ULONG* poffExe32)
{
  BYTE
    aBytes[SIZ_OLDEXEHEADER];
  ULONG
    cBytes;
  ULONG
    rc;

  rc = DosRead(hFile, aBytes, SIZ_OLDEXEHEADER, &cBytes);

  if (rc != NO_ERROR)
    return rc;
  
  // If we can't read the old header, the file can't possibly be an
  // executable. 
  
  if (cBytes != SIZ_OLDEXEHEADER)
    return ERROR_BAD_EXE_FORMAT;

  struct exe
    *exeOld = (struct exe*) aBytes;

  // EXEID comes from exe.h
  
  if (exeOld->eid != EXEID)
    return ERROR_BAD_EXE_FORMAT;
  
  if (exeOld->ereloff != TAG_EXE32)
    return ERROR_BAD_EXE_FORMAT;
  
  *poffExe32 = *((ULONG*)&aBytes[IDX_NEWEXEHEADER]);
  
  return NO_ERROR;
}


/*****************************************************************************
 *
 * FUNCTION: ULONG (Pre)Parse...(...);
 * 
 * RETURN:
 *    The size of the current entry.
 *
 * NOTES:
 *    All these function return the size of the current import entry.
 *
 *****************************************************************************/
//
//               +-----+-----+-----+-----+
//           00h | SRC |FLAGS|SRCOFF/CNT*|
//               +-----+-----+-----+-----+
//               ...
//
//             * These fields are variable size.
//             @ These fields are optional.

static ULONG PreParseReference(BYTE* pbRecord)
{
  ULONG
    ulSize = 2; // The "Source Type" and "Target Flags" bytes.
  BYTE
    sourceType = pbRecord[0];

  // If the "Source List" flag is set, the target data is followed by a
  // list of source offsets and the item following target flags bytes
  // is a byte specifying the number of source offsets. If it is
  // clear, the item following is a word directly specifying the
  // single source offset. 
  
  if (sourceType & NRCHAIN) // "Source List" flag.
    {
      ulSize += 1;  // The byte itself.

      BYTE
	sourceCount = pbRecord[2];
      
      ulSize += sourceCount * 2; // The actual offsets..
    }
  else
    ulSize += 2;

  return ulSize;
}


/*****************************************************************************/
//
//               +-----+-----+-----+-----+
//           00h | SRC |FLAGS|SRCOFF/CNT*|
//               +-----+-----+-----+-----+-----+-----+
//       03h/04h |  OBJECT * |        TRGOFF * @     |
//               +-----+-----+-----+-----+-----+-----+
//               | SRCOFF1 @ |   . . .   | SRCOFFn @ |
//               +-----+-----+----   ----+-----+-----+
//
//             * These fields are variable size.
//             @ These fields are optional.

static ULONG ParseInternalReference(BYTE* pbRecord)
{
  ULONG
    ulSize = PreParseReference(pbRecord);
  BYTE
    sourceType  = pbRecord[0],
    flTarget = pbRecord[1];

  if (flTarget & NR16OBJMOD) // "16-bit Object Number/Module Ordinal" flag.
    ulSize += 2;
  else
    ulSize += 1;
  
  // The target offset field is not present if this pbRecord represents
  // a 16-bit Selector fixup.

  if ((sourceType & NRSTYP) == NRSSEG) // "16-bit Selector Fixup"
    ;
  else
    {
      // The value may be a 2 or 4 byte entry

      if (flTarget & NR32BITOFF) // "32-bit Target Offset" flag
	ulSize += 4;
      else
	ulSize += 2;
    }

  return ulSize;
}


/*****************************************************************************/
//
//               +-----+-----+-----+-----+
//           00h | SRC |FLAGS|SRCOFF/CNT*|
//               +-----+-----+-----+-----+-----+-----+-----+-----+
//       03h/04h | MOD ORD# *|IMPORT ORD*|     ADDITIVE * @      |
//               +-----+-----+-----+-----+-----+-----+-----+-----+
//               | SRCOFF1 @ |   . . .   | SRCOFFn @ |
//               +-----+-----+----   ----+-----+-----+
//
//             * These fields are variable size.
//             @ These fields are optional.

static ULONG ParseOrdinalImportReference(BYTE* pbRecord, RMXMODULE* pModules)
{
  ULONG
    ulSize = PreParseReference(pbRecord);
  BYTE
    sourceType  = pbRecord[0],
    flTarget = pbRecord[1];

  ULONG
    index = 3;

  if (!(sourceType & NRCHAIN)) // "Source List" flag.
    index++;

  USHORT
    usModule = 0;
  
  if (flTarget & NR16OBJMOD) // "16-bit Object Number/Module Ordinal" flag
    {
      usModule = *((USHORT*) &pbRecord[index]);
      index += 2;
      ulSize  += 2;
    }
  else
    {
      usModule = (UCHAR) pbRecord[index];
      index += 1;
      ulSize  += 1;
    }
  
  // I don't use it, but ordinal is the actual ordinal of the function
  // that will be resolved from the DLL. You can use it in order to
  // make a tdump or exehdr like utility. The warning about the unused
  // variable is turned off with the following pragma. Works only for
  // Borland.  

#pragma warn -aus

  ULONG
    ordinal = 0;

  if (flTarget & NR8BITORD) // "8-bit Ordinal" flag
    {
      ordinal = (UCHAR) pbRecord[index];
      ulSize  += 1;
    }
  else if (flTarget & NR32BITOFF) // "32-bit Target Offset" flag
    {
      ordinal = *((ULONG*) &pbRecord[index]);
      ulSize  += 4;
    }
  else
    {
      ordinal = *((USHORT*) &pbRecord[index]);
      ulSize  += 2;
    }

  if (flTarget & NRADD) // "Additive Fixup" flag
    {
      if (flTarget & NR32BITADD) // "32-bit Additive Fixup" flag
	ulSize += 4;
      else
	ulSize += 2;
    }

  UpdateFixupInfo(sourceType, &pModules[usModule - 1]);

  return ulSize;
}

// We turn the warning of unused variables on again.
#pragma warn .aus


/*****************************************************************************/
//
//               +-----+-----+-----+-----+
//           00h | SRC |FLAGS|SRCOFF/CNT*|
//               +-----+-----+-----+-----+-----+-----+-----+-----+-----+-----+
//       03h/04h | MOD ORD# *| PROCEDURE NAME OFFSET*|     ADDITIVE * @      |
//               +-----+-----+-----+-----+-----+-----+-----+-----+-----+-----+
//               | SRCOFF1 @ |   . . .   | SRCOFFn @ |
//               +-----+-----+----   ----+-----+-----+
//
//             * These fields are variable size.
//             @ These fields are optional.

static ULONG ParseNameImportReference(BYTE* pbRecord, RMXMODULE* pModules)
{
  ULONG
    ulSize = PreParseReference(pbRecord);
  BYTE
    sourceType  = pbRecord[0],
    flTarget = pbRecord[1];

  ULONG
    index = 3;

  if (!(sourceType & NRCHAIN)) // "Source List" flag.
    index++;

  USHORT
    usModule = 0;
  
  if (flTarget & NR16OBJMOD) // "16-bit Object Number/Module Ordinal" flag
    {
      usModule = *((USHORT*) &pbRecord[index]);
      index += 2;
      ulSize  += 2;
    }
  else
    {
      usModule = (UCHAR) pbRecord[index];
      index += 1;
      ulSize  += 1;
    }

  // We don't use it, but procedure is a direct offset into the import
  // procedure name table. You can use it in order to make a tdump or
  // exehdr like utility. Keep in mind that the strings are not NULL
  // terminated. The warning about the unused variable is turned off
  // with the following pragma. Works only for Borland. 

#pragma warn -aus

  ULONG
    procedure = 0;

  if (flTarget & NR32BITOFF) // "32-bit Target Offset" flag
    {
      procedure = *((ULONG*) &pbRecord[index]);
      index += 4;
      ulSize  += 4;
    }
  else
    {
      procedure = *((USHORT*) &pbRecord[index]);
      index += 2;
      ulSize  += 2;
    }

  if (flTarget & NRADD) // "Additive Fixup" flag
    {
      if (flTarget & NR32BITADD) // "32-bit Additive Fixup" flag
	ulSize += 4;
      else
	ulSize += 2;
    }

  UpdateFixupInfo(sourceType, &pModules[usModule - 1]);

  return ulSize;
}

// We turn the waring of unused variables on again.
#pragma warn .aus


/*****************************************************************************/
//
//               +-----+-----+-----+-----+
//           00h | SRC |FLAGS|SRCOFF/CNT*|
//               +-----+-----+-----+-----+-----+-----+
//       03h/04h |  ORD # *  |     ADDITIVE * @      |
//               +-----+-----+-----+-----+-----+-----+
//               | SRCOFF1 @ |   . . .   | SRCOFFn @ |
//               +-----+-----+----   ----+-----+-----+
//
//             * These fields are variable size.
//             @ These fields are optional.

static ULONG ParseInternalEntryReference(BYTE* pbRecord)
{
  ULONG
    ulSize = PreParseReference(pbRecord);
  BYTE
    flTarget = pbRecord[1];

  if (flTarget & NR16OBJMOD) // "16-bit Object Number/Module Ordinal" flag.
    ulSize += 2;
  else
    ulSize += 1;

  if (flTarget & NRADD) // "Additive Fixup" flag
    {
      if (flTarget & NR32BITADD) // "32-bit Additive Fixup" flag
	ulSize += 4;
      else
	ulSize += 2;
    }

  return ulSize;
}



/*****************************************************************************
 *
 * FUNCTION: void UpdateFixupInfo(BYTE sourceType, RMXMODULE* pModule);
 * 
 * INPUT:
 *    sourceType: The type of the current ficup entry.
 *
 * OUTPUT:
 *    *pModule:   The module for which the current fixup entry is.
 *
 * NOTES:
 *    This function is called once for each fixup entry of a module.
 *
 *****************************************************************************/

static void UpdateFixupInfo(BYTE sourceType, RMXMODULE* pModule)
{
  switch (sourceType & NRSTYP)
    {
    case NRSBYT:
      // Havn't got a clue what this actually means.
      pModule->fixup8           = TRUE;
      break;
      
    case NRSSEG:
      pModule->fixup16Selector  = TRUE;
      break;
      
    case NRSPTR:
      pModule->fixup1616Pointer = TRUE;
      break;
      
    case NRSOFF:
      pModule->fixup16Offset    = TRUE;
      break;
      
    case NRPTR48:
      pModule->fixup1632Pointer = TRUE;
      break;
      
    case NROFF32:
      pModule->fixup32Offset    = TRUE;
      break;
      
    case NRSOFF32:
      pModule->fixup32SROffset  = TRUE;
      break;
      
    default:
      // Hmm, what should we do about this. Let's ignore it and hope
      // it'll not bomb. Anyway, it should occur.
      ;
    }
}
