(**************************************************************************)
(*                                                                        *)
(*  Setup for web server                                                  *)
(*  Copyright (C) 2021   Peter Moylan                                     *)
(*                                                                        *)
(*  This program is free software: you can redistribute it and/or modify  *)
(*  it under the terms of the GNU General Public License as published by  *)
(*  the Free Software Foundation, either version 3 of the License, or     *)
(*  (at your option) any later version.                                   *)
(*                                                                        *)
(*  This program is distributed in the hope that it will be useful,       *)
(*  but WITHOUT ANY WARRANTY; without even the implied warranty of        *)
(*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *)
(*  GNU General Public License for more details.                          *)
(*                                                                        *)
(*  You should have received a copy of the GNU General Public License     *)
(*  along with this program.  If not, see <http://www.gnu.org/licenses/>. *)
(*                                                                        *)
(*  To contact author:   http://www.pmoylan.org   peter@pmoylan.org       *)
(*                                                                        *)
(**************************************************************************)

IMPLEMENTATION MODULE RealmsPage;

        (************************************************************)
        (*                                                          *)
        (*                  PM Setup for web server                 *)
        (*             The list of realms for a domain              *)
        (*                                                          *)
        (*    Started:        9 September 2021                      *)
        (*    Last edited:    17 September 2021                     *)
        (*    Status:         OK                                    *)
        (*                                                          *)
        (************************************************************)



IMPORT SYSTEM, OS2, OS2RTL, DID, PMInit, Strings, CommonSettings,
       INIData, RINIData, Remote, OneLine, RealmEditor;

FROM Names IMPORT
    (* type *)  DomainName, HostNameIndex, HostName, FilenameIndex, FilenameString;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

FROM MiscFuncs IMPORT
    (* type *)  CharArrayPointer;

FROM Languages IMPORT
    (* type *)  LangHandle,
    (* proc *)  StrToBuffer, StrToBufferA;

FROM LowLevel IMPORT
    (* proc *)  EVAL;

(************************************************************************)

CONST
    Nul = CHR(0);

TYPE
    LanguageString = ARRAY [0..31] OF CHAR;
    RealmType = ARRAY [0..63] OF CHAR;

VAR
    OurLang: LangHandle;
    INIname, SetupINIname: ARRAY [0..15] OF CHAR;
    OurHWND: OS2.HWND;
    DomName: DomainName;
    ChangeInProgress, Changed: BOOLEAN;
    PageFont: CommonSettings.FontName;

(************************************************************************)
(*                       SETTING LABELS AND FONTS                       *)
(************************************************************************)

PROCEDURE SetLanguage (lang: LangHandle);

    (* Relabels this page in the new language. *)

    VAR stringval: ARRAY [0..511] OF CHAR;

    BEGIN
        OurLang := lang;
        StrToBufferA (lang, "RealmList.Title", DomName, stringval);
        OS2.WinSetWindowText (OurHWND, stringval);
        StrToBuffer (lang, "RealmList.RealmListLabel", stringval);
        OS2.WinSetDlgItemText (OurHWND, DID.RealmListLabel, stringval);
        StrToBuffer (lang, "Buttons.Add", stringval);
        OS2.WinSetDlgItemText (OurHWND, DID.AddRealm, stringval);
        StrToBuffer (lang, "Buttons.Rename", stringval);
        OS2.WinSetDlgItemText (OurHWND, DID.RenameRealm, stringval);
        StrToBuffer (lang, "Buttons.Edit", stringval);
        OS2.WinSetDlgItemText (OurHWND, DID.EditRealm, stringval);
        StrToBuffer (lang, "Buttons.Promote", stringval);
        OS2.WinSetDlgItemText (OurHWND, DID.PromoteRealm, stringval);
        StrToBuffer (lang, "Buttons.Delete", stringval);
        OS2.WinSetDlgItemText (OurHWND, DID.DeleteRealm, stringval);
        StrToBuffer (lang, "Buttons.OK", stringval);
        OS2.WinSetDlgItemText (OurHWND, DID.RlistOK, stringval);
        StrToBuffer (lang, "Buttons.Cancel", stringval);
        OS2.WinSetDlgItemText (OurHWND, DID.RlistCancel, stringval);
    END SetLanguage;

(************************************************************************)

PROCEDURE SetPageFonts;

    (* Changes the font of the notebook pages to the font recorded in   *)
    (* the INI file as belonging to this dialogue.                      *)

    VAR NewFontName: CommonSettings.FontName;

    BEGIN
        CommonSettings.CurrentFont (CommonSettings.DomainDlg, NewFontName);
        IF NOT Strings.Equal (NewFontName, PageFont) THEN
            PageFont := NewFontName;
        END (*IF*);
    END SetPageFonts;

(************************************************************************)
(*                         LOADING DIALOGUE DATA                        *)
(************************************************************************)

PROCEDURE LoadDialogueData (hwnd: OS2.HWND);

    VAR state: RINIData.StringReadState;
        name: RealmType;

    BEGIN
        EVAL (RINIData.OpenINIFile (INIname));

        (* The list of realms for this domain. *)

        RINIData.GetStringList (DomName, "Realms", state);
        REPEAT
            RINIData.NextString (state, name);
            IF name[0] <> Nul THEN
                (* Add name to the listbox. *)

                OS2.WinSendDlgItemMsg (hwnd, DID.RealmList, OS2.LM_INSERTITEM,
                     OS2.MPFROMSHORT(OS2.LIT_END), SYSTEM.ADR(name));

            END (*IF*);
        UNTIL name[0] = Nul;
        RINIData.CloseStringList (state);

        RINIData.CloseINIFile;

    END LoadDialogueData;

(************************************************************************)
(*                           STORING THE DATA                           *)
(************************************************************************)

PROCEDURE StoreList (hwnd: OS2.HWND);

    (* Stores a RealmList into the INI file.  The hwnd parameter is the *)
    (* handle of the listbox.  We assume that the INI file is already   *)
    (* open.                                                            *)

    CONST NameLength = MAX(HostNameIndex)+1;

    VAR bufptr: CharArrayPointer;
        BufferSize: CARDINAL;
        j, k, count, index: CARDINAL;
        name: HostName;

    BEGIN
        (* Work out how much buffer space we need. *)

        BufferSize := 0;
        count := OS2.ULONGFROMMR(OS2.WinSendMsg (hwnd, OS2.LM_QUERYITEMCOUNT, NIL, NIL));
        IF count > 0 THEN
            FOR index := 0 TO count-1 DO
                INC (BufferSize,
                     OS2.ULONGFROMMR(OS2.WinSendMsg (hwnd, OS2.LM_QUERYITEMTEXTLENGTH,
                                   OS2.MPFROMUSHORT(index), NIL)) + 1);
            END (*FOR*);
        END (*IF*);

        (* Create the string buffer. *)

        IF BufferSize = 0 THEN
            bufptr := NIL;
        ELSE
            INC (BufferSize);
            ALLOCATE (bufptr, BufferSize);
        END (*IF*);

        (* Store all the strings into the buffer. *)

        IF count > 0 THEN
            j := 0;
            FOR index := 0 TO count-1 DO
                OS2.WinSendMsg (hwnd, OS2.LM_QUERYITEMTEXT,
                                OS2.MPFROM2USHORT(index, NameLength), SYSTEM.ADR(name));
                k := 0;
                REPEAT
                    bufptr^[j] := name[k];
                    INC (k);  INC (j);
                UNTIL (name[k] = Nul) OR (k = NameLength);
                bufptr^[j] := Nul;
                INC (j);
            END (*FOR*);

            bufptr^[j] := Nul;

        END (*IF*);

        (* Write the buffer to the INI file. *)

        IF BufferSize = 0 THEN
            RINIData.INIPutBinary (DomName, "Realms", j, 0);
        ELSE
            RINIData.INIPutBinary (DomName, "Realms", bufptr^, BufferSize);
        END (*IF*);

        (* Deallocate the buffer space. *)

        IF BufferSize > 0 THEN
            DEALLOCATE (bufptr, BufferSize);
        END (*IF*);

    END StoreList;

(************************************************************************)

PROCEDURE StoreData (hwnd: OS2.HWND);

    BEGIN
        EVAL (RINIData.OpenINIFile (INIname));

        (* List of realms for the domain. *)

        StoreList (OS2.WinWindowFromID (hwnd, DID.RealmList));

        RINIData.CloseINIFile;

    END StoreData;

(************************************************************************)
(*                     DELETING THE DATA FOR A REALM                    *)
(************************************************************************)

PROCEDURE DeleteRealmKey (name: ARRAY OF CHAR);

    (* Removes the key for this realm. *)

    VAR RealmKey: ARRAY [0..255] OF CHAR;

    BEGIN
        EVAL (RINIData.OpenINIFile (INIname));
        Strings.Assign ("realm-", RealmKey);
        Strings.Append (name, RealmKey);
        RINIData.INIDeleteKey (DomName, RealmKey);
        RINIData.CloseINIFile;
    END DeleteRealmKey;

(************************************************************************)
(*                WINDOW PROCEDURE FOR SUBCLASSED CASE                  *)
(************************************************************************)

PROCEDURE ["SysCall"] SubWindowProc (hwnd     : OS2.HWND;
                                     msg      : OS2.ULONG;
                                     mp1, mp2 : OS2.MPARAM): OS2.MRESULT;

    (* Window procedure to intercept some of the things that happen in  *)
    (* the notebook subwindow.  We want this here mainly so that we can *)
    (* detect a new font dropped on the notebook tabs.  If the message  *)
    (* is something we don't want to deal with here, we pass it         *)
    (* to the parent window procedure.                                  *)

    VAR OldWndProc: OS2.PFNWP;
        owner: OS2.HWND;
        length, AttrFound: CARDINAL;
        NewFontName: CommonSettings.FontName;

    BEGIN
        OldWndProc := SYSTEM.CAST (OS2.PFNWP, OS2.WinQueryWindowPtr (hwnd, OS2.QWL_USER));
        owner := OS2.WinQueryWindow(hwnd,OS2.QW_OWNER);

        (* Because of the interaction between subclassing and DragText, *)
        (* some messages will go lost if we use the obvious strategy of *)
        (* sending them through to OldWndProc.  To get around this, we  *)
        (* have to send those messages directly to the target window.   *)

        IF (msg = OS2.WM_BUTTON2DOWN) OR (msg = OS2.DM_DRAGOVER)
                   OR (msg = OS2.DM_DRAGLEAVE) OR (msg = OS2.DM_DROP) THEN

            RETURN OS2.WinSendMsg (owner, msg, mp1, mp2);

        (* Check for font or colour change. *)

        ELSIF msg = OS2.WM_PRESPARAMCHANGED THEN

            IF ChangeInProgress THEN
                RETURN OS2.WinDefDlgProc(hwnd, msg, mp1, mp2);
            ELSE
                ChangeInProgress := TRUE;
                length := OS2.WinQueryPresParam (hwnd, OS2.PP_FONTNAMESIZE, 0,
                                             AttrFound, CommonSettings.FontNameSize, NewFontName,
                                              0(*OS2.QPF_NOINHERIT*));
                IF length < CommonSettings.FontNameSize THEN
                    NewFontName[length] := Nul;
                END (*IF*);
                ChangeInProgress := FALSE;
                RETURN NIL;
            END (*IF*);

        END (*IF*);

        RETURN OldWndProc (hwnd, msg, mp1, mp2);

    END SubWindowProc;

(************************************************************************)
(*                   WINDOW PROCEDURE FOR THIS DIALOGUE                 *)
(************************************************************************)

PROCEDURE ["SysCall"] DialogueProc(hwnd     : OS2.HWND
                     ;msg      : OS2.ULONG
                     ;mp1, mp2 : OS2.MPARAM): OS2.MRESULT;

    VAR Title: ARRAY [0..255] OF CHAR;
        listwindow: OS2.HWND;
        ButtonID, NotificationCode: CARDINAL;
        index: INTEGER;
        oldname, name: ARRAY HostNameIndex OF CHAR;
        message: ARRAY [0..127] OF CHAR;

    BEGIN
        index := OS2.LONGFROMMR(
                   OS2.WinSendDlgItemMsg (hwnd, DID.RealmList, OS2.LM_QUERYSELECTION, NIL, NIL));
        CASE msg OF
           |  OS2.WM_INITDLG:
                   Title := "RealmListDlg";
                   INIData.SetInitialWindowPosition (hwnd, SetupINIname,
                                                                     Title);
                   OS2.WinSetWindowPtr (OurHWND, OS2.QWL_USER,
                               SYSTEM.CAST(SYSTEM.ADDRESS,
                                    OS2.WinSubclassWindow (OurHWND,
                                                           SubWindowProc)));
                   OS2.WinEnableWindow (OS2.WinWindowFromID(hwnd, DID.RenameRealm), FALSE);
                   OS2.WinEnableWindow (OS2.WinWindowFromID(hwnd, DID.EditRealm), FALSE);
                   OS2.WinEnableWindow (OS2.WinWindowFromID(hwnd, DID.PromoteRealm), FALSE);
                   OS2.WinEnableWindow (OS2.WinWindowFromID(hwnd, DID.DeleteRealm), FALSE);

           |  CommonSettings.FONTCHANGED:

                   IF ChangeInProgress THEN
                       RETURN OS2.WinDefDlgProc(hwnd, msg, mp1, mp2);
                   ELSE
                       ChangeInProgress := TRUE;
                       SetPageFonts;
                       ChangeInProgress := FALSE;
                       RETURN NIL;
                   END (*IF*);

           |  OS2.WM_COMMAND:

                   listwindow := OS2.WinWindowFromID(hwnd,DID.RealmList);
                   ButtonID := OS2.SHORT1FROMMP(mp1);

                   IF ButtonID = DID.RlistOK THEN
                       StoreData (hwnd);
                       OS2.WinPostMsg (hwnd, OS2.WM_CLOSE, mp1, mp2)
                   ELSIF ButtonID = DID.RlistCancel THEN
                       OS2.WinPostMsg (hwnd, OS2.WM_CLOSE, mp1, mp2)

                   ELSIF ButtonID = DID.AddRealm THEN
                          name := "";
                          StrToBuffer (OurLang, "Local.EnterName", message);
                          OneLine.Edit (hwnd, SetupINIname, message, name);
                          IF name[0] <> Nul THEN
                              IF index = OS2.LIT_NONE THEN
                                  index := 0;
                              ELSE
                                  INC(index);
                              END (*IF*);
                              OS2.WinSendDlgItemMsg (hwnd, DID.RealmList, OS2.LM_INSERTITEM,
                                     OS2.MPFROMSHORT(index), SYSTEM.ADR(name));
                              OS2.WinSendDlgItemMsg (hwnd, DID.RealmList, OS2.LM_SELECTITEM,
                                     OS2.MPFROMSHORT(index), OS2.MPFROMSHORT(ORD(TRUE)));
                          END (*IF*);
                          RealmEditor.Edit (hwnd, DomName, name);
                          Changed := TRUE;

                   ELSIF ButtonID = DID.RenameRealm THEN
       
                          OS2.WinSendMsg (listwindow, OS2.LM_QUERYITEMTEXT,
                                          OS2.MPFROM2USHORT(index, MAX(HostNameIndex)+1), SYSTEM.ADR(name));
                          oldname := name;
                          StrToBuffer (OurLang, "RealmList.NewName", message);
                          OneLine.Edit (hwnd, SetupINIname, message, name);
                          OS2.WinSendDlgItemMsg (hwnd, DID.RealmList,
                                  OS2.LM_SETITEMTEXT, OS2.MPFROMSHORT(index), SYSTEM.ADR(name));
                          Changed := TRUE;

                   ELSIF ButtonID = DID.EditRealm THEN

                          OS2.WinSendMsg (listwindow, OS2.LM_QUERYITEMTEXT,
                                          OS2.MPFROM2USHORT(index, MAX(HostNameIndex)+1), SYSTEM.ADR(name));
                          RealmEditor.Edit (hwnd, DomName, name);
                          Changed := TRUE;

                   ELSIF ButtonID = DID.PromoteRealm THEN
       
                          OS2.WinSendMsg (listwindow, OS2.LM_QUERYITEMTEXT,
                                          OS2.MPFROM2USHORT(index, MAX(HostNameIndex)+1), SYSTEM.ADR(name));
                          OS2.WinSendMsg (listwindow, OS2.LM_DELETEITEM,
                                                 OS2.MPFROMSHORT(index), NIL);
                          DEC (index);
                          OS2.WinSendMsg (listwindow, OS2.LM_INSERTITEM,
                                 OS2.MPFROMSHORT(index), SYSTEM.ADR(name));
                          OS2.WinSendMsg (listwindow, OS2.LM_SELECTITEM,
                                 OS2.MPFROMSHORT(index), OS2.MPFROMSHORT(ORD(TRUE)));
                          Changed := TRUE;

                   ELSIF ButtonID = DID.DeleteRealm THEN

                          OS2.WinSendMsg (listwindow, OS2.LM_QUERYITEMTEXT,
                                          OS2.MPFROM2USHORT(index, MAX(HostNameIndex)+1), SYSTEM.ADR(name));
                          DeleteRealmKey (name);
                          OS2.WinSendDlgItemMsg (hwnd, DID.RealmList, OS2.LM_DELETEITEM,
                                                 OS2.MPFROMSHORT(index), NIL);
                          OS2.WinEnableWindow (OS2.WinWindowFromID(hwnd, DID.RenameRealm), FALSE);
                          OS2.WinEnableWindow (OS2.WinWindowFromID(hwnd, DID.EditRealm), FALSE);
                          OS2.WinEnableWindow (OS2.WinWindowFromID(hwnd, DID.PromoteRealm), FALSE);
                          OS2.WinEnableWindow (OS2.WinWindowFromID(hwnd, DID.DeleteRealm), FALSE);
                          Changed := TRUE;
                   END (*IF*);

                   IF index > 0 THEN
                       OS2.WinEnableWindow (OS2.WinWindowFromID(hwnd, DID.PromoteRealm), TRUE);
                   ELSE
                       OS2.WinEnableWindow (OS2.WinWindowFromID(hwnd, DID.PromoteRealm), FALSE);
                   END (*IF*);

                   RETURN NIL;

           |  OS2.WM_CONTROL:

                   NotificationCode := OS2.ULONGFROMMP(mp1);
                   ButtonID := NotificationCode MOD 65536;
                   NotificationCode := NotificationCode DIV 65536;

                   IF ButtonID = DID.RealmList THEN
                       IF NotificationCode = OS2.LN_SELECT THEN
                           OS2.WinEnableWindow (OS2.WinWindowFromID(hwnd, DID.RenameRealm), TRUE);
                           OS2.WinEnableWindow (OS2.WinWindowFromID(hwnd, DID.EditRealm), TRUE);
                           OS2.WinEnableWindow (OS2.WinWindowFromID(hwnd, DID.DeleteRealm), TRUE);
                           IF index > 0 THEN
                               OS2.WinEnableWindow (OS2.WinWindowFromID(hwnd, DID.PromoteRealm), TRUE);
                           ELSE
                               OS2.WinEnableWindow (OS2.WinWindowFromID(hwnd, DID.PromoteRealm), FALSE);
                           END (*IF*);
                           RETURN NIL;
                       ELSIF NotificationCode = OS2.LN_ENTER THEN
                           (* Treat this one as if the edit button had been clicked. *)
                           OS2.WinSendMsg (hwnd, OS2.WM_COMMAND,
                                 OS2.MPFROMSHORT(DID.EditRealm), NIL);
                           RETURN NIL;
                       ELSE
                           RETURN OS2.WinDefDlgProc(hwnd, msg, mp1, mp2);
                       END (*IF*);
                   ELSE
                       RETURN OS2.WinDefDlgProc(hwnd, msg, mp1, mp2);
                   END (*IF*);

           |  OS2.WM_CLOSE:
                   CommonSettings.EnableFontChanges(FALSE);
                   RETURN OS2.WinDefDlgProc(hwnd, msg, mp1, mp2);

        ELSE    (* default *)
           RETURN OS2.WinDefDlgProc(hwnd, msg, mp1, mp2);
        END (*CASE*);
        RETURN NIL;
    END DialogueProc;

(**********************************************************************)
(*                            MAIN PROGRAM                            *)
(**********************************************************************)

PROCEDURE Edit (owner: OS2.HWND;  VAR (*INOUT*) dname: DomainName);

    (* Opens and allows editing of the Realm list for this domain. *)

    VAR Lang: LangHandle;
        LangName: LanguageString;

    BEGIN
        OS2.DosError (OS2.FERR_DISABLEHARDERR);
        CommonSettings.GetINIFilename (INIname);
        SetupINIname := "Setup.INI";
        SetupINIname[6] := INIname[LENGTH(INIname)-3];
        Strings.Assign (dname, DomName);
        Changed := FALSE;
        OurHWND := OS2.WinLoadDlg(OS2.HWND_DESKTOP, owner,
                       DialogueProc,    (* dialogue procedure *)
                       0,                   (* use resources in EXE *)
                       DID.RealmListDlg,                (* dialogue ID *)
                       NIL);               (* creation parameters *)
        Remote.SetInitialWindowPosition (OurHWND, "RealmListDlg");
        CommonSettings.CurrentLanguage (Lang, LangName);
        SetLanguage (Lang);
        LoadDialogueData (OurHWND);

        OS2.WinShowWindow (owner, FALSE);
        OS2.WinProcessDlg(OurHWND);
        OS2.WinShowWindow (owner, TRUE);

        Remote.StoreWindowPosition (OurHWND, "RealmListDlg", TRUE);
        OS2.WinDestroyWindow (OurHWND);
    END Edit;

(************************************************************************)

BEGIN
    ChangeInProgress := FALSE;
END RealmsPage.

