/***************************************************************************/
/* MsgPost Version 1.00 Jan 1992           (C) 1992 by CodeLand Australia, */
/*                                                    All Rights Reserved. */
/* Written by Colin Wheat of Fidonet 3:690/613                             */
/* Compiled using Turbo C++ v1.00                                          */
/*                                                                         */
/* MsgPost uses the Squish Message Base Level 0 MsgAPI                     */
/* Squish is a trademark of Scott J. Dudley                                */
/*                                                                         */
/* Command line:                                                           */
/* -T<name>  Message text file path and name                               */
/* -C<name>  Configuration file path and name                              */
/* -M<name>  Message area path and name                                    */
/* -N<addr>  Netmail format - Send to address\n"                           */
/* -S[##]    Split long messages to ## Kb size (0 - 16)                    */
/* -PX[X]    Message priority flag(s)                                      */
/* -K        Kill text file after processing                               */
/* -?        Program help screen                                           */
/*                                                                         */
/* Error levels:                                                           */
/* 0 - Normal exit                                                         */
/* 1 - Syntax exit                                                         */
/* 2 - Out of memory                                                       */
/* 3 - Text file not found                                                 */
/* 4 - No system address set                                               */
/* 5 - Text file is blank                                                  */
/* 6 - Message base open failed                                            */
/*                                                                         */
/***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <sys/timeb.h>
#include <io.h>
/* #include <dir.h> */

#if defined(__EMX__)
  #include <sys/param.h>
  #include <dos.h>
#else
  #define INCL_NOPM
  #define INCL_ORDINALS
  #include <os2.h>
  #define MAXPATHLEN ORD_DOSMAXPATHLEN
#endif

#include "prog.h"
#include "msgapi.h"             /* Squish API header */

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

#define VERSION   "1.0/32"      /* MsgPost version   */
#define MAX_BLOCK 256000        /* Maximum body size */
#define MAX_LINE  10000         /* Maximum lines     */

int kill_txt=0;                 /* Txtfile kill flag */
int netmail=0;                  /* Netmail area      */
int ga=0;                       /* Got address flag  */
int split_k=12;                 /* Message split     */
unsigned long seed;             /* MSGID seed        */
time_t time_now;                /* Creation time     */
int mn;                         /* Split unique #    */

dword attr=MSGLOCAL;            /* Message attr      */
struct _netaddr myaddr;         /* System address    */
struct _netaddr toaddr;         /* Netmail to addr   */

char homepath[MAXPATHLEN+1];              /* Executable path   */
char msgarea[MAXPATHLEN+1];               /* Message area name */
char msgtext[MAXPATHLEN+1];               /* Import text file  */
char cfgname[MAXPATHLEN+1];               /* MsgPost CFG file  */

char str_to[XMSG_TO_SIZE];
char str_from[XMSG_FROM_SIZE];
char str_subj[XMSG_SUBJ_SIZE];
char str_orig[80];
char str_chrs[80];
char *def_orig = "TeX-Box +49-6034-1455 24h V.32b V.42b";

int linesnum=0;
char *lines[MAX_LINE];
long linesbytes=0L;

long body_len;                  /* Body length       */
char *body;                     /* Message body      */

static void write_msg (MSG *ap);
static int get_num (int splitbytes);
static int build_body (int limit);
static void read_cfg (void);
static int read_txt (void);
static int set_msgcfg (char *s);
static void set_attr (char *p);
static int read_orig (void);
static void build_tear (char *s);
static void build_hdr (XMSG *x, int num, int maxnum);
static void build_ctrl (char *str, int *len, int num, int maxnum);
static unsigned long hsec_time (void);
static void get_addr (char *str, struct _netaddr *addr);
static void add_slash (char *str);
static void strip_slash(char *str);
static void strip_cr (char *str);
static void strtrim (char *str);
static int strblank (char *str);
static void cvt_us (char *s);
static char *fancy_s (char *string);
static void make_ourpath (char *pth);
static void setup (int argc, char *argv[]);
static void parse_cmd_line (int argc, char *argv[]);
static void syntax_exit (void);
static void near InitCvt(void);
struct tm * _fast DosDate_to_TmDate(union stamp_combo *dosdate,
				    struct tm *tmdate);
union stamp_combo *TmDate_to_DosDate(struct tm *tmdate,
				     union stamp_combo *dosdate);


void main (int argc, char *argv[]) {
  int areatyp;                /* Message area type */
  struct _minf mi;            /* API structure     */
  MSG *ap;                    /* API area pointer  */

  printf("\nMsgPost " VERSION " - the Fidonet/Squish Message Base Writer"
	 "\n   (C) Copyright 1992 by CodeLand, All Rights Reserved\n\n");

  setup(argc,argv);
  read_cfg(); /* Setup */
  if (!read_txt()) { printf("\nError; Text file not found!\n\n"); exit(3); }
  if (!linesbytes) { printf("\nError; Text file is blank!\n\n"); exit(5); }
  parse_cmd_line(argc,argv); /* Get command line overrides */

  /* Check we have a system address */
  if (!ga) { printf("\nError; No address set!\n\n"); exit(4); }
  mi.def_zone=myaddr.zone; mi.req_version=0; /* Set api */

  /* If no origin set, try from the message area, else use default */
  if (!*str_orig  && !read_orig())
    strcpy(str_orig,def_orig);

  MsgOpenApi(&mi);
  areatyp=*msgarea=='$' ? MSGTYPE_SQUISH : MSGTYPE_SDM;
  if (areatyp==MSGTYPE_SDM && !netmail) areatyp|=MSGTYPE_ECHO;

  /* Open the message base */
  if ((ap=MsgOpenArea((byte *)msgarea+(*msgarea=='$'),
		      MSGAREA_NORMAL,areatyp))==NULL) {
    printf("\nError; Message base open failed!\n\n");
    exit(6);
  }

  MsgLock(ap);            /* Lock the base        */
  write_msg(ap);          /* Write the message(s) */
  MsgCloseArea(ap);       /* Close the base       */

  MsgCloseApi();
  if (kill_txt) unlink(msgtext);
  printf("\nDone!\n");
  exit(0);
}


/*
** write_msg ()
** write message(s)
*/

static void write_msg (MSG *ap) {
  register i;
  register split_num=1;       /* How many msgs     */
  register splitbytes;        /* Split size        */
  MSGH *msg;                  /* Message pointer   */
  XMSG xmsg;                  /* Xmsg structure    */
  int ctrllen;                /* Control info size */
  char ctrl[256];             /* Control info      */
  char tear[256];             /* Tear line, origin */

  time_now=time(NULL); /* Get message time */
  seed=hsec_time();    /* Get MSGID seed */
  srand((unsigned)seed); mn=rand(); /* Get split messages unique number */
  build_tear(tear); /* Make tear/origin line for echomail */
  
  /* Set each message size */
  if (!split_k) splitbytes=MAX_BLOCK;
  else splitbytes=split_k*1024;
  if (splitbytes>MAX_BLOCK) splitbytes=MAX_BLOCK;
  
  /* Adjust for tear and header */
  if (!netmail) splitbytes-=strlen(tear);
  splitbytes-=250;
  
  if (linesbytes>splitbytes) { /* Split or cut */
    if (split_k) {
      split_num=get_num(splitbytes); /* Compute number of messages */
      /* Set to average message size for neatness :-) */
      splitbytes=(int)(linesbytes/(long)split_num);
      for (;;) { /* And adjust size back up so no text is lost */
	if (get_num(splitbytes)<=split_num) break;
	splitbytes++;
      }
    }
  }
  
  printf("Writing \"%s\"  01/%02u ",fancy_s(msgarea),split_num);
  for (i=0; i<split_num; i++) {
    printf("\b\b\b\b\b\b%02u/%02u ",i+1,split_num);
    if (!build_body(splitbytes)) break;
    if (!netmail) { strcat(body,tear); body_len+=(long)strlen(tear); }
    if (i) attr &= ~MSGFILE;	/* Kill file attaches on 2nd, 3rd, etc */
    build_hdr(&xmsg,i+1,split_num);
    msg=MsgOpenMsg(ap,MOPEN_CREATE,0L); /* Create new message */
    build_ctrl(ctrl,&ctrllen,i+1,split_num);
    MsgWriteMsg(msg,FALSE,&xmsg,NULL,0L,body_len,ctrllen,(byte *)ctrl);
    MsgWriteMsg(msg,TRUE,NULL,(byte *)body,body_len,body_len,0L,NULL);
    MsgCloseMsg(msg);
  }
  printf("\n");
}


/*
 ** get_num ()
 ** Compute how many split messages to generate
 */

static int get_num (int splitbytes) {
  register i, result=1;
  long len=0L;

  /* Scan through the lines */
  for (i=0; i<linesnum; i++) {
    len+=(long)strlen(lines[i]);
    if (len>(long)splitbytes) {
      result++;
      len=(long)strlen(lines[i]);
    }
  }
  return result;
}


/*
 ** build_body ()
 ** Fill the body buffer from the lines array
 */

static int build_body (int limit) {
  register len;
  static int linescount=0;
  char *p;

  if (linescount>=linesnum) return 0;
  p=body; body_len=0L;

  for (;;) {
    len=strlen(lines[linescount]);
    if (body_len+(long)len>limit) break;
    memcpy(p,lines[linescount],len);
    p+=len; body_len+=(long)len;
    if (++linescount>=linesnum) break;
  }
  *p='\0'; body_len++;

  return 1;
}


/*
 ** read_cfg ()
 ** read configuration file
 */

static void read_cfg (void) {
  FILE *fp;
  char line[258];

  printf("Reading \"%s\" ",fancy_s(cfgname));
  if ((fp=fopen(cfgname,"r"))==NULL) {
    printf("(Not found)\n");
    return;
  }
  while (fgets(line,256,fp))
    if (*line!=';')
      set_msgcfg(line);
  fclose(fp);
  printf("\n");
}


/*
 ** read_txt ()
 ** read text file into memory
 */

static int read_txt (void) {
  register llen, checkcfg=1, trimleading=1;
  FILE *fp;
  char *fbuf=NULL;
  char line[258];

  printf("Reading \"%s\"\n",fancy_s(msgtext));
  if ((fp=fopen(msgtext,"r"))==NULL)
    return 0;
  fbuf=(char *)malloc(4096);	/* Get extended file buffer */
  if (fbuf)
    setvbuf(fp,fbuf,_IOFBF,4096);

  while (fgets(line,256,fp)) {

    if (checkcfg) {		/* Check for override configuration */
      if (*line==';') continue;
      if (set_msgcfg(line)) continue;
      else checkcfg=0;
    }

    strip_cr(line); strtrim(line); /* Clean up */

    /* Ignore leading blank lines */
    if (trimleading) {
      if (strblank(line)) continue;
      else trimleading=0;
    }

    strcat(line,"\r"); llen=strlen(line);
    if ((lines[linesnum]=(char *)malloc(llen+1))==NULL) {
      printf("\nError; Out of memory!\n\n");
      fclose(fp);
      exit(2);
    }

    memcpy(lines[linesnum],line,llen+1);
    linesbytes+=(long)llen;	/* Add to total body bytes */
    if (++linesnum>=MAX_LINE) {
      fprintf(stderr,"Maximum number of lines (%d) reached.\n",MAX_LINE);
      break;
    }
  }

  if (linesnum<1) {
    printf("\nError; no control information in file!\n\n");
    fclose(fp);
    exit(2);
  }

  /* Strip trailing blank lines */
  for (;;) {			/* Step back */
    if (!strblank(lines[linesnum-1])) break;
    linesbytes-=(long)strlen(lines[linesnum-1]);
    if (!--linesnum) break;
  }

  fclose(fp);
  if (fbuf) free(fbuf);
  return 1;
}


/*
 ** set_msgcfg ()
 ** Set message configuration from configuration string
 */

static int set_msgcfg (char *s) {
  register status=0;
  char *p, *q, *r, sbuf[258];

  strcpy(sbuf,s);  /* Clone the line so we don't change the original */

  if ((p=strtok(sbuf," \t\n\r"))!=NULL) {
    if ((q=strtok(NULL,"\n\r"))!=NULL) {
      r=q;
      while (*r==' '||*r=='\t')
	r++; /* Strip leading */
      strtrim(r);		/* Strip trailing */
      if (*r) {
	if (!stricmp(p,"To:")) {
	  strncpy(str_to,r,XMSG_TO_SIZE-1);
	  str_to[XMSG_TO_SIZE-1]='\0';
	  status=1;
	}
	else if (!stricmp(p,"From:")) {
	  strncpy(str_from,r,XMSG_FROM_SIZE-1);
	  str_from[XMSG_FROM_SIZE-1]='\0';
	  status=1;
	}
	else if (!stricmp(p,"Subj:")) {
	  strncpy(str_subj,r,XMSG_SUBJ_SIZE-1);
	  str_subj[XMSG_SUBJ_SIZE-1]='\0';
	  status=1;
	}
	else if (!stricmp(p,"Origin:")) {
	  strncpy(str_orig,r,59);
	  str_orig[59]='\0';
	  status=1;
	}
	else if (!stricmp(p,"Charset:")) {
	  strncpy(str_chrs,r,79);
	  str_chrs[79]='\0';
	  status=1;
	}
	else if (!stricmp(p,"Address:")) {
	  get_addr(r,&myaddr); ga++;
	  status=1;
	}
	else if (!stricmp(p,"Attr:")) {
	  set_attr(r);
	  status=1;
	}
	else if (!stricmp(p,"Area:")) {
	  strncpy(msgarea,r,78); msgarea[78]='\0';
	  strip_slash(msgarea);
	  status=1;
	}
	else if (!stricmp(p,"Netmail:")) {
	  get_addr(r,&toaddr); netmail=1;
	  status=1;
	}
      }
    }
  }

  return status;
}


/*
 ** set_msgattr ()
 ** Set message attribute flags
 */

static void set_attr (char *p) {
  while (*p) {
    *p=tolower(*p);
    switch (*p) {
    case 'c':			/* Crash */
      attr |= MSGCRASH;
      attr &= ~MSGHOLD;
      break;
    case 'f':			/* File attach */
      attr |= MSGFILE;
      break;
    case 'h':			/* Hold */
      attr |= MSGHOLD;
      attr &= ~MSGCRASH;
      break;
    case 'k':			/* Kill after sending */
      attr |= MSGKILL;
      break;
    case 'p':			/* Private */
      attr |= MSGPRIVATE;
      break;
    case 'r':			/* File Request */
      attr |= MSGFRQ;
      break;
    case 'u':			/* File Update Request */
      attr |= MSGURQ;
      break;
    }
    p++;
  }
}


/*
 ** read_orig ()
 ** read message area origin file
 */

static int read_orig (void) {
  FILE *fp;
  char *p, buf[128];

  if (*msgarea=='$') { strcpy(buf,msgarea+1); strcat(buf,".SQO"); }
  else { strcpy(buf,msgarea); strcat(buf,"\\ORIGIN"); }
  printf("Reading \"%s\"\n",fancy_s(buf));

  if ((fp=fopen(buf,"r"))==NULL) return 0;
  fgets(buf,128,fp);
  fclose(fp);

  p=buf;
  while (*p==' '||*p=='\t')
    p++; /* Strip leading */
  if (strlen(p)>59) p[59]='\0';
  strip_cr(p); strtrim(p);
  strcpy(str_orig,p);
  return 1;
}


/*
 ** build_tear ()
 ** read text file
 */

static void build_tear (char *s) {
  char line[90], sbuf[20];

  *s='\0';
  if (!netmail) {
    /* Tear line */
    strcpy(s,"\r--- MsgPost/2 " VERSION "\r");
    /* Origin line */
    if (myaddr.point) sprintf(sbuf,".%u",myaddr.point);
    else *sbuf='\0';
    sprintf(line," * Origin: %s (%u:%u/%u%s)\r",
	    str_orig,myaddr.zone,myaddr.net,myaddr.node,sbuf);
    strcat(s,line);
    /* Seen-by line */
    sprintf(line,"SEEN-BY: %u/%u\r",myaddr.net,myaddr.node);
    strcat(s,line);
  }
}


/*
 ** build_hdr ()
 ** Set message header information
 */

static void build_hdr (XMSG *x, int num, int maxnum) {
  union stamp_combo combo;
  struct tm *tmdate;

  memset(x,'\0',sizeof(XMSG));

  x->attr=attr;			/* Set message attributes */

  x->orig=x->dest=myaddr;
  if (netmail) x->dest=toaddr;

  strcpy((char *)x->to,str_to);
  strcpy((char *)x->from,str_from);

  if (maxnum>1) {
    sprintf((char *)x->subj,"(#%d / %d) ",num,maxnum);
    strncat((char *)x->subj,str_subj,
	    XMSG_SUBJ_SIZE-1-strlen((char *)x->subj));
    (x->subj)[XMSG_SUBJ_SIZE-1]='\0';
  }
  else strcpy((char *)x->subj,str_subj);

  tmdate=localtime(&time_now);
  x->date_written=
  x->date_arrived=(TmDate_to_DosDate(tmdate,&combo))->msg_st;
}


/*
 ** build_ctrl ()
 ** set message control information
 */

static void build_ctrl (char *str, int *len, int num, int maxnum) {
  struct tm *t;
  unsigned long time_new;
  char ubuf[20], sbuf[80];

  *len=0;
  time_new=hsec_time(); if (seed<time_new) seed=time_new;

  if (myaddr.point)
    sprintf(str,"\01MSGID: %u:%u/%u.%u %08lx",
	    myaddr.zone,myaddr.net,myaddr.node,myaddr.point,seed);
  else
    sprintf(str,"\01MSGID: %u:%u/%u %08lx",
	    myaddr.zone,myaddr.net,myaddr.node,seed);

  if (*str_chrs) {
    sprintf(sbuf,"\001CHRS: %s 2",str_chrs);
    strcat(str,sbuf);
  }
  /* ^ASPLIT: 30 Mar 90 11:12:34 @494/4       12345 02/03 +++++++++++ */
  if (maxnum>1) {
    t=localtime(&time_now);
    sprintf(ubuf,"@%u/%u",myaddr.net,myaddr.node); ubuf[12]='\0';
    strftime(sbuf,sizeof(sbuf)-2,"\01SPLIT: %d %h %y %T",t);
    sprintf(sbuf+strlen(sbuf),
	    " %-12s %05u %02u/%02u +++++++++++",
	    ubuf,mn,num,maxnum);
    strcat(str,sbuf);
  }

  *len=strlen(str)+1; seed++;
}


/*
 ** hsec_time ()
 ** get system date/time in hsec's
*/

static unsigned long hsec_time (void) {
  struct timeb t;

  ftime(&t);
  return t.millitm;
}


/*
** get_addr ()
** get command line system address
*/

static void get_addr (char *str, struct _netaddr *addr) {
  char *p;

  if ((p=strchr(str,':'))!=NULL) {
    addr->zone=atoi(str);
    str=p+1;
  }
  else addr->zone=myaddr.zone;
    
  if ((p=strchr(str,'/'))!=NULL) {
    addr->net=atoi(str);
    str=p+1;
  }
  else addr->net=0;
    
  if ((p=strchr(str,'.'))!=NULL) {
    addr->node=atoi(str);
    str=p+1;
    addr->point=atoi(str);
  }
  else {
    addr->node=atoi(str);
    addr->point=0;
  }
}


/*
 ** add_slash (char *str)
 ** Add trailing back slash
 */

static void add_slash (char *str) {
  char *p;

  p=str;
  while (*p) p++;
  if (p != str) {
    p--;
    if (*p != '\\')
      strcat(str,"\\");
  }
}


/*
 ** strip_slash ()
 ** strip trailing back slash
 */

static void strip_slash(char *str) {
  register i;

  for (i=strlen(str)-1; (str[i]=='\\') && i>=0; i--)
    ;
  str[i+1]='\0';
}


/*
 ** strip_cr ()
 ** strip trailing cr/lf
 */

static void strip_cr (char *str) {
  register i;

  for (i=strlen(str)-1; (str[i]=='\r'||str[i]=='\n') && i>=0; i--) ;
  str[i+1]='\0';
}


/*
 ** strtrim ()
 ** strip trailing white space
 */

static void strtrim (char *str) {
  register int i;

  for (i=strlen(str)-1; isspace(str[i]) && i>=0; i--) ;
  str[i+1]='\0';
}


/*
 ** strblank ()
 ** is string blank?
 */

static int strblank (char *str) {
  register char *p;
  register int blank=1;

  for (p=str; *p; p++) {
    if (!isspace(*p)) {
      blank=0;
      break;
    }
  }
  return blank;
}


/*
 ** cvt_us ()
 ** convert underscore characters to spaces
 */

static void cvt_us (char *s) {
  while (*s) {
    if (*s=='_')
      *s=' ';
    s++;
  }
}

/*
 ** fancy_s ()
 ** Make It Look Like This
 */

static char *fancy_s (char *string) {
  register flag=0;
  char *s;

  s=string;
  while (*string) {
    if (isalpha(*string)) {	/* If alphabetic,     */
      if (flag)
	*string=tolower(*string); /* already saw one?   */
      else {
	flag = 1;		/* first one, flag it */
	*string=toupper(*string); /* Uppercase it       */
      }
    }
    else
      flag=0;		/* reset alpha flag   */
    string++;
  }
  return s;
}


/*
 ** make_ourpath ()
 ** get executable path
 */

static void make_ourpath (char *pth) {
  char drive[_MAX_DRIVE], dir[_MAX_DIR], name[_MAX_FNAME], ext[_MAX_EXT];

  _splitpath(pth,drive,dir,name,ext);
  strcpy(homepath,drive); strcat(homepath,dir);
  add_slash(homepath);
}


/*
 ** setup ()
 ** program setup
 */

static void setup (int argc, char *argv[]) {
  register i;
  char *p;

  /* Get the text storage */
  if ((body=malloc(MAX_BLOCK+1024))==NULL) {
    printf("\nError; Out of memory!\n\n");
    exit(2);
  }
  memset(body,'\0',MAX_BLOCK+1024); /* Clear it */

  *str_orig=*msgarea=*msgtext='\0';
  make_ourpath(argv[0]);
  strcpy(cfgname,homepath); strcat(cfgname,"MsgPost.Cfg");
  strcpy(str_to,"All");
  strcpy(str_from,"MsgPost " VERSION);
  strcpy(str_subj,"Automated Posting");

  /* Get file names */
  for (i=1; i<argc; i++) {
    p=argv[i];
    if (*p=='-'||*p=='/') {
      switch(tolower(*(++p))) {
      case 'c': strcpy(cfgname,++p); break;
      case 't': strcpy(msgtext,++p); break;
      case 'a':
      case 'f':
      case 'j':
      case 'k':
      case 'm':
      case 'n':
      case 'p':
      case 'r':
      case 's': break;
      case '?':
      default : syntax_exit();
      }
    }
    else syntax_exit();
  }
}


/*
 ** parse_cmd_line ()
 ** get command line parameters
 */

static void parse_cmd_line (int argc, char *argv[]) {
  register i;
  char *p, s[60];

    for (i=1; i<argc; i++) {
    p=argv[i];
    if (*p=='-'||*p=='/') {
      switch(tolower(*(++p))) {
      case 'a':
	strcpy(s,++p);
	get_addr(s,&myaddr);
	ga=TRUE;
	break;
      case 'f':
	strncpy(str_from,++p,XMSG_FROM_SIZE-1);
	str_from[XMSG_FROM_SIZE-1]='\0';
	cvt_us(str_from);
	break;
      case 'r':
	strncpy(str_chrs,++p,78);
	str_chrs[79]='\0';
	cvt_us(str_chrs);
	break;
      case 'm':
	strcpy(msgarea,++p);
	strip_slash(msgarea);
	break;
      case 'n':
	strcpy(s,++p);
	get_addr(s,&toaddr);
	netmail=TRUE;
	break;
      case 'p':
	set_attr(++p);
	break;
      case 'j':
	strncpy(str_subj,++p,XMSG_SUBJ_SIZE-1);
	str_subj[XMSG_SUBJ_SIZE-1]='\0';
	cvt_us(str_subj);
	break;
      case 's':
	split_k=atoi(++p);
	if (split_k<0 || split_k>16)
	  split_k=12;
	break;
      case 'k':
	kill_txt=1;
	break;
      }
    }
  }
}


/*
 ** syntax_exit ()
 ** syntax command line display and exit
 */

static void syntax_exit (void) {
  puts("    Syntax:  MsgPost [-switch -switch ... ]\n\n"
       "\t -T<name>    Text file path and name\n"
       "\t -C<name>    Optional configuration file path and name\n"
       "\t -M<name>    Message area path and name override\n"
       "\t -J<subj>    Message subject override (use _ for space)\n"
       "\t -A<addr>    Set my aka to address\n"
       "\t -F<name>    Set name as From:-Name\n"
       "\t -N<addr>    Netmail format - Send to address\n"
       "\t -S[##]      Split long messages to ## Kb size (0 - 16)\n"
       "\t -R<name>    Name of charset used (eg. Latin-1)\n"
       "\t -K          Kill text file after processing\n"
       "\t -?          Program help screen\n"
       );

  exit(1);
}

/***************************************************************************
 *                                                                         *
 *  Squish Developers Kit Source, Version 2.00                             *
 *  Copyright 1989-1994 by SCI Communications.  All rights reserved.       *
 *                                                                         *
 *  USE OF THIS FILE IS SUBJECT TO THE RESTRICTIONS CONTAINED IN THE       *
 *  SQUISH DEVELOPERS KIT LICENSING AGREEMENT IN SQDEV.PRN.  IF YOU DO NOT *
 *  FIND THE TEXT OF THIS AGREEMENT IN THE AFOREMENTIONED FILE, OR IF YOU  *
 *  DO NOT HAVE THIS FILE, YOU SHOULD IMMEDIATELY CONTACT THE AUTHOR AT    *
 *  ONE OF THE ADDRESSES LISTED BELOW.  IN NO EVENT SHOULD YOU PROCEED TO  *
 *  USE THIS FILE WITHOUT HAVING ACCEPTED THE TERMS OF THE SQUISH          *
 *  DEVELOPERS KIT LICENSING AGREEMENT, OR SUCH OTHER AGREEMENT AS YOU ARE *
 *  ABLE TO REACH WITH THE AUTHOR.                                         *
 *                                                                         *
 *  You can contact the author at one of the address listed below:         *
 *                                                                         *
 *  Scott Dudley       FidoNet     1:249/106                               *
 *  777 Downing St.    Internet    sjd@f106.n249.z1.fidonet.org            *
 *  Kingston, Ont.     CompuServe  >INTERNET:sjd@f106.n249.z1.fidonet.org  *
 *  Canada  K7M 5N3    BBS         1-613-634-3058, V.32bis                 *
 *                                                                         *
 ***************************************************************************/

static int is_dst=-1;

/* Find out the current status of daylight savings time */

static void near InitCvt(void) {
  time_t tnow;

  tnow=time(NULL);
  is_dst=!! (localtime(&tnow)->tm_isdst);
}


/* Convert a DOS-style bitmapped date into a 'struct tm'-type date. */

struct tm * _fast DosDate_to_TmDate(union stamp_combo *dosdate,
				    struct tm *tmdate) {
  if (is_dst==-1)
    InitCvt();

  tmdate->tm_mday=dosdate->msg_st.date.da;
  tmdate->tm_mon =dosdate->msg_st.date.mo-1;
  tmdate->tm_year=dosdate->msg_st.date.yr+80;

  tmdate->tm_hour=dosdate->msg_st.time.hh;
  tmdate->tm_min =dosdate->msg_st.time.mm;
  tmdate->tm_sec =dosdate->msg_st.time.ss << 1;

  tmdate->tm_isdst=is_dst;

  return tmdate;
}

/* Convert a 'struct tm'-type date into an Opus/DOS bitmapped date */

union stamp_combo *TmDate_to_DosDate(struct tm *tmdate,
				     union stamp_combo *dosdate) {
  dosdate->msg_st.date.da=tmdate->tm_mday;
  dosdate->msg_st.date.mo=tmdate->tm_mon+1;
  dosdate->msg_st.date.yr=(tmdate->tm_year < 80) ? 0 : tmdate->tm_year-80;

  dosdate->msg_st.time.hh=tmdate->tm_hour;
  dosdate->msg_st.time.mm=tmdate->tm_min;
  dosdate->msg_st.time.ss=tmdate->tm_sec >> 1;

  return dosdate;
}

char *sc_time(union stamp_combo *sc, char *string) {
  struct tm t;

  if (sc->msg_st.date.yr==0)
    *string='\0';
  else {
    DosDate_to_TmDate(sc,&t);
    strftime(string,strlen(string)-1,"%d %e %y %T",&t);
  }

  return string;
}
