/*----------------------------------------------------------
   Stores a value in an INI or TNI file. Although this can
   be run from the command line, it is primarly intended to
   be called from another script.

   Note that it is not easy for a Rexx program to save state
   information from one invocation to the next. On each call
   the INI or TNI file has to be consulted all over again.
   If you need to store a number of INI values, it might
   be more efficient to do the job in a different language.

           Author:       Peter Moylan (peter@pmoylan.org)
           Last revised: 16 November 2020

   Can't yet handle, or untested:
        <nothing at present>

   Usage:
                CALL INI_put filename, app, key, value

            where
                filename is the name of an INI or TNI file,
                including a path if it is not in the
                current working directory. The file is
                assumed to be an INI file if the name
                ends with ".INI", and a TNI file otherwise.
                If the file does not exist then we
                return without storing anything.

                app and key are the labels of an
                (application,key) pair within that file.

                value is a byte string.

------------------------------------------------------------*/

/****************************************************************/
/*                       MAIN PROGRAM                           */
/****************************************************************/

CALL RxFuncAdd SysLoadFuncs, RexxUtil, SysLoadFuncs
CALL SysLoadFuncs

PARSE ARG INIfile, app, key, value
INIfile = STRIP(INIfile)
IF STREAM(INIfile, 'C', 'QUERY EXISTS') = '' THEN DO
    SAY "ERROR: File "INIfile" does not exist"
    RETURN
END
IF (app = "") | (key = "") THEN DO
    SAY "ERROR: Application and key must both be non-null"
    RETURN
END

TNIoption = 'T'
pos = LASTPOS('.', INIfile)
base = INIfile
if pos > 0 THEN
    DO
        extension = TRANSLATE(RIGHT(INIfile, LENGTH(INIfile) - pos))
        IF extension = "INI" THEN TNIoption = 'I'
        base = LEFT(INIfile,pos)
    END

FMTfile = base||"FMT"
IF STREAM(FMTfile, 'C', 'QUERY EXISTS') = '' THEN frmat = ''
ELSE frmat = INI_get( FMTfile, app, key)

CALL INIstore INIfile, TNIoption, app, key, value, frmat
RETURN

/****************************************************************/
/*                SETTING AN INI OR TNI VALUE                   */
/****************************************************************/

INIstore: PROCEDURE

    /* Stores one INI or TNI value, depending on TNIoption. */

    PARSE ARG filename, TNIoption, application, key, value, frmat
    filename = STRIP(filename)
    IF TNIoption = 'T' THEN DO
        CALL SetVal filename, application, key, value, frmat
    END
    ELSE DO
        CALL SysIni filename, application, key, value
    END
    RETURN

/****************************************************************/

SetVal: PROCEDURE

    /* Stores or modifies a value in a TNI file.  */

    PARSE ARG srcfile,app,key,value,frmat
    frmat = STRIP(frmat)

    /* We need the two extra files introduced in this routine   */
    /* to be in the same directory as srcfile, so let us        */
    /* separate the srcfile name into path and filename.   The  */
    /* path must either be empty or end with a '\'.             */

    k = LASTPOS('\', srcfile)
    IF k = 0 THEN DO
        srcpath = ''
        srcfilename = srcfile
    END
    ELSE DO
        srcpath = LEFT(srcfile, k)
        srcfilename = SUBSTR(srcfile, k+1)
    END

    dstfile = SysTempFileName(srcpath||'testp?????.tmp')
    dummy = STREAM(srcfile, 'C', 'OPEN READ')
    mustclose = MoveToApp( srcfile,dstfile,app )
    oldline = MoveToKey(srcfile,dstfile,app,key)
    IF oldline == '' THEN mustclose = 1
    IF frmat == '' THEN frmat = ChooseFormat(oldline)

    /* Special case: skip over continuation lines. */

    DO WHILE RIGHT(oldline,1) = '+'
        oldline = LINEIN(srcfile)
    END

    CALL PutKey srcfile, dstfile, key, value, frmat
    IF mustclose THEN CALL LINEOUT dstfile, '[/'app']'
    CALL CopyRemainder srcfile,dstfile
    CALL STREAM srcfile, 'C', 'CLOSE'
    CALL STREAM dstfile, 'C', 'CLOSE'
    BAKname = srcfilename'.BAK'
    '@DEL 'srcpath||BAKname' 2>NUL'
    '@RENAME 'srcfile' 'BAKname
    '@RENAME 'dstfile' 'srcfilename
    RETURN

/****************************************************************/
/*             FINDING THE CORRECT LINE IN A TNI FILE           */
/****************************************************************/

MoveToApp: PROCEDURE

    /* Copies srcfile to dstfile until we match the given app   */
    /* or reach the end of file.  If no match we create a new   */
    /* app.  On return the srcfile file pointer is at the first */
    /* line after the [app] line.  Normally returns 0, but      */
    /* returns 1 if a [/app] line must be appended.             */

    PARSE ARG srcfile,dstfile,app
    app = STRIP(app)
    DO FOREVER
        IF CHARS(srcfile) = 0 THEN DO
            CALL LINEOUT dstfile, '['app']'
            RETURN 1
        END
        line0 = LINEIN(srcfile)
        line = STRIP(line0)
        IF LENGTH(line) \= 0 THEN DO
            CALL LINEOUT dstfile, line0
            IF LEFT(line,1) \= '[' THEN DO
                 SAY "Unexpected line "line
                 RETURN 0
            END
            ELSE DO
                PARSE VAR line '['thisapp']'
                thisapp = STRIP(thisapp)
                IF thisapp = app THEN RETURN 0
                ELSE CALL SkipSection srcfile dstfile thisapp 0
            END
        END
    END

/****************************************************************/

MoveToKey: PROCEDURE

    /* Copies srcfile to dstfile until we match the given key   */
    /* or reach the end of the current app.  On a match we      */
    /* return the line that matches (without copying it to      */
    /* dstfile).  If no match we return a null string.          */

    PARSE ARG srcfile,dstfile,app,key
    DO FOREVER
        IF CHARS(srcfile) = 0 THEN RETURN ""
        line0 = LINEIN(srcfile)
        line = STRIP(line0)
        IF LENGTH(line) = 0 THEN
            DO
                /* skip blank line */
            END
        ELSE IF LEFT(line,1) = '[' THEN
            DO
                IF SUBSTR(line,2,1) = '/' THEN
                    RETURN ""
                ELSE
                    DO
                        PARSE VAR line '['thiskey']'
                        thiskey = STRIP(thiskey)
                        IF thiskey = key THEN
                            RETURN line
                        ELSE DO
                            CALL LINEOUT dstfile,line0
                            CALL SkipSection srcfile dstfile thiskey 1
                        END
                    END
            END
        ELSE
            DO
                PARSE VAR line thiskey"="v
                thiskey = STRIP(thiskey)
                IF thiskey = key THEN
                    RETURN line
                ELSE CALL LINEOUT dstfile,line0
            END
    END

/****************************************************************/

SkipSection: PROCEDURE

    /* Copies srcfile to dstfile until we have passed the end   */
    /* of the specified application.                            */

    PARSE ARG srcfile dstfile app nested
    DO FOREVER
        IF CHARS(srcfile) = 0 THEN RETURN
        line = LINEIN(srcfile)
        CALL LineOut dstfile, line
        line = STRIP(line)
        IF LENGTH(line) = 0 THEN
            DO
                /* do nothing */
            END
        ELSE IF LEFT(line,1) = '[' THEN DO
            PARSE VAR line '['label']'
            label = STRIP(label)
            IF LEFT(label,1) = '/' THEN
                DO
                    label = DELSTR(label,1,1)
                    IF label = app THEN RETURN
                    ELSE SAY "Mismatched terminator [/"label"]"
                END
            ELSE IF nested = 0 THEN CALL Skipsection srcfile dstfile label 1
        END
    END

/****************************************************************/
/*                      MOVING TO END OF FILE                   */
/****************************************************************/

CopyRemainder: PROCEDURE

    /* Copies what is left in srcfile to dstfile.  */

    PARSE ARG srcfile,dstfile
    DO WHILE CHARS(srcfile) > 0
        line = LINEIN(srcfile)
        CALL LINEOUT dstfile, line
    END
    RETURN

/****************************************************************/
/*              PUTTING A NEW VALUE INTO A TNI FILE             */
/****************************************************************/

PutKey: PROCEDURE

    /* Inserts a key=value line, or the equivalent, in the      */
    /* destination file.                                        */

    PARSE ARG srcfile, dstfile, key, value, frmat

    IF frmat = '"' THEN DO

        /****************************/
        /* TYPE = STRING OF STRINGS */
        /****************************/

        Nul = '00'X
        CALL LINEOUT dstfile,'   ['key']'
        DO WHILE (value \= '') & (value \= Nul)
            k = POS(Nul, value)
            str = LEFT(value, k-1)
            value = DELSTR(value,1,k)
            IF str \= '' THEN CALL LINEOUT dstfile,'      'str
        END

        /* Delete previous value, if any */

        DO UNTIL LEFT(line,2) = '[/'
            line = STRIP(LINEIN(srcfile))
        END
        CALL LINEOUT dstfile,'   [/'key']'
    END
    ELSE DO
        CALL CHAROUT dstfile, '   'key'='

        IF LEFT(frmat,1) = '(' THEN DO

            /******************/
            /* TYPE = NUMERIC */
            /******************/

            CALL CHAROUT dstfile,frmat
            base = 10
            PARSE VAR frmat '(' bytespernum ')'
            IF bytespernum = 'X' THEN DO
                base = 16
                bytespernum = 1
            END

            DO WHILE value \== ''
                IF LENGTH(value) > bytespernum THEN DO
                    part = LEFT(value,bytespernum)
                    value = DELSTR(value,1,bytespernum)
                END
                ELSE DO
                    part = value
                    value = ''
                END

                /* Convert little-endian string to number. */

                result = 0
                scale = 1
                DO WHILE LENGTH(part) > 0
                    result = scale*result + C2D(LEFT(part,1))
                    part = DELSTR(part,1,1)
                    scale = base*scale
                END
                IF base = 16 THEN result = D2X(result)
                CALL CHAROUT dstfile,result
                IF value \== '' THEN CALL CHAROUT dstfile,' '
            END
            CALL LINEOUT dstfile,''
        END
        ELSE IF LEFT(frmat,1) = "'" THEN DO

            /****************************/
            /* TYPE = CHARACTER STRING  */
            /****************************/

            IF (RIGHT(frmat,1) = '0') & (RIGHT(value,1) \= '00'X)
                                    THEN value = value||'00'X
            CALL PutString dstfile,value
        END

        ELSE DO

            /****************************/
            /* UNKNOWN FORMAT, SHOULD NOT OCCUR  */
            /****************************/

            SAY "ERROR: unknown format code "frmat
        END
    END
    RETURN

/****************************************************************/

PutString: PROCEDURE

    /* Output of properly bracketed string to dstfile. */

    PARSE ARG dstfile,string
    addnul = 0
    IF RIGHT(string,1) = '00'X THEN DO
        addnul = 1
        string = DELSTR(string,LENGTH(string))
    END
    DO WHILE string \= ''
        IF LENGTH(string) > 70 THEN DO
            CALL PutStringNoEOL dstfile,LEFT(string,70)
            string = DELSTR(string,1,70)
            IF string \== '' THEN DO
                CALL LINEOUT dstfile,'+'
                CALL CHAROUT dstfile,'        '
            END
        END
        ELSE DO
            CALL PutStringNoEOL dstfile,string
            string = ''
        END
    END
    IF addnul THEN CALL LINEOUT dstfile,'0'
    ELSE CALL LINEOUT dstfile,''
    RETURN

/****************************************************************/

PutStringNoEOL: PROCEDURE

    /* A variant of PutString that handles short strings and    */
    /* does not add a line terminator.                          */

    PARSE ARG dstfile,string
    DO WHILE string \= ''
        pos1 = POS('"',string)
        pos2 = POS("'",string)
        IF pos1 = 0 THEN DO
            IF pos2 = 0 THEN DO
                /* both zero */
                CALL CHAROUT dstfile,'"'string'"'
                string = ''
            END
            ELSE DO
                /* single quote at pos2. */
                CALL CHAROUT dstfile,'"'||LEFT(string,pos2)||'"'
                string = DELSTR(string,1,pos2)
            END
        END
        ELSE DO
            IF pos2 = 0 THEN DO
                /* double quote at pos1. */
                CALL CHAROUT dstfile, "'"||LEFT(string,pos1)||"'"
                string = DELSTR(string,1,pos1)
            END
            ELSE DO
                /* string contains both kinds of quote. */
                IF pos1 < pos2 THEN DO
                    CALL CHAROUT dstfile, "'"||LEFT(string,pos1)||"'"
                    string = DELSTR(string,1,pos1)
                END
                ELSE DO
                    CALL CHAROUT dstfile,'"'||LEFT(string,pos2)||'"'
                    string = DELSTR(string,1,pos2)
                END
            END
        END

        /* I could probably make the above more compact by      */
        /* combining common code, but it might hurt readability. */

        IF string \== '' THEN CALL CHAROUT dstfile,'+'
    END
    RETURN

/****************************************************************/

ChooseFormat: PROCEDURE

    /* Use the leading characters of oldline to decide what */
    /* format to use.  Default to hexadecimal.              */

    PARSE ARG oldline

    ch = LEFT(oldline,1)
    IF ch == '' THEN RETURN '(X)'
    ELSE IF ch = '[' THEN RETURN '"'
    ELSE DO
        PARSE VAR oldline k'='oldval
        oldval = STRIP(oldval)
        ch = LEFT(oldval,1)
        IF (ch = '"') | (ch = "'") THEN DO
            IF RIGHT(oldval,1) = '00'X THEN frmat = "'0"
            ELSE frmat = "'"
            RETURN frmat
        END
        ELSE IF ch = '(' THEN
            DO
                /* This can mean one of several numeric formats. */
                pos = POS(')',oldval)
                IF pos = 2 THEN RETURN '(4)'
                ELSE RETURN LEFT(oldval,pos)
            END
        ELSE IF DATATYPE(ch, 'W') > 0 THEN RETURN '(4)'
        ELSE RETURN '(X)'
    END

/****************************************************************/

