/*----------------------------------------------------------
   Returns a value from an INI or TNI or FMT file. Although
   this can be run from the command line, it is primarily
   intended to be called from another script.

   Note that it is not easy for a Rexx program to save state
   information from one invocation to the next. On each call
   the INI or TNI file has to be consulted all over again.
   If you need to look up a number of INI values, it might
   be more efficient to do the job in a different language.

           Author:       Peter Moylan (peter@pmoylan.org)
           Last revised: 14 February 2019

   Usage:
                result = INIget(filename, app, key)

            where
                filename is the name of an INI or TNI or
                FMT file, including a path if it is not in
                the current working directory. The file is
                assumed to be in INI format if the name
                ends with ".INI", in FMT format if the name
                ends with ".FMT, and TNI format otherwise.

                app and key are the labels of an
                (application,key) pair within that file.

            The result is a string of bytes. It is up to the caller
            to decide whether this should be interpreted as a
            character string or as binary data.

    Special cases:

        If the file is absent or cannot be opened the result
        is the string "NO FILE".

        If the file can be opened but the (app,key) pair does
        not exist, the result is the null string.

        If you omit the key, or the key is a null string, you
        get a list of all keys for that app. The list is in
        the "string of strings" format: each key is followed by
        the null character '00'X, and there is an extra
        final null.

        If you omit the app, or the app is a null string, you
        get a list of all apps.

------------------------------------------------------------*/

/****************************************************************/
/*                       MAIN PROGRAM                           */
/****************************************************************/

CALL RxFuncAdd SysLoadFuncs, RexxUtil, SysLoadFuncs
CALL SysLoadFuncs

PARSE ARG INIfile, app, key
INIfile = STRIP(INIfile)

IF STREAM(INIfile, 'C', 'QUERY EXISTS') = '' THEN DO
    SAY "ERROR: File "INIfile" does not exist"
    RETURN "NO FILE"
END

IF (app = '') & (key \= '') THEN DO
    SAY "ERROR: Empty application and non-empty key is forbidden"
    RETURN ''
END

TNIoption = 'T'
pos = LASTPOS('.', INIfile)
if pos > 0 THEN
    DO
        extension = TRANSLATE(RIGHT(INIfile, LENGTH(INIfile) - pos))
        IF extension = "INI" THEN TNIoption = 'I'
        ELSE IF extension = "FMT" THEN TNIoption = 'F'
    END

RETURN INIval(INIfile, TNIoption, app, key)

/****************************************************************/
/*               RETURNING AN INI OR TNI VALUE                  */
/****************************************************************/

INIval: PROCEDURE

    /* Returns one INI or TNI value, depending on TNIoption. */

    PARSE ARG filename, TNIoption, application, key
    key = STRIP(key)
    IF TNIoption = 'I' THEN DO
        IF application = '' THEN DO
            answer = SysIni(filename,'ALL:','stem.')
            IF answer \= "ERROR:" THEN answer = UnpackTree(stem)
        END
        ELSE IF key = '' THEN DO
            answer = SysIni(filename,application,'ALL:','stem.')
            IF answer \= "ERROR:" THEN answer = UnpackTree(stem)
        END
        ELSE answer = SysIni(filename, application, key)
        IF answer = "ERROR:" THEN answer = ""
    END
    ELSE DO
        dummy = STREAM(filename, 'C', 'OPEN READ')
        IF application = '' THEN answer = ListAllApps(filename)
        ELSE IF key = '' THEN answer = ListAllKeys(filename, application)
        ELSE answer = GetVal(filename, application, key, TNIoption)
    END
    RETURN answer

/****************************************************************/

UnpackTree:

    /* Turns a stem list into string-of-strings format.  Note   */
    /* that, because of quirk of Rexx, we have to declare this  */
    /* as a non-procedure in order to pass a stem variable.     */

    stem = ARG(1)
    list = '00'X
    IF stem.0 > 0 THEN
        DO k = 1 TO stem.0
            list = stem.k||'00'X||list
        END
    RETURN list

/****************************************************************/

GetVal: PROCEDURE

    /* Returns a key value from the TNI file.       */
    /* Returns a null string for value not found.   */

    PARSE ARG filename,app,key,TNIoption
    TNIoption = STRIP(TNIoption)
    IF MoveToApp(filename,app) \= 0 THEN RETURN ""
    DO FOREVER
        IF CHARS(filename) = 0 THEN RETURN ""
        line = STRIP(LINEIN(filename))
        IF LENGTH(line) = 0 THEN
            DO
                /* skip blank line */
            END
        ELSE IF LEFT(line,1) = '[' THEN
            DO
                IF SUBSTR(line,2,1) = '/' THEN
                    RETURN ""
                ELSE
                    DO
                        PARSE VAR line '['thiskey']'
                        thiskey = STRIP(thiskey)
                        IF thiskey = key THEN
                            RETURN GetStringOfStrings(filename, key)
                        ELSE CALL SkipSection filename thiskey 1
                    END
            END
        ELSE
            DO
                PARSE VAR line thiskey"="v
                thiskey = STRIP(thiskey)
                IF thiskey = key THEN DO
                    IF TNIoption = 'F' THEN RETURN v
                    ELSE RETURN DecodeValue(filename,v)
                END
            END
    END

/****************************************************************/
/*                  A COUPLE OF SPECIAL CASES                   */
/****************************************************************/

ListAllApps: PROCEDURE

    PARSE ARG filename
    list = ''
    DO WHILE CHARS(filename) > 0
        line = STRIP(LINEIN(filename))
        IF LENGTH(line) > 0 THEN DO
            IF LEFT(line,1) \= '[' THEN DO
                 SAY "Unexpected line "line
            END
            ELSE DO
                PARSE VAR line '['app']'
                app = STRIP(app)
                IF app \= '' THEN list = list||app||'00'X
                CALL SkipSection filename app 0
            END
        END
    END
    RETURN list||'00'X

/****************************************************************/

ListAllKeys: PROCEDURE

    PARSE ARG filename,app
    list = ''
    IF MoveToApp(filename,app) \= 0 THEN RETURN list
    DO FOREVER
        IF CHARS(filename) = 0 THEN RETURN list||'00'X
        line = STRIP(LINEIN(filename))
        IF LENGTH(line) = 0 THEN
            DO
                /* skip blank line */
            END
        ELSE IF LEFT(line,1) = '[' THEN
            DO
                IF SUBSTR(line,2,1) = '/' THEN

                    /* End of this app */

                    RETURN list
                ELSE
                    DO
                        /* Nested block. */

                        PARSE VAR line '['key']'
                        key = STRIP(key)
                        IF key \= '' THEN list = list||key||'00'X
                        CALL SkipSection filename key 1
                    END
            END
        ELSE
            DO
                /* Normal key=value line. */
                PARSE VAR line key"="v
                key = STRIP(key)
                IF key \= '' THEN list = list||key||'00'X

                /* Skip over continuation lines. */

                DO WHILE RIGHT(line,1) = '+'
                    line = LINEIN(filename)
                END
            END
    END

/****************************************************************/
/*          MOVING TO THE RIGHT PLACE IN A TNI FILE             */
/****************************************************************/

MoveToApp: PROCEDURE

    /* Sets the file pointer to the specified application.  */
    /* Returns 0 for success, -1 for "no such app".         */

    PARSE ARG filename,app
    app = STRIP(app)
    DO FOREVER
        IF CHARS(filename) = 0 THEN RETURN -1
        line = STRIP(LINEIN(filename))
        IF LENGTH(line) = 0 THEN
            DO
                /* do nothing */
            END
        ELSE IF LEFT(line,1) \= '[' THEN DO
             SAY "Unexpected line "line
             RETURN -1
        END
        PARSE VAR line '['thisapp']'
        thisapp = STRIP(thisapp)
        IF thisapp = app THEN RETURN 0
        ELSE CALL SkipSection filename thisapp 0
    END

/****************************************************************/

SkipSection: PROCEDURE

    /* Sets the file pointer past the specified application.  */

    PARSE ARG filename app nested
    filename = STRIP(filename)
    DO FOREVER
        IF CHARS(filename) = 0 THEN RETURN
        line = STRIP(LINEIN(filename))
        IF LENGTH(line) = 0 THEN
            DO
                /* do nothing */
            END
        ELSE IF LEFT(line,1) = '[' THEN DO
            PARSE VAR line '['label']'
            label = STRIP(label)
            IF LEFT(label,1) = '/' THEN
                DO
                    label = DELSTR(label,1,1)
                    IF label = app THEN RETURN
                    ELSE SAY "Mismatched terminator [/"label"]"
                END
            ELSE IF nested = 0 THEN CALL Skipsection filename label 1
        END
    END

/****************************************************************/
/*                FETCHING AND DECODING A VALUE                 */
/****************************************************************/

DecodeValue: PROCEDURE

    /* Turns a value string in the TNI format into a string of  */
    /* bytes.  Whether that string is to be treated as a        */
    /* character string otherwise, a number, or some other      */
    /* data structure is up to the caller.                      */

    PARSE ARG filename,str
    filename = STRIP(filename)
    str = STRIP(str)
    val = ''
    number = -1
    base = 10
    bytespernum = 4
    ch = LEFT(str,1)
    IF ch = '(' THEN
        DO
            /* This can mean one of several numeric formats. */
            number = 0
            str = DELSTR(str,1,1)
            ch = LEFT(str,1)
            IF ch = 'X' THEN
                DO
                    base = 16
                    bytespernum = 1
                    str = DELSTR(str,1,1)
                END
            ELSE IF DATATYPE(ch, 'N') > 0 THEN
                DO
                    bytespernum = ch
                    str = DELSTR(str,1,1)
                END
            /* Get rid of the closing ')'  */
            str = DELSTR(str,1,1)
        END
    ELSE IF DATATYPE(ch, 'W') > 0 THEN
        DO
            /* Default numeric format: 4-byte decimal. */
            number = 0
        END

    /* If we have detected any of the numeric formats, do   */
    /* the conversion.                                      */

    IF number = 0 THEN
        DO
            val = ""
            str = STRIP(str)
            DO WHILE LENGTH(str) > 0
                PARSE VAR str number' 'str
                number = STRIP(number)
                str = STRIP(str)
                DO WHILE number = '+'
                    str = STRIP(DELSTR(str,1,1))
                    IF LENGTH(str) = 0 THEN
                        str = STRIP(LINEIN(filename))
                    PARSE VAR str number ' ' str
                    number = STRIP(number)
                END
                IF base = 16 THEN number = X2D(number)

                /* Now we have to store number as the   */
                /* required number of bytes.            */

                DO BytesPerNum
                    val = val||D2C(number // 256)
                    number = number % 256
                END
            END
        END
    ELSE
        DO
            /* str is a character string, possibly with quote   */
            /* marks. We also have to allow for the possibility */
            /* of concanetated substrings, possible overflowing */
            /* to multiple lines.                               */

            str = STRIP(str)
            ch = LEFT(str,1)
            IF \((ch = '"') | (ch = "'")) THEN
                /* No quote marks, string is rest of line. */
                val = str
            ELSE
                DO
                    IF ch = "'" THEN
                        PARSE VAR str "'"part"'"rest
                    ELSE
                        PARSE VAR str '"'part'"'rest
                    val = val||part
                    str = STRIP(rest)

                    /* Now deal with strings concatenated with  */
                    /* '+', if any. This might require fetching */
                    /* continuation lines.                      */

                    DO WHILE LEFT(str,1) = '+'
                        str = STRIP(DELSTR(str,1,1))
                        IF LENGTH(str) = 0 THEN
                            str = STRIP(LINEIN(filename))
                        ch = LEFT(str,1)
                        IF ch = "'" THEN
                            PARSE VAR str "'"part"'"rest
                        ELSE
                            PARSE VAR str '"'part'"'rest
                        val = val||part
                        str = STRIP(rest)
                    END
                END
        END

    RETURN val

/****************************************************************/

GetStringOfStrings: PROCEDURE

    /* Reads one string per line of a file, stops when we reach */
    /* [/key]. The result is a string of Nul-terminated strings.*/

    PARSE ARG filename, key
    filename = STRIP(filename)
    Nul = '00'X
    codedlist = ''
    DO FOREVER
        newname = STRIP(LINEIN(filename))
        IF newname = '[/'||key||']' THEN LEAVE
        IF newname \= '' THEN codedlist = codedlist||newname||Nul
    END
    codedlist = codedlist||Nul
    RETURN codedlist

/****************************************************************/

