/*
 * viewer.c - Glyph Viewer
 */

#include <stdio.h>
#include <stdlib.h>

#define INCL_PM
#include <os2.h>

#include "xfont.h"
#include "xfontres.h"

/*
 * Glyph Viewer have two external entry points
 *      initViewer  - Create Window for Glyph Viwer
 *      termViewer  - Destroy Window for Glyph Viewer
 *  All window operation is doe with "ViewerWndProc".
 */

MRESULT EXPENTRY ViewerWndProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2) ;

HWND    initViewer(HAB  hab)
{
    HWND    hwndFrame    ;
    ULONG   flFrameStyle ;    
    ULONG   flFrameFlags ;

    WinRegisterClass(hab, XfdProgName, ViewerWndProc, CS_SIZEREDRAW, 0L) ;

    if (XfdSwpInit.fl & (SWP_SIZE | SWP_MOVE)) {
        flFrameStyle = 0 ;
    } else {
        flFrameStyle = WS_VISIBLE ;
    }
    XfdSwpInit.fl |= SWP_ACTIVATE | SWP_ZORDER | SWP_SHOW ;

    flFrameFlags = (FCF_TITLEBAR | FCF_SYSMENU |
                    FCF_SIZEBORDER | FCF_MINMAX |
		    FCF_MENU | FCF_ACCELTABLE | FCF_ICON |
		    FCF_HORZSCROLL | FCF_VERTSCROLL |
		    FCF_SHELLPOSITION | FCF_TASKLIST) ;

    hwndFrame = WinCreateStdWindow(HWND_DESKTOP, flFrameStyle, 
            &flFrameFlags, XfdProgName, XfdProgName, 0L, 0, ID_XFONT, NULL) ;

    if (hwndFrame != NULLHANDLE && flFrameStyle == 0) {
        WinSetWindowPos(hwndFrame, HWND_TOP, XfdSwpInit.x, XfdSwpInit.y,
	        XfdSwpInit.cx, XfdSwpInit.cy, XfdSwpInit.fl) ;
    }
    return hwndFrame ;
}

void    termViewer(HWND frame)
{
    WinDestroyWindow(frame) ;
}

/*
 * Window Handles
 */

static  HWND    hwndClient  ;
static  HWND    hwndFrame   ;
static  HWND    hwndTitle   ;
static  HWND    hwndMenu    ;
static  HWND    hwndHScroll ;
static  HWND    hwndVScroll ;

static  HWND    hwndPopFile ;
static  HWND    hwndPopChar ;

/*
 * Font to Display
 */

LISTPTR curFile = NULL ;    /* Current Font File    */
FontPtr curFont = NULL ;    /* Current Font Info    */
int     curRow, curCol ;    /* Current Code         */

/*
 * Size of Font Area, Name and Glyph
 *      Name, Glyph area size is depend on Font to Display.
 *      Calc. them on Load Complete timing.
 *      Size of client area is determined on WM_SIZE message.
 */
 
SIZEL   szChar = { 0 } ;        /* Max Extent           */
SIZEL   szCell = { 0 } ;        /* Glyph Cell           */
SIZEL   szFont = { 0 } ;        /* Size of Glyph Area   */
SIZEL   szName = { 0 } ;        /* Size of Name  Area   */
SIZEL   szCode = { 0 } ;        /* Size of Code  Area   */
SIZEL   szArea = { 0 } ;        /* Size of Client Area  */

/*
 * Where to Draw Glyph, FontName
 */

static  POINTL  ptName = { 0 } ;        /* for Font Name String */
static  POINTL  ptCode = { 0 } ;        /* for Selected Code    */
static  POINTL  ptFont = { 0 } ;        /* for Font Glyphs      */

static  LONG    maxHorz = 0 ;
static  LONG    maxVert = 0 ;
static  LONG    adjHorz = 0 ;
static  LONG    adjVert = 0 ;

/*
 * Format of Code Display
 */

static  char    codeSmp[] = "Range XXXX - XXXX [Current XXXX]" ;
static  char    codeFmt[] = "Range %02x%02x - %02x%02x [Current %02x%02x]" ;
static  char    codeBuf[64] ;

static  char    *codeStr(void)
{
    if (curFont == NULL) {
        return "" ;
    }
    sprintf(codeBuf, codeFmt, curRow, curFont->firstCol,
	    curRow, curFont->lastCol, curRow, curCol) ;
    return codeBuf ;
}

/*
 * Menu Control
 *      menuInit    to Initial status
 *      menuLoad    disable on loading
 *      menuFile    as font load complete
 *      menuPage    as page changed
 *      menuChar    as select a character
 */

static  USHORT  menuInitEna[] = {       /* Initially Enabled    */
    IDM_LOAD, IDM_EXIT, 0
} ;
static  USHORT  menuInitDis[] = {       /* Initially Disabled   */
    IDM_SELECT, IDM_PAGE, IDM_GLYPH,
    IDM_PAGE_TOP, IDM_PAGE_BOT, IDM_PAGE_PREV, IDM_PAGE_NEXT,
    IDM_GLYPH_MAGN, IDM_GLYPH_REVS, IDM_GLYPH_CLIP, 0
} ;

static  void    menuInit(void)
{
    PUSHORT pItem ;

    for (pItem = menuInitEna ; *pItem != 0 ; pItem++) {
        WinEnableMenuItem(hwndMenu,    *pItem, TRUE) ;
        WinEnableMenuItem(hwndPopFile, *pItem, TRUE) ;
        WinEnableMenuItem(hwndPopChar, *pItem, TRUE) ;
    }
    for (pItem = menuInitDis ; *pItem != 0 ; pItem++) {
        WinEnableMenuItem(hwndMenu,    *pItem, FALSE) ;
        WinEnableMenuItem(hwndPopFile, *pItem, FALSE) ;
        WinEnableMenuItem(hwndPopChar, *pItem, FALSE) ;
    }
}
    
static  USHORT  menuLoadDis[] = {   /* Disabled on Loading  */
    IDM_LOAD, IDM_SELECT, 0
} ;

static  void    menuLoad(void)
{
    PUSHORT pItem ;

    for (pItem = menuLoadDis ; *pItem != 0 ; pItem++) {
        WinEnableMenuItem(hwndMenu,    *pItem, FALSE) ;
        WinEnableMenuItem(hwndPopFile, *pItem, FALSE) ;
    }
} 

static  void    menuFile(void)
{
    LISTPTR top  ;
    BOOL    flag ;

    /*
     * 'Load' is always OK
     */
     
    WinEnableMenuItem(hwndMenu,    IDM_LOAD, TRUE) ;
    WinEnableMenuItem(hwndPopFile, IDM_LOAD, TRUE) ;
    
    /*
     * 'Select' will enabled if there are multi Font Files
     */
    
    if ((top = getFontList(XfdFontList)) == NULL) {
        flag = FALSE ;
    } else if (top->next == NULL) {
        flag = FALSE ;
    } else {
        flag = TRUE ;
    }
    WinEnableMenuItem(hwndMenu,    IDM_SELECT, flag) ;
    WinEnableMenuItem(hwndPopFile, IDM_SELECT, flag) ;
}

static  USHORT  menuPageAll[] = {     /* Menus for Pages  */
    IDM_PAGE, IDM_PAGE_TOP, IDM_PAGE_BOT, IDM_PAGE_PREV, IDM_PAGE_NEXT, 0
} ;
static  USHORT  menuPageTop[] = {       /* Enabled on Top Page      */
    IDM_PAGE, IDM_PAGE_BOT, IDM_PAGE_NEXT, 0
} ;
static  USHORT  menuPageBot[] = {       /* Enabled on Bottom Page   */
    IDM_PAGE, IDM_PAGE_TOP, IDM_PAGE_PREV, 0
} ;

static  void    menuPage(void)
{
    PUSHORT pItem ;    

    /*
     * Initally disable all page menus
     */

    for (pItem = menuPageAll ; *pItem != 0 ; pItem++) {
        WinEnableMenuItem(hwndMenu,    *pItem, FALSE) ;
        WinEnableMenuItem(hwndPopFile, *pItem, FALSE) ;
    }

    /*
     * There are no font or not multi-page font, no menu will be enabled
     */

    if (curFont == NULL || curFont->firstRow == curFont->lastRow) {
	return ;
    }

    /*
     * Enable page menus for multi-page font
     */
     
    if (curRow == curFont->firstRow) {
        pItem = menuPageTop ;
    } else if (curRow == curFont->lastRow) {
        pItem = menuPageBot ;
    } else {
        pItem = menuPageAll ;
    }
    for ( ; *pItem != 0 ; pItem++) {
        WinEnableMenuItem(hwndMenu,    *pItem, TRUE) ;
        WinEnableMenuItem(hwndPopFile, *pItem, TRUE) ;
    }
}

static  USHORT  menuCharList[] = {
    IDM_GLYPH, IDM_GLYPH_MAGN, IDM_GLYPH_REVS, IDM_GLYPH_CLIP, 0
} ;

static  void    menuChar(void)
{
    BOOL        flag ;
    PUSHORT     pItem ;
    
    if (curFont == NULL) {
        flag = FALSE ;
    } else if (GetCharInfo(curFont, curRow, curCol) == NULL) {
        flag = FALSE ;
    } else {
        flag = TRUE ;
    }
    for (pItem = menuCharList ; *pItem != 0 ; pItem++) {
        WinEnableMenuItem(hwndMenu,    *pItem, flag) ;
        WinEnableMenuItem(hwndPopChar, *pItem, flag) ;
    }
}

/*
 * Adjusting Display POsitions
 */

#define YSLACK   3

static  LONG    maxWidth(void)
{
    LONG    mx = 0 ;

    if (mx < szName.cx) mx = szName.cx ;
    if (mx < szCode.cx) mx = szCode.cx ;
    if (mx < szFont.cx) mx = szFont.cx ;

    return mx ;
}

static  LONG    maxHeight(void)
{
    return szName.cy + szCode.cy + szFont.cy + YSLACK ;
}

static  void    adjustPos(void)
{
    LONG    mx = maxWidth()  ;
    LONG    my = maxHeight() ;

    /*
     * Adjusting Horizontal Position
     */
     
    ptName.x = (szArea.cx / 2) - (szName.cx / 2) ;
    ptCode.x = (szArea.cx / 2) - (szCode.cx / 2) ;
    ptFont.x = (szArea.cx / 2) - (szFont.cx / 2) ;
    maxHorz = adjHorz = 0 ;

    if (mx < szArea.cx) {
        WinEnableWindow(hwndHScroll, FALSE) ;
    } else {
        WinEnableWindow(hwndHScroll, TRUE) ;
        maxHorz = mx - szArea.cx ;
	ptName.x += (maxHorz / 2) ;
	ptCode.x += (maxHorz / 2) ;
	ptFont.x += (maxHorz / 2) ;
        WinSendMsg(hwndHScroll, SBM_SETTHUMBSIZE,
                MPFROM2SHORT(szArea.cx, mx), NULL) ;
        WinSendMsg(hwndHScroll, SBM_SETSCROLLBAR,
	        MPFROM2SHORT(adjHorz, 0), MPFROM2SHORT(0, maxHorz)) ;
    }
    
    /*
     * Adjusting Vertical Position
     */
    
    ptName.y = szArea.cy - szName.cy ;
    ptCode.y = ptName.y - szCode.cy  ;
    ptFont.y = ptCode.y - YSLACK     ;
    maxVert = adjVert = 0 ;

    if (my < szArea.cy) {
        WinEnableWindow(hwndVScroll, FALSE) ;
    } else {
        WinEnableWindow(hwndVScroll, TRUE) ;
        maxVert = my - szArea.cy ;
        WinSendMsg(hwndVScroll, SBM_SETTHUMBSIZE,
                MPFROM2SHORT(szArea.cy, my), NULL) ;
        WinSendMsg(hwndVScroll, SBM_SETSCROLLBAR,
	        MPFROM2SHORT(adjVert, 0), MPFROM2SHORT(0, maxVert)) ;
    }
}

/*
 * display Font Name, Glyphs
 *      showBack    background of client area
 *      showName    Font Name
 *      showCode    Selected Code
 *      showFont    Glyphs
 */

#define nDispRow(font)  (((font)->lastCol < 128) ? 8 : 16)

static  void    showBack(HWND hwnd, HPS hps)
{
    RECTL   rct ;
    
    WinQueryWindowRect(hwnd, &rct)    ;
    WinFillRect(hps, &rct, CLR_WHITE) ;
}

static  void    showName(HWND hwnd, HPS hps)
{
    RECTL   rct ;

    if (curFont == NULL) {
        return ;
    }

    /*
     * Font Name at ptName, pointing lower-left conner of font name area
     */

    rct.xLeft   = ptName.x    - adjHorz   ;
    rct.xRight  = rct.xLeft   + szName.cx ;
    rct.yBottom = ptName.y    + adjVert   ;
    rct.yTop    = rct.yBottom + szName.cy ;

    WinDrawText(hps, -1, curFont->fontname,
        &rct, 0, 0, DT_CENTER | DT_VCENTER | DT_ERASERECT | DT_TEXTATTRS) ;
}

static  void    showCode(HWND hwnd, HPS hps)
{
    RECTL       rct  ;
    
    if (curFont == NULL) {
        return ;
    }

    /*
     * Selected Code at ptCode, pointing lower-left conner of code area
     */
     
    rct.xLeft   = ptCode.x    - adjHorz   ;
    rct.xRight  = rct.xLeft   + szCode.cx ;
    rct.yBottom = ptCode.y    + adjVert   ;
    rct.yTop    = rct.yBottom + szCode.cy ;

    WinDrawText(hps, -1, codeStr(),
        &rct, 0, 0, DT_CENTER | DT_VCENTER | DT_ERASERECT | DT_TEXTATTRS) ;
}

static  void    showFont(HWND hwnd, HPS hps)
{
    int         row, col, nrows, len ;
    POINTL      pt1, pt2 ;
    SIZEL       sz    ;
    ULONG       color ;
    CharInfoPtr pci ;
    
    if (curFont == NULL) {
        return ;
    }
    
    nrows = nDispRow(curFont) ;
    color = GpiQueryColor(hps) ;
        
    /*
     * Glyphs
     *      'pt1' as Cell Position (upper-left) and
     *      'pt2' as Glyph Image's  base position
     */

    GpiSetColor(hps, CLR_BLACK) ;
    pt1.y = ptFont.y + adjVert ;

    for (row = 0 ; row < nrows ; row++, pt1.y -= szCell.cy) {

	pt1.x = ptFont.x - adjHorz ;
        
        for (col = 0 ; col < 16 ; col++, pt1.x += szCell.cx) {

	    if ((pci = GetCharInfo(curFont, curRow, (row * 16 + col))) == NULL) {
	        continue ;
	    }

            /*
	     * Adjuting base position from Cell
	     */
	     
	    pt2.x = pt1.x + 2
	        - curFont->minbound.leftSideBearing
		+ pci->metrics.leftSideBearing ;
            pt2.y = pt1.y - 2
	        - curFont->maxbound.ascent
		+ pci->metrics.ascent ;

            /*
	     * Size of Glyph Rectangle
	     */
	     
	    sz.cx = pci->metrics.rightSideBearing
	                - pci->metrics.leftSideBearing ;
            sz.cy = pci->metrics.ascent
	                + pci->metrics.descent ;
#if 0
            len = ((sz.cx + 7) >> 3) * sz.cy ;
#endif
#if 1
            sz.cx = ((sz.cx + 7) >> 3) * 8 ;
	    len = (sz.cx >> 3) * sz.cy ;
#endif
            /*
	     * Draw Glyph
	     */

	    GpiMove(hps, &pt2) ;
	    GpiImage(hps, 0L, &sz, len, pci->glyph) ;
        }
    }

    /*
     * Glid for Character Cell
     *      ptFont points upper-left conner of Glyph Area
     */

    GpiSetColor(hps, CLR_PALEGRAY) ;

    pt1.y = pt2.y = ptFont.y + adjVert     ;    /* Horizontal Lines */
    pt1.x = ptFont.x - adjHorz             ;
    pt2.x = ptFont.x - adjHorz + szFont.cx ;
    
    for (row = 0 ; row <= nrows ; row++) {
        GpiMove(hps, &pt1) ;
	GpiLine(hps, &pt2) ;
	pt1.y -= szCell.cy ;
	pt2.y -= szCell.cy ;
    }
    
    pt1.x = pt2.x = ptFont.x - adjHorz     ;    /* Vertical Lines   */
    pt1.y = ptFont.y + adjVert             ;
    pt2.y = ptFont.y + adjVert - szFont.cy ;
    
    for (col = 0 ; col <= 16 ; col++) {
        GpiMove(hps, &pt1) ;
	GpiLine(hps, &pt2) ;
	pt1.x += szCell.cx ;
	pt2.x += szCell.cx ;
    }

    /*
     * Restore Color
     */

    GpiSetColor(hps, color) ;    
}

/*
 * Hit Check - checks (x, y) pos. with Character Glyph
 */

static  int hitCol(int mx)
{
    int     col, x ;
    
    x = ptFont.x - adjHorz ;

    for (col = 0 ; col < 16 ; col++, x += szCell.cx) {
        if (mx >= x && mx <= (x + szCell.cx)) {
	    return col ;
        }
    }
    return -1 ;
}

static  int hitRow(int my)
{
    int     row, y, nrows ;
    
    nrows = nDispRow(curFont) ;

    y = ptFont.y + adjVert ;
    
    for (row = 0 ; row < nrows ; row++, y -= szCell.cy) {
        if (my <= y && my >= (y - szCell.cy)) {
	    return row ;
	}
    }
    return -1 ;
}

/*
 * viewCreate - When Viewer Window was created
 */

static  void    viewCreate(HWND hwnd, MPARAM mp1, MPARAM mp2)
{
    LISTPTR     new ;
    
    /*
     * Save Frequently Used Window Handles
     */

    hwndClient  = hwnd ;
    hwndFrame   = WinQueryWindow(hwnd, QW_PARENT) ;
    hwndTitle   = WinWindowFromID(hwndFrame, FID_TITLEBAR)   ;
    hwndMenu    = WinWindowFromID(hwndFrame, FID_MENU)       ;
    hwndHScroll = WinWindowFromID(hwndFrame, FID_HORZSCROLL) ;
    hwndVScroll = WinWindowFromID(hwndFrame, FID_VERTSCROLL) ;

    WinEnableWindow(hwndHScroll, FALSE) ;
    WinEnableWindow(hwndVScroll, FALSE) ;

    hwndPopChar = WinLoadMenu(hwnd, NULLHANDLE, ID_POPCHAR) ;
    hwndPopFile = WinLoadMenu(hwnd, NULLHANDLE, ID_POPFILE) ;

    menuInit() ;
    
    /*
     * If there font to load, start load now
     */

    if ((new = getFontList(XfdFontList)) != NULL) {
        WinPostMsg(hwnd, WM_LOAD_FONT, MPFROMP(new), NULL) ;
    }
}

/*
 * viewRedraw
 */
 
static  void    viewRedraw(HWND hwnd, HPS hps)
{
    showBack(hwnd, hps) ;
    showName(hwnd, hps) ;
    showCode(hwnd, hps) ;
    showFont(hwnd, hps) ;
}

/*
 * viewResize - Client Window was Resized, adjust geometories
 */

static  void    viewResize(HWND hwnd, MPARAM mp1, MPARAM mp2)
{
    szArea.cx = SHORT1FROMMP(mp2) ;
    szArea.cy = SHORT2FROMMP(mp2) ;

    adjustPos() ;
}

/*
 * viewHorzScroll - Horizontal Scroll Operation
 */

static  void    viewHorzScroll(HWND hwnd, MPARAM mp1, MPARAM mp2)
{
    USHORT  cmd = SHORT2FROMMP(mp2) ;
    SHORT   pos = SHORT1FROMMP(mp2) ;
    LONG    new ;
    BOOL    set = TRUE ;
    
    switch (cmd) {

    case SB_SLIDERTRACK    :
    case SB_SLIDERPOSITION :
        new = pos   ;
	set = FALSE ;
	break ;
    case SB_LINELEFT  :
        new = adjHorz - 2 ;
	break ;
    case SB_LINERIGHT :
        new = adjHorz + 2 ;
	break ;
    case SB_PAGELEFT  :
        new = adjHorz - szCell.cx ;
	break ;
    case SB_PAGERIGHT :
        new = adjHorz + szCell.cx ;
	break ;
    default :
        return ;
    }

    if (new < 0)       new = 0       ;
    if (new > maxHorz) new = maxHorz ;

    if (set) {
        WinSendMsg(hwndHScroll, SBM_SETSCROLLBAR,
	    MPFROM2SHORT(new, 0), MPFROM2SHORT(0, maxHorz)) ;
    }
    if (new != adjHorz) {
        adjHorz = new ;
	WinInvalidateRect(hwnd, NULL, FALSE) ;
    }
}

/*
 * viewVertScroll - Vertical Scroll Operation
 */

static  void    viewVertScroll(HWND hwnd, MPARAM mp1, MPARAM mp2)
{
    USHORT  cmd = SHORT2FROMMP(mp2) ;
    SHORT   pos = SHORT1FROMMP(mp2) ;
    LONG    new ;
    BOOL    set = TRUE ;
    
    switch (cmd) {

    case SB_SLIDERTRACK    :
    case SB_SLIDERPOSITION :
        new = pos   ;
	set = FALSE ;
	break ;
    case SB_LINEUP   :
        new = adjVert - 2 ;
	break ;
    case SB_LINEDOWN :
        new = adjVert + 2 ;
	break ;
    case SB_PAGEUP   :
        new = adjVert - szCell.cy ;
	break ;
    case SB_PAGEDOWN :
        new = adjVert + szCell.cy ;
	break ;
    default :
        return ;
    }

    if (new < 0)       new = 0       ;
    if (new > maxVert) new = maxVert ;

    if (set) {
        WinSendMsg(hwndVScroll, SBM_SETSCROLLBAR,
	    MPFROM2SHORT(new, 0), MPFROM2SHORT(0, maxVert)) ;
    }
    if (new != adjVert) {
        adjVert = new ;
	WinInvalidateRect(hwnd, NULL, FALSE) ;
    }
}

/*
 * viewLoadFont - Request to Load File
 */
 
static  BOOL    loading = FALSE ;

static  void    loadDone(LISTPTR font)
{
    loading = FALSE ;
    WinPostMsg(hwndClient, WM_LOAD_DONE, MPFROMP(font), NULL) ;
}

static  void    viewLoadFont(HWND hwnd, MPARAM mp1, MPARAM mp2)
{
    LISTPTR     font = (LISTPTR) mp1 ;
    
    static  UCHAR   buff[MAXNAM] ;

    if (font == NULL) {
        return ;
    }
    if (loading == TRUE) {
        return ;
    }

    menuLoad() ;
    
    if (font->finfo == NULL) {
        sprintf(buff, "%s : Loading %s", XfdProgName, font->base) ;
    } else {
        sprintf(buff, "%s : %s", XfdProgName, font->base) ;
    }
    WinSetWindowText(hwndTitle,buff) ;
    
    if (font->finfo != NULL) {
        WinPostMsg(hwndClient, WM_LOAD_DONE, MPFROMP(font), NULL) ;
    } else {
        loading = TRUE ;
	loadFontInfo(font, loadDone) ;
    }
}

/*
 * viewLoadDone - Font Loading Complete
 */

static  void    viewLoadDone(HWND hwnd, MPARAM mp1, MPARAM mp2)
{
    LISTPTR     font = (LISTPTR) mp1 ;
    POINTL      apt[TXTBOX_COUNT] ;
    HPS         hps ;
    static  UCHAR   buff[MAXNAM] ;

    /*
     * If failed, resume previous title text
     */
            
    if ((font == NULL) || (font->finfo == NULL)) {
        if (curFile == NULL) {
            strcpy(buff, XfdProgName) ;
        } else {
	    sprintf(buff, "%s : %s", XfdProgName, curFile->base) ;
	}
	WinSetWindowText(hwndTitle, buff) ;
	menuFile() ;
	menuPage() ;
	return ;
    }
    
    /*
     * Font Load Complete, change current font to new one
     */

    sprintf(buff, "%s : %s", XfdProgName, font->base) ;
    WinSetWindowText(hwndTitle, buff) ;
    
    curFile = font        ;
    curFont = font->finfo ;

    curRow = curFont->firstRow ;
    curCol = curFont->firstCol ;
    
    /*
     * Calc. new geometry for Font
     */

    szChar.cx = curFont->maxbound.rightSideBearing
                    - curFont->minbound.leftSideBearing ;
    szChar.cy = curFont->maxbound.ascent
                    + curFont->maxbound.descent ;
    szCell.cx = szChar.cx + 3 ;
    szCell.cy = szChar.cy + 3 ;
    szFont.cx = szCell.cx * 16 ;
    szFont.cy = szCell.cy * nDispRow(curFont) ;

    hps = WinGetPS(hwnd) ;

    GpiQueryTextBox(hps, 
        strlen(curFont->fontname), curFont->fontname, TXTBOX_COUNT, apt) ;
    szName.cx = apt[TXTBOX_TOPRIGHT].x - apt[TXTBOX_TOPLEFT].x   ;
    szName.cy = apt[TXTBOX_TOPLEFT].y - apt[TXTBOX_BOTTOMLEFT].y ;

    GpiQueryTextBox(hps, strlen(codeSmp), codeSmp, TXTBOX_COUNT, apt) ;
    szCode.cx = apt[TXTBOX_TOPRIGHT].x - apt[TXTBOX_TOPLEFT].x   ;
    szCode.cy = apt[TXTBOX_TOPLEFT].y - apt[TXTBOX_BOTTOMLEFT].y ;

    WinReleasePS(hps) ;

    /*
     * Adjust Menu & Positions
     */
     
    adjustPos() ;
    menuFile()  ;   /* Loaded new Font File */
    menuPage()  ;   /* Selected new Page    */
    menuChar()  ;   /* Selected a Char      */
    
    WinInvalidateRect(hwnd, NULL, FALSE) ;
}

/*
 * viewSelect - Selection on Client Area
 */

static  BOOL    selectChar(HWND hwnd, SHORT x, SHORT y)
{
    HPS     hps ;
    int     col, row ;

    if (curFile == NULL || curFont == NULL) {
        return FALSE ;
    }
    if ((col = hitCol(x)) < 0 || (row = hitRow(y)) < 0) {
        return FALSE ;
    }

    curCol = row * 16 + col ;
    hps = WinGetPS(hwnd)    ;
    showCode(hwnd, hps)     ;
    WinReleasePS(hps)       ;
    menuChar()              ;
    
    return TRUE ;
}

static  void    viewSelect(HWND hwnd, MPARAM mp1, MPARAM mp2)
{
    selectChar(hwnd, SHORT1FROMMP(mp1), SHORT2FROMMP(mp1)) ;
}

/*
 * viewAction - Request Default Action on Client Area
 */

static  void    viewAction(HWND hwnd, MPARAM mp1, MPARAM mp2)
{
    USHORT  item ;
    LISTPTR top  ;
    
    if (selectChar(hwnd, SHORT1FROMMP(mp1), SHORT2FROMMP(mp1)) == TRUE) {
        item = IDM_GLYPH_MAGN ;
    } else if ((top = getFontList(XfdFontList)) == NULL) {
        item = IDM_LOAD ;
    } else if (top->next == NULL) {
        item = IDM_LOAD ;
    } else {
        item = IDM_SELECT ;
    }
    WinSendMsg(hwnd, WM_COMMAND, MPFROMSHORT(item), NULL) ;
}

/*
 * viewPopup - Popup Menu for pointed object
 */

static  void    viewPopup(HWND hwnd, MPARAM mp1, MPARAM mp2)
{
    int     x, y   ;
    BOOL    fGlyph ;
    LISTPTR top  ;
    HWND    menu ;
    ULONG   item ;
    ULONG   opts ;

    fGlyph = selectChar(hwnd, x = SHORT1FROMMP(mp1), y = SHORT2FROMMP(mp1)) ;
    
    opts = PU_POSITIONONITEM| PU_HCONSTRAIN | PU_VCONSTRAIN |
        PU_MOUSEBUTTON2DOWN |
	PU_KEYBOARD | PU_MOUSEBUTTON1 | PU_MOUSEBUTTON2 ;

    if (fGlyph == TRUE) {
        if (GetCharInfo(curFont, curRow, curCol) == NULL) {
	    menu = NULLHANDLE ;
	} else {
	    menu = hwndPopChar    ;
	    item = IDM_GLYPH_MAGN ;
	}
    } else {
        menu = hwndPopFile ;
        if ((top = getFontList(XfdFontList)) == NULL) {
	    item = IDM_LOAD ;
	} else if (top->next == NULL) {
	    item = IDM_LOAD ;
	} else {
	    item = IDM_SELECT ;
	}
    }
    if (menu) {
        WinPopupMenu(hwnd, hwnd, menu, x, y, item, opts) ;
    }
}

/*
 * viewPage - display Top, Bottom, Prev, Next page
 */

static  void    viewPage(HWND hwnd, SHORT what)
{
    BOOL    redraw = FALSE ;
    
    if (curFont == NULL || curFont->firstRow == curFont->lastRow) {
        return ;
    }
    
    switch (what) {

    case IDM_PAGE_TOP :
        if (curRow != curFont->firstRow) {
	    curRow = curFont->firstRow ;
	    curCol = curFont->firstCol ;
	    redraw = TRUE ;
	}
	break ;
    
    case IDM_PAGE_BOT :
        if (curRow != curFont->lastRow) {
	    curRow = curFont->lastRow  ;
	    curCol = curFont->firstCol ;
	    redraw = TRUE ;
	}
	break ;
	
    case IDM_PAGE_PREV :
        if (curRow > curFont->firstRow) {
	    curRow -= 1 ;
	    curCol = curFont->firstCol ;
	    redraw = TRUE ;
	}
	break ;

    case IDM_PAGE_NEXT :
        if (curRow < curFont->lastRow) {
	    curRow += 1 ;
	    curCol = curFont->firstCol ;
	    redraw = TRUE ;
	}
	break ;
    }

    if (redraw) {
        menuPage() ;
	menuChar() ;
        WinInvalidateRect(hwnd, NULL, FALSE) ;
    }
}

/*
 * viewMagnify - Magnify Selected Character
 */

static  void    viewMagnify(HWND hwnd)
{
    CharInfoPtr pci ;
    
    if ((pci = GetCharInfo(curFont, curRow, curCol)) == NULL) {
        return ;
    }
    glyphMagnify(hwnd, pci) ;
}

/*
 * viewClipon - Glyph bitmap into clipboard
 */

static  void    viewClipon(HWND hwnd, BOOL revs)
{
    CharInfoPtr pci ;
    
    if ((pci = GetCharInfo(curFont, curRow, curCol)) == NULL) {
        return ;
    }
    glyphClipon(hwnd, pci, revs) ;
}

/*
 * ViewerWndProc - Window Procedure for XFONT Glyph Viewer
 */
 
MRESULT EXPENTRY ViewerWndProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    HPS     hps ;
    
    switch(msg) {

    case WM_CREATE :
        viewCreate(hwnd, mp1, mp2) ;
	return (MRESULT) 0 ;

    case WM_DESTROY :
	return (MRESULT) 0 ;
        
    case WM_PAINT :
        hps = WinBeginPaint(hwnd, NULLHANDLE, NULL) ;
	viewRedraw(hwnd, hps) ;
        WinEndPaint(hps) ;
        return (MRESULT) 0 ;

    case WM_ERASEBACKGROUND :
        return (MRESULT) 1 ;

    case WM_SIZE :
        viewResize(hwnd, mp1, mp2) ;
	return (MRESULT) 0 ;

    case WM_HSCROLL :
        viewHorzScroll(hwnd, mp1, mp2) ;
	return (MRESULT) 0 ;

    case WM_VSCROLL :
        viewVertScroll(hwnd, mp1, mp2) ;
	return (MRESULT) 0 ;

    case WM_LOAD_FONT :
        viewLoadFont(hwnd, mp1, mp2) ;
	return (MRESULT) 0 ;
	
    case WM_LOAD_DONE :
        viewLoadDone(hwnd, mp1, mp2) ;
	return (MRESULT) 0 ;

    case WM_BUTTON1DOWN :
        viewSelect(hwnd, mp1, mp2) ;
	return WinDefWindowProc(hwnd, msg, mp1, mp2) ;

    case WM_BUTTON1DBLCLK :
        viewAction(hwnd, mp1, mp2) ;
        return (MRESULT) 0 ;

    case WM_BUTTON2DOWN :
        viewPopup(hwnd, mp1, mp2) ;
        return (MRESULT) 0 ;
	
    case WM_COMMAND :
        switch (SHORT1FROMMP(mp1)) {

	case IDM_EXIT :
	    positionSave(hwndFrame) ;
	    WinSendMsg(hwnd, WM_CLOSE, NULL, NULL) ;
	    return (MRESULT) 0 ;

	case IDM_LOAD   :
	    dialogLoadFile(hwnd) ;
	    return (MRESULT) 0 ;

        case IDM_SELECT :
	    dialogSelectFont(hwnd) ;
	    return (MRESULT) 0 ;

	case IDM_PAGE_TOP  :
	case IDM_PAGE_BOT  :
	case IDM_PAGE_PREV :
	case IDM_PAGE_NEXT :
	    viewPage(hwnd, SHORT1FROMMP(mp1)) ;
	    return (MRESULT) 0 ;

        case IDM_GLYPH_MAGN :
	    viewMagnify(hwnd) ;
	    return (MRESULT) 0 ;
	case IDM_GLYPH_CLIP :
	    viewClipon(hwnd, FALSE) ;
	    return (MRESULT) 0 ;
        case IDM_GLYPH_REVS :
	    viewClipon(hwnd, TRUE) ;
            return (MRESULT) 0 ;	    
        }
    }
    return WinDefWindowProc(hwnd, msg, mp1, mp2) ;
}
