/* #! /usr/bin/regina */
/* Enhanced Directory Tree Utility by Al Heath */

/* UNDOCUMENTED /DBG option:
    /DBG or /DBG1 = Pause after mask breakdown and starting directory information
    /DBG2 = display intermediary 'listDirectory' results
    /DBG3 = turn on trace("?R") to allow setting 'gblTrace' to part of a filename
            for more debugging of 'filterAllFiles'.                                */

versionString = '2.0.0 2022-12-14'

/* define some default values */
numeric digits 16        /* increase the precision for byte totals */
pauseOpt = 0             /* a '/p' option */
linesOutput = 1          /*  for use with the /p option */
maxRecursionLevel = 200  /* in practicality, will never have this many nested subdirectories */
dbg = 0                  /* internally debugging, 0 = production no debug mode */
fsSeparator = '\'        /* typical OS/2 or Windows path separator vs Linux/Unix/Aix */

parse source opSys . ourFullName .
parse value filespec('name',ourFullName) with exname '.' exExtension
exExtension = '.'||exExtension    /* implied naming convention for Rexx Execs */

if OpSys = 'UNIX' then fsSeparator = '/'
if translate(left(OpSys,7)) = 'WINDOWS' | translate(left(OpSys,4)) = 'OS/2' then do
   /* Insure the SysXXXXX functions are ready to use */
   Call RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
   Call SysLoadFuncs
end

/* Read screen size to implement paging the output if requested by '/p' */
parse value SysTextScreenSize() with rows cols

parse arg args
if word(args,1) = '?' | word(args,1) = '-?' | translate(word(args,1)) = '-H' then do
   say 'Enhanced directory command for a drive, somewhat similar to "dir /s".'
   say '    ' filespec('name',ourFullName) 'Version:' versionString
   say '  A file mask may be specified.  Defaults to *'
   say '  Implied path of ".\" to start from the current relative subdirectory.'
   say '  Quote the "mask" with Single or Double quotes when it has embedded blanks.'
   say ''
   say 'Syntax of "'||ourFullName||'":'
   say ' ' filespec('name',exname) '<mask> <modifiers> ( <DIR | FILES | BOTH> <COUNT | ROLLUP>'
   say ''
   say 'Optionally, The <mask> may be have multi fragments WITHIN double quotes.'
   say '  The <mask> is NOT a regular expression.  A leading caret "!" means NOT.'
   say 'For Example:'
   say '  "*.dll+*.cmd"  will return all *.dll PLUS all *.cmd matches.'
   say '  "!*.cmd"  will return all that are NOT *.cmd.'
   say '  A fragment within parenthesis means the subdirectory must contain "mask".'
   say '     "(*.jpg)" ignore the ENTIRE subdirectory unless it contains a *.jpg file.'
   say '  "*.htm*+(*.jpg)" is *.htm* files in subdirectories with 1 or more jpg files.'
   say '  "*.htm*+(!*.jpg)" is *.htm* files in subdirectories without a jpg file.'
   say ''
   say ' NOTE: "e*+!*.cmd" returns "e*" plus "!*.cmd", thus "e.cmd" would be returned'
   say '    as it matched the "e*" fragment. Likewise "a.dll" would also be returned!'
   say '    That may not be what you were thinking it might do!'
   say ' Try "/E" to Explain how specific masks will function.'
   if rows < 40 then '@pause'
   say ''
   say 'Options:'
   say '  Default is to display only files in ALL subdirectories.'
   say '  Option DIR means only the directory tree in ALL subdirectories.'
   say '  **NOTE: using "/a:d" will also display the directory tree.'
   say '  Option FILES means only Files in the ONE SPECIFIED subdirectory.'
   say '  **NOTE: The FILES option is deprecated.  Use "/r0" instead.'
   say '  Option BOTH displays all files and all directory names.'
   say ''
   say '  Option COUNT will output only summary information of # matches and bytes'
   say '       used by each directory.'
   say '    A minimum COUNT may be specified, i.e. "COUNT 2", to display only the'
   say '       entries that have 2 or more mask matches, or "COUNT 0" for 0 or more.'
   say '       Default is 1 or more matches.'
   say '  Option ROLLUP is same as COUNT with the addition of lower level subdirectory'
   say '    summary count information is rolled up into the parent directory.'
   '@pause'
   say ''
   say 'Modifiers may be:'
   say ' Comparators:'
   say '   /d: signals a date comparison for greater than or equal to specified date.'
   say '            Note: date format is mm-dd-yy.  mm/dd/yy is not allowed.'
   say '            Year may be 2 digit "yy" or for fully "yyyy".'
   say '        if /d: is given without a date, defaults to "equals today".'
   say '        optionally, /de: for date equal to,'
   say '                 or /dl: for date less than or equal.'
   say '                 or /dg: for date greater than or equal.'
   if rows < 25 then '@pause'
   say '   /t: signals a time comparison for greater than or equal to specified time.'
   say '        if /t: given without a date, the date defaults to today.'
   say '     Typically, if a /dg: is specified, specify only /t: to select those'
   say '           files newer than that date and time.  If a /tg: is specified,'
   say '           then BOTH the Date and the Time must be greater.'
   say '           That probably would not be what you intended.'
   say '        optionally, /te: for time equal to,'
   say '                 or /tl: for time less than or equal.'
   say '                 or /tg: for time greater than or equal.'
   say '   /s: signals a size comparison.  (defaults to /sg:1  >=1 or not zero bytes).'
   say '        Similarly there are the /sl: and /se: options available.'
   say '   /a: attributes must have.  For example /a:a-r for Archive AND not readonly'
   say '                                  or /a:hs for hidden AND system files.'
   '@pause'
   say ''
   say 'Output control:'
   say '  /l /l+ display the .LONGNAME extended attribute as the file name'
   say '       instead of the actual short name on a VFAT file system.'
   say '  /f  shows only the fully qualified file name.'
   say '  /p  to page the output considering the screen size.'
   say "      In /exec mode, pause after each exec'd command so output can be checked."
   say '  /r# maximum subdirectory recursion level.'
   say '      Default is unlimited.  Specify "/r0" for no recursion.'
   say '  /cmd xxx'
   say '      also implies the /f modifier.'
   say '      xxx is a command to prepend to the fully qualified name.'
   say '       OR a command template where % indicates where to substitue the name.'
   say '         % is full drive:\path\name, %D Drive only, %P Path only, %N Name only.'
   say '     Typically one would redirect this output to a file to further manipulate.'
   say '  /exec xxx is similiar to /cmd except it will execute it immediately!'
   say ''
   if  translate(left(OpSys,4)) = 'OS/2' then do
      say 'Enter "G" to Explain Sample masks, or any other key to exit this Help section...'
      pull response
      if translate(left(response,1)) <> 'G'
         then exit 100
   end
   else do
      '@pause'
      exit 100
   end

   /* our examples should work with 'boot drive' & '\os2' */
   Call RxFuncAdd 'RxLoadFuncs', 'RXUTILS', 'RxLoadFuncs'
   Call RxLoadFuncs('QUIET')
   forceDrive = filespec('drive',directory())
   forceDir = filespec('path',directory()||fsSeparator)
   if translate(forceDrive) <> RxBootDrive()
      then forceDrive = RxBootDrive()
      else forceDrive = ''
   if translate(forceDir) <> '\OS2\'
      then forceDir = '\OS2\'
      else forceDir = ''
   ExampleArgs.1 = '"'||forceDrive||'\os2\*.ini" Simple example that forces the starting path as "\os2" and traverses that tree looking for "*.ini".',
                                       '  If the current directory was already "\os2" then it is equivalent to a simple mask of just "*.ini".'
   ExampleArgs.2 = '"'||forceDrive||forceDir||'boot\*.ini+\os2\*.ini" Notice how this complex mask operates differently from "'||RxBootDrive()||forceDir||'*.ini" ',
                                       'since the 2nd mask is fixed from the drive root.  With the first mask specifying the starting directory ',
                                       'as "\os2\boot", the search will NOT find ini files in "\os2\install" nor "\mdos\winos2"'
   ExampleArgs.3 = '"'||forceDrive||'\ecs\*.ini+..\os2\os2.ini" This one is tricky to understand since the ..\os2 is applied as the \ecs tree is traversed.',
                                       '  Thus when in \ecs it will look at \os2 but then in \ecs\boot it will look for \ecs\os2 which likely does not exist.'
   ExampleArgs.4 = '"'||forceDrive||forceDir||'*inst\*.ini" Only those in subdirectories with a path matching "*inst".'
   if forceDir = ''
      Then exampleArgs.4 = '"'||forceDrive||'.\*inst\*.ini" Only those in subdirectories with a path matching "*inst". ' ,
                                        'NOTE: In this case, the ".\" is VERY important to make it a relative in the tree!'
   ExampleArgs.5 = '"'||forceDrive||forceDir||'*.ini+(!shut*.exe)" In this case, the subdirectory MUST ALSO NOT CONTAIN 1 or more shut*.exe files. ',
                                       'A "contains" clause, denoted by ( ), is a simple true/false logic test to pass before continuing.'
   ExampleArgs.6 = '"'||forceDrive||forceDir||'*.ir+.\*inst\*ini" Demonstrates the importance of ".\" in the 2nd specification.' ,
                                       'Without the ".\", only *inst subdirectories directly under \os2 are searched.'
   ExampleArgs.7 = '"'||forceDrive||forceDir||'*.ini+(*.exe)+(!write.exe)" Returns INI files in directories that contain *.exe but NOT write.exe' ,
                                       'In this extremely concocted case, ',
                                       'excluding those MDOS\WINOS2\*.INI files as WRITE.EXE exists there, but still includes MDOS\WINOS2\SYSTEM\*.ini ',
                                       'files as SYSTEM has an EXE but not WRITE.EXE.'
   ExampleArgs.0 = 7

   say ''
   if forceDrive <> '' | forceDir <> '' then do
      say 'These specific examples may over specify the starting directory so they'
      say '   will actually run regardless of your actual current drive and directory.'
      say '   In practice, you will likely run the exec assuming the current drive'
      say '   and probably the current working directory.'
      say ''
   end
   else do
      say 'Changing your default drive and directory will affect how these examples'
      say 'are defined! They have been designed to search the OS2 directory on your'
      say 'boot drive even when this exec has been invoked from a different location.'
      say 'Only the example masks will change but the result will be consistent.'
   end
   say 'Note, Your current directory is:' substr(directory(),3) 'and thus is implied!'

   l = 6+2
   do i = 1 to ExampleArgs.0
      parse value ExampleArgs.i with '"' Sample '"' Text
      say i '"'||word(Sample,1)||'"'
      l = l + 1
      if Text <> '' then do until Text = ''
         if l >= rows then do
            '@pause'
            l = 2
         end
         j = ''
         do until length(j) + 1 + length(word(Text,1)) > cols - 10
            j = j word(Text,1)
            Text = subword(Text,2)
            if Text = '' then leave
         end
         say '    ' j
         l = l + 1
      end
   end
   say ''

   say 'Select an example, or just press Enter to exit this help section...'
   say '    (Remember you can always use the "/E" modifier to Explain a mask.)'
   parse pull response
   if response = '' | datatype(response) = 'NUM'
      then if response < 0 | response > ExampleArgs.0
         then exit 100
   if datatype(response) = 'NUM'
      then parse value ExampleArgs.response with '"' args '"' .
      else args = response
   args = args '/E'                   /* we always want to Explain this run */
   parse arg . '/' i '(' .      /* include any additional args we were passed */
   if i <> ''
      then args = args '/'||i
end

/* take an early check for a /DBG option to help us debug parsing */
if pos('/DBG',translate(args)) > 0 then do     /* 'Debug' */
   dbg = 1
   i = pos('/DBG',translate(args))+4
   if substr(args,i,1) <> ' ' & datatype(word(substr(args,i),1)) = 'NUM'
      then dbg = word(substr(args,i),1)
end

/* first lets pull the mask specification from the args */
mask = ''
do i = 1 to words(args)
   /* find the first word that isn't a modifier option */
   if left(word(args,i),1) <> '/' then do
      /* we found the mask, so lets split it out separately */
      ch = left(word(args,i),1)        /* if could have been single or double quoted */
      if (ch = '"' | ch = "'") then do
         mask = subword(args,i)        /* pull out the start of the mask */
         i = pos(mask,args)            /* where the mask starts in the original args */
         j = pos(ch,substr(mask,2))    /* the end of the actual quoted mask */
         if j > 0 then do
            mask = substr(mask,2,j-1)  /* limit the mask to the ending quote */
            args = strip(left(args,i-1)) strip(substr(args,i+length(mask)+2))  /* remove mask from the arg string */
         end
         else do
            say 'Error parsing the mask.  Looks like it is missing a ending quote:' ch
            say '    ' args
            exit 8
         end
      end
      else do
         mask = word(args,i)
         args = delword(args,i,1)
      end

      /* insure the mask is cleaned up */
      mask = strip(strip(mask),'B','"')
      leave
   end
end
If mask = '' Then mask = '*'
maskArg = mask

/* now lets make the initial separation of the options from the modifier args */
i = wordpos('/CMD',translate(args))
if i = 0
   then i = wordpos('/EXEC',translate(args))
if i > 0 then do
   /* handle an embedded '(' within a CMD or EXEC template */
   ch = left(word(args,i+1),1)
   if (ch = '"' | ch = "'") then do
      sz = subword(args,i+1)
      c = pos(ch,substr(sz,2))
      if c > 0 then do
         j = pos('(',sz,c)
         if j > 0 then do
            opts = substr(sz,j+1)
            args = subword(args,1,i) substr(sz,1,j-1)
         end
         else opts = ''     /* there is no '(' to worry about */
      end
      else do
         say 'Error parsing the' word(args,i) 'template.  Looks like it is missing a ending quote:' ch
         say '   ' sz
         exit 8
      end
   end
   else do
      /* the template wasn't quoted so it is also a simple parse */
      parse arg . '/' args '(' opts
      args = '/'||strip(args)
   end
end
else do   /* else nothing special to consider when parsing */
   parse var args . '(' opts
   parse var args . '/' args '(' .
   args = '/'||strip(args)
end
opts = translate(opts)        /* we expect upper case */

/* Debug: dump out our parsing results */
if dbg > 0 then do
   parse arg i
   say 'Parsing:' i
   say '  Mask: "'||mask||'"'
   say '  Modifiers:' args
   say '  Options:' opts
end

/* handle some old options that may be deprecated */
what = 'F'      /* only files listed by default */
If wordpos('DIR',translate(strip(opts))) > 0 Then what = 'D'
If wordpos('BOTH',translate(strip(opts))) > 0 Then what = 'B'

nesting = 1   /* assume we will traverse all nested subdirectories */
If wordpos('FILES',translate(strip(opts))) > 0 Then nesting = 0

countOnly = -1 /* by default, names will be displayed */
rollup = 0     /* by default, counts are not rolled up to parent directory */
i = wordpos('COUNT',translate(strip(opts)))
If i = 0 then do
   i = wordpos('ROLLUP',translate(strip(opts)))
   If i > 0
      then rollup = 1
End
If i > 0 Then Do
   countOnly = 1
   if words(opts) > i & datatype(word(opts,i+1)) = 'NUM'
      then countOnly = word(opts,i+1)
end

/* set special processing options from the modifier args */
if pos('/DBG',translate(args)) > 0 then do     /* 'Debug' */
   dbg = 1
   i = pos('/DBG',translate(args))+4
   if substr(args,i,1) <> ' ' & datatype(word(substr(args,i),1)) = 'NUM'
      then dbg = word(substr(args,i),1)
end
if pos('/P',translate(args)) > 0             /* 'Pause' arg */
   then pauseOpt = 1
if pos('/R',translate(args)) > 0 then do     /* 'Recursion' arg */
   i = pos('/R',translate(args))+2
   if substr(args,i,1) <> ' ' & datatype(word(substr(args,i),1)) = 'NUM'
      then maxRecursionLevel = word(substr(args,i),1)
end

show_longname = 0
i = pos('/L',translate(args))
if i > 0 then do
   parse value substr(args,i) with _OptValue .
   show_longname = 1
   if length(_OptValue > 2) & right(_OptValue,1) = '+'
      then show_longname = 2
   args = delstr(args,i,length(_OptValue))
end

showFullNameOnly = 0
if pos('/F',translate(args)) > 0
   then showFullNameOnly = 1

execTemplate = ''
i = wordpos('/CMD',translate(args))
if i = 0
   then i = wordpos('/EXEC',translate(args))
if i > 0 then if words(args) > i then do
   showFullNameOnly = 2
   if wordpos('/EXEC',translate(args)) > 0
      then showFullNameOnly = 3
   execTemplate = word(args,i+1)
   ch = left(execTemplate,1)
   if ch = '"' | ch = "'" then do
      execTemplate = substr(subword(args,i+1),2)
      c = pos(ch,execTemplate)
      execTemplate = left(execTemplate,c-1)
   end
   args = delword(args,i+1,words(execTemplate))
   if pos('%',execTemplate) = 0
      then execTemplate = execTemplate '%'
end

/* date comparisons */
if pos('/d:',args) > 0 | pos('/de:',args) > 0 | pos('/dg:',args) > 0 | pos('/dl:',args) > 0 then do
   parse var args . '/d' comparator ':' begin_date . '/' .
   comparator = translate(comparator)
   if begin_date = '' then do
      begin_date = date('U')

      if dbg > 0
        then say '   defaulting date to today =' begin_date
   end

   if pos('-',begin_date) > 0
      then parse var begin_date mm '-' dd '-' yy
      else  parse var begin_date mm '/' dd '/' yy

   if yy = '' then yy = left(date('S'),4)
   if yy <= 99 & yy >= 70
      then yy = yy + 1900
      else if yy < 100 then yy = yy + 2000

   if (( mm < 1 | mm > 12) | (dd < 1 | dd > 31)) then do
      say '  Date entered was not in proper month-day-year form:'
      say '     a date of' begin_date 'is not appropriate.'
      exit 8
   end

   begin_date = yy||'-'||right(100+mm,2)||'-'||right(100+dd,2)
end
else do
  begin_date = 0
  comparator = ''
end

/* time comparisons */
parse var args . '/t' t_comparator ':' time_arg . '/' .
t_comparator = translate(t_comparator)
if time_arg = '' then time_arg = '00:00'
pm = 0
parse var time_arg hr ':' min ':' .
if min = '' then min = 0
if right(min,1) = 'p' then do
   pm = 12
   min = left(min,length(min)-1)
end
begin_time = ((hr + pm) * 60) + min

if begin_time > 0 & begin_date = 0
   then begin_date = substr(date('S'),1,4)||'-'||substr(date('S'),5,2)||'-'||substr(date('S'),7,2)

/* is there a "size" option? */
s_comparator = ''
if pos('/S',translate(args)) > 0 then do
   parse var args . '/s' s_comparator ':' size_arg . '/' .
   s_comparator = translate(s_comparator)
   if s_comparator = '' then s_comparator = 'G'
   if size_arg = '' then size_arg = 1
end

tAttrib = "*****"   /* ADHRS attribute mask for SysFileTree */
parse var args . '/a:' attributeFilter .
do while attributeFilter \= ''
   /* parse out the target attribute specification... i.e.  '/a:-r' for not read only */
   anAttr = left(attributeFilter,1)
   anAttrMask = '+'
   if anAttr = '-' then do
      anAttr = left(attributeFilter,2)
      anAttrMask = '-'
   end
   if anAttr = '+' then do
      anAttr = left(attributeFilter,2)
      anAttrMask = '+'
   end
   attributeFilter = substr(attributeFilter,length(anAttr)+1)

   anAttr = translate(right(anAttr,1))
   maskPos = pos(anAttr,'ADHRS')

   tAttrib = overlay(anAttrMask,tAttrib,maskPos,1)
end
if substr(tAttrib,2,1) = '+'
   then what = 'D'

/* On Unix/Linux, Hidden files start with a . */
if OpSys = 'UNIX' & substr(tAttrib,3,1) = '+' & left(mask,1) \= '.' then do
   mask = '.'||mask
   tAttrib = substr(tAttrib,1,2) || '*' || substr(tAttrib,4)
end

/* decompose a complex mask into its subordinate pieces
   ordering "contains" clause(s) first in processing order */
maskParts.0 = 0
fullMask = ''
NotMask = 0
primaryMaskIndex = 0

do  pass = 1 to 2
   parsingMask = mask
   do while parsingMask <> ''
      parse var parsingMask m '+' parsingMask
      if pass = 1 then do      /* first pass is any contains clauses */
         if left(m,1) <> '('
            then iterate
      end
      else if left(m,1) = '('  /* next pass is anything else */
            then iterate

      /* add the mask to the processing list */
      i = maskParts.0 + 1
      maskParts.i = translate(m)
      maskParts.0 = i

      /* the first "non contains" mask can specify a starting path */
      if pass = 2 & fullMask = '' then do
         fullMask = m
         if left(mask,1) = '!' then do
            NotMask = 1
            m = substr(m,2)
         end
         if filespec('drive',m) <> ''
            then m = filespec('path',m)||filespec('name',m)
         primaryMaskIndex = i
         p = filespec('path',m)
         if p <> '' & pos('*',p) = 0 & pos('?',p) = 0 then do
            maskParts.i = translate(filespec('name',m))
            if NotMask = 1
               then maskParts.i = '!'||maskParts.i
         end
      end

      /* remove any drive specifier in the masks as 'fullMask' has that
         and will cause the proper definition of the root drive.
         Our subsequent processing assumes an active mask has no drive info */
      m = maskParts.i
      j = 1
      if left(m,1) = '('
         then j = 2
      ch = substr(m,j,1)
      if ch <> '!'
         then ch = ''
         else j = j + 1
      /* after using scratch variable 'j' as an index we reuse it to reassemble the mask */
      j = strip(substr(m,j),'T',')')
      j = ch||filespec('path',j)||filespec('name',j)
      if left(maskParts.i,1) = '('
         then j = '('||j||')'
      maskParts.i = j
   end
end

/* the first 'non contains' clause can specify a starting directory,
   so we strip off any leading '!' that signals a "not" condition */
if left(fullMask,1) = '!' then do
   fullMask = substr(fullMask,2)
end

if dbg > 0 then do i = 1 to maskParts.0
   say 'Ordering Mask' i 'as "'||maskParts.i||'"'
end

/* set up for recursion down the directory tree */
gblWarnings = ''
currentRecursionLevel = 0

/* if the mask doesn't specify a drive */
rootDrive = filespec('drive',fullMask)
if rootDrive = ''
   then rootDrive = filespec('drive',directory())  /* starting a current drive */
rootDrive = translate(rootDrive)
if dbg > 0 then say 'Drive is' rootDrive

/* Now considering the primary mask, see if the drive and/or current directory
   are defaulted. */
relPath = filespec('path',fullMask)
if relPath = '' then do
   /* no path explicitly specified, pick up the current directory from the drive */
   '@setlocal'
   relPath = translate(filespec('path',directory(rootDrive)||fsSeparator))
   '@endlocal'
end
else do         /* else there is a path to consider */

   /* partial paths can't be used to set the initial root directory */

   /* a ".\" means all subdirs are relative in the tree */
   if left(relPath,2) = '.'||fsSeparator
      then relPath = ''     /* there is no modification to the initial root directory */

   /* we can't set wild cards in the initial root directory */
   j = pos('*',relPath)
   if j > 0
      then relPath = filespec('path',left(relPath,j-1))
   j = pos('?',relPath)
   if j > 0
      then relPath = filespec('path',left(relPath,j-1))

   /* we'll start at the current directory on the requested drive to fully resolve it */
   '@setlocal'
   d = directory(rootDrive)                   /* save where it is */
   if right(d,1) <> fsSeparator               /* checking if x:\ or x:\path */
      then d = d||fsSeparator

   if left(relPath,1) = fsSeparator           /* does it specify starting at root */
      then testDir = filespec('drive',d)||relPath||'.'
      else testDir = d||relPath||'.'

   if dbg >= 2 then say 'Verifying "'||testDIr'" exists...'

   parse value translate(directory(testDir)) with . ':' relPath
   i = directory(d)                           /* restore things back */
   '@endlocal'

   /* lets examine the specified path making sure the relevant parts of the path are valid */
   ques = translate(filespec('path',fullMask))
   if ques <> '' then do
      /* ignore generic parent directory specifications */
      do while left(ques,3) = '..'||fsSeparator
         ques = substr(ques,4)
      end
      if left(ques,2) <>'.'||fsSeparator then   /* reserved meanings for '.' in specifying a path */
         itsRelative = 'False'
      else do
         ques = substr(ques,3)
         itsRelative = 'True'
      end

      /* partial paths are not important at this point as we are checking for definite directories */
      j = pos('*',ques)
      if j > 0 then
         ques = filespec('path',left(ques,j-1))
      j = pos('?',ques)
      if j > 0
         then ques = filespec('path',left(ques,j-1))

      /* is a relevant path portion of the mask actually present? */
      if ques <> '' & pos(ques,translate(relPath||fsSeparator)) = 0 then do
         say "Please verify your initial mask path.  It doesn't appear to be valid."
         say '    For:' d '  "'||fullMask||'"'
         exit 8
      end
      else do
         /* it appears to be valid, so remove that part from the mask as it is
            defined by our initial root directory. */
         i = pos(translate(ques),translate(maskParts.primaryMaskIndex))
         if i > 0 & itsRelative = 'False' then do
            /* we make the remaining parts relative and preserve the 'Not' flag */
            maskParts.primaryMaskIndex = substr(maskParts.primaryMaskIndex,i+length(ques))
            if filespec('path',maskParts.primaryMaskIndex) <> ''
               then maskParts.primaryMaskIndex = '.\'||maskParts.primaryMaskIndex
            if NotMask = 1
               then maskParts.primaryMaskIndex = '!'||maskParts.primaryMaskIndex
         end
         if dbg >=2 then say 'relevant path in mask:' ques '& primary mask:' maskParts.primaryMaskIndex
      end
   end
end
if dbg > 0 then say 'Initial path is' relPath

rootDir = rootDrive||relPath
topLevelDir = rootDir
if wordpos('/E',translate(args)) > 0
   then call ExplainMasks

/* before we rock 'n roll, pause and allow trace to be set when Debugging */
if dbg >= 1 then do
   say 'Ready to begin processing...'
   '@pause'
end
if dbg >= 3 then do
   say 'Option here to MANUALLY set "gblTrace" to part of a file name to trace for debugging "filterAllFiles"'
   trace("?R")
end
gblTrace = ''
if dbg >= 3
   then say 'Global Tracing: "'||gblTrace||'"'

/* process the directory recursively */
call listDirectory rootDir
parse pull thisCount ',' thisSize

if countOnly >= 0 then do
   say '"'||rootDir||'"' thisCount 'entries' prettyNum(thisSize) 'bytes'
end

if gblWarnings <> ''
   then say gblWarnings

exit

prettyNum: procedure
parse arg aNumber .
   prettyNumber = reverse(aNumber)
   do nCommas = 1 while length(prettyNumber) >= 4*nCommas
      prettyNumber = substr(prettyNumber,1,4*(nCommas)-1)||','||substr(prettyNumber,4*(nCommas))
   end
   prettyNumber = reverse(prettyNumber)
return prettyNumber

/*************************
 listDirectory proceedure
*************************/

listDirectory: procedure expose exExtension maskParts. begin_date begin_time comparator t_comparator s_comparator size_arg show_longname ,
                                nesting what tAttrib countOnly rollup showFullNameOnly execTemplate fsSeparator OpSys pauseOpt ,
                                rows cols linesOutput currentRecursionLevel maxRecursionLevel gblWarnings topLevelDir gblTrace dbg
parse arg rootDir
rootDir = strip(rootDir,'T',fsSeparator)

/* Check for exceeding the recursive subdirectory nesting */
if currentRecursionLevel > maxRecursionLevel then do
   if pos('Subdirectory recursion level',gblWarnings) = 0 & maxRecursionLevel > 0 & countOnly > 0
      then gblWarnings = gblWarnings 'Subdirectory recursion level' currentRecursionLevel 'reached, ignoring lower level subdirectories.'
   push '0,0'
   return
end
currentRecursionLevel = currentRecursionLevel + 1

/* list the contents this directory */
options = what||'L' /* 'L' returns date as YYYY-MM-DD HH:MM:SS */

thisCount = 0
thisSize = 0
consolidatedFiles.0 = 0
dirs.0 = 0

      /********* I believe rootDir having a '*' or '?' has been deprecated E **********/
wCard = pos('*',rootDir) + pos('?',rootDir)
if wCard <> 0 then do
   rc = SysFileTree(rootDir,dirs,'D',tAttrib)
   do i = 1 to dirs.0
       aSubDir = subword(dirs.i,5)
       call listDirectory aSubDir||fsSeparator
       parse pull subCount ',' subSize
       thisCount = thisCount + subCount
       thisSize = thisSize + subSize
   end
end
else do    /* check for the mask having a wild card path component */
   do m = 1 to maskParts.0
      /* When mask is not a "contains" mask */
      mask = maskParts.m
      if left(mask,1) <> '(' then do
         /* and it contains a wild card in the path */
         if left(mask,1) = '!' then mask = substr(mask,2)
         wCard = wCard + pos('*',filespec('path',mask)) + pos('?',filespec('path',mask))
         if wCard <> 0 then do
            /* if we previously had a mask with a subdirectory spec */
            if dirs.0 <> 0 then do
               /* this situation is not expected nor coded for.  Error out */
               say 'Error.  Mask specifications too complex.  Can not have multiple paths.'
               do i = 1 to maskParts.0
                  say '   ' maskParts.i
               end
               exit 16
            end

            /* separate out the path specification containing the wild card */
            i = pos('?',mask)
            j = pos('*',mask)
            if j > 0 & (j < i | i = 0)
               then i = j

            /* find the end of this particular subdirectory specification */
            j = pos(fsSeparator,mask,i)

            /* Temporarily replace this mask */
            saveMask = maskParts.m
            maskParts.m = substr(mask,j+1)
            startingAt = rootDir || fsSeparator

            /* preserve the Not flag */
            if left(saveMask,1) = '!'
               then maskParts.m = '!'||maskParts.m

            /* if there is a path specification */
            if filespec('path',mask) <> '' then do
               /* skip over the '.\' relative indicator to traverse the tree */
               if left(mask,2) = '.'||fsSeparator then do
                  mask = substr(mask,3)
                  j = j - 2
               end
               else do    /* always start at the top level directory, so only needed once */
                  startingAt = topLevelDir || fsSeparator
                  if currentRecursionLevel > 1 then do
                     maskParts.m = saveMask
                     iterate
                  end
               end
            end

            /* Enumerate the subdirectories that fit this mask */
            rc = SysFileTree(startingAt||left(mask,j-1),dirs,'D',tAttrib)

            /* recurse the directories */
            do i = 1 to dirs.0
                aSubDir = subword(dirs.i,5)
                call listDirectory aSubDir||fsSeparator
                parse pull subCount ',' subSize
                if countOnly >= 0 & subCount >= countOnly then do
                   say '"'||aSubDir'"' subCount 'entries' prettyNum(subSize) 'bytes'
                end
                if rollup > 0 then do
                   thisCount = thisCount + subCount
                   thisSize = thisSize + subSize
                end
            end

            /* restore the mask */
            maskParts.m = saveMask
         end
      end
   end

   /* if there wasn't an embedded path spec in the mask that we just handled */
   if dirs.0 = 0 then do m = 1 to maskParts.0
      /* process each mask putting all the individual results into files. */
      mask = maskParts.m
      ContainsClause = 0
      if left(mask,1) = '(' then do
         ContainsClause = 1
         parse var mask . '(' mask ')'
      end

      NotMask = 0
      if left(mask,1) = '!' then do
         NotMask = 1
         mask = substr(mask,2)
      end

      /* mask syntax and meaning, where '!' inverts the test:
            "mask"            search current 'rootDir' directory
            ".\path\mask"     search current 'rootDIr' with relative path
            "path\mask"       search initial 'path' starting directory
            "\path\mask"      search path from ROOT of the drive
      */
      startingAt = rootDir || fsSeparator
      if filespec('path',mask) <> '' then do
         /* we must consider how the 'path' affects the searching */
         if left(mask,3) = '..\' then do
            /* traverse back up the tree to the parent */
            do while left(mask,3) = '..\'
               mask = substr(mask,4)
               startingAt = filespec('drive',startingAt)||filespec('path',strip(startingAt,'T',fsSeparator))
            end
         end
         else if left(mask,1) = '.'
            /* it is relative to where we are presently in the directory tree,
                   so we remove the '.\' which means the current directory. */
            then mask = substr(mask,pos(fsSeparator,mask)+1)
         else if left(mask,1) = fsSeparator
            then startingAt = filespec('drive',topLevelDir)    /* fixated on the Drive */
            else do
               /* signal a fixed search that we only perform once by setting mask starting with \ */
               mask = fsSeparator||strip(mask,'L',fsSeparator)
               startingAt = topLevelDir
            end
      end

      if dbg >= 2 then do
         if ContainsClause = 1
            then j = '"Contains" '
            else j = ''

         say 'currently in:' rootDir 'processing' j||'mask' maskParts.m
         if NotMask = 1
            then say '  filtering rules: NOT' startingAt||mask
            else say '  filtering rules:' startingAt||mask
         say 'Enter S to Stop, T to Trace, otherwise continue...'
         pull response
         if translate(left(response,1)) = 'S' then exit 1
         if translate(left(response,1)) = 'T' then trace("?R")
      end

      /* if this mask is specifying the subdirectory must (or must not) contain
         at least 1 of the files in the mask */
      if ContainsClause = 1 then do
         /* contains clause syntax and meaning, where '!' inverts the test:
               (mask)            current directory contains a file
               (path\mask)       initial starting directory + path contains a file
               (.\path\mask)     path relative to where we are contains a file
               (\path\mask)      hard path at top of the drive contains a file
         */
         startingAt = rootDir || fsSeparator
         if filespec('path',mask) <> '' then do
            /* we must consider how the 'path' affects the searching */
            if left(mask,1) = '.' then do
               /* it is relative to where we are presently in the directory tree,
                      so we remove the '.\' which means the current directory.
                      Note, a '..\' syntax is not supported */
               mask = substr(mask,pos(fsSeparator,mask)+1)
            end
            else if left(mask,1) = fsSeparator
               then startingAt = filespec('drive',topLevelDir)    /* fixated on the Drive */
               else startingAt = topLevelDir || fsSeparator       /* fixated where we started */
         end

         if dbg >= 2 then do
            say 'currently in:' rootDir 'processing "contains" mask' maskParts.m
            if NotMask = 1
               then say '  filtering rules: NOT' startingAt||mask
               else say '  filtering rules:' startingAt||mask
         end

         if OpSys = 'UNIX' & substr(tAttrib,3,1) = '-'
            /* Hidden files work differently on Unix/Linux */
            then rc = SysFileTree(startingAt || mask ,files,options,substr(tAttrib,1,2)||'*'||substr(tAttrib,4))
            else rc = SysFileTree(startingAt || mask ,files,options,tAttrib)
          fileCount = files.0
          drop files.
          if dbg >= 2 then say '      found' fileCount 'matching files.'

         /* if NO files were found */
         if fileCount = 0 then do
            /* Then IF the "not" condition is false, then we must find something to continue on */
            if NotMask = 0 then leave
         end
         else /* some files were found, so ... */
            if NotMask = 1 then leave     /* if we didn't want to find them, then we leave the loop */

         /* we've processed the 'contains' clause and can conntinue to the next mask */
         iterate
      end

      /* any path hard coded to the root is processed only once */
      if currentRecursionLevel > 1 & left(mask,1) = fsSeparator
         then iterate

      /* we handled any path portion in the setup of the starting search directory */
      /*
      mask = filespec('name',mask)
      */

      /* can the system query handle the type of mask we were given? */
      if filespec('path',maskParts.m) = '' & NotMask = 0 then do
         /* Hidden files work differently on Unix/Linux */
         if OpSys = 'UNIX' & substr(tAttrib,3,1) = '-'
            then rc = SysFileTree(startingAt || mask ,files,options,substr(tAttrib,1,2)||'*'||substr(tAttrib,4))
            else rc = SysFileTree(startingAt || mask ,files,options,tAttrib)
          fileCount = files.0
      end
      else do
         if filespec('path',maskParts.m) <> '' & left(maskParts.m,1) <> '.' then do
            /* there is some parts of a specific path to match
               If starts with a '\' then hard path on the drive,
                  else hard path in our initial starting directory */
            if left(maskParts.m,1) = fsSeparator
               then mask = filespec('drive',topLevelDIr)||maskParts.m
               else mask = filespec('drive',topLevelDIr)||filespec('path',topLevelDIr||fsSeparator||'.')||maskParts.m
         end
         else mask = startingAt||mask
         if NotMask = 1 | pos('?',maskParts.m) > 0 | pos('*',filespec('path',mask)) > 0
            /* Looking for NOT Matching or single match character rules */
               then fileCount = filterAllFiles(mask)
         else do
            /* Hidden files work differently on Unix/Linux */
            if OpSys = 'UNIX' & substr(tAttrib,3,1) = '-'
               then rc = SysFileTree( mask ,files,options,substr(tAttrib,1,2)||'*'||substr(tAttrib,4))
               else rc = SysFileTree( mask ,files,options,tAttrib)
             fileCount = files.0
         end
      end

      /* if we got some new files */
      if fileCount > 0 then do      /* merge new ones with the old ones */
         rc = SysStemCopy(files,consolidatedFiles,1,consolidatedFiles.0+1)
      end
   end

   drop files.
   files.0 = 0

   /* As necessary, sort the results so we can remove duplicates */
   if consolidatedFiles.0 > 0 then do
      rc = SysStemSort(consolidatedFiles, 'A', 'I', 1, consolidatedFiles.0, 42, 290)

      /* copy the 'consolidated' results into 'files'. removing any duplicates */
      j = 1
      files.0 = j
      files.j = consolidatedFiles.1
      do i = 2 to consolidatedFiles.0
         lnA = files.j
         lnB = consolidatedFiles.i
         if lnA <> lnB then do
            j = j + 1
            files.j = lnB
            files.0 = j
         end
      end
      drop consolidatedFiles.
   end

   /* Filter on Date/Time/Size as necessary and Format the results for output */
   Do i = 1 to files.0
      l = files.i
      parse var l dt tm size attr fname
      fname = strip(fname,'L')      /* remove any leading blanks */
      shortName = filespec('name',fname)

      /* if on Unix/Linux and requesting non hidden files,
                 then file name must not start with a '.'             */
      if OpSys = 'UNIX' & substr(tAttrib,3,1) = '-' & left(shortname,1) = '.'
         then iterate

      spot = pos(fname,l)-1

      if show_longname > 0 then do
         rc = SysGetEA(fname,'.LONGNAME','longname')
         if (rc = 0 & length(longname) > 4) then do
            fname = filespec('drive',fname)||filespec('path',fname)||delstr(longname,1,4)
         end
         else shortName = ''
      end

      /* filter by Modification date (and time) */
      if begin_date > 0 then do
         if pos('/',dt) > 0 then do
            parse var dt mm '/' dd '/' yy
            if yy <= 99 & yy >= 70
               then yy = yy + 1900
               else yy = yy + 2000
            compare_date = yy||'-'||right(100+mm,2)||'-'||right(100+dd,2)
         end
         else do
            compare_date = dt
         end

         select
           when comparator = '' then do
                if compare_date < begin_date then iterate
             end
           when comparator = 'E' then do
                if compare_date \= begin_date then iterate
             end
           when comparator = 'G' then do
                if compare_date < begin_date then iterate
             end
           when comparator = 'L' then do
                if compare_date > begin_date then iterate
             end
           otherwise if compare_date < begin_date then iterate
         end

         pm = 0
         if right(tm,1) = 'p' then pm = 12
         parse value left(tm,length(tm)-1) with hr ':' min ':' sec
         minutes = ((hr + pm) * 60) + min
         select
           when t_comparator = '' then do
                /* interpret the time comparator using date rules */
                if compare_date = begin_date then do
                   if comparator = 'L' then do
                      if minutes >= begin_time then iterate
                   end
                   else do
                      if minutes < begin_time then iterate
                   end
                end
                /* else the time component is irrelevant as the date argument
                   has already filtered the file's timestamp */
             end
           when t_comparator = 'E' then do
                if minutes \= begin_time then iterate
             end
           when t_comparator = 'G' then do
                if minutes < begin_time then iterate
             end
           when t_comparator = 'L' then do
                if minutes > begin_time then iterate
             end
           otherwise if minutes < begin_time then iterate
         end

      end

      /* filter by Size */
      if s_comparator \= '' then do
         select
            when s_comparator = 'E' then do
                 if size \= size_arg then iterate
              end
            when s_comparator = 'G' then do
                 if size < size_arg then iterate
              end
            when s_comparator = 'L' then do
                 if size > size_arg then iterate
              end
            otherwise do
                 say 'Error in logic.  s_comparator value "'||s_comparator||'" is unprogrammed!'
              end
         end
      end

      /* reformat the output to enclose the file name in quotes in case of blanks */
      output = left(l,spot)||'"'||fname||'"'
      if show_longname > 1 then
         output = left(l,spot)|| left(shortName||"            ",12) '"'||fname||'"'
      if countOnly < 0 then do
         if showFullNameOnly >= 1 then do
            parse var output with '"' fullPathName '"'
            if execTemplate <> '' then do
               output = execTemplate
               spot = pos('%',output)
               do while spot > 0
                  /* substituting the parts of the file path name */
                  if substr(output,spot+1,1) = 'D'        /* drive only */
                     then rc = filespec('drive',fullPathName)
                  else if substr(output,spot+1,1) = 'P'   /* path only */
                     then rc = filespec('path',fullPathName)
                  else if substr(output,spot+1,1) = 'N'   /* name only */
                     then rc = filespec('name',fullPathName)
                  else rc = fullPathName

                  if rc = fullPathName   /* was it only a % or a %D or %P or %N */
                     then output = left(output,spot-1)||rc||substr(output,spot+1)
                     else output = left(output,spot-1)||rc||substr(output,spot+2)

                  spot = pos('%',output)
               end
            end
         end

         /* if the '/EXEC' modifier option was requested */
         if showFullNameOnly >= 3 then do
            /* we will Execute the command */
            if right(translate(word(execTemplate,1)),4) = translate(exExtension)  /* typically '.CMD' or perhaps '.REX' */
               then output = '@call' output
            interpret "'"||output||"'"

            /* should we pause after each invocation so results can be checked? */
            if pauseOpt then do
               call charout ,'Press enter to continue...(or stop)'
               pull rc
               if rc <> '' then do
                  say '  ... Early exit request acknowledged.'
                  exit 2
               end
            end
         end
         else do     /* just output the formatted results */
            if pauseOpt then do
               /* considering a possible line wrap, count another line of output */
               linesNeeded = trunc(length(output)/cols)+1
               linesOutput = linesOutput + linesNeeded
               if linesOutput >= rows then do
                  /* place the prompt at the bottom and no extra linefeed */
                  call charout ,'Press enter to continue...(or stop)'
                  pull rc
                  if rc <> '' then do
                     say '  ... Early exit request acknowledged.'
                     exit 2
                  end
                  linesOutput = linesNeeded
               end
            end
            say strip(output)
         end
      end

      thisCount = thisCount + 1
      thisSize = thisSize + size

   End
End

/* if to display the nested directory information */;
if nesting \= 0 & dirs.0 = 0 then do
  rc = SysFileTree(rootDir || fsSeparator||'*',subDirs,'O',"*+***")

  Do d = 1 to subDirs.0
     call listDirectory subDirs.d
     parse pull subCount ',' subSize
     if countOnly >= 0 & subCount >= countOnly then do
        say '"'||subdirs.d||'"' subCount 'entries' prettyNum(subSize) 'bytes'
     end
     if rollup > 0 then do
        thisCount = thisCount + subCount
        thisSize = thisSize + subSize
     end
  End
End

push thisCount||','||thisSize
currentRecursionLevel = currentRecursionLevel - 1
return

/*************************************
 proceedure to handle "NOT" conditions
*************************************/

filterAllFiles:
   arg currentMask        /* assumption being currentMask has been translated to upper case for optimization */

   /* Looking for NOT Matching, so list all files and then remove those that match */
   if OpSys = 'UNIX' & substr(tAttrib,3,1) = '-'
      then rc = SysFileTree(rootDir || fsSeparator || '*' ,allFiles,options,substr(tAttrib,1,2)||'*'||substr(tAttrib,4))
      else rc = SysFileTree(rootDir || fsSeparator || '*' ,allFiles,options,tAttrib)

   files.0 = 0
   Do i = 1 to allFiles.0
      l = allFiles.i
      parse var l dt tm size attr fname

traceIt = 0
if pos(translate(gblTrace),translate(filespec('name',fname))) > 0 then do
   trace("?R")
   say 'Ready to trace file' fname
   traceIt = 1
end

      if filespec('path',currentMask) = ''
         then fname = translate(filespec('name',strip(fname,'L')))     /* disregard path part & rest is case independent */
      else do
         j = filespec('drive',fname)
         fname = strip(fname,'L')                                      /* disregard the drive keeping path & name case independent */
         fname = translate(filespec('path',fname)||filespec('name',fname))
         if filespec('drive',currentMask) <> ''                        /* however, mask had a drive spec */
            then fname = translate(strip(j))||fname                    /* so we put it back in for the comparisons */
      end

      /* process the mask character by character */
      maskP = 1
      fnameP = 1
      do while fnameP <= length(fname) & maskP <= length(currentMask)
         maskC = substr(currentMask,maskP,1)
         Select
            when maskC = '?' then do
               /* wild card match one character position only */
               fnameP = fnameP + 1
               maskP = maskP + 1
            end
            when maskC = '*' then do
if traceIt = 1 then trace("?R")
               /* find the position in the mask of the next wild card (if any) */
               consumeUpTo = pos('*',currentMask,maskP + 1)
               nextWildCardQuesP = pos('?',currentMask,maskP + 1)
               if consumeUpTo = 0 | (nextWildCardQuesP > 0 & nextWildCardQuesP < consumeUpTo)
                  then consumeUpTo = nextWildCardQuesP

               /* set the search string we are to match */
               if consumeUpTo > 0
                  then toMatch = substr(currentMask,maskP+1,consumeUpTo - maskP - 1)
                  else toMatch = substr(currentMask,maskP+1)

               /* if there is more to match */
               if toMatch <> '' then do
                  /* be greedy in matching a '*' wild card to the most we can grab */
                  rc = fnameP   /* remember how far we've gone thru the candidate */
                  fnameP = lastpos(toMatch,fname)
                  if fnameP > 0 & fnameP > rc
                     then fnameP = fnameP + length(toMatch)
                     else leave
               end
               else fnameP = length(fname)+1     /* final wild card matches all that's left */

               maskP = maskP + 1 + length(toMatch)
            end
            otherwise do
               /* if not a matching character, we are done and failed to match */
               if substr(fname,fnameP,1) <> maskC
                  then leave

               fnameP = fnameP + 1
               maskP = maskP + 1
            end
         End
      end

if traceIt = 1 then trace("?R")
      /* if it DID NOT match, i.e.:
         More "fname" left, or more "mask", or not at an ending '*' in the mask,
         this one should be included  as a "no match" */
      if length(fname) >= fnameP | length(currentMask) > maskP | (substr(currentMask,maskP,1) <> '*' & maskP = length(currentMask)) then do
         if NotMask = 1 then do
            rc = files.0 + 1
            files.rc = l
            files.0 = rc
         end
      end
      else if NotMask = 0 then do
         rc = files.0 + 1
         files.rc = l
         files.0 = rc
      end
   end
return files.0

/* this routine explains what the specified masks will attempt to do */
ExplainMasks:
   say ''
   say 'Processing order of mask: "'||maskArg||'"'
   say '    Analyzing first part of the mask, operation begins at:' topLevelDir
   say ''
   orderWords = 'First, Secondly Next Then'
   do m = 1 to maskParts.0
      if m < words(orderWords)-1
         then Phrase = word(orderWords,m)
         else Phrase = word(orderWords,words(orderWords))

      mask = maskParts.m
      maskInternals = mask
      NotPhrase = ''
      Clause = 'match'
      kind = 'files'

      if left(maskInternals,1) = '(' then do
         Clause = 'CONTAIN 1 or more'
         kind = 'a directory'
         parse var maskInternals . '(' maskInternals ')' .
      end

      if left(maskInternals,1) = '!' then do
         NotPhrase = ' NOT'
         maskInternals = substr(maskInternals,2)
      end

      say m||'.' Phrase mask 'means' kind 'must'||NotPhrase Clause maskInternals
      msg2 = ''
      msg3 = ''
      posSeparator = pos(fsSeparator,maskInternals)
      if posSeparator > 0 then do
         if posSeparator = 1
            then posSeparator = pos(fsSeparator,maskInternals,posSeparator+1)
         firstSubdir = substr(maskInternals,1,posSeparator-1)
         msg1 = 'As it starts with "'.'" it is relative to nested subdirectories in the tree'
         ch = left(firstSubdir,1)
         if ch = fsSeparator
            then msg1 = 'Since it starts with "'||fsSeparator||'", it is processed ALWAYS from the root' filespec('drive',rootDrive)||fsSeparator
         else if ch <> '.' then do
            msg1 = 'As it has a path "'||firstSubdir||'" that is not relative ".\",'
            msg2 = 'It is searched under initial directory' topLevelDir
            msg3 = "AND it's tree is not recursively traversed!"
         end
      end
      else do
          msg1 ='As it contains no path information,'
          msg2 ='each subdirectory is searched as the tree is traversed.'
      end

      say '    ' msg1
      if msg2 <> '' then say '         ' msg2
      if msg3 <> '' then say '         ' msg3
   end

   say 'Enter "Q" to Quit, Otherwise proceed...'
   pull response
   if translate(left(response,1)) = 'Q'
      then exit 2
return
