#include <ctype.h>
#include <limits.h>
#include <dnpap.h>
#include <message.h>
#include <stdio.h>
#include <stdlib.h>
#include <config.h>
#include <string.h>
#include <mibdf.h>

#ifndef OS2
#include <floatingpoint.h>
#endif
      

#include "defs.h"
#include "funcs.h"
#include "hosts.h"


#define MAXNAME         256
#define MAXLINE         2048
#define MAXBYTES        2048
#define MAXLONGS        1024
#define MAXIPADDR       4



enum CFG2INI_ERRORS
{                      
        NO_ERR,
        USAGE_ERR,
        OPTION_ERR,
        MIB_ERR,
        OPEN_ERR,
        LOAD_ERR,
        SAVE_ERR,
        TYPESYNTAX_ERR,
        NAMESYNTAX_ERR,
        PREGARBAGE_ERR,
        BOOLEANPTR_ERR,
        SHORTPTR_ERR,
        LONGPTR_ERR,
        IPADDRPTR_ERR,
        DOUBLEPTR_ERR,
        STRINGPTR_ERR,
        BYTESPTR_ERR,
        LONGBUFPTR_ERR,
        MIBIDPTR_ERR,
        NULLPTR_ERR,
        CONFIGCREATE_ERR,
        CONFIGLEVEL_ERR,
        CONFIGVALUE_ERR
};

         
static CHAR *home;        /*  global variable that contains home directory  */



static BOOLEAN ParseFile(CHAR* name);
static BOOLEAN ParseLine(CHAR* fname, CHAR* line, INT linenr);
static VOID RemoveComment(CHAR *str);
static VOID RemoveBlanks(CHAR *str);
static VOID Usage(VOID);
static BOOLEAN strtot(CHAR* ptr, CHAR** eptr, BOOLEAN* b, INT* n);
static BOOLEAN strtob(CHAR* ptr, CHAR** eptr, BYTE* buf, INT* n);
static BOOLEAN strtolb(CHAR* ptr, CHAR** eptr, LONG* buf, INT* n);
static BOOLEAN strtoid(CHAR* ptr, CHAR** eptr, LONG* buf, INT* n);
static BOOLEAN strtoip(CHAR* ptr, CHAR** eptr, IPADDR* buf, INT* n);
static BOOLEAN strtohost(CHAR* ptr, CHAR** eptr, LONG* buf, INT* n);



int main(int argc, char *argv[])
{
INT rc = TRUE;
INT argi, i;
                                       
        MessageInit("stdio:err", MSGFILE, MSGSTDIO, NULL);
        MessageConfig(NO_ERR, "Cfg2Ini");
        
        if (argc < 2)
        {
                MESSAGE(DMC_FATAL, USAGE_ERR, "argument missing");
                Usage();
                DnpapExit(1);
        }

        if ((home = getenv("HOME")) == NULL)
                home = ".";

        if (MibInit() != 0 || MibFullMib() != 0)
        {
                MESSAGE(DMC_FATAL, MIB_ERR, "MIB initialization failed");
                DnpapExit(1);
        }
        
        for (argi = 1; argi < argc; argi++)
        {
                if (argv[argi][0] == '-')
                        for (i = 1; i < STRLEN(argv[argi]); i++)
                                switch (argv[argi][i])
                                {
                                case 'h':
                                        Usage();
                                        DnpapExit(0);
                                default:
                                        MESSAGE(DMC_FATAL, OPTION_ERR, "unrecognized option");
                                        Usage();
                                        DnpapExit(1);
                                }
                else
                if (ParseFile(argv[argi]) == FALSE)
                        rc = FALSE;
        }

        if (rc == FALSE)
                DnpapExit(1);
        else
                DnpapExit(0);
        return 0;
}


BOOLEAN ParseFile(CHAR* name)
{
BOOLEAN rc;
FILE *infh;
CHAR line[MAXLINE];
INT linenr;
CHAR base[MAXNAME];        
extern BYTE configBase[];
          
        rc = TRUE;
        
        if ((infh = fopen(name, "r")) == NULL)
        {
                MESSAGE(DMC_ERROR, OPEN_ERR, "open of input file %s failed", name);
                return FALSE;
        }

        Basename(name, base, sizeof(base));
        strcpy(configBase, base);


        fgets(line, MAXLINE, infh);
        linenr = 1;

        while (!feof(infh))
        {
                RemoveComment(line);
                RemoveBlanks(line);
                if (STRLEN(line) != 0 &&
                    (rc = ParseLine(name, line, linenr)) == FALSE)
                        break;
                fgets(line, MAXLINE, infh);
                linenr++;
        }
                                              
        if (rc == TRUE)
        {
                /*  write variables to opened files          */                  
                /*  for now we only save to the current dir  */
                if (ConfigSave() == FALSE)
                {
                        MESSAGE(DMC_FATAL, SAVE_ERR, "saving of INI files failed");
                        return FALSE;
                }
        }
                        
        fclose(infh);

        return rc;
}


BOOLEAN ParseLine(CHAR* fname, CHAR* line, INT linenr)
{
INT i, j, tmpi;
USHORT level;
CHAR name[MAXLINE];
CHAR tmps[MAXLINE];
USHORT type;
CHAR *ptr, *eptr;
INT n;
BOOLEAN bval;
SHORT sval;
LONG lval;
IPADDR ival;
DOUBLE dval;
BYTE buf[MAXBYTES];
LONG lbuf[MAXLONGS];
CONFIG_ENTRY* e;

        /*  find first '=' token  */
        for (i = 0; i < STRLEN(line); i++)
                if (line[i] == '=')
                        break;

        if (i == STRLEN(line))          /*  no '=' token then error  */
        {
                MESSAGE(DMC_FATAL, TYPESYNTAX_ERR, "missing '=', file %s, line %d, column %d", fname, linenr, i+1);
                return FALSE;
        }

        tmpi = i;

        /*  go 1 token further and skip whitespace  */
        for (i++ ; i < STRLEN(line) && isspace(line[i]); i++)
                ;

        if (line[i] != '(')
        {
                MESSAGE(DMC_FATAL, TYPESYNTAX_ERR, "missing '(' in type specification, file %s, line %d, column %d", fname, linenr, i+1);
                return FALSE;
        }

        for (j = i++ ; j < STRLEN(line) && line[j] != ')'; j++)
                ;

        if (line[j] != ')')
        {
                MESSAGE(DMC_FATAL, TYPESYNTAX_ERR, "missing ')' in type specification, file %s, line %d, column %d", fname, linenr, j+1);
                return FALSE;
        }

        strncpy(tmps, line+i, j-i);
        tmps[j-i] = '\0';

        RemoveBlanks(tmps);

        if (strcasecmp(tmps, "Boolean") == 0)
                type = CONFIG_TYPE_BOOLEAN;
        else
        if (strcasecmp(tmps, "Short") == 0)
                type = CONFIG_TYPE_SHORT;
        else
        if (strcasecmp(tmps, "Long") == 0)
                type = CONFIG_TYPE_LONG;
        else
        if (strcasecmp(tmps, "IPAddr") == 0)
                type = CONFIG_TYPE_IPADDR;
        else
        if (strcasecmp(tmps, "Double") == 0)
                type = CONFIG_TYPE_DOUBLE;
        else
        if (strcasecmp(tmps, "String") == 0)
                type = CONFIG_TYPE_STRING;
        else
        if (strcasecmp(tmps, "Bytes") == 0)
                type = CONFIG_TYPE_BYTES;
        else
        if (strcasecmp(tmps, "LongBuf") == 0)
                type = CONFIG_TYPE_LONGBUF;
        else
        if (strcasecmp(tmps, "MibID") == 0)
                type = CONFIG_TYPE_MIBID;
        else
        if (strcasecmp(tmps, "Null") == 0)
                type = CONFIG_TYPE_NULL;
        else
        {
                MESSAGE(DMC_FATAL, TYPESYNTAX_ERR, "invalid type, file %s, line %d, column %d", fname, linenr, i+1);
                return FALSE;
        }

        ptr = &line[j+1];       /*  let 'ptr' point to begin of specified value  */

        /*  go back from '=' position and skip whitespace  */
        for (i = tmpi-1; i >= 0 && isspace(line[i]); i--)
                ;

        if (i < 0)
        {
                MESSAGE(DMC_FATAL, NAMESYNTAX_ERR, "no name defined, file %s, line %d, column %d", fname, linenr, i+1);
                return FALSE;
        }

        for (j = i ; j >= 0 && !isspace(line[j]); j--)
                ;

        strncpy(name, line+j+1, i-j);
        name[i-j] = '\0';

        /*  go back and skip whitespace  */
        for (i = j; i >= 0 && isspace(line[i]); i--)
                ;

        if (i < 0)
                level = 0;   /*  assume variable on level 0  */
        else
        {
                MESSAGE(DMC_FATAL, PREGARBAGE_ERR, "garbage at begin of line, file %s, line %d, column %d", fname, linenr, j+1);
                return FALSE;
        }

        if ((e = ConfigCreate(name)) == NULL)
        {
                MESSAGE(DMC_FATAL, CONFIGCREATE_ERR, "couldn't create entry", fname, linenr);
                return FALSE;
        }
        if (ConfigSetLevel(e, level) == FALSE)
        {
                MESSAGE(DMC_FATAL, CONFIGLEVEL_ERR, "couldn't set level", fname, linenr);
                return FALSE;
        }

        RemoveBlanks(eptr = ptr);      /*  remove blanks around value  */

        switch (type)
        {
        case CONFIG_TYPE_BOOLEAN:
                if (STRSPN(ptr, "01trueTRUEfalseFALSEyesYESnoNOdownDOWNupUP") != STRLEN(ptr))
                {
                        MESSAGE(DMC_FATAL, BOOLEANPTR_ERR, "illegal characters in boolean value, file %s, line %d, column %d", fname, linenr, ptr-line+1);
                        return FALSE;
                }  
                if (strtot(ptr, &eptr, &bval, &n) == FALSE || eptr == ptr)
                {
                        MESSAGE(DMC_FATAL, BOOLEANPTR_ERR, "boolean value can't be parsed, file %s, line %d, column %d", fname, linenr, eptr-line+1);
                        return FALSE;
                }                 
                else
                if (STRLEN(eptr) > 0)
                {
                        MESSAGE(DMC_WARNING, BOOLEANPTR_ERR, "trailing garbage ignored, file %s, line %d, column %d", fname, linenr, eptr-line+1);
                }
                if (ConfigSetValueBoolean(e, bval) == FALSE)
                {
                        MESSAGE(DMC_FATAL, CONFIGVALUE_ERR, "couldn't set value", fname, linenr);
                        return FALSE;
                }
                break;
        case CONFIG_TYPE_SHORT:
                if (STRSPN(ptr, "+-x0123456789abcdefABCDEF") != STRLEN(ptr))
                {
                        MESSAGE(DMC_FATAL, SHORTPTR_ERR, "illegal characters in short value, file %s, line %d, column %d", fname, linenr, ptr-line+1);
                        return FALSE;
                }  
                if ((lval = strtol(ptr, &eptr, 0)) < SHRT_MIN || lval > SHRT_MAX)
                {
                        MESSAGE(DMC_FATAL, SHORTPTR_ERR, "short value out of range, file %s, line %d, column %d", fname, linenr, ptr-line+1);
                        return FALSE;
                }  
                if (eptr == ptr)
                {
                        MESSAGE(DMC_FATAL, SHORTPTR_ERR, "short value can't be parsed, file %s, line %d, column %d", fname, linenr, eptr-line+1);
                        return FALSE;
                }                 
                else
                if (STRLEN(eptr) > 0)
                {
                        MESSAGE(DMC_WARNING, SHORTPTR_ERR, "trailing garbage ignored, file %s, line %d, column %d", fname, linenr, eptr-line+1);
                }
                sval = (SHORT) lval;
                if (ConfigSetValueShort(e, sval) == FALSE)
                {
                        MESSAGE(DMC_FATAL, CONFIGVALUE_ERR, "couldn't set value", fname, linenr);
                        return FALSE;
                }
                break;
        case CONFIG_TYPE_LONG:
                if (STRSPN(ptr, "+-x0123456789abcdefABCDEF") != STRLEN(ptr))
                {
                        MESSAGE(DMC_FATAL, LONGPTR_ERR, "illegal characters in long value, file %s, line %d, column %d", fname, linenr, ptr-line+1);
                        return FALSE;
                }
                lval = strtol(ptr, &eptr, 0);
                if (eptr == ptr)
                {
                        MESSAGE(DMC_FATAL, LONGPTR_ERR, "long value can't be parsed, file %s, line %d, column %d", fname, linenr, eptr-line+1);
                        return FALSE;
                }  
                else
                if (STRLEN(eptr) > 0)
                {
                        MESSAGE(DMC_WARNING, LONGPTR_ERR, "trailing garbage ignored, file %s, line %d, column %d", fname, linenr, eptr-line+1);
                }
                if (ConfigSetValueLong(e, lval) == FALSE)
                {
                        MESSAGE(DMC_FATAL, CONFIGVALUE_ERR, "couldn't set value", fname, linenr);
                        return FALSE;
                }
                break;
        case CONFIG_TYPE_IPADDR:
                if (STRSPN(ptr, "0123456789.") != STRLEN(ptr))
                {
                        MESSAGE(DMC_FATAL, IPADDRPTR_ERR, "illegal characters in IP address, file %s, line %d, column %d", fname, linenr, ptr-line+1);
                        return FALSE;
                }
                if (strtoip(ptr, &eptr, &ival, &n) == FALSE || eptr == ptr)
                {
                        MESSAGE(DMC_FATAL, IPADDRPTR_ERR, "IP address can't be parsed, file %s, line %d, column %d", fname, linenr, eptr-line+1);
                        return FALSE;
                }  
                else
                if (STRLEN(eptr) > 0)
                {
                        MESSAGE(DMC_WARNING, IPADDRPTR_ERR, "trailing garbage ignored, file %s, line %d, column %d", fname, linenr, eptr-line+1);
                }
                if (ConfigSetValueIPAddr(e, ival) == FALSE)
                {
                        MESSAGE(DMC_FATAL, CONFIGVALUE_ERR, "couldn't set value", fname, linenr);
                        return FALSE;
                }
                break;
        case CONFIG_TYPE_DOUBLE:
                if (STRSPN(ptr, "+-0123456789.E") != STRLEN(ptr))
                {
                        MESSAGE(DMC_FATAL, DOUBLEPTR_ERR, "illegal characters in double value, file %s, line %d, column %d", fname, linenr, ptr-line+1);
                        return FALSE;
                }
                dval = strtod(ptr, &eptr);
                if (eptr == ptr)
                {
                        MESSAGE(DMC_FATAL, DOUBLEPTR_ERR, "double value can't be parsed, file %s, line %d, column %d", fname, linenr, eptr-line+1);
                        return FALSE;
                }  
                else
                if (STRLEN(eptr) > 0)
                {
                        MESSAGE(DMC_WARNING, DOUBLEPTR_ERR, "trailing garbage ignored, file %s, line %d, column %d", fname, linenr, eptr-line+1);
                }
                if (ConfigSetValueDouble(e, dval) == FALSE)
                {
                        MESSAGE(DMC_FATAL, CONFIGVALUE_ERR, "couldn't set value", fname, linenr);
                        return FALSE;
                }
                break;
        case CONFIG_TYPE_STRING:
                if (*ptr == '\"' && *(ptr+STRLEN(ptr)-1) != '\"')
                    MESSAGE(DMC_WARNING, STRINGPTR_ERR, "only quotes at the start of string value, file %s, line %d", fname, linenr);
                if (*ptr != '\"' && *(ptr+STRLEN(ptr)-1) == '\"')
                    MESSAGE(DMC_WARNING, STRINGPTR_ERR, "only quotes at the end of string value, file %s, line %d", fname, linenr);
                if (*ptr == '\"' && *(ptr+STRLEN(ptr)-1) == '\"')
                {
                        ptr++;
                        *(ptr+STRLEN(ptr)-1) = '\0';
                }
                if (ConfigSetValueString(e, ptr) == FALSE)
                {
                        MESSAGE(DMC_FATAL, CONFIGVALUE_ERR, "couldn't set value", fname, linenr);
                        return FALSE;
                }
                break;
        case CONFIG_TYPE_BYTES:
                if (*ptr == '\"' && *(ptr+STRLEN(ptr)-1) != '\"')
                    MESSAGE(DMC_WARNING, BYTESPTR_ERR, "only quotes at the start of bytes value, file %s, line %d", fname, linenr);
                if (*ptr != '\"' && *(ptr+STRLEN(ptr)-1) == '\"')
                    MESSAGE(DMC_WARNING, BYTESPTR_ERR, "only quotes at the end of bytes value, file %s, line %d", fname, linenr);
                if ((ptr[0] != '\"' || *(ptr+STRLEN(ptr)-1) != '\"') &&
                    STRSPN(ptr, "x0123456789abcdefABCDEF:") != STRLEN(ptr))
                {
                        MESSAGE(DMC_FATAL, BYTESPTR_ERR, "illegal characters in bytes value, file %s, line %d, column %d", fname, linenr, ptr-line+1);
                        return FALSE;
                }
                if (strtob(ptr, &eptr, buf, &n) == FALSE || eptr == ptr)
                {
                        MESSAGE(DMC_FATAL, BYTESPTR_ERR, "bytes value can't be parsed, file %s, line %d, column %d", fname, linenr, eptr-line+1);
                        return FALSE;
                }  
                else
                if (STRLEN(eptr) > 0)
                {
                        MESSAGE(DMC_WARNING, BYTESPTR_ERR, "trailing garbage ignored, file %s, line %d, column %d", fname, linenr, eptr-line+1);
                }
                if (ConfigSetValueBytes(e, buf, n) == FALSE)
                {
                        MESSAGE(DMC_FATAL, CONFIGVALUE_ERR, "couldn't set value", fname, linenr);
                        return FALSE;
                }
                break;
        case CONFIG_TYPE_LONGBUF:
                if (STRSPN(ptr, "x0123456789abcdefABCDEF.") != STRLEN(ptr))
                {
                        MESSAGE(DMC_FATAL, LONGBUFPTR_ERR, "illegal characters in longbuf value, file %s, line %d, column %d", fname, linenr, ptr-line+1);
                        return FALSE;
                }
                if (strtolb(ptr, &eptr, lbuf, &n) == FALSE || eptr == ptr)
                {
                        MESSAGE(DMC_FATAL, LONGBUFPTR_ERR, "longbuf value can't be parsed, file %s, line %d, column %d", fname, linenr, eptr-line+1);
                        return FALSE;
                }  
                else
                if (STRLEN(eptr) > 0)
                {
                        MESSAGE(DMC_WARNING, LONGBUFPTR_ERR, "trailing garbage ignored, file %s, line %d, column %d", fname, linenr, eptr-line+1);
                }
                if (ConfigSetValueLongBuf(e, lbuf, n) == FALSE)
                {
                        MESSAGE(DMC_FATAL, CONFIGVALUE_ERR, "couldn't set value", fname, linenr);
                        return FALSE;
                }
                break;
        case CONFIG_TYPE_MIBID:
                if (STRSPN(ptr, "\\/,;") != 0)
                {
                        MESSAGE(DMC_FATAL, MIBIDPTR_ERR, "illegal characters in MIB identifier, file %s, line %d, column %d", fname, linenr, ptr-line+1);
                        return FALSE;
                }
                if (strtoid(ptr, &eptr, lbuf, &n) == FALSE || eptr == ptr)
                {
                        MESSAGE(DMC_FATAL, MIBIDPTR_ERR, "MIB identifier can't be parsed, file %s, line %d, column %d", fname, linenr, eptr-line+1);
                        return FALSE;
                }  
                else
                if (STRLEN(eptr) > 0)
                {
                        MESSAGE(DMC_WARNING, MIBIDPTR_ERR, "trailing garbage ignored, file %s, line %d, column %d", fname, linenr, eptr-line+1);
                }
                if (ConfigSetValueLongBuf(e, lbuf, n) == FALSE)
                {
                        MESSAGE(DMC_FATAL, CONFIGVALUE_ERR, "couldn't set value", fname, linenr);
                        return FALSE;
                }
                break;
        case CONFIG_TYPE_NULL:
        default:                              
                if (STRLEN(ptr) > 0)
                {
                        MESSAGE(DMC_WARNING, NULLPTR_ERR, "trailing garbage ignored, file %s, line %d, column %d", fname, linenr, ptr-line+1);
                }
                break;
        }         
        
        return TRUE;
}


VOID RemoveComment(CHAR *str)
{
CHAR *p = str;

        while ((p = strchr(p, '#')) != NULL)
            if (p > str && p[-1] == '\\')
                strcpy(p-1, p);
            else
            {
                p[0] = '\0';
                break;
            }
}


VOID RemoveBlanks(CHAR *str)
{
INT i1, i2;

        for (i1 = 0; i1 < STRLEN(str) && isspace(str[i1]); i1++)
                ;

        for (i2 = STRLEN(str)-1; i2 >= 0 && isspace(str[i2]); i2--)
                ;

        if (i2 < i1)
                i1 = i2+1;
                
        strncpy(str, str+i1, i2-i1+1);
        str[i2-i1+1] = '\0';
}


VOID Usage(VOID)
{
        MESSAGE(DMC_MESSAGE, USAGE_ERR, "usage: cfg2ini filename ...");
}


BOOLEAN strtot(CHAR* ptr, CHAR** eptr, BOOLEAN* b, INT* n)
{
LONG lval;

    if (strcasecmp(ptr, "TRUE") == 0)
    {   
        *eptr = *eptr + (*n = strlen("TRUE"));
        *b = TRUE;
        return TRUE;
    }
    else
    if (strcasecmp(ptr, "YES") == 0)
    {   
        *eptr = *eptr + (*n = strlen("YES"));
        *b = TRUE;
        return TRUE;
    }
    else
    if (strcasecmp(ptr, "UP") == 0)
    {   
        *eptr = *eptr + (*n = strlen("UP"));
        *b = TRUE;
        return TRUE;
    }
    else
    if (strcasecmp(ptr, "FALSE") == 0)
    {   
        *eptr = *eptr + (*n = strlen("FALSE"));
        *b = FALSE;
        return TRUE;
    }
    else
    if (strcasecmp(ptr, "NO") == 0)
    {   
        *eptr = *eptr + (*n = strlen("NO"));
        *b = FALSE;
        return TRUE;
    }
    else
    if (strcasecmp(ptr, "DOWN") == 0)
    {   
        *eptr = *eptr + (*n = strlen("DOWN"));
        *b = FALSE;
        return TRUE;
    }
    else
    if ((lval = strtol(ptr, eptr, 0)) > 0)
    {   
        *n = *eptr - ptr;
        *b = TRUE;
        return TRUE;
    }
    else
    if (lval == 0 && ptr != *eptr)
    {
        *n = *eptr - ptr;
        *b = FALSE;
        return TRUE;
    }
    
    return FALSE;
}

                    
BOOLEAN strtob(CHAR* ptr, CHAR** eptr, BYTE* buf, INT* n)
{         
CHAR* p;
LONG lval;

        *n = 0;
        p =ptr;
 
        if (p[0] == '\"' && p[strlen(p)-1] == '\"')
        {
            *(p+STRLEN(p)-1) = '\0';
            p++;

            while (*p != '\0')
            {
                buf[(*n)++] = *p++;
                if (*p != '\0' && *n == MAXBYTES)
                    return FALSE;
            }
            *eptr = p;
        }
        else
        {
            p = strtok(ptr, ":");
            while (p != NULL)
            {
                    if ((lval = strtol(p, eptr, 0)) < 0 || lval > UCHAR_MAX || STRLEN(*eptr) > 0)
                            return FALSE;
                    buf[(*n)++] = (BYTE)lval;
                    p = strtok(NULL, ":");
                    if (p != NULL && *n == MAXBYTES)
                            return FALSE;
            }                                
        }
        return TRUE;
}


BOOLEAN strtolb(CHAR* ptr, CHAR** eptr, LONG* buf, INT* n)
{
CHAR* p;
LONG lval;

        *n = 0;
        
        p = strtok(ptr, ".");
        while (p != NULL)
        {
                lval = strtol(p, eptr, 0);
                if (STRLEN(*eptr) > 0)
                        return FALSE;
                buf[(*n)++] = lval;
                p = strtok(NULL, ".");
                if (p != NULL && *n == MAXLONGS)
                        return FALSE;
        }                                
        return TRUE;
}


BOOLEAN strtoip(CHAR* ptr, CHAR** eptr, IPADDR* buf, INT *n)
{
CHAR* p;
LONG lval;

        *n = 0;
        *buf = 0;
        
        p = strtok(ptr, ".");
        while (p != NULL)
        {
                lval = strtol(p, eptr, 0);
                if ((lval = strtol(p, eptr, 0)) < 0 || lval > UCHAR_MAX || STRLEN(*eptr) > 0)
                        return FALSE;
                *buf = (*buf << 8) | (BYTE)lval;
                (*n)++;
                p = strtok(NULL, ".");
        }
        if (*n != MAXIPADDR)
                return FALSE;
        return TRUE;
}


BOOLEAN strtoid(CHAR* ptr, CHAR** eptr, LONG* buf, INT* n)
{
CHAR* p;
INT n2;
LONG lval;

        *n = 0;
        
        p = strtok(ptr, ".");
        while (p != NULL)
        {
                if (isdigit(p[0]) && STRSPN(p, "x0123456789abcdefABCDEF") == STRLEN(p))
                {
                        lval = strtol(p, eptr, 0);
                        if (*eptr == p)
                                return FALSE;
                        buf[(*n)++] = lval;
                }
                else
                if (strtohost(p, eptr, buf+*n, &n2) == TRUE)
                {
                        *n += n2;
                }
                else
                if ((n2 = MibName2ID(buf+*n, p)) >= 0)
                {
                        *n += n2;
                } 
                else
                        return FALSE;
                p = strtok(NULL, ".");
                if (p != NULL && *n >= MAXLONGS-20)
                        return FALSE;
        }                                
        return TRUE;
}


BOOLEAN strtohost(CHAR* ptr, CHAR** eptr, LONG* buf, INT* n)
{
INT i, j;

    *n = 0;

    for (i = 0; i < sizeof(hosts)/sizeof(Host); i++)
        if (strcmp(ptr, hosts[i].name) == 0)
        {
            for (j = 0; j < HOSTADDRSIZE; j++)
                buf[j] = hosts[i].address[j];
            *n = HOSTADDRSIZE;
            return TRUE;
        }
    return FALSE;
}
