#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <dnpap.h>
#include <snmp.h>
#include "prototyp.h"
#include "dp.h"

#define REC_FRAME       0xF8        /* Marks start of a frame */
#define REC_TIMESTAMP   0xFB
#define REC_HEADER          0xfe            /* Dump file record types */
#define REC_VERSION         0xfd
#define REC_HDRINFO         0xfc
#define FR_DUMP_HDRSIZE     (1+sizeof(unsigned int)) /* ID + len */
#define MAXFRAMES          1000

static char buff[16*1024];
static long fpos[MAXFRAMES];
FILE *dumpfile;
int nframes;

struct rec_hdr {
    BYTE  type;
    short len;
};

/* Compare DP.H _DPBUF for the first 6 items */
static struct frame {
    short Dev;                      /* Logic dev# [0..DPMAXDEV-1]  */
    LWORD ClockMs;                  /* Timestamp in micro seconds  */
    WORD Status;                    /* Status bits - internal use  */
    WORD Size;                      /* Packet length in bytes      */
    BYTE *pBuf;                     /* Pointer to packet's content */
    struct frame *pNext;            /* Pointer - internal use      */
    BYTE Data[1600];                /* Packet data contents        */
};

int ReadByteFromFile (FILE *file, BYTE *b)
{
    return ((fread(b, 1, 1, file) == 1) ? 0 : EOF);
}

int ReadIntFromFile (FILE *file, short *i)
{
    BYTE t1, t2;

    if (ReadByteFromFile(file, &t1) == EOF)
        return (EOF);
    if (ReadByteFromFile(file, &t2) == EOF)
        return (EOF);
    *i = (t2 << 8) | t1;
    return (0);
}

int ReadLongFromFile (FILE *file, long *l)
{
    BYTE t1, t2, t3, t4;

    if (ReadByteFromFile(file, &t1) == EOF)
        return (EOF);
    if (ReadByteFromFile(file, &t2) == EOF)
        return (EOF);
    if (ReadByteFromFile(file, &t3) == EOF)
        return (EOF);
    if (ReadByteFromFile(file, &t4) == EOF)
        return (EOF);
    *l = (t4 << 24) | (t3 << 16) | (t2 << 8) | t1;
    return (0);
}

/*****************************************************************
** NAME:        ReadFile
** SYNOPSIS:    void ReadFile(void)
** PARAMETERS:  - none
** DESCRIPTION: - reads the pktcapt.dmp file and gets the number of
**                the frames
** REMARKS:     
** RETURNS:     no value
*******************************************************************/

void ReadFile(void)
{
    char TimeStamp[20];

    DpuDumpInit();
    nframes = DpuDumpCount(fpos ,MAXFRAMES, TimeStamp);
}

/*****************************************************************
** NAME:        ReadPacket
** SYNOPSIS:    BOOLEAN ReadPacket(PROT_PKT *Pkt, int i)
** PARAMETERS:  - Pkt - pointer to the packet
**              - i - the number of the packet in the file
** DESCRIPTION: - reads the packet number i from the file
** REMARKS:     
** RETURNS:     FALSE - if there is attempt to read packet with
**                      higher number than is the whole number
**                      of the packets in the file
**              TRUE  - otherwise
*******************************************************************/

BOOLEAN  ReadPacket(PROT_PKT *Pkt, int i)
{
    struct frame buf;
    WORD Ether = MAC_TYPE_ETHERNET_CSMACD;
    static LONG ID = 0;
    BYTE Packet[1700];

    if(i >= nframes)
     return(FALSE);

    DpuDumpSeek(fpos[i]);
    DpuDumpReadFrame(&buf,buff);
    Packet[0] = ((BYTE*) &Ether)[0];
    Packet[1] = ((BYTE*) &Ether)[1];
    Packet[2] = ((BYTE*) &Ether)[0];
    Packet[3] = ((BYTE*) &Ether)[1];
    Packet[4] = ((BYTE*) &ID)[0];
    Packet[5] = ((BYTE*) &ID)[1];
    Packet[6] = ((BYTE*) &ID)[2];
    Packet[7] = ((BYTE*) &ID)[3];
    Packet[8] = ((BYTE*) &buf.Size)[0];
    Packet[9] = ((BYTE*) &buf.Size)[1];
    Packet[10] = ((BYTE*) &buf.ClockMs)[0];
    Packet[11] = ((BYTE*) &buf.ClockMs)[1];
    Packet[12] = ((BYTE*) &buf.ClockMs)[2];
    Packet[13] = ((BYTE*) &buf.ClockMs)[3];
    memcpy(&Packet[14],buf.pBuf, 1600);
    ProtFile (Pkt, Packet);
    PacketSize = Pkt->Frame->Beholder.Len;
    ID++;
    return(TRUE);
}

/*****************************************************************
** NAME:        OpenFile
** SYNOPSIS:    FILE *OpenFile(void)
** PARAMETERS:  - none
** DESCRIPTION: - opens the pktcapt.dmp file
** REMARKS:     
** RETURNS:     NULL - opening not successful
**              - otherwise - pointer to the opened file
*******************************************************************/

FILE *OpenFile(void)
{
 FILE *dumpfile;

 return(dumpfile = fopen("pktcapt.dmp", "r"));
}

/*****************************************************************
** NAME:        ConvertToString
** SYNOPSIS:    BOOLEAN ConvertToString(PROT_PKT *Pkt, BYTE **String, int flag)
** PARAMETERS:  - Pkt - pointer to the packet
**              - String - returning string
**              - flag - which beholder field will be converted         
** DESCRIPTION: - converts a beholder field number 'flag' to the string
** REMARKS:     
** RETURNS:     TRUE - field was found
**              FALSE - otherwise 
*******************************************************************/

BOOLEAN ConvertToString (PROT_PKT *Pkt, BYTE **String, int flag)
{
    unsigned int i;

 
    switch (flag)
    {
        case 1:                                 /* time */
            sprintf(protocolString, "%u:%03u:%03u",
                (WORD) ((Pkt->Frame->Beholder.Time / 1000000L) % 1000),
                (WORD) ((Pkt->Frame->Beholder.Time / 1000L) % 1000),
                (WORD) (Pkt->Frame->Beholder.Time % 1000));
                *String = protocolString;
            return (TRUE);
        case 2:                                 /* ID */
            sprintf(protocolString, "%d", Pkt->Frame->Beholder.ID);
                *String = protocolString;
            return (TRUE);
        case 3:                                 /* len */
            sprintf(protocolString, "%d", Pkt->Frame->Beholder.Len);
                *String = protocolString;
            return (TRUE);
        case 5:                                 /* index */
            sprintf(protocolString, "%d", Pkt->Frame->Beholder.IfIndex);
                *String = protocolString;
            return (TRUE);
        case 4:
            for(i = 0;i < 3;i++)
            {
              if(i == Pkt->ChildProt)
               *String = ProtPtr[i].Name;
            }                   
            return(TRUE);
        case 6:
            Pkt->DataLen = PacketSize;
            ProtPtr[Pkt->ChildProt].Header(Pkt);
            *String = ProtPtr[Pkt->ChildProt].Name;
            return(TRUE);
        default:
            return (FALSE);
    }
}

/*****************************************************************
** NAME:        GetData
** SYNOPSIS:    BOOLEAN GetData(PROT_PKT *Pkt, int frame)
** PARAMETERS:  - Pkt - pointer to the packet
**              - frame - number of the frame in the file         
** DESCRIPTION: - reads a frame number 'frame'
** REMARKS:     
** RETURNS:     no value
*******************************************************************/

VOID  GetData(PROT_PKT *Pkt, int frame)
{
    struct frame buf;
    WORD Ether = 0;

    DpuDumpSeek(fpos[frame]);
    DpuDumpReadFrame(&buf,buff);
    Pkt->Ptr = buf.pBuf;
}

/*****************************************************************
**              DPU functions for reading of the file 
*******************************************************************/

int DpuDumpInit(void)
{
    setvbuf(dumpfile, buff, _IOFBF, 16*1024);
    return 0;
}

unsigned int DpuDumpCount(long fpos[],unsigned int  maxframes, char *TimeStamp)
{
    unsigned int nframes = 0;
    long fptr = 0L;
    unsigned int len;
    int rec;

    if (dumpfile)
    {
        fseek(dumpfile, fptr, SEEK_SET);
        while (((rec = fgetc(dumpfile)) != EOF) && (nframes < maxframes))
        {
            fread(&len, sizeof(len), 1, dumpfile);
            switch (rec)
            {
/*            case REC_HDRINFO:
                fread(hdrinfo, sizeof(DPHDRINFO), 1, dumpfh);
                break; */
            case REC_TIMESTAMP:
                fread(TimeStamp, sizeof(char), len-FR_DUMP_HDRSIZE, dumpfile);
                break;
            case REC_FRAME:
                if (fpos)
                    fpos[nframes++] = fptr;
                break;
            }
            fptr += len;
            fseek(dumpfile, fptr, SEEK_SET);
        }
    }
    return nframes;
}

void DpuDumpSeek(long fpos)
{
    fseek(dumpfile, fpos+FR_DUMP_HDRSIZE, SEEK_SET);
}

/************************************************************************
** NAME:        DpuDumpReadFrame
** SYNOPSIS:    void DpuDumpReadFrame(frp, buf)
**              DPBUF *frp          Pkt description struct
**              BYTE *buf           Pkt data buffer
** DESCRIPTION: Reads a packet from dumpfile.
** RETURNS:     void
** SEE ALSO:    DpuDumpInit, DumpEnd, DpuDumpWriteFrame
*************************************************************************/
void DpuDumpReadFrame(struct frame *buf,BYTE *buff)
{
    if (dumpfile)
    {
        fread(buf, sizeof(DPBUF), 1, dumpfile);
        buf->pBuf = buff;
        fread(buff, sizeof(BYTE), buf->Size, dumpfile);
        PacketSize = buf->Size;
    }
}



