/*
 * printer.c - manipulating printer informations
 */

#include <stdio.h>
#include <stdlib.h>

#include "nup.h"

/*
 * Use as Buffer for Printer List
 */
 
static  PPRQINFO3   bufPrtInfo = NULL ;
static  ULONG       numPrtInfo = 0    ;

/*
 * printerListup - listup printers
 */

static  PPRQINFO3   printerListup(PULONG nPrinters)
{
    SPLERR      stat ;
    PPRQINFO3   base ;
    ULONG       leng, need, total, saved ;
    
    *nPrinters = 0 ;    /* as default */
    
    /*
     * already listed, use them
     */
     
    if (bufPrtInfo != NULL) {
        *nPrinters = numPrtInfo ;
	return bufPrtInfo ;
    }
    
    /*
     * query number of printer queues
     */

    stat = SplEnumQueue(NULL, 3, NULL, 0, &saved, &total, &need, NULL) ;

    if (stat != ERROR_MORE_DATA) {
        return NULL ;
    }
    
    /*
     * prepare buffer for printer list
     */
    
    if ((base = (PPRQINFO3) malloc(leng = need)) == NULL) {
        return NULL ;
    }
    
    /*
     * query queue list
     */

    stat = SplEnumQueue(NULL, 3, base, leng, &saved, &total, &need, NULL) ;
    
    if (stat != NO_ERROR) {
        free(base)  ;
        return NULL ;
    }

    /*
     * return printer list (array of PPRQINFO3)
     */
    
    bufPrtInfo = base  ;
    numPrtInfo = total ;

    *nPrinters = total ;
    return base ;
}

/*
 * printerDefault - query default printer
 */

PPRQINFO3   printerDefault(void)
{
    PPRQINFO3   base, pprq ;
    ULONG       nprt, i ;
    
    if ((base = printerListup(&nprt)) == NULL) {
        return NULL ;
    }

    /*
     * search default printer
     */
     
    for (i = 0, pprq = base ; i < nprt ; i++, pprq++) {
        if (pprq->fsType & PRQ3_TYPE_APPDEFAULT) {
	    return pprq ;
	}
    }
    return NULL ;
}

/*
 * printerRefer - refer to named printer
 */

PPRQINFO3   printerRefer(PSZ name)
{
    PPRQINFO3   base, pprq ;
    ULONG       nprt, i ;
    
    if ((base = printerListup(&nprt)) == NULL) {
        return NULL ;
    }

    for (i = 0, pprq = base ; i < nprt ; i++, pprq++) {
        if (stricmp(name, pprq->pszName) == 0) {
	    return pprq ;
	}
    }
    return NULL ;
}

/*
 * printerFillup - fill listbox with printer (queue) names
 */

void    printerFillup(HWND hwndLbox)
{
    PPRQINFO3   base, pprq ;
    ULONG       i, nprt    ;
    
    if ((base = printerListup(&nprt)) == NULL) {
        return ;
    }
    WinSendMsg(hwndLbox, LM_DELETEALL, NULL, NULL) ;

    for (i = 0, pprq = base ; i < nprt ; i++, pprq++) {
        WinSendMsg(hwndLbox, LM_INSERTITEM, 
	    MPFROM2SHORT(LIT_SORTASCENDING, 0), MPFROMP(pprq->pszName)) ;
    }
}

/*
 * printerSelect - mark selection in printer list
 */

void    printerSelect(HWND hwndLbox, PSZ name)
{
    LONG    idx ;

    idx = (LONG) WinSendMsg(hwndLbox, LM_SEARCHSTRING, 
                MPFROM2SHORT(0, LIT_FIRST), MPFROMP(name)) ;
    if (idx != LIT_NONE) {
        WinSendMsg(hwndLbox, LM_SELECTITEM, 
	            MPFROMLONG(idx), MPFROMSHORT(TRUE)) ;
    }
}

/*
 * printerProp - set Job Property for Pinter
 */

int     printerProp(HAB hab, PPRQINFO3 prt)
{
    int     i, len, stat ;
    UCHAR   szDriverName[128]  ;
    UCHAR   szDeviceName[128]  ;
    UCHAR   szPrinterName[128] ;
    PUCHAR  sp, dp ;
    
    /*
     * extract (single) printer name from 'pprq->pszPrinters'
     */

    sp = prt->pszPrinters ;
    dp = szPrinterName    ;
    for (i = 0 ; i < 128 ; i++) {
        if (*sp == '\0' || *sp == '.') {
	    break ;
	}
	*dp++ = *sp++ ;
    }
    *dp =  '\0' ;
    
    /*
     * separate Driver.Device from 'pprq->pszDriverName'
     */

    sp = prt->pszDriverName ;
    dp = szDriverName ;
    for (i = 0 ; i < 128 ; i++) {
        if (*sp == '\0' || *sp == '.') {
	    break ;
	}
	*dp++ = *sp++ ;
    }
    *dp = '\0' ;
    
    if (*sp == '.') sp++ ;
    dp = szDeviceName    ;
    for(i = 0 ; i < 128 ; i++) {
        if (*sp == '\0') {
	    break ;
	}
	*dp++ = *sp++ ;
    }
    *dp = '\0' ;
    
    /*
     * query length of Driver Data
     */
     
    len = DevPostDeviceModes(
            hab,
	    (PDRIVDATA) NULL, 
	    szDriverName,
	    szDeviceName,
	    szPrinterName,
	    DPDM_POSTJOBPROP) ;
    if (len <= 0) {
        return -2 ;
    }
    if (len > prt->pDriverData->cb) {
        return -3 ;
    }
    
    /*
     * set job property (via dialog)
     */

    stat = DevPostDeviceModes(
            hab,
	    prt->pDriverData,
            szDriverName,
            szDeviceName,
            szPrinterName,
            DPDM_POSTJOBPROP) ;

    if (stat < 0) {
        return -4 ;
    } else {
        return 0 ;
    }
}
