/*
 * setup.c - setup 'nup' print job
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "nup.h"
#include "resource.h"

/*
 * controlButton - enable/disable buttons
 *  
 *  will enabled if
 *      Property  - printer was selected
 *      Next/Prev - always
 *      Edit      - always
 *      Append    - always
 *      Delete    - file was selected
 *      Print     - printer, file exist
 *      Exit      - always
 */

static  void    controlButton(HWND hwnd)
{
    BOOL    flag ;
    LONG    idx  ;
    
    /*
     * enable 'property' if printer was selected
     */

    flag = (thePrt != NULL) ? TRUE : FALSE ;
    WinEnableWindow(WinWindowFromID(hwnd, IDS_PROPERTY), flag) ;

    /*
     * enable 'delete' if file was selected
     */

    idx = (LONG) WinSendMsg(WinWindowFromID(hwnd, IDS_DOCS),
                    LM_QUERYSELECTION, MPFROMP(LIT_FIRST), NULL) ;
    flag = (idx == LIT_NONE) ? FALSE : TRUE ;
    WinEnableWindow(WinWindowFromID(hwnd, IDS_DELETE), flag) ;
    
    /*
     * enable 'print' if printer, document exist
     */
    
    flag = TRUE ;
    if (thePrt == NULL)    flag = FALSE ;
    if (docEmpty(&theDoc)) flag = FALSE ;
    WinEnableWindow(WinWindowFromID(hwnd, IDS_PRINT), flag) ;
}

/*
 * fillup Printer, Document list
 */

static  void    fillPrinter(HWND hwndDlg)
{
    HWND    hwnd = WinWindowFromID(hwndDlg, IDS_PRINTER) ;
    
    printerFillup(hwnd) ;
    
    if (thePrt != NULL) {
        printerSelect(hwnd, thePrt->pszName) ;
    }
}

static  void    fillDocument(HWND hwndDlg)
{
    DOCPTR  pDoc ;
    HWND    hwnd = WinWindowFromID(hwndDlg, IDS_DOCS) ;
    
    WinSendMsg(hwnd, LM_DELETEALL, NULL, NULL) ;
    
    pDoc = docFirst(&theDoc) ;
    
    while (pDoc != NULL) {
        WinSendMsg(hwnd, LM_INSERTITEM,
	        MPFROM2SHORT(LIT_END, 0), MPFROMP(pDoc->title)) ;
        pDoc = docNext(&theDoc, pDoc) ;
    }
}

/*
 * manages Icons used in this dialog
 */
 
typedef struct _ICONMAP {
    int         id   ;
    HPOINTER    icon ;
} ICONMAP ;

static  ICONMAP    icons[] = {
    { IDI_PORT1, NULLHANDLE } ,
    { IDI_PORT2, NULLHANDLE } ,
    { IDI_PORT4, NULLHANDLE } ,
    { IDI_LAND1, NULLHANDLE } ,
    { IDI_LAND2, NULLHANDLE } ,
    { IDI_LAND4, NULLHANDLE } ,
    { 0,         NULLHANDLE }
} ;

static  HPOINTER    getIcon(int id)
{
    ICONMAP *p ;

    for (p = icons ; p->id != 0 ; p++) {
        if (p->id != id) {
	    continue ;
	}
	if (p->icon == NULLHANDLE) {
	    p->icon = WinLoadPointer(HWND_DESKTOP, NULLHANDLE, p->id) ;
	}
	return p->icon ;
    }
    return NULLHANDLE ;
}

/*
 * manages layout selection
 */

static  int     curLayout ;

static  int     layoutIcons[] = {
    IDI_PORT1, IDI_PORT2, IDI_PORT4, IDI_LAND1, IDI_LAND2, IDI_LAND4 
} ;

static  void    showLayout(HWND hwnd)
{
    WinSendDlgItemMsg(hwnd, IDS_LAYOUT, 
            SM_SETHANDLE, MPFROMP(getIcon(layoutIcons[curLayout])), NULL) ;
}

/*
 * refer to selected printer
 */

static  UCHAR   prtname[128] ;

static  PPRQINFO3   getPrinter(HWND hwnd)
{
    WinQueryWindowText(WinWindowFromID(hwnd, IDS_PRINTER), 128, prtname) ;
    return printerRefer(prtname) ;
}

/*
 * refer to selected layout
 */
 
static  LAYPTR  getLayout(HWND hwnd)
{
    return layoutInfo(curLayout) ;
}

/*
 * append/delete file to/from document list
 */

static  UCHAR   openDriv[4]          = "" ;
static  UCHAR   openPath[CCHMAXPATH] = "" ;
static  UCHAR   openBase[CCHMAXPATH] = "" ;
static  UCHAR   openExtn[CCHMAXPATH] = "" ;

MRESULT EXPENTRY    openDlgProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    if (msg == WM_INITDLG) {
        dialogAtMouse(hwnd) ;
    }
    return WinDefFileDlgProc(hwnd, msg, mp1, mp2) ;
}

static  void    appendFile(HWND hwndDlg)
{
    HWND    hwndOpen ;
    FILEDLG openDlg  ;
    DOCPTR  pDoc     ;

    memset(&openDlg, 0, sizeof(openDlg)) ;

    openDlg.cbSize = sizeof(openDlg) ;
    openDlg.pszTitle = "t@C" ;
    openDlg.pszOKButton = "ǉ" ;
    openDlg.fl = FDS_OPEN_DIALOG | FDS_PRELOAD_VOLINFO ;
    openDlg.pfnDlgProc = openDlgProc ;

    if (strlen(openDriv) > 0) {
        openDlg.pszIDrive = openDriv ;
    }
    if (strlen(openPath) > 0) {
        sprintf(openDlg.szFullFile, "%s*.*", openPath) ;
    } else {
        sprintf(openDlg.szFullFile, "*.*") ;
    }

    hwndOpen = WinFileDlg(HWND_DESKTOP, hwndDlg, &openDlg) ;

    if (hwndOpen == NULLHANDLE) {
        return ;
    }
    if (openDlg.lReturn != DID_OK) {
        return ;
    }
    _splitpath(openDlg.szFullFile, openDriv, openPath, openBase, openExtn) ;

    if ((pDoc = docFromFile(openDlg.szFullFile)) == NULL) {
        return ;
    }
    docAppend(&theDoc, pDoc) ;
    fillDocument(hwndDlg)  ;
    controlButton(hwndDlg) ;
}

static  void    deleteFile(HWND hwndDlg)
{
    LONG    idx, i ;
    DOCPTR  pDoc   ;

    idx = (LONG) WinSendMsg(WinWindowFromID(hwndDlg, IDS_DOCS),
                        LM_QUERYSELECTION, MPFROMSHORT(LIT_FIRST), NULL) ;
    if (idx == LIT_NONE) {
        return ;
    }

    pDoc = docFirst(&theDoc) ;
    for (i = 0 ; i < idx ; i++) {
        pDoc = docNext(&theDoc, pDoc) ;
    }
    if (pDoc != NULL) {
        docDelete(&theDoc, pDoc) ;
        docDispose(pDoc) ;
    }
    fillDocument(hwndDlg)  ;
    controlButton(hwndDlg) ;
}

/*
 * flistProc - subclassed ListBox with Drag & Drop
 */

static  PFNWP   pfnFlist ;

MRESULT EXPENTRY flistProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    PDRAGINFO   pDragInfo ;
    PDRAGITEM   pDragItem ;
    int         i  ;
    SHORT       op ;
    DOCPTR      pDoc ;
    HWND        hwndDlg ;

static  UCHAR   path[256] ;
static  UCHAR   base[256] ;
static  UCHAR   name[512] ;
    
    switch (msg) {

    case DM_DRAGOVER :
        pDragInfo = (PDRAGINFO) mp1  ;
	DrgAccessDraginfo(pDragInfo) ;
	for (i = 0, op = DOR_NEVERDROP ; i < pDragInfo->cditem ; i++) {
	    pDragItem = DrgQueryDragitemPtr(pDragInfo, i) ;
	    if (DrgVerifyType(pDragItem, DRT_TEXT)) {
	        op = DOR_DROP ;
	    }
	}
        return (MPFROM2SHORT(op, DO_UNKNOWN)) ;

    case DM_DROP :
        pDragInfo = (PDRAGINFO) mp1  ;
	DrgAccessDraginfo(pDragInfo) ;
	for (i = 0, op = DOR_NEVERDROP ; i < pDragInfo->cditem ; i++) {
	    pDragItem = DrgQueryDragitemPtr(pDragInfo, i) ;
	    if (! DrgVerifyType(pDragItem, DRT_TEXT)) {
                continue ;
	    }
	    DrgQueryStrName(pDragItem->hstrContainerName, 256, path) ;
	    DrgQueryStrName(pDragItem->hstrSourceName, 256, base)    ;
	    strcpy(name, path) ;
	    strcat(name, base) ;
	    if ((pDoc = docFromFile(name)) != NULL) {
	        docAppend(&theDoc, pDoc) ;
	    }
	}
	hwndDlg = WinQueryWindow(hwnd, QW_PARENT) ;
	fillDocument(hwndDlg)  ;
	controlButton(hwndDlg) ;
        return (MRESULT) 0 ;

    default :
        return (*pfnFlist) (hwnd, msg, mp1, mp2) ;
    }
}

/*
 * setupProc - dialog procedure for setup dialog
 */

static  UCHAR   titleSetup[64] ;
static  UCHAR   titlePrint[64] ;
static  UCHAR   titleSuspn[64] ;
static  UCHAR   titleDone[64]  ;
static  BOOL    printImmediate ;

MRESULT EXPENTRY setupProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    LAYPTR      pLay ;
    PPRQINFO3   pPrt ;
    
    switch (msg) {
    
    case WM_INITDLG :
        sprintf(titleSetup, "%s - ݒ",     ProgramName) ;
        sprintf(titlePrint, "%s - ",   ProgramName) ;
        sprintf(titleSuspn, "%s - f", ProgramName) ;
        sprintf(titleDone,  "%s - I", ProgramName) ;

	WinSendMsg(hwnd, WM_SETICON,
	    MPFROMP(WinLoadPointer(HWND_DESKTOP, NULLHANDLE, ID_NUP)), NULL) ;
        WinSetWindowText(hwnd, titleSetup) ;
	dialogAtMouse(hwnd) ;
	
        pfnFlist = WinSubclassWindow(WinWindowFromID(hwnd, IDS_DOCS), flistProc) ;

	fillPrinter(hwnd)   ;
	fillDocument(hwnd)  ;
	curLayout = layoutCur() ;
	showLayout(hwnd)    ;
	controlButton(hwnd) ;
	
        if (printImmediate) {
	    WinPostMsg(hwnd, WM_COMMAND, MPFROM2SHORT(IDS_PRINT, 0), NULL) ;
        }
        return (MRESULT) 0 ;

    case WM_CLOSE :
        WinSendDlgItemMsg(hwnd, IDS_LAYOUT, SM_SETHANDLE, NULL, NULL) ;
        WinPostMsg(0, WM_QUIT, NULL, NULL) ;
	return (MRESULT) 0 ;        

    case WM_COMMAND :
        switch (SHORT1FROMMP(mp1)) {

        case IDS_EXIT :
	    WinPostMsg(hwnd, WM_CLOSE, NULL, NULL) ;
	    break ;

	case IDS_PRINT :
	    if ((pPrt = getPrinter(hwnd)) != NULL) {
	        thePrt = pPrt ;
	    }
	    if ((pLay = getLayout(hwnd)) != NULL) {
	        theLay = pLay ;
	    }
	    if (thePrt != NULL && theLay != NULL) {
                printDocument(hwnd) ;
            }
	    break ;
	    
        case IDS_PROPERTY :
	    if ((pPrt = getPrinter(hwnd)) != NULL) {
	        printerProp(habMain, pPrt) ;
	    }
            WinSetWindowText(hwnd, titleSetup) ;
	    break ;
	    
	case IDS_EDIT :
            editLayout(hwnd, curLayout) ;
            WinSetWindowText(hwnd, titleSetup) ;
	    break ;
	    
        case IDS_LPREV :
	    curLayout = layoutPrev(curLayout) ;
	    showLayout(hwnd) ;
            WinSetWindowText(hwnd, titleSetup) ;
	    break ;
	case IDS_LNEXT :
	    curLayout = layoutNext(curLayout) ;
	    showLayout(hwnd) ;
            WinSetWindowText(hwnd, titleSetup) ;
	    break ;
	    
	case IDS_APPEND :
	    appendFile(hwnd) ;
            WinSetWindowText(hwnd, titleSetup) ;
	    break ;
	case IDS_DELETE :
	    deleteFile(hwnd) ;
            WinSetWindowText(hwnd, titleSetup) ;
	    break ;
        }
	return (MRESULT) 0 ;

    case WM_CONTROL :
        switch (SHORT1FROMMP(mp1)) {
	case IDS_PRINTER :
	    if (SHORT2FROMMP(mp1) == CBN_EFCHANGE) {
	        controlButton(hwnd) ;
	    }
            WinSetWindowText(hwnd, titleSetup) ;
	    break ;
        case IDS_DOCS :
	    if (SHORT2FROMMP(mp1) == LN_SELECT) {
	        controlButton(hwnd) ;
	    }
            WinSetWindowText(hwnd, titleSetup) ;
	    break ;
        }
	return (MRESULT) 0 ;

    case WM_USER_PRINT_FILE :
        WinSetWindowText(hwnd, titlePrint) ;
	return (MRESULT) 0 ;
	
    case WM_USER_PRINT_SUSP :
        WinSetWindowText(hwnd, titleSuspn) ;
	return (MRESULT) 0 ;

    case WM_USER_PRINT_DONE :
        WinSetWindowText(hwnd, titleDone) ;
	if (printImmediate) {
       	    WinPostMsg(hwnd, WM_CLOSE, NULL, NULL) ;
	}
        return (MRESULT) 0 ;
    }
    return WinDefDlgProc(hwnd, msg, mp1, mp2) ;
}

/*
 * setupPrint - print setup & print
 */

void    setupPrint(BOOL immediate)
{
    printImmediate = immediate ;
    
    WinDlgBox(HWND_DESKTOP, HWND_DESKTOP, 
                        setupProc, NULLHANDLE, ID_SETUP, NULL) ;
}

