// Advanced listbox class

PCOLUMN         getColumnFromX (PLISTBOXDATA plbd, int x)
{
   int i;
   PCOLUMN pc;

   pc = plbd->firstColumn;

   for (i = 0; i < x && pc; i ++)
   {
      pc = pc->nextColumn;
   }

return pc;
}

PITEMPACK       getItemFromY (PCOLUMN pc, int y)
{
   int i;
   PITEMPACK pip;

   pip = pc->firstItem;

   for (i = 0; i < y && pip; i++)
   {
      pip = pip->nextItem;
   }

return pip;
}

PITEMPACK       getItemFromXY (PLISTBOXDATA plbd, int x, int y)
{
   PITEMPACK pip;
   PCOLUMN pc;

   pc = getColumnFromX (plbd, x);

   if (!pc) return NULL;

   pip = getItemFromY (pc, y);

return pip;
}

void    queryTextBox (HPS hps, PSZ pszText, PSIZEL pszl)
{
   POINTL txtPointl[TXTBOX_COUNT];

   GpiQueryTextBox (hps, strlen(pszText), pszText, TXTBOX_COUNT,
                   (PPOINTL)&txtPointl[0]);

   pszl->cx = txtPointl[TXTBOX_TOPRIGHT].x - txtPointl[TXTBOX_TOPLEFT].x + 2;

   pszl->cy = txtPointl[TXTBOX_TOPLEFT].y - txtPointl[TXTBOX_BOTTOMLEFT].y + 2;
}

void    queryBitmapSize (HPS hps, HBITMAP hbm, PSIZEL pszl)
{
   BITMAPINFOHEADER bmp;

   bmp.cbFix = sizeof (BITMAPINFOHEADER);
   GpiQueryBitmapParameters (hbm, &bmp);

   pszl->cx = bmp.cx + 2;
   pszl->cy = bmp.cy + 2;
}

int     queryItemSize (HPS hps, PITEMPACK pip, PSIZEL pszl)
{

   switch (pip->item.itemType & ITEMTYPE_MASK)
   {
      case ITEMTYPE_TEXT:
      queryTextBox (hps, pip->item.text, pszl);
      break;

      case ITEMTYPE_ICON:
      pszl->cy = WinQuerySysValue (HWND_DESKTOP, SV_CYICON) + 2;
      pszl->cx = WinQuerySysValue (HWND_DESKTOP, SV_CXICON) + 2;
      break;

      case ITEMTYPE_MINIICON:
      pszl->cy = WinQuerySysValue (HWND_DESKTOP, SV_CYICON) / 2 + 2;
      pszl->cx = WinQuerySysValue (HWND_DESKTOP, SV_CXICON) / 2 + 2;
      break;

      case ITEMTYPE_BITMAP:
      queryBitmapSize (hps, pip->item.imageHandle, pszl);
      break;

      case ITEMTYPE_OBJECT:
      {
         SIZEL szl;

         queryTextBox (hps, pip->item.text, &szl);

         pszl->cy = WinQuerySysValue (HWND_DESKTOP, SV_CYICON) / 2 + 2;
         pszl->cx = WinQuerySysValue (HWND_DESKTOP, SV_CXICON) / 2 + 2;

         if(szl.cy > pszl->cy) pszl->cy = szl.cy;
         pszl->cx += szl.cx + 1; // preserve some space

      break;
      }

      default: // so what?
      pszl->cx = pszl->cy = 0;
   }

return pszl->cx;
}

int     convertTextAttr (PITEM pi)
{
   if (pi->itemAlign == ITEMALIGN_CENTER) return DT_CENTER;
   if (pi->itemAlign == ITEMALIGN_RIGHT) return DT_RIGHT;

return 0;
}

void    drawItem (HPS hps, PITEMPACK pip, BOOL fSelection, BOOL fFocus)
{
   int i, j, textAttr = convertTextAttr (&pip->item) | DT_VCENTER | DT_ERASERECT;
   long fgColor = fSelection ?
                              (fFocus ? WinQuerySysColor(HWND_DESKTOP,
                                                         SYSCLR_HILITEFOREGROUND, 0)
                                      : WinQuerySysColor(HWND_DESKTOP,
                                                         SYSCLR_WINDOWTEXT, 0))
                             : pip->item.fgColor,
        bgColor = fSelection ?
                              (fFocus ? WinQuerySysColor(HWND_DESKTOP,
                                                         SYSCLR_HILITEBACKGROUND, 0)
                                      : WinQuerySysColor(HWND_DESKTOP,
                                                         SYSCLR_DIALOGBACKGROUND, 0))
                             : pip->item.bgColor;

   if (pip->item.itemFlags & ITEMFLAGS_NOTSELECT)
   {
      fgColor = pip->item.fgColor;
      bgColor = pip->item.bgColor;
   }

   switch (pip->item.itemType & ITEMTYPE_MASK)
   {
      case ITEMTYPE_TEXT:
      WinDrawText (hps, -1L, pip->item.text, &pip->rcl,
                   fgColor, bgColor, textAttr);
      break;

      case ITEMTYPE_ICON:
      WinFillRect (hps, &pip->rcl, bgColor);

      j = (pip->rcl.yTop - pip->rcl.yBottom -
           WinQuerySysValue (HWND_DESKTOP, SV_CYICON)) / 2;

      i = (pip->rcl.xRight - pip->rcl.xLeft -
           WinQuerySysValue (HWND_DESKTOP, SV_CXICON)) / 2;

      WinDrawPointer (hps,
                      pip->rcl.xLeft + i,
                      pip->rcl.yBottom + j + 1, pip->item.imageHandle, DP_NORMAL);
      break;

      case ITEMTYPE_MINIICON:
      WinFillRect (hps, &pip->rcl, bgColor);

      j = (pip->rcl.yTop - pip->rcl.yBottom -
           WinQuerySysValue (HWND_DESKTOP, SV_CYICON) / 2) / 2;

      i = (pip->rcl.xRight - pip->rcl.xLeft -
           WinQuerySysValue (HWND_DESKTOP, SV_CXICON) / 2) / 2;

      WinDrawPointer (hps,
                      pip->rcl.xLeft + i,
                      pip->rcl.yBottom + j + 1, pip->item.imageHandle, DP_MINI);

      break;

      case ITEMTYPE_OBJECT:
      {
         RECTL rcl;

         rcl = pip->rcl;
         rcl.xRight = rcl.xLeft + WinQuerySysValue (HWND_DESKTOP, SV_CXICON) / 2;
         rcl.xRight ++;

         WinFillRect (hps, &rcl, bgColor);

         j = (rcl.yTop - rcl.yBottom -
              WinQuerySysValue (HWND_DESKTOP, SV_CYICON) / 2) / 2;

         i = (rcl.xRight - rcl.xLeft -
              WinQuerySysValue (HWND_DESKTOP, SV_CXICON) / 2) / 2;

         WinDrawPointer (hps,
                         pip->rcl.xLeft + i,
                         pip->rcl.yBottom + j + 1, pip->item.imageHandle, DP_MINI);

         rcl = pip->rcl;
         rcl.xLeft += WinQuerySysValue (HWND_DESKTOP, SV_CXICON) / 2 + 1;

         WinDrawText (hps, -1L, pip->item.text, &rcl,
                      fgColor, bgColor, textAttr);
      break;
      }

      case ITEMTYPE_BITMAP:

      if (pip->item.itemAlign == ITEMALIGN_STRETCH)
      {
         SIZEL szl;

         queryBitmapSize (hps, pip->item.imageHandle, &szl);

         j = (pip->rcl.yTop - pip->rcl.yBottom - szl.cy) / 2;
         i = (pip->rcl.xRight - pip->rcl.xLeft - szl.cx) / 2;

         WinFillRect (hps, &pip->rcl, bgColor); // erase background under bitmap
         WinDrawBitmap (hps, pip->item.imageHandle, NULL, (PPOINTL)&pip->rcl,
                        fgColor, bgColor, DBM_NORMAL);

      }
      else
      {
         WinDrawBitmap (hps, pip->item.imageHandle, NULL, (PPOINTL)&pip->rcl,
                        fgColor, bgColor, DBM_STRETCH);
      }
      break;
   }
}

int     formatColumnHeaders (HPS hps, PLISTBOXDATA plbd)
{
   int i, height = 0, width = 0;
   PCOLUMN pc = plbd->firstColumn;

   for (i = 0; i < plbd->columns && pc; i++)
   {
      SIZEL szl;

      queryItemSize (hps, &pc->columnItem, &szl);

      if (pc->columnWidth == -1) pc->columnWidth = szl.cx + 2;

      if (height < (szl.cy + 2)) height = szl.cy + 2;

      pc->columnItem.rcl.xLeft = width + 1;
      pc->columnItem.rcl.xRight = width + pc->columnWidth - 1;
      //pc->columnItem.rcl.yBottom = 1;

      width += pc->columnWidth;
      pc = pc->nextColumn;
   }

   for (pc = plbd->firstColumn; pc; pc = pc->nextColumn)
   {
      pc->columnItem.rcl.yTop = plbd->windowHeight - 1;
      pc->columnItem.rcl.yBottom = plbd->windowHeight - height - 1;
   }

   plbd->boxWidth = width;
   plbd->headerHeight = height;

return height;
}

void     scrollItems (PLISTBOXDATA plbd, int xDelta, int yDelta)
{
   PCOLUMN pc;

   for (pc = plbd->firstColumn; pc; pc = pc->nextColumn)
   {
      PITEMPACK pip;

      for (pip = pc->firstItem; pip; pip = pip->nextItem)
      {
         pip->rcl.yTop += yDelta;
         pip->rcl.yBottom += yDelta;
         pip->rcl.xLeft += xDelta;
         pip->rcl.xRight += xDelta;
      }

      pc->columnItem.rcl.xLeft += xDelta;
      pc->columnItem.rcl.xRight += xDelta;
   }

}

int     formatItems (HPS hps, PLISTBOXDATA plbd)
{
   int i, j, rowHeight, height = 0, width = 0;
   PCOLUMN pc = plbd->firstColumn;

   for (i = 0, pc = plbd->firstColumn; i < plbd->columns && pc; i++)
   {
      pc->tempItem = pc->firstItem;
      width += pc->columnWidth;
      pc = pc->nextColumn;
   }

   //plbd->boxWidth = width; // set already in formatColumnHeaders?

   do
   {
      width = rowHeight = 0;
      pc = plbd->firstColumn;
      j = 0;

      // look about highest row in all columns

      for (i = 0; i < plbd->columns && pc; i++)
      {
         SIZEL szl;

         if (pc->tempItem)
         {
            queryItemSize (hps, pc->tempItem, &szl);

            if (rowHeight < (szl.cy + 2)) rowHeight = szl.cy + 2;

            //pc->tempItem.rcl.yTop = ;
            pc->tempItem->rcl.xLeft = width + 1;
            pc->tempItem->rcl.xRight = width + pc->columnWidth;
            //pc->tempItem = pc->tempItem->nextItem;
            j ++;
         }

         width += pc->columnWidth;

         pc = pc->nextColumn;
      }

      // autoadjust row heights

      for (i = 0, pc = plbd->firstColumn; i < plbd->columns && pc && j; i++)
      {
         if (pc->tempItem)
         {
            pc->tempItem->rcl.yTop = plbd->windowHeight - height
                                   - 2 - plbd->headerHeight;
            pc->tempItem->rcl.yBottom = plbd->windowHeight - height
                                      - rowHeight - 1 - plbd->headerHeight;
            pc->tempItem = pc->tempItem->nextItem;
         }

         pc = pc->nextColumn;
      }

      height += rowHeight;
   } while (j);

   plbd->boxHeight = height;

return height;
}

void     paintListHeaders (HPS hps, PLISTBOXDATA plbd, PRECTL prclBound)
{
   HAB          hab = WinQueryAnchorBlock (plbd->hwnd);
   PCOLUMN      pc;

   for (pc = plbd->firstColumn; pc; pc = pc->nextColumn)
   {
      PITEMPACK pip = &pc->columnItem;
      RECTL  rclTemp;

      rclTemp = pip->rcl;
      rclTemp.xLeft --; rclTemp.yTop ++;
      rclTemp.xRight ++; rclTemp.yBottom --;

      if (WinIntersectRect (hab, &rclTemp, prclBound, &rclTemp))
      {
         rclTemp = pip->rcl;
         rclTemp.xLeft --; rclTemp.yTop ++;
         rclTemp.xRight ++; rclTemp.yBottom --;

         drawItem (hps, pip, FALSE, FALSE);
         WinDrawBorder (hps, &rclTemp, 1, 1, 0, 0, 0x400);
      }

   }

   if ((plbd->boxWidth - plbd->currentX) < plbd->windowWidth)
   {
      RECTL rclReg;

      rclReg.xLeft = plbd->boxWidth - plbd->currentX;
      rclReg.xRight = plbd->windowWidth - plbd->scrollbarWidth;
      rclReg.yTop = plbd->windowHeight - plbd->headerHeight - 1;
      rclReg.yBottom = plbd->scrollbarWidth;

      if (WinIntersectRect (hab, &rclReg, prclBound, &rclReg))
      {
         WinFillRect (hps, &rclReg,
                      WinQuerySysColor(HWND_DESKTOP, SYSCLR_WINDOW, 0));
      }

      rclReg.xLeft = plbd->boxWidth - plbd->currentX;
      rclReg.xRight = plbd->windowWidth;
      rclReg.yTop = plbd->windowHeight;
      rclReg.yBottom = plbd->windowHeight - plbd->headerHeight - 2;

      if (WinIntersectRect (hab, &rclReg, prclBound, &rclReg))
      {
         WinFillRect (hps, &rclReg,
                      WinQuerySysColor(HWND_DESKTOP, SYSCLR_DIALOGBACKGROUND, 0));
      }

      rclReg.xLeft = plbd->boxWidth - plbd->currentX;
      rclReg.xRight = plbd->windowWidth - 1;
      rclReg.yTop = plbd->windowHeight;
      rclReg.yBottom = plbd->windowHeight - plbd->headerHeight - 2;
      WinDrawBorder (hps, &rclReg, 1, 1, 0, 0, 0x400);
   }
}

void     paintListItems (HPS hps, PLISTBOXDATA plbd, PRECTL prclBound)
{
   HAB          hab = WinQueryAnchorBlock (plbd->hwnd);
   PCOLUMN      pc;

   for (pc = plbd->firstColumn; pc; pc = pc->nextColumn)
   {
      PITEMPACK pip;
      int       i;

      for (i = 0, pip = pc->firstItem; pip; i++, pip = pip->nextItem)
      {
         RECTL  rclTemp;

         rclTemp = pip->rcl;
         rclTemp.xLeft --; rclTemp.yTop ++;
         rclTemp.xRight ++; rclTemp.yBottom --;

         if (WinIntersectRect (hab, &rclTemp, prclBound, &rclTemp))
         {
            POINTL ptl;
            long bgColor = (i == plbd->currentRow) ?
                           (plbd->haveFocus ? WinQuerySysColor(HWND_DESKTOP,
                                              SYSCLR_HILITEBACKGROUND, 0)
                                            : WinQuerySysColor(HWND_DESKTOP,
                                              SYSCLR_DIALOGBACKGROUND, 0) )
                                                   : pip->item.bgColor;

            drawItem (hps, pip, (i == plbd->currentRow), plbd->haveFocus);
            ptl.x = pip->rcl.xLeft - 1; ptl.y = pip->rcl.yBottom - 1;

            GpiSetColor (hps, (pip->item.itemFlags & ITEMFLAGS_NOTSELECT) ?
                              pip->item.bgColor : bgColor);
            GpiMove (hps, &ptl);
            ptl.x = pip->rcl.xRight; ptl.y = pip->rcl.yTop;
            GpiBox (hps, DRO_OUTLINE, &ptl, 0, 0);
         }
      }
   }

   if (plbd->windowWidth - plbd->headerHeight - plbd->scrollbarWidth >
       plbd->boxHeight - plbd->currentY)
   {
      RECTL rclReg;

      rclReg.xLeft = 0;
      rclReg.xRight = plbd->boxWidth;
      rclReg.yTop = plbd->windowHeight - plbd->boxHeight
                  + plbd->currentY - plbd->headerHeight;
      rclReg.yBottom = plbd->scrollbarWidth;

      if (WinIntersectRect (hab, &rclReg, prclBound, &rclReg))
      {
         WinFillRect (hps, &rclReg,
                      WinQuerySysColor(HWND_DESKTOP, SYSCLR_WINDOW, 0));
      }
   }
}

void    destroyListbox (PLISTBOXDATA plbd)
{
   PCOLUMN pc, pcPrev;

   for (pcPrev = NULL, pc = plbd->firstColumn; pc; pc = pc->nextColumn)
   {
      PITEMPACK pip, pipPrev;

      for (pipPrev = NULL, pip = pc->firstItem; pip; pip = pip->nextItem)
      {
         if (pipPrev) free (pipPrev);
         pipPrev = pip;
      }

      if (pipPrev) free (pipPrev);

      if (pcPrev) free (pcPrev);

      pcPrev = pc;
   }

   if (pcPrev) free (pcPrev);

   //_heapmin (); // what's that, dear Watson?
}

int    destroyColumn (PLISTBOXDATA plbd, int columnIndex)
{
   PCOLUMN pc, pcPrev = NULL;
   int i;

   for (i = 0, pc = plbd->firstColumn; i ++, pc; pc = pc->nextColumn)
   {
      if (columnIndex == i)
      {
         PITEMPACK pip, pipPrev;

         for (pipPrev = NULL, pip = pc->firstItem; pip; pip = pip->nextItem)
         {
            if (pipPrev) free (pipPrev);
            pipPrev = pip;
         }

         if (pipPrev) free (pipPrev);

         if (pcPrev)
         {
            // remove this column from linked list
            pcPrev->nextColumn = pc->nextColumn;
         } plbd->firstColumn = pc->nextColumn; // we removed 1st column in list

         free (pc);
         plbd->columns --;

      break;
      }

      pcPrev = pc;
   }

return plbd->columns;
}

int    destroyItem (PCOLUMN pc, int itemIndex)
{
   PITEMPACK pip, pipPrev = NULL;
   int i;

   for (i = 0, pip = pc->firstItem; pip; pip = pip->nextItem, i++)
   {
      if (itemIndex == i)
      {

         if (pipPrev)
         {
            // remove this column from linked list
            pipPrev->nextItem = pip->nextItem;
         } else pc->firstItem = pip->nextItem; // we removed 1st column in list

         free (pip);
         pc->items --;

      break;
      }

      pipPrev = pip;
   }

return pc->items;
}

int     insertColumn (PLISTBOXDATA plbd, PITEM columnItem,
                      PCOLUMN pcNew, int index, int width)
{
   PCOLUMN pc;

   if (!pcNew)
   {
      pcNew = (PCOLUMN)malloc (sizeof (*pcNew));

      if (!pcNew) return 0;

      memset (pcNew, 0, sizeof (*pcNew));

      pcNew->columnWidth = width;
      pcNew->columnItem.item = *columnItem;
   }

   if (index == -1) // last column
   {
      if (!plbd->firstColumn) // none...
      {
         plbd->firstColumn = pcNew;
      } else
      {
         pc = plbd->firstColumn;

         while (pc->nextColumn) pc = pc->nextColumn;

         pc->nextColumn = pcNew;
      }
   } else if (index == 0) // first column
   {
      pcNew->nextColumn = plbd->firstColumn;
      plbd->firstColumn = pcNew;
   } else // somewhere 0 << last
   {
      int i;

      for (i = 0, pc = plbd->firstColumn;
           i < index - 1 && pc;
           i ++, pc = pc->nextColumn);

      if (!pc) // not found
      {
         return insertColumn (plbd, columnItem, pcNew, -1, width); // last, recall now
      }
      else
      {
         pcNew->nextColumn = pc->nextColumn;
         pc->nextColumn = pcNew;
      }
   }

   plbd->columns ++;

return plbd->columns;
}

int     insertItem (PLISTBOXDATA plbd, PITEM pitem, int x, int y)
{
   PCOLUMN pc = getColumnFromX (plbd, x);
   int rc = 0;

   if (pc)
   {
      PITEMPACK pipNew;

      pipNew = (PITEMPACK) malloc (sizeof (*pipNew));
      memset (pipNew, 0, sizeof (*pipNew));
      pipNew->item = *pitem;

      if ((y == 0) || !pc->firstItem)
      {
         pipNew->nextItem = pc->firstItem;
         pc->firstItem = pipNew;
      } else if (y == 65535) // last item
      {
         PITEMPACK pipTemp = pc->firstItem;

         while (pipTemp->nextItem) pipTemp = pipTemp->nextItem;

         pipNew->nextItem = NULL;
         pipTemp->nextItem = pipNew;

      } else
      {
         int i;
         PITEMPACK pipTemp = pc->firstItem;
         for (i = 0;
              i < y - 1 && pipTemp;
              i ++, pipTemp = pipTemp->nextItem);

         pipNew->nextItem = pipTemp->nextItem;
         pipTemp->nextItem = pipNew;
      }

   rc = 1; pc->items ++;
   }

return rc;
}

int     columnControlResize (PLISTBOXDATA plbd, SHORT x, SHORT y)
{
   PCOLUMN pc;
   int i;

   for (i = 0, pc = plbd->firstColumn; pc; pc = pc->nextColumn, i ++)
   {
      if ((x > pc->columnItem.rcl.xRight - 3) &&
          (x < pc->columnItem.rcl.xRight + 3) &&
          (y > plbd->windowHeight - plbd->headerHeight) &&
          ((pc->columnItem.item.itemFlags & ITEMFLAGS_FIXEDSIZE) == 0)) return i;
   }

return -1;
}

int     checkSelectedRow (PLISTBOXDATA plbd, SHORT x, SHORT y)
{
   PCOLUMN pc;

   if ((y < plbd->scrollbarWidth + 1) ||
       (y > plbd->windowHeight - plbd->headerHeight)) return -1;

   for (pc = plbd->firstColumn; pc; pc = pc->nextColumn)
   {
      PITEMPACK pip;
      int i;

      for (i = 0, pip = pc->firstItem; pip; pip = pip->nextItem, i ++)
      {
         if ((pip->rcl.yTop >= y) &&
             (pip->rcl.xLeft <= x) &&
             (pip->rcl.xRight >= x) &&
             (pip->rcl.yBottom <= y))
            return i;
      }
   }

return -1;
}

void    selectRowUpdate (PLISTBOXDATA plbd, int updateRow)
{
   PCOLUMN pc;
   int saveRow = plbd->currentRow;

   plbd->currentRow = updateRow;

   for (pc = plbd->firstColumn; pc; pc = pc->nextColumn)
   {
      PITEMPACK pip;
      int i;

      for (i = 0, pip = pc->firstItem; pip; pip = pip->nextItem, i ++)
      {
         if ((i == updateRow) || (i == saveRow))
         {
            RECTL rclTemp;

            rclTemp = pip->rcl;
            rclTemp.xLeft --; rclTemp.xRight ++;
            rclTemp.yTop ++; rclTemp.yBottom --;
            WinInvalidateRect (plbd->hwnd, &rclTemp, FALSE);
         }
      }
   }
}

void    formatScrollbars (PLISTBOXDATA plbd)
{
   HWND hwndHScroll = WinWindowFromID (plbd->hwnd, WID_HSCROLLBAR),
        hwndVScroll = WinWindowFromID (plbd->hwnd, WID_VSCROLLBAR);
   int width = WinQuerySysValue (HWND_DESKTOP, SV_CYHSCROLL);

   plbd->scrollbarWidth = width;

   WinSetWindowPos (hwndHScroll, HWND_TOP,
                    0, 0, plbd->windowWidth - width, width,
                    SWP_MOVE | SWP_SIZE | SWP_SHOW);

   WinSetWindowPos (hwndVScroll, HWND_TOP,
                    plbd->windowWidth - width, width,
                    width, plbd->windowHeight - width,
                    SWP_MOVE | SWP_SIZE | SWP_SHOW);


   if ((plbd->windowWidth - width) < plbd->boxWidth)
   {
      WinEnableWindow (hwndHScroll, TRUE);

      WinSendMsg (hwndHScroll, SBM_SETSCROLLBAR,
                  MPFROM2SHORT (plbd->currentX, 0),
                  MPFROM2SHORT (0, plbd->boxWidth -
                                (plbd->windowWidth - width)));

      WinSendMsg (hwndHScroll, SBM_SETTHUMBSIZE,
                  MPFROM2SHORT (plbd->windowWidth - width, plbd->boxWidth),
                  NULL);
   } else
   {
      WinEnableWindow (hwndHScroll, FALSE);
      if (plbd->currentX)
      {
         plbd->currentX = 0;
         plbd->reformatItems = TRUE;
         plbd->reformatHeaders = TRUE;
         WinInvalidateRect (plbd->hwnd, NULL, TRUE);
      }
   }

   if ((plbd->windowHeight - width - plbd->headerHeight) < plbd->boxHeight)
   {
/*      HPS hps = WinGetPS (plbd->hwnd);

      plbd->reformatHeaders = FALSE;
      formatColumnHeaders (hps, plbd);

      plbd->reformatItems = FALSE;
      formatItems (hps, plbd);

      WinReleasePS (hps);*/

      WinEnableWindow (hwndVScroll, TRUE);

//      plbd->currentY = (SHORT)WinSendMsg (hwndVScroll, SBM_QUERYPOS, (MPARAM)NULL, (MPARAM)NULL);

//      scrollItems (plbd, 0, plbd->currentY);

      WinSendMsg (hwndVScroll, SBM_SETSCROLLBAR,
                  MPFROM2SHORT (plbd->currentY, 0),
                  MPFROM2SHORT (0, plbd->boxHeight - (plbd->windowHeight
                                   - width - plbd->headerHeight)));
                                //plbd->boxHeight));

      WinSendMsg (hwndVScroll, SBM_SETTHUMBSIZE,
                  MPFROM2SHORT (plbd->windowHeight - width - plbd->headerHeight,
                                plbd->boxHeight),
                  NULL);

   } else
   {
      WinEnableWindow (hwndVScroll, FALSE);
      if (plbd->currentY)
      {
         plbd->currentY = 0;
         plbd->reformatItems = TRUE;
         plbd->reformatHeaders = TRUE;
         WinInvalidateRect (plbd->hwnd, NULL, TRUE);
      }

   }

}

void    sliderVScroll (PLISTBOXDATA plbd, MPARAM mp1, MPARAM mp2)
{
   HWND hwndVScroll = WinWindowFromID (plbd->hwnd, WID_VSCROLLBAR);
   int oldY = plbd->currentY;
   RECTL rcl;

   switch (SHORT2FROMMP (mp2))
   {
      case SB_LINEUP:
      plbd->currentY -= 4 ;
      break ;

      case SB_LINEDOWN:
      plbd->currentY += 4 ;
      break ;

      case SB_PAGEUP:
      plbd->currentY -= 12 ;
      break ;

      case SB_PAGEDOWN:
      plbd->currentY += 12 ;
      break ;

      case SB_SLIDERTRACK:
      case SB_SLIDERPOSITION:
      plbd->currentY = SHORT1FROMMP (mp2) ;
      break ;
   }

   if (plbd->currentY < 0) plbd->currentY = 0;

   if (plbd->currentY != SHORT1FROMMR (WinSendMsg (hwndVScroll,
                                       SBM_QUERYPOS, NULL, NULL)))
   {
      WinSendMsg (hwndVScroll, SBM_SETPOS,
                  MPFROMSHORT (plbd->currentY), NULL) ;
   }

   scrollItems (plbd, 0, - (oldY - plbd->currentY));

   rcl.xLeft = 0;
   rcl.xRight = plbd->windowWidth - plbd->scrollbarWidth;
   rcl.yBottom = plbd->scrollbarWidth;
   rcl.yTop = plbd->windowHeight - plbd->headerHeight - 2;

/*   if (oldY < plbd->currentY) // scrolls down
   {
      rcl.yTop = plbd->windowHeight - plbd->headerHeight - plbd->currentY + oldY - 1;
      rcl.yBottom = plbd->scrollbarWidth;
   }
   else
   {
      rcl.yTop = plbd->windowHeight - plbd->headerHeight - 1;
      rcl.yBottom = plbd->scrollbarWidth - plbd->currentY + oldY;
   }*/
   WinScrollWindow (plbd->hwnd, 0, plbd->currentY - oldY, &rcl, &rcl,
                    NULLHANDLE, NULL, SW_INVALIDATERGN);

/*   rcl.xLeft = 0;
   rcl.yBottom = plbd->scrollbarWidth;
   rcl.yTop = plbd->windowHeight - plbd->headerHeight - 1;
   rcl.xRight = plbd->windowWidth - plbd->scrollbarWidth;

   scrollItems (plbd, 0, - (oldY - plbd->currentY));
   WinInvalidateRect (plbd->hwnd, &rcl, TRUE);*/
}

void    sliderHScroll (PLISTBOXDATA plbd, MPARAM mp1, MPARAM mp2)
{
   HWND hwndHScroll = WinWindowFromID (plbd->hwnd, WID_HSCROLLBAR);
   int oldX = plbd->currentX;
   RECTL rcl;

   switch (SHORT2FROMMP (mp2))
   {
      case SB_LINELEFT:
      plbd->currentX -= 2 ;
      break ;

      case SB_LINERIGHT:
      plbd->currentX += 2 ;
      break ;

      case SB_PAGELEFT:
      plbd->currentX -= 20 ;
      break ;

      case SB_PAGERIGHT:
      plbd->currentX += 20 ;
      break ;

      case SB_SLIDERTRACK:
      case SB_SLIDERPOSITION:
      plbd->currentX = SHORT1FROMMP (mp2) ;
      break ;
   }

   if (plbd->currentX < 0) plbd->currentX = 0;
   if (plbd->currentX + plbd->windowWidth - plbd->scrollbarWidth >
       plbd->boxWidth)
   {
      plbd->currentX = plbd->boxWidth - (plbd->windowWidth - plbd->scrollbarWidth);
      if (plbd->boxWidth < plbd->windowWidth - plbd->scrollbarWidth)
         plbd->currentX = 0;
   }

   if (plbd->currentX != SHORT1FROMMR (WinSendMsg (hwndHScroll,
                                       SBM_QUERYPOS, NULL, NULL)))
   {
      WinSendMsg (hwndHScroll, SBM_SETPOS,
                  MPFROMSHORT (plbd->currentX), NULL) ;
   }

   scrollItems (plbd, (oldX - plbd->currentX), 0);

   rcl.xLeft = 0;
   rcl.xRight = plbd->windowWidth - plbd->scrollbarWidth;
   rcl.yBottom = plbd->scrollbarWidth;
   rcl.yTop = plbd->windowHeight;

   WinScrollWindow (plbd->hwnd, - (plbd->currentX - oldX), 0, &rcl, &rcl,
                    NULLHANDLE, NULL, SW_INVALIDATERGN);
}

void    trackColumn (PLISTBOXDATA plbd, int index)
{
   PCOLUMN pc = getColumnFromX (plbd, index);

   if (pc)
   {
      TRACKINFO tiStruct;

      tiStruct.cxBorder = 2;
      tiStruct.cyBorder = 2;
      tiStruct.cxGrid = 0;
      tiStruct.cyGrid = 0;

      tiStruct.rclTrack.xLeft = pc->columnItem.rcl.xRight - 1;
      tiStruct.rclTrack.xRight = pc->columnItem.rcl.xRight + 1;
      tiStruct.rclTrack.yTop = pc->columnItem.rcl.yTop;
      tiStruct.rclTrack.yBottom = plbd->scrollbarWidth;

      tiStruct.rclBoundary.xLeft = pc->columnItem.rcl.xLeft - 1;
      tiStruct.rclBoundary.xRight = WinQuerySysValue (HWND_DESKTOP, SV_CXSCREEN);//plbd->windowWidth - plbd->scrollbarWidth;
      tiStruct.rclBoundary.yTop = pc->columnItem.rcl.yTop;
      tiStruct.rclBoundary.yBottom = plbd->scrollbarWidth;

      tiStruct.ptlMinTrackSize.x = 0;
      tiStruct.ptlMinTrackSize.y = plbd->windowHeight - plbd->scrollbarWidth;
      tiStruct.ptlMaxTrackSize.y = plbd->windowHeight - plbd->scrollbarWidth;
      tiStruct.ptlMaxTrackSize.x = 3;
      tiStruct.fs = TF_ALLINBOUNDARY | TF_MOVE;

      if (WinTrackRect (plbd->hwnd, (HPS)0L, &tiStruct))
      {
         RECTL rcl;
         rcl.xLeft = 0;
         rcl.yBottom = plbd->scrollbarWidth;
         rcl.xRight = plbd->windowWidth - plbd->scrollbarWidth;
         rcl.yTop = plbd->windowHeight;

         pc->columnWidth = tiStruct.rclTrack.xLeft - pc->columnItem.rcl.xLeft + 3;
         plbd->reformatHeaders = TRUE;
         plbd->reformatItems = TRUE;
         WinInvalidateRect (plbd->hwnd, &rcl, FALSE);
      }
   }
}

void    processKey (PLISTBOXDATA plbd, int key)
{
   switch (key)
   {
      case VK_LEFT:
      sliderHScroll (plbd, MPFROMSHORT (WID_HSCROLLBAR),
                     MPFROM2SHORT (0, SB_PAGELEFT));

      break;

      case VK_RIGHT:
      sliderHScroll (plbd, MPFROMSHORT (WID_HSCROLLBAR),
                     MPFROM2SHORT (0, SB_PAGERIGHT));

      break;

      case VK_UP:
      if (plbd->currentRow > 0)
      {
         PITEMPACK pip = getItemFromXY (plbd,
                                        0,
                                        plbd->currentRow - 1);

         selectRowUpdate (plbd, plbd->currentRow - 1);

//         plbd->currentRow --;

         if (pip && (pip->rcl.yTop > plbd->windowHeight - plbd->headerHeight - 2))
         {
            sliderVScroll (plbd, MPFROMSHORT (WID_VSCROLLBAR),
                           MPFROM2SHORT (plbd->currentY
                           - (pip->rcl.yTop - (plbd->windowHeight
                           - plbd->headerHeight - 2)),
                           SB_SLIDERPOSITION));
         }
      }
      break;

      case VK_DOWN:
      if (plbd->currentRow == -1) return;

      if (plbd->currentRow < (plbd->firstColumn->items - 1))
      {
         PITEMPACK pip = getItemFromXY (plbd,
                                        0,
                                        plbd->currentRow + 1);

         selectRowUpdate (plbd, plbd->currentRow + 1);

//         plbd->currentRow --;

         if (pip && (pip->rcl.yBottom < plbd->scrollbarWidth))
         {
            sliderVScroll (plbd, MPFROMSHORT (WID_VSCROLLBAR),
                           MPFROM2SHORT (plbd->currentY
                           + (plbd->scrollbarWidth - pip->rcl.yBottom),
                           SB_SLIDERPOSITION));
         }
      }
      break;

   }
}

MRESULT EXPENTRY listboxWindowProc (HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
   PLISTBOXDATA plbd;

   if (msg == WM_CREATE)
   {
      plbd = (PLISTBOXDATA) malloc (sizeof (*plbd));

      if (plbd) // initialize stuff
      {
         PCREATESTRUCT pcs = (PCREATESTRUCT)mp2;

         memset (plbd, 0, sizeof (*plbd));
         plbd->hwnd = hwnd;
         plbd->currentRow = -1;

         WinSetWindowULong (hwnd, 0L, (ULONG)plbd);

         WinCreateWindow (hwnd, WC_SCROLLBAR, NULL, SBS_HORZ,
                          0, 0, 0, 0, hwnd, HWND_TOP, WID_HSCROLLBAR, NULL, NULL);

         WinCreateWindow (hwnd, WC_SCROLLBAR, NULL, SBS_VERT,
                          0, 0, 0, 0, hwnd, HWND_TOP, WID_VSCROLLBAR, NULL, NULL);
         plbd->windowWidth = pcs->cx;
         plbd->windowHeight = pcs->cy;
         plbd->reformatHeaders = TRUE;
         plbd->reformatItems = TRUE;
//         formatScrollbars (plbd);
      }
   } else
   {
      RECTL rcl, rclReg;
      HRGN hrgn, hrgnOld;
      HPS hps;
      int rc;

      plbd = (PLISTBOXDATA) WinQueryWindowULong (hwnd, 0L); // get instance data

      switch (msg)
      {
         // control messages
         case ALM_INSERTCOLUMN:
         rc = insertColumn (plbd, (PITEM)mp1, NULL,
                            SHORT1FROMMP(mp2), SHORT2FROMMP(mp2));

         plbd->reformatHeaders = TRUE;
         return (MRESULT) rc;

         case ALM_REMOVECOLUMN:
         rc = destroyColumn (plbd, (INT)mp2);
         plbd->reformatHeaders = TRUE;
         plbd->reformatItems = TRUE;
         return (MRESULT) rc;

         case ALM_CHANGECOLUMN:
         {
            PCOLUMN pc = getColumnFromX (plbd, (INT)mp2);

            if (pc)
            {
               pc->columnItem.item = *((PITEM)mp2);
               plbd->reformatHeaders = TRUE;
               plbd->reformatItems = TRUE;               // workaround...
            }

         return (MRESULT) plbd->reformatHeaders;
         }

         case ALM_INSERTITEM:
         rc = insertItem (plbd, (PITEM)mp1,
                          (int)SHORT1FROMMP (mp2), (int)SHORT2FROMMP (mp2));

         plbd->reformatItems = TRUE;
         return (MRESULT) rc;

         case ALM_CHANGEITEM:
         {
            PITEMPACK pip = getItemFromXY (plbd,
                                           SHORT1FROMMP (mp1),
                                           SHORT2FROMMP (mp1));

            if (pip)
            {
               pip->item = *((PITEM)mp2);
               plbd->reformatHeaders = TRUE;             // workaround...
               plbd->reformatItems = TRUE;

               return (MRESULT) 1;
            }
            return (MRESULT) 0L;
         }

         case ALM_QUERYITEM:
         {
            PITEMPACK pip = getItemFromXY (plbd,
                                           SHORT1FROMMP (mp1),
                                           SHORT2FROMMP (mp1));

            if (pip)
            {
               *((PITEM)mp2) = pip->item;

               return (MRESULT) 1;
            }
            return (MRESULT) 0L;
         }

         case ALM_QUERYITEMRECT:
         {
            PITEMPACK pip = getItemFromXY (plbd,
                                           SHORT1FROMMP (mp1),
                                           SHORT2FROMMP (mp1));

            if (pip)
            {
               *((PRECTL)mp2) = pip->rcl;

               return (MRESULT) 1;
            }
            return (MRESULT) 0L;
         }

         case ALM_QUERYITEMCOUNT:
         {
            PCOLUMN pc = getColumnFromX (plbd, (INT)mp2);

            if (pc) return (MRESULT)pc->items;

         return (MRESULT) 0;
         }

         case ALM_REMOVEITEM:
         {
            PCOLUMN pc = getColumnFromX (plbd, SHORT1FROMMP (mp2));

            if (pc)
            {
               plbd->reformatItems = TRUE;
               if (SHORT2FROMMP(mp2) == plbd->currentRow)
               {
                  plbd->currentRow = -1;
/*                  WinSendMsg (WinQueryWindow (hwnd, QW_OWNER), WM_CONTROL,
                              MPFROM2SHORT (WinQueryWindowUShort(hwnd, QWS_ID),
                              ALN_SELECT), (MPARAM)plbd->currentRow);*/

               }
               return (MRESULT)destroyItem (pc, SHORT2FROMMP (mp2));
            }
         return (MRESULT) -1L;
         }

         case ALM_QUERYSELECTION:
         return (MPARAM)plbd->currentRow;

         case WM_CHAR:
            if ((SHORT1FROMMP (mp1) & (KC_VIRTUALKEY | KC_KEYUP)) == KC_VIRTUALKEY)
            {
               processKey (plbd, SHORT2FROMMP (mp2));
            }
         break;

         case WM_SETFOCUS:
         plbd->haveFocus = (BOOL)mp2;
         selectRowUpdate (plbd, plbd->currentRow);
         return (MRESULT)0;

         case WM_SIZE:
         plbd->windowWidth = SHORT1FROMMP (mp2);
         plbd->windowHeight = SHORT2FROMMP (mp2);
         plbd->reformatHeaders = TRUE;
         plbd->reformatItems = TRUE;
//         formatScrollbars (plbd);
         break;

         case WM_VSCROLL:
         sliderVScroll (plbd, mp1, mp2);
         break;

         case WM_HSCROLL:
         sliderHScroll (plbd, mp1, mp2);
         break;

         case WM_BUTTON1DOWN:
         WinFocusChange (HWND_DESKTOP, hwnd, 0);
         WinSetActiveWindow (HWND_DESKTOP, WinQueryWindow (hwnd, QW_OWNER));
         rc = columnControlResize (plbd, SHORT1FROMMP (mp1), SHORT2FROMMP (mp1));

         if (rc != -1) trackColumn (plbd, rc);
         else
         {
            rc = checkSelectedRow (plbd, SHORT1FROMMP (mp1), SHORT2FROMMP (mp1));

            if (rc > -1)
            {
               selectRowUpdate (plbd, rc);
               plbd->leftDown = TRUE;
               WinSetCapture (HWND_DESKTOP, hwnd);
               WinSendMsg (WinQueryWindow (hwnd, QW_OWNER), WM_CONTROL,
                           MPFROM2SHORT (WinQueryWindowUShort(hwnd, QWS_ID),
                           ALN_SELECT), (MPARAM)rc);

            }

         }
         return (MRESULT)0;

         case WM_BUTTON1DBLCLK:
         rc = checkSelectedRow (plbd, SHORT1FROMMP (mp1), SHORT2FROMMP (mp1));

         WinSendMsg (WinQueryWindow (hwnd, QW_OWNER), WM_CONTROL,
                     MPFROM2SHORT (WinQueryWindowUShort(hwnd, QWS_ID),
                     ALN_OPEN), (MPARAM)rc);

         return (MRESULT)0;

         case WM_BEGINDRAG:
         rc = checkSelectedRow (plbd, SHORT1FROMMP (mp1), SHORT2FROMMP (mp1));

         WinSendMsg (WinQueryWindow (hwnd, QW_OWNER), WM_CONTROL,
                     MPFROM2SHORT (WinQueryWindowUShort(hwnd, QWS_ID),
                     ALN_DRAG), (MPARAM)rc);

         return (MRESULT)0;

         case WM_BUTTON1UP:

         if (plbd->leftDown)
         {
            plbd->leftDown = FALSE;
            WinSetCapture (HWND_DESKTOP, NULLHANDLE);
         }
         return (MRESULT)0;

         case WM_MOUSEMOVE:

         if (plbd->leftDown)
         {
/*            if (SHORT2FROMMP(mp1) < plbd->scrollbarWidth)
            {
               return WinSendMsg (hwnd, WM_VSCROLL,
                                  MPFROMSHORT (WID_VSCROLLBAR),
                                  MPFROM2SHORT (plbd->currentY, SB_PAGEDOWN));
            }

            if (SHORT2FROMMP(mp1) > plbd->windowHeight - plbd->headerHeight)
            {
               return WinSendMsg (hwnd, WM_VSCROLL,
                                  MPFROMSHORT (WID_VSCROLLBAR),
                                  MPFROM2SHORT (plbd->currentY, SB_PAGEUP));
            }*/

            rc = checkSelectedRow (plbd, SHORT1FROMMP (mp1), SHORT2FROMMP (mp1));

            if ((rc > -1) && (rc != plbd->currentRow))
            {
               selectRowUpdate (plbd, rc);
               WinSendMsg (WinQueryWindow (hwnd, QW_OWNER), WM_CONTROL,
                           MPFROM2SHORT (WinQueryWindowUShort(hwnd, QWS_ID),
                           ALN_SELECT), (MPARAM)rc);
            }

         return (MRESULT)0;
         }

         rc = columnControlResize (plbd, SHORT1FROMMP(mp1), SHORT2FROMMP(mp1));

         if (rc > -1)
         {
            WinSetPointer (HWND_DESKTOP, WinQuerySysPointer (HWND_DESKTOP,
                           SPTR_SIZEWE, FALSE));
//            return (MRESULT) 1L;
         } else
         {
            WinSetPointer (HWND_DESKTOP, WinQuerySysPointer (HWND_DESKTOP,
                           SPTR_ARROW, FALSE));
//            return (MRESULT) 1L;
         }
         return (MRESULT)0;

         case WM_CONTEXTMENU:
         rc = checkSelectedRow (plbd, SHORT1FROMMP (mp1), SHORT2FROMMP (mp1));

         WinSendMsg (WinQueryWindow (hwnd, QW_OWNER), WM_CONTROL,
                     MPFROM2SHORT (WinQueryWindowUShort(hwnd, QWS_ID),
                     ALN_CONTEXTMENU), (MPARAM)rc);
         break;


/*         case WM_SINGLESELECT:
         rc = checkSelectedRow (plbd, SHORT1FROMMP(mp1), SHORT2FROMMP(mp1));

         if (rc > -1)
         {
            selectRowUpdate (plbd, rc);
         }
         break;*/

         case WM_PAINT: // paint stuff

         hps = WinBeginPaint (hwnd, 0L, &rcl);

         GpiCreateLogColorTable(hps, 0L, LCOLF_RGB, 0L, 0L, (PLONG) NULL);

         if (plbd->reformatHeaders)
         {
            plbd->reformatHeaders = FALSE;
            formatColumnHeaders (hps, plbd);
//            scrollItems (plbd, plbd->currentX, 0);
            plbd->reformatScrollbars = TRUE;
         }

         if (plbd->reformatItems)
         {
            plbd->reformatItems = FALSE;
            formatItems (hps, plbd);
            scrollItems (plbd, -plbd->currentX, plbd->currentY);
            plbd->reformatScrollbars = TRUE;
         }

         paintListHeaders (hps, plbd, &rcl);

         rclReg.xLeft = 0;
         rclReg.yBottom = plbd->scrollbarWidth;
         rclReg.xRight = plbd->windowWidth - plbd->scrollbarWidth;
         rclReg.yTop = plbd->windowHeight  - plbd->headerHeight - 2;

         if (WinIntersectRect (WinQueryAnchorBlock (hwnd),
                               &rclReg, &rcl, &rclReg))
         {
            hrgn = GpiCreateRegion (hps, 1, &rclReg);
            GpiSetClipRegion (hps, hrgn, &hrgnOld);
            paintListItems (hps, plbd, &rclReg); // special clip...
            GpiSetClipRegion (hps, hrgnOld, &hrgn);
            GpiDestroyRegion (hps, hrgn);
         }

         rclReg.xLeft = plbd->windowWidth - WinQuerySysValue (HWND_DESKTOP, SV_CYHSCROLL);
         rclReg.yBottom = 0;
         rclReg.xRight = plbd->windowWidth;
         rclReg.yTop = WinQuerySysValue (HWND_DESKTOP, SV_CYHSCROLL);

         if (WinIntersectRect (WinQueryAnchorBlock (hwnd),
                               &rclReg, &rcl, &rclReg))
         {
            WinFillRect (hps, &rclReg,
                         WinQuerySysColor(HWND_DESKTOP, SYSCLR_DIALOGBACKGROUND, 0));

         }
         WinEndPaint (hps);

         if (plbd->reformatScrollbars)
         {
            plbd->reformatScrollbars = FALSE;
            formatScrollbars (plbd);
         }
         return (MRESULT) 1;

         case WM_DESTROY:
         destroyListbox (plbd); // free memory
         free (plbd);
         break;
      }
   }

return WinDefWindowProc (hwnd, msg, mp1, mp2);
}

void EXPENTRY  registerListbox (HAB hab)
{
   WinRegisterClass (hab, WC_ALISTBOX, (PFNWP)listboxWindowProc,
                     CS_SIZEREDRAW, sizeof (ULONG));
}

