/*
 ***********************************************************************
 *                                                                     *
 * i860.h: i860 APX Driver definitions.                                *
 *                                                                     *
 ***********************************************************************
 */

/*
 * i860_NPROC should be the max number of user processes to run
 * concurrently + 3. (there are 3 system processes running all time).
*/
#define i860_NPROC 12

/*
 * MAX_BOOT_TIMEOUT defines the number of checks the i860 boot driver
 * does to see if the EV is ready to read information.
 * The checks are done every 1/10 second.
*/
#define MAX_BOOT_TIMEOUT 32

/*
 * Maximum amount of data in a control block
 */
#define CB_MAX_DATA 0x4000

/* Ioctl requests for i860boot */

#define RESET		0		/* Halt the i860 */
#define GO		1		/* Start the i860 running */
#define XFACE_INFO	2		/* Display xface info on console */
#define REQUEST_NPROC	3		/* Get value of i860_NPROC */
#define	I860DEBUG	4		/* Toggle debugging printfs */
#define	I860INB		5		/* inb() hack */
#define	I860OUTB	6		/* outb() hack */
#define	I860MEMSIZE	7		/* Return physical memory size of 860 */
#define	I860MEMSTART	8		/* Return physical start addr of 860 */
#define	I860SETARB	9		/* Set arbitration level */
#define	I860MAPBIOS	10		/* Remap ROM BIOS to BIOS area */
#define	I486MEMSIZE	11		/* Return physical memory size of 486 */

/* Ioctl requests for i860comm */

#define IOCTL_XFACE_ADDR	1
#define IOCTL_RESET_DRIVER	2
#define IOCTL_REQUEST_NPROC	3
#define IOCTL_SEND_SIGNAL	4
#define	IOCTL_PRINTF_ADDR	5
#define	IOCTL_PROD		6

/* i860_NUM_PTRS should not be changed at any case */
#define i860_NUM_PTRS		0x12	/* 18 pointer pairs */

typedef struct {
	struct {
		unsigned short    ack;
		unsigned short    op;
		unsigned short    io_index;
		unsigned short    filler;
	} i860;
	struct {
		unsigned short    ack;
		unsigned short    op;
		unsigned short    io_index;
		unsigned short    filler;
	} host;
} i860_xface_t;

typedef struct {
	unsigned long     p_addr;
	unsigned long     size;
} i860_ptr_pair;

typedef struct {
	i860_ptr_pair    ptrs[i860_NUM_PTRS];
} i860_ptr_blk;

typedef struct {
	i860_ptr_blk     *shm_p;
	unsigned short    pid;
	unsigned short    status;
} i860_proc_t;

#ifdef DRV860
/*
 * Common type shorthands
 */
#define	byte		unsigned char
#define	bool		unsigned char
#define	word		unsigned short
#define	dword		unsigned long

/*
 * Common constants
 */
#define	TRUE		1
#define	FALSE		0
#define	NIL		((char *)0)

/*
 * Macro shorthands
 */
#define	min(a, b)	(((a) < (b)) ? (a) : (b))
#define	max(a, b)	(((a) > (b)) ? (a) : (b))

/*
 * Hauppague 4860 I/O Ports
 * See "Hauppague 4860 Programmer's Reference Manual, Rev. 0.3; March 16, 1990"
 *	Page 1-1
 */
#define	i4860_CR	0xf8		/* Mother board configuration reg */
#define	i486_ATTI	0xfb		/* 486 Attention interrupt */
#define	i860_ATTI	0xfc		/* 860 Attention interrupt */
#define	i860_CR		0xfd		/* 860 Configuration reg */
#define	i860_INTCTLR	0xfe		/* 860 Interrupt Controller */
#define	i860_INTMASK	0xff		/* 860 Interrupt Cont. Mask */
#define	ECR_PAGE_REG	0xc00		/* CMOS page register */
#define	ECR_DATA_PAGE	0x800		/* Get CMOS data address */
#define	HCW_CFR		0x810		/* CMOS copy of config reg */
#define	HCW_860CFR	0x811		/* CMOS copy of 860 config reg */
#define	HCW_STATUS	0x820		/* Status info */
#define	HCW_860_RAM	0x834		/* How much 860 RAM */
#define	MEM_BIOS	0xffff0000	/* Where the ROM is in 486 */

/*
 * HCW status bits for HCW_STATUS
 */
#define	HCW_STATUS_PAGE	0xfe		/* Where status page is in CMOS */
#define	HCW_SETUP_PAGE	0xff		/* Where setup page is in CMOS */
#define	ST_i860		0x40		/* Is 860 preset? */
#define	ST_HCW_CHECKSUM	0x02		/* Is checksum good? */

/*
 * HCW_PAGE structure, from CMOS
 */
struct	mr_desc	{
	byte	mr_count;
	byte	mr_attr;
	byte	mr_map486;
	byte	mr_map860;
};
#define	MR_IMAGE_SIZE	47

struct	hcw_page	{
	byte	h_setup_status;		/* +00		setup	*/
	byte	h_setup_program;	/* +01		setup	*/
	byte	h_setup_version;	/* +02		setup	*/

	byte	h_fill03[13];

	byte	h_config_reg;		/* +10	status	setup	*/
	byte	h_config860_reg;	/* +11	status	setup	*/
	byte	h_speed;		/* +12	status	setup	*/

	byte	h_fill13;

	byte	h_simm_cfg[4];		/* +14	status	setup	*/

	byte	h_fill18[8];

	byte	h_status;		/* +20	status		*/
	byte	h_bist;			/* +21	status		*/
	byte	h_comp_id;		/* +22	status		*/
	byte	h_comp_step;		/* +23	status		*/

	byte	h_fill24[12];

	word	h_conv_ram;		/* +30		setup	*/
	word	h_ext_ram;		/* +32		setup	*/
	word	h_ram_860;		/* +34		setup	*/

	byte	h_fill33[9];

	byte	h_bios_map;		/* +3f		setup	*/
	byte	h_mr_count_lo;		/* +40	status	setup	*/
	byte	h_mr_count_hi;		/* +41	status	setup	*/
	struct mr_desc h_mr_image[MR_IMAGE_SIZE];
	word	h_ecram_chksm;		/* +fe		setup	*/
};
typedef	struct hcw_page	hcw_page_t;

/*
 * i4860_CR values
 */
#define	CFR_MREN	0x01
#define	CFR_MRWRITE	0x02
#define	CFR_MR860	0x04
#define	CFR_MR486	0x00
#define	CFR_PNORM	0x00
#define	CFR_PERROR	0x80
#define	CFR_BUS5	0x00
#define	CFR_BUS4	0x10
#define	CFR_BUS3	0x20
#define	CFR_BUS2	0x30
#define	CFR_PORTEN	0x40
#define	CFR_CACHEDIS	0x80

/*
 * i860_CR values
 */
#define	ARB_FAIR	0x00		/* Arbiter Mode Fair */
#define	ARB_486		0x01		/* Priority 486 */
#define	ARB_860		0x02		/* Priority 860 */
#define	ARB_64BIT	0x03		/* Priority 64 bit slot */
#define	i486_IRQ13	0x00		/* Int. Request 13 on 486 */
#define	i486_NMI	0x04		/* Send NMI */
#define	i860_HALT	0x80		/* Halt i860 */
#define	i860_RUN	0x00		/* Let 860 run */

/*
 * h4860 memory map register
 */
#define	MR_RW		0x00
#define	MR_RO		0x01
#define	MR_CACHE	0x00
#define	MR_NOCACHE	0x02
#define	MR_FAST		0x00
#define	MR_SLOW		0x04
#define	MR_ENABLE	0x00
#define	MR_DISABLE	0x10
#define	MR_FRONT	0x00
#define	MR_BACK		0x20
#define	MR_BANK0	0x00
#define	MR_BANK1	0x40
#define	MR_BANK2	0x80
#define	MR_BANK3	0xc0
#define	MR_ROM		0x80


/*
 * CMOS constants
 */
#define	CMOS_CR		0x5a
#define	CMOS_CFR	0x5e

/*
 * i860 shared memory constants
 */
#define	INTR_ON		((word)0x0002)
#define	INDIR		0x8000

/*
 * Fixed processes
 */
#define	ADM860		((word)0)
#define	PMGR		((word)1)

/*
 * Implementation constants
 */
#define	TIME_OUT	i860HZ
#define	MAXDELAYS	20	/* Wait this many 10ths before giving up */
#define	SWPR_MASK	0x8000
#define	PARITY_ERR_MASK	0x0001
/* Must also change constants in krprint.c in ker860 directory */
#define	PRBUF_SIZE	2048

/*
 * Xface structure ACK values
 */
#define	FREE		0
#define	BUSY		1

/*
 * Xface structure "op" values
 */
#define	OP_READ		7
#define	OP_WRITE	8

/*
 * Process list status values
 */
#define	UNUSED		0		/* Slot is unused (free) */
#define	READY		1		/* Slot is in use */
#define	WAIT_IO		2		/* Waiting for result from i860 */
#define	IO_DONE		3		/* Interrupt received from i860 */

/*
 * Control block structure
 */
typedef struct	{
	unsigned short	pid;		/* Executor ID */
	unsigned char	tcode;		/* Transaction code */
	unsigned char	scode;		/* Sub-code */
	unsigned long	info;		/* Additional info */
	unsigned long	info2;		/* Error code or more info */
	unsigned short	size;		/* Tail data size */
	unsigned char	reserved[2];	/* Reserved for future use */
} CB_header_t;
	/* Tail data immediately follows the CB_header_t, 'size' bytes worth */

#define	LAST_PMGR_TCODE	0xf
#define	C_CRPROC	0x1		/* Create a new process */
#define	C_GOPROC	0x7		/* Start a new process */
#define	C_CTRL		0x5		/* 'adm860 -s or -k' */
#define	C_SIGNAL	0xc		/* Signal a process */
#define	C_EXIT		0xe		/* Terminate a process */

/*
 * Minor device numbers
 */
#define	BOOT_DEVICE	0		/* /dev/i860boot */
#define	COMM_DEVICE	1		/* /dev/i860comm */
#endif
