#ifdef __KERNEL__
#include <linux/string.h>
#else // shell
#include <string.h>
#endif

#include "logtime.h"

time_t ConvertHWLogTime(time_t timeHW) {
	int			 nDSTFlag;
	time_t		 timeLog;
	struct rtc_time LocalTime, *pLocalTime;
	const time_t	 nSecPerDay = ( 24 * 60 * 60 );
	
	// We cannot adjust for GMT when the time is not "real" time.
	if( timeHW < nSecPerDay )
		return( timeHW );
	
	// The hardware does not take timezones into account, so this
	// call actually returns the local time of when the hardware event
	// was logged.
	pLocalTime = gmtimek( &timeHW );
	memcpy( &LocalTime, pLocalTime, sizeof(struct rtc_time) );
	
	// Ignore the return value; we are only interested in the DST
	// flag returned in the LocalTime structure.
	timeLog = maketime( LocalTime.tm_year+1900, LocalTime.tm_mon+1 ,
				   LocalTime.tm_mday     , LocalTime.tm_hour,
				   LocalTime.tm_min      , LocalTime.tm_sec  );
	nDSTFlag = LocalTime.tm_isdst;
	
	// Get the LocalTime again, because maketime changes the LocalTime 
	// structure to something we can not use.
	pLocalTime = gmtimek( &timeHW );
	memcpy( &LocalTime, pLocalTime, sizeof(struct rtc_time) );
	
	// Set the DST Flag in the LocalTime structure to the appropriate value.
	LocalTime.tm_isdst = nDSTFlag;
	
	// The LocalTime structure will now be converted into a timestamp value
	// representing GM time.
	timeLog = maketime( LocalTime.tm_year+1900, LocalTime.tm_mon+1 ,
				   LocalTime.tm_mday     , LocalTime.tm_hour,
				   LocalTime.tm_min      , LocalTime.tm_sec  );
	
	return ( timeLog );
}

static struct rtc_time _TM;

#define	SECS_PER_HOUR	(60 * 60)
#define	SECS_PER_DAY	(SECS_PER_HOUR * 24)

#define	__isleap(year)	((year) % 4 == 0 && ((year) % 100 != 0 || (year) % 400 == 0))

const unsigned short int __mon_yday[2][13] = {
	/* Normal years.  */
	{ 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334, 365 },
	/* Leap years.  */
	{ 0, 31, 60, 91, 121, 152, 182, 213, 244, 274, 305, 335, 366 }
};

static int __offtime(const time_t *t, long int offset, struct rtc_time *tp) {
	long int days, rem, y;
	const unsigned short int *ip;
	
	days = *t / SECS_PER_DAY;
	rem = *t % SECS_PER_DAY;
	rem += offset;
	while (rem < 0) {
		rem += SECS_PER_DAY;
		--days;
	}
	while (rem >= SECS_PER_DAY) {
		rem -= SECS_PER_DAY;
		++days;
	}
	tp->tm_hour = rem / SECS_PER_HOUR;
	rem %= SECS_PER_HOUR;
	tp->tm_min = rem / 60;
	tp->tm_sec = rem % 60;
	/* January 1, 1970 was a Thursday.  */
	tp->tm_wday = (4 + days) % 7;
	if (tp->tm_wday < 0)
		tp->tm_wday += 7;
	y = 1970;

#define DIV(a, b) ((a) / (b) - ((a) % (b) < 0))
#define LEAPS_THRU_END_OF(y) (DIV (y, 4) - DIV (y, 100) + DIV (y, 400))

	while (days < 0 || days >= (__isleap (y) ? 366 : 365)) {
		/* Guess a corrected year, assuming 365 days per year.  */
		long int yg = y + days / 365 - (days % 365 < 0);
		
		/* Adjust DAYS and Y to match the guessed year.  */
		days -= ((yg - y) * 365 + LEAPS_THRU_END_OF (yg - 1) - LEAPS_THRU_END_OF (y - 1));
		
		y = yg;
	}
	tp->tm_year = y - 1900;
	if (tp->tm_year != y - 1900)
		return 0;
	tp->tm_yday = days;
	ip = __mon_yday[__isleap(y)];
	for (y = 11; days < (long int) ip[y]; --y)
		continue;
	days -= ip[y];
	tp->tm_mon = y;
	tp->tm_mday = days + 1;
	return 1;
}

struct rtc_time *gmtimek( time_t *timer ) {
	struct rtc_time *tp;
	
	if (timer == NULL)
		return NULL;
	
	tp = &_TM;
	tp->tm_isdst = 0;
	//tp->tm_zone = "GMT";
	//tp->tm_gmtoff = 0L;
	
	if (! __offtime (timer, 0L, tp) )
		tp = NULL;
		
	return tp;
}

unsigned long maketime ( unsigned int year, unsigned int mon, 
					unsigned int day, unsigned int hour,
					unsigned int min, unsigned int sec) {
        if (0 >= (int) (mon -= 2)) {    /* 1..12 -> 11,12,1..10 */
                mon += 12;              /* Puts Feb last since it has leap day */
                year -= 1;
        }

        return (((
                (unsigned long) (year/4 - year/100 + year/400 + 367*mon/12 + day) +
                        year*365 - 719499
            )*24 + hour /* now have hours */
          )*60 + min /* now have minutes */
        )*60 + sec; /* finally seconds */
}
