/*
 * (c) Copyright 2001 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

/**************************************************************************
**  Copyright (C) 1996, Hewlett-Packard Co. All rights reserved.
**
**  File name:  mmcdrv_m.c
**  Summary:    Middle layer of the MMC device driver
**  Product:    Motherboard Management Controller
**  Owner:      Sonny Talag
**
**  START AUTOMATIC VERSION CONTROL INFORMATION
**  DO NOT MANUALLY EDIT THIS SECTION!
**  $Workfile:   MMCDRV_M.C  $
**  $Revision: 1.2 $
**  $Date: 2001/04/24 13:22:06 $
**  $Author: hamilton $
**  END AUTOMATIC VERSION CONTROL INFORMATION
**
**  Description:
**      OS-independent layer of the device driver that manages the message
**      transfer with the MMC.  This layer sits between the OS tunneling
**      functions above and the OS-dependent port input and output
**      functions below.
**
**  Exports:
**      MMCDRVR_OpenDriver()
**      MMCDRVR_CloseDriver()
**      MMCDRVR_SendRcvMessage()
**      MMCDRVR_GetVersionNumber()
**
***************************************************************************
**  START AUTOMATIC VERSION CONTROL LOG
**  DO NOT MANUALLY EDIT THIS SECTION!
**
**  $Log: mmcdrv_m.c,v $
**  Revision 1.2  2001/04/24 13:22:06  hamilton
**  - Added copyright notice;
**  - Changed Makefile for compiling in rh70.
**
**  Revision 1.1.1.1  2000/08/21 18:45:45  hamilton
**  Initial import.
**
**  Revision 1.1.1.1  2000/08/21 12:49:14  hamilton
**  Initial import.
**
** 
**    Rev 1.36   10 Oct 1997 16:24:00   stalag
** Released for BETA
** 
**    Rev 1.35   06 Oct 1997 16:57:52   stalag
** Modified the read message retry mechanism to 25 retries with 100 msec in between
** 
**    Rev 1.34   26 Sep 1997 16:06:08   stalag
** Needed to track down a bug by modifying parameters.  Bug fixed via architectural change in firmware update utility.
** 
**    Rev 1.33   05 Sep 1997 10:53:28   stalag
** no intentional changes.
** 
** 
**    Rev 1.32   27 Aug 1997 16:21:08   engelman
** modified global variable names to have a g_ prefix
** 
**    Rev 1.31   14 Aug 1997 17:53:16   stalag
** Made changes necessary to support LC4-23.bin firmware.
** 
**    Rev 1.30   14 Aug 1997 16:04:00   stalag
** Fixed all Code Review items assigned to me on 8/13/97.
** 1.  Added comments for TIME_PROCESS 
** 2.  Added comments for ENABLE_TIMEOUTS
** 3.  Cleaned up indentations.
** 4.  Rewrote MMCDRVR_Get_Version_Number description.
** 5.  Rewrote part of ReceiveMessage per Elecia's comment.
** 6.  Renamed GetMMC* to ReadMMC*Byte
** 
**    Rev 1.29   11 Aug 1997 15:56:38   stalag
** Found another debug problem and fixed it.
** 
**    Rev 1.28   11 Aug 1997 15:45:12   stalag
** Fixed an error that came up in DEBUG mode due to a 
** #define
** 
**    Rev 1.27   06 Aug 1997 10:43:02   engelman
** Finally got the strcopy section working without bugs
** 
**    Rev 1.26   05 Aug 1997 16:44:10   engelman
** fixed get driver version, printfs
** 
**    Rev 1.25   05 Aug 1997 14:25:12   engelman
** changed printfs to use MMCPrint function, removed "magic numbers"
** (retval = 1 now is retval = GENERIC_ERROR).
** 
**    Rev 1.24   05 Aug 1997 10:18:26   stalag
** Tried to fix flash bug but didn't happen.
** 
**    Rev 1.23   05 Aug 1997 09:25:16   engelman
** fixing more typos
** 
** 
**    Rev 1.22   05 Aug 1997 09:23:24   engelman
** cleaning up typos
** 
** 
**    Rev 1.21   05 Aug 1997 09:01:52   MThomas
** added print function, still need clean up stuff
** 
**    Rev 1.20   04 Aug 1997 12:56:48   engelman
** removed data section, removed strpy, added far *
** 
**    Rev 1.19   04 Aug 1997 09:29:38   stalag
** Fixed timing problems which used to show up in the MMC/ERA flash program.
** 
**    Rev 1.18   31 Jul 1997 16:36:28   stalag
** Reworked timeouts to test for the required condition rather than do the
** MMCWait function.
** 
**    Rev 1.17   23 Jul 1997 10:11:28   engelman
** fixed some bugs introduced when I cleaned up the file, also changes 
** memcopy to something more manual since SCO doesn't have access to memcopy
** 
**    Rev 1.16   08 Jul 1997 08:01:22   engelman
** took out era_defs header
** 
**    Rev 1.15   26 Jun 1997 08:35:58   engelman
** cleaned up code a little. The hardware check for addresses only happenes
** the first time the driver is called. Took out as many magic numbers as 
** I could, made #def TRUE and FALSE and ONE_MILLISECOND so that the code 
** was clearer. Need to change some 
** 
**    Rev 1.14   19 Jun 1997 08:21:56   engelman
** changed memcpy to strcpy the version function becuase mecopy doesn't care
** if there is a '\0'
** 
**    Rev 1.13   12 Jun 1997 17:51:16   engelman
** <No Description>
** 
**    Rev 1.12   12 Jun 1997 15:29:10   stalag
** Fixed more timing problems regarding receive message retries
** 
**    Rev 1.11   11 Jun 1997 16:34:24   stalag
** Tracking down a memory leak?
** 
**    Rev 1.10   11 Jun 1997 15:35:54   stalag
** Added function headers as required by ERA coding guidelines
** 
**    Rev 1.9   10 Jun 1997 09:41:06   engelman
** took out all use of clock function and replaced it with and MMCWait 
** function that uses the x80 port to delay. Also replaced Flag variable 
** in several functions. Now instead of carrying a failed flag, the 
** function will return if it has nothing to do
** 
**    Rev 1.8   09 Jun 1997 11:58:14   engelman
** changed all of the printfs to fprintfs
** 
** 
**    Rev 1.7   06 Jun 1997 16:51:22   engelman
** <No Description>
** 
**    Rev 1.6   06 Jun 1997 12:42:36   stalag
** Fixed a timing bug where CC_SMS_GET_STATUS was timing out rather than 
** completing correctly
** 
**    Rev 1.5   06 Jun 1997 11:02:10   stalag
** Returned the variables with driver wide scope to this file.  Elecia has
** incorporated "extern" declarations into the associated header
** 
**    Rev 1.4   05 Jun 1997 17:01:36   stalag
** Removed delay per Myron's and Bob's suggestions
** 
**    Rev 1.3   05 Jun 1997 13:26:04   engelman
** moved global variables to mmcdrv_m.h
** 
**    Rev 1.2   05 Jun 1997 13:21:54   stalag
** Fixed a problem with the SendMessage function where it wrote the last 
** byte as zero
** 
**    Rev 1.1   05 Jun 1997 10:45:06   stalag
** Removed the TX_DATA_RDY requirement from the SendByte function to allow
** it to work with Bob's SMIC interface.
**
**  END AUTOMATIC VERSION CONTROL LOG
**************************************************************************/

/** includes **/
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include "imb_std.h"
#include "imb_isa.h"
#include "mmc.h"
#include "osdepdrv.h"
#include "mmcdrv_m.h"

/** local constants **/
#define DRIVERRECEIVERETRIES    25
            /* Device driver loops at most */
            /* ten times while waiting for */
            /* the results from the MMC. */

#define XMITTIMEOUT             5000L
            /* 5 milliseconds before */
            /* failing the byte transmit*/
            /* This value is the number of */
            /* times to access an I/O port */
            /* which takes ~1usec per */

#define RECEIVETIMEOUT          5000L
            /* 5 milliseconds before */
            /* failing the byte receive*/
            /* This value is the number of */
            /* times to access an I/O port */
            /* which takes ~1usec per */

#define ENABLETIMEOUTS          1
            /* Enable the command timeout */
            /* code -- otherwise wait until */
            /* the condition is met. This */
            /* is used to test whether the */
            /* MMC is actually hanging or */
            /* if it just taking a long time. */

#define RECEIVE_WAIT            100000L
            /* 100 milliseconds before */
            /* retrying the message receive */
            /* This value is the number of */
            /* times to access an I/O port */
            /* which takes ~1usec per */

#define MESSAGE_ABORT           -1
            /* return used by the SendMessage */
            /* and ReceiveMessage functions */
            /* to tell higher level to either */
            /* restart the whole message or */
            /* fail */

#define RECEIVE_BUFFER_FULL     -2
            /* return value used to indicate */
            /* the MMC is returning too many */
            /* bytes. */

#define ONE_MILLISECOND         1000
            /* one thousand micro seconds */

#define ONE_MICROSECOND         1

#define TRUE                    1
#define FALSE                   0

#define GENERIC_ERROR           1

// #define TIME_PROCESS            1
            /* used to identify timing */
            /* problems.  Defining this */
            /* enables a basic method */
            /* of timing how long portions */
            /* of the driver are taking to */
            /* complete so that the slow */
            /* portion can be tracked down. */
            /* This method has only been */
            /* tested in DOS using MSVC */
            /* version 1.52. */

/** local typedefs **/

/** local forward prototypes **/
static int ReceiveMessage(UINT8 FAR_PNTR * buffer, UINT8 len);

static int SendMessage(UINT8 FAR_PNTR * inputbufferptr,
        UINT8 commandlength);

static int SendMMCData(UINT8 command, UINT8 data);

static int ReceiveMMCData(UINT8 command);

static UINT8 ReadMMCDataByte(UINT8 FAR_PNTR * buffer);

static UINT8 ReadMMCFlagsByte(UINT8 FAR_PNTR * buffer);

static void MMCWait(UINT32 waitTime);

static void MMCPrint(char *szFormat,...);

#ifdef TIME_PROCESS
    clock_t timeSend, timeReceive;
    
    float GetTotalSendTime(void)
    {
    float retval = (float)timeSend / (float)CLOCKS_PER_SEC;
    timeSend = 0;
    return retval;
    }
    
    float GetTotalReceiveTime(void)
    {
    float retval = (float)timeReceive / (float)CLOCKS_PER_SEC;
    timeReceive = 0;
    return retval;
    }
#endif

//#define SAVE_MMC_RESULTS
#ifdef SAVE_MMC_RESULTS
	#define SAVEFILE	"era_cmds.out"
	FILE *saveFile;
#endif


#if defined( WATCOM_DOS ) || defined( NLM )
static int GetBIOS32SvcDirEntryPoint( void **ppfnBIOS32SvcDirRet );
static int GetPCIBIOSSvcEntryPoint( void **ppfnPCIBIOSSvcRet, 
                                    void *pfnBIOS32SvcDir );
static int FindPCISMIC();
#endif


#if   !defined ( WATCOM_DOS ) && !defined( NLM ) && !defined( WINNT ) && \
      !defined ( SCO_UNIX ) && !defined ( OS2 ) && !defined ( LINUX )
#include "dosdev.h"
#endif


/**************************************************************************
**  MMCDRVR_OpenDriver -- Discover the hardware and set the In Use flag.
**
**  Description:
**      This function is mapped by the DOS TunnelOpenDriver command as the
**      OpenDriver function.  It performs the required hardware discovery
**      and driver initialization so that the subsequent calls can access
**      the MMC.
**
**  Parameters: None
**  Globals Used:
**      g_dataRegisterAddresses, g_dataRegister, g_controlStatusRegister, 
**      g_flagRegister, g_driverInUse
**  Globals Modified:
**      g_dataRegister, g_controlStatusRegister, g_flagRegister, g_driverInUse
**
**************************************************************************/
int MMCDRVR_OpenDriver(void)
{
   int error = EINUSE;
   if ( g_driverInUse ) goto AllDone;
   if ( g_dataRegister )
   {
      g_driverInUse = TRUE;
      error = MMC_SUCCESS;
      goto AllDone;
   } // if

   // Try finding the ISA SMIC first
   error = ENOHARDWARE;
   (void) OutputByte( g_dataRegisterAddresses[ 0 ], 0xAA );
   (void) InputByte( g_dataRegisterAddresses[ 0 ] + 1 );
   if ( InputByte( g_dataRegisterAddresses[ 0 ] ) == 0xAA )
   {
      (void) OutputByte( g_dataRegisterAddresses[ 0 ], 0x55 );
      (void) InputByte( g_dataRegisterAddresses[ 0 ] + 1 );

      if ( InputByte( g_dataRegisterAddresses[ 0 ] ) == 0x55 )
      {
         g_dataRegister = g_dataRegisterAddresses[ 0 ];
         g_controlStatusRegister = g_dataRegisterAddresses[ 0 ] + 1;
         g_flagRegister = g_dataRegisterAddresses[ 0 ] + 2;
         g_driverInUse = TRUE;
         error = MMC_SUCCESS;
         goto AllDone;
      } /* end if the data register sticks to 0x55 */
   } /* end if the data register sticks to 0xAA */

   // Now try finding the PCI SMIC
#if defined( WATCOM_DOS ) || defined( NLM )
   error = FindPCISMIC();
#endif

#if   !defined ( WATCOM_DOS ) && !defined( NLM ) && !defined( WINNT ) && \
      !defined ( SCO_UNIX ) && !defined ( OS2 ) && !defined( LINUX )
   error = MMCDRVR_DoPciDiscover();
#endif

   if (!error)
   {
       if ( g_dataRegister )
       {
	   g_controlStatusRegister = g_dataRegister + 1;
	   g_flagRegister = g_dataRegister + 2;
	   (void) OutputByte( g_dataRegister, 0xAA );
	   (void) InputByte( g_controlStatusRegister  );
	   if ( InputByte( g_dataRegister ) == 0xAA )
	   {
	       (void) OutputByte( g_dataRegister, 0x55 );
	       (void) InputByte( g_controlStatusRegister );
	       if ( InputByte( g_dataRegister ) == 0x55 )
	       {
		   g_driverInUse = TRUE;
		   error = MMC_SUCCESS;
		   goto AllDone;
	       } /* end if g_dataRegister sticks to 0x55 */
	   } /* end if g_dataRegister sticks to 0xAA */
       } /* end if g_dataRegister has a value */
   } /* end if PCI search turned up an MMC */
   /* A successful path hasn't been followed -- reset all */
   error = ENOHARDWARE;
   g_driverInUse = FALSE;
   g_dataRegister = 0;
   g_controlStatusRegister = 0;
   g_flagRegister = 0;

AllDone:
   return error;

} /* end MMCDRVR_OpenDriver() */


/**************************************************************************
**  MMCDRVR_CloseDriver -- Clears the In Use flag.
**
**  Description:
**      This is the complement to MMCDRVR_OpenDriver, freeing the device
**      driver for use by another process.
**
**  Parameters: None
**  Globals Used:
**      g_driverInUse
**  Globals Modified:
**      g_driverInUse
**
**************************************************************************/
int MMCDRVR_CloseDriver(void)
{
    if(g_driverInUse)
    { 
        g_driverInUse = FALSE;
        return MMC_SUCCESS;
    }
    else
        return EDEVDRV;
} /* end MMCDRVR_CloseDriver(void) */


/**************************************************************************
**  MMCDRVR_SendRcvMessage -- Manages message sending and receiving
**
**  Description:
**      This function defines the overall message sending and receiving
**      strategy -- the number of receive retries, the delay between
**      retries, etc.
**
**  Parameters:
**      UINT8* sbuffer          Message buffer containing data built up by
**                              the MMC_SendRcvMsg function
**      UINT8 slength           number of bytes of data pointed to by 
**                              sbuffer
**      UINT8* rbuffer          Message buffer to be filled with the result
**                              of the process
**      UINT8* rlength          length buffer to be filled with the number
**                              of bytes being returned as the result
**
**  Globals Used: None
**
**  Globals Modified: None
**
**************************************************************************/
int MMCDRVR_SendRcvMessage(UINT8 FAR_PNTR * sbuffer, UINT8 slength, 
    UINT8 FAR_PNTR * rbuffer, UINT8 FAR_PNTR * rlength)
{
    int retval = ETIMEOUT;
    int bytesReturned = 0;
    int count = 0;
    #ifdef TIME_PROCESS 
    clock_t oldClock;
    #endif
    
    #ifdef SAVE_MMC_RESULTS
    int i;
    char sStream[160], rStream[160];
    char textBuffer[80];
    #endif
    
    MMCPrint("\nEntered MMCDRVR_SendRcvMessage");
    
    if (!rlength)
    {
	MMCPrint("\nError: No return buffer size buffer passed");
        return EDEVDRV;
    }

    #ifdef TIME_PROCESS
    oldClock = clock();
    #endif
    if (!SendMessage(sbuffer, slength))
    {
        #ifdef TIME_PROCESS
        timeSend += clock() - oldClock;
        #endif
    
        
        MMCPrint("\nSendMessage completed.");
        
        /* Get the response */
        
        while ((bytesReturned < 1) && (count < DRIVERRECEIVERETRIES))
        {
            #ifdef TIME_PROCESS
            oldClock = clock();
            #endif
            bytesReturned = ReceiveMessage(rbuffer, *rlength);
            #ifdef TIME_PROCESS
            timeReceive += clock() - oldClock;
            #endif
            if (bytesReturned == 0)
            {
                /* Didn't receive a message before timing out. */                
                MMCPrint("\nReceiveMessage timed out.");                
                MMCWait(RECEIVE_WAIT);
            }
            else if (bytesReturned < 0)
            {
                /* Some error message was signaled by the code */
                MMCPrint("\nReceiveMessage indicated a failure.");
                retval = EABORTED;
                break; /* break out of the while */
            }
            else
            {
                /* Successfully received bytesReturned bytes of data */
				#ifdef SAVE_MMC_RESULTS
                	memset(sStream, 0, sizeof(sStream));
                	memset(rStream, 0, sizeof(rStream));
                	for (i = 0; i < slength; i++)
                	{
	                	sprintf(textBuffer, "%2x ", sbuffer[i]);
	                	strcat(sStream, textBuffer);
                	}

                	for (i = 0; i < bytesReturned; i++)
                	{
	                	sprintf(textBuffer, "%2x ", rbuffer[i]);
	                	strcat(rStream, textBuffer);
                	}

					saveFile = fopen(SAVEFILE, "a");
					if (saveFile)
					{
						fprintf(saveFile, "\nMMCDRVR_SendRcvMessage -- "
	                		 "\nsbuffer[] = %s"
	                		 "\nslength = %u"
	                		 "\nrbuffer[] = %s"
	                		 "\n*rlength = %u"
	                		 "\nbytesReturned = %u", sStream, slength, 
	                		 rStream, *rlength, bytesReturned);
	                	fclose(saveFile);
					}
				#endif

                if (*rlength < (UINT8) bytesReturned)
                    retval = EBUFFER;
                else
                    retval = MMC_SUCCESS;
                
                *rlength = (UINT8) bytesReturned;
            }
            count++;
        } /* end while (we haven't received anything and we haven't retried */
          /* too many times) */
    } /* end if (SendMessage returned an OK) */
    else
    {
        *rlength = 0;
        retval = EDEVDRV;
    } /* end else (SendMessage failed) */
    
    return retval;
} /* end MMCDRVR_SendRcvMessage() */

/**************************************************************************
**  MMCDRVR_GetVersionNumber -- Retrieves the driver version number
**
**  Description:
**      This function corresponds to the "MMC_GetKernelDriverVersion"
**      function in the MMC API.  It retrieves the version information
**      specific to the "kernel" layer of the driver.  For NT for example
**      the version information for the MMC API DLL can be different from
**      that of this driver.
**
**  Parameters:
**      char* oem_name              char[8] buffer always filled with "HP"
**      char* version_letter        one letter designating version number
**      UINT8* version_major        byte indicating major version number
**      UINT8* version_minor        byte indicating minor version number
**
**  Globals Used: None
**
**  Globals Modified: None
**
**************************************************************************/
int MMCDRVR_GetVersionNumber(char FAR_PNTR * oem_name, 
    char FAR_PNTR * version_letter, UINT8 FAR_PNTR * version_major,
    UINT8 FAR_PNTR * version_minor)
{
    int retval = MMC_SUCCESS;
    UINT8 i = 0; 
    
    for (i=0; i < OEM_NAME_LENGTH; i++)
       oem_name[i] = OEM_NAME[i];
    oem_name[i] = '\0';

    *version_letter = KERNEL_VERSION_LETTER;
    *version_major = KERNEL_VERSION_MAJOR;
    *version_minor = KERNEL_VERSION_MINOR;
    return retval;
} /* end MMCDRVR_GetVersionNumber() */


/*************************************************************************/
/**  START LOCAL FUNCTION DEFINITIONS                                   **/
/*************************************************************************/


/**************************************************************************
**  SendMessage -- Manages sending a stream of bytes
**
**  Description:
**      This function defines a state machine which will manage the 
**      transfer of the contents of a data buffer to the MMC hardware.
**      It makes sure the "frame" is built up, sending the appropriate
**      command codes.
**
**  Parameters:
**      UINT8* inputbufferptr       Points to buffer of data to be sent
**      commandlength               Number of bytes in the buffer
**
**  Returns:
**      0 indicates the message was sent successfully.
**      Any other value indicates an error.
**
**  Globals Used: None
**
**  Globals Modified: None
**
**************************************************************************/
static int SendMessage(UINT8 FAR_PNTR * inputbufferptr, UINT8 commandlength)
{
    int count = 1;
    int retval = 0;
    UINT8 FAR_PNTR * currentbyte;
    int sendretval;

    currentbyte = inputbufferptr;
    
    sendretval = SendMMCData(CC_SMS_WR_START, *currentbyte);
    currentbyte++;
    
    /* Send the first byte to the MMC */
    if (sendretval != SC_SMS_WR_START)
    {
	MMCPrint("\ndid not receive SC_SMS_WR_START");
        retval =  GENERIC_ERROR;
    }
    
    /* Now send all but the last */
    else
    {
        while (count < (commandlength - 1))
        {
            sendretval = SendMMCData(CC_SMS_WR_NEXT, *currentbyte);
            
            currentbyte++;
            count++;
            if (sendretval != SC_SMS_WR_NEXT)
            {            
		MMCPrint("\ndid not receive SC_SMS_WR_NEXT");
                retval = GENERIC_ERROR;
                break;
            }
        } 
        
        /* Send the last byte */
        if (!retval)
        {
            sendretval = SendMMCData(CC_SMS_WR_END, *currentbyte);
            if (sendretval != SC_SMS_WR_END)
            {            
		MMCPrint("\ndid not receive SC_SMS_WR_END");
                retval = GENERIC_ERROR;     
            }
        }
    } /* else we received the SC_SMS_WR_START message */
    return retval;
} /*end SendMessage() */


/**************************************************************************
**  ReceiveMessage -- Manages receiving a stream of bytes
**
**  Description:
**      This function defines a state machine which will manage the 
**      transfer of the contents of a data buffer from the MMC hardware.
**      It fills the buffer passed to it with the results of a request
**      sent by SendMessage by retrieving and managing the framing 
**      information from the MMC.
**
**  Parameters:
**      UINT8* buffer       Points to buffer which is to be filled
**      int len             Size of the buffer in bytes
**
**  Returns:
**      Number of bytes retrieved from the MMC.  Zero indicates the MMC
**      didn't have data.
**
**  Globals Used: None
**
**  Globals Modified: None
**
**************************************************************************/
static int ReceiveMessage(UINT8 FAR_PNTR * buffer, UINT8 len)
{
    UINT8 FAR_PNTR * currentbyte;
    int sendretval;
    UINT8 tempbuffer[MAX_BUFFER_SIZE];
    int length = 0;
    int flag = FALSE;
    int count = 0;
    int retval = 0;
    UINT8 i = 0;
    
    MMCPrint("\nReceiveMessage begun, len = %u", len);
    
    /* Wait for the receive data ready flag to go high: */
    while ((!flag) && (count < DRIVERRECEIVERETRIES))
    {
        /* copy any messages to temporary buffer */
        currentbyte = tempbuffer;
    
        /* For a read we need to send the MMC/BMC the prompt that we want */
        /* to pull data from it. */
        sendretval = ReceiveMMCData(CC_SMS_RD_START);
    
        switch (sendretval)
        {
            case SC_SMS_RD_START:
                /* BMC has stuff to transfer: */
                length = 1;
                ReadMMCDataByte(&tempbuffer[0]);
                
                sendretval = ReceiveMMCData(CC_SMS_RD_NEXT);
                while (sendretval == SC_SMS_RD_NEXT)
                {
                    /* Read another byte, and at least one more is */
                    /* coming in */
                    if (length < MAX_BUFFER_SIZE)
                    {
                        ReadMMCDataByte(&tempbuffer[length]);
                        length++;
                            
                    }
                    else
                    {
                        length = RECEIVE_BUFFER_FULL;
                        flag = TRUE; /* fail the big while's conditions */
                        break;  /* break out of the nested while */
                    }
                    sendretval = ReceiveMMCData(CC_SMS_RD_NEXT);
                }

                if (sendretval == SC_SMS_RD_END)
                {
                    /* Last byte -- pull it in. */
                    ReadMMCDataByte(&tempbuffer[length]);
                    length++;
                    if (SendMMCData(CC_SMS_RD_END, 0) != SC_SMS_READY)
                    {
                        // Didn't retrieve the SC_SMS_READY byte --
                    }
                }
                else if ((sendretval == SC_SMS_READY) ||
                    (sendretval == SC_SMM_READY))
                {
                    /* receive aborted for some reason */
                    length = MESSAGE_ABORT;
                    flag = TRUE; /* fail the the big while's conditions */
                    break;  /* break out of the nested while */
                }
                else
                {
                    // Timeout? -- need to retry from the beginning
                    length = 0;
                    break;  /* break out of the switch */
                }
                
                for (i=0; i < __min((UINT8)length, len); i++)
                {
                    buffer[i] = tempbuffer[i];
                }
                    
                flag = TRUE;
                break;
            
            case SC_SMS_RD_END:
                /* BMC had one byte in the receive buffer: */
                length = 1;
                ReadMMCDataByte(&tempbuffer[0]);
                
                if (SendMMCData(CC_SMS_RD_END, 0) != SC_SMS_READY)
                {
                // Didn't retrieve the SC_SMS_READY byte --
                MMCPrint("\nDidn't receive an "
                "SC_SMS_READY after a CC_SMS_RD_END.");
                
                }
                break; /* break out of the switch */
            
            case MESSAGE_ABORT:
                /* Someone has aborted this message -- need to fail the */
                /* entire thing. */
                length = 0;
                flag = TRUE; /* fail the the big while's conditions */
                break;
                
            default:
                length = sendretval;
                count++;
                /* For some reason the BMC didn't return the start byte: */
                
        } /* end of switch on the CommandStatus register contents */
    } /* end of while */
    
    retval = length;
    
    return retval;
} /* end ReceiveMessage() */


/**************************************************************************
**  SendMMCData -- Sends a byte of data with the control code given
**
**  Description:
**      This function sends a byte of data through to the MMC using the
**      Madrona SMIC protocol.  It has timeouts built in which allow a
**      process to continue if there is an error at this level.
**
**  Parameters:
**      UINT8 command       byte to put into the CommandStatus register
**      UINT8 data          byte put into the Data register
**
**  Returns:
**      This function returns the result placed into the CommandStatus
**      register by the MMC after the BUSY bit returns low.  On a timeout
**      it returns zero.
**
**  Globals Used: None
**
**  Globals Modified: None
**
**************************************************************************/
static int SendMMCData(UINT8 command, UINT8 data)
{
    UINT8 flags = 0;
    long count = 0;
    int retval = 0;

    #ifdef ENABLETIMEOUTS
    count = 0;
    while (ReadMMCFlagsByte(NULL) & IF_BUSY && (count < XMITTIMEOUT))
    {
        MMCWait(ONE_MICROSECOND);
        count++;        
    }
    if (count == XMITTIMEOUT)
    {
	MMCPrint("\nSendMMCData timed out");
        return 0; /* Timed out */
    }
    #else
    while (ReadMMCFlagsByte(NULL) & IF_BUSY);    
    #endif
    
    OutputByte(g_controlStatusRegister, command);
    OutputByte(g_dataRegister, data);
    OutputByte(g_flagRegister, flags | IF_BUSY);
    #ifdef ENABLETIMEOUTS
    count = 0;
    while (InputByte(g_flagRegister) & IF_BUSY && (count < XMITTIMEOUT))
    {
        MMCWait(ONE_MICROSECOND);
        count++;
    }
    if (count == XMITTIMEOUT)
    {
        MMCPrint("\nNo data returned");
        return (MESSAGE_ABORT); // no data returned
    }
    #else
    while (InputByte(g_flagRegister) & IF_BUSY);
    #endif
    
    retval = InputByte(g_controlStatusRegister);
       
    MMCPrint("\nSendMMCData read %2x", retval);
    return retval;
} /* end SendMMCData() */


/**************************************************************************
**  ReceiveMMCData -- Retrieves a byte of data using the control code given
**
**  Description:
**      This function retrieves a byte of data from the MMC using the
**      Madrona SMIC protocol.  It has timeouts built in which allow a
**      process to continue if there is an error at this level.
**
**  Parameters:
**      UINT8 command       byte to put into the CommandStatus register
**
**  Returns:
**      This function returns the result placed into the CommandStatus 
**      register by the MMC after the BUSY bit returns low.  On a timeout
**      it returns zero.
**
**  Globals Used: None
**
**  Globals Modified: None
**
**************************************************************************/
static int ReceiveMMCData(UINT8 command)
{
    UINT8 flags;
    int retval = 0;
    long count = 0;

    
    #ifdef ENABLETIMEOUTS
    count = 0;
    while (ReadMMCFlagsByte(NULL) & IF_BUSY &&(count <RECEIVETIMEOUT))
    {
        MMCWait(ONE_MICROSECOND);
        count++;  
    }
    if ( count == RECEIVETIMEOUT)
    {
	MMCPrint("\nReceiveMMCData timed out -- checkpoint 1");
        return 0; // no byte ready -- first checkpoint
    }
    #else
        while (ReadMMCFlagsByte(NULL) & IF_BUSY);
    #endif

    #ifdef ENABLETIMEOUTS
    count = 0;
    while ((!(ReadMMCFlagsByte(&flags) & IF_RX_DATA_READY))
            && (count < RECEIVETIMEOUT))
    {
        MMCWait(ONE_MICROSECOND);
        count++;
    }

    if (count == RECEIVETIMEOUT)
    {
	MMCPrint("\nReceiveMMCData timed out -- checkpoint 2");
        return 0; // no byte ready -- second checkpoint
    }
    #else
    while (!(ReadMMCFlagsByte(&flags) & IF_RX_DATA_READY));
    #endif
    
    OutputByte(g_controlStatusRegister, command);
    OutputByte(g_flagRegister, flags | IF_BUSY);
    #ifdef ENABLETIMEOUTS
    count = 0;
    while (ReadMMCFlagsByte(NULL) & IF_BUSY && (count < RECEIVETIMEOUT))
    {
        MMCWait(ONE_MICROSECOND);
        count++;
    }
    if (count == RECEIVETIMEOUT)
    {
	MMCPrint("\nReceiveMMCData timed out -- checkpoint 3");
        return (MESSAGE_ABORT); /* no byte ready -- since we */
            /* have already sent a byte */
            /* this is a bad sign */
    }
    #else
    while (ReadMMCFlagsByte(NULL) & IF_BUSY);
    #endif            

    retval = InputByte(g_controlStatusRegister);        

    return retval;
} /* end ReceiveMMCData() */


/**************************************************************************
**  ReadMMCDataByte -- Retrieves the contents of the MMC's Data register
**
**  Description:
**      This function is used to get the value of the Data register.  It
**      can fill in a buffer if passed in, and will always return the
**      contents.
**
**  Parameters:
**      UINT8* buffer       buffer to be filled in -- NULL means don't save
**                          the value retrieved.
**
**  Returns:
**      This function returns the byte in the MMC's Data register.
**
**  Globals Used: g_dataRegister
**
**  Globals Modified: None
**
**************************************************************************/
static UINT8 ReadMMCDataByte(UINT8 FAR_PNTR * buffer)
{
    UINT8 retval;
    retval = InputByte(g_dataRegister);
    if (buffer)
        *buffer = retval;
    return (retval);
} /* end ReadMMCDataByte() */


/**************************************************************************
**  ReadMMCFlagsByte -- Retrieves the contents of the MMC's Flags register
**
**  Description:
**      This function is used to get the value of the Flags register.  It
**      can fill in a buffer if passed in, and will always return the
**      contents.
**
**  Parameters:
**      UINT8* buffer       buffer to be filled in -- NULL means don't save
**                          the value retrieved.
**
**  Returns:
**      This function returns the byte in the MMC's Flags register.
**
**  Globals Used: g_flagRegister
**
**  Globals Modified: None
**
**************************************************************************/
static UINT8 ReadMMCFlagsByte(UINT8 FAR_PNTR * buffer)
{
    UINT8 retval;
    retval = InputByte(g_flagRegister);
    if (buffer)
        *buffer = retval;
    return retval;
} /* end ReadMMCFlagsByte() */

/**************************************************************************
**  MMCWait -- implements a delay
**
**  Description:
**      This function is used to generate a delay of the requested length
**
**  Parameters:
**      UINT32 waitTime     number of 1 microsecond increments to wait
**
**  Returns:
**      Always returns zero.
**
**  Globals Used: None
**
**  Globals Modified: None
**
**************************************************************************/
static void MMCWait(UINT32 waitTime)
{   
    for (; waitTime > 0; waitTime--)
        DelayByte();
 } /* end MMCWait() */

/**************************************************************************
**  MMCPrint -- prints a message, NOS dependent
**
**  Description:
**      This function prints the input message to the screen or debugger
**      if the DEBUG is defined.
**
**  Parameters:
**      
**
**  Returns:
**      Nothing.
**
**  Globals Used: None
**
**  Globals Modified: None
**
**************************************************************************/
static void MMCPrint(char *szFormat,...)
{
#ifdef _DEBUG
   char szBuffer[256];
   char *pArguments;
   int bytesWritten;

   pArguments = (char *) &szFormat + sizeof(szFormat);
   vsprintf(szBuffer, szFormat, pArguments);

   #ifdef MS_DOS
      printf(szBuffer);
   #endif /* MS_DOS */
   #ifdef NLM
      printf(szBuffer);
   #endif /* NLM */
   #ifdef SCO_UNIX
      printf(szBuffer);
   #endif /* SCO_UNIX */
   #ifdef WINNT
      KdPrint((szBuffer));
   #endif /* endif WINNT */

/* no action taken if OS2 */
      
#endif /*DEBUG*/
} /* MMCPrint */


#if defined( WATCOM_DOS ) || defined( NLM )
/**************************************************************************
**  GetBIOS32SvcDirEntryPoint
**
**  Description:
**       This function scans the SM BIOS tables (starting at physical
**       address 0xE0000) for the BIOS32 header.  If the header is found,
**       *ppfnBIOS32SvcDirRet is set to the BIOS32 Service Directory
**       entry point.  Otherwise, an error is returned.
**
**  Parameters: 
**    ppfnBIOS32SvcDirRet
**       A pointer to a function pointer where the
**       BIOS32 Service Directory entry point will be
**       stored
**
**  Globals Used:
**      None
**  Globals Modified:
**      None
**
**************************************************************************/
static int GetBIOS32SvcDirEntryPoint( void **ppfnBIOS32SvcDirRet )
{
   int error = ENOHARDWARE;
   char rgbHeader[] = "_32_";
   UINT8 *pbBIOSData = (UINT8 *) 0xE0000;
   UINT8 *pbBIOS32Struct = NULL;
   UINT8 bChecksum = 0;
   int ibPage;
   int ibBIOS32Struct = 0;

   if ( !ppfnBIOS32SvcDirRet ) goto AllDone;
   *ppfnBIOS32SvcDirRet = NULL;
   // Check for 32-bit BIOS
   // Scan for "_32_" header within 128k starting at 0xE0000
   // The "_32_" should start on a 16-byte boundary
   for ( ibPage = 0; 
         (  ( ibPage < 0x20000 ) && 
            ( memcmp( &pbBIOSData[ ibPage ], rgbHeader, 4 ) != 0 ) ); 
         ibPage += 0x10 );
   if ( ibPage > 0x1FFF0 ) goto AllDone;

   // Validate the checksum (which is stored in offset 0x0a)
   pbBIOS32Struct = &pbBIOSData[ ibPage ];
   for ( ibBIOS32Struct = 0;
         ibBIOS32Struct <= 0x0a;
         ibBIOS32Struct++ )
   {
      bChecksum += pbBIOS32Struct[ ibBIOS32Struct ];
   } // for
   if ( bChecksum != 0 ) goto AllDone;

   // Get 32-bit BIOS service directory program entry point
   *ppfnBIOS32SvcDirRet = (void *) *( (UINT32 *) &pbBIOSData[ ibPage + 4 ] );
   if ( !( *ppfnBIOS32SvcDirRet ) ) goto AllDone;
   error = MMC_SUCCESS;   

AllDone:
   return error;
} // GetBIOS32SvcDirEntryPoint


/**************************************************************************
**  GetPCIBIOSSvcEntryPoint
**
**  Description:
**       This function calls the BIOS32 Service Directory entry point
**       to obtain the PCI BIOS Service entry point (if it exists).
**       If the PCI BIOS Service entry does exist, it is stored in
**       *ppfnPCIBIOSSvcRet.  The PCI BIOS Service entry point is then
**       called to determine if the PCI BIOS is present.  If the PCI
**       BIOS Service entry point cannot be obtained, or the PCI BIOS
**       is not present, then an error will be returned.
**
**  Parameters: 
**    ppfnPCIBIOSSvcRet
**       A pointer to a function pointer where the
**       PCI BIOS Service entry point will be
**       stored
**
**    pfnBIOS32SvcDir
**       The function pointer for the BIOS32 Service
**       Directory entry point (obtained through
**       GetBIOS32SvcDirEntryPoint)
**
**  Globals Used:
**      None
**  Globals Modified:
**      None
**
**************************************************************************/
static int GetPCIBIOSSvcEntryPoint( void **ppfnPCIBIOSSvcRet, 
                                    void *pfnBIOS32SvcDir )
{
   int error = ENOHARDWARE;
   UINT32 dwPCIBIOSSvcStartAddress = 0;
   void *pfnPCIBIOSSvc = NULL;

   if ( !ppfnPCIBIOSSvcRet || !pfnBIOS32SvcDir ) goto AllDone;
   // Determine whether or not the PCI BIOS exists.  If it does exist,
   // then the PCI BIOS Service entry point will be obtained.
   _asm
   {
      pushad
      pusha
      pushf
      pushfd

      mov   eax,49435024h     // eax = "$PCI"
      xor   ebx,ebx           // ebx = 0

      mov   ecx,pfnBIOS32SvcDir

      push  cs                // far call
      call  ecx

      and   eax,000000FFh
      mov   error,eax

      mov   pfnPCIBIOSSvc,ebx
      mov   dwPCIBIOSSvcStartAddress,edx

      popfd
      popf
      popa
      popad
   }; // _asm
   if ( error ) goto AllDone;
   if ( !pfnPCIBIOSSvc )
   {
      error = ENOHARDWARE;
      goto AllDone;
   } // if

   pfnPCIBIOSSvc =
      (void *) (  (UINT32) pfnPCIBIOSSvc +
                     dwPCIBIOSSvcStartAddress );

   // Check for PCI BIOS presence
   _asm
   {
      pushad
      pusha
      pushf
      pushfd

      xor   eax,eax
      mov   al,01h
      mov   ah,0B1h

      mov   ecx,pfnPCIBIOSSvc

      push  cs
      call  ecx
      jc    PCIBIOSNotPresent

      cmp   edx,20494350h  // "PCI "
      jnz   PCIBIOSNotPresent

      and   eax,0000FF00h  // AH contains status
      mov   error,eax
      jmp   Done

   PCIBIOSNotPresent:
      mov   error,ENOHARDWARE

   Done:
      popfd
      popf
      popa
      popad
   }; // _asm
   if ( error ) goto AllDone;
   error = MMC_SUCCESS;
   *ppfnPCIBIOSSvcRet = pfnPCIBIOSSvc;

AllDone:
   return error;
} // GetPCIBIOSSvcEntryPoint


/**************************************************************************
**  FindPCISMIC
**
**  Description:
**       This function obtains the BIOS32 Service Directory
**       entry point and uses it to obtain the PCI BIOS Service entry
**       point (through calls to GetBIOS32SvcDirEntryPoint and 
**       GetPCIBIOSSvcEntryPoint).  If the PCI BIOS Service entry point
**       was successfully obtained, then this function uses it to scan for
**       the PCI SMIC.  If any of the above steps fail, an error is returned.
**       Otherwise, the g_dataRegister variable is set to the base address
**       of the PCI SMIC.
**
**  Parameters: 
**    None
**
**  Globals Used:
**      None
**  Globals Modified:
**      g_dataRegister
**
**************************************************************************/
static int FindPCISMIC()
{
   int error = ENOHARDWARE;
   void *pfnBIOS32SvcDir = NULL;
   void *pfnPCIBIOSSvc = NULL;

   g_dataRegister = 0;
   error = GetBIOS32SvcDirEntryPoint( &pfnBIOS32SvcDir );
   if ( error ) goto AllDone;
   error = GetPCIBIOSSvcEntryPoint( &pfnPCIBIOSSvc, pfnBIOS32SvcDir );
   if ( error ) goto AllDone;

   // Scan for a PCI SMIC using "Find PCI class code" function in
   // the PCI BIOS.  One device is found, get base address by calling
   // "Read Configuration Word" function on address 0x10 (which is
   // Base Address 1).
   _asm
   {
      pushad
      pusha
      pushf
      pushfd

      // We should be scanning for the SMIC using PCI class codes.
      // The T7 BIOS did not program the class code for the SMIC
      // at the time this function was written.  So, as a workaround,
      // we will scan for the SMIC using the VendorId/DeviceId pair.
#if 0
      xor   eax,eax
      mov   al,03h
      mov   ah,0B1h
      xor   ecx,ecx
      mov   ecx,000C0600h     // 0C/06/00 = Serial/IPMI/SMIC
      xor   si,si

      mov   edx,pfnPCIBIOSSvc

      push  cs                // far call
      call  edx
#else
      push  ebp
      mov   ebp,pfnPCIBIOSSvc
      
      xor   eax,eax
      mov   al,02h
      mov   ah,0B1h
      xor   ecx,ecx
      mov   cx,121Ah // Device Id (smic should be 121ah)
      xor   edx,edx
      mov   dx,103Ch // Vendor Id (smic should be 103ch)
      xor   si,si
      
      push  cs
      call  ebp
      pop   ebp
#endif // if 0

      jc    DeviceNotFound

      and   eax,0000FF00h
      mov   error,eax
      cmp   error,00000000h
      jnz   DeviceNotFound

      // Read configuration word (PCI register 0x10 which
      // indicates the base address of the device)
      mov   al,9h
      mov   ah,0B1h
      mov   di,0010h

      mov   edx,pfnPCIBIOSSvc
      push  cs
      call  edx    
      jc    DeviceNotFound

      and   cx,0FFFEh         // bit 0 indicates PIO vs Memory mapped
                              // we'll assume PIO for now
      mov   g_dataRegister,cx
      and   eax,0000FF00h
      mov   error,eax
      jmp   Done

   DeviceNotFound:
      mov   error,ENOHARDWARE

   Done:

      popfd
      popf
      popa
      popad      
   }; // _asm
   if ( error ) goto AllDone;
   error = MMC_SUCCESS;

   // The base address for the SMIC is not used.  The data
   // register is base address + 1.
   g_dataRegister++;

AllDone:
   return error;
} // FindPCISMIC

#endif



