#ifdef __KERNEL__

#include <linux/module.h>

#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))

# ifndef MODULE
#  define MODULE
# endif

# include <linux/autoconf.h>
# if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#  define MODVERSIONS
# endif

# ifdef MODVERSIONS
#  include <linux/modversions.h>
# endif

#endif

#include <linux/kmod.h>
#include <linux/proc_fs.h>
#include <linux/errno.h>

#else // shell

#include <stdio.h>
#define	printk	printf
#define	off_t	long
#define	EBUSY	16

#endif

#include "dev_sel.h"
#include "mmc_dev.h"
#include "mmc_func.h"
#include "show_sel.h"
#include "sdrdesc.h"
#include "logtime.h"

#define LIMITE_SIZE 128

#define FIXUP(t)	do {				\
				if ( count > limit) { 	\
					goto t;		\
				}			\
			} while (0)

UINT16 nNextRecID;
short  endfile;

#ifdef __KERNEL__

static struct semaphore open_sem;
static struct file_operations *old_fops;
static struct proc_dir_entry *entry;

#endif

int sel_read_proc(char *buf, char **start, off_t offset, int size, int *eof, void *data) {
	int         nReturnCode = HPEL_SUCCESS;
	int         count, limit;
	SEL         selHW;
	UINT16      nReservation, nBeforeRecID;
	const UINT8 nReadBytes = sizeof( SEL );
	const UINT8 nOffset = 0;
	limit = size - LIMITE_SIZE;
	
	if (offset == 0) {
		nNextRecID = FIRST_RECORD;
		endfile    = 0;
		count = sprintf(buf, "System Event Log Device - version: 1.1 (seldev)\n" );
	} else
		count = 0;
	
	if (!endfile) {
		
		FIXUP(Done);
	
		nReturnCode = MMC_OpenDevice();
		if( nReturnCode != HPEL_SUCCESS ) {
			if (nReturnCode == -EBUSY)
				count += sprintf(&buf[count], "\nMMC is in use by other proccess.\n\n" );
			else
				count += sprintf(&buf[count], "\nUnable to open MMC (%04X).\n\n", nReturnCode );			
			goto Done;
		}
		
		nReturnCode = MMCReserveSEL( &nReservation );
		if( nReturnCode != HPEL_SUCCESS ) {
			count += sprintf(&buf[count], "\nUnable to reserve SEL\n\n" );
			goto AllDone;
		}
		
		/* header 
Index  Date       Time      Severity       Description
-----  ---------- --------  -------------  -------------------------------------
    #  mm-dd-yyyy hh:mm:ss  Informational  XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
		*/  
	
		/* data */
		while (nNextRecID != LAST_RECORD) {
			nBeforeRecID = nNextRecID;
			
			nReturnCode = MMCGetSELEntry( nReservation,  nNextRecID, nOffset, nReadBytes, &nNextRecID, &selHW );
			if( nReturnCode != HPEL_SUCCESS ) {
				if (nBeforeRecID == FIRST_RECORD)
					count += sprintf( &buf[count], "\nEvent Log is empty.\n" );
				else
					count += sprintf( &buf[count], "\nCannot get missing SEL entry 0x%.4x\n", nNextRecID );
				goto AllDone;
			}
			
			if (nBeforeRecID == FIRST_RECORD) {
				count += sprintf( &buf[count], "\nIndex  Date       Time      Severity       Description" );
				count += sprintf( &buf[count], "\n-----  ---------- --------  -------------  -------------------------------------\n" );
			}
			//selHW.timestamp = ConvertHWLogTime( (time_t)selHW.timestamp );
			count += SEL_ShowEvent( buf, (nBeforeRecID+1), &selHW, count );
			//count += sprintf(&buf[count], "%5d  ---------- --------  -------------  ---[%p] %p\n", nBeforeRecID, &data, data );
			FIXUP(AllDone);
		}
	
AllDone:
		MMC_CloseDevice();
	
Done:
		*start = buf;
		if ((nNextRecID == LAST_RECORD) || (nReturnCode != HPEL_SUCCESS)) {
			nNextRecID = FIRST_RECORD;
			endfile = 1;
			*eof = endfile;
		}
	} else {
		endfile = 0;
		*eof = 1;
		HPEL_SdrCleanup();
	}
	return count;
}


#ifdef __KERNEL__

static ssize_t sel_read(struct file *file, char *buf, size_t count, loff_t *ppos) {
	if (!old_fops) {
		printk("<1> SELDEV: The read function has not been defined.\n");
		return 0;
	}
	return old_fops->read(file, buf, count, ppos);
}

static int sel_open(struct inode *inode, struct file *filp) {
	if (down_interruptible(&open_sem))
		return -ERESTARTSYS;
	return 0;
}

static int sel_release(struct inode *inode, struct file *filp) {
	up(&open_sem);
	return 0;
}

static struct file_operations sel_file_operations = {
        read: sel_read,
        open: sel_open,
        release: sel_release
};


int init_module(void) {

	printk("<1> SELDEV module loaded.\n" );
	
	entry = create_proc_entry("sel", 0, NULL);
	if (entry) {
		sema_init(&open_sem, 1);
		entry->mode = S_IFREG | S_IRUGO;
		entry->nlink = 2;
		entry->read_proc = sel_read_proc;
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
		old_fops = entry->ops->default_file_ops;
		entry->ops->default_file_ops = &sel_file_operations;
#else
		old_fops = entry->proc_fops;
		entry->proc_fops = &sel_file_operations;
#endif
		return 0;
	} else {
		return 1;
	}
}

void cleanup_module(void){
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
	entry->ops->default_file_ops = old_fops;
#endif
	remove_proc_entry("sel",NULL);
	printk("<1> unloaded SELDEV module.\n" );
}

#else // shell

int main() {
	int eof=0;
	char saida[4096];
	char *start;
	off_t offset;
	
	do {
		start  = NULL;
		offset = 0;
		offset += sel_read_proc( saida , &start, offset, sizeof(saida), &eof, NULL);
		printk(saida);
		if (eof) {
			offset += sel_read_proc( saida , &start, offset, sizeof(saida), &eof, NULL);
		}
	} while (!eof);		
	
	return 1;
}

#endif
