#ifdef __KERNEL__

#ifdef __NO_VERSION__
#error __NO_VERSION__ must not be defined
#endif

#include <linux/version.h>

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))

# ifndef __KERNEL__
#  define __KERNEL__
# endif

# ifndef MODULE
#  define MODULE
# endif

# include <linux/autoconf.h>
# if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
#  define MODVERSIONS
# endif

# ifdef MODVERSIONS
#  include <linux/modversions.h>
# endif

#endif

#include <linux/slab.h>
#include <asm/string.h>

#else // shell 

#include <malloc.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#define	printk		printf
#define	GFP_KERNEL	0
#define	kmalloc(a,b)	malloc(a)		
#define	kfree(a)	free(a)

#endif

#include "hpelint.h"
#include "mmcapi.h"
#include "sdrdesc.h"

static SDR *g_pSdrTable     = NULL;
static unsigned int g_nSdrs = 0;
static int g_fTableError    = FALSE;

/*
** BuildSharedSdr:
**
** Modify the provided shared SDR according to iSdr.
**
** INPUTS:
**    iSdr is a 0-based index used to modify the necessary fields
**       in pSdr
**    pSdr is a pointer to the original unmodified shared SDR
**
** OUTPUTS:
**    *pSdr is modified according to iSdr
*/
static void BuildSharedSdr( unsigned int iSdr, SDR *pSdr ) {
	unsigned int Offset = 0;
	unsigned int cbDescription = 0;
	unsigned int ibDescription = 0;
	
	if ( !pSdr )
		goto AllDone;
	if ( (pSdr->recordType != SDR09_RT_DIGITAL_DISCRETE) || pSdr->recordType != SDR10_RT_COMPACT_SENSOR_RECORD )
		goto AllDone;
	switch ( pSdr->sdrVersion ) {
		case 0x10:
		case 0x90:
			pSdr->body.t09_02.sensorNumber += iSdr;
			if ( pSdr->body.t09_02.sensorRecordSharing & 0x8000 )
				pSdr->body.t09_02.entityId += iSdr;
			Offset = (((pSdr->body.t09_02.sensorRecordSharing) >> 8) & 0x7f);
			cbDescription = pSdr->body.t09_02.nameTypeLength & 0x1f;
			ibDescription = cbDescription;
			if ( ibDescription >= 16 )
				ibDescription = 15;
			if ( pSdr->body.t09_02.sensorRecordSharing & 0x0010 ) {
				// Only support one letter (A-Z)
				if ( (Offset + iSdr) >= 26 )
					goto AllDone;
				pSdr->body.t09_02.name[ ibDescription ] = (Offset + iSdr + 'A');
			} else  {
				if ( (Offset + iSdr) >= 10 ) {
					char szDesc[10];
					
					if ( ibDescription >= 15 ) ibDescription = 14;
						sprintf( szDesc, "%.2d", (Offset + iSdr) );
					pSdr->body.t09_02.name[ ibDescription ] = szDesc[0];
					pSdr->body.t09_02.name[ ibDescription + 1 ] = szDesc[1];
					if ( cbDescription <= 14 ) {
						pSdr->body.t09_02.nameTypeLength += 1;
						pSdr->recordLength += 1;
					}
				} else  {
					pSdr->body.t09_02.name[ ibDescription ] = (Offset + iSdr + '0');
				}
			}
			if ( cbDescription >= 16 )
				goto AllDone;
			pSdr->body.t09_02.nameTypeLength += 1;
			pSdr->recordLength += 1;
			break;
		case 0x01:
			pSdr->body.t10_02.sensorNumber += iSdr;
			if ( pSdr->body.t10_02.sensorRecordSharing[1] & 0x80 )
				pSdr->body.t10_02.entityId += iSdr;
			Offset = pSdr->body.t10_02.sensorRecordSharing[1] & 0x7f;
			cbDescription = pSdr->body.t10_02.nameTypeLength & 0x1f;
			ibDescription = cbDescription;
			if ( ibDescription >= 16 )
				ibDescription = 15;
			if ( pSdr->body.t10_02.sensorRecordSharing[0] & 0x10 ) {
				// Only support one letter (A-Z)
				if ( (Offset + iSdr) >= 26 ) goto AllDone;
					pSdr->body.t10_02.name[ ibDescription ] = (Offset + iSdr + 'A');
			} else  {
				if ( (Offset + iSdr) >= 10 ) {
					char szDesc[10];
					
					if ( ibDescription >= 15 ) ibDescription = 14;
						sprintf( szDesc, "%.2d", (Offset + iSdr) );
					pSdr->body.t10_02.name[ ibDescription ] = szDesc[0];
					pSdr->body.t10_02.name[ ibDescription + 1 ] = szDesc[1];
					if ( cbDescription <= 14 ) {
						pSdr->body.t10_02.nameTypeLength += 1;
						pSdr->recordLength += 1;
					}
				} else  {
					pSdr->body.t10_02.name[ ibDescription ] = (Offset + iSdr + '0');
				}
			}
			if ( cbDescription >= 16 )
				goto AllDone;
			pSdr->body.t10_02.nameTypeLength += 1;
			pSdr->recordLength += 1;
			break;
		default:
			break;
	}
	
AllDone:
	return;
}

struct ListSDR {
	int pos;
	SDR *sSdr;
	struct ListSDR *next;
};

#define NEW_SDR_NODO( a )	a = (struct ListSDR*)kmalloc( sizeof(struct ListSDR), GFP_KERNEL );	\
				if ( !a ) {								\
					error = 1;							\
					goto AllDone;							\
				}									\
				a->pos  = g_nSdrs++;							\
				a->sSdr = NULL;								\
				a->next = NULL;								\
				if ( !StartSDR ) {							\
					StartSDR = a;							\
				} else {								\
					EndSDR->next = a;						\
				}									\
				EndSDR = a


#define NEW_SDR( a )		a = (SDR*) kmalloc( sizeof(SDR), GFP_KERNEL );				\
				if ( !a ) {								\
					error = 1;							\
					goto AllDone;							\
				} while(0)



static void BuildSdrTable() {
	int error = 0;
	unsigned int iSdrTable = 0;
	UINT16 wReservation;
	UINT16 wRecordId = 0;
	UINT16 wNextRecordId = 0;
	UINT8 bOffset;
	UINT8 bBytesRead;
	UINT8 bBytesToRead;
	SDR *SharedSdr;
	SDR *CurrentSdr;
	struct ListSDR *StartSDR, *EndSDR, *NodoSDR;
	
	unsigned int ShareCount = 0;
	unsigned int iShareCount = 0;
	
	StartSDR = NULL;
	EndSDR   = NULL;
	NodoSDR  = NULL;

	if ( g_pSdrTable || g_nSdrs )
		goto AllDone;
	if ( g_fTableError )
		goto AllDone;
		
	error = MMC_ReserveSDRRepository( &wReservation );
	if ( error )
		goto AllDone;
		
	while ( wRecordId != LAST_RECORD ) {
		
		NEW_SDR_NODO( NodoSDR );
		NEW_SDR( CurrentSdr );
		NodoSDR->sSdr = CurrentSdr;
			
		/* geting its record header */
		bOffset = 0;
		error = MMC_GetSDR( wReservation, wRecordId, bOffset, sizeof(SDR_RECORD_HEADER), &wNextRecordId, CurrentSdr );
		if ( error )
			goto AllDone;
		
		/* geting its record data */
		bBytesRead = 0;
		bOffset = sizeof(SDR_RECORD_HEADER);
		while ( bBytesRead < CurrentSdr->recordLength ) {
			bBytesToRead = CurrentSdr->recordLength - bBytesRead;
			if ( bBytesToRead > 16 )
				bBytesToRead = 16;
			error = MMC_GetSDR( wReservation, wRecordId, bOffset, bBytesToRead, &wNextRecordId, CurrentSdr );
			if ( error )
				goto AllDone;
			bBytesRead += bBytesToRead;
			bOffset += bBytesToRead;
		}
		
		/* evalueting the necessity to replay the structure */	
		// Only store SDRs that describe sensors (type 01h and 02h)
		// same for both IPMI 0.9 and 1.0
		if ( CurrentSdr->recordType == 0x02 ) {
			ShareCount = 0;
			switch ( CurrentSdr->sdrVersion ) {
				case 0x10:
				case 0x90:
					ShareCount = (CurrentSdr->body.t09_02.sensorRecordSharing & 0x000f);
					break;
				case 0x01:
					ShareCount = (CurrentSdr->body.t10_02.sensorRecordSharing[0] & 0x0f);
					break;
			}
			if ( ShareCount > 1 ) {
				iShareCount = 0;
				do	{
					SharedSdr = CurrentSdr;
					BuildSharedSdr( iShareCount, SharedSdr );
					iShareCount++;
					if ( iShareCount < ShareCount ) {
						NEW_SDR_NODO( NodoSDR );
						NEW_SDR( CurrentSdr );
						NodoSDR->sSdr = CurrentSdr;
					}
				} while ( iShareCount < ShareCount );
			}
		}
		wRecordId = wNextRecordId;
	}
	error = 0;
	
AllDone:
	
	if ( error ) {
		g_fTableError = TRUE;
		HPEL_SdrCleanup();
	} else {
		g_pSdrTable = (SDR*) kmalloc( g_nSdrs * sizeof(SDR), GFP_KERNEL );
		if ( !g_pSdrTable ) {
			error = 1;
			goto AllDone;
		}
		NodoSDR = StartSDR;
		for (iSdrTable = 0; ((iSdrTable < g_nSdrs) && (NodoSDR)) ; iSdrTable++) {
			memcpy( &g_pSdrTable[ iSdrTable ], NodoSDR->sSdr, sizeof(SDR) );
			NodoSDR = NodoSDR->next;
		}
		g_nSdrs = iSdrTable;
		
	}
	
	/* disallocate all structures allocated in this function */
	while (StartSDR) {
		NodoSDR = StartSDR->next;
		if (StartSDR->sSdr)
			kfree(StartSDR->sSdr);
		kfree(StartSDR);
		StartSDR = NodoSDR;
	}
	return;
}


/*
** HPEL_GetSdrDescription:
**
** Get the SDR description using the generator ID and sensor
** number provided by the SEL entry.
**
** INPUTS:
**    pSelEntry is a pointer to a SEL record
**    pszSdrDescription is a pointer to a buffer where the Sdr
**       description will be written (must be at least 17 bytes long)
**
** OUTPUTS:
**    pszSdrDescription is filled in and terminated with a NULL character
*/
int HPEL_GetSdrDescription( SEL *pSelEntry, char *pszSdrDescription, size_t iSizeDesc ) {
	int error = HPEL_ILLEGAL_PARAMETER;
	UINT8 bOwnerId = 0;
	UINT8 bSensorNumber = 0;
	UINT8 rgbDescription[ 16 ];
	unsigned int iSdrTable = 0;
	unsigned int cbDescription = 0;
	
	if ( !pSelEntry || !pszSdrDescription )
		goto AllDone;
		
	if ( !g_pSdrTable )
		BuildSdrTable();
	
	error = HPEL_GENERAL_ERROR;
	if ( !g_pSdrTable )
		goto AllDone;
	
	while ( iSdrTable < g_nSdrs ) {
		
		switch ( g_pSdrTable[ iSdrTable ].sdrVersion ) {
			case 0x10:
			case 0x90:
				switch ( g_pSdrTable[ iSdrTable ].recordType ) {
					case SDR09_RT_8BIT_THRESHOLD:
						bOwnerId      = (g_pSdrTable[ iSdrTable ].body.t09_01.sensorOwnerId)  & 0xfe;
						bSensorNumber =  g_pSdrTable[ iSdrTable ].body.t09_01.sensorNumber;
						cbDescription = (g_pSdrTable[ iSdrTable ].body.t09_01.nameTypeLength) & 0x1f;
						if ( cbDescription > 16 )
							cbDescription = 16;
						memcpy( rgbDescription, g_pSdrTable[ iSdrTable ].body.t09_01.name, cbDescription );
						error = 0;
						break;
					case SDR09_RT_DIGITAL_DISCRETE:
						bOwnerId      = (g_pSdrTable[ iSdrTable ].body.t09_02.sensorOwnerId)  & 0xfe;
						bSensorNumber =  g_pSdrTable[ iSdrTable ].body.t09_02.sensorNumber;
						cbDescription = (g_pSdrTable[ iSdrTable ].body.t09_02.nameTypeLength) & 0x1f;
						if ( cbDescription > 16 )
							cbDescription = 16;
						memcpy( rgbDescription, g_pSdrTable[ iSdrTable ].body.t09_02.name, cbDescription );
						error = 0;
						break;
					default:
						error = 1;
					break;
				}
				break;
			case 0x01:
				switch ( g_pSdrTable[ iSdrTable ].recordType ) {
					case SDR10_RT_FULL_SENSOR_RECORD:
						bOwnerId      = (g_pSdrTable[ iSdrTable ].body.t10_01.sensorOwnerId)  & 0xfe;
						bSensorNumber =  g_pSdrTable[ iSdrTable ].body.t10_01.sensorNumber;
						cbDescription = (g_pSdrTable[ iSdrTable ].body.t10_01.nameTypeLength) & 0x1f;
						if ( cbDescription > 16 )
							cbDescription = 16;
						memcpy( rgbDescription, g_pSdrTable[ iSdrTable ].body.t10_01.name, cbDescription );
						error = 0;
						break;
					case SDR10_RT_COMPACT_SENSOR_RECORD:
						bOwnerId      = (g_pSdrTable[ iSdrTable ].body.t10_02.sensorOwnerId)  & 0xfe;
						bSensorNumber =  g_pSdrTable[ iSdrTable ].body.t10_02.sensorNumber;
						cbDescription = (g_pSdrTable[ iSdrTable ].body.t10_02.nameTypeLength) & 0x1f;
						if ( cbDescription > 16 )
							cbDescription = 16;
						memcpy( rgbDescription, g_pSdrTable[ iSdrTable ].body.t10_02.name, cbDescription );
						error = 0;
						break;
					default:
						error = 1;
						break;
				}
				break;
			default:
				error = 1;
				break;
		}
		
		if ( !error ) {
			if ( ( (pSelEntry->generatorId & 0xfe) == bOwnerId) && (pSelEntry->sensorNumber == bSensorNumber) ) {
				if ( cbDescription ) {
					if (cbDescription > iSizeDesc)
						cbDescription = iSizeDesc;
					memcpy( pszSdrDescription, rgbDescription, cbDescription );
					pszSdrDescription[ cbDescription ] = 0;
				} else
					strncpy( pszSdrDescription, "No Description", iSizeDesc );
				goto AllDone;
			}
		}
		
		iSdrTable ++;
	}
	error = HPEL_GENERAL_ERROR;
	
AllDone:
	return error;
}


/*
** HPEL_SdrCleanup:
**
** Free any memory that was allocated by building the SDR table used
** be HPEL_GetSdrDescription.  This funciton must be called on application
** exit or memory leaks will occur.
*/
void HPEL_SdrCleanup() {
	if ( g_pSdrTable )
		kfree( g_pSdrTable );
	g_fTableError = FALSE;
	g_pSdrTable   = NULL;
	g_nSdrs       = 0;
	return;
}
