/*
 * (c) Copyright 2001 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

/**************************************************************************
**  Copyright (C) 2000, Hewlett-Packard Co. All rights reserved.
**
**  File name:  main.c
**  Summary:    Linux MMC device driver main file
**  Product:    Motherboard Management Controller
**  Created by: magoga - T&T, based in WinNT implementation
**  Owner:
**
**  START AUTOMATIC VERSION CONTROL INFORMATION
**  DO NOT MANUALLY EDIT THIS SECTION!
**  $Workfile:    $
**  $Revision: 1.16.4.1 $
**  $Date: 2002/04/30 21:50:33 $
**  $Author: rita $
**  END AUTOMATIC VERSION CONTROL INFORMATION
**
**  Description:
**              The driver needs code which will tunnel through the
**              operating system layers. This file provides that.
**
**  OS callable functions:
**              init_module     - called on insmod
**              cleanup_module  - called on rmmod
**              mmcdev_open     - called when user opens /dev/mmcdev
**              mmcdev_close    - called when user closes /dev/mmcdev
**              mmcdev_ioctl    - called when user does a ioctl on /dev/mmcdev
**
***************************************************************************
**  START AUTOMATIC VERSION CONTROL LOG
**  DO NOT MANUALLY EDIT THIS SECTION!
**  $Log: main.c,v $
**  Revision 1.16.4.1  2002/04/30 21:50:33  rita
**  - included macro MODULE_LICENSE("GPL") to define this module as GPL.
**    Note: this change was only tested for Red Hat 7.1/7.2, kernel 2.4.9-31.
**
**  Revision 1.16  2002/01/07 15:47:16  gustavo
**  *** empty log message ***
**
**  Revision 1.4  2002/01/05 15:43:06  gustavo
**  *** empty log message ***
**
**  Revision 1.3  2002/01/04 16:13:29  gustavo
**  *** empty log message ***
**
**  Revision 1.1  2002/01/02 16:19:10  gustavo
**  *** empty log message ***
**
**  Revision 1.14  2001/12/26 13:57:29  gustavo
**
**  added inter module functions registry.
**
**  Revision 1.13  2001/12/20 17:07:48  hamilton
**  Fixed bug in request_region().
**
**  Revision 1.12  2001/12/17 16:18:46  hamilton
**  Fixed for 2.2 compiling.
**
**  Revision 1.11  2001/12/17 15:57:16  hamilton
**  Added support for T11/ST2 SMICs.
**
**  Revision 1.10  2001/07/31 16:08:10  hamilton
**  Fixed bug in spin_lock initialization.
**
**  Revision 1.9  2001/07/27 14:10:03  hamilton
**  - Added support for 2.4.x kernels.
**
**  Revision 1.8  2001/06/27 14:23:40  hamilton
**  - Hack to work on an 2.4.x kernel.
**
**  Revision 1.7  2001/04/24 13:22:06  hamilton
**  - Added copyright notice;
**  - Changed Makefile for compiling in rh70.
**
**  Revision 1.6  2001/02/19 18:55:10  magoga
**  Corrected arguments passing for various ReadPort and WritePort functions
**
**  Revision 1.5  2000/09/08 14:43:37  magoga
**  Debugging messages removed.
**
**  Revision 1.4  2000/09/08 14:37:42  magoga
**  Bugs related to PCI SMIC corrected.
**
**  Revision 1.3  2000/08/30 21:21:06  magoga
**  Module initialization and finalization improved.
**
**  Revision 1.2  2000/08/21 22:10:47  hamilton
**  Structure alignment fiexed.
**
**  Revision 1.1.1.1  2000/08/21 18:45:45  hamilton
**  Initial import.
**
**  Revision 1.2  2000/08/21 16:58:29  hamilton
**  - Fix bug in call to copy_to_user() in function LinuxMMCTunnelSendReceive().
**
**
**  END AUTOMATIC VERSION CONTROL LOG
**************************************************************************/

#ifndef __KERNEL__
# define __KERNEL__
#endif
#ifndef MODULE
# define MODULE
#endif

#include <linux/autoconf.h>
#if defined(CONFIG_MODVERSIONS) && !defined(MODVERSIONS)
# define MODVERSIONS
#endif

#ifdef MODVERSIONS
# include <linux/modversions.h>
#endif

#include <linux/module.h>
#include <linux/kernel.h> /* printk() */
#include <linux/slab.h>   /* kmalloc() */
#include <linux/fs.h>     /* everything... */
#include <linux/errno.h>  /* error codes */
#include <linux/types.h>  /* size_t */
#include <linux/proc_fs.h>
#include <linux/fcntl.h>  /* O_ACCMODE */
#include <linux/pci.h>    /* pci_* */
#include <linux/ioport.h> /* *_region */
#include <asm/uaccess.h>  /* copy_*_user */
#include <asm/system.h>   /* cli(), *_flags */
#include <asm/segment.h>
#include <asm/io.h>

#include <imb_std.h>
#include <osdepdrv.h>
#include <mmcdrv_m.h>
#include <mmc.h>
#include <linux_tunnl.h>

#include "main.h"        /* local definitions */

#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif

/* Local prototypes */
static int mmcdev_open(struct inode *, struct file *);
static int mmcdev_close(struct inode *, struct file *);
static int mmcdev_ioctl(struct inode *, struct file *, unsigned int, unsigned long);
static int mmcdev_getmajor();

EXPORT_SYMBOL_NOVERS( mmcdev_open     );
EXPORT_SYMBOL_NOVERS( mmcdev_close    );
EXPORT_SYMBOL_NOVERS( mmcdev_getmajor );
EXPORT_SYMBOL_NOVERS( MMCDRVR_SendRcvMessage );

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
static int mmcdev_proc_read(char *, char **, off_t, int, int);
#endif

static int LinuxMMCTunnelSendReceive(unsigned long);
static int LinuxMMCTunnelGetVersionNumber(unsigned long);
static int LinuxMMCTunnelReadDataPort(unsigned long);
static int LinuxMMCTunnelReadStatusPort(unsigned long);
static int LinuxMMCTunnelReadFlagPort(unsigned long);
static int LinuxMMCTunnelWriteDataPort(unsigned long);
static int LinuxMMCTunnelWriteStatusPort(unsigned long);
static int LinuxMMCTunnelWriteFlagPort(unsigned long);

static BOOL MMCFindISA_SMIC(void);
static int MMCFindPCI_SMIC(void);

/* External prototypes */

/* Local variables */
static BOOL smic_found = 0;		   /* 0 - none, 1 - isa, 2 - pci */
static int mmcdev_count = 0;
static int mmcdev_major = 0;	   /* use dynamic allocation */
static spinlock_t mmcdev_lock;

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
/*
 * Describes our /proc entry (/proc/mmcdev)
 */
static struct proc_dir_entry mmcdev_proc_entry =
{
	0,                 /* low_ino: the inode -- dynamic */
	6, "mmcdev",       /* len of name and name */
	S_IFREG | S_IRUGO, /* mode */
	1, 0, 0,           /* nlinks, owner, group */
	0, NULL,           /* size - unused; operations -- use default */
	mmcdev_proc_read,  /* function used to read data */
};
#endif

/*
 * The file operations we can handle.
 */

static struct file_operations mmcdev_fops =
{
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0))
	THIS_MODULE,			/* owner */
#endif
	NULL,                  /* lseek */
	NULL,                  /* read */
	NULL,                  /* write */
	NULL,                  /* readdir */
	NULL,                  /* poll */
	mmcdev_ioctl,          /* ioctl */
	NULL,                  /* mmap */
	mmcdev_open,           /* open */
	NULL,                  /* flush */
	mmcdev_close,          /* close */
};

/**************************************************************************
 ** init_module - initialize the module
 **
 **  Description:
 **              This routine is called on insmod mmcdev. Initializes the
 **              module, creating the entrys /dev/mmcdev and /proc/mmcdev.
 **              Also calls the initialization code of MMC API (kernel side).
 **
 **  Prototype:
 **              int init_module(void)
 **
 **  Return Value:
 **              0 - success, otherwise return the result of register_chrdev()
 **
 **  Globals Used:  mmcdev_major.
 **  Globals Modified: mmcdev_major.
 **
 **************************************************************************/
int init_module(void)
{
	int result = 0;

	spin_lock_init(&mmcdev_lock);

	result = register_chrdev(mmcdev_major, "mmcdev", &mmcdev_fops);

	if (result < 0)
	{
		printk(KERN_ERR "mmcdev: register_chrdev() failed (%d).\n", result);
		return result;
	}

	if (mmcdev_major == 0)
		mmcdev_major = result;         /* dynamic */

	if (MMCFindISA_SMIC() == FALSE)
		if (MMCFindPCI_SMIC() == FALSE)
			goto fail_hpinit;

	EXPORT_NO_SYMBOLS;                 /* hide global symbols */

#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
	proc_register(&proc_root, &mmcdev_proc_entry);
#endif
	/* if sucess, registry the inter module functions */
	//inter_module_register( "mmcdev_open"           , THIS_MODULE, mmcdev_open            );
	//inter_module_register( "mmcdev_close"          , THIS_MODULE, mmcdev_close           );
	//inter_module_register( "MMCDRVR_SendRcvMessage", THIS_MODULE, MMCDRVR_SendRcvMessage );
	//inter_module_register( "mmcdev_getmajor"       , THIS_MODULE, mmcdev_getmajor        );
	return 0;                          /* success */

fail_hpinit:
	unregister_chrdev(mmcdev_major, "mmcdev");

	return result;
}
// init_module

/**************************************************************************
 ** cleanup_module - clears the module
 **
 **  Description:
 **              This routine is called on rmmod mmcdev. Unregister the module
 **              and remove the entry /proc/mmcdev.
 **
 **  Prototype:
 **              int init_module(void)
 **
 **  Return Value:
 **              None
 **
 **  Globals Used:  none.
 **  Globals Modified: none.
 **
 **************************************************************************/
void cleanup_module(void)
{
	//inter_module_unregister( "mmcdev_open"            );
	//inter_module_unregister( "mmcdev_close"           );
	//inter_module_unregister( "MMCDRVR_SendRcvMessage" );
	//inter_module_unregister( "mmcdev_getmajor"        );
	
	if (g_dataRegister)
		release_region(g_dataRegister, 3);
	if (mmcdev_major)
		unregister_chrdev(mmcdev_major, "mmcdev");
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
	proc_unregister(&proc_root, mmcdev_proc_entry.low_ino);
#endif
}
// cleanup_module


/**************************************************************************
 ** mmcdev_getmajor - Return the dynamic MAJOR code
 **
 **  Description:
 **              Return the dynamic MAJOR code
 **
 **  Prototype:
 **              int mmcdev_getmajor()
 **
 **  Return value:
 **              MAJOR CODE
 **
 **  Globals Used:  mmcdev_major
 **
 **************************************************************************/
int mmcdev_getmajor() { return mmcdev_major; }
// mmcdev_getmajor

/**************************************************************************
 ** mmcdev_open - Open the driver device
 **
 **  Description:
 **              Opens and locks the driver so only one request at a time is
 **              being handled.
 **
 **  Prototype:
 **              int mmcdev_open(struct inode *inode, struct file *filp)
 **                              inode - pointer to struct inode
 **                              filp - pointer to struct file
 **
 **  Return value:
 **              -ENODEV - No MMC hardware or dev num > 0.
 **              -EBUSY  - anyone else is using the device
 **              0       - success
 **
 **  Globals Used:  mmcdev_lock, mmcdev_count.
 **  Globals Modified: mmcdev_lock, mmcdev_count.
 **
 **************************************************************************/
int mmcdev_open(struct inode *inode, struct file *filp)
{
	int num = MINOR(inode->i_rdev);

	if (num > 0 || smic_found == 0)
	{
		return -ENODEV;                       /* 1 device only */
	}

	spin_lock(&mmcdev_lock);

	if (mmcdev_count)
	{
		spin_unlock (&mmcdev_lock);
		return -EBUSY;                        /* already open */
	}

	mmcdev_count++;
	spin_unlock(&mmcdev_lock);

	MOD_INC_USE_COUNT;
	return 0;                                 /* success */
}
// mmcdev_open

/**************************************************************************
 ** mmcdev_close - Close the driver device
 **
 **  Description:
 **              Close and unlocks the driver so another process can use it.
 **
 **  Prototype:
 **              int mmcdev_close(struct inode *inode, struct file *filp)
 **                               inode - pointer to struct inode
 **                               filp - pointer to struct file
 **
 **  Return value:
 **              0 - success
 **
 **  Globals Used:  mmcdev_count.
 **  Globals Modified: mmcdev_count.
 **
 **************************************************************************/
int mmcdev_close(struct inode *inode, struct file *filp)
{
	if (mmcdev_count)
	{
		mmcdev_count--;                           /* release driver */
		MOD_DEC_USE_COUNT;
		return 0;                                 /* success */
	}
	else
	{
		return -EBADF;
	}
}
// mmcdev_close

/**************************************************************************
 ** mmcdev_ioctl - Does the ioctl syscall
 **
 **  Description:
 **              All driver functionality is implemented here. Handle the
 **              commands.
 **
 **  Prototype:
 **              int mmcdev_ioctl(struct inode *inode, struct file *filp,
 **                               unsigned int cmd, unsigned long arg)
 **                               inode - pointer to struct inode
 **                               filp - pointer to struct file
 **                               cmd - which command
 **                               arg - THE only arg
 **
 **  Return value:
 **              0 - success
 **
 **  Globals Used:  mmcdev_count.
 **  Globals Modified: mmcdev_count.
 **
 **************************************************************************/
int mmcdev_ioctl(struct inode *inode, struct file *filp,
				 unsigned int cmd, unsigned long arg)
{
	int ret;

	switch (cmd)
	{
	case IOCTL_MMC_GET_KERNEL_VERSION:
		ret = LinuxMMCTunnelGetVersionNumber(arg);
		break;
	case IOCTL_MMC_SEND_RCV:
		ret = LinuxMMCTunnelSendReceive(arg);
		break;
	case IOCTL_MMC_PORT_READ_DATA:
		ret = LinuxMMCTunnelReadDataPort(arg);
		break;
	case IOCTL_MMC_PORT_READ_STATUS:
		ret = LinuxMMCTunnelReadStatusPort(arg);
		break;
	case IOCTL_MMC_PORT_READ_FLAG:
		ret = LinuxMMCTunnelReadFlagPort(arg);
		break;
	case IOCTL_MMC_PORT_WRITE_DATA:
		ret = LinuxMMCTunnelWriteDataPort(arg);
		break;
	case IOCTL_MMC_PORT_WRITE_STATUS:
		ret = LinuxMMCTunnelWriteStatusPort(arg);
		break;
	case IOCTL_MMC_PORT_WRITE_FLAG:
		ret = LinuxMMCTunnelWriteFlagPort(arg);
		break;
	default:
		return -ENOTTY;
	}
	return ret;
}
// mmcdev_ioctl

/**************************************************************************
 ** mmcdev_proc_read - Prints statistics
 **
 **  Description:
 **              Called when the user read /proc/mmcdev.
 **
 **  Prototype:
 **              int mmcdev_proc_read(char *buf, char **start, off_t offset,
 **                                   int count, int *eof, void *data)
 **
 **  Return value:
 **              the number of bytes written.
 **
 **  Globals Used:  none.
 **  Globals Modified: none.
 **
 **************************************************************************/
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
int mmcdev_proc_read(char *buf, char **start, off_t offset,
					 int count, int eof)
{
	if (smic_found == 0)
		strcpy(buf, "SMIC not found.\n");
	else if (smic_found == 1)
		sprintf(buf, "ISA SMIC at address: 0x%x.\n", g_dataRegister);
	else
		sprintf(buf, "PCI SMIC at address: 0x%x.\n", g_dataRegister);

	return strlen(buf);
}
// mmcdev_proc_read
#endif

/* These will call the os-independent code... */

/**************************************************************************
 ** LinuxMMCTunnelSendReceive - Send/Receive to/from MMC
 **
 **  Description:
 **              Send bytes out the port to the MMC
 **
 **  Prototype:
 **
 **              int LinuxMMCTunnelSendReceive(unsigned long arg)
 **                                            arg - the arg
 **
 **  Return value:
 **
 **
 **  Globals Used:  None.
 **  Globals Modified: None.
 **
 **************************************************************************/
int LinuxMMCTunnelSendReceive(unsigned long arg)
{
	SEND_RCV sr;
	/* first copy the structure from userspace */
	if (copy_from_user(&sr, (int *)arg, sizeof(SEND_RCV)))
		return -EFAULT;
	
	/* allocate an area to copy the input buffer into */
	if ((sr.input = kmalloc(sr.ilen, GFP_KERNEL)) == NULL)
		return -EFAULT;
	
	/* now copy the input buffer from userspace */
	if (copy_from_user( sr.input, ((PSEND_RCV)arg)->input, sr.ilen))
	{
		kfree(sr.input);
		return -EFAULT;
	}

	/* allocate an area to output buffer */
	if ((sr.output = kmalloc(sr.olen, GFP_KERNEL)) == NULL)
	{
		kfree(sr.input);
		return -EFAULT;
	}
	sr.retcode = MMCDRVR_SendRcvMessage(sr.input, sr.ilen, sr.output, &sr.olen);
	/* copy to userspace the output buffer, its len and the retcode */
	if (copy_to_user(&(((PSEND_RCV)arg)->olen), &sr.olen, sizeof(sr.olen)) ||
		copy_to_user(&(((PSEND_RCV)arg)->retcode), &sr.retcode, sizeof(sr.retcode)) ||
		copy_to_user(((PSEND_RCV)arg)->output, sr.output, sr.olen))
	{
		kfree(sr.input);
		kfree(sr.output);
		return -EFAULT;
	}

	kfree(sr.input);
	kfree(sr.output);

	return 0;

}
// LinuxMMCTunnelSendReceive

/**************************************************************************
 ** LinuxTunnelGetVersionNumber - Driver version information
 **
 **  Description:
 **              Provides driver information
 **
 **  Prototype:
 **              int LinuxMMCTunnelGetVersionNumber(arg)
 **                                                 arg - the arg
 **
 **  Return value:
 **
 **
 **  Globals Used:  None.
 **  Globals Modified: None.
 **
 **************************************************************************/
int LinuxMMCTunnelGetVersionNumber(unsigned long arg)
{
	KERNEL_VERSION kernelVersion;

	kernelVersion.retcode =
		MMCDRVR_GetVersionNumber(&kernelVersion.oem_name[0],
								 &kernelVersion.version_letter,
								 &kernelVersion.version_major,
								 &kernelVersion.version_minor);

	if (copy_to_user((int *)arg, &kernelVersion, sizeof(KERNEL_VERSION)))
		return -EFAULT;
	else
		return 0;
}
// LinuxMMCTunnelGetVersionNumber

/**************************************************************************
 ** LinuxMMCTunnelReadDataPort - Read the data register
 **
 **  Description:
 **              ...
 **
 **  Prototype:
 **              int LinuxMMCTunnelReadDataPort(arg)
 **                                             arg - the arg
 **
 **  Return value:
 **
 **
 **  Globals Used:  None.
 **  Globals Modified: None.
 **
 **************************************************************************/
int LinuxMMCTunnelReadDataPort(unsigned long arg)
{
	SEND_RCV sr;
	UINT8 chRead;

	sr.retcode = MMC_SUCCESS;
	sr.output = &chRead;
	sr.olen = 1;

	chRead = InputByte(g_dataRegister);

	/* copy to userspace the output buffer, its len and the retcode */
	if (copy_to_user(&(((PSEND_RCV)arg)->olen), &sr.olen, sizeof(sr.olen)) ||
		copy_to_user(&(((PSEND_RCV)arg)->retcode), &sr.retcode, sizeof(sr.retcode)) ||
		copy_to_user(((PSEND_RCV)arg)->output, sr.output, sr.olen))
	{
		return -EFAULT;
	}
	else
		return 0;
}
// LinuxMMCTunnelReadDataPort

/**************************************************************************
 ** LinuxMMCTunnelReadStatusPort - Read the status register
 **
 **  Description:
 **              ...
 **
 **  Prototype:
 **              int LinuxMMCTunnelReadStatusPort(arg)
 **                                               arg - the arg
 **
 **  Return value:
 **
 **
 **  Globals Used:  None.
 **  Globals Modified: None.
 **
 **************************************************************************/
int LinuxMMCTunnelReadStatusPort(unsigned long arg)
{
	SEND_RCV sr;
	UINT8 chRead;

	sr.retcode = MMC_SUCCESS;
	sr.output = &chRead;
	sr.olen = 1;

	chRead = InputByte(g_controlStatusRegister);

	/* copy to userspace the output buffer, its len and the retcode */
	if (copy_to_user(&(((PSEND_RCV)arg)->olen), &sr.olen, sizeof(sr.olen)) ||
		copy_to_user(&(((PSEND_RCV)arg)->retcode), &sr.retcode, sizeof(sr.retcode)) ||
		copy_to_user(((PSEND_RCV)arg)->output, sr.output, sr.olen))
	{
		return -EFAULT;
	}
	else
		return 0;
}
// LinuxMMCTunnelReadStatusPort

/**************************************************************************
 ** LinuxMMCTunnelReadFlagPort - Read the flag register
 **
 **  Description:
 **              ...
 **
 **  Prototype:
 **              int LinuxMMCTunnelReadFlagPort(arg)
 **                                             arg - the arg
 **
 **  Return value:
 **
 **
 **  Globals Used:  None.
 **  Globals Modified: None.
 **
 **************************************************************************/
int LinuxMMCTunnelReadFlagPort(unsigned long arg)
{
	SEND_RCV sr;
	UINT8 chRead;

	sr.retcode = MMC_SUCCESS;
	sr.output = &chRead;
	sr.olen = 1;

	chRead = InputByte(g_flagRegister);

	/* copy to userspace the output buffer, its len and the retcode */
	if (copy_to_user(&(((PSEND_RCV)arg)->olen), &sr.olen, sizeof(sr.olen)) ||
		copy_to_user(&(((PSEND_RCV)arg)->retcode), &sr.retcode, sizeof(sr.retcode)) ||
		copy_to_user(((PSEND_RCV)arg)->output, sr.output, sr.olen))
	{
		return -EFAULT;
	}
	else
		return 0;
}
// LinuxMMCTunnelReadFlagPort

/**************************************************************************
 ** LinuxMMCTunnelWriteDataPort - Write the data register
 **
 **  Description:
 **              ...
 **
 **  Prototype:
 **              int LinuxMMCTunnelWriteDataPort(arg)
 **                                              arg - the arg
 **
 **  Return value:
 **
 **
 **  Globals Used:  None.
 **  Globals Modified: None.
 **
 **************************************************************************/
int LinuxMMCTunnelWriteDataPort(unsigned long arg)
{
	UINT16 retCode;
	UINT8 chWrite;
	int retVal;

	retVal = get_user(chWrite, ((PSEND_RCV)arg)->input);

	if (retVal)
		return retVal;

	OutputByte(g_dataRegister, chWrite);

	retCode = MMC_SUCCESS;

	/* copy to userspace the retcode */
	if (copy_to_user(&(((PSEND_RCV)arg)->retcode), &retCode, sizeof(retCode)))
		return -EFAULT;
	else
		return 0;
}
// LinuxMMCTunnelWriteDataPort

/**************************************************************************
 ** LinuxMMCTunnelWriteStatusPort - Write the status register
 **
 **  Description:
 **              ...
 **
 **  Prototype:
 **              int LinuxMMCTunnelWriteStatusPort(arg)
 **                                                arg - the arg
 **
 **  Return value:
 **
 **
 **  Globals Used:  None.
 **  Globals Modified: None.
 **
 **************************************************************************/
int LinuxMMCTunnelWriteStatusPort(unsigned long arg)
{
	UINT16 retCode;
	UINT8 chWrite;
	int retVal;

	retVal = get_user(chWrite, ((PSEND_RCV)arg)->input);

	if (retVal)
		return retVal;

	OutputByte(g_controlStatusRegister, chWrite);

	retCode = MMC_SUCCESS;

	/* copy to userspace the retcode */
	if (copy_to_user(&(((PSEND_RCV)arg)->retcode), &retCode, sizeof(retCode)))
		return -EFAULT;
	else
		return 0;
}
// LinuxMMCTunnelWriteStatusPort

/**************************************************************************
 ** LinuxMMCTunnelWriteFlagPort - Write the flag register
 **
 **  Description:
 **              Write the flag register
 **
 **  Prototype:
 **              int LinuxMMCTunnelWriteFlagPort(arg)
 **                                              arg - the arg
 **
 **  Return value:
 **
 **
 **  Globals Used:  None.
 **  Globals Modified: None.
 **
 **************************************************************************/
int LinuxMMCTunnelWriteFlagPort(unsigned long arg)
{
	UINT16 retCode;
	UINT8 chWrite;
	int retVal;

	retVal = get_user(chWrite, ((PSEND_RCV)arg)->input);

	if (retVal)
		return retVal;

	OutputByte(g_flagRegister, chWrite);

	retCode = MMC_SUCCESS;

	/* copy to userspace the retcode */
	if (copy_to_user(&(((PSEND_RCV)arg)->retcode), &retCode, sizeof(retCode)))
		return -EFAULT;
	else
		return 0;
}
// LinuxMMCTunnelWriteFlagPort

/*******************************************************************
 ** MMCFindISA_SMIC - Find and probe an ISA SMIC
 **
 **  Description:
 **              This tries to find an ISA SMIC and its address.
 **              It first tries the address passed in. If that is
 **              zero or it fails, it tries each of the addresses in
 **              g_dataRegisterAddresses.
 **
 **  Prototype:
 **              int MMCFindISA_SMIC(void)
 **
 **  Return value:
 **              TRUE  - address found
 **              FALSE - address not found
 **
 **  Globals Used: g_dataRegisterAddresses, DelayPort
 **  Globals Modified: g_dataRegister, g_controlStatusRegister, g_flagRegister
 **
 ************************************************************************/
BOOL MMCFindISA_SMIC(void)
{
	if (check_region(g_dataRegisterAddresses[0], 3) < 0)
	{
		printk(KERN_ERR "mmcdev: ISA region in use!\n");
		return FALSE;
	}

	request_region(g_dataRegisterAddresses[0], 3, "mmcdev");

	OutputByte(g_dataRegisterAddresses[0], 0xAA);
	InputByte(g_dataRegisterAddresses[0] + 1);
	if (InputByte(g_dataRegisterAddresses[0]) == 0xAA)
	{
		OutputByte(g_dataRegisterAddresses[0], 0x55);
		InputByte(g_dataRegisterAddresses[0] + 1);
		if (InputByte(g_dataRegisterAddresses[0]) == 0x55)
		{
			printk(KERN_INFO "mmcdev: ISA SMIC present.\n");
			smic_found = 1;
			g_dataRegister = g_dataRegisterAddresses[0];
			g_controlStatusRegister = g_dataRegisterAddresses[0] + 1;
			g_flagRegister = g_dataRegisterAddresses[0] + 2;
			return TRUE;
		}
	}
	release_region(g_dataRegisterAddresses[0], 3);
	return FALSE;
}
// MMCFindISA_SMIC

/*******************************************************************
 ** MMCFindPCI_SMIC - Find and probe a PCI SMIC
 **
 **  Description:
 **              This tries to find an PCI SMIC and its address.
 **              It first tries the address passed in. If that is
 **              zero or it fails, it tries each of the addresses in
 **              g_dataRegisterAddresses.
 **
 **  Prototype:
 **              int MMCFindPCI_SMIC(void)
 **
 **  Return value:
 **              TRUE  - address found
 **              FALSE - address not found
 **
 **  Globals Used: g_dataRegisterAddresses, DelayPort
 **  Globals Modified: g_dataRegister, g_controlStatusRegister, g_flagRegister
 **
 ************************************************************************/
int MMCFindPCI_SMIC(void)
{
	int error;
	struct pci_dev *pDev;
	UINT16 wBaseAddr;
	int feRmc = FALSE;
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
	UINT16 wSubVendorId;
#endif

	if (!pci_present())
		return FALSE;

	if ((pDev = pci_find_device(PCI_HP_VENDOR_ID, PCI_MMC_DEVICE_ID, NULL)))
		;
	else if ((pDev = pci_find_class(PCI_ERMC_CLASSCODE, NULL)) &&
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0))
			 pci_read_config_word(pDev, PCI_SUBSYSTEM_VENDOR_ID, &wSubVendorId) &&
			 wSubVendorId == PCI_HP_VENDOR_ID)
#else
		pDev->subsystem_vendor == PCI_HP_VENDOR_ID)
#endif
		feRmc = TRUE;
	else
		return FALSE;
	
	error = pci_read_config_word(pDev, PCI_MMC_ADDR_CW, &wBaseAddr);
	if (error)
	{
		printk(KERN_ERR "mmcdev: pci_read_config_word() failed (%d).\n", error);
		return FALSE;
	}
	
	/* Bit 0: 1 specifies programmed I/O, 0 specifies memory mapped I/O */
	if (!(wBaseAddr & 0x0001))
	{
		printk(KERN_ERR "mmcdev: memory mapped I/O not supported.\n");
		return FALSE;
	}
	
	wBaseAddr &= 0xFFFE;
	if (!feRmc)
		/* Data register starts at base address + 1 in eRMC */
		++wBaseAddr;
	
	if (check_region(wBaseAddr, 3) < 0)
	{
		printk(KERN_ERR "mmcdev: PCI region in use (0x%X).\n", wBaseAddr);
		return FALSE;
	}
	request_region(wBaseAddr, 3, "mmcdev");
	
	g_dataRegister = wBaseAddr;
	smic_found = 2;
	g_controlStatusRegister = g_dataRegister + 1;
	g_flagRegister = g_dataRegister + 2;
	
	printk(KERN_INFO "mmcdev: PCI SMIC present.\n");
	
	return TRUE;
}
// MMCFindPCI_SMIC
