/*
 *
 * A PCMCIA ethernet driver for NS8390-based cards
 *
 * This driver supports the D-Link DE-650 and Linksys EthernetCard
 * cards, the newer D-Link and Linksys combo cards, Accton EN2212
 * cards, the RPTI EP400, and the PreMax PE-200 in non-shared-memory
 * mode, and the IBM Credit Card Adapter, the NE4100, the Thomas
 * Conrad ethernet card, and the Kingston KNE-PCM/x in shared-memory
 * mode.  It will also handle the Socket EA card in either mode.
 *
 * Copyright (C) 1998 David A. Hinds -- dhinds@hyper.stanford.edu
 *
 * pcnet_cs.c 1.85 1999/02/13 06:47:20
 *  
 * The network driver code is based on Donald Becker's NE2000 code:
 *
 * Written 1992,1993 by Donald Becker.
 * Copyright 1993 United States Government as represented by the
 * Director, National Security Agency.  This software may be used and
 * distributed according to the terms of the GNU Public License,
 * incorporated herein by reference.
 * Donald Becker may be reached at becker@cesdis1.gsfc.nasa.gov
 * 
 * Based also on Keith Moore's changes to Don Becker's code, for IBM
 * CCAE support.  Drivers merged back together, and shared-memory
 * Socket EA support added, by Ken Raeburn, September 1995.
 *
 */

#include <pcmcia/config.h>
#include <pcmcia/k_compat.h>

#include <linux/version.h>
#include <linux/config.h>
#include <linux/module.h>
#include <linux/modversions.h>
#include <linux/skbuff.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/ptrace.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/delay.h>
#include <asm/io.h>
#include <asm/system.h>

#include <linux/netdevice.h>
#include "8390.h"

#include <pcmcia/driver_ops.h> 
#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/ciscode.h>
#include <pcmcia/ds.h>
#include <pcmcia/cisreg.h>

#ifdef GENERIC
#include "fpc0105tx.h"
#include "fpc0105tx_oem.h"
#endif

#define PCNET_CMD       0x00
#define PCNET_DATAPORT  0x10    /* NatSemi-defined port window offset. */
#define PCNET_RESET     0x1f    /* Issue a read to reset, a write to clear. */
#define PCNET_MISC      0x18    /* For IBM CCAE and Socket EA cards */

#define PCNET_START_PG  0x40    /* First page of TX buffer */
#define PCNET_STOP_PG   0x80    /* Last page +1 of RX ring */

/* Socket EA cards have a larger packet buffer */
#define SOCKET_START_PG 0x01
#define SOCKET_STOP_PG  0xff

#define PCNET_RDC_TIMEOUT 0x02    /* Max wait in jiffies for Tx RDC */

static char *if_names[] = { "Auto", "10baseT", "10base2"};

#ifdef PCMCIA_DEBUG
static int pc_debug = PCMCIA_DEBUG;
MODULE_PARM(pc_debug, "i");
#define DEBUG(n, args...) if (pc_debug>(n)) printk(KERN_DEBUG args)
static char *version =
"pcnet_cs.c 1.85 1999/02/13 06:47:20 (David Hinds)";
#else
#define DEBUG(n, args...)
#endif

/* Parameters that can be set with 'insmod' */

/* Bit map of interrupts to choose from */
static u_int irq_mask = 0xdeb8;
static int irq_list[4] = { -1 };

/* Transceiver type, for Socket EA and IBM CC cards. */
static int if_port = 1;

/* Use 64K packet buffer, for Socket EA cards. */
static int use_big_buf = 1;

/* Shared memory speed, in ns */
static int mem_speed = 0;

/* Insert a pause in block_output after sending a packet */
static int delay_output = 0;

/* Length of delay, in microseconds */
static int delay_time = 4;

/* Use shared memory or polled IO?  Default is to use hw_info */
static int use_shmem = -1;

/* Ugh!  Let the user hardwire the hardware address for queer cards */
static int hw_addr[6] = { 0, /* ... */ };

MODULE_PARM(irq_mask, "i");
MODULE_PARM(irq_list, "1-4i");
MODULE_PARM(if_port, "i");
MODULE_PARM(use_big_buf, "i");
MODULE_PARM(mem_speed, "i");
MODULE_PARM(delay_output, "i");
MODULE_PARM(delay_time, "i");
MODULE_PARM(use_shmem, "i");
MODULE_PARM(hw_addr, "6i");

static void pcnet_config(dev_link_t *link);
static void pcnet_release(u_long arg);
static int pcnet_event(event_t event, int priority,
               event_callback_args_t *args);

static int pcnet_open(struct device *dev);
static int pcnet_close(struct device *dev);

static void pcnet_reset_8390(struct device *dev);

static int set_config(struct device *dev, struct ifmap *map);

static int setup_shmem_window(dev_link_t *link, int start_pg,
                  int stop_pg, int cm_offset);
static int setup_dma_config(dev_link_t *link, int start_pg,
                int stop_pg);

static dev_info_t dev_info = "pcnet_cs";

static dev_link_t *pcnet_attach(void);
static void pcnet_detach(dev_link_t *);

static dev_link_t *dev_list;

typedef struct hw_info_t {
    u_long  offset;
    u_char  a0, a1, a2;
    u_long  flags;
} hw_info_t;

#define DELAY_OUTPUT    0x01
#define HAS_MISC_REG    0x02
#define USE_SHMEM       0x04
#define USE_BIG_BUF     0x08
#define HAS_IBM_MISC    0x10
#define IS_DL10019A     0x20

static hw_info_t hw_info[] = {
    { /* Accton EN2212 */ 0x0ff0, 0x00, 0x00, 0xe8, DELAY_OUTPUT }, 
    { /* Allied Telesis LA-PCM */ 0x0ff0, 0x00, 0x00, 0xf4, USE_SHMEM },
    { /* APEX MultiCard */ 0x03f4, 0x00, 0x20, 0xe5, 0 },
    { /* ASANTE FriendlyNet */ 0x4910, 0x00, 0x00, 0x94,
      DELAY_OUTPUT | HAS_IBM_MISC },
    { /* Danpex EN-6200P2 */ 0x0110, 0x00, 0x40, 0xc7, 0 },
    { /* DataTrek NetCard */ 0x0ff0, 0x00, 0x20, 0xe8, 0 },
    { /* Dayna CommuniCard E */ 0x0110, 0x00, 0x80, 0x19, 0 },
    { /* D-Link DE-650 */ 0x0040, 0x00, 0x80, 0xc8, 0 },
    { /* EP-210 Ethernet */ 0x0110, 0x00, 0x40, 0x33, 0 },
    { /* EP4000 Ethernet */ 0x01c0, 0x00, 0x00, 0xb4, 0 },
    { /* Epson EEN10B */ 0x0ff0, 0x00, 0x00, 0x48,
      USE_SHMEM | HAS_MISC_REG | HAS_IBM_MISC },
    { /* ELECOM Laneed LD-CDWA */ 0xb8, 0x08, 0x00, 0x42, 0 },
    { /* Hypertec Ethernet */ 0x01c0, 0x00, 0x40, 0x4c, 0 },
    { /* IBM CCAE */ 0x0ff0, 0x08, 0x00, 0x5a,
      USE_SHMEM | HAS_MISC_REG | HAS_IBM_MISC },
    { /* IBM CCAE */ 0x0ff0, 0x00, 0x04, 0xac,
      USE_SHMEM | HAS_MISC_REG | HAS_IBM_MISC },
    { /* IBM CCAE */ 0x0ff0, 0x00, 0x06, 0x29,
      USE_SHMEM | HAS_MISC_REG | HAS_IBM_MISC },
    { /* IBM FME */ 0x0374, 0x08, 0x00, 0x5a,
      USE_SHMEM | HAS_MISC_REG | HAS_IBM_MISC },
    { /* IBM FME */ 0x0374, 0x00, 0x04, 0xac,
      USE_SHMEM | HAS_MISC_REG | HAS_IBM_MISC },
    { /* NSC DP83903 */ 0x0374, 0x00, 0xc0, 0xa8,
      USE_SHMEM | HAS_MISC_REG | HAS_IBM_MISC },
    { /* NSC DP83903 */ 0x0374, 0x00, 0xa0, 0xb0,
      USE_SHMEM | HAS_MISC_REG | HAS_IBM_MISC },
    { /* NSC DP83903 */ 0x0198, 0x00, 0x20, 0xe0,
      USE_SHMEM | HAS_MISC_REG | HAS_IBM_MISC },
    { /* I-O DATA PCLA/T */ 0x0ff0, 0x00, 0xa0, 0xb0, 0 },
    { /* Katron PE-520 */ 0x0110, 0x00, 0x40, 0xf6, 0 },
    { /* Kingston KNE-PCM/x */ 0x0ff0, 0x00, 0xc0, 0xf0,
      USE_SHMEM | HAS_MISC_REG | HAS_IBM_MISC },
    { /* Kingston KNE-PCM/x */ 0x0ff0, 0xe2, 0x0c, 0x0f,
      USE_SHMEM | HAS_MISC_REG | HAS_IBM_MISC },
    { /* Kingston KNE-PC2 */ 0x0180, 0x00, 0xc0, 0xf0, 0 },
    { /* Longshine LCS-8534 */ 0, 0x08, 0x00, 0x00, 0 },
    { /* Maxtech PCN2000 */ 0x5000, 0x00, 0x00, 0xe8, 0 },
    { /* NDC Instant-Link */ 0x003a, 0x00, 0x80, 0xc6, 0 },

#ifdef GENERIC_IN0100AF
    { /* Generic iPort */ 0x0ff0, 0x00, 0x02, 0xDD, 0 },
#endif

#ifdef DDC 
    { /* Digital Data Communications */ 0x0ff0, 0x00, 0x02, 0xDD, 0 },
#endif

#ifdef GENERIC_BIN0100A
    { /* BroMax iPort */ 0x0ff0, 0x00, 0x02, 0xDD, 0 },
#endif

#ifdef MELCO 
    { /* MELCO LPC3-TX */ 0x0ff0, 0x00, 0x40, 0x26, 0 },
    { /* MELCO LPC3-TX */ 0x0ff0, 0x00, 0x02, 0xDD, 0 },
#endif

#ifdef BUFFALO_CLX
    { /* BUFFALO LPC3-CLX */ 0x0ff0, 0x00, 0x40, 0x26, 0 },
    { /* BUFFALO LPC3-CLX */ 0x0ff0, 0x00, 0x02, 0xDD, 0 },
#endif

#ifdef COREGA
    { /* corega FEther PPC-TXD */ 0xff0, 0x00, 0x90, 0x99, 0 },
#endif

    { /* NE2000 Compatible */ 0x0ff0, 0x00, 0xa0, 0x0c, 0 },
    { /* Network General Sniffer */ 0x0ff0, 0x00, 0x00, 0x65,
      USE_SHMEM | HAS_MISC_REG | HAS_IBM_MISC },
    { /* Olicom GoCard */ 0x00ae, 0x00, 0x00, 0x24, 0 },
    { /* Panasonic VEL211 */ 0x0ff0, 0x00, 0x80, 0x45, 
      USE_SHMEM | HAS_MISC_REG | HAS_IBM_MISC },
    { /* PreMax PE-200 */ 0x07f0, 0x00, 0x20, 0xe0, 0 },
    { /* RPTI EP400 */ 0x0110, 0x00, 0x40, 0x95, 0 },
    { /* SCM Ethernet */ 0x0ff0, 0x00, 0x20, 0xcb, 0 },
    { /* Socket EA */ 0x4000, 0x00, 0xc0, 0x1b,
      DELAY_OUTPUT | HAS_MISC_REG | USE_BIG_BUF },
    { /* SuperSocket RE450T */ 0x0110, 0x00, 0xe0, 0x98, 0 },
    { /* PCMCIA 10/100 Ethernet Card */ 0x0400, 0x88, 0x19, 0x00, 0 },
    { /* Volktek NPL-402CT */ 0x0060, 0x00, 0x40, 0x05, 0 }
};

#define NR_INFO     (sizeof(hw_info)/sizeof(hw_info_t))

static hw_info_t default_info =
{ /* Unknown NE2000 Clone */ 0x00, 0x00, 0x00, 0x00, 0 };

static hw_info_t dl_fast_info =
{ /* D-Link EtherFast */ 0x00, 0x00, 0x00, 0x00, IS_DL10019A };

typedef struct pcnet_dev_t {
    struct device   dev;
    dev_node_t      node;
    u_long          flags;
    caddr_t         base;
} pcnet_dev_t;

#ifdef GENERIC
extern dev_link_t *asix_link;
#endif

static void cs_error(client_handle_t handle, int func, int ret) {
    error_info_t err = { func, ret };
    CardServices(ReportError, handle, &err);
}

/*
 *  We never need to do anything when a pcnet device is "initialized"
 *  by the net software, because we only register already-found cards.
 */
static int pcnet_init(struct device *dev) {
    return(0);
}

/*
 * pcnet_attach() creates an "instance" of the driver, allocating
 * local data structures for one device.  The device is registered
 * with Card Services.
 */
static dev_link_t *pcnet_attach(void) {
    client_reg_t client_reg;
    dev_link_t *link;
    pcnet_dev_t *info;
    struct device *dev;
    int i, ret;

    DEBUG(0, "pcnet_attach()\n");

    /* Create new ethernet device */
    link = kmalloc(sizeof(struct dev_link_t), GFP_KERNEL);
    memset(link, 0, sizeof(struct dev_link_t));
    link->release.function = &pcnet_release;
    link->release.data = (u_long) link;
    link->irq.Attributes = IRQ_TYPE_EXCLUSIVE;
    link->irq.IRQInfo1 = IRQ_INFO2_VALID | IRQ_LEVEL_ID;

    if (irq_list[0] == -1)
        link->irq.IRQInfo2 = irq_mask;
    else
        for (i = 0; i < 4; i++)
            link->irq.IRQInfo2 |= 1 << irq_list[i];

    link->conf.Attributes = CONF_ENABLE_IRQ;
    link->conf.Vcc = 50;
    link->conf.IntType = INT_MEMORY_AND_IO;

    info = kmalloc(sizeof(struct pcnet_dev_t), GFP_KERNEL);
    memset(info, 0, sizeof(struct pcnet_dev_t));
    dev = &info->dev;
    ethdev_init(dev);
    dev->name = info->node.dev_name;
    dev->init = &pcnet_init;
    dev->open = &pcnet_open;
    dev->stop = &pcnet_close;
    dev->set_config = &set_config;
    dev->tbusy = 1;
    link->priv = info;

    /* Register with Card Services */
    link->next = dev_list;
    dev_list = link;
    client_reg.dev_info = &dev_info;
    client_reg.Attributes = INFO_IO_CLIENT | INFO_CARD_SHARE;
    client_reg.EventMask = CS_EVENT_CARD_INSERTION | CS_EVENT_CARD_REMOVAL |
                           CS_EVENT_RESET_PHYSICAL | CS_EVENT_CARD_RESET |
                           CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
    client_reg.event_handler = &pcnet_event;
    client_reg.Version = 0x0210;
    client_reg.event_callback_args.client_data = link;
    ret = CardServices(RegisterClient, &link->handle, &client_reg);
    if (ret != CS_SUCCESS) {
        cs_error(link->handle, RegisterClient, ret);
        pcnet_detach(link);

        return(NULL);
    }

    return(link);
} /* pcnet_attach */

/*
 * This deletes a driver "instance".  The device is de-registered
 * with Card Services.  If it has been released, all local data
 * structures are freed.  Otherwise, the structures will be freed
 * when the device is released.
 */
static void pcnet_detach(dev_link_t *link) {
    dev_link_t **linkp;
    long flags;

    DEBUG(0, "pcnet_detach(0x%p)\n", link);

    /* Locate device structure */
    for (linkp = &dev_list; *linkp; linkp = &(*linkp)->next)
        if (*linkp == link)
            break;

    if (*linkp == NULL)
        return;

    save_flags(flags);
    cli();
    if (link->state & DEV_RELEASE_PENDING) {
        del_timer(&link->release);
        link->state &= ~DEV_RELEASE_PENDING;
    }
    restore_flags(flags);

    if (link->state & DEV_CONFIG) {
        pcnet_release((u_long) link);
        if (link->state & DEV_STALE_CONFIG) {
            link->state |= DEV_STALE_LINK;

            return;
        }
    }

    if (link->handle)
        CardServices(DeregisterClient, link->handle);

    /* Unlink device structure, free bits */
    *linkp = link->next;
      if (link->priv) {
          struct device *dev = link->priv;
          if (dev->priv)
              kfree_s(dev->priv, sizeof(struct ei_device));
          kfree_s(dev, sizeof(struct pcnet_dev_t));
      }
    kfree_s(link, sizeof(struct dev_link_t));

} /* pcnet_detach */

/*
 *  For the Linksys EtherFast 10/100 card.
 */
static hw_info_t *get_dl_fast(dev_link_t *link) {
    struct device *dev = link->priv;
    int i;
    u_char sum;

    for (sum = 0, i = 0x14; i < 0x1c; i++)
        sum += inb_p(dev->base_addr + i);

    if (sum != 0xff)
        return(NULL);

    for (i = 0; i < 6; i++)
        dev->dev_addr[i] = inb_p(dev->base_addr + 0x14 + i);

    return(&dl_fast_info);
}

/*
 * This probes for a card's hardware address, for card types that encode this
 * information in their CIS.
 */
static hw_info_t *get_hwinfo(dev_link_t *link) {
    struct device *dev = link->priv;
    win_req_t req;
    memreq_t mem;
    u_char *base, *virt;
    int i, j;

    /* Allocate a 4K memory window */
    req.Attributes = WIN_DATA_WIDTH_8 | WIN_MEMORY_TYPE_AM | WIN_ENABLE;
    req.Base = 0; req.Size = 0x1000;
    req.AccessSpeed = 0;
    link->win = (window_handle_t) link->handle;
    i = CardServices(RequestWindow, &link->win, &req);

    if (i != CS_SUCCESS) {
        cs_error(link->handle, RequestWindow, i);

        return(NULL);
    }

    virt = ioremap(req.Base, 0x1000);
    mem.Page = 0;
    mem.CardOffset = 0;
    CardServices(MapMemPage, link->win, &mem);
    base = &virt[0];

    writeb(0x47, base + 0x3c0);
    writeb(0x47, base + 0x3e0);
    writeb(readb(base + 0x3e2) | 0x08, base + 0x3e8);

    writeb(dev->base_addr, base + 0x3ca);
    writeb(dev->base_addr >> 8, base + 0x3cc);

    writeb(link->io.BasePort2, base + 0x3ea);
    writeb(link->io.BasePort2 >> 8, base + 0x3ec);

    for (i = 0; i < NR_INFO; i++) {
        mem.CardOffset = hw_info[i].offset & ~0x0fff;
        CardServices(MapMemPage, link->win, &mem);
        base = &virt[hw_info[i].offset & 0x0fff];

        if ((readb(base+0) == hw_info[i].a0) &&
            (readb(base+2) == hw_info[i].a1) &&
            (readb(base+4) == hw_info[i].a2))
            break;
    }

    if (i < NR_INFO)
        for (j = 0; j < 6; j++)
            dev->dev_addr[j] = readb(base + (j<<1));
    
    iounmap(virt);
    j = CardServices(ReleaseWindow, link->win);

    if (j != CS_SUCCESS)
        cs_error(link->handle, ReleaseWindow, j);

    return((i < NR_INFO) ? hw_info+i : NULL);
} /* get_hwinfo */

/*
 * This probes for a card's hardware address by reading the PROM.
 * It checks the address against a list of known types, then falls
 * back to a simple NE2000 clone signature check.
 */
static hw_info_t *get_prom(dev_link_t *link) {
    struct device *dev = link->priv;
    u16 prom[16];
    int i, ioaddr;

    /* This is lifted straight from drivers/net/ne.c */
    struct {
        unsigned char value, offset;
    } program_seq[] = {
          {E8390_NODMA + E8390_PAGE0+E8390_STOP, E8390_CMD},
                                            /* Select page 0*/
          {0x01, EN0_DCFG},                 /* Set byte-wide (0x48) access. */
          {0x00, EN0_RCNTLO},               /* Clear the count regs. */
          {0x00, EN0_RCNTHI},
          {0x00, EN0_IMR},                  /* Mask completion irq. */
          {0xFF, EN0_ISR},
          {E8390_RXOFF |0x40, EN0_RXCR},    /* 0x20  Set to monitor */
          {E8390_TXOFF, EN0_TXCR},          /* 0x02  and loopback mode. */
          {32, EN0_RCNTLO},
          {0x00, EN0_RCNTHI},
          {0x00, EN0_RSARLO},               /* DMA starting at 0x0000. */
          {0x04, EN0_RSARHI},
          {E8390_RREAD+E8390_START, E8390_CMD},
    };

    ioaddr = dev->base_addr;

    pcnet_reset_8390(dev);
    udelay(10000);

    for (i = 0; i < sizeof(program_seq) / sizeof(program_seq[0]); i++)
    outb_p(program_seq[i].value, ioaddr + program_seq[i].offset);

    for (i = 0; i < 16; i++)
         prom[i] = inw(ioaddr + PCNET_DATAPORT);

    for (i = 0; i < NR_INFO; i++) {
        if (((prom[0] & 0xff) == hw_info[i].a0) &&
            ((prom[0] >>8) == hw_info[i].a1) &&
            ((prom[1] & 0xff) == hw_info[i].a2))
            break;
    }

    if ((i < NR_INFO) || ((prom[28] == 0x57) && (prom[30] == 0x57))) {
        dev->dev_addr[0] = prom[0] & 0xff;
        dev->dev_addr[1] = prom[0] >>8;
        dev->dev_addr[2] = prom[1] & 0xff;
        dev->dev_addr[3] = prom[1] >>8;
        dev->dev_addr[4] = prom[2] & 0xff;
        dev->dev_addr[5] = prom[2] >>8;

        return((i < NR_INFO) ? hw_info+i : &default_info);
    }

    return(NULL);
} /* get_prom */

/*
 * This should be totally unnecessary... but when we can't figure
 * out the hardware address any other way, we'll let the user hard
 * wire it when the module is initialized.
 */
static hw_info_t *get_hwired(dev_link_t *link) {
    struct device *dev = link->priv;
    int i;

    for (i = 0; i < 6; i++)
        if (hw_addr[i] != 0)
            break;

    if (i == 6)
        return(NULL);

    for (i = 0; i < 6; i++)
        dev->dev_addr[i] = hw_addr[i];

    return(&default_info);
} /* get_hwired */

/*
 * pcnet_config() is scheduled to run after a CARD_INSERTION event
 * is received, to configure the PCMCIA socket, and to make the
 * ethernet device available to the system.
 */
static int try_io_port(dev_link_t *link) {
    int j, ret;

    if (link->io.NumPorts1 == 32) {
        link->io.Attributes1 = IO_DATA_PATH_WIDTH_AUTO;
        if (link->io.NumPorts2 > 0) {
            /* for master/slave multifunction cards */
            link->io.Attributes2 = IO_DATA_PATH_WIDTH_8;
            link->irq.Attributes = IRQ_TYPE_EXCLUSIVE | IRQ_FIRST_SHARED;
        }
    } else {
        /* This should be two 16-port windows */
        link->io.Attributes1 = IO_DATA_PATH_WIDTH_8;
        link->io.Attributes2 = IO_DATA_PATH_WIDTH_16;
    }

    if (link->io.BasePort1 == 0) {
        for (j = 0; j < 0x400; j += 0x20) {
            link->io.BasePort1 = j ^ 0x300;
            link->io.BasePort2 = (j ^ 0x300) + 0x10;
            ret = CardServices(RequestIO, link->handle, &link->io);
            if (ret == CS_SUCCESS)
                return(ret);
        }

        return(ret);
    }
    else
        return(CardServices(RequestIO, link->handle, &link->io));
}

static void pcnet_config(dev_link_t *link) {
    client_handle_t handle;
    tuple_t tuple;
    cisparse_t parse;
    pcnet_dev_t *info;
    struct device *dev;
    int i, last_ret, last_fn, start_pg, stop_pg, cm_offset;
    int manfid = 0, prodid = 0, slave = 0;
    u_short buf[64];
    hw_info_t *hw_info;

    handle = link->handle;
    info = link->priv;
    dev = &info->dev;

    DEBUG(0, "pcnet_config(0x%p)\n", link);

    tuple.Attributes = 0;
    tuple.TupleData = (cisdata_t *)buf;
    tuple.TupleDataMax = sizeof(buf);
    tuple.TupleOffset = 0;
    tuple.DesiredTuple = CISTPL_CONFIG;
    if ((last_ret = CardServices((last_fn = GetFirstTuple), handle, &tuple)) ||
        (last_ret = CardServices((last_fn = GetTupleData), handle, &tuple)) ||
        (last_ret = CardServices((last_fn = ParseTuple), handle, &tuple,
        &parse))) {
        cs_error(link->handle, last_fn, last_ret);
        pcnet_release((u_long) link);

        return;
    } 

    link->conf.ConfigBase = parse.config.base;
    link->conf.Present = parse.config.rmask[0];

    /* Configure card */
    link->state |= DEV_CONFIG;

    tuple.DesiredTuple = CISTPL_MANFID;
    tuple.Attributes = TUPLE_RETURN_COMMON;
    if ((CardServices((last_fn = GetFirstTuple), handle, &tuple) == CS_SUCCESS)
        && (CardServices((last_fn = GetTupleData), handle, &tuple) ==
        CS_SUCCESS)) {
        manfid = le16_to_cpu(buf[0]);
        prodid = le16_to_cpu(buf[1]);

#ifdef GENERIC
        if (!is_ken0100a(manfid, prodid)) {
            kernel_message("pcnet_cs: Invalid manfucature ID or product ID.");

            return;
        }
#endif

        slave = ((manfid == MANFID_OLICOM) && (prodid == PRODID_OLICOM_OC2232));
    }
    
    tuple.DesiredTuple = CISTPL_CFTABLE_ENTRY;
    tuple.Attributes = 0;

    if (last_ret = CardServices((last_fn = GetFirstTuple), handle, &tuple)) {
        cs_error(link->handle, last_fn, last_ret);
        pcnet_release((u_long) link);

        return;
    }

    while (last_ret == CS_SUCCESS) {
        cistpl_cftable_entry_t *cfg = &(parse.cftable_entry);
        cistpl_io_t *io = &(parse.cftable_entry.io);
    
        if ((last_ret = CardServices((last_fn = GetTupleData), handle, &tuple))
            || (last_ret = CardServices((last_fn = ParseTuple), handle, &tuple,
            &parse))) {
            last_ret = CardServices(GetNextTuple, handle, &tuple);
            if (last_ret != CS_SUCCESS) {
                cs_error(handle, RequestIO, last_ret);
                pcnet_release((u_long) link);

                return;
            }
        }

        if ((cfg->index == 0) || (cfg->io.nwin == 0) ||
            (slave && ((io->nwin != 2) || (io->win[1].len != 8)))) {
            last_ret = CardServices((last_fn = GetNextTuple), handle, &tuple);
            if (last_ret != CS_SUCCESS) {
                pcnet_release((u_long) link);

                return;
            }
        }

        link->conf.ConfigIndex = cfg->index;

        /*
         * For multifunction cards, by convention, we configure the
         * network function with window 0, and serial with window 1.
         */
        if (io->nwin > 1) {
            i = (io->win[1].len > io->win[0].len);
            link->io.BasePort2 = io->win[1-i].base;
            link->io.NumPorts2 = io->win[1-i].len;
        } else
            i = link->io.NumPorts2 = 0;

        link->io.BasePort1 = io->win[i].base;
        link->io.NumPorts1 = io->win[i].len;

        if (link->io.NumPorts1 + link->io.NumPorts2 >= 32) {
            last_ret = try_io_port(link);
            if (last_ret == CS_SUCCESS)
                break;
            else {
                last_ret = CardServices(GetNextTuple, handle, &tuple);
                if (last_ret != CS_SUCCESS) {
                    cs_error(handle, RequestIO, last_ret);
                    pcnet_release((u_long) link);

                    return;
                }
            }
        }
    }  /* while */

    if (last_ret = CardServices((last_fn = RequestIRQ), handle, &link->irq)) {
        cs_error(link->handle, last_fn, last_ret);
        pcnet_release((u_long) link);

        return;
    }
    
    if (link->io.NumPorts2 == 8) {
        link->conf.Attributes |= CONF_ENABLE_SPKR;
        link->conf.Status = CCSR_AUDIO_ENA;
    }

    if ((manfid == MANFID_IBM) && (prodid == PRODID_IBM_HOME_AND_AWAY))
    link->conf.ConfigIndex |= 0x10;
    
    if (last_ret = CardServices((last_fn = RequestConfiguration), handle,
        &link->conf)) {
        cs_error(link->handle, last_fn, last_ret);
        pcnet_release((u_long) link);

        return;
    }
    dev->irq = link->irq.AssignedIRQ;
    dev->base_addr = link->io.BasePort1;

    if ((if_port == 1) || (if_port == 2))
        dev->if_port = if_port;
    else
        kernel_message("pcnet_cs: invalid if_port requested.");
    dev->tbusy = 0;

    if (register_netdev(dev) != 0) {
        kernel_message("pcnet_cs: register_netdev() failed.");
        pcnet_release((u_long) link);

        return;
    }

    hw_info = get_hwinfo(link);
    if (hw_info == NULL)
        hw_info = get_prom(link);

    if (hw_info == NULL)
        hw_info = get_dl_fast(link);

    if (hw_info == NULL)
        hw_info = get_hwired(link);

    if (hw_info == NULL) {
        kernel_message("pcnet_cs: unable to read MAC address.");
        unregister_netdev(dev);
        pcnet_release((u_long) link);

        return;
    }

    info->flags = hw_info->flags;

    /* Check for user overrides */
    info->flags |= (delay_output) ? DELAY_OUTPUT : 0;

    if ((manfid == MANFID_SOCKET) && (prodid == PRODID_SOCKET_LPE))
        info->flags &= ~USE_BIG_BUF;

    if (!use_big_buf)
        info->flags &= ~USE_BIG_BUF;

    if (use_shmem != -1) {
        info->flags &= ~USE_SHMEM;
        info->flags |= (use_shmem) ? USE_SHMEM : 0;
    }
    
    if (info->flags & USE_BIG_BUF) {
        start_pg = SOCKET_START_PG;
        stop_pg = SOCKET_STOP_PG;
        cm_offset = 0x10000;
    } else {
        start_pg = PCNET_START_PG;
        stop_pg = PCNET_STOP_PG;
        cm_offset = 0;
    }

    if (info->flags & USE_SHMEM)
        setup_shmem_window(link, start_pg, stop_pg, cm_offset);
    else
        setup_dma_config(link, start_pg, stop_pg);

    if (info->flags & USE_SHMEM) {
        char s[256];

        sprintf(s, "%s: mem %#5lx," , dev->name, dev->mem_start);
        kernel_message(s);
        unregister_netdev(dev);
        pcnet_release((u_long) link);

        return;
    }

    ei_status.name = "NE2000";
    ei_status.word16 = 1;
    ei_status.reset_8390 = &pcnet_reset_8390;

    link->dev = &info->node;
    link->state &= ~DEV_CONFIG_PENDING;

    printk(KERN_NOTICE "\n%s: NE2000 Compatible: port %#3lx, irq %d,",
           dev->name, dev->base_addr, dev->irq);

    if (info->flags & HAS_MISC_REG)
        printk(" %s port,", if_names[dev->if_port]);
        printk(" hw_addr ");

    for (i = 0; i < 6; i++)
        printk("%02X%s", dev->dev_addr[i], ((i<5) ? ":" : "\n"));

#ifdef GENERIC
    ken0100a_setup_base(link->io.BasePort1);
#endif

    return;
} /* pcnet_config */

/*
 * After a card is removed, pcnet_release() will unregister the net
 * device, and release the PCMCIA configuration.  If the device is
 * still open, this will be postponed until it is closed.
 */
static void pcnet_release(u_long arg) {
    dev_link_t *link = (dev_link_t *) arg;
    pcnet_dev_t *info = link->priv;
    int i;

    DEBUG(0, "pcnet_release(0x%p)\n", link);

    if (link->open) {
        DEBUG(1, "pcnet_cs: release postponed, '%s' still open\n",
              info->node.dev_name);
        link->state |= DEV_STALE_CONFIG;

        return;
    }

    if (info->flags & USE_SHMEM) {
        iounmap(info->base);
        CardServices(ReleaseWindow, link->win);
    }

    CardServices(ReleaseConfiguration, link->handle);
    CardServices(ReleaseIO, link->handle, &link->io);
    CardServices(ReleaseIRQ, link->handle, &link->irq);

    if (link->dev)
        unregister_netdev(&info->dev);

    link->dev = NULL;

    link->state &= ~(DEV_CONFIG | DEV_RELEASE_PENDING);
    if (link->state & DEV_STALE_LINK)
        pcnet_detach(link);

} /* pcnet_release */

/*
 * The card status event handler.  Mostly, this schedules other
 * stuff to run after an event is received.  A CARD_REMOVAL event
 * also sets some flags to discourage the net drivers from trying
 * to talk to the card any more.
 */
static int pcnet_event(event_t event, int priority,
                       event_callback_args_t *args) {
    dev_link_t *link = args->client_data;
    pcnet_dev_t *info = link->priv;

    DEBUG(2, "pcnet_event(0x%06x)\n", event);

    switch (event) {
        case CS_EVENT_CARD_REMOVAL:
            link->state &= ~DEV_PRESENT;

            if (link->state & DEV_CONFIG) {
                info->dev.tbusy = 1; info->dev.start = 0;
                link->release.expires = RUN_AT(HZ/20);
                link->state |= DEV_RELEASE_PENDING;
                add_timer(&link->release);
            }
            break;

        case CS_EVENT_CARD_INSERTION:
            link->state |= DEV_PRESENT;
            asix_link = link;
            pcnet_config(link);
            break;

        case CS_EVENT_PM_SUSPEND:
            link->state |= DEV_SUSPEND;
            /* Fall through... */

        case CS_EVENT_RESET_PHYSICAL:
            if (link->state & DEV_CONFIG) {
                if (link->open) {
                    info->dev.tbusy = 1; info->dev.start = 0;
                }
                CardServices(ReleaseConfiguration, link->handle);
            }
            break;

        case CS_EVENT_PM_RESUME:
            link->state &= ~DEV_SUSPEND;
            /* Fall through... */

        case CS_EVENT_CARD_RESET:
            if (link->state & DEV_CONFIG) {
                CardServices(RequestConfiguration, link->handle, &link->conf);
                if (link->open) {
                    pcnet_reset_8390(&info->dev);
                    NS8390_init(&info->dev, 1);
                    info->dev.tbusy = 0; info->dev.start = 1;
                }
            }
            break;
    }

    return(0);
} /* pcnet_event */

static void set_misc_reg(struct device *dev) {
    int nic_base = dev->base_addr;
    pcnet_dev_t *info = (pcnet_dev_t *)dev;
    u_char tmp;
    
    if (info->flags & HAS_MISC_REG) {
        tmp = inb_p(nic_base + PCNET_MISC) & ~3;
        if (dev->if_port == 2)
            tmp |= 1;

        if (info->flags & USE_BIG_BUF)
            tmp |= 2;

        if (info->flags & HAS_IBM_MISC)
            tmp |= 8;

        outb_p(tmp, nic_base + PCNET_MISC);
    }
}

static int pcnet_open(struct device *dev) {
    pcnet_dev_t *info = (pcnet_dev_t *)dev;
    dev_link_t *link;
    
    DEBUG(2, "pcnet_open('%s')\n", dev->name);

    for (link = dev_list; link; link = link->next)

    if (link->priv == dev)
        break;

    if (!DEV_OK(link))
        return(-ENODEV);

    link->open++;
    MOD_INC_USE_COUNT;

    /* For D-Link EtherFast, wait for something(?) to happen */
    if (info->flags & IS_DL10019A) {
        int i;

        for (i = 0; i < 20; i++) {
            if ((inb(dev->base_addr+0x1c) & 0x01) == 0)
                break;

            current->state = TASK_INTERRUPTIBLE;
            schedule_timeout(HZ/10);
        }
    }
    
    set_misc_reg(dev);
    REQUEST_IRQ(dev->irq, ei_interrupt, SA_SHIRQ, dev_info, dev);

    return(ei_open(dev));
} /* pcnet_open */

static int pcnet_close(struct device *dev) {
    dev_link_t *link;

    DEBUG(2, "pcnet_close('%s')\n", dev->name);

    for (link = dev_list; link; link = link->next)
        if (link->priv == dev)
            break;

    if (link == NULL)
        return(-ENODEV);

	pcnet_reset_8390(dev);
    FREE_IRQ(dev->irq, dev);
    link->open--; dev->start = 0;
    if (link->state & DEV_STALE_CONFIG) {
        link->release.expires = RUN_AT(HZ/20);
        link->state |= DEV_RELEASE_PENDING;
        add_timer(&link->release);
    }

    MOD_DEC_USE_COUNT;

    return(0);
} /* pcnet_close */

/*
 * Hard reset the card.  This used to pause for the same period that
 * a 8390 reset command required, but that shouldn't be necessary.
 */
static void pcnet_reset_8390(struct device *dev) {
    int nic_base = dev->base_addr;
    int i;

    ei_status.txing = 0;

#ifdef GENERIC
    if (!is_ken0100a(0, 0))
        ei_status.dmaing = 0;
#else
    ei_status.dmaing = 0;
#endif

    outb(inb(nic_base + PCNET_RESET), nic_base + PCNET_RESET);

    for (i = 0; i < 100; i++) {
        if ((inb_p(nic_base + EN0_ISR) & ENISR_RESET) != 0)
            break;

        udelay(100L);
    }

    outb_p(ENISR_RESET, nic_base + EN0_ISR); /* Ack intr. */
    
    if (i == 100) {
        char s[256];

        sprintf(s, "%s: pcnet_reset_8390() did not complete.", dev->name);
        kernel_message(s);
    }

    set_misc_reg(dev);
    
} /* pcnet_reset_8390 */

static int set_config(struct device *dev, struct ifmap *map) {
    pcnet_dev_t *info = (pcnet_dev_t *)dev;

    if ((map->port != (u_char)(-1)) && (map->port != dev->if_port)) {
        if ((map->port != 0) && !(info->flags & HAS_MISC_REG)) {
            char s[256];

            sprintf(s, "%s: transceiver selection not "
                   "implemented\n", dev->name);
            kernel_message(s);

            return(-EINVAL);
        }

        if ((map->port == 1) || (map->port == 2)) {
            char s[256];

            dev->if_port = map->port;
            sprintf(s, "%s: switched to %s port\n", dev->name,
                    if_names[dev->if_port]);
            kernel_message(s);
        } else
            return(-EINVAL);
    }

    return(0);
}

static void dma_get_8390_hdr(struct device *dev, struct e8390_pkt_hdr *hdr,
    int ring_page) {
    int nic_base = dev->base_addr;

    outb_p(0, nic_base + EN0_RSARLO);        /* On page boundary */
    outb_p(ring_page, nic_base + EN0_RSARHI);
    outb_p(E8390_RREAD+E8390_START, nic_base + PCNET_CMD);

    insw_ns(nic_base + PCNET_DATAPORT, hdr, sizeof(struct e8390_pkt_hdr)>>1);
}

static void dma_block_input(struct device *dev, int count, struct sk_buff *skb,
     int ring_offset) {
    int nic_base = dev->base_addr;
    int xfer_count = count;
    char *buf = skb->data;

#ifdef PCMCIA_DEBUG
    if ((ei_debug > 4) && (count != 4))
    printk(KERN_DEBUG "%s: [bi=%d]\n", dev->name, count+4);
#endif

    outb_p(ring_offset & 0xff, nic_base + EN0_RSARLO);
    outb_p(ring_offset >> 8, nic_base + EN0_RSARHI);
    outb_p(E8390_RREAD+E8390_START, nic_base + PCNET_CMD);

    insw_ns(nic_base + PCNET_DATAPORT,buf,count>>1);
    if (count & 0x01)
        buf[count-1] = inb(nic_base + PCNET_DATAPORT), xfer_count++;

#ifdef PCMCIA_DEBUG
    if (ei_debug > 4) {        // DMA termination address check...
        int addr, tries = 20;
        do {
            /*
             * DON'T check for 'inb_p(EN0_ISR) & ENISR_RDC' here
             * it's broken for Rx on some cards!
             */
            int high = inb_p(nic_base + EN0_RSARHI);
            int low = inb_p(nic_base + EN0_RSARLO);

            addr = (high << 8) + low;
            if (((ring_offset + xfer_count) & 0xff) == (addr & 0xff))
            break;
        } while (--tries > 0);

        if (tries <= 0) {
            char s[256];
    
            sprintf(s, "%s: RX transfer address mismatch,"
                   "%#4.4x (expected) vs. %#4.4x (actual).\n",
                   dev->name, ring_offset + xfer_count, addr);
            kernel_message(s);
        }
    }
#endif

}

static void dma_block_output(struct device *dev, int count,
    const unsigned char *buf, const int start_page) {
    unsigned long flags;
    int nic_base = dev->base_addr;
    pcnet_dev_t *info = (pcnet_dev_t *)dev;

#ifdef PCMCIA_DEBUG
    int retries = 0;
#endif

    u_long dma_start;

#ifdef PCMCIA_DEBUG
    if (ei_debug > 4)
    printk(KERN_DEBUG "%s: [bo=%d]\n", dev->name, count);
#endif

    /*
     * Round the count up for word writes.  Do we need to do this?
     * What effect will an odd byte count have on the 8390?
     * I should check someday.
     */
    if (count & 0x01)
        count++;

    save_flags(flags);
    cli();

#ifdef PCMCIA_DEBUG
  retry:
#endif

    outb_p(0x00, nic_base + EN0_RSARLO);
    outb_p(start_page, nic_base + EN0_RSARHI);
    outb_p(E8390_RWRITE+E8390_START, nic_base + PCNET_CMD);
    outsw_ns(nic_base + PCNET_DATAPORT, buf, count>>1);

    dma_start = jiffies;

#ifdef PCMCIA_DEBUG
    /*
     * This was for the ALPHA version only, but enough people have
     * encountering problems that it is still here.
     */
    if (ei_debug > 4) {    /* DMA termination address check... */
        int addr, tries = 20;

        do {
            int high = inb_p(nic_base + EN0_RSARHI);
            int low = inb_p(nic_base + EN0_RSARLO);
            addr = (high << 8) + low;
            if ((start_page << 8) + count == addr)
            break;
        } while (--tries > 0);

        if (tries <= 0) {
            char s[256];

            sprintf(s, "%s: Tx packet transfer address mismatch,"
                   "%#4.4x (expected) vs. %#4.4x (actual).\n", dev->name,
                   (start_page << 8) + count, addr);
            kernel_message(s);

            if (retries++ == 0)
                goto retry;
        }
    }
#endif

    restore_flags(flags);
}

static int setup_dma_config(dev_link_t *link, int start_pg, int stop_pg) {
    struct device *dev = link->priv;

    ei_status.tx_start_page = start_pg;
    ei_status.rx_start_page = start_pg + TX_PAGES;
    ei_status.stop_page = stop_pg;

    /* set up block i/o functions */
    ei_status.get_8390_hdr = &dma_get_8390_hdr;
    ei_status.block_input = &dma_block_input;
    ei_status.block_output = &dma_block_output;

    return(0);
}

static void copyin(unsigned char *dest, unsigned char *src, int c) {
    unsigned short *d = (unsigned short *) dest;
    unsigned short *s = (unsigned short *) src;
    int odd;

    if (c <= 0)
        return;

    odd = (c & 01); c >>= 1;

    if (c) {
        do {
            *d++ = readw_ns(s++);
           } while (--c);
    }

    /* get last byte by fetching a word and masking */
    if (odd)
        *((unsigned char *) d) = readw(s) & 0xff;
}

static void copyout(unsigned char *dest, const unsigned char *src, int c) {
    volatile unsigned short *d = (unsigned short *) dest;
    unsigned short *s = (unsigned short *) src;
    int odd;

    if (c <= 0)
        return;

    odd = (c & 01); c >>= 1;

    if (c) {
        do {
            writew_ns(*s++, d++);
           } while (--c);
    }

    /* copy last byte doing a read-modify-write */
    if (odd)
        writew((readw(d) & 0xff00) | *(u_char *)s, d);
}

static void shmem_get_8390_hdr(struct device *dev, struct e8390_pkt_hdr *hdr,
    int ring_page) {
    void *xfer_start = (void *)(dev->rmem_start + (ring_page << 8)
                - (ei_status.rx_start_page << 8));
    
    copyin((void *)hdr, xfer_start, sizeof(struct e8390_pkt_hdr));
}

static void shmem_block_input(struct device *dev, int count,
    struct sk_buff *skb, int ring_offset) {
    void *xfer_start = (void *)(dev->rmem_start + ring_offset
                       - (ei_status.rx_start_page << 8));
    char *buf = skb->data;
    
    if (xfer_start + count > (void *)dev->rmem_end) {
        /* We must wrap the input move. */
        int semi_count = (void*)dev->rmem_end - xfer_start;
        copyin(buf, xfer_start, semi_count);
        buf += semi_count;
        ring_offset = ei_status.rx_start_page << 8;
        xfer_start = (void *)dev->rmem_start;
        count -= semi_count;
    }

    copyin(buf, xfer_start, count);
}

static void shmem_block_output(struct device *dev, int count,
    const unsigned char *buf, const int start_page) {
    void *shmem = (void *)dev->mem_start + (start_page << 8);
    shmem -= ei_status.tx_start_page << 8;

    if (ei_debug > 4) {
        char s[256];

        sprintf(s, "[bo=%d @ %x]\n", count, start_page);
        kernel_message(s);
    }

    copyout(shmem, buf, count);
}

static int setup_shmem_window(dev_link_t *link, int start_pg, int stop_pg,
                              int cm_offset) {
    win_req_t req;
    memreq_t mem;
    int offset = 0, last_ret, last_fn;
    struct device *dev = link->priv;
    pcnet_dev_t *info = link->priv;
    int window_size;

    window_size = (stop_pg - start_pg) << 8;
    if (window_size > 32 * 1024)
    window_size = 32 * 1024;

    /* Make sure it's a power of two.  */
    while ((window_size & (window_size - 1)) != 0)
    window_size += window_size & ~(window_size - 1);

    /* Allocate a memory window */
    req.Attributes = WIN_DATA_WIDTH_16 | WIN_MEMORY_TYPE_CM | WIN_ENABLE |
                     WIN_USE_WAIT;
    req.Base = 0; req.Size = window_size;
    req.AccessSpeed = mem_speed;
    link->win = (window_handle_t)link->handle;
    if (last_ret = CardServices((last_fn = RequestWindow), &link->win,
        &req)) {
        cs_error(link->handle, last_fn, last_ret);
        return(1);
    }

    mem.CardOffset = (start_pg << 8) + cm_offset;
    offset = mem.CardOffset % window_size;
    mem.CardOffset -= offset;
    mem.Page = 0;
    if (last_ret = CardServices((last_fn = MapMemPage), link->win, &mem)) {
        cs_error(link->handle, last_fn, last_ret);
        return(1);
    }

    info->base = ioremap(req.Base, window_size);
    dev->mem_start = (u_long) info->base + offset;
    dev->rmem_start = dev->mem_start + (TX_PAGES<<8);
    dev->mem_end = dev->rmem_end = (u_long) info->base + req.Size;

    ei_status.tx_start_page = start_pg;
    ei_status.rx_start_page = start_pg + TX_PAGES;
    ei_status.stop_page = start_pg + ((req.Size - offset) >> 8);

    /* set up block i/o functions */
    ei_status.get_8390_hdr = &shmem_get_8390_hdr;
    ei_status.block_input = &shmem_block_input;
    ei_status.block_output = &shmem_block_output;

    return(0);
}

int init_module(void) {
    servinfo_t serv;
    DEBUG(0, "%s\n", version);
    CardServices(GetCardServicesInfo, &serv);
    if (serv.Revision != CS_RELEASE_CODE) {
        char s[256];

        sprintf(s, "pcnet_cs: Card Services release "
               "does not match!\n");
        kernel_message(s);

        return(-1);
    }

    register_pcmcia_driver(&dev_info, &pcnet_attach, &pcnet_detach);

#ifdef GENERIC
    ken0100a_init();
#endif

    return(0);
}

void cleanup_module(void) {
    DEBUG(0, "pcnet_cs: unloading\n");
    unregister_pcmcia_driver(&dev_info);
    while (dev_list != NULL)
        pcnet_detach(dev_list);
}
