#include <stdio.h>
#include <string.h>
#include "latex2lyx.h"

/*----------------------------------------------------------------------------*/
/*      Terminals - Constants definitions                                     */
/*----------------------------------------------------------------------------*/

#define L_documentstyle 1
#define L_pagenumbering 2
#define L_pagestyle 3
#define L_maketitle 4
#define L_tableofcontents 5
#define L_begin 6
#define L_end 7
#define L_item 8
#define L_bibitem 9
#define L_label 10
#define L_ref 11
#define L_cite 12
#define L_noindent 13
#define L_input 14
#define L_include 15
#define L_includeonly 16
#define L_centerline 17
#define L_mbox 18
#define L_fbox 19
#define L_framebox 20
#define L_sbox 21
#define L_savebox 22
#define L_newsavebox 23
#define L_usebox 24
#define L_newpage 25
#define L_clearpage 26
#define L_samepage 27
#define L_linebreak 28
#define L_nolinebreak 29
#define L_sloppy 30
#define L_hline 31
#define L_cline 32
#define L_multicolumn 33
#define L_caption 34
#define L_kill 35
#define L_vspace 36
#define L_hspace 37
#define L_vfill 38
#define L_hfill 39
#define L_put 40
#define L_multiput 41
#define L_makebox 42
#define L_dashbox 43
#define L_line 44
#define L_vector 45
#define L_shortstack 46
#define L_circle 47
#define L_oval 48
#define L_frame 49
#define L_thinlines 50
#define L_thicklines 51
#define L_sqrt 52
#define L_frac 53
#define L_overline 54
#define L_appendix 55
#define L_tabset 56
#define L_tabskip 57
#define L_pmod 58
#define L_vec 59
#define L_stackrel 60
#define L_centering 61
#define L_lbracket 62
#define L_rbracket 63
#define L_lcurly 64
#define L_rcurly 65
#define L_word 66
#define L_linefeed 67
#define L_bibliographystyle 68
#define L_newline 69
#define L_smallspace 70
#define L_today 71
#define L_footnote 72
#define L_subscript 73
#define L_superscript 74
#define L_thinspace 75
#define L_negspace 76
#define L_mediumspace 77
#define L_thickspace 78
#define L_plus 79
#define L_div 80
#define L_equal 81
#define L_mathbegin 82
#define L_mathend 83
#define L_less 84
#define L_more 85
#define L_amper 86
#define L_marginpar 87
#define L_lparen 88
#define L_rparen 89
#define L_tilde 90
#define L_dollar 91
#define L_backslash 92
#define L_mathspace 93
#define L_box 94
#define L_textwidth 95
#define L_bibliography 96
#define L_linespacing 97
#define L_pagebreak 98
#define L_nopagebreak 99
#define L_Heading 100
#define L_FontShape 101
#define L_FontSize 102
#define L_Environment 103
#define L_DocStyle 104
#define L_MathGreek 105
#define L_MathSymbol 106
#define L_MathDelim 107
#define L_accent 108
#define L_LatexMisc 109
#define L_space 110
#define L_document 111
#define L_Heading_star 112
#define L_pageref 113
#define L_rule 114
#define L_bezier 115


/*============================================================================*/
/* 	Set of routines to emulate a stack of integer			      */
/*============================================================================*/

inline void	push (int *stack, int *top, int data)
{
    stack[*top] = data;
    (*top)++;
}

inline int	pop  (int *stack, int *top)
{
    (*top)--;
    return stack[*top];
}


/*============================================================================*/
/*	get_isolat - Convert a accent and a letter to a isolatin1 char	      */
/*============================================================================*/

char		get_isolat (int accent, int c)
{
    switch (accent) 
    {
      case 0x27:
	switch (c)
	{
	  case 'a': return 0xe1;	  case 'e': return 0xe9;
	  case 'i': return 0xed;	  case 'o': return 0xf3;
	  case 'u': return 0xfa;	  case 'y': return 0xfd;
	  case 'A': return 0xc1;	  case 'E': return 0xc9;
	  case 'I': return 0xcd;	  case 'O': return 0xd3;
	  case 'U': return 0xda;
	}
      case '`':
	switch (c)
	{
	  case 'a': return 0xe0;	  case 'e': return 0xe8;
	  case 'i': return 0xec;	  case 'o': return 0xf2;
	  case 'u': return 0xf9;
	  case 'A': return 0xc0;	  case 'E': return 0xc8;
	  case 'I': return 0xcc;	  case 'O': return 0xd2;
	  case 'U': return 0xd9;
	}
      case '^':
	switch (c)
	{
	  case 'a': return 0xe2;	  case 'e': return 0xea;
	  case 'i': return 0xee;	  case 'o': return 0xf4;
	  case 'u': return 0xfb;
	  case 'A': return 0xc2;	  case 'E': return 0xca;
	  case 'I': return 0xce;	  case 'O': return 0xd4;
	  case 'U': return 0xdb;
	}
      case '~':
	switch (c)
	{
	  case 'a': return 0xe3;	  case 'o': return 0xf5;
	  case 'n': return 0xf1;
	  case 'A': return 0xc3;	  case 'O': return 0xd5;
	  case 'N': return 0xd1;
	}
      case '"':
	switch (c)
	{
	  case 'a': return 0xe4;	  case 'e': return 0xeb;
	  case 'i': return 0xef;	  case 'o': return 0xf6;
	  case 'u': return 0xfc;	  case 'y': return 0xff;
	  case 'A': return 0xc4;	  case 'E': return 0xcb;
	  case 'I': return 0xcf;	  case 'O': return 0xd6;
	  case 'U': return 0xdc;
	}
      case 'c':
	switch (c)
	{
	  case 'c': return 0xe7;	  case 'C': return 0xc7;
	}
      case '.':
      case 'd':
      case 'b':
      case 'v':
      case 'H':
      case 't':
      case 'u':
      default:
	return 0x00;
    }
}


/*============================================================================*/
/*	print_lyx - Output LyX code for a given latex parse tree	      */
/*============================================================================*/

int		lf_c = 0;
int		math_mode = 0;
int		latex_mode = 0;
int		align = 0;
int		layout = 0;
int		paperfontsize = 10;
int		env_stack[32];
int		environ = -1, env_depth = 0;
int		fam_stack[32], ser_stack[32], shp_stack[32], siz_stack[32], bar_stack[32];
int		fam_top = 0, ser_top = 0, shp_top = 0, siz_top = 0, bar_top = 0;
int		family = -1, series = -1, shape = -1, size = -1, bar = -1;
int		change = 0;

char		opt_buf[1000];
char		pack_buf[1000];

#define SKIP 	while(tree->last){if(tree==start)return;tree=tree->next;}\
                if(tree==start)return;tree=tree->next;continue


void		print_lyx (FILE *fout, Node *start, Node *tree)
{
    Node	*t, *t2;
    char	c;
    char	*ptr, *ptr2, *ptr3;
    int		k;

    do {

	if (tree->token != L_linefeed) {
	    if (lf_c >= 2) {
		if (tree->token != L_item && tree->token != L_Environment &&
		    tree->token != L_Heading && tree->token != L_Heading_star) {
		    if (layout != 0)
		        fprintf (fout, "\n\\layout %s\n",EnvironmentName[tree->data.spec]);
		    else 
		        fprintf(fout, "\n\\layout Standard\n\n");
		}
		if (align == 'l')
		    fprintf(fout, "\n\\layout Standard\n\\align left\n");
		else if (align == 'r')
		    fprintf(fout, "\n\\layout Standard\n\\align right\n");
		else if (align == 'c')
		    fprintf(fout, "\n\\layout Standard\n\\align center\n");
		/****** A new paragraph reset all defaults */
		family=-1; series=-1; shape=-1; size=-1; bar=-1;
		fam_top=0; ser_top=0; shp_top=0; siz_top=0; bar_top=0;
	    }
	    lf_c = 0;
	}


	switch (tree->token)
	{
	  case L_latex:
	    t = tree->down;

	    /****** Is there a head ? */
	    if (t->token != L_documentstyle) {
		/* SKIP IT: print_lyx (fout, t, t); */
		t = t->next;
	    }

	    /****** Process document style */
	    fprintf(fout, "#This file was created by latex2lyx 1.0 (public copyright)\n");
	    fprintf(fout, "\\lyxformat 2.1\n");
	    fprintf(fout, "\\textclass %s\n", DocStyleName[t->data.spec]);

	    /****** Is there options ? */
	    if (t->down != NULL) {
		opt_buf[0] = '\0';
		pack_buf[0] = '\0';
		ptr = t->down->down->data.ptr;
		ptr2 = pack_buf;
		do {
		    if (strncmp(ptr, "10pt", 4) == 0)
		        paperfontsize = 10;
		    else if (strncmp(ptr, "11pt", 4) == 0)
		        paperfontsize = 11;
		    else if (strncmp(ptr, "12pt", 4) == 0)
		        paperfontsize = 12;
		    else {
			k = 0;
			if (strncmp(ptr, "twosided", 8) == 0)
			    k = 8;
			else if(strncmp(ptr, "dvips", 5) == 0)
			    k = 5;
			else if (strncmp(ptr, "oneside", 7) == 0)
			    k = 7;
			else if (strncmp(ptr, "onecolumn", 9) == 0)
			    k = 9;

			if (k != 0) {
			    if (opt_buf[0] != '\0')
			        strcat (opt_buf, ",");
			    strncat (opt_buf, ptr, k);
			} else {
			    ptr3 = ptr;
			    if (pack_buf[0] != '\0')
			        *ptr2++ = ',';
			    while (*ptr3 && *ptr3 != ',')
			        *ptr2++ = *ptr3++;
			    *ptr2 = '\0';
			}
		    }
		    while (*ptr && *ptr != ',')
		        ptr++;
		    if (*ptr == ',')
		        ptr++;
		} while (*ptr);
		if (opt_buf[0] != '\0')
		    fprintf(fout, "\\options %s\n", opt_buf);
		if (pack_buf[0] != '\0') {
		    fprintf(fout, "\\begin_preamble\n\\usepackage{%s}\n\\end_preamble\n",
			    pack_buf);
		}
	    }
	    t = t->next;
	    
	    /****** Is there a preambule ? */
	    if (t->last == 0) {
		print_lyx (fout, t, t);
		t = t->next;
	    }
	    /****** Output other LyX structures */
	    fprintf(fout, "\\language default\n");
	    fprintf(fout, "\\inputencoding default\n");	
	    fprintf(fout, "\\fontscheme default\n");
	    fprintf(fout, "\\epsfig none\n");
	    fprintf(fout, "\\papersize usletter\n");
	    fprintf(fout, "\\paperfontsize %d\n", paperfontsize);	
	    fprintf(fout, "\\baselinestretch 1.00\n");
	    fprintf(fout, "\\secnumdepth 3\n");
	    fprintf(fout, "\\tocdepth 3\n");
	    fprintf(fout, "\\paragraph_separation indent\n");	
	    fprintf(fout, "\\quotes_language english\n");
	    fprintf(fout, "\\quotes_times 2\n");
	    fprintf(fout, "\\paperorientation portrait\n");
	    fprintf(fout, "\\papercolumns 1\n");
	    fprintf(fout, "\\papersides 1\n");
	    fprintf(fout, "\\paperpagestyle plain\n");
	    fprintf(fout, "\n\\layout Standard\n\n");

	    /****** Process document body */
	    tree = t;
	    print_lyx(fout, tree, tree);
	    return;
	    break;

	  case L_word:
	    fprintf (fout, "%s", tree->data.ptr);
	    break;

	  case L_space:
	    fprintf (fout, " ");
	    break;

	  case L_linefeed:
	    if (lf_c == 0) {
		fprintf (fout, "\n ");
	    }
	    lf_c ++;
	    break;

	  case L_backslash:
	    if (math_mode) {
		fprintf(fout, "\\");
		print_lyx (fout, tree->down, tree->down);
	    } else {
		if (!latex_mode) {
		    fprintf(fout, "\n\\latex latex\n\\backslash\n");
		    latex_mode = 1; print_lyx (fout, tree->down, tree->down); latex_mode = 0;
		    fprintf(fout, "\n\\latex default\n");
		} else {
		    fprintf(fout, "\n\\backslash\n");
		    print_lyx (fout, tree->down, tree->down);
		}
	    }
	    SKIP;
	    break;

	  case L_latex_nop:
	    switch (tree->data.spec)
	    {
		/****** Those commands are latex commands that has no lyx equivalent. We
		  must translate then as latex, until the day lyx can handle them */
	      case L_appendix:
	      case L_box:
	      case L_maketitle:
	      case L_samepage:
	      case L_sloppy:
	      case L_textwidth:
	      case L_today:
	      case L_centering:
		if (!latex_mode) {
		    fprintf(fout, "\n\\latex latex\n\\backslash\n%s",CommandName[tree->data.spec]);
		    fprintf(fout, "\n\\latex default\n");
		} else {
		    fprintf(fout, "\n\\backslash\n%s",CommandName[tree->data.spec]);
		}
		break;
	      case L_noindent:
	      case L_hfill:    
		fprintf(fout, "\n\\%s\n", CommandName[tree->data.spec]);
		break;
	      case L_newpage:
	      case L_clearpage:
		fprintf(fout, "\\pagebreak_top\n");
		break;
	      case L_vfill:
		fprintf(fout, "\\fill_top\n");
		break;
	    }
	    break;

	  case L_latex_1op:
	    switch (tree->data.spec)
	    {
		/****** Those commands are latex commands that has no lyx equivalent. We
		  must translate then as latex, until the day lyx can handle them */
	      case L_bibliography:
	      case L_bibliographystyle:
	      case L_input:
	      case L_fbox:
	      case L_mbox:
	      case L_newsavebox:
	      case L_pagenumbering: /* @@@ translate to lyx */
	      case L_pagestyle:     /* @@@ translate to lyx \paperpagestyle */
	      case L_usebox:
	      case L_vspace: /* @@@ move to paragraph */
	      case L_hspace: /* @@@ move to paragraph */
	      case L_cite:
	      case L_circle:
	      case L_frame:
	      case L_centerline:
	      case L_linespacing:
	      case L_cline:
	      case L_include:
	      case L_includeonly:
		/****** Those are without optional argument */
	      case L_bibitem:
	      case L_shortstack:
		if (math_mode) {
		    fprintf(fout, "\\%s{", CommandName[tree->data.spec]);
		    print_lyx(fout, tree->down, tree->down);
		    fprintf(fout, "}");
		} else {
		    if (!latex_mode) {
			fprintf(fout, "\n\\latex latex\n\\backslash\n%s{",CommandName[tree->data.spec]);
			latex_mode = 1; print_lyx(fout, tree->down, tree->down); latex_mode = 0;
			fprintf(fout, "}\n\\latex default\n");
		    } else {
			fprintf(fout, "\n\\backslash\n%s{",CommandName[tree->data.spec]);
			print_lyx(fout, tree->down, tree->down); 
			fprintf(fout, "}\n");
		    }
		}
		break;
	      case L_footnote:
	      case L_marginpar:
		fprintf(fout, "\n\\begin_float %s\n\\layout Standard\n", CommandName[tree->data.spec]);
		print_lyx(fout, tree->down, tree->down); 
		fprintf(fout, "\n\\end_float\n");
		break;
	      case L_label:
		if (math_mode) {
		    fprintf(fout, "\n\\label{");
		    print_lyx(fout, tree->down, tree->down); 
		    fprintf(fout, "}\n");
		} else {
		    fprintf(fout, "\n\\begin_inset Label ");
		    print_lyx(fout, tree->down, tree->down); 
		    fprintf(fout, "\n\\end_inset\n");
		}
		break;
	      case L_ref:
	      case L_pageref:
		fprintf(fout, "\n\\begin_inset LatexDel \\%s{", CommandName[tree->data.spec]);
		print_lyx(fout, tree->down, tree->down); 
		fprintf(fout, "}\n\\end_inset\n");
		break;
	      case L_overline:
	      case L_pmod:
	      case L_sqrt:
	      case L_vec:
		fprintf(fout, "\\%s{", CommandName[tree->data.spec]);
		print_lyx(fout, tree->down, tree->down);
		fprintf(fout, "}");
		break;
	      case L_subscript:
	      case L_superscript:
		if (math_mode) {
		    fprintf(fout, "%s{", CommandName[tree->data.spec]);
		    print_lyx(fout, tree->down, tree->down);
		    fprintf(fout, "}");
		} else {
		    fprintf(fout, "_");
		    print_lyx(fout, tree->down, tree->down);
		}
		break;
	      case L_caption:
		fprintf(fout, "\n\\layout Caption\n");
		print_lyx(fout, tree->down, tree->down); 
		break;
	    }
	    SKIP;
	    break;

	  case L_latex_2op:
	    switch (tree->data.spec)
	    {
		/****** Those commands are latex commands that has no lyx equivalent. We
		  must translate then as latex, until the day lyx can handle them */
	      case L_sbox:
	      case L_rule:
		/* Arguments for those are {} {} */
		if (!latex_mode) {
		    fprintf(fout, "\n\\latex latex\n\\backslash\n%s{",CommandName[tree->data.spec]);
		    latex_mode = 1; print_lyx(fout, tree->down, tree->down);
		    fprintf(fout, "}{");
		    print_lyx(fout, tree->down->next, tree->down->next); latex_mode = 0;
		    fprintf(fout, "}\n\\latex default\n");
		} else {
		    fprintf(fout, "\n\\backslash\n%s{",CommandName[tree->data.spec]);
		    print_lyx(fout, tree->down, tree->down); 
		    fprintf(fout, "}{");
		    print_lyx(fout, tree->down->next, tree->down->next);
		    fprintf(fout, "}\n");
		}
		break;
	      case L_line:
	      case L_vector:
	      case L_put:
		/****** Those are without optional argument */
	      case L_makebox:
	      case L_framebox:
		/* Arguments for those are () {} */
		if (!latex_mode) {
		    fprintf(fout, "\n\\latex latex\n\\backslash\n%s(",CommandName[tree->data.spec]);
		    latex_mode = 1; print_lyx(fout, tree->down, tree->down); 
		    fprintf(fout, "){");
		    print_lyx(fout, tree->down->next, tree->down->next); latex_mode = 0;
		    fprintf(fout, "}\n\\latex default\n");
		} else { 
		    fprintf(fout, "\n\\backslash\n%s(",CommandName[tree->data.spec]);
		    print_lyx(fout, tree->down, tree->down); 
		    fprintf(fout, "){");
		    print_lyx(fout, tree->down->next, tree->down->next);
		    fprintf(fout, "}\n");
		}
		break;
	      case L_bibitem:
	      case L_shortstack:
		/* Arguments for those are [] {} */
		if (!latex_mode) {
		    fprintf(fout, "\n\\latex latex\n\\backslash\n%s[",CommandName[tree->data.spec]);
		    latex_mode = 1; print_lyx(fout, tree->down, tree->down); 
		    fprintf(fout, "]{");
		    print_lyx(fout, tree->down->next, tree->down->next); latex_mode = 0;
		    fprintf(fout, "}\n\\latex default\n");
		} else {
		    fprintf(fout, "\n\\backslash\n%s[",CommandName[tree->data.spec]);
		    print_lyx(fout, tree->down, tree->down); 
		    fprintf(fout, "]{");
		    print_lyx(fout, tree->down->next, tree->down->next);
		    fprintf(fout, "}\n");
		}
		break;
	      case L_frac:
	      case L_stackrel:
		fprintf(fout, "\\%s{", CommandName[tree->data.spec]);
		print_lyx(fout, tree->down, tree->down);
		fprintf(fout, "}{");
		print_lyx(fout, tree->down->next, tree->down->next);
		fprintf(fout, "}");
		break;
	      case L_caption: /* @@@ Don't know what to do with [] option */
		/* Arguments for those are [] {} */
		fprintf(fout, "\n\\layout Caption\n");
		print_lyx(fout, tree->down->next, tree->down->next); 
		break;
	    }
	    SKIP;
	    break;

	  case L_latex_3op:
	    switch (tree->data.spec)
	    {
		/****** Should never occur */
	      case L_multicolumn:
		fprintf(stderr, "*");
		break;
	      case L_makebox:
	      case L_framebox:
		/* Arguments for those are () [] {} */
		if (!latex_mode) {
		    fprintf(fout, "\n\\latex latex\n\\backslash\n%s(",CommandName[tree->data.spec]);
		    latex_mode = 1; print_lyx(fout, tree->down, tree->down); 
		    fprintf(fout, ")[");
		    print_lyx(fout, tree->down->next, tree->down->next);
		    fprintf(fout, "]{");
		    print_lyx(fout, tree->down->next->next, tree->down->next->next);
		    latex_mode = 0;
		    fprintf(fout, "}\n\\latex default\n");
		} else {
		    fprintf(fout, "\n\\backslash\n%s(",CommandName[tree->data.spec]);
		    print_lyx(fout, tree->down, tree->down); 
		    fprintf(fout, ")[");
		    print_lyx(fout, tree->down->next, tree->down->next);
		    fprintf(fout, "]{");
		    print_lyx(fout, tree->down->next->next, tree->down->next->next);
		    fprintf(fout, "}\n");
		}
		break;
	      case L_rule:
		/* Arguments for those are [] {} {} */
		if (!latex_mode) {
		    fprintf(fout, "\n\\latex latex\n\\backslash\n%s[",CommandName[tree->data.spec]);
		    latex_mode = 1; print_lyx(fout, tree->down, tree->down); 
		    fprintf(fout, "]{");
		    print_lyx(fout, tree->down->next, tree->down->next);
		    fprintf(fout, "}{");
		    print_lyx(fout, tree->down->next->next, tree->down->next->next);
		    latex_mode = 0;
		    fprintf(fout, "}\n\\latex default\n");
		} else {
		    fprintf(fout, "\n\\backslash\n%s[",CommandName[tree->data.spec]);
		    print_lyx(fout, tree->down, tree->down); 
		    fprintf(fout, "]{");
		    print_lyx(fout, tree->down->next, tree->down->next);
		    fprintf(fout, "}{");
		    print_lyx(fout, tree->down->next->next, tree->down->next->next);
		    fprintf(fout, "}\n");
		}
		break;
		/****** Those are without optional argument */
	      case L_dashbox:
	      case L_savebox:
		/* Arguments for those are {} () {} */
		if (!latex_mode) {
		    fprintf(fout, "\n\\latex latex\n\\backslash\n%s{",CommandName[tree->data.spec]);
		    latex_mode = 1; print_lyx(fout, tree->down, tree->down); 
		    fprintf(fout, "}(");
		    print_lyx(fout, tree->down->next, tree->down->next);
		    fprintf(fout, "){");
		    print_lyx(fout, tree->down->next->next, tree->down->next->next);
		    latex_mode = 0;
		    fprintf(fout, "}\n\\latex default\n");
		} else {
		    fprintf(fout, "\n\\backslash\n%s{",CommandName[tree->data.spec]);
		    print_lyx(fout, tree->down, tree->down); 
		    fprintf(fout, "}(");
		    print_lyx(fout, tree->down->next, tree->down->next);
		    fprintf(fout, "){");
		    print_lyx(fout, tree->down->next->next, tree->down->next->next);
		    fprintf(fout, "}\n");
		}
		break;
	    }
	    SKIP;
	    break;

	  case L_latex_4op:
	    switch (tree->data.spec)
	    {
		/****** Those commands are latex commands that has no lyx equivalent. We
		  must translate then as latex, until the day lyx can handle them */
	      case L_multiput:
		/* Arguments for those are: () () {} {} */
		if (!latex_mode) {
		    fprintf(fout, "\n\\latex latex\n\\backslash\n%s(",CommandName[tree->data.spec]);	
		    latex_mode = 1; print_lyx(fout, tree->down, tree->down); 
		    fprintf(fout, ")(");
		    print_lyx(fout, tree->down->next, tree->down->next);
		    fprintf(fout, "){");
		    print_lyx(fout, tree->down->next->next, tree->down->next->next);
		    fprintf(fout, "}{");
		    print_lyx(fout, tree->down->next->next->next, tree->down->next->next->next);
		    latex_mode = 0;
		    fprintf(fout, "}\n\\latex default\n");
		} else {
		    fprintf(fout, "\n\\backslash\n%s(",CommandName[tree->data.spec]);	
		    print_lyx(fout, tree->down, tree->down); 
		    fprintf(fout, ")(");
		    print_lyx(fout, tree->down->next, tree->down->next);
		    fprintf(fout, "){");
		    print_lyx(fout, tree->down->next->next, tree->down->next->next);
		    fprintf(fout, "}{");
		    print_lyx(fout, tree->down->next->next->next, tree->down->next->next->next);
		    fprintf(fout, "}\n");
		}
		break;
	      case L_bezier:
		/* Arguments for those are: {} () () () */
		if (!latex_mode) {
		    fprintf(fout, "\n\\latex latex\n\\backslash\n%s{",CommandName[tree->data.spec]);	
		    latex_mode = 1; print_lyx(fout, tree->down, tree->down); 
		    fprintf(fout, "}(");
		    print_lyx(fout, tree->down->next, tree->down->next);
		    fprintf(fout, ")(");
		    print_lyx(fout, tree->down->next->next, tree->down->next->next);
		    fprintf(fout, ")(");
		    print_lyx(fout, tree->down->next->next->next, tree->down->next->next->next);
		    latex_mode = 0;
		    fprintf(fout, ")\n\\latex default\n");
		} else {
		    fprintf(fout, "\n\\backslash\n%s{",CommandName[tree->data.spec]);	
		    print_lyx(fout, tree->down, tree->down); 
		    fprintf(fout, "}(");
		    print_lyx(fout, tree->down->next, tree->down->next);
		    fprintf(fout, ")(");
		    print_lyx(fout, tree->down->next->next, tree->down->next->next);
		    fprintf(fout, ")(");
		    print_lyx(fout, tree->down->next->next->next, tree->down->next->next->next);
		    fprintf(fout, ")\n");
		}
		break;
	      case L_dashbox:
	      case L_savebox:
		/* Arguments for those are: {} () [] {} */
		if (!latex_mode) {
		    fprintf(fout, "\n\\latex latex\n\\backslash\n%s{",CommandName[tree->data.spec]);
		    latex_mode = 1; print_lyx(fout, tree->down, tree->down); 
		    fprintf(fout, "}(");
		    print_lyx(fout, tree->down->next, tree->down->next);
		    fprintf(fout, ")[");
		    print_lyx(fout, tree->down->next->next, tree->down->next->next);
		    fprintf(fout, "]{");
		    print_lyx(fout, tree->down->next->next->next, tree->down->next->next->next);
		    latex_mode = 0;
		    fprintf(fout, "}\n\\latex default\n");
		} else {
		    fprintf(fout, "\n\\backslash\n%s{",CommandName[tree->data.spec]);
		    print_lyx(fout, tree->down, tree->down); 
		    fprintf(fout, "}(");
		    print_lyx(fout, tree->down->next, tree->down->next);
		    fprintf(fout, ")[");
		    print_lyx(fout, tree->down->next->next, tree->down->next->next);
		    fprintf(fout, "]{");
		    print_lyx(fout, tree->down->next->next->next, tree->down->next->next->next);
		    fprintf(fout, "}\n");
		}
		break;

	    }
	    SKIP;
	    break;

	  case L_Heading:
	    fprintf(fout, "\n\\layout %s\n\n", HeadingName[tree->data.spec]);
	    t = tree->down;
	    if (t->token == L_optional_spec)
	        /****** A option for section was made, no lyx equivalent, discard */
	        t = t->next;
	    print_lyx(fout, t, t);
	    fprintf(fout, "\n\\layout Standard\n\n");
	    SKIP;
	    break;
	
	  case L_Heading_star:
	    fprintf(fout, "\n\\layout %s*\n\n", HeadingName[tree->data.spec]);
	    print_lyx(fout, tree->down, tree->down);
	    fprintf(fout, "\n\\layout Standard\n\n");
	    SKIP;
	    break;
	
	  case L_dollar:
	    fprintf(fout, "\n\\begin_inset Formula\n\\%c ", tree->data.spec);
	    math_mode = 1; print_lyx(fout, tree->down, tree->down); math_mode = 0;
	    fprintf(fout, " \\%c\n\\end_inset\n\n", tree->data.spec=='['?']':')');
	    SKIP;
	    break;

	  case L_lcurly:
	    /****** Start of a group of words */
	    push (fam_stack, &fam_top, family);
	    push (ser_stack, &ser_top, series);
	    push (shp_stack, &shp_top, shape);
	    push (siz_stack, &siz_top, size);
	    push (bar_stack, &bar_top, bar);

	    print_lyx (fout, tree->down, tree->down);

	    /****** End of a group of words */
	    change = (family != fam_stack[fam_top-1]);
	    family = pop (fam_stack, &fam_top);
	    if (change)
		fprintf(fout, "\n\\family %s\n", family<0?"default":FontShapeName[family]);
	    change = (series != ser_stack[ser_top-1]);
	    series = pop (ser_stack, &ser_top);
	    if (change)
		fprintf(fout, "\n\\series %s\n", series<0?"default":FontShapeName[series]);
	    change = (shape != shp_stack[shp_top-1]);
	    shape = pop (shp_stack, &shp_top);
	    if (change)
		fprintf(fout, "\n\\shape %s\n", shape<0?"default":FontShapeName[shape]);
	    change = (size != siz_stack[siz_top-1]);
	    size = pop (siz_stack, &siz_top);
	    if (change)
		fprintf(fout, "\n\\size %s\n", size<0?"default":FontShapeName[size]);
	    change = (bar != bar_stack[bar_top-1]);
	    bar = pop (bar_stack, &bar_top);
	    if (change)
		fprintf(fout, "\n\\bar %s\n", bar<0?"no_bar":FontShapeName[bar]);
	    SKIP;
	    break;

	  case L_optional_spec:
	    /****** optional argument */
	    break;

	  case L_word_list:
	    switch(tree->data.spec)
	    {
	      case L_lcurly:
		/****** May be used to group words together */
		break;
	      default:
		/****** composition of other command, skip */
		break;
	    }
	    break;

	  case L_FontShape:
	    if (math_mode) {
		fprintf(fout, "%s ", FontShapeMathName[tree->data.spec]);
	    } else {
		switch (tree->data.spec)
		{
		  case EM:
		    if (shape != IT)
			shape = IT;
		    else
		        shape = EM;
		    fprintf(fout, "\n\\shape %s\n", FontShapeName[shape]);
		    break;
		  case RM:
		  case SF:
		  case TT:
		    family = tree->data.spec;
		    fprintf(fout, "\n\\family %s\n", FontShapeName[family]);
		    break;
		  case SC:
		  case IT:
		  case SL:
		  case UP:
		    shape = tree->data.spec;
		    fprintf(fout, "\n\\shape %s\n", FontShapeName[shape]);
		    break;
		  case BF:	
		  case MD:
		    series = tree->data.spec;
		    fprintf(fout, "\n\\series %s\n", FontShapeName[series]);
		    break;
		}
	    }
	    break;

	  case L_FontSize:
	    size = tree->data.spec;
	    fprintf(fout, "\n\\size %s\n", FontSizeName[size]);
	    break;

	  case L_MathGreek:
	    fprintf(fout, "%s ", MathGreekName[tree->data.spec]);
	    break;

	  case L_MathSymbol:
	    fprintf(fout, "%s ", MathSymbolName[tree->data.spec]);
	    break;

	  case L_MathDelim:
	    fprintf(fout, "%s ", MathDelimName[tree->data.spec]);
	    break;

	  case L_plus:
	    fprintf(fout, "+");
	    break;

	  case L_div:
	    fprintf(fout, "/");
	    break;

	  case L_equal:
	    fprintf(fout, "=");
	    break;

	  case L_less:
	    fprintf(fout, "<");
	    break;

	  case L_more:
	    fprintf(fout, ">");
	    break;

	  case L_lparen:
	    fprintf(fout, "(");
	    break;

	  case L_rparen:
	    fprintf(fout, ")");
	    break;

	  case L_lbracket:
	    fprintf(fout, "[");
	    break;

	  case L_rbracket:
	    fprintf(fout, "]");
	    break;

	  case L_tilde:
	    if (math_mode)
	        fprintf(fout, "~");
	    else
	        fprintf(fout, "\n\\protected_separator\n");
	    break;

	  case L_thinspace:
	    fprintf(fout, "\\,");
	    break;

	  case L_negspace:
	    fprintf(fout, "\\!");
	    break;

	  case L_mediumspace:
	    fprintf(fout, "\\:");
	    break;

	  case L_thickspace:
	    fprintf(fout, "\\;");
	    break;

	  case L_mathspace:
	    fprintf(fout, "\\ ");
	    break;

	  case L_LatexMisc:
	    switch(tree->data.spec)
	    {
	      case 'S':
		fprintf(fout, "%c", 0xa7);
		break;
	      case 'P':
		fprintf(fout, "%c", 0xb6);
		break;
	      case 'c':
		fprintf(fout, "%c", 0xa9);
		break;
	      case 'p':
		fprintf(fout, "%c", 0xa3);
		break;
	      case '-':
		fprintf(fout, "\n\\begin_inset LatexDel \\-\n\\end_inset\n");
		break;
	      case '%':
	      case '#':
	      case '$':
	      case '&':
		fprintf(fout, "%c", tree->data.spec);
		break;
	    }
	    break;
	    
	  case L_thinlines:
	  case L_thicklines:
	    if (!latex_mode) {
 		fprintf(fout, "\n\\latex latex\n\\backslash\n%s",CommandName[tree->data.spec]);
		fprintf(fout, "\n\\latex default\n");
	    } else {
 		fprintf(fout, "\n\\backslash\n%s",CommandName[tree->data.spec]);
	    }
	    break;

	  case L_tableofcontents:
	    fprintf(fout, "\n\\begin_inset LatexDel \\%s",CommandName[tree->data.spec]);
	    fprintf(fout, "\n\\end_inset\n");
	    break;

	  case L_newline:
	    t = tree->next;
	    if (t->token == L_optional_spec) {
		tree = tree->next;
		if (!latex_mode) {
		    fprintf(fout, "\n\\latex latex\n\\backslash\n\\backslash\n[");
		    latex_mode = 1; print_lyx(fout, tree, tree); latex_mode = 0;
		    fprintf(fout, "]\n\\newline\n\\latex default\n");
		} else {
		    fprintf(fout, "\n\\backslash\n\\backslash\n[");
		    print_lyx(fout, tree, tree); 
		    fprintf(fout, "]\n\\newline\n");
		}
		SKIP;
	    } else {
		if (math_mode)
		    fprintf(fout, "\\\\\n");
		else if (latex_mode)
		    fprintf(fout, "\n\\backslash\n\\backslash\n\\newline\n");
		else
		    fprintf(fout, "\n\\newline\n");
	    }
	    break;

	  case L_accent:
	    t2 = tree->down;
	    if (t2->token == L_backslash) {
		if (t2->down->token != L_word)
		    fprintf(stderr, "don't know this node \n");
	        /* must be and \ i in latex */
	        c = *(t2->down->data.ptr);
	    } else  if (t2->token == L_word) {
	        /* must be a single char */
	        c = *(t2->data.ptr);
	    } else {
		fprintf(stderr, "don't know char to be accented\n");
	        exit (1);
	    }
	    c = get_isolat (tree->data.spec, c);
	    if (c == 0x00)
	        /* no isolatin, must go in latex mode @@@ complete */
	        ;
	    else 
	        fprintf(fout, "%c", c);
	    SKIP;
	    break;

	  case L_Environment:
	    switch(tree->data.spec)
	    {
	      case MATH:
		fprintf(fout, "\n\\begin_inset Formula\n\\( ");
		math_mode = 1; print_lyx(fout, tree->down, tree->down); math_mode = 0;
		fprintf(fout, " \\)\n\\end_inset\n\n");
		break;
	      case DISPLAYMATH:
		fprintf(fout, "\n\\begin_inset Formula\n\\[ ");
		math_mode = 1; print_lyx(fout, tree->down, tree->down); math_mode = 0;
		fprintf(fout, " \\]\n\\end_inset\n\n");
		break;
	      case EQNARRAY:
	      case EQNARRAYs:
	      case EQUATION:
		fprintf(fout, "\n\\begin_inset Formula\n");
		fprintf(fout, "\\begin%s\n", EnvironmentName[tree->data.spec]);
		math_mode=1; print_lyx(fout, tree->down, tree->down); math_mode=0;
		fprintf(fout, "\\end%s\n", EnvironmentName[tree->data.spec]);
		fprintf(fout, "\n\\end_inset\n\n");
		break;
	      case DESCRIPTION:
	      case ENUMERATE:
	      case ITEMIZE:
	      case LIST:
		push (env_stack, &env_depth, environ);
		environ = tree->data.spec;
		if (env_depth > 1)
		    fprintf(fout, "\n\\begin_deeper\n");
		print_lyx (fout, tree->down, tree->down);
		if (env_depth > 1)
		    fprintf(fout, "\n\\end_deeper\n");
		environ = pop (env_stack, &env_depth);
		break;
	      case TABLE:
	      case FIGURE:
		fprintf (fout, "\n\\begin_float %s\n", EnvironmentName[tree->data.spec]);
		t = tree->down->down;
		if (t->token == L_optional_spec) {
		    switch (*(t->down->data.ptr))
		    {
		      case 't':
			fprintf(fout, "\n\\align top\n");
			break;
		      case 'b':
			fprintf(fout, "\n\\align botton\n");
			break;
		    }
		    t = t->next;
		}
		print_lyx (fout, tree->down, t);
		fprintf (fout, "\n\\end_float\n");
		break;
	      case TABULAR:
		t = tree->down->down;
		fprintf(fout, "\\layout Standard\n");
		if (t->token == L_optional_spec) {
		    switch (*(t->down->data.ptr))
		    {
		      case 'c':
			fprintf(fout, "\\align center ");
			break;
		      case 'l':
			fprintf(fout, "\\align left ");
			break;
		      case 'r':
			fprintf(fout, "\\align right ");
			break;
		    }
		    t = t->next;
		} else
		    fprintf(fout, "\\align center ");
		fprintf(fout, "\\LyXTable\n");
		print_table (fout, tree->down, t->next, t->down->data.ptr);
		break;
	      case ABSTRACT:
	      case QUOTATION:
	      case QUOTE:
	      case VERSE:
		fprintf (fout, "\n\\layout %s\n",EnvironmentName[tree->data.spec]);
		layout = tree->data.spec; 
		print_lyx (fout, tree->down, tree->down);
		layout = 0;
		break;
	      case FLUSHLEFT:
		fprintf(fout, "\n\\layout Standard\n\\align left\n");
		align = 'l'; print_lyx(fout, tree->down, tree->down); align = 0;
		break;
	      case FLUSHRIGHT:
		fprintf(fout, "\n\\layout Standard\n\\align right\n");
		align = 'r'; print_lyx(fout, tree->down, tree->down); align = 0;
		break;
	      case CENTER:  
		fprintf(fout, "\n\\layout Standard\n\\align center\n");
		align = 'c'; print_lyx(fout, tree->down, tree->down); align = 0;
		break;
		/* @@@ environments with no easy translation to lyx */
	      case TRIVLIST:
	      case ARRAY:
	      case MINIPAGE:
	      case TABBING:
	      case VERBATIN:
	      case TITLEPAGE:
	      case THEINDEX:
	      case THEBIBLIOGRAPHY:
	      case SLOPPYPAR:
	      case PICTURE:
		fprintf(fout, "\\layout LaTeX\n\\backslash\nbegin%s\n", 
			EnvironmentName[tree->data.spec]);
		latex_mode = 1; print_lyx (fout, tree->down, tree->down); latex_mode = 0;
		fprintf(fout, "\\backslash\nend%s\n\n\\layout Standard",
			EnvironmentName[tree->data.spec]);
		break;
	    }
	    SKIP;
	    break;

	  case L_kill:
	  case L_tabset:
	  case L_tabskip:
	    /****** I am assuming these commands are printed within a LaTeX layout */
	    fprintf(fout, "\\backslash\n%s", CommandName[tree->token]);
	    break;

	  case L_item:
	    t = tree->next;
	    if (t->token == L_optional_spec) {
		/****** This an optional argument for item,
		  very important for the description environtmet @@@ complete */
		tree = tree->next;
		/* print_lyx(fout, tree, tree); */
		SKIP;
	    } else {
		fprintf(fout, "\n\\layout %s\n", EnvironmentName[environ]);
	    }
	    break;

	    /****** Those are useless, discard them */
	  case L_document:		
	  case L_end:
	  case L_smallspace: /* Latex2e can handle that automatically */
	    break;

	    /****** Those '*' stderr outputs should never occur */
	  case L_hline:
	    fprintf(stderr, "*");
	    break;
	  case L_amper:
	    if (math_mode)
	        fprintf(fout, "&");
	    else
	        fprintf(stderr, "*");
	    break;


	    /****** @@@ all those must be worked on */
	    /****** []		*/
	  case L_linebreak:
	  case L_nolinebreak:
	  case L_pagebreak:
	  case L_nopagebreak:
	    /****** () []	*/
	  case L_oval:
	  default:
	    fprintf(stderr, ".");
	    break;
	}

	if (tree->down != NULL) {
	    tree = tree->down;
	} else {
	    SKIP;
	}

    } while (tree != start);
}




