/*;-*-C-*-; 
** Copyright (c) Massachusetts Institute of Technology 1994, 1995, 1996.
**          All Rights Reserved.
**          Unpublished rights reserved under the copyright laws of
**          the United States.
**
** THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
** OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
**
** This code is distributed freely and may be used freely under the 
** following conditions:
**
**     1. This notice may not be removed or altered.
**
**     2. This code may not be re-distributed or modified
**        without permission from MIT (contact 
**        lclint-request@larch.lcs.mit.edu.)  
**
**        Modification and re-distribution are encouraged,
**        but we want to keep track of changes and
**        distribution sites.
*/

Digit			[0-9]
Letter			[a-zA-Z_$]
H			[a-fA-F0-9]
E			[Ee][+-]?{Digit}+
U                       (u|U)
L                       (l|L)
FS			(f|F|l|L)
IS			(u|U|l|L)*
ULSuffix                ({U}{L}|{L}{U})

%{
/*
** based on original C lexer by Nate Osgood
**    from hacrat@catfish.lcs.mit.edu Mon Jun 14 13:06:32 1993
**
*/

# include "flex.reset"

# include "lclintMacros.nf"
# include "basic.h"
# include "cgrammar.h"
# include "cgrammar_tokens.h"
# include "fileIdList.h"
# include "pcpp.h"


static bool lastWasString = FALSE;
static char savechar = '\0';

/*@notfunction@*/
# define yyinput() (incColumn (), getc (yyin))

/*@-noparams@*/
extern /*@external@*/ int read ();
/*@=noparams@*/

static /*@owned@*/ cstring lastidprocessed = cstring_undefined;

static int lminput (void);
static int tokLength = 0;
static bool inSpecPart = FALSE;
static bool continueLine = FALSE;

static int ninput (void);
static char processChar (void);
static double processFloat (void);
static /*@only@*/ cstring processString (void);
static long processDec (void);
static long processHex (void);
static int processIdentifier (/*@only@*/ cstring id)
   /*@globals undef lastidprocessed@*/ ;
static bool processHashIdentifier (/*@only@*/ cstring id)
   /*@globals undef lastidprocessed@*/ ;

static int processSpec (int tok);
static bool handleSpecial (char *s);
static int handleLlSpecial (void);
static void handleMacro (void);
static void processMacro (void);
static /*@only@*/ cstring makeIdentifier (char *s);

/* yes, this is exported! */
bool GlobalExpectingTypeName = TRUE; /* beginning of file can be type name! */

static int returnInt (ctype, long);
static int returnFloat (ctype, double);
static int returnChar (char);
static void setTokLength (int len) /*@modifies currentloc@*/ ;
static void setTokLengthT (size_t len) /*@modifies currentloc@*/ ;

static void advanceLine (void)
{
  tokLength = 0;
  beginLine ();
}
    
/*@-allmacros@*/
# define RETURN_INT(c,i) \
  do { lastWasString = FALSE; \
       return (returnInt (c, i)); } while (FALSE)

# define RETURN_FLOAT(c,f) \
  do { lastWasString = FALSE; \
       return (returnFloat (c, f)); \
       } while (FALSE)

# define RETURN_CHAR(c) \
  do { lastWasString = FALSE; \
       return (returnChar (c)); \
     } while (FALSE)

# define RETURN_TOK(t) \
  do { yylval.tok = lltok_create (t, fileloc_decColumn (currentloc, tokLength)); \
       tokLength = 0; \
       lastWasString = FALSE; \
       return (t); } while (FALSE)

# define RETURN_TYPE(t, ct) \
  do { yylval.ctyp = ct; tokLength = 0; return (t); } while (FALSE)

# define RETURN_STRING(c) \
 do { yylval.expr = exprNode_stringLiteral (c, fileloc_decColumn (currentloc, tokLength));  \
      tokLength = 0; \
      lastWasString = TRUE; \
      return (CCONSTANT); } while (FALSE)

static void setTokLength (int len) 
{
  addColumn (len);
  tokLength = len;
}

static void setTokLengthT (size_t len)
{
  setTokLength (size_toInt (len));
}

# include "flex.head"
%}

%%

"/*"		{ llfatalbug (cstring_makeLiteral ("Comment in pre-processor output")); }

"#"{Letter}({Letter}|{Digit})*  { 
                 context_saveLocation (); 
                 setTokLength (longUnsigned_toInt (mstring_length (yytext))); 

                 if (processHashIdentifier (makeIdentifier (yytext + 1)))
                   {
		     if (lastWasString)
		       {
			 ;
		       }
		     else
		       {
			 RETURN_STRING (cstring_makeLiteral ("\"\""));
		       }
		   }
                 else
                   { 
		     if (handleSpecial (yytext)) 
                       { 
			 setTokLength (1); 
			 RETURN_TOK (0); 
		       }
		   }
                } 
"#"             { if (handleSpecial (yytext)) 
                    { 
                       setTokLength (1); RETURN_TOK (0); 
                     }
                }
"..."           { setTokLength (3); RETURN_TOK (CTOK_ELIPSIS); }
"break"		{ setTokLength (5); RETURN_TOK (BREAK); }
"case"		{ setTokLength (4); RETURN_TOK (CASE); }
"continue"	{ setTokLength (8); RETURN_TOK (CONTINUE); }
"default"	{ setTokLength (7); RETURN_TOK (DEFAULT); }
"do"		{ setTokLength (2); RETURN_TOK (DO); }
"else"		{ setTokLength (4); RETURN_TOK (CELSE); }
"for"		{ setTokLength (3); RETURN_TOK (CFOR); }
"goto"		{ setTokLength (4); RETURN_TOK (GOTO); }
"if"		{ setTokLength (2); RETURN_TOK (CIF); }
"return"	{ setTokLength (6); RETURN_TOK (RETURN); }
"sizeof"	{ setTokLength (6); RETURN_TOK (CSIZEOF); }
"offsetof"	{ setTokLength (8); RETURN_TOK (COFFSETOF); }
"switch"	{ setTokLength (6); RETURN_TOK (SWITCH); }
"while"		{ setTokLength (5); RETURN_TOK (WHILE); }
"va_arg"        { setTokLength (6); RETURN_TOK (VA_ARG); }   
"va_dcl"        { setTokLength (6); RETURN_TOK (VA_DCL); }   
"inline"        { 
                  /* gcc extension...this might not be appropriate */
                  setTokLength (6); RETURN_TOK (QINLINE); }

"struct"        { setTokLength (6); RETURN_TOK (CSTRUCT); }  
"typedef"       { setTokLength (7); RETURN_TOK (CTYPEDEF); }

"union"         { setTokLength (5); RETURN_TOK (CUNION); }
"enum"          { setTokLength (4); RETURN_TOK (CENUM); }

"void"		{ setTokLength (4); RETURN_TYPE (CVOID, ctype_void); }
"int"		{ setTokLength (3); RETURN_TYPE (CINT, ctype_int); }
"double"	{ setTokLength (6); RETURN_TYPE (CDOUBLE, ctype_double); }
"char"		{ setTokLength (4); RETURN_TYPE (CGCHAR, ctype_char); }
"float"		{ setTokLength (5); RETURN_TYPE (CGFLOAT, ctype_float); }

"long"		{ setTokLength (4); RETURN_TOK (QLONG); }
"short"		{ setTokLength (5); RETURN_TOK (QSHORT); }
"unsigned"	{ setTokLength (8); RETURN_TOK (QUNSIGNED); }
"signed"	{ setTokLength (6); RETURN_TOK (QSIGNED); }

"volatile"	{ setTokLength (8); RETURN_TOK (QVOLATILE); }
"const"		{ setTokLength (5); RETURN_TOK (QCONST); }

                        /* some systems expect this! [gack!] */ 
"__const"	{ setTokLength (7); RETURN_TOK (QCONST); }

"extern"	{ setTokLength (6); RETURN_TOK (QEXTERN); }
"auto"		{ setTokLength (4); RETURN_TOK (QAUTO); }
"register"	{ setTokLength (8); RETURN_TOK (QREGISTER); }
"static"	{ setTokLength (6); RETURN_TOK (QSTATIC); }

\"(\\.|[^\\"])*\"([ \t\n]*\"(\\.|[^\\"])*\")* { RETURN_STRING (processString ()); }
"out"                   { return (processSpec (QOUT)); }
"in"                    { return (processSpec (QIN)); }
"partial"               { return (processSpec (QPARTIAL)); }
"special"               { return (processSpec (QSPECIAL)); }
"anytype"               { return (processSpec (QANYTYPE)); }
"integraltype"          { return (processSpec (QINTEGRALTYPE)); }
"keep"                  { return (processSpec (QKEEP)); }
"null"                  { return (processSpec (QNULL)); } 
"notnull"               { return (processSpec (QNOTNULL)); } 
"isnull"                { return (processSpec (QISNULL)); } 
"truenull"              { return (processSpec (QTRUENULL)); } 
"falsenull"             { return (processSpec (QFALSENULL)); } 
"relnull"               { return (processSpec (QRELNULL)); }
"reldef"                { return (processSpec (QRELDEF)); }
"exposed"               { return (processSpec (QEXPOSED)); }
"newref"                { return (processSpec (QNEWREF)); }
"tempref"               { return (processSpec (QTEMPREF)); }
"killref"               { return (processSpec (QKILLREF)); }
"refcounted"            { return (processSpec (QREFCOUNTED)); }
"checked"               { return (processSpec (QCHECKED)); }
"checkmod"              { return (processSpec (QCHECKMOD)); }
"checkedstrict"         { return (processSpec (QCHECKEDSTRICT)); }
"unchecked"             { return (processSpec (QUNCHECKED)); }
"only"                  { return (processSpec (QONLY)); }
"owned"                 { return (processSpec (QOWNED)); }
"observer"              { return (processSpec (QOBSERVER)); }
"dependent"             { return (processSpec (QDEPENDENT)); }
"unused"                { return (processSpec (QUNUSED)); }
"external"              { return (processSpec (QEXTERNAL)); }
"sef"                   { return (processSpec (QSEF)); }
"shared"                { return (processSpec (QSHARED)); }
"yield"                 { return (processSpec (QYIELD)); }
"undef"                 { return (processSpec (QUNDEF)); }
"killed"                { return (processSpec (QKILLED)); }
{Letter}({Letter}|{Digit})* { int tok; 
			      context_saveLocation (); 
			      setTokLength (longUnsigned_toInt (mstring_length (yytext))); 
			      tok = processIdentifier (makeIdentifier (yytext)); 
			      if (tok != BADTOK)
				{
				  return (tok);
				}
			    }
0[xX]{H}+	        { setTokLengthT (mstring_length (yytext)); 
			  RETURN_INT (ctype_uint, processHex ()); 
			}
0[xX]{H}+{L}	        { setTokLengthT (mstring_length (yytext)); 
			  RETURN_INT (ctype_lint, processHex ()); }
0[xX]{H}+{U}	        { setTokLengthT (mstring_length (yytext)); 
			  RETURN_INT (ctype_lint, processHex ()); }
0[xX]{H}+{ULSuffix}     { setTokLengthT (mstring_length (yytext)); 
			  RETURN_INT (ctype_ulint, processHex ()); }
0{Digit}+	        { setTokLengthT (mstring_length (yytext)); 
			  RETURN_INT (ctype_int, processDec ()); } 
0{Digit}+{U}	       { setTokLengthT (mstring_length (yytext)); 
			  RETURN_INT (ctype_uint, processDec ()); } 
0{Digit}+{L}	       { setTokLengthT (mstring_length (yytext)); 
 		         RETURN_INT (ctype_lint, processDec ()); } 
0{Digit}+{ULSuffix}    { setTokLengthT (mstring_length (yytext)); 
	   	         RETURN_INT (ctype_ulint, processDec ()); } 
{Digit}+	       { setTokLengthT (mstring_length (yytext)); 
		         RETURN_INT (ctype_int, processDec ()); } 
{Digit}+{U}	       { setTokLengthT (mstring_length (yytext)); 
		         RETURN_INT (ctype_uint, processDec ()); } 
{Digit}+{L}	       { setTokLengthT (mstring_length (yytext)); 
	      	         RETURN_INT (ctype_lint, processDec ()); } 
{Digit}+{ULSuffix}     { setTokLengthT (mstring_length (yytext)); 
	    	         RETURN_INT (ctype_ulint, processDec ()); } 
'(\\.|[^\\'])+'	       { setTokLengthT (mstring_length (yytext)); 
                         RETURN_CHAR (processChar ()); }
{Digit}+{E}[fF]        { setTokLengthT (mstring_length (yytext)); 
			 RETURN_FLOAT (ctype_float, processFloat ()); }
{Digit}+{E}[lL]        { setTokLengthT (mstring_length (yytext)); 
			 RETURN_FLOAT (ctype_ldouble, processFloat ()); }
{Digit}+{E}            { setTokLengthT (mstring_length (yytext)); 
			 RETURN_FLOAT (ctype_double, processFloat ()); }

{Digit}*"."{Digit}+({E})?[fF] { setTokLengthT (mstring_length (yytext)); 
				RETURN_FLOAT (ctype_float, processFloat ()); }
{Digit}*"."{Digit}+({E})?[lL] { setTokLengthT (mstring_length (yytext)); 
				RETURN_FLOAT (ctype_ldouble, processFloat ()); }
{Digit}*"."{Digit}+({E})?     { setTokLengthT (mstring_length (yytext)); 
				RETURN_FLOAT (ctype_double, processFloat ()); }

{Digit}+"."{Digit}*({E})?[fF]   { setTokLengthT (mstring_length (yytext)); 
				  RETURN_FLOAT (ctype_float, processFloat ()); }
{Digit}+"."{Digit}*({E})?[lL]   { setTokLengthT (mstring_length (yytext)); 
				  RETURN_FLOAT (ctype_ldouble, processFloat ()); }
{Digit}+"."{Digit}*({E})?       { setTokLengthT (mstring_length (yytext)); 
				  RETURN_FLOAT (ctype_double, processFloat ()); }

">>="		{ setTokLength (3); RETURN_TOK (RIGHT_ASSIGN); }
"<<="		{ setTokLength (3); RETURN_TOK (LEFT_ASSIGN); }
"+="		{ setTokLength (2); RETURN_TOK (ADD_ASSIGN); }
"-="		{ setTokLength (2); RETURN_TOK (SUB_ASSIGN); }
"*="		{ setTokLength (2); RETURN_TOK (MUL_ASSIGN); }
"/="		{ setTokLength (2); RETURN_TOK (DIV_ASSIGN); }
"%="		{ setTokLength (2); RETURN_TOK (MOD_ASSIGN); }
"&="		{ setTokLength (2); RETURN_TOK (AND_ASSIGN); }
"^="		{ setTokLength (2); RETURN_TOK (XOR_ASSIGN); }
"|="		{ setTokLength (2); RETURN_TOK (OR_ASSIGN); }
">>"		{ setTokLength (2); RETURN_TOK (RIGHT_OP); }
"<<"		{ setTokLength (2); RETURN_TOK (LEFT_OP); }
"++"		{ setTokLength (2); RETURN_TOK (INC_OP); }
"--"		{ setTokLength (2); RETURN_TOK (DEC_OP); }
"->"		{ setTokLength (2); RETURN_TOK (ARROW_OP); }
"&&"		{ setTokLength (2); RETURN_TOK (AND_OP); }
"||"		{ setTokLength (2); RETURN_TOK (OR_OP); }
"<="		{ setTokLength (2); RETURN_TOK (LE_OP); }
">="		{ setTokLength (2); RETURN_TOK (GE_OP); }
"=="		{ setTokLength (2); RETURN_TOK (CTOK_EQOP); }
"!="		{ setTokLength (2); RETURN_TOK (NE_OP); }
";"		{ setTokLength (1); RETURN_TOK (TSEMI); }
"{"		{ setTokLength (1); RETURN_TOK (TLBRACE); }
"}"		{ setTokLength (1); RETURN_TOK (TRBRACE); }
","		{ setTokLength (1); RETURN_TOK (TCOMMA); }
":"		{ setTokLength (1); RETURN_TOK (TCOLON); }
"="		{ setTokLength (1); RETURN_TOK (TEQ); }
"("		{ setTokLength (1); RETURN_TOK (TLPAREN); }
")"		{ setTokLength (1); RETURN_TOK (TRPAREN); }
"["		{ setTokLength (1); RETURN_TOK (TLSQBR); }
"]"		{ setTokLength (1); RETURN_TOK (TRSQBR); }
"."		{ setTokLength (1); RETURN_TOK (TDOT); }
"&"		{ setTokLength (1); RETURN_TOK (TAMPERSAND); }
"!"		{ setTokLength (1); RETURN_TOK (TEXCL); }


"~"		{ setTokLength (1); RETURN_TOK (TTILDE); }
"-"		{ setTokLength (1); RETURN_TOK (TMINUS); }
"+"		{ setTokLength (1); RETURN_TOK (TPLUS); }
"*"		{ setTokLength (1); RETURN_TOK (TMULT); }
"/"		{ setTokLength (1); RETURN_TOK (TDIV); }
"%"		{ setTokLength (1); RETURN_TOK (TPERCENT); }
"<"		{ setTokLength (1); RETURN_TOK (TLT); }
">"		{ setTokLength (1); RETURN_TOK (TGT); }
"^"		{ setTokLength (1); RETURN_TOK (TCIRC); }
"|"		{ setTokLength (1); RETURN_TOK (TBAR); }
"?"		{ setTokLength (1); RETURN_TOK (TQUEST); }

[ \t\v\f]	{ incColumn (); }
\n              { context_incLineno ();
		  if (continueLine)
		    {
		      continueLine = FALSE;
		    }
                 else 
		   {
		     if (context_inMacro ())
		       {
			 /* Don't use RETURN_TOK */
			 yylval.tok = lltok_create (TENDMACRO, currentloc);
			 lastWasString = FALSE;
			 return (TENDMACRO);
		       }  
		   }
		}
"@@MR@@"        { setTokLength (6); 
		  processMacro ();

		  if (context_inIterDef ()) 
		    { 
		      RETURN_TOK (LLMACROITER); 
		    }
		  if (context_inIterEnd ())
		    {
		      RETURN_TOK (LLMACROEND); 
		    }
		  if (context_inMacro ())
		    {
		      RETURN_TOK (LLMACRO); 
		    }
		}
"@QLMRX"        { if (context_inHeader () || context_inFunction ())
		    { 
		      handleMacro ();
		    }
                  else
		    {
		      setTokLength (6); 
		      processMacro ();
		      if (context_inIterDef ()) 
			{
			  RETURN_TOK (LLMACROITER); 
			}
		      if (context_inIterEnd ())
			{
			  RETURN_TOK (LLMACROEND); 
			}
		      if (context_inMacro ())
			{ 
			  RETURN_TOK (LLMACRO); 
			}
		    }
		}
"@.CT"          { setTokLength (4); llmsg (ctype_unparseTable ()); }
"@.F"           { setTokLength (3); 
		  llmsg (message ("%q: *** marker ***", fileloc_unparse (currentloc)));
		}
"@.L"           { setTokLength (3); usymtab_printLocal (); }
"@.A"           { setTokLength (3); llmsg (usymtab_unparseAliases ()); }
"@.C"           { setTokLength (3); llmsg (context_unparse ()); }
"@.W"           { setTokLength (3); llmsg (context_unparseClauses ()); }
"@.G"           { setTokLength (3); usymtab_printGuards (); }
"@.S"           { setTokLength (3); usymtab_printOut (); }
"@.X"           { setTokLength (3); usymtab_printAll (); }
"@.Z"           { setTokLength (3); usymtab_printComplete (); }
"@.T"           { setTokLength (3); usymtab_printTypes (); }
"@.K"           { setTokLength (3); llmsg (usymtab_unparseStack ()); }
"@.M"           { setTokLength (3); 
		  llmsg (message ("Can modify: %q", 
				  sRefSet_unparse (context_modList ()))); 
		}
"@"             { int tok; 
		  incColumn (); 
		  tok = handleLlSpecial (); 
		  if (tok != BADTOK)
		    {
		      RETURN_TOK (tok); 
		    }
		}
"\\"            { incColumn (); continueLine = TRUE; }
.		{ incColumn (); 
		  voptgenerror
		    (FLG_SYNTAX, 
		     message ("Invalid character (ascii: %d), skipping character",
			      (int)(*yytext)),
		     currentloc);
		}
%%

# include "flex.reset"

# define isAnyTypeMacro(s)    (cstring_equalLit (s, "anytype"))
# define isIntegralTypeMacro(s)    (cstring_equalLit (s, "integraltype"))
# define isOutMacro(s)        (cstring_equalLit (s, "out"))
# define isInMacro(s)         (cstring_equalLit (s, "in"))
# define isOnlyMacro(s)       (cstring_equalLit (s, "only"))
# define isOwnedMacro(s)      (cstring_equalLit (s, "owned"))
# define isDependentMacro(s)  (cstring_equalLit (s, "dependent"))
# define isPartialMacro(s)    (cstring_equalLit (s, "partial"))
# define isSpecialMacro(s)    (cstring_equalLit (s, "special"))
# define isTrueNullMacro(s)   (cstring_equalLit (s, "truenull"))
# define isFalseNullMacro(s)  (cstring_equalLit (s, "falsenull"))
# define isKeepMacro(s)       (cstring_equalLit (s, "keep"))
# define isKeptMacro(s)       (cstring_equalLit (s, "kept"))
# define isNotNullMacro(s)    (cstring_equalLit (s, "notnull"))
# define isAbstractMacro(s)   (cstring_equalLit (s, "abstract"))
# define isConcreteMacro(s)   (cstring_equalLit (s, "concrete"))
# define isMutableMacro(s)    (cstring_equalLit (s, "mutable"))
# define isImmutableMacro(s)  (cstring_equalLit (s, "immutable"))
# define isUnusedMacro(s)     (cstring_equalLit (s, "unused"))
# define isExternalMacro(s)   (cstring_equalLit (s, "external"))
# define isSefMacro(s)        (cstring_equalLit (s, "sef"))
# define isUniqueMacro(s)     (cstring_equalLit (s, "unique"))
# define isReturnedMacro(s)   (cstring_equalLit (s, "returned"))
# define isExposedMacro(s)    (cstring_equalLit (s, "exposed"))
# define isRefCountedMacro(s) (cstring_equalLit (s, "refcounted"))
# define isRefsMacro(s)       (cstring_equalLit (s, "refs"))
# define isNewRefMacro(s)     (cstring_equalLit (s, "newref"))
# define isTempRefMacro(s)    (cstring_equalLit (s, "tempref"))
# define isKillRefMacro(s)    (cstring_equalLit (s, "killref"))
# define isNullMacro(s)       (cstring_equalLit (s, "null"))
# define isRelNullMacro(s)    (cstring_equalLit (s, "relnull"))
# define isRelDefMacro(s)     (cstring_equalLit (s, "reldef"))
# define isObserverMacro(s)   (cstring_equalLit (s, "observer"))
# define isExitsMacro(s)      (cstring_equalLit (s, "exits"))
# define isMayExitMacro(s)    (cstring_equalLit (s, "mayexit"))
# define isTrueExitMacro(s)   (cstring_equalLit (s, "trueexit"))
# define isFalseExitMacro(s)  (cstring_equalLit (s, "falseexit"))
# define isNeverExitMacro(s)  (cstring_equalLit (s, "neverexit"))
# define isTempMacro(s)       (cstring_equalLit (s, "temp"))
# define isSharedMacro(s)     (cstring_equalLit (s, "shared"))
# define isRefMacro(s)        (cstring_equalLit (s, "ref"))
# define isBlockMacro(s)      (cstring_equalLit (s, "block"))
# define isUncheckedMacro(s)  (cstring_equalLit (s, "unchecked"))
# define isCheckedMacro(s)    (cstring_equalLit (s, "checked"))
# define isCheckModMacro(s)  (cstring_equalLit (s, "checkmod"))
# define isCheckedStrictMacro(s)  (cstring_equalLit (s, "checkedstrict"))
# define isInnerContinueMacro(s)  (cstring_equalLit (s, "innercontinue"))
# define isInnerBreakMacro(s)  (cstring_equalLit (s, "innerbreak"))
# define isLoopBreakMacro(s)   (cstring_equalLit (s, "loopbreak"))
# define isSwitchBreakMacro(s) (cstring_equalLit (s, "switchbreak"))
# define isSafeBreakMacro(s)   (cstring_equalLit (s, "safebreak"))
# define isFallThrough(s)      (cstring_equalLit (s, "fallthrough"))
# define isLintFallThrough(s)  (cstring_equalLit (s, "fallthrou"))
# define isLintFallThru(s)     (cstring_equalLit (s, "fallth"))
# define isNotReached(s)       (cstring_equalLit (s, "notreached"))
# define isLintNotReached(s)   (cstring_equalLit (s, "notreach"))
# define isPrintfLike(s)       (cstring_equalLit (s, "printflike"))
# define isLintPrintfLike(s)   (cstring_equalLit (s, "printfli"))
# define isScanfLike(s)        (cstring_equalLit (s, "scanflike"))
# define isMessageLike(s)      (cstring_equalLit (s, "messagelike"))
# define isLintArgsUsed(s)     (cstring_equalLit (s, "argsus"))

/*
** would be better if these weren't hard coded...
*/

static bool isArtificial (cstring s)
{
  return (cstring_equalLit (s, "MODIFIS") 
	  || cstring_equalLit (s, "EM") 
	  || cstring_equalLit (s, "GLOBLS") 
	  || cstring_equalLit (s, "EG") 
	  || cstring_equalLit (s, "AL") 
	  || cstring_equalLit (s, "EA"));
}

void swallowMacro (void)
{
  int i;
  bool skipnext = FALSE;

  while ((i = lminput ()) != EOF)
    {
      char c = (char) i;
      
      DPRINTF (("swallow: %c", c));

      if (c == '\\')
	{
	  skipnext = TRUE;
	}
      else if (c == '\n')
	{
	  if (skipnext)
	    {
	      skipnext = FALSE;
	    }
	  else
	    {
	      checkUngetc (i, yyin);
	      return;
	    }
	}
    }

  if (i != EOF)
    {
      checkUngetc (i, yyin);
    }
}

static 
int tokenMacroCode (cstring s)
{
  if (isAnyTypeMacro (s))    return QANYTYPE;
  if (isIntegralTypeMacro (s))    return QINTEGRALTYPE;
  if (isOutMacro (s))        return QOUT;
  if (isInMacro (s))         return QIN;
  if (isOwnedMacro (s))      return QOWNED;
  if (isDependentMacro (s))  return QDEPENDENT;
  if (isPartialMacro (s))    return QPARTIAL;
  if (isSpecialMacro (s))    return QSPECIAL;
  if (isOnlyMacro (s))       return QONLY;
  if (isKeepMacro (s))       return QKEEP;
  if (isKeptMacro (s))       return QKEPT;
  if (isExposedMacro (s))    return QEXPOSED;
  if (isUniqueMacro (s))     return QUNIQUE;
  if (isReturnedMacro (s))   return QRETURNED;
  if (isNullMacro (s))       return QNULL;
  if (isRelNullMacro (s))    return QRELNULL;
  if (isObserverMacro (s))   return QOBSERVER;
  if (isTempMacro (s))       return QTEMP;
  if (isSharedMacro (s))     return QSHARED;
  if (isRefMacro (s))        return QREF;
  if (isUncheckedMacro (s))  return QUNCHECKED;
  if (isCheckedMacro (s))    return QCHECKED;
  if (isCheckModMacro (s))   return QCHECKMOD;
  if (isCheckedStrictMacro (s)) return QCHECKEDSTRICT;
  if (isInnerContinueMacro(s))  return QINNERCONTINUE;
  if (isInnerBreakMacro(s))  return QINNERBREAK;
  if (isLoopBreakMacro(s))   return QLOOPBREAK;
  if (isSwitchBreakMacro(s)) return QSWITCHBREAK;
  if (isSafeBreakMacro(s))   return QSAFEBREAK;
  if (isFallThrough(s))      return QFALLTHROUGH;
  if (isLintFallThrough(s))
    {
      voptgenerror (FLG_WARNLINTCOMMENTS,
		    cstring_makeLiteral
		    ("Traditional lint comment /*FALLTHROUGH*/ used.  "
		     "This is interpreted by "
		     "LCLint in the same way as most Unix lints, but it is "
		     "preferable to replace it with the /*@fallthrough@*/ "
		     "stylized comment"),
		    currentloc);
      return QFALLTHROUGH;
    }
  if (isLintFallThru(s))
    {
      voptgenerror (FLG_WARNLINTCOMMENTS,
		    cstring_makeLiteral
		    ("Traditional lint comment /*FALLTHRU*/ used.  "
		     "This is interpreted by "
		     "LCLint in the same way as most Unix lints, but it is "
		     "preferable to replace it with the /*@fallthrough@*/ "
		     "stylized comment"),
		    currentloc);
      return QFALLTHROUGH;
    }
  if (isNotReached(s))       return QNOTREACHED;
  if (isLintNotReached(s)) 
    {
      voptgenerror (FLG_WARNLINTCOMMENTS,
		    cstring_makeLiteral
		    ("Traditional lint comment /*NOTREACHED*/ used.  "
		     "This is interpreted by "
		     "LCLint in the same way as most Unix lints, but it is "
		     "preferable to replace it with the /*@notreached@*/ "
		     "stylized comment."),
		    currentloc);
      
      return QNOTREACHED;
    }
  if (isLintPrintfLike(s)) 
    {
      voptgenerror (FLG_WARNLINTCOMMENTS,
		    cstring_makeLiteral
		    ("Traditional lint comment /*PRINTFLIKE*/ used.  "
		     "This is interpreted by "
		     "LCLint in the same way as most Unix lints, but it is "
		     "preferable to replace it with either /*@printflike@*/, "
		     "/*@scanflike@*/ or /*@messagelike@*/."),
		    currentloc);
      
      setSpecialFunction (QU_PRINTFLIKE);
      return SKIPTOK;
    }
  if (isPrintfLike (s))
    {
      setSpecialFunction (QU_PRINTFLIKE);
      return SKIPTOK;
    }
  if (isMessageLike (s))
    {
      setSpecialFunction (QU_MESSAGELIKE);
      return SKIPTOK;
    }
  if (isScanfLike (s))
    {
      setSpecialFunction (QU_SCANFLIKE);
      return SKIPTOK;
    }
  if (isLintArgsUsed(s)) 
    {
      voptgenerror (FLG_WARNLINTCOMMENTS,
		    cstring_makeLiteral
		    ("Traditional lint comment /*ARGSUSED*/ used.  "
		     "This is interpreted by "
		     "LCLint in the same way as most Unix lints, but it is "
		     "preferable to use /*@unused@*/ annotations on "
		     "the unused parameters."),
		    currentloc);
      
      setArgsUsed ();
      return SKIPTOK;
    }
  if (isNotNullMacro (s))    return QNOTNULL;
  if (isTrueNullMacro (s))   return QTRUENULL;
  if (isFalseNullMacro (s))  return QFALSENULL;
  if (isAbstractMacro (s))   return QABSTRACT;
  if (isConcreteMacro (s))   return QCONCRETE;
  if (isMutableMacro (s))    return QMUTABLE;
  if (isImmutableMacro (s))  return QIMMUTABLE;
  if (isExitsMacro (s))      return QEXITS;
  if (isMayExitMacro (s))    return QMAYEXIT;
  if (isTrueExitMacro (s))   return QTRUEEXIT;
  if (isFalseExitMacro (s))  return QFALSEEXIT;
  if (isNeverExitMacro (s))  return QNEVEREXIT;
  if (isUnusedMacro (s))     return QUNUSED;
  if (isExternalMacro (s))   return QEXTERNAL;
  if (isSefMacro (s))        return QSEF;
  if (isRefCountedMacro (s)) return QREFCOUNTED;
  if (isRefsMacro (s))       return QREFS;
  if (isNewRefMacro (s))     return QNEWREF;
  if (isTempRefMacro (s))    return QTEMPREF;
  if (isKillRefMacro (s))    return QKILLREF;
  if (isRelDefMacro (s))     return QRELDEF;
  else  
    {
      return BADTOK;
    }
}

static int lminput ()
{
  if (savechar == '\0')
    {
      incColumn ();
      return (input ());
    }
  else
    {
      int save = (int) savechar;
      savechar = '\0';
      return save;
    }
}

static void lmsavechar (char c)
{
  if (savechar == '\0') savechar = c;
  else
    {
      llbuglit ("lmsavechar: override");
    }
}

static int returnFloat (ctype ct, double f)
{
  yylval.expr = exprNode_floatLiteral (f, ct, cstring_fromChars (yytext), 
				       fileloc_decColumn (currentloc, tokLength));
  tokLength = 0; 
  return (CCONSTANT);
}

static int returnInt (ctype ct, long i)
{
  ctype c;

  if (i == 0)
    {
      c = context_typeofZero ();
    }
  else if (i == 1)
    {
      c = context_typeofOne ();
    }
  else
    {
      c = ct;
    }
  
  yylval.expr = exprNode_numLiteral (c, cstring_fromChars (yytext), 
				     fileloc_decColumn (currentloc, tokLength), i);   
  tokLength = 0; 
  return (CCONSTANT);
}

static int returnChar (char c)
{
  yylval.expr = exprNode_charLiteral (c, cstring_fromChars (yytext), 
				      fileloc_decColumn (currentloc, tokLength));
  tokLength = 0; 
  return (CCONSTANT);
}

static int ninput ()  
{
  int c = lminput ();

  if (c != EOF && ((char)c == '\n'))
    {
      context_incLineno ();
    }

  return c;
}

static char macro_nextChar ()
{
  static bool in_quote = FALSE, in_escape = FALSE, in_char = FALSE;
  int ic;
  char c;

  ic = lminput ();
  c = char_fromInt (ic);
  
  if (!in_quote && !in_char && (c == '\\' || c == '@'))
    {
      if (c == '\\')
	{
	  while ((c = char_fromInt (lminput ())) != '\0' && c != '\n')
	    {
	      ; /* skip to newline */
	    }
	  
	  context_incLineno ();
	  
	  if (c != '\0')
	    {
	      return macro_nextChar ();
	    }
	  else 
	    {
	      return c;
	    }
	}
      else /* if (c == '@') */
	{
	  if (handleLlSpecial () != BADTOK)
	    {
	      llerrorlit (FLG_SYNTAX, "Macro cannot use special syntax");
	    }

	  return macro_nextChar ();
	}
    }
  else if (!in_escape && c == '\"')
    {
      in_quote = !in_quote;
    }
  else if (!in_escape && c == '\'')
    {
      in_char = !in_char;
    }
  else if ((in_quote || in_char) && c == '\\')
    {
      in_escape = !in_escape;
    }
  else if ((in_quote || in_char) && in_escape)
    {
      in_escape = FALSE;
    }
  else if (!in_quote && c == '/')
    {
      char c2;
      
      if ((c2 = char_fromInt (lminput ())) == '*')
	{
	  while (c2 != '\0')
	    {
	      while ((c2 = char_fromInt (lminput ())) != '\0'
		     && c2 != '\n' && c2 != '*')
		{
		  ;
		}
	      
	      if (c2 == '*')
		{
		  while ((c2 = char_fromInt (lminput ())) != '\0' 
			 && c2 == '*')
		    {
		      ;
		    }

		  if (c2 == '/')
		    {
		      goto outofcomment;
		    }
		}
	      else 
		{
		  llfatalerror (cstring_makeLiteral ("Macro: bad comment!"));
		}
	    }
	outofcomment:
	  return macro_nextChar ();
	}
      else
	{
          /*** putchar does not work!  why?  puts to stdio...??! ***/
          lmsavechar (c2);
	}
    }
  return c;
}

/*
** keeps stylized comments
*/

static char macro_nextCharC ()
{
  static bool in_quote = FALSE, in_escape = FALSE, in_char = FALSE;
  char c;

  c = char_fromInt (lminput ());

  if (!in_quote && !in_char && c == '\\')
    {
      while ((c = char_fromInt (lminput ())) != '\0' && c != '\n')
	{
	  ; /* skip to newline */
	}
      
      context_incLineno ();
      
      if (c != '\0')
	{
	  return macro_nextCharC ();
	}
      else
	{
	  return c;
	}
    }
  else if (!in_escape && c == '\"')
    {
      in_quote = !in_quote;
    }
  else if (!in_escape && c == '\'')
    {
      in_char = !in_char;
    }
  else if ((in_quote || in_char) && c == '\\')
    {
      in_escape = !in_escape;
    }
  else if ((in_quote || in_char) && in_escape)
    {
      in_escape = FALSE;
    }
  else if (!in_quote && c == '/')
    {
      char c2;
      
      if ((c2 = char_fromInt (lminput ())) == '*')
	{
	  while (c2 != '\0')
	    {
	      while ((c2 = char_fromInt (lminput ())) != '\0' 
		     && c2 != '\n' && c2 != '*')
		{
		  ;
		}
	      
	      if (c2 == '*')
		{
		  while ((c2 = char_fromInt (lminput ())) != '\0'
			 && c2 == '*')
		    {
		      ;
		    }

		  if (c2 == '/') 
		    {
		      goto outofcomment;
		    }
		}
	      else 
		{
		  llfatalerror (cstring_makeLiteral ("Macro: bad comment!"));
		}
	    }
	outofcomment:
	  return macro_nextCharC ();
	}
      else
	{
	  lmsavechar (c2);
	}
    }
  return c;
}

/*
** skips whitespace (handles line continuations)
** returns first non-whitespace character
*/

static char skip_whitespace ()
{
  char c;

  while ((c = macro_nextChar ()) == ' ' || c == '\t')
    {
      ;
    }

  return c;
}

static void handleMacro ()
{
  cstring mac = cstring_undefined;
  int macrocode;
  char c;

  while (currentColumn () > 2)
    {
      mac = cstring_appendChar (mac, ' ');
      setTokLength (-1);
    }
  
  while (((c = macro_nextCharC ()) != '\0') && (c != '\n'))
    {
      mac = cstring_appendChar (mac, c);
    }

  DPRINTF (("handle: %s", mac));

  macrocode = tokenMacroCode (mac);

  if (macrocode == BADTOK && !isArtificial (mac))
    {
      DPRINTF (("this add: <%s>", mac));
      context_addMacroCache (mac);
    }
  else
    {
      cstring_free (mac);
    }

  if (c == '\n')
    {
      context_incLineno ();
    }
}

static void processMacro (void)
{
  uentry e2;
  ctype ct;
  int noparams = 0;
  cstring fname = cstring_undefined;
  bool isspecfcn = FALSE;
  bool isiter = FALSE;
  bool skipparam = FALSE;
  bool isenditer = FALSE;
  bool unknownm = FALSE;
  bool hasParams = FALSE;
  bool emptyMacro = FALSE;
  char c = skip_whitespace ();
  fileloc loc = fileloc_noColumn (currentloc);

  /* are both of these necessary?  what do they mean? */
  uentryList specparams = uentryList_undefined;
  uentryList pn = uentryList_undefined;

  context_resetMacroMissingParams ();

  if (c == '\0' || c == '\n')
    {
      llcontbug (cstring_makeLiteral ("Bad macro"));
      fileloc_free (loc);
      return;
    }
  
  fname = cstring_appendChar (fname, c);  

  while ((c = macro_nextChar ()) != '(' && c != '\0'
	 && c != ' ' && c != '\t' && c != '\n')
    {
      fname = cstring_appendChar (fname, c);
    }

  if (c == ' ' || c == '\t' || c == '\n')
    {
      char oldc = c;

      if (c != '\n')
	{
	  while (c == ' ' || c == '\t')
	    {
	      c = macro_nextChar ();
	    }
	  unput (c);
	}

      if (c == '\n')
	{
	  emptyMacro = TRUE;
	  unput (c);
	}

      c = oldc;
    }

  hasParams = (c == '(');

  DPRINTF (("process: %s", fname));

  if (usymtab_exists (fname))
    {
      e2 = usymtab_lookupExpose (fname);
      ct = uentry_getType (e2);

      DPRINTF (("entry: %s", uentry_unparseFull (e2)));

      if (uentry_isCodeDefined (e2) 
	  && fileloc_isUser (uentry_whereDefined (e2)))
	{
	  if (optgenerror 
	      (FLG_MACROREDEF,
	       message ("Macro %s already defined", fname),
	       loc))
	    {
	      uentry_showWhereDefined (e2);
	      uentry_clearDefined (e2);
	    }

	  if (uentry_isFunction (e2))
	    {
	      uentry_setType (e2, ctype_unknown);
	      ct = ctype_unknown;
	      unknownm = TRUE;
	      context_enterUnknownMacro (e2); 
	    }
	  else
	    {
	      context_enterConstantMacro (e2);
	    }
	}
      else
	{
	  if (uentry_isForward (e2) && uentry_isFunction (e2))
	    {
	      unknownm = TRUE;

	      voptgenerror 
		(FLG_MACROFCNDECL,
		 message
		 ("Parameterized macro has no prototype or specification: %s ", 
		  fname),
		 loc);
	      
	      ct = ctype_unknown;
	      uentry_setType (e2, ctype_unknown);
	      
	      DPRINTF (("set forward defined: %s / %s", 
			uentry_unparse (e2),
			fileloc_unparse (loc)));
	      uentry_setFunctionDefined (e2, loc); 
	      uentry_setUsed (e2, fileloc_undefined);
	      context_enterUnknownMacro (e2); 
	    }
	  else
	    {
	      if (uentry_isIter (e2))
		{
		  isiter = TRUE;
		  specparams = uentry_getParams (e2);
		  noparams = uentryList_size (specparams);
		  uentry_setDefined (e2, loc);
		  context_enterIterDef (e2); 
		}
	      else if (uentry_isEndIter (e2))
		{
		  isenditer = TRUE;
		  uentry_setDefined (e2, loc);
		  context_enterIterEnd (e2); /* don't care about it now */
		  /* but should parse like an iter! */
		}
	      else if (uentry_isConstant (e2))
		{
		  if (hasParams)
		    {
		      voptgenerror 
			(FLG_INCONDEFS, 
			 message ("Constant %s implemented as parameterized macro",
				  fname),
			 currentloc);
		      
		      uentry_showWhereSpecified (e2);
		      uentry_setType (e2, ctype_unknown);
		      uentry_makeVarFunction (e2);
		      uentry_setDefined (e2, currentloc);
		      uentry_setFunctionDefined (e2, currentloc);
		      context_enterUnknownMacro (e2); 
		    }
		  else
		    {
		      if (!uentry_isSpecified (e2))
			{
			  fileloc oloc = uentry_whereDeclared (e2);

			  if (fileloc_isLib (oloc))
			    {
			      ;
			    }
			  else if (fileloc_isUndefined (oloc)
				   || fileloc_isPreproc (oloc))
			    {
			      if (!emptyMacro)
				{
				  voptgenerror
				    (FLG_MACROCONSTDECL,
				     message 
				     ("Macro constant %s not declared",
				      uentry_getName (e2)),
				     loc);			 
				}
			    }
			  else if (!fileloc_withinLines (oloc, loc, 2))
			    { /* bogus!  will give errors if there is too much whitespace */
			      voptgenerror
				(FLG_SYNTAX,
				 message 
				 ("Macro constant name %s does not match name in "
				  "previous constant declaration.  This constant "
				  "is declared at %q", fname, 
				  fileloc_unparse (oloc)),
				 loc);
			    }
			}

		      context_enterConstantMacro (e2);	      
		      cstring_free (fname);
		      fileloc_free (loc);
		      return;
		    }

		}
	      else if (ctype_isFunction (ct))
		{
		  isspecfcn = TRUE;
		  specparams = ctype_argsFunction (ct);
		  noparams = uentryList_size (specparams);
		  
		  uentry_setFunctionDefined (e2, loc); 
		  context_enterMacro (e2);
		}
	      else if (uentry_isVar (e2))
		{
		  if (hasParams)
		    {
		      voptgenerror
			(FLG_INCONDEFS,
			 message ("Variable %s implemented as parameterized macro", 
				  fname),
			 loc);

		      uentry_showWhereSpecified (e2);
		      uentry_setType (e2, ctype_unknown);
		      uentry_makeVarFunction (e2);
		      uentry_setDefined (e2, currentloc);
		      uentry_setFunctionDefined (e2, currentloc);
		      context_enterUnknownMacro (e2); 
		    }
		  else
		    {
		      uentry ucons = uentry_makeConstant (fname,
							  ctype_unknown,
							  loc);
		      if (uentry_isExpandedMacro (e2))
			{
			  ; /* okay */
			}
		      else
			{
			  if (optgenerror 
			      (FLG_INCONDEFS,
			       message ("Variable %s implemented by a macro",
					fname),
			       loc))
			    {
			      uentry_showWhereSpecified (e2);
			    }
			}

		      uentry_setDefined (e2, loc);
		      uentry_setUsed (ucons, loc);

		      context_enterConstantMacro (ucons);
		      cstring_free (fname);
		      return;
		    }
		}
	      else
		{
		  if (uentry_isDatatype (e2))
		    {
		      voptgenerror (FLG_SYNTAX,
				    message ("Type implemented as macro: %s", 
					     uentry_getName (e2)),
				    currentloc);
		      context_enterConstantMacro (e2);
		      swallowMacro ();
		    }
		  else
		    {
		      llcontbug 
			(message ("Unexpanded macro not function or constant: %q", 
				  uentry_unparse (e2)));
		      uentry_setType (e2, ctype_unknown);
		      
		      if (hasParams)
			{
			  uentry_makeVarFunction (e2);
			  uentry_setDefined (e2, currentloc);
			  uentry_setFunctionDefined (e2, currentloc);
			  context_enterUnknownMacro (e2); 
			}
		    }
		}
	    }
	}
    }
  else
    {
      uentry ce;

      voptgenerror 
	(FLG_MACROMATCHNAME,
	 message ("Unexpanded macro %s does not match name of a constant "
		  "or iter declaration.  The name used in the control "
		  "comment on the previous line should match.  "
		  "(Assuming macro defines a constant.)", 
		  fname),
	 loc);


      ce = uentry_makeConstant (fname, ctype_unknown, fileloc_undefined);      
      uentry_setUsed (ce, loc); /* perhaps bogus? */
      e2 = usymtab_supEntryReturn (ce);
      
      context_enterConstantMacro (e2);	      
      cstring_free (fname);
      fileloc_free (loc);
      return;
    }
  
  /* in macros, ( must follow immediatetly after name */
  
  if (hasParams)
    {
      int paramno = 0;
      
      c = skip_whitespace ();

      while (c != ')' && c != '\0')
	{
	  uentry  param;
	  bool    suppress = context_inSuppressRegion ();
	  cstring paramname = cstring_undefined;

	  /*
	  ** save the parameter location
	  */

	  decColumn ();
	  context_saveLocation ();
	  incColumn ();

	  while (c != ' ' && c != '\t' && c != ',' && c != '\0' && c != ')')
	    {
	      paramname = cstring_appendChar (paramname, c);
	      c = macro_nextChar ();
	    }
	  
	  if (c == ' ' || c == '\t') c = skip_whitespace ();

	  if (c == ',')
	    {
	      c = macro_nextChar ();
	      if (c == ' ' || c == '\t') c = skip_whitespace ();
	    }
	  
	  if (c == '\0')
	    {
	      llfatalerror (cstring_makeLiteral ("Bad macro syntax: uentryList"));
	    }
	  
	  if ((isspecfcn || isiter) && (paramno < noparams)
	      && !uentry_isElipsisMarker (uentryList_getN (specparams, paramno)))
	    {
	      uentry decl = uentryList_getN (specparams, paramno);
	      sRef sr;
	      
	      param = uentry_nameCopy (paramname, decl);

	      DPRINTF (("param: %s", sRef_unparseFull (param->sref)));
	      DPRINTF (("decl:  %s", uentry_unparseFull (decl)));
	      
	      uentry_setParam (param);
	      sr = sRef_makeParam (paramno, uentry_getType (param));

	      if (sRef_getNullState (sr) == NS_ABSNULL)
		{
		  ctype pt = ctype_realType (uentry_getType (param));

		  if (ctype_isUser (pt))
		    {
		      uentry te = usymtab_getTypeEntrySafe (ctype_typeId (pt));
		      
		      if (uentry_isValid (te))
			{
			  sRef_setStateFromUentry (sr, te);
			}
		    }
		  else
		    {
		      sRef_setNullState (sr, NS_UNKNOWN, currentloc);
		    }
		}

	      uentry_setSref (param, sr);
	      uentry_setDeclaredForceOnly (param, context_getSaveLocation ());

	      DPRINTF (("macroParam: %s, %s", uentry_unparseFull (param),
			fileloc_unparse (uentry_whereDeclared (param))));
	      
	      skipparam = isiter && uentry_isOut (uentryList_getN (specparams, paramno));
	    }
	  else
	    {
	      fileloc sloc = context_getSaveLocation ();

	      param = uentry_makeVariableSrefParam 
		(paramname, ctype_unknown, sRef_makeParam (paramno, ctype_unknown));
	      cstring_free (paramname);

	      DPRINTF (("possibly null: %s (%d / %d / %d)", uentry_getName (param),
			isiter, paramno, noparams));
	      sRef_setPosNull  (uentry_getSref (param), sloc);

	      uentry_setDeclaredForce (param, sloc);

	      skipparam = FALSE;
	      fileloc_free (sloc);
	    }

	  if (!skipparam)
	    {
	      llassert (!uentry_isElipsisMarker (param));

	      if (!suppress)
		{
		  sRef_makeUnsafe (uentry_getSref (param));
		  DPRINTF (("unsafe: %s", uentry_unparseFull (param)));
		}
	      
	      DPRINTF (("add param: %s / %s", 
			uentry_unparseFull (param),
			fileloc_unparse (uentry_whereDeclared (param))));
	      
	      pn = uentryList_add (pn, uentry_copy (param));
	      usymtab_supEntry (param);
	    }
	  else
	    {
	      /* don't add param */
	      uentry_free (param);
	    }

	  if (c == ',') 
	    {
	      (void) macro_nextChar ();
	      c = skip_whitespace ();
	    }

	  paramno++;
	}
      
      if (c == ')')
	{
	  if (isspecfcn || isiter)
	    {
	      if (paramno != noparams && noparams >= 0)
		{
		  advanceLine ();

		  voptgenerror 
		    (FLG_INCONDEFS,
		     message ("Macro %s specified with %d args, defined with %d", 
			      fname, noparams, paramno),
		     currentloc);

		  uentry_showWhereSpecified (e2);
		  uentry_resetParams (e2, pn);
		}
	    }
	  else
	    {
	      uentry_resetParams (e2, pn);
	    }
	}
    }
  else
    {
      /*
      ** the form should be:
      **
      ** # define newname oldname
      ** where oldname refers to a function matching the specification
      ** of newname.
      */

      if (unknownm)
	{
	  sRef_setGlobalScope ();
	  usymtab_supGlobalEntry (uentry_makeVariableLoc (fname, ctype_unknown));
	  sRef_clearGlobalScope ();
	}
      else
	{
	  context_setMacroMissingParams ();
	}
    }
  
  DPRINTF (("macro params: %s", uentryList_unparseFull (pn)));

  /* context_setuentryList (pn); */
  usymtab_enterScope ();

  fileloc_free (loc);
  cstring_free (fname);

  return;
}

static bool handleSpecial (char *yyt)
{
  char *l = mstring_create (MAX_NAME_LENGTH);
  char *fname = mstring_create (MAX_NAME_LENGTH);
  static bool reportcpp = FALSE;
  int lineno = 0;
  char c;
  char *ol;
  cstring olc;
  char *ofname = fname;

  DPRINTF (("handleSpecial: +%s+", yyt));

  strcpy (l, yyt + 1);

  /* Need to safe original l for deallocating. */
  ol = l;

  l += strlen (yyt) - 1;
  
  while ((c = char_fromInt (lminput ())) != '\n' && c != '\0')
    {
      *l++ = c;
    }

  *l = '\0';
  olc = cstring_fromChars (ol);
  DPRINTF (("olc: %s", olc));

  if (cstring_equalPrefix (olc, "pragma"))
    {
      char *pname = mstring_create (longUnsigned_fromInt (MAX_PRAGMA_LEN));
      char *opname = pname;
      char *ptr = ol + 6; /* pragma is six characters, plus space */
      int len = 0;
      
      DPRINTF (("pragma: %s / %s", olc, ptr));

      /* skip whitespace */
      while (((c = *ptr) != '\0') && isspace (c))
	{
	  ptr++;
	}

      DPRINTF (("ptr: %s", ptr));

      while (((c = *ptr) != '\0') && !isspace (c))
	{
	  len++;

	  if (len > MAX_PRAGMA_LEN)
	    {
	      break;
	    }

	  ptr++;
	  *pname++ = c;
	}

      *pname = '\0';
      DPRINTF (("opname = %s", opname));

      if (len == PRAGMA_LEN_EXPAND && mstring_equal (opname, PRAGMA_EXPAND))
	{
	  cstring exname = cstring_undefined;
	  uentry ue;
	  
	  ptr++; 
	  while (((c = *ptr) != '\0') && !isspace (c))
	    {
	      exname = cstring_appendChar (exname, c);
	      ptr++;
	    }
	     
	  DPRINTF (("exname = :%s:", exname));

	  ue = usymtab_lookupExposeGlob (exname);
	  
	  if (uentry_isExpandedMacro (ue))
	    {
	      if (fileloc_isPreproc (uentry_whereDefined (ue)))
		{
		  fileloc_setColumn (currentloc, 1);
		  uentry_setDefined (ue, currentloc);
		}
	    }
	}
    }
  else if (cstring_equalPrefix (olc, "ident"))
    {
      /* Some pre-processors will leave these in the code.  Ignore rest of line */
    }
  else if (sscanf (ol, " %d \"%s", &lineno, fname) == 2)
    {
      char *tmp = fname + strlen (fname);
      fileId fid;
      
      while (*tmp-- != '\"' && tmp > fname)
	{
	  ;
	}

      *++tmp = '\0';

      fname = removePreDirs (fname);
      fid = fileTable_lookupBase (context_fileTable (),
				  cstring_fromChars (fname));
      
      if (!(fileId_isValid (fid)))
	{
	  fid = fileTable_addFile (context_fileTable (), 
				   cstring_fromChars (fname));
	}

      DPRINTF (("setFilename: %s", fname));
      setFileLine (fid, lineno);
    }
  else if (sscanf (ol, " %d", &lineno) == 1)
    {
      setLine (lineno); /* next line is <cr> */
    }
  else
    {
      if (!reportcpp)
	{
	  llerror (FLG_SYNTAX,
		   message ("File contains preprocessor command: #%s.  " 
			    "Check LCLINT_CPPCMD environment variable.  "
			    "(Further instances unreported)",
			    cstring_fromChars (ol)));
	  reportcpp = TRUE;
	}
      sfree (ofname);
      sfree (ol);
      return TRUE;
    }

  sfree (ol);
  sfree (ofname);
  return FALSE;
}
  
static int handleLlSpecial ()
{ 
  int ic; 
  char c;
  char *s = mstring_createEmpty ();
  char *os; 
  int tok;
  int charsread = 0;
  bool isstart = FALSE;

  while (((ic = ninput ()) != (int)'@') && (ic != (int)'\0') && (ic != EOF))
    {
      c = (char)ic;
      s = mstring_append (s, c);
      charsread++;
    }

  os = s;

  if (charsread == 0 && ic == (int)'@')
    {
      DPRINTF (("processing: %d", isProcessingGlobMods ()));

      if (isProcessingGlobMods () && (*s == '\0'))
	{
	  sfree (os);
	  return QNOMODS; /* special token no modifications token */
	}
      else
	{
	  ;
	}
    }

  DPRINTF (("Handle Special: <%s>", os));

  tok = commentMarkerToken (os, &isstart);

  if (tok != BADTOK)
    {
      tokLength = charsread;
      sfree (os);
      inSpecPart = isstart;
      return tok;
    }

  while (*s == ' ' || *s == '\t' || *s == '\n') 
    {
      s++;
    }

  if (*s == '-' || *s == '+' || *s == '=') /* setting flags */
    {
      c = *s;

      while (c == '-' || c == '+' || c == '=')
	{
	  ynm set = ynm_fromCodeChar (c);
	  cstring thisflag;

	  s++;
	  
	  thisflag = cstring_fromChars (s);
	  
	  while ((c = *s) != '\0' && (c != '-') && (c != '=')
		 && (c != '+') && (c != ' ') && (c != '\t') && (c != '\n'))
	    {
	      s++;
	    }

	  *s = '\0';

	  if (!context_getFlag (FLG_NOCOMMENTS))
	    {
	      cstring flagname = thisflag;
	      flagcode fflag = identifyFlag (flagname);

	      if (flagcode_isSkip (fflag))
		{
		  ;
		}
	      else if (flagcode_isInvalid (fflag))
		{
		  if (isMode (flagname))
		    {
		      if (ynm_isMaybe (set))
			{
			  llerror
			    (FLG_BADFLAG, 
			     message 
			     ("Stylized comment attempts to restore flag %s.  "
			      "A mode flag cannot be restored.",
			      flagname));
			}
		      else
			{
			  context_setMode (flagname);
			}
		    }
		  else
		    {
		      llerror
			(FLG_BADFLAG, 
			 message ("Unrecognized option in stylized comment: %s", 
				  flagname));
		    }
		}
	      else if (flagcode_isGlobalFlag (fflag))
		{
		  llerror
		    (FLG_BADFLAG, 
		     message 
		     ("Stylized comment attempts to set global flag %s.  "
		      "A global flag cannot be set locally.",
		      flagname));
		}
	      else
		{
		  context_fileSetFlag (fflag, set);

		  if (flagcode_hasArgument (fflag))
		    {
		      if (ynm_isMaybe (set))
			{
			  llerror
			    (FLG_BADFLAG, 
			     message 
			     ("Stylized comment attempts to restore flag %s.  "
			      "A flag for setting a value cannot be restored.",
			      flagname));
			}
		      else
			{ /* cut-and-pastied from llmain...blecch */
			  cstring extra = cstring_undefined;
			  char *rest;
			  char *orest;
			  char rchar;
			  
			  *s = c;
			  rest = mstring_copy (s);
			  orest = rest;
			  *s = '\0';
			  
			  while ((rchar = *rest) != '\0'
				 && (isspace (rchar)))
			    {
			      rest++;
			      s++;
			    }
			  
			  while ((rchar = *rest) != '\0'
				 && !isspace (rchar))
			    {
			      extra = cstring_appendChar (extra, rchar);
			      rest++; 
			      s++;
			    }
			  
			  sfree (orest);

			  if (cstring_isUndefined (extra))
			    {
			      llerror 
				(FLG_BADFLAG,
				 message
				 ("Flag %s (in stylized comment) must be followed by an argument",
				  flagcode_unparse (fflag)));
			    }
			  else
			    {
			      s--;
			      
			      DPRINTF (("extra: %s / %s",
					flagcode_unparse (fflag),
					extra));

			      if (flagcode_hasValue (fflag))
				{
				  setValueFlag (fflag, extra);
				}
			      else if (flagcode_hasString (fflag))
				{
				  setStringFlag (fflag, extra);
				}
			      else
				{
				  BADEXIT;
				}
			    }
			}
		    }
		}
	    }
	  else
	    {
	      DPRINTF (("comment ignored"));
	    }

	  *s = c;
	  while ((c == ' ') || (c == '\t') || (c == '\n'))
	    {
	      c = *(++s);
	    }
	} 

      if (context_inHeader () && !isArtificial (cstring_fromChars (os)))
	{
	  DPRINTF (("add comment: %s", os));
	  context_addComment (cstring_fromCharsNew (os));
	}
      else
	{
	  DPRINTF (("don't add: %s", os));
	}
    }
  else
    {
      char *t = s;
      int macrocode;
      char tchar = '\0';

      while (*s != '\0' && *s != ' ' && *s != '\t' && *s != '\n') 
	{
	  s++;
	}

      if (*s != '\0') 
	{
	  tchar = *s;
	  *s = '\0';
	  s++;
	}

      DPRINTF (("comment: <%s>", t));

      t = cstring_toCharsSafe (cstring_downcase (cstring_fromChars (t)));

      macrocode = tokenMacroCode (cstring_fromChars (t));

      if (macrocode != BADTOK)
	{
	  tokLength = mstring_length (t);

	  sfree (t);
	  sfree (os);

	  if (macrocode == SKIPTOK)
	    {
	      return BADTOK;
	    }

	  return macrocode;
	}
      
      if (context_inHeader ())
	{
	  if (tchar != '\0')
	    {
	      *(s-1) = tchar;
	    }
	  
	  if ((context_inMacro () || context_inGlobalContext ())
	      && macrocode != SKIPTOK
	      && !isArtificial (cstring_fromChars (os)))
	    {
	      DPRINTF (("add comment: %s", os));
	      context_addComment (cstring_fromCharsNew (os));
	    }
	  else
	    {
	      /* sfree (os); */
	    }
	  
	  if (tchar != '\0')
	    {
	      *(s-1) = '\0';
	    }
	}

      DPRINTF (("t = %s", t));

      if (mstring_equal (t, "ignore"))
	{
	  if (!context_getFlag (FLG_NOCOMMENTS))
	    {
	      context_enterSuppressRegion ();
	    }
	}
      else if ((*t == 'i' || *t == 't')
	       && (*(t + 1) == '\0'))
	{
	  if (!context_getFlag (FLG_NOCOMMENTS)
	      && (*t == 'i' || context_getFlag (FLG_TMPCOMMENTS)))
	    {
	      context_enterSuppressLine (-1); /* infinite suppression */
	    }
	}
      else if (((*t == 'i') || (*t == 't'))
	       && ((*(t + 1) >= '0' && *(t + 1) <= '9')))
	{
	  bool tmpcomment = (*t == 't');
	  int val = -1; 
	  char *tt = t; /* don't mangle t, since it is free'd */
	  char lc = *(++tt);

	  if (lc >= '0' && lc <= '9')
	    {
	      val = (int)(lc - '0');
	      
	      lc = *(++tt);	  
	      while (lc >= '0' && lc <= '9')
		{
		  val *= 10;
		  val += lc - '0';
		  lc = *(++tt);
		}
	    }

	  if (!context_getFlag (FLG_NOCOMMENTS)
	      && (!tmpcomment || context_getFlag (FLG_TMPCOMMENTS)))
	    {
	      context_enterSuppressLine (val);
	    }
	}
      else if (mstring_equal (t, "end"))
	{
	  if (!context_getFlag (FLG_NOCOMMENTS))
	    {
	      context_exitSuppressRegion ();
	    }
	}
      else if (mstring_equal (t, "notfunction"))
	{
	 ; /* handled by pcpp */
	}
      else if (mstring_equal (t, "access"))
	{
	  cstring tname;
	  
	  while (TRUE)
	    {
	      while ((c = *s) && (c == ' ' || c == '\t' || c == '\n'))
		{
		  s++;
		}
	      
	      if (c == '\0')
		{
                   break;
		}

	      tname = cstring_fromChars (s);
	      
	      while ((c = *s) != '\0' && c != ' ' 
		     && c != '\t' && c != '\n' && c != ',') 
		{
		  s++;
		}

	      *s = '\0';

	      DPRINTF (("access %s", tname));

	      if (!context_getFlag (FLG_NOCOMMENTS) 
		  && !context_getFlag (FLG_NOACCESS))
		{
		  if (usymtab_existsType (tname))
		    {
		      usymId uid = usymtab_getTypeId (tname);

		      DPRINTF (("addAccess: %s (%d)", tname, uid));
		      context_addFileAccessType (uid);
		    }
		  else
		    {
		      if (!(context_inSuppressRegion ()
			    || context_inSuppressZone (currentloc)))
			{
			  llmsg 
			    (message
			     ("%q: Unrecognized type %s used in access comment",
			      fileloc_unparse (currentloc), tname));
			}
		    }
		}
	      
	      if (c != '\0') 
		{
		  s++;
		}
	      
	      if (c != ',' && c != ' ')
		{
		  break;
		}
	    }
	}
      else if (mstring_equal (t, "noaccess"))
	{
	  cstring tname;
	  char lc;
	  
	  while (TRUE)
	    {
	      while ((lc = *s) && (lc == ' ' || lc == '\t' || lc == '\n')) 
		{
		  s++;
		}
	      
	      if (lc == '\0')
		{
		 break;
		}

	      tname = cstring_fromChars (s);
	      
	      while ((lc = *s) != '\0' && lc != ' ' && lc != '\t' 
		     && lc != '\n' && lc != ',') 
		{
		  s++;
		}

	      *s = '\0';

	      if (!context_getFlag (FLG_NOCOMMENTS) 
		  && !context_getFlag (FLG_NOACCESS))
		{
		  if (usymtab_existsType (tname))
		    {
		      typeId tuid = usymtab_getTypeId (tname);
		      
		      if (context_couldHaveAccess (tuid))
			{
			  context_removeFileAccessType (tuid);
			}
		      else
			{
			  if (!(context_inSuppressRegion () 
				|| context_inSuppressZone (currentloc)))
			    {
			      uentry ue = usymtab_getTypeEntry (tuid);
			      
			      if (uentry_isAbstractDatatype (ue))
				{
				  llmsg
				    (message
				     ("%q: Non-accessible abstract type %s used in noaccess comment",
				      fileloc_unparse (currentloc), tname));
				}
			      else
				{
				  llmsg
				    (message
				     ("%q: Non-abstract type %s used in noaccess comment",
				      fileloc_unparse (currentloc), tname));
				}
			    }
			}
		    }
		  else
		    {
		      if (!(context_inSuppressRegion () 
			    || context_inSuppressZone (currentloc)))
			{
			  llmsg
			    (message
			     ("%q: Unrecognized type %s used in noaccess comment",
			      fileloc_unparse (currentloc), tname));
			}
		    }
		}
	      
	      if (lc != '\0') 
		{
		  s++;
		}
	      
	      if (lc != ',' && lc != ' ')
		{
		  break;
		}
	    }
	}
      else
	{
	  setTokLength (- (2 + charsread));

	  voptgenerror (FLG_UNRECOGCOMMENTS, 
			message ("Stylized comment unrecognized: %s", 
				 cstring_fromChars (os)), 
			currentloc);
	}

      sfree (t);
    }
  
  sfree (os); 
  return BADTOK;
}

static /*@only@*/ cstring makeIdentifier (char *s)
{
  char *c = mstring_create (size_toInt (strlen (s)) + 1);
  cstring id = cstring_fromChars (c);

  while (isalnum (*s) || (*s == '_') || (*s == '$')) 
    {
      *c++ = *s++;
    }

  *c = '\0';
  return (id);
}

/*@observer@*/ /*@dependent@*/ uentry coerceId (ctype c)
{
  cstring cn = LastIdentifier ();

  DPRINTF (("coerce id: %s", cn));

  if (!(usymtab_exists (cn)))
    {
      if (ctype_isUnknown (c))
	{
	  uentry ce;
	  fileloc loc = fileloc_createExternal ();
	  bool infunc = sRef_modInFunction ();

	  if (infunc)
	    {
	      sRef_setGlobalScope ();
	    }

	  ce = uentry_makeVariableLoc (cn, c);
	  uentry_setDefined (ce, loc);
	  fileloc_free (loc);

	  if (!context_inIterEnd ())
	    {
	      voptgenerror 
		(FLG_SYSTEMUNRECOG, 
		 message ("Unrecognized (possibly system) identifier: %q", 
			  uentry_getName (ce)), 
		 uentry_whereLast (ce));
	    }

	  uentry_setHasNameError (ce);
	  ce = usymtab_supReturnFileEntry (ce);

	  if (infunc)
	    {
	      sRef_clearGlobalScope ();
	    }
	}
      else
	{
	  llcontbug (message ("coerceId: bad call: %t", c));
	  return (uentry_makeVariableLoc (cn, ctype_unknown));
	}
    }
  
  return (usymtab_lookup (cn));
}

/*
** like, coerceId, but doesn't supercede for iters
*/

/*@observer@*/ uentry coerceIterId (ctype c)
{
  cstring cn = LastIdentifier ();

  if (!(usymtab_exists (cn)))
    {
      if (ctype_isUnknown (c))
	{
	  return uentry_undefined;
	}
      else
	{
	  llcontbug (message ("coerceId: bad call: %t", c));
	  return (uentry_makeVariableLoc (cn, ctype_unknown));
	}
    }
  
  return (usymtab_lookup (cn));
}

/*@observer@*/ cstring LastIdentifier ()
{
  return (lastidprocessed);
}

static int processIdentifier (cstring id)
{
  uentry le;

  DPRINTF (("processing: %s (%d)", id,
	    GlobalExpectingTypeName));
  context_clearJustPopped ();

  lastidprocessed = id; 

  if (context_getFlag (FLG_GNUEXTENSIONS))
    {
      int tok = BADTOK;
      
      if (cstring_equalLit (id, "__volatile__"))
	{
	  tok = QVOLATILE;
	}
      else if (cstring_equalLit (id, "__signed"))
	{
	  tok = QSIGNED;
	}
      else if (cstring_equalLit (id, "__unsigned"))
	{
	  tok = QUNSIGNED;
	}
      else if (cstring_equalLit (id, "__const__"))
	{
	  tok = QCONST;
	}
      else if (cstring_equalLit (id, "__attribute__")
	       || cstring_equalLit (id, "__asm__")
	       || cstring_equalLit (id, "__asm"))
	{
	  int depth = 0;
	  int ic;
	  
	  DPRINTF (("attribute!"));
	  
	  while ((ic = input ()) != EOF)
	    {
	      DPRINTF (("ic = %c", ic));
	      
	      if (ic == '(')
		{
		  depth++;
		}
	      else if (ic == ')')
		{
		  depth--;
		  if (depth == 0) break;
		}
	      else if (ic == '\n')
		{
		  context_incLineno ();
		}
	      else
		{
		  ;
		}
	    }
	  
	  llassert (ic == ')');
	  
	  return BADTOK;
	}
      else if (cstring_equalLit (id, "inline")
	       || cstring_equalLit (id, "__inline__"))
	{
	  tok = QINLINE;
	}
      
      if (tok != BADTOK)
	{
	  RETURN_TOK (tok);
	}
    }

  le = usymtab_lookupSafe (id);

  /*@-dependenttrans@*/

  if (uentry_isIter (le))
    {
      yylval.entry = le;
      return (ITER_NAME);
    }
  else if (uentry_isEndIter (le))
    {
      yylval.entry = le;
      return (ITER_ENDNAME);
    }
  else if (uentry_isUndefined (le))
    {
      yylval.cname = id;

      /* avoid parse errors for certain system built ins */

      if (GlobalExpectingTypeName && (cstring_firstChar (id) == '_')
	  && (cstring_secondChar (id) == '_'))
	{
	  yylval.ctyp = ctype_unknown;
	  return (TYPE_NAME_OR_ID);
	}

      return (NEW_IDENTIFIER);
    }
  else if (!uentry_isDeclared (le) && !uentry_isCodeDefined (le))
    {
      if (uentry_isDatatype (le))
	{
	  DPRINTF (("code not defined: %s", uentry_unparse (le)));
	  yylval.cname = id;
	  return (NEW_IDENTIFIER);
	}
      else
	{
	  yylval.entry = le;
	  DPRINTF (("id entry: %s", uentry_unparseFull (le)));
	  
	  return (IDENTIFIER); 
	}
    }
  else if (uentry_isDatatype (le))
    {
      if (!GlobalExpectingTypeName)
	{
	  yylval.cname = id;
	  return (NEW_IDENTIFIER);
	}
      else
	{
	  yylval.ctyp = uentry_getAbstractType (le);

	  DPRINTF (("abs: %s", ctype_unparse (yylval.ctyp)));
	  uentry_setUsed (le, currentloc);
	  DPRINTF (("type name (%s): %s", id, uentry_unparse (le)));
	  return (TYPE_NAME);
	}
    }
  else
    {
      yylval.entry = le;
      DPRINTF (("process id: %s", uentry_unparseFull (le)));
      
      return (IDENTIFIER); 
    }

  /*@=dependenttrans@*/
}

static bool processHashIdentifier (/*@only@*/ cstring id)
{
  DPRINTF (("processhash: %s", id));

  if (context_inMacro () || context_inIterDef () ||
      context_inIterEnd ())
    {
      uentry le;
      
      context_clearJustPopped ();

      lastidprocessed = id; 
      le = usymtab_lookupSafe (id);

      if (uentry_isParam (le) || uentry_isRefParam (le))
	{
	  return TRUE;
	}
      else
	{
	  return FALSE;
	}
    }
  else
    {
      return FALSE;
    }
}


static /*@only@*/ cstring processString ()
{
  char *nl = strchr (yytext, '\n');
  cstring ns = cstring_fromCharsNew (yytext);

  if (nl == NULL)
    {
      setTokLength (longUnsigned_toInt (cstring_length (ns)));
    }
  else
    {
      char *lastnl = nl;

      context_incLineno ();
      
      while ((nl = strchr ((nl + 1), '\n')) != NULL)
	{
	  context_incLineno ();
	  lastnl = nl;
	}
      setTokLengthT (strlen (lastnl) - 1);
    }

  DPRINTF (("string: %s", ns));
  return (ns);
}

static 
char processChar ()
{
  char fchar = *(yytext + 1);
  char next;

  if (fchar != '\\') return fchar;
  
  next = *(yytext + 2);
  switch (next)
    {
    case 'n': return '\n';
    case 't': return '\t';
    case '\"': return '\"';
    case '\'': return '\'';
    case '\\': return '\\';
    default: return '\0';
    }
}

static
double processFloat ()
{
  double ret = atof (yytext);

  DPRINTF (("process float: %s -> %lf", yytext, ret));
  return (ret);
}

static
long processHex ()
{
  return (atol (yytext));
}

static
long processDec ()
{
  return (atol (yytext));
}

static int
processSpec (int tok)
{
  size_t length = strlen (yytext);

  if (inSpecPart)
    {
      setTokLengthT (length);
      RETURN_TOK (tok);
    }
  else
    {
      
      context_saveLocation ();
      setTokLengthT (length);
      return (processIdentifier (makeIdentifier (yytext)));
    }
}
