/*  MRXXTIME.CMD  */
/*============================================================================*/
/*      Program  ID    :  MRXXTIME  - IBM Confidential                        */
/*      Filename       :  MRXXTIME.CMD                                        */
/*      Purpose        :  Calcluate total elapsed run time of a stress test.  */
/*      Author(s)      :  Jack Kipp, IBM Austin, TX                           */
/*      Date Written   :  13 Feb 1997                                         */
/*      Based on       :  REXXTIME.CMD                                        */
/*      Comments       :  This is the Merlin version of REXXTIME.CMD.         */
/*                                                                            */
/*      Command line for this test :                                          */
/*                                                                            */
/*         MRXXTIME xxxxxxx.yyy                                               */
/*                                                                            */
/*         xxxxxxx.yyy - name of run time log file                            */
/*                       For Merlin stress tests, the file to check           */
/*                       usually will be 'switch', the test script log.       */
/*                                                                            */
/*      REXX calls     :  RXQUEUE('Create'), RXQUEUE('Delete')                */
/*                                                                            */
/*      Comments       :  assumes tests will take six days or less to run;    */
/*                        spans wrap-around to the following day or week.     */
/******************************************************************************/
/*   PSEUDOCODE -                                                             */
/*   Read and parse command line arguments.                                   */
/*   Open the run time test log and load it into string memory.               */
/*   Find and read start and end date and time.                               */
/*   Calculate total run time.                                                */
/*   Exit.                                                                    */
/******************************************************************************/
/*                            revision history                                */
/*                                                                            */
/* Changes for OS/2 SE 1.30:                                                  */
/*                                                                            */
/*   Date       Name                 Description                              */
/* --------  -----------  ----------------------------------------------------*/
/*  9/15/90  J. Kipp      base version                                        */
/*  9/18/90  J. Kipp      don't show days; just show hours and minutes        */
/*  9/18/90  J. Kipp      show both digits of minutes even if less than 10    */
/*  3/15/93  J. Kipp      add to total hours hours in number of full days     */
/******************************************************************************/

   arg logname

/*ͻ*/
/* (main)               Here is the start of the test                     */
/*ͼ*/

      call defaults;                    /* set defaults and initial values */
      call openlog logname;             /* open log */
      logstring = loadlog(logname);     /* put log in string memory */
      starttime = findstring('starting' 'at' '1' '1' logstring);
      startdate = findstring('starting' 'at' '3' '1' logstring);
      posstop = LASTPOS('stopping',logstring);
      endstring = SUBSTR(logstring,posstop);
      endedtime = findstring('stopping' 'at' '1' '1' endstring);
      endeddate = findstring('stopping' 'at' '3' '1' endstring);

      jsday = julian(startdate);
      jeday = julian(endeddate);

      elapsed = difference(startdate starttime endeddate endedtime);

      say 'Total elapsed run time (hours:minutes): ' elapsed ;

exit;   /* end of main program */

defaults:
/*ͻ*/
/* (defaults)           Set default values and initialize variables       */
/*ͼ*/

      /* set default values */

      if logname = '' then logname = 'runtime.log';

      /* date and time separator initial values                           */
      /*   these are U.S. values; other countries may be different        */

      dsep = '/';
      tsep = ':';
      psep = '.';

return;

openlog:  procedure
parse arg logname
/*ͻ*/
/* (openlog) subroutine to open the run time log file                     */
/*ͼ*/

      if stream( logname, 'C', 'query exists') = '' then do;
         message = 'Cannot find run time log file ' || logname || '; exiting.';
         say message;
         exit;
      end;

      cond = stream( logname, 'C', 'open read' ) ;    /* open log */

return;

loadlog: procedure
parse arg InputName
/*ͻ*/
/* (loadlog) function to load the run time log file into a string and     */
/*           return that string to the caller                             */
/*ͼ*/

      stringout = '';
      IsLine = lines( InputName ) ;
      I = 1 ;

      Do While IsLine = 1
          stringin = linein( InputName,,1 ) ;
          stringout = stringout || stringin || ' ';
          IsLine = lines( InputName ) ;
          I = I + 1 ;
      End

return stringout;

findstring:  procedure
parse arg marker1 marker2 incr nwords stringin
/*ͻ*/
/* (findstring) function to find the date or time immediately following   */
/*              marker1 and marker2 using separator in stringin           */
/*ͼ*/

        pos1 = WORDPOS(marker1,stringin,1);
        pos2 = WORDPOS(marker2,stringin,pos1);
        pos3 = pos2 + incr;
        stringout = SUBWORD(stringin,pos3,nwords);

return stringout;

difference:  procedure expose dsep tsep psep jsday jeday
parse arg startdate starttime endeddate endedtime
/*ͻ*/
/* (difference) function to return the elapsed time from the four input   */
/*              strings startdate, starttime, enddate, and endtime; this  */
/*              is where numeric values are extracted from the strings    */
/*              and are checked against a calendar to calculate number    */
/*              of days and number of hours that the test has run;        */
/*              a string is returned showing elapsed time in              */
/*              hours:minutes.                                            */
/*ͼ*/

        /* get numeric values of start and end time */

        parse value starttime with shh (tsep) smm (tsep) sss (psep) suu .;
        parse value endedtime with ehh (tsep) emm (tsep) ess (psep) euu .;

        /* get numeric value of dates */

        parse value startdate with smon (dsep) sday (dsep) syear .;
        parse value endeddate with emon (dsep) eday (dsep) eyear .;

        /* initialize cumulative number of days count */
        days = 0;

        /* did we cross a year boundary? */
        if eyear > syear then do;
           /* subtract start year from end year */
           dyear = eyear - syear;
           /* did cross more than one year boundary? */
           if dyear > 1 then do;
              /* for each full year, add number of days in that year */
              fyear = dyear - 1;
              /* check each full year individually */
              do i = 1 to fyear by 1;
                 tyear = eyear - i;
                 /* is that year a leap year? */
                 if tyear // 4 = 0 then do;
                    days = days + 366;
                 end;
                 else do;
                    days = days + 365;
                 end;
              end;
              /* was last year a leap year? */
              if syear // 4 == 0 then do;
                 /* calculate number of days till end of last year */
                 jsrdays = 366 - jsday;
              end;
              /* if not a leap year */
              else do;
                 /* calculate number of days till end of last year */
                 jsrdays = 365 - jsday;
              end;
              /* add days */
              days = days + jsrdays + jeday;
           end;
           /* we did not cross more than one year boundary */
           else do;
              /* was last year a leap year? */
              if syear // 4 == 0 then do;
                 /* calculate number of days till end of last year */
                 jsrdays = 366 - jsday;
              end;
              /* if not a leap year */
              else do;
                 /* calculate number of days till end of last year */
                 jsrdays = 365 - jsday;
              end;
              /* add days */
              days = jsrdays + jeday;
           end;
        end;
        /* start and end date are in the same year */
        else do;
             days = jeday - jsday;
        end;

        /* if test started late at night and ended early in the morning,    */
        /* normalize the times and days so that elapsed time can be derived */

        if ehh < shh then do;
           days = days - 1;
           dhh = 24 * days;
           nhh = 24 + ehh + dhh;
        end;
        else nhh = ehh;

        hours = nhh - shh;

        if emm < smm then do;
           hours = hours - 1;
           nmm = 60 + emm;
        end;
        else nmm = emm;

        minutes = nmm - smm;

        if minutes < 10 then do;
          stringout = hours || tsep || '0' || minutes;
        end;
        else do;
          stringout = hours || tsep || minutes;
        end;

return stringout;

getday:  procedure
parse arg stringin
/*ͻ*/
/* (getday) function to convert the day of the week from a string to a    */
/*          number between one and seven                                  */
/*ͼ*/

        if COMPARE('Sun',stringin) == 0 then dayout = 1;
        if COMPARE('Mon',stringin) == 0 then dayout = 2;
        if COMPARE('Tue',stringin) == 0 then dayout = 3;
        if COMPARE('Wed',stringin) == 0 then dayout = 4;
        if COMPARE('Thu',stringin) == 0 then dayout = 5;
        if COMPARE('Fri',stringin) == 0 then dayout = 6;
        if COMPARE('Sat',stringin) == 0 then dayout = 7;

return dayout;

julian:  procedure expose dsep
parse arg inputdate
/*ͻ*/
/* (julian) converts a mm/dd/yyyy date to the julian number of that date. */
/*ͼ*/

        parse value inputdate with imon (dsep) iday (dsep) iyear .;

        /* initialize number of days per month in a non-leap year */
        mdays.1  = 31;          /* January */
        mdays.2  = 28;          /* February */
        mdays.3  = 31;          /* March */
        mdays.4  = 30;          /* April */
        mdays.5  = 31;          /* May */
        mdays.6  = 30;          /* June */
        mdays.7  = 31;          /* July */
        mdays.8  = 31;          /* August */
        mdays.9  = 30;          /* September */
        mdays.10 = 31;          /* October */
        mdays.11 = 30;          /* November */
        mdays.12 = 31;          /* December */

        /* initialize cumulative count of julian days */
        jdays = 0;

        /* how many full months are there? */
        fmon = imon - 1;

        /* if one or more months to add */
        if fmon > 0 then do;
           /* check each full month */
           do i = 1 to fmon by 1;
              /* if it is February */
              if i = 2 then do;
                 /* if it is a leap year */
                 if iyear // 4 = 0 then do;
                     adddays = 29;
                 end;
                 else do;
                     adddays = mdays.i;
                 end;
              end;
              /* in months other than February */
              else do;
                 adddays = mdays.i;
              end;
              /* now add that month's days */
              jdays = jdays + adddays;
           end;
        end;

        /* now add the days in the current month */
        jdays = jdays + iday;

return jdays;
