/*  REXXTIME.CMD  */
/*============================================================================*/
/*      Program  ID    :  REXXTIME  - IBM Confidential                        */
/*      Filename       :  REXXTIME.CMD                                        */
/*      Purpose        :  Calcluate total elapsed run time of a stress test.  */
/*      Author(s)      :  Jack Kipp, IBM System Test Group, Delray Beach, FL  */
/*      Date Written   :   9/15/90                                            */
/*                                                                            */
/*      Command line for this test :                                          */
/*                                                                            */
/*         REXXTIME runtime.log                                               */
/*                                                                            */
/*         runtime.log - name of run time log file                            */
/*                                                                            */
/*      REXX calls     :  RXQUEUE('Create'), RXQUEUE('Delete')                */
/*                                                                            */
/*      Comments       :  assumes tests will take six days or less to run;    */
/*                        spans wrap-around to the following day or week.     */
/******************************************************************************/
/*   PSEUDOCODE -                                                             */
/*   Read and parse command line arguments.                                   */
/*   Open the run time test log and load it into string memory.               */
/*   Find and read start and end date and time.                               */
/*   Calculate total run time.                                                */
/*   Exit.                                                                    */
/******************************************************************************/
/*                            revision history                                */
/*                                                                            */
/* Changes for OS/2 SE 1.30:                                                  */
/*                                                                            */
/*   Date       Name                 Description                              */
/* --------  -----------  ----------------------------------------------------*/
/*  9/15/90  J. Kipp      base version                                        */
/*  9/18/90  J. Kipp      don't show days; just show hours and minutes        */
/*  9/18/90  J. Kipp      show both digits of minutes even if less than 10    */
/*  3/15/93  J. Kipp      add to total hours hours in number of full days     */
/******************************************************************************/

   arg logname

/*ͻ*/
/* (main)               Here is the start of the test                     */
/*ͼ*/

      call defaults;                    /* set defaults and initial values */
      call openlog logname;             /* open log */
      logstring = loadlog(logname);     /* put log in string memory */
      startdate = findstring('START:' 'date' '1' logstring);
      starttime = findstring('START:' 'time' '1' logstring);
      endeddate = findstring('FINISH:' 'date' '1' logstring);
      endedtime = findstring('FINISH:' 'time' '1' logstring);

      elapsed = difference(startdate starttime endeddate endedtime);

      say 'Total elapsed run time (hours:minutes): ' elapsed ;

exit;   /* end of main program */

defaults:
/*ͻ*/
/* (defaults)           Set default values and initialize variables       */
/*ͼ*/

      /* set default values */

      if logname = '' then logname = 'runtime.log';

      /* date and time separator initial values                           */
      /*   these are U.S. values; other countries may be different        */

      dsep = '-';
      tsep = ':';
      psep = '.';

return;

openlog:  procedure
parse arg logname
/*ͻ*/
/* (openlog) subroutine to open the run time log file                     */
/*ͼ*/

      if stream( logname, 'C', 'query exists') = '' then do;
         message = 'Cannot find run time log file ' || logname || '; exiting.';
         say message;
         exit;
      end;

      cond = stream( logname, 'C', 'open read' ) ;    /* open log */

return;

loadlog: procedure
parse arg InputName
/*ͻ*/
/* (loadlog) function to load the run time log file into a string and     */
/*           return that string to the caller                             */
/*ͼ*/

      stringout = '';
      IsLine = lines( InputName ) ;
      I = 1 ;

      Do While IsLine = 1
          stringin = linein( InputName,,1 ) ;
          stringout = stringout || stringin || ' ';
          IsLine = lines( InputName ) ;
          I = I + 1 ;
      End

return stringout;

findstring:  procedure
parse arg marker1 marker2 nwords stringin
/*ͻ*/
/* (findstring) function to find the date or time immediately following   */
/*              marker1 and marker2 using separator in stringin           */
/*ͼ*/

        pos1 = WORDPOS(marker1,stringin,1);
        pos2 = WORDPOS(marker2,stringin,pos1);
        stringout = SUBWORD(stringin,pos2+2,nwords);

return stringout;

difference:  procedure expose dsep tsep psep
parse arg startdate starttime endeddate endedtime
/*ͻ*/
/* (difference) function to return the elapsed time from the four input   */
/*              strings startdate, starttime, enddate, and endtime; this  */
/*              is where numeric values are extracted from the strings    */
/*              and are checked against a calendar to calculate number    */
/*              of days and number of hours that the test has run;        */
/*              a string is returned showing elapsed time in              */
/*              hours:minutes.                                            */
/*ͼ*/

        /* get numeric values of start and end time */

        parse value starttime with shh (tsep) smm (tsep) sss (psep) suu .;
        parse value endedtime with ehh (tsep) emm (tsep) ess (psep) euu .;

        /* get numeric value of day of week */

        sday = getday(startdate);
        eday = getday(endeddate);

        /* if test started Saturday and ended Sunday or after, normalize */
        /* the two days so that number of days can be derived            */

        if eday < sday then nday = 7 + eday
        else nday = eday;
        days = nday - sday;

        /* if test started late at night and ended early in the morning,    */
        /* normalize the times and days so that elapsed time can be derived */

        if ehh < shh then do;
           days = days - 1;
           dhh = 24 * days;
           nhh = 24 + ehh + dhh;
        end;
        else nhh = ehh;

        hours = nhh - shh;

        if emm < smm then do;
           hours = hours - 1;
           nmm = 60 + emm;
        end;
        else nmm = emm;

        minutes = nmm - smm;

        if days > 0 then do;
           hours = hours + (24 * days);
        end;

        if minutes < 10 then do;
          stringout = hours || tsep || '0' || minutes;
        end;
        else do;
          stringout = hours || tsep || minutes;
        end;

return stringout;

getday:  procedure
parse arg stringin
/*ͻ*/
/* (getday) function to convert the day of the week from a string to a    */
/*          number between one and seven                                  */
/*ͼ*/

        if COMPARE('Sun',stringin) == 0 then dayout = 1;
        if COMPARE('Mon',stringin) == 0 then dayout = 2;
        if COMPARE('Tue',stringin) == 0 then dayout = 3;
        if COMPARE('Wed',stringin) == 0 then dayout = 4;
        if COMPARE('Thu',stringin) == 0 then dayout = 5;
        if COMPARE('Fri',stringin) == 0 then dayout = 6;
        if COMPARE('Sat',stringin) == 0 then dayout = 7;

return dayout;
