/*  RXQLOADQ.CMD  */
/*============================================================================*/
/*      Program  ID    :  RXQLOADQ  - IBM Confidential                        */
/*      Filename       :  RXQLOADQ.CMD                                        */
/*      Author(s)      :  Jack Kipp, IBM System Test, Delray Beach, FL        */
/*      Based on       :  RXQMAXQS.CMD amd RXQSINGQ.CMD, written by J. Kipp   */
/*      Date Written   :   4/12/89                                            */
/*      Date Revised   :  na                                                  */
/*                                                                            */
/*      Command line for this test :                                          */
/*                                                                            */
/*      RXQLOADQ method startlq lngqi maxlngq rxqmlogn loglevel temp          */
/*                                                                            */
/*      method  - 'FIFO' or 'LIFO' access ('FIFO' is default)                 */
/*      startlq - starting length of longname to append to queue name         */
/*      lngqi   - number of characters to add to longname on each iteration   */
/*      maxlngq - maximum length  of longname to append to queue name         */
/*      rxqmlogn- name of logfile (maxqs default; .log extension is appended) */
/*      loglevel- for logging any messages (6 default)                        */
/*                                                                            */
/*      REXX calls     :  RXQUEUE('Create'), RXQUEUE('Delete')                */
/*                        RXQUEUE('Set'), PUSH, QUEUE, PULL                   */
/*                                                                            */
/*      Comments       :  Queue limits test,                                  */
/*                           max # of elements in a queue                     */
/******************************************************************************/
/*   PSEUDOCODE -                                                             */
/*   Read command line arguments.                                             */
/*   For increasing lengths of longnames:                                     */
/*      Make longname.                                                        */
/*      Create one Q with that longname.                                      */
/*      Load that Q with data.                                                */
/*      Show count of Q entries.                                              */
/*      Cleanup Q.                                                            */
/*   Exit.                                                                    */
/******************************************************************************//* jak */
/*                            revision history                                *//* jak */
/*                                                                            *//* jak */
/*   Date     OS/2       Name                 Description                     *//* jak */
/* --------  -------  -----------  -----------------------------------------  *//* jak */
/*  4/14/89  SE 1.20  J. Kipp      no specified number of items to fill q     *//* jak */
/******************************************************************************//* jak */

   arg method startlq lngqi maxlngq rxqmlogn loglevel dsize

/*ͻ*/
/* (main)               Here is the start of the test                     */
/*ͼ*/

      call defaults;  /* set defaults and initialize variables     */
      call filllong;  /* load up the character table for longnames */

      do lngq=startlq by lngqi to maxlngq;

         if lngq=0 then do;   /* requested longname length is zero */
            lnbfr='';         /* so append a null string in that case */
         end;
         else do;             /* requested longname length is nonzero */
            call makelong;    /* generate a longname of length lngq */
         end;

         /* in the event of Ctrl-Break, go clean up the open queue */

         call on halt name cleanup;

         call makequeue;  /* create queue named with 'Q' & longname */
         call activate;   /* activate queue & fill it with data */
         call purgeq;     /* purge queue */
         call deleteq;    /* delete queue */
         call passfail;   /* did we pass or fail for this longname length? */
      end;

      call ending;

exit;   /* end of main program */

novalue:
/*ͻ*/
/* (novalue)            Error trapping for uninitialized variables        */
/*ͼ*/

      mlogmsg = '(novalue) No value ' || rc || ' at line ' || sigl;
      whatlevel = 2;
      call logit;
      say mlogmsg;
      mlogmsg = '(novalue) Program cannot continue.  Sorry about that.'
      call logit;
      say mlogmsg;

exit;  /* end of novalue */

defaults:
/*ͻ*/
/* (defaults)           Set default values:                               */
/*                                                                        */
/*                      Method of access:           FIFO                  */
/*                      Starting queue name length:    0                  */
/*                      Queue name length increment:  15                  */
/*                      Maximum queue name length:   255                  */
/*                      Logging level:                 6                  */
/*                      Data item size in characters:  5                  */
/*                                                                        */
/*                      Log file prefix:           loadq                  */
/*ͼ*/

      /* queue method default */

      if method='' then method='FIFO';
      call STRIP method,'t'   /* strip trailing blanks from access method */

      /* queue size defaults */

      if startlq='' then startlq=0;
      if lngqi='' then lngqi=15;
      if maxlngq='' then maxlngq=255;

      /* size of each data item in a queue entry */                             /* jak */

      if dsize='' then dsize=5;

      /* log file name default */

      if loglevel='' then loglevel=6;
      if rxqmlogn='' then rxqmlogn='loadq';
      call STRIP rxqmlogn,'t'   /* strip trailing blanks from log filename */

      rxqmlog = rxqmlogn || '.log';

      erase rxqmlog; /* if a previous version exists */

      /* start logging */

      mlogmsg = 'ENTER RXQLOADQ STRESS TEST';
      whatlevel = 1;
      call logit;
      say mlogmsg;

      /* log the longname parameters if we are debugging */

      mlogmsg = '(main) argument startlq = ' || startlq;
      whatlevel = 9;
      call logit;
      mlogmsg = '(main) argument maxlngq = ' || maxlngq;
      call logit;
      mlogmsg = '(main) argument lngqi   =' || lngqi;
      call logit;

return;  /* end of defaults */

filllong:
/*ͻ*/
/* (filllong)           Fill the character table for making longnames     */
/*                                                                        */
/*                      The table has upper, lower and digit character    */
/*                      graphics.                                         */
/*                                                                        */
/*                      Input:                                            */
/*                         lngq        requested queue name length        */
/*                      Output:                                           */
/*                         lnbfr       long name of the requested length  */
/*ͼ*/

      s.1=64; f.1=90;          /* Initialize byte value range of digit numb.*/
      s.2=47; f.2=57;          /* Initialize byte value range of upper alph.*/
      s.3=96; f.3=122;         /* Initialize byte value range of lower alph.*/

        i=0;
        do while i < 256;       /* Build character array of initialized chrs */
         do j=1 to 3;           /* of the three ranges specified             */
          n=s.j;                /* by starting and                           */
          do while n < f.j;     /* stopping at each range                    */
           if i>255 then leave; /* to the limit of the byte value.           */
           i=i+1; n=n+1;
           ctbl.i=d2c(n);       /* assign character to the array.            */

          end;                  /* loop of range.                            */
         end;                   /* loop of 3 ranges.                         */
        end;                    /* loop of byte value.                       */

return; /* end of filllong */

makelong:
/*ͻ*/
/* (makelong)           Build a longname of length requested              */
/*                                                                        */
/*                      The longname has upper, lower and digit character */
/*                      graphics, and is taken from the character table   */
/*                      built by the 'filllong' routine.                  */
/*                                                                        */
/*                      Input:                                            */
/*                         lngq        requested queue name length        */
/*                         ctbl        the character table                */
/*                      Output:                                           */
/*                         lnbfr       long name of the requested length  */
/*ͼ*/

      mlogmsg = '(makelong) Generating long names of length ' || lngq;
      whatlevel = 5;
      call logit;
      say mlogmsg;

/*      signal off novalue;             */
        lnbfr='';
/*      signal on novalue;              */
        if lngq<>'' & lngq^=0 then  /* For the size of the queue name     */
         do l=1 to lngq;
          lnbfr = lnbfr||ctbl.l;      /* assign characters of the array to  */
         end;                         /* string buffer.                     */

return; /* end of makelong */

makequeue:
/*ͻ*/
/* (makequeue)          Create queue                                      */
/*                                                                        */
/*                      The queue will be given a unique name starting    */
/*                      with the letter 'Q' and appending the long name.  */
/*                                                                        */
/*                             Qlnbfr                                     */
/*                                                                        */
/*                      where                                             */
/*                             lnbfr  is the long name appended           */
/*                                                                        */
/*                      Input:                                            */
/*                         lnbfr      the longname created by the         */
/*                                    longname routine                    */
/*ͼ*/

      newqname = 'Q';   /* just a starting value for new queue name */

      qname='Q' || lnbfr;
      mlogmsg = '(makequeue) Attempting to create queue ' || qname;
      whatlevel = 9;
      call logit;

/*    signal off novalue;            */
      newqname=RXQUEUE('Create',qname);
/*    signal on novalue;             */

      if length(newqname) = 0 then do;
         mlogmsg = '(makequeue) Cannot open queue name ' || qname;
         whatlevel = 4;
         call logit;
         say mlogmsg;
      end;
      else do;
         mlogmsg = '(makequeue) Queue created; name: ' || newqname;
         whatlevel = 7;
         call logit;
      end;

return;  /* end of makequeue */

activate:
/*ͻ*/
/* (activate)           Make this queue the active one                    */
/*ͼ*/

      signal off novalue;
      oq = RXQUEUE('Set',qname);
      signal on novalue;
      call fillq;                       /* fill q and verify its contents */

return;  /* end of activate */

fillq:
/*ͻ*/
/* (fillq)              Fill the queue with data.  Each queue element     */
/*                      will contain 3 items:  queue size, queue element  */
/*                      number, and a hash value.  Each number is to      */
/*                      occupy the number of spaces specified by dsize,   */
/*                      right-justified with blank padding to the left.   */
/*                      The numbers will be separated from each other     */
/*                      by one space.                                     */
/*                                                                        */
/*                      NOTE:  dsize = 5 by default; if you need > 5      */
/*                      digits for each number, REMEMBER, whatever length */
/*                      you choose...                                     */
/*                                                                        */
/*                         1.  All elements are assumed the SAME LENGTH.  */
/*                         2.  Change dsize argument value to start this. */
/*                                                                        */
/*                      Input:                                            */
/*                        qname   name of queue to be filled              */
/*ͼ*/

      logmsg = '(fillq) Filling queue named ' || qname;
      whatlevel = 5;
      call logit;
      say logmsg;

      addcount = 0;     /* total number of q elements added: start at zero */

      qrecnum = 0;      /* initializes q record number */

      call on error name qfull;  /* when q is full we stop the following loop */

      do forever;       /* this loop fills the queue */

         qrecnum = qrecnum + 1;         /* next q record number */

         /* pad the data element with blanks up to required size;       */
         /* right-justified, no decimal point and no fractional value   */

         dqrecnum = format(qrecnum,dsize,0);

         /* do either LIFO or FIFO insertion now */

         logmsg = '(fillq) Adding to queue ' qname;
         logmsg = logmsg || ' element ' || qrecnum || ':  ';
         logmsg = logmsg || dqrecnum;
         whatlevel = 7;

         select
            when method = 'LIFO' then do;
               logmsg = logmsg || ' in LIFO order';
               call logit;
               PUSH dqrecnum dqrecnum dqrecnum;
            end;
            otherwise do;               /* queue is FIFO */
               logmsg = logmsg || ' in FIFO order';
               call logit;
               QUEUE dqrecnum dqrecnum dqrecnum;
            end;
         end;

         addcount = addcount + 1;

      end;

qfull:
/*ͻ*/
/* (fillq)              The queue is full now.  Show how many elements    */
/*                      were written to it.                               */
/*                                                                        */
/*                      Control passes here from the do-forever loop when */
/*                      a PUSH or QUEUE command cannot be done because    */
/*                      the queue is full.                                */
/*ͼ*/


      logmsg = '(fillq) ' || addcount || ' elements added to queue ' || qname;
      whatlevel = 5;
      call logit;

      call off error;

return;  /* end of fillq */

purgeq:
/*ͻ*/
/* (purgeq)             Purge the active queue                            */
/*                                                                        */
/*                      Input:                                            */
/*                        addcount   number of elements in the queue      */
/*ͼ*/

       logmsg = '(purgeq) About to purge ' || addcount || ' elements from queue ' || qname;
       say logmsg;

       /* purge the queue */

       purgecount = 0;
       do qrecnum = 1 by 1 to addcount;
          PULL dqsize dqrecnum dhash;
          purgecount = purgecount + 1;
       end;

       /* log purge results */

       logmsg = '(purgeq) Elements added = ' || addcount || '; elements purged = ' || purgecount;
       whatlevel = 5;
       call logit;
       say logmsg;

return;  /* end of purgeq */

deleteq:
/*ͻ*/
/* (deleteq)            Delete the active queue                           */
/*                                                                        */
/*                      Input:                                            */
/*                        qname      the name of the queue to clean up    */
/*ͼ*/

       /* delete queue and log deletion results */

       dq = RXQUEUE('Delete',qname);
       if dq ^= 0 then do;
          logmsg = '(deleteq) Failed to delete queue ' || qname || '; result is ' || dq || '; should be 0';
          whatlevel = 4;
       end;
       else do;
          logmsg = '(deleteq) Deleted queue ' || qname;
          whatlevel = 5;
       end;
       call logit;
       say logmsg;

return;  /* end of deleteq */

passfail:
/*ͻ*/
/* (passfail)           Print PASS or FAIL message for this longname      */
/*                      length                                            */
/*                                                                        */
/*                      Right now the only way we know if this longname   */
/*                      length passes is whether we get this far without  */
/*                      bombing out on a REXX error.                      */
/*ͼ*/

      mlogmsg = '(passfail) RXQLOADQ PASSED for long name length ' || lngq;
      whatlevel = 1;
      call logit;
      say mlogmsg;

return;  /* end of passfail */

logit:
/*ͻ*/
/* (logit)              Log a message if <= requested logging level       */
/*                                                                        */
/*                      Input:                                            */
/*                        whatlevel  level at or above which to log       */
/*                        mlogmsg    the message to be logged             */
/*                                                                        */
/*                      Globals:                                          */
/*                        rxqmlog    the name of the logfile              */
/*                        loglevel   logging level requested at startup   */
/*ͼ*/

      if whatlevel <= loglevel then do;
         msgcnt = lineout(rxqmlog,mlogmsg);
      end;

return;  /* end of logit */

cleanup:
/*ͻ*/
/* (cleanup)            Ctrl-Break and Ctrl-C handler.                    */
/*                                                                        */
/*                      We come here to clean up if Ctrl-Break or Ctrl-C  */
/*                      was pressed.  Clean-up consists of purging and    */
/*                      deleting the active queue.                        */
/*                                                                        */
/*                      Input:                                            */
/*                        qname      the open queue                       */
/*ͼ*/

      mlogmsg = '(cleanup) Ctrl-Break or Ctrl-C was pressed; clean-up begins';
      whatlevel = 2;
      call logit;
      say mlogmsg;

      call purgeq;  /* purge the queue */
      call deleteq; /* delete the queue */
      call passfail; /* did we pass or fail for this longname length? */
      call ending;

exit;  /* end of cleanup */

ending:
/*ͻ*/
/* (ending)             End of the test                                   */
/*ͼ*/

      mlogmsg = 'EXIT RXQLOADQ STRESS TEST';
      whatlevel = 1;
      call logit;
      say mlogmsg;
/*    call off halt;        */  /* turn off Ctrl-Break handler */

return;  /* end of ending */
