/*  RXQREADS.CMD  */
/*============================================================================*/
/*      Program  ID    :  RXQREADS  - IBM Confidential                        */
/*      Filename       :  RXQREADS.CMD                                        */
/*      Author(s)      :  Jack Kipp, IBM System Test, Delray Beach, FL        */
/*      Based on       :  RXQREADQ.CMD, written by J. Kipp                    */
/*      Date Written   :   4/24/89                                            */
/*      Date Revised   :  na                                                  */
/*                                                                            */
/*    ***** IMPORTANT ******************************************************* */
/*    * The code in this program is intended to be identical in function    * */
/*    * to the code of RXQREADQ.CMD.  So, if one changes, the other should. * */
/*    *********************************************************************** */
/*                                                                            */
/*      Command line for this test :                                          */
/*                                                                            */
/*      RXQREADS method dqname tag loglevel logfile dsize sempre sleepcount   */
/*                                                                            */
/*      method  - 'FIFO' or 'LIFO' access ('FIFO' default)                    */
/*      dqname   - name of queue ('SESSION' default)                          */
/*      tag     - unique id for each run of this program ('A' default)        */
/*      loglevel- log messages <= specified level (6 default)                 */
/*      logfile - name of logfile ('RXQREADQ.log' default)                    */
/*      dsize   - size of each numeric data item in a queue entry (5 default) */
/*      sempre  - sem file prefix; one letter long ('X' default)              */
/*      sleepcount - # times to do sleep loop for FIFO queue (300 default)    */
/*                                                                            */
/*      REXX calls     :  RXQUEUE('Set'), PULL, PUSH, QUEUE,                  */
/*                        RXQUEUE('Delete')                                   */
/*                                                                            */
/*      Comments       :  Queue churning test.  This part of it reads the q,  */
/*                        eating entries of the same tag as used by one of    */
/*                        the RXQWRITQ tasks.  Entries with other tags are    */
/*                        put back on the queue for other RXQREADQ tasks to   */
/*                        eat.                                                */
/*                                                                            */
/*      Instructions:     Run as many of these as you run RXQWRITQ tasks      */
/*                        from a switcher script.  Give this run the same     */
/*                        <tag> as one of the RXQWRITQ tasks.  For example,   */
/*                        you could run:                                      */
/*                                                                            */
/*                        (1)  a RXQWRITEQ and a RXQREADQ with <tag> A;       */
/*                        (2)  a RXQWRITEQ and a RXQREADQ with <tag> B;       */
/*                        (3)  a RXQWRITEQ and a RXQREADQ with <tag> C.       */
/*                                                                            */
/*                        Each RXQREADQ task is to eat only queue entries     */
/*                        that have the specified <tag>.  So, RXQREADQ task A */
/*                        eats only A entries, B eats only B's, etc.          */
/******************************************************************************/
/*   PSEUDOCODE -                                                             */
/*   Read command line arguments.                                             */
/*   Set default and initial values.                                          */
/*   Activate <dqname>.  Assumed this q already created by RXQWRITQ.CMD.      */
/*   Do forever:                                                              */
/*      Read an entry from <dqname>.                                          */
/*      If not requested <tag>, put entry back on queue;                      */
/*      Else, eat the entry.                                                  */
/*   Cleanup Q when Ctrl-C is detected:                                       */
/*      Make control queue active.                                            */
/*      Wait till the control queue is no longer empty.                       */
/*      Pull the record from the control queue.                               */
/*      Save number of tags written.                                          */
/*      Delete the control queue.                                             */
/*      Make the data queue active again.                                     */
/*                                                                            */
/*      Until number of tags eaten equals or exceeds                          */
/*      number of tags written:                                               */
/*         Pull elements from the data queue.                                 */
/*                                                                            */
/*   Exit.                                                                    */
/******************************************************************************/
/*                            revision history                                */
/*                                                                            */
/*  Changes for OS/2 SE 1.20:                                                 */
/*                                                                            */
/*   Date       Name                 Description                              */
/* --------  -----------  ----------------------------------------------------*/
/*  4/24/89  J. Kipp      base version of the test                            */
/*  5/ 1/89  J. Kipp      allows option of testing SESSION queue              */
/*  5/ 1/89  J. Kipp      makes SESSION queue default queue                   */
/*  6/14/89  J. Kipp      replace ^ with                                     */
/*  6/23/89  J. Kipp      replace  with ^ till  is implemented              */
/*  7/21/89  J. Kipp      replace ^ with                                     */
/*  9/26/89  J. Kipp      purge q of our tags after Ctrl-C comes              */
/*  9/28/89  J. Kipp      set tags eaten to zero in defaults routine          */
/* 10/ 5/89  J. Kipp      put elapsed time (sec.) on each log message         */
/* 10/ 6/89  J. Kipp      if FIFO q has < 10 items, sleep for a while         */
/* 10/ 6/89  J. Kipp      make sleepcount an argument; default 300            */
/* 10/ 6/89  J. Kipp      make REXX source code fit in first 80 columns       */
/* 10/ 7/89  J. Kipp      fix screen group lingering problem with control q;  */
/*                        end this program only after getting Ctrl-C and      */
/*                        after eating number of tags written by corresponding*/
/*                        RXQWRITQ.CMD; number of tags data is in control q   */
/* 10/ 7/89  J. Kipp      use time('L') rather than time('E')                 */
/* 10/ 7/89  J. Kipp      use dqname for data queue & cqname for control q    */
/* 10/ 7/89  J. Kipp      if LIFO q has < 10 items, sleep for a while         */
/* 10/ 9/89  J. Kipp      set itispulled flag before PULLing a q item         */
/*                        hopefully this fixes screen group lingering problem */
/* 10/10/89  J. Kipp      check for non-empty q at the very last possible     */
/*                        moment before pulling a queue entry; trying to      */
/*                        fix the screen group lingering problem              */
/* 10/10/89  J. Kipp      correct misspelled isitpulled; was itispulled       */
/* 10/10/89  J. Kipp      do a RXQUEUE('Get') to see which queue is active    */
/* 10/12/89  J. Kipp      correct misspelled itispulled; was isitpulled       */
/* 10/17/89  J. Kipp      use external function rxqueue_get_flag in           */
/*                        RXSTRESS.DLL, written by Eric Mintz                 */
/* 10/17/89  J. Kipp      tell reader of <tag> when to begin reading, which   */
/*                        will be after writer of <tag> gets Ctrl-C and stops */
/*                        writing to the data q                               */
/* 10/17/89  J. Kipp      clear up the logic where a q item is pulled         */
/* 10/17/89  J. Kipp      all done if current reader is eating last tag in q  */
/* 10/18/89  J. Kipp      make logit parameter driven so that each call takes */
/*                        fewer commands, and so less time                    */
/* 10/18/89  J. Kipp      discontinue using control q                         */
/* 10/18/89  J. Kipp      discontinue using deleteq routine                   */
/* 10/18/89  J. Kipp      count the number of times tags were re-queued       */
/* 10/19/89  J. Kipp      get full pathname of logfile so that old version    */
/*                        of logfile may be deleted properly                  */
/* 10/19/89  J. Kipp      fix main program's call to pullone; needs word call */
/* 12/13/90  J. Kipp      correct the comments above                          */
/*                                                                            */
/*  Changes for OS/2 SE 2.00:                                                 */
/*                                                                            */
/* 12/13/90  J. Kipp      use a file as a semaphore between this program and  */
/*                        RXQREADQ.EXE; there will be 1 sem file per qsize;   */
/*                        presence of a sem file means that RXQREADQ.CMD is   */
/*                        being given control to empty the queue; RXQREADQ.CMD*/
/*                        is responsible for deleting the sem file after      */
/*                        deleting the queue                                  */
/*  2/21/91  J. Kipp      use a sem file prefix to uniquely indentify each    */
/*                        screen group's sem file                             */
/*  3/14/91  J. Kipp      change STRIP calls to function calls in defaults()  */
/*  4/19/91  J. Kipp      fix purgeq to use semfile to decide when to stop,   */
/*                        rather than using the external function rxqflag,    */
/*                        which was not 32-bitified                           */
/*  6/ 1/91  J. Kipp      corrects comments                                   */
/*  6/ 1/91  J. Kipp      wipes out source text past column 80                */
/*  6/ 1/91  J. Kipp      sleep only if the q is empty or has one element     */
/*  6/ 1/91  J. Kipp      stop eating tags if enough have been eaten          */
/*  6/ 1/91  J. Kipp      do not purge q after Ctrl-C comes; causes PULL from */
/*                        the keyboard                                        */
/******************************************************************************/

   arg method dqname tag loglevel logfile dsize sempre sleepcount .

/*ͻ*/
/* (main)               Here is the start of the program                  */
/*ͼ*/

     call off halt;                   /* don't allow Ctrl-C yet */
     call defaults;                   /* set default and initial values */
     qname = dqname;                  /* get ready to activate data queue */
     call activate;                   /* make the requested queue active */
     call on halt name cleanup;       /* go clean-up if Ctrl-C is detected */
     do forever;                      /* till Ctrl-C comes */
        call waittogo;                /* wait till writer of <tag>s says go */
        call eattags;                 /* read & eat of q entries with <tag> */
     end;

exit;  /* end of main program */

defaults:
/*ͻ*/
/* (defaults)           Set default and initial values:                   */
/*                                                                        */
/*                      Method of access:           FIFO                  */
/*                      Queue name:             RXQREADQ                  */
/*                      Tag:                           A                  */
/*                      Logging level:                 6                  */
/*                      Log file name:      RXQREADQ.LOG                  */
/*                                                                        */
/*                      Data item size in characters:  5                  */
/*ͼ*/

      /* logging defaults */
      if loglevel = '' then loglevel = 6;
      if logfile = '' then logfile = 'RXQREADQ.LOG'
      logfile = directory() || '\' || logfile;          /* tack on cur. dir. */
      logfile = strip(logfile,'t');                 /* strip trailing blanks */

      /* tag */
      if tag = '' then tag = 'A';
      tag = strip(tag,'t');                         /* strip trailing blanks */
      tag = substr(tag,1,1);    /* make sure tag is only one character */
      msgtag = '(RXQREADQ-' || tag || ') ';

      /* erase old log file if a previous version exists */
      if stream( logfile, 'c', 'query exists' ) = '' then do;
         logmsg = 'Erasing old log file ' || logfile;
         call logit logmsg, 5;
         erase logfile;
      end;

      /* start logging */
      logmsg = 'ENTER RXQREADQ STRESS TEST';
      call logit logmsg, 1;

      /* queue method default */
      if method='' then method='FIFO';
      method=strip(method,'t');                     /* strip trailing blanks */

      /* data q name */
      if dqname = '' then dqname = 'SESSION';

      /* semfile prefix */
      if sempre = '' then sempre = 'X';

      /* failure count; if zero we pass the test */
      failcount = 0;

      /* data items for each queue entry */
      dqsize = 0;       /* q size */
      dqrecnum = 0;     /* q record # */

      /* data item size */
      if dsize = '' then dsize = 5;
      logmsg = 'dsize = ' || dsize;
      call logit logmsg, 7;

      /* sleep count */
      if sleepcount = '' then sleepcount = 300;

      /* flag saying whether we have one more tag to eat or re-queue */
      itispulled = 0;   /* no tag to deal with right yet */

      /* counters */
      tagseaten = 0;    /* number of tags eaten */
      tagsreqd  = 0;    /* number of times tags were re-queued */

return;  /* end of defaults */

activate:
/*ͻ*/
/* (activate)           Make the requested queue the active one           */
/*ͼ*/

      signal off novalue;
      logmsg = '(activate) Activating queue ' || qname;
      call logit logmsg, 5;
      oq = RXQUEUE('Set',qname);
      currentq = RXQUEUE('Get');
      logmsg = '(activate) Currently active queue is ' || currentq;
      call logit logmsg, 5;
      signal on novalue;

return;  /* end of activate */

waittogo:
/*ͻ*/
/* (waittogo)           Wait till the writer of <tag>s says go            */
/*                      Presence of the sem file gives the go-ahead.      */
/*ͼ*/

      semname = sempre || 'RXQSEM' || tag;

      do while stream( semname, 'C', 'query exists') = '';
         nop;
      end;

return;  /* end of waittogo */

eattags:
/*ͻ*/
/* (eattags)            Read the queue and eat only entries with <tag>.   */
/*                      Put the non-<tag> entries back on the queue.      */
/*                                                                        */
/*                      NOTE:  dsize = 5 by default; if you need > 5      */
/*                      digits for each numeric item, REMEMBER, whatever  */
/*                      length you choose...                              */
/*                                                                        */
/*                         1.  All numeric items are assumed SAME LENGTH. */
/*                         2.  Change dsize argument value to start this. */
/*                                                                        */
/*                      Input:                                            */
/*                        method  'FIFO' or 'LIFO'                        */
/*                        qname   name of queue                           */
/*                        dsize   size of each numeric data item          */
/*ͼ*/

      logmsg = '(eattags) Eating ' || tag || ' tags from ';
      logmsg = logmsg || method || ' queue ' || qname;
      call logit logmsg, 5;

      tagsate = 0;                /* number of tags eaten this pass */

      do while stream( semname, 'C', 'query exists') = ''; /* while not done */
         call pullone;                                           /* eat a tag */
      end;

return;  /* end of eattags */

pullone:
/*ͻ*/
/* (pullone)            Pull a q entry; if it has the <tag> we are        */
/*                      are looking for, eat it; otherwise, put it        */
/*                      back on the queue                                 */
/*ͼ*/

         if queued() = 0 then nop                 /* do nothing if q is empty */
         else do;                                           /* q is not empty */
            itispulled = 1;                              /* flag a tag pulled */
            PULL dqsize dqrecnum dqtag;                       /* pull the tag */
            call isitours;                                /* see if it's ours */
            itispulled = 0;                              /* reset pulled flag */
         end;

return;  /* end of pullone */

isitours:
/*ͻ*/
/* (isitours)           A q entry was pulled; if it has the <tag> we are  */
/*                      are looking for, eat it; otherwise, put it        */
/*                      back on the queue                                 */
/*                                                                        */
/*                      Input:                                            */
/*                        method      queue access type:  FIFO or LIFO    */
/*                        tag         tag we look for in each q entry     */
/*                        sleepcount  # counts to sleep for FIFO q        */
/*ͼ*/

      itsourtag = compare(dqtag,tag);  /* is this our tag? */

      if itsourtag = 0 then do;       /* no, it's not */

         select
            when method = 'LIFO' then do;
               logmsg = 'Putting entry back on q in LIFO order: ';
               logmsg = logmsg || dqsize || ' ' dqrecnum || ' ' || dqtag;
               call logit logmsg, 7;
               PUSH dqsize dqrecnum dqtag;
               itispulled = 0;   /* we are done with current q entry */
            end;
            otherwise do;               /* queue is FIFO */
               logmsg = 'Putting entry back on q in FIFO order: ';
               logmsg = logmsg || dqsize || ' ' dqrecnum || ' ' || dqtag;
               call logit logmsg, 7;
               QUEUE dqsize dqrecnum dqtag;
               itispulled = 0;   /* we are done with current q entry */
            end;
         end;  /* end select FIFO or LIFO */

         tagsreqd = tagsreqd + 1;

         /* if FIFO or LIFO q has <= 1 items, sleep for a while     */

         if queued() <= 1 then do;
             waitcount = 0;
             do while waitcount < sleepcount;
                waitcount = waitcount + 1;
             end;
         end;

       end   /* end if it's not our tag */

       else do;                        /* YIPPEE!  It is our tag! */

         itispulled = 0;               /* we are done with current q entry */
         tagsate = tagsate + 1;        /* eat the tag */
         logmsg = 'Eating a queue entry: ';
         logmsg = logmsg || dqsize || ' ' dqrecnum || ' ' || tag;
         call logit logmsg, 7;

         /* if enough tags have been eaten, tell the tag writer to resume */
         if tagsate >= dqsize then do;
            cond = stream( semname, 'C', 'close');
            deletesem = 'del ' || semname;
            deletesem;
            tagseaten = tagseaten + tagsate;    /* add to grand total */
            tagsate = 0;                        /* reset tags eaten per pass */
         end;

       end;  /* end else it is our tag */

return;  /* end of isitours */

didwepass:
/*ͻ*/
/* (didwepass)          Print PASS or FAIL message                        */
/*                                                                        */
/*                      Input:                                            */
/*                        failcount  the # of times we just plain goofed  */
/*ͼ*/

      logmsg = '(didwepass) Number of ' || tag || ' tags eaten = ' || tagseaten;
      call logit logmsg, 5;
      logmsg = '(didwepass) Number of times tags were re-queued = ' || tagsreqd;
      call logit logmsg, 5;
      if failcount = 0 then do;
         logmsg = '(didwepass) RXQREADQ PASSED';
      end;
      else do;
         logmsg = '(didwepass) RXQREADQ FAILED';
      end;
      call logit logmsg, 1;

return;  /* end of didwepass */

logit:  procedure expose logfile loglevel msgtag
parse arg whatmsg, whatlevel
/*ͻ*/
/* (logit)              Log a message if <= requested logging level       */
/*                                                                        */
/*                      Input:                                            */
/*                        whatlevel  level at or above which to log       */
/*                        whatmsg   the message to be logged              */
/*                                                                        */
/*                      Globals:                                          */
/*                        logfile    the name of the logfile              */
/*                        loglevel   logging level requested at startup   */
/*                        msgtag     tag to be used on each message       */
/*ͼ*/

      newmsg = time('L') || ' ' || msgtag;
      newmsg = newmsg || whatmsg;
      if whatlevel <= loglevel then do;
         msgcnt = lineout(logfile,newmsg);
         say newmsg;
      end;

return;  /* end of logit */

cleanup:
/*ͻ*/
/* (cleanup)            Ctrl-Break and Ctrl-C handler.                    */
/*                                                                        */
/*                      We come here to clean up if Ctrl-Break or Ctrl-C  */
/*                      was pressed.  Clean-up consists of eating or re-  */
/*                      queueing the current tag, if it is different than */
/*                      the previous tag pulled.                          */
/*                                                                        */
/*                      Input:                                            */
/*                        qname      the open queue                       */
/*                        itispulled flag saying that a tag was pulled    */
/*ͼ*/

      logmsg = '(cleanup) Ctrl-Break or Ctrl-C was pressed; clean-up begins';
      call logit logmsg, 2;

      /* eat or re-queue current tag */
      if itispulled = 1 then call isitours;
/*    call purgeq;      */                  /* eat all the rest of our tags */               /* jak */
      call didwepass;                       /* did we pass or fail? */
      call ending;

exit;  /* end of cleanup */

purgeq:
/*ͻ*/
/* (purgeq)             Purge the active queue                            */
/*                                                                        */
/*                      Input:                                            */
/*                        qname      the name of the queue to clean up    */
/*                        tag        tag; says who put entry in q         */
/*                        addcount   how many tags to purge               */
/*                                                                        */
/*                      Assumed each q element has 3 data items:          */
/*                                                                        */
/*                        dqsize     q size                               */
/*                        dqrecnum   q record number                      */
/*                        dqtag      tag; says who put entry in q         */
/*ͼ*/

       logmsg = '(purgeq) About to purge queue ' || qname;
       logmsg = logmsg || ' of ' || tag || ' tags';
       call logit logmsg, 5;

       /* purge the queue of <addcount> <tag> tags */
       do while stream( semname, 'C', 'query exists') = '';/* while not done */
          call pullone;
       end;

       logmsg = '(purgeq) Finished purging queue ' || qname;
       logmsg = logmsg || ' of ' || tag || ' tags';
       call logit logmsg, 5;

return;  /* end of purgeq */

ending:
/*ͻ*/
/* (ending)             End of the test                                   */
/*ͼ*/

      logmsg = 'EXIT RXQREADQ STRESS TEST';
      call logit logmsg, 1;

return;  /* end of ending */

