/*  RXQSINGQ.CMD  */
/*============================================================================*/
/*    Program  ID    :  RXQSINGQ.CMD - IBM Confidential                       */
/*    Filename       :  RXQSINGQ.CMD                                          */
/*    Author(s)      :  Jack Kipp                                             */
/*                      IBM System Test Group, Delray Beach, FL               */
/*    Based on       :  Tony Ingenoso's STRQ01B.C                             */
/*                      from OS/2 base queue stress system test suite         */
/*    Date Written   :  1/6/89                                                */
/*    Date Revised   :  na                                                    */
/*                                                                            */
/*    Command line for this test :                                            */
/*                                                                            */
/*    RXQSINGQ method variety initqty incrqty maxqty loglevel logprefx        */
/*                                                                            */
/*    method  - 'FIFO' or 'LIFO' access ('FIFO' is default)                   */
/*    variety - 'NAMED' or 'SESSION' queue  ('NAMED' is default)              */
/*    initqty - initial number of queue entrys to start with (100 default)    */
/*    incrqty - number of entries to increase by each iteration (100 default) */
/*    maxqty  - max number of queue entrys to try (4000 default)              */
/*    loglevel- for logging any messages (6 default)                          *//* jak */
/*    logprefx- log file prefix (maximum 4 chars, prefixes a 4-digit #)       */
/*    dsize   - size in bytes of each data item in each queue element         */
/*                                                                            */
/*    REXX calls     :  RXQUEUE('Create'), RXQUEUE('Set'), PUSH, QUEUE        */
/*                                                                            */
/*    Comments       :  Tests FIFO & LIFO Qs, named and session queues.       */
/*                                                                            */
/*                      This REXX program works with C program RXQSINGL.EXE.  */
/*                      The REXX program sets up each queue and fills it.     */
/*                      The C program verifies the queue contents.            */
/*                                                                            */
/******************************************************************************/
/*                                                                            */
/*  PSEUDOCODE -                                                              */
/*  Read command line arguments.                                              */
/*  repeat the following steps with increasing Q sizes                        */
/*                                                                            */
/*    REXX program:                                                           */
/*     1.    Setup the queue.                                                 */
/*     2.    Write to the queue n items.                                      */
/*    C program:                                                              */
/*     3.    Query the Queue for the proper number items.                     */
/*     4.    Peek through the Queue and verify all items are present.         */
/*     5.    Delete the queue.                                                */
/*                                                                            */
/******************************************************************************//* jak */
/*                            revision history                                *//* jak */
/*                                                                            *//* jak */
/*  Changes for OS/2 SE 1.20:                                                 *//* jak */
/*                                                                            *//* jak */
/*   Date       Name                 Description                              *//* jak */
/* --------  -----------  ----------------------------------------------------*//* jak */
/*  2/21/89  J. Kipp      SINGLEQR.CMD renamed to RXQSINGR.CMD                *//* jak */
/*  2/21/89  J. Kipp      SINGLEQ.CMD renamed to RXQSINGQ.CMD                 *//* jak */
/*  2/21/89  J. Kipp      SINGLEQS.EXE renamed to RXQSINGL.EXE                *//* jak */
/*  3/ 2/89  J. Kipp      erase old copy of log file                          *//* jak */
/*  3/20/89  J. Kipp      log msgs only at or below level requested           *//* jak */
/*  3/20/89  J. Kipp      count & log number of queue elements added          *//* jak */
/*  3/23/89  J. Kipp      put all setup into defaults routine                 *//* jak */
/*  3/23/89  J. Kipp      comments to distinguish setup steps                 *//* jak */
/*  3/23/89  J. Kipp      clean up after Ctrl-Break or Ctrl-C                 *//* jak */
/*  3/23/89  J. Kipp      show msg saying Ctrl-Break was pressed              *//* jak */
/*  3/23/89  J. Kipp      put ending messages in a separate routine           *//* jak */
/*  3/23/89  J. Kipp      flag to say whether we have started C pgm.          *//* jak */
/*  3/23/89  J. Kipp      display novalue failure                             *//* jak */
/*  4/14/89  J. Kipp      fix chaining error for reading session qs           *//* jak */
/*  5/19/89  J. Kipp      do forever so it runs till Ctrl-C comes             *//* jak */
/*  6/14/89  J. Kipp      replace ^ with                                     *//* jak */
/*  6/23/89  J. Kipp      replace  with ^ till  is implemented              *//* jak */
/*  7/21/89  J. Kipp      replace ^ with                                     *//* jak */
/* 10/ 6/89  J. Kipp      put elapsed time (sec.) on each log message         *//* jak */
/* 10/ 7/89  J. Kipp      use time('L') rather than time('E')                 *//* jak */
/* 10/10/89  J. Kipp      wait till active queue is emptied by RXQSINGL.EXE   *//* jak */
/* 10/11/89  J. Kipp      purge until is queue is empty, rather than purging  *//* jak */
/*                        by getting the number of items added and purging    *//* jak */
/*                        that many; this is so that we don't go to the       *//* jak */
/*                        keyboard when the queue is already empty            *//* jak */
/* 10/12/89  J. Kipp      start C program RXQSINGL.EXE from here; eliminate   *//* jak */
/*                        the need for RXQSINGR.CMD to start RXQSINGL.EXE;    *//* jak */
/*                        eliminating RXQSINGR.CMD may fix timing problems    *//* jak */
/*                        that cause q size mismatches and scrambled q data   *//* jak */
/*  9/24/90  J. Kipp      use a file as a semaphore between this program and  */
/*                        RXQSINGL.EXE; there will be 1 sem file per qsize;   */
/*                        presence of a sem file means that the C program is  */
/*                        being given control to empty the queue; C program   */
/*                        is responsible for deleting the sem file after      */
/*                        deleting the queue                                  */
/******************************************************************************//* jak */

   arg method variety initqty incrqty maxqty loglevel logprefx dsize

/*ͻ*/
/* (main)               Here is the start of the test                     */
/*ͼ*/

      call defaults;  /* set defaults if parameters are not given  */

      call on halt name cleanup;                                                /* jak */

      do forever;     /* or at least till Ctrl-C comes */                       /* jak */

         do qsize = initqty by incrqty to maxqty;

             /* In the event of Ctrl-Break, go clean up the active queue. */    /* jak */
             /* Otherwise, if the sem file for this q size isn't present, */
             /* create the queue and fill it.                             */

             semname = 'SEM' || qsize;

             if stream( semname, 'C', 'query exists') = '' then do;

                if variety = 'NAMED' then do;
                   call buildq;             /*  create, activate, and fill q */
                end;
                else do;
                   qname = 'SESSION';
                   call activate;           /*  fill the session queue  */
                end;

             end;

         end;  /* end do qsize = initqty by incrqty to maxqty; */               /* jak */

      end;  /* end do forever */                                                /* jak */

      call ending;                                                              /* jak */

exit;  /* end of main program */

novalue:
/*ͻ*/
/* (novalue)            Error trapping                                    */
/*ͼ*/

      logmsg = time('L') || ' ' || msgtag;                                      /* jak */
      logmsg = logmsg || '(novalue) No value ' || rc || ' at line ' || sigl;
      whatlevel = 2;                                                            /* jak */
      call logit;                                                               /* jak */
      say logmsg;                                                               /* jak */
      logmsg = time('L') || ' ' || msgtag;                                      /* jak */
      logmsg = logmsg || '(novalue) Program cannot continue.  Sorry about that.';
      call logit;                                                               /* jak */
      say logmsg;                                                               /* jak */

exit;  /* end of novalue */

defaults:
/*ͻ*/
/* (defaults)           Set default values                                */
/*                                                                        */
/*             Type of queue:  FIFO NAMED                                 */  /* jak */
/*                                                                        */
/*             Initial number of queue entries to start with:     100     */
/*             Number of entries to increment on each iteration:  100     */
/*             Maximum number of queue entries to try:           4000     */
/*                                                                        */
/*             Logging file and level:  SFNREXX.LOG, 6                    */  /* jak */
/*             Size of each data item:  5                                 */  /* jak */
/*ͼ*/

      /* defaults for queue size and type */                                    /* jak */

      if method   = '' then method   = 'FIFO';
      if variety  = '' then variety  = 'NAMED';
      if initqty  = '' then initqty  = 100;
      if incrqty  = '' then incrqty  = 100;
      if maxqty   = '' then maxqty   = 4000;
      if loglevel = '' then loglevel = 6;

      /* size of each data item in a queue entry */                             /* jak */

      if dsize    = '' then dsize    = 5;

      /* log file name */                                                       /* jak */

      if logprefx = '' then logprefx = 'SFN';

      rxqslog = logprefx || 'rexx.log';

      erase rxqslog; /* if a previous version exists */                         /* jak */

      /* start logging */                                                       /* jak */

      msgtag = '(RXQSINGQ) ';                                                   /* jak */
      logmsg = time('L') || ' ' || msgtag;                                      /* jak */
      logmsg = logmsg || 'ENTER RXQSINGQ STRESS TEST';                                    /* jak */
      whatlevel = 1;                                                            /* jak */
      call logit;                                                               /* jak */
      say logmsg;

      logmsg = time('L') || ' ' || msgtag;                                      /* jak */
      logmsg = logmsg || '(defaults) initqty = ' || initqty;
      whatlevel = 9;                                                            /* jak */
      call logit;                                                               /* jak */

      logmsg = time('L') || ' ' || msgtag;                                      /* jak */
      logmsg = logmsg || '(defaults) incrqty = ' || incrqty;
      call logit;                                                               /* jak */

      logmsg = time('L') || ' ' || msgtag;                                      /* jak */
      logmsg = logmsg || '(defaults) maxqty  = ' || maxqty;
      call logit;                                                               /* jak */

      logmsg = time('L') || ' ' || msgtag;                                      /* jak */
      logmsg = logmsg || '(defaults) logprefx= ' || logprefx;
      call logit;                                                               /* jak */

      /* also initialize variables */                                           /* jak */

      qsize = 0;        /* to start */
      addcount = 0;     /* number of elements added to current queue */         /* jak */

return;  /* end of defaults */

buildq:
/*ͻ*/
/* (buildq)             Build each queue and fill it with entries         */
/*                                                                        */
/*                      Input:                                            */
/*                        qname   name of queue to be filled              */
/*                        qsize   number of elements to put in queue      */
/*ͼ*/

      /* create queue */

      logmsg = time('L') || ' ' || msgtag;                                      /* jak */
      logmsg = logmsg || '(buildq) Attempting to create queue';
      whatlevel = 8;                                                            /* jak */
      call logit;                                                               /* jak */

      signal off novalue;
      qname=RXQUEUE('Create');
      signal on novalue;

      if length(qname) = 0 then do;
         logmsg = time('L') || ' ' || msgtag;                                   /* jak */
         logmsg = logmsg || '(buildq) Cannot open queue';
         whatlevel = 4;                                                         /* jak */
         call logit;                                                            /* jak */
         say logmsg;
      end;
      else do;
         logmsg = time('L') || ' ' || msgtag;                                   /* jak */
         logmsg = logmsg || '(buildq) Queue created; name:  ' || qname;
         whatlevel = 5;                                                         /* jak */
         call logit;                                                            /* jak */
         say logmsg;

         /* make the queue active, fill it, and verify its contents */

         call activate;
      end;

return;  /* end of buildq */

activate:
/*ͻ*/
/* (activate)           Make this queue the active one                    */
/*ͼ*/

      signal off novalue;
      oq = RXQUEUE('Set',qname);
      signal on novalue;
      call fillq;                       /* fill q and verify its contents */

return;  /* end of activate */

fillq:
/*ͻ*/
/* (fillq)              Fill the queue with data.  Each queue element     */
/*                      will contain 3 items:  queue size, queue element  */
/*                      number, and a hash value.  Each number is to      */
/*                      occupy the number of spaces specified by dsize,   */
/*                      right-justified with blank padding to the left.   */
/*                      The numbers will be separated from each other     */
/*                      by one space.                                     */
/*                                                                        */
/*                      NOTE:  dsize = 5 by default; if you need > 5      */
/*                      digits for each number, REMEMBER, whatever length */
/*                      you choose...                                     */
/*                                                                        */
/*                         1.  All elements are assumed the SAME LENGTH.  */
/*                         2.  Change dsize argument value to start this. */
/*                         3.  Change NUM_LEN in RXQSINGL.C accordingly.  */  /* jak */
/*                                                                        */
/*                      Input:                                            */
/*                        qname   name of queue to be filled              */
/*                        qsize   number of elements to put in queue      */
/*ͼ*/

      logmsg = time('L') || ' ' || msgtag;                                      /* jak */
      logmsg = logmsg || '(fillq) Filling queue named ' || qname || ' with ' || qsize || ' entries';
      whatlevel = 5;
      call logit;                                                               /* jak */
      say logmsg;

      addcount = 0;     /* total number of q elements added: start at zero */

      do qrecnum = 1 by 1 to qsize;

         /* come up with the hash value */

         hash = (qsize + qrecnum) // 16;

         /* pad each data element with blanks up to required size;      */
         /* right-justified, no decimal point and no fractional value   */

         dqsize   = format(qsize,dsize,0);
         dqrecnum = format(qrecnum,dsize,0);
         dhash    = format(hash,dsize,0);

         /* do either LIFO or FIFO insertion now */

         logmsg = time('L') || ' ' || msgtag;                                   /* jak */
         logmsg = logmsg || '(fillq) Adding to queue ' qname;
         logmsg = logmsg || ' element ' || qrecnum || ':  ';
         logmsg = logmsg || dqsize || ' ' || dqrecnum || ' ' || dhash;
         whatlevel = 7;

         select
            when method = 'LIFO' then do;
               logmsg = logmsg || ' in LIFO order';
               call logit;                                                      /* jak */
               PUSH dqsize dqrecnum dhash;
               addcount = addcount + 1;                                         /* jak */
            end;
            otherwise do;               /* queue is FIFO */
               logmsg = logmsg || ' in FIFO order';
               call logit;                                                      /* jak */
               QUEUE dqsize dqrecnum dhash;
               addcount = addcount + 1;                                         /* jak */
            end;
         end;
      end;

      /* now that we have loaded the queue, put the sem file in */
      /* place so we can run the C program                      */

      semname = 'SEM' || qsize;
      cond = stream( semname, 'C', 'open write' ) ;    /* open sem file */
      cond = lineout( semname, semname ) ;             /* write one line */
      cond = stream( semname, 'C', 'close' ) ;         /* close sem file */

      logmsg = time('L') || ' ' || msgtag;                                      /* jak */
      logmsg = logmsg || '(fillq) ' || addcount || ' elements added to queue ' || qname;  /* jak */
      whatlevel = 5;                                                            /* jak */
      call logit;                                                               /* jak */

      call verifyq;     /* verify, purge, and delete queue with C program */

return;  /* end of fillq */

verifyq:
/*ͻ*/
/* (verifyq)            Verify the contents of the queue and then delete  */
/*                      the queue.  All this will be done by C program    */
/*                      RXQSINGL.C, which is started in a separate        */  /* jak */
/*                      session by REXX program RXQSINGR.CMD, which we    */  /* jak */
/*                      call here.                                        */
/*                                                                        */
/*                      Input:                                            */
/*                        method     'FIFO' or 'LIFO'                     */
/*                        variety    'NAMED' or 'SESSION'                 */
/*                        qsize      number of elements in the queue      */
/*                        loglevel   for TESTLOG utility                  */
/*                        logprefx   log file prefix                      */
/*ͼ*/

      call off halt;            /* turn off Ctrl-Break handler */               /* jak */

      rxqsclog = logprefx || qsize || '.LOG';    /* build child logfile name */

      cmdline = 'rxqsingl.exe';                                                                                         /* jak */
      cmdline = cmdline || ' -Crxqsingl..' || method || '..' || variety || '..' || qsize || '..in..progress..';         /* jak */
      cmdline = cmdline || ' -V'    || qname;     /* qname to Verify */
      cmdline = cmdline || ' -n'    || qsize;     /* size of queue */
      cmdline = cmdline || ' -q'    || method;    /* FIFO or LIFO */
      cmdline = cmdline || ' -v'    || variety;   /* NAMED or SESSION */
      cmdline = cmdline || ' -f'    || rxqsclog;  /* WRITELOG file */
      cmdline = cmdline || ' -l'    || loglevel;  /* WRITELOG logging level */

/*ͻ*/
/*                      NOW RUN THE C PROGRAM                             */
/*                                                                        */
/*                                                                        */
/*                      If using the session queue, run the C program in  */
/*                      the same session, so that it may be read.         */
/*                                                                        */
/*                      If using a named queue, run the C program in      */
/*                      a different session so that we may verify that    */
/*                      the named queue can be read properly from another */
/*                      session.                                          */
/*ͼ*/

      logmsg = time('L') || ' ' || msgtag;                                      /* jak */
      logmsg = logmsg || '(verifyq) About to run the C part of this test as follows:';
      whatlevel = 5;                                                            /* jak */
      call logit;                                                               /* jak */
      say logmsg;

      if variety = 'SESSION' then do;                                          /* jak */
         cmdline = 'start /c ' || cmdline;
      end;
      else do;
         cmdline = 'call ' || cmdline;  /* 'call ...' fixes chaining error */   /* jak */
      end;

      logmsg = time('L') || ' ' || msgtag;                                      /* jak */
      logmsg = logmsg || '(verifyq) ' || cmdline;
      call logit;                                                               /* jak */
      say logmsg;
      cmdline;

      /* re-activate the Ctrl-Break handler */                                  /* jak */

      call on halt name cleanup;                                                /* jak */

return;  /* end of verifyq */

logit:                                                                          /* jak */
/*ͻ*/  /* jak */
/* (logit)              Log a message if <= requested logging level       */  /* jak */
/*                                                                        */  /* jak */
/*                      Input:                                            */  /* jak */
/*                        whatlevel  level at or above which to log       */  /* jak */
/*                        logmsg     the message to be logged             */  /* jak */
/*                                                                        */  /* jak */
/*                      Globals:                                          */  /* jak */
/*                        rxqslog    the name of the logfile              */  /* jak */
/*                        loglevel   logging level requested at startup   */  /* jak */
/*ͼ*/  /* jak */

      if whatlevel <= loglevel then do;                                         /* jak */
         msgcnt = lineout(rxqslog,logmsg);                                      /* jak */
      end;                                                                      /* jak */

return;  /* end of logit */                                                     /* jak */

cleanup:
/*ͻ*/  /* jak */
/* (cleanup)            Ctrl-Break and Ctrl-C handler.                    */  /* jak */
/*                                                                        */  /* jak */
/*                      We come here to clean up if Ctrl-Break or Ctrl-C  */  /* jak */
/*                      was pressed.  Clean-up consists of purging and    */  /* jak */
/*                      deleting the last queue we created.  This purging */  /* jak */
/*                      and deleting is not done here if the C pgm. has   */  /* jak */
/*                      been called.  In that case, the C program takes   */  /* jak */
/*                      care of cleaning up.                              */  /* jak */
/*                                                                        */  /* jak */
/*                      Input:                                            */  /* jak */
/*                        qname      the name of the queue to clean up    */  /* jak */
/*                        addcount   number of elements in the queue      */  /* jak */
/*ͼ*/  /* jak */

      logmsg = time('L') || ' ' || msgtag;                                      /* jak */
      logmsg = logmsg || '(cleanup) Ctrl-Break or Ctrl-C was pressed; clean-up begins';   /* jak */
      whatlevel = 2;                                                            /* jak */
      call logit;                                                               /* jak */
      say logmsg;                                                               /* jak */

      semname = 'SEM' || qsize;

      if stream( semname, 'C', 'query exists') = '' then do;

         logmsg = time('L') || ' ' || msgtag;                                   /* jak */
         logmsg = logmsg || '(cleanup) Ctrl-Break handler will clean up queue ' || qname; /* jak */
         whatlevel = 2;                                                         /* jak */
         call logit;                                                            /* jak */
         say logmsg;                                                            /* jak */

         call purgeq;   /* remove all elements from the queue */                /* jak */

         if variety = 'SESSION' then do;                                       /* jak */
            call deleteq;  /* delete queue if it is not session queue */        /* jak */
         end;                                                                   /* jak */

      end;                                                                      /* jak */

      else do;                                                                  /* jak */
         logmsg = time('L') || ' ' || msgtag;                                   /* jak */
         logmsg = logmsg || '(cleanup) C program will clean-up queue ' || qname;          /* jak */
         whatlevel = 2;                                                         /* jak */
         call logit;                                                            /* jak */
         say logmsg;                                                            /* jak */
      end;                                                                      /* jak */

      call ending;                                                              /* jak */

exit;  /* end of cleanup */                                                     /* jak */

purgeq:
/*ͻ*/  /* jak */
/* (purgeq)             Purge the active queue                            */  /* jak */
/*                                                                        */  /* jak */
/*                      We come here to clean up if Ctrl-Break or Ctrl-C  */  /* jak */
/*                      was pressed, and the C program was not started    */  /* jak */
/*                      up.  Normally the C program takes care of purging */  /* jak */
/*                      and deleting the queue.  But if the C pgm. has    */  /* jak */
/*                      not been called, the Ctrl-Break handler calls     */  /* jak */
/*                      this routine as part of cleaning up.              */  /* jak */
/*                                                                        */  /* jak */
/*                      Input:                                            */  /* jak */
/*                        addcount   number of elements in the queue      */  /* jak */
/*ͼ*/  /* jak */

       /* purge the queue */                                                    /* jak */

       purgecount = 0;                                                          /* jak */
       do while queued() > 0;                                                   /* jak */
          PULL dqsize dqrecnum dhash;                                           /* jak */
          purgecount = purgecount + 1;                                          /* jak */
       end;                                                                     /* jak */

       /* log purge results */                                                  /* jak */

       logmsg = time('L') || ' ' || msgtag;                                     /* jak */
       logmsg = logmsg || '(purgeq) Elements added = ' || addcount || '; elements purged = ' || purgecount;     /* jak */
       whatlevel = 5;                                                           /* jak */
       call logit;                                                              /* jak */
       say logmsg;                                                              /* jak */

return;  /* end of purgeq */                                                    /* jak */

deleteq:
/*ͻ*/  /* jak */
/* (deleteq)            Purge the active queue                            */  /* jak */
/*                                                                        */  /* jak */
/*                      We come here to clean up if Ctrl-Break or Ctrl-C  */  /* jak */
/*                      was pressed, and the C program was not started    */  /* jak */
/*                      up.  Normally the C program takes care of purging */  /* jak */
/*                      and deleting the queue.  But if the C pgm. has    */  /* jak */
/*                      not been called, the Ctrl-Break handler calls     */  /* jak */
/*                      this routine as part of cleaning up.              */  /* jak */
/*                                                                        */  /* jak */
/*                      Input:                                            */  /* jak */
/*                        qname      the name of the queue to clean up    */  /* jak */
/*ͼ*/  /* jak */

       /* delete queue and log deletion results */                              /* jak */

       dq = RXQUEUE('Delete',qname);                                            /* jak */
       if dq = 0 then do;                                                      /* jak */
          logmsg = time('L') || ' ' || msgtag;                                  /* jak */
          logmsg = logmsg || '(deleteq) Tried to delete queue ' || qname || '; result is ' || dq || '; should be 0';    /* jak */
          whatlevel = 4;                                                        /* jak */
       end;                                                                     /* jak */
       else do;                                                                 /* jak */
          logmsg = time('L') || ' ' || msgtag;                                  /* jak */
          logmsg = logmsg || '(deleteq) Deleted queue ' || qname;                         /* jak */
          whatlevel = 5;                                                        /* jak */
       end;                                                                     /* jak */
       call logit;                                                              /* jak */
       say logmsg;                                                              /* jak */

return;  /* end of deleteq */

ending:                                                                         /* jak */
/*ͻ*/  /* jak */
/* (ending)             End of the test                                   */  /* jak */
/*ͼ*/  /* jak */

      logmsg = time('L') || ' ' || msgtag;                                      /* jak */
      logmsg = logmsg || 'EXIT RXQSINGQ STRESS TEST';                                     /* jak */
      whatlevel = 1;                                                            /* jak */
      call logit;                                                               /* jak */
      say logmsg;

/*    call off halt;        */  /* turn off Ctrl-Break handler */               /* jak */

return;  /* end of ending */
