/*  RXQWRITS.CMD  */
/*============================================================================*/
/*      Program  ID    :  RXQWRITS  - IBM Confidential                        */
/*      Filename       :  RXQWRITS.CMD                                        */
/*      Author(s)      :  Jack Kipp, IBM System Test, Delray Beach, FL        */
/*      Based on       :  RXQWRITQ.CMD; written by J. Kipp                    */
/*      Date Written   :   4/24/89                                            */
/*      Date Revised   :  na                                                  */
/*                                                                            */
/*    ***** IMPORTANT ******************************************************* */
/*    * The code in this program is intended to be identical in function    * */
/*    * to the code of RXQWRITQ.CMD.  So, if one changes, the other should. * */
/*    *********************************************************************** */
/*                                                                            */
/*      Command line for this test :                                          */
/*                                                                            */
/*      RXQWRITS method dqname qszst qszinc qszmax tag loglevel logfile dsize */
/*               makeit sempre                                                */
/*                                                                            */
/*      method  - 'FIFO' or 'LIFO' access ('FIFO' default)                    */
/*      dqname  - name of queue (SESSION queue default)                       */
/*      qsizest - starting queue size, number of entries (10 default)         */
/*      qsizeinc- queue size increment, each iteration (10 default)           */
/*      qsizemax- maximum queue size (30 default)                             */
/*      tag     - unique id for each run of this program ('A' default)        */
/*      loglevel- log messages <= specified level (6 default)                 */
/*      logfile - name of logfile ('rxqwritq.log' default)                    */
/*      dsize   - size of each numeric data item in a queue entry (5 default) */
/*      makeit  - 'CREATE' to create queue; null otherwise (no default)       */
/*      sempre  - sem file prefix; one letter long ('X' default)              */
/*                                                                            */
/*      REXX calls     :  RXQUEUE('Create'), RXQUEUE('Set'), PUSH, QUEUE,     */
/*                        RXQUEUE('Delete')                                   */
/*                                                                            */
/*      Comments       :  Queue churning test.  This part of it writes to the */
/*                        queue, putting in each queue entry the unique tag   */
/*                        you provide at startup.                             */
/*                                                                            */
/*      Instructions:     Run two or three of these at once from a switcher   */
/*                        script.  Give each run a unique <tag>.  For each    */
/*                        run of this program, run a RXQREADQ using the same  */
/*                        <tag>.  So, for example, you could run:             */
/*                                                                            */
/*                        (1)  a RXQWRITEQ and a RXQREADQ with <tag> A;       */
/*                        (2)  a RXQWRITEQ and a RXQREADQ with <tag> B;       */
/*                        (3)  a RXQWRITEQ and a RXQREADQ with <tag> C.       */
/*                                                                            */
/*                        Each RXQREADQ task is to eat only queue entries     */
/*                        that have the specified <tag>.  So, RXQREADQ task A */
/*                        eats only A entries, B eats only B's, etc.          */
/******************************************************************************/
/*   PSEUDOCODE -                                                             */
/*   Read command line arguments.                                             */
/*   Set default and initial values.                                          */
/*   Create queue <dqname>.                                                   */
/*   Activate <dqname>.                                                       */
/*   Do forever:                                                              */
/*      For increasing queue sizes from <qszst> to <qszmax>:                  */
/*          Write <qsize> entries to <dqname> with <tag> in each entry.       */
/*          Wait for queue to be emptied by RXQREADQ tasks.                   */
/*   Cleanup when Ctrl-C is detected:                                         */
/*      Create control queue and make it active.                              */
/*      Add a record to the control q, showing how many <tag> tags were added.*/
/*   Exit.                                                                    */
/******************************************************************************/
/*                            revision history                                */
/*                                                                            */
/*  Changes for OS/2 SE 1.20:                                                 */
/*                                                                            */
/*   Date       Name                 Description                              */
/* --------  -----------  ----------------------------------------------------*/
/*  4/24/89  J. Kipp      base version of the test                            */
/*  5/ 1/89  J. Kipp      allows option of testing SESSION queue              */
/*  5/ 1/89  J. Kipp      makes SESSION queue default queue                   */
/*  6/19/89  J. Kipp      shows return code from failed q create              */
/*  9/28/89  J. Kipp      initialize addcount zero before Ctrl-C is allowed   */
/* 10/ 5/89  J. Kipp      put elapsed time (sec.) on each log message         */
/* 10/ 6/89  J. Kipp      make REXX source code fit in first 80 columns       */
/* 10/ 7/89  J. Kipp      tell control q how many <tag> tags were written     */
/* 10/ 7/89  J. Kipp      use dqname for data queue & cqname for control q    */
/* 10/ 7/89  J. Kipp      use time('L') on each log message                   */
/* 10/ 7/89  J. Kipp      delete q if name requested is not created           */
/* 10/ 7/89  J. Kipp      add cqname at the end of the argument list          */
/* 10/ 7/89  J. Kipp      move addcount initialization from loadq to fillq    */
/* 10/ 9/89  J. Kipp      move addcount initialization before do forever      */
/* 10/10/89  J. Kipp      do a RXQUEUE('Get') to see which queue is active    */
/* 10/17/89  J. Kipp      use external function rxqueue_get_flag in           */
/*                        RXSTRESS.DLL, written by Eric Mintz                 */
/* 10/17/89  J. Kipp      tell reader of <tag> when to begin reading, which   */
/*                        will be after this writer gets Ctrl-C and stops     */
/*                        writing to the data q                               */
/* 10/18/89  J. Kipp      make logit parameter driven so that each call takes */
/*                        fewer commands, and so less time                    */
/* 10/18/89  J. Kipp      discontinue using control q                         */
/* 10/19/89  J. Kipp      get full pathname of logfile so that old version    */
/*                        of logfile may be deleted properly                  */
/*                                                                            */
/*  Changes for OS/2 SE 2.00:                                                 */
/*                                                                            */
/* 12/13/90  J. Kipp      use a file as a semaphore between this program and  */
/*                        RXQREADQ.EXE; there will be 1 sem file per qsize;   */
/*                        presence of a sem file means that RXQREADQ.CMD is   */
/*                        being given control to empty the queue; RXQREADQ.CMD*/
/*                        is responsible for deleting the sem file after      */
/*                        deleting the queue                                  */
/*  2/21/91  J. Kipp      use a sem file prefix to uniquely indentify each    */
/*                        screen group's sem file                             */
/*  3/14/91  J. Kipp      change STRIP calls to function calls in defaults()  */
/*  3/14/91  J. Kipp      check for rc 5 when checking for a full queue       */
/*  3/14/91  J. Kipp      SIGNAL ON ERROR and SIGNAL ON FAILURE are spurious  */
/*  4/19/91  J. Kipp      disable CALL OFF HALT; it gets error 25:  invalid   */
/*                        sub-keyword                                         */
/*  5/15/91  J. Kipp      cleans up cleanup routine; when it is time to let   */
/*                        the reader programs take over, write to the sem     */
/*                        file, instead of calling rxqflag of rxstress.dll,   */
/*                        which is 16-bit and so is not used in OS/2 2.00     */
/*  6/ 1/91  J. Kipp      corrects comments                                   */
/*  6/ 1/91  J. Kipp      wipes out source text past column 80                */
/*  8/28/91  J. Kipp      if a queue cannot be deleted by this program,       */
/*                        it is all right if it is because the queue          */
/*                        name is not valid, or if the named queue does       */
/*                        not exist; but if it cannot be deleted for any      */
/*                        other reason, then fail the test.                   */
/*                                                                            */
/******************************************************************************/

   arg method dqname qszst qszinc qszmax tag loglevel logfile dsize makeit sempre .

/*ͻ*/
/* (main)               Here is the start of the test                     */
/*ͼ*/

/*    call off halt;    */        /* don't allow Ctrl-C yet */
      call defaults;              /* set default and initial values */

      /* make q if not SESSION & not made yet */
      qname = dqname;
      if wecreateit = 0 then call makequeue;

      call activate;              /* make that queue active */
      call on halt name cleanup;  /* go clean-up if Ctrl-C is detected */
      call fillq;                 /* over and over again till Ctrl-C comes */

exit;   /* end of main program */

novalue:
/*ͻ*/
/* (novalue)            Error trapping for uninitialized variables        */
/*ͼ*/

      logmsg = 'No value ' || rc || ' at line ' || sigl;
      call logit logmsg, 2;
      logmsg = 'Program cannot continue.  Sorry about that.'
      call logit logmsg, 2;

exit;  /* end of novalue */

defaults:
/*ͻ*/
/* (defaults)           Set default and initial values:                   */
/*                                                                        */
/*                      Method of access:           FIFO                  */
/*                      Queue name:             RXQMULTI                  */
/*                      Starting queue size:          10                  */
/*                      Queue size increment:         10                  */
/*                      Maximum queue size:           30                  */
/*                      Tag:                           A                  */
/*                      Logging level:                 6                  */
/*                      Log file name:      RXQWRITQ.LOG                  */
/*                                                                        */
/*                      Data item size in characters:  5                  */
/*ͼ*/

      /* logging defaults */
      if loglevel = '' then loglevel = 6;
      if logfile = '' then logfile = 'RXQWRITQ.LOG'
      logfile = directory() || '\' || logfile;          /* tack on cur. dir. */
      logfile = strip(logfile,'t');                 /* strip trailing blanks */

      /* tag */
      if tag = '' then tag = 'A';
      tag = strip(tag,'t');                        /* strip trailing blanks */
      msgtag = '(RXQWRITQ-' || tag || ') ';

      /* erase old log file if a previous version exists */
      if stream( logfile, 'c', 'query exists' ) = '' then do;
         logmsg = 'Erasing old log file ' || logfile;
         call logit logmsg, 5;
         erase logfile;
      end;

      /* start logging */
      logmsg = 'ENTER RXQWRITQ STRESS TEST';
      call logit logmsg, 1;

      /* queue method default */
      if method='' then method='FIFO';
      method = strip(method,'t');              /* strip trailing blanks */

      /* queue size defaults */
      if qsizest = ''   then qsizest = 10;        /* default */
      if qsizest > 1000 then qsizest = 1000;      /* upper bound */
      if qsizest < 1    then qsizest = 1;         /* lower bound */
      if qsizeinc = ''   then qsizeinc = 10;      /* default */
      if qsizeinc > 1000 then qsizeinc = 1000;    /* upper bound */
      if qsizeinc < 0    then qsizeinc = 0;       /* lower bound */
      if qsizemax = ''    then qsizemax = 30;     /* default */
      if qsizemax > 10000 then qsizemax = 10000;  /* upper bound */
      if qsizemax < 10    then qsizemax = 10;     /* lower bound */

      /* data q name */
      if dqname = '' then dqname = 'SESSION';

      /* semfile prefix */
      if sempre = '' then sempre = 'X';

      /* failure count; if zero we pass the test */
      failcount = 0;

      /* data items for each queue entry */
      dqsize = 0;       /* q size */
      dqrecnum = 0;     /* q record # */

      /* data item size */
      if dsize = '' then dsize = 5;
      logmsg = '(defaults) dsize = ' || dsize;
      call logit logmsg, 7;

      /* create-q option */
      makeit = strip(makeit,'t');                 /* strip trailing blanks */
      logmsg = '(defaults) makeit = ' || makeit;
      call logit logmsg, 7;

      /* session queue flag */
      isitsession = compare(dqname,'SESSION');  /* is it SESSION q? */

      /*****************************************************************/
      /* if using SESSION queue, don't allow creating a queue, even if */
      /* operator goofed and asked for a queue to be created           */
      /*                                                               */
      /* in other words, the CREATE option has no effect when using    */
      /* the session queue                                             */
      /*****************************************************************/

      if isitsession = 0 then makeit='';       /* then don't create q */
      wecreateit = compare(makeit,'CREATE');   /* do we create q?  0 = yes */

      /* set initial value for total number of queue entries added */
      addcount = 0;                       /* total # of q entries added */

return;  /* end of defaults */

makequeue:
/*ͻ*/
/* (makequeue)          Create queue                                      */
/*                                                                        */
/*                      The queue will be given the name you request at   */
/*                      startup, or the default name.                     */
/*ͼ*/

      newqname = '';   /* just a starting value for new queue name */

      logmsg = '(makequeue) Attempting to create queue ' || qname;
      call logit logmsg, 9;

      do until length(newqname) > 0 & compare(newqname,qname) = 0;
/*       signal off novalue;            */
         newqname=RXQUEUE('Create',qname);
/*       signal on novalue;             */
         if length(newqname) = 0 then do;
            logmsg = '(makequeue) Cannot open queue name ';
            logmsg = logmsg || qname || '; return code is ' || rc;
            call logit logmsg, 4;
         end;
         else do;
            logmsg = '(makequeue) Queue created; name: ' || newqname;
            call logit logmsg, 5;
            if newqname = qname then do;  /* q created but name is wrong */
                intendedq = qname;         /* save intended qname */
                qname = newqname;          /* load arg for deleteq */
                call deleteq;              /* deleteq with wrong name */
                qname = intendedq;         /* reload intended qname */
            end;
         end;
      end;

return;  /* end of makequeue */

activate:
/*ͻ*/
/* (activate)           Make this queue the active one                    */
/*ͼ*/

      signal off novalue;
      logmsg = '(activate) Activating queue ' || qname;
      call logit logmsg, 5;
      oq = RXQUEUE('Set',qname);
      currentq = RXQUEUE('Get');
      logmsg = '(activate) Currently active queue is ' || currentq;
      call logit logmsg, 5;
      signal on novalue;

return;  /* end of activate */

fillq:
/*ͻ*/
/* (fillq)              Fill the queue with data.  Each queue element     */
/*                      will contain 3 items:  queue size, queue record   */
/*                      number, and the tag.  Each number is to occupy    */
/*                      the number of spaces specified by dsize, right-   */
/*                      justified with blank padding to the left.  The    */
/*                      data items will be separated from each other      */
/*                      by one space.                                     */
/*                                                                        */
/*                      NOTE:  dsize = 5 by default; if you need > 5      */
/*                      digits for each number, REMEMBER, whatever length */
/*                      you choose...                                     */
/*                                                                        */
/*                         1.  All numeric items are assumed SAME LENGTH. */
/*                         2.  Change dsize argument value to start this. */
/*                                                                        */
/*                      Input:                                            */
/*                        method  'FIFO' or 'LIFO'                        */
/*                        qname   name of queue to be filled              */
/*                        qszst   initial queue size                      */
/*                        qszinc  queue size increment                    */
/*                        qszmax  maximum queue size                      */
/*                        dsize   size of each numeric data item          */
/*ͼ*/

      addcount = 0;                   /* total # of q entries added */

      do forever;                               /* till Ctrl-C comes */

         do qsize = qszst by qszinc to qszmax;  /* for increasing q sizes */

            call loadq;         /* load q with <qsize> items */

            logmsg = '(fillq) Waiting for queue ' || qname || ' to empty';
            call logit logmsg, 5;

            /* now write the sem file to let queue readers take over */

            semname = sempre || 'RXQSEM' || tag;
            cond = stream( semname, 'C', 'open write' ) ; /* open semfile */
            cond = lineout( semname, semname ) ;          /* write 1 line */
            cond = stream( semname, 'C', 'close' ) ;      /* close semfile */

            /* now wait for the readers to empty the q; a reader that */
            /* empties q will signal the writers by deleting sem file */

            do until stream( semname, 'C', 'query exists') = '';
               nop;                    /* wait for RXQREADQs to empty q */
            end;

         end; /* end qszst by qszinc to qszmax */

      end;  /* end do forever */

return;  /* end of fillq */

loadq:
/*ͻ*/
/* (loadq)              Load up queue <qname> with <qsize> entries        */
/*ͼ*/

      logmsg = '(loadq) Loading ' || method || ' queue ' || qname;
      logmsg = logmsg || ' with ' || qsize || ' tag '|| tag || ' entries';
      call logit logmsg, 5;

      dqsize = format(qsize,dsize,0); /* right-justify qsize, pad with blanks */
      qrecnum = 0;                        /* initial q record number */

      signal on syntax name isqfull;   /* if q is full, empty it */
/*    signal on error name isqfull;       if q is full, empty it */
/*    signal on failure name isqfull;     if q is full, empty it */

      do qrecnum = 1 by 1 to qsize;       /* fill up with qsize entries */

         dqrecnum = format(qrecnum,dsize,0);    /* right-justify, pad blanks */

         logmsg = '(loadq) Adding to ' || qname || ':  ';
         logmsg = logmsg || dqsize || ' ' || dqrecnum || ' ' || tag;

         select
            when method = 'LIFO' then do;
               logmsg = logmsg || ' in LIFO order';
               call logit logmsg, 7;
               PUSH dqsize dqrecnum tag;
            end;
            otherwise do;               /* queue is FIFO */
               logmsg = logmsg || ' in FIFO order';
               call logit logmsg, 7;
               QUEUE dqsize dqrecnum tag;
            end;
         end;  /* end select FIFO or LIFO */

         addcount = addcount + 1;      /* we've added one more entry */

      end; /* end 1 by 1 to qsize */

isqfull:
      signal off syntax;
      signal off error;
      signal off failure;
                                        /* q is full */
      if rc = '48' then do;
          logmsg = '(loadq) Queue ' || qname || ' is full as of line ' || sigl;
          call logit logmsg, 4;
      end;                              /* end if queue is full */
      if rc = '5' then do;
          logmsg = '(loadq) Queue ' || qname || ' is out of resource ' ||
                   'as of line ' || sigl;
          call logit logmsg, 4;
      end;                              /* end if queue is out of resources */

return;  /* end of loadq */

didwepass:
/*ͻ*/
/* (didwepass)          Print PASS or FAIL message; show how many entries */
/*                      were added to the q                               */
/*                                                                        */
/*                      Input:                                            */
/*                        failcount  the # of times we just plain goofed  */
/*ͼ*/

      logmsg = '(didwepass) Elements added to queue ';
      logmsg = logmsg || qname || ' = ' || addcount;
      call logit logmsg, 5;
      if failcount = 0 then do;
         logmsg = '(didwepass) RXQWRITQ PASSED';
      end;
      else do;
         logmsg = '(didwepass) RXQWRITQ FAILED';
      end;
      call logit logmsg, 1;

return;  /* end of didwepass */

logit:  procedure expose logfile loglevel msgtag
parse arg whatmsg, whatlevel
/*ͻ*/
/* (logit)              Log a message if <= requested logging level       */
/*                                                                        */
/*                      Input:                                            */
/*                        whatlevel  level at or above which to log       */
/*                        whatmsg   the message to be logged              */
/*                                                                        */
/*                      Globals:                                          */
/*                        logfile    the name of the logfile              */
/*                        loglevel   logging level requested at startup   */
/*                        msgtag     tag to be used on each message       */
/*ͼ*/

      newmsg = time('L') || ' ' || msgtag;
      newmsg = newmsg || whatmsg;
      if whatlevel <= loglevel then do;
         msgcnt = lineout(logfile,newmsg);
         say newmsg;
      end;

return;  /* end of logit */

cleanup:
/*ͻ*/
/* (cleanup)            Ctrl-Break and Ctrl-C handler.                    */
/*                                                                        */
/*                      We come here to clean up if Ctrl-Break or Ctrl-C  */
/*                      was pressed.                                      */
/*                                                                        */
/*                      Input:                                            */
/*                        qname      the open queue                       */
/*ͼ*/

      logmsg = '(cleanup) Ctrl-Break or Ctrl-C was pressed; clean-up begins';
      call logit logmsg, 2;

      /* now write the sem file to let queue readers take over */

      semname = sempre || 'RXQSEM' || tag;
      cond = stream( semname, 'C', 'open write' ) ; /* open semfile */
      cond = lineout( semname, semname ) ;          /* write 1 line */
      cond = stream( semname, 'C', 'close' ) ;      /* close semfile */

      call didwepass;    /* did we pass or fail? */
      call ending;

exit;  /* end of cleanup */

ending:
/*ͻ*/
/* (ending)             End of the test                                   */
/*ͼ*/

      logmsg = 'EXIT RXQWRITQ STRESS TEST';
      call logit logmsg, 1;

return;  /* end of ending */

deleteq:
/*ͻ*/
/* (deleteq)            Delete a queue                                    */
/*                                                                        */
/*                      If a queue cannot be deleted by this routine,     */
/*                      it is all right if it is because the queue        */
/*                      name is not valid, or if the named queue does     */
/*                      not exist; but if it cannot be deleted for any    */
/*                      other reason, then fail the test.                 */
/*                                                                        */
/*                      Input:                                            */
/*                        qname      the name of the queue to delete      */
/*ͼ*/

       /* delete queue and log deletion results */

       dq = RXQUEUE('Delete',qname);
       if dq = 0 then do;
          logmsg = '(deleteq) Could not delete queue ' || qname;
          logmsg = logmsg || '; result is ' || dq || '; should be 0';
          if dq =  5    /* not a valid queue name */
           & dq =  9    /* named queue does not exist */
                then do;
            failcount = failcount + 1;
          end;
       end;
       else do;
          logmsg = '(deleteq) Deleted queue ' || qname;
       end;
       call logit logmsg, 4;

return;  /* end of deleteq */

