/* Author: Shaun Tancheff */
/* Copyright (C) Medtronic, Inc. All Rights Reserved. */

#define  INCL_DOSERRORS
#define  INCL_DOSMODULEMGR
#include <os2.h>
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include "DosQProc.h"

#define Kbytes       1024
#define BUFSZ        (128 * Kbytes)
#define INC_ALLPIDS  0

#ifndef True
#define True 1
#define False 0
#endif

static qsPrec_t * _psNextProcess(qsPrec_t * );
static qsFrec_t * ofh(qsPtrRec_t * pSysState, USHORT hSysFile );
static void       lsfn(qsPtrRec_t * pSysState, int bAll );
static qsLrec_t * mte(qsPtrRec_t * pSysState, USHORT hMte );
static int        isnfh(qsPtrRec_t * pSysState, ULONG pid, USHORT fh, PUSHORT usl, int ct );
static qsPrec_t * findpid(qsPtrRec_t * pSysState, PID pid );
static int        duplicate(USHORT fh, PUSHORT usl, int i);
static int        printfilerec( FILE * fp, char * s, qsFrec_t * pFile );
static void       named_ofh( qsPtrRec_t * pSysState, char * fn );
static void       lsof( void );


int main( int argc, char * argv[] )
{
   lsof();
   return 0;
}

/*
 * DosQuerySysState()
 *   - For each running process
 *     - report open files
 *   - List all remaining open files
 */
static void lsof( void )
{
   qsPtrRec_t * pSysState = NULL;

   pSysState = (qsPtrRec_t *) malloc( BUFSZ );
   if (pSysState) {
      APIRET rcApi;
      qsPrec_t * pProcess;
   
      rcApi = DosQuerySysState( QS_SUPPORTED, RESERVED, INC_ALLPIDS, RESERVED, pSysState, BUFSZ);

      if ( rcApi == NO_ERROR ) {
         fprintf( stdout, "PID\tPPID\t#OPEN\tPROCESS\n" );

         for ( pProcess = pSysState -> pProcRec;
                          pProcess -> RecType == 1;
                          pProcess = _psNextProcess( pProcess ) ) {
   
            APIRET apiReturnCode;
            UCHAR  pszNameBuf[1024]; /* Temporary Buffer */
            UCHAR  pszBuffer[1024];  /* Temporary Buffer */
      
            apiReturnCode = DosQueryModuleName(pProcess -> hMte, sizeof(pszNameBuf), pszNameBuf);
            if (apiReturnCode == 6) {
                /* DosQueryModuleName cannot find the winos2:session name.
                */
                qsLrec_t * lib;
                lib = mte( pSysState, pProcess->hMte );
                if (lib) {
                   strcpy( pszBuffer, "winos2" );
                }
            } else {
                int iLen;
                char * pszEnd;
      
                named_ofh( pSysState, pszNameBuf );

                pszEnd = strrchr( pszNameBuf, '.' ); /* right most . */
                if ( pszEnd ) *pszEnd = '\0';
                pszEnd = strrchr( pszNameBuf, '\\' ); /* right most \ */
                if ( pszEnd ) pszEnd++;
                else pszEnd = pszNameBuf;
                strcpy ( pszBuffer, strlwr( pszEnd ) );
            }
            fprintf( stdout, "%d\t%d\t#%d\t%s\n", pProcess->pid, pProcess->ppid, pProcess->cFH, pszBuffer );

            /* now list files opened by this process */
            if (pProcess->pFSRec == NULL) {
               /* Pre W4-FP13 */
               if (pProcess->pShrMemRec) {
                  pProcess->pFSRec = pProcess->pShrMemRec + pProcess->cShrMem;
               }
            }

            if (pProcess->pFSRec) {
               int file;

               for ( file = 0; file < pProcess->cFH; file++ ) {
#if DEBUG
                  fprintf( stdout, "\t(debug) sfh: %d\n", pProcess->pFSRec[file] );
#endif
                  if ( pProcess->pFSRec[file] != 0 && pProcess->pFSRec[file] != 0xffff ) {
                     if ( isnfh( pSysState, pProcess->ppid, pProcess->pFSRec[file], pProcess->pFSRec, file) ) {
                        qsFrec_t * pFile;

                        pFile = ofh( pSysState, pProcess->pFSRec[file] );
                        if (pFile) {
                          strlwr(pFile->name);
                          printfilerec( stdout, "\t", pFile );
                        }
                     }
                  } else {
                     /* My testing seems to indicate that once an invalid fh is encountered,
                        the rest of the fh left are not valid ... it isn't clear what the 
                        expected behaviour would be. */
                     break;
                  }
               }
            } else {
               fprintf( stdout, "\t??? null ???\n" );
            }
         }
         /* pFSRec */
         fprintf( stdout, "\n\n" );
         /*
          * Note: The 'executable' file that our process was launched from will be
          *       reported again. The SFN for our process isn't known, though we *could*
          *       do a lookup by name to make the output below slightly cleaner.
          */
         lsfn( pSysState, False );
      }
   }
}


/*
 * Scan process information for pid.
 */
static qsPrec_t * findpid( qsPtrRec_t * pSysState, PID pid )
{
   qsPrec_t * found = NULL;
   qsPrec_t * pProcess;

   for ( pProcess = pSysState -> pProcRec; pProcess -> RecType == 1; pProcess = _psNextProcess( pProcess ) ) { 
      if (pProcess->pid == pid ) {
         found = pProcess;
         break;
      }
   }
   return found;
}

/*
 *  Is this a 'new' file handle (ie not inherited from a parent process).
 *  Make output cleaner.
 *  Example:
 *     lsof > lsof.txt
 *  lsof.txt will be reported as open for cmd but not for lsof (it's child).
 *  also \dev\con won't be reported 3 times as open (stdin, stdout, stderr)
 */
static int isnfh(qsPtrRec_t * pSysState, ULONG pid, USHORT fh, PUSHORT usl, int ct )
{
   int dup = False;

   dup = duplicate(fh, usl, ct);
   if ( !dup ) {
      qsPrec_t * pProcess;
      
      do {
         pProcess = findpid( pSysState, pid );
         if ( pProcess ) {
            dup = duplicate( fh, pProcess->pFSRec, pProcess->cFH );
            if (dup) {
               break;
            }
            pid = pProcess->ppid;
         }
      } while ( pProcess );
   }

   return !dup;
}

/*
 * Detect duplicates in the pProcess->pFSRec list
 */
static int duplicate(USHORT fh, PUSHORT usl, int ct) 
{
   int i;
   int dup = False;

   for ( i = 0; i < ct; i++ ) {
      if (usl[i] != 0 && usl[i] != 0xffff) {
         if (fh == usl[i]) {
            dup = True;
            break;
         }
      } else {
         break;
      }
   }
   return dup;
}

/* 
 * List SFN 
 *   When this gets run time strlwr() has been called on all 
 *   the SFN (open files) assocated with every 'known' process.
 *   Set bAll to non-zero for the complete list of SFN or 
 *   to false for all the files 'not yet listed'.
 */
static void lsfn( qsPtrRec_t * pSysState, int bAll )
{
   qsFrec_t * found = NULL;
   qsFrec_t * files;
   qsSft_t  * finfo;
   files = pSysState -> pFSRec;

   fprintf( stdout, "SFN\t  SIZE   ATTRIB PATH\n" );
   while ( files && files->RecType == 8 ) {
      qsFrec_t * pFile;
      pFile = files;
      if (pFile) {
         int bShow;
         bShow = bAll;
         if ( !bShow ) {
            bShow = ( pFile->name[0] != '\\' ? isupper(pFile->name[0]) 
                                             : isupper(pFile->name[1]) );
         }
         if (bShow) {
            printfilerec( stdout, "", pFile );
         }
      }
      files = files -> pNextRec;
   }
}

/*
 * Common routine to print out file information.
 */
static int printfilerec( FILE * fp, char * s, qsFrec_t * pFile )
{
   int attrib;

   attrib = pFile->pSft->attr;

   return fprintf( fp, "%s%d\t%8d %c%c%c%c%c%c 0x%04x %s\n", 
                    s, pFile->pSft->sfn, pFile->pSft->size,
                        (attrib&0x20) ? 'a' : '-',
                        (attrib&0x10) ? 'd' : '-',
                        (attrib&0x08) ? 'l' : '-',
                        (attrib&0x04) ? 's' : '-',
                        (attrib&0x02) ? 'h' : '-',
                        (attrib&0x01) ? 'r' : '-',
                         pFile->pSft->mode, pFile->name );
 
}


/*
 * Scan all SFN returned from DosQuerySysState() for a match
 *   against hSysFile.
 */
static qsFrec_t * ofh( qsPtrRec_t * pSysState, USHORT hSysFile )
{
   qsFrec_t * found = NULL;
   qsFrec_t * files;
   qsSft_t  * finfo;
   files = pSysState -> pFSRec;
   while ( files && files->RecType == 8 ) {
      finfo = files -> pSft;

      if ( finfo->sfn == hSysFile ) {
         found = files;
         break;
      }
      files = files -> pNextRec;
   }
   return found;
}

/*
 * Find 'filename' and lowercase it.
 *  -- suppress redundant mentioning of 'process' file(s).
 */
static void named_ofh( qsPtrRec_t * pSysState, char * fn )
{
   qsFrec_t * files;
   files = pSysState -> pFSRec;
   while ( files && files->RecType == 8 ) {
      if ( stricmp(fn, files->name) == 0 ) {
         strlwr( files->name );
      }
      files = files -> pNextRec;
   }
}


/*
 * Scan the lib list for a matching mte.
 */
static qsLrec_t * mte(qsPtrRec_t * pSysState, USHORT hMte )
{
   qsLrec_t * found = NULL;
   qsLrec_t * libs;
   libs = pSysState -> pLibRec;

   while (libs) {
      if ( libs->hmte == hMte ) {
         found = libs;
         break;
      }
      libs = libs->pNextRec;
   }
   return found;
}


/*
 * Return address of the 'next' process
 *  - this is made difficult due to poor documentation but ...
 *    it appears that the 'last' set of arrays for a process record entry
 *    will be blocks of contiguous ThreadControlBlocks (TCB) that we have
 *    a pointer to the start of, and the number of TCBs associated with this
 *    process. So next_record = StartOfThreadControlBlocks+Count*SizeOfEachTCB
 */
static qsPrec_t * _psNextProcess( qsPrec_t * Record )
{
  char * pNext;

  pNext = (char *) Record -> pThrdRec;
  pNext += Record -> cTCB * sizeof(qsTrec_t);

  return (qsPrec_t *) pNext;
}

