/*
 *      HTEXT.CMD - HyperText/2 - V1.02 - C.Langanke 2001
 *
 *    Syntax: htext <input_file> [<ipf_file>] [<inf_or_hlp_file>] [rch_file]
 *                  [/Verbose] [/Noview] [/Hlp] [/Rch[:define_prefix]]
 *
 *    By default, HyperText/2 creates an IPF and INF or HLP file with the
 *    same name as the input file, but with extensions .ipf and .inf or hlp.
 *    Optionally you can specify the IPF and INF/HLP file to override the
 *    default names (and directories).
 *
 *    Note:
 *    - in order to specify the resulting INF or HLP file, you have also
 *      to specify the IPF file
 *    - in order to specify a resource header file, you have also to specify
 *      the IPF and INF/HLP file
 *    - if you specify the IPF file and no INF file, the INF file name
 *      will match the (path)name of the IPF file
 *    - If not specified otherwise, the target files are created in the
 *      directory of the sourcefile
 *    - if the INF file can be successfully generated, the INF file viewer
 *      is started to show it, unless the parameter /Noview is specified.
 */
/* The first comment is used as online help text */

 SIGNAL ON HALT NAME HALT

 TitleLine = STRIP(SUBSTR(SourceLine(2), 3));
 PARSE VAR TitleLine CmdName'.CMD 'Info
 Title = CmdName Info
 env       = 'OS2ENVIRONMENT';
 TRUE      = (1 = 1);
 FALSE     = (0 = 1);
 CrLf      = '0d0a'x;
 Redirection = '1>NUL 2>&1';
 '@ECHO OFF'
 rcx = SETLOCAL();

 /* some OS/2 Error codes */
 ERROR.NO_ERROR           =   0;
 ERROR.INVALID_FUNCTION   =   1;
 ERROR.FILE_NOT_FOUND     =   2;
 ERROR.PATH_NOT_FOUND     =   3;
 ERROR.ACCESS_DENIED      =   5;
 ERROR.NOT_ENOUGH_MEMORY  =   8;
 ERROR.INVALID_FORMAT     =  11;
 ERROR.INVALID_DATA       =  13;
 ERROR.NO_MORE_FILES      =  18;
 ERROR.WRITE_FAULT        =  29;
 ERROR.READ_FAULT         =  30;
 ERROR.SHARING_VIOLATION  =  32;
 ERROR.GEN_FAILURE        =  31;
 ERROR.INVALID_PARAMETER  =  87;
 ERROR.ENVVAR_NOT_FOUND   = 204;

 GlobalVars = 'Title CmdName env TRUE FALSE CrLf Redirection ERROR.';

 /* load RexxUtil */
 CALL RxFuncAdd    'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs';
 CALL SysLoadFuncs;

 /* some defaults */
 GlobalVars        = GlobalVars 'Flag. Valid. IpfSymbol. HtmlSymbol. Char. Pages. Bitmap. If. Launch.';
 rc                = ERROR.NO_ERROR;

 Flag.fHelp        = FALSE;
 Flag.fDebug       = FALSE;
 Flag.fVerbose     = FALSE;
 Flag.fNoView      = FALSE;
 Flag.fHelpFile    = FALSE;
 Flag.fRchFile     = FALSE;

 SourceFile        = '';
 TargetFile        = '';
 InfFile           = '';
 IpfDefaultOptions = '-c:850';
 RchFile           = '';
 RchDefineStem     = '';

 IPFC._Exec        = 'IPFC.EXE';
 IPFC._EnvVar      = 'IPFC';

 Valid._Colors     = 'DEFAULT BLACK BLUE RED PINK GREEN CYAN YELLOW BROWN DARKGRAY DARKBLUE DARKRED DARKPINK DARKGREEN DARKCYAN PALEGRAY';
 Valid._Fonts      = 'DEFAULT HELV TMS_RMN COURIER SYSTEM_MONOSPACED SYSTEM_PROPORTIONAL';

 /* used by MakeIPFLine */
 Char._EscapeChars = '*#_';
 Char._Attrs       = 'BOLD ITALIC UNDERLINED';
 Char._LastAttrs   = '';
 Char._LinkChars   = '[]';

 /* symbols for conversion may not include escape chars above ! */
 IpfSymbol._chars   = '&:'
 IpfSymbol._strings = '&amp. &colon.';

 Pages.             = '';
 Pages.0            = 0;
 Pages._MaxLevel    = 0;
 Pages._ClearLevels = 1;

 Bitmap.            = '';

 /* special launchname to allow patch of INF files */
 /* created by HyperText/2 through HTLAUNCH.EXE    */
 Launch._Exec       = 'nEtScApE.eXe';

 If._fIfOpen        = FALSE;
 If._fIfElse        = FALSE;
 If._fIncludeSource = FALSE;

 call RxFuncAdd    'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
 call SysLoadFuncs

 /* read commandline parameters */
 PARSE ARG Parms;
 DO i = 1 TO WORDS(Parms)
    ThisParm = WORD(Parms, i);
    PARSE VAR ThisParm ThisTag':'ThisValue;
    ThisTag = TRANSLATE( ThisTag);

    SELECT
       WHEN (POS(ThisParm, '/DEBUG') = 1) THEN
          Flag.fDebug = TRUE;

       WHEN (POS(ThisParm, '/VERBOSE') = 1) THEN
          Flag.fVerbose = TRUE;

       WHEN (POS(ThisParm, '/NOVIEW') = 1) THEN
          Flag.fNoView = TRUE;

       WHEN (POS(ThisParm, '/HLP') = 1) THEN
          Flag.fHelpFile = TRUE;

       WHEN (POS(ThisTag, '/RCH') = 1) THEN
       DO
          Flag.fRchFile = TRUE;
          IF (POS( ':', ThisParm) > 0) THEN
             RchDefineStem = ThisValue;
       END;

       WHEN (POS(ThisTag, '/?') = 1) THEN
       DO
          Flag.fHelp =TRUE;
       END;

       OTHERWISE
       DO
          SELECT
             WHEN (SourceFile = '') THEN
                 SourceFile = ThisParm;

             WHEN (TargetFile = '') THEN
                 TargetFile = ThisParm;

             WHEN (InfFile = '') THEN
                 InfFile = ThisParm;

             WHEN (RchFile = '') THEN
                 RchFile = ThisParm;

             OTHERWISE
             DO
                SAY 'error: invalid parameters.'
                SAY;
                'PAUSE'
                EXIT( ERROR.INVALID_PARAMETER);
             END;
          END;
       END;
    END;
 END;

 IF (Flag.fVerbose) THEN
    SAY;

 DO UNTIL (TRUE)

    /* display help */
    IF (Flag.fHelp) THEN
    DO
       rcx = SHowHelp();
       LEAVE;
    END;

    /* check for IPF compiler */
    IF (SysSearchPath( 'PATH', IPFC._Exec) = '') THEN
    DO
       SAY 'error: IPF compiler not found.';
       SAY;
       'PAUSE'
       rc = ERROR.FILE_NOT_FOUND;
       LEAVE;
    END;

    IF (VALUE( IPFC._EnvVar,,env) = '') THEN
    DO
       SAY 'error: IPF compiler is not properly installed.';
       SAY '       environment variable' IPFC._EnvVar 'is missing.';
       SAY;
       'PAUSE'
       rc = ERROR.ENVVAR_NOT_FOUND;
       LEAVE;
    END;

    /* check input file */
    IF (SourceFile = '') THEN
    DO
       SAY 'error: no  sourcefile given.';
       rc = ERROR.INVALID_PARAMETER;
       LEAVE;
    END;
    IF (\FileExist( SourceFile)) THEN
    DO
       SAY 'error:  sourcefile' SourceFile 'not found.';
       rc = ERROR.FILE_NOT_FOUND;
       LEAVE;
    END;

    /* determine out file and write header */
    IF (TargetFile = '') THEN
    DO
       BaseName = FILESPEC( 'N', SourceFile);
       TargetFile = ChangeExtension( Basename, '.ipf');
    END;

    IF (InfFile = '') THEN
    DO
       BaseName = FILESPEC( 'N', TargetFile);
       IF (Flag.fHelpFile) THEN
       DO
          IpfTypeOption = '';
          InfFile = ChangeExtension( Basename, '.hlp');
       END;
       ELSE
       DO
          IpfTypeOption = '/INF';
          InfFile = ChangeExtension( Basename, '.inf');
       END;
    END;
    ELSE
       IpfTypeOption = '/INF';

    IpfOptions = IpfDefaultOptions IpfTypeOption;

    IF (Flag.fRchFile) THEN
    DO
       IF (RchFile = '') THEN
          RchFile = ChangeExtension( Basename, '.rch');
    END;

    /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

    /* check for bitmaps */
    Bitmap._Netscape     = GetBitmap( 'ns.bmp');
    Bitmap._NetscapeMail = GetBitmap( 'nsmail.bmp');
    Bitmap._Infbook      = GetBitmap( 'book.bmp');

    /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

    List.0    = 0;
    rcProcess = ProcessSourceFile( SourceFile, TargetFile, 0);

    /* close files */
    rcx = STREAM( SourceFile, 'C', 'CLOSE');

    IF (rcProcess > 0) THEN
    DO
       rc = rcProcess;
       SAY 'aborting due to error, rc=' rc;
       LEAVE;
    END;

    /* determine used levels */
    DocLevels = '';
    DO i = 1 TO Pages._MaxLevel
       DocLevels = DocLevels''i;
    END;

    /* write header of target file */
    rcx = SysFileDelete( TargetFile);
    rcx = LINEOUT( TargetFile, ':userdoc.');
    rcx = LINEOUT( TargetFile, ':docprof toc='DocLevels'.');

    if (Pages._DocTitle \= '') THEN
       rcx = LINEOUT( TargetFile, ':title.'ReplaceLinks( MakeIPFLine( Pages._DocTitle)));

    /* go thru all pages resolving links, sublink lists and write data*/
    DO p = 1 TO Pages.0

       SourceWindowAttrs = '';

       /* - - - - - create sublink list - - - - - */

       /* sublink commands supported only in headers, not in footnotes ! */
       IF ((Pages.p._Level < 6) & (Pages.p._SublinkPos > 0)) THEN
       DO
          SublinkPercent = Pages.p._SublinkPercent

          /* check starting point of sublink tree */
          StartPage = p;
          SublinkAnchor = Pages.StartPage._SublinkAnchor;
          IF (SublinkAnchor \= '') THEN
          DO
             StartPage = GetPageFromAnchor( SublinkAnchor);
             IF (StartPage = 0) THEN
             DO
                SAY SourceFile'(***): anchor' SublinkAnchor 'for .SUBLINK command not found.';
                rc = ERROR.INVALID_DATA;
                EXIT(rc);
             END;
          END;

          /* determine dimensions of pages */
          IF (SublinkPercent \= '') THEN
             Pages.StartPage._Dimensions = ModifyDimensions( Pages.StartPage._Dimensions, 'SOURCE', SublinkPercent);

          /* check values for negative level offset */
          SELECT
             /* show all pages one level above */
             WHEN (Pages.p._SublinkLevels = -1) THEN
             DO
                ThisLevel = Pages.StartPage._Level - 1;
                SubLevel  = 1;
             END;

             OTHERWISE
             DO
                ThisLevel = Pages.StartPage._Level;
                SubLevel  = Pages.StartPage._SublinkLevels;
             END;
          END;

          /* go thru all subpages */
          LastLevel = ThisLevel;
          LinkList  = '';
          DownCount = 0;
          PARSE VAR Pages.StartPage._SublinkType ItemTag'|'StartTag'|'EndTag'|'DownTag'|'UpTag;
          DO s = StartPage + 1 TO Pages.0
             LinkStart = ':link reftype=hd viewport dependent refid='s'.';
             LinkEnd   = ':elink.';
             SELECT

                /* ignore footnotes with level 7 */
                WHEN (Pages.s._Level > 6) THEN
                   ITERATE;

                /* Only modify dimensions on windows of lower level than desired */
                WHEN ((SubLevel > 0) & (Pages.s._Level - ThisLevel > SubLevel)) THEN
                DO
                   Pages.s._Dimensions = ModifyDimensions( Pages.s._Dimensions, 'TARGET', SublinkPercent);
                   ITERATE;
                END;

                /* level is in range, so add to list if */
                /* step down in list indent when reachin lower level than before */
                WHEN (Pages.s._Level = ThisLevel + 1) THEN
                DO
                   IF (Pages.s._Level < LastLevel) THEN
                   DO
                      LinkList = LinkList''CrLf''UpTag''CrLf;
                      DownCount = DownCount - 1;
                   END;

                   IF (Pages.p._SublinkType \= '') THEN
                   DO
                      NewLink = ''CrLf''ItemTag''LinkStart''MakeIPFLine( Pages.s._Title)''LinkEnd;
                      LinkList = LinkList''NewLink;
                   END;

                END;

                /* level is in range, so add to list if */
                /* step up in list indent when reachin upper level than before */
                WHEN (Pages.s._Level > ThisLevel) THEN
                DO
                   IF (Pages.s._Level > LastLevel) THEN
                   DO
                      LinkList = LinkList''CrLf''DownTag;
                      DownCount = DownCount + 1;
                   END;

                   IF (Pages.p._SublinkType \= '') THEN
                   DO
                      NewLink = ''CrLf':li.'LinkStart''MakeIPFLine( Pages.s._Title)''LinkEnd;
                      LinkList = LinkList''NewLink;
                   END;

                END;

                OTHERWISE LEAVE;
             END;

             /* modify dimensions of subpage */
             Pages.s._Dimensions = ModifyDimensions( Pages.s._Dimensions, 'TARGET', SublinkPercent);

             LastLevel = Pages.s._Level;

             /* add link to list */
             IF (Pages.p._AutoLink = '') THEN
                Pages.p._AutoLink = ':link reftype=hd auto viewport dependent refid='s'.'
          END;

          /* append end tags for open lists */
          DO i = 1 TO DownCount
             LinkList = LinkList''CrLf''UpTag''CrLf;
          END;
          DownCount = 0;

          IF (LinkList \= '') THEN
          DO
             /* make list complete */
             LinkList = StartTag''LinkList''CrLf''EndTag;
             IF (LENGTH( Pages.p._Contents) = 0) THEN
                Pages.p._Contents = LinkList;
             ELSE
                Pages.p._Contents = INSERT( LinkList, Pages.p._Contents, Pages.p._SublinkPos);
          END;

       END;
    END;

    /* go thru all pages copying contents of pages */
    DO p = 1 TO Pages.0
       IF (Pages.p._MirrorAnchor \= '') THEN
       DO
          SourcePage = GetPageFromAnchor( Pages.p._MirrorAnchor);
          IF (SourcePage = 0) THEN
          DO
             SAY SourceFile'(***): anchor' Pages.p._MirrorAnchor 'for .MIRROR command not found.';
             rc = ERROR.INVALID_DATA;
             EXIT(rc);
          END;
          ELSE
          DO
             /* mirror contents to this page */
             Pages.p._Contents       = Pages.SourcePage._Contents;

             /* also copy sublink attributes */
             Pages.p._SublinkPercent = Pages.SourcePage._SublinkPercent;
             Pages.p._Dimensions     = Pages.SourcePage._Dimensions;
             Pages.p._AutoLink       = Pages.SourcePage._AutoLink;
          END;
       END;
    END;

    /* reset all attributes previously set */
    Char._LastAttrs   = '';

    /* write data*/
    DO p = 1 TO Pages.0

       /* invoke header creation trace */
       IF (Pages.p._HTrace) THEN
       DO
          SAY 'Trace for header' p':'Pages.p._Title;
          Trace ?i
       END;

       /* - - - - - - - - - - - - - - - - - - - - */

       /* determine parameters for header */
       HeaderAttributes = '';
       IpfTitle = ReplaceLinks( MakeIPFLine( Pages.p._Title));
       IF (\Flag.fHelpFile) THEN
       DO
          Dimensions = Pages.p._Dimensions;
          IF (Pages.p._Dimensions \= '') THEN
          DO
             PARSE VAR Pages.p._Dimensions _x _y _cx _cy;
             HeaderAttributes = HeaderAttributes 'x='_x'% y='_y'% width='_cx'% height='_cy'%';
          END;
       END;
       IF (Pages.p._Clear = 1) THEN
          HeaderAttributes = HeaderAttributes 'clear';
       IF (Pages.p._Resid \= '') THEN
          HeaderAttributes = HeaderAttributes 'res='Pages.p._Resid;

       /* write header or footnote */
       SELECT
          /* create footnote panel */
          WHEN (Pages.p._Level > 6) THEN
          DO
             rcx = LINEOUT( TargetFile, ':fn id='p'.');
             Pages.p._Contents = Pages.p._Contents''CrLf':efn.';
          END;

          /* create header panel */
          OTHERWISE
             rcx = LINEOUT( TargetFile, ':h'Pages.p._Level' id='p''HeaderAttributes'.'IpfTitle);
       END;

       /* write index data */
       IF (Pages.p._Index) THEN
          rcx = LINEOUT( TargetFile, ':i1.'IpfTitle);

       /* for INF file write autolink */
       IF (\Flag.fHelpFile) THEN
       DO
          IF ((Pages.p._SublinkPercent \= '') & (Pages.p._AutoLink \=  '')) THEN
             rcx = LINEOUT( TargetFile, Pages.p._AutoLink);
       END;

       /* replace links in contents */
       Pages.p._Contents = ReplaceLinks( Pages.p._Contents);

       /* write contents of page */
       rcx = LINEOUT( TargetFile, Pages.p._Contents);

    END;

    rcx = LINEOUT( TargetFile, ':euserdoc.');
    rcx = STREAM( TargetFile, 'C', 'CLOSE');

    /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

    /* write RCH file */
    IF (Flag.fRchFile) THEN
    DO
       BaseName = TRANSLATE( TRANSLATE( FILESPEC( 'N', RchFile)), '_', '.');
       rcx = SysFileDelete( RchFile);
       rcx = LINEOUT( RchFile, '');
       rcx = LINEOUT( RchFile, '/*' FILESPEC( 'N', RchFile) ' generated by HyperText/2 - ' DATE() TIME() '*/');
       rcx = LINEOUT( RchFile, '');
       rcx = LINEOUT( RchFile, '#ifndef' BaseName);
       rcx = LINEOUT( RchFile, '#define' BaseName);
       rcx = LINEOUT( RchFile, '');

       /* check all pages for resource ids */
       DO p = 1 TO Pages.0
          IF (Pages.p._Resid = '') THEN ITERATE;
          AnchorList = Pages.p._AnchorList;
          IF (AnchorList = '') THEN
          DO
          SAY
             SAY SourceFile'(***): warning: no anchor name specified for panel with resource id' Pages.p._Resid'.';
             ITERATE;
          END;
          DO WHILE (AnchorList \= '')
             PARSE VAR AnchorList ThisAnchorName AnchorList ;
             rcx = LINEOUT( RchFile, '#define' RchDefineStem''LEFT( ThisAnchorName, 40) RIGHT( Pages.p._Resid, 6));
          END;
       END;

       rcx = LINEOUT( RchFile, '');
       rcx = LINEOUT( RchFile, '#endif /*' BaseName '*/');
       rcx = LINEOUT( RchFile, '');
       rcx = STREAM( RchFile, 'C', 'CLOSE');
    END;

    /* - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

    /* now compile - handle strange rc of IPFC */
    IF (Flag.fVerbose) THEN
    DO
       SAY 'Compiling INF';
       SAY;
    END;

    TmpFile = SysTempFileName( VALUE('TMP',,env)'\htext.???');
    'IPFC' IpfOptions TargetFile InfFile '>' TmpFile

    IF ((rc \= ERROR.NO_ERROR) | (Flag.fVerbose)) THEN
       rcx = ShowIPFCOutput( TmpFile);
    rcx = SysFileDelete( TmpFile);

    IF ((rc = ERROR.NO_ERROR) & (\Flag.fNoView)) THEN
       'START VIEW' InfFile

 END;

 EXIT( rc);

/* ------------------------------------------------------------------------- */

HALT:
 SAY;
 SAY 'Interrupted by user.';
 EXIT(ERROR.GEN_FAILURE);

/* ------------------------------------------------------------------------- */
ShowHelp: PROCEDURE EXPOSE (GlobalVars)

 PARSE SOURCE . . ThisFile

 SAY;
 SAY Title;
 SAY;

 /* skip header */
 DO i = 1 TO 3
    rc = LINEIN(ThisFile);
 END;

 /* show help */
 DO WHILE (ThisLine \= ' */')
    ThisLine = LINEIN(Thisfile);
    SAY SUBSTR(ThisLine, 7);
 END;

 /* close file */
 rc = LINEOUT(Thisfile);

 RETURN('');

/* ------------------------------------------------------------------------- */
LASTWORD: PROCEDURE
 PARSE ARG String;
 IF (String = '') THEN
    RETURN( '');
 ELSE
    RETURN( WORD( String, WORDS( String)));

/* ------------------------------------------------------------------------- */
FileExist: PROCEDURE
 ARG FileName

 RETURN(STREAM(Filename, 'C', 'QUERY EXISTS') > '');

/* ------------------------------------------------------------------------- */
GetCalldir: PROCEDURE
PARSE SOURCE . . CallName
 CallDir = FILESPEC('Drive', CallName)||FILESPEC('Path', CallName);
 RETURN(LEFT(CallDir, LENGTH(CallDir) - 1));

/* ========================================================================= */
GetBitmap: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Filename, Align;

 FoundFilename = '';
 Link          = '';
 CallDir       = GetCallDir();

 DO UNTIL (TRUE)

    /* search given path */
    IF (FileExist( Filename)) THEN
    DO
       FoundFilename = FileName;
       LEAVE;
    END;

    /* now search files without path */
    Filename = FILESPEC( 'N', Filename);

    /* search in call dir and subdir */
    CheckName = CallDir'\'Filename;
    IF (FileExist( CheckName)) THEN
    DO
       FoundFilename = CheckName;
       LEAVE;
    END;
    CheckName = CallDir'\htext\'Filename;
    IF (FileExist( CheckName)) THEN
    DO
       FoundFilename = CheckName;
       LEAVE;
    END;

    /* search in environment vars */
    /* do not use SysSearchPath, it returns fully qualified pathnames */
    FoundFilename = SearchPath( 'INCLUDE', Filename);
    IF (FoundFilename \= '') THEN
       LEAVE;
    FoundFilename = SearchPath( 'HTINCLUDE', Filename);
    IF (FoundFilename \= '') THEN
       LEAVE;
 END;

 IF (Align = '') THEN
    Align = 'runin';
 ELSE
    Align = ' align='Align;


 IF (FoundFilename \= '') THEN
    Link = ":artwork name='"FoundFilename"' "Align".";

 RETURN( Link);

/* ========================================================================= */
DumpDimensions:
 PARSE ARG x y cx cy;
 RETURN( RIGHT( x, 3) RIGHT( y, 3) RIGHT( cx, 3) RIGHT( cy, 3));

/* ========================================================================= */
ModifyDimensions: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Dimensions, Type, Modifier;

 IF (Modifier = '') THEN
    RETURN( Dimensions);

 fSource = FALSE;
 fTarget = FALSE;

 IF (Dimensions = '') THEN
    Dimensions = '0 0 100 100';


 Type = TRANSLATE( Type);
 SELECT
    WHEN (POS( Type, 'SOURCE') = 1) THEN fSource = TRUE;
    WHEN (POS( Type, 'TARGET') = 1) THEN fTarget = TRUE;
    OTHERWISE RETURN Dimensions;
 END;

 PARSE VAR Dimensions x y cx cy;
 PARSE VAR Modifier Direction +1 Percentage;
 Direction = TRANSLATE( Direction);

/* CALL CHAROUT, Type ':' Modifier '-' DumpDimensions( Dimensions) ' -> '; */

 SELECT
    WHEN ( Direction = 'V') THEN
    DO
       IF (fSource) THEN
          cx = Percentage;

       IF (fTarget) THEN
       DO
          x  = x  + Percentage;
          cx = cx - Percentage;
       END;
    END;

    WHEN ( Direction = 'H') THEN
    DO
       IF (fSource) THEN
       DO
          IF (cy \= 100) THEN
             y =  cy - Percentage;
          ELSE
             y  = 100 + y - Percentage;
          cy = Percentage;
       END;

       IF (fTarget) THEN
       DO
          y  = 0;
          cy = cy - Percentage;
       END;
    END;
    OTHERWISE NOP;
 END;
 Dimensions = x y cx cy;

/* SAY DumpDimensions( Dimensions); */

 RETURN Dimensions;

/* ========================================================================= */
ChangeExtension: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Filename, NewExtension;

 BaseName   = FILESPEC( 'N', Filename);
 BaseExtPos = POS( '.', BaseName);
 IF (BaseExtPos = 0) THEN
    NewName = FileName''NewExtension;
 ELSE
    NewName = OVERLAY( NewExtension, FileName, LENGTH( Filename) - LENGTH( BaseName) + BaseExtPos);

 RETURN( NewName);

/* ========================================================================= */
MakeIPFLine: PROCEDURE EXPOSE (GlobalVars)
PARSE ARG OldStr;

 NewStr = '';
 fLinkOpen = FALSE;

 /* now check for IPF symbols */
 DO WHILE (OldStr \= '')
    PARSE VAR OldStr char +1 OldStr;
    NextChar = LEFT( OldStr, 1);

    /* ignore links */
    SELECT
       /* handle double bracket */
       WHEN ((char = '[') & (NextChar = '[')) THEN
       DO
          NewStr = NewStr''char;
          OldStr = SUBSTR( OldStr, 2);
       END;
       WHEN ((char = ']') & (NextChar = ']')) THEN
       DO
          NewStr = NewStr''char;
          OldStr = SUBSTR( OldStr, 2);
       END;

       WHEN (char = '[') THEN
       DO
          NewStr = NewStr''char;
          fLinkOpen = TRUE;
          ITERATE;
       END;
       WHEN (char = ']') THEN
       DO
          NewStr = NewStr''char;
          fLinkOpen = FALSE;
          ITERATE;
       END;
       WHEN (fLinkOpen) THEN
       DO
          NewStr = NewStr''char;
          ITERATE;
       END;

       OTHERWISE NOP;
    END;

    /* examine char */
    charIndex  = POS( char, IpfSymbol._chars);
    echarIndex = POS( char, Char._EscapeChars);
    SELECT
       WHEN (charIndex > 0) THEN
          NewStr = NewStr''WORD( IpfSymbol._strings, charIndex);

       WHEN ((echarIndex > 0) & (char = LEFT( OldStr, 1))) THEN
       DO
          /* skip char if given twice */
          NewStr = NewStr''char;
          OldStr = SUBSTR( OldStr, 2);
       END;

       WHEN (echarIndex > 0) THEN
       DO

          /* any attribute active ? - end it first */
          IF (Char._LastAttrs \= '') THEN
          DO
             IpfAttr = GetTextAttr( Char._LastAttrs);
             PARSE VAR IpfAttr StartAttr EndAttr;
             NewStr = NewStr''EndAttr;
          END;

          /* check for switched attribute */
          NewAttr = WORD( Char._Attrs, echarIndex);
          AttrPos = WORDPOS( NewAttr, Char._LastAttrs);
          IF (AttrPos = 0) THEN
             /* add attribute */
             Char._LastAttrs = Char._LastAttrs NewAttr;
          ELSE
             /* remove attribute */
             Char._LastAttrs = STRIP( DELWORD( Char._LastAttrs, AttrPos, 1));

          /* start with new attribute */
          IF (Char._LastAttrs \= '') THEN
          DO
             IpfAttr = GetTextAttr( Char._LastAttrs);
             PARSE VAR IpfAttr StartAttr EndAttr;
             NewStr = NewStr''StartAttr;
          END;

       END;

       OTHERWISE
          NewStr = NewStr''char;
    END;
 END;

 RETURN( NewStr);

/* ========================================================================= */
MakeUrlLine: PROCEDURE EXPOSE (GlobalVars)
PARSE ARG OldStr;

 NewStr = '';

 /* now check for IPF symbols */
 DO WHILE (OldStr \= '')
    PARSE VAR OldStr char +1 OldStr;
    DVal = C2D( char);
    /* examine char */
    IF ((DVal <= 32) | (DVal > 127)) THEN
       Char = '%'D2X( DVal);

    NewStr = NewStr''char;
 END;

 RETURN( NewStr);

/* ========================================================================= */
MakeExternalLink: PROCEDURE EXPOSE (GlobalVars)
PARSE ARG Link;

 fLinkTextCopied = FALSE;

 DO UNTIL (TRUE)

    /* is it an external link ? */
    PARSE VAR Link LinkTag':'LinkData;
    PARSE VAR LinkData LinkSource LinkText;
    LinkTag = TRANSLATE( LinkTag);
    SELECT

       WHEN ( WORDPOS( LinkTag, 'HTTP FTP MAILTO') > 0) THEN
       DO
          Program    = Launch._Exec;
          Bitmap     = Bitmap._Netscape;
          PARSE VAR LINK LinkSource .;

          /* special handling for mailto tags */
          IF (LinkTag = 'MAILTO') THEN
          DO
             Bitmap = Bitmap._NetscapeMail;
             IF (LinkText = '') THEN
                LinkText = LinkData;
                /* LinkSource = '-mail' LinkSource; */
          END;
          ELSE
          DO
             IF (LinkText = '') THEN
             DO
                LinkText = LinkSource;
                fLinkTextCopied = TRUE;
             END;
          END;
       END;

       WHEN ( WORDPOS( LinkTag, 'INF') > 0) THEN
       DO
          Program = 'view.exe';
          Bitmap = Bitmap._Infbook;

          PARSE VAR LinkData Link LinkData;
          IF (LinkData \= '') THEN
             LinkSource = LinkSource '"'LinkData'"';

          IF (LinkText = '') THEN
             LinkText = LinkData;
          IF (LinkText = '') THEN
             LinkText = Link;
       END;

       OTHERWISE LEAVE;
    END;

    /* take care for colons in linktext */
    /* do not call MakeIpfLine, it will mask the URL again ! */
    ColonPos = POS( ':', LinkText);
    IF (fLinkTextCopied) THEN
    DO
       IpfLink = '';
       DO WHILE (LinkText \= '')
          PARSE VAR LinkText ch +1 LinkText
          chIndex = POS( ch, IpfSymbol._chars);
          IF (chIndex = 0) THEN
             IpfLink = IpfLink''ch;
          ELSE
             IpfLink = IpfLink''WORD( IpfSymbol._strings, chIndex);
       END;
    END;
    ELSE
       IpfLink = MakeIPFLine( LinkText);

    /* create the program link */
    Link = Bitmap":link reftype=launch object='"Program"' data='"LinkSource"'."IpfLink":elink.";

 END;

 RETURN( Link);

/* ========================================================================= */
GetPageFromAnchor: PROCEDURE EXPOSE (GlobalVars)
PARSE ARG Anchor;

 Page = 0;

 DO UNTIL (TRUE)
    /* anchor name must start with period ! */
    IF (LEFT( Anchor, 1) \= '.') THEN
       LEAVE;

    /* prepare anchor name */
    PARSE VAR Anchor . +1 AnchorName;

    /* search link in pages */
    DO p = 1 TO Pages.0
       IF (WORDPOS( AnchorName, Pages.p._AnchorList) > 0) THEN
       DO
          Page = p;
          LEAVE;
       END;
    END;
 END;

 RETURN( Page);

/* ========================================================================= */
ReplaceLinks: PROCEDURE EXPOSE (GlobalVars)
PARSE ARG Str, fIgnoreAttributes;

 /* defaults */
 IF (fIgnoreAttributes = '') THEN
    fIgnoreAttributes = FALSE;

 /* check for links */
 LinkStart = POS( '[', Str);
 DO WHILE (LinkStart > 0)

    /* ignore, if escape char is given twice */
    IF (SUBSTR( Str, LinkStart + 1, 1) = '[') THEN
    DO
       LinkEnd  =  POS( ']]', Str, LinkStart);
       IF (LinkEnd > 0) THEN
       DO
          Str = DELSTR( Str, LinkStart, 1);
          Str = DELSTR( Str, LinkEnd, 1);
       END;
       LinkStart = POS( '[', Str, LinkStart + 2);
       ITERATE;
    END;

    /* check for end of link */
    LinkEnd =  POS( ']', Str, LinkStart);
    IF (LinkEnd = 0) THEN LEAVE;

    /* remove link from string */
    Link = SUBSTR( Str, LinkStart + 1, LinkEnd - LinkStart - 1);
    Str  = DELSTR( Str, LinkStart, LinkEnd - LinkStart + 1);

    CheckLink = TRANSLATE( Link);
    PARSE VAR CheckLink LinkTag':'.
    fLinkFound  = FALSE;
    fAnchorLink = FALSE;

    SELECT

       /* is it a text replacement only ? */
       WHEN (LEFT( Link, 1) = '=') THEN
       DO
          PARSE VAR Link . +1 ReplacementName;
          SELECT
             /* text defined by anchor */
             WHEN (LEFT( ReplacementName, 1) = '.') THEN
             DO
                LinkedPage = GetPageFromAnchor( ReplacementName);
                IF (LinkedPage > 0) THEN
                   Link = Pages.LinkedPage._Title;
             END;

             /* text defined by env var */
             OTHERWISE
                Link = VALUE( ReplacementName,,env);
          END;

          /* make it IPF conform */
          IF (\fIgnoreAttributes) THEN
             Link = MakeIpfLine( Link);
       END;


       /* is it an anchor link ? */
       WHEN (LEFT( Link, 1) = '.') THEN
       DO
          fAnchorLink = TRUE;
          PARSE VAR Link LinkAnchor Link;
          LinkedPage = GetPageFromAnchor( LinkAnchor);
          IF ((LinkedPage > 0) & (STRIP( Link) = '')) THEN
             Link  = Pages.LinkedPage._Title;
          Link = MakeIpfLine( Link)
          IF (LinkedPage > 0) THEN
             SELECT
                WHEN (Pages.LinkedPage._Level > 6) THEN
                   Link = ':link reftype=fn refid='LinkedPage'.'Link':elink.';
                OTHERWISE
                   Link = ':link reftype=hd viewport dependent refid='LinkedPage'.'Link':elink.';
             END;
          ELSE
          DO
             PARSE VAR LinkAnchor '.'LinkAnchorName;
             /* allow anchor links to refids outside the resulting ipf source */
             /* this makes sense, when the resulting IPF source is being embedded */
             Link = ':link reftype=hd viewport dependent refid='LinkAnchorName'.'Link':elink.';
          END;
       END;

       /* is it an external link ? */
       WHEN ( WORDPOS( LinkTag, 'HTTP FTP MAILTO INF') > 0) THEN
          Link = MakeExternalLink( Link);


       OTHERWISE
       DO
          /* search link in pages */
          DO p = 1 TO Pages.0
             fLinkFound = (TRANSLATE( Pages.p._Title) = CheckLink);

             IF (fLinkFound) THEN
             DO
                SELECT
                   WHEN (Pages.p._Level > 6) THEN
                      Link = ':link reftype=fn id='p'.'Link':elink.';
                   OTHERWISE
                      Link = ':link reftype=hd viewport dependent refid='p'.'Link':elink.';
                END;
                LEAVE;
             END;
          END;
       END;

    END; /* SELECT */

    /* add link to string */
    Str = INSERT( Link, Str, LinkStart - 1);

    /* search next link */
    LinkStart = POS( '[', Str, LinkStart + LENGTH( Link));
 END;


 RETURN( Str);


/* ========================================================================= */
GetTextAttr: PROCEDURE EXPOSE (GlobalVars)
PARSE ARG Parms;

 fUseBold       = FALSE;
 fUseItalic     = FALSE;
 fUseUnderlined = FALSE;
 fUseHilite     = FALSE;
 TextAttr       = '';

 NewParms = '';
 DO WHILE (Parms \= '')
    PARSE VAR Parms ThisParm Parms;
    ThisParm = TRANSLATE( ThisParm);
    SELECT
       WHEN (POS( ThisParm, 'BOLD')       = 1) THEN fUseBold       = TRUE;
       WHEN (POS( ThisParm, 'ITALIC')     = 1) THEN fUseItalic     = TRUE;
       WHEN (POS( ThisParm, 'UNDERLINED') = 1) THEN fUseUnderlined = TRUE;
       WHEN (POS( ThisParm, 'HILITE')     = 1) THEN fUseHilite     = TRUE;
       OTHERWISE NOP;
    END;
 END;
 SELECT
    WHEN (fUseHilite) THEN
       TextAttr = ':hp8. :ehp8.';
    WHEN ((fUseItalic) & (fUseUnderlined)) THEN
       TextAttr = ':hp6. :ehp6.';
    WHEN ((fUseBold) & (fUseUnderlined)) THEN
       TextAttr = ':hp7. :ehp7.';
    WHEN ((fUseBold) & (fUseItalic)) THEN
       TextAttr = ':hp3. :ehp3.';
    WHEN (fUseBold) THEN
       TextAttr = ':hp2. :ehp2.';
    WHEN (fUseItalic) THEN
       TextAttr = ':hp1. :ehp1.';
    WHEN (fUseUnderlined)  THEN
       TextAttr = ':hp5. :ehp5.';
    OTHERWISE NOP;
 END;

 RETURN( TextAttr);

/* ========================================================================= */
GetFontAttr: PROCEDURE EXPOSE (GlobalVars)
PARSE ARG Parms;

 FontAttrs       = '';
 ForegroundColor = '';
 BackgroundColor = '';
 FontName        = '';
 FontSize        = '';

 QuoteChar       = "'";

 DO UNTIL (TRUE)
    DO WHILE (Parms \= '')
       PARSE VAR Parms ThisParm Parms;
       PARSE VAR ThisParm ThisTag'='ThisParm;
       ThisTag = TRANSLATE( ThisTag);
       ThisParm = STRIP( ThisParm);
       SELECT
          WHEN ( ThisTag = 'FC') THEN
             ForegroundColor = ThisParm;

          WHEN ( ThisTag = 'BC') THEN
             BackgroundColor = ThisParm;

          WHEN ( POS( ThisTag, 'FONT') = 1) THEN
          DO
             /* take care for quotes */
             IF (LEFT( ThisParm, 1) = QuoteChar) THEN
             DO
                /* both quotes in this parm ? */
                ThisParm = SUBSTR( ThisParm, 2);
                IF (RIGHT( ThisParm, 1) = QuoteChar) THEN
                   PARSE VAR ThisParm ThisParm"'";
                ELSE
                DO
                   /* quote in rest of string */
                   QuotePos = POS( QuoteChar, Parms);
                   IF (QuotePos = 0) THEN
                   DO
                      ThisParm = ThisParm Parms;
                      Parms    = '';
                   END;
                   ELSE
                   DO
                      ThisParm = ThisParm LEFT( Parms, QuotePos - 1);
                      Parms    = SUBSTR( Parms, QuotePos + 1);
                   END;
                END;
             END;

             PARSE VAR ThisParm FontSize'.'FontName;
             CheckFontName = TRANSLATE( TRANSLATE( FontName), '_', ' ');
          END;

          OTHERWISE NOP;
       END;
    END;

    /* validate values */
    /* return error message instead of font attributes ... */
    SELECT
       WHEN ((ForegroundColor \= '') & (WORDPOS( TRANSLATE( ForegroundColor), Valid._Colors) = 0)) THEN
          FontAttrs = 'invalid foreground color' ForegroundColor 'specified.';

       WHEN ((BackgroundColor \= '') & (WORDPOS( TRANSLATE( BackgroundColor), Valid._Colors) = 0)) THEN
          FontAttrs = 'invalid background color' BackgroundColor 'specified.';

       WHEN (TRANSLATE( FontSize) = 'DEFAULT') THEN NOP;

       WHEN ((FontName \= '') & (WORDPOS( CheckFontName, Valid._Fonts) = 0)) THEN
          FontAttrs = 'invalid bitmap font' Fontname 'specified.';

       WHEN ((FontSize \= '') & (DATATYPE( FontSize) \= 'NUM')) THEN
          FontAttrs = 'font size specified is not numeric.';

       OTHERWISE NOP;
    END;
    IF( FontAttrs \= '') THEN
    DO
       FontAttrs = 'error:' FontAttrs;
       LEAVE;
    END;

    /* concatenate values */
    SELECT
       WHEN ((ForegroundColor \= '')  & (BackgroundColor \= '')) THEN
          FontAttrs = FontAttrs':color fc='ForegroundColor' bc='BackgroundColor'.';
       WHEN (ForegroundColor \= '') THEN
          FontAttrs = FontAttrs':color fc='ForegroundColor'.';
       WHEN (BackgroundColor \= '') THEN
          FontAttrs = FontAttrs':color bc='BackgroundColor'.';
       OTHERWISE NOP;
    END;
    SELECT
       WHEN (TRANSLATE( FontSize) = 'DEFAULT') THEN
          FontAttrs = FontAttrs':font.'
       WHEN (FontName \= '') THEN
          FontAttrs = FontAttrs":font facename='"FontName"' size="FontSize"x"FontSize".";
       OTHERWISE NOP;
    END;

 END;

 RETURN( FontAttrs);

/* ========================================================================= */
DiscardTextAttrs: PROCEDURE
 PARSE ARG Parms

 NewParms = '';
 DO WHILE (Parms \= '')
    PARSE VAR Parms ThisParm Parms;
    CheckParm = TRANSLATE( ThisParm);
    SELECT
       WHEN (POS( CheckParm, 'BOLD')       = 1) THEN NOP;
       WHEN (POS( CheckParm, 'ITALIC')     = 1) THEN NOP;
       WHEN (POS( CheckParm, 'UNDERLINED') = 1) THEN NOP;
       WHEN (POS( CheckParm, 'HILITE')     = 1) THEN NOP;
       OTHERWISE NewParms = NewParms ThisParm;
    END;
 END;
 RETURN( NewParms);


/* ========================================================================= */
/* clone of SysSearchPath, which does not return fully qualified name ! */
SearchPath: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG EnvVar, Filename;

 FoundFilename = '';

 DO UNTIL (TRUE)
    EnvPath = VALUE( EnvVar,,env);

    DO WHILE (EnvPath \= '')
       PARSE VAR EnvPath ThisDir';'EnvPath;
       ThisFile = ThisDir'\'Filename;
       IF (FileExist( ThisFile)) THEN
       DO
          FoundFilename = ThisFile;
          LEAVE;
       END;
    END;
 END;

 RETURN( FoundFilename);

/* ========================================================================= */
SearchIncludeFile: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG Filename;

 FoundFilename = '';
 Filename = STRIP( Filename);

 DO UNTIL (TRUE)
    /* check filename as is */
    IF (FileExist( Filename)) THEN
    DO
       FoundFilename = Filename;
       LEAVE;
    END;

    /* if drive letter given, we cannot search further */
    IF (POS( ':',  Filename) > 1) THEN
       LEAVE;

    /* search in environment vars */
    /* do not use SysSearchPath, it returns fully qualified pathnames */
    FoundFilename = SearchPath( 'INCLUDE', Filename);
    IF (FoundFilename \= '') THEN
       LEAVE;
    FoundFilename = SearchPath( 'HTINCLUDE', Filename);
    IF (FoundFilename \= '') THEN
       LEAVE;

 END;

 RETURN( FoundFilename);

/* ========================================================================= */
EvaluateCondition: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG CompileCondition

 Condition.0 = 0;

 DO WHILE (CompileCondition \= '')
    PARSE VAR CompileCondition ThisAndCondition'&'CompileCondition;
    DO WHILE (ThisAndCondition \= '')
       PARSE VAR ThisAndCondition ThisOrCondition'|'ThisAndCondition;
       c           = Condition.0 + 1;
       Condition.c = STRIP( ThisOrCondition);
       Condition.0 = c;
       IF (ThisAndCondition = '') THEN
          Condition.c._Operator = '&';
       ELSE
          Condition.c._Operator = '|';
    END;
 END;


 /* check all conditions and reconcatenate */
 NewCondition = '';
 LastOperator = '';
 DO c = 1 TO Condition.0
    PARSE VAR Condition.c SymbolName'='SymbolValue;
    SymbolName  = TRANSLATE( STRIP( SymbolName));
    SymbolValue = STRIP( SymbolValue);
    IF (SymbolValue = '') THEN
       /* just return symbol value */
       Condition.c._Value = VALUE( SymbolName,,env);
    ELSE
       Condition.c._Value = (VALUE( SymbolName,,env) = SymbolValue);

    NewCondition = NewCondition LastOperator Condition.c._Value;
    LastOperator = Condition.c._Operator;
 END;

 RETURN( NewCondition);

/* ========================================================================= */
ProcessSourceFile: PROCEDURE EXPOSE (GlobalVars) List. PList.
 PARSE ARG SourceFile, TargetFile, IncludeLevel;

 LineCount = 0;
 rc        = ERROR.NO_ERROR;
 fTrace    = FALSE;

 ListAttrOrdered     = ':li.|:ol compact.|:eol.|:ol compact.|:eol.';
 ListAttrUnordered   = ':li.|:ul compact.|:eul.|:ul compact.|:eul.';
 ListAttrSimple      = ':li.|:sl compact.|:esl.|:sl compact.|:esl.';
 ListAttrBreaks      = '.br'CrLf'|'CrLf'|.br|:ul compact.|:eul.';
 ListAttrNoList      = '';

    IF (Flag.fVerbose) THEN
       SAY COPIES( ' ', IncludeLevel * 2)'-' SourceFile;
    IncludeLevel = IncludeLevel + 1;
    rcx = STREAM( SourceFile, 'C', 'OPEN READ');
    DO WHILE (LINES( SourceFile));

       /* read line */
       LineCount = LineCount + 1;
       ThisLine = LINEIN( SourceFile);

       /* filter out EXTPROC command setting */
       IF (LineCount = 1) THEN
          IF (TRANSLATE( WORD( ThisLine, 1)) = 'EXTPROC') THEN
             ITERATE;

       /* skip comments */
       IF (LEFT( ThisLine, 2) = '..') THEN ITERATE;

       /* is trace activated ? */
       IF (fTrace) THEN
          TRACE ?i

       PARSE VAR ThisLine Command CommandParms;

       /* resolve text replacement in command parms first */
       CommandParms = ReplaceLinks( CommandParms, TRUE);

       /* - - - - - - - - - - - - - */

       x = List.0;
       IF (LEFT( ThisLine, 1) = '.') THEN
       DO
          Command = TRANSLATE( Command);
          CommandName = SUBSTR( Command, 2);
          fIsHeader = (DATATYPE( CommandName) = 'NUM');

          SELECT

             /* handle ifdef */
             WHEN (POS( '.IF', Command) = 1) THEN
             DO
                IF (If._fIfOpen) THEN
                DO
                   SAY SourceFile'('LineCount'): error: nested .IF/.IFDEF/.IFNDEF invalid.';
                   rc = ERROR.FILE_NOT_FOUND;
                   ITERATE;
                END;

                /* determine condition */
                ConditionResult = EvaluateCondition( CommandParms);
                /* SAY '>' CommandParms '->' ConditionResult */
                SELECT
                   WHEN (Command = '.IFDEF')  THEN If._fIncludeSource = (ConditionResult \= '');
                   WHEN (Command = '.IFNDEF') THEN If._fIncludeSource = (ConditionResult  = '');
                   WHEN (Command = '.IF')     THEN INTERPRET( 'If._fIncludeSource = (('ConditionResult') == TRUE);');
                   OTHERWISE
                   DO
                      SAY SourceFile'('LineCount'): error: invalid command' Command'.';
                      rc = ERROR.FILE_NOT_FOUND;
                      ITERATE;
                   END;
                END
                If._fIfOpen        = TRUE;
                If._fIfElse        = FALSE;
                ITERATE;
             END;

             /* handle else */
             WHEN ( Command = '.ELSE') THEN
             DO
                IF ((\If._fIfOpen) | (If._fIfElse)) THEN
                DO
                   SAY SourceFile'('LineCount'): error: .ELSE invalid.';
                   rc = ERROR.FILE_NOT_FOUND;
                   ITERATE;
                END;
                If._fIncludeSource = \If._fIncludeSource;
                If._fIfElse        = TRUE;
                ITERATE;
             END;

             /* handle endif */
             WHEN ( Command = '.ENDIF') THEN
             DO
                IF (\If._fIfOpen) THEN
                DO
                   SAY SourceFile'('LineCount'): error: .ENDIF invalid.';
                   rc = ERROR.FILE_NOT_FOUND;
                   ITERATE;
                END;
                If._fIfOpen        = FALSE;
                If._fIfElse        = FALSE;
                If._fIncludeSource = FALSE;
                ITERATE;
             END;

             /* skip section with wrong conditions */
             /* do that after checking for .IF .ELSE .ENDIF !! */
             WHEN ((If._fIfOpen) & (\If._fIncludeSource)) THEN ITERATE;

             /* gibe line breaks */
             WHEN (Command = '.') THEN
             DO
                ThisLine = '.br'
             END;

             /* handle headers */
             WHEN (((fIsHeader) & (CommandName < 7)) | (Command = '.FN')) THEN
             DO

                p                       = Pages.0 + 1;
                /* take care for footnotes */
                IF (DATATYPE( CommandName) \= 'NUM') THEN
                   Pages.p._Level = 7;
                ELSE
                DO
                   /* check for a panel title */
                   IF (STRIP( CommandParms) = '') THEN
                   DO
                      SAY SourceFile'('LineCount'): error: header directive has no title text.';
                      rc = ERROR.INVALID_FORMAT;
                   END;

                   Pages._MaxLevel         = MAX( Pages._MaxLevel, CommandName);
                   Pages.p._Level          = CommandName;
                   Pages.p._Title          = CommandParms;
                END;
                Pages.p._Contents       = '';
                Pages.p._MirrorAnchor     = '';
                Pages.p._Index          = (POS( Pages.p._Level, Pages._IndexLevels) > 0);
                Pages.p._Clear          = (POS( Pages.p._Level, Pages._ClearLevels) > 0);
                Pages.p._SublinkPos     = 0;
                Pages.p._Dimensions     = '';
                Pages.p._SublinkPercent = '';
                Pages.p._SublinkLevels  = 0;
                Pages.p._AnchorList     = '';
                Pages.p._Resid          = '';
                Pages.p._HTrace         = FALSE;
                Pages.0                 = p;

                /* reset previous text attributes */
                Char._LastAttrs = '';
                ITERATE;
             END;

             /* no more command with less than two letters from here */
             WHEN (LENGTH( Command) < 3) THEN
             DO
                SAY SourceFile'('LineCount'): error: invalid command' Command'.';
                rc = ERROR.FILE_NOT_FOUND;
             END;

             /* handle setting of variables */
             WHEN (POS( Command, '.SET') = 1) THEN
             DO
                /* check parameters */
                PARSE VAR CommandParms SymbolName'='SymbolValue;
                SymbolName  = STRIP( SymbolName);
                SymbolValue = STRIP( SymbolValue);
                IF ((SymbolName = '') | (SymbolValue = '')) THEN
                DO
                   SAY SourceFile'('LineCount'): error: invalid SET command.';
                   rc = ERROR.FILE_NOT_FOUND;
                END;
                ELSE
                   rcx = VALUE( SymbolName, ReplaceLinks( SymbolValue), env);
                ITERATE;
             END;

             /* handle titles */
             WHEN (POS( Command, '.TITLE') = 1) THEN
             DO
                Pages._DocTitle = CommandParms;
             END;

             /* handle bitmaps */
             WHEN (POS( Command, '.BITMAP') = 1) THEN
             DO
                IF (STRIP( CommandParms) = '') THEN
                DO
                   SAY SourceFile'('LineCount'): error: no bitmap specified.';
                   rc = ERROR.FILE_NOT_FOUND;
                END;
                ELSE
                DO
                   /* get list of bitmaps and align attribute */
                   Align      = '';
                   BitmapList = '';
                   ThisLine   = '';
                   DO WHILE (CommandParms \= '')
                      PARSE VAR CommandParms ThisParm CommandParms;
                      SELECT
                         WHEN (WORDPOS( TRANSLATE( ThisParm), 'LEFT RIGHT CENTER') > 0)
                            THEN Align = ThisParm
                         OTHERWISE
                            BitmapList = BitmapList ThisParm;
                      END;
                   END;

                   /* get all bitmaps */
                   DO WHILE (BitmapList \= '')
                      PARSE VAR BitmapList ThisBitmap BitmapList;
                      ThisLine = ThisLine''GetBitmap( ThisBitmap, Align);
                   END;

                END;
             END;

             /* handle index */
             WHEN (POS( Command, '.INDEX') = 1) THEN
             DO
                IF (STRIP( CommandParms) = '') THEN
                DO
                   p = Pages.0;
                   Pages.p._Index = TRUE;
                   ITERATE;
                END;
                ELSE
                   ThisLine = ':i1.'MakeIPFLine( CommandParms);
             END;

             /* handle automatic header index */
             WHEN (POS( Command, '.HINDEX') = 1) THEN
             DO
                Pages._IndexLevels = CommandParms;
                ITERATE;
             END;

             /* handle automatic header clearance */
             WHEN (POS( Command, '.HCLEAR') = 1) THEN
             DO
                Pages._ClearLevels = CommandParms;
                ITERATE;
             END;

             /* handle left margin */
             WHEN (POS( Command, '.LMARGIN') = 1) THEN
             DO
                IF (STRIP( CommandParms) = '') THEN
                   Margin = 1;
                ELSE
                DO
                   IF (DATATYPE( CommandParms) \= 'NUM') THEN
                   DO
                      SAY SourceFile'('LineCount'): error: value for left margin not numeric.';
                      rc = ERROR.FILE_NOT_FOUND;
                   END;
                   Margin = CommandParms
                END;
                ThisLine = ':lm margin='Margin'.';
             END;

             /* handle right margin */
             WHEN (POS( Command, '.RMARGIN') = 1) THEN
             DO
                IF (STRIP( CommandParms) = '') THEN
                   Margin = 1;
                ELSE
                DO
                   IF (DATATYPE( CommandParms) \= 'NUM') THEN
                   DO
                      SAY SourceFile'('LineCount'): error: value for right margin not numeric.';
                      rc = ERROR.FILE_NOT_FOUND;
                   END;
                   Margin = CommandParms
                END;
                ThisLine = ':rm margin='Margin'.';
             END;

             /* handle external resource ids*/
             WHEN (POS( Command, '.RESID') = 1) THEN
             DO
                IF (Pages.p._Resid \= '') THEN
                DO
                   SAY SourceFile'('LineCount'): error: resource id already specified.';
                   rc = ERROR.FILE_NOT_FOUND;
                END;
                IF (DATATYPE( CommandParms) \= 'NUM') THEN
                DO
                   SAY SourceFile'('LineCount'): error: value for resourceid not numeric.';
                   rc = ERROR.FILE_NOT_FOUND;
                END;
                Pages.p._Resid = CommandParms;

                ITERATE;
             END;

             /* handle anchor names */
             WHEN (POS( Command, '.ANCHOR') = 1) THEN
             DO
                AnchorName = TRANSLATE( CommandParms);
                SELECT
                   WHEN (STRIP( CommandParms) = '') THEN
                   DO
                      SAY SourceFile'('LineCount'): error: no anchor name specified.';
                      rc = ERROR.FILE_NOT_FOUND;
                   END;

                   WHEN (WORDS( CommandParms) \= 1) THEN
                   DO
                      SAY SourceFile'('LineCount'): error: anchorname contains blanks.'
                      rc = ERROR.FILE_NOT_FOUND;
                   END;

                   WHEN (WORDPOS( AnchorName, Pages._AnchorList) > 0) THEN
                   DO
                      SAY SourceFile'('LineCount'): error: anchorname' CommandParms 'already exists.';
                      rc = ERROR.FILE_NOT_FOUND;
                   END;

                   OTHERWISE
                   DO
                      /* store anchor name */
                      p = Pages.0;
                      Pages.p._AnchorList = Pages.p._AnchorList AnchorName;
                      Pages._AnchorList   = Pages._AnchorList AnchorName;
                   END;
                END;
                ITERATE;
             END;

             /* copy contens of other section */
             WHEN (POS( Command, '.MIRROR') = 1) THEN
             DO
                AnchorName = TRANSLATE( CommandParms);
                SELECT
                   WHEN (STRIP( CommandParms) = '') THEN
                   DO
                      SAY SourceFile'('LineCount'): error: no anchor name specified.';
                      rc = ERROR.FILE_NOT_FOUND;
                   END;

                   WHEN (WORDS( CommandParms) \= 1) THEN
                   DO
                      SAY SourceFile'('LineCount'): error: anchorname contains blanks.'
                      rc = ERROR.FILE_NOT_FOUND;
                   END;

                   OTHERWISE
                   DO
                      /* store anchor name */
                      p = Pages.0;
                      Pages.p._MirrorAnchor = '.'AnchorName;
                   END;
                END;
                ITERATE;
             END;

             /* handle attributes */
             WHEN (POS( Command, '.ATTRIBUTE') = 1) THEN
             DO
                ThisLine = '';

                /* cleanup old highlight attributes */
                NewIpfAttr = GetTextAttr( CommandParms);
                IF ((NewIpfAttr \= '') | (CommandParms = '')) THEN
                DO
                   /* any attribute active ? - end it first */
                   IF (Char._LastAttrs \= '') THEN
                   DO
                      IpfAttr = GetTextAttr( Char._LastAttrs);
                      PARSE VAR IpfAttr StartAttr EndAttr;
                      ThisLine = EndAttr;
                      Char._LastAttrs = '';
                   END;
                END;

                /* take care for color and font attributes */
                IF (CommandParms = '') THEN
                   ThisLine = ThisLine':font.:color fc=default bc=default.';
                ELSE
                DO
                   /* new font attributes ? */
                   FontAttrs = GetFontAttr( CommandParms);
                   SELECT
                      WHEN (WORD( FontAttrs, 1) = 'error:') THEN
                      DO
                         SAY SourceFile'('LineCount'):' FontAttrs;
                         rc = ERROR.INVALID_DATA;
                         LEAVE;
                      END;
                      WHEN (FontAttrs \= '') THEN
                         ThisLine = ThisLine''FontAttrs;

                      OTHERWISE NOP;
                   END;
                END;

                IF (NewIpfAttr \= '') THEN
                DO
                   /* now set new hilight attributes */
                   PARSE VAR NewIpfAttr StartAttr EndAttr;
                   ThisLine = ThisLine''StartAttr;
                   Char._LastAttrs = CommandParms;
                END;

             END;

             /* handle xmp */
             WHEN (POS( Command, '.FORMAT') = 1) THEN
             DO
                CommandParms = TRANSLATE( STRIP( CommandParms));
                SELECT
                   WHEN (CommandParms = 'OFF')  THEN ThisLine = ':xmp.';
                   WHEN (CommandParms = 'ON')   THEN ThisLine = ':exmp.';
                   OTHERWISE
                   DO
                      SAY SourceFile'('LineCount'): error: invalid parameter for command .FORMAT';
                      rc = ERROR.INVALID_DATA;
                      LEAVE;
                   END;
                END;
             END;

             /* handle trace option */
             WHEN (POS( Command, '.TRACE') = 1) THEN
             DO
                fTrace = TRUE;
                ITERATE;
             END;
             /* handle trace option */
             WHEN (POS( Command, '.HTRACE') = 1) THEN
             DO
                Pages.p._HTrace = TRUE;
                ITERATE;
             END;

             /* handle sublink list */
             WHEN (POS( Command, '.SUBLINK') = 1) THEN
             DO
                p = Pages.0;
                IF (Pages.p._SublinkPos > 0) THEN
                DO
                   SAY SourceFile'('LineCount'): error: .SUBLINK command used more than once within same page.';
                   rc = ERROR.INVALID_DATA;
                   LEAVE;
                END;
                Pages.p._Contents = Pages.p._Contents''CrLf;
                Pages.p._SublinkPos   = LENGTH( Pages.p._Contents);
                Pages.p._SublinkType  = ListAttrUnordered;
                Pages.p._SublinkPercent = '';
                Pages.p._SublinkAnchor  = '';

                /* are attributes valid ? */

                DO WHILE (CommandParms \= '')
                   PARSE VAR CommandParms ThisParm CommandParms;
                   ThisParm = TRANSLATE( ThisParm);
                   SELECT
                      WHEN (POS( LEFT( ThisParm, 1), 'HV') > 0) THEN
                      DO
                        IF (Pages.p._SublinkAnchor \= '') THEN
                        DO
                           SAY SourceFile'('LineCount'): error: .SUBLINK parameters for both anchor name and split windows specified.';
                           rc = ERROR.INVALID_DATA;
                           LEAVE;
                        END;
                         PARSE VAR ThisParm . +1 Percentage;
                         IF (DATATYPE( Percentage) \= 'NUM') THEN
                         DO
                            SAY SourceFile'('LineCount'): error: invalid attribute for .SUBLINK command: no numeric percentage specified';
                            rc = ERROR.INVALID_DATA;
                            ITERATE;
                         END;
                         IF ((Percentage < 20) | (Percentage > 80)) THEN
                         DO
                            SAY SourceFile'('LineCount'): error: invalid attribute for .SUBLINK command: percentage must be between 20 and 80.';
                            rc = ERROR.INVALID_DATA;
                            ITERATE;
                         END;

                         Pages.p._SublinkPercent = ThisParm;
                      END;

                      WHEN (DATATYPE( ThisParm) = 'NUM') THEN
                      DO
                         Pages.p._SublinkLevels = ThisParm;
                      END;

                      WHEN (LEFT( ThisParm, 1) = '.') THEN
                      DO
                        IF (Pages.p._SublinkPercent \= '') THEN
                        DO
                           SAY SourceFile'('LineCount'): error: .SUBLINK parameters for both anchor name and split windows specified.';
                           rc = ERROR.INVALID_DATA;
                           LEAVE;
                        END;
                         Pages.p._SublinkAnchor = ThisParm;
                      END;

                      WHEN (POS( ThisParm, 'CLEAR') = 1) THEN
                         Pages.p._Clear = 1;

                      WHEN (POS( ThisParm, 'ORDERED') = 1) THEN
                         Pages.p._SublinkType = ListAttrOrdered;

                      WHEN (POS( ThisParm, 'UNORDERED') = 1) THEN
                         Pages.p._SublinkType = ListAttrUnordered;

                      WHEN (POS( ThisParm, 'SIMPLE') = 1) THEN
                         Pages.p._SublinkType = ListAttrSimple;

                      WHEN (POS( ThisParm, 'BREAKS') = 1) THEN
                         Pages.p._SublinkType = ListAttrBreaks;

                      WHEN (POS( ThisParm, 'NOLIST') = 1) THEN
                         Pages.p._SublinkType = ListAttrNoList;

                      OTHERWISE
                      DO
                         SAY SourceFile'('LineCount'): error: invalid attribute for .SUBLINK command.';
                         rc = ERROR.INVALID_DATA;
                      END;
                   END;

                END;

                ITERATE;
             END;

             /* handle lists */
             WHEN ((POS( Command, '.ULIST') = 1) | ,
                   (POS( Command, '.OLIST') = 1) | ,
                   (POS( Command, '.SLIST') = 1) | ,
                   (POS( Command, '.PLIST') = 1)) THEN
             DO
                x             = List.0 + 1;
                List.x._Type  = TRANSLATE( SUBSTR( Command, 2, 1), 'uosp', 'UOSP');
                List.0        = x

                /* save current pos as first for this list */
                IF (x = 1) THEN
                   List.x._Pos = 1
                ELSE
                   List.x._Pos = '';

                SELECT
                   WHEN (List.x._Type = 'p') THEN
                   DO
                      List.x._TAttr  = GetTextAttr( CommandParms);
                      List.x._Attr  = DiscardTextAttrs(CommandParms);
                      List.x._Start = ':parml' List.x._Attr'.';
                      List.x._End   = ':eparml.';
                   END;

                   OTHERWISE
                   DO
                      List.x._Attr  = CommandParms;
                      List.x._Start = ':'List.x._Type'l' List.x._Attr'.';
                      List.x._End   = ':e'List.x._Type'l.';
                   END;
                END;

                ThisLine      = List.x._Start;
             END;

             /* handle lists */
             WHEN (POS( Command, '.ELIST') = 1) THEN
             DO
                IF (List.0 = 0) THEN
                DO
                   SAY SourceFile'('LineCount'): error: list not open.';
                   rc = ERROR.FILE_NOT_FOUND;
                END;
                ELSE
                DO
                   /* close ipf list */
                   ThisLine = List.x._End''CrLf'.br';

                   /* delete last level or delete list entry */
                   IF (WORDS( List.x._Pos) > 1) THEN
                      List.x._Pos = DELWORD( List.x._Pos, WORDS( List.x._Pos));
                   ELSE
                      List.0 = List.0 - 1;
                END;
             END;


             /* handle titles */
             WHEN (POS( Command, '.INCLUDE') = 1) THEN
             DO
                IncludeFile = SearchIncludeFile( CommandParms);
                IF (IncludeFile = '') THEN
                DO
                   SAY SourceFile'('LineCount'): error: include file' CommandParms 'not found.';
                   rc = ERROR.FILE_NOT_FOUND;
                END;
                ELSE
                DO
                   rcInclude = ProcessSourceFile( IncludeFile, TargetFile, IncludeLevel);
                   IF (rcInclude > 0) THEN
                      rc = rcInclude;
                   ITERATE;
                END;
             END;


             OTHERWISE NOP;
          END;

       END;
       ELSE

       /* skip section with wrong conditions */
       IF ((If._fIfOpen) & (\If._fIncludeSource)) THEN
          ITERATE;
       ELSE
       /* - - - - - - - - - - - - - */

       IF (ThisLine = '') THEN
       DO
          IF (List.0 > 0) THEN
          DO
             /* close open lists*/
             p = Pages.0;
             DO x = List.0 TO 1 BY -1
                DO i = 1 TO WORDS( List.x._Pos)
                   Pages.p._Contents = Pages.p._Contents''CrLf''List.x._End;
                END;
             END;
             DROP( List.);
             List.0 = 0;
          END;

          /* use empty lines to separate paragraphs */
          ThisLine = '';
          IF (Pages.0 > 0) THEN
             ThisLine = ':p.';

       END;
       ELSE

       /* - - - - - - - - - - - - - */

       IF (List.0 > 0) THEN
       DO
          /* check for list items */
          FirstWord = WORD( ThisLine, 1);
          NewList._Pos = WORDINDEX( ThisLine, 1);
          ThisLine = STRIP( ThisLine)

          SELECT
             WHEN (List.x._Type = 'p') THEN
                fIsItem = (WORDPOS( FirstWord, '- =') > 0);

             OTHERWISE
                fIsItem = (FirstWord = '-');
          END;

          IF (fIsItem) THEN
          DO

             /* is this the current list level ? */
             /* list type may change !!! */
             CurrentList._Pos = LASTWORD( List.x._Pos);

             SELECT
                /* new list - do nothing */
                WHEN (CurrentList._Pos = '') THEN
                   List.x._Pos = NewList._Pos;

                /* go one level deeper */
                WHEN (NewList._Pos > CurrentList._Pos) THEN
                DO
                   List.x._Pos   = List.x._Pos NewList._Pos;
                   p = Pages.0;
                   Pages.p._Contents = Pages.p._Contents''CrLf''List.x._Start;
                END;

                /* keep same level */
                WHEN (NewList._Pos = CurrentList._Pos) THEN
                   NOP;

                /* go up again */
                WHEN (NewList._Pos < CurrentList._Pos) THEN
                DO
                   TodoList = REVERSE( List.x._Pos);
                   DO WHILE (TodoList \= '')
                      PARSE VAR TodoList ThisReversePos TodoList;
                      ThisPos = REVERSE( ThisReversePos);
                      IF (NewList._Pos < ThisPos) THEN
                      DO
                         p = Pages.0;
                         Pages.p._Contents = Pages.p._Contents''CrLf''List.x._End;

                         /* if lowest level of this list reached, close it */
                         IF (TodoList = '') THEN
                         DO
                            x      = List.0 - 1;
                            List.0 = x;
                         END;
                      END;
                      ELSE
                         /* level reached */
                      DO
                         List.x._Pos  =  REVERSE( ThisPos TodoList);
                         TodoList = '';
                      END;
                   END;
                END;
             END;

             /* prepare line */
             SELECT
                WHEN (List.x._Type = 'p') THEN
                DO
                   SELECT
                      WHEN (FirstWord = '-') THEN
                      DO
                         PARSE VAR List.x._TAttr AttrStart AttrEnd;
                         ThisLine = ':pt.'AttrStart''MakeIPFLine( SUBSTR( STRIP( ThisLine), 3))''AttrEnd;
                      END;

                      WHEN (FirstWord = '=') THEN
                      DO
                         ThisLine = ':pd.'MakeIPFLine( SUBSTR( STRIP( ThisLine), 3));
                      END;

                      OTHERWISE
                         ThisLine = MakeIPFLine( STRIP( ThisLine));
                   END;
                END;

                OTHERWISE
                   ThisLine = ':li.'MakeIPFLine( SUBSTR( ThisLine, 3));
             END;

          END;
          ELSE
          DO
             ThisLine = MakeIPFLine( ThisLine);

             /* handle dot as first character */
             IF (LEFT( ThisLine, 1) = '.') THEN
                ThisLine = ' 'ThisLine;
          END;
       END;
       ELSE
          ThisLine = MakeIPFLine( ThisLine);

       /* add line */
       p = Pages.0;
       Pages.p._Contents = Pages.p._Contents''CrLf''ThisLine

    END;

 /* close source file, otherwise we might run out of handles */
 rcx = STREAM( SourceFile, 'C', 'CLOSE');
 RETURN( rc);

/* ========================================================================= */
ShowIPFCOutput: PROCEDURE EXPOSE (GlobalVars)
 PARSE ARG OutputFile;

 fShowOutput = FALSE;

 DO WHILE (LINES( OutputFile) > 0)
    ThisLine = LINEIN( OutputFile);

    SELECT
       WHEN (ThisLine = '')                           THEN NOP;
       WHEN (LEFT( ThisLine, 1) = '.')                THEN fShowOutput = TRUE;
       WHEN (POS( 'Statistics :', ThisLine) = 1)      THEN fShowOutput = FALSE;
       WHEN (POS( 'Document title :', ThisLine) = 1)  THEN
       DO
          SAY;
          SAY ThisLine;
       END;
       WHEN (POS( '(INF)', ThisLine) > 0)             THEN SAY ThisLine;

       WHEN (LEFT( ThisLine, 1) = '<') THEN
       DO
          PARSE VAR ThisLine '<'ThisFile':'ThisLine'>'ThisInfo;
          SAY ThisFile'('ThisLine'):'ThisInfo;
       END;

       WHEN ( fShowOutput)                          THEN SAY ThisLine;
       OTHERWISE NOP;
    END;
 END;
 rcx = STREAM( OutputFile, 'C', 'CLOSE');

 RETURN( ERROR.NO_ERROR);

