/*****************************************************************************/
/* Module: database_client.c                                                 */
/*                                                                           */
/* Description:                                                              */
/*    This module obtains and sets the binding information registered in     */
/*    the name service. After setting the binding handle, this module calls  */
/*    the database_access() that provides an online interface allowing you   */
/*    to access the database managed by the database_server.exe.             */
/*                                                                           */
/*****************************************************************************/
#ifdef _WINDOWS
#include <dce/dcewin.h>
#endif
#include "db.h"
#include "errorchk.h"
#include <stdio.h>
#include <stdlib.h>
#ifdef _WINDOWS
#include <dce/pthreadx.h>
#else
#include <pthread_exc.h>
#endif

void database_access( db_context_t );

#define MAX_CALL_REQUESTS 3

int main( int argc, char *argv[] )
{
   unsigned32 st;
   rpc_ns_handle_t       import_context;
   rpc_binding_handle_t  binding_h;
   db_context_t          context_h;

#ifndef _WINDOWS
   sigset_t              sigset;
   pthread_t             this_thread = pthread_self();
#endif

#ifdef IBMOS2
   pthread_inst_exception_handler();
#endif

   rpc_ns_binding_import_begin(
      rpc_c_ns_syntax_default,         /* use default syntax                 */
      ENTRY_NAME,                      /* defined in IDL file                */
      Database_v1_0_c_ifspec,          /* ifspec generated by IDL compiler   */
      NULL,                            /* not working with object UUID       */
      &import_context,                 /* import context obtained            */
      &st );
   ERRORCK( "rpc_ss_binding_import_begin", st );

   while ( 1 ) {
      rpc_ns_binding_import_next(
         import_context,
         &binding_h,
         &st);
      ERRORCK( "rpc_ss_binding_import_next", st );

      if ( binding_h == NULL ) exit( 1 );

      db_open( binding_h, &context_h, &st );
                                       /* Obtain the context handle          */
      if ( st == rpc_s_ok )
         break;
      else
         rpc_binding_free( &binding_h, &st );
   }

#ifndef _WINDOWS
   sigemptyset ( &sigset );
   sigaddset ( &sigset, SIGINT );
   sigaddset ( &sigset, SIGTERM );
   pthread_signal_to_cancel_np ( &sigset, &this_thread );
#endif

   TRY
      if ( context_h != NULL ) {
         printf( "Context handle: %08X\n\n", context_h );
         database_access( context_h );
      } else
         printf( "   <Error: %d>\n", st );
   FINALLY
      db_close( &context_h, &st );
      do {
         rpc_binding_free( &binding_h, &st );
         rpc_ns_binding_import_next(
            import_context,            /* using this import context,         */
            &binding_h,                /* import the first binding handle    */
            &st);
      } while( binding_h != NULL && st == rpc_s_ok );

      rpc_ns_binding_import_done(
         &import_context,              /* free the import context            */
         &st );

#ifdef IBMOS2
      pthread_dinst_exception_handler();
#endif
      exit( 0 );
   ENDTRY
}

