/************************************************************************/
/**                        IBM LAN Server                              **/
/**                 Copyright(c) IBM Corp., 1995                       **/
/************************************************************************/
/*
 * Name:  VNETDLS.CMD
 *
 * Description:
 *        VNETDLS.CMD installs the DOS LAN Services Windows network drivers
 *        and user interface on an OS/2 LAN Requester for use with the
 *        Virtual DOS LAN API device driver (VNETAPI.OS2).  Once installed,
 *        this configuration enables network functions within each WIN-OS/2
 *        session without the use of the DOS LAN Services redirector.  The
 *        program files are installed to a drive:\path specified by the
 *        user and the AUTOEXEC.BAT on the boot drive is modified, updating
 *        the PATH= statement to include the installation path.
 *
 * Dependencies:
 *        Requires PKUNZIP2.EXE somewhere in the OS/2 PATH environment
 *        variable.  Since this command file is executed after OS/2 LAN
 *        Requester is installed, we can safely assume that PKUNZIP2.EXE is
 *        in the d:\IBMLAN\INSTALL directory.  Also, error and
 *        informational messages are read from the NET.MSG/NETH.MSG files
 *        in the d:\IBMLAN\NETPROG directory.  This command file is
 *        installed when the user installs Virtual DOS LAN API Support
 *        with the OS/2 LAN Services Installation/Configuration program.
 *
 * Error checking:
 *        - checks that VNETAPI.OS2 is installed
 *        - checks that WIN-OS/2 is installed
 *        - checks that PKUNZIP2.EXE is installed
 *        - renames NETAPI.DLL and PMSPL.DLL in the Windows directory
 *          to NETAPI.$LL and PMSPL.$LL, respectively.  This ensures
 *          that the NETAPI.DLL and PMSPL.DLL in the target directory
 *          are used by DLSNET.DRV instead of the versions shipped
 *          with Windows.
 */

call RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
call SysLoadFuncs

alphabet = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ'

/* LAN Server messages */
message_file = 'NET.MSG'
help_file = 'NETH.MSG'

/* OS/2 system messages */
osmsg_file = 'OSO001.MSG'
oshelp_file = 'OSO001H.MSG'

/* See if the user supplied a target installation path as
 * a command-line parameter.  If not, we prompt them for
 * the path.
 */
parse upper arg target workdrive .

/* User may want syntax help, so we check for '?', '/?', and 'HELP'.
 */
if (target = '?' | target = '/?' | target = 'HELP' | target = '/HELP')
   then signal syntax

/* ---------------------Error checking--------------------------- */
/* We check if WIN-OS/2, VNETAPI.OS2, and PKUNZIP2.EXE are installed.
 * The installation process fails if any one is not present.
 */

/* Is WIN-OS/2 installed? */
fspec = SysSearchPath('PATH','WINOS2.COM')
if (fspec = '') then signal no_winos2
else
do
  /* Rename NETAPI.DLL and PMSPL.DLL, if necessary.
   */
  netapi = filespec("drive",fspec)||filespec("path",fspec)||'SYSTEM\NETAPI.DLL'
  pmspl = filespec("drive",fspec)||filespec("path",fspec)||'SYSTEM\PMSPL.DLL'

  call SysFileTree netapi, 'file', 'F'
  if (file.0 \= 0) then
  do
     /* If NETAPI.DLL exists, rename it to NETAPI.$LL
      */
     '@rename' netapi 'NETAPI.$LL'
  end

  call SysFileTree pmspl, 'file', 'F'
  if (file.0 \= 0) then
  do
     /* If PMSPL.DLL exists, rename it to PMSPL.$LL
      */
     '@rename' pmspl 'PMSPL.$LL'
  end
end

/* Is VNETAPI installed? */
fspec = SysSearchPath('PATH','VNETAPI.OS2')
if (fspec = '') then signal no_vnetapi
else
do
  /* The program files are installed, but is it
   * loaded in the CONFIG.SYS?
   */
  fspec = BootDrive() || ':\CONFIG.SYS'
  if (fspec \= '') then
  do
    call SysFileSearch 'VNETAPI.OS2', fspec, 'stem.'
    if stem.0 = 0 then signal no_vnetapi
  end
end

/* Is PKUNZIP2.EXE installed? */
fspec = SysSearchPath('PATH','PKUNZIP2.EXE')
if (fspec = '') then signal no_pkunzip2


/* ---------------------Verify Target Path--------------------------- */
/* User may have supplied a command-line parameter for the
 * target path. If not, then we must prompt for a target.
 */
check_target:
if target = '' then
do
   /* "Type the target path and Enter when ready."
    */
   say SysGetMessage(2610, message_file )
   pull target

   /* If they didn't supply a target, then display help and exit.
    */
   if target = '' then signal check_target
end

/* If the user did not specify a subdirectory for the target path
 * (such as 'C:\'), then we display an error.
 */
if (translate(right(target,1)) \= '\') then
   target = target || '\'

path = filespec('path', target )
/* If the path is not valid then display an error
 */
if ( length(path) = 1 ) then signal error_help

/* Strip off the backslash we just added.
 */
target = strip(target,'T','\')

/* Calling MakeDirectory ensures the target directory is
 * a valid DOS directory and creates it, if necessary.
 */
rc = MakeDirectory( target )
if ( rc \= 0 ) then
do
   /* If error from MakeDirectory, display message and exit. */
   say SysGetMessage( rc, osmsg_file )
   exit
end

/* ---------------------Verify Install Drive------------------------- */
/* If the user supplied a workdrive, then we try to use it.
 */
check_wrkdrv:
if ( workdrive \= '' ) then
do
   /* Append a colon (':') to the workdrive, if necessary.
    */
   if ( length(workdrive) = 1 ) then
      workdrive = workdrive || ':'
   else
   do
     /* If string is >= 2 bytes, and the second byte is not a
      * colon (':'), then signal a syntax error.
      */
     if ( substr(workdrive,2,1) \= ':' ) then
        signal syntax
   end

   if ( length(workdrive) = 3 ) then
   do
     /* If the user specifies something like 'A:\', where the drive
      * letter is followed by a backslash, we need to remove the
      * backslash.  Otherwise, we will think that they're providing
      * an absolute path.
      */
     if ( substr(workdrive,3,1) = '\' ) then
        workdrive = substr(workdrive,1,2)
   end

   /* Verify that workdrive is a legal drive.  If not, then
    * prompt the user for a valid drive letter.
    */
   drive = translate(left(workdrive,1))
   if (verify(drive,alphabet) <> 0) then
   do
      workdrive = ''
      signal check_wrkdrv
   end

   /* Determine, as best we can, what type of drive is
    * the install (source) drive.  It could be either a
    * diskette drive, for which we'll prompt for the DLS
    * diskettes, or a fixed disk on which WARP Server has
    * been preloaded, or a CD-ROM drive.
    */
   drivetype = GetDriveType( substr(workdrive,1,2) )

   select
     when (drivetype = 0) then do             /* Drive is fixed */
       /* If not installing via diskette, then the install drive can
        * include both a drive and path.
        */
       media = 'fixed'

     end
     when (drivetype = 1) then do             /* Drive is diskette */
       media = 'diskette'

       /* All we want is the drive letter for the diskette drive.  So
        * we cut off any remaining path the user may have supplied.
        */
       if ( length(workdrive) > 2 ) then
          workdrive = substr(workdrive,1,2)

     end
     when (drivetype = 2) then do             /* Drive is removeable */

       /* The default is 'fixed,' but if it's A: or B:, then we
        * assume the install drive is a diskette drive.
        */
       media = 'fixed'

       /* If the user specifies something like 'A:\DIR1', then we
        * assume they mean just 'A:'.  This assumption is made only
        * if they specify drives 'A:' or 'B:' as part of the path.
        */
       if ( length(workdrive) >= 2 ) then
       do
          worktemp = substr(workdrive,1,2)
          if (worktemp  = 'A:' | worktemp = 'B:') then
          do
             workdrive = worktemp
             media = 'diskette'
          end
       end
     end
   end  /* select */
end  /* workdrive \= '' */
/* If user didn't supply a drive letter for the install drive,
 * we prompt them to enter one.  This includes both diskette and
 * "fixed" installations done via a hard drive, or CD-ROM.
 */
else if ( workdrive = '' ) then
do
   /* Type the drive and path of the DOS LAN Services program files,
    * and select Enter when ready.
    */
   say SysGetMessage(2611, message_file)
   pull workdrive
   signal check_wrkdrv
end

/* ------------------ Unpack the DLS program files --------------------- */
if (media = 'diskette') then
do
   /* Insert DOS LAN Services Disk 1 in workdrive and select Enter when ready.
   */
   msg = SysGetMessage(2609, message_file, '1', substr(workdrive,1,2) )
   say ''
   say msg
   pull ready

   /* Make sure the user gave us Disk 1...
    */
   call SysFileTree workdrive||'\NETWORK.INI', 'file', 'F'
   if ( file.0 = 0 ) then
   do
     /* Insert DOS LAN Services Disk 1 in workdrive and select Enter when ready.
     */
     msg = SysGetMessage(2609, message_file, '1', substr(workdrive,1,2) )
     file.0 = 0
     do while (file.0 = 0)
       say ''
       say msg
       pull ready

       /* Have to determine if this is the correct diskette.
        */
       call SysFileTree workdrive||'\NETWORK.INI', 'file', 'F'
     end
   end

   /* Determine what diskette the DLSWIN.ZIP and DLSMSG.ZIP
    * files are located on.
    */
   setup_inf = workdrive||'\SETUP.INF'

   '@copy' workdrive||'\NETWORK.INI' target
   do while lines(setup_inf) > 0
     line = linein(setup_inf)
     queue line
   end

   do queued()
      parse pull line
      /* Does this line point to either file?  If it does, then
       * store the diskette number.
       */
      if ( translate(substr(line,4,10)) = 'DLSWIN.ZIP' ) then
         diskette_win = translate(left(line,1))
      if ( translate(substr(line,4,10)) = 'DLSMSG.ZIP' ) then
         diskette_msg = translate(left(line,1))
   end

   /* Now we're ready to copy the files ...
    */
   dlswin = workdrive||'\DLSWIN.ZIP'

   if (diskette_win \= '1') then
   do
     /* Insert DOS LAN Services Disk' diskette_win,
      * in' workdrive 'and select Enter when ready.'
      */
     msg = SysGetMessage(2609, message_file, diskette_win, substr(workdrive,1,2) )
     file.0 = 0
     do while (file.0 = 0)
       say ''
       say msg
       pull ready

       /* Have to determine if this is the correct diskette.
        */
       call SysFileTree dlswin, 'file', 'F'
     end
   end

   /* Unpack the files from the DLSWIN.ZIP file */
   call UnPack_DLSWIN  dlswin, target

   dlsmsg = workdrive||'\DLSMSG.ZIP'

   /* Insert DOS LAN Services Disk' diskette_msg,
    * in' workdrive 'and select Enter when ready.'
    */
   msg = SysGetMessage(2609, message_file, diskette_msg, substr(workdrive,1,2) )
   file.0 = 0
   do while (file.0 = 0)
     say ''
     say msg
     pull ready

     /* Have to determine if this is the correct diskette.
      */
     call SysFileTree dlsmsg, 'file', 'F'
   end

   /* Unpack the files from the DLSMSG.ZIP file */
   call UnPack_DLSMSG  dlsmsg, target

end
else   /* media = 'fixed' */
do
   setup_inf = workdrive||'\SETUP.INF'

   /* Look for SETUP.INF on the specified source drive.  This
    * gives us a good indication that the DLS files are on the
    * same path.
    */
   call SysFileTree setup_inf, 'file', 'FSO'
   if (file.0 \= 0) then
   do
      /* Find DLSWIN.ZIP on the install drive.
       */
      dlswin = workdrive||'\DLSWIN.ZIP'
      call SysFileTree dlswin, 'file', 'FSO'
      if (file.0 \= 0) then
      do
         dlswin = file.1

         /* Unpack the required files from DLSWIN.ZIP. */
         call UnPack_DLSWIN  dlswin, target
      end
      else
      do
         /* "DLSWIN.ZIP file not found" */
         say SysGetMessage(1533, osmsg_file, dlswin )
         say SysGetMessage(1533, oshelp_file )
         exit
      end

      /* Find DLSMSG.ZIP on the install drive.
       */
      dlsmsg = workdrive||'\DLSMSG.ZIP'
      call SysFileTree dlsmsg, 'file', 'FSO'
      if (file.0 \= 0) then
      do
         dlsmsg = file.1

         /* Unpack the required files from DLSMSG.ZIP. */
         call UnPack_DLSMSG  dlsmsg, target
      end
      else
      do
         /* "DLSMSG.ZIP file not found" */
         say SysGetMessage(1533, osmsg_file, dlsmsg )
         say SysGetMessage(1533, oshelp_file )
         exit
      end
   end
   else
   do
      /* If we got to here then we think we're in the right place
       * on the diskette (or hard disk, CD-ROM, etc.), but we couldn't
       * find SETUP.INF.
       */
      /* "SETUP.INF file not found" */
      say SysGetMessage(1533, osmsg_file, setup_inf )
      say SysGetMessage(1533, oshelp_file )
      exit
   end
end    /* if( media = 'fixed' ) */


/* ----------------------Update the AUTOEXEC.BAT----------------------- */

autoexec = BootDrive() || ":\AUTOEXEC.BAT"
autotemp = BootDrive() || ":\AUTOEXEC.TMP"

/* First we check if the install path is already in the PATH.
 * If it is, then we do no work.
 */
automod = 0                     /* Reset flag */

'@copy' autoexec autotemp       /* Copy AUTOEXEC.BAT to AUTOEXEC.TMP */
do while lines(autotemp) > 0
  line = linein(autotemp)
  queue line
end
call lineout autotemp           /* Close AUTOEXEC.TMP */
'@del' autotemp                 /* Now delete AUTOEXEC.TMP, since we will
                                 * recreate it later after we modify it.
                                 */

/* We're now going to rebuild AUTOEXEC.TMP.  If we
 * encounter a 'PATH=' or 'PATH ' statement, append
 * the target path to it before writing it.
 */
do queued()
  parse pull line
  part = translate(left(line,5))
  if ( part = 'PATH ' | part = 'PATH=' ) then
  do
    /* We found a PATH= statement.  We append the install
     * directory only if it's not already in the path.
     */
    target = target || ';'
    found = POS(target, line)
    if ( found = 0 ) then
    do
      if (translate(right(line,1)) = ';') then      /* Is last char a semicolon? */
         call lineout autotemp, line||target
      else
         call lineout autotemp, line||';'||target

      /* Update flag to say that we modified AUTOEXEC.BAT
       */
      automod = 1
    end
    else
      call lineout autotemp, line
  end
  else
    call lineout autotemp, line
end
call lineout autotemp           /* Close AUTOEXEC.TMP */

/* Make AUTOEXEC.TMP the new AUTOEXEC.BAT
 */
autobak = BootDrive() || ':\AUTOEXEC.BAK'
'@copy' autoexec autobak
'@del' autoexec
'@rename' autotemp 'AUTOEXEC.BAT'

/* Start a Full-Screen DOS session, load Windows and execute DLSSETUP.EXE,
 * then exit the DOS session.
 */
if (translate(right(target,1)) = ';') then  /* Is last char a semicolon? */
   /* Strip off the semicolon */
   target = strip(target,'T',';')

'start /fs/dos/c win 'target'\dlssetup.exe /silent'

/* ----------------------------Messages----------------------------------
 */
call SysCls
/* Installation has successfully completed.
 */
say ''
say SysGetMessage(2602, message_file)
if (automod = 1) then
do
  /* Changes were made to' autoexec'. The previous version
   * was renamed to' autobak'.
   */
  say ''
  say SysGetMessage(2603, message_file, autoexec, autobak )
end
exit

error_help:
/* You did not supply a drive/path target for the installation.
 * Example: VNETAPI C:\NET
 */
say SysGetMessage(2604, message_file)
say ''
say SysGetMessage(3515, message_file, 2604)
exit

syntax:
/* Syntax:  VNETAPI drive:\path [install_drive]
 */
say SysGetMessage(2605, message_file)
say SysGetMessage(2608, message_file)
exit

no_winos2:
/* WIN-OS/2 is not installed.
 */
say SysGetMessage(2606, message_file)
say ''
say SysGetMessage(3515, message_file, 2606)
exit

no_vnetapi:
/* Virtual DOS LAN API support is not installed.
 */
say SysGetMessage(2607, message_file)
say ''
say SysGetMessage(3515, message_file, 2607)
exit

no_pkunzip2:
/* "PKUNZIP2.EXE file not found"
 */
say SysGetMessage(1533, osmsg_file, 'PKUNZIP2.EXE' )
say SysGetMessage(1533, oshelp_file )
exit

/* ---------------------------Subroutines------------------------------ */

/* Name: BootDrive
 */
BootDrive:
record = value('PATH',,'OS2ENVIRONMENT')
record = translate(record)
os2_pos = pos(':\OS2;',record)
if (os2_pos=0) then
    bootdrive='A'
else do
            bootdrive_pos = os2_pos - 1
            bootdrive = substr(record,bootdrive_pos,1)
end
if  (verify(bootdrive,alphabet) <> 0) then return 'A' /* presume A */
return bootdrive


/* Name: UnPack_DLSWIN( source, target )
 * Parameters:
 *       source - fully qualified path to DLSWIN.ZIP
 *       target - path to install program files
 */
UnPack_DLSWIN:
   source = arg(1)
   target = arg(2)

   'pkunzip2 ' source ' ' target 'dlsnet.drv'
   'pkunzip2 ' source ' ' target 'dlsnet.hlp'
   'pkunzip2 ' source ' ' target 'dlssetup.exe'
   'pkunzip2 ' source ' ' target 'fmshare.dll'
   'pkunzip2 ' source ' ' target 'netapi.dll'
   'pkunzip2 ' source ' ' target 'pmspl.dll'
   'pkunzip2 ' source ' ' target 'runlsapp.exe'
   'pkunzip2 ' source ' ' target 'wdls.exe'
   'pkunzip2 ' source ' ' target 'windls.dll'
   'pkunzip2 ' source ' ' target 'dlstb.dll'
   'pkunzip2 ' source ' ' target 'setini.exe'
return


/* Name: UnPack_DLSMSG( source, target )
 * Parameters:
 *       source - full qualified path to DLSMSG.ZIP
 *       target - path to install program files
 */
UnPack_DLSMSG:
   source = arg(1)
   target = arg(2)

   'pkunzip2 ' source ' ' target 'gui.msg'
   'pkunzip2 ' source ' ' target 'guih.msg'
   'pkunzip2 ' source ' ' target 'net.msg'
   'pkunzip2 ' source ' ' target 'neth.msg'
return


/*
 * Name:  GetDriveType( drive )
 * Purpose:
 *        Determines if a given drive is a diskette drive, fixed drive,
 *        or removeable media.
 *
 * Input:
 *        drive - drive letter to test, for example 'A:'
 *
 * Output:
 *        0 - drive is not a diskette drive (it's a fixed drive)
 *        1 - drive is a diskette drive
 *        2 - drive is removeable media (i.e., diskette, CD-ROM, etc.)
 */
GetDriveType:

drive = arg(1)
size144 = 1.44 * 1024 * 1000        /* Total bytes for 1.44MB diskette */

   /* Retrieve a list of local drives, starting with drive A:,
    * and verify that this drive is a valid device.  We are
    * interested only in local drives.
    */
   locallist = SysDriveMap( 'A:' ,'LOCAL');

   found = POS(drive, locallist)
   if ( found = 0 ) then        /* Did not find the drive ... */
   do
     /* SYS0003: The system cannot find the path specified. */
     say SysGetMessage(3, osmsg_file )
     exit
   end

   driveinfo = SysDriveInfo(drive)
   parse var driveinfo localdrive freebytes totalbytes label

   if (totalbytes \= '') then
   do
     /* Anything with capacity equal to or less than a 1.44MB
      * drive will be considered a diskette drive.
      */
     if (totalbytes <= size144) then
       rc = 1            /* It's a diskette drive */
     else
       rc = 0            /* It's not a diskette drive */
   end
   /* If we got this far and information could not be obtained,
    * then we only know that it's removeable media -- it could
    * be a CD-ROM drive without a CD-ROM present or a diskette
    * drive.
    */
   else
       rc = 2            /* It's removeable media */

return rc


/*
 * Name:  MakeDirectory( dirpath )
 * Purpose:
 *        Handles creation of nested paths, such as 'C:\NET\GUI\WINDOWS'
 *
 * Input:
 *        dirpath - fully qualified directory path
 *
 * Output:
 *        0 - directory path was created successfully
 *        non-zero - directory path was not created successfully; this
 *                   is set to the return code from SysMkDir()
 */
MakeDirectory:
dirpath = arg(1)
exists = 0

  /* Check for files of the same name as the target directory. */
  call SysFileTree  dirpath, 'file', 'F'
  if (file.0 \= 0) then
  do
    /* SYS1248: A subdirectory of file %1 already exists */
    say SysGetMessage(1248, osmsg_file, dirpath )
    exit
  end

  /* Check the length of the target directory -- it can't be more
   * than 63 bytes.
   */
  if ( length(dirpath) > 63 ) then
  do
    /* SYS1694: The path name is too long. */
    say SysGetMessage(1694, osmsg_file )
    say SysGetMessage(1694, oshelp_file )
    exit
  end

  /* Check if the target directory already exists.  If it does, then
   * we set the 'exists' flag.  Since this directory may have a name
   * greater than 8, we still want to parse the specified target path
   * so as to detect any directories that are not compatible with DOS.
   */
  call SysFileTree  dirpath, 'file', 'D'
  if (file.0 \= 0) then
     exists = 1

  /* Append a backslash to the target, if necessary.
   */
  if (translate(right(dirpath,1)) <> '\') then
      dirpath = dirpath || '\'

  /* Split the path into pieces */
  drive = filespec('drive',dirpath)
  path = filespec('path',dirpath)

  /* We support up to 8 nested directories -- any more than that
   * are ignored.
   */
  parse var path '\'dir.1'\'dir.2'\'dir.3'\'dir.4'\'dir.5'\'dir.6'\'dir.7'\'dir.8'\'

  /* Now we're ready to create the target directory.
   * If a directory is encountered that has more than
   * 8 bytes, the process fails.  This is to ensure
   * that the target directory adheres to the DOS
   * directory format.  Directory names with an extension
   * (i.e., the "8.3" format) are not supported.
   */
  i = 1
  savedir = drive
  do while ( (dir.i <> '') & (i <= 8) )
    if ( length(dir.i) <= 8 ) then
    do
      dir.i = savedir || '\' || dir.i
      rc = SysMkDir( dir.i )
      savedir = dir.i
      i = i + 1
    end
    else
    do
      /* Any directory greater than 8 bytes in length is invalid.
       *
       * "SYS1057: The file or path name entered is invalid in a DOS session."
       */
      rc = 1057
      return rc
    end
  end

  /* If the directory already exists, then SysMkDir() will
   * return Error 5: Access denied.  This is OK, so we fixup
   * the return code to reflect success.
   */
  if ( (rc = 5) & (exists = 1) ) then
     rc = 0

return rc
