#!/bin/sh
#
# Install FIELDVIEW from multi-platform CD-ROM.
#

echo 'Enter the name of the directory where you will be installing FIELDVIEW:'
read targetdir
if test -z "$targetdir"
then
    echo 'Empty directory name - aborting the installation.'
    exit 1
fi

first_char=`echo $targetdir | sed -e 's/^\(.\).*/\1/'`
if test "$first_char" != "/"
then
    echo "'$targetdir' is not a complete path."
    echo "Please supply a complete path name, starting with '/'."
    exit 1
fi

if test ! -d $targetdir
then
    echo "Directory $targetdir does not exist, or is not a directory."
    echo "Please create the directory before installing FIELDVIEW."
    exit 1
fi

if test ! -w $targetdir
then
    echo "You do not have permission to write into directory $targetdir."
    echo 'Aborting the installation.'
    exit 1
fi

ostype=`uname`

echo
echo "Choose one of the following versions of FIELDVIEW to be installed:"
echo
echo "    1. SGI Irix"
echo "    2. Sun Solaris"
echo "    3. IBM AIX"
echo "    4. Hewlett-Packard HP-UX"
echo "    5. Linux (Intel x86)"
echo "    6. Linux (Intel IA64)"
echo "    7. Linux (AMD 64, Intel EM64T)"
echo "    8. Mac OS X"
echo
echo "    9. none of the above"
echo
echo "Please enter a number from 1 to 9:"

while :
do
    read version
    case $version in
    1) tarfile='[iI][rR][iI][sS]64.[zZ]*'; break;;
    2) tarfile='[sS][oO][lL][aA][rR][iI][sS].[zZ]*'; break;;
    3) tarfile='[iI][bB][mM]_[rR][sS].[zZ]*'; break;;
    4) tarfile='[hH][pP]_700.[zZ]*'; break;;
    5) tarfile='[lL][iI][nN][uU][xX]_[xX]86.[zZ]*'; break;;
    6) tarfile='[lL][iI][nN][uU][xX]_[iI][aA]64.[zZ]*'; break;;
    7) tarfile='[lL][iI][nN][uU][xX]_[aA][mM][dD]64.[zZ]*'; break;;
    8) tarfile='[mM][aA][cC]_[pP][pP][cC].[zZ]*'; break;;
    9) exit;;

    *) echo "Please enter a number from 1 to 9:"
    esac
done

comfile='[cC][oO][mM][mM][oO][nN].[zZ]*'

if test ! \( -f $tarfile -a -f $comfile \)
then
    echo
    echo "Could not find the FIELDVIEW archive file in the"
    echo "current directory."
    echo "Make sure you are in the CD-ROM directory that contains"
    echo "the Unix FIELDVIEW installation files."
    exit 1
fi

type zcat >/dev/null
if test $? -ne 0
then
    echo
    echo "Could not find the \"zcat\" command on your system." 1>&2
    echo "This command is needed to uncompress the FIELDVIEW" 1>&2
    echo "archive file retrieved from CD-ROM." 1>&2
    echo "See your system administrator." 1>&2
    exit 1
fi

echo "Restoring fv directory from CD-ROM..."
curd=`pwd`
(cd $targetdir; cat $curd/$tarfile | zcat | tar xvf - fv)
if test $? -ne 0
then
    echo
    echo "Error restoring FIELDVIEW files from CD-ROM." 1>&2
    exit 5
fi
(cd $targetdir; cat $curd/$comfile | zcat | tar xvf - fv)
if test $? -ne 0
then
    echo
    echo "Error restoring FIELDVIEW files from CD-ROM." 1>&2
    exit 5
fi
echo "Restore completed."

case "$ostype" in
SunOS*)
    case "`uname -r`" in
    4*)  sum_cmd="/bin/sum"; need_name=1;;
    *)   sum_cmd="/bin/sum -r"; need_name=0;;
    esac
    ;;
Darwin)  sum_cmd="/usr/bin/sum"; need_name=0;;
Linux*)  sum_cmd="/usr/bin/sum"; need_name=1;;
*)       sum_cmd="/bin/sum -r"; need_name=0;;
esac

checknew="/tmp/checknew.$$"
checkold="/tmp/checkold.$$"

trap '/bin/rm -f $checknew $checkold; exit 15' 1 2 15

cd $targetdir/fv
if test $? -ne 0
then
    echo
    echo "Unable to change directory to $targetdir/fv" 1>&2
    echo "Unable to verify the install." 1>&2
    exit 6
fi

echo "Verifying install of fv directory..."

case $version in
1)
# checksums for 64-bit SGI version of fv directory
cat > $checkold <<EOF
00023 ./lib/tk8.2/tkfbox.tcl
00052 ./lib/tcl8.2/encoding/big5.enc
00064 ./lib/tk8.2/text.tcl
00273 ./lib/tcl8.2/encoding/cp1253.enc
00281 ./lib/tcl8.2/encoding/gb12345.enc
00300 ./user/user_results_file_read.f
00328 ./demo/iso1.iso
00349 ./bin/fv_to_printer.sh.sample
00455 ./lib/tcl8.2/encoding/cp857.enc
00685 ./examples/rectangular_duct/rect_duct_044.uns
00715 ./examples/rectangular_duct/rect_duct_048.uns
00795 ./lib/tcl8.2/encoding/jis0212.enc
01283 ./examples/rectangular_duct/rect_duct_029.uns
01383 ./lib/tk8.2/button.tcl
01478 ./fvx_and_restarts/animate_streamlines.map
01645 ./bin/mv_position
01677 ./demo/f18i9b_g_bin.fvbnd
01677 ./examples/f18/f18i9b_g_bin.fvbnd
02009 ./examples/rectangular_duct/rect_duct_024.uns
02213 ./examples/rectangular_duct/rect_duct_012.uns
02225 ./fvx_and_restarts/plot3d_eqns.frm
02235 ./lib/tk8.2/msgbox.tcl
02266 ./lib/tcl8.2/encoding/cp949.enc
02317 ./data/xdb_vars
02325 ./lib/blt2.4/dd_protocols/dd-number.tcl
02326 ./uil/iris64/iso_surface.uid
02345 ./user/user_grid_file_read.f
02492 ./examples/rectangular_duct/rect_duct_062.uns
02682 ./doc/Readme_InstallNotes_12.pdf
02722 ./data/colormaps/c3.col
02780 ./uns/write_split_ascii_uns.f
02844 ./examples/rectangular_duct/rect_duct_067.uns
03030 ./fvx_and_restarts/bluntfinq.bin
03294 ./lib/blt2.4/bltDragdrop.tcl
03299 ./examples/rectangular_duct/rect_duct_030.uns
03321 ./demo/coord3.crd
03329 ./fvx_and_restarts/set_view_TEST.fvx
03592 ./dore/data/il/def/parsekey.sym
03816 ./geometry/cyln48.atm
04150 ./fvx_and_restarts/seed_rectangle_proto.fvx
04449 ./examples/rectangular_duct/rect_duct_039.uns
04525 ./fvx_and_restarts/seed_circ_GUI.fvx
04824 ./lib/tcl8.2/encoding/macThai.enc
04871 ./uil/iris64/transient_datav2.uid
04999 ./lib/tcl8.2/encoding/macIceland.enc
05048 ./examples/rectangular_duct/rect_duct_040.uns
05077 ./lib/tcl8.2/http2.1/pkgIndex.tcl
05136 ./lib/tcl8.2/tcltest1.0/pkgIndex.tcl
05176 ./lib/tcl8.2/encoding/koi8-r.enc
05298 ./uil/iris64/coordinate_surface.uid
05986 ./demo/aerospace_LDM.bnd
06058 ./demo/f18i9b_q_bin
06058 ./examples/f18/f18i9b_q_bin
06106 ./lib/tcl8.2/encoding/cp737.enc
06139 ./dore/data/fonts/romant.df
06194 ./lib/tcl8.2/encoding/gb2312.enc
06517 ./bin/fvsrv
06966 ./lib/tcl8.2/encoding/iso8859-8.enc
07340 ./geometry/cyln12.atm
07365 ./examples/rectangular_duct/rect_duct_016.uns
07605 ./demo/stream4.stl
07614 ./demo/t3.ttl
07885 ./bin/fv.template
08136 ./lib/tcl8.2/encoding/euc-jp.enc
08160 ./lib/tcl8.2/encoding/cp1254.enc
08221 ./demo/t.ttl
08221 ./demo/t10.ttl
08320 ./examples/rectangular_duct/rect_duct_043.uns
08917 ./user/user_defined_functions.f
09118 ./lib/tcl8.2/encoding/cp862.enc
09161 ./examples/rectangular_duct/rect_duct_018.uns
09510 ./examples/rectangular_duct/rect_duct_015.uns
09563 ./examples/rectangular_duct/rect_duct_068.uns
09906 ./uil/iris64/title_panel.uid
10046 ./demo/lift.fvx
10063 ./bin/lib/ImageMagick-5.5.6-Q16/log.mgk
10147 ./uil/iris64/comp_surface.uid
10314 ./lib/tk8.2/obsolete.tcl
10394 ./examples/rectangular_duct/rect_duct_073.uns
10827 ./lib/tcl8.2/http2.1/http.tcl
10855 ./lib/tk8.2/bgerror.tcl
10967 ./doc/FVReference_Manual.pdf
11013 ./examples/rectangular_duct/rect_duct_055.uns
11049 ./uil/iris64/data_input.uid
11200 ./dore/data/fonts/romans.df
11396 ./examples/rectangular_duct/rect_duct_011.uns
11412 ./lib/tcl8.2/encoding/cp864.enc
11478 ./uil/iris64/xdb_info.uid
11868 ./lib/tcl8.2/tclIndex
11903 ./lib/tcl8.2/encoding/euc-cn.enc
11963 ./dore/data/fonts/italicc.df
12109 ./uil/iris64/plot.uid
12263 ./dore/data/fonts/cyrillic.df
12397 ./demo/t11.ttl
12721 ./examples/rectangular_duct/rect_duct_028.uns
12833 ./lib/tcl8.2/tcltest1.0/tcltest.tcl
13082 ./fvx_and_restarts/set_view.fvx
13243 ./bin/stereo_on
13258 ./lib/tcl8.2/encoding/euc-kr.enc
13293 ./fvx_and_restarts/bio.frm
13398 ./demo/t2.ttl
13461 ./examples/rectangular_duct/rect_duct_009.uns
13652 ./examples/rectangular_duct/rect_duct_078.uns
13709 ./uil/iris64/dynamic_subset.uid
13726 ./bin/lib/ImageMagick-5.5.6-Q16/delegates.mgk
14078 ./uil/iris64/flip.uid
14117 ./lib/blt2.4/NEWS
14544 ./dore/data/fonts/greekp.df
14611 ./geometry/allobs.ob
15065 ./examples/rectangular_duct/rect_duct_033.uns
15106 ./uil/iris64/formula.uid
15605 ./lib/tcl8.2/encoding/macCentEuro.enc
15825 ./lib/tcl8.2/encoding/macCroatian.enc
15887 ./lib/tcl8.2/encoding/macUkraine.enc
16097 ./lib/tcl8.2/encoding/iso8859-6.enc
16373 ./examples/rectangular_duct/rect_duct_008.uns
16561 ./lib/tk8.2/scrlbar.tcl
16650 ./geometry/poly12.atm
16658 ./fvx_and_restarts/seed_circular_proto.fvx
16709 ./lib/tcl8.2/msgcat1.0/pkgIndex.tcl
16741 ./lib/tcl8.2/parray.tcl
16795 ./dore/data/fonts/gothgrt.df
16841 ./lib/tcl8.2/encoding/cp950.enc
16950 ./fvx_and_restarts/animate_streamlines.prd
17062 ./uil/iris64/dataset_controls.uid
17116 ./lib/tcl8.2/auto.tcl
17184 ./demo/start.plt
17317 ./demo/start.dat
17332 ./demo/bound1.bnd
17427 ./lib/tcl8.2/encoding/cp775.enc
17570 ./lib/tcl8.2/encoding/iso2022.enc
17635 ./examples/rectangular_duct/rect_duct_031.uns
17647 ./geometry/poly48.atm
17790 ./lib/tcl8.2/encoding/iso8859-7.enc
17844 ./lib/tcl8.2/encoding/cp932.enc
17920 ./lib/tcl8.2/msgcat1.0/msgcat.tcl
18023 ./examples/rectangular_duct/rect_duct_020.uns
18271 ./lib/tcl8.2/encoding/cp863.enc
18315 ./lib/tcl8.2/encoding/cp869.enc
18328 ./examples/rectangular_duct/rect_duct_019.uns
18393 ./uil/iris64/boundary_conditions.uid
18422 ./bin/fv_user_print.sh.sample
18484 ./fvx_and_restarts/animate_streamlines.fvx
18619 ./lib/tcl8.2/encoding/macDingbats.enc
18713 ./dore/data/fonts/gothitt.df
19151 ./demo/aerospace_LDM.fvx
19329 ./lib/tcl8.2/encoding/macGreek.enc
19500 ./lib/tcl8.2/encoding/jis0208.enc
19545 ./lib/tcl8.2/encoding/cp861.enc
20094 ./examples/rectangular_duct/rect_duct_069.uns
20144 ./examples/rectangular_duct/rect_duct_056.uns
20242 ./bin/lib/ImageMagick-5.5.6-Q16/modules.mgk
20418 ./examples/rectangular_duct/rect_duct_007.uns
20545 ./examples/rectangular_duct/rect_duct_060.uns
20873 ./dore/data/fonts/unused.df
21460 ./lib/tcl8.2/http1.0/http.tcl
21496 ./data/xdb_vars_FV11.0
21564 ./lib/tcl8.2/ldAout.tcl
21798 ./demo/t7.ttl
21894 ./lib/blt2.4/bltTabset.tcl
21927 ./bin/mv_record
21960 ./dore/data/fonts/astrology.df
21969 ./examples/heat_exchanger/counter.uns
22040 ./uns/write_unformatted_uns.F
22438 ./lib/tcl8.2/encoding/cp1257.enc
22490 ./uil/iris64/datasetBrowser.uid
22569 ./examples/rectangular_duct/rect_duct_035.uns
22765 ./data/license.dat
23141 ./uil/iris64/region_control.uid
23217 ./demo/wingloadplots.fvx
23286 ./dore/data/fonts/italiccs.df
23323 ./demo/coord1.crd
23344 ./uns/write_ascii_uns.f
23526 ./lib/tk8.2/scale.tcl
23533 ./lib/tcl8.2/encoding/iso8859-5.enc
23541 ./examples/rectangular_duct/rect_duct_005.uns
23709 ./lib/tk8.2/images/logo64.gif
23711 ./bin/plugins/fluent_reader.so
23752 ./lib/tk8.2/dialog.tcl
23776 ./uil/iris64/vr.uid
24149 ./bin/animate
24188 ./examples/rectangular_duct/rect_duct_001.uns
24661 ./examples/rectangular_duct/rect_duct_010.uns
24673 ./data/colormaps/c2.col
24852 ./demo/aerospace_LDM.dat
24881 ./lib/blt2.4/htresize.cur
25000 ./bin/convert
25189 ./lib/tk8.2/tk.tcl
25502 ./examples/rectangular_duct/rect_duct_054.uns
25536 ./bin/mpeg2encode
25598 ./user/user_unstruct_combined.f
25605 ./lib/tcl8.2/encoding/macCyrillic.enc
25703 ./lib/tk8.2/tclIndex
26148 ./user/user_combined_file_read.f
26216 ./examples/rectangular_duct/rect_duct_057.uns
26225 ./uil/iris64/legend_control.uid
26345 ./bin/lib/ImageMagick-5.5.6-Q16/magic.mgk
26356 ./lib/tk8.2/images/README
26380 ./lib/tk8.2/focus.tcl
26644 ./bin/plugins/lsdyna_reader.so
26668 ./geometry/sph24.atm
26962 ./uil/iris64/vortex_core.uid
27073 ./fvx_and_restarts/seed_rect_GUI.fvx
27081 ./lib/blt2.4/bltGraph.tcl
27188 ./lib/tk8.2/images/pwrdLogo200.gif
27513 ./geometry/star.atm
27920 ./bin/lib/ImageMagick-5.5.6-Q16/type-ghostscript.mgk
28042 ./demo/coord0.crd
28310 ./examples/rectangular_duct/rect_duct_003.uns
28324 ./uns/write_split_binary_uns.c
28358 ./lib/blt2.4/dd_protocols/dd-file.tcl
28436 ./user/ftn_fv_close.f
28601 ./demo/coord5.crd
28632 ./examples/rectangular_duct/rect_duct_034.uns
28734 ./lib/blt2.4/dd_protocols/dd-color.tcl
28893 ./examples/rectangular_duct/rect_duct_053.uns
29100 ./lib/tk8.2/listbox.tcl
29194 ./lib/tk8.2/images/pwrdLogo75.gif
29195 ./examples/rectangular_duct/rect_duct_046.uns
29424 ./bin/lib/ImageMagick-5.5.6-Q16/colors.mgk
29488 ./examples/rectangular_duct/rect_duct_066.uns
29543 ./lib/tcl8.2/encoding/macRomania.enc
29643 ./fvx_and_restarts/animate_streamlines.vct
29657 ./demo/stream3.stl
29787 ./lib/tk8.2/optMenu.tcl
29826 ./lib/tcl8.2/init.tcl
29845 ./examples/rectangular_duct/rect_duct_023.uns
29936 ./examples/rectangular_duct/rect_duct_014.uns
29942 ./lib/tcl8.2/package.tcl
30113 ./lib/tk8.2/palette.tcl
30177 ./dore/data/fonts/meteorology.df
30254 ./lib/blt2.4/bltDnd.tcl
30349 ./lib/tk8.2/images/tai-ku.gif
30506 ./lib/tcl8.2/encoding/iso2022-kr.enc
30674 ./uil/iris64/integration.uid
30792 ./dore/data/fonts/romanc.df
31089 ./lib/tcl8.2/encoding/iso8859-1.enc
31428 ./bin/lib/ImageMagick-5.5.6-Q16/type-windows.mgk
31591 ./lib/tcl8.2/encoding/cp866.enc
31691 ./lib/tcl8.2/encoding/cp865.enc
32060 ./examples/rectangular_duct/rect_duct_025.uns
32217 ./examples/rectangular_duct/rect_duct_006.uns
32418 ./uil/iris64/color_mixer.uid
32492 ./lib/tcl8.2/encoding/cp852.enc
32548 ./demo/aerospace_LDM.vct
32643 ./fvx_and_restarts/basic_volume_integral.fvx
32649 ./examples/rectangular_duct/rect_duct_051.uns
32725 ./examples/rectangular_duct/rect_duct_022.uns
32837 ./examples/rectangular_duct/rect_duct_050.uns
32927 ./lib/tk8.2/tkAppInit.c
33001 ./lib/tcl8.2/encoding/cp874.enc
33248 ./demo/moments.fvx
33321 ./lib/blt2.4/dd_protocols/dd-text.tcl
33396 ./demo/coord2.crd
33520 ./doc/Readme_Server_Toolkit.txt
33547 ./uns/write_binary_uns.c
34182 ./lib/blt2.4/dd_protocols/tclIndex
34626 ./geometry/sph12.atm
34669 ./lib/tcl8.2/encoding/gb1988.enc
34710 ./fvx_and_restarts/wind.frm
34959 ./dore/data/fonts/lcmath.df
35021 ./fvx_and_restarts/animate_streamlines.frm
35130 ./bin/display
35313 ./dore/data/fonts/romanp.df
35384 ./geometry/square.atm
35398 ./demo/t4.ttl
35629 ./bin/README_Copyrights.txt
35670 ./doc/FVUsers_Guide.pdf
35707 ./lib/tcl8.2/encoding/shiftjis.enc
35818 ./examples/rectangular_duct/rect_duct_063.uns
35841 ./bin/node_id
36318 ./examples/rectangular_duct/rect_duct_080.uns
36575 ./fvx_and_restarts/bluntfinx.bin.fvbnd
36582 ./uil/iris64/vector_params.uid
36641 ./lib/tk8.2/images/pwrdLogo150.gif
36865 ./uns/ftn_fv_reader_tags.h
36925 ./examples/rectangular_duct/rect_duct_004.uns
37055 ./lib/tcl8.2/tclAppInit.c
37259 ./bin/fvbin
37272 ./lib/tk8.2/clrpick.tcl
37307 ./lib/tcl8.2/encoding/ksc5601.enc
37409 ./lib/tcl8.2/history.tcl
37439 ./lib/tk8.2/images/pwrdLogo.eps
37477 ./lib/tcl8.2/encoding/cp1258.enc
37544 ./doc/Whats_New_FV12.pdf
37688 ./user/ftn_register_functions.f
38287 ./fvx_and_restarts/README_PLOT3D_formula_restart
38461 ./demo/f18i9b_g_bin
38461 ./examples/f18/f18i9b_g_bin
38501 ./dore/data/fonts/italict.df
38531 ./examples/rectangular_duct/rect_duct_036.uns
38545 ./doc/Help_Intro.pdf
38574 ./lib/tk8.2/prolog.ps
38666 ./lib/tk8.2/menu.tcl
38861 ./lib/tk8.2/images/pwrdLogo100.gif
38994 ./lib/tcl8.2/encoding/macTurkish.enc
39109 ./lib/tk8.2/images/pwrdLogo175.gif
39350 ./bin/contest
39676 ./lib/tcl8.2/encoding/iso2022-jp.enc
39766 ./bin/stereo_off
39768 ./bin/mv_reset
39874 ./demo/f18show.scr
40066 ./user/ftn_register_data_readers.f
40442 ./demo/col1.map
40541 ./user/register_data_readers.c
40552 ./lib/tcl8.2/encoding/symbol.enc
41107 ./lib/tcl8.2/encoding/cp1256.enc
41134 ./bin/fvsrvbin
41169 ./geometry/geometry_formats.ps
41346 ./uil/iris64/streamlines.uid
41358 ./demo/stream2.stl
41364 ./lib/tk8.2/images/logo100.gif
41501 ./bin/plugins/lib/libcgns.so
41566 ./lib/blt2.4/pkgIndex.tcl
41734 ./demo/iso2.iso
42046 ./lib/tcl8.2/encoding/iso8859-9.enc
42054 ./demo/t9.ttl
42217 ./lib/tcl8.2/encoding/macJapan.enc
42218 ./lib/tcl8.2/encoding/cp437.enc
42270 ./examples/rectangular_duct/rect_duct_026.uns
42627 ./dore/data/fonts/ucmath.df
42717 ./uil/iris64/misc_support.uid
42775 ./uil/iris64/xform_controls.uid
43278 ./dore/data/fonts/symbols.df
43508 ./lib/tk8.2/images/logo.eps
43701 ./examples/rectangular_duct/rect_duct_041.uns
43751 ./examples/heat_exchanger/cocurrent.uns.fvreg
43751 ./examples/heat_exchanger/counter.uns.fvreg
43946 ./fvx_and_restarts/animate_streamlines.dat
43960 ./user/user_unstruct_grid.f
44519 ./lib/tcl8.2/ldAix
44814 ./examples/rectangular_duct/rect_duct_042.uns
44867 ./demo/start.vct
44875 ./user/user_unstruct_results.f
44878 ./lib/tcl8.2/encoding/jis0201.enc
44926 ./examples/rectangular_duct/rect_duct_037.uns
45085 ./sconfig/srv.template
45438 ./examples/rectangular_duct/rect_duct_032.uns
45642 ./dore/data/fonts/gothgbt.df
45830 ./demo/iso3.iso
45880 ./lib/tcl8.2/encoding/cp1250.enc
46000 ./examples/rectangular_duct/rect_duct_013.uns
46023 ./lib/tk8.2/images/logoMed.gif
46093 ./demo/view2.vct
46103 ./examples/rectangular_duct/rect_duct_047.uns
46238 ./demo/start.map
46455 ./lib/tcl8.2/encoding/cp1255.enc
46496 ./lib/blt2.4/bltHierbox.tcl
46844 ./geometry/cone12.atm
47054 ./fvx_and_restarts/seed_sphere_proto.fvx
47193 ./bin/plugins/cgns_reader.so
47357 ./uns/write_split_unformatted_uns.F
47481 ./bin/server_name
47891 ./demo/view6.vct
47930 ./demo/drag.fvx
47970 ./lib/tcl8.2/safe.tcl
47994 ./bin/fv_platform
48023 ./lib/tcl8.2/encoding/cp936.enc
48142 ./lib/blt2.4/bltHiertable.tcl
48291 ./data/colormaps/c7.col
48295 ./fvx_and_restarts/xcolor_defines.fvx
48438 ./examples/rectangular_duct/rect_duct_059.uns
48906 ./dore/data/fonts/greekcs.df
49045 ./fvx_and_restarts/seed_sphere_GUI.fvx
49051 ./demo/view4.vct
49338 ./demo/view3.vct
49417 ./lib/tcl8.2/encoding/ascii.enc
49617 ./demo/start.bnd
49824 ./lib/tcl8.2/encoding/iso8859-3.enc
49896 ./geometry/cube.atm
49920 ./fvx_and_restarts/lsdyna_formulas.frm
49926 ./data/colormaps/c4.col
49932 ./demo/t8.ttl
49964 ./uil/iris64/misc_file_input.uid
50329 ./demo/view7.vct
50415 ./examples/rectangular_duct/rect_duct_061.uns
50427 ./examples/rectangular_duct/rect_duct_077.uns
50441 ./bin/prolog.ps
50605 ./examples/rectangular_duct/rect_duct_038.uns
51123 ./examples/rectangular_duct/rect_duct_021.uns
51494 ./bin/fv_convert.template
51948 ./bin/mv_init_tape
51968 ./examples/rectangular_duct/rect_duct_065.uns
52013 ./lib/tcl8.2/encoding/cp855.enc
52057 ./examples/rectangular_duct/rect_duct_076.uns
52309 ./lib/tcl8.2/opt0.4/pkgIndex.tcl
52340 ./examples/rectangular_duct/rect_duct_049.uns
52415 ./demo/start.ttl
52522 ./uil/iris64/surface_plot.uid
52682 ./uil/iris64/keyframe.uid
52745 ./dore/data/fonts/romand.df
52833 ./lib/tcl8.2/encoding/cp850.enc
53221 ./lib/blt2.4/tclIndex
53293 ./examples/rectangular_duct/rect_duct_052.uns
53473 ./lib/tk8.2/comdlg.tcl
53516 ./examples/rectangular_duct/rect_duct_058.uns
53572 ./bin/lmhostid
53968 ./user/toolkit.c
54002 ./lib/tcl8.2/encoding/cp860.enc
54102 ./fvx_and_restarts/calc_volume_integral.fvx
54228 ./sconfig/manual.srv
54250 ./demo/plot1.lpt
54260 ./demo/plot2.lpt
54331 ./bin/make_fv
54533 ./dore/data/fonts/romancs.df
54905 ./dore/data/fonts/helvetica.df
55047 ./examples/rectangular_duct/rect_duct.trk
55148 ./bin/mpeg2decode
55208 ./fvx_and_restarts/Q-criterion.frm
55231 ./lib/tcl8.2/http1.0/pkgIndex.tcl
55361 ./lib/tk8.2/xmfbox.tcl
55552 ./examples/rectangular_duct/rect_duct_074.uns
55614 ./examples/rectangular_duct/rect_duct_045.uns
55673 ./lib/tcl8.2/encoding/cp1251.enc
55709 ./examples/rectangular_duct/rect_duct_075.uns
55734 ./dore/data/fonts/scriptc.df
55885 ./lib/tk8.2/images/logoLarge.gif
55991 ./uil/iris64/subset_params.uid
56003 ./data/colormaps/c5.col
56186 ./dore/data/fonts/greeks.df
56255 ./examples/rectangular_duct/rect_duct_002.uns
56320 ./demo/t6.ttl
56378 ./examples/rectangular_duct/rect_duct_070.uns
56431 ./uil/iris64/particle_paths.uid
56695 ./lib/tcl8.2/encoding/iso8859-2.enc
56916 ./lib/blt2.4/bltTabnotebook.tcl
56960 ./dore/data/fonts/greekc.df
57234 ./user/user_unstruct_functions.f
57480 ./demo/aerospace_LDM.frm
57686 ./lib/blt2.4/Problems
57698 ./fvx_and_restarts/FVX_Templates.pdf
57711 ./lib/tk8.2/entry.tcl
57757 ./examples/rectangular_duct/rect_duct_064.uns
57941 ./lib/tcl8.2/encoding/dingbats.enc
58071 ./lib/tcl8.2/opt0.4/optparse.tcl
58217 ./bin/lib/ImageMagick-5.5.6-Q16/type.mgk
58430 ./fvx_and_restarts/animate_streamlines.bnd
58616 ./fvx_and_restarts/plot_along_streamline.fvx
58635 ./demo/start.frm
58764 ./lib/tk8.2/tearoff.tcl
59310 ./demo/t0.ttl
59391 ./lib/tcl8.2/encoding/iso8859-4.enc
59497 ./doc/How_to.pdf
59695 ./examples/rectangular_duct/rect_duct_071.uns
60258 ./dore/data/errmsg/errmsgfile
60621 ./data/colormaps/c6.col
60693 ./examples/rectangular_duct/rect_duct_079.uns
60841 ./bin/lib/ImageMagick-5.5.6-Q16/type-solaris.mgk
60862 ./uil/iris64/presentation.uid
61035 ./lib/blt2.4/bltGraph.pro
61091 ./bin/node_id.il
61117 ./lib/tcl8.2/word.tcl
61247 ./demo/blank.ttl
61328 ./uil/iris64/colormap_specification.uid
61446 ./uns/fv_reader_tags.h
61497 ./dore/data/fonts/music.df
61541 ./lib/tk8.2/console.tcl
61734 ./demo/coord4.crd
61852 ./demo/view5.vct
62390 ./user/register_functions.c
62413 ./demo/t5.ttl
62662 ./demo/t1.ttl
62834 ./uil/iris64/point_probe.uid
62907 ./demo/stream1.stl
62918 ./fvx_and_restarts/bluntfinx.bin
63021 ./examples/rectangular_duct/rect_duct_017.uns
63102 ./lib/tcl8.2/encoding/macRoman.enc
63106 ./examples/heat_exchanger/cocurrent.uns
63221 ./lib/tk8.2/safetk.tcl
63321 ./bin/ld_fv
63340 ./demo/iso4.iso
63384 ./examples/rectangular_duct/rect_duct_027.uns
63671 ./geometry/cone48.atm
63850 ./lib/tcl8.2/encoding/cp1252.enc
64512 ./examples/rectangular_duct/rect_duct_072.uns
64515 ./data/colormaps/c8.col
64986 ./lib/blt2.4/Readme
65154 ./lib/blt2.4/bltCanvEps.pro
65212 ./uil/iris64/func.uid
65304 ./data/colormaps/c1.col
65467 ./dore/data/fonts/scripts.df
EOF
;;
2)
# checksums for Solaris 2 version of fv directory
cat > $checkold <<EOF
00023 ./lib/tk8.2/tkfbox.tcl
00052 ./lib/tcl8.2/encoding/big5.enc
00064 ./lib/tk8.2/text.tcl
00273 ./lib/tcl8.2/encoding/cp1253.enc
00281 ./lib/tcl8.2/encoding/gb12345.enc
00300 ./user/user_results_file_read.f
00328 ./demo/iso1.iso
00349 ./bin/fv_to_printer.sh.sample
00455 ./lib/tcl8.2/encoding/cp857.enc
00521 ./bin/mv_reset
00685 ./examples/rectangular_duct/rect_duct_044.uns
00715 ./examples/rectangular_duct/rect_duct_048.uns
00770 ./bin/node_id.il
00795 ./lib/tcl8.2/encoding/jis0212.enc
01283 ./examples/rectangular_duct/rect_duct_029.uns
01383 ./lib/tk8.2/button.tcl
01478 ./fvx_and_restarts/animate_streamlines.map
01677 ./demo/f18i9b_g_bin.fvbnd
01677 ./examples/f18/f18i9b_g_bin.fvbnd
02009 ./examples/rectangular_duct/rect_duct_024.uns
02213 ./examples/rectangular_duct/rect_duct_012.uns
02225 ./fvx_and_restarts/plot3d_eqns.frm
02235 ./lib/tk8.2/msgbox.tcl
02266 ./lib/tcl8.2/encoding/cp949.enc
02317 ./data/xdb_vars
02325 ./lib/blt2.4/dd_protocols/dd-number.tcl
02345 ./user/user_grid_file_read.f
02492 ./examples/rectangular_duct/rect_duct_062.uns
02682 ./doc/Readme_InstallNotes_12.pdf
02722 ./data/colormaps/c3.col
02780 ./uns/write_split_ascii_uns.f
02844 ./examples/rectangular_duct/rect_duct_067.uns
03030 ./fvx_and_restarts/bluntfinq.bin
03255 ./uil/solaris/xdb_info.uid
03294 ./lib/blt2.4/bltDragdrop.tcl
03299 ./examples/rectangular_duct/rect_duct_030.uns
03321 ./demo/coord3.crd
03329 ./fvx_and_restarts/set_view_TEST.fvx
03592 ./dore/data/il/def/parsekey.sym
03816 ./geometry/cyln48.atm
04150 ./fvx_and_restarts/seed_rectangle_proto.fvx
04449 ./examples/rectangular_duct/rect_duct_039.uns
04525 ./fvx_and_restarts/seed_circ_GUI.fvx
04824 ./lib/tcl8.2/encoding/macThai.enc
04999 ./lib/tcl8.2/encoding/macIceland.enc
05048 ./examples/rectangular_duct/rect_duct_040.uns
05077 ./lib/tcl8.2/http2.1/pkgIndex.tcl
05136 ./lib/tcl8.2/tcltest1.0/pkgIndex.tcl
05176 ./lib/tcl8.2/encoding/koi8-r.enc
05986 ./demo/aerospace_LDM.bnd
06058 ./demo/f18i9b_q_bin
06058 ./examples/f18/f18i9b_q_bin
06106 ./lib/tcl8.2/encoding/cp737.enc
06139 ./dore/data/fonts/romant.df
06194 ./lib/tcl8.2/encoding/gb2312.enc
06517 ./bin/fvsrv
06966 ./lib/tcl8.2/encoding/iso8859-8.enc
07146 ./uil/solaris/presentation.uid
07340 ./geometry/cyln12.atm
07365 ./examples/rectangular_duct/rect_duct_016.uns
07492 ./uil/solaris/title_panel.uid
07605 ./demo/stream4.stl
07614 ./demo/t3.ttl
07885 ./bin/fv.template
07909 ./uil/solaris/vector_params.uid
08136 ./lib/tcl8.2/encoding/euc-jp.enc
08160 ./lib/tcl8.2/encoding/cp1254.enc
08221 ./demo/t.ttl
08221 ./demo/t10.ttl
08280 ./uil/solaris/integration.uid
08320 ./examples/rectangular_duct/rect_duct_043.uns
08917 ./user/user_defined_functions.f
09118 ./lib/tcl8.2/encoding/cp862.enc
09161 ./examples/rectangular_duct/rect_duct_018.uns
09510 ./examples/rectangular_duct/rect_duct_015.uns
09563 ./examples/rectangular_duct/rect_duct_068.uns
10046 ./demo/lift.fvx
10063 ./bin/lib/ImageMagick-5.5.6-Q16/log.mgk
10314 ./lib/tk8.2/obsolete.tcl
10394 ./examples/rectangular_duct/rect_duct_073.uns
10596 ./uil/solaris/region_control.uid
10725 ./bin/display
10827 ./lib/tcl8.2/http2.1/http.tcl
10855 ./lib/tk8.2/bgerror.tcl
10860 ./uil/solaris/iso_surface.uid
10967 ./doc/FVReference_Manual.pdf
11013 ./examples/rectangular_duct/rect_duct_055.uns
11200 ./dore/data/fonts/romans.df
11329 ./uil/solaris/streamlines.uid
11396 ./examples/rectangular_duct/rect_duct_011.uns
11412 ./lib/tcl8.2/encoding/cp864.enc
11868 ./lib/tcl8.2/tclIndex
11903 ./lib/tcl8.2/encoding/euc-cn.enc
11963 ./dore/data/fonts/italicc.df
12225 ./uil/solaris/xform_controls.uid
12263 ./dore/data/fonts/cyrillic.df
12397 ./demo/t11.ttl
12501 ./bin/fv_vnc
12721 ./examples/rectangular_duct/rect_duct_028.uns
12833 ./lib/tcl8.2/tcltest1.0/tcltest.tcl
13082 ./fvx_and_restarts/set_view.fvx
13258 ./lib/tcl8.2/encoding/euc-kr.enc
13293 ./fvx_and_restarts/bio.frm
13398 ./demo/t2.ttl
13461 ./examples/rectangular_duct/rect_duct_009.uns
13652 ./examples/rectangular_duct/rect_duct_078.uns
13726 ./bin/lib/ImageMagick-5.5.6-Q16/delegates.mgk
14117 ./lib/blt2.4/NEWS
14491 ./uil/solaris/transient_datav2.uid
14544 ./dore/data/fonts/greekp.df
14611 ./geometry/allobs.ob
14677 ./bin/fvsrv64/plugins/lsdyna_reader.so
15065 ./examples/rectangular_duct/rect_duct_033.uns
15605 ./lib/tcl8.2/encoding/macCentEuro.enc
15825 ./lib/tcl8.2/encoding/macCroatian.enc
15887 ./lib/tcl8.2/encoding/macUkraine.enc
16097 ./lib/tcl8.2/encoding/iso8859-6.enc
16373 ./examples/rectangular_duct/rect_duct_008.uns
16561 ./lib/tk8.2/scrlbar.tcl
16650 ./geometry/poly12.atm
16658 ./fvx_and_restarts/seed_circular_proto.fvx
16709 ./lib/tcl8.2/msgcat1.0/pkgIndex.tcl
16741 ./lib/tcl8.2/parray.tcl
16795 ./dore/data/fonts/gothgrt.df
16841 ./lib/tcl8.2/encoding/cp950.enc
16950 ./fvx_and_restarts/animate_streamlines.prd
17028 ./bin/fvsrv64/fvsrv64
17116 ./lib/tcl8.2/auto.tcl
17153 ./lib/start_python.luac64
17184 ./demo/start.plt
17206 ./bin/mv_init_tape
17317 ./demo/start.dat
17332 ./demo/bound1.bnd
17427 ./lib/tcl8.2/encoding/cp775.enc
17570 ./lib/tcl8.2/encoding/iso2022.enc
17635 ./examples/rectangular_duct/rect_duct_031.uns
17647 ./geometry/poly48.atm
17790 ./lib/tcl8.2/encoding/iso8859-7.enc
17844 ./lib/tcl8.2/encoding/cp932.enc
17920 ./lib/tcl8.2/msgcat1.0/msgcat.tcl
18023 ./examples/rectangular_duct/rect_duct_020.uns
18271 ./lib/tcl8.2/encoding/cp863.enc
18315 ./lib/tcl8.2/encoding/cp869.enc
18328 ./examples/rectangular_duct/rect_duct_019.uns
18422 ./bin/fv_user_print.sh.sample
18484 ./fvx_and_restarts/animate_streamlines.fvx
18619 ./lib/tcl8.2/encoding/macDingbats.enc
18713 ./dore/data/fonts/gothitt.df
19151 ./demo/aerospace_LDM.fvx
19329 ./lib/tcl8.2/encoding/macGreek.enc
19500 ./lib/tcl8.2/encoding/jis0208.enc
19545 ./lib/tcl8.2/encoding/cp861.enc
19776 ./lib/start_python.luac
19875 ./bin/server_name
20094 ./examples/rectangular_duct/rect_duct_069.uns
20144 ./examples/rectangular_duct/rect_duct_056.uns
20186 ./bin/vnc_disp
20242 ./bin/lib/ImageMagick-5.5.6-Q16/modules.mgk
20418 ./examples/rectangular_duct/rect_duct_007.uns
20545 ./examples/rectangular_duct/rect_duct_060.uns
20859 ./uil/solaris/datasetBrowser.uid
20873 ./dore/data/fonts/unused.df
21460 ./lib/tcl8.2/http1.0/http.tcl
21496 ./data/xdb_vars_FV11.0
21564 ./lib/tcl8.2/ldAout.tcl
21692 ./bin/mpeg2encode
21798 ./demo/t7.ttl
21894 ./lib/blt2.4/bltTabset.tcl
21960 ./dore/data/fonts/astrology.df
21969 ./examples/heat_exchanger/counter.uns
22040 ./uns/write_unformatted_uns.F
22438 ./lib/tcl8.2/encoding/cp1257.enc
22569 ./examples/rectangular_duct/rect_duct_035.uns
22765 ./data/license.dat
23217 ./demo/wingloadplots.fvx
23286 ./dore/data/fonts/italiccs.df
23323 ./demo/coord1.crd
23344 ./uns/write_ascii_uns.f
23526 ./lib/tk8.2/scale.tcl
23533 ./lib/tcl8.2/encoding/iso8859-5.enc
23541 ./examples/rectangular_duct/rect_duct_005.uns
23709 ./lib/tk8.2/images/logo64.gif
23752 ./lib/tk8.2/dialog.tcl
24188 ./examples/rectangular_duct/rect_duct_001.uns
24661 ./examples/rectangular_duct/rect_duct_010.uns
24673 ./data/colormaps/c2.col
24852 ./demo/aerospace_LDM.dat
24881 ./lib/blt2.4/htresize.cur
25189 ./lib/tk8.2/tk.tcl
25218 ./uil/solaris/surface_plot.uid
25502 ./examples/rectangular_duct/rect_duct_054.uns
25505 ./lib/solaris/dummy/libGLU.so.1
25598 ./user/user_unstruct_combined.f
25605 ./lib/tcl8.2/encoding/macCyrillic.enc
25703 ./lib/tk8.2/tclIndex
26148 ./user/user_combined_file_read.f
26216 ./examples/rectangular_duct/rect_duct_057.uns
26335 ./bin/animate
26345 ./bin/lib/ImageMagick-5.5.6-Q16/magic.mgk
26356 ./lib/tk8.2/images/README
26380 ./lib/tk8.2/focus.tcl
26668 ./geometry/sph24.atm
26764 ./uil/solaris/func.uid
26896 ./uil/solaris/color_mixer.uid
27073 ./fvx_and_restarts/seed_rect_GUI.fvx
27081 ./lib/blt2.4/bltGraph.tcl
27188 ./lib/tk8.2/images/pwrdLogo200.gif
27513 ./geometry/star.atm
27920 ./bin/lib/ImageMagick-5.5.6-Q16/type-ghostscript.mgk
27965 ./uil/solaris/data_input.uid
28042 ./demo/coord0.crd
28088 ./uil/solaris/subset_params.uid
28310 ./examples/rectangular_duct/rect_duct_003.uns
28324 ./uns/write_split_binary_uns.c
28358 ./lib/blt2.4/dd_protocols/dd-file.tcl
28403 ./lib/solaris/liblualib.so.4.0
28436 ./user/ftn_fv_close.f
28601 ./demo/coord5.crd
28632 ./examples/rectangular_duct/rect_duct_034.uns
28734 ./lib/blt2.4/dd_protocols/dd-color.tcl
28893 ./examples/rectangular_duct/rect_duct_053.uns
29100 ./lib/tk8.2/listbox.tcl
29194 ./lib/tk8.2/images/pwrdLogo75.gif
29195 ./examples/rectangular_duct/rect_duct_046.uns
29424 ./bin/lib/ImageMagick-5.5.6-Q16/colors.mgk
29488 ./examples/rectangular_duct/rect_duct_066.uns
29543 ./lib/tcl8.2/encoding/macRomania.enc
29559 ./uil/solaris/flip.uid
29643 ./fvx_and_restarts/animate_streamlines.vct
29657 ./demo/stream3.stl
29787 ./lib/tk8.2/optMenu.tcl
29826 ./lib/tcl8.2/init.tcl
29845 ./examples/rectangular_duct/rect_duct_023.uns
29873 ./bin/mv_position
29936 ./examples/rectangular_duct/rect_duct_014.uns
29942 ./lib/tcl8.2/package.tcl
30113 ./lib/tk8.2/palette.tcl
30177 ./dore/data/fonts/meteorology.df
30254 ./lib/blt2.4/bltDnd.tcl
30349 ./lib/tk8.2/images/tai-ku.gif
30506 ./lib/tcl8.2/encoding/iso2022-kr.enc
30584 ./uil/solaris/vortex_core.uid
30728 ./uil/solaris/particle_paths.uid
30792 ./dore/data/fonts/romanc.df
30976 ./bin/plugins/fluent_reader.so
31089 ./lib/tcl8.2/encoding/iso8859-1.enc
31428 ./bin/lib/ImageMagick-5.5.6-Q16/type-windows.mgk
31591 ./lib/tcl8.2/encoding/cp866.enc
31667 ./bin/fvsrv64/plugins/fluent_reader.so
31691 ./lib/tcl8.2/encoding/cp865.enc
32060 ./examples/rectangular_duct/rect_duct_025.uns
32217 ./examples/rectangular_duct/rect_duct_006.uns
32492 ./lib/tcl8.2/encoding/cp852.enc
32548 ./demo/aerospace_LDM.vct
32643 ./fvx_and_restarts/basic_volume_integral.fvx
32649 ./examples/rectangular_duct/rect_duct_051.uns
32725 ./examples/rectangular_duct/rect_duct_022.uns
32837 ./examples/rectangular_duct/rect_duct_050.uns
32927 ./lib/tk8.2/tkAppInit.c
33001 ./lib/tcl8.2/encoding/cp874.enc
33248 ./demo/moments.fvx
33321 ./lib/blt2.4/dd_protocols/dd-text.tcl
33396 ./demo/coord2.crd
33520 ./doc/Readme_Server_Toolkit.txt
33547 ./uns/write_binary_uns.c
34182 ./lib/blt2.4/dd_protocols/tclIndex
34369 ./bin/fvsrv64/plugins/lib/libcgns.so
34385 ./bin/Xvnc
34626 ./geometry/sph12.atm
34669 ./lib/tcl8.2/encoding/gb1988.enc
34710 ./fvx_and_restarts/wind.frm
34959 ./dore/data/fonts/lcmath.df
35021 ./fvx_and_restarts/animate_streamlines.frm
35071 ./lib/solaris/dummy/libGL.so.1
35313 ./dore/data/fonts/romanp.df
35384 ./geometry/square.atm
35398 ./demo/t4.ttl
35629 ./bin/README_Copyrights.txt
35670 ./doc/FVUsers_Guide.pdf
35707 ./lib/tcl8.2/encoding/shiftjis.enc
35818 ./examples/rectangular_duct/rect_duct_063.uns
35841 ./bin/node_id
36318 ./examples/rectangular_duct/rect_duct_080.uns
36575 ./fvx_and_restarts/bluntfinx.bin.fvbnd
36641 ./lib/tk8.2/images/pwrdLogo150.gif
36671 ./uil/solaris/point_probe.uid
36865 ./uns/ftn_fv_reader_tags.h
36925 ./examples/rectangular_duct/rect_duct_004.uns
37055 ./lib/tcl8.2/tclAppInit.c
37222 ./uil/solaris/coordinate_surface.uid
37262 ./uil/solaris/comp_surface.uid
37272 ./lib/tk8.2/clrpick.tcl
37307 ./lib/tcl8.2/encoding/ksc5601.enc
37409 ./lib/tcl8.2/history.tcl
37431 ./bin/lmhostid
37439 ./lib/tk8.2/images/pwrdLogo.eps
37477 ./lib/tcl8.2/encoding/cp1258.enc
37544 ./doc/Whats_New_FV12.pdf
37596 ./uil/solaris/legend_control.uid
37688 ./user/ftn_register_functions.f
38176 ./bin/fvsrvbin
38287 ./fvx_and_restarts/README_PLOT3D_formula_restart
38461 ./demo/f18i9b_g_bin
38461 ./examples/f18/f18i9b_g_bin
38501 ./dore/data/fonts/italict.df
38531 ./examples/rectangular_duct/rect_duct_036.uns
38545 ./doc/Help_Intro.pdf
38574 ./lib/tk8.2/prolog.ps
38666 ./lib/tk8.2/menu.tcl
38861 ./lib/tk8.2/images/pwrdLogo100.gif
38928 ./bin/mpeg2decode
38994 ./lib/tcl8.2/encoding/macTurkish.enc
39109 ./lib/tk8.2/images/pwrdLogo175.gif
39676 ./lib/tcl8.2/encoding/iso2022-jp.enc
39874 ./demo/f18show.scr
40052 ./lib/solaris/liblua.so.4.0
40066 ./user/ftn_register_data_readers.f
40351 ./bin/convert
40442 ./demo/col1.map
40463 ./bin/fv_platform
40541 ./user/register_data_readers.c
40552 ./lib/tcl8.2/encoding/symbol.enc
41107 ./lib/tcl8.2/encoding/cp1256.enc
41169 ./geometry/geometry_formats.ps
41358 ./demo/stream2.stl
41364 ./lib/tk8.2/images/logo100.gif
41448 ./bin/plugins/lib/libcgns.so
41566 ./lib/blt2.4/pkgIndex.tcl
41734 ./demo/iso2.iso
42046 ./lib/tcl8.2/encoding/iso8859-9.enc
42054 ./demo/t9.ttl
42217 ./lib/tcl8.2/encoding/macJapan.enc
42218 ./lib/tcl8.2/encoding/cp437.enc
42270 ./examples/rectangular_duct/rect_duct_026.uns
42627 ./dore/data/fonts/ucmath.df
43241 ./bin/fvsrv64/fvsrvbin64
43278 ./dore/data/fonts/symbols.df
43508 ./lib/tk8.2/images/logo.eps
43701 ./examples/rectangular_duct/rect_duct_041.uns
43751 ./examples/heat_exchanger/cocurrent.uns.fvreg
43751 ./examples/heat_exchanger/counter.uns.fvreg
43946 ./fvx_and_restarts/animate_streamlines.dat
43960 ./user/user_unstruct_grid.f
44519 ./lib/tcl8.2/ldAix
44814 ./examples/rectangular_duct/rect_duct_042.uns
44867 ./demo/start.vct
44875 ./user/user_unstruct_results.f
44878 ./lib/tcl8.2/encoding/jis0201.enc
44926 ./examples/rectangular_duct/rect_duct_037.uns
45085 ./sconfig/srv.template
45438 ./examples/rectangular_duct/rect_duct_032.uns
45442 ./uil/solaris/misc_file_input.uid
45540 ./lib/fv.pyc
45642 ./dore/data/fonts/gothgbt.df
45830 ./demo/iso3.iso
45880 ./lib/tcl8.2/encoding/cp1250.enc
46000 ./examples/rectangular_duct/rect_duct_013.uns
46023 ./lib/tk8.2/images/logoMed.gif
46093 ./demo/view2.vct
46103 ./examples/rectangular_duct/rect_duct_047.uns
46238 ./demo/start.map
46313 ./uil/solaris/misc_support.uid
46455 ./lib/tcl8.2/encoding/cp1255.enc
46496 ./lib/blt2.4/bltHierbox.tcl
46643 ./uil/solaris/boundary_conditions.uid
46844 ./geometry/cone12.atm
47054 ./fvx_and_restarts/seed_sphere_proto.fvx
47266 ./bin/fvbin
47357 ./uns/write_split_unformatted_uns.F
47834 ./X11/XKeysymDB
47891 ./demo/view6.vct
47930 ./demo/drag.fvx
47970 ./lib/tcl8.2/safe.tcl
48023 ./lib/tcl8.2/encoding/cp936.enc
48142 ./lib/blt2.4/bltHiertable.tcl
48202 ./uil/solaris/plot.uid
48291 ./data/colormaps/c7.col
48295 ./fvx_and_restarts/xcolor_defines.fvx
48438 ./examples/rectangular_duct/rect_duct_059.uns
48906 ./dore/data/fonts/greekcs.df
49045 ./fvx_and_restarts/seed_sphere_GUI.fvx
49051 ./demo/view4.vct
49338 ./demo/view3.vct
49417 ./lib/tcl8.2/encoding/ascii.enc
49617 ./demo/start.bnd
49824 ./lib/tcl8.2/encoding/iso8859-3.enc
49896 ./geometry/cube.atm
49920 ./fvx_and_restarts/lsdyna_formulas.frm
49926 ./data/colormaps/c4.col
49932 ./demo/t8.ttl
50129 ./uil/solaris/dynamic_subset.uid
50329 ./demo/view7.vct
50415 ./examples/rectangular_duct/rect_duct_061.uns
50427 ./examples/rectangular_duct/rect_duct_077.uns
50441 ./bin/prolog.ps
50605 ./examples/rectangular_duct/rect_duct_038.uns
50854 ./uil/solaris/vr.uid
51123 ./examples/rectangular_duct/rect_duct_021.uns
51494 ./bin/fv_convert.template
51968 ./examples/rectangular_duct/rect_duct_065.uns
52013 ./lib/tcl8.2/encoding/cp855.enc
52057 ./examples/rectangular_duct/rect_duct_076.uns
52180 ./uil/solaris/dataset_controls.uid
52309 ./lib/tcl8.2/opt0.4/pkgIndex.tcl
52340 ./examples/rectangular_duct/rect_duct_049.uns
52415 ./demo/start.ttl
52745 ./dore/data/fonts/romand.df
52763 ./bin/fvsrv64/plugins/cgns_reader.so
52833 ./lib/tcl8.2/encoding/cp850.enc
52942 ./uil/solaris/keyframe.uid
53221 ./lib/blt2.4/tclIndex
53293 ./examples/rectangular_duct/rect_duct_052.uns
53473 ./lib/tk8.2/comdlg.tcl
53516 ./examples/rectangular_duct/rect_duct_058.uns
53968 ./user/toolkit.c
54002 ./lib/tcl8.2/encoding/cp860.enc
54102 ./fvx_and_restarts/calc_volume_integral.fvx
54228 ./sconfig/manual.srv
54250 ./demo/plot1.lpt
54260 ./demo/plot2.lpt
54331 ./bin/make_fv
54533 ./dore/data/fonts/romancs.df
54905 ./dore/data/fonts/helvetica.df
55047 ./examples/rectangular_duct/rect_duct.trk
55208 ./fvx_and_restarts/Q-criterion.frm
55231 ./lib/tcl8.2/http1.0/pkgIndex.tcl
55361 ./lib/tk8.2/xmfbox.tcl
55552 ./examples/rectangular_duct/rect_duct_074.uns
55614 ./examples/rectangular_duct/rect_duct_045.uns
55673 ./lib/tcl8.2/encoding/cp1251.enc
55709 ./examples/rectangular_duct/rect_duct_075.uns
55734 ./dore/data/fonts/scriptc.df
55885 ./lib/tk8.2/images/logoLarge.gif
56003 ./data/colormaps/c5.col
56186 ./dore/data/fonts/greeks.df
56255 ./examples/rectangular_duct/rect_duct_002.uns
56320 ./demo/t6.ttl
56378 ./examples/rectangular_duct/rect_duct_070.uns
56644 ./uil/solaris/colormap_specification.uid
56695 ./lib/tcl8.2/encoding/iso8859-2.enc
56916 ./lib/blt2.4/bltTabnotebook.tcl
56923 ./bin/plugins/cgns_reader.so
56960 ./dore/data/fonts/greekc.df
57234 ./user/user_unstruct_functions.f
57480 ./demo/aerospace_LDM.frm
57686 ./lib/blt2.4/Problems
57698 ./fvx_and_restarts/FVX_Templates.pdf
57711 ./lib/tk8.2/entry.tcl
57757 ./examples/rectangular_duct/rect_duct_064.uns
57941 ./lib/tcl8.2/encoding/dingbats.enc
58071 ./lib/tcl8.2/opt0.4/optparse.tcl
58217 ./bin/lib/ImageMagick-5.5.6-Q16/type.mgk
58430 ./fvx_and_restarts/animate_streamlines.bnd
58550 ./uil/solaris/formula.uid
58616 ./fvx_and_restarts/plot_along_streamline.fvx
58635 ./demo/start.frm
58764 ./lib/tk8.2/tearoff.tcl
59034 ./lib/solaris/lua-python.so
59310 ./demo/t0.ttl
59343 ./bin/mv_record
59391 ./lib/tcl8.2/encoding/iso8859-4.enc
59497 ./doc/How_to.pdf
59695 ./examples/rectangular_duct/rect_duct_071.uns
60258 ./dore/data/errmsg/errmsgfile
60390 ./bin/plugins/lsdyna_reader.so
60621 ./data/colormaps/c6.col
60693 ./examples/rectangular_duct/rect_duct_079.uns
60841 ./bin/lib/ImageMagick-5.5.6-Q16/type-solaris.mgk
61035 ./lib/blt2.4/bltGraph.pro
61117 ./lib/tcl8.2/word.tcl
61247 ./demo/blank.ttl
61446 ./uns/fv_reader_tags.h
61497 ./dore/data/fonts/music.df
61541 ./lib/tk8.2/console.tcl
61734 ./demo/coord4.crd
61852 ./demo/view5.vct
62390 ./user/register_functions.c
62413 ./demo/t5.ttl
62662 ./demo/t1.ttl
62907 ./demo/stream1.stl
62918 ./fvx_and_restarts/bluntfinx.bin
63021 ./examples/rectangular_duct/rect_duct_017.uns
63102 ./lib/tcl8.2/encoding/macRoman.enc
63106 ./examples/heat_exchanger/cocurrent.uns
63221 ./lib/tk8.2/safetk.tcl
63321 ./bin/ld_fv
63340 ./demo/iso4.iso
63384 ./examples/rectangular_duct/rect_duct_027.uns
63599 ./bin/contest
63671 ./geometry/cone48.atm
63850 ./lib/tcl8.2/encoding/cp1252.enc
64512 ./examples/rectangular_duct/rect_duct_072.uns
64515 ./data/colormaps/c8.col
64986 ./lib/blt2.4/Readme
65154 ./lib/blt2.4/bltCanvEps.pro
65304 ./data/colormaps/c1.col
65467 ./dore/data/fonts/scripts.df
EOF
;;
3)
# checksums for IBM version of fv directory
cat > $checkold <<EOF
00023 ./lib/tk8.2/tkfbox.tcl
00052 ./lib/tcl8.2/encoding/big5.enc
00064 ./lib/tk8.2/text.tcl
00273 ./lib/tcl8.2/encoding/cp1253.enc
00281 ./lib/tcl8.2/encoding/gb12345.enc
00300 ./user/user_results_file_read.f
00328 ./demo/iso1.iso
00349 ./bin/fv_to_printer.sh.sample
00455 ./lib/tcl8.2/encoding/cp857.enc
00685 ./examples/rectangular_duct/rect_duct_044.uns
00715 ./examples/rectangular_duct/rect_duct_048.uns
00795 ./lib/tcl8.2/encoding/jis0212.enc
01283 ./examples/rectangular_duct/rect_duct_029.uns
01383 ./lib/tk8.2/button.tcl
01478 ./fvx_and_restarts/animate_streamlines.map
01578 ./bin/display
01677 ./demo/f18i9b_g_bin.fvbnd
01677 ./examples/f18/f18i9b_g_bin.fvbnd
02009 ./examples/rectangular_duct/rect_duct_024.uns
02213 ./examples/rectangular_duct/rect_duct_012.uns
02225 ./fvx_and_restarts/plot3d_eqns.frm
02235 ./lib/tk8.2/msgbox.tcl
02266 ./lib/tcl8.2/encoding/cp949.enc
02317 ./data/xdb_vars
02325 ./lib/blt2.4/dd_protocols/dd-number.tcl
02345 ./user/user_grid_file_read.f
02492 ./examples/rectangular_duct/rect_duct_062.uns
02539 ./uil/ibm_rs/vortex_core.uid
02682 ./doc/Readme_InstallNotes_12.pdf
02722 ./data/colormaps/c3.col
02780 ./uns/write_split_ascii_uns.f
02844 ./examples/rectangular_duct/rect_duct_067.uns
03030 ./fvx_and_restarts/bluntfinq.bin
03294 ./lib/blt2.4/bltDragdrop.tcl
03299 ./examples/rectangular_duct/rect_duct_030.uns
03321 ./demo/coord3.crd
03329 ./fvx_and_restarts/set_view_TEST.fvx
03550 ./bin/mpeg2decode
03592 ./dore/data/il/def/parsekey.sym
03816 ./geometry/cyln48.atm
04150 ./fvx_and_restarts/seed_rectangle_proto.fvx
04449 ./examples/rectangular_duct/rect_duct_039.uns
04525 ./fvx_and_restarts/seed_circ_GUI.fvx
04824 ./lib/tcl8.2/encoding/macThai.enc
04999 ./lib/tcl8.2/encoding/macIceland.enc
05048 ./examples/rectangular_duct/rect_duct_040.uns
05077 ./lib/tcl8.2/http2.1/pkgIndex.tcl
05136 ./lib/tcl8.2/tcltest1.0/pkgIndex.tcl
05176 ./lib/tcl8.2/encoding/koi8-r.enc
05986 ./demo/aerospace_LDM.bnd
06058 ./demo/f18i9b_q_bin
06058 ./examples/f18/f18i9b_q_bin
06106 ./lib/tcl8.2/encoding/cp737.enc
06139 ./dore/data/fonts/romant.df
06194 ./lib/tcl8.2/encoding/gb2312.enc
06517 ./bin/fvsrv
06966 ./lib/tcl8.2/encoding/iso8859-8.enc
07340 ./geometry/cyln12.atm
07365 ./examples/rectangular_duct/rect_duct_016.uns
07605 ./demo/stream4.stl
07614 ./demo/t3.ttl
07885 ./bin/fv.template
08136 ./lib/tcl8.2/encoding/euc-jp.enc
08160 ./lib/tcl8.2/encoding/cp1254.enc
08221 ./demo/t.ttl
08221 ./demo/t10.ttl
08320 ./examples/rectangular_duct/rect_duct_043.uns
08837 ./uil/ibm_rs/dataset_controls.uid
08917 ./user/user_defined_functions.f
09118 ./lib/tcl8.2/encoding/cp862.enc
09161 ./examples/rectangular_duct/rect_duct_018.uns
09510 ./examples/rectangular_duct/rect_duct_015.uns
09563 ./examples/rectangular_duct/rect_duct_068.uns
09914 ./uil/ibm_rs/keyframe.uid
10046 ./demo/lift.fvx
10063 ./bin/lib/ImageMagick-5.5.6-Q16/log.mgk
10314 ./lib/tk8.2/obsolete.tcl
10394 ./examples/rectangular_duct/rect_duct_073.uns
10827 ./lib/tcl8.2/http2.1/http.tcl
10855 ./lib/tk8.2/bgerror.tcl
10967 ./doc/FVReference_Manual.pdf
11013 ./examples/rectangular_duct/rect_duct_055.uns
11073 ./bin/convert
11200 ./dore/data/fonts/romans.df
11396 ./examples/rectangular_duct/rect_duct_011.uns
11412 ./lib/tcl8.2/encoding/cp864.enc
11868 ./lib/tcl8.2/tclIndex
11903 ./lib/tcl8.2/encoding/euc-cn.enc
11963 ./dore/data/fonts/italicc.df
12263 ./dore/data/fonts/cyrillic.df
12397 ./demo/t11.ttl
12721 ./examples/rectangular_duct/rect_duct_028.uns
12833 ./lib/tcl8.2/tcltest1.0/tcltest.tcl
13046 ./uil/ibm_rs/transient_datav2.uid
13082 ./fvx_and_restarts/set_view.fvx
13258 ./lib/tcl8.2/encoding/euc-kr.enc
13293 ./fvx_and_restarts/bio.frm
13398 ./demo/t2.ttl
13461 ./examples/rectangular_duct/rect_duct_009.uns
13652 ./examples/rectangular_duct/rect_duct_078.uns
13726 ./bin/lib/ImageMagick-5.5.6-Q16/delegates.mgk
13935 ./uil/ibm_rs/integration.uid
14117 ./lib/blt2.4/NEWS
14544 ./dore/data/fonts/greekp.df
14611 ./geometry/allobs.ob
15065 ./examples/rectangular_duct/rect_duct_033.uns
15605 ./lib/tcl8.2/encoding/macCentEuro.enc
15825 ./lib/tcl8.2/encoding/macCroatian.enc
15887 ./lib/tcl8.2/encoding/macUkraine.enc
16097 ./lib/tcl8.2/encoding/iso8859-6.enc
16373 ./examples/rectangular_duct/rect_duct_008.uns
16561 ./lib/tk8.2/scrlbar.tcl
16650 ./geometry/poly12.atm
16658 ./fvx_and_restarts/seed_circular_proto.fvx
16709 ./lib/tcl8.2/msgcat1.0/pkgIndex.tcl
16741 ./lib/tcl8.2/parray.tcl
16795 ./dore/data/fonts/gothgrt.df
16841 ./lib/tcl8.2/encoding/cp950.enc
16950 ./fvx_and_restarts/animate_streamlines.prd
17116 ./lib/tcl8.2/auto.tcl
17184 ./demo/start.plt
17317 ./demo/start.dat
17332 ./demo/bound1.bnd
17427 ./lib/tcl8.2/encoding/cp775.enc
17570 ./lib/tcl8.2/encoding/iso2022.enc
17635 ./examples/rectangular_duct/rect_duct_031.uns
17647 ./geometry/poly48.atm
17745 ./uil/ibm_rs/formula.uid
17790 ./lib/tcl8.2/encoding/iso8859-7.enc
17844 ./lib/tcl8.2/encoding/cp932.enc
17920 ./lib/tcl8.2/msgcat1.0/msgcat.tcl
18023 ./examples/rectangular_duct/rect_duct_020.uns
18271 ./lib/tcl8.2/encoding/cp863.enc
18315 ./bin/lmhostid
18315 ./lib/tcl8.2/encoding/cp869.enc
18328 ./examples/rectangular_duct/rect_duct_019.uns
18422 ./bin/fv_user_print.sh.sample
18484 ./fvx_and_restarts/animate_streamlines.fvx
18619 ./lib/tcl8.2/encoding/macDingbats.enc
18713 ./dore/data/fonts/gothitt.df
19151 ./demo/aerospace_LDM.fvx
19329 ./lib/tcl8.2/encoding/macGreek.enc
19500 ./lib/tcl8.2/encoding/jis0208.enc
19545 ./lib/tcl8.2/encoding/cp861.enc
19753 ./uil/ibm_rs/region_control.uid
20094 ./examples/rectangular_duct/rect_duct_069.uns
20144 ./examples/rectangular_duct/rect_duct_056.uns
20242 ./bin/lib/ImageMagick-5.5.6-Q16/modules.mgk
20418 ./examples/rectangular_duct/rect_duct_007.uns
20545 ./examples/rectangular_duct/rect_duct_060.uns
20873 ./dore/data/fonts/unused.df
21460 ./lib/tcl8.2/http1.0/http.tcl
21496 ./data/xdb_vars_FV11.0
21564 ./lib/tcl8.2/ldAout.tcl
21798 ./demo/t7.ttl
21844 ./lib/ibm_rs/dummy/libGLU.a
21894 ./lib/blt2.4/bltTabset.tcl
21960 ./dore/data/fonts/astrology.df
21969 ./examples/heat_exchanger/counter.uns
22040 ./uns/write_unformatted_uns.F
22438 ./lib/tcl8.2/encoding/cp1257.enc
22569 ./examples/rectangular_duct/rect_duct_035.uns
22765 ./data/license.dat
23217 ./demo/wingloadplots.fvx
23286 ./dore/data/fonts/italiccs.df
23323 ./demo/coord1.crd
23344 ./uns/write_ascii_uns.f
23526 ./lib/tk8.2/scale.tcl
23533 ./lib/tcl8.2/encoding/iso8859-5.enc
23541 ./examples/rectangular_duct/rect_duct_005.uns
23709 ./lib/tk8.2/images/logo64.gif
23752 ./lib/tk8.2/dialog.tcl
24188 ./examples/rectangular_duct/rect_duct_001.uns
24246 ./bin/fvsrvbin
24661 ./examples/rectangular_duct/rect_duct_010.uns
24673 ./data/colormaps/c2.col
24852 ./demo/aerospace_LDM.dat
24881 ./lib/blt2.4/htresize.cur
25189 ./lib/tk8.2/tk.tcl
25502 ./examples/rectangular_duct/rect_duct_054.uns
25598 ./user/user_unstruct_combined.f
25605 ./lib/tcl8.2/encoding/macCyrillic.enc
25685 ./uil/ibm_rs/misc_file_input.uid
25703 ./lib/tk8.2/tclIndex
26148 ./user/user_combined_file_read.f
26216 ./examples/rectangular_duct/rect_duct_057.uns
26345 ./bin/lib/ImageMagick-5.5.6-Q16/magic.mgk
26356 ./lib/tk8.2/images/README
26380 ./lib/tk8.2/focus.tcl
26668 ./geometry/sph24.atm
26782 ./bin/animate
27073 ./fvx_and_restarts/seed_rect_GUI.fvx
27081 ./lib/blt2.4/bltGraph.tcl
27188 ./lib/tk8.2/images/pwrdLogo200.gif
27513 ./geometry/star.atm
27920 ./bin/lib/ImageMagick-5.5.6-Q16/type-ghostscript.mgk
28042 ./demo/coord0.crd
28310 ./examples/rectangular_duct/rect_duct_003.uns
28324 ./uns/write_split_binary_uns.c
28358 ./lib/blt2.4/dd_protocols/dd-file.tcl
28436 ./user/ftn_fv_close.f
28494 ./uil/ibm_rs/vector_params.uid
28601 ./demo/coord5.crd
28632 ./examples/rectangular_duct/rect_duct_034.uns
28734 ./lib/blt2.4/dd_protocols/dd-color.tcl
28893 ./examples/rectangular_duct/rect_duct_053.uns
28999 ./uil/ibm_rs/point_probe.uid
29100 ./lib/tk8.2/listbox.tcl
29194 ./lib/tk8.2/images/pwrdLogo75.gif
29195 ./examples/rectangular_duct/rect_duct_046.uns
29424 ./bin/lib/ImageMagick-5.5.6-Q16/colors.mgk
29488 ./examples/rectangular_duct/rect_duct_066.uns
29543 ./lib/tcl8.2/encoding/macRomania.enc
29643 ./fvx_and_restarts/animate_streamlines.vct
29657 ./demo/stream3.stl
29787 ./lib/tk8.2/optMenu.tcl
29826 ./lib/tcl8.2/init.tcl
29845 ./examples/rectangular_duct/rect_duct_023.uns
29936 ./examples/rectangular_duct/rect_duct_014.uns
29942 ./lib/tcl8.2/package.tcl
30113 ./lib/tk8.2/palette.tcl
30177 ./dore/data/fonts/meteorology.df
30254 ./lib/blt2.4/bltDnd.tcl
30349 ./lib/tk8.2/images/tai-ku.gif
30506 ./lib/tcl8.2/encoding/iso2022-kr.enc
30792 ./dore/data/fonts/romanc.df
30906 ./uil/ibm_rs/datasetBrowser.uid
30978 ./bin/mv_reset
31089 ./lib/tcl8.2/encoding/iso8859-1.enc
31428 ./bin/lib/ImageMagick-5.5.6-Q16/type-windows.mgk
31591 ./lib/tcl8.2/encoding/cp866.enc
31654 ./bin/fv_platform
31691 ./lib/tcl8.2/encoding/cp865.enc
32060 ./examples/rectangular_duct/rect_duct_025.uns
32217 ./examples/rectangular_duct/rect_duct_006.uns
32492 ./lib/tcl8.2/encoding/cp852.enc
32548 ./demo/aerospace_LDM.vct
32643 ./fvx_and_restarts/basic_volume_integral.fvx
32649 ./examples/rectangular_duct/rect_duct_051.uns
32725 ./examples/rectangular_duct/rect_duct_022.uns
32837 ./examples/rectangular_duct/rect_duct_050.uns
32927 ./lib/tk8.2/tkAppInit.c
33001 ./lib/tcl8.2/encoding/cp874.enc
33248 ./demo/moments.fvx
33321 ./lib/blt2.4/dd_protocols/dd-text.tcl
33396 ./demo/coord2.crd
33520 ./doc/Readme_Server_Toolkit.txt
33547 ./uns/write_binary_uns.c
33827 ./bin/node_id.il
34182 ./lib/blt2.4/dd_protocols/tclIndex
34626 ./geometry/sph12.atm
34669 ./lib/tcl8.2/encoding/gb1988.enc
34710 ./fvx_and_restarts/wind.frm
34959 ./dore/data/fonts/lcmath.df
35021 ./fvx_and_restarts/animate_streamlines.frm
35313 ./dore/data/fonts/romanp.df
35384 ./geometry/square.atm
35398 ./demo/t4.ttl
35629 ./bin/README_Copyrights.txt
35670 ./doc/FVUsers_Guide.pdf
35693 ./uil/ibm_rs/dynamic_subset.uid
35707 ./lib/tcl8.2/encoding/shiftjis.enc
35818 ./examples/rectangular_duct/rect_duct_063.uns
35841 ./bin/node_id
36318 ./examples/rectangular_duct/rect_duct_080.uns
36382 ./uil/ibm_rs/flip.uid
36575 ./fvx_and_restarts/bluntfinx.bin.fvbnd
36641 ./lib/tk8.2/images/pwrdLogo150.gif
36865 ./uns/ftn_fv_reader_tags.h
36925 ./examples/rectangular_duct/rect_duct_004.uns
37055 ./lib/tcl8.2/tclAppInit.c
37176 ./uil/ibm_rs/presentation.uid
37272 ./lib/tk8.2/clrpick.tcl
37307 ./lib/tcl8.2/encoding/ksc5601.enc
37409 ./lib/tcl8.2/history.tcl
37439 ./lib/tk8.2/images/pwrdLogo.eps
37477 ./lib/tcl8.2/encoding/cp1258.enc
37544 ./doc/Whats_New_FV12.pdf
37549 ./uil/ibm_rs/iso_surface.uid
37688 ./user/ftn_register_functions.f
38287 ./fvx_and_restarts/README_PLOT3D_formula_restart
38331 ./uil/ibm_rs/xform_controls.uid
38461 ./demo/f18i9b_g_bin
38461 ./examples/f18/f18i9b_g_bin
38501 ./dore/data/fonts/italict.df
38531 ./examples/rectangular_duct/rect_duct_036.uns
38545 ./doc/Help_Intro.pdf
38574 ./lib/tk8.2/prolog.ps
38666 ./lib/tk8.2/menu.tcl
38782 ./uil/ibm_rs/coordinate_surface.uid
38861 ./lib/tk8.2/images/pwrdLogo100.gif
38994 ./lib/tcl8.2/encoding/macTurkish.enc
39109 ./lib/tk8.2/images/pwrdLogo175.gif
39300 ./uil/ibm_rs/data_input.uid
39346 ./uil/ibm_rs/boundary_conditions.uid
39530 ./bin/plugins/cgns_reader.so
39676 ./lib/tcl8.2/encoding/iso2022-jp.enc
39874 ./demo/f18show.scr
40066 ./user/ftn_register_data_readers.f
40442 ./demo/col1.map
40541 ./user/register_data_readers.c
40552 ./lib/tcl8.2/encoding/symbol.enc
41107 ./lib/tcl8.2/encoding/cp1256.enc
41169 ./geometry/geometry_formats.ps
41224 ./uil/ibm_rs/title_panel.uid
41358 ./demo/stream2.stl
41364 ./lib/tk8.2/images/logo100.gif
41566 ./lib/blt2.4/pkgIndex.tcl
41734 ./demo/iso2.iso
42046 ./lib/tcl8.2/encoding/iso8859-9.enc
42054 ./demo/t9.ttl
42217 ./lib/tcl8.2/encoding/macJapan.enc
42218 ./lib/tcl8.2/encoding/cp437.enc
42270 ./examples/rectangular_duct/rect_duct_026.uns
42627 ./dore/data/fonts/ucmath.df
43278 ./dore/data/fonts/symbols.df
43508 ./lib/tk8.2/images/logo.eps
43701 ./examples/rectangular_duct/rect_duct_041.uns
43751 ./examples/heat_exchanger/cocurrent.uns.fvreg
43751 ./examples/heat_exchanger/counter.uns.fvreg
43946 ./fvx_and_restarts/animate_streamlines.dat
43960 ./user/user_unstruct_grid.f
44519 ./lib/tcl8.2/ldAix
44567 ./uil/ibm_rs/func.uid
44814 ./examples/rectangular_duct/rect_duct_042.uns
44867 ./demo/start.vct
44875 ./user/user_unstruct_results.f
44878 ./lib/tcl8.2/encoding/jis0201.enc
44926 ./examples/rectangular_duct/rect_duct_037.uns
44952 ./bin/mpeg2encode
45085 ./sconfig/srv.template
45137 ./uil/ibm_rs/particle_paths.uid
45438 ./examples/rectangular_duct/rect_duct_032.uns
45642 ./dore/data/fonts/gothgbt.df
45830 ./demo/iso3.iso
45880 ./lib/tcl8.2/encoding/cp1250.enc
46000 ./examples/rectangular_duct/rect_duct_013.uns
46023 ./lib/tk8.2/images/logoMed.gif
46093 ./demo/view2.vct
46103 ./examples/rectangular_duct/rect_duct_047.uns
46238 ./demo/start.map
46455 ./lib/tcl8.2/encoding/cp1255.enc
46496 ./lib/blt2.4/bltHierbox.tcl
46844 ./geometry/cone12.atm
47054 ./fvx_and_restarts/seed_sphere_proto.fvx
47357 ./uns/write_split_unformatted_uns.F
47891 ./demo/view6.vct
47930 ./demo/drag.fvx
47970 ./lib/tcl8.2/safe.tcl
48023 ./lib/tcl8.2/encoding/cp936.enc
48079 ./uil/ibm_rs/streamlines.uid
48142 ./lib/blt2.4/bltHiertable.tcl
48291 ./data/colormaps/c7.col
48295 ./fvx_and_restarts/xcolor_defines.fvx
48403 ./uil/ibm_rs/subset_params.uid
48438 ./examples/rectangular_duct/rect_duct_059.uns
48651 ./bin/contest
48906 ./dore/data/fonts/greekcs.df
49045 ./fvx_and_restarts/seed_sphere_GUI.fvx
49051 ./demo/view4.vct
49338 ./demo/view3.vct
49417 ./lib/tcl8.2/encoding/ascii.enc
49617 ./demo/start.bnd
49824 ./lib/tcl8.2/encoding/iso8859-3.enc
49896 ./geometry/cube.atm
49920 ./fvx_and_restarts/lsdyna_formulas.frm
49926 ./data/colormaps/c4.col
49932 ./demo/t8.ttl
50092 ./bin/mv_position
50329 ./demo/view7.vct
50415 ./examples/rectangular_duct/rect_duct_061.uns
50427 ./examples/rectangular_duct/rect_duct_077.uns
50441 ./bin/prolog.ps
50605 ./examples/rectangular_duct/rect_duct_038.uns
51123 ./examples/rectangular_duct/rect_duct_021.uns
51494 ./bin/fv_convert.template
51968 ./examples/rectangular_duct/rect_duct_065.uns
52013 ./lib/tcl8.2/encoding/cp855.enc
52057 ./examples/rectangular_duct/rect_duct_076.uns
52153 ./uil/ibm_rs/surface_plot.uid
52309 ./lib/tcl8.2/opt0.4/pkgIndex.tcl
52340 ./examples/rectangular_duct/rect_duct_049.uns
52394 ./uil/ibm_rs/legend_control.uid
52415 ./demo/start.ttl
52745 ./dore/data/fonts/romand.df
52833 ./lib/tcl8.2/encoding/cp850.enc
52928 ./uil/ibm_rs/colormap_specification.uid
53221 ./lib/blt2.4/tclIndex
53293 ./examples/rectangular_duct/rect_duct_052.uns
53473 ./lib/tk8.2/comdlg.tcl
53516 ./examples/rectangular_duct/rect_duct_058.uns
53968 ./user/toolkit.c
54002 ./lib/tcl8.2/encoding/cp860.enc
54102 ./fvx_and_restarts/calc_volume_integral.fvx
54228 ./sconfig/manual.srv
54250 ./demo/plot1.lpt
54260 ./demo/plot2.lpt
54331 ./bin/make_fv
54533 ./dore/data/fonts/romancs.df
54905 ./dore/data/fonts/helvetica.df
55021 ./uil/ibm_rs/xdb_info.uid
55047 ./examples/rectangular_duct/rect_duct.trk
55208 ./fvx_and_restarts/Q-criterion.frm
55231 ./lib/tcl8.2/http1.0/pkgIndex.tcl
55361 ./lib/tk8.2/xmfbox.tcl
55552 ./examples/rectangular_duct/rect_duct_074.uns
55614 ./examples/rectangular_duct/rect_duct_045.uns
55673 ./lib/tcl8.2/encoding/cp1251.enc
55709 ./examples/rectangular_duct/rect_duct_075.uns
55734 ./dore/data/fonts/scriptc.df
55885 ./lib/tk8.2/images/logoLarge.gif
56003 ./data/colormaps/c5.col
56118 ./bin/mv_init_tape
56186 ./dore/data/fonts/greeks.df
56255 ./examples/rectangular_duct/rect_duct_002.uns
56320 ./demo/t6.ttl
56378 ./examples/rectangular_duct/rect_duct_070.uns
56677 ./uil/ibm_rs/vr.uid
56695 ./lib/tcl8.2/encoding/iso8859-2.enc
56916 ./lib/blt2.4/bltTabnotebook.tcl
56960 ./dore/data/fonts/greekc.df
57234 ./user/user_unstruct_functions.f
57480 ./demo/aerospace_LDM.frm
57686 ./lib/blt2.4/Problems
57698 ./fvx_and_restarts/FVX_Templates.pdf
57711 ./lib/tk8.2/entry.tcl
57757 ./examples/rectangular_duct/rect_duct_064.uns
57941 ./lib/tcl8.2/encoding/dingbats.enc
58071 ./lib/tcl8.2/opt0.4/optparse.tcl
58217 ./bin/lib/ImageMagick-5.5.6-Q16/type.mgk
58430 ./fvx_and_restarts/animate_streamlines.bnd
58437 ./uil/ibm_rs/comp_surface.uid
58616 ./fvx_and_restarts/plot_along_streamline.fvx
58635 ./demo/start.frm
58764 ./lib/tk8.2/tearoff.tcl
59310 ./demo/t0.ttl
59391 ./lib/tcl8.2/encoding/iso8859-4.enc
59440 ./bin/fvbin
59497 ./doc/How_to.pdf
59695 ./examples/rectangular_duct/rect_duct_071.uns
60232 ./bin/mv_record
60258 ./dore/data/errmsg/errmsgfile
60416 ./uil/ibm_rs/color_mixer.uid
60621 ./data/colormaps/c6.col
60693 ./examples/rectangular_duct/rect_duct_079.uns
60841 ./bin/lib/ImageMagick-5.5.6-Q16/type-solaris.mgk
61035 ./lib/blt2.4/bltGraph.pro
61117 ./lib/tcl8.2/word.tcl
61247 ./demo/blank.ttl
61397 ./lib/ibm_rs/dummy/libGL.a
61446 ./uns/fv_reader_tags.h
61497 ./dore/data/fonts/music.df
61541 ./lib/tk8.2/console.tcl
61734 ./demo/coord4.crd
61852 ./demo/view5.vct
62315 ./uil/ibm_rs/misc_support.uid
62390 ./user/register_functions.c
62413 ./demo/t5.ttl
62662 ./demo/t1.ttl
62907 ./demo/stream1.stl
62918 ./fvx_and_restarts/bluntfinx.bin
63021 ./examples/rectangular_duct/rect_duct_017.uns
63102 ./lib/tcl8.2/encoding/macRoman.enc
63106 ./examples/heat_exchanger/cocurrent.uns
63221 ./lib/tk8.2/safetk.tcl
63321 ./bin/ld_fv
63340 ./demo/iso4.iso
63384 ./examples/rectangular_duct/rect_duct_027.uns
63671 ./geometry/cone48.atm
63850 ./lib/tcl8.2/encoding/cp1252.enc
64052 ./bin/server_name
64512 ./examples/rectangular_duct/rect_duct_072.uns
64515 ./data/colormaps/c8.col
64986 ./lib/blt2.4/Readme
65154 ./lib/blt2.4/bltCanvEps.pro
65257 ./uil/ibm_rs/plot.uid
65304 ./data/colormaps/c1.col
65356 ./bin/plugins/lsdyna_reader.so
65363 ./bin/plugins/fluent_reader.so
65467 ./dore/data/fonts/scripts.df
EOF
;;
4)
# checksums for HP version of fv directory
cat > $checkold <<EOF
00023 ./lib/tk8.2/tkfbox.tcl
00052 ./lib/tcl8.2/encoding/big5.enc
00064 ./lib/tk8.2/text.tcl
00273 ./lib/tcl8.2/encoding/cp1253.enc
00281 ./lib/tcl8.2/encoding/gb12345.enc
00300 ./user/user_results_file_read.f
00328 ./demo/iso1.iso
00349 ./bin/fv_to_printer.sh.sample
00455 ./lib/tcl8.2/encoding/cp857.enc
00685 ./examples/rectangular_duct/rect_duct_044.uns
00715 ./examples/rectangular_duct/rect_duct_048.uns
00795 ./lib/tcl8.2/encoding/jis0212.enc
01283 ./examples/rectangular_duct/rect_duct_029.uns
01383 ./lib/tk8.2/button.tcl
01478 ./fvx_and_restarts/animate_streamlines.map
01677 ./demo/f18i9b_g_bin.fvbnd
01677 ./examples/f18/f18i9b_g_bin.fvbnd
01903 ./bin/fv_platform
02009 ./examples/rectangular_duct/rect_duct_024.uns
02213 ./examples/rectangular_duct/rect_duct_012.uns
02225 ./fvx_and_restarts/plot3d_eqns.frm
02235 ./lib/tk8.2/msgbox.tcl
02266 ./lib/tcl8.2/encoding/cp949.enc
02317 ./data/xdb_vars
02325 ./lib/blt2.4/dd_protocols/dd-number.tcl
02345 ./user/user_grid_file_read.f
02492 ./examples/rectangular_duct/rect_duct_062.uns
02596 ./bin/fvsrv64/fvsrvbin64
02618 ./bin/contest
02682 ./doc/Readme_InstallNotes_12.pdf
02722 ./data/colormaps/c3.col
02780 ./uns/write_split_ascii_uns.f
02844 ./examples/rectangular_duct/rect_duct_067.uns
03030 ./fvx_and_restarts/bluntfinq.bin
03294 ./lib/blt2.4/bltDragdrop.tcl
03299 ./examples/rectangular_duct/rect_duct_030.uns
03321 ./demo/coord3.crd
03329 ./fvx_and_restarts/set_view_TEST.fvx
03451 ./uil/hp_700/iso_surface.uid
03592 ./dore/data/il/def/parsekey.sym
03816 ./geometry/cyln48.atm
04075 ./bin/display
04150 ./fvx_and_restarts/seed_rectangle_proto.fvx
04449 ./examples/rectangular_duct/rect_duct_039.uns
04525 ./fvx_and_restarts/seed_circ_GUI.fvx
04824 ./lib/tcl8.2/encoding/macThai.enc
04999 ./lib/tcl8.2/encoding/macIceland.enc
05048 ./examples/rectangular_duct/rect_duct_040.uns
05077 ./lib/tcl8.2/http2.1/pkgIndex.tcl
05136 ./lib/tcl8.2/tcltest1.0/pkgIndex.tcl
05176 ./lib/tcl8.2/encoding/koi8-r.enc
05986 ./demo/aerospace_LDM.bnd
06058 ./demo/f18i9b_q_bin
06058 ./examples/f18/f18i9b_q_bin
06106 ./lib/tcl8.2/encoding/cp737.enc
06139 ./dore/data/fonts/romant.df
06194 ./lib/tcl8.2/encoding/gb2312.enc
06517 ./bin/fvsrv
06806 ./uil/hp_700/misc_support.uid
06966 ./lib/tcl8.2/encoding/iso8859-8.enc
07125 ./uil/hp_700/dataset_controls.uid
07338 ./bin/stereo_hp
07340 ./geometry/cyln12.atm
07365 ./examples/rectangular_duct/rect_duct_016.uns
07605 ./demo/stream4.stl
07614 ./demo/t3.ttl
07885 ./bin/fv.template
08136 ./lib/tcl8.2/encoding/euc-jp.enc
08160 ./lib/tcl8.2/encoding/cp1254.enc
08221 ./demo/t.ttl
08221 ./demo/t10.ttl
08320 ./examples/rectangular_duct/rect_duct_043.uns
08437 ./bin/mv_position
08917 ./user/user_defined_functions.f
08972 ./bin/vnc_disp
09118 ./lib/tcl8.2/encoding/cp862.enc
09161 ./examples/rectangular_duct/rect_duct_018.uns
09510 ./examples/rectangular_duct/rect_duct_015.uns
09563 ./examples/rectangular_duct/rect_duct_068.uns
10046 ./demo/lift.fvx
10063 ./bin/lib/ImageMagick-5.5.6-Q16/log.mgk
10314 ./lib/tk8.2/obsolete.tcl
10394 ./examples/rectangular_duct/rect_duct_073.uns
10827 ./lib/tcl8.2/http2.1/http.tcl
10855 ./lib/tk8.2/bgerror.tcl
10967 ./doc/FVReference_Manual.pdf
10973 ./bin/mpeg2encode
11013 ./examples/rectangular_duct/rect_duct_055.uns
11200 ./dore/data/fonts/romans.df
11203 ./uil/hp_700/func.uid
11274 ./bin/fvbin
11396 ./examples/rectangular_duct/rect_duct_011.uns
11412 ./lib/tcl8.2/encoding/cp864.enc
11868 ./lib/tcl8.2/tclIndex
11903 ./lib/tcl8.2/encoding/euc-cn.enc
11963 ./dore/data/fonts/italicc.df
12263 ./dore/data/fonts/cyrillic.df
12397 ./demo/t11.ttl
12501 ./bin/fv_vnc
12721 ./examples/rectangular_duct/rect_duct_028.uns
12833 ./lib/tcl8.2/tcltest1.0/tcltest.tcl
13082 ./fvx_and_restarts/set_view.fvx
13258 ./lib/tcl8.2/encoding/euc-kr.enc
13293 ./fvx_and_restarts/bio.frm
13398 ./demo/t2.ttl
13461 ./examples/rectangular_duct/rect_duct_009.uns
13652 ./examples/rectangular_duct/rect_duct_078.uns
13715 ./bin/fvsrv64/plugins/fluent_reader.so
13726 ./bin/lib/ImageMagick-5.5.6-Q16/delegates.mgk
14117 ./lib/blt2.4/NEWS
14204 ./bin/fvsrvbin
14434 ./uil/hp_700/region_control.uid
14544 ./dore/data/fonts/greekp.df
14611 ./geometry/allobs.ob
15059 ./uil/hp_700/coordinate_surface.uid
15065 ./examples/rectangular_duct/rect_duct_033.uns
15123 ./uil/hp_700/keyframe.uid
15605 ./lib/tcl8.2/encoding/macCentEuro.enc
15825 ./lib/tcl8.2/encoding/macCroatian.enc
15887 ./lib/tcl8.2/encoding/macUkraine.enc
16097 ./lib/tcl8.2/encoding/iso8859-6.enc
16373 ./examples/rectangular_duct/rect_duct_008.uns
16561 ./lib/tk8.2/scrlbar.tcl
16650 ./geometry/poly12.atm
16658 ./fvx_and_restarts/seed_circular_proto.fvx
16709 ./lib/tcl8.2/msgcat1.0/pkgIndex.tcl
16741 ./lib/tcl8.2/parray.tcl
16795 ./dore/data/fonts/gothgrt.df
16841 ./lib/tcl8.2/encoding/cp950.enc
16950 ./fvx_and_restarts/animate_streamlines.prd
17028 ./bin/fvsrv64/fvsrv64
17116 ./lib/tcl8.2/auto.tcl
17153 ./lib/start_python.luac64
17184 ./demo/start.plt
17317 ./demo/start.dat
17332 ./demo/bound1.bnd
17427 ./lib/tcl8.2/encoding/cp775.enc
17570 ./lib/tcl8.2/encoding/iso2022.enc
17635 ./examples/rectangular_duct/rect_duct_031.uns
17647 ./geometry/poly48.atm
17790 ./lib/tcl8.2/encoding/iso8859-7.enc
17844 ./lib/tcl8.2/encoding/cp932.enc
17920 ./lib/tcl8.2/msgcat1.0/msgcat.tcl
18023 ./examples/rectangular_duct/rect_duct_020.uns
18271 ./lib/tcl8.2/encoding/cp863.enc
18315 ./lib/tcl8.2/encoding/cp869.enc
18328 ./examples/rectangular_duct/rect_duct_019.uns
18422 ./bin/fv_user_print.sh.sample
18484 ./fvx_and_restarts/animate_streamlines.fvx
18619 ./lib/tcl8.2/encoding/macDingbats.enc
18713 ./dore/data/fonts/gothitt.df
18890 ./uil/hp_700/vector_params.uid
19151 ./demo/aerospace_LDM.fvx
19329 ./lib/tcl8.2/encoding/macGreek.enc
19500 ./lib/tcl8.2/encoding/jis0208.enc
19545 ./lib/tcl8.2/encoding/cp861.enc
19638 ./bin/plugins/lsdyna_reader.so
19668 ./uil/hp_700/streamlines.uid
19689 ./uil/hp_700/plot.uid
19776 ./lib/start_python.luac
19890 ./lib/hp_700/dummy/libGLU.1
20094 ./examples/rectangular_duct/rect_duct_069.uns
20144 ./examples/rectangular_duct/rect_duct_056.uns
20242 ./bin/lib/ImageMagick-5.5.6-Q16/modules.mgk
20418 ./examples/rectangular_duct/rect_duct_007.uns
20545 ./examples/rectangular_duct/rect_duct_060.uns
20873 ./dore/data/fonts/unused.df
21460 ./lib/tcl8.2/http1.0/http.tcl
21496 ./data/xdb_vars_FV11.0
21564 ./lib/tcl8.2/ldAout.tcl
21798 ./demo/t7.ttl
21894 ./lib/blt2.4/bltTabset.tcl
21960 ./dore/data/fonts/astrology.df
21969 ./examples/heat_exchanger/counter.uns
22040 ./uns/write_unformatted_uns.F
22438 ./lib/tcl8.2/encoding/cp1257.enc
22569 ./examples/rectangular_duct/rect_duct_035.uns
22765 ./data/license.dat
23217 ./demo/wingloadplots.fvx
23286 ./dore/data/fonts/italiccs.df
23323 ./demo/coord1.crd
23344 ./uns/write_ascii_uns.f
23526 ./lib/tk8.2/scale.tcl
23533 ./lib/tcl8.2/encoding/iso8859-5.enc
23541 ./examples/rectangular_duct/rect_duct_005.uns
23709 ./lib/tk8.2/images/logo64.gif
23752 ./lib/tk8.2/dialog.tcl
24130 ./uil/hp_700/vortex_core.uid
24188 ./examples/rectangular_duct/rect_duct_001.uns
24661 ./examples/rectangular_duct/rect_duct_010.uns
24673 ./data/colormaps/c2.col
24852 ./demo/aerospace_LDM.dat
24881 ./lib/blt2.4/htresize.cur
25189 ./lib/tk8.2/tk.tcl
25502 ./examples/rectangular_duct/rect_duct_054.uns
25598 ./user/user_unstruct_combined.f
25605 ./lib/tcl8.2/encoding/macCyrillic.enc
25703 ./lib/tk8.2/tclIndex
25807 ./uil/hp_700/vr.uid
26148 ./user/user_combined_file_read.f
26216 ./examples/rectangular_duct/rect_duct_057.uns
26345 ./bin/lib/ImageMagick-5.5.6-Q16/magic.mgk
26356 ./lib/tk8.2/images/README
26380 ./lib/tk8.2/focus.tcl
26668 ./geometry/sph24.atm
26809 ./bin/server_name
27073 ./fvx_and_restarts/seed_rect_GUI.fvx
27081 ./lib/blt2.4/bltGraph.tcl
27188 ./lib/tk8.2/images/pwrdLogo200.gif
27513 ./geometry/star.atm
27920 ./bin/lib/ImageMagick-5.5.6-Q16/type-ghostscript.mgk
28042 ./demo/coord0.crd
28310 ./examples/rectangular_duct/rect_duct_003.uns
28324 ./uns/write_split_binary_uns.c
28358 ./lib/blt2.4/dd_protocols/dd-file.tcl
28436 ./user/ftn_fv_close.f
28601 ./demo/coord5.crd
28632 ./examples/rectangular_duct/rect_duct_034.uns
28734 ./lib/blt2.4/dd_protocols/dd-color.tcl
28893 ./examples/rectangular_duct/rect_duct_053.uns
28954 ./lib/hp_700/dummy/libGL.1
29100 ./lib/tk8.2/listbox.tcl
29194 ./lib/tk8.2/images/pwrdLogo75.gif
29195 ./examples/rectangular_duct/rect_duct_046.uns
29424 ./bin/lib/ImageMagick-5.5.6-Q16/colors.mgk
29434 ./uil/hp_700/integration.uid
29488 ./examples/rectangular_duct/rect_duct_066.uns
29543 ./lib/tcl8.2/encoding/macRomania.enc
29643 ./fvx_and_restarts/animate_streamlines.vct
29657 ./demo/stream3.stl
29787 ./lib/tk8.2/optMenu.tcl
29826 ./lib/tcl8.2/init.tcl
29845 ./examples/rectangular_duct/rect_duct_023.uns
29908 ./bin/fvsrv64/plugins/lib/libadb.sl
29923 ./bin/plugins/fluent_reader.so
29936 ./examples/rectangular_duct/rect_duct_014.uns
29942 ./lib/tcl8.2/package.tcl
30113 ./lib/tk8.2/palette.tcl
30127 ./uil/hp_700/xform_controls.uid
30177 ./dore/data/fonts/meteorology.df
30254 ./lib/blt2.4/bltDnd.tcl
30259 ./uil/hp_700/xdb_info.uid
30349 ./lib/tk8.2/images/tai-ku.gif
30506 ./lib/tcl8.2/encoding/iso2022-kr.enc
30792 ./dore/data/fonts/romanc.df
31089 ./lib/tcl8.2/encoding/iso8859-1.enc
31428 ./bin/lib/ImageMagick-5.5.6-Q16/type-windows.mgk
31591 ./lib/tcl8.2/encoding/cp866.enc
31691 ./lib/tcl8.2/encoding/cp865.enc
32060 ./examples/rectangular_duct/rect_duct_025.uns
32133 ./uil/hp_700/legend_control.uid
32149 ./lib/hp_700/lua-python.so
32217 ./examples/rectangular_duct/rect_duct_006.uns
32378 ./uil/hp_700/data_input.uid
32492 ./lib/tcl8.2/encoding/cp852.enc
32548 ./demo/aerospace_LDM.vct
32643 ./fvx_and_restarts/basic_volume_integral.fvx
32649 ./examples/rectangular_duct/rect_duct_051.uns
32725 ./examples/rectangular_duct/rect_duct_022.uns
32837 ./examples/rectangular_duct/rect_duct_050.uns
32927 ./lib/tk8.2/tkAppInit.c
33001 ./lib/tcl8.2/encoding/cp874.enc
33248 ./demo/moments.fvx
33321 ./lib/blt2.4/dd_protocols/dd-text.tcl
33396 ./demo/coord2.crd
33456 ./bin/convert
33520 ./doc/Readme_Server_Toolkit.txt
33547 ./uns/write_binary_uns.c
34100 ./bin/mpeg2decode
34182 ./lib/blt2.4/dd_protocols/tclIndex
34626 ./geometry/sph12.atm
34669 ./lib/tcl8.2/encoding/gb1988.enc
34710 ./fvx_and_restarts/wind.frm
34846 ./uil/hp_700/point_probe.uid
34959 ./dore/data/fonts/lcmath.df
35021 ./fvx_and_restarts/animate_streamlines.frm
35313 ./dore/data/fonts/romanp.df
35384 ./geometry/square.atm
35398 ./demo/t4.ttl
35629 ./bin/README_Copyrights.txt
35670 ./doc/FVUsers_Guide.pdf
35707 ./lib/tcl8.2/encoding/shiftjis.enc
35724 ./bin/lmhostid
35818 ./examples/rectangular_duct/rect_duct_063.uns
35841 ./bin/node_id
36305 ./bin/fvsrv64/plugins/cgns_reader.so
36318 ./examples/rectangular_duct/rect_duct_080.uns
36575 ./fvx_and_restarts/bluntfinx.bin.fvbnd
36641 ./lib/tk8.2/images/pwrdLogo150.gif
36865 ./uns/ftn_fv_reader_tags.h
36925 ./examples/rectangular_duct/rect_duct_004.uns
37055 ./lib/tcl8.2/tclAppInit.c
37272 ./lib/tk8.2/clrpick.tcl
37307 ./lib/tcl8.2/encoding/ksc5601.enc
37409 ./lib/tcl8.2/history.tcl
37439 ./lib/tk8.2/images/pwrdLogo.eps
37477 ./lib/tcl8.2/encoding/cp1258.enc
37544 ./doc/Whats_New_FV12.pdf
37688 ./user/ftn_register_functions.f
38070 ./uil/hp_700/title_panel.uid
38119 ./uil/hp_700/transient_datav2.uid
38287 ./fvx_and_restarts/README_PLOT3D_formula_restart
38461 ./demo/f18i9b_g_bin
38461 ./examples/f18/f18i9b_g_bin
38501 ./dore/data/fonts/italict.df
38531 ./examples/rectangular_duct/rect_duct_036.uns
38545 ./doc/Help_Intro.pdf
38574 ./lib/tk8.2/prolog.ps
38666 ./lib/tk8.2/menu.tcl
38861 ./lib/tk8.2/images/pwrdLogo100.gif
38994 ./lib/tcl8.2/encoding/macTurkish.enc
39109 ./lib/tk8.2/images/pwrdLogo175.gif
39676 ./lib/tcl8.2/encoding/iso2022-jp.enc
39874 ./demo/f18show.scr
40066 ./user/ftn_register_data_readers.f
40442 ./demo/col1.map
40541 ./user/register_data_readers.c
40552 ./lib/tcl8.2/encoding/symbol.enc
40850 ./bin/Xvnc
41107 ./lib/tcl8.2/encoding/cp1256.enc
41169 ./bin/fvsrv64/plugins/lsdyna_reader.so
41169 ./geometry/geometry_formats.ps
41358 ./demo/stream2.stl
41364 ./lib/tk8.2/images/logo100.gif
41566 ./lib/blt2.4/pkgIndex.tcl
41734 ./demo/iso2.iso
42046 ./lib/tcl8.2/encoding/iso8859-9.enc
42054 ./demo/t9.ttl
42217 ./lib/tcl8.2/encoding/macJapan.enc
42218 ./lib/tcl8.2/encoding/cp437.enc
42270 ./examples/rectangular_duct/rect_duct_026.uns
42627 ./dore/data/fonts/ucmath.df
43278 ./dore/data/fonts/symbols.df
43508 ./lib/tk8.2/images/logo.eps
43545 ./uil/hp_700/flip.uid
43701 ./examples/rectangular_duct/rect_duct_041.uns
43751 ./examples/heat_exchanger/cocurrent.uns.fvreg
43751 ./examples/heat_exchanger/counter.uns.fvreg
43815 ./uil/hp_700/particle_paths.uid
43946 ./fvx_and_restarts/animate_streamlines.dat
43960 ./user/user_unstruct_grid.f
44028 ./lib/hp_700/liblualib.sl.4.0
44519 ./lib/tcl8.2/ldAix
44689 ./bin/fvsrv64/plugins/lib/libcgns.sl
44814 ./examples/rectangular_duct/rect_duct_042.uns
44867 ./demo/start.vct
44875 ./user/user_unstruct_results.f
44878 ./lib/tcl8.2/encoding/jis0201.enc
44926 ./examples/rectangular_duct/rect_duct_037.uns
45085 ./sconfig/srv.template
45415 ./uil/hp_700/presentation.uid
45438 ./examples/rectangular_duct/rect_duct_032.uns
45540 ./lib/fv.pyc
45642 ./dore/data/fonts/gothgbt.df
45830 ./demo/iso3.iso
45880 ./lib/tcl8.2/encoding/cp1250.enc
46000 ./examples/rectangular_duct/rect_duct_013.uns
46023 ./lib/tk8.2/images/logoMed.gif
46093 ./demo/view2.vct
46103 ./examples/rectangular_duct/rect_duct_047.uns
46238 ./demo/start.map
46455 ./lib/tcl8.2/encoding/cp1255.enc
46496 ./lib/blt2.4/bltHierbox.tcl
46697 ./bin/animate
46844 ./geometry/cone12.atm
47054 ./fvx_and_restarts/seed_sphere_proto.fvx
47357 ./uns/write_split_unformatted_uns.F
47891 ./demo/view6.vct
47930 ./demo/drag.fvx
47970 ./lib/tcl8.2/safe.tcl
48023 ./lib/tcl8.2/encoding/cp936.enc
48142 ./lib/blt2.4/bltHiertable.tcl
48291 ./data/colormaps/c7.col
48295 ./fvx_and_restarts/xcolor_defines.fvx
48438 ./examples/rectangular_duct/rect_duct_059.uns
48449 ./bin/mv_record
48906 ./dore/data/fonts/greekcs.df
49045 ./fvx_and_restarts/seed_sphere_GUI.fvx
49051 ./demo/view4.vct
49338 ./demo/view3.vct
49417 ./lib/tcl8.2/encoding/ascii.enc
49468 ./bin/mv_init_tape
49617 ./demo/start.bnd
49824 ./lib/tcl8.2/encoding/iso8859-3.enc
49867 ./uil/hp_700/surface_plot.uid
49896 ./geometry/cube.atm
49920 ./fvx_and_restarts/lsdyna_formulas.frm
49926 ./data/colormaps/c4.col
49932 ./demo/t8.ttl
50329 ./demo/view7.vct
50415 ./examples/rectangular_duct/rect_duct_061.uns
50427 ./examples/rectangular_duct/rect_duct_077.uns
50441 ./bin/prolog.ps
50605 ./examples/rectangular_duct/rect_duct_038.uns
51123 ./examples/rectangular_duct/rect_duct_021.uns
51494 ./bin/fv_convert.template
51742 ./uil/hp_700/subset_params.uid
51968 ./examples/rectangular_duct/rect_duct_065.uns
52013 ./lib/tcl8.2/encoding/cp855.enc
52057 ./examples/rectangular_duct/rect_duct_076.uns
52309 ./lib/tcl8.2/opt0.4/pkgIndex.tcl
52340 ./examples/rectangular_duct/rect_duct_049.uns
52415 ./demo/start.ttl
52574 ./lib/hp_700/liblua.sl.4.0
52720 ./bin/fvsrv64/plugins/acusolve_reader.so
52745 ./dore/data/fonts/romand.df
52755 ./uil/hp_700/comp_surface.uid
52833 ./lib/tcl8.2/encoding/cp850.enc
53034 ./bin/node_id.il
53221 ./lib/blt2.4/tclIndex
53293 ./examples/rectangular_duct/rect_duct_052.uns
53473 ./lib/tk8.2/comdlg.tcl
53516 ./examples/rectangular_duct/rect_duct_058.uns
53968 ./user/toolkit.c
53979 ./uil/hp_700/boundary_conditions.uid
54002 ./lib/tcl8.2/encoding/cp860.enc
54004 ./uil/hp_700/datasetBrowser.uid
54102 ./fvx_and_restarts/calc_volume_integral.fvx
54228 ./sconfig/manual.srv
54250 ./demo/plot1.lpt
54260 ./demo/plot2.lpt
54331 ./bin/make_fv
54533 ./dore/data/fonts/romancs.df
54905 ./dore/data/fonts/helvetica.df
55047 ./examples/rectangular_duct/rect_duct.trk
55047 ./uil/hp_700/colormap_specification.uid
55208 ./fvx_and_restarts/Q-criterion.frm
55231 ./lib/tcl8.2/http1.0/pkgIndex.tcl
55361 ./lib/tk8.2/xmfbox.tcl
55552 ./examples/rectangular_duct/rect_duct_074.uns
55614 ./examples/rectangular_duct/rect_duct_045.uns
55673 ./lib/tcl8.2/encoding/cp1251.enc
55709 ./examples/rectangular_duct/rect_duct_075.uns
55734 ./dore/data/fonts/scriptc.df
55885 ./lib/tk8.2/images/logoLarge.gif
56003 ./data/colormaps/c5.col
56186 ./dore/data/fonts/greeks.df
56255 ./examples/rectangular_duct/rect_duct_002.uns
56320 ./demo/t6.ttl
56378 ./examples/rectangular_duct/rect_duct_070.uns
56695 ./lib/tcl8.2/encoding/iso8859-2.enc
56916 ./lib/blt2.4/bltTabnotebook.tcl
56960 ./dore/data/fonts/greekc.df
57234 ./user/user_unstruct_functions.f
57480 ./demo/aerospace_LDM.frm
57686 ./lib/blt2.4/Problems
57698 ./fvx_and_restarts/FVX_Templates.pdf
57711 ./lib/tk8.2/entry.tcl
57757 ./examples/rectangular_duct/rect_duct_064.uns
57941 ./lib/tcl8.2/encoding/dingbats.enc
58071 ./lib/tcl8.2/opt0.4/optparse.tcl
58217 ./bin/lib/ImageMagick-5.5.6-Q16/type.mgk
58430 ./fvx_and_restarts/animate_streamlines.bnd
58497 ./uil/hp_700/formula.uid
58616 ./fvx_and_restarts/plot_along_streamline.fvx
58635 ./demo/start.frm
58729 ./uil/hp_700/color_mixer.uid
58764 ./lib/tk8.2/tearoff.tcl
59143 ./uil/hp_700/dynamic_subset.uid
59310 ./demo/t0.ttl
59391 ./lib/tcl8.2/encoding/iso8859-4.enc
59497 ./doc/How_to.pdf
59695 ./examples/rectangular_duct/rect_duct_071.uns
60258 ./dore/data/errmsg/errmsgfile
60621 ./data/colormaps/c6.col
60693 ./examples/rectangular_duct/rect_duct_079.uns
60841 ./bin/lib/ImageMagick-5.5.6-Q16/type-solaris.mgk
61035 ./lib/blt2.4/bltGraph.pro
61091 ./bin/mv_reset
61117 ./lib/tcl8.2/word.tcl
61247 ./demo/blank.ttl
61446 ./uns/fv_reader_tags.h
61497 ./dore/data/fonts/music.df
61541 ./lib/tk8.2/console.tcl
61734 ./demo/coord4.crd
61805 ./uil/hp_700/misc_file_input.uid
61852 ./demo/view5.vct
62390 ./user/register_functions.c
62413 ./demo/t5.ttl
62662 ./demo/t1.ttl
62907 ./demo/stream1.stl
62918 ./fvx_and_restarts/bluntfinx.bin
63021 ./examples/rectangular_duct/rect_duct_017.uns
63102 ./lib/tcl8.2/encoding/macRoman.enc
63106 ./examples/heat_exchanger/cocurrent.uns
63221 ./lib/tk8.2/safetk.tcl
63321 ./bin/ld_fv
63340 ./demo/iso4.iso
63384 ./examples/rectangular_duct/rect_duct_027.uns
63671 ./geometry/cone48.atm
63850 ./lib/tcl8.2/encoding/cp1252.enc
64512 ./examples/rectangular_duct/rect_duct_072.uns
64515 ./data/colormaps/c8.col
64986 ./lib/blt2.4/Readme
65154 ./lib/blt2.4/bltCanvEps.pro
65304 ./data/colormaps/c1.col
65467 ./dore/data/fonts/scripts.df
EOF
;;
5)
# checksums for Linux_x86 version of fv directory
cat > $checkold <<EOF
00023 ./lib/tk8.2/tkfbox.tcl
00052 ./lib/tcl8.2/encoding/big5.enc
00064 ./lib/tk8.2/text.tcl
00273 ./lib/tcl8.2/encoding/cp1253.enc
00281 ./lib/tcl8.2/encoding/gb12345.enc
00300 ./user/user_results_file_read.f
00328 ./demo/iso1.iso
00349 ./bin/fv_to_printer.sh.sample
00455 ./lib/tcl8.2/encoding/cp857.enc
00685 ./examples/rectangular_duct/rect_duct_044.uns
00715 ./examples/rectangular_duct/rect_duct_048.uns
00795 ./lib/tcl8.2/encoding/jis0212.enc
01283 ./examples/rectangular_duct/rect_duct_029.uns
01383 ./lib/tk8.2/button.tcl
01478 ./fvx_and_restarts/animate_streamlines.map
01677 ./demo/f18i9b_g_bin.fvbnd
01677 ./examples/f18/f18i9b_g_bin.fvbnd
01748 ./uil/linux_x86/formula.uid
02009 ./examples/rectangular_duct/rect_duct_024.uns
02213 ./examples/rectangular_duct/rect_duct_012.uns
02225 ./fvx_and_restarts/plot3d_eqns.frm
02235 ./lib/tk8.2/msgbox.tcl
02266 ./lib/tcl8.2/encoding/cp949.enc
02317 ./data/xdb_vars
02325 ./lib/blt2.4/dd_protocols/dd-number.tcl
02345 ./user/user_grid_file_read.f
02492 ./examples/rectangular_duct/rect_duct_062.uns
02682 ./doc/Readme_InstallNotes_12.pdf
02722 ./data/colormaps/c3.col
02780 ./uns/write_split_ascii_uns.f
02844 ./examples/rectangular_duct/rect_duct_067.uns
03030 ./fvx_and_restarts/bluntfinq.bin
03294 ./lib/blt2.4/bltDragdrop.tcl
03299 ./examples/rectangular_duct/rect_duct_030.uns
03321 ./demo/coord3.crd
03329 ./fvx_and_restarts/set_view_TEST.fvx
03592 ./dore/data/il/def/parsekey.sym
03788 ./bin/server_name
03816 ./geometry/cyln48.atm
04150 ./fvx_and_restarts/seed_rectangle_proto.fvx
04449 ./examples/rectangular_duct/rect_duct_039.uns
04525 ./fvx_and_restarts/seed_circ_GUI.fvx
04824 ./lib/tcl8.2/encoding/macThai.enc
04999 ./lib/tcl8.2/encoding/macIceland.enc
05048 ./examples/rectangular_duct/rect_duct_040.uns
05077 ./lib/tcl8.2/http2.1/pkgIndex.tcl
05136 ./lib/tcl8.2/tcltest1.0/pkgIndex.tcl
05176 ./lib/tcl8.2/encoding/koi8-r.enc
05448 ./uil/linux_x86/datasetBrowser.uid
05986 ./demo/aerospace_LDM.bnd
06058 ./demo/f18i9b_q_bin
06058 ./examples/f18/f18i9b_q_bin
06106 ./lib/tcl8.2/encoding/cp737.enc
06139 ./dore/data/fonts/romant.df
06194 ./lib/tcl8.2/encoding/gb2312.enc
06517 ./bin/fvsrv
06656 ./uil/linux_x86/misc_support.uid
06709 ./uil/linux_x86/colormap_specification.uid
06966 ./lib/tcl8.2/encoding/iso8859-8.enc
07340 ./geometry/cyln12.atm
07365 ./examples/rectangular_duct/rect_duct_016.uns
07605 ./demo/stream4.stl
07614 ./demo/t3.ttl
07885 ./bin/fv.template
08136 ./lib/tcl8.2/encoding/euc-jp.enc
08160 ./lib/tcl8.2/encoding/cp1254.enc
08221 ./demo/t.ttl
08221 ./demo/t10.ttl
08320 ./examples/rectangular_duct/rect_duct_043.uns
08785 ./uil/linux_x86/vortex_core.uid
08917 ./user/user_defined_functions.f
09118 ./lib/tcl8.2/encoding/cp862.enc
09161 ./examples/rectangular_duct/rect_duct_018.uns
09510 ./examples/rectangular_duct/rect_duct_015.uns
09563 ./examples/rectangular_duct/rect_duct_068.uns
10046 ./demo/lift.fvx
10063 ./bin/lib/ImageMagick-5.5.7/log.mgk
10314 ./lib/tk8.2/obsolete.tcl
10394 ./examples/rectangular_duct/rect_duct_073.uns
10827 ./lib/tcl8.2/http2.1/http.tcl
10855 ./lib/tk8.2/bgerror.tcl
10967 ./doc/FVReference_Manual.pdf
10998 ./uil/linux_x86/surface_plot.uid
11013 ./examples/rectangular_duct/rect_duct_055.uns
11200 ./dore/data/fonts/romans.df
11218 ./uil/linux_x86/func.uid
11396 ./examples/rectangular_duct/rect_duct_011.uns
11412 ./lib/tcl8.2/encoding/cp864.enc
11868 ./lib/tcl8.2/tclIndex
11903 ./lib/tcl8.2/encoding/euc-cn.enc
11963 ./dore/data/fonts/italicc.df
12263 ./dore/data/fonts/cyrillic.df
12397 ./demo/t11.ttl
12501 ./bin/fv_vnc
12721 ./examples/rectangular_duct/rect_duct_028.uns
12833 ./lib/tcl8.2/tcltest1.0/tcltest.tcl
13082 ./fvx_and_restarts/set_view.fvx
13258 ./lib/tcl8.2/encoding/euc-kr.enc
13293 ./fvx_and_restarts/bio.frm
13398 ./demo/t2.ttl
13461 ./examples/rectangular_duct/rect_duct_009.uns
13635 ./bin/mpich/ch_p4/bin/mpirun.ch_p4
13652 ./examples/rectangular_duct/rect_duct_078.uns
13726 ./bin/lib/ImageMagick-5.5.7/delegates.mgk
14117 ./lib/blt2.4/NEWS
14544 ./dore/data/fonts/greekp.df
14611 ./geometry/allobs.ob
14678 ./bin/lib/ImageMagick-5.5.7/modules.mgk
14913 ./uil/linux_x86/vr.uid
15065 ./examples/rectangular_duct/rect_duct_033.uns
15605 ./lib/tcl8.2/encoding/macCentEuro.enc
15825 ./lib/tcl8.2/encoding/macCroatian.enc
15887 ./lib/tcl8.2/encoding/macUkraine.enc
16097 ./lib/tcl8.2/encoding/iso8859-6.enc
16373 ./examples/rectangular_duct/rect_duct_008.uns
16561 ./lib/tk8.2/scrlbar.tcl
16650 ./geometry/poly12.atm
16658 ./fvx_and_restarts/seed_circular_proto.fvx
16709 ./lib/tcl8.2/msgcat1.0/pkgIndex.tcl
16741 ./lib/tcl8.2/parray.tcl
16795 ./dore/data/fonts/gothgrt.df
16841 ./lib/tcl8.2/encoding/cp950.enc
16950 ./fvx_and_restarts/animate_streamlines.prd
17116 ./lib/tcl8.2/auto.tcl
17153 ./lib/start_python.luac64
17184 ./demo/start.plt
17317 ./demo/start.dat
17332 ./demo/bound1.bnd
17427 ./lib/tcl8.2/encoding/cp775.enc
17570 ./lib/tcl8.2/encoding/iso2022.enc
17634 ./bin/plugins/lsdyna_reader.so
17635 ./examples/rectangular_duct/rect_duct_031.uns
17647 ./geometry/poly48.atm
17790 ./lib/tcl8.2/encoding/iso8859-7.enc
17844 ./lib/tcl8.2/encoding/cp932.enc
17920 ./lib/tcl8.2/msgcat1.0/msgcat.tcl
17929 ./uil/linux_x86/streamlines.uid
18023 ./examples/rectangular_duct/rect_duct_020.uns
18271 ./lib/tcl8.2/encoding/cp863.enc
18315 ./lib/tcl8.2/encoding/cp869.enc
18328 ./examples/rectangular_duct/rect_duct_019.uns
18422 ./bin/fv_user_print.sh.sample
18484 ./fvx_and_restarts/animate_streamlines.fvx
18619 ./lib/tcl8.2/encoding/macDingbats.enc
18713 ./dore/data/fonts/gothitt.df
18859 ./bin/plugins/lib/libadb.so
19151 ./demo/aerospace_LDM.fvx
19329 ./lib/tcl8.2/encoding/macGreek.enc
19500 ./lib/tcl8.2/encoding/jis0208.enc
19545 ./lib/tcl8.2/encoding/cp861.enc
19776 ./lib/start_python.luac
20046 ./uil/linux_x86/xform_controls.uid
20094 ./bin/plugins/lib/libflsrdr.so
20094 ./examples/rectangular_duct/rect_duct_069.uns
20144 ./examples/rectangular_duct/rect_duct_056.uns
20418 ./examples/rectangular_duct/rect_duct_007.uns
20545 ./examples/rectangular_duct/rect_duct_060.uns
20546 ./uil/linux_x86/subset_params.uid
20641 ./uil/linux_x86/misc_file_input.uid
20873 ./dore/data/fonts/unused.df
21393 ./bin/vnc_disp
21460 ./lib/tcl8.2/http1.0/http.tcl
21496 ./data/xdb_vars_FV11.0
21519 ./bin/mpeg2encode
21564 ./lib/tcl8.2/ldAout.tcl
21798 ./demo/t7.ttl
21894 ./lib/blt2.4/bltTabset.tcl
21960 ./dore/data/fonts/astrology.df
21969 ./examples/heat_exchanger/counter.uns
22040 ./uns/write_unformatted_uns.F
22075 ./bin/plugins/acusolve_reader.so
22438 ./lib/tcl8.2/encoding/cp1257.enc
22569 ./examples/rectangular_duct/rect_duct_035.uns
22661 ./bin/fvsrvbin
22732 ./bin/fvsrv_shmem
22765 ./data/license.dat
23217 ./demo/wingloadplots.fvx
23286 ./dore/data/fonts/italiccs.df
23323 ./demo/coord1.crd
23344 ./uns/write_ascii_uns.f
23526 ./lib/tk8.2/scale.tcl
23533 ./lib/tcl8.2/encoding/iso8859-5.enc
23541 ./examples/rectangular_duct/rect_duct_005.uns
23614 ./bin/mpich/ch_shmem/bin/mpirun.ch_shmem
23709 ./lib/tk8.2/images/logo64.gif
23752 ./lib/tk8.2/dialog.tcl
24188 ./examples/rectangular_duct/rect_duct_001.uns
24259 ./bin/mpich/ch_p4/bin/mpirun.args
24259 ./bin/mpich/ch_shmem/bin/mpirun.args
24661 ./examples/rectangular_duct/rect_duct_010.uns
24673 ./data/colormaps/c2.col
24852 ./demo/aerospace_LDM.dat
24881 ./lib/blt2.4/htresize.cur
25189 ./lib/tk8.2/tk.tcl
25502 ./examples/rectangular_duct/rect_duct_054.uns
25598 ./user/user_unstruct_combined.f
25605 ./lib/tcl8.2/encoding/macCyrillic.enc
25703 ./lib/tk8.2/tclIndex
26148 ./user/user_combined_file_read.f
26216 ./examples/rectangular_duct/rect_duct_057.uns
26356 ./lib/tk8.2/images/README
26361 ./bin/mpich/ch_shmem/bin/mpirun
26380 ./lib/tk8.2/focus.tcl
26400 ./lib/linux_x86/lua-python.so
26668 ./geometry/sph24.atm
27073 ./fvx_and_restarts/seed_rect_GUI.fvx
27081 ./lib/blt2.4/bltGraph.tcl
27188 ./lib/tk8.2/images/pwrdLogo200.gif
27513 ./geometry/star.atm
27721 ./bin/animate
27920 ./bin/lib/ImageMagick-5.5.7/type-ghostscript.mgk
27971 ./bin/lib/ImageMagick-5.5.7/magic.mgk
28042 ./demo/coord0.crd
28310 ./examples/rectangular_duct/rect_duct_003.uns
28324 ./uns/write_split_binary_uns.c
28358 ./lib/blt2.4/dd_protocols/dd-file.tcl
28393 ./bin/plugins/lib/libcgns.so
28436 ./user/ftn_fv_close.f
28601 ./demo/coord5.crd
28632 ./examples/rectangular_duct/rect_duct_034.uns
28734 ./lib/blt2.4/dd_protocols/dd-color.tcl
28893 ./examples/rectangular_duct/rect_duct_053.uns
29100 ./lib/tk8.2/listbox.tcl
29194 ./lib/tk8.2/images/pwrdLogo75.gif
29195 ./examples/rectangular_duct/rect_duct_046.uns
29424 ./bin/lib/ImageMagick-5.5.7/colors.mgk
29488 ./examples/rectangular_duct/rect_duct_066.uns
29543 ./lib/tcl8.2/encoding/macRomania.enc
29643 ./fvx_and_restarts/animate_streamlines.vct
29657 ./demo/stream3.stl
29787 ./lib/tk8.2/optMenu.tcl
29826 ./lib/tcl8.2/init.tcl
29845 ./examples/rectangular_duct/rect_duct_023.uns
29936 ./examples/rectangular_duct/rect_duct_014.uns
29942 ./lib/tcl8.2/package.tcl
30113 ./lib/tk8.2/palette.tcl
30177 ./dore/data/fonts/meteorology.df
30254 ./lib/blt2.4/bltDnd.tcl
30349 ./lib/tk8.2/images/tai-ku.gif
30442 ./uil/linux_x86/data_input.uid
30506 ./lib/tcl8.2/encoding/iso2022-kr.enc
30792 ./dore/data/fonts/romanc.df
31089 ./lib/tcl8.2/encoding/iso8859-1.enc
31428 ./bin/lib/ImageMagick-5.5.7/type-windows.mgk
31591 ./lib/tcl8.2/encoding/cp866.enc
31691 ./lib/tcl8.2/encoding/cp865.enc
32060 ./examples/rectangular_duct/rect_duct_025.uns
32217 ./examples/rectangular_duct/rect_duct_006.uns
32492 ./lib/tcl8.2/encoding/cp852.enc
32548 ./demo/aerospace_LDM.vct
32643 ./fvx_and_restarts/basic_volume_integral.fvx
32649 ./examples/rectangular_duct/rect_duct_051.uns
32725 ./examples/rectangular_duct/rect_duct_022.uns
32837 ./examples/rectangular_duct/rect_duct_050.uns
32927 ./lib/tk8.2/tkAppInit.c
33001 ./lib/tcl8.2/encoding/cp874.enc
33248 ./demo/moments.fvx
33321 ./lib/blt2.4/dd_protocols/dd-text.tcl
33396 ./demo/coord2.crd
33520 ./doc/Readme_Server_Toolkit.txt
33547 ./uns/write_binary_uns.c
34009 ./bin/plugins/lib/libifport.so.5
34182 ./lib/blt2.4/dd_protocols/tclIndex
34626 ./geometry/sph12.atm
34669 ./lib/tcl8.2/encoding/gb1988.enc
34707 ./bin/mpich/share/machines.sample
34710 ./fvx_and_restarts/wind.frm
34959 ./dore/data/fonts/lcmath.df
35007 ./bin/mpeg2decode
35021 ./fvx_and_restarts/animate_streamlines.frm
35118 ./bin/plugins/lib/libsvml.so
35313 ./dore/data/fonts/romanp.df
35384 ./geometry/square.atm
35398 ./demo/t4.ttl
35629 ./bin/README_Copyrights.txt
35670 ./doc/FVUsers_Guide.pdf
35707 ./lib/tcl8.2/encoding/shiftjis.enc
35809 ./uil/linux_x86/color_mixer.uid
35818 ./examples/rectangular_duct/rect_duct_063.uns
35841 ./bin/node_id
36055 ./bin/mpich/ch_p4/bin/mpirun.pg
36318 ./examples/rectangular_duct/rect_duct_080.uns
36575 ./fvx_and_restarts/bluntfinx.bin.fvbnd
36641 ./lib/tk8.2/images/pwrdLogo150.gif
36865 ./uns/ftn_fv_reader_tags.h
36925 ./examples/rectangular_duct/rect_duct_004.uns
36979 ./bin/mpich/ch_p4/bin/tarch
36979 ./bin/mpich/ch_shmem/bin/tarch
37055 ./lib/tcl8.2/tclAppInit.c
37068 ./uil/linux_x86/region_control.uid
37272 ./lib/tk8.2/clrpick.tcl
37307 ./lib/tcl8.2/encoding/ksc5601.enc
37409 ./lib/tcl8.2/history.tcl
37439 ./lib/tk8.2/images/pwrdLogo.eps
37477 ./lib/tcl8.2/encoding/cp1258.enc
37544 ./doc/Whats_New_FV12.pdf
37688 ./user/ftn_register_functions.f
37735 ./bin/plugins/lib/libirc.so
38287 ./fvx_and_restarts/README_PLOT3D_formula_restart
38460 ./uil/linux_x86/boundary_conditions.uid
38461 ./demo/f18i9b_g_bin
38461 ./examples/f18/f18i9b_g_bin
38501 ./dore/data/fonts/italict.df
38531 ./examples/rectangular_duct/rect_duct_036.uns
38545 ./doc/Help_Intro.pdf
38574 ./lib/tk8.2/prolog.ps
38666 ./lib/tk8.2/menu.tcl
38861 ./lib/tk8.2/images/pwrdLogo100.gif
38994 ./lib/tcl8.2/encoding/macTurkish.enc
39109 ./lib/tk8.2/images/pwrdLogo175.gif
39676 ./lib/tcl8.2/encoding/iso2022-jp.enc
39872 ./bin/convert
39874 ./demo/f18show.scr
40066 ./user/ftn_register_data_readers.f
40442 ./demo/col1.map
40541 ./user/register_data_readers.c
40552 ./lib/tcl8.2/encoding/symbol.enc
40610 ./bin/plugins/flow3d_reader.so
40862 ./lib/linux_x86/dummy/libGLU.so.1
41107 ./lib/tcl8.2/encoding/cp1256.enc
41169 ./geometry/geometry_formats.ps
41358 ./demo/stream2.stl
41364 ./lib/tk8.2/images/logo100.gif
41566 ./lib/blt2.4/pkgIndex.tcl
41734 ./demo/iso2.iso
41740 ./bin/mpich/ch_p4/bin/mpirun.rand
42046 ./lib/tcl8.2/encoding/iso8859-9.enc
42054 ./demo/t9.ttl
42074 ./uil/linux_x86/flip.uid
42217 ./lib/tcl8.2/encoding/macJapan.enc
42218 ./lib/tcl8.2/encoding/cp437.enc
42270 ./examples/rectangular_duct/rect_duct_026.uns
42286 ./uil/linux_x86/keyframe.uid
42490 ./bin/contest
42627 ./dore/data/fonts/ucmath.df
43206 ./uil/linux_x86/integration.uid
43278 ./dore/data/fonts/symbols.df
43508 ./lib/tk8.2/images/logo.eps
43701 ./examples/rectangular_duct/rect_duct_041.uns
43751 ./examples/heat_exchanger/cocurrent.uns.fvreg
43751 ./examples/heat_exchanger/counter.uns.fvreg
43757 ./bin/plugins/lib/libimf.so
43946 ./fvx_and_restarts/animate_streamlines.dat
43960 ./user/user_unstruct_grid.f
44519 ./lib/tcl8.2/ldAix
44814 ./examples/rectangular_duct/rect_duct_042.uns
44867 ./demo/start.vct
44875 ./user/user_unstruct_results.f
44878 ./lib/tcl8.2/encoding/jis0201.enc
44926 ./examples/rectangular_duct/rect_duct_037.uns
45037 ./lib/linux_x86/dummy/libGL.so.1
45085 ./sconfig/srv.template
45438 ./examples/rectangular_duct/rect_duct_032.uns
45540 ./lib/fv.pyc
45642 ./dore/data/fonts/gothgbt.df
45830 ./demo/iso3.iso
45880 ./lib/tcl8.2/encoding/cp1250.enc
46000 ./examples/rectangular_duct/rect_duct_013.uns
46023 ./lib/tk8.2/images/logoMed.gif
46093 ./demo/view2.vct
46103 ./examples/rectangular_duct/rect_duct_047.uns
46238 ./demo/start.map
46455 ./lib/tcl8.2/encoding/cp1255.enc
46496 ./lib/blt2.4/bltHierbox.tcl
46844 ./geometry/cone12.atm
46884 ./bin/plugins/lib/libifcore.so.5
47054 ./fvx_and_restarts/seed_sphere_proto.fvx
47357 ./uns/write_split_unformatted_uns.F
47640 ./bin/fvsrv_p4
47685 ./bin/lmhostid
47891 ./demo/view6.vct
47930 ./demo/drag.fvx
47970 ./lib/tcl8.2/safe.tcl
47971 ./bin/plugins/cgns_reader.so
48023 ./lib/tcl8.2/encoding/cp936.enc
48142 ./lib/blt2.4/bltHiertable.tcl
48291 ./data/colormaps/c7.col
48295 ./fvx_and_restarts/xcolor_defines.fvx
48438 ./examples/rectangular_duct/rect_duct_059.uns
48906 ./dore/data/fonts/greekcs.df
49045 ./fvx_and_restarts/seed_sphere_GUI.fvx
49051 ./demo/view4.vct
49338 ./demo/view3.vct
49417 ./lib/tcl8.2/encoding/ascii.enc
49617 ./demo/start.bnd
49824 ./lib/tcl8.2/encoding/iso8859-3.enc
49896 ./geometry/cube.atm
49920 ./fvx_and_restarts/lsdyna_formulas.frm
49926 ./data/colormaps/c4.col
49932 ./demo/t8.ttl
49968 ./uil/linux_x86/legend_control.uid
50065 ./lib/linux_x86/liblualib.so.4.0
50329 ./demo/view7.vct
50415 ./examples/rectangular_duct/rect_duct_061.uns
50427 ./examples/rectangular_duct/rect_duct_077.uns
50441 ./bin/prolog.ps
50605 ./examples/rectangular_duct/rect_duct_038.uns
50988 ./uil/linux_x86/point_probe.uid
51123 ./examples/rectangular_duct/rect_duct_021.uns
51494 ./bin/fv_convert.template
51558 ./uil/linux_x86/iso_surface.uid
51840 ./uil/linux_x86/presentation.uid
51968 ./examples/rectangular_duct/rect_duct_065.uns
52013 ./lib/tcl8.2/encoding/cp855.enc
52057 ./examples/rectangular_duct/rect_duct_076.uns
52248 ./uil/linux_x86/transient_datav2.uid
52309 ./lib/tcl8.2/opt0.4/pkgIndex.tcl
52340 ./examples/rectangular_duct/rect_duct_049.uns
52415 ./demo/start.ttl
52512 ./uil/linux_x86/plot.uid
52587 ./bin/fvbin
52745 ./dore/data/fonts/romand.df
52833 ./lib/tcl8.2/encoding/cp850.enc
53221 ./lib/blt2.4/tclIndex
53293 ./examples/rectangular_duct/rect_duct_052.uns
53473 ./lib/tk8.2/comdlg.tcl
53516 ./examples/rectangular_duct/rect_duct_058.uns
53788 ./uil/linux_x86/comp_surface.uid
53968 ./user/toolkit.c
54002 ./lib/tcl8.2/encoding/cp860.enc
54036 ./uil/linux_x86/dataset_controls.uid
54102 ./fvx_and_restarts/calc_volume_integral.fvx
54228 ./sconfig/manual.srv
54250 ./demo/plot1.lpt
54260 ./demo/plot2.lpt
54331 ./bin/make_fv
54533 ./dore/data/fonts/romancs.df
54905 ./dore/data/fonts/helvetica.df
55047 ./examples/rectangular_duct/rect_duct.trk
55208 ./fvx_and_restarts/Q-criterion.frm
55231 ./lib/tcl8.2/http1.0/pkgIndex.tcl
55361 ./lib/tk8.2/xmfbox.tcl
55552 ./examples/rectangular_duct/rect_duct_074.uns
55614 ./examples/rectangular_duct/rect_duct_045.uns
55673 ./lib/tcl8.2/encoding/cp1251.enc
55709 ./examples/rectangular_duct/rect_duct_075.uns
55734 ./dore/data/fonts/scriptc.df
55885 ./lib/tk8.2/images/logoLarge.gif
56003 ./data/colormaps/c5.col
56186 ./dore/data/fonts/greeks.df
56243 ./lib/linux_x86/liblua.so.4.0
56255 ./examples/rectangular_duct/rect_duct_002.uns
56320 ./demo/t6.ttl
56378 ./examples/rectangular_duct/rect_duct_070.uns
56695 ./lib/tcl8.2/encoding/iso8859-2.enc
56801 ./bin/mpich/ch_p4/bin/mpirun.p4shmem
56862 ./bin/mpich/ch_p4/bin/mpirun.ch_p4.args
56916 ./lib/blt2.4/bltTabnotebook.tcl
56960 ./dore/data/fonts/greekc.df
57234 ./user/user_unstruct_functions.f
57480 ./demo/aerospace_LDM.frm
57686 ./lib/blt2.4/Problems
57698 ./fvx_and_restarts/FVX_Templates.pdf
57711 ./lib/tk8.2/entry.tcl
57757 ./examples/rectangular_duct/rect_duct_064.uns
57908 ./bin/fv_platform
57941 ./lib/tcl8.2/encoding/dingbats.enc
58071 ./lib/tcl8.2/opt0.4/optparse.tcl
58217 ./bin/lib/ImageMagick-5.5.7/type.mgk
58234 ./uil/linux_x86/particle_paths.uid
58430 ./fvx_and_restarts/animate_streamlines.bnd
58616 ./fvx_and_restarts/plot_along_streamline.fvx
58635 ./demo/start.frm
58686 ./uil/linux_x86/coordinate_surface.uid
58764 ./lib/tk8.2/tearoff.tcl
59310 ./demo/t0.ttl
59391 ./lib/tcl8.2/encoding/iso8859-4.enc
59497 ./doc/How_to.pdf
59695 ./examples/rectangular_duct/rect_duct_071.uns
60106 ./uil/linux_x86/title_panel.uid
60258 ./dore/data/errmsg/errmsgfile
60621 ./data/colormaps/c6.col
60693 ./examples/rectangular_duct/rect_duct_079.uns
60841 ./bin/lib/ImageMagick-5.5.7/type-solaris.mgk
60867 ./bin/mpich/ch_p4/bin/mpirun
61035 ./lib/blt2.4/bltGraph.pro
61052 ./uil/linux_x86/xdb_info.uid
61117 ./lib/tcl8.2/word.tcl
61247 ./demo/blank.ttl
61446 ./uns/fv_reader_tags.h
61497 ./dore/data/fonts/music.df
61541 ./lib/tk8.2/console.tcl
61734 ./demo/coord4.crd
61852 ./demo/view5.vct
62208 ./uil/linux_x86/dynamic_subset.uid
62390 ./user/register_functions.c
62413 ./demo/t5.ttl
62662 ./demo/t1.ttl
62907 ./demo/stream1.stl
62918 ./fvx_and_restarts/bluntfinx.bin
63021 ./examples/rectangular_duct/rect_duct_017.uns
63102 ./lib/tcl8.2/encoding/macRoman.enc
63106 ./examples/heat_exchanger/cocurrent.uns
63221 ./lib/tk8.2/safetk.tcl
63321 ./bin/ld_fv
63340 ./demo/iso4.iso
63384 ./examples/rectangular_duct/rect_duct_027.uns
63497 ./uil/linux_x86/vector_params.uid
63671 ./geometry/cone48.atm
63850 ./lib/tcl8.2/encoding/cp1252.enc
64512 ./examples/rectangular_duct/rect_duct_072.uns
64515 ./data/colormaps/c8.col
64642 ./bin/display
64986 ./lib/blt2.4/Readme
65039 ./bin/plugins/fluent_reader.so
65154 ./lib/blt2.4/bltCanvEps.pro
65304 ./data/colormaps/c1.col
65467 ./dore/data/fonts/scripts.df
EOF
;;
6)
# checksums for Linux_ia64 version of fv directory
cat > $checkold <<EOF
00023 ./lib/tk8.2/tkfbox.tcl
00052 ./lib/tcl8.2/encoding/big5.enc
00064 ./lib/tk8.2/text.tcl
00273 ./lib/tcl8.2/encoding/cp1253.enc
00281 ./lib/tcl8.2/encoding/gb12345.enc
00300 ./user/user_results_file_read.f
00328 ./demo/iso1.iso
00349 ./bin/fv_to_printer.sh.sample
00455 ./lib/tcl8.2/encoding/cp857.enc
00685 ./examples/rectangular_duct/rect_duct_044.uns
00715 ./examples/rectangular_duct/rect_duct_048.uns
00795 ./lib/tcl8.2/encoding/jis0212.enc
01283 ./examples/rectangular_duct/rect_duct_029.uns
01383 ./lib/tk8.2/button.tcl
01408 ./uil/linux_ia64/xform_controls.uid
01424 ./bin/server_name
01478 ./fvx_and_restarts/animate_streamlines.map
01677 ./demo/f18i9b_g_bin.fvbnd
01677 ./examples/f18/f18i9b_g_bin.fvbnd
02009 ./examples/rectangular_duct/rect_duct_024.uns
02081 ./uil/linux_ia64/vortex_core.uid
02213 ./examples/rectangular_duct/rect_duct_012.uns
02225 ./fvx_and_restarts/plot3d_eqns.frm
02235 ./lib/tk8.2/msgbox.tcl
02266 ./lib/tcl8.2/encoding/cp949.enc
02317 ./data/xdb_vars
02325 ./lib/blt2.4/dd_protocols/dd-number.tcl
02345 ./user/user_grid_file_read.f
02492 ./examples/rectangular_duct/rect_duct_062.uns
02592 ./uil/linux_ia64/surface_plot.uid
02682 ./doc/Readme_InstallNotes_12.pdf
02722 ./data/colormaps/c3.col
02780 ./uns/write_split_ascii_uns.f
02844 ./examples/rectangular_duct/rect_duct_067.uns
03030 ./fvx_and_restarts/bluntfinq.bin
03294 ./lib/blt2.4/bltDragdrop.tcl
03299 ./examples/rectangular_duct/rect_duct_030.uns
03321 ./demo/coord3.crd
03329 ./fvx_and_restarts/set_view_TEST.fvx
03592 ./dore/data/il/def/parsekey.sym
03708 ./uil/linux_ia64/dataset_controls.uid
03816 ./geometry/cyln48.atm
04150 ./fvx_and_restarts/seed_rectangle_proto.fvx
04378 ./bin/fvbin
04449 ./examples/rectangular_duct/rect_duct_039.uns
04525 ./fvx_and_restarts/seed_circ_GUI.fvx
04824 ./lib/tcl8.2/encoding/macThai.enc
04999 ./lib/tcl8.2/encoding/macIceland.enc
05048 ./examples/rectangular_duct/rect_duct_040.uns
05077 ./lib/tcl8.2/http2.1/pkgIndex.tcl
05136 ./lib/tcl8.2/tcltest1.0/pkgIndex.tcl
05176 ./lib/tcl8.2/encoding/koi8-r.enc
05986 ./demo/aerospace_LDM.bnd
06058 ./demo/f18i9b_q_bin
06058 ./examples/f18/f18i9b_q_bin
06106 ./lib/tcl8.2/encoding/cp737.enc
06139 ./dore/data/fonts/romant.df
06194 ./lib/tcl8.2/encoding/gb2312.enc
06517 ./bin/fvsrv
06966 ./lib/tcl8.2/encoding/iso8859-8.enc
07340 ./geometry/cyln12.atm
07365 ./examples/rectangular_duct/rect_duct_016.uns
07417 ./uil/linux_ia64/color_mixer.uid
07605 ./demo/stream4.stl
07614 ./demo/t3.ttl
07885 ./bin/fv.template
07969 ./uil/linux_ia64/legend_control.uid
08136 ./lib/tcl8.2/encoding/euc-jp.enc
08160 ./lib/tcl8.2/encoding/cp1254.enc
08221 ./demo/t.ttl
08221 ./demo/t10.ttl
08320 ./examples/rectangular_duct/rect_duct_043.uns
08917 ./user/user_defined_functions.f
09118 ./lib/tcl8.2/encoding/cp862.enc
09161 ./examples/rectangular_duct/rect_duct_018.uns
09510 ./examples/rectangular_duct/rect_duct_015.uns
09563 ./examples/rectangular_duct/rect_duct_068.uns
09565 ./bin/convert
10046 ./demo/lift.fvx
10063 ./bin/lib/ImageMagick-5.5.7/log.mgk
10314 ./lib/tk8.2/obsolete.tcl
10394 ./examples/rectangular_duct/rect_duct_073.uns
10827 ./lib/tcl8.2/http2.1/http.tcl
10855 ./lib/tk8.2/bgerror.tcl
10967 ./doc/FVReference_Manual.pdf
11013 ./examples/rectangular_duct/rect_duct_055.uns
11200 ./dore/data/fonts/romans.df
11217 ./uil/linux_ia64/flip.uid
11396 ./examples/rectangular_duct/rect_duct_011.uns
11412 ./lib/tcl8.2/encoding/cp864.enc
11868 ./lib/tcl8.2/tclIndex
11903 ./lib/tcl8.2/encoding/euc-cn.enc
11963 ./dore/data/fonts/italicc.df
11986 ./bin/plugins/lib/libadb.so
12061 ./uil/linux_ia64/misc_support.uid
12233 ./bin/fvsrvbin
12263 ./dore/data/fonts/cyrillic.df
12397 ./demo/t11.ttl
12501 ./bin/fv_vnc
12721 ./examples/rectangular_duct/rect_duct_028.uns
12833 ./lib/tcl8.2/tcltest1.0/tcltest.tcl
13082 ./fvx_and_restarts/set_view.fvx
13258 ./lib/tcl8.2/encoding/euc-kr.enc
13293 ./fvx_and_restarts/bio.frm
13398 ./demo/t2.ttl
13461 ./examples/rectangular_duct/rect_duct_009.uns
13635 ./bin/mpich/ch_p4/bin/mpirun.ch_p4
13652 ./examples/rectangular_duct/rect_duct_078.uns
13726 ./bin/lib/ImageMagick-5.5.7/delegates.mgk
14117 ./lib/blt2.4/NEWS
14544 ./dore/data/fonts/greekp.df
14611 ./geometry/allobs.ob
14678 ./bin/lib/ImageMagick-5.5.7/modules.mgk
14785 ./uil/linux_ia64/misc_file_input.uid
15065 ./examples/rectangular_duct/rect_duct_033.uns
15605 ./lib/tcl8.2/encoding/macCentEuro.enc
15825 ./lib/tcl8.2/encoding/macCroatian.enc
15887 ./lib/tcl8.2/encoding/macUkraine.enc
16097 ./lib/tcl8.2/encoding/iso8859-6.enc
16325 ./uil/linux_ia64/integration.uid
16373 ./examples/rectangular_duct/rect_duct_008.uns
16561 ./lib/tk8.2/scrlbar.tcl
16650 ./geometry/poly12.atm
16658 ./fvx_and_restarts/seed_circular_proto.fvx
16709 ./lib/tcl8.2/msgcat1.0/pkgIndex.tcl
16741 ./lib/tcl8.2/parray.tcl
16795 ./dore/data/fonts/gothgrt.df
16841 ./lib/tcl8.2/encoding/cp950.enc
16950 ./fvx_and_restarts/animate_streamlines.prd
17116 ./lib/tcl8.2/auto.tcl
17153 ./lib/start_python.luac64
17184 ./demo/start.plt
17317 ./demo/start.dat
17332 ./demo/bound1.bnd
17427 ./lib/tcl8.2/encoding/cp775.enc
17570 ./lib/tcl8.2/encoding/iso2022.enc
17635 ./examples/rectangular_duct/rect_duct_031.uns
17647 ./geometry/poly48.atm
17790 ./lib/tcl8.2/encoding/iso8859-7.enc
17844 ./lib/tcl8.2/encoding/cp932.enc
17920 ./lib/tcl8.2/msgcat1.0/msgcat.tcl
18023 ./examples/rectangular_duct/rect_duct_020.uns
18271 ./lib/tcl8.2/encoding/cp863.enc
18315 ./lib/tcl8.2/encoding/cp869.enc
18328 ./examples/rectangular_duct/rect_duct_019.uns
18422 ./bin/fv_user_print.sh.sample
18472 ./bin/mpich/ch_shmem/bin/mpirun
18484 ./fvx_and_restarts/animate_streamlines.fvx
18619 ./lib/tcl8.2/encoding/macDingbats.enc
18713 ./dore/data/fonts/gothitt.df
18713 ./uil/linux_ia64/comp_surface.uid
19151 ./demo/aerospace_LDM.fvx
19329 ./lib/tcl8.2/encoding/macGreek.enc
19500 ./lib/tcl8.2/encoding/jis0208.enc
19545 ./lib/tcl8.2/encoding/cp861.enc
19776 ./lib/start_python.luac
20094 ./examples/rectangular_duct/rect_duct_069.uns
20144 ./examples/rectangular_duct/rect_duct_056.uns
20418 ./examples/rectangular_duct/rect_duct_007.uns
20504 ./uil/linux_ia64/dynamic_subset.uid
20545 ./examples/rectangular_duct/rect_duct_060.uns
20873 ./dore/data/fonts/unused.df
21157 ./uil/linux_ia64/xdb_info.uid
21460 ./lib/tcl8.2/http1.0/http.tcl
21496 ./data/xdb_vars_FV11.0
21564 ./lib/tcl8.2/ldAout.tcl
21798 ./demo/t7.ttl
21894 ./lib/blt2.4/bltTabset.tcl
21960 ./dore/data/fonts/astrology.df
21969 ./examples/heat_exchanger/counter.uns
22040 ./uns/write_unformatted_uns.F
22410 ./uil/linux_ia64/vector_params.uid
22438 ./lib/tcl8.2/encoding/cp1257.enc
22569 ./examples/rectangular_duct/rect_duct_035.uns
22765 ./data/license.dat
22773 ./uil/linux_ia64/datasetBrowser.uid
22951 ./bin/display
23217 ./demo/wingloadplots.fvx
23286 ./dore/data/fonts/italiccs.df
23323 ./demo/coord1.crd
23344 ./uns/write_ascii_uns.f
23526 ./lib/tk8.2/scale.tcl
23533 ./lib/tcl8.2/encoding/iso8859-5.enc
23541 ./examples/rectangular_duct/rect_duct_005.uns
23548 ./bin/vnc_disp
23614 ./bin/mpich/ch_shmem/bin/mpirun.ch_shmem
23709 ./lib/tk8.2/images/logo64.gif
23752 ./lib/tk8.2/dialog.tcl
23817 ./bin/lmhostid
24188 ./examples/rectangular_duct/rect_duct_001.uns
24249 ./lib/linux_ia64/lua-python.so
24259 ./bin/mpich/ch_p4/bin/mpirun.args
24259 ./bin/mpich/ch_shmem/bin/mpirun.args
24661 ./examples/rectangular_duct/rect_duct_010.uns
24673 ./data/colormaps/c2.col
24693 ./uil/linux_ia64/presentation.uid
24852 ./demo/aerospace_LDM.dat
24881 ./lib/blt2.4/htresize.cur
25189 ./lib/tk8.2/tk.tcl
25502 ./examples/rectangular_duct/rect_duct_054.uns
25598 ./user/user_unstruct_combined.f
25605 ./lib/tcl8.2/encoding/macCyrillic.enc
25703 ./lib/tk8.2/tclIndex
25902 ./bin/plugins/lsdyna_reader.so
26148 ./user/user_combined_file_read.f
26216 ./examples/rectangular_duct/rect_duct_057.uns
26356 ./lib/tk8.2/images/README
26380 ./lib/tk8.2/focus.tcl
26668 ./geometry/sph24.atm
26887 ./bin/mpeg2decode
26892 ./lib/linux_ia64/liblualib.so.4.0
27073 ./fvx_and_restarts/seed_rect_GUI.fvx
27081 ./lib/blt2.4/bltGraph.tcl
27188 ./lib/tk8.2/images/pwrdLogo200.gif
27513 ./geometry/star.atm
27816 ./bin/mpeg2encode
27920 ./bin/lib/ImageMagick-5.5.7/type-ghostscript.mgk
27971 ./bin/lib/ImageMagick-5.5.7/magic.mgk
28042 ./demo/coord0.crd
28310 ./examples/rectangular_duct/rect_duct_003.uns
28324 ./uns/write_split_binary_uns.c
28358 ./lib/blt2.4/dd_protocols/dd-file.tcl
28436 ./user/ftn_fv_close.f
28503 ./uil/linux_ia64/func.uid
28544 ./uil/linux_ia64/data_input.uid
28601 ./demo/coord5.crd
28632 ./examples/rectangular_duct/rect_duct_034.uns
28734 ./lib/blt2.4/dd_protocols/dd-color.tcl
28893 ./examples/rectangular_duct/rect_duct_053.uns
29100 ./lib/tk8.2/listbox.tcl
29194 ./lib/tk8.2/images/pwrdLogo75.gif
29195 ./examples/rectangular_duct/rect_duct_046.uns
29424 ./bin/lib/ImageMagick-5.5.7/colors.mgk
29488 ./examples/rectangular_duct/rect_duct_066.uns
29543 ./lib/tcl8.2/encoding/macRomania.enc
29643 ./fvx_and_restarts/animate_streamlines.vct
29657 ./demo/stream3.stl
29787 ./lib/tk8.2/optMenu.tcl
29826 ./lib/tcl8.2/init.tcl
29845 ./examples/rectangular_duct/rect_duct_023.uns
29936 ./examples/rectangular_duct/rect_duct_014.uns
29942 ./lib/tcl8.2/package.tcl
30113 ./lib/tk8.2/palette.tcl
30177 ./dore/data/fonts/meteorology.df
30254 ./lib/blt2.4/bltDnd.tcl
30349 ./lib/tk8.2/images/tai-ku.gif
30506 ./lib/tcl8.2/encoding/iso2022-kr.enc
30792 ./dore/data/fonts/romanc.df
30929 ./uil/linux_ia64/streamlines.uid
31089 ./lib/tcl8.2/encoding/iso8859-1.enc
31428 ./bin/lib/ImageMagick-5.5.7/type-windows.mgk
31591 ./lib/tcl8.2/encoding/cp866.enc
31691 ./lib/tcl8.2/encoding/cp865.enc
31972 ./uil/linux_ia64/coordinate_surface.uid
32060 ./examples/rectangular_duct/rect_duct_025.uns
32217 ./examples/rectangular_duct/rect_duct_006.uns
32492 ./lib/tcl8.2/encoding/cp852.enc
32548 ./demo/aerospace_LDM.vct
32643 ./fvx_and_restarts/basic_volume_integral.fvx
32649 ./examples/rectangular_duct/rect_duct_051.uns
32725 ./examples/rectangular_duct/rect_duct_022.uns
32837 ./examples/rectangular_duct/rect_duct_050.uns
32927 ./lib/tk8.2/tkAppInit.c
33001 ./lib/tcl8.2/encoding/cp874.enc
33248 ./demo/moments.fvx
33321 ./lib/blt2.4/dd_protocols/dd-text.tcl
33396 ./demo/coord2.crd
33520 ./doc/Readme_Server_Toolkit.txt
33547 ./uns/write_binary_uns.c
34182 ./lib/blt2.4/dd_protocols/tclIndex
34626 ./geometry/sph12.atm
34669 ./lib/tcl8.2/encoding/gb1988.enc
34707 ./bin/mpich/share/machines.sample
34710 ./fvx_and_restarts/wind.frm
34959 ./dore/data/fonts/lcmath.df
35021 ./fvx_and_restarts/animate_streamlines.frm
35313 ./dore/data/fonts/romanp.df
35384 ./geometry/square.atm
35398 ./demo/t4.ttl
35566 ./bin/contest
35629 ./bin/README_Copyrights.txt
35670 ./doc/FVUsers_Guide.pdf
35707 ./lib/tcl8.2/encoding/shiftjis.enc
35818 ./examples/rectangular_duct/rect_duct_063.uns
35841 ./bin/node_id
36055 ./bin/mpich/ch_p4/bin/mpirun.pg
36062 ./bin/plugins/fluent_reader.so
36318 ./examples/rectangular_duct/rect_duct_080.uns
36575 ./fvx_and_restarts/bluntfinx.bin.fvbnd
36641 ./lib/tk8.2/images/pwrdLogo150.gif
36865 ./uns/ftn_fv_reader_tags.h
36925 ./examples/rectangular_duct/rect_duct_004.uns
36979 ./bin/mpich/ch_p4/bin/tarch
36979 ./bin/mpich/ch_shmem/bin/tarch
37055 ./lib/tcl8.2/tclAppInit.c
37272 ./lib/tk8.2/clrpick.tcl
37307 ./lib/tcl8.2/encoding/ksc5601.enc
37409 ./lib/tcl8.2/history.tcl
37439 ./lib/tk8.2/images/pwrdLogo.eps
37477 ./lib/tcl8.2/encoding/cp1258.enc
37544 ./doc/Whats_New_FV12.pdf
37647 ./uil/linux_ia64/boundary_conditions.uid
37670 ./uil/linux_ia64/particle_paths.uid
37688 ./user/ftn_register_functions.f
38287 ./fvx_and_restarts/README_PLOT3D_formula_restart
38461 ./demo/f18i9b_g_bin
38461 ./examples/f18/f18i9b_g_bin
38501 ./dore/data/fonts/italict.df
38531 ./examples/rectangular_duct/rect_duct_036.uns
38545 ./doc/Help_Intro.pdf
38574 ./lib/tk8.2/prolog.ps
38666 ./lib/tk8.2/menu.tcl
38861 ./lib/tk8.2/images/pwrdLogo100.gif
38994 ./lib/tcl8.2/encoding/macTurkish.enc
39109 ./lib/tk8.2/images/pwrdLogo175.gif
39456 ./uil/linux_ia64/region_control.uid
39676 ./lib/tcl8.2/encoding/iso2022-jp.enc
39874 ./demo/f18show.scr
39957 ./uil/linux_ia64/plot.uid
40066 ./user/ftn_register_data_readers.f
40442 ./demo/col1.map
40541 ./user/register_data_readers.c
40552 ./lib/tcl8.2/encoding/symbol.enc
41107 ./lib/tcl8.2/encoding/cp1256.enc
41169 ./geometry/geometry_formats.ps
41277 ./bin/fv_platform
41358 ./demo/stream2.stl
41364 ./lib/tk8.2/images/logo100.gif
41566 ./lib/blt2.4/pkgIndex.tcl
41734 ./demo/iso2.iso
41740 ./bin/mpich/ch_p4/bin/mpirun.rand
42046 ./lib/tcl8.2/encoding/iso8859-9.enc
42054 ./demo/t9.ttl
42217 ./lib/tcl8.2/encoding/macJapan.enc
42218 ./lib/tcl8.2/encoding/cp437.enc
42270 ./examples/rectangular_duct/rect_duct_026.uns
42627 ./dore/data/fonts/ucmath.df
43022 ./bin/fvsrv_p4
43137 ./bin/animate
43278 ./dore/data/fonts/symbols.df
43508 ./lib/tk8.2/images/logo.eps
43701 ./examples/rectangular_duct/rect_duct_041.uns
43751 ./examples/heat_exchanger/cocurrent.uns.fvreg
43751 ./examples/heat_exchanger/counter.uns.fvreg
43852 ./uil/linux_ia64/subset_params.uid
43946 ./fvx_and_restarts/animate_streamlines.dat
43960 ./user/user_unstruct_grid.f
44519 ./lib/tcl8.2/ldAix
44814 ./examples/rectangular_duct/rect_duct_042.uns
44845 ./bin/mpich/ch_p4/bin/mpirun
44867 ./demo/start.vct
44875 ./user/user_unstruct_results.f
44878 ./lib/tcl8.2/encoding/jis0201.enc
44926 ./examples/rectangular_duct/rect_duct_037.uns
45001 ./lib/linux_ia64/liblua.so.4.0
45085 ./sconfig/srv.template
45260 ./bin/plugins/acusolve_reader.so
45438 ./examples/rectangular_duct/rect_duct_032.uns
45540 ./lib/fv.pyc
45642 ./dore/data/fonts/gothgbt.df
45830 ./demo/iso3.iso
45880 ./lib/tcl8.2/encoding/cp1250.enc
46000 ./examples/rectangular_duct/rect_duct_013.uns
46023 ./lib/tk8.2/images/logoMed.gif
46093 ./demo/view2.vct
46103 ./examples/rectangular_duct/rect_duct_047.uns
46182 ./uil/linux_ia64/point_probe.uid
46238 ./demo/start.map
46455 ./lib/tcl8.2/encoding/cp1255.enc
46496 ./lib/blt2.4/bltHierbox.tcl
46844 ./geometry/cone12.atm
47054 ./fvx_and_restarts/seed_sphere_proto.fvx
47357 ./uns/write_split_unformatted_uns.F
47891 ./demo/view6.vct
47930 ./demo/drag.fvx
47970 ./lib/tcl8.2/safe.tcl
48023 ./lib/tcl8.2/encoding/cp936.enc
48142 ./lib/blt2.4/bltHiertable.tcl
48291 ./data/colormaps/c7.col
48295 ./fvx_and_restarts/xcolor_defines.fvx
48438 ./examples/rectangular_duct/rect_duct_059.uns
48906 ./dore/data/fonts/greekcs.df
49045 ./fvx_and_restarts/seed_sphere_GUI.fvx
49051 ./demo/view4.vct
49338 ./demo/view3.vct
49417 ./lib/tcl8.2/encoding/ascii.enc
49617 ./demo/start.bnd
49824 ./lib/tcl8.2/encoding/iso8859-3.enc
49896 ./geometry/cube.atm
49920 ./fvx_and_restarts/lsdyna_formulas.frm
49926 ./data/colormaps/c4.col
49932 ./demo/t8.ttl
50329 ./demo/view7.vct
50415 ./examples/rectangular_duct/rect_duct_061.uns
50427 ./examples/rectangular_duct/rect_duct_077.uns
50441 ./bin/prolog.ps
50605 ./examples/rectangular_duct/rect_duct_038.uns
51123 ./examples/rectangular_duct/rect_duct_021.uns
51155 ./uil/linux_ia64/vr.uid
51494 ./bin/fv_convert.template
51968 ./examples/rectangular_duct/rect_duct_065.uns
52013 ./lib/tcl8.2/encoding/cp855.enc
52057 ./examples/rectangular_duct/rect_duct_076.uns
52309 ./lib/tcl8.2/opt0.4/pkgIndex.tcl
52340 ./examples/rectangular_duct/rect_duct_049.uns
52415 ./demo/start.ttl
52745 ./dore/data/fonts/romand.df
52833 ./lib/tcl8.2/encoding/cp850.enc
53221 ./lib/blt2.4/tclIndex
53293 ./examples/rectangular_duct/rect_duct_052.uns
53473 ./lib/tk8.2/comdlg.tcl
53516 ./examples/rectangular_duct/rect_duct_058.uns
53968 ./user/toolkit.c
54002 ./lib/tcl8.2/encoding/cp860.enc
54102 ./fvx_and_restarts/calc_volume_integral.fvx
54228 ./sconfig/manual.srv
54250 ./demo/plot1.lpt
54260 ./demo/plot2.lpt
54331 ./bin/make_fv
54533 ./dore/data/fonts/romancs.df
54905 ./dore/data/fonts/helvetica.df
55047 ./examples/rectangular_duct/rect_duct.trk
55208 ./fvx_and_restarts/Q-criterion.frm
55231 ./lib/tcl8.2/http1.0/pkgIndex.tcl
55281 ./uil/linux_ia64/title_panel.uid
55361 ./lib/tk8.2/xmfbox.tcl
55552 ./examples/rectangular_duct/rect_duct_074.uns
55614 ./examples/rectangular_duct/rect_duct_045.uns
55673 ./lib/tcl8.2/encoding/cp1251.enc
55709 ./examples/rectangular_duct/rect_duct_075.uns
55734 ./dore/data/fonts/scriptc.df
55885 ./lib/tk8.2/images/logoLarge.gif
56003 ./data/colormaps/c5.col
56186 ./dore/data/fonts/greeks.df
56255 ./examples/rectangular_duct/rect_duct_002.uns
56320 ./demo/t6.ttl
56378 ./examples/rectangular_duct/rect_duct_070.uns
56516 ./uil/linux_ia64/colormap_specification.uid
56695 ./lib/tcl8.2/encoding/iso8859-2.enc
56801 ./bin/mpich/ch_p4/bin/mpirun.p4shmem
56862 ./bin/mpich/ch_p4/bin/mpirun.ch_p4.args
56916 ./lib/blt2.4/bltTabnotebook.tcl
56960 ./dore/data/fonts/greekc.df
57234 ./user/user_unstruct_functions.f
57480 ./demo/aerospace_LDM.frm
57686 ./lib/blt2.4/Problems
57698 ./fvx_and_restarts/FVX_Templates.pdf
57711 ./lib/tk8.2/entry.tcl
57757 ./examples/rectangular_duct/rect_duct_064.uns
57941 ./lib/tcl8.2/encoding/dingbats.enc
58071 ./lib/tcl8.2/opt0.4/optparse.tcl
58217 ./bin/lib/ImageMagick-5.5.7/type.mgk
58430 ./fvx_and_restarts/animate_streamlines.bnd
58616 ./fvx_and_restarts/plot_along_streamline.fvx
58635 ./demo/start.frm
58764 ./lib/tk8.2/tearoff.tcl
59310 ./demo/t0.ttl
59391 ./lib/tcl8.2/encoding/iso8859-4.enc
59497 ./doc/How_to.pdf
59606 ./uil/linux_ia64/transient_datav2.uid
59695 ./examples/rectangular_duct/rect_duct_071.uns
60258 ./dore/data/errmsg/errmsgfile
60621 ./data/colormaps/c6.col
60693 ./examples/rectangular_duct/rect_duct_079.uns
60841 ./bin/lib/ImageMagick-5.5.7/type-solaris.mgk
61035 ./lib/blt2.4/bltGraph.pro
61117 ./lib/tcl8.2/word.tcl
61247 ./demo/blank.ttl
61446 ./uns/fv_reader_tags.h
61497 ./dore/data/fonts/music.df
61541 ./lib/tk8.2/console.tcl
61734 ./demo/coord4.crd
61852 ./demo/view5.vct
62091 ./uil/linux_ia64/keyframe.uid
62390 ./user/register_functions.c
62413 ./demo/t5.ttl
62662 ./demo/t1.ttl
62705 ./lib/linux_ia64/dummy/libGLU.so.1
62727 ./bin/fvsrv_shmem
62774 ./uil/linux_ia64/iso_surface.uid
62907 ./demo/stream1.stl
62918 ./fvx_and_restarts/bluntfinx.bin
63021 ./examples/rectangular_duct/rect_duct_017.uns
63102 ./lib/tcl8.2/encoding/macRoman.enc
63106 ./examples/heat_exchanger/cocurrent.uns
63221 ./lib/tk8.2/safetk.tcl
63321 ./bin/ld_fv
63340 ./demo/iso4.iso
63384 ./examples/rectangular_duct/rect_duct_027.uns
63671 ./geometry/cone48.atm
63850 ./lib/tcl8.2/encoding/cp1252.enc
64367 ./uil/linux_ia64/formula.uid
64512 ./examples/rectangular_duct/rect_duct_072.uns
64515 ./data/colormaps/c8.col
64986 ./lib/blt2.4/Readme
65154 ./lib/blt2.4/bltCanvEps.pro
65304 ./data/colormaps/c1.col
65467 ./dore/data/fonts/scripts.df
65505 ./lib/linux_ia64/dummy/libGL.so.1
EOF
;;
7)
# checksums for Linux_amd64 version of fv directory
cat > $checkold <<EOF
00023 ./lib/tk8.2/tkfbox.tcl
00052 ./lib/tcl8.2/encoding/big5.enc
00064 ./lib/tk8.2/text.tcl
00273 ./lib/tcl8.2/encoding/cp1253.enc
00281 ./lib/tcl8.2/encoding/gb12345.enc
00300 ./user/user_results_file_read.f
00328 ./demo/iso1.iso
00349 ./bin/fv_to_printer.sh.sample
00455 ./lib/tcl8.2/encoding/cp857.enc
00685 ./examples/rectangular_duct/rect_duct_044.uns
00715 ./examples/rectangular_duct/rect_duct_048.uns
00795 ./lib/tcl8.2/encoding/jis0212.enc
01035 ./bin/fvsrvbin
01283 ./examples/rectangular_duct/rect_duct_029.uns
01383 ./lib/tk8.2/button.tcl
01478 ./fvx_and_restarts/animate_streamlines.map
01677 ./demo/f18i9b_g_bin.fvbnd
01677 ./examples/f18/f18i9b_g_bin.fvbnd
01823 ./bin/plugins/lib/libimf.so
02009 ./examples/rectangular_duct/rect_duct_024.uns
02213 ./examples/rectangular_duct/rect_duct_012.uns
02225 ./fvx_and_restarts/plot3d_eqns.frm
02235 ./lib/tk8.2/msgbox.tcl
02266 ./lib/tcl8.2/encoding/cp949.enc
02317 ./data/xdb_vars
02325 ./lib/blt2.4/dd_protocols/dd-number.tcl
02345 ./user/user_grid_file_read.f
02492 ./examples/rectangular_duct/rect_duct_062.uns
02682 ./doc/Readme_InstallNotes_12.pdf
02722 ./data/colormaps/c3.col
02780 ./uns/write_split_ascii_uns.f
02844 ./examples/rectangular_duct/rect_duct_067.uns
03030 ./fvx_and_restarts/bluntfinq.bin
03294 ./lib/blt2.4/bltDragdrop.tcl
03299 ./examples/rectangular_duct/rect_duct_030.uns
03321 ./demo/coord3.crd
03329 ./fvx_and_restarts/set_view_TEST.fvx
03592 ./dore/data/il/def/parsekey.sym
03816 ./geometry/cyln48.atm
04150 ./fvx_and_restarts/seed_rectangle_proto.fvx
04449 ./examples/rectangular_duct/rect_duct_039.uns
04525 ./fvx_and_restarts/seed_circ_GUI.fvx
04824 ./lib/tcl8.2/encoding/macThai.enc
04999 ./lib/tcl8.2/encoding/macIceland.enc
05048 ./examples/rectangular_duct/rect_duct_040.uns
05076 ./uil/linux_amd64/presentation.uid
05077 ./lib/tcl8.2/http2.1/pkgIndex.tcl
05136 ./lib/tcl8.2/tcltest1.0/pkgIndex.tcl
05176 ./lib/tcl8.2/encoding/koi8-r.enc
05986 ./demo/aerospace_LDM.bnd
06058 ./demo/f18i9b_q_bin
06058 ./examples/f18/f18i9b_q_bin
06106 ./lib/tcl8.2/encoding/cp737.enc
06139 ./dore/data/fonts/romant.df
06194 ./lib/tcl8.2/encoding/gb2312.enc
06517 ./bin/fvsrv
06966 ./lib/tcl8.2/encoding/iso8859-8.enc
07001 ./uil/linux_amd64/transient_datav2.uid
07162 ./bin/plugins/lib/libirc.so
07205 ./bin/vnc_disp
07340 ./geometry/cyln12.atm
07365 ./examples/rectangular_duct/rect_duct_016.uns
07605 ./demo/stream4.stl
07614 ./demo/t3.ttl
07885 ./bin/fv.template
08136 ./lib/tcl8.2/encoding/euc-jp.enc
08160 ./lib/tcl8.2/encoding/cp1254.enc
08221 ./demo/t.ttl
08221 ./demo/t10.ttl
08320 ./examples/rectangular_duct/rect_duct_043.uns
08771 ./lib/linux_amd64/dummy/libGL.so.1
08917 ./user/user_defined_functions.f
09118 ./lib/tcl8.2/encoding/cp862.enc
09161 ./examples/rectangular_duct/rect_duct_018.uns
09510 ./examples/rectangular_duct/rect_duct_015.uns
09563 ./examples/rectangular_duct/rect_duct_068.uns
10046 ./demo/lift.fvx
10063 ./bin/lib/ImageMagick-5.5.7/log.mgk
10314 ./lib/tk8.2/obsolete.tcl
10394 ./examples/rectangular_duct/rect_duct_073.uns
10827 ./lib/tcl8.2/http2.1/http.tcl
10855 ./lib/tk8.2/bgerror.tcl
10967 ./doc/FVReference_Manual.pdf
11013 ./examples/rectangular_duct/rect_duct_055.uns
11200 ./dore/data/fonts/romans.df
11396 ./examples/rectangular_duct/rect_duct_011.uns
11412 ./lib/tcl8.2/encoding/cp864.enc
11868 ./lib/tcl8.2/tclIndex
11903 ./lib/tcl8.2/encoding/euc-cn.enc
11963 ./dore/data/fonts/italicc.df
12263 ./dore/data/fonts/cyrillic.df
12397 ./demo/t11.ttl
12501 ./bin/fv_vnc
12721 ./examples/rectangular_duct/rect_duct_028.uns
12833 ./lib/tcl8.2/tcltest1.0/tcltest.tcl
13082 ./fvx_and_restarts/set_view.fvx
13129 ./uil/linux_amd64/vortex_core.uid
13258 ./lib/tcl8.2/encoding/euc-kr.enc
13293 ./fvx_and_restarts/bio.frm
13398 ./demo/t2.ttl
13461 ./examples/rectangular_duct/rect_duct_009.uns
13493 ./uil/linux_amd64/surface_plot.uid
13591 ./bin/plugins/lsdyna_reader.so
13635 ./bin/mpich/ch_p4/bin/mpirun.ch_p4
13652 ./examples/rectangular_duct/rect_duct_078.uns
13726 ./bin/lib/ImageMagick-5.5.7/delegates.mgk
14117 ./lib/blt2.4/NEWS
14544 ./dore/data/fonts/greekp.df
14611 ./geometry/allobs.ob
14678 ./bin/lib/ImageMagick-5.5.7/modules.mgk
14832 ./uil/linux_amd64/color_mixer.uid
15065 ./examples/rectangular_duct/rect_duct_033.uns
15299 ./uil/linux_amd64/coordinate_surface.uid
15605 ./lib/tcl8.2/encoding/macCentEuro.enc
15825 ./lib/tcl8.2/encoding/macCroatian.enc
15887 ./lib/tcl8.2/encoding/macUkraine.enc
16097 ./lib/tcl8.2/encoding/iso8859-6.enc
16373 ./examples/rectangular_duct/rect_duct_008.uns
16561 ./lib/tk8.2/scrlbar.tcl
16590 ./uil/linux_amd64/boundary_conditions.uid
16650 ./geometry/poly12.atm
16658 ./fvx_and_restarts/seed_circular_proto.fvx
16709 ./lib/tcl8.2/msgcat1.0/pkgIndex.tcl
16741 ./lib/tcl8.2/parray.tcl
16795 ./dore/data/fonts/gothgrt.df
16841 ./lib/tcl8.2/encoding/cp950.enc
16950 ./fvx_and_restarts/animate_streamlines.prd
17084 ./bin/plugins/fluent_reader.so
17116 ./lib/tcl8.2/auto.tcl
17153 ./lib/start_python.luac64
17181 ./bin/server_name
17184 ./demo/start.plt
17317 ./demo/start.dat
17332 ./demo/bound1.bnd
17427 ./lib/tcl8.2/encoding/cp775.enc
17570 ./lib/tcl8.2/encoding/iso2022.enc
17635 ./examples/rectangular_duct/rect_duct_031.uns
17647 ./geometry/poly48.atm
17790 ./lib/tcl8.2/encoding/iso8859-7.enc
17844 ./lib/tcl8.2/encoding/cp932.enc
17920 ./lib/tcl8.2/msgcat1.0/msgcat.tcl
18023 ./examples/rectangular_duct/rect_duct_020.uns
18271 ./lib/tcl8.2/encoding/cp863.enc
18315 ./lib/tcl8.2/encoding/cp869.enc
18328 ./examples/rectangular_duct/rect_duct_019.uns
18422 ./bin/fv_user_print.sh.sample
18484 ./fvx_and_restarts/animate_streamlines.fvx
18619 ./lib/tcl8.2/encoding/macDingbats.enc
18688 ./bin/mpeg2decode
18713 ./dore/data/fonts/gothitt.df
19151 ./demo/aerospace_LDM.fvx
19329 ./lib/tcl8.2/encoding/macGreek.enc
19500 ./lib/tcl8.2/encoding/jis0208.enc
19545 ./lib/tcl8.2/encoding/cp861.enc
19776 ./lib/start_python.luac
20055 ./uil/linux_amd64/datasetBrowser.uid
20094 ./examples/rectangular_duct/rect_duct_069.uns
20144 ./examples/rectangular_duct/rect_duct_056.uns
20418 ./examples/rectangular_duct/rect_duct_007.uns
20545 ./examples/rectangular_duct/rect_duct_060.uns
20688 ./bin/animate
20873 ./dore/data/fonts/unused.df
21460 ./lib/tcl8.2/http1.0/http.tcl
21496 ./data/xdb_vars_FV11.0
21564 ./lib/tcl8.2/ldAout.tcl
21798 ./demo/t7.ttl
21894 ./lib/blt2.4/bltTabset.tcl
21951 ./bin/plugins/lib/libsvml.so
21960 ./dore/data/fonts/astrology.df
21969 ./examples/heat_exchanger/counter.uns
22040 ./uns/write_unformatted_uns.F
22438 ./lib/tcl8.2/encoding/cp1257.enc
22491 ./uil/linux_amd64/integration.uid
22569 ./examples/rectangular_duct/rect_duct_035.uns
22756 ./uil/linux_amd64/misc_file_input.uid
22765 ./data/license.dat
22895 ./uil/linux_amd64/streamlines.uid
23217 ./demo/wingloadplots.fvx
23286 ./dore/data/fonts/italiccs.df
23323 ./demo/coord1.crd
23344 ./uns/write_ascii_uns.f
23526 ./lib/tk8.2/scale.tcl
23533 ./lib/tcl8.2/encoding/iso8859-5.enc
23541 ./examples/rectangular_duct/rect_duct_005.uns
23614 ./bin/mpich/ch_shmem/bin/mpirun.ch_shmem
23709 ./lib/tk8.2/images/logo64.gif
23752 ./lib/tk8.2/dialog.tcl
24188 ./examples/rectangular_duct/rect_duct_001.uns
24259 ./bin/mpich/ch_p4/bin/mpirun.args
24259 ./bin/mpich/ch_shmem/bin/mpirun.args
24661 ./examples/rectangular_duct/rect_duct_010.uns
24673 ./data/colormaps/c2.col
24742 ./uil/linux_amd64/comp_surface.uid
24852 ./demo/aerospace_LDM.dat
24881 ./lib/blt2.4/htresize.cur
25189 ./lib/tk8.2/tk.tcl
25461 ./uil/linux_amd64/iso_surface.uid
25502 ./examples/rectangular_duct/rect_duct_054.uns
25598 ./user/user_unstruct_combined.f
25605 ./lib/tcl8.2/encoding/macCyrillic.enc
25682 ./bin/display
25703 ./lib/tk8.2/tclIndex
26148 ./user/user_combined_file_read.f
26216 ./examples/rectangular_duct/rect_duct_057.uns
26240 ./lib/linux_amd64/liblualib.so.4.0
26356 ./lib/tk8.2/images/README
26361 ./bin/mpich/ch_shmem/bin/mpirun
26380 ./lib/tk8.2/focus.tcl
26664 ./uil/linux_amd64/vector_params.uid
26668 ./geometry/sph24.atm
27067 ./uil/linux_amd64/formula.uid
27073 ./fvx_and_restarts/seed_rect_GUI.fvx
27081 ./lib/blt2.4/bltGraph.tcl
27188 ./lib/tk8.2/images/pwrdLogo200.gif
27513 ./geometry/star.atm
27920 ./bin/lib/ImageMagick-5.5.7/type-ghostscript.mgk
27971 ./bin/lib/ImageMagick-5.5.7/magic.mgk
28042 ./demo/coord0.crd
28156 ./bin/plugins/lib/libifport.so.5
28310 ./examples/rectangular_duct/rect_duct_003.uns
28324 ./uns/write_split_binary_uns.c
28358 ./lib/blt2.4/dd_protocols/dd-file.tcl
28414 ./uil/linux_amd64/title_panel.uid
28436 ./user/ftn_fv_close.f
28539 ./uil/linux_amd64/plot.uid
28601 ./demo/coord5.crd
28632 ./examples/rectangular_duct/rect_duct_034.uns
28734 ./lib/blt2.4/dd_protocols/dd-color.tcl
28893 ./examples/rectangular_duct/rect_duct_053.uns
29100 ./lib/tk8.2/listbox.tcl
29194 ./lib/tk8.2/images/pwrdLogo75.gif
29195 ./examples/rectangular_duct/rect_duct_046.uns
29424 ./bin/lib/ImageMagick-5.5.7/colors.mgk
29488 ./examples/rectangular_duct/rect_duct_066.uns
29543 ./lib/tcl8.2/encoding/macRomania.enc
29643 ./fvx_and_restarts/animate_streamlines.vct
29657 ./demo/stream3.stl
29787 ./lib/tk8.2/optMenu.tcl
29826 ./lib/tcl8.2/init.tcl
29845 ./examples/rectangular_duct/rect_duct_023.uns
29936 ./examples/rectangular_duct/rect_duct_014.uns
29942 ./lib/tcl8.2/package.tcl
30113 ./lib/tk8.2/palette.tcl
30177 ./dore/data/fonts/meteorology.df
30254 ./lib/blt2.4/bltDnd.tcl
30349 ./lib/tk8.2/images/tai-ku.gif
30506 ./lib/tcl8.2/encoding/iso2022-kr.enc
30792 ./dore/data/fonts/romanc.df
31089 ./lib/tcl8.2/encoding/iso8859-1.enc
31428 ./bin/lib/ImageMagick-5.5.7/type-windows.mgk
31591 ./lib/tcl8.2/encoding/cp866.enc
31691 ./lib/tcl8.2/encoding/cp865.enc
31956 ./lib/linux_amd64/dummy/libGLU.so.1
32060 ./examples/rectangular_duct/rect_duct_025.uns
32217 ./examples/rectangular_duct/rect_duct_006.uns
32327 ./lib/linux_amd64/lua-python.so
32492 ./lib/tcl8.2/encoding/cp852.enc
32548 ./demo/aerospace_LDM.vct
32643 ./fvx_and_restarts/basic_volume_integral.fvx
32649 ./examples/rectangular_duct/rect_duct_051.uns
32725 ./examples/rectangular_duct/rect_duct_022.uns
32837 ./examples/rectangular_duct/rect_duct_050.uns
32927 ./lib/tk8.2/tkAppInit.c
33001 ./lib/tcl8.2/encoding/cp874.enc
33248 ./demo/moments.fvx
33321 ./lib/blt2.4/dd_protocols/dd-text.tcl
33396 ./demo/coord2.crd
33433 ./uil/linux_amd64/misc_support.uid
33448 ./bin/plugins/lib/libflsrdr.so
33520 ./doc/Readme_Server_Toolkit.txt
33547 ./uns/write_binary_uns.c
34001 ./bin/plugins/flow3d_reader.so
34182 ./lib/blt2.4/dd_protocols/tclIndex
34355 ./uil/linux_amd64/dynamic_subset.uid
34510 ./uil/linux_amd64/keyframe.uid
34626 ./geometry/sph12.atm
34669 ./lib/tcl8.2/encoding/gb1988.enc
34707 ./bin/mpich/share/machines.sample
34710 ./fvx_and_restarts/wind.frm
34959 ./dore/data/fonts/lcmath.df
35021 ./fvx_and_restarts/animate_streamlines.frm
35313 ./dore/data/fonts/romanp.df
35384 ./geometry/square.atm
35398 ./demo/t4.ttl
35629 ./bin/README_Copyrights.txt
35670 ./doc/FVUsers_Guide.pdf
35707 ./lib/tcl8.2/encoding/shiftjis.enc
35818 ./examples/rectangular_duct/rect_duct_063.uns
35841 ./bin/node_id
36055 ./bin/mpich/ch_p4/bin/mpirun.pg
36318 ./examples/rectangular_duct/rect_duct_080.uns
36575 ./fvx_and_restarts/bluntfinx.bin.fvbnd
36641 ./lib/tk8.2/images/pwrdLogo150.gif
36697 ./bin/plugins/acusolve_reader.so
36861 ./uil/linux_amd64/func.uid
36865 ./uns/ftn_fv_reader_tags.h
36925 ./examples/rectangular_duct/rect_duct_004.uns
36979 ./bin/mpich/ch_p4/bin/tarch
36979 ./bin/mpich/ch_shmem/bin/tarch
37055 ./lib/tcl8.2/tclAppInit.c
37060 ./bin/fv_platform
37272 ./lib/tk8.2/clrpick.tcl
37307 ./lib/tcl8.2/encoding/ksc5601.enc
37409 ./lib/tcl8.2/history.tcl
37439 ./lib/tk8.2/images/pwrdLogo.eps
37477 ./lib/tcl8.2/encoding/cp1258.enc
37544 ./doc/Whats_New_FV12.pdf
37582 ./bin/fvsrv_p4
37688 ./user/ftn_register_functions.f
37689 ./lib/linux_amd64/liblua.so.4.0
38131 ./bin/lmhostid
38287 ./fvx_and_restarts/README_PLOT3D_formula_restart
38461 ./demo/f18i9b_g_bin
38461 ./examples/f18/f18i9b_g_bin
38501 ./dore/data/fonts/italict.df
38531 ./examples/rectangular_duct/rect_duct_036.uns
38545 ./doc/Help_Intro.pdf
38574 ./lib/tk8.2/prolog.ps
38666 ./lib/tk8.2/menu.tcl
38861 ./lib/tk8.2/images/pwrdLogo100.gif
38994 ./lib/tcl8.2/encoding/macTurkish.enc
39109 ./lib/tk8.2/images/pwrdLogo175.gif
39278 ./uil/linux_amd64/particle_paths.uid
39676 ./lib/tcl8.2/encoding/iso2022-jp.enc
39874 ./demo/f18show.scr
40066 ./user/ftn_register_data_readers.f
40442 ./demo/col1.map
40541 ./user/register_data_readers.c
40552 ./lib/tcl8.2/encoding/symbol.enc
41107 ./lib/tcl8.2/encoding/cp1256.enc
41169 ./geometry/geometry_formats.ps
41358 ./demo/stream2.stl
41364 ./lib/tk8.2/images/logo100.gif
41566 ./lib/blt2.4/pkgIndex.tcl
41734 ./demo/iso2.iso
41740 ./bin/mpich/ch_p4/bin/mpirun.rand
42046 ./lib/tcl8.2/encoding/iso8859-9.enc
42054 ./demo/t9.ttl
42217 ./lib/tcl8.2/encoding/macJapan.enc
42218 ./lib/tcl8.2/encoding/cp437.enc
42270 ./examples/rectangular_duct/rect_duct_026.uns
42357 ./bin/plugins/lib/libifcore.so.5
42468 ./uil/linux_amd64/dataset_controls.uid
42627 ./dore/data/fonts/ucmath.df
43278 ./dore/data/fonts/symbols.df
43463 ./uil/linux_amd64/subset_params.uid
43508 ./lib/tk8.2/images/logo.eps
43701 ./examples/rectangular_duct/rect_duct_041.uns
43751 ./examples/heat_exchanger/cocurrent.uns.fvreg
43751 ./examples/heat_exchanger/counter.uns.fvreg
43946 ./fvx_and_restarts/animate_streamlines.dat
43960 ./user/user_unstruct_grid.f
44519 ./lib/tcl8.2/ldAix
44814 ./examples/rectangular_duct/rect_duct_042.uns
44867 ./demo/start.vct
44875 ./user/user_unstruct_results.f
44878 ./lib/tcl8.2/encoding/jis0201.enc
44926 ./examples/rectangular_duct/rect_duct_037.uns
45085 ./sconfig/srv.template
45438 ./examples/rectangular_duct/rect_duct_032.uns
45458 ./uil/linux_amd64/legend_control.uid
45540 ./lib/fv.pyc
45642 ./dore/data/fonts/gothgbt.df
45830 ./demo/iso3.iso
45880 ./lib/tcl8.2/encoding/cp1250.enc
46000 ./examples/rectangular_duct/rect_duct_013.uns
46023 ./lib/tk8.2/images/logoMed.gif
46093 ./demo/view2.vct
46100 ./bin/fvbin
46103 ./examples/rectangular_duct/rect_duct_047.uns
46238 ./demo/start.map
46455 ./lib/tcl8.2/encoding/cp1255.enc
46496 ./lib/blt2.4/bltHierbox.tcl
46844 ./geometry/cone12.atm
47054 ./fvx_and_restarts/seed_sphere_proto.fvx
47357 ./uns/write_split_unformatted_uns.F
47891 ./demo/view6.vct
47930 ./demo/drag.fvx
47970 ./lib/tcl8.2/safe.tcl
48023 ./lib/tcl8.2/encoding/cp936.enc
48142 ./lib/blt2.4/bltHiertable.tcl
48291 ./data/colormaps/c7.col
48295 ./fvx_and_restarts/xcolor_defines.fvx
48337 ./bin/plugins/lib/libfv_dr.so
48438 ./examples/rectangular_duct/rect_duct_059.uns
48757 ./bin/plugins/lib/libadb.so
48906 ./dore/data/fonts/greekcs.df
49045 ./fvx_and_restarts/seed_sphere_GUI.fvx
49051 ./demo/view4.vct
49338 ./demo/view3.vct
49417 ./lib/tcl8.2/encoding/ascii.enc
49617 ./demo/start.bnd
49824 ./lib/tcl8.2/encoding/iso8859-3.enc
49896 ./geometry/cube.atm
49920 ./fvx_and_restarts/lsdyna_formulas.frm
49926 ./data/colormaps/c4.col
49932 ./demo/t8.ttl
50329 ./demo/view7.vct
50415 ./examples/rectangular_duct/rect_duct_061.uns
50427 ./examples/rectangular_duct/rect_duct_077.uns
50441 ./bin/prolog.ps
50605 ./examples/rectangular_duct/rect_duct_038.uns
50761 ./uil/linux_amd64/point_probe.uid
51123 ./examples/rectangular_duct/rect_duct_021.uns
51494 ./bin/fv_convert.template
51968 ./examples/rectangular_duct/rect_duct_065.uns
52013 ./lib/tcl8.2/encoding/cp855.enc
52057 ./examples/rectangular_duct/rect_duct_076.uns
52309 ./lib/tcl8.2/opt0.4/pkgIndex.tcl
52340 ./examples/rectangular_duct/rect_duct_049.uns
52415 ./demo/start.ttl
52745 ./dore/data/fonts/romand.df
52833 ./lib/tcl8.2/encoding/cp850.enc
53221 ./lib/blt2.4/tclIndex
53293 ./examples/rectangular_duct/rect_duct_052.uns
53401 ./bin/plugins/cgns_reader.so
53473 ./lib/tk8.2/comdlg.tcl
53516 ./examples/rectangular_duct/rect_duct_058.uns
53968 ./user/toolkit.c
54002 ./lib/tcl8.2/encoding/cp860.enc
54102 ./fvx_and_restarts/calc_volume_integral.fvx
54228 ./sconfig/manual.srv
54250 ./demo/plot1.lpt
54260 ./demo/plot2.lpt
54331 ./bin/make_fv
54533 ./dore/data/fonts/romancs.df
54905 ./dore/data/fonts/helvetica.df
55047 ./examples/rectangular_duct/rect_duct.trk
55208 ./fvx_and_restarts/Q-criterion.frm
55231 ./lib/tcl8.2/http1.0/pkgIndex.tcl
55361 ./lib/tk8.2/xmfbox.tcl
55397 ./uil/linux_amd64/vr.uid
55552 ./examples/rectangular_duct/rect_duct_074.uns
55614 ./examples/rectangular_duct/rect_duct_045.uns
55673 ./lib/tcl8.2/encoding/cp1251.enc
55709 ./examples/rectangular_duct/rect_duct_075.uns
55734 ./dore/data/fonts/scriptc.df
55847 ./uil/linux_amd64/region_control.uid
55885 ./lib/tk8.2/images/logoLarge.gif
56003 ./data/colormaps/c5.col
56033 ./bin/mpeg2encode
56109 ./bin/fvsrv_shmem
56186 ./dore/data/fonts/greeks.df
56255 ./examples/rectangular_duct/rect_duct_002.uns
56320 ./demo/t6.ttl
56378 ./examples/rectangular_duct/rect_duct_070.uns
56695 ./lib/tcl8.2/encoding/iso8859-2.enc
56801 ./bin/mpich/ch_p4/bin/mpirun.p4shmem
56862 ./bin/mpich/ch_p4/bin/mpirun.ch_p4.args
56916 ./lib/blt2.4/bltTabnotebook.tcl
56960 ./dore/data/fonts/greekc.df
57052 ./bin/convert
57136 ./bin/plugins/lib/libcgns.so
57234 ./user/user_unstruct_functions.f
57480 ./demo/aerospace_LDM.frm
57686 ./lib/blt2.4/Problems
57698 ./fvx_and_restarts/FVX_Templates.pdf
57711 ./lib/tk8.2/entry.tcl
57757 ./examples/rectangular_duct/rect_duct_064.uns
57941 ./lib/tcl8.2/encoding/dingbats.enc
58071 ./lib/tcl8.2/opt0.4/optparse.tcl
58217 ./bin/lib/ImageMagick-5.5.7/type.mgk
58430 ./fvx_and_restarts/animate_streamlines.bnd
58616 ./fvx_and_restarts/plot_along_streamline.fvx
58635 ./demo/start.frm
58764 ./lib/tk8.2/tearoff.tcl
59310 ./demo/t0.ttl
59391 ./lib/tcl8.2/encoding/iso8859-4.enc
59497 ./doc/How_to.pdf
59695 ./examples/rectangular_duct/rect_duct_071.uns
60016 ./uil/linux_amd64/colormap_specification.uid
60258 ./dore/data/errmsg/errmsgfile
60621 ./data/colormaps/c6.col
60693 ./examples/rectangular_duct/rect_duct_079.uns
60841 ./bin/lib/ImageMagick-5.5.7/type-solaris.mgk
60867 ./bin/mpich/ch_p4/bin/mpirun
60968 ./uil/linux_amd64/xdb_info.uid
61035 ./lib/blt2.4/bltGraph.pro
61117 ./lib/tcl8.2/word.tcl
61247 ./demo/blank.ttl
61446 ./uns/fv_reader_tags.h
61497 ./dore/data/fonts/music.df
61541 ./lib/tk8.2/console.tcl
61734 ./demo/coord4.crd
61852 ./demo/view5.vct
62390 ./user/register_functions.c
62413 ./demo/t5.ttl
62662 ./demo/t1.ttl
62907 ./demo/stream1.stl
62918 ./fvx_and_restarts/bluntfinx.bin
63021 ./examples/rectangular_duct/rect_duct_017.uns
63102 ./lib/tcl8.2/encoding/macRoman.enc
63106 ./examples/heat_exchanger/cocurrent.uns
63221 ./lib/tk8.2/safetk.tcl
63321 ./bin/ld_fv
63340 ./demo/iso4.iso
63384 ./examples/rectangular_duct/rect_duct_027.uns
63619 ./uil/linux_amd64/flip.uid
63671 ./geometry/cone48.atm
63850 ./lib/tcl8.2/encoding/cp1252.enc
64512 ./examples/rectangular_duct/rect_duct_072.uns
64515 ./data/colormaps/c8.col
64761 ./uil/linux_amd64/data_input.uid
64772 ./bin/contest
64927 ./uil/linux_amd64/xform_controls.uid
64986 ./lib/blt2.4/Readme
65154 ./lib/blt2.4/bltCanvEps.pro
65304 ./data/colormaps/c1.col
65467 ./dore/data/fonts/scripts.df
EOF
;;
8)
# checksums for Mac OS X version of fv directory
cat > $checkold <<EOF
00023 ./lib/tk8.2/tkfbox.tcl
00052 ./lib/tcl8.2/encoding/big5.enc
00064 ./lib/tk8.2/text.tcl
00273 ./lib/tcl8.2/encoding/cp1253.enc
00281 ./lib/tcl8.2/encoding/gb12345.enc
00300 ./user/user_results_file_read.f
00328 ./demo/iso1.iso
00349 ./bin/fv_to_printer.sh.sample
00444 ./uil/mac_ppc/xform_controls.uid
00455 ./lib/tcl8.2/encoding/cp857.enc
00543 ./bin/mpeg2encode
00685 ./examples/rectangular_duct/rect_duct_044.uns
00715 ./examples/rectangular_duct/rect_duct_048.uns
00795 ./lib/tcl8.2/encoding/jis0212.enc
01283 ./examples/rectangular_duct/rect_duct_029.uns
01383 ./lib/tk8.2/button.tcl
01478 ./fvx_and_restarts/animate_streamlines.map
01677 ./demo/f18i9b_g_bin.fvbnd
01677 ./examples/f18/f18i9b_g_bin.fvbnd
02009 ./examples/rectangular_duct/rect_duct_024.uns
02213 ./examples/rectangular_duct/rect_duct_012.uns
02225 ./fvx_and_restarts/plot3d_eqns.frm
02235 ./lib/tk8.2/msgbox.tcl
02266 ./lib/tcl8.2/encoding/cp949.enc
02317 ./data/xdb_vars
02325 ./lib/blt2.4/dd_protocols/dd-number.tcl
02345 ./user/user_grid_file_read.f
02492 ./examples/rectangular_duct/rect_duct_062.uns
02682 ./doc/Readme_InstallNotes_12.pdf
02722 ./data/colormaps/c3.col
02780 ./uns/write_split_ascii_uns.f
02844 ./examples/rectangular_duct/rect_duct_067.uns
03030 ./fvx_and_restarts/bluntfinq.bin
03051 ./uil/mac_ppc/coordinate_surface.uid
03294 ./lib/blt2.4/bltDragdrop.tcl
03299 ./examples/rectangular_duct/rect_duct_030.uns
03319 ./uil/mac_ppc/point_probe.uid
03321 ./demo/coord3.crd
03329 ./fvx_and_restarts/set_view_TEST.fvx
03531 ./uil/mac_ppc/comp_surface.uid
03592 ./dore/data/il/def/parsekey.sym
03816 ./geometry/cyln48.atm
04150 ./fvx_and_restarts/seed_rectangle_proto.fvx
04277 ./uil/mac_ppc/misc_support.uid
04449 ./examples/rectangular_duct/rect_duct_039.uns
04517 ./uil/mac_ppc/boundary_conditions.uid
04525 ./fvx_and_restarts/seed_circ_GUI.fvx
04824 ./lib/tcl8.2/encoding/macThai.enc
04999 ./lib/tcl8.2/encoding/macIceland.enc
05048 ./examples/rectangular_duct/rect_duct_040.uns
05077 ./lib/tcl8.2/http2.1/pkgIndex.tcl
05136 ./lib/tcl8.2/tcltest1.0/pkgIndex.tcl
05176 ./lib/tcl8.2/encoding/koi8-r.enc
05986 ./demo/aerospace_LDM.bnd
06058 ./demo/f18i9b_q_bin
06058 ./examples/f18/f18i9b_q_bin
06106 ./lib/tcl8.2/encoding/cp737.enc
06139 ./dore/data/fonts/romant.df
06194 ./lib/tcl8.2/encoding/gb2312.enc
06517 ./bin/fvsrv
06966 ./lib/tcl8.2/encoding/iso8859-8.enc
07340 ./geometry/cyln12.atm
07365 ./examples/rectangular_duct/rect_duct_016.uns
07605 ./demo/stream4.stl
07614 ./demo/t3.ttl
07885 ./bin/fv.template
08136 ./lib/tcl8.2/encoding/euc-jp.enc
08160 ./lib/tcl8.2/encoding/cp1254.enc
08221 ./demo/t.ttl
08221 ./demo/t10.ttl
08320 ./examples/rectangular_duct/rect_duct_043.uns
08420 ./uil/mac_ppc/streamlines.uid
08917 ./user/user_defined_functions.f
09118 ./lib/tcl8.2/encoding/cp862.enc
09161 ./examples/rectangular_duct/rect_duct_018.uns
09510 ./examples/rectangular_duct/rect_duct_015.uns
09563 ./examples/rectangular_duct/rect_duct_068.uns
09664 ./bin/convert
10046 ./demo/lift.fvx
10063 ./bin/lib/ImageMagick-5.5.7/log.mgk
10297 ./uil/mac_ppc/flip.uid
10314 ./lib/tk8.2/obsolete.tcl
10394 ./examples/rectangular_duct/rect_duct_073.uns
10581 ./uil/mac_ppc/subset_params.uid
10827 ./lib/tcl8.2/http2.1/http.tcl
10855 ./lib/tk8.2/bgerror.tcl
10967 ./doc/FVReference_Manual.pdf
11013 ./examples/rectangular_duct/rect_duct_055.uns
11200 ./dore/data/fonts/romans.df
11349 ./uil/mac_ppc/dynamic_subset.uid
11396 ./examples/rectangular_duct/rect_duct_011.uns
11412 ./lib/tcl8.2/encoding/cp864.enc
11868 ./lib/tcl8.2/tclIndex
11903 ./lib/tcl8.2/encoding/euc-cn.enc
11963 ./dore/data/fonts/italicc.df
12263 ./dore/data/fonts/cyrillic.df
12397 ./demo/t11.ttl
12721 ./examples/rectangular_duct/rect_duct_028.uns
12833 ./lib/tcl8.2/tcltest1.0/tcltest.tcl
13082 ./fvx_and_restarts/set_view.fvx
13258 ./lib/tcl8.2/encoding/euc-kr.enc
13293 ./fvx_and_restarts/bio.frm
13398 ./demo/t2.ttl
13461 ./examples/rectangular_duct/rect_duct_009.uns
13652 ./examples/rectangular_duct/rect_duct_078.uns
13726 ./bin/lib/ImageMagick-5.5.7/delegates.mgk
14117 ./lib/blt2.4/NEWS
14544 ./dore/data/fonts/greekp.df
14611 ./geometry/allobs.ob
14614 ./uil/mac_ppc/keyframe.uid
14678 ./bin/lib/ImageMagick-5.5.7/modules.mgk
15065 ./examples/rectangular_duct/rect_duct_033.uns
15605 ./lib/tcl8.2/encoding/macCentEuro.enc
15825 ./lib/tcl8.2/encoding/macCroatian.enc
15887 ./lib/tcl8.2/encoding/macUkraine.enc
16097 ./lib/tcl8.2/encoding/iso8859-6.enc
16373 ./examples/rectangular_duct/rect_duct_008.uns
16561 ./lib/tk8.2/scrlbar.tcl
16641 ./uil/mac_ppc/dataset_controls.uid
16650 ./geometry/poly12.atm
16658 ./fvx_and_restarts/seed_circular_proto.fvx
16709 ./lib/tcl8.2/msgcat1.0/pkgIndex.tcl
16741 ./lib/tcl8.2/parray.tcl
16795 ./dore/data/fonts/gothgrt.df
16841 ./lib/tcl8.2/encoding/cp950.enc
16950 ./fvx_and_restarts/animate_streamlines.prd
17116 ./lib/tcl8.2/auto.tcl
17184 ./demo/start.plt
17317 ./demo/start.dat
17332 ./demo/bound1.bnd
17427 ./lib/tcl8.2/encoding/cp775.enc
17570 ./lib/tcl8.2/encoding/iso2022.enc
17635 ./examples/rectangular_duct/rect_duct_031.uns
17647 ./geometry/poly48.atm
17790 ./lib/tcl8.2/encoding/iso8859-7.enc
17844 ./lib/tcl8.2/encoding/cp932.enc
17920 ./lib/tcl8.2/msgcat1.0/msgcat.tcl
18023 ./examples/rectangular_duct/rect_duct_020.uns
18271 ./lib/tcl8.2/encoding/cp863.enc
18315 ./lib/tcl8.2/encoding/cp869.enc
18328 ./examples/rectangular_duct/rect_duct_019.uns
18422 ./bin/fv_user_print.sh.sample
18484 ./fvx_and_restarts/animate_streamlines.fvx
18565 ./bin/fvsrvbin
18619 ./lib/tcl8.2/encoding/macDingbats.enc
18713 ./dore/data/fonts/gothitt.df
19151 ./demo/aerospace_LDM.fvx
19329 ./lib/tcl8.2/encoding/macGreek.enc
19500 ./lib/tcl8.2/encoding/jis0208.enc
19545 ./lib/tcl8.2/encoding/cp861.enc
20094 ./examples/rectangular_duct/rect_duct_069.uns
20144 ./examples/rectangular_duct/rect_duct_056.uns
20418 ./examples/rectangular_duct/rect_duct_007.uns
20545 ./examples/rectangular_duct/rect_duct_060.uns
20873 ./dore/data/fonts/unused.df
21460 ./lib/tcl8.2/http1.0/http.tcl
21496 ./data/xdb_vars_FV11.0
21564 ./lib/tcl8.2/ldAout.tcl
21639 ./bin/mpeg2decode
21791 ./uil/mac_ppc/datasetBrowser.uid
21798 ./demo/t7.ttl
21894 ./lib/blt2.4/bltTabset.tcl
21960 ./dore/data/fonts/astrology.df
21969 ./examples/heat_exchanger/counter.uns
22040 ./uns/write_unformatted_uns.F
22438 ./lib/tcl8.2/encoding/cp1257.enc
22569 ./examples/rectangular_duct/rect_duct_035.uns
22765 ./data/license.dat
23070 ./uil/mac_ppc/color_mixer.uid
23111 ./uil/mac_ppc/region_control.uid
23217 ./demo/wingloadplots.fvx
23286 ./dore/data/fonts/italiccs.df
23323 ./demo/coord1.crd
23344 ./uns/write_ascii_uns.f
23526 ./lib/tk8.2/scale.tcl
23533 ./lib/tcl8.2/encoding/iso8859-5.enc
23541 ./examples/rectangular_duct/rect_duct_005.uns
23709 ./lib/tk8.2/images/logo64.gif
23752 ./lib/tk8.2/dialog.tcl
24020 ./uil/mac_ppc/title_panel.uid
24188 ./examples/rectangular_duct/rect_duct_001.uns
24661 ./examples/rectangular_duct/rect_duct_010.uns
24673 ./data/colormaps/c2.col
24852 ./demo/aerospace_LDM.dat
24881 ./lib/blt2.4/htresize.cur
25119 ./bin/display
25189 ./lib/tk8.2/tk.tcl
25502 ./examples/rectangular_duct/rect_duct_054.uns
25598 ./user/user_unstruct_combined.f
25605 ./lib/tcl8.2/encoding/macCyrillic.enc
25703 ./lib/tk8.2/tclIndex
26148 ./user/user_combined_file_read.f
26216 ./examples/rectangular_duct/rect_duct_057.uns
26356 ./lib/tk8.2/images/README
26380 ./lib/tk8.2/focus.tcl
26668 ./geometry/sph24.atm
27073 ./fvx_and_restarts/seed_rect_GUI.fvx
27081 ./lib/blt2.4/bltGraph.tcl
27179 ./bin/plugins/fluent_reader.so
27188 ./lib/tk8.2/images/pwrdLogo200.gif
27513 ./geometry/star.atm
27920 ./bin/lib/ImageMagick-5.5.7/type-ghostscript.mgk
27971 ./bin/lib/ImageMagick-5.5.7/magic.mgk
28042 ./demo/coord0.crd
28310 ./examples/rectangular_duct/rect_duct_003.uns
28324 ./uns/write_split_binary_uns.c
28358 ./lib/blt2.4/dd_protocols/dd-file.tcl
28436 ./user/ftn_fv_close.f
28520 ./uil/mac_ppc/presentation.uid
28601 ./demo/coord5.crd
28632 ./examples/rectangular_duct/rect_duct_034.uns
28690 ./uil/mac_ppc/misc_file_input.uid
28734 ./lib/blt2.4/dd_protocols/dd-color.tcl
28893 ./examples/rectangular_duct/rect_duct_053.uns
29100 ./lib/tk8.2/listbox.tcl
29192 ./bin/fv_platform
29194 ./lib/tk8.2/images/pwrdLogo75.gif
29195 ./examples/rectangular_duct/rect_duct_046.uns
29424 ./bin/lib/ImageMagick-5.5.7/colors.mgk
29488 ./examples/rectangular_duct/rect_duct_066.uns
29543 ./lib/tcl8.2/encoding/macRomania.enc
29643 ./fvx_and_restarts/animate_streamlines.vct
29657 ./demo/stream3.stl
29787 ./lib/tk8.2/optMenu.tcl
29826 ./lib/tcl8.2/init.tcl
29845 ./examples/rectangular_duct/rect_duct_023.uns
29936 ./examples/rectangular_duct/rect_duct_014.uns
29942 ./lib/tcl8.2/package.tcl
30113 ./lib/tk8.2/palette.tcl
30177 ./dore/data/fonts/meteorology.df
30254 ./lib/blt2.4/bltDnd.tcl
30349 ./lib/tk8.2/images/tai-ku.gif
30506 ./lib/tcl8.2/encoding/iso2022-kr.enc
30792 ./dore/data/fonts/romanc.df
31089 ./lib/tcl8.2/encoding/iso8859-1.enc
31321 ./bin/lmhostid
31428 ./bin/lib/ImageMagick-5.5.7/type-windows.mgk
31591 ./lib/tcl8.2/encoding/cp866.enc
31691 ./lib/tcl8.2/encoding/cp865.enc
32060 ./examples/rectangular_duct/rect_duct_025.uns
32217 ./examples/rectangular_duct/rect_duct_006.uns
32492 ./lib/tcl8.2/encoding/cp852.enc
32548 ./demo/aerospace_LDM.vct
32643 ./fvx_and_restarts/basic_volume_integral.fvx
32649 ./examples/rectangular_duct/rect_duct_051.uns
32725 ./examples/rectangular_duct/rect_duct_022.uns
32837 ./examples/rectangular_duct/rect_duct_050.uns
32927 ./lib/tk8.2/tkAppInit.c
33001 ./lib/tcl8.2/encoding/cp874.enc
33248 ./demo/moments.fvx
33321 ./lib/blt2.4/dd_protocols/dd-text.tcl
33396 ./demo/coord2.crd
33520 ./doc/Readme_Server_Toolkit.txt
33547 ./uns/write_binary_uns.c
34182 ./lib/blt2.4/dd_protocols/tclIndex
34228 ./bin/fvbin
34523 ./uil/mac_ppc/plot.uid
34626 ./geometry/sph12.atm
34669 ./lib/tcl8.2/encoding/gb1988.enc
34710 ./fvx_and_restarts/wind.frm
34959 ./dore/data/fonts/lcmath.df
35021 ./fvx_and_restarts/animate_streamlines.frm
35313 ./dore/data/fonts/romanp.df
35384 ./geometry/square.atm
35398 ./demo/t4.ttl
35629 ./bin/README_Copyrights.txt
35670 ./doc/FVUsers_Guide.pdf
35707 ./lib/tcl8.2/encoding/shiftjis.enc
35818 ./examples/rectangular_duct/rect_duct_063.uns
35841 ./bin/node_id
36318 ./examples/rectangular_duct/rect_duct_080.uns
36575 ./fvx_and_restarts/bluntfinx.bin.fvbnd
36625 ./uil/mac_ppc/particle_paths.uid
36641 ./lib/tk8.2/images/pwrdLogo150.gif
36865 ./uns/ftn_fv_reader_tags.h
36925 ./examples/rectangular_duct/rect_duct_004.uns
37055 ./lib/tcl8.2/tclAppInit.c
37272 ./lib/tk8.2/clrpick.tcl
37307 ./lib/tcl8.2/encoding/ksc5601.enc
37409 ./lib/tcl8.2/history.tcl
37439 ./lib/tk8.2/images/pwrdLogo.eps
37477 ./lib/tcl8.2/encoding/cp1258.enc
37544 ./doc/Whats_New_FV12.pdf
37688 ./user/ftn_register_functions.f
38287 ./fvx_and_restarts/README_PLOT3D_formula_restart
38461 ./demo/f18i9b_g_bin
38461 ./examples/f18/f18i9b_g_bin
38501 ./dore/data/fonts/italict.df
38531 ./examples/rectangular_duct/rect_duct_036.uns
38545 ./doc/Help_Intro.pdf
38574 ./lib/tk8.2/prolog.ps
38666 ./lib/tk8.2/menu.tcl
38861 ./lib/tk8.2/images/pwrdLogo100.gif
38908 ./bin/animate
38994 ./lib/tcl8.2/encoding/macTurkish.enc
39109 ./lib/tk8.2/images/pwrdLogo175.gif
39676 ./lib/tcl8.2/encoding/iso2022-jp.enc
39874 ./demo/f18show.scr
40066 ./user/ftn_register_data_readers.f
40442 ./demo/col1.map
40541 ./user/register_data_readers.c
40552 ./lib/tcl8.2/encoding/symbol.enc
41107 ./lib/tcl8.2/encoding/cp1256.enc
41169 ./geometry/geometry_formats.ps
41358 ./demo/stream2.stl
41364 ./lib/tk8.2/images/logo100.gif
41566 ./lib/blt2.4/pkgIndex.tcl
41734 ./demo/iso2.iso
41850 ./bin/contest
42046 ./lib/tcl8.2/encoding/iso8859-9.enc
42054 ./demo/t9.ttl
42217 ./lib/tcl8.2/encoding/macJapan.enc
42218 ./lib/tcl8.2/encoding/cp437.enc
42270 ./examples/rectangular_duct/rect_duct_026.uns
42627 ./dore/data/fonts/ucmath.df
42702 ./uil/mac_ppc/integration.uid
42904 ./uil/mac_ppc/xdb_info.uid
43278 ./dore/data/fonts/symbols.df
43508 ./lib/tk8.2/images/logo.eps
43701 ./examples/rectangular_duct/rect_duct_041.uns
43751 ./examples/heat_exchanger/cocurrent.uns.fvreg
43751 ./examples/heat_exchanger/counter.uns.fvreg
43946 ./fvx_and_restarts/animate_streamlines.dat
43960 ./user/user_unstruct_grid.f
44519 ./lib/tcl8.2/ldAix
44715 ./uil/mac_ppc/vr.uid
44814 ./examples/rectangular_duct/rect_duct_042.uns
44867 ./demo/start.vct
44875 ./user/user_unstruct_results.f
44878 ./lib/tcl8.2/encoding/jis0201.enc
44926 ./examples/rectangular_duct/rect_duct_037.uns
45085 ./sconfig/srv.template
45118 ./uil/mac_ppc/func.uid
45438 ./examples/rectangular_duct/rect_duct_032.uns
45642 ./dore/data/fonts/gothgbt.df
45830 ./demo/iso3.iso
45880 ./lib/tcl8.2/encoding/cp1250.enc
46000 ./examples/rectangular_duct/rect_duct_013.uns
46023 ./lib/tk8.2/images/logoMed.gif
46093 ./demo/view2.vct
46103 ./examples/rectangular_duct/rect_duct_047.uns
46125 ./uil/mac_ppc/data_input.uid
46238 ./demo/start.map
46455 ./lib/tcl8.2/encoding/cp1255.enc
46496 ./lib/blt2.4/bltHierbox.tcl
46844 ./geometry/cone12.atm
47054 ./fvx_and_restarts/seed_sphere_proto.fvx
47357 ./uns/write_split_unformatted_uns.F
47891 ./demo/view6.vct
47930 ./demo/drag.fvx
47970 ./lib/tcl8.2/safe.tcl
48023 ./lib/tcl8.2/encoding/cp936.enc
48142 ./lib/blt2.4/bltHiertable.tcl
48291 ./data/colormaps/c7.col
48295 ./fvx_and_restarts/xcolor_defines.fvx
48438 ./examples/rectangular_duct/rect_duct_059.uns
48906 ./dore/data/fonts/greekcs.df
49045 ./fvx_and_restarts/seed_sphere_GUI.fvx
49051 ./demo/view4.vct
49338 ./demo/view3.vct
49417 ./lib/tcl8.2/encoding/ascii.enc
49617 ./demo/start.bnd
49824 ./lib/tcl8.2/encoding/iso8859-3.enc
49896 ./geometry/cube.atm
49920 ./fvx_and_restarts/lsdyna_formulas.frm
49926 ./data/colormaps/c4.col
49932 ./demo/t8.ttl
50012 ./uil/mac_ppc/vortex_core.uid
50329 ./demo/view7.vct
50348 ./uil/mac_ppc/legend_control.uid
50415 ./examples/rectangular_duct/rect_duct_061.uns
50427 ./examples/rectangular_duct/rect_duct_077.uns
50441 ./bin/prolog.ps
50605 ./examples/rectangular_duct/rect_duct_038.uns
50726 ./uil/mac_ppc/formula.uid
51123 ./examples/rectangular_duct/rect_duct_021.uns
51369 ./uil/mac_ppc/vector_params.uid
51494 ./bin/fv_convert.template
51968 ./examples/rectangular_duct/rect_duct_065.uns
52013 ./lib/tcl8.2/encoding/cp855.enc
52057 ./examples/rectangular_duct/rect_duct_076.uns
52309 ./lib/tcl8.2/opt0.4/pkgIndex.tcl
52340 ./examples/rectangular_duct/rect_duct_049.uns
52415 ./demo/start.ttl
52745 ./dore/data/fonts/romand.df
52833 ./lib/tcl8.2/encoding/cp850.enc
53221 ./lib/blt2.4/tclIndex
53293 ./examples/rectangular_duct/rect_duct_052.uns
53473 ./lib/tk8.2/comdlg.tcl
53516 ./examples/rectangular_duct/rect_duct_058.uns
53968 ./user/toolkit.c
54002 ./lib/tcl8.2/encoding/cp860.enc
54080 ./bin/plugins/lsdyna_reader.so
54102 ./fvx_and_restarts/calc_volume_integral.fvx
54228 ./sconfig/manual.srv
54250 ./demo/plot1.lpt
54260 ./demo/plot2.lpt
54331 ./bin/make_fv
54533 ./dore/data/fonts/romancs.df
54725 ./uil/mac_ppc/iso_surface.uid
54905 ./dore/data/fonts/helvetica.df
55047 ./examples/rectangular_duct/rect_duct.trk
55208 ./fvx_and_restarts/Q-criterion.frm
55231 ./lib/tcl8.2/http1.0/pkgIndex.tcl
55361 ./lib/tk8.2/xmfbox.tcl
55552 ./examples/rectangular_duct/rect_duct_074.uns
55614 ./examples/rectangular_duct/rect_duct_045.uns
55673 ./lib/tcl8.2/encoding/cp1251.enc
55709 ./examples/rectangular_duct/rect_duct_075.uns
55734 ./dore/data/fonts/scriptc.df
55885 ./lib/tk8.2/images/logoLarge.gif
56003 ./data/colormaps/c5.col
56186 ./dore/data/fonts/greeks.df
56255 ./examples/rectangular_duct/rect_duct_002.uns
56320 ./demo/t6.ttl
56378 ./examples/rectangular_duct/rect_duct_070.uns
56695 ./lib/tcl8.2/encoding/iso8859-2.enc
56916 ./lib/blt2.4/bltTabnotebook.tcl
56960 ./dore/data/fonts/greekc.df
57234 ./user/user_unstruct_functions.f
57241 ./uil/mac_ppc/transient_datav2.uid
57480 ./demo/aerospace_LDM.frm
57686 ./lib/blt2.4/Problems
57698 ./fvx_and_restarts/FVX_Templates.pdf
57711 ./lib/tk8.2/entry.tcl
57757 ./examples/rectangular_duct/rect_duct_064.uns
57941 ./lib/tcl8.2/encoding/dingbats.enc
58071 ./lib/tcl8.2/opt0.4/optparse.tcl
58217 ./bin/lib/ImageMagick-5.5.7/type.mgk
58430 ./fvx_and_restarts/animate_streamlines.bnd
58616 ./fvx_and_restarts/plot_along_streamline.fvx
58635 ./demo/start.frm
58718 ./uil/mac_ppc/colormap_specification.uid
58764 ./lib/tk8.2/tearoff.tcl
59310 ./demo/t0.ttl
59391 ./lib/tcl8.2/encoding/iso8859-4.enc
59497 ./doc/How_to.pdf
59695 ./examples/rectangular_duct/rect_duct_071.uns
60258 ./dore/data/errmsg/errmsgfile
60621 ./data/colormaps/c6.col
60693 ./examples/rectangular_duct/rect_duct_079.uns
60841 ./bin/lib/ImageMagick-5.5.7/type-solaris.mgk
61035 ./lib/blt2.4/bltGraph.pro
61117 ./lib/tcl8.2/word.tcl
61247 ./demo/blank.ttl
61446 ./uns/fv_reader_tags.h
61497 ./dore/data/fonts/music.df
61541 ./lib/tk8.2/console.tcl
61734 ./demo/coord4.crd
61852 ./demo/view5.vct
62104 ./bin/server_name
62390 ./user/register_functions.c
62413 ./demo/t5.ttl
62662 ./demo/t1.ttl
62907 ./demo/stream1.stl
62918 ./fvx_and_restarts/bluntfinx.bin
63021 ./examples/rectangular_duct/rect_duct_017.uns
63102 ./lib/tcl8.2/encoding/macRoman.enc
63106 ./examples/heat_exchanger/cocurrent.uns
63221 ./lib/tk8.2/safetk.tcl
63281 ./uil/mac_ppc/surface_plot.uid
63321 ./bin/ld_fv
63340 ./demo/iso4.iso
63384 ./examples/rectangular_duct/rect_duct_027.uns
63671 ./geometry/cone48.atm
63850 ./lib/tcl8.2/encoding/cp1252.enc
64512 ./examples/rectangular_duct/rect_duct_072.uns
64515 ./data/colormaps/c8.col
64986 ./lib/blt2.4/Readme
65154 ./lib/blt2.4/bltCanvEps.pro
65304 ./data/colormaps/c1.col
65467 ./dore/data/fonts/scripts.df
EOF
;;
*)
echo "Internal error in install: unknown version number $version" 1>&2
exit 7
;;
esac

# Use default C locale for sorting.
unset LC_COLLATE
unset LC_CTYPE
LANG=C
export LANG

if test $need_name -eq 1
then
    # sum command does not print single filename - trick it
    find . -type f -exec $sum_cmd {} _shesnotthere_ \; 2>/dev/null |
	awk '{printf "%05d %s\n", $1, $3}' | sort > $checknew
else
    # sum command does print single filename
    find . -type f -exec $sum_cmd {} \; |
	awk '{printf "%05d %s\n", $1, $3}' | sort > $checknew
fi
badsums=`comm -23 $checkold $checknew | wc -l`
if test $badsums -ne 0
then
    echo
    echo "Verify failed for directory $targetdir/fv"
    echo "The following expected file checksums were not found:"
    comm -23 $checkold $checknew
    echo
    exit 8
else
    echo "Verified OK."
fi

/bin/rm -f $checknew $checkold

echo
echo 'The FIELDVIEW startup script contains an environment variable'
echo 'called FV_HOME that points to the subdirectory fv in the'
echo 'directory where FIELDVIEW is installed.'
echo 'This directory is apparently:'
echo "   $targetdir"
echo
echo 'The default value for the FV_HOME environment variable in the'
echo 'FIELDVIEW startup script will be set to this subdirectory.'
echo
echo 'If you move the installation, or if you simply wish to use'
echo 'a different name for this directory, you must edit the file'
echo "   $targetdir/fv/bin/fv"
echo 'and change the values of the variable FV_HOME.'
echo

ftmp=$targetdir/fv/bin/fv.template
sed -e "s;FV_HOME_VALUE;$targetdir;" $ftmp >$targetdir/fv/bin/fv
chmod +x $targetdir/fv/bin/fv
rm $ftmp

ftmp=$targetdir/fv/bin/fv_convert.template
sed -e "s;FV_HOME_VALUE;$targetdir;" $ftmp >$targetdir/fv/bin/fv_convert
chmod +x $targetdir/fv/bin/fv_convert
rm $ftmp

echo "! This server config file uses the local server on the client system." \
    >$targetdir/fv/sconfig/local.srv

srv_name=""
case $version in
2|4)
    # solaris or hp_700
    echo
    echo "Choose a default server for this platform:"
    echo
    echo "    1. 32-bit server"
    echo "    2. 64-bit server"
    echo
    echo "Please enter a number from 1 to 2:"

    while :
    do
	read server_version
	case $server_version in
	1) break;;
	2) srv_name="$targetdir/fv/bin/fvsrv64/fvsrv64"
	   break;;
	*) echo "Please enter a number from 1 to 2:";;
	esac
    done
    ;;
esac

if test -n "$srv_name"
then
    echo "ServerProgram: $srv_name" >> $targetdir/fv/sconfig/local.srv
fi

echo
echo "This installation already includes a FIELDVIEW server for this"
echo "platform.  Do you also want to install a directory containing"
echo "compressed servers for all platforms (answer y for yes) ?"

read response
case $response in
[yY]*)
    if test ! -d $targetdir/fv/servers
    then
	mkdir $targetdir/fv/servers
    fi
    cd $curd
    echo "Copying compressed servers into $targetdir/fv/servers/"
    cp servers/* $targetdir/fv/servers/
    echo
    ;;
esac

echo "To run FIELDVIEW with a floating license, you must install the"
echo "floating license manager (FLEXlm), and receive passwords for"
echo "this option from Intelligent Light."
echo
echo "Do you want to install the floating license manager (answer y for yes) ?"

read response
case $response in
[yY]*)
    cd $curd
    ./installcd_flexlm.sh $targetdir
    ;;
*)
    echo
    echo "You may proceed with the next step in the installation instructions."
    ;;
esac
