(* :Title: InstallNET.m *)

(* :Author:
        Todd Gayley
        tgayley@wolfram.com
*)

(* :Package Version: 1.3 *)

(* :Mathematica Version: 5.0 *)
             
(* :Copyright: .NET/Link source code (c) 2003-2007, Wolfram Research, Inc. All rights reserved.

   Use is governed by the terms of the .NET/Link license agreement, which can be found at
   www.wolfram.com/solutions/mathlink/netlink.
*)

(* :Discussion:
   InstallNET, UninstallNET and related.
    
   This file is a component of the .NET/Link Mathematica source code.
   It is not a public API, and should never be loaded directly by users or programmers.

   .NET/Link uses a special system wherein one package context (NETLink`) has its implementation
   split among a number of .m files. Each component file has its own private context, and also
   potentially introduces public symbols (in the NETLink` context) and so-called "package" symbols,
   where the term "package" comes from Java terminology, referring to symbols that are visible
   everywhere within the implementation of .NET/Link, but not to clients. The NETLink.m file itself
   is produced by an automated tool from the component files and contains only declarations.
   
   Do not modify the special comment markers that delimit Public- and Package-level exports.
*)


(*<!--Public From InstallNET.m

InstallNET::usage =
"InstallNET[] launches the .NET runtime and prepares it to be used from Mathematica. Only one .NET runtime is ever launched; \
subsequent calls to InstallNET after the first have no effect."

UninstallNET::usage =
"UninstallNET[] shuts down the .NET runtime that was started by InstallNET. It is provided mainly for developers who are \
actively recompiling .NET types for use in Mathematica and therefore need to shut down and restart the .NET runtime to reload \
the modified types. Users generally have no reason to call UninstallNET. The .NET runtime is a shared resource used by \
potentially many Mathematica programs. You should leave it running unless you are absolutely sure you need to shut it down."

ReinstallNET::usage =
"ReinstallNET[] is a convenience function that calls UninstallNET[] followed by InstallNET[]. See the usage messages for \
InstallNET and UninstallNET for more information."

NETLink::usage =
"NETLink[] returns the MathLink LinkObject that is used to communicate with the .NET/Link .NET runtime. It will return \
Null if .NET is not running."

NETUILink::usage =
"NETUILink[] returns the MathLink LinkObject used by calls to Mathematica that originate from .NET user-interface actions, or Null if no such link is present."

-->*)


(*<!--Package From InstallNET.m

$inPreemptiveCallFromNET

-->*)


(* Current context will be NETLink`. *)

Begin["`InstallNET`Private`"]


InstallNET::fail = "A link to the .NET runtime could not be established."

InstallNET::uifail = "The separate .NET user-interface link could not be established."

NET::init = "The .NET runtime is not running. You must call InstallNET[] to start the .NET runtime."

General::netlink = "`1` cannot operate because InstallNET[] failed to launch the .NET runtime."

NETLink::net =
"The .NET Framework is not installed on this machine. It must be installed before .NET/Link can be used. \
You can install the .NET Framework at Microsoft's Windows Update web site: http://windowsupdate.microsoft.com. \
If you do not see the .NET Framework listed among the available updates you probably need to update your version \
of Internet Explorer first."

NETLink::netvers =
".NET/Link requires a more recent version of the .NET Framework than is currently installed on this machine. \
You can install the latest .NET Framework at Microsoft's Windows Update web site: http://windowsupdate.microsoft.com."



If[!ValueQ[$nlink], $nlink = Null]
If[!ValueQ[$uilink], $uilink = Null]

Internal`SetValueNoTrack[$nlink, True]
Internal`SetValueNoTrack[$uilink, True]


NETLink[] := $nlink

NETUILink[] := $uilink


(* InstallNET currently has no options but we leave in possible opts___ for legacy. *)

InstallNET[opts___?OptionQ] := InstallNET[Null, opts]

InstallNET[link:(_LinkObject | Null), opts___?OptionQ] :=
	preemptProtect[
		Module[{netlinkPath, gacSystemDir, gac2SystemDir, verStrings, verNumbers, useUILink, protocol},
			(* Bail out right away if link is already open and OK. *)
			If[MemberQ[Links[], $nlink],
				(* Only do the check on the health of the link if not already in a preemptive
				   transaction with .NET. It's obviously not necessary in such cases, but it also
				   can cause the link to die if we call LinkReadyQ on a link when we happen
				   to already be blocking in its yield function in the main computation.
				*)
				If[$inExternalCall,
					Return[$nlink]
				];
				LinkReadyQ[$nlink]; (* Hit link to force LinkError to give current value. *)
				If[First[LinkError[$nlink]] === 0,
					Return[$nlink],
				(* else *)
					UninstallNET[]
				],
			(* else *)
				(* This extra test (nlink has a value, but it is not in Links[]) is to catch cases where
				   user has improperly shut down .NET (e.g., by calling LinkClose).
				*)
				If[Head[$nlink] === LinkObject,
					resetMathematica[];
					$nlink = Null
				]
			];
			(* Check for the existence of the correct .NET Framework on this machine (>= 1.1). *)
			If[StringQ[Environment["WINDIR"]],
				gacSystemDir = ToFileName[{Environment["WINDIR"], "assembly", "gac", "system"}];
				gac2SystemDir = ToFileName[{Environment["WINDIR"], "assembly", "gac_msil", "system"}];
				verStrings = Join[
								StringTake[#, {StringLength[gacSystemDir] + 1, StringLength[gacSystemDir] + 5}]& /@
									Select[FileNames["*", gacSystemDir], StringMatchQ[#, "*.*.*"]&],
								 StringTake[#, {StringLength[gac2SystemDir] + 1, StringLength[gac2SystemDir] + 5}]& /@
									Select[FileNames["*", gac2SystemDir], StringMatchQ[#, "*.*.*"]&]
							 ];
				(* verStrings looks like {"1.0.3", "1.0.5", "2.0.0"} *)
				If[Length[verStrings] == 0,
					Message[NETLink::net];
					Return[$Failed]
				];
				verNumbers = ToExpression /@ Select[StringReplace[#, "."->""]& /@ verStrings, DigitQ];
				If[Max[verNumbers] < 105,
					Message[NETLink::netvers];
					Return[$Failed]

				]
			];
			If[Head[link] === LinkObject,
				$nlink = Install[link],
			(* else *)
				netlinkPath = 
					Scan[If[FileType[#] === File, Return[#]]&,
						{ToFileName[$netlinkDir, "InstallableNET.exe"],  (* First try to find InstallableNET.exe next to NETLink.m *)
						 ToFileName[{$TopDirectory, "AddOns", "NETLink"}, "InstallableNET.exe"],  (* For >= 5.0 *)
						 ToFileName[{$TopDirectory, "AddOns", "Applications", "NETLink"}, "InstallableNET.exe"],
						 ToFileName[{$PreferencesDirectory, "Applications", "NETLink"}, "InstallableNET.exe"],  (* For >= 5.0 *)
						 ToFileName[{$PreferencesDirectory, "AddOns", "Applications", "NETLink"}, "InstallableNET.exe"]}
					];
				If[netlinkPath === Null,
					(* If it wasn't found in any of the standard places, just launch it by name and see if it is on PATH. *)
					netlinkPath = "InstallableNET.exe"
				];
				If[!osIsWindows[],
					netlinkPath = "mono " <> netlinkPath
				];
				protocol = If[osIsWindows[] && $VersionNumber < 5.1, "filemap", Automatic];
				(* On Windows and Unix, LinkQuote is not needed, and can interfere
				   with finding the program to launch (e.g., if CommandLine option has a full pathname with backslashes).
				   Because LinkQuote is wrapped around another function we don't want called (FindFile), we replace it
				   with Hold. 
				*)
				Block[{System`Dump`LinkQuote = Hold},
					$nlink = Install[netlinkPath, LinkProtocol->protocol]
				]
			];
	        
			(* Don't want to set up uiLink if we are calling InstallNET[$ParentLink] from NET (this happens
			   during IKernelLink.EnableObjectReferences()). In that case there won't be a Reader thread, so
			   we don't want the uiLink.
			*)
			useUILink = useUILinkQ[] && link =!= $ParentLink;
			
			If[Head[$nlink] === LinkObject,
				MathLink`LinkAddInterruptMessageHandler[$nlink];
				$uilink = initNET[useUILink]
			];
			If[Head[$nlink] === LinkObject && (Head[$uilink] === LinkObject || !useUILink),
				$nlink,
			(* else *)
				Message[InstallNET::fail];
				If[Head[$nlink] === LinkObject, LinkClose[$nlink]];
				$nlink = Null;
				If[Head[$uilink] === LinkObject, LinkClose[$uilink]];
				$uilink = Null;
				$Failed
			]
		]
	]


UninstallNET[] :=
	preemptProtect[
		Module[{res = Null, wasOn},
			If[Head[$nlink] === LinkObject,
				(* To avoid potentially many errors, only call onUnloadType methods if nlink is alive and well. *)
				If[MemberQ[Links[], $nlink],
					LinkReadyQ[$nlink]; (* Hit link to force LinkError to give current value. *)
					If[First[LinkError[$nlink]] === 0,
						(* TODO: Not wired up yet. *)
						callAllUnloadTypeMethods[]
					]
				];
				UnshareFrontEnd[$nlink];
				If[MemberQ[SharingLinks[], $nlink],
					UnshareKernel[$nlink]
				];
				resetMathematica[];
				(* Before calling Uninstall, turn off the linkn message. This prevents seeing messages warning that
				   the link is dead while we are closing it. They would otherwise show up when re-installing after a
				   crash or forced quit of .NET.
				*)
				wasOn = Head[LinkObject::linkn] =!= $Off;
				Off[LinkObject::linkn];
				res = Uninstall[$nlink];
				If[Head[$uilink] === LinkObject,
					MathLink`RemoveSharingLink[$uilink];
					LinkClose[$uilink]
				];
				If[wasOn, On[LinkObject::linkn]];
				$nlink = Null;
				$uilink = Null
			];
			res
		]
	]


Options[ReinstallNET] = Options[InstallNET]

ReinstallNET[args___] := preemptProtect[UninstallNET[]; InstallNET[args]]


initNET[setupUILink:(True | False)] :=
    Module[{link, prot},
        (* Load all assemblies in the special WRI SystemFiles/assembly dir. *)
        LoadNETAssembly[FileNames[{"*.dll", "*.exe"}, ToFileName[{$TopDirectory, "SystemFiles", "assembly"}]]];
        If[isPreemptiveKernel[] && setupUILink,
	        (* Set up the UI link. *)
	        prot = If[osIsWindows[] (* || $VersionNumber >= 6.0 *), "SharedMemory", "TCPIP"];
	        link = LinkCreate[LinkProtocol->prot];
	        If[TrueQ[nUILink[First[link], prot]],
	            LinkConnect[link];
				MathLink`AddSharingLink[link,
						MathLink`LinkSwitchPre -> linkSwitchPreFunc,
						MathLink`LinkSwitchPost -> linkSwitchPostFunc,
						MathLink`AllowPreemptive -> True,
						MathLink`ImmediateStart -> True
				];
				MathLink`LinkAddInterruptMessageHandler[link];
				(* UI link is a daemon link, meaning that kernel should not stay alive just for it. *)
				MathLink`SetDaemon[link, True];
	            link,
	        (* else *)
	            Message[InstallNET::uifail];
	            LinkClose[link];
	            Null
	        ],
	    (* else *)
	    	(* Version 5.0 or earlier--no UI Link *)
	    	Null
	    ]
	]


If[!ValueQ[$inPreemptiveCallFromNET], $inPreemptiveCallFromNET = False]


linkSwitchPreFunc[___] :=
	Block[{uiLink, oldFrontEnd},
	    uiLink = NETUILink[];
		If[MathLink`IsPreemptive[], $inPreemptiveCallFromNET = True];
		oldFrontEnd = 
			If[isServiceFrontEnd[],
				MathLink`SetServiceFrontEnd[],
			(* else *)
				If[FrontEndSharedQ[NETLink[]],
					(* Note that for legacy reasons, users call ShareFrontEnd[NETLink[]], but really
					   it is the NETUILink[] ($activeNETLink at the moment) that the FE-specific traffic
					   goes out on.
					*)
					MathLink`SetFrontEnd[uiLink],
				(* else *)
					MathLink`SetFrontEnd[Null];
					MathLink`SetMessageLink[uiLink]
				]
			];
		If[First[oldFrontEnd] === False,
			(* There was no ServiceLink, and the MessageLink was set to Null.
			   FE services won't work, but at least we can set the MessageLink
			   to the active link (uiLink), so that side-effect output will come to
			   .NET and not get completely lost.
			*)
			MathLink`SetMessageLink[uiLink]
		];
		oldFrontEnd
	]
	
linkSwitchPostFunc[oldFrontEnd_] :=
	(
		MathLink`RestoreFrontEnd[oldFrontEnd];
		$inPreemptiveCallFromNET = False;
	)
	


(* The Mathematica-side things that must be done when starting a fresh .NET session. After this func is run, it should
    be true that the kernel is in the same state it was in before any .NET sessions were started. The possible exception
    to this is that certain contexts may exist that weren't present before, but their contents should be empty. This function
    is only used during InstallNET; it is not user-visible, and no attempt is made to provide users with a way to
    "reset" their Mathematica-.NET session.
*)
resetMathematica[] :=
    AbortProtect[
        clearNETDefs[];
        resetNETBlock[];
        clearComplexClass[];
    ]


(* Tells whether to set up the separate UI link. Don't want to setup uiLink if not a preemptive kernel, or a standalone kernel. *)
useUILinkQ[] :=
	isPreemptiveKernel[] && ValueQ[$ParentLink] && $ParentLink =!= Null;


End[]
