//////////////////////////////////////////////////////////////////////////////////////
//
//   .NET/Link source code (c) 2003, Wolfram Research, Inc. All rights reserved.
//
//   Use is governed by the terms of the .NET/Link license agreement, which can be found at
//   www.wolfram.com/solutions/mathlink/netlink.
//
//   Author: Todd Gayley
//
//////////////////////////////////////////////////////////////////////////////////////

using System;
using System.Collections;
using System.Threading;
using Wolfram.NETLink.Internal;


namespace Wolfram.NETLink {

/// <summary>
/// StdLink is a container for some methods and state related to the link back to the kernel.
/// </summary>
/// <remarks>
/// You never create or use an instance of StdLink. It contains static members only.
/// <para>
/// The name is inspired by the 'stdlink' global variable that holds the link in C-language "installable"
/// MathLink programs generated by the mprep tool.
/// </para>
/// </remarks>
/// 
public class StdLink {
    
    // mainLink is the link that the Reader thread was started with. All UI-triggered computations will
    // be sent to this link.
    private static IKernelLink mainLink;
    private static bool mainLinkHasReader = false;

    // In 5.1 and later, there are two links to M. This one is for computations initiated by
    // events in .NET, such as mouse clicks, menus, etc.
    private static IKernelLink uiLink;

    private static Hashtable stdLinkHash = new Hashtable(4, 0.75F);
    private static object stdLinkLock = new object();
    private static object uiLock = new object();


    // Give this a private ctor just so that a default public one doesn't show up in the NDoc-generated docs.
    private StdLink() {}


    // A Stack is used to handle recursive calls to handleCallPacket. Each thread is associated
    // with a stack of KernelLinks representing each time the handleCallPacket function is entered
    // (setup() is called at this time). When we leave handleCallPacket we call remove(),
    // which pops from this stack, and then removes a thread record from the
    // hashtable only when all recursive calls have unwound. Of course, all this is somewhat
    // unlikely. Recursive calls aren't all that unlikely, but those could be handled by associating
    // a simple count with a thread-link pair. We need a stack of potentially different links on each
    // thread only to handle the exceedingly rare cases when a call into .NET would trigger a call
    // to another kernel, which would then in turn call back into .NET.
    // This scheme is absurdly complicated for virtually all conceivable uses. But it works,
    // and it's safe. The vast majority of the time, though, 'return mainLink;'
    // would be a sufficient implementation of getLink(), and setup() and remove() could go away completely.


    /// <summary>
    /// Gets or sets the link that points back to the <i>Mathematica</i> kernel.
    /// </summary>
    /// <remarks>
    /// When called during a session when .NET is "installed" into <i>Mathematica</i> (i.e., InstallNET[] has been called),
    /// this property gives the <see cref="IKernelLink"/> that points back to <i>Mathematica</i>. It returns null if .NET is
    /// not being used from <i>Mathematica</i>. .NET methods need to obtain the link back to <i>Mathematica</i> in a number
    /// of circumstances, such as if they want to return a result to <i>Mathematica</i> manually, or trigger computations
    /// in <i>Mathematica</i> before they return, or if they are called from the user interface in response to a
    /// user action like clicking a button.
    /// <example>
    /// <para>
    /// Here is an example of how it might be used in a method that wants to return a result to <i>Mathematica</i> manually,
    /// instead of having its normal return value sent back. 
    /// <code>
    /// IKernelLink ml = StdLink.Link;
    /// if (ml != null) {
    ///     ml.BeginManual();
    ///     // Here we put a result to Mathematica manually:
    ///     ml.Put(42);
    /// }
    /// </code>
    /// </para>
    /// </example>
    /// Do not forget to call <see cref="RequestTransaction"/> on the link if you are calling back to <i>Mathematica</i> in
    /// response to some user action initiated in .NET (like clicking a button).
    /// <para>
    /// You might want to <b>set</b> this property yourself in a .NET program that launches and controls the kernel
    /// (that is, <i>not</i> a program that is itself called from <i>Mathematica</i>, as then the property will be set automatically).
    /// One reason to set Link yourself is if you have classes that call <i>Mathematica</i> that you want to use in both a
    /// standalone program and scripted from <i>Mathematica</i>. For scripted uses, they need to use the Link property to get the
    /// link back to <i>Mathematica</i>. If you set this property to be the link you have created in your standalone program, then
    /// your classes will automatically acquire the correct link in both cases.
    /// In other words, .NET/Link lets you assign the "StdLink" yourself, so that you don't have to modify all the code
    /// that calls StdLink.Link just to use that code in a program that is not being called from <i>Mathematica</i>.
    /// </para>
    /// </remarks>
    /// <seealso cref="RequestTransaction"/>
    ///
    public static IKernelLink Link {
        get {    
            lock (stdLinkLock) {
                object key = Thread.CurrentThread;
                Stack s = (Stack) stdLinkHash[key];
                if (s != null && s.Count > 0) {
                    // This call is from code that is itself being called from Mathematica (handleCallPacket is on the stack).
                    return (IKernelLink) s.Peek();
                } else if (uiLink == null) {
                    // Mathematica 5.0 or earlier has no uiLink.
                    return mainLink;
                } else if (Reader.isInModalState) {
                    // The UI link is active but DoNETModal is running.
                    return mainLink;
                } else {
                    // This call is not being made as part of a chain that originated in Mathematica (most likely
                    // it is being made as a result of a user action in .NET).
                    return uiLink;
                }
            }
        }
        set {
            mainLink = value;
        }
    }
 

    /// <summary>
    /// Must always be called before calling into <i>Mathematica</i> from a .NET user-interface action. 
    /// </summary>
    /// <remarks>
    /// You should always call RequestTransaction whenever you are preparing to call into <i>Mathematica</i>
    /// from a .NET user-interface action, like clicking a button. This only applies for code executing in
    /// a .NET runtime launched from <i>Mathematica</i> by InstallNET[], not if you are launching and controlling
    /// the kernel from a standalone .NET program.
    /// <para>
    /// The only time you do <b>not</b> need to call RequestTransaction before calling into <i>Mathematica</i>
    /// is when the call is being made during the handling of a call from <i>Mathematica</i> into .NET. In other words,
    /// when <i>Mathematica</i> is currently in the middle of a call into .NET, it is always safe for calls
    /// to be made from .NET back into <i>Mathematica</i> (and RequestTransaction does nothing in this circumstance).
    /// You need RequestTransaction to guard calls that <i>originate</i> in .NET. User interface actions are typical
    /// examples of such calls, as <i>Mathematica</i> might not currently be ready to accept incoming requests from .NET.
    /// </para>
    /// It is safe to call RequestTransaction <i>whenever</i> you call back into <i>Mathematica</i>--if it is not necessary
    /// then RequestTransaction will simply return immediately. In other words, if you do not understand the exact circumstances
    /// in which it is necessary, you can err on the side of caution and call it more often than needed.
    /// <para>
    /// What RequestTransaction does is check whether it is safe to call into <i>Mathematica</i>. If it is safe, it returns
    /// immediately. If it is not safe, a <see cref="MathematicaNotReadyException"/> is thrown. You can typically ignore this
    /// exception and let it propagate up into the .NET event-dispatching mechanism, where it will be caught
    /// by the internals of .NET/Link and cause a MessageBox to be displayed.
    /// </para>
    /// It is safe to call into <i>Mathematica</i> in several circumstances. The first is if the <i>Mathematica</i> functions
    /// DoNETModal or DoNETModeless are currently running. This is the normal situation when user interface actions that
    /// call <i>Mathematica</i> are being executed. It is also safe to call into <i>Mathematica</i> when <i>Mathematica</i>
    /// is in the middle of calling .NET. Finally, it is safe to call into <i>Mathematica</i> if you have launched the
    /// kernel from your own program instead of running in the .NET runtime launched from <i>Mathematica</i> by InstallNET[].
    /// <para>
    /// Many programmers will never use this method, because they will use the <i>Mathematica</i> function AddEventHandler
    /// to establish callbacks from .NET into <i>Mathematica</i>. Event handlers created by AddEventHandler handle all details
    /// of the communication with <i>Mathematica</i> for you (and of course they call RequestTransaction internally). 
    /// </para>
    /// <example>
    /// Here is an example of .NET code that is intended to be executed as a result of a user interface action.
    /// <code>
    /// IKernelLink ml = StdLink.Link;
    /// if (ml != null) {
    ///     StdLink.RequestTransaction();
    ///     // Always lock the link before using it.
    ///     lock (ml) {
    ///         ml.Print("You clicked a button");
    ///     }    
    /// }
    /// </code>
    /// </example>
    /// <para>
    /// Note again that this is not the "normal" way you would wire up a Print action to a button click. Instead,
    /// you would just write something like this in <i>Mathematica</i>:
    /// <code>
    /// (* Mathematica code *)
    /// AddEventHandler[myButton@Click, Print["You clicked a button"]&amp;];
    /// </code>
    /// This discussion about RequestTransaction only applies to programmers who are writing their own custom .NET code.
    /// </para>
    /// </remarks>
    /// <exception cref="MathematicaNotReadyException">
    /// If the kernel is not in a state where it is receptive to calls originating in .NET, as described above.
    /// </exception>
    ///
    public static void RequestTransaction() {
        
        // requestTransaction must be a no-op under two circumstances. There is no reason
        // to call requestTransaction in these circumstances, but it must be safe:
        // - calls being made from a .NET program to a kernel it is $ParentLink'ed to. That is, no Reader thread running.
        // - calls being made as part of a callback from Mathematica.
        
        // This tells us if the Reader thread is not running.
        if (mainLink == null || !mainLinkHasReader)
            return;
        
        // This is the link that the user's subsequent call to StdLink.getLink() will return. Depending on which link
        // this is, we need to do different things below.
        IKernelLink linkThatWillBeUsed = Link;

        // This is an error condition (the user will get an exception later when they call Link and
        // try to use it), but there is nothing to do here.
        if (linkThatWillBeUsed == null)
            return;
		
        // It's always safe to call on the UI link.
        if (linkThatWillBeUsed == uiLink)
            return;
        
        // We only get here if we are not going to be using the uiLink. This could be because it is a
        // 5.0 kernel and there is no uiLink, or because we're in the modal state or we're in the
        // middle of a call from Mathematica.

        // We cannot use the mainLink if we are already in the middle of NextPacket() on that link.
        // The only likely scenario for this is that we are in a 5.0 kernel and we are not in the ShareKernel
        // or modal state.
        if (Reader.isInsideNextPacket) {
            //System.Diagnostics.Debug.WriteLine("inside next packet, throwing MathematicaNotReadyException");
            throw new MathematicaNotReadyException(MathematicaNotReadyException.KERNEL_NOT_SHARED);
        }

        if (Reader.isInModalState)
            return;

        // If we are in the middle of a callback from Mathematica, handleCallPacket is on the stack, which we can
        // detect by having the thread represented in stdLinkHash.
        lock (stdLinkLock) {
            object key = Thread.CurrentThread;
            Stack s = (Stack) stdLinkHash[key];
            if (s != null && s.Count > 0)
                return;
        }
    
        // We now know that mainLink is the correct link to communicate with.

        // Kernel is shared, but another link has its attention.
        if (!Reader.allowUIComputations) {
            System.Diagnostics.Debug.WriteLine("!allowUIComputations, throwing MathematicaNotReadyException");
            throw new MathematicaNotReadyException(MathematicaNotReadyException.FE_HAS_KERNEL_ATTENTION);
        }
    }


    /*********************************  End Public API  ********************************/

    internal static IKernelLink UILink {
        get { return uiLink; }
        set { uiLink = value; }
    }

    
    // Marks this link as having a Reader thread. In other words, it marks the link as being a "true" StdLink.
    // Because Link is a public property, some of the links that users will call it with do
    // not have the properties of a StdLink. Specifically, the RequestTransaction() method needs to distinguish
    // links that have a Reader thread associated with them (and thus all the issues that make RequestTransaction()
    // necessary) and those that do not. With Link public, it no longer suffices to examine whether there
    // is a mainLink in RequestTransaction(). We need to specifically know whether the mainLink is a Reader
    // link or not. Thus a new property (not public) was needed.
    internal static bool HasReader {
        get { return mainLinkHasReader; }
        set { mainLinkHasReader = value; }
    }
    

    internal static void setup(IKernelLink ml) {
        
        lock (stdLinkLock) {
            object key = Thread.CurrentThread;
            Stack s = (Stack) stdLinkHash[key];
            if (s == null) {
                // One-time only operation for any given thread.
                s = new Stack();
                stdLinkHash.Add(key, s);
            }
            s.Push(ml);
        }
    }
    
    internal static void remove() {
        
        lock (stdLinkLock) {
            object key = Thread.CurrentThread;
            Stack s = (Stack) stdLinkHash[key];
            // s will never be null.
            s.Pop();
        }
    }

}

}
