(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     52708,       1825]
NotebookOptionsPosition[     44355,       1540]
NotebookOutlinePosition[     46265,       1585]
CellTagsIndexPosition[     46222,       1582]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "TutorialColorBar"],

Cell[BoxData[GridBox[{
   {Cell["NUMERICAL CALCULUS PACKAGE TUTORIAL", "PacletNameCell"], Cell[
    TextData[Cell[BoxData[
     PopupMenuBox[
      Dynamic[{"ActionMenu", None}, 
       Part[{"EulerSum" :> 
         Documentation`HelpLookup["paclet:NumericalCalculus/ref/EulerSum"], 
         "Infinity" :> Documentation`HelpLookup["paclet:ref/Infinity"], "ND" :> 
         Documentation`HelpLookup["paclet:NumericalCalculus/ref/ND"], 
         "NIntegrate" :> Documentation`HelpLookup["paclet:ref/NIntegrate"], 
         "NLimit" :> 
         Documentation`HelpLookup["paclet:NumericalCalculus/ref/NLimit"], 
         "NResidue" :> 
         Documentation`HelpLookup["paclet:NumericalCalculus/ref/NResidue"], 
         "NSeries" :> 
         Documentation`HelpLookup["paclet:NumericalCalculus/ref/NSeries"], 
         "PrecisionGoal" :> 
         Documentation`HelpLookup["paclet:ref/PrecisionGoal"], "Radius" :> 
         Documentation`HelpLookup["paclet:NumericalCalculus/ref/Radius"], 
         "WorkingPrecision" :> 
         Documentation`HelpLookup["paclet:ref/WorkingPrecision"]}, #, 2]& , 
       Evaluator -> Automatic], {1->"\<\"EulerSum\"\>", 2->"\<\"Infinity\"\>",
       3->"\<\"ND\"\>", 4->"\<\"NIntegrate\"\>", 5->"\<\"NLimit\"\>", 
      6->"\<\"NResidue\"\>", 7->"\<\"NSeries\"\>", 8->"\<\"PrecisionGoal\"\>",
       9->"\<\"Radius\"\>", 
      10->"\<\"WorkingPrecision\"\>"}, "\<\"functions\"\>", 
      StyleBox["\<\"functions\"\>",
       Background->Automatic],
      Appearance->Automatic,
      ImageSize->Automatic,
      MenuAppearance->"Menu"]],
     FontSize->9]], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell["Numerical Calculus Package", "Title",
 CellID->509267359],

Cell[TextData[{
 "The functions defined in the ",
 Cell[BoxData["NumericalCalculus`"], "InlineFormula"],
 " context provide support for finding numerical solutions to \
calculus-related problems."
}], "Text",
 CellID->1534169418],

Cell["This loads the package:", "MathCaption",
 CellID->1825449234],

Cell[BoxData[
 RowBox[{"<<", "NumericalCalculus`"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1373894796],

Cell[CellGroupData[{

Cell["Numerical Calculation of Limits", "Section",
 CellID->2591],

Cell[TextData[{
 "The built\[Hyphen]in function ",
 Cell[BoxData[
  ButtonBox["Limit",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Limit"]], "InlineFormula"],
 " computes limits using symbolic and analytic methods. The function ",
 Cell[BoxData[
  ButtonBox["NLimit",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/NLimit"]], "InlineFormula"],
 " contained in the ",
 Cell[BoxData[
  ButtonBox["NumericalCalculus",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/guide/NumericalCalculusPackage"]], 
  "InlineFormula"],
 " package works by numerically evaluating a short sequence of function \
values as the argument approaches the specified point. The result of this \
calculation is passed to a routine that uses either Wynn\[CloseCurlyQuote]s \
",
 Cell[BoxData[
  FormBox["\[Epsilon]", TraditionalForm]], "InlineFormula"],
 "\[Hyphen]algorithm or a generalized Euler transformation to find an \
approximation to the limit. "
}], "Text",
 CellID->16706],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["NLimit",
      BaseStyle->"Link",
      ButtonData->"paclet:NumericalCalculus/ref/NLimit"], "[", 
     StyleBox["expr", "TI"], ",", 
     StyleBox["x", "TI"], "->", 
     SubscriptBox[
      StyleBox["x", "TI"], "0"], "]"}], Cell[TextData[{
     "numerically find the limit as ",
     Cell[BoxData[
      StyleBox["x", "TI"]], "InlineFormula"],
     " approaches ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["x", "TI"], "0"]], "InlineFormula"]
    }], "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {{False}}, 
   "RowsIndexed" -> {}}},
 CellID->19362],

Cell["Computing numerical limits. ", "Caption",
 CellID->19031],

Cell["Here is the numerical computation of a limit. ", "MathCaption",
 CellID->6421],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NLimit", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Zeta", "[", "s", "]"}], "-", 
    FractionBox["1", 
     RowBox[{"s", "-", "1"}]]}], ",", 
   RowBox[{"s", "\[Rule]", "1"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->27111],

Cell[BoxData["0.577215659712782`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->191037609]
}, Open  ]],

Cell["\<\
The limit gives the number known as Euler\[CloseCurlyQuote]s gamma. \
\>", "MathCaption",
 CellID->7348],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "EulerGamma", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->3464],

Cell[BoxData["0.5772156649015329`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->498198997]
}, Open  ]],

Cell[TextData[{
 "You can also compute limits as ",
 Cell[BoxData[
  FormBox[
   StyleBox["x", "TI"], TraditionalForm]], "InlineFormula"],
 " approaches infinity. "
}], "MathCaption",
 CellID->25569],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NLimit", "[", 
  RowBox[{
   FractionBox[
    RowBox[{
     RowBox[{"2", " ", 
      SuperscriptBox["x", "3"]}], "+", 
     RowBox[{"Sin", "[", "x", "]"}]}], 
    RowBox[{
     RowBox[{"5", " ", 
      SuperscriptBox["x", "3"]}], "+", 
     RowBox[{"Log", "[", "x", "]"}]}]], ",", 
   RowBox[{"x", "\[Rule]", "\[Infinity]"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->2989],

Cell[BoxData["0.3999999999999999`"], "Output",
 ImageSize->{25, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->654426462]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {
    ButtonBox["WorkingPrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/WorkingPrecision"], 
    ButtonBox["MachinePrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MachinePrecision"], Cell[
    "number of digits of precision to be used ", "TableText"]},
   {
    ButtonBox["Scale",
     BaseStyle->"Link",
     ButtonData->"paclet:NumericalCalculus/ref/Scale"], "1", Cell[
    "initial step size ", "TableText"]},
   {
    ButtonBox["Terms",
     BaseStyle->"Link",
     ButtonData->"paclet:NumericalCalculus/ref/Terms"], "7", Cell[
    "total number of terms generated in the sequence ", "TableText"]},
   {
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    ButtonBox["EulerSum",
     BaseStyle->"Link",
     ButtonData->"paclet:NumericalCalculus/ref/EulerSum"], Cell[TextData[{
     "method of evaluation, either ",
     StyleBox[ButtonBox["EulerSum",
      BaseStyle->"Link",
      ButtonData->"paclet:NumericalCalculus/ref/EulerSum"], "InlineCode"],
     " or SequenceLimit "
    }], "TableText"]},
   {"WynnDegree", "1", Cell["degree to use in the Wynn method", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->2017],

Cell[TextData[{
 "Options for ",
 Cell[BoxData[
  ButtonBox["NLimit",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/NLimit"]], "InlineFormula"],
 "."
}], "Caption",
 CellID->21763],

Cell[TextData[{
 "There are many specialized techniques for accelerating the convergence of a \
series. Faster convergence means that in a numerical evaluation fewer terms \
are needed to give a result of the desired accuracy. One such technique, \
Euler\[CloseCurlyQuote]s transformation, is used by ",
 Cell[BoxData[
  ButtonBox["EulerSum",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/EulerSum"]], "InlineFormula"],
 ". "
}], "Text",
 CellID->17753],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["EulerSum",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/EulerSum"]], "InlineFormula"],
 " is particularly good for working with alternating series and series whose \
terms have the form ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    StyleBox["p", "TI"], 
    RowBox[{"(", 
     StyleBox["n", "TI"], ")"}], 
    SuperscriptBox[
     StyleBox["r", "TI"], 
     StyleBox["n", "TI"]]}], TraditionalForm]], "InlineFormula"],
 " where ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    StyleBox["p", "TI"], 
    RowBox[{"(", 
     StyleBox["n", "TI"], ")"}]}], TraditionalForm]], "InlineFormula"],
 " is a polynomial in ",
 Cell[BoxData[
  FormBox[
   StyleBox["n", "TI"], TraditionalForm]], "InlineFormula"],
 ". It is a useful adjunct to the built\[Hyphen]in ",
 Cell[BoxData[
  ButtonBox["NSum",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NSum"]], "InlineFormula"],
 ", which is designed to be an effective general case algorithm. "
}], "Text",
 CellID->27158],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["EulerSum",
      BaseStyle->"Link",
      ButtonData->"paclet:NumericalCalculus/ref/EulerSum"], "[", 
     StyleBox["f", "TI"], ",", "{", 
     StyleBox["i", "TI"], ",", 
     StyleBox["imin", "TI"], ",", 
     ButtonBox["Infinity",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Infinity"], "}", "]"}], Cell[TextData[{
     "use Euler\[CloseCurlyQuote]s transformation to numerically evaluate ",
     Cell[BoxData[
      RowBox[{
       UnderoverscriptBox["\[Sum]", 
        StyleBox["imin", "TI"], "\[Infinity]",
        LimitsPositioning->True], " ", 
       StyleBox["f", "TI"]}]], "InlineFormula"],
     " "
    }], "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {{False}}, 
   "RowsIndexed" -> {}}},
 CellID->25550],

Cell["\<\
Numerical evaluation of sums using Euler\[CloseCurlyQuote]s transformation. \
\>", "Caption",
 CellID->9037],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {
    ButtonBox["WorkingPrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/WorkingPrecision"], 
    ButtonBox["MachinePrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MachinePrecision"], Cell[
    "number of digits of precision to be used ", "TableText"]},
   {
    ButtonBox["Terms",
     BaseStyle->"Link",
     ButtonData->"paclet:NumericalCalculus/ref/Terms"], "5", Cell[
    "total number of terms generated before extrapolation ", "TableText"]},
   {
    ButtonBox["ExtraTerms",
     BaseStyle->"Link",
     ButtonData->"paclet:NumericalCalculus/ref/ExtraTerms"], "7", Cell["\<\
number of terms to be used in the extrapolation process; must be at least 2\
\>", "TableText"]},
   {
    ButtonBox["EulerRatio",
     BaseStyle->"Link",
     ButtonData->"paclet:NumericalCalculus/ref/EulerRatio"], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "the fixed ratio to be used in the transformation ", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->29072],

Cell[TextData[{
 "Options for ",
 Cell[BoxData[
  ButtonBox["EulerSum",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/EulerSum"]], "InlineFormula"],
 "."
}], "Caption",
 CellID->26739],

Cell[TextData[{
 "This uses ",
 Cell[BoxData[
  ButtonBox["EulerSum",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/EulerSum"]], "InlineFormula"],
 " to compute a numerical approximation to the sum of an alternating series. \
"
}], "MathCaption",
 CellID->1781],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"EulerSum", "[", 
  RowBox[{
   FractionBox[
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"-", "1"}], ")"}], "k"], 
    RowBox[{
     RowBox[{"2", " ", "k"}], "+", "1"}]], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "0", ",", "\[Infinity]"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->26975],

Cell[BoxData["0.7853983570594825`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->705840830]
}, Open  ]],

Cell["\<\
Increasing the precision and the number of terms does not change the answer \
in the first six decimal places. This provides a check on the accuracy of the \
numerically computed result. \
\>", "MathCaption",
 CellID->19289],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"EulerSum", "[", 
  RowBox[{
   FractionBox[
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"-", "1"}], ")"}], "k"], 
    RowBox[{
     RowBox[{"2", " ", "k"}], "+", "1"}]], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "0", ",", "\[Infinity]"}], "}"}], ",", 
   RowBox[{"WorkingPrecision", "\[Rule]", "40"}], ",", 
   RowBox[{"Terms", "\[Rule]", "30"}], ",", 
   RowBox[{"ExtraTerms", "\[Rule]", "30"}]}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->7358],

Cell[BoxData["0.7853981633974483096156608457913032254024560638073841639228`39.\
14491374808192"], "Output",
 ImageSize->{291, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->230069580]
}, Open  ]],

Cell["\<\
The result is also in good agreement with the known exact result. \
\>", "MathCaption",
 CellID->27254],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"%", "-", 
  RowBox[{"N", "[", 
   RowBox[{
    FractionBox["\[Pi]", "4"], ",", "40"}], "]"}]}]], "Input",
 CellLabel->"In[7]:=",
 CellID->23509],

Cell[BoxData[
 RowBox[{"-", 
  "2.85724956468362860363922913235`10.649016130894474*^-29"}]], "Output",
 ImageSize->{137, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->211871406]
}, Open  ]],

Cell[TextData[{
 "There are occasionally instances in which it is difficult or impossible to \
analytically compute the derivative of a function. In these cases you can \
compute the derivative using the function ",
 Cell[BoxData[
  ButtonBox["ND",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/ND"]], "InlineFormula"],
 " instead of the built\[Hyphen]in function ",
 Cell[BoxData[
  ButtonBox["D",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/D"]], "InlineFormula"],
 ". "
}], "Text",
 CellID->26307],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["ND",
      BaseStyle->"Link",
      ButtonData->"paclet:NumericalCalculus/ref/ND"], "[", 
     StyleBox["f", "TI"], ",", 
     StyleBox["x", "TI"], ",", 
     SubscriptBox[
      StyleBox["x", "TI"], "0"], "]"}], Cell[TextData[{
     "give a numerical approximation to ",
     Cell[BoxData[
      FractionBox[
       RowBox[{"\[PartialD]", 
        StyleBox["f", "TI"]}], 
       RowBox[{"\[PartialD]", 
        StyleBox["x", "TI"]}]]], "InlineFormula"],
     " evaluated at the point\[NonBreakingSpace]",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["x", "TI"], "0"]], "InlineFormula"]
    }], "TableText"]},
   {
    RowBox[{
     ButtonBox["ND",
      BaseStyle->"Link",
      ButtonData->"paclet:NumericalCalculus/ref/ND"], "[", 
     StyleBox["f", "TI"], ",", "{", 
     StyleBox["x", "TI"], ",", 
     StyleBox["n", "TI"], "}", ",", 
     SubscriptBox[
      StyleBox["x", "TI"], "0"], "]"}], Cell[TextData[{
     "give the ",
     Cell[BoxData[
      SuperscriptBox[
       StyleBox["n", "TI"], "th"]], "InlineFormula"],
     " derivative"
    }], "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {{False}}, 
   "RowsIndexed" -> {}}},
 CellID->6442],

Cell["Numerical computation of derivatives. ", "Caption",
 CellID->27758],

Cell[TextData[{
 "Here is a numerical approximation of the derivative of ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox[
    StyleBox["e", "TI"], 
    RowBox[{"sin", 
     RowBox[{"(", 
      StyleBox["x", "TI"], ")"}]}]], TraditionalForm]], "InlineFormula"],
 " at ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    StyleBox["x", "TI"], "=", "2"}], TraditionalForm]], "InlineFormula"],
 ". "
}], "MathCaption",
 CellID->6249],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ND", "[", 
  RowBox[{
   SuperscriptBox["\[ExponentialE]", 
    RowBox[{"Sin", "[", "x", "]"}]], ",", "x", ",", "2"}], "]"}]], "Input",
 CellLabel->"In[8]:=",
 CellID->13196],

Cell[BoxData[
 RowBox[{"-", "1.0331168751058337`"}]], "Output",
 ImageSize->{61, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[8]=",
 CellID->143307490]
}, Open  ]],

Cell[TextData[{
 "After increasing the setting of the option ",
 Cell[BoxData[
  ButtonBox["Terms",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/Terms"]], "InlineFormula"],
 ", the difference between the symbolically computed result and the numerical \
result for a third derivative is quite small. "
}], "MathCaption",
 CellID->29449],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"ND", "[", 
   RowBox[{
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"Sin", "[", "x", "]"}]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "3"}], "}"}], ",", "2", ",", 
    RowBox[{"Terms", "\[Rule]", "10"}]}], "]"}], "-", 
  RowBox[{"N", "[", 
   RowBox[{
    RowBox[{
     SubscriptBox["\[PartialD]", 
      RowBox[{"{", 
       RowBox[{"x", ",", "3"}], "}"}]], 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"Sin", "[", "x", "]"}]]}], "/.", "\[InvisibleSpace]", 
    RowBox[{"x", "\[Rule]", "2"}]}], "]"}]}]], "Input",
 CellLabel->"In[9]:=",
 CellID->23083],

Cell[BoxData["2.239886525146062`*^-7"], "Output",
 ImageSize->{89, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[9]=",
 CellID->187596744]
}, Open  ]],

Cell["\<\
This is the error in the numerical derivative using Cauchy\[CloseCurlyQuote]s \
integral formula. \
\>", "MathCaption",
 CellID->15204],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"ND", "[", 
   RowBox[{
    SuperscriptBox["\[ExponentialE]", 
     RowBox[{"Sin", "[", "x", "]"}]], ",", 
    RowBox[{"{", 
     RowBox[{"x", ",", "3"}], "}"}], ",", "2", ",", 
    RowBox[{"Method", "\[Rule]", "NIntegrate"}]}], "]"}], "-", 
  RowBox[{"N", "[", 
   RowBox[{
    RowBox[{
     SubscriptBox["\[PartialD]", 
      RowBox[{"{", 
       RowBox[{"x", ",", "3"}], "}"}]], 
     SuperscriptBox["\[ExponentialE]", 
      RowBox[{"Sin", "[", "x", "]"}]]}], "/.", "\[InvisibleSpace]", 
    RowBox[{"x", "\[Rule]", "2"}]}], "]"}]}]], "Input",
 CellLabel->"In[10]:=",
 CellID->31242],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", "1.3500311979441904`*^-13"}], "+", 
  RowBox[{"1.790234627208065`*^-15", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{215, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[10]=",
 CellID->556571991]
}, Open  ]],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {
    ButtonBox["WorkingPrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/WorkingPrecision"], 
    ButtonBox["MachinePrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MachinePrecision"], Cell[
    "number of digits of precision to be used", "TableText"]},
   {
    ButtonBox["Scale",
     BaseStyle->"Link",
     ButtonData->"paclet:NumericalCalculus/ref/Scale"], "1", Cell[
    "size of steps in the evaluation ", "TableText"]},
   {
    ButtonBox["Terms",
     BaseStyle->"Link",
     ButtonData->"paclet:NumericalCalculus/ref/Terms"], "7", Cell[
    "total number of terms generated in the sequence ", "TableText"]},
   {
    ButtonBox["Method",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Method"], 
    ButtonBox["EulerSum",
     BaseStyle->"Link",
     ButtonData->"paclet:NumericalCalculus/ref/EulerSum"], Cell[TextData[{
     "method of evaluation, either ",
     StyleBox[ButtonBox["EulerSum",
      BaseStyle->"Link",
      ButtonData->"paclet:NumericalCalculus/ref/EulerSum"], "InlineCode"],
     " (numerical limit of difference quotients) or ",
     StyleBox[ButtonBox["NIntegrate",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/NIntegrate"], "InlineCode"],
     " (Cauchy\[CloseCurlyQuote]s integral formula) "
    }], "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->5317],

Cell[TextData[{
 "Options for ",
 Cell[BoxData[
  ButtonBox["ND",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/ND"]], "InlineFormula"],
 "."
}], "Caption",
 CellID->32656],

Cell[TextData[{
 "You should realize that with sufficiently pathological examples, the \
algorithms used for ",
 Cell[BoxData[
  ButtonBox["EulerSum",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/EulerSum"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["NLimit",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/NLimit"]], "InlineFormula"],
 ", or ",
 Cell[BoxData[
  ButtonBox["ND",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/ND"]], "InlineFormula"],
 " can give wrong answers. In most cases, you can test your answer by looking \
at the sensitivity to changes in the setting of options. "
}], "Text",
 CellID->11452]
}, Open  ]],

Cell[CellGroupData[{

Cell["Numerical Calculation of Residues", "Section",
 CellID->30767],

Cell[TextData[{
 "The ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " function ",
 Cell[BoxData[
  ButtonBox["Residue",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Residue"]], "InlineFormula"],
 " symbolically finds the residue of an expression at a point in the complex \
plane. Because it is symbolic in nature it is sometimes unable to get a \
result. "
}], "Text",
 CellID->3951],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NResidue",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/NResidue"]], "InlineFormula"],
 " is the numerical version of ",
 Cell[BoxData[
  ButtonBox["Residue",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Residue"]], "InlineFormula"],
 ". It works by numerically integrating around a small circle centered at the \
point at which the residue is being sought. The obvious problem with this \
approach is that it in fact finds the sum of the residues at all of the \
points contained within the circle. By making the radius of the circle \
sufficiently small you can exclude all singularities but the one in question. \
"
}], "Text",
 CellID->16117],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["NResidue",
      BaseStyle->"Link",
      ButtonData->"paclet:NumericalCalculus/ref/NResidue"], "[", 
     StyleBox["expr", "TI"], ",", "{", 
     StyleBox["x", "TI"], ",", 
     SubscriptBox[
      StyleBox["x", "TI"], "0"], "}", "]"}], Cell[TextData[{
     "numerically find the residue of ",
     Cell[BoxData[
      StyleBox["expr", "TI"]], "InlineFormula"],
     " at the point ",
     Cell[BoxData[
      RowBox[{
       StyleBox["x", "TI"], "="}]], "InlineFormula"],
     " ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["x", "TI"], "0"]], "InlineFormula"],
     " "
    }], "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {{False}}, 
   "RowsIndexed" -> {}}},
 CellID->8707],

Cell["Numerical evaluation of residues. ", "Caption",
 CellID->28587],

Cell[BoxData[GridBox[{
   {
    ButtonBox["Radius",
     BaseStyle->"Link",
     ButtonData->"paclet:NumericalCalculus/ref/Radius"], Cell["\<\
specifies the radius of the circle around which the integration is performed \
\
\>", "TableText"]},
   {
    ButtonBox["PrecisionGoal",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/PrecisionGoal"], Cell[TextData[{
     "the ",
     StyleBox[ButtonBox["PrecisionGoal",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/PrecisionGoal"], "InlineCode"],
     " to be used by ",
     StyleBox[ButtonBox["NIntegrate",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/NIntegrate"], "InlineCode"],
     " "
    }], "TableText"]},
   {
    ButtonBox["WorkingPrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/WorkingPrecision"], Cell[TextData[{
     "the ",
     StyleBox[ButtonBox["WorkingPrecision",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/WorkingPrecision"], "InlineCode"],
     " to be used by ",
     StyleBox[ButtonBox["NIntegrate",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/NIntegrate"], "InlineCode"],
     " "
    }], "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {{False}}, 
   "RowsIndexed" -> {}}},
 CellID->32556],

Cell[TextData[{
 "Options for ",
 Cell[BoxData[
  ButtonBox["NResidue",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/NResidue"]], "InlineFormula"],
 ". "
}], "Caption",
 CellID->4207],

Cell[TextData[{
 "Find the residue of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"1", "/", 
    StyleBox["z", "TI"]}], TraditionalForm]], "InlineFormula"],
 " at the origin. "
}], "MathCaption",
 CellID->16622],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NResidue", "[", 
  RowBox[{
   FractionBox["1", "z"], ",", 
   RowBox[{"{", 
    RowBox[{"z", ",", "0"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->16390],

Cell[BoxData["1.`"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->7936742]
}, Open  ]],

Cell["Define an expression whose residue you will find. ", "MathCaption",
 CellID->22729],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"f", "=", 
  FractionBox["1", 
   RowBox[{"Expand", "[", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"z", "-", "1.7"}], ")"}], " ", 
     RowBox[{"(", 
      RowBox[{"z", "+", ".2", "+", 
       RowBox[{".5", " ", "\[ImaginaryI]"}]}], ")"}], " ", 
     RowBox[{"(", 
      RowBox[{"z", "+", ".2", "-", 
       RowBox[{".5", " ", "\[ImaginaryI]"}]}], ")"}]}], "]"}]]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->27477],

Cell[BoxData[
 RowBox[{"1", "/", 
  RowBox[{"(", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "0.493`"}], "-", 
      RowBox[{"7.209944447028604`*^-18", " ", "\[ImaginaryI]"}]}], ")"}], "-",
     
    RowBox[{
     RowBox[{"(", 
      RowBox[{"0.39`", "\[InvisibleSpace]", "+", 
       RowBox[{"0.`", " ", "\[ImaginaryI]"}]}], ")"}], " ", "z"}], "-", 
    RowBox[{
     RowBox[{"(", 
      RowBox[{"1.3`", "\[InvisibleSpace]", "+", 
       RowBox[{"0.`", " ", "\[ImaginaryI]"}]}], ")"}], " ", 
     SuperscriptBox["z", "2"]}], "+", 
    SuperscriptBox["z", "3"]}], ")"}]}]], "Output",
 ImageSize->{435, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->264026320]
}, Open  ]],

Cell[TextData[{
 "Find the residue. Strictly speaking, ",
 StyleBox["f", "MR"],
 " has no singularity at ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    StyleBox["z", "TI"], "=", "1.7"}], TraditionalForm]], "InlineFormula"],
 ", but it has one very near to 1.7. "
}], "MathCaption",
 CellID->20455],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Residue", "[", 
  RowBox[{"f", ",", 
   RowBox[{"{", 
    RowBox[{"z", ",", "1.7"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->20334],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->998948]
}, Open  ]],

Cell["Numerically find the residue. ", "MathCaption",
 CellID->23883],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NResidue", "[", 
  RowBox[{"f", ",", 
   RowBox[{"{", 
    RowBox[{"z", ",", "1.7"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->19662],

Cell[BoxData[
 RowBox[{"0.2590673575129538`", "\[InvisibleSpace]", "-", 
  RowBox[{"3.296178321255481`*^-17", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{174, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->174038953]
}, Open  ]],

Cell["This is another way to find the residue. ", "MathCaption",
 CellID->13284],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  FractionBox["1", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"z", "+", ".2", "+", 
      RowBox[{".5", " ", "\[ImaginaryI]"}]}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{"z", "+", ".2", "-", 
      RowBox[{".5", " ", "\[ImaginaryI]"}]}], ")"}]}]], "/.", 
  "\[InvisibleSpace]", 
  RowBox[{"z", "\[Rule]", "1.7"}]}]], "Input",
 CellLabel->"In[6]:=",
 CellID->29804],

Cell[BoxData[
 RowBox[{"0.25906735751295334`", "\[InvisibleSpace]", "+", 
  RowBox[{"0.`", " ", "\[ImaginaryI]"}]}]], "Output",
 ImageSize->{98, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->58319039]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Numerical Calculation of Series Expansions", "Section",
 CellID->32197],

Cell[TextData[{
 "The ",
 StyleBox["Mathematica",
  FontSlant->"Italic"],
 " function ",
 Cell[BoxData[
  ButtonBox["Series",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Series"]], "InlineFormula"],
 " finds a power series expansion of a function about a point by evaluating \
derivatives. However, sometimes you only want a numerical approximation to a \
series and the derivatives are difficult to evaluate. You could use ",
 Cell[BoxData[
  ButtonBox["ND",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/ND"]], "InlineFormula"],
 " to evaluate the derivatives, but that would involve repeating most of the \
work (",
 StyleBox["i.e.",
  FontSlant->"Italic"],
 ", evaluating the function itself at several points) for each derivative. If \
the function is analytic and can be evaluated in the complex plane, the \
solution is to use the ",
 Cell[BoxData[
  ButtonBox["NSeries",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/NSeries"]], "InlineFormula"],
 ". "
}], "Text",
 CellID->19192],

Cell[BoxData[GridBox[{
   {
    RowBox[{
     ButtonBox["NSeries",
      BaseStyle->"Link",
      ButtonData->"paclet:NumericalCalculus/ref/NSeries"], "[", 
     StyleBox["f", "TI"], ",", "{", 
     StyleBox["x", "TI"], ",", 
     SubscriptBox[
      StyleBox["x", "TI"], "0"], ",", 
     StyleBox["n", "TI"], "}", "]"}], Cell[TextData[{
     "give a numerical approximation to the series expansion of ",
     Cell[BoxData[
      StyleBox["f", "TI"]], "InlineFormula"],
     " about ",
     Cell[BoxData[
      RowBox[{
       StyleBox["x", "TI"], "=", 
       SubscriptBox[
        StyleBox["x", "TI"], "0"]}]], "InlineFormula"],
     " including the terms ",
     Cell[BoxData[
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{
         StyleBox["x", "TI"], "-", 
         SubscriptBox[
          StyleBox["x", "TI"], "0"]}], ")"}], 
       RowBox[{"-", 
        StyleBox["n", "TI"]}]]], "InlineFormula"],
     " through ",
     Cell[BoxData[
      SuperscriptBox[
       RowBox[{"(", 
        RowBox[{
         StyleBox["x", "TI"], "-", 
         SubscriptBox[
          StyleBox["x", "TI"], "0"]}], ")"}], 
       StyleBox["n", "TI"]]], "InlineFormula"],
     " "
    }], "TableText"]}
  }]], "DefinitionBox",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, "Rows" -> {{False}}, 
   "RowsIndexed" -> {}}},
 CellID->19045],

Cell["Numerically finding series expansions. ", "Caption",
 CellID->7357],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NSeries",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/NSeries"]], "InlineFormula"],
 " works by evaluating the function at discrete points on a circle centered \
at the expansion point. It then uses ",
 Cell[BoxData[
  ButtonBox["InverseFourier",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InverseFourier"]], "InlineFormula"],
 " to express the function as a sum of orthogonal basis functions, each of \
which is trivial to integrate around the circle. By Cauchy\[CloseCurlyQuote]s \
integral formula it is seen that ",
 Cell[BoxData[
  ButtonBox["InverseFourier",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/InverseFourier"]], "InlineFormula"],
 " essentially gives us the required series coefficients; very little \
additional work is required and ",
 Cell[BoxData[
  ButtonBox["NSeries",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/NSeries"]], "InlineFormula"],
 " does it all for you. Because it is based on Cauchy\[CloseCurlyQuote]s \
integral formula, care should be taken to ensure that no singularities of the \
function occur inside the circle of sample points. "
}], "Text",
 CellID->25290],

Cell[BoxData[GridBox[{
   {Cell["option name", "TableHeader"], Cell["default value", "TableHeader"], 
    " "},
   {
    ButtonBox["WorkingPrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/WorkingPrecision"], 
    ButtonBox["MachinePrecision",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/MachinePrecision"], Cell[
    "precision of the arithmetic to use in calculations ", "TableText"]},
   {
    ButtonBox["Radius",
     BaseStyle->"Link",
     ButtonData->"paclet:NumericalCalculus/ref/Radius"], "1", Cell[
    "radius of the circle on which the function is sampled ", "TableText"]}
  }]], "DefinitionBox3Col",
 GridBoxOptions->{
 GridBoxDividers->{
  "Columns" -> {{False}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {False, True, {False}, False}, "RowsIndexed" -> {}}},
 CellID->3225],

Cell[TextData[{
 "Options for ",
 Cell[BoxData[
  ButtonBox["NSeries",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/NSeries"]], "InlineFormula"],
 "."
}], "Caption",
 CellID->16825],

Cell[TextData[{
 "This gives an approximation to the series expansion of ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox[
    StyleBox["e", "TI"], 
    StyleBox["x", "TI"]], TraditionalForm]], "InlineFormula"],
 " about the origin. Note that ",
 Cell[BoxData[
  ButtonBox["NSeries",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/NSeries"]], "InlineFormula"],
 " is unable to recognize small numbers that should in fact be zero. "
}], "MathCaption",
 CellID->26053],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NSeries", "[", 
  RowBox[{
   SuperscriptBox["\[ExponentialE]", "x"], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->19391],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   FractionBox[
    RowBox[{"1.605902378646316`*^-10", "+", 
     RowBox[{"9.357319639010834`*^-17", " ", "\[ImaginaryI]"}]}], 
    SuperscriptBox["x", "3"]], "+", 
   FractionBox[
    RowBox[{"1.1470626338827344`*^-11", "-", 
     RowBox[{"4.7362466919702726`*^-17", " ", "\[ImaginaryI]"}]}], 
    SuperscriptBox["x", "2"]], "+", 
   FractionBox[
    RowBox[{"7.647002927662619`*^-13", "-", 
     RowBox[{"6.792549346518558`*^-17", " ", "\[ImaginaryI]"}]}], "x"], "+", 
   RowBox[{"(", 
    RowBox[{"1.0000000000000477`", "\[InvisibleSpace]", "-", 
     RowBox[{"1.6266193360102852`*^-17", " ", "\[ImaginaryI]"}]}], ")"}], "+",
    
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1.0000000000000027`", "\[InvisibleSpace]", "-", 
      RowBox[{"1.7179883785287495`*^-16", " ", "\[ImaginaryI]"}]}], ")"}], 
    " ", "x"}], "+", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"0.5000000000000002`", "\[InvisibleSpace]", "-", 
      RowBox[{"9.770671717270932`*^-17", " ", "\[ImaginaryI]"}]}], ")"}], " ",
     
    SuperscriptBox["x", "2"]}], "+", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"0.1666666666666667`", "\[InvisibleSpace]", "-", 
      RowBox[{"3.904336402970845`*^-17", " ", "\[ImaginaryI]"}]}], ")"}], " ",
     
    SuperscriptBox["x", "3"]}], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "4"],
    SeriesData[x, 0, {}, -3, 4, 1],
    Editable->False]}],
  SeriesData[x, 0, {
    Complex[1.60590237864632*^-10, 9.35731963901083*^-17], 
    Complex[1.14706263388273*^-11, -4.73624669197027*^-17], 
    Complex[7.64700292766262*^-13, -6.79254934651856*^-17], 
    Complex[1.00000000000005, -1.62661933601029*^-17], 
    Complex[1., -1.71798837852875*^-16], 
    Complex[0.5, -9.77067171727093*^-17], 
    Complex[0.166666666666667, -3.90433640297085*^-17]}, -3, 4, 1],
  Editable->False]], "Output",
 ImageSize->{533, 86},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->342387]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Chop",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Chop"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["Rationalize",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Rationalize"]], "InlineFormula"],
 " can often clean up the result. "
}], "MathCaption",
 CellID->27664],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Rationalize", "[", 
  RowBox[{"Chop", "[", "%", "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->6364],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   FractionBox["1.605902378646316`*^-10", 
    SuperscriptBox["x", "3"]], "+", "1", "+", "x", "+", 
   FractionBox[
    SuperscriptBox["x", "2"], "2"], "+", 
   FractionBox[
    SuperscriptBox["x", "3"], "6"], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "4"],
    SeriesData[x, 0, {}, -3, 4, 1],
    Editable->False]}],
  SeriesData[x, 0, {1.60590237864632*^-10, 0, 0, 1, 1, 
    Rational[1, 2], 
    Rational[1, 6]}, -3, 4, 1],
  Editable->False]], "Output",
 ImageSize->{235, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->48540072]
}, Open  ]],

Cell["If the radius is too small, cancellation becomes a problem. ", \
"MathCaption",
 CellID->6827],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Rationalize", "[", 
  RowBox[{"Chop", "[", 
   RowBox[{"NSeries", "[", 
    RowBox[{
     SuperscriptBox["\[ExponentialE]", "x"], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "5"}], "}"}], ",", 
     RowBox[{"Radius", "\[Rule]", 
      FractionBox["1", "8"]}]}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->17078],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"1", "+", "x", "+", 
   FractionBox[
    SuperscriptBox["x", "2"], "2"], "+", 
   FractionBox[
    SuperscriptBox["x", "3"], "6"], "+", 
   RowBox[{"0.04166666666663863`", " ", 
    SuperscriptBox["x", "4"]}], "+", 
   RowBox[{"0.008333333333300637`", " ", 
    SuperscriptBox["x", "5"]}], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "6"],
    SeriesData[x, 0, {}, 0, 6, 1],
    Editable->False]}],
  SeriesData[x, 0, {1, 1, 
    Rational[1, 2], 
    Rational[1, 6], 0.0416666666666386, 0.00833333333330064}, 0, 6, 1],
  Editable->False]], "Output",
 ImageSize->{325, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->688756]
}, Open  ]],

Cell[TextData[{
 "Increasing the ",
 Cell[BoxData[
  ButtonBox["WorkingPrecision",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/WorkingPrecision"]], "InlineFormula"],
 " will usually solve this problem. "
}], "MathCaption",
 CellID->6182],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Rationalize", "[", 
  RowBox[{"Chop", "[", 
   RowBox[{"NSeries", "[", 
    RowBox[{
     SuperscriptBox["\[ExponentialE]", "x"], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "5"}], "}"}], ",", 
     RowBox[{"WorkingPrecision", "\[Rule]", "40"}], ",", 
     RowBox[{"Radius", "\[Rule]", 
      FractionBox["1", "8"]}]}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[5]:=",
 CellID->25862],

Cell[BoxData[
 InterpretationBox[
  RowBox[{"1", "+", "x", "+", 
   FractionBox[
    SuperscriptBox["x", "2"], "2"], "+", 
   FractionBox[
    SuperscriptBox["x", "3"], "6"], "+", 
   FractionBox[
    SuperscriptBox["x", "4"], "24"], "+", 
   FractionBox[
    SuperscriptBox["x", "5"], "120"], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "6"],
    SeriesData[x, 0, {}, 0, 6, 1],
    Editable->False]}],
  SeriesData[x, 0, {1, 1, 
    Rational[1, 2], 
    Rational[1, 6], 
    Rational[1, 24], 
    Rational[1, 120]}, 0, 6, 1],
  Editable->False]], "Output",
 ImageSize->{204, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[5]=",
 CellID->210061539]
}, Open  ]],

Cell["\<\
If the radius is too large, the sample points become too scattered and it \
starts looking like poles exist. \
\>", "MathCaption",
 CellID->20963],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Rationalize", "[", 
  RowBox[{"Chop", "[", 
   RowBox[{"NSeries", "[", 
    RowBox[{
     SuperscriptBox["\[ExponentialE]", "x"], ",", 
     RowBox[{"{", 
      RowBox[{"x", ",", "0", ",", "5"}], "}"}], ",", 
     RowBox[{"Radius", "\[Rule]", "4"}]}], "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[6]:=",
 CellID->31555],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   FractionBox["1.6929016799505088`*^-9", 
    SuperscriptBox["x", "5"]], "+", "1", "+", "x", "+", 
   FractionBox[
    SuperscriptBox["x", "2"], "2"], "+", 
   FractionBox[
    SuperscriptBox["x", "3"], "6"], "+", 
   FractionBox[
    SuperscriptBox["x", "4"], "24"], "+", 
   FractionBox[
    SuperscriptBox["x", "5"], "120"], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", "x", "]"}], "6"],
    SeriesData[x, 0, {}, -5, 6, 1],
    Editable->False]}],
  SeriesData[x, 0, {1.69290167995051*^-9, 0, 0, 0, 0, 1, 1, 
    Rational[1, 2], 
    Rational[1, 6], 
    Rational[1, 24], 
    Rational[1, 120]}, -5, 6, 1],
  Editable->False]], "Output",
 ImageSize->{299, 34},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[6]=",
 CellID->111197973]
}, Open  ]],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["NSeries",
   BaseStyle->"Link",
   ButtonData->"paclet:NumericalCalculus/ref/NSeries"]], "InlineFormula"],
 " also recognizes poles. "
}], "MathCaption",
 CellID->23438],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Chop", "[", 
  RowBox[{"NSeries", "[", 
   RowBox[{
    RowBox[{"Zeta", "[", "s", "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"s", ",", "1", ",", "5"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[7]:=",
 CellID->14136],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   FractionBox["1", 
    RowBox[{"s", "-", "1"}]], "+", "0.5772156649015326`", "+", 
   RowBox[{"0.07281584548367676`", " ", 
    RowBox[{"(", 
     RowBox[{"s", "-", "1"}], ")"}]}], "-", 
   RowBox[{"0.004845181596436386`", " ", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"s", "-", "1"}], ")"}], "2"]}], "-", 
   RowBox[{"0.00034230573671693886`", " ", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"s", "-", "1"}], ")"}], "3"]}], "+", 
   RowBox[{"0.00009689041939406385`", " ", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"s", "-", "1"}], ")"}], "4"]}], "-", 
   RowBox[{"6.611031810638576`*^-6", " ", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{"s", "-", "1"}], ")"}], "5"]}], "+", 
   InterpretationBox[
    SuperscriptBox[
     RowBox[{"O", "[", 
      RowBox[{"s", "-", "1"}], "]"}], "6"],
    SeriesData[s, 1, {}, -1, 6, 1],
    Editable->False]}],
  SeriesData[
  s, 1, {1., 0.577215664901533, 
   0.0728158454836768, -0.00484518159643639, -0.000342305736716939, 
   0.0000968904193940638, -6.61103181063858*^-6}, -1, 6, 1],
  Editable->False]], "Output",
 ImageSize->{496, 47},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[7]=",
 CellID->634463235]
}, Open  ]]
}, Open  ]]
}, Open  ]],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"Numerical Calculus Package - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 21, 4, 18.6558336}", "context" -> "", 
    "keywords" -> {}, "index" -> True, "label" -> 
    "Numerical Calculus Package Tutorial", "language" -> "en", "paclet" -> 
    "NumericalCalculus Package", "status" -> "None", "summary" -> 
    "The functions defined in the NumericalCalculus` context provide support \
for finding numerical solutions to calculus-related problems. This loads the \
package: The built-in function Limit computes limits using symbolic and \
analytic methods. The function NLimit contained in the NumericalCalculus \
package works by numerically evaluating a short sequence of function values \
as the argument approaches the specified point. The result of this \
calculation is passed to a routine that uses either Wynn's \
\\[Epsilon]-algorithm or a generalized Euler transformation to find an \
approximation to the limit. ", "synonyms" -> {}, "title" -> 
    "Numerical Calculus Package", "type" -> "Tutorial", "uri" -> 
    "NumericalCalculus/tutorial/NumericalCalculus"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->Notebook[{
   Cell[
    StyleData[
    StyleDefinitions -> FrontEnd`FileName[{"Wolfram"}, "Reference.nb"]]], 
   Cell[
    StyleData["Input"], CellContext -> "Global`"], 
   Cell[
    StyleData["Output"], CellContext -> "Global`"]}, Visible -> False, 
  FrontEndVersion -> "6.0 for Microsoft Windows (32-bit) (April 17, 2007)", 
  StyleDefinitions -> "Default.nb"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 29, 0, 8, "TutorialColorBar"],
Cell[600, 23, 1631, 32, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2256, 59, 63, 1, 70, "Title",
 CellID->509267359],
Cell[2322, 62, 229, 6, 70, "Text",
 CellID->1534169418],
Cell[2554, 70, 67, 1, 70, "MathCaption",
 CellID->1825449234],
Cell[2624, 73, 106, 3, 70, "Input",
 CellID->1373894796],
Cell[CellGroupData[{
Cell[2755, 80, 65, 1, 70, "Section",
 CellID->2591],
Cell[2823, 83, 995, 26, 70, "Text",
 CellID->16706],
Cell[3821, 111, 712, 23, 70, "DefinitionBox",
 CellID->19362],
Cell[4536, 136, 63, 1, 70, "Caption",
 CellID->19031],
Cell[4602, 139, 84, 1, 70, "MathCaption",
 CellID->6421],
Cell[CellGroupData[{
Cell[4711, 144, 257, 9, 70, "Input",
 CellID->27111],
Cell[4971, 155, 175, 5, 36, "Output",
 CellID->191037609]
}, Open  ]],
Cell[5161, 163, 114, 3, 70, "MathCaption",
 CellID->7348],
Cell[CellGroupData[{
Cell[5300, 170, 101, 3, 70, "Input",
 CellID->3464],
Cell[5404, 175, 176, 5, 36, "Output",
 CellID->498198997]
}, Open  ]],
Cell[5595, 183, 199, 7, 70, "MathCaption",
 CellID->25569],
Cell[CellGroupData[{
Cell[5819, 194, 406, 14, 70, "Input",
 CellID->2989],
Cell[6228, 210, 176, 5, 36, "Output",
 CellID->654426462]
}, Open  ]],
Cell[6419, 218, 1470, 40, 70, "DefinitionBox3Col",
 CellID->2017],
Cell[7892, 260, 201, 8, 70, "Caption",
 CellID->21763],
Cell[8096, 270, 474, 11, 70, "Text",
 CellID->17753],
Cell[8573, 283, 1021, 34, 70, "Text",
 CellID->27158],
Cell[9597, 319, 876, 26, 70, "DefinitionBox",
 CellID->25550],
Cell[10476, 347, 118, 3, 70, "Caption",
 CellID->9037],
Cell[10597, 352, 1316, 35, 70, "DefinitionBox3Col",
 CellID->29072],
Cell[11916, 389, 205, 8, 70, "Caption",
 CellID->26739],
Cell[12124, 399, 282, 9, 70, "MathCaption",
 CellID->1781],
Cell[CellGroupData[{
Cell[12431, 412, 339, 12, 70, "Input",
 CellID->26975],
Cell[12773, 426, 176, 5, 36, "Output",
 CellID->705840830]
}, Open  ]],
Cell[12964, 434, 234, 5, 70, "MathCaption",
 CellID->19289],
Cell[CellGroupData[{
Cell[13223, 443, 489, 15, 70, "Input",
 CellID->7358],
Cell[13715, 460, 238, 6, 36, "Output",
 CellID->230069580]
}, Open  ]],
Cell[13968, 469, 113, 3, 70, "MathCaption",
 CellID->27254],
Cell[CellGroupData[{
Cell[14106, 476, 168, 6, 70, "Input",
 CellID->23509],
Cell[14277, 484, 233, 7, 36, "Output",
 CellID->211871406]
}, Open  ]],
Cell[14525, 494, 526, 15, 70, "Text",
 CellID->26307],
Cell[15054, 511, 1311, 43, 70, "DefinitionBox",
 CellID->6442],
Cell[16368, 556, 73, 1, 70, "Caption",
 CellID->27758],
Cell[16444, 559, 420, 16, 70, "MathCaption",
 CellID->6249],
Cell[CellGroupData[{
Cell[16889, 579, 198, 6, 70, "Input",
 CellID->13196],
Cell[17090, 587, 193, 6, 36, "Output",
 CellID->143307490]
}, Open  ]],
Cell[17298, 596, 357, 9, 70, "MathCaption",
 CellID->29449],
Cell[CellGroupData[{
Cell[17680, 609, 611, 19, 70, "Input",
 CellID->23083],
Cell[18294, 630, 179, 5, 36, "Output",
 CellID->187596744]
}, Open  ]],
Cell[18488, 638, 145, 4, 70, "MathCaption",
 CellID->15204],
Cell[CellGroupData[{
Cell[18658, 646, 621, 19, 70, "Input",
 CellID->31242],
Cell[19282, 667, 280, 8, 36, "Output",
 CellID->556571991]
}, Open  ]],
Cell[19577, 678, 1609, 43, 70, "DefinitionBox3Col",
 CellID->5317],
Cell[21189, 723, 193, 8, 70, "Caption",
 CellID->32656],
Cell[21385, 733, 690, 20, 70, "Text",
 CellID->11452]
}, Open  ]],
Cell[CellGroupData[{
Cell[22112, 758, 68, 1, 70, "Section",
 CellID->30767],
Cell[22183, 761, 395, 13, 70, "Text",
 CellID->3951],
Cell[22581, 776, 716, 17, 70, "Text",
 CellID->16117],
Cell[23300, 795, 839, 28, 70, "DefinitionBox",
 CellID->8707],
Cell[24142, 825, 69, 1, 70, "Caption",
 CellID->28587],
Cell[24214, 828, 1313, 41, 70, "DefinitionBox",
 CellID->32556],
Cell[25530, 871, 205, 8, 70, "Caption",
 CellID->4207],
Cell[25738, 881, 207, 8, 70, "MathCaption",
 CellID->16622],
Cell[CellGroupData[{
Cell[25970, 893, 194, 7, 70, "Input",
 CellID->16390],
Cell[26167, 902, 158, 5, 36, "Output",
 CellID->7936742]
}, Open  ]],
Cell[26340, 910, 89, 1, 70, "MathCaption",
 CellID->22729],
Cell[CellGroupData[{
Cell[26454, 915, 443, 14, 70, "Input",
 CellID->27477],
Cell[26900, 931, 734, 23, 39, "Output",
 CellID->264026320]
}, Open  ]],
Cell[27649, 957, 295, 10, 70, "MathCaption",
 CellID->20455],
Cell[CellGroupData[{
Cell[27969, 971, 173, 6, 70, "Input",
 CellID->20334],
Cell[28145, 979, 155, 5, 36, "Output",
 CellID->998948]
}, Open  ]],
Cell[28315, 987, 69, 1, 70, "MathCaption",
 CellID->23883],
Cell[CellGroupData[{
Cell[28409, 992, 174, 6, 70, "Input",
 CellID->19662],
Cell[28586, 1000, 277, 7, 36, "Output",
 CellID->174038953]
}, Open  ]],
Cell[28878, 1010, 80, 1, 70, "MathCaption",
 CellID->13284],
Cell[CellGroupData[{
Cell[28983, 1015, 393, 13, 70, "Input",
 CellID->29804],
Cell[29379, 1030, 256, 7, 36, "Output",
 CellID->58319039]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[29684, 1043, 77, 1, 70, "Section",
 CellID->32197],
Cell[29764, 1046, 1035, 29, 70, "Text",
 CellID->19192],
Cell[30802, 1077, 1376, 46, 70, "DefinitionBox",
 CellID->19045],
Cell[32181, 1125, 73, 1, 70, "Caption",
 CellID->7357],
Cell[32257, 1128, 1201, 28, 70, "Text",
 CellID->25290],
Cell[33461, 1158, 808, 21, 70, "DefinitionBox3Col",
 CellID->3225],
Cell[34272, 1181, 203, 8, 70, "Caption",
 CellID->16825],
Cell[34478, 1191, 484, 14, 70, "MathCaption",
 CellID->26053],
Cell[CellGroupData[{
Cell[34987, 1209, 220, 7, 70, "Input",
 CellID->19391],
Cell[35210, 1218, 2015, 53, 107, "Output",
 CellID->342387]
}, Open  ]],
Cell[37240, 1274, 332, 12, 70, "MathCaption",
 CellID->27664],
Cell[CellGroupData[{
Cell[37597, 1290, 133, 4, 70, "Input",
 CellID->6364],
Cell[37733, 1296, 673, 22, 55, "Output",
 CellID->48540072]
}, Open  ]],
Cell[38421, 1321, 100, 2, 70, "MathCaption",
 CellID->6827],
Cell[CellGroupData[{
Cell[38546, 1327, 370, 11, 70, "Input",
 CellID->17078],
Cell[38919, 1340, 760, 24, 54, "Output",
 CellID->688756]
}, Open  ]],
Cell[39694, 1367, 242, 8, 70, "MathCaption",
 CellID->6182],
Cell[CellGroupData[{
Cell[39961, 1379, 428, 12, 70, "Input",
 CellID->25862],
Cell[40392, 1393, 727, 26, 54, "Output",
 CellID->210061539]
}, Open  ]],
Cell[41134, 1422, 156, 4, 70, "MathCaption",
 CellID->20963],
Cell[CellGroupData[{
Cell[41315, 1430, 345, 10, 70, "Input",
 CellID->31555],
Cell[41663, 1442, 843, 28, 55, "Output",
 CellID->111197973]
}, Open  ]],
Cell[42521, 1473, 213, 7, 70, "MathCaption",
 CellID->23438],
Cell[CellGroupData[{
Cell[42759, 1484, 248, 8, 70, "Input",
 CellID->14136],
Cell[43010, 1494, 1279, 39, 68, "Output",
 CellID->634463235]
}, Open  ]]
}, Open  ]]
}, Open  ]],
Cell[44328, 1538, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

