(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     30828,        971]
NotebookOptionsPosition[     25923,        806]
NotebookOutlinePosition[     27252,        843]
CellTagsIndexPosition[     27167,        838]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Special Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpecialFunctions"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Special Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"AiryAi" :> Documentation`HelpLookup["paclet:ref/AiryAi"], 
          "AiryBiZero" :> Documentation`HelpLookup["paclet:ref/AiryBiZero"], 
          "BesselJZero" :> Documentation`HelpLookup["paclet:ref/BesselJZero"],
           "FindRoot" :> Documentation`HelpLookup["paclet:ref/FindRoot"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"AiryAi\"\>", 
       2->"\<\"AiryBiZero\"\>", 3->"\<\"BesselJZero\"\>", 
       4->"\<\"FindRoot\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Bessel-Related Functions" :> 
          Documentation`HelpLookup["paclet:guide/BesselRelatedFunctions"], 
          "Functions Used in Quantum Mechanics" :> 
          Documentation`HelpLookup[
           "paclet:guide/FunctionsUsedInQuantumMechanics"], 
          "Inverse Functions" :> 
          Documentation`HelpLookup["paclet:guide/InverseFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Bessel-Related Functions\"\>", 
       2->"\<\"Functions Used in Quantum Mechanics\"\>", 
       3->"\<\"Inverse Functions\"\>", 
       4->"\<\"New in 6.0: Mathematical Functions\"\>", 
       5->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["AiryAiZero", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["AiryAiZero",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/AiryAiZero"], "[", 
       StyleBox["k", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]represents the ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     Cell[BoxData[
      FormBox[
       SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
     " zero of the Airy function ",
     Cell[BoxData[
      FormBox[
       RowBox[{"Ai", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["AiryAiZero",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/AiryAiZero"], "[", 
       RowBox[{
        StyleBox["k", "TI"], ",", 
        SubscriptBox[
         StyleBox["x", "TI"], 
         StyleBox["0", "TR"]]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]represents the ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     Cell[BoxData[
      FormBox[
       SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
     " zero less than ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["0", "TR"]]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["N",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/N"], "[", 
   RowBox[{"AiryAiZero", "[", 
    StyleBox["k", "TI"], "]"}], "]"}]], "InlineFormula"],
 " gives a numerical approximation so long as the specified zero exists."
}], "Notes",
 CellID->43381924],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["AiryAiZero",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AiryAiZero"], "[", 
   StyleBox["k", "TI"], "]"}]], "InlineFormula"],
 " represents the ",
 Cell[BoxData[
  StyleBox["k", "TI"]], "InlineFormula"],
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
 " zero less than 0."
}], "Notes",
 CellID->1067943069],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["AiryAiZero",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AiryAiZero"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["AiryAiZero",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AiryAiZero"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->79237617],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->118228550],

Cell["Evaluate numerically:", "ExampleText",
 CellID->24457934],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{"AiryAiZero", "[", "1", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->846423094],

Cell[BoxData[
 RowBox[{"-", "2.338107410459767`"}]], "Output",
 ImageSize->{61, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->446589090]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->107280062],

Cell["Evaluate symbolically:", "ExampleText",
 CellID->94628078],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AiryAi", "[", 
  RowBox[{"AiryAiZero", "[", "1", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->16966966],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->336655762]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->4879934],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"AiryAiZero", "[", "10", "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->210060541],

Cell[BoxData[
 RowBox[{"-", 
  "12.8287767528657572004067294072418244773864155995734199414869668243688`50.\
"}]], "Output",
 ImageSize->{369, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->185418562]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->81909017],

Cell[TextData[{
 "Find the first zero of ",
 Cell[BoxData[
  FormBox[
   RowBox[{"Ai", "(", "x", ")"}], TraditionalForm]], "InlineMath"],
 " less than -1000:"
}], "ExampleText",
 CellID->25126027],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{"AiryAiZero", "[", 
   RowBox[{"1", ",", 
    RowBox[{"-", "1000"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->726288814],

Cell[BoxData[
 RowBox[{"-", "1000.0187153968427`"}]], "Output",
 ImageSize->{61, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->127325059]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->219294962],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["AiryAiZero",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AiryAiZero"]], "InlineFormula"],
 " ",
 "threads element-wise over lists:"
}], "ExampleText",
 CellID->65490277],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"AiryAiZero", "[", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "3"}], "}"}], "]"}], "//", "N"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->298902341],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "2.338107410459767`"}], ",", 
   RowBox[{"-", "4.087949444130971`"}], ",", 
   RowBox[{"-", "5.520559828095551`"}]}], "}"}]], "Output",
 ImageSize->{211, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->354515689]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->465855712],

Cell[TextData[{
 "Asymptotic behavior of ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["AiryAiZero",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AiryAiZero"], "[", 
   StyleBox["k", "TI"], "]"}]], "InlineFormula"],
 " for large ",
 Cell[BoxData[
  StyleBox["k", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->14555186],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"AiryAiZero", "[", "k", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "\[Infinity]", ",", "5"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->125185909],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", 
   FractionBox[
    RowBox[{"77125", " ", 
     SuperscriptBox[
      RowBox[{"(", 
       FractionBox["1", 
        RowBox[{
         RowBox[{"-", 
          FractionBox["1", "4"]}], "+", "k"}]], ")"}], 
      RowBox[{"16", "/", "3"}]]}], 
    RowBox[{"314928", " ", 
     SuperscriptBox["2", 
      RowBox[{"2", "/", "3"}]], " ", 
     SuperscriptBox["3", 
      RowBox[{"1", "/", "3"}]], " ", 
     SuperscriptBox["\[Pi]", 
      RowBox[{"16", "/", "3"}]]}]]}], "+", 
  FractionBox[
   RowBox[{"10", " ", 
    SuperscriptBox[
     RowBox[{"(", 
      FractionBox["2", "3"], ")"}], 
     RowBox[{"1", "/", "3"}]], " ", 
    SuperscriptBox[
     RowBox[{"(", 
      FractionBox["1", 
       RowBox[{
        RowBox[{"-", 
         FractionBox["1", "4"]}], "+", "k"}]], ")"}], 
     RowBox[{"10", "/", "3"}]]}], 
   RowBox[{"243", " ", 
    SuperscriptBox["\[Pi]", 
     RowBox[{"10", "/", "3"}]]}]], "-", 
  FractionBox[
   RowBox[{"5", " ", 
    SuperscriptBox[
     RowBox[{"(", 
      FractionBox["1", 
       RowBox[{
        RowBox[{"-", 
         FractionBox["1", "4"]}], "+", "k"}]], ")"}], 
     RowBox[{"4", "/", "3"}]]}], 
   RowBox[{"36", " ", 
    SuperscriptBox["2", 
     RowBox[{"2", "/", "3"}]], " ", 
    SuperscriptBox["3", 
     RowBox[{"1", "/", "3"}]], " ", 
    SuperscriptBox["\[Pi]", 
     RowBox[{"4", "/", "3"}]]}]], "-", 
  FractionBox[
   SuperscriptBox[
    RowBox[{"(", 
     FractionBox[
      RowBox[{"3", " ", "\[Pi]"}], "2"], ")"}], 
    RowBox[{"2", "/", "3"}]], 
   SuperscriptBox[
    RowBox[{"(", 
     FractionBox["1", 
      RowBox[{
       RowBox[{"-", 
        FractionBox["1", "4"]}], "+", "k"}]], ")"}], 
    RowBox[{"2", "/", "3"}]]]}]], "Output",
 ImageSize->{465, 81},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->8866833]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell["Display zeros on the plot:", "ExampleText",
 CellID->32898500],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"AiryAi", "[", "z", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"z", ",", 
     RowBox[{"-", "20"}], ",", "5"}], "}"}], ",", 
   RowBox[{"Prolog", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"PointSize", "[", "0.03", "]"}], ",", 
      RowBox[{"Point", "[", 
       RowBox[{"Table", "[", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{
           RowBox[{"AiryAiZero", "[", "k", "]"}], ",", "0"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"k", ",", "19"}], "}"}]}], "]"}], "]"}]}], "}"}]}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->184083053],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 110},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->211156527]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["AiryAi",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AiryAi"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["AiryBiZero",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AiryBiZero"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BesselJZero",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BesselJZero"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FindRoot",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FindRoot"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpecialFunctions"]], "Tutorials",
 CellID->23022]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Bessel-Related Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/BesselRelatedFunctions"]], "MoreAbout",
 CellID->341409251],

Cell[TextData[ButtonBox["Functions Used in Quantum Mechanics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/FunctionsUsedInQuantumMechanics"]], "MoreAbout",
 CellID->127332795],

Cell[TextData[ButtonBox["Inverse Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/InverseFunctions"]], "MoreAbout",
 CellID->346358290],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->114397374],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->5799687]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"AiryAiZero - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 23, 48.2202084}", "context" -> "System`", 
    "keywords" -> {"Airy Ai zero", "zero of Airy function"}, "index" -> True, 
    "label" -> "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "AiryAiZero[k] represents the k\\[Null]^th zero of the Airy function \
Ai(x). AiryAiZero[k, x_0] represents the k\\[Null]^th zero less than x_0.", 
    "synonyms" -> {"airy ai zero"}, "title" -> "AiryAiZero", "type" -> 
    "Symbol", "uri" -> "ref/AiryAiZero"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[8297, 257, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->79237617]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 27025, 831}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2791, 64, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3414, 91, 1575, 37, 70, "ObjectNameGrid"],
Cell[4992, 130, 1473, 49, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[6502, 184, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6980, 201, 120, 4, 70, "Notes",
 CellID->20640],
Cell[7103, 207, 316, 10, 70, "Notes",
 CellID->43381924],
Cell[7422, 219, 405, 15, 70, "Notes",
 CellID->1067943069],
Cell[7830, 236, 222, 7, 70, "Notes",
 CellID->31226],
Cell[8055, 245, 205, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[8297, 257, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->79237617],
Cell[CellGroupData[{
Cell[8681, 271, 148, 5, 70, "ExampleSection",
 CellID->118228550],
Cell[8832, 278, 63, 1, 70, "ExampleText",
 CellID->24457934],
Cell[CellGroupData[{
Cell[8920, 283, 134, 4, 28, "Input",
 CellID->846423094],
Cell[9057, 289, 192, 6, 36, "Output",
 CellID->446589090]
}, Open  ]],
Cell[9264, 298, 125, 3, 70, "ExampleDelimiter",
 CellID->107280062],
Cell[9392, 303, 64, 1, 70, "ExampleText",
 CellID->94628078],
Cell[CellGroupData[{
Cell[9481, 308, 138, 4, 70, "Input",
 CellID->16966966],
Cell[9622, 314, 158, 5, 36, "Output",
 CellID->336655762]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[9829, 325, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[10053, 334, 68, 1, 70, "ExampleText",
 CellID->4879934],
Cell[CellGroupData[{
Cell[10146, 339, 160, 5, 70, "Input",
 CellID->210060541],
Cell[10309, 346, 254, 8, 36, "Output",
 CellID->185418562]
}, Open  ]],
Cell[10578, 357, 124, 3, 70, "ExampleDelimiter",
 CellID->81909017],
Cell[10705, 362, 196, 7, 70, "ExampleText",
 CellID->25126027],
Cell[CellGroupData[{
Cell[10926, 373, 181, 6, 70, "Input",
 CellID->726288814],
Cell[11110, 381, 193, 6, 36, "Output",
 CellID->127325059]
}, Open  ]],
Cell[11318, 390, 125, 3, 70, "ExampleDelimiter",
 CellID->219294962],
Cell[11446, 395, 218, 8, 70, "ExampleText",
 CellID->65490277],
Cell[CellGroupData[{
Cell[11689, 407, 189, 6, 70, "Input",
 CellID->298902341],
Cell[11881, 415, 322, 10, 36, "Output",
 CellID->354515689]
}, Open  ]],
Cell[12218, 428, 125, 3, 70, "ExampleDelimiter",
 CellID->465855712],
Cell[12346, 433, 333, 13, 70, "ExampleText",
 CellID->14555186],
Cell[CellGroupData[{
Cell[12704, 450, 232, 7, 70, "Input",
 CellID->125185909],
Cell[12939, 459, 1862, 68, 102, "Output",
 CellID->8866833]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[14850, 533, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[15080, 542, 68, 1, 70, "ExampleText",
 CellID->32898500],
Cell[CellGroupData[{
Cell[15173, 547, 651, 21, 70, "Input",
 CellID->184083053],
Cell[15827, 570, 7008, 119, 70, "Output",
 Evaluatable->False,
 CellID->211156527]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[22896, 696, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[23215, 707, 951, 35, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[24203, 747, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[24524, 758, 144, 3, 70, "Tutorials",
 CellID->23022]
}, Open  ]],
Cell[CellGroupData[{
Cell[24705, 766, 299, 8, 70, "MoreAboutSection"],
Cell[25007, 776, 158, 3, 70, "MoreAbout",
 CellID->341409251],
Cell[25168, 781, 178, 3, 70, "MoreAbout",
 CellID->127332795],
Cell[25349, 786, 145, 3, 70, "MoreAbout",
 CellID->346358290],
Cell[25497, 791, 174, 3, 70, "MoreAbout",
 CellID->114397374],
Cell[25674, 796, 177, 3, 70, "MoreAbout",
 CellID->5799687]
}, Open  ]],
Cell[25866, 802, 27, 0, 70, "History"],
Cell[25896, 804, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

