(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     36038,       1305]
NotebookOptionsPosition[     28548,       1043]
NotebookOutlinePosition[     29904,       1081]
CellTagsIndexPosition[     29818,       1076]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Algebraic Number Fields" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/AlgebraicNumberFields"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Algebraic Number Fields\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"AlgebraicNumberTrace" :> 
          Documentation`HelpLookup["paclet:ref/AlgebraicNumberTrace"], 
          "MinimalPolynomial" :> 
          Documentation`HelpLookup["paclet:ref/MinimalPolynomial"], 
          "NumberFieldNormRepresentatives" :> 
          Documentation`HelpLookup[
           "paclet:ref/NumberFieldNormRepresentatives"], 
          "NumberFieldClassNumber" :> 
          Documentation`HelpLookup["paclet:ref/NumberFieldClassNumber"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"AlgebraicNumberTrace\"\>", 
       2->"\<\"MinimalPolynomial\"\>", 
       3->"\<\"NumberFieldNormRepresentatives\"\>", 
       4->"\<\"NumberFieldClassNumber\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Algebraic Number Theory" :> 
          Documentation`HelpLookup["paclet:guide/AlgebraicNumberTheory"], 
          "Number Theoretic Functions" :> 
          Documentation`HelpLookup["paclet:guide/NumberTheoreticFunctions"], 
          "New in 6.0: Symbolic Computation" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60AlgebraicComputing"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"], 
          "New in 6.0: Number Theory & Integer Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Algebraic Number Theory\"\>", 
       2->"\<\"Number Theoretic Functions\"\>", 
       3->"\<\"New in 6.0: Symbolic Computation\"\>", 
       4->"\<\"New in 6.0: Mathematics & Algorithms\"\>", 
       5->"\<\"New in 6.0: Number Theory & Integer Functions\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["AlgebraicNumberNorm", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["AlgebraicNumberNorm",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/AlgebraicNumberNorm"], "[", 
       StyleBox["a", "TI"], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the norm of the algebraic number ",
     Cell[BoxData[
      StyleBox["a", "TI"]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "The norm of ",
 Cell[BoxData[
  StyleBox["a", "TI"]], "InlineFormula"],
 " is defined to be the product of the roots of its minimal polynomial."
}], "Notes",
 CellID->4932284],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["AlgebraicNumberNorm",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AlgebraicNumberNorm"], "[", 
   RowBox[{
    StyleBox["a", "TI"], ",", 
    RowBox[{
     ButtonBox["Extension",
      BaseStyle->"Link",
      ButtonData->"paclet:ref/Extension"], "->", 
     StyleBox["\[Theta]", "TR"]}]}], "]"}]], "InlineFormula"],
 " finds the norm of ",
 Cell[BoxData[
  StyleBox["a", "TI"]], "InlineFormula"],
 " over the field ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[DoubleStruckCapitalQ]", "[", "\[Theta]", "]"}], 
   TraditionalForm]], "InlineMath"],
 "."
}], "Notes",
 CellID->251581752]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->482340736],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->57022428],

Cell["Norms of algebraic numbers:", "ExampleText",
 CellID->28913612],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", "[", 
  RowBox[{"Sqrt", "[", "2", "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->389984823],

Cell[BoxData[
 RowBox[{"-", "2"}]], "Output",
 ImageSize->{19, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->99371058]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", "[", 
  RowBox[{
   RowBox[{"E", "^", 
    RowBox[{"(", 
     RowBox[{"2", "Pi", " ", "I"}], ")"}]}], "/", "3"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->77381826],

Cell[BoxData[
 FractionBox["1", "3"]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{16, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->24351645]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", "[", "GoldenRatio", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->1894405153],

Cell[BoxData[
 RowBox[{"-", "1"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{19, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->859922084]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(4)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->172245298],

Cell["Integers and rational numbers:", "ExampleText",
 CellID->1216392566],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", "[", "2", "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2013020969],

Cell[BoxData["2"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->162983737]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", "[", 
  RowBox[{
   RowBox[{"-", "2"}], "/", "3"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1606475006],

Cell[BoxData[
 RowBox[{"-", 
  FractionBox["2", "3"]}]], "Output",
 ImageSize->{23, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1226017834]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->2140009798],

Cell["Radical expressions:", "ExampleText",
 CellID->1088555030],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", "[", 
  RowBox[{"1", "/", 
   RowBox[{"Sqrt", "[", 
    RowBox[{
     RowBox[{"Sqrt", "[", "2", "]"}], "+", "3"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->931200417],

Cell[BoxData[
 FractionBox["1", "7"]], "Output",
 ImageSize->{16, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->576143073]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->247893264],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["Root",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Root"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["AlgebraicNumber",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumber"]], "InlineFormula"],
 " objects:"
}], "ExampleText",
 CellID->223812868],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", "[", 
  RowBox[{"Root", "[", 
   RowBox[{
    RowBox[{
     RowBox[{
      RowBox[{"-", "1"}], "+", "#1", "+", 
      RowBox[{"#1", "^", "2"}], "+", 
      RowBox[{"#1", "^", "3"}], "+", 
      RowBox[{"#1", "^", "4"}]}], " ", "&"}], " ", ",", "1"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1379006579],

Cell[BoxData[
 RowBox[{"-", "1"}]], "Output",
 ImageSize->{19, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->232863588]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", "[", 
  RowBox[{"AlgebraicNumber", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"Sqrt", "[", "2", "]"}], " ", "I"}], ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "2"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1002952795],

Cell[BoxData["9"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1023118682]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1805255463],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["AlgebraicNumberNorm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumberNorm"]], "InlineFormula"],
 " automatically threads over lists:"
}], "ExampleText",
 CellID->760339003],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"2", 
     RowBox[{"Sqrt", "[", "2", "]"}]}], ",", " ", 
    RowBox[{"E", "^", 
     RowBox[{"(", 
      RowBox[{"Pi", "*", 
       RowBox[{"I", "/", "8"}]}], ")"}]}], ",", 
    RowBox[{"1", "+", "I"}]}], "}"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->404604487],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"-", "8"}], ",", "1", ",", "2"}], "}"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{70, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->654126845]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Options",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->14825509],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Extension",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->38080918],

Cell[TextData[{
 "Norm of ",
 Cell[BoxData[
  FormBox[
   SqrtBox["2"], TraditionalForm]], "InlineMath"],
 " over ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[DoubleStruckCapitalQ]", "[", 
    SuperscriptBox["\[ExponentialE]", 
     FractionBox["\[Pi]\[ImaginaryI]", "4"]], "]"}], TraditionalForm]], 
  "InlineMath"],
 ":"
}], "ExampleText",
 CellID->741064791],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", "[", 
  RowBox[{
   RowBox[{"Sqrt", "[", "2", "]"}], ",", " ", 
   RowBox[{"Extension", " ", "\[Rule]", " ", 
    RowBox[{"E", "^", 
     RowBox[{"(", 
      RowBox[{"Pi", " ", 
       RowBox[{"I", "/", "4"}]}], ")"}]}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->322502872],

Cell[BoxData["4"], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->941094922]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{"9", "+", 
    SqrtBox["10"]}], TraditionalForm]], "InlineMath"],
 "is irreducible in ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[DoubleStruckCapitalZ]", "[", 
    SqrtBox["10"], "]"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->1150475409],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", "[", 
  RowBox[{"9", "+", 
   RowBox[{"Sqrt", "[", "10", "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1177890050],

Cell[BoxData["71"], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->765293899]
}, Open  ]],

Cell[TextData[{
 "Since ",
 Cell[BoxData[
  ButtonBox["AlgebraicNumberNorm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumberNorm"]], "InlineFormula"],
 " is multiplicative, having a prime norm implies the original number is \
prime:"
}], "ExampleText",
 CellID->107560674],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PrimeQ", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1847380980],

Cell[BoxData["True"], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{32, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1014998610]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->220216355],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["AlgebraicNumberNorm",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/AlgebraicNumberNorm"]], "InlineFormula"],
 " is multiplicative:"
}], "ExampleText",
 CellID->746832672],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"2", ",", " ", 
     RowBox[{"Sqrt", "[", "5", "]"}]}], "}"}], ",", 
   RowBox[{"Extension", " ", "\[Rule]", " ", 
    RowBox[{"Sqrt", "[", "5", "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1777904245],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"4", ",", 
   RowBox[{"-", "5"}]}], "}"}]], "Output",
 GeneratedCell->True,
 CellAutoOverwrite->True,
 ImageSize->{51, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1619030149]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", "[", 
  RowBox[{
   RowBox[{"2", "  ", 
    RowBox[{"Sqrt", "[", "5", "]"}]}], ",", 
   RowBox[{"Extension", " ", "\[Rule]", " ", 
    RowBox[{"Sqrt", "[", "5", "]"}]}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->967435411],

Cell[BoxData[
 RowBox[{"-", "20"}]], "Output",
 ImageSize->{26, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->274423106]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->16060344],

Cell[TextData[{
 "Units in a number field have norm ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[PlusMinus]", "1"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->141061500],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", " ", "/@", " ", 
  RowBox[{"NumberFieldFundamentalUnits", "[", 
   RowBox[{
    RowBox[{"Sqrt", "[", "2", "]"}], "+", 
    RowBox[{"Sqrt", "[", "3", "]"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2146167859],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", 
   RowBox[{"-", "1"}], ",", "1"}], "}"}]], "Output",
 ImageSize->{70, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->306453705]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", " ", "/@", " ", 
  RowBox[{"NumberFieldFundamentalUnits", "[", 
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"16", "-", 
       RowBox[{"20", " ", 
        SuperscriptBox["#1", "2"]}], "+", 
       SuperscriptBox["#1", "4"]}], "&"}], ",", "4"}], "]"}], 
   "]"}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1254816875],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", "1", ",", "1"}], "}"}]], "Output",
 ImageSize->{62, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->880098572]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->268694455],

Cell[TextData[{
 "Compute the smallest field that includes ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SqrtBox["3"], "+", 
    SqrtBox[
     RowBox[{"-", "5"}]]}], TraditionalForm]], "InlineMath"],
 ", i.e. ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[DoubleStruckCapitalQ]", "[", 
    RowBox[{
     SqrtBox["3"], ",", 
     SqrtBox[
      RowBox[{"-", "5"}]]}], "]"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->326261594],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"e", "=", 
  RowBox[{
   RowBox[{"ToNumberField", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"Sqrt", "[", "3", "]"}], ",", 
       RowBox[{"Sqrt", "[", 
        RowBox[{"-", "5"}], "]"}]}], "}"}], ",", "All"}], "]"}], "[", 
   RowBox[{"[", 
    RowBox[{"1", ",", "1"}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->96374045],

Cell[BoxData[
 RowBox[{"Root", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"64", "+", 
     RowBox[{"4", " ", 
      SuperscriptBox["#1", "2"]}], "+", 
     SuperscriptBox["#1", "4"]}], "&"}], ",", "4"}], "]"}]], "Output",
 ImageSize->{163, 18},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->19594863]
}, Open  ]],

Cell["Compute the norm in that field:", "ExampleText",
 CellID->556366196],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"AlgebraicNumberNorm", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"Sqrt", "[", "3", "]"}], "+", 
    RowBox[{"Sqrt", "[", 
     RowBox[{"-", "5"}], "]"}]}], ",", 
   RowBox[{"Extension", "\[Rule]", "e"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->24571468],

Cell[BoxData["64"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->119247608]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Neat Examples",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->589267740],

Cell[TextData[{
 "Plot of norms of elements in ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[DoubleStruckCapitalQ]", "[", 
    SqrtBox[
     RowBox[{"-", "2"}]], "]"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->927354934],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{"AlgebraicNumberNorm", " ", "@", "  ", 
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"AlgebraicNumber", "[", 
      RowBox[{
       RowBox[{"Sqrt", "[", 
        RowBox[{"-", "2"}], "]"}], " ", ",", "p"}], "]"}], ",", 
     RowBox[{"{", 
      RowBox[{"p", ",", 
       RowBox[{"Tuples", "[", 
        RowBox[{
         RowBox[{"Range", "[", 
          RowBox[{
           RowBox[{"-", "10"}], ",", "10"}], "]"}], ",", "2"}], "]"}]}], 
      "}"}]}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->498838206],

Cell[BoxData[
 GraphicsBox[{{}, 
   {Hue[0.67, 0.6, 0.6], PointBox[CompressedData["
1:eJxdmUuyFEUUhlNHDGvAgAGD8oZhGIZhtO+3FOAb0BZQQQSLFlR8QHtRLsKk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     "]]}, {}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->True,
  AxesOrigin->{0, 0},
  PlotRange->{{0., 441.}, {-10., 300.}},
  PlotRangeClipping->True,
  PlotRangePadding->{
    Scaled[0.02], 
    Scaled[0.02]}]], "Output",
 ImageSize->{184, 116},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->27849947]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["AlgebraicNumberTrace",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AlgebraicNumberTrace"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["MinimalPolynomial",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/MinimalPolynomial"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NumberFieldNormRepresentatives",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NumberFieldNormRepresentatives"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NumberFieldClassNumber",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NumberFieldClassNumber"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Algebraic Number Fields",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/AlgebraicNumberFields"]], "Tutorials",
 CellID->341631938]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->416357456],

Cell[TextData[ButtonBox["Algebraic Number Theory",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/AlgebraicNumberTheory"]], "MoreAbout",
 CellID->16511685],

Cell[TextData[ButtonBox["Number Theoretic Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NumberTheoreticFunctions"]], "MoreAbout",
 CellID->340534559],

Cell[TextData[ButtonBox["New in 6.0: Symbolic Computation",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60AlgebraicComputing"]], "MoreAbout",
 CellID->244008812],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->84629967],

Cell[TextData[ButtonBox["New in 6.0: Number Theory & Integer Functions",
 BaseStyle->"Link",
 ButtonData->
  "paclet:guide/NewIn60NumberTheoryAndIntegerFunctions"]], "MoreAbout",
 CellID->113942120]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"AlgebraicNumberNorm - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 23, 55.6578988}", "context" -> "System`", 
    "keywords" -> {
     "absolute value of algebraic number", "algebraic number invariant", 
      "norm of algebraic number"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "AlgebraicNumberNorm[a] gives the norm of the algebraic number a.", 
    "synonyms" -> {"algebraic number norm"}, "title" -> "AlgebraicNumberNorm",
     "type" -> "Symbol", "uri" -> "ref/AlgebraicNumberNorm"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7262, 212, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->482340736]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 29675, 1069}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3076, 70, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3699, 97, 1584, 37, 70, "ObjectNameGrid"],
Cell[5286, 136, 583, 18, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[5906, 159, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6384, 176, 193, 6, 70, "Notes",
 CellID->4932284],
Cell[6580, 184, 645, 23, 70, "Notes",
 CellID->251581752]
}, Closed]],
Cell[CellGroupData[{
Cell[7262, 212, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->482340736],
Cell[CellGroupData[{
Cell[7647, 226, 147, 5, 70, "ExampleSection",
 CellID->57022428],
Cell[7797, 233, 69, 1, 70, "ExampleText",
 CellID->28913612],
Cell[CellGroupData[{
Cell[7891, 238, 146, 4, 28, "Input",
 CellID->389984823],
Cell[8040, 244, 174, 6, 36, "Output",
 CellID->99371058]
}, Open  ]],
Cell[CellGroupData[{
Cell[8251, 255, 218, 7, 28, "Input",
 CellID->77381826],
Cell[8472, 264, 225, 8, 51, "Output",
 CellID->24351645]
}, Open  ]],
Cell[CellGroupData[{
Cell[8734, 277, 126, 3, 28, "Input",
 CellID->1894405153],
Cell[8863, 282, 223, 8, 36, "Output",
 CellID->859922084]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[9135, 296, 220, 7, 70, "ExampleSection",
 CellID->172245298],
Cell[9358, 305, 74, 1, 70, "ExampleText",
 CellID->1216392566],
Cell[CellGroupData[{
Cell[9457, 310, 116, 3, 70, "Input",
 CellID->2013020969],
Cell[9576, 315, 158, 5, 36, "Output",
 CellID->162983737]
}, Open  ]],
Cell[CellGroupData[{
Cell[9771, 325, 158, 5, 70, "Input",
 CellID->1606475006],
Cell[9932, 332, 197, 7, 51, "Output",
 CellID->1226017834]
}, Open  ]],
Cell[10144, 342, 126, 3, 70, "ExampleDelimiter",
 CellID->2140009798],
Cell[10273, 347, 64, 1, 70, "ExampleText",
 CellID->1088555030],
Cell[CellGroupData[{
Cell[10362, 352, 229, 7, 70, "Input",
 CellID->931200417],
Cell[10594, 361, 178, 6, 51, "Output",
 CellID->576143073]
}, Open  ]],
Cell[10787, 370, 125, 3, 70, "ExampleDelimiter",
 CellID->247893264],
Cell[10915, 375, 321, 12, 70, "ExampleText",
 CellID->223812868],
Cell[CellGroupData[{
Cell[11261, 391, 367, 12, 70, "Input",
 CellID->1379006579],
Cell[11631, 405, 175, 6, 36, "Output",
 CellID->232863588]
}, Open  ]],
Cell[CellGroupData[{
Cell[11843, 416, 292, 9, 70, "Input",
 CellID->1002952795],
Cell[12138, 427, 159, 5, 36, "Output",
 CellID->1023118682]
}, Open  ]],
Cell[12312, 435, 126, 3, 70, "ExampleDelimiter",
 CellID->1805255463],
Cell[12441, 440, 233, 7, 70, "ExampleText",
 CellID->760339003],
Cell[CellGroupData[{
Cell[12699, 451, 363, 12, 70, "Input",
 CellID->404604487],
Cell[13065, 465, 280, 10, 36, "Output",
 CellID->654126845]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[13394, 481, 221, 7, 70, "ExampleSection",
 CellID->14825509],
Cell[CellGroupData[{
Cell[13640, 492, 229, 7, 70, "ExampleSubsection",
 CellID->38080918],
Cell[13872, 501, 359, 14, 70, "ExampleText",
 CellID->741064791],
Cell[CellGroupData[{
Cell[14256, 519, 330, 10, 70, "Input",
 CellID->322502872],
Cell[14589, 531, 206, 7, 36, "Output",
 CellID->941094922]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[14856, 545, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[15086, 554, 316, 12, 70, "ExampleText",
 CellID->1150475409],
Cell[CellGroupData[{
Cell[15427, 570, 172, 5, 70, "Input",
 CellID->1177890050],
Cell[15602, 577, 207, 7, 36, "Output",
 CellID->765293899]
}, Open  ]],
Cell[15824, 587, 289, 9, 70, "ExampleText",
 CellID->107560674],
Cell[CellGroupData[{
Cell[16138, 600, 103, 3, 70, "Input",
 CellID->1847380980],
Cell[16244, 605, 210, 7, 36, "Output",
 CellID->1014998610]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[16503, 618, 237, 7, 70, "ExampleSection",
 CellID->220216355],
Cell[16743, 627, 218, 7, 70, "ExampleText",
 CellID->746832672],
Cell[CellGroupData[{
Cell[16986, 638, 308, 9, 70, "Input",
 CellID->1777904245],
Cell[17297, 649, 271, 10, 36, "Output",
 CellID->1619030149]
}, Open  ]],
Cell[CellGroupData[{
Cell[17605, 664, 277, 8, 70, "Input",
 CellID->967435411],
Cell[17885, 674, 176, 6, 36, "Output",
 CellID->274423106]
}, Open  ]],
Cell[18076, 683, 124, 3, 70, "ExampleDelimiter",
 CellID->16060344],
Cell[18203, 688, 192, 7, 70, "ExampleText",
 CellID->141061500],
Cell[CellGroupData[{
Cell[18420, 699, 266, 7, 70, "Input",
 CellID->2146167859],
Cell[18689, 708, 232, 8, 36, "Output",
 CellID->306453705]
}, Open  ]],
Cell[CellGroupData[{
Cell[18958, 721, 386, 12, 70, "Input",
 CellID->1254816875],
Cell[19347, 735, 213, 7, 36, "Output",
 CellID->880098572]
}, Open  ]],
Cell[19575, 745, 125, 3, 70, "ExampleDelimiter",
 CellID->268694455],
Cell[19703, 750, 446, 18, 70, "ExampleText",
 CellID->326261594],
Cell[CellGroupData[{
Cell[20174, 772, 385, 13, 70, "Input",
 CellID->96374045],
Cell[20562, 787, 349, 12, 39, "Output",
 CellID->19594863]
}, Open  ]],
Cell[20926, 802, 74, 1, 70, "ExampleText",
 CellID->556366196],
Cell[CellGroupData[{
Cell[21025, 807, 286, 9, 70, "Input",
 CellID->24571468],
Cell[21314, 818, 159, 5, 36, "Output",
 CellID->119247608]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[21522, 829, 228, 7, 70, "ExampleSection",
 CellID->589267740],
Cell[21753, 838, 243, 9, 70, "ExampleText",
 CellID->927354934],
Cell[CellGroupData[{
Cell[22021, 851, 577, 18, 70, "Input",
 CellID->498838206],
Cell[22601, 871, 2686, 53, 137, "Output",
 CellID->27849947]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[25348, 931, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[25667, 942, 1059, 35, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[26763, 982, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[27084, 993, 159, 3, 70, "Tutorials",
 CellID->341631938]
}, Open  ]],
Cell[CellGroupData[{
Cell[27280, 1001, 319, 9, 70, "MoreAboutSection",
 CellID->416357456],
Cell[27602, 1012, 155, 3, 70, "MoreAbout",
 CellID->16511685],
Cell[27760, 1017, 162, 3, 70, "MoreAbout",
 CellID->340534559],
Cell[27925, 1022, 169, 3, 70, "MoreAbout",
 CellID->244008812],
Cell[28097, 1027, 178, 3, 70, "MoreAbout",
 CellID->84629967],
Cell[28278, 1032, 198, 4, 70, "MoreAbout",
 CellID->113942120]
}, Open  ]],
Cell[28491, 1039, 27, 0, 70, "History"],
Cell[28521, 1041, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

