(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     63028,       1692]
NotebookOptionsPosition[     56279,       1461]
NotebookOutlinePosition[     57685,       1500]
CellTagsIndexPosition[     57599,       1495]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Special Functions" :> 
          Documentation`HelpLookup["paclet:tutorial/SpecialFunctions"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Special Functions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"BesselJ" :> Documentation`HelpLookup["paclet:ref/BesselJ"], 
          "BesselYZero" :> Documentation`HelpLookup["paclet:ref/BesselYZero"],
           "AiryAiZero" :> Documentation`HelpLookup["paclet:ref/AiryAiZero"], 
          "FindRoot" :> Documentation`HelpLookup["paclet:ref/FindRoot"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"BesselJ\"\>", 
       2->"\<\"BesselYZero\"\>", 3->"\<\"AiryAiZero\"\>", 
       4->"\<\"FindRoot\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Bessel-Related Functions" :> 
          Documentation`HelpLookup["paclet:guide/BesselRelatedFunctions"], 
          "Functions Used in Optics" :> 
          Documentation`HelpLookup["paclet:guide/FunctionsUsedInOptics"], 
          "Inverse Functions" :> 
          Documentation`HelpLookup["paclet:guide/InverseFunctions"], 
          "Special Functions" :> 
          Documentation`HelpLookup["paclet:guide/SpecialFunctions"], 
          "New in 6.0: Mathematical Functions" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicalFunctions"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Bessel-Related Functions\"\>", 
       2->"\<\"Functions Used in Optics\"\>", 3->"\<\"Inverse Functions\"\>", 
       4->"\<\"Special Functions\"\>", 
       5->"\<\"New in 6.0: Mathematical Functions\"\>", 
       6->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["BesselJZero", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["BesselJZero",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/BesselJZero"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["k", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]represents the ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     Cell[BoxData[
      FormBox[
       SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
     " zero of the Bessel function ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        SubscriptBox["J", "n"], "(", "x", ")"}], TraditionalForm]], 
      "InlineMath"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["BesselJZero",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/BesselJZero"], "[", 
       RowBox[{
        StyleBox["n", "TI"], ",", 
        StyleBox["k", "TI"], ",", 
        SubscriptBox[
         StyleBox["x", "TI"], 
         StyleBox["0", "TR"]]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]represents the ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     Cell[BoxData[
      FormBox[
       SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
     " zero greater than ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["0", "TR"]]], "InlineFormula"],
     "."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->982511436]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell["\<\
Mathematical function, suitable for both symbolic and numerical manipulation.\
\
\>", "Notes",
 CellID->20640],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["N",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/N"], "[", 
   RowBox[{"BesselJZero", "[", 
    RowBox[{
     StyleBox["n", "TI"], ",", 
     StyleBox["k", "TI"]}], "]"}], "]"}]], "InlineFormula"],
 " gives a numerical approximation so long as the specified zero exists."
}], "Notes",
 CellID->43381924],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["BesselJZero",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BesselJZero"], "[", 
   RowBox[{
    StyleBox["n", "TI"], ",", 
    StyleBox["k", "TI"]}], "]"}]], "InlineFormula"],
 " represents the ",
 Cell[BoxData[
  StyleBox["k", "TI"]], "InlineFormula"],
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
 " zero greater than 0."
}], "Notes",
 CellID->1067943069],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BesselJZero",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BesselJZero"]], "InlineFormula"],
 " can be evaluated to arbitrary numerical precision."
}], "Notes",
 CellID->31226],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BesselJZero",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BesselJZero"]], "InlineFormula"],
 " automatically threads over lists."
}], "Notes",
 CellID->28162]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->133756076],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->211303757],

Cell["Evaluate numerically:", "ExampleText",
 CellID->271583785],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{"BesselJZero", "[", 
   RowBox[{"0", ",", "1"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->20770762],

Cell[BoxData["2.404825557695773`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->126301692]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->347055722],

Cell["Evaluate symbolically:", "ExampleText",
 CellID->375713404],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"BesselJ", "[", 
  RowBox[{"0", ",", 
   RowBox[{"BesselJZero", "[", 
    RowBox[{"0", ",", "1"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->139609482],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->368038474]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1293636265],

Cell["Evaluate to high precision:", "ExampleText",
 CellID->4879934],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"BesselJZero", "[", 
    RowBox[{"0", ",", "10"}], "]"}], ",", "50"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->210060541],

Cell[BoxData["30.\
6346064684319751175495789268542327372735716291781471907550221552131`50."], \
"Output",
 ImageSize->{361, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->159930780]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->81909017],

Cell[TextData[{
 "Find the first zero of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    SubscriptBox["J", "0"], "(", "x", ")"}], TraditionalForm]], 
  "InlineMath"],
 " greater than 100:"
}], "ExampleText",
 CellID->25126027],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{"BesselJZero", "[", 
   RowBox[{"0", ",", "1", ",", "100"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->726288814],

Cell[BoxData["102.8883742541948`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->97376225]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->219294962],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BesselJZero",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BesselJZero"]], "InlineFormula"],
 " ",
 "threads element-wise over lists:"
}], "ExampleText",
 CellID->65490277],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"BesselJZero", "[", 
   RowBox[{"0", ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "2", ",", "3"}], "}"}]}], "]"}], "//", "N"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->298902341],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "2.404825557695773`", ",", "5.520078110286311`", ",", 
   "8.653727912911013`"}], "}"}]], "Output",
 ImageSize->{188, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->213699606]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Generalizations & Extensions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->1020263627],

Cell[TextData[{
 "Find a zero of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{
      SubscriptBox["J", "\[Nu]"], "(", "x", ")"}], " ", 
     RowBox[{"cos", "(", "\[Alpha]", ")"}]}], "-", 
    RowBox[{
     RowBox[{
      SubscriptBox["Y", "\[Nu]"], "(", "x", ")"}], " ", 
     RowBox[{"sin", "(", "\[Alpha]", ")"}]}]}], TraditionalForm]], 
  "InlineMath"],
 " using ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["BesselJZero",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BesselJZero"], "[", 
   RowBox[{
    StyleBox["\[Nu]", "TR"], ",", 
    RowBox[{
     StyleBox["k", "TI"], "-", 
     RowBox[{
      StyleBox["\[Alpha]", "TR"], "/", 
      StyleBox["\[Pi]", "TR"]}]}]}], "]"}]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->759007334],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", 
  RowBox[{
   RowBox[{"BesselJZero", "[", 
    RowBox[{"0", ",", 
     RowBox[{"1", "-", 
      RowBox[{"2", "/", "3"}]}]}], "]"}], ",", "20"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->309352248],

Cell[BoxData["0.4321180692235360887779401405446913456`20."], "Output",
 ImageSize->{158, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->31581649]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"BesselJ", "[", 
    RowBox[{"0", ",", "%"}], "]"}], 
   RowBox[{"Cos", "[", 
    RowBox[{"2", 
     RowBox[{"Pi", "/", "3"}]}], "]"}]}], "-", 
  RowBox[{
   RowBox[{"BesselY", "[", 
    RowBox[{"0", ",", "%"}], "]"}], 
   RowBox[{"Sin", "[", 
    RowBox[{"2", 
     RowBox[{"Pi", "/", "3"}]}], "]"}]}]}]], "Input",
 CellLabel->"In[2]:=",
 CellID->585149652],

Cell[BoxData["0``19.389596821303588"], "Output",
 ImageSize->{59, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->298868616]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->736573821],

Cell[TextData[{
 "Find the derivative of Bessel zero with respect to ",
 Cell[BoxData[
  StyleBox["k", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->5396862],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"D", "[", 
  RowBox[{
   RowBox[{"BesselJZero", "[", 
    RowBox[{"\[Nu]", ",", "k"}], "]"}], ",", "k"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->889963302],

Cell[BoxData[
 RowBox[{
  FractionBox["1", "2"], " ", 
  SuperscriptBox["\[Pi]", "2"], " ", 
  RowBox[{"BesselJZero", "[", 
   RowBox[{"\[Nu]", ",", "k"}], "]"}], " ", 
  RowBox[{"(", 
   RowBox[{
    SuperscriptBox[
     RowBox[{"BesselJ", "[", 
      RowBox[{"\[Nu]", ",", 
       RowBox[{"BesselJZero", "[", 
        RowBox[{"\[Nu]", ",", "k"}], "]"}]}], "]"}], "2"], "+", 
    SuperscriptBox[
     RowBox[{"BesselY", "[", 
      RowBox[{"\[Nu]", ",", 
       RowBox[{"BesselJZero", "[", 
        RowBox[{"\[Nu]", ",", "k"}], "]"}]}], "]"}], "2"]}], 
   ")"}]}]], "Output",
 ImageSize->{444, 49},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->188620940]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->258228157],

Cell["\<\
Find the first ten eigenmodes of a circular drum with Dirichlet boundary \
conditions:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->23472583],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"modes", " ", "=", 
  RowBox[{"N", "[", 
   RowBox[{"BesselJZero", "[", 
    RowBox[{"0", ",", 
     RowBox[{"Range", "[", "10", "]"}]}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->559586426],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "2.404825557695773`", ",", "5.520078110286311`", ",", "8.653727912911013`", 
   ",", "11.791534439014281`", ",", "14.930917708487787`", ",", 
   "18.071063967910924`", ",", "21.21163662987926`", ",", 
   "24.352471530749302`", ",", "27.493479132040253`", ",", 
   "30.634606468431976`"}], "}"}]], "Output",
 ImageSize->{368, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->186146956]
}, Open  ]],

Cell["\<\
Construct an amplitude comprising a certain mixture of modes:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->37342946],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"ampl", "[", "r_", "]"}], "=", 
   RowBox[{
    RowBox[{"Sin", "[", "modes", "]"}], ".", 
    RowBox[{"BesselJ", "[", 
     RowBox[{"0", ",", 
      RowBox[{"r", " ", "modes"}]}], "]"}]}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->178525005],

Cell["Circular density plot:", "ExampleText",
 CellID->54566326],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ParametricPlot", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"Cos", "[", "\[Phi]", "]"}], " ", "r"}], ",", 
     RowBox[{
      RowBox[{"Sin", "[", "\[Phi]", "]"}], " ", "r"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\[Phi]", ",", "0", ",", 
     RowBox[{"2", "Pi"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"r", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"ColorFunction", "\[Rule]", 
    RowBox[{"Function", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"x", ",", "y", ",", "\[Phi]", ",", "r"}], "}"}], ",", 
      RowBox[{"GrayLevel", "[", 
       RowBox[{"ampl", "[", "r", "]"}], "]"}]}], "]"}]}], ",", 
   RowBox[{"Mesh", "\[Rule]", "False"}], ",", 
   RowBox[{"Axes", "\[Rule]", "False"}], ",", 
   RowBox[{"BoundaryStyle", "\[Rule]", "None"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->130025248],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 172},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->723083224]
}, Open  ]],

Cell["Radial drum displacement profile:", "ExampleText",
 CellID->139382388],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"ampl", "[", "r", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"r", ",", "0", ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->54809775],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 113},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->183434337]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->474590927],

Cell[TextData[{
 "Find the coefficient in the ",
 ButtonBox["Rayleigh criterion",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://scienceworld.wolfram.com/physics/AiryDisk.html"], None}],
 " for diffraction-limited optics:"
}], "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->215833738],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"BesselJZero", "[", 
    RowBox[{"1", ",", "1"}], "]"}], "/", "\[Pi]"}], " ", "//", "N"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->371271319],

Cell[BoxData["1.2196698912665045`"], "Output",
 ImageSize->{53, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->94621189]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->2123667759],

Cell[TextData[{
 "Asymptotic behavior of ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["BesselJZero",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BesselJZero"], "[", 
   RowBox[{
    StyleBox["\[Nu]", "TR"], ",", 
    StyleBox["k", "TI"]}], "]"}]], "InlineFormula"],
 " for large ",
 Cell[BoxData[
  StyleBox["k", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellID->14555186],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Series", "[", 
  RowBox[{
   RowBox[{"BesselJZero", "[", 
    RowBox[{"\[Nu]", ",", "k"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "Infinity", ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->70795635],

Cell[BoxData[
 RowBox[{
  RowBox[{"\[Pi]", " ", 
   RowBox[{"(", 
    RowBox[{
     RowBox[{"-", 
      FractionBox["1", "4"]}], "+", "k", "+", 
     FractionBox["\[Nu]", "2"]}], ")"}]}], "-", 
  FractionBox[
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "1"}], "+", 
      RowBox[{"2", " ", "\[Nu]"}]}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{"1", "+", 
      RowBox[{"2", " ", "\[Nu]"}]}], ")"}]}], 
   RowBox[{"8", " ", "\[Pi]", " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", 
       FractionBox["1", "4"]}], "+", "k", "+", 
      FractionBox["\[Nu]", "2"]}], ")"}]}]], "-", 
  FractionBox[
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "1"}], "+", 
      RowBox[{"2", " ", "\[Nu]"}]}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{"1", "+", 
      RowBox[{"2", " ", "\[Nu]"}]}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "31"}], "+", 
      RowBox[{"28", " ", 
       SuperscriptBox["\[Nu]", "2"]}]}], ")"}]}], 
   RowBox[{"384", " ", 
    SuperscriptBox["\[Pi]", "3"], " ", 
    SuperscriptBox[
     RowBox[{"(", 
      RowBox[{
       RowBox[{"-", 
        FractionBox["1", "4"]}], "+", "k", "+", 
       FractionBox["\[Nu]", "2"]}], ")"}], "3"]}]]}]], "Output",
 ImageSize->{430, 45},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->374804007]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->1255426704],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["BesselJ",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BesselJ"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BesselYZero",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BesselYZero"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["AiryAiZero",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/AiryAiZero"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["FindRoot",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/FindRoot"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->929782353]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->250839057],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/SpecialFunctions"]], "Tutorials",
 CellID->23022]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->128959009],

Cell[TextData[ButtonBox["Bessel-Related Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/BesselRelatedFunctions"]], "MoreAbout",
 CellID->418982243],

Cell[TextData[ButtonBox["Functions Used in Optics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/FunctionsUsedInOptics"]], "MoreAbout",
 CellID->13274966],

Cell[TextData[ButtonBox["Inverse Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/InverseFunctions"]], "MoreAbout",
 CellID->562052629],

Cell[TextData[ButtonBox["Special Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/SpecialFunctions"]], "MoreAbout",
 CellID->385708847],

Cell[TextData[ButtonBox["New in 6.0: Mathematical Functions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicalFunctions"]], "MoreAbout",
 CellID->722832632],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->362349083]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"BesselJZero - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 25, 48.7389186}", "context" -> "System`", 
    "keywords" -> {
     "Bessel functions", "Bessel zero", "cylindrical functions", 
      "zeros of Bessel functions"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "BesselJZero[n, k] represents the k\\[Null]^th zero of the Bessel \
function J_n (x). BesselJZero[n, k, x_0] represents the k\\[Null]^th zero \
greater than x_0.", "synonyms" -> {"bessel zero"}, "title" -> "BesselJZero", 
    "type" -> "Symbol", "uri" -> "ref/BesselJZero"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[8628, 267, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->133756076]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 57456, 1488}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2886, 65, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3509, 92, 1576, 37, 70, "ObjectNameGrid"],
Cell[5088, 131, 1604, 54, 70, "Usage",
 CellID->982511436]
}, Open  ]],
Cell[CellGroupData[{
Cell[6729, 190, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[7207, 207, 120, 4, 70, "Notes",
 CellID->20640],
Cell[7330, 213, 365, 12, 70, "Notes",
 CellID->43381924],
Cell[7698, 227, 456, 17, 70, "Notes",
 CellID->1067943069],
Cell[8157, 246, 224, 7, 70, "Notes",
 CellID->31226],
Cell[8384, 255, 207, 7, 70, "Notes",
 CellID->28162]
}, Closed]],
Cell[CellGroupData[{
Cell[8628, 267, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->133756076],
Cell[CellGroupData[{
Cell[9013, 281, 148, 5, 70, "ExampleSection",
 CellID->211303757],
Cell[9164, 288, 64, 1, 70, "ExampleText",
 CellID->271583785],
Cell[CellGroupData[{
Cell[9253, 293, 158, 5, 28, "Input",
 CellID->20770762],
Cell[9414, 300, 175, 5, 36, "Output",
 CellID->126301692]
}, Open  ]],
Cell[9604, 308, 125, 3, 70, "ExampleDelimiter",
 CellID->347055722],
Cell[9732, 313, 65, 1, 70, "ExampleText",
 CellID->375713404],
Cell[CellGroupData[{
Cell[9822, 318, 190, 6, 70, "Input",
 CellID->139609482],
Cell[10015, 326, 158, 5, 36, "Output",
 CellID->368038474]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[10222, 337, 221, 7, 70, "ExampleSection",
 CellID->1293636265],
Cell[10446, 346, 68, 1, 70, "ExampleText",
 CellID->4879934],
Cell[CellGroupData[{
Cell[10539, 351, 186, 6, 70, "Input",
 CellID->210060541],
Cell[10728, 359, 236, 7, 36, "Output",
 CellID->159930780]
}, Open  ]],
Cell[10979, 369, 124, 3, 70, "ExampleDelimiter",
 CellID->81909017],
Cell[11106, 374, 223, 9, 70, "ExampleText",
 CellID->25126027],
Cell[CellGroupData[{
Cell[11354, 387, 171, 5, 70, "Input",
 CellID->726288814],
Cell[11528, 394, 174, 5, 36, "Output",
 CellID->97376225]
}, Open  ]],
Cell[11717, 402, 125, 3, 70, "ExampleDelimiter",
 CellID->219294962],
Cell[11845, 407, 220, 8, 70, "ExampleText",
 CellID->65490277],
Cell[CellGroupData[{
Cell[12090, 419, 218, 8, 70, "Input",
 CellID->298902341],
Cell[12311, 429, 272, 9, 36, "Output",
 CellID->213699606]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[12632, 444, 244, 7, 70, "ExampleSection",
 CellID->1020263627],
Cell[12879, 453, 769, 29, 70, "ExampleText",
 CellID->759007334],
Cell[CellGroupData[{
Cell[13673, 486, 238, 8, 70, "Input",
 CellID->309352248],
Cell[13914, 496, 200, 5, 36, "Output",
 CellID->31581649]
}, Open  ]],
Cell[CellGroupData[{
Cell[14151, 506, 404, 15, 70, "Input",
 CellID->585149652],
Cell[14558, 523, 178, 5, 36, "Output",
 CellID->298868616]
}, Open  ]],
Cell[14751, 531, 125, 3, 70, "ExampleDelimiter",
 CellID->736573821],
Cell[14879, 536, 170, 6, 70, "ExampleText",
 CellID->5396862],
Cell[CellGroupData[{
Cell[15074, 546, 188, 6, 70, "Input",
 CellID->889963302],
Cell[15265, 554, 707, 23, 70, "Output",
 CellID->188620940]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[16021, 583, 227, 7, 70, "ExampleSection",
 CellID->258228157],
Cell[16251, 592, 273, 8, 70, "ExampleText",
 CellID->23472583],
Cell[CellGroupData[{
Cell[16549, 604, 229, 7, 70, "Input",
 CellID->559586426],
Cell[16781, 613, 479, 12, 54, "Output",
 CellID->186146956]
}, Open  ]],
Cell[17275, 628, 248, 7, 70, "ExampleText",
 CellID->37342946],
Cell[17526, 637, 296, 10, 70, "Input",
 CellID->178525005],
Cell[17825, 649, 64, 1, 70, "ExampleText",
 CellID->54566326],
Cell[CellGroupData[{
Cell[17914, 654, 887, 25, 70, "Input",
 CellID->130025248],
Cell[18804, 681, 27675, 458, 70, "Output",
 Evaluatable->False,
 CellID->723083224]
}, Open  ]],
Cell[46494, 1142, 76, 1, 70, "ExampleText",
 CellID->139382388],
Cell[CellGroupData[{
Cell[46595, 1147, 213, 7, 70, "Input",
 CellID->54809775],
Cell[46811, 1156, 3043, 54, 70, "Output",
 Evaluatable->False,
 CellID->183434337]
}, Open  ]],
Cell[49869, 1213, 125, 3, 70, "ExampleDelimiter",
 CellID->474590927],
Cell[49997, 1218, 320, 9, 70, "ExampleText",
 CellID->215833738],
Cell[CellGroupData[{
Cell[50342, 1231, 191, 7, 70, "Input",
 CellID->371271319],
Cell[50536, 1240, 175, 5, 36, "Output",
 CellID->94621189]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[50760, 1251, 238, 7, 70, "ExampleSection",
 CellID->2123667759],
Cell[51001, 1260, 385, 15, 70, "ExampleText",
 CellID->14555186],
Cell[CellGroupData[{
Cell[51411, 1279, 258, 8, 70, "Input",
 CellID->70795635],
Cell[51672, 1289, 1369, 49, 66, "Output",
 CellID->374804007]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[53102, 1345, 316, 9, 70, "SeeAlsoSection",
 CellID->1255426704],
Cell[53421, 1356, 953, 35, 70, "SeeAlso",
 CellID->929782353]
}, Open  ]],
Cell[CellGroupData[{
Cell[54411, 1396, 318, 9, 70, "TutorialsSection",
 CellID->250839057],
Cell[54732, 1407, 144, 3, 70, "Tutorials",
 CellID->23022]
}, Open  ]],
Cell[CellGroupData[{
Cell[54913, 1415, 319, 9, 70, "MoreAboutSection",
 CellID->128959009],
Cell[55235, 1426, 158, 3, 70, "MoreAbout",
 CellID->418982243],
Cell[55396, 1431, 156, 3, 70, "MoreAbout",
 CellID->13274966],
Cell[55555, 1436, 145, 3, 70, "MoreAbout",
 CellID->562052629],
Cell[55703, 1441, 145, 3, 70, "MoreAbout",
 CellID->385708847],
Cell[55851, 1446, 174, 3, 70, "MoreAbout",
 CellID->722832632],
Cell[56028, 1451, 179, 3, 70, "MoreAbout",
 CellID->362349083]
}, Open  ]],
Cell[56222, 1457, 27, 0, 70, "History"],
Cell[56252, 1459, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

