(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     43435,       1499]
NotebookOptionsPosition[     35972,       1236]
NotebookOutlinePosition[     37415,       1275]
CellTagsIndexPosition[     37329,       1270]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Discrete Distributions" :> 
          Documentation`HelpLookup[
           "paclet:tutorial/DiscreteDistributions"]}, #, 2]& , Evaluator -> 
        Automatic], {
       1->"\<\"Discrete Distributions\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"BinomialDistribution" :> 
          Documentation`HelpLookup["paclet:ref/BinomialDistribution"], 
          "BetaDistribution" :> 
          Documentation`HelpLookup["paclet:ref/BetaDistribution"], 
          "BetaNegativeBinomialDistribution" :> 
          Documentation`HelpLookup[
           "paclet:ref/BetaNegativeBinomialDistribution"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"BinomialDistribution\"\>", 
       2->"\<\"BetaDistribution\"\>", 
       3->"\<\"BetaNegativeBinomialDistribution\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Discrete Statistical Distributions" :> 
          Documentation`HelpLookup[
           "paclet:guide/DiscreteStatisticalDistributions"]}, #, 2]& , 
        Evaluator -> Automatic], {
       1->"\<\"Discrete Statistical Distributions\"\>"}, "\<\"more about\"\>",
        
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["BetaBinomialDistribution", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["BetaBinomialDistribution",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/BetaBinomialDistribution"], "[", 
       RowBox[{
        StyleBox["\[Alpha]", "TR"], ",", 
        StyleBox["\[Beta]", "TR"], ",", 
        StyleBox["n", "TI"]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]represents a beta binomial mixture distribution with \
beta distribution parameters ",
     Cell[BoxData[
      FormBox["\[Alpha]", TraditionalForm]], "InlineMath"],
     " and ",
     Cell[BoxData[
      FormBox["\[Beta]", TraditionalForm]], "InlineMath"],
     ", and ",
     Cell[BoxData[
      FormBox["n", TraditionalForm]], "InlineMath"],
     " binomial trials."
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {{None}}, 
   "RowsIndexed" -> {}}},
 CellID->10419]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "The beta-binomial distribution is a binomial distribution with ",
 Cell[BoxData[
  FormBox["n", TraditionalForm]], "InlineMath"],
 " trials whose probability parameter ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 " follows a beta distribution with shape parameters ",
 Cell[BoxData[
  StyleBox["\[Alpha]", "TR"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["\[Beta]", "TR"]], "InlineFormula"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->311311635,
  ButtonNote->"311311635"]
}], "Notes",
 CellID->272931153],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BetaBinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BetaBinomialDistribution"]], "InlineFormula"],
 " allows ",
 Cell[BoxData[
  StyleBox["\[Alpha]", "TR"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["\[Beta]", "TR"]], "InlineFormula"],
 " to be any positive real numbers, and ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " to be any positive integer."
}], "Notes",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0.7, 0},
 ScriptMinSize->9,
 CellID->10777],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BetaBinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BetaBinomialDistribution"]], "InlineFormula"],
 " can be used with such functions as ",
 Cell[BoxData[
  ButtonBox["Mean",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Mean"]], "InlineFormula"],
 ", ",
 Cell[BoxData[
  ButtonBox["CDF",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CDF"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["RandomInteger",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RandomInteger"]], "InlineFormula"],
 ".",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->15726,
  ButtonNote->"15726"]
}], "Notes",
 CellID->33143606]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->157164439],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->150687490],

Cell["Mean and variance of a beta binomial distribution:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 CellFrameLabels->{{None, None}, {None, None}},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->15726],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"BetaBinomialDistribution", "[", 
   RowBox[{"\[Alpha]", ",", "\[Beta]", ",", "n"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2922],

Cell[BoxData[
 FractionBox[
  RowBox[{"n", " ", "\[Alpha]"}], 
  RowBox[{"\[Alpha]", "+", "\[Beta]"}]]], "Output",
 ImageSize->{35, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->422575976]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Variance", "[", 
  RowBox[{"BetaBinomialDistribution", "[", 
   RowBox[{"\[Alpha]", ",", "\[Beta]", ",", "n"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->23529],

Cell[BoxData[
 FractionBox[
  RowBox[{"n", " ", "\[Alpha]", " ", "\[Beta]", " ", 
   RowBox[{"(", 
    RowBox[{"n", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}]}], 
  RowBox[{
   SuperscriptBox[
    RowBox[{"(", 
     RowBox[{"\[Alpha]", "+", "\[Beta]"}], ")"}], "2"], " ", 
   RowBox[{"(", 
    RowBox[{"1", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}]}]]], "Output",
 ImageSize->{118, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->71098824]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->23627],

Cell["Probability density function:", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->9425],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"BetaBinomialDistribution", "[", 
    RowBox[{"\[Alpha]", ",", "\[Beta]", ",", "n"}], "]"}], ",", "k"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19900],

Cell[BoxData[
 FractionBox[
  RowBox[{
   RowBox[{"Beta", "[", 
    RowBox[{
     RowBox[{"k", "+", "\[Alpha]"}], ",", 
     RowBox[{
      RowBox[{"-", "k"}], "+", "n", "+", "\[Beta]"}]}], "]"}], " ", 
   RowBox[{"Binomial", "[", 
    RowBox[{"n", ",", "k"}], "]"}]}], 
  RowBox[{"Beta", "[", 
   RowBox[{"\[Alpha]", ",", "\[Beta]"}], "]"}]]], "Output",
 ImageSize->{241, 32},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->749144913]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListPlot", "[", 
  RowBox[{
   RowBox[{"Table", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"k", ",", 
       RowBox[{"PDF", "[", 
        RowBox[{
         RowBox[{"BetaBinomialDistribution", "[", 
          RowBox[{"5", ",", "15", ",", "50"}], "]"}], ",", "k"}], "]"}]}], 
      "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"k", ",", "0", ",", "50"}], "}"}]}], "]"}], ",", 
   RowBox[{"Filling", "\[Rule]", "Axis"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->7975],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztXUuPFUUUrrk994LXRySBiIkM18mY6D/AhZr4jAs38lq4MJOJCsGZIUB0
YaILlVFGZWGiQXA0wvADMIxLFReCGliCCxaDCxZoAiaTGIlpT3V1nz5dr+77
6lv9mGS6q07XrfrOd06dOtV9p+eF6UN7XpmdPrR3Zrrz3IHp/Xv2zhzsPDt/
AETeGGNsFX6PbmZQ9qEU/K76PpzV4zY4etCgAecGnDtwHif1reF1L6xP8k/x
Bn70E32KtvKCvh9Ntg17ZV4smeCHpqjjUWm1ldSN163jdDQ9cKLG+Dkj1vRR
0nBuCbllTVQ3bDk5AHw6HeXRG5E80ZMWd6JFHvgmUnuYMGrQMV5pZETH2WEt
2qI3G9sxRr4eW15lcpsfzyF+flCqcwTpM3KS4IhnJJ35myTkF7TSBX54fPnU
pSceO/bxRz9BLbV86uRFKCweOQfCkZfHA602wPFSyDJjGwMdAS9IAK+wVfP3
KzcUy7DW7dv/Xb36l5BCWStfW/sX5VD+49pNVX7r1j/Xr/8t5FDWykH45421
wGLhOfBIKIvr0BCaoxzKWjkMK+QABMqpclBFyEEhKDciP0U5MCP8J+G9XIpc
4AXBSyB/kcq//uo39JRdO5ZB/sxTJ4LyaZCL8u6dp1F+/ufVudkz361cgfKF
89dAjuW52W9FGYSiDMLX951ZOXsZG2M5f/k48THUaBA+BmbT+hjKDT42TuUN
ycNiTxNepHoUlVPPEWXUxSSPPKolvEuRK5x4SakgRGhyH2Xk5DcXxfyVvIh6
CJTn5+SyEuPkejOl7hELo29qLJyYIdTC0Qzp18JQzsvCJjmJGQkL01gic6Ja
GFoLpu4Eq2KcgNlDrRpa73mtNbgNcd2trZDVCvegNJX47cizug5YLGpk3cQu
lRN2DawXhOc2SqVQJbjdHXMb56CYUEn91uwSdgPpOpqHYvxnLyGrPDIwvjCj
Z9ecpnDqIVXsZV/jnei7NZNpTAYeyXNZyqScX9QMBnJkAaf0TM1ZVs5wysac
8cjXqrlSucL4VftXCmcessBerbkSnLRVrnwSs15Dnio6//T8YHwS/IRbN4a7
fuJT0T1LL0zc6LWg3qI5n2k3V3pGMYqFjCbuOAc8JSQyy3JdMEu3fxVgVhP3
9g2QTcz+qsammKCLR871ziavrreyVdJ7NzgFB+OJleQQY2PNYZccesghmyPk
iB+ezdDnkc3wfEeF+ZEz4hL6Slvlwicxfr7aPGC8jnko6ZzQ6483XSvjB5rY
cLDWveK6l3TOd2lv+iixRLrjWmfTPcveuYC6o1o6X6ffsiiRzqiW0DntbhX/
9LjmbtVdChcF3O17KGdvCBpSdlMNwgKrWfGJC1n2mNVkRfYN3guv1mxEbBTW
R9oqK360kn74wY9vmhjJ8E3sEvKCq2zIS4a7LncXX2tcZ4XWcl5VKLtqosBb
tVYF1Kpwc6uCNiqnVrHnmfbRzmqD67rNRqYdsrNaIWCbVqY9sLNaIWC9Vvf6
8nxybgdPMuu3fTlfMu3QPZI7bSiijn3uHgpp11pnRedgtNCXuapl1DHtbkkZ
dS6QL7dVnX2SA7xDjJjcwzoch/U6YQYgdHJ4rdTjx6ezzuPXzKN3C43ZMf8u
Hb9FxBz7BN3huYlV5Zfu34qCme7O3Me80Yott50jruRJz93kBjp0Q4Eube8n
51VyG0c4R0cNteozQx6pVh7K2WGTNk0qidTjV7gSzqNfrz6lEkf5jRHOa2Lc
W9ZaDFkLzf5IF6tGHnkV5E6tWm6ga6vo/GgdXTj8wwIiy82aekS4dsaIHOFK
/LHUytnLI0Sm8bFFCxr6Zqu80cR+lCUrz4+TLPl2Hmg2W/1l0Fl+0i73j2Ds
bp6AjJqnfu+E1pgNmHG96R0zr+aCFcNWNqymnC4XrHgXNTtW5a+B88FMsqFP
fDku0CyTnocYrxQ8OcZp58Z2g2tT1pLH2PQVaHmPbcpMhmPnLYqde8t6gkmK
cf2oVr8HBjkWxuUcxkKTxGMNkrtPfTkCmt41oltZBqVpiGIg33SuMZkx8erQ
sSh3LXU59dBRdJVtCDQ06neHhkSg99/7Xp1TpqxCzOJdO5Yx4vcwrohGTz95
Qoyrj0b0PSs9jIFRSD9Gx9KXeZX4fCh9RW+FdwXP6HVj3vFjvwjPZp9ZcC19
+avO/w197tweeq21T2yVqc/4Pca2Puk7rG19Hvezrm1ybODyhyw8k977jDxT
NmsSu32RabS0J15TNjsTi/Y+Gq9O2SzvxW+gHaBO62h8Q/3YkmmEHtbwh4N6
/N7XJQIvGeXpm9l4/ZFI79C3xSf1/2GC/g8KNvY/YC5eng==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 143},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->351137597]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->21413],

Cell["\<\
Generate a set of pseudorandom numbers that have a beta binomial \
distribution:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->19096],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RandomInteger", "[", 
  RowBox[{
   RowBox[{"BetaBinomialDistribution", "[", 
    RowBox[{"3", ",", "7.5", ",", "20"}], "]"}], ",", "10"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->17016],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "8", ",", "3", ",", "3", ",", "6", ",", "1", ",", "4", ",", "6", ",", "8", 
   ",", "3", ",", "4"}], "}"}]], "Output",
 ImageSize->{189, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->570895079]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->26905],

Cell["\<\
Properties based on higher\[Hyphen]order moments:\
\>", "ExampleText",
 CellID->20699375],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Skewness", "[", 
  RowBox[{"BetaBinomialDistribution", "[", 
   RowBox[{"\[Alpha]", ",", "\[Beta]", ",", "n"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7251],

Cell[BoxData[
 RowBox[{"-", 
  FractionBox[
   RowBox[{
    RowBox[{"(", 
     RowBox[{"\[Alpha]", "-", "\[Beta]"}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"2", " ", "n"}], "+", "\[Alpha]", "+", "\[Beta]"}], ")"}]}], 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"2", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}], " ", 
    SqrtBox[
     FractionBox[
      RowBox[{"n", " ", "\[Alpha]", " ", "\[Beta]", " ", 
       RowBox[{"(", 
        RowBox[{"n", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}]}], 
      RowBox[{"1", "+", "\[Alpha]", "+", "\[Beta]"}]]]}]]}]], "Output",
 ImageSize->{160, 56},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->413794163]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Kurtosis", "[", 
  RowBox[{"BetaBinomialDistribution", "[", 
   RowBox[{"\[Alpha]", ",", "\[Beta]", ",", "n"}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->27703],

Cell[BoxData[
 RowBox[{
  RowBox[{"(", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"1", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"3", " ", 
       SuperscriptBox["n", "2"], " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"\[Alpha]", " ", 
          RowBox[{"(", 
           RowBox[{
            RowBox[{"-", "2"}], "+", "\[Beta]"}], ")"}], " ", "\[Beta]"}], 
         "+", 
         RowBox[{"2", " ", 
          SuperscriptBox["\[Beta]", "2"]}], "+", 
         RowBox[{
          SuperscriptBox["\[Alpha]", "2"], " ", 
          RowBox[{"(", 
           RowBox[{"2", "+", "\[Beta]"}], ")"}]}]}], ")"}]}], "+", 
      RowBox[{"3", " ", "n", " ", 
       RowBox[{"(", 
        RowBox[{
         RowBox[{"2", " ", 
          SuperscriptBox["\[Alpha]", "2"], " ", 
          SuperscriptBox["\[Beta]", "2"]}], "+", 
         RowBox[{"2", " ", 
          SuperscriptBox["\[Beta]", "3"]}], "+", 
         RowBox[{"\[Alpha]", " ", 
          SuperscriptBox["\[Beta]", "3"]}], "+", 
         RowBox[{
          SuperscriptBox["\[Alpha]", "3"], " ", 
          RowBox[{"(", 
           RowBox[{"2", "+", "\[Beta]"}], ")"}]}]}], ")"}]}], "+", 
      RowBox[{
       SuperscriptBox[
        RowBox[{"(", 
         RowBox[{"\[Alpha]", "+", "\[Beta]"}], ")"}], "2"], " ", 
       RowBox[{"(", 
        RowBox[{
         SuperscriptBox["\[Alpha]", "2"], "+", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{
            RowBox[{"-", "1"}], "+", "\[Beta]"}], ")"}], " ", "\[Beta]"}], 
         "-", 
         RowBox[{"\[Alpha]", " ", 
          RowBox[{"(", 
           RowBox[{"1", "+", 
            RowBox[{"4", " ", "\[Beta]"}]}], ")"}]}]}], ")"}]}]}], ")"}]}], 
   ")"}], "/", 
  RowBox[{"(", 
   RowBox[{"n", " ", "\[Alpha]", " ", "\[Beta]", " ", 
    RowBox[{"(", 
     RowBox[{"2", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{"3", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{"n", "+", "\[Alpha]", "+", "\[Beta]"}], ")"}]}], 
   ")"}]}]], "Output",
 ImageSize->{527, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->33979992]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->113],

Cell[TextData[{
 "The ",
 Cell[BoxData[
  FormBox["0.7`", TraditionalForm]], "InlineMath"],
 " quantile of a beta binomial distribution with ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Alpha]", "=", "2"}], TraditionalForm]], "InlineMath"],
 ", ",
 Cell[BoxData[
  FormBox[
   RowBox[{"\[Beta]", "=", "5"}], TraditionalForm]], "InlineMath"],
 " and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"n", "=", "100"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->48521834],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Quantile", "[", 
  RowBox[{
   RowBox[{"BetaBinomialDistribution", "[", 
    RowBox[{"2", ",", "5", ",", "100"}], "]"}], ",", ".7"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->4591],

Cell[BoxData["36"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->98408037]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->17011],

Cell[TextData[{
 "The probability of more than 50 successes in 100 trials assuming a beta \
distribution on ",
 Cell[BoxData[
  StyleBox["p", "TI"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->32514],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"1", "-", 
  RowBox[{"CDF", "[", 
   RowBox[{
    RowBox[{"BetaBinomialDistribution", "[", 
     RowBox[{"4", ",", "5", ",", "100"}], "]"}], ",", "50"}], 
   "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10642],

Cell[BoxData[
 FractionBox["398435", "1113121"]], "Output",
 ImageSize->{62, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->388258411]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"N", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->3834],

Cell[BoxData["0.3579440150711378`"], "Output",
 ImageSize->{60, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->14696202]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->86062964],

Cell["\<\
Plot the cumulative distribution function of a beta binomial distribution:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->13370],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Plot", "[", 
  RowBox[{
   RowBox[{"CDF", "[", 
    RowBox[{
     RowBox[{"BetaBinomialDistribution", "[", 
      RowBox[{"10", ",", "5", ",", "20"}], "]"}], ",", "k"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"k", ",", "0", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->26124],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWT1oFEEUntzeH6JiZ0CLmM4kRqt04mn8QUVtDGqjXA7hToyR5FRQLLQw
hVb+gSCehRGMFoISsRBEwUoSkCiSFHKFxCJBMKQICOt7b+cys7Oze7u3t4eH
LpeZffve+968mTdv520OZIv5kwPZYiGXbdszlD2bL+SG23YPDsEjo4UxVoa/
260M7k24o7+yaUIvtxloDWB18z4GfQz6Lk6zOInxqxOIuCRlELMXWxQ26SJd
Fhe0JS9LdCi0odAbsUlY9HJLKOhYC/YRWe7QIkRttVOi9Z5u5zO+2Z8tIcH1
un1ZkHG6/VlKyJYYjxyWlPXc5nEHH9smU0RTJfZiPPRiPOpkvhV1chSvUUY2
4edp6cFH+LH10KyYnPjOjIWFJS45SpIroZ3iI3aipQLSaYVOEg024dHgmRcs
jU0I/Gr2wuLVG9/QzuoWrdb+vaVtW+/Bj6WwMY70PbYWi62roq/Seqs7iQbU
2dlfeDviLXXr5geMnBQ2xtOxT5x/2UPr/btvPN4sLUCBmLNrYWwfImkIBQdX
xjwYTGrk2lthefTR5NLSb2QNe2gBsG28uf5nr8a/umsdJRoM1SiFvh8LiaGV
unTxNTqQJi/EDs97aM1Mz9l8hxmEHlkFrVY/jh8kTtWFizORqxsiD3i65LcO
0ZXXu7u2Lf8nFH2dLjNgWfheTdNetaYOWcc1mEkFQ30znq5gwlLi7QkNhppZ
8E2DZK26Qe3TPDifVEeR1ibJVz7hS1Nd87BSaHmgIZacUp+nfuzqvS/iBTIN
5x/Wap0nGhI1pFC87YtECmfkQkMs/Vu+zc8vwstcnHesAEHWPlPdE3TrOF+q
O8TypFz+Ca8sPMal6CwHoWTHDZR1rhDtjYF7PM57VPWrI2dQ1Rc/GAlTnyki
0JXWQnt6q4qgxkBYKRz71YZYilYKNortPAtbgp9nMzZfb1Sk+bk0o8W83nRS
f5NvbickvNXtFReUgCeC2lEQpLFjMLxmwaUW6tFgqRj68bhXSj2O9UqZ4kuB
jHSHaFEf9WjjIawUWr7bEEv/fdNKudS1XR5a7nWt0ELfH5K0qDL1mKXIpYJm
JxeUgJmhcSjyiQRVm2D8Pqq2gCtu+0YxMz3H+Rs8tNRvFADBa632ZS0c2RhJ
i0qsXYv5pOmkIvMNEoJYCysNIWuthxakHUg+Yi3AECQnZLVqtYiF2JFK4Qy9
DIYBvmPxlKYKCio1zl/loQVS54rjWMqlqJ4r5J9bBTxbbYpNVbnUryu6vfOG
aFFNWjiqJm1Cz91bXxw4nluHdsdXCiMALgpM1DiOcHq6rPWFHjn/J8ha/gBe
m77m\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 148},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->50592151]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->540041761],

Cell[TextData[{
 "Density functions for beta binomial distributions depend heavily on the \
parameters ",
 Cell[BoxData[
  StyleBox["\[Alpha]", "TR"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["\[Beta]", "TR"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->24052],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"PDF", "[", 
     RowBox[{
      RowBox[{"BetaBinomialDistribution", "[", 
       RowBox[{"\[Alpha]", ",", "7.5", ",", "20"}], "]"}], ",", "k"}], "]"}], 
    ",", 
    RowBox[{"{", 
     RowBox[{"k", ",", "1", ",", "12"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"\[Alpha]", ",", "2", ",", "24", ",", "2"}], "}"}]}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->28041],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzt279rGmEYB3Btmj/gvkMRQYxOlYCKDi6SQUIFaUFUcFJQKSRDabEuEiI4
ZMxiIaCLiAkOurqIgiAKDoaQJRmiIigOig7+Cmq0957p3CVQrM/Be7x393w/
43Pv3ekXT+Dk6zdP4NTnOTD7PT9OTn0/Dz599/On9sQikajFhkTEz9f8bDP2
1ms2YVudH+LN4R67JmyiOr8Tbwre9Ow7fod+v49sNgu/3w+j0Qi5XM5JJBIo
FAro9XpYLBZ4vV4Eg0FcXl6+J4kkkkjaZanVaiEajcJut+Pw8BBKpZLTaDQw
m83w+XwIhUK4urpCOp1GPp8niSSSSNpp6f7+HmdnZ0LnValUMBgMnMPhQCAQ
QCQSQSaTQbFYxN3dHZ6enkgiiSSSdloqFApwuVzQarVMgdPp5M7PzxGPx5HL
5XB7e4t6vY5ut4vBYEASSSSRtNNSMpnE8fExdDodbDYbW6FyNzc3KJVKeHh4
QKfTYQLG4zGen59JIokkknZauri4EN6AmkwmYZV6fX3NVSoVoeP2ej2MRiMm
YLFY4OXlhSSSSCJppyX2pUitVsPtdiMWi6FcLnPNZlPouNPpFPP5HMvlEqvV
6gMppJBCyrYqR0dHwlN5OBxmX3nQaDS44XCI2Wz2p1My4SMJJJBAwr8QZDIZ
rFYrUqkUHh8fwac5tpJ77W6fKUlJSm5jcn9/X3hWq1arwm/4+BT32g/clKIU
pbYnxR8ikUig3W4L93Y+EaZqqqbqv1dLpVLUajVMJhNW+YsqqfI/rnzLf721
NtfEvwEAp221\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{225, 225},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->343669660]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(2)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->21358],

Cell[TextData[{
 "The probability of getting negative integers, integers beyond ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 ", or non-integer numbers is zero:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->6782],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"PDF", "[", 
  RowBox[{
   RowBox[{"BetaBinomialDistribution", "[", 
    RowBox[{"\[Alpha]", ",", "\[Beta]", ",", "n"}], "]"}], ",", "\[Pi]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17831],

Cell[BoxData["0"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->14312290]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->61963951],

Cell[TextData[{
 "Beta binomial is a mixture of ",
 Cell[BoxData[
  ButtonBox["BinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BinomialDistribution"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["BetaDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BetaDistribution"]], "InlineFormula"],
 ":"
}], "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->311311635],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"binom", "=", 
  RowBox[{"PDF", "[", 
   RowBox[{
    RowBox[{"BinomialDistribution", "[", 
     RowBox[{"10", ",", "p"}], "]"}], ",", "k"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->221558863],

Cell[BoxData[
 RowBox[{
  SuperscriptBox[
   RowBox[{"(", 
    RowBox[{"1", "-", "p"}], ")"}], 
   RowBox[{"10", "-", "k"}]], " ", 
  SuperscriptBox["p", "k"], " ", 
  RowBox[{"Binomial", "[", 
   RowBox[{"10", ",", "k"}], "]"}]}]], "Output",
 ImageSize->{188, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->265385930]
}, Open  ]],

Cell[BoxData[
 RowBox[{
  RowBox[{"dist", "=", 
   RowBox[{"BetaDistribution", "[", 
    RowBox[{"2", ",", "4"}], "]"}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->32781365],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ExpectedValue", "[", 
  RowBox[{"binom", ",", "dist", ",", "p", ",", 
   RowBox[{"Assumptions", "\[Rule]", 
    RowBox[{"10", ">=", "k", "\[GreaterEqual]", "0"}]}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->759718569],

Cell[BoxData[
 RowBox[{"-", 
  FractionBox[
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "13"}], "+", "k"}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "12"}], "+", "k"}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "11"}], "+", "k"}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{"1", "+", "k"}], ")"}]}], "18018"]}]], "Output",
 ImageSize->{232, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->283373837]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"PDF", "[", 
   RowBox[{
    RowBox[{"BetaBinomialDistribution", "[", 
     RowBox[{"2", ",", "4", ",", "10"}], "]"}], ",", "k"}], "]"}], "//", 
  "FunctionExpand"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->104120125],

Cell[BoxData[
 RowBox[{"-", 
  FractionBox[
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "13"}], "+", "k"}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "12"}], "+", "k"}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"-", "11"}], "+", "k"}], ")"}], " ", 
    RowBox[{"(", 
     RowBox[{"1", "+", "k"}], ")"}]}], "18018"]}]], "Output",
 ImageSize->{232, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->3624490]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(3)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->21111],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BetaBinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BetaBinomialDistribution"]], "InlineFormula"],
 " is not defined when ",
 Cell[BoxData[
  StyleBox["\[Alpha]", "TR"]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  StyleBox["\[Beta]", "TR"]], "InlineFormula"],
 " is non-positive:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->28540],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"BetaBinomialDistribution", "[", 
   RowBox[{
    RowBox[{"-", "4"}], ",", "6.2", ",", "40"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->31597],

Cell[BoxData[
 RowBox[{
  RowBox[{"BetaBinomialDistribution", "::", "\<\"posparm\"\>"}], ":", 
  " ", "\<\"Parameter \\!\\(-4\\) is expected to be positive.\"\>"}]], \
"Message", "MSG",
 CellID->438769290],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"BetaBinomialDistribution", "[", 
   RowBox[{
    RowBox[{"-", "4"}], ",", "6.2`", ",", "40"}], "]"}], "]"}]], "Output",
 ImageSize->{301, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->33150362]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->577907513],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["BetaBinomialDistribution",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/BetaBinomialDistribution"]], "InlineFormula"],
 " is not defined when ",
 Cell[BoxData[
  StyleBox["n", "TI"]], "InlineFormula"],
 " is not a positive integer:"
}], "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->436266755],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"BetaBinomialDistribution", "[", 
   RowBox[{"5", ",", "6.2", ",", 
    RowBox[{"-", "10"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->871243724],

Cell[BoxData[
 RowBox[{
  RowBox[{"BetaBinomialDistribution", "::", "\<\"posint\"\>"}], ":", 
  " ", "\<\"Parameter \\!\\(-10\\) is expected to be a positive \
integer.\"\>"}]], "Message", "MSG",
 CellID->30754757],

Cell[BoxData[
 RowBox[{"Mean", "[", 
  RowBox[{"BetaBinomialDistribution", "[", 
   RowBox[{"5", ",", "6.2`", ",", 
    RowBox[{"-", "10"}]}], "]"}], "]"}]], "Output",
 ImageSize->{301, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->280355742]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->19046],

Cell["\<\
Substitution of invalid parameters into symbolic outputs gives results that \
are not meaningful:\
\>", "ExampleText",
 CellBracketOptions->{"Color"->RGBColor[0.269993, 0.308507, 0.6]},
 ShowAutoStyles->False,
 ParagraphSpacing->{0, 6},
 ScriptMinSize->9,
 CellID->16727],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"Mean", "[", 
   RowBox[{"BetaBinomialDistribution", "[", 
    RowBox[{"\[Alpha]", ",", "\[Beta]", ",", "n"}], "]"}], "]"}], "/.", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"\[Alpha]", "\[Rule]", 
     RowBox[{"3", "+", "I"}]}], ",", 
    RowBox[{"\[Beta]", "\[Rule]", "7"}], ",", 
    RowBox[{"n", "\[Rule]", "23"}]}], "}"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25838],

Cell[BoxData[
 RowBox[{
  FractionBox["713", "101"], "+", 
  FractionBox[
   RowBox[{"161", " ", "\[ImaginaryI]"}], "101"]}]], "Output",
 ImageSize->{78, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->154987007]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->17436],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["BinomialDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BinomialDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BetaDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BetaDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BetaNegativeBinomialDistribution",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BetaNegativeBinomialDistribution"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->16825]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->10883],

Cell[TextData[ButtonBox["Discrete Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/DiscreteDistributions"]], "Tutorials",
 CellID->4738]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection"],

Cell[TextData[ButtonBox["Discrete Statistical Distributions",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DiscreteStatisticalDistributions"]], "MoreAbout",
 CellID->611803265]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"BetaBinomialDistribution - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 25, 50.7858460}", "context" -> "System`", 
    "keywords" -> {"beta-binomial distribution", "mixture distribution"}, 
    "index" -> True, "label" -> "Built-in Mathematica Symbol", "language" -> 
    "en", "paclet" -> "Mathematica", "status" -> "None", "summary" -> 
    "BetaBinomialDistribution[\\[Alpha], \\[Beta], n] represents a beta \
binomial mixture distribution with beta distribution parameters \\[Alpha] and \
\\[Beta], and n binomial trials.", 
    "synonyms" -> {"beta binomial distribution"}, "title" -> 
    "BetaBinomialDistribution", "type" -> "Symbol", "uri" -> 
    "ref/BetaBinomialDistribution"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[7821, 245, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->157164439]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 37186, 1263}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 2181, 55, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2804, 82, 1589, 37, 70, "ObjectNameGrid"],
Cell[4396, 121, 945, 28, 70, "Usage",
 CellID->10419]
}, Open  ]],
Cell[CellGroupData[{
Cell[5378, 154, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[5856, 171, 590, 19, 70, "Notes",
 CellID->272931153],
Cell[6449, 192, 616, 20, 70, "Notes",
 CellID->10777],
Cell[7068, 214, 716, 26, 70, "Notes",
 CellID->33143606]
}, Closed]],
Cell[CellGroupData[{
Cell[7821, 245, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->157164439],
Cell[CellGroupData[{
Cell[8206, 259, 148, 5, 70, "ExampleSection",
 CellID->150687490],
Cell[8357, 266, 274, 6, 70, "ExampleText",
 CellID->15726],
Cell[CellGroupData[{
Cell[8656, 276, 193, 5, 28, "Input",
 CellID->2922],
Cell[8852, 283, 244, 8, 51, "Output",
 CellID->422575976]
}, Open  ]],
Cell[CellGroupData[{
Cell[9133, 296, 198, 5, 28, "Input",
 CellID->23529],
Cell[9334, 303, 490, 15, 54, "Output",
 CellID->71098824]
}, Open  ]],
Cell[9839, 321, 121, 3, 70, "ExampleDelimiter",
 CellID->23627],
Cell[9963, 326, 204, 5, 70, "ExampleText",
 CellID->9425],
Cell[CellGroupData[{
Cell[10192, 335, 221, 7, 70, "Input",
 CellID->19900],
Cell[10416, 344, 485, 16, 53, "Output",
 CellID->749144913]
}, Open  ]],
Cell[CellGroupData[{
Cell[10938, 365, 508, 16, 47, "Input",
 CellID->7975],
Cell[11449, 383, 2579, 46, 164, "Output",
 Evaluatable->False,
 CellID->351137597]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[14077, 435, 216, 7, 70, "ExampleSection",
 CellID->21413],
Cell[14296, 444, 264, 8, 70, "ExampleText",
 CellID->19096],
Cell[CellGroupData[{
Cell[14585, 456, 221, 7, 70, "Input",
 CellID->17016],
Cell[14809, 465, 291, 9, 36, "Output",
 CellID->570895079]
}, Open  ]],
Cell[15115, 477, 121, 3, 70, "ExampleDelimiter",
 CellID->26905],
Cell[15239, 482, 99, 3, 70, "ExampleText",
 CellID->20699375],
Cell[CellGroupData[{
Cell[15363, 489, 197, 5, 70, "Input",
 CellID->7251],
Cell[15563, 496, 712, 22, 77, "Output",
 CellID->413794163]
}, Open  ]],
Cell[CellGroupData[{
Cell[16312, 523, 198, 5, 70, "Input",
 CellID->27703],
Cell[16513, 530, 2218, 67, 58, "Output",
 CellID->33979992]
}, Open  ]],
Cell[18746, 600, 119, 3, 70, "ExampleDelimiter",
 CellID->113],
Cell[18868, 605, 619, 22, 70, "ExampleText",
 CellID->48521834],
Cell[CellGroupData[{
Cell[19512, 631, 212, 6, 70, "Input",
 CellID->4591],
Cell[19727, 639, 158, 5, 36, "Output",
 CellID->98408037]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[19934, 650, 223, 7, 70, "ExampleSection",
 CellID->17011],
Cell[20160, 659, 344, 11, 70, "ExampleText",
 CellID->32514],
Cell[CellGroupData[{
Cell[20529, 674, 238, 8, 70, "Input",
 CellID->10642],
Cell[20770, 684, 189, 6, 51, "Output",
 CellID->388258411]
}, Open  ]],
Cell[CellGroupData[{
Cell[20996, 695, 92, 3, 70, "Input",
 CellID->3834],
Cell[21091, 700, 175, 5, 36, "Output",
 CellID->14696202]
}, Open  ]],
Cell[21281, 708, 124, 3, 70, "ExampleDelimiter",
 CellID->86062964],
Cell[21408, 713, 258, 7, 70, "ExampleText",
 CellID->13370],
Cell[CellGroupData[{
Cell[21691, 724, 328, 10, 70, "Input",
 CellID->26124],
Cell[22022, 736, 1558, 30, 70, "Output",
 Evaluatable->False,
 CellID->50592151]
}, Open  ]],
Cell[23595, 769, 125, 3, 70, "ExampleDelimiter",
 CellID->540041761],
Cell[23723, 774, 419, 14, 70, "ExampleText",
 CellID->24052],
Cell[CellGroupData[{
Cell[24167, 792, 477, 15, 70, "Input",
 CellID->28041],
Cell[24647, 809, 1018, 21, 70, "Output",
 Evaluatable->False,
 CellID->343669660]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[25714, 836, 233, 7, 70, "ExampleSection",
 CellID->21358],
Cell[25950, 845, 347, 10, 70, "ExampleText",
 CellID->6782],
Cell[CellGroupData[{
Cell[26322, 859, 225, 7, 70, "Input",
 CellID->17831],
Cell[26550, 868, 157, 5, 36, "Output",
 CellID->14312290]
}, Open  ]],
Cell[26722, 876, 124, 3, 70, "ExampleDelimiter",
 CellID->61963951],
Cell[26849, 881, 430, 14, 70, "ExampleText",
 CellID->311311635],
Cell[CellGroupData[{
Cell[27304, 899, 226, 7, 70, "Input",
 CellID->221558863],
Cell[27533, 908, 373, 13, 36, "Output",
 CellID->265385930]
}, Open  ]],
Cell[27921, 924, 181, 6, 70, "Input",
 CellID->32781365],
Cell[CellGroupData[{
Cell[28127, 934, 250, 6, 70, "Input",
 CellID->759718569],
Cell[28380, 942, 518, 19, 51, "Output",
 CellID->283373837]
}, Open  ]],
Cell[CellGroupData[{
Cell[28935, 966, 253, 8, 70, "Input",
 CellID->104120125],
Cell[29191, 976, 516, 19, 51, "Output",
 CellID->3624490]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[29756, 1001, 226, 7, 70, "ExampleSection",
 CellID->21111],
Cell[29985, 1010, 521, 17, 70, "ExampleText",
 CellID->28540],
Cell[CellGroupData[{
Cell[30531, 1031, 204, 6, 70, "Input",
 CellID->31597],
Cell[30738, 1039, 205, 5, 70, "Message",
 CellID->438769290],
Cell[30946, 1046, 297, 9, 36, "Output",
 CellID->33150362]
}, Open  ]],
Cell[31258, 1058, 125, 3, 70, "ExampleDelimiter",
 CellID->577907513],
Cell[31386, 1063, 456, 14, 70, "ExampleText",
 CellID->436266755],
Cell[CellGroupData[{
Cell[31867, 1081, 208, 6, 70, "Input",
 CellID->871243724],
Cell[32078, 1089, 214, 5, 70, "Message",
 CellID->30754757],
Cell[32295, 1096, 298, 9, 36, "Output",
 CellID->280355742]
}, Open  ]],
Cell[32608, 1108, 121, 3, 70, "ExampleDelimiter",
 CellID->19046],
Cell[32732, 1113, 281, 8, 70, "ExampleText",
 CellID->16727],
Cell[CellGroupData[{
Cell[33038, 1125, 410, 12, 70, "Input",
 CellID->25838],
Cell[33451, 1139, 266, 9, 51, "Output",
 CellID->154987007]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[33778, 1155, 311, 9, 70, "SeeAlsoSection",
 CellID->17436],
Cell[34092, 1166, 784, 26, 70, "SeeAlso",
 CellID->16825]
}, Open  ]],
Cell[CellGroupData[{
Cell[34913, 1197, 314, 9, 70, "TutorialsSection",
 CellID->10883],
Cell[35230, 1208, 153, 3, 70, "Tutorials",
 CellID->4738]
}, Open  ]],
Cell[CellGroupData[{
Cell[35420, 1216, 299, 8, 70, "MoreAboutSection"],
Cell[35722, 1226, 178, 3, 70, "MoreAbout",
 CellID->611803265]
}, Open  ]],
Cell[35915, 1232, 27, 0, 70, "History"],
Cell[35945, 1234, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

