(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[    643148,      16066]
NotebookOptionsPosition[    599117,      14508]
NotebookOutlinePosition[    601622,      14562]
CellTagsIndexPosition[    601534,      14557]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Cellular Automata" :> 
          Documentation`HelpLookup["paclet:tutorial/CellularAutomata"]}, #, 
         2]& , Evaluator -> Automatic], {
       1->"\<\"Cellular Automata\"\>"}, "\<\"tutorials\"\>", 
       StyleBox["\<\"tutorials\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"ListConvolve" :> 
          Documentation`HelpLookup["paclet:ref/ListConvolve"], 
          "TuringMachine" :> 
          Documentation`HelpLookup["paclet:ref/TuringMachine"], "Partition" :> 
          Documentation`HelpLookup["paclet:ref/Partition"], "BitXor" :> 
          Documentation`HelpLookup["paclet:ref/BitXor"], "BitShiftLeft" :> 
          Documentation`HelpLookup["paclet:ref/BitShiftLeft"], "ArrayPlot" :> 
          Documentation`HelpLookup["paclet:ref/ArrayPlot"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"ListConvolve\"\>", 
       2->"\<\"TuringMachine\"\>", 3->"\<\"Partition\"\>", 
       4->"\<\"BitXor\"\>", 5->"\<\"BitShiftLeft\"\>", 
       6->"\<\"ArrayPlot\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Bitwise Operations" :> 
          Documentation`HelpLookup["paclet:guide/BitwiseOperations"], 
          "Computational Systems" :> 
          Documentation`HelpLookup[
           "paclet:guide/ComputationalSystemsAndDiscovery"], 
          "Cryptographic Number Theory" :> 
          Documentation`HelpLookup["paclet:guide/CryptographicNumberTheory"], 
          "Discrete Mathematics" :> 
          Documentation`HelpLookup["paclet:guide/DiscreteMathematics"], 
          "Image Processing" :> 
          Documentation`HelpLookup["paclet:guide/ImageProcessing"], 
          "Precollege Education" :> 
          Documentation`HelpLookup["paclet:guide/PrecollegeEducation"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Bitwise Operations\"\>", 
       2->"\<\"Computational Systems\"\>", 
       3->"\<\"Cryptographic Number Theory\"\>", 
       4->"\<\"Discrete Mathematics\"\>", 5->"\<\"Image Processing\"\>", 
       6->"\<\"Precollege Education\"\>", 
       7->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["CellularAutomaton", "ObjectName"], Cell[BoxData[
     TooltipBox[
      TogglerBox[Dynamic[
        FrontEnd`CurrentValue[
         FrontEnd`EvaluationNotebook[], {
         TaggingRules, "ModificationHighlight"}]], {True->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVVly2zAMDQhxEylSsizLaxJn73KafvQAuUDv/1eApJuosceeTDAWF4B8
eFgk71///HoVV1dXK3p+0wNfIzdCLL4I6p1sYBYuPy3SI2AmRBpuaE/TPRsE
3Isg8mrB9l6ILQhhLkbd93MBuwVxAthdZwPr5wVVzBc8wAvMX8TFqPkqE8q6
rDCsKPjZ/g1guz+PKvLFfGsDU1R4A8zDhgLangX9x/WGM7A1qSQ7rraYkaLf
LHZ5FWY9H7rew/n4/6vWPKT2mXFBaH/NGejzqi/VnFEFJ2Rb1Lo+jtofllPD
RVJjdZLr51Eb/ED1K+QI109LcIg+sdSIWqa5g85L6MiNoRloTws+pdBKiFGl
mx3N0jcAlcsKeFivVkteVOgMSNsC18sYS/iIMdZoK43WSdQVK1SLlYy09bZ2
aEO+aoPiACNak0EBbtfLRIz9KJRcroaSKwE11BprpRHqiJ0BrxtJG4pLYqWc
RZO4BQwSbGT4rFivaLjjQdscjSvlih48Mu+WDisOBcB5ehgeDEZQlikkKh3b
KA1FMTDg9yFxLSe4XC37begHTDh5seS1I/gaW025Yo/yUFbmSalShzon1Kch
JTNn2hKS1hyWVIkf5n2LHEQgt0zeUNTkMaYrDdkMZdo3fJCDHx7hxzAC+8rs
XYBokTPfEJk6JVlZx3Yl0UnTYMPvSUt6zghf8dhIi5VDnwO+Hcb1MHK1msK+
oy5qkYS7wVC3cfiq8ojBcMtFMJXnDiIcLPFJClDFGkJBhXGEZV6VcCQFoey7
tyvBF6lw0uKH6nyQ5fqhrKx/Kxe8+3coucpGN4U5+RYux7vnhGO5P7hKenqz
xSO3ivjisf345fg5Pt1ScCRkkzR1U1Dur+NCec4LZ49Ynx9P04GmOvlnqcKZ
79BfJfcjkg==
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlLlKQ0EYRi/iA9gpWIgvIXbjM1iKWAjW2vgeZnWN+AgWZhEULCxELFyi
ZnfJavbeJOr/33xXcj8CNnY5EA7k3GWYmTuzq+uLa2OWZS38/MYt0DbDHH8Q
8uTB64bRNrmsQkaPRoQSGX1ZqZPR9/eEazL6klIlowf8wiUZ3ecV3snowYDw
SEafUmJk9HmlQh4cX4GM7vUId2T0VFIho28or2T0w5DQIKN3PoUaGX1FaZLR
0ymh6sz7hHLk9PMzoU5Gz6SFmvFsCbd9Xzj9qyc0nfe49xfuTyYUEzoQbmh8
+N8e37HTbVpk9MSzUCajbypFMvq0UiOjzyklMnqvKzRMJCwU+/vjzemTSpns
nh97feJmd0e4cvppTKiQ0bsdoW5ecorZDgpPtD6xqFA2fp+Qp/vtdUj1v797
Oqdwnb2Ov++fUVpk9PCJUCC7n4t5wz5xj9/+rj6ceXbPP86NbEb58/k56qP+
n/0bdGyYqQ==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}], False->
       GraphicsBox[RasterBox[CompressedData["
1:eJytVdd22zAMDUBxSaJm5HgljrPTJN177/nWh35AfqD//1aAVGrrJLbbk+BI
FEmQFxcXlDQ9//3zHNfW1gZ0/6IbbsYOhahvCGrO9qHM/mnhmMMLoKv0RD6V
cPyZHTTcDI5NkYnQq4UQ0BdiAkKYZagl1OOw5fWtcgyvPp7AnefsGG0DBIeA
9RZVrNfcwBasb4nlbP9u4cbc/XD/zXcIgC2qYYazJTQiGSYHS0Hr8Yu5LV/h
ePvR5oMOKswAQ0M8R5OloFMWSNRE+IAVeALv38KX26wAV1uU5Ojv16PQy8q+
gMP+9ABW5C9KbkZi6qv19DGc/ID9b+CrZTjbbVagH3p91oIcL+GZ6JDNUet4
HpWUXxx5BaWZxRh1FQgH6JqoKcarF/2/dblez7IE0XmWGlFL/6ygchIqCmPo
CTSmDq9SaCUUhfI7K3pKlwJESZiAveHGxoA7ESYGpM2B62WMJXzEoojRRhpt
IlFHPKFyjGRBQ2fjBG0WttpMcYIFWhNAAXaHA0+M4yiUXK6UxJWAGmKNsdII
cYGVAadTSQPKS2KkEovGc8swk2ALhg8Tww1qdrjRNmSTtOUqHDhk3jktVpwK
QOLoZngwWICyTMFTqdhHMrQTDQPeazzXdgWXK+e4KV3AhH0US1Ergo8x16QV
R5QXZWWeJJW6qLNHPWq8mEFpS0hac1pSeX4YxjlyEhmFZfKGsqaIhd+Sks+Q
0i7lhZx8cwYPmx5wrMA+yaCwyMqnRCb2IiubsF9JTKRJMeX3JKd5VoS3OEyl
xShBFxLebXrDpsfVSlv2FZ2iHMn4NBg6bZy+ihxiZvjIFWAixyeIcLDNT1KC
qogha1Gh14NB6LXpSEpC2bm3y8O3FmHniF9U55INhnttz7pZuWDu79BqFZxJ
F2bhWzjo7Zx6HMvng6ukuztzvGJXa66NmF/+crzrHe1ScmTkk/SouqB8vq42
0jl0EnuF9/RsMR1Io4U/S5Wt+A79AaDyK/w=
         "], {{0, 0}, {86, 25}},
         ColorFunction->(Apply[RGBColor, 
           Part[CompressedData["
1:eJztlUtrE1EYhgfxBwguFBTEvyAiFhfzHxRFvC0E17rQhYLiT7D3u/gTXNg2
3sBFFyIutE1qm17UZJKZZJK0KFjBtur7Td4jnhe7dTMPhAfyQObMnO9Mjl69
fvraniAI7v/+7A3IRri7/8VGWJwHVTH72qohZp+ZBnUx+yWjJWYfHwNvxOwX
jKaYfXAAzIrZ+/tARcw+NAhKYvaTRiJmP2gUxH+vLxKzl5cMMXtfL3gvZn84
Cdpi9hvGJzH7yxegJWa/YnTE7Mtl0HTPfZ/xyPWVZZCGvQ/Au65fSV9aNMLJ
CfDW9a0fIBWz/9wBHbcOf/74PX83W99j128ZNTH71BMQidmfFkAiZj9irIvZ
Fz+AWMx+yEjF7KeMVOw/32x/iuHoCHjt+s42aIfTU6DWna/Prm9vgVb4cc0I
h4fAgusZ62L2A0YsZj9h1MX+9bN9LHfP75zrZ42a2L9/3+wXjVTM7q8jm9NE
5q8wA+JwoB/8eX+eNxIx+00jErP3GA2xvy7uG88R+z2jI2Y/bLTE/v1l76WG
mxP/+r7ZN7+Bupj9mFERsz9/Bppi9myeEzH7QgnUxOy3jYaY/a7RFrNfNtpi
9nNGXcw+NgoaYnbOk2///PJ/bXXFcP37JojF7CPDIBazf/0CKmL2+TlQEbPf
MZpi9olxkIrZS0UQidn3G1Ux+3EjErOfMapi9mBX8p73vOc973n/P/0XnFJ2
Rg==
            "], #]]& ),
         ColorFunctionScaling->False],
        Background->None,
        ImageSize->{86, 25},
        PlotRange->{{0, 86}, {0, 25}}]}, "\"\<Click me\>\""],
      "Toggle changes in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["CellularAutomaton",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/CellularAutomaton"], "[", 
       RowBox[{
        StyleBox["rule", "TI"], ",", 
        StyleBox["init", "TI"], ",", 
        StyleBox["t", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]generates a list representing the evolution of the \
cellular automaton with the specified rule from initial condition ",
     Cell[BoxData[
      StyleBox["init", "TI"]], "InlineFormula"],
     " for ",
     Cell[BoxData[
      StyleBox["t", "TI"]], "InlineFormula"],
     " steps. "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["CellularAutomaton",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/CellularAutomaton"], "[", 
       RowBox[{
        StyleBox["rule", "TI"], ",", 
        StyleBox["init", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the result of evolving ",
     Cell[BoxData[
      StyleBox["init", "TI"]], "InlineFormula"],
     " for one step. "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["CellularAutomaton",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/CellularAutomaton"], "[", 
       RowBox[{
        StyleBox["rule", "TI"], ",", 
        StyleBox["init", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["tspec", "TI"], ",", 
          StyleBox["xspec", "TI"], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives only those parts of the evolution specified by ",
     Cell[BoxData[
      StyleBox["tspec", "TI"]], "InlineFormula"],
     ", ",
     Cell[BoxData[
      StyleBox["xspec", "TI"]], "InlineFormula"],
     ", etc. "
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["CellularAutomaton",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/CellularAutomaton"], "[", 
       RowBox[{
        StyleBox["rule", "TI"], ",", 
        StyleBox["init", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["t", "TI"], ",", 
          ButtonBox["All",
           BaseStyle->"Link",
           ButtonData->"paclet:ref/All"], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]includes at each step all cells that could be affected \
over the course of ",
     Cell[BoxData[
      StyleBox["t", "TI"]], "InlineFormula"],
     " steps. "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, {None}}, "RowsIndexed" -> {}}},
 CellID->24448]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 "Possible forms for ",
 Cell[BoxData[
  StyleBox["rule", "TI"]], "InlineFormula"],
 " are: "
}], "Notes",
 CellID->9303],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], 
    StyleBox["n", "TI"], Cell[TextData[{
     Cell[BoxData[
      FormBox[
       RowBox[{"k", "=", "2"}], TraditionalForm]], "InlineMath"],
     ", ",
     Cell[BoxData[
      FormBox[
       RowBox[{"r", "=", "1"}], TraditionalForm]], "InlineMath"],
     ", elementary rule with rule number ",
     Cell[BoxData[
      StyleBox["n", "TI"]], "InlineFormula"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      StyleBox["n", "TI"], ",", 
      StyleBox["k", "TI"]}], "}"}], Cell[TextData[{
     "general nearest\[Hyphen]neighbor rule with ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     " colors "
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      StyleBox["n", "TI"], ",", 
      StyleBox["k", "TI"], ",", 
      StyleBox["r", "TI"]}], "}"}], Cell[TextData[{
     "general rule with ",
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     " colors and range ",
     Cell[BoxData[
      StyleBox["r", "TI"]], "InlineFormula"]
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      StyleBox["n", "TI"], ",", 
      StyleBox["k", "TI"], ",", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["r", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["r", "TI"], 
         StyleBox["2", "TR"]], ",", 
        StyleBox["\[Ellipsis]", "TR"], ",", 
        SubscriptBox[
         StyleBox["r", "TI"], 
         StyleBox["d", "TI"]]}], "}"}]}], "}"}], Cell[TextData[{
     Cell[BoxData[
      StyleBox["d", "TI"]], "InlineFormula"],
     "\[Hyphen]dimensional rule with ",
     Cell[BoxData[
      FormBox[
       RowBox[{
        RowBox[{"(", 
         RowBox[{
          RowBox[{"2", " ", 
           SubscriptBox["r", "1"]}], "+", "1"}], ")"}], "\[Cross]", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"2", " ", 
           SubscriptBox["r", "2"]}], "+", "1"}], ")"}], "\[Cross]", 
        "\[Ellipsis]", "\[Cross]", 
        RowBox[{"(", 
         RowBox[{
          RowBox[{"2", " ", 
           SubscriptBox["r", "d"]}], "+", "1"}], ")"}]}], TraditionalForm]], 
      "InlineMath"],
     " neighborhood "
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       StyleBox["n", "TI"], ",", 
       StyleBox["k", "TI"], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          SubscriptBox[
           StyleBox["off", "TI"], 
           StyleBox["1", "TR"]], "}"}], ",", 
         RowBox[{"{", 
          SubscriptBox[
           StyleBox["off", "TI"], 
           StyleBox["2", "TR"]], "}"}], ",", 
         StyleBox["\[Ellipsis]", "TR"], ",", 
         RowBox[{"{", 
          SubscriptBox[
           StyleBox["off", "TI"], 
           StyleBox["s", "TI"]], "}"}]}], "}"}]}], "}"}], 
     StyleBox[" ", "TR"]}], Cell[
    "rule with neighbors at specified offsets ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      StyleBox["n", "TI"], ",", 
      StyleBox["k", "TI"], ",", 
      StyleBox["rspec", "TI"], ",", 
      StyleBox["s", "TI"]}], "}"}], Cell[TextData[{
     "order\[Hyphen]",
     Cell[BoxData[
      StyleBox["s", "TI"]], "InlineFormula"],
     " rule "
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      StyleBox["n", "TI"], ",", 
      RowBox[{"{", 
       RowBox[{
        StyleBox["k", "TI"], ",", "1"}], "}"}]}], "}"}], Cell[TextData[{
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     "\[Hyphen]color nearest\[Hyphen]neighbor totalistic rule"
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      StyleBox["n", "TI"], ",", 
      RowBox[{"{", 
       RowBox[{
        StyleBox["k", "TI"], ",", "1"}], "}"}], ",", 
      StyleBox["r", "TI"]}], "}"}], Cell[TextData[{
     Cell[BoxData[
      StyleBox["k", "TI"]], "InlineFormula"],
     "\[Hyphen]color range ",
     Cell[BoxData[
      StyleBox["r", "TI"]], "InlineFormula"],
     " totalistic rule "
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      StyleBox["n", "TI"], ",", 
      RowBox[{"{", 
       RowBox[{
        StyleBox["k", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          SubscriptBox[
           StyleBox["wt", "TI"], 
           StyleBox["1", "TR"]], ",", 
          SubscriptBox[
           StyleBox["wt", "TI"], 
           StyleBox["2", "TR"]], ",", 
          StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "}"}], ",", 
      StyleBox["rspec", "TI"]}], "}"}], Cell[TextData[{
     "rule in which neighbor ",
     Cell[BoxData[
      StyleBox["i", "TI"]], "InlineFormula"],
     " is assigned weight ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["wt", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"],
     " "
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       SubscriptBox[
        StyleBox["lhs", "TI"], 
        StyleBox["1", "TR"]], "->", 
       SubscriptBox[
        StyleBox["rhs", "TI"], 
        StyleBox["1", "TR"]]}], ",", 
      RowBox[{
       SubscriptBox[
        StyleBox["lhs", "TI"], 
        StyleBox["2", "TR"]], "->", 
       SubscriptBox[
        StyleBox["rhs", "TI"], 
        StyleBox["2", "TR"]]}], ",", 
      StyleBox["\[Ellipsis]", "TR"]}], "}"}], Cell[
    "explicit replacements for lists of neighbors ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      StyleBox["fun", "TI"], ",", 
      RowBox[{"{", "}"}], ",", 
      StyleBox["rspec", "TI"]}], "}"}], Cell[TextData[{
     "function ",
     Cell[BoxData[
      StyleBox["fun", "TI"]], "InlineFormula"],
     " to apply to each list of neighbors "
    }], "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {
    None, None, None, None, None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{
     True, True, True, True, True, True, True, True, True, True, True, 
      True}}}},
 CellID->24715],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["CellularAutomaton",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/CellularAutomaton"], "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      StyleBox["n", "TI"], ",", 
      StyleBox["k", "TI"]}], "}"}], ",", 
    StyleBox["\[Ellipsis]", "TR"]}], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["CellularAutomaton",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/CellularAutomaton"], "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      StyleBox["n", "TI"], ",", 
      RowBox[{"{", 
       RowBox[{
        StyleBox["k", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{
           StyleBox["k", "TI"], "^", "2"}], ",", 
          StyleBox["k", "TI"], ",", "1"}], "}"}]}], "}"}]}], "}"}], ",", 
    StyleBox["\[Ellipsis]", "TR"]}], "]"}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->22959],

Cell["Common forms for 2D cellular automata include: ", "Notes",
 CellID->15156],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      StyleBox["n", "TI"], ",", 
      RowBox[{"{", 
       RowBox[{
        StyleBox["k", "TI"], ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1", ",", "1"}], "}"}]}], "}"}], Cell[
    "9\[Hyphen]neighbor totalistic rule ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    StyleBox[
     RowBox[{"{", 
      RowBox[{
       StyleBox["n", "TI"], ",", 
       RowBox[{"{", 
        RowBox[{
         StyleBox["k", "TI"], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"0", ",", "1", ",", "0"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"1", ",", "1", ",", "1"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"0", ",", "1", ",", "0"}], "}"}]}], "}"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", "1"}], "}"}]}], "}"}],
     GridElementStyleOptions -> {ColumnAlignments -> Left}], 
    "\[SpanFromLeft]"},
   {Cell["      ", "TableRowIcon"], "", Cell[
    "5\[Hyphen]neighbor totalistic rule ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    StyleBox[
     RowBox[{"{", 
      RowBox[{
       StyleBox["n", "TI"], ",", 
       RowBox[{"{", 
        RowBox[{
         StyleBox["k", "TI"], ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"0", ",", 
             StyleBox["k", "TI"], ",", "0"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{
             StyleBox["k", "TI"], ",", "1", ",", 
             StyleBox["k", "TI"]}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"0", ",", 
             StyleBox["k", "TI"], ",", "0"}], "}"}]}], "}"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", "1"}], "}"}]}], "}"}],
     GridElementStyleOptions -> {ColumnAlignments -> Left}], 
    "\[SpanFromLeft]"},
   {Cell["      ", "TableRowIcon"], "", Cell["\<\
5\[Hyphen]neighbor outer totalistic rule \
\>", "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, {None}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, False, True, False, True}}}},
 CellID->27972],

Cell["\<\
The number of possible cellular automaton rules is as follows: \
\>", "Notes",
 CellID->23274],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], Cell["elementary rules", "TableText"], 
    Cell["256 ", "TableText"]},
   {Cell["      ", "TableRowIcon"], Cell["1D general rules", "TableText"], 
    Cell[TextData[Cell[BoxData[
     FormBox[
      SuperscriptBox["k", 
       SuperscriptBox["k", 
        RowBox[{"2", "r", "+", "1"}]]], TraditionalForm]], "InlineMath"]], 
     "TableText"]},
   {Cell["      ", "TableRowIcon"], Cell["1D totalistic rules", "TableText"], 
    Cell[TextData[Cell[BoxData[
     FormBox[
      SuperscriptBox["k", 
       RowBox[{
        RowBox[{"(", 
         RowBox[{"k", "-", "1"}], ")"}], 
        RowBox[{"(", 
         RowBox[{"2", "r", "+", "1"}], ")"}], "+", "1"}]], TraditionalForm]], 
     "InlineMath"]], "TableText"]},
   {Cell["      ", "TableRowIcon"], Cell["2D general rules", "TableText"], 
    Cell[TextData[Cell[BoxData[
     FormBox[
      SuperscriptBox["k", 
       SuperscriptBox["k", 
        SuperscriptBox[
         RowBox[{"(", 
          RowBox[{"2", "r", "+", "1"}], ")"}], "2"]]], TraditionalForm]], 
     "InlineMath"]], "TableText"]},
   {Cell["      ", "TableRowIcon"], Cell[
    "2D 9\[Hyphen]neighbor totalistic rules", "TableText"], Cell[TextData[
    Cell[BoxData[
     FormBox[
      SuperscriptBox["k", 
       RowBox[{"9", 
        RowBox[{"(", 
         RowBox[{"k", "-", "1"}], ")"}], "+", "1"}]], TraditionalForm]], 
     "InlineMath"]], "TableText"]},
   {Cell["      ", "TableRowIcon"], Cell[
    "2D 5\[Hyphen]neighbor totalistic rules", "TableText"], Cell[TextData[
    Cell[BoxData[
     FormBox[
      SuperscriptBox["k", 
       RowBox[{"5", 
        RowBox[{"(", 
         RowBox[{"k", "-", "1"}], ")"}], "+", "1"}]], TraditionalForm]], 
     "InlineMath"]], "TableText"]},
   {Cell["      ", "TableRowIcon"], Cell["\<\
2D 5\[Hyphen]neighbor outer totalistic rules\
\>", "TableText"], Cell[TextData[Cell[BoxData[
     FormBox[
      SuperscriptBox["k", 
       RowBox[{"k", 
        RowBox[{"(", 
         RowBox[{"4", 
          RowBox[{"(", 
           RowBox[{"k", "-", "1"}], ")"}], "+", "1"}], ")"}]}]], 
      TraditionalForm]], "InlineMath"]], "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, True, True}}}},
 CellID->12134],

Cell[TextData[{
 "Normally, all elements in ",
 Cell[BoxData[
  StyleBox["init", "TI"]], "InlineFormula"],
 " and the evolution list are integers between 0 and ",
 Cell[BoxData[
  FormBox[
   RowBox[{"k", "-", "1"}], TraditionalForm]], "InlineMath"],
 ". "
}], "Notes",
 CellID->7892],

Cell[TextData[{
 "When a general function or replacement list is used, the elements of ",
 Cell[BoxData[
  StyleBox["init", "TI"]], "InlineFormula"],
 " and the evolution list can be any expressions. ",
 ButtonBox[" \[RightGuillemet]",
  BaseStyle->"ExampleLink",
  ButtonData->26557,
  ButtonNote->"26557"]
}], "Notes",
 CellID->31317],

Cell[TextData[{
 "Explicit replacement rules ",
 Cell[BoxData[
  RowBox[{
   SubscriptBox[
    StyleBox["lhs", "TI"], 
    StyleBox["i", "TI"]], "->", 
   SubscriptBox[
    StyleBox["rhs", "TI"], 
    StyleBox["i", "TI"]]}]], "InlineFormula"],
 " can contain patterns. "
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->22815],

Cell[TextData[{
 "In a 1D cellular automaton, replacement rules or an explicit function ",
 StyleBox["fun", "TI"],
 " are always taken to apply to a 1D list of neighbors. If the neighbors are \
specified by explicit offsets, they are given in the order of the offsets."
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->23759],

Cell[TextData[{
 "When the neighborhood in a multidimensional cellular automaton is defined \
by a range specification such as ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    SubscriptBox[
     StyleBox["r", "TI"], 
     StyleBox["1", "TR"]], ",", " ", 
    SubscriptBox[
     StyleBox["r", "TI"], 
     StyleBox["2", "TR"]], ",", " ", 
    StyleBox["\[Ellipsis]", "TR"]}], "}"}]], "InlineFormula"],
 ", the list of neighbors is taken to be a full array with dimensions ",
 Cell[BoxData[
  RowBox[{
   RowBox[{"2", 
    RowBox[{"{", 
     RowBox[{
      SubscriptBox[
       StyleBox["r", "TI"], 
       StyleBox["1", "TR"]], ",", " ", 
      SubscriptBox[
       StyleBox["r", "TI"], 
       StyleBox["2", "TR"]], ",", " ", 
      StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "+", "1"}]], 
  "InlineFormula"],
 ". "
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->21964],

Cell["\<\
If the neighbors in a multidimensional cellular automaton are specified by an \
explicit list of offsets, the neighbors are supplied in a one-dimensional \
list in the order of the offsets. \
\>", "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->29209],

Cell[TextData[{
 "If an explicit function ",
 Cell[BoxData[
  StyleBox["fun", "TI"]], "InlineFormula"],
 " is given, the first argument supplied to it is the list of neighbors. The \
second argument is the step number starting at 0. "
}], "Notes",
 CellID->3347],

Cell[TextData[{
 "In an order\[Hyphen]",
 Cell[BoxData[
  StyleBox["s", "TI"]], "InlineFormula"],
 " cellular automaton, specified by ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    StyleBox["rule", "TI"], ",", 
    StyleBox["kspec", "TI"], ",", 
    StyleBox["rspec", "TI"], ",", 
    StyleBox["s", "TI"]}], "}"}]], "InlineFormula"],
 ", each step depends on ",
 Cell[BoxData[
  StyleBox["s", "TI"]], "InlineFormula"],
 " preceding steps. "
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->7030],

Cell[TextData[{
 "Initial conditions are constructed from ",
 Cell[BoxData[
  StyleBox["init", "TI"]], "InlineFormula"],
 " as follows: "
}], "Notes",
 CellID->1413],

Cell[BoxData[GridBox[{
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      SubscriptBox[
       StyleBox["a", "TI"], 
       StyleBox["1", "TR"]], ",", 
      SubscriptBox[
       StyleBox["a", "TI"], 
       StyleBox["2", "TR"]], ",", 
      StyleBox["\[Ellipsis]", "TR"]}], "}"}], Cell[TextData[{
     "explicit list of values ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["a", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"],
     ", assumed cyclic "
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["a", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["a", "TI"], 
         StyleBox["2", "TR"]], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
      StyleBox["b", "TI"]}], "}"}], Cell[TextData[{
     "values ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["a", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"],
     " superimposed on a ",
     Cell[BoxData[
      StyleBox["b", "TI"]], "InlineFormula"],
     " background "
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["a", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["a", "TI"], 
         StyleBox["2", "TR"]], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["b", "TI"], 
         StyleBox["1", "TR"]], ",", 
        SubscriptBox[
         StyleBox["b", "TI"], 
         StyleBox["2", "TR"]], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}]}], "}"}], Cell[TextData[{
     "values ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["a", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"],
     " superimposed on a background of repetitions of ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["b", "TI"], 
       StyleBox["1", "TR"]]], "InlineFormula"],
     ", ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["b", "TI"], 
       StyleBox["2", "TR"]]], "InlineFormula"],
     ", \[Ellipsis] "
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            SubscriptBox[
             StyleBox["a", "TI"], 
             StyleBox["11", "TR"]], ",", 
            SubscriptBox[
             StyleBox["a", "TI"], 
             StyleBox["12", "TR"]], ",", 
            StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
          SubscriptBox[
           StyleBox["off", "TI"], 
           StyleBox["1", "TR"]]}], "}"}], ",", " ", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{
            SubscriptBox[
             StyleBox["a", "TI"], 
             StyleBox["21", "TR"]], ",", 
            StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
          SubscriptBox[
           StyleBox["off", "TI"], 
           StyleBox["2", "TR"]]}], "}"}], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
      StyleBox["bspec", "TI"]}], "}"}], "\[SpanFromLeft]"},
   {Cell["      ", "TableRowIcon"], "", Cell[TextData[{
     "values ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["a", "TI"], 
       StyleBox["ij", "TI"]]], "InlineFormula"],
     " at offsets ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["off", "TI"], 
       StyleBox["i", "TI"]]], "InlineFormula"],
     " on a background "
    }], "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["a", "TI"], 
         StyleBox["11", "TR"]], ",", 
        SubscriptBox[
         StyleBox["a", "TI"], 
         StyleBox["12", "TR"]], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        SubscriptBox[
         StyleBox["a", "TI"], 
         StyleBox["21", "TR"]], ",", 
        StyleBox["\[Ellipsis]", "TR"]}], "}"}], ",", 
      StyleBox["\[Ellipsis]", "TR"]}], "}"}], Cell[
    "explicit list of values in two dimensions ", "TableText"]},
   {Cell["      ", "TableRowIcon"], 
    RowBox[{"{", 
     RowBox[{
      StyleBox["aspec", "TI"], ",", 
      StyleBox["bspec", "TI"]}], "}"}], Cell[TextData[{
     "values in ",
     Cell[BoxData[
      StyleBox["d", "TI"]], "InlineFormula"],
     " dimensions with ",
     Cell[BoxData[
      StyleBox["d", "TI"]], "InlineFormula"],
     "\[Hyphen]dimensional padding "
    }], "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, 
   "Rows" -> {None, None, None, None, None, None, {None}}, 
   "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, False, True, True, True}}}},
 CellID->11813],

Cell[TextData[{
 "The first element of ",
 Cell[BoxData[
  StyleBox["aspec", "TI"]], "InlineFormula"],
 " is superimposed on the background at the first position in the positive \
direction in each coordinate relative to the origin. This means that ",
 Cell[BoxData[
  RowBox[{
   StyleBox["bspec", "TI"], "[", 
   RowBox[{"[", 
    RowBox[{"1", ",", "1", ",", 
     StyleBox["\[Ellipsis]", "TR"]}], "]"}], "]"}]], "InlineFormula"],
 " is aligned with ",
 Cell[BoxData[
  RowBox[{
   StyleBox["aspec", "TI"], "[", 
   RowBox[{"[", 
    RowBox[{"1", ",", "1", ",", 
     StyleBox["\[Ellipsis]", "TR"]}], "]"}], "]"}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->19603],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["CellularAutomaton",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/CellularAutomaton"], "[", 
   RowBox[{
    StyleBox["rule", "TI"], ",", 
    StyleBox["init", "TI"], ",", 
    StyleBox["t", "TI"]}], "]"}]], "InlineFormula"],
 " generates an evolution list of length ",
 Cell[BoxData[
  FormBox[
   RowBox[{"t", "+", "1"}], TraditionalForm]], "InlineMath"],
 "."
}], "Notes",
 CellID->591],

Cell[TextData[{
 "For an order\[Hyphen]",
 Cell[BoxData[
  StyleBox["s", "TI"]], "InlineFormula"],
 " cellular automaton, ",
 Cell[BoxData[
  StyleBox["init", "TI"]], "InlineFormula"],
 " is a list giving the initial ",
 Cell[BoxData[
  StyleBox["s", "TI"]], "InlineFormula"],
 " steps in the evolution of the system."
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->24629],

Cell[TextData[{
 "Time specifications ",
 Cell[BoxData[
  StyleBox["tspec", "TI"]], "InlineFormula"],
 " in ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    StyleBox["tspec", "TI"], ",", 
    StyleBox["xspec", "TI"], ",", 
    StyleBox["\[Ellipsis]", "TR"]}], "}"}]], "InlineFormula"],
 " can be as follows: "
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->29344],

Cell[BoxData[GridBox[{
   {Cell["\[FilledVerySmallSquare]", "TableRowIcon",
     FontColor->None], 
    StyleBox["t", "TI"], Cell[TextData[{
     "all steps ",
     Cell[BoxData[
      FormBox["0", TraditionalForm]], "InlineMath"],
     " through ",
     Cell[BoxData[
      StyleBox["t", "TI"]], "InlineFormula"]
    }], "TableText"]},
   {Cell["\[FilledVerySmallSquare]", "TableRowIcon",
     FontColor->None], 
    RowBox[{"{", 
     StyleBox["t", "TI"], "}"}], Cell[TextData[{
     "a list containing only step ",
     Cell[BoxData[
      StyleBox["t", "TI"]], "InlineFormula"]
    }], "TableText"]},
   {Cell["\[FilledVerySmallSquare]", "TableRowIcon",
     FontColor->None], 
    RowBox[{"{", 
     RowBox[{"{", 
      StyleBox["t", "TI"], "}"}], "}"}], Cell[TextData[{
     "step ",
     Cell[BoxData[
      StyleBox["t", "TI"]], "InlineFormula"],
     " alone"
    }], "TableText"]},
   {Cell["\[FilledVerySmallSquare]", "TableRowIcon",
     FontColor->None], 
    RowBox[{"{", 
     RowBox[{
      SubscriptBox[
       StyleBox["t", "TI"], 
       StyleBox["1", "TR"]], ",", 
      SubscriptBox[
       StyleBox["t", "TI"], 
       StyleBox["2", "TR"]]}], "}"}], Cell[TextData[{
     "steps ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["t", "TI"], 
       StyleBox["1", "TR"]]], "InlineFormula"],
     " through ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["t", "TI"], 
       StyleBox["2", "TR"]]], "InlineFormula"]
    }], "TableText"]},
   {Cell["\[FilledVerySmallSquare]", "TableRowIcon",
     FontColor->None], 
    RowBox[{"{", 
     RowBox[{
      SubscriptBox[
       StyleBox["t", "TI"], 
       StyleBox["1", "TR"]], ",", 
      SubscriptBox[
       StyleBox["t", "TI"], 
       StyleBox["2", "TR"]], ",", 
      StyleBox["dt", "TI"]}], "}"}], Cell[TextData[{
     "steps ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["t", "TI"], 
       StyleBox["1", "TR"]]], "InlineFormula"],
     ", ",
     Cell[BoxData[
      RowBox[{
       SubscriptBox[
        StyleBox["t", "TI"], 
        StyleBox["1", "TR"]], "+", 
       StyleBox["dt", "TI"]}]], "InlineFormula"],
     ", \[Ellipsis] "
    }], "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {
     Dynamic[
      If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
       RGBColor[0.92, 1, 0.59], None]], 
     Dynamic[
      If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
       RGBColor[0.92, 1, 0.59], None]], 
     Dynamic[
      If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
       RGBColor[0.92, 1, 0.59], None]], 
     Dynamic[
      If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
       RGBColor[0.92, 1, 0.59], None]], {
      Dynamic[
       If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
        RGBColor[0.92, 1, 0.59], None]]}}, "RowsIndexed" -> {}},
 GridBoxDividers->{"Rows" -> {{True, True, True, True, True, True}}}},
 CellID->31416],

Cell["The initial condition is considered to be at step 0. ", "Notes",
 CellID->6485],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   ButtonBox["CellularAutomaton",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/CellularAutomaton"], "[", 
   RowBox[{
    StyleBox["rule", "TI"], ",", 
    StyleBox["init", "TI"], ",", 
    RowBox[{"{", 
     StyleBox["tspec", "TI"], "}"}]}], "]"}]], "InlineFormula"],
 " uses the default ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 " for ",
 Cell[BoxData[
  StyleBox["xspec", "TI"]], "InlineFormula"],
 ". "
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->31227],

Cell[TextData[{
 "Space specifications ",
 Cell[BoxData[
  StyleBox["xspec", "TI"]], "InlineFormula"],
 " can be as follows: "
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->16133],

Cell[BoxData[GridBox[{
   {Cell["\[FilledVerySmallSquare]", "TableRowIcon",
     FontColor->None], 
    ButtonBox["All",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/All"], Cell["\<\
all cells that can be affected by the specified initial condition\
\>", "TableText"]},
   {Cell["\[FilledVerySmallSquare]", "TableRowIcon",
     FontColor->None], 
    ButtonBox["Automatic",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/Automatic"], Cell[
    "all cells in the region that differs from the background ", 
     "TableText"]},
   {Cell["\[FilledVerySmallSquare]", "TableRowIcon",
     FontColor->None], "0", Cell[TextData[{
     "cell aligned with beginning of ",
     Cell[BoxData[
      StyleBox["aspec", "TI"]], "InlineFormula"]
    }], "TableText"]},
   {Cell["\[FilledVerySmallSquare]", "TableRowIcon",
     FontColor->None], 
    StyleBox["x", "TI"], Cell[TextData[{
     "cells at offsets up to ",
     Cell[BoxData[
      StyleBox["x", "TI"]], "InlineFormula"],
     " on the right "
    }], "TableText"]},
   {Cell["\[FilledVerySmallSquare]", "TableRowIcon",
     FontColor->None], 
    RowBox[{"-", 
     StyleBox["x", "TI"]}], Cell[TextData[{
     "cells at offsets up to ",
     Cell[BoxData[
      StyleBox["x", "TI"]], "InlineFormula"],
     " on the left "
    }], "TableText"]},
   {Cell["\[FilledVerySmallSquare]", "TableRowIcon",
     FontColor->None], 
    RowBox[{"{", 
     StyleBox["x", "TI"], "}"}], Cell[TextData[{
     "cell at offset ",
     Cell[BoxData[
      StyleBox["x", "TI"]], "InlineFormula"],
     " to the right "
    }], "TableText"]},
   {Cell["\[FilledVerySmallSquare]", "TableRowIcon",
     FontColor->None], 
    RowBox[{"{", 
     RowBox[{"-", 
      StyleBox["x", "TI"]}], "}"}], Cell[TextData[{
     "cell at offset ",
     Cell[BoxData[
      StyleBox["x", "TI"]], "InlineFormula"],
     " to the left "
    }], "TableText"]},
   {Cell["\[FilledVerySmallSquare]", "TableRowIcon",
     FontColor->None], 
    RowBox[{"{", 
     RowBox[{
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["1", "TR"]], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["2", "TR"]]}], "}"}], Cell[TextData[{
     "cells at offsets ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["1", "TR"]]], "InlineFormula"],
     " through ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["2", "TR"]]], "InlineFormula"]
    }], "TableText"]},
   {Cell["\[FilledVerySmallSquare]", "TableRowIcon",
     FontColor->None], 
    RowBox[{"{", 
     RowBox[{
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["1", "TR"]], ",", 
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["2", "TR"]], ",", 
      StyleBox["dx", "TI"]}], "}"}], Cell[TextData[{
     "cells ",
     Cell[BoxData[
      SubscriptBox[
       StyleBox["x", "TI"], 
       StyleBox["1", "TR"]]], "InlineFormula"],
     ", ",
     Cell[BoxData[
      RowBox[{
       SubscriptBox[
        StyleBox["x", "TI"], 
        StyleBox["1", "TR"]], "+", 
       StyleBox["dx", "TI"]}]], "InlineFormula"],
     ", \[Ellipsis] "
    }], "TableText"]}
  }]], "2ColumnTableMod",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {
     Dynamic[
      If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
       RGBColor[0.92, 1, 0.59], None]], 
     Dynamic[
      If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
       RGBColor[0.92, 1, 0.59], None]], 
     Dynamic[
      If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
       RGBColor[0.92, 1, 0.59], None]], 
     Dynamic[
      If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
       RGBColor[0.92, 1, 0.59], None]], 
     Dynamic[
      If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
       RGBColor[0.92, 1, 0.59], None]], 
     Dynamic[
      If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
       RGBColor[0.92, 1, 0.59], None]], 
     Dynamic[
      If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
       RGBColor[0.92, 1, 0.59], None]], 
     Dynamic[
      If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
       RGBColor[0.92, 1, 0.59], None]], {
      Dynamic[
       If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
        RGBColor[0.92, 1, 0.59], None]]}}, "RowsIndexed" -> {}},
 GridBoxDividers->{
  "Rows" -> {{True, True, True, True, True, True, True, True, True, True}}}},
 CellID->9639],

Cell[TextData[{
 "In one dimension, the first element of ",
 Cell[BoxData[
  StyleBox["aspec", "TI"]], "InlineFormula"],
 " is taken by default to have space offset 0. "
}], "Notes",
 CellID->20594],

Cell[TextData[{
 "In any number of dimensions, ",
 Cell[BoxData[
  RowBox[{
   StyleBox["aspec", "TI"], "[", 
   RowBox[{"[", 
    RowBox[{"1", ",", "1", ",", "1", ",", 
     StyleBox["\[Ellipsis]", "TR"]}], "]"}], "]"}]], "InlineFormula"],
 " is taken by default to have space offset ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{"0", ",", "0", ",", "0", ",", 
    StyleBox["\[Ellipsis]", "TR"]}], "}"}]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->29739],

Cell[TextData[{
 "Each element of the evolution list produced by ",
 Cell[BoxData[
  ButtonBox["CellularAutomaton",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CellularAutomaton"]], "InlineFormula"],
 " is always the same size. "
}], "Notes",
 CellID->7204],

Cell[TextData[{
 "With an initial condition specified by an ",
 Cell[BoxData[
  StyleBox["aspec", "TI"]], "InlineFormula"],
 " of width ",
 Cell[BoxData[
  FormBox["w", TraditionalForm]], "InlineMath"],
 ", the region that can be affected after ",
 Cell[BoxData[
  FormBox["t", TraditionalForm]], "InlineMath"],
 " steps by a cellular automaton with a rule of range ",
 Cell[BoxData[
  FormBox["r", TraditionalForm]], "InlineMath"],
 " has width ",
 Cell[BoxData[
  FormBox[
   RowBox[{"w", "+", "2", "r", "t"}], TraditionalForm]], "InlineMath"],
 ". "
}], "Notes",
 CellID->11387],

Cell[TextData[{
 "If no ",
 Cell[BoxData[
  StyleBox["bspec", "TI"]], "InlineFormula"],
 " background is specified, space offsets of ",
 Cell[BoxData[
  ButtonBox["All",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/All"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 " will include every cell in ",
 Cell[BoxData[
  StyleBox["aspec", "TI"]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->23967],

Cell[TextData[{
 "A space offset of ",
 Cell[BoxData[
  ButtonBox["All",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/All"]], "InlineFormula"],
 " includes all cells that can be affected by the initial condition. "
}], "Notes",
 CellID->29415],

Cell[TextData[{
 "A space offset of ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 " can be used to trim off background from the sides of a cellular automaton \
pattern. "
}], "Notes",
 CellID->2922],

Cell[TextData[{
 "In working out how wide a region to keep, ",
 Cell[BoxData[
  ButtonBox["Automatic",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Automatic"]], "InlineFormula"],
 " only looks at results on steps specified by ",
 Cell[BoxData[
  SubscriptBox[
   StyleBox["off", "TI"], 
   StyleBox["t", "TI"]]], "InlineFormula"],
 ". "
}], "Notes",
 CellID->384],

Cell[TextData[{
 Cell[BoxData[
  RowBox[{
   RowBox[{"CellularAutomaton", "[", 
    StyleBox["rule", "TI"], "]"}], "[", 
   StyleBox["init", "TI"], "]"}]], "InlineFormula"],
 " is equivalent to ",
 Cell[BoxData[
  RowBox[{
   ButtonBox["CellularAutomaton",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/CellularAutomaton"], "[", 
   RowBox[{
    StyleBox["rule", "TI"], ",", 
    StyleBox["init", "TI"]}], "]"}]], "InlineFormula"],
 ". "
}], "Notes",
 Background->Dynamic[
   If[CurrentValue[{TaggingRules, "ModificationHighlight"}] === True, 
    RGBColor[0.92, 1, 0.59], None]],
 CellID->17996]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->12012548],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(2)", "ExampleCount"]
}], "ExampleSection",
 CellID->524365467],

Cell["Run rule 30 for 2 steps:", "ExampleText",
 CellID->30991],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"30", ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "1", ",", "0", ",", "0", ",", "0"}],
     "}"}], ",", "2"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->406],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "1", ",", "0", ",", "0", ",", "0"}],
     "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "1", ",", "1", ",", "0", ",", "0"}],
     "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "1", ",", "0", ",", "0", ",", "1", ",", "0"}],
     "}"}]}], "}"}]], "Output",
 ImageSize->{431, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->21442]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1323],

Cell["Run for 50 steps from a single 1 on a background of 0s:", "ExampleText",
 CellID->30617],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"30", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "50"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->20737],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy1lVFSwzAMRCVuwhm4CUdghm/u/0eLrd2VsWM3iUPJtImj9dNKyvvXz+fX
m5l9PP6/7Xm4Pz7PU/3+OD//vP76u1dXxMVyo1zFfSsRLBaXBYxZbnpdEI8b
49TdaKi4+vcLUjVK2WeNkjaHBVgXcTxwPQHWy+YiFYtlg7I5wwKsi1wiB1ge
YQ3wErMBTZsb+IIoVHGgCTyhkQpNAFm6voSd4gDRAE9opCInYOKLR0KQO0FD
4VnwMxU5ARNfkH4XeT5pjImNwtJWZdovJvLMHsPDisi8t8dKv6h8fCS8bJQ3
+ixH/aLykT0JHxXiodRVmfQL86XZU9vdRXakMukXXmb2JDxaZsQx9cU1X6bM
6I3I85hj6oucPDNHAbLYDjTm/ZKnFtqSbdjtjxHLqF8wPGBxJeMUnnAoy6Bf
MDyMsODwyPSKykG/eAyP+IKxiOcXNGb94t58Mczlw/4YsfR8MUkSTAq2JT9a
lo4voiylxIJb1jjyRZWl/Vhwr6n0fUnKrLD++2OVpfUlKTsrbK0/RiytLwzK
CgPfGZWOLzJNWGpl/QmNkS8yTZDFV/pjxKK+mEri9XWWQ1nUF5kmGO/nOZRF
fUHHsNMvcCiLjlyO/MC4ptHzhSO/il/kUBadJhr8OgdCuvqSg9+h8d+X6MfX
3h8LKsmX6JNr/dFRSb6A6zaOqqK+ROZu5Kgq6ks4c6tG68vZ98eCSop+P0dV
SdF3aCRfNvghKuHLJo6qgjG/79Axv1Vlox+ispmjqOzXOMXxC/ECEP0=
   "], {{0, 0}, {101, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->9375]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(66)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->9806],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "One-Dimensional Rules",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(25)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->26724],

Cell["Elementary rule 73:", "ExampleText",
 CellID->10293],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"73", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "4"}], "]"}], "//", 
  "Grid"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->28596],

Cell[BoxData[
 TagBox[GridBox[{
    {"0", "0", "0", "0", "1", "0", "0", "0", "0"},
    {"1", "1", "1", "0", "0", "0", "1", "1", "1"},
    {"0", "0", "1", "0", "1", "0", "1", "0", "0"},
    {"1", "0", "0", "0", "0", "0", "0", "0", "1"},
    {"1", "0", "1", "1", "1", "1", "1", "0", "1"}
   }],
  "Grid"]], "Output",
 ImageSize->{131, 73},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->30927]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"73", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "50"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->8548],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy1V0FywzAIhP6kb+hP+oTO9Nz/35qOY7G7gCwljWccOxgBu4uI8/718/n1
ZmYft/P7dpqb/x3HXdz7/dtpO+7sPMQ+Ps3Q97zHQJTo+D6exyInX/YrEhx2
SyjGI0wXARJi8JMEd6aGKSVy53QRICEOyiq6jmujiwUVXCHRRX41XcS7JgIq
hGOqkShLeozkjS5IhS5Ca24O0GMgaXQJ0WURFMV+hR6BZNLGTJdpUeKX9dBl
dRszXako8iv0QCS1Lo7OwXyyVu2LoY3z5TYOZ94nZO31YCRtG2sZTYhGD3Eu
dSFn2SdobfVQJJUuqIPuE7ISGg1rOSclimRpn/CIbPTISLo2TkqgTRMojpSz
0KVWQp70elRICl1wcEgxOAAbPUokhS44OLCYZvo2ETMWSkSDw0tbr0eTN+kS
VZv0lLZvH6/GUkzj3FPavl24+iC6nMjxwjbRo0dStvEgEkfnpR6T3KoL/ZrT
6IQEs1g9Fmlj7Smw93pMkYguOL3G5FrRY46kamOmSNv3ESTy6uNCEbbvdZw5
Fm5jSr+kxyUS0oWnl7TvQpQ5lgiDsytKuMJxWYOJFoZyL+mxggR14dm1qMcS
Enolpdm1pscaEmnjQSE+eR4JYTjTLuuxisSB/dg1tqbHMhJkH657q9ewAFVs
+S8kJlSt67GDBAe87+ixhST/ddtcuY7FsX1fgUTftl6DJN5ZdpftHU+t2sGy
i2O7JtvW4xEkvo/D/RcCrA2Y
   "], {{0, 0}, {101, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->29670]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->4046],

Cell["3-color rule 679458:", "ExampleText",
 CellID->22842],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"679458", ",", " ", "3"}], "}"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", " ", "0"}], "}"}], ",", " ", "50"}], 
   "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->24383],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy9WFFShUAM2/YmnsGbeARn/Pb+f/LYNk27DG92AZlRkQfJpkkA/fj+/frW
1trn9vWzfYmqNGkqotvutv/6tm+vQ7r/7Hs48XWsb/20Zr/jaNuPSQfTHRow
fM2O71faHvM5nvrqMrr4J335DMPXYLlYaOJroeUQHZ8AOuk3XMGlsfA4cR+A
rVIO0V1vQMe4Fbhxqca6leaMVR6ja8wqBLDRdgEbRSfCeuganYdejY3GDRvo
UlgoriGF7CxXvNG4YUMxSolZcsjOcpVYjvrS+orYQkqam/8mV0XL2BdxNoq0
EYvA/De5KlrGvpjRNirTBQI9RB9yVbSMfQFlh28K69n8N7kqWsa+0EKhi8cF
9LNcFS1jX1L7Ob6S0U9zVbSMfUnt5/jO5KpoGftC7W9KCqZyVbSUvuT2v2ig
YCpXRUvuS2m/2zWfq6Il9yWWCOsjcDO5KlpyX2KJQics5KpoSX1p3Ylk/Vqu
ipbUF3PCBLUcuKlcFS3cF3fCcCQHzo9PzApauC9ACetXc1W0cF+AIgG8mKui
JfpSrKcHwEKuipboS7aeHwALuSpayO1kvUtVHF7kKH3RZH2gL+UqsVBfegkp
B5dylVg0+lKsl2u5Sizl0WvUeJhdnZVhR1+q9RdzlViiL9X6i7lKLAyUrb9l
Vg5FQHTD15tm5VD1z8W7cpVYoi84cuesDFMjUnTDv5Uj9YWf9bdyhC/5hn/3
Zr7EfwYe4IAv5YZ/O4vgvfKRWRmLv1Y8NCtj8XE9yKH8ov8kS7zoP8ny8KyM
5R844h45sf0BObolJw==
   "], {{0, 0}, {101, 51}}, {0, 2}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->14663]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{
   RowBox[{"CellularAutomaton", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"679458", ",", " ", "3"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", "1", "}"}], ",", " ", "0"}], "}"}], ",", " ", "50"}], 
    "]"}], ",", 
   RowBox[{"ColorRules", "->", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"1", "->", "Pink"}], ",", 
      RowBox[{"2", "->", "Blue"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->28182],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztW21vE0cQvvh8d/bFCdCKNoK2QIsoiL4gaEVB0BbRhjelL4jCRxShSvCh
akX5zl/iA/9vuM3t45t9dtZ2ITR2TCTHe3e7s/PszjwzO5dsbT97/Odf28+e
PNo+fuPp9j+Pnzz69/jm30+bW/lKlmUvm8/DjaxpS9NqP7mIa7ifF81npb3M
3bOdn+xF82ul7fDmd4vxXffz/HnWdx1y13L6SNZrmj3o1bTdE3dPXxdNC/1c
u3bt5tuJcd0Kf92j+z317ca2cscKZYPmetQ0KjW+fd7pMfD3R5Hecf88ktPO
U3md+uZq9RIICqWZnhmSdhpKhrusvIxCXReBVuEqY3UOqNXqU7+RgSiFnMda
8lI7Xrxbnf+8On1CNEwgB8Ky+S6b75pWra/vKdlYpdxrNfIrBo+chNpCZqEv
SKberXeIJSu9pa96JAUhKbTFK3Zir8CKrHl5tZq9ImTwomoKIqwYEPFuYCWH
S4YMTGNFO62NhVAj0tFN37essVZjwCIDb3G52AxXJbwnhboWMFY3LzwpXoWh
TOfXglAV0vEsOD4Y10iv1bha+WuttO55u8JK1AkdRvtM81UJ+VHHPCtOulmR
48Dmx9ZGcQiWlIqxLp9al3S+xL49KVdaX0J0jr9Kv/spRmCPheVwLAF6xD8w
wKEZNWcWsmNbyKsHTVTrEvMSa8DcFmpkj8PeD8gW4C+VR95m/uls3ULKEZqR
H1pypAel47Ewg2ollWomp1Vu9GULsiJ94VHCXyvvh+yfqcwa6MABQy/r/SVC
9R5JhRSMxjUjsbxes8OYTz0CPXaovKAkrWfJKdzjwyYah9FFVs7LsR+4hval
dLwYc1aYHWEfLA2nReMP9oG2h2U6fxX03PIBnKu7nQ0tKGXvdTiv6Mwp8Euv
D+x+YwnQfCg2N+lYb3nwqujsP3121acK1589GEhmzRc400L7yBKi25Au3msv
ZzaARkA86XyKE4pmg1k1L6VjI443R00EDldwsk74Cjhen55jbu80trLIVIaC
eIUz7ccLrulHEvJTYK/MGQn7N/lFuvPpcCxbPd9ZkXhOK8NnRCPD5o8tIbpP
xPLQuDKts/WxdGMsrFBrwhbHcd41T5jaHZO4Rqr9f1XhjNeu4yWsD7RJRTK+
B7v/dAG1OyFhxrYG6RJy6cD3S9qpivL2OSEdRU4umGafGRK46slaBFWWCf3g
z4hb0Ihjl35PNAkVrKaU2JtOmehOKsl6vbkGqGfWrBJXrLuKDDQqJY4any+A
NqckXY/Tvsrcx7U5i3nPzMmMp6XLKPTK4QytpaTyTs48Uhyvd8kNObsgGp2R
LnNlj05lTSPSgiP0eGbv7dNy/CjCK67/wtT6LPmMlR/otWS/6Rt9gHDW90up
arTr/9U+0vpLiaM+8y981XrGfM18zvXvc3OsxdfScuxI7YZVERtImOfCt8+b
Us/JbPUO+EQqvwILaIbkPb4wpxqcl3QFN1fSeJ+4/7qEflNT/2/ncPZvIntt
Vzl+x8s8mj4B4y+teAzeY100NXH6IT4xK+vKs3Wu0+fp6Ixo2MF3c6rBRZnu
C1a+gmoj28jlPZzlUsKuwaDpaKPeZZLlXNnjmS5LWJPjbERnKmsSe4a7vmpK
vuJt74DEsTZVj//hLUv6nuxkUjWmkvC0kPL9a3s804+SrsFYtfhZ3mBfN2e6
pmwF3sP2kMqzrT36aQ9nuS5hfXRStgeP23zLkn6m3UrVD/QZz3pjeNOUvinx
yZi5w6wkNr9u/U8Sb9Le6zOSu76zi6Nu+30bv9FWZ4CtXRpxR+zzoOWbbid/
MaVs0TrjTfJvu9D7V7V2VsS8u4ujfpfubwdcr3tmr7sS5h5/vEGve34tHrzG
0/u0P3av1/t/gJfts5VXNofZZA==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->289680340]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->770],

Cell["3-color totalistic rule with code 867:", "ExampleText",
 CellID->14721],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"867", ",", " ", 
      RowBox[{"{", 
       RowBox[{"3", ",", " ", "1"}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", " ", "0"}], "}"}], ",", " ", "50"}], 
   "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->23071],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy1WFt2WjEMnNFOuobupEvIOfnu/v+KNSPZpoGYADdwgWtZo5c1dn59/P3z
EQB+X96flzcD4yfJGH/jO+LycfkOXu7E+BpjfPwIXWA/izFZspEzU0FqG5OR
qpgifjx+Jd6YD2kZUzROjFu+kEaFxBIPQi0FMjnNvHyG1MUcSJOtqXwMPQwZ
E0LRLwmBnhXTYLmecsOc9JCw0QqCNLHUBeS3JkjA0BS+Amv3QD+hI8yMibKA
DGzY5wwwKw/2wpaiJluICkzmmJqFmmBzrBkOrLx0gJdkOydrEbQQPUpns6Y0
ShVGB7ZMzpSF6y1cN/r0Yws5Tpo/S3S9ujCiK6fnoQwMdE66lqLzGF2BsZTo
jpJpSgOcavS8pd5gj2cttUWIrkBrusaQecSsHMzwd37bQi0zD1SiMCvQmv6/
cmF0vm01W5YuZ+g2K4LVKbrApqYvUDjqF+oMZfWUdQ+p2zowvVMMcFVXuy+u
YIyXobaAerGsyZZ1qR7ZoLKGb/jhkMgwKKpXstXzZrKndcoD3Jlu+uGQKGTu
eMA+7GRcBwOofuqA38PwEqgerhYcC5B6yOZgStArz6G6j6HKUPdRPwO6f3gF
0uuyHhJaG4RnfuOHfElXCE2nmxSsD6qI6aJEMiF03/oWQ3FSF6NLtZemywdh
+kJFyQkxD55c5ko68V5ltY6q+KqllIhK+35d7b6U0yMqqNaC5lLTnDnSIisP
nvnSxKesqnlMLtWq5C6Cg7rafWniw2zHbutu+t18sVLnI1cRn5a+7DR5hInF
vqLNOKur3RdzW8M1eVQdtOqWfRCjuXKBK54qBpuqv+bBM19Ce5epwvXgvjJ3
gTd48MwXJcTsVV0luq/QYzd58MwXovYXLDavWz3jPR488wXuMaiYmNKjn93n
wTNfYGtRm0q7OJ/d58EzXxjNddsO8mrsOZRYuM6uVEfZxp5DmVwHLXsu2/IT
HjxCCSs1mdHkZNAX+JEotdfX7pNRLHvMg0cofa6kqDxqo/psXW0orHNl9kMf
Fx7hwSOUPldq6xXbefBlKH2urL3dozx4hNLnSnxxHnwZSp8rvWhe7keq7384
/JQHj1DqlPVjHjxCEUnh5zx4hNLb1PdhmCvfUlcbCp/lwSOUt9XVhvJ2PxLl
8Sn/AKAyItc=
   "], {{0, 0}, {101, 51}}, {0, 2}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->6497]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->28668],

Cell[TextData[{
 "2-color range ",
 Cell[BoxData[
  FormBox[
   RowBox[{"r", "=", 
    RowBox[{"3", "/", "2"}]}], TraditionalForm]], "InlineMath"],
 " (two neighbors on the left and one on the right) rule 23898:"
}], "ExampleText",
 CellID->2040],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"23898", ",", "2", ",", 
      RowBox[{"3", "/", "2"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", " ", "0"}], "}"}], ",", " ", "40"}], 
   "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->8657],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJzFlVt2gzAMRDXdST+znS4hG+h3dx8K1sso5WFZhXAgIHvOtSzN5/P76/lB
RI/l+lkuAgi/N3lY7rT+QPzE/9tLe65DSSZaz20KcMDuOwty7DY16LTw9v5I
xn2XqPUFx7b4k8KseYd2O7Z4asPOE7fRt2hZmTWY/BzxCK0yE4+6QDxCq8yy
wy8Qj9C6PF8SHqV1eeYlPxCWuh+idXmGpPtP4hxal+c2FTr8nph7zxity7Om
G/1m8/nNoFVmqWfFD5c6i1aZpZ7btLrgfqmzaJVZ6plbN0RT0RJplVmNyuwj
2ePH1nBDFl09t+Euxy0sk1aZ3/kza2TTKrPsa+/PUnDZtMoc+7PZcsm0ymzW
1fTuebTKHPnzTFpllnq2xBNpWTny59gaEmXxzp9ja0iURezPJI0Ls2QR+/O+
WabLIvbn2bSsbLvGvllOkkXszwWyiP25QBaRP5fIYufPVbLo/blMFq6eK2Xh
6rlSFqaea2Vh6rlWFqaey49Wx/+inHy8AKV8EDU=
   "], {{0, 0}, {121, 41}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 66},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->20301]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->13426],

Cell[TextData[{
 "A general range ",
 Cell[BoxData[
  FormBox[
   RowBox[{"r", "=", "2"}], TraditionalForm]], "InlineMath"],
 " rule:"
}], "ExampleText",
 CellID->24618],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"39421042", ",", "2", ",", "2"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", " ", "0"}], "}"}], ",", " ", "40"}], 
   "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->8124],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJzNldFBwzAMRGU24ZN1GIEFmIlNKU2ke5JtEmjaxrQkdhzpfDpdXz8+3z9e
zOzt8v26fK0tn3b5XMbPZFm53lyvy4Pme9u6o8V0XVx2Xf+ab7JrTH8n7fC3
4hOhW2TxG80bMulq8VbEE6QGsDppYAES7DAr6IyhRYrYWd70EKKxpACJYBtg
A14sAIMpUkUHgFE6YyRLIUCj4IpEFSJqrBKtlUB9lDnfW4aJO7AHiLlCODqL
6YTmhK4EKYj1wceIIgm56Wlhr2cxUOBNpQ/JB3dUAo+IVivlg44Is2pvAJFa
RB0yb04AaIhz8IgWS7XKoaPgShmn8DKLUhgpUc3M2UxKqEcEOZl0aS5UO9He
nMW2smQzX/R/hcF8xNR6jRsKpdvs9Sxu+eIavjLIxGgc8pu6dlt7cxY9pWgM
JAwvqdYnXMozAdyjvTmLzlmcUHBb3hIcl8Rpjhnv/sBez6IvDH3RwzvsYDN1
J4LhbFu+t5/F33wxKi+LjjdZdjKo1lamf42ixSS2ljNDaWyUNUzE+7Pv7YBI
oSWxlcxoCig0uVPqiP9qbwCRWhz7oj/unMcSDNjPLdobQKQWUVZltu7nWg1A
d0pVvp09QKQWZTT0RVylhOpOucoHwfMcv/livsYhILebfW8HRGqx+qK6G3UE
7aGEA7U3gEgtVl+kE7HvuXy09gYQ574IflnHg31vB8TMD31RRLGFjva9HRCn
vpjaKPN6T+0NIFKLRJPbiP0Sbz5kzHyxqcz4gXmI9gYQWyrogg5tcnff2wFx
5Iss7719bwfEgS/G9BG+twNi74tR3qdobwCx80WjGJ/HHiB2vvhg39saA198
tvbq6HzRTsOej84Xz8OeD2rxZOz5QBOfEd7PCP09G8h82HnZ83E4vG+7uhaa

   "], {{0, 0}, {161, 41}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 52},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->23622]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->10900],

Cell[TextData[{
 Cell[BoxData[
  FormBox[
   RowBox[{"r", "=", "2"}], TraditionalForm]], "InlineMath"],
 " totalistic rule with code 10:"
}], "ExampleText",
 CellID->4825],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"10", ",", 
      RowBox[{"{", 
       RowBox[{"2", ",", "1"}], "}"}], ",", "2"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", " ", "0"}], "}"}], ",", " ", "40"}], 
   "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->941],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJzNV9F1wyAMFN2kn12nI2SBztRNm2cH6046wA2JbfWZOOKQDkVI9PP28337
MLOv+/N7f6xUseWxx3v9K9uDYgJfCMszIxsm8Btq4bgRXTgvz4q2x/RDvY0V
6WvNGQfNHhulqjcewMpodxwdAy2PHEPcfdbssRGjTKx4d8EX7X/9zkiMCVsr
Ao/xy54sWZO7iyvjXERiTNiawmP8sqdsbdt3RdXEXD8MtIUwMcraWm+19pSt
pf1wFln4VN961nqrtaem+G+PpwjPGo4661RG9lZHTx166z44Aq5VXuK6dkb2
VkdPA+Ed+XuRcfEBcQUnRKxHnkYUG7vTcVGZY2m2ZaPlaUSxVY1gp5iv2FO4
fzhOx6jtaUQRxjhjhKlvqI85Flfu8zSimDKpDrEz5NMXo+Z4ZXB37uW95TzT
nSFHIEbN8crq06LyrFD02qcvIqIFtv88RXrj3hujmNdGZDs/ZyimWliTx6tc
aCJ0q2SkqoHzFJv1z+OhOolGaquzFKFaeZ3D38xrGaMYa2TppZLPLjGA2Yhi
7ETdG1KEuphzkvMpdpOy7743TRGyCrUqn3TOTde9IUVxi2vlk+rBb8i95Bbe
Wl0kYv9535sVdYvTN+qn73vTFEUcVR3UyENEVTOsfNZFHkMRRtepU/q2ujcS
dV/E8QX3vWmK1G0zxwPq3pCivNlp7UmiK9yhdW8krf/WlPYk0Vl2au5F0Z3k
UqI7yaVEd5JLyQXq3kgumHtRXk7vDy/xF8I=
   "], {{0, 0}, {161, 41}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 52},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2744]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->9648],

Cell["\<\
Rule 30 specified by giving explicit offsets for cells in its neighborhood:\
\>", "ExampleText",
 CellID->20076],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"30", ",", "2", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"-", "1"}], "}"}], ",", 
        RowBox[{"{", "0", "}"}], ",", 
        RowBox[{"{", "1", "}"}]}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "50"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2431],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy1lVFSwzAMRCVuwhm4CUdghm/u/0eLrd2VsWM3iUPJtImj9dNKyvvXz+fX
m5l9PP6/7Xm4Pz7PU/3+OD//vP76u1dXxMVyo1zFfSsRLBaXBYxZbnpdEI8b
49TdaKi4+vcLUjVK2WeNkjaHBVgXcTxwPQHWy+YiFYtlg7I5wwKsi1wiB1ge
YQ3wErMBTZsb+IIoVHGgCTyhkQpNAFm6voSd4gDRAE9opCInYOKLR0KQO0FD
4VnwMxU5ARNfkH4XeT5pjImNwtJWZdovJvLMHsPDisi8t8dKv6h8fCS8bJQ3
+ixH/aLykT0JHxXiodRVmfQL86XZU9vdRXakMukXXmb2JDxaZsQx9cU1X6bM
6I3I85hj6oucPDNHAbLYDjTm/ZKnFtqSbdjtjxHLqF8wPGBxJeMUnnAoy6Bf
MDyMsODwyPSKykG/eAyP+IKxiOcXNGb94t58Mczlw/4YsfR8MUkSTAq2JT9a
lo4voiylxIJb1jjyRZWl/Vhwr6n0fUnKrLD++2OVpfUlKTsrbK0/RiytLwzK
CgPfGZWOLzJNWGpl/QmNkS8yTZDFV/pjxKK+mEri9XWWQ1nUF5kmGO/nOZRF
fUHHsNMvcCiLjlyO/MC4ptHzhSO/il/kUBadJhr8OgdCuvqSg9+h8d+X6MfX
3h8LKsmX6JNr/dFRSb6A6zaOqqK+ROZu5Kgq6ks4c6tG68vZ98eCSop+P0dV
SdF3aCRfNvghKuHLJo6qgjG/79Axv1Vlox+ispmjqOzXOMXxC/ECEP0=
   "], {{0, 0}, {101, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->4574]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->32678],

Cell["An analog of rule 30 with modified offsets:", "ExampleText",
 CellID->18067],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"30", ",", "2", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"-", "1"}], "}"}], ",", 
        RowBox[{"{", "0", "}"}], ",", 
        RowBox[{"{", "2", "}"}]}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "50"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->4660],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJzNlo11gzAMhH3dpDN0k46QBTJO1y1pCdZJwljGxigvDcg/OR+X7/Xz8fx+
fKSUvpb3z/JOeL1elfC+Tttd7vIsqBXYenm1LtlPaj5gVCx//mptrlfrNXI/
b5OSXpF7/zPFvdOnceTRvL/rAJ9Pnsm69F7BXrPLuq+fTTL7KwfMOUBnsi4l
6bBcIc+d+/bZ8Ko8Rg7oc+xnTK+Q/vK97Ntnw6vkmHGsLmNeUngn+vTSCT3K
+5vz12XMSwrvxL7bb7KjVhU5Vpcx63Dus9faBZUr5XfBsbqM2bRFOAa1/54q
eA7wufNZeX6cY+yjmyvfsbEc0z6WVZFjdRmLc4zSVc6V71hdxuIcsz5WVTRj
QY65PtYKC2UswrE9H2uFhTJWzzHeP+AVn7M3x+DMiQoLZeyYY+xNKFdqYShj
RxzTKtpUoSfHPOK3qkJPjnnEPycslLF9jmnen6x+HNub3y4slDGfY3JeD1Xo
wzH+ZfYTFsqY5Vju91OF8xyDM6eXsEaOyZHeqnCGY514VRDWyLEBuVLCGjk2
IFfqa5o4NtCrTVgDx0arQhvHxqtCC8cuqjjHrhMW5Nh1woIcu05YkGMX1WHG
pqjCUcZmqUI5Y/NUoZyxqeVmbLoqeBm7gyrYjM3Ws5X/H/wNajqv9upWuZJ1
q1zJalD1C/epGaw=
   "], {{0, 0}, {151, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 66},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->6790]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->6812],

Cell["A rule with 2 neighbors:", "ExampleText",
 CellID->32554],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"9", ",", "2", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"-", "1"}], "}"}], ",", 
        RowBox[{"{", "2", "}"}]}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "50"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->21317],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJzNVkEOwzAII/vJ3rCf7An9wJ6z766nqupCFBtDarUIRdRQZFl5bp/39jCz
1/5+97eZHXEGZfUN+coK66GvKusb+KCg+RsYicE4/vOcMzkxGMf/v71xTgzG
8Xv6iehKwu/pJ6IrCb83c3xXQf7xJuOg+VV+JedX+ZWcX+VXcn6VX8n5M3Ql
4c/QlYQ/b1dB/tSpIvyez6ie4B/lxchgXZ9R5ZHBUnManh5U55HBrMemOqfh
/aPqnIbHpjqngfqP3K8GjbpRVU8D9R+5Xw0adXNVPQ1UJ6m6ujQypEuqri6N
oC4FuzoapdbT8HypzK8Gg0GxDDN3quJdHU2hvAxl9ysUZfcrFGX3KxRl9ysU
y/3Kw3K/8rDcrzws9ysPt9LVGbfS1RnUrloBiCY/7eEEsQ==
   "], {{0, 0}, {151, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 66},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->277]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->10994],

Cell["\<\
Rule 30 specified by giving explicit weights for cells in its neighborhood:\
\>", "ExampleText",
 CellID->15968],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"30", ",", 
      RowBox[{"{", 
       RowBox[{"2", ",", 
        RowBox[{"{", 
         RowBox[{"4", ",", "2", ",", "1"}], "}"}]}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "50"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->4149],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy1lVFSwzAMRCVuwhm4CUdghm/u/0eLrd2VsWM3iUPJtImj9dNKyvvXz+fX
m5l9PP6/7Xm4Pz7PU/3+OD//vP76u1dXxMVyo1zFfSsRLBaXBYxZbnpdEI8b
49TdaKi4+vcLUjVK2WeNkjaHBVgXcTxwPQHWy+YiFYtlg7I5wwKsi1wiB1ge
YQ3wErMBTZsb+IIoVHGgCTyhkQpNAFm6voSd4gDRAE9opCInYOKLR0KQO0FD
4VnwMxU5ARNfkH4XeT5pjImNwtJWZdovJvLMHsPDisi8t8dKv6h8fCS8bJQ3
+ixH/aLykT0JHxXiodRVmfQL86XZU9vdRXakMukXXmb2JDxaZsQx9cU1X6bM
6I3I85hj6oucPDNHAbLYDjTm/ZKnFtqSbdjtjxHLqF8wPGBxJeMUnnAoy6Bf
MDyMsODwyPSKykG/eAyP+IKxiOcXNGb94t58Mczlw/4YsfR8MUkSTAq2JT9a
lo4voiylxIJb1jjyRZWl/Vhwr6n0fUnKrLD++2OVpfUlKTsrbK0/RiytLwzK
CgPfGZWOLzJNWGpl/QmNkS8yTZDFV/pjxKK+mEri9XWWQ1nUF5kmGO/nOZRF
fUHHsNMvcCiLjlyO/MC4ptHzhSO/il/kUBadJhr8OgdCuvqSg9+h8d+X6MfX
3h8LKsmX6JNr/dFRSb6A6zaOqqK+ROZu5Kgq6ks4c6tG68vZ98eCSop+P0dV
SdF3aCRfNvghKuHLJo6qgjG/79Axv1Vlox+ispmjqOzXOMXxC/ECEP0=
   "], {{0, 0}, {101, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->30922]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->23928],

Cell["A 3-color totalistic rule:", "ExampleText",
 CellID->27160],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"2049", ",", " ", 
      RowBox[{"{", 
       RowBox[{"3", ",", " ", "1"}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", " ", "0"}], "}"}], ",", " ", "50"}], 
   "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->4716],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy1WFuS4yAMVOsme4a9yR5hquZ77/+3RmokHlqPk9hUYgcMatFSA86vr79/
vlREfh/f7+MrkFY9LmgVoF1aXbXdrCB+bSUf4RjBBrMyGtb2GOwhVj06HS3A
iyCtHOOas7SlzTDsR8M6wND9cXTIyyBmxox3W96irbjv3t6eWCelO5dBfERz
Um02rcXmpV6sb3tqfpgDY0gugtgY2HihNThXRDEfjDKIzwz4H8hcGXvBPW89
aE0TwxijC9JntoP061SbetFzISHJVTBGFzwFxpCEpWCU92WSPgbdjIxcBWPq
Oa5OMHYQchD3BYRjwoyuGKqFKjfiyWjcF5BNg3vZVLkRj+4B77KEa9dggbKq
ciM+PNB1Jqq1BguUVZUL8WCSM7h7uAoNlozNqlwseVqCwS3CVWiwZGxW5QIS
KbyCnGiwZGxSZVqienoKs2UNV6XBkrFJleluS5pIYfhynSBnGiwZG1WZllo9
U5gupE7ONFgyNqhyABlEZLOaZ3KqwZKxQZUDSE9dYxLrTE41WDKGISSDPU9d
T+UF5AcNloylKj2rPLv7yiHIRrmmwZKxVKVvetrPMNrPJb3xogYLlFSlcNND
54RMIrfPaxosUDAuLBpLLJLJaLyqwQJlWlhIDM88grHxugYLFB3XQXT2Ndab
PGxc1mCBgqDDqPHoGv3MieHxG1wRhar0/ahHF8NK7c9e0WCBQlU6Lz26mRN2
/H1RgwUKc0mn6GZO6BsaLFBclZijGzmBdzRYoHgElujmAvOOBgsUj8ASXebE
mxosUDwCS3R70+dchb0iusyJG7gKe35ImVs/0mCBIr7Az+UzDRYoypV9Kvdx
RRTwJWf63ItBVfI9x8VzL1dE8S0Xcb2ZK6Ig3nOAmzRYoAhPW/zb4ZkSr6s3
arBA6XviQ1wRJfbHJwv/WXoUw1X5NIYWK+UTZV8pfyz/APRaHEo=
   "], {{0, 0}, {101, 51}}, {0, 2}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->29767]
}, Open  ]],

Cell["\<\
Totalistic rules have weight 1 for each offset in the neighborhood:\
\>", "ExampleText",
 CellID->11743],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"2049", ",", " ", 
      RowBox[{"{", 
       RowBox[{"3", ",", " ", 
        RowBox[{"{", 
         RowBox[{"1", ",", "1", ",", "1"}], "}"}]}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", " ", "0"}], "}"}], ",", " ", "50"}], 
   "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->21647],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy1WFuS4yAMVOsme4a9yR5hquZ77/+3RmokHlqPk9hUYgcMatFSA86vr79/
vlREfh/f7+MrkFY9LmgVoF1aXbXdrCB+bSUf4RjBBrMyGtb2GOwhVj06HS3A
iyCtHOOas7SlzTDsR8M6wND9cXTIyyBmxox3W96irbjv3t6eWCelO5dBfERz
Um02rcXmpV6sb3tqfpgDY0gugtgY2HihNThXRDEfjDKIzwz4H8hcGXvBPW89
aE0TwxijC9JntoP061SbetFzISHJVTBGFzwFxpCEpWCU92WSPgbdjIxcBWPq
Oa5OMHYQchD3BYRjwoyuGKqFKjfiyWjcF5BNg3vZVLkRj+4B77KEa9dggbKq
ciM+PNB1Jqq1BguUVZUL8WCSM7h7uAoNlozNqlwseVqCwS3CVWiwZGxW5QIS
KbyCnGiwZGxSZVqienoKs2UNV6XBkrFJleluS5pIYfhynSBnGiwZG1WZllo9
U5gupE7ONFgyNqhyABlEZLOaZ3KqwZKxQZUDSE9dYxLrTE41WDKGISSDPU9d
T+UF5AcNloylKj2rPLv7yiHIRrmmwZKxVKVvetrPMNrPJb3xogYLlFSlcNND
54RMIrfPaxosUDAuLBpLLJLJaLyqwQJlWlhIDM88grHxugYLFB3XQXT2Ndab
PGxc1mCBgqDDqPHoGv3MieHxG1wRhar0/ahHF8NK7c9e0WCBQlU6Lz26mRN2
/H1RgwUKc0mn6GZO6BsaLFBclZijGzmBdzRYoHgElujmAvOOBgsUj8ASXebE
mxosUDwCS3R70+dchb0iusyJG7gKe35ImVs/0mCBIr7Az+UzDRYoypV9Kvdx
RRTwJWf63ItBVfI9x8VzL1dE8S0Xcb2ZK6Ig3nOAmzRYoAhPW/zb4ZkSr6s3
arBA6XviQ1wRJfbHJwv/WXoUw1X5NIYWK+UTZV8pfyz/APRaHEo=
   "], {{0, 0}, {101, 51}}, {0, 2}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->21315]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->30541],

Cell["A 3-color outer totalistic rule:", "ExampleText",
 CellID->366],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"50257", ",", 
      RowBox[{"{", 
       RowBox[{"3", ",", 
        RowBox[{"{", 
         RowBox[{"3", ",", "1", ",", "3"}], "}"}]}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "50"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->15423],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy1WNF15DAIRHSSGq6TKyEpIP3/nQUMIBvtrs4yyTqyjRAzox375ev79+83
E9Gf4/NDPZgaS7TWqDU5cutnLD+SQsc56V3JOIYs18myNBvzG0r2lH6nT7D5
57Uk+/hzZLHcbdJQH9p1ZKXK3kXvzKppl3ZnXKsfrRmdxw1Dlv6Q5ZWB1aBb
td4Q4ryW8MS4zHJiiaxMnSoHmwq9RUNG3mWt3oowKnd7qo37MGeh2qBDVAs9
+LIW1FeW2LkKfXNlPukQ90OPjrmz49GcUaXMuUodHWHzLjo4jqSH6GfpSX1n
KbhyFaQ8J76yDo4j6YFdlDnOLAVXnuG7q9Kh0gO7aFA/sZT2lakQu6vSodID
uyirn1lyrlzf2F2VDpUe2EVQH/wohODKtUq7q9Kh0iP6UPXJ+FEIbFeSVo64
1sFxFH5F6Mz4YWMJiGniS7Oo/MoqOEvgKnlm6UtTHIVfSW/KkuwCPcoiroT3
+EIHzyn8qvdmz4505Iz3Ix2i86tfsS6tUASAluI3vjTFUfiVUmxQBEA89V75
0hRH4VdKMSkIARBPvVe+NMVR+JW2Chie9NaXpjgKv7LVm3PxoS9NcRR+Fd8w
a/pDX5riKPwqvmF0eeqt6OA4Cr+KGuen3poOUeXqV+zvUWu+NMVR+FW8R635
0hRH4VfxvFvzpVlUfkV5F9zQAXH1qzbughs6OI7Rr+w9XEfSww0dHMfoV3Da
//KlWYx+5U67RQfE6FfutFt0QIx+5U67RQfE6Fftli/NYvSre740i9Gv9uqA
GP1qrw5ec/CrvTp4zdBjuw4I1QO/e3VAiB760rNdB4T6hx0fWQEvnfSIDoj4
f8pzQen4VISLPBf0oA6IdQy8Gusz/gGg5xbh
   "], {{0, 0}, {99, 51}}, {0, 2}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 97},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->3164]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->13767],

Cell["\<\
Specify rule 90 by giving explicit replacements for neighborhoods:\
\>", "ExampleText",
 CellID->23030],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", "_", ",", "1"}], "}"}], "->", "0"}], ",", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", "_", ",", "0"}], "}"}], "->", "1"}], ",", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"0", ",", "_", ",", "1"}], "}"}], "->", "1"}], ",", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"0", ",", "_", ",", "0"}], "}"}], "->", "0"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "50"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->8756],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy1ldENwyAMRE036QzdpCNE6nf3/2uqCqGQxnf2GSJkIXI8DFh3397P7WZm
j72/9m6ttW+/ajXz5v7Tiub9f6rmDXyocXoDHVMYfeddRSYXrDcQmVywfj5/
9j5i+vn82fuI6WduJA9e/y+3GIXRR+sjp4/WR04frY+cPlofOX32PmL67H3E
9EoevF5jsHrkD6r/jL0onaX4/qD6z8hFiSzF9wfVf8aePH9Q/WfsTRmzFG3M
Uvz3r/rP2Jv3/lX/6av471/1n77KMUbnWYp/3qr/9FX8far+01fBXqo3xksr
KBX+gSkV/oEpFf6BKX6solT4B6ZU+AemVPgHpvjjKsqK+jhTVtTHmbKiPs6U
Y1xFWXkfg7I6jx9lPSOVxwfoHBS3
   "], {{0, 0}, {101, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->9226]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->13576],

Cell["\<\
Specify rule 90 by giving a single \"algebraic\" replacement rule:\
\>", "ExampleText",
 CellID->24702],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"x_", ",", "_", ",", "y_"}], "}"}], ":>", 
      RowBox[{"Mod", "[", 
       RowBox[{
        RowBox[{"x", "+", "y"}], ",", "2"}], "]"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "50"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7419],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy1ldENwyAMRE036QzdpCNE6nf3/2uqCqGQxnf2GSJkIXI8DFh3397P7WZm
j72/9m6ttW+/ajXz5v7Tiub9f6rmDXyocXoDHVMYfeddRSYXrDcQmVywfj5/
9j5i+vn82fuI6WduJA9e/y+3GIXRR+sjp4/WR04frY+cPlofOX32PmL67H3E
9EoevF5jsHrkD6r/jL0onaX4/qD6z8hFiSzF9wfVf8aePH9Q/WfsTRmzFG3M
Uvz3r/rP2Jv3/lX/6av471/1n77KMUbnWYp/3qr/9FX8far+01fBXqo3xksr
KBX+gSkV/oEpFf6BKX6solT4B6ZU+AemVPgHpvjjKsqK+jhTVtTHmbKiPs6U
Y1xFWXkfg7I6jx9lPSOVxwfoHBS3
   "], {{0, 0}, {101, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->23643]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->27493],

Cell[TextData[{
 "Use an explicit Boolean formula for rule 30, operating on ",
 Cell[BoxData[
  ButtonBox["True",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/True"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  ButtonBox["False",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/False"]], "InlineFormula"],
 " states:"
}], "ExampleText",
 CellID->5046],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"p_", ",", "q_", ",", "r_"}], "}"}], "->", 
     RowBox[{"Xor", "[", 
      RowBox[{"p", ",", 
       RowBox[{"Or", "[", 
        RowBox[{"q", ",", "r"}], "]"}]}], "]"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "True", "}"}], ",", "False"}], "}"}], ",", "2"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->9123],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"False", ",", "False", ",", "True", ",", "False", ",", "False"}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"False", ",", "True", ",", "True", ",", "True", ",", "False"}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"True", ",", "True", ",", "False", ",", "False", ",", "True"}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{455, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->32177]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Boole", "[", "%", "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->9398],

Cell[BoxData[
 GraphicsBox[
  RasterBox[{{0, 0, 1, 1, 0}, {1, 0, 0, 0, 1}, {1, 1, 0, 1, 1}}, {{0, 0}, {5, 
   3}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{104, 62},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{100, Automatic}},
 CellLabel->"Out[2]=",
 CellID->22532]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->11575],

Cell["\<\
Values in a cellular automaton can be any symbolic expression:\
\>", "ExampleText",
 CellFrameLabels->{{None, None}, {None, None}},
 CellID->26557],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"{", 
       RowBox[{"a", ",", "_", ",", "b"}], "}"}], "->", "a"}], ",", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"b", ",", "_", ",", "a"}], "}"}], "->", "b"}], ",", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"x_", ",", "_", ",", "x_"}], "}"}], "->", "a"}]}], "}"}], ",", 
   
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "b", "}"}], ",", "a"}], "}"}], ",", "4"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->13182],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"b", ",", "a", ",", "a", ",", "a", ",", "a"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "a", ",", "a", ",", "a"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "a", ",", "b", ",", "a", ",", "a"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "a", ",", "a", ",", "b", ",", "a"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "a", ",", "a", ",", "a", ",", "b"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{532, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->8496]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->23662],

Cell[TextData[{
 "Use an arbitrary symbolic function ",
 Cell[BoxData["f"], "InlineFormula"],
 " as the rule to apply to range-1 neighbors:"
}], "ExampleText",
 CellID->30760],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"f", "[", "#", "]"}], "&"}], ",", 
     RowBox[{"{", "}"}], ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", "1"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->27980],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"f", "[", 
      RowBox[{"{", 
       RowBox[{"c", ",", "a", ",", "b"}], "}"}], "]"}], ",", 
     RowBox[{"f", "[", 
      RowBox[{"{", 
       RowBox[{"a", ",", "b", ",", "c"}], "}"}], "]"}], ",", 
     RowBox[{"f", "[", 
      RowBox[{"{", 
       RowBox[{"b", ",", "c", ",", "a"}], "}"}], "]"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{359, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->25303]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->14720],

Cell["Apply the function to neighbors with offsets -1 and 0:", "ExampleText",
 CellID->11548],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"f", "[", "#", "]"}], "&"}], ",", 
     RowBox[{"{", "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"-", "1"}], "}"}], ",", 
       RowBox[{"{", "0", "}"}]}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "2"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->21301],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWktPwzAMztq9AG1sO3EcJ8QfgRM/opqQtgMCjf19KE22lMpKZztxH6oy
adWcJn7GyeIvb9lp//6RnQ67bPt6zL72h9339uXzWDSlI6WSJ6XU44MqfufF
L/O1n+fi+0+eqTv9TMte6rZ4JAU5JbbfAPoe0OPLuAloXwnz48qRtmc9MHuo
8VkF2jMGdKofxppqK5ylcBT23npnTZBO6Uflh9FUedL2bQZuHzV+mB9C6SR3
zWbdOilIymhKP2116sitWU07xm8aKJc6nktT+Yb6zVf/WaB+vvFq22/S+ofO
t3lDfqNlcwpo0g7H6G+kTkra6KDK/XRW+xazTI+ei+oC/xtI+N9HS1kP+1jl
GuczO8Mi3AffylslP2/njXkYX02w2QUtg/7n9k+dti6crRivpYhseCLBuEhr
JeNBrhWhVvtFrGvfyVghM++WjXgQW4lGRk4//kO4vY+tW5j3abJddYgQmquV
jAe5VmBZj433i1jXvpOxQmbeubO+nb0+Zn3MeorUmPVDznrp6p9kli+CZEpn
N0eb9rReEqW1k8XSPpLROmwe9Slb4x7d930m7tH+vot79DnrIf5okQbd24WC
Y++Ty/v7Sj+ohcUpqXy4qDqFP8eOa3I2BD66fYX4o0k7KPKp8fC1g3b7Ib98
XPd1ZCgsK7BbI9jqCnfRvt9OaeI2UpX+NV7HqqfWS1YbeIvB3m7g3o7p6vZE
07dpLP+c5F0uak1FsyEW1dXtg1CU3xdFD0XvpW49/OSUWVC3YtfjkTy00oX8
9gOP5mKmV/n0BLmVw8Y1t/MqnSC9aRgWDdlaMGU1g0Vy8TT43g9Jk0HxZHBQ
/TC+Z68fnOhjZ85uTr7c05tzfEdnRplTt34Y+TH6MfodRb/dGja1PisT7bBq
8JCiHHO86+hT5mC1UlRXwXOdzW2/kEoapQIF+/nyuWZfXWVDPwyvMiDc2pMa
/QFrSdu4\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{551, 33},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->239097570]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Grid", "[", "%", "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->15928],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWktPAjEQLrsuqIhgYoxH7t5FT9705AH9BYSYwMFoEE28GH+Pj9+4brcU
SNPdmbbTXYgl4THtdB5fZ3Z2O9yO5pOHx9F8Oh71b2aj58l0/NK/fpplQ3GD
sShhjJ2dsux3mv3K3/LVyt4rUlD72WeUMTU5FS+5C8f3FLqr0DuLdYky3iOW
J8dPgv212s+/RRSps+t0G5iX3h4BfDGSDysPoo+DX1vlV8Q/WK6WiVcsYxRY
lSD4uNhYkwutgnFIXtNRL3Y9RLeRclxxsrUXu67IPtv98Y0Ttb2u8USFk8jC
SBmFpMaG/Ln8ZEmnIuVlXWoVzkKe8NW7pLaoNdY1Xw9qRdbGGxtMi+TY72yd
mNJ7Qx+nfpDFXQ1UPFUpEH+stb2jHYVkHZLoVu+kISnQPBShfhA09cIUO3Xe
bsfqxo7GC5q484NgnTVd7xF03YHQxOnWPQ9vcp2h8cIUO5W227G6saPxgibu
/CAYstgOt5DFKys2Hbv/ksW+TrkoPek46aTOWh/nN+5WYzGqJjupMaKx2i2O
Qi0VdKil9l6EWmrvxebXUnkizmd1XVJoPlrMd9f4VK2yv4WVU9Z1LRrvAvJN
/CjTg/GDj/cAPHz6gdFvsh82fuT8y4DV/f8CT10V2Ijt4H/WvP5y6UVZPmI7
wR8bJmeQe3eeVtMhfUXyuXYYsXpcO4oCvYEyWm2f6s1Qe9G1wK6LYqrdRycH
twt++gaQ/35O3E21Up64U8a86d2sa6zb3YvRxLid7oB29WhjqyENythaRYOu
SQUOMby9MXyR0jwxvSPkYJ5YMHKkPWVyxN24xMPtyaSMGqYiT74X9vwsvu/y
2XvFui/tTulHh0ECqQS+IzwyflPdTkl+saus8QfTq3a7\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{517, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->168139396]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->7809],

Cell["\<\
Set up a \"Pascal\[CloseCurlyQuote]s triangle cellular automaton\":\
\>", "ExampleText",
 CellID->30800],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"Total", "[", "#", "]"}], "&"}], ",", 
     RowBox[{"{", "}"}], ",", 
     RowBox[{"1", "/", "2"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "3"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->26698],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "2", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "3", ",", "3", ",", "1"}], "}"}]}], "}"}]], "Output",
 ImageSize->{354, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->5017]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1445],

Cell["Specify rule 90 as an explicit function:", "ExampleText",
 CellID->25227],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"Mod", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"#", "[", 
           RowBox[{"[", "1", "]"}], "]"}], "+", 
          RowBox[{"#", "[", 
           RowBox[{"[", "3", "]"}], "]"}]}], ",", "2"}], "]"}], "&"}], ",", 
      RowBox[{"{", "}"}], ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "50"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17896],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy1ldENwyAMRE036QzdpCNE6nf3/2uqCqGQxnf2GSJkIXI8DFh3397P7WZm
j72/9m6ttW+/ajXz5v7Tiub9f6rmDXyocXoDHVMYfeddRSYXrDcQmVywfj5/
9j5i+vn82fuI6WduJA9e/y+3GIXRR+sjp4/WR04frY+cPlofOX32PmL67H3E
9EoevF5jsHrkD6r/jL0onaX4/qD6z8hFiSzF9wfVf8aePH9Q/WfsTRmzFG3M
Uvz3r/rP2Jv3/lX/6av471/1n77KMUbnWYp/3qr/9FX8far+01fBXqo3xksr
KBX+gSkV/oEpFf6BKX6solT4B6ZU+AemVPgHpvjjKsqK+jhTVtTHmbKiPs6U
Y1xFWXkfg7I6jx9lPSOVxwfoHBS3
   "], {{0, 0}, {101, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->13334]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->19826],

Cell["Additive cellular automaton modulo 4:", "ExampleText",
 CellID->3996],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"Mod", "[", 
        RowBox[{
         RowBox[{"Total", "[", "#", "]"}], ",", "4"}], "]"}], "&"}], ",", 
      RowBox[{"{", "}"}], ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "50"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->12432],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy1V1uS4zAI5HGSOcPeZI8wVfu99/8bC4GEW1JkxxnHLgKEtIPpAF/f//9+
CxH9Oa5/xyVMLKrK9lJFHeVdf9VFiu2wMptEHeVdf9WrTZVdoo7yrr/qQvVk
qmcYhOvp5mZXqmcExAfDHgFhcCyWI4Eld2w5JLfxcZHbONlLHLmtxJB/Nuzx
Peo2Ryk/Q8geVpGHYCk1cdxeya3VR7IXWfSjgswv9r7b43tIOkbJG9vFLktG
7Td4bgns6jq5X8Ae35Mx1J6IWv7Vs10SU+6zBHDzl8yHLFkPP5XEeJzac6r+
81GfBXvOq6z32eudLP9dZj5QipPkB5SWf2rSKjHVe81/l9mf47IfUFr+u8R6
Z5DZTxA/w9DGjah5Kxa7K/a816wHZ9CP8fMjuBE1H/9K2vJeuRGcQT/GL1C8
3inlO66ad2n8mPkxfoGS+BH5jqvmvfNj5sf4BYqVaP9M8CAOshKlpR/jFyhQ
38GDhuL8WfkxfoEC+WTmkx78Wfnn/BhQQGfQaeO/gtH7C/aVOFFXOp/Xjugv
2Fco8SDr0Uf0JT8GFO8v2FcyD2Z95DU/BpRFX5GFHn3kDoYO/SOYvdLn/WOL
Av0jmL3S5/1jiwL9I66Vfo0fA4qe+wdv9Hcwen+JmUzsX0uMDDGPlfev+8cW
xXmgab5S50PMY7rtH1sU50Ger4IPMY/t+8cWxXmQ5yvR8zy27x9bFO8Peb7C
eew+PwaUyXyF89hTjHHfmM1jz4/ZfJX1T2CM+8t8/3iMAvvLfP94jAL7S8hn
/BhQYO+Y7x+PUWB/ea9/bFFgf3mvf2xRYH/5DD8GFL2yfzxGubR/PEa5tH88
Rrm0fzxG+Uj/2KJ8pH9sUX6FHwPKrz6PhnI/5AdKOzNN
   "], {{0, 0}, {101, 51}}, {0, 3}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->8394]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->31371],

Cell["The second argument to the function gives the step number:", \
"ExampleText",
 CellID->17127],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"f", "[", 
       RowBox[{"#1", ",", "#2"}], "]"}], "&"}], ",", 
     RowBox[{"{", "}"}], ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", "1"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10792],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"f", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"c", ",", "a", ",", "b"}], "}"}], ",", "1"}], "]"}], ",", 
     RowBox[{"f", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"a", ",", "b", ",", "c"}], "}"}], ",", "1"}], "]"}], ",", 
     RowBox[{"f", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"b", ",", "c", ",", "a"}], "}"}], ",", "1"}], "]"}]}], 
    "}"}]}], "}"}]], "Output",
 ImageSize->{413, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->88775419]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->21819],

Cell[TextData[{
 "Change the rule at successive steps; ",
 Cell[BoxData["#2"], "InlineFormula"],
 " gives the step number:"
}], "ExampleText",
 CellID->16585],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"Mod", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Total", "[", "#", "]"}], "+", "#2"}], ",", "4"}], "]"}], 
       "&"}], ",", 
      RowBox[{"{", "}"}], ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "30"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10752],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJyNlO11wzAIRfkYpKeDdImMkAW6/7/yMKgSRlZeQgBZsmSuyff79/UWIvoy
+zFTUSFh4YjFjEmII6652jybbWsQq1psZqko20xVi8THuxzz2NcoMZOFML6G
LGbPaJNzzMe4jfrJSP08hC3UbZeTPwPW2olsxE6PpwgvYbsc3he6cBMrxPBp
uzw2FXXJdbDh+ZDHspBV0w6AotrD4KFA4zoihjHkMRJzsYpCyRhzNbhS8ARn
jThZr0qOivIH1+SJnP3axT3FQ8lRJq40cZaJ+13+xtkWqAl+BGRIRp7X/yWT
OPiRf1AZRh1HzgvfqrwZWAbHJS+brcAGP04WJS/TixxusMY3vcOluvZGbGKt
k6eWb8cbUFCj2fN9owZa8tXie74db/Qt2PirPPq3qiUWfYt68a1/T8q+9fJP
/Vu1IRZsqe3fk7K/1/6t2hKb/rfPfDvetX/PnGt/f8a3493zfeI88/6cb8f7
SQdiz5f/AClqDdk=
   "], {{0, 0}, {61, 31}}, {0, 3}],
  Frame->Automatic,
  FrameLabel->{None, None},
  FrameTicks->{None, None},
  PlotRangeClipping->True]], "Output",
 ImageSize->{184, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->170013987]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->22760],

Cell["Use continuous values for cells:", "ExampleText",
 CellID->16613],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"Mod", "[", 
        RowBox[{
         RowBox[{
          RowBox[{"Total", "[", "#", "]"}], "/", "2"}], ",", "1"}], "]"}], 
       "&"}], ",", 
      RowBox[{"{", "}"}], ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "50"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->15555],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->1439720]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{
   RowBox[{"CellularAutomaton", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{
        RowBox[{"Mod", "[", 
         RowBox[{
          RowBox[{
           RowBox[{"Total", "[", "#", "]"}], "/", "2"}], ",", "1"}], "]"}], 
        "&"}], ",", 
       RowBox[{"{", "}"}], ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "50"}], "]"}], ",", 
   RowBox[{"ColorFunction", "->", "Hue"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1022],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->108011751]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->14371]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Initial Conditions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(8)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->26673],

Cell["Explicit initial conditions are assumed cyclic:", "ExampleText",
 CellID->30660],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"30", ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0"}], "}"}], 
   ",", "2"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->18312],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "0", ",", "0", ",", "0", ",", "1"}], "}"}], 
   ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "1", ",", "1"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{377, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->950]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"30", ",", 
    RowBox[{"{", 
     RowBox[{"1", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0"}], "}"}], 
    ",", "2"}], "]"}], "//", "Grid"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->6128],

Cell[BoxData[
 TagBox[GridBox[{
    {"1", "0", "0", "0", "0", "0"},
    {"1", "1", "0", "0", "0", "1"},
    {"0", "0", "1", "0", "1", "1"}
   }],
  "Grid"]], "Output",
 ImageSize->{86, 43},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->24420]
}, Open  ]],

Cell["\<\
The left neighbor of the leftmost cell is the rightmost cell, and vice versa:\
\
\>", "ExampleText",
 CellID->8532],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{
      RowBox[{"f", "[", "#", "]"}], "&"}], ",", 
     RowBox[{"{", "}"}], ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c", ",", "d"}], "}"}], ",", "1"}], 
  "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->25744],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"a", ",", "b", ",", "c", ",", "d"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"f", "[", 
      RowBox[{"{", 
       RowBox[{"d", ",", "a", ",", "b"}], "}"}], "]"}], ",", 
     RowBox[{"f", "[", 
      RowBox[{"{", 
       RowBox[{"a", ",", "b", ",", "c"}], "}"}], "]"}], ",", 
     RowBox[{"f", "[", 
      RowBox[{"{", 
       RowBox[{"b", ",", "c", ",", "d"}], "}"}], "]"}], ",", 
     RowBox[{"f", "[", 
      RowBox[{"{", 
       RowBox[{"c", ",", "d", ",", "a"}], "}"}], "]"}]}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{466, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->18918]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->28213],

Cell["Random initial conditions:", "ExampleText",
 CellID->8191],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"30", ",", 
    RowBox[{"RandomInteger", "[", 
     RowBox[{"1", ",", "100"}], "]"}], ",", "50"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7022],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJxtlNG13EYMQ8l0khrSSUrwyX/6/4u9K+Beytlj+UmjIQcAAf3549+/f/wx
M3/9vP6ZX7/dz7W/fj/v5nN9nj//z3fTd+fz5ns3edCWp3ae4k+T7w0bdtP9
u/c5Xii+ayn+Nm3PIEvxc6Zah0TqvhxfKGZfgLf31O7cM9V6IZIGD3yVvAAj
K7WzPrNkH8iDEuEYFfVGxciq+pBBwkeMLkyLOghp1GJkWp1nCfK6s+nCqqhO
K4iqg0xzzkOCvM4QvACLLeaCGKPSObceC1WY1cKahVTkOFC5ketj2dTl8AVp
tIn+mfb2STJ5BB46hq8VB6QnoPavnuyx+1eDqeEVpiIjoPZvc7il2W8KajCY
Gn6EIkzWbLcL5BCaVTtGh8I1fFnUkrAdLZTYwFFeWzGs4SPv6AicJesygjGl
hvQEDepalIcG3BKPEawpxSYNEOMVyaueEehIDXiKX1+dWJP0aoaBIQ8Vgcbc
p8lXIiF1Np3e0YFHPWxUixOtmr/z3RgzQJVeH3jUu+LPgdW4dr6VuipWYy5n
z6FRmIo7cGq7Zh0V0TgXRiS428ZoIncrDpTUW9J4L1vbY9qY9NjdigNGS4ys
cegcmMqU6oDAWNOtB+Uw1MaCF6YypfR0ZhprdKkVSm7Lz27m21D9b3oqEvYN
k7Al6+LHyoI8+t/0IBJ2YtTp1T/zQpKR++jjcoQkHPnJKF0uGDZFyxUiuVyJ
djjyz1PPajnpxZQ8IrzTo8H100IucZk5dVMH3+nUgKSnwWk4uEVBucIReLdG
A9kXkXLw9NYBEJlGAGmcI8LnUckmzVgtbs+KKmA7+IWtPRplV5uxgKCajIbo
wSNcQyqwVQubI/vxrOrrTbUNHjoTJVuJKYfiCaCSlpDZk+gChmCTyvVNKMqz
qm+TiCf0fISIEoE8OalkW+qS0YllqlLrRKn7b04qWR1JTdC+DHXVWhGLJGA6
XiJ/ip+0LXqSdoNOU2bUhMiSj5aMdujVcJE0B11NS68JwUJFvby2TzkEfJnP
aYohKjLOidjXa87vnKb17aupGellWQk8jmSHUKHJq2nsq5SSCpHSOmCceUa+
vzVV0LE88azIzZd8BnvpMb81PUHHgs5xDRl8TUstpub7P00viBVUMSZbOdrG
o/lpYNnm1Q2ilUvZIvoty9BeDSTbgo/51N5BivOoEeDiUAOFGHwoLXt77EMV
s/wPnvAMOw==
   "], {{0, 0}, {100, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 96},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->28651]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->29317],

Cell["\<\
Start with a \"seed\" consisting of the block 11101 surrounded by 0s:\
\>", "ExampleText",
 CellID->11573],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"126", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"1", ",", "1", ",", "1", ",", "0", ",", "1"}], "}"}], ",", 
      "0"}], "}"}], ",", "20"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->11677],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzdV9FKw0AQTDhISUkwEKgWLAQKhYJCoVgoVCoUFAXxG0IR2gdRar/Eh/7u
ulvncBMs+pJcbCC3d7szS25vbyCP+Xb1/JJv18s8u9vkb6v18j27fd2wy/ie
5+34fTjzeE482787IvoK8GPEs3+8Dx78P3pjngY8CdjGbFtsr9iKv6X8Ys0R
ZzBgRWxTtteKlSK7W1aAqN3PQqElW+ic1QY6QOXvsZYsEfbnlhUjKqgn1RNJ
7eik1JW20rbC5oDfnkz677OmiMY4wRA2ouK97RwZu1OqjtgELLuWLN3Gsbt0
WFF/UuDzxmfpKVSKk9TKnDlnZVRU3hMqKnPfOatPReU9paIyD5yzBqoneqi4
7G/YcPRQdUhZZ+VcLpyzLlUv61s3qh09Up1gFU06ZFw7eozvtRW0yjSpHT1B
tI1KThuCmqKOVmdmlaNm9N3VUr955agbHkKgxL2oHPXbXx/+E/1PlKqzXw==
\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{150, 74},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{150, Automatic}},
 CellLabel->"Out[1]=",
 CellID->222513779]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1430],

Cell["Start from a single 0 surrounded by 1s:", "ExampleText",
 CellID->16061],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"150", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "0", "}"}], ",", "1"}], "}"}], ",", "20"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->13591],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzzTSzJSM1NLMlMTlRwL0osyMhMLlZwyy8CCjEzMjAwTAPiAAkGIPs/kAXG
ICEgYAaxwYBhEpBgpEiUBUhwgYSANDeUlkTj4xIH6WMf+qZCZZihKmF8eSjN
QkCcGY0/agN5NrADmSxoNoBodTQ+LnFmqH72YW0DKD1zARm6QJobSnMhpfOh
ooMPKmOKRvMNqA5BpFQKS+24+MxEqgPRgqM20tVGUSSTkGsBXHxi1YmOaBsl
GCC5mxlKC6LRMHGJYW6CDFrISaLRMHGZQW0CrpqbmNyIXtOP2kC+DcpIKsnJ
vzBaeVjboMaAPbeqDTEdWgzY86vWgOrQZcCsaWFlIbK47jA3wRBHOkVPr4aD
2gQTBtSaDEabDDEdFjhCwGJAdVgzYC/hrYeYDnu09ASj7QdUhxM0VTgNsAo3
qIwbTVUQGqmDjvgxAgB+aIA8\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{150, 80},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{150, Automatic}},
 CellLabel->"Out[1]=",
 CellID->1699977]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->31974],

Cell["Start from 111 on a background of repeated 10 blocks:", "ExampleText",
 CellID->4779],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"22", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"1", ",", "1", ",", "1"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1", ",", "0"}], "}"}]}], "}"}], ",", "20"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->31728],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztmj0LwjAQhlNaLLQg1KlOujk5OjkKioOgk6MUEewgSvWXOPh3651cMEbF
UD9jr9C+uWtyeQzXXIWOkt1ysUp26TxpDrJks0zn22Z/nYHLdYQQBzjHsYB2
Dq3TiS44XGyfDrGHi/OU10MTGgGop2nnjh/7y3F2RgzhgmZAvaR2NTugftWS
RMHV88GMaBV7pBH5/Z8YhWZMv2dIGpPf9lGoFSWPJ0peS799UdTMnGqZWPYo
LuX7jNT9iyi4Hov8vAdzlMua5IvrmiV3RJP7TPFqCr1Kqmp6nyn+k0JmYY2y
LySt3chCnrXorPItCTXSbJ7V7ln1fbye39/HmeJTFOqO2dBspig3hczSlkFW
MsWnKFDbhv2YolwUal3R/zuZZi1T2kZ5q0KYVAymZMp3U8qnJizw1DClrZSo
UcFxTMmU76REDZ8cz7TfpH30dQN9JeEcAfzR0y8=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{150, 80},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{150, Automatic}},
 CellLabel->"Out[1]=",
 CellID->253880348]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->179],

Cell["Specify the \"seed\" as a sparse array:", "ExampleText",
 CellID->31703],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"150", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"SparseArray", "[", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"1", "->", "1"}], ",", 
         RowBox[{"10", "->", "1"}]}], "}"}], "]"}], ",", "0"}], "}"}], ",", 
    "20"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->26352],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJyNU9sRwzAIE92hA2SFjNIRskD3/6tjwKg2CfEdx+OsIOGwHd/P8QLwbrY3
g4h0s+D0d3mzFkLrGlS5gjrYgyrvDdV7UOVGj4MqV1kwjo9yo+hjMLkm2kYj
Q0N8BRL3XGlgSQNTjHuDzKjJDIhxg+hwbQHEuIHQ8f9gSRc4QbBPNLAW58/+
ooNrmXVcaGAtPNZRuoHY6PUPTN8h0zLtQ3WSfSgh6z7UXZZ9qLvMAZ8fqnAE
6A==
   "], {{0, 0}, {50, 21}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{154, 67},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{150, Automatic}},
 CellLabel->"Out[1]=",
 CellID->9545]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->27583],

Cell[TextData[{
 "Use a ",
 Cell[BoxData[
  ButtonBox["SparseArray",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/SparseArray"]], "InlineFormula"],
 " to give the complete cyclic initial condition:"
}], "ExampleText",
 CellID->21133],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"150", ",", 
    RowBox[{"SparseArray", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"1", "->", "1"}], ",", 
        RowBox[{"10", "->", "1"}]}], "}"}], ",", "50"}], "]"}], ",", "20"}], 
   "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->29570],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJyNU9EVgzAIPLpDB+gKHaUjuED3/6vNEbgEo/KiHBiEg8dr+362B4Dn/rz3
B2bGF7XaFfAGiP7AVdgVUFmLb2C2K2B8cxBgsivoJstNdQYAPwu7eNguUuLx
mkXCieyeQWKz9MziTgDCK2NtChBn/FNmMXJQLjkTzYaZg3Lp049sUfKh9A8D
jyj5OGRog7Z3KXnDqViZwxEXbwJbsOagXIb9OM9gcWlYg8uQugbXWcoa3Mgi
Jd6SH6YkBOg=
   "], {{0, 0}, {50, 21}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{154, 67},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{150, Automatic}},
 CellLabel->"Out[1]=",
 CellID->17591]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->22609],

Cell["Start from block 101 at offset -8 and block 11 at offset 9:", \
"ExampleText",
 CellID->23942],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"150", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"1", ",", "0", ",", "1"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"-", "8"}], "}"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"1", ",", "1"}], "}"}], ",", 
          RowBox[{"{", "9", "}"}]}], "}"}]}], "}"}], ",", "0"}], "}"}], ",", 
    "20"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19660],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJyNU9sBwiAMvLiDAziHWzhCF3D/PxFJcnnQCoU2ScnlOHgc79dxA3Af4zkG
xPu3DYeNzmYHphMekpVwY7NjoXloNtna7JjQHOJEjc0O5FA0i80O/IDtkQs7
1SigORhrX8IcS5gdyh1UjnJTWXL5rK1YjfZtHAtxVRaOrPKpjkXgyFcYWeXT
jyJw5BuQ6biA3y1uQuZzGxwdLmivnaIJu13p+trpVpaNnh1fMVxd3uvZ8WWN
64U9w433t17YM1yfNdkVT/81VfcHTyqxJGvaB5kyBXY=
   "], {{0, 0}, {59, 21}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{154, 58},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{150, Automatic}},
 CellLabel->"Out[1]=",
 CellID->1396]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Time Steps",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(7)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->32380],

Cell["Two steps of evolution:", "ExampleText",
 CellID->18199],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"30", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "2"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->15392],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "1", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "0", ",", "0", ",", "1"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{322, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->29350]
}, Open  ]],

Cell["Alternative form:", "ExampleText",
 CellID->18339],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"30", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", "2", "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->29353],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "1", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "0", ",", "0", ",", "1"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{322, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->30867]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->789],

Cell["A list containing only the second step:", "ExampleText",
 CellID->15165],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"30", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"{", "2", "}"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->18025],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"{", 
   RowBox[{"1", ",", "1", ",", "0", ",", "0", ",", "1"}], "}"}], 
  "}"}]], "Output",
 ImageSize->{112, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->6635]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->20610],

Cell["The second step, not in a list:", "ExampleText",
 CellID->10454],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"30", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"{", "2", "}"}], "}"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->15681],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", "1", ",", "0", ",", "0", ",", "1"}], "}"}]], "Output",
 ImageSize->{98, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->12991]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->13142],

Cell["Steps 50 through 80:", "ExampleText",
 CellID->21766],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"30", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{"50", ",", "80"}], "}"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->21977],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJydlIF120AMQ8ku0peVOkIW6EzdtIktfIDKyZKsl1gn3YkEQJAfn3//fP6q
qt9f///q++r++uvn4vH7vGmxLbXu7cXzu9pWj6W/er7bXj7urfe1/9DxqpRj
g6MYCrMdfN6I3RFWNJTO0NtHO/IqrMHvPtRXbDbZQCsNt/A6DyuHNY3tePAv
CVcjr86Zsj/kMB9pX0C2VNJQBYzYkbuThvSppOXqkpdzUFb9y4e5s5/idGi4
9GEPhgLQThxlHCCHjmUik/POo+140i81XPpwMARAJEbyYcGhYxv04Gwd3RDZ
Xonw2IezCwXSibFy1Y+8PER9FCakckOMMk2Exz6EIS08EnfinnnjAQXgYUHK
UdJte4THPqQtvI5Bkbiju7hJZak0OiisMJDt8Z340EuTDTcF7uguPxQHFSIa
ZrRvRaQFwmMfquysJWrtcZsKDyZPiL2ja+Rf6JcaLn0oh1gyy5e4zUoPWyDU
6xG0xwtIr6+XPlzMQ5elJ6suA6XZQ5mdo/06PPNSw6UPLS7uyXEgVg2wHDnI
SeVtYgsWhE80vDoPnVQCloB1TDOqEYAwcQoWhE80XPoQ1kOB6Gvph2rZQcQI
7Tr1ow5nCG/NwzHEpJ9SjQ7iObSb+hHmkoZLH/6Yhw0Pwc392fkYLE0sh+g6
xXfiQ2dgzbSjvJR8dH6bMNPFcPm7hvDYhztV2gQETghiCEVTE6QyWHXgvq7h
0oe7eVgk1yETG5KBVHtxppPwHQ2XPrS47c3waHR0ELKi4Q+a7HT+HWl4ZR66
dScCZMdyjXs7wvv3zvXSh+JqERBrEosyB9LwRxP9ln6p4dKHmmQWgSmTxEaZ
QeoiWNi7+qWGSx+OeciehxvKGhN+wHoh7G39UsOzeRh7LcmbsSJMYRZRYWC/
o19quPQhIyz3GG5yJmRiRIkFs+89/cAhUV7MQ/aQPBuHJ8rp1vcQehvhpXkY
gtpxyiuxopwenXfn3wLhpXkoKey4dl41T5QTzOf++w9RGw2w
   "], {{0, 0}, {161, 31}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 41},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->32747]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->8582],

Cell["The initial condition is step 0:", "ExampleText",
 CellID->11507],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"30", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"0", ",", "2"}], "}"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2783],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "1", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "0", ",", "0", ",", "1"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{322, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->15797]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->6096],

Cell["Show every third step from 0 to 100: ", "ExampleText",
 CellID->26710],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"30", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{"0", ",", "100", ",", "3"}], "}"}], "}"}]}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->30259],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJzVV4tVwzAMlNgERmIEFmBM1oI2vo9M4qQhbYPeo01ix9LpdFJ5/fh8/3iJ
iLefv6+4WF4/8nJx+Zy+EivZ1i5Pc7qd9kS093Sf2H3de13SEdlW+STTdsBx
NsfTUgst9QCPU864iUHaizi2RUsPIawIJoiaZ+BJGlDDhny01x0Q75VBf0BA
eMwXMoXJkxktdB5JH6CicQOAiR2hzQQX0YWXYrYACvlN820JNEpxqgxhICsB
6uiVcEmF6sUYU/mV+kjtREEwMUGfpQBx68WE7CreXzbUiS4Zj8oqPArPC8u2
ObfqY0A6VmHAO3OrO2Vo2YY6gcZKCdpTy7mYLfmOEl8Rjb3IV0vRucAWeJjj
ZFYnyEGlgliDrl2bKssuIhVyfRHnMdp+7zYb6kRZxyq7Rt+zhED1qUyLQr7I
zcY6YRjLm22okz6bYF71IRGqJXiM3M4OloLlrJuuVvSwxsmsTtSHeG314TWn
itQXck+plsoUXPnPW3nokCzrpOiv+TEJEo4qko0mqZVwkEVx1f9+DESyrBNW
kIoHroPQrecUoYZOl6yVmd7/n22oE11avpnDEAiTChlkwUtkySzADsFAJMs6
kT4pEzQj4RczlA8AWBsSw1ash9pQJ+ws4Mx7qxelQSHJJQfJ0w/loUOyaZ7Y
WlpAHCupJHBNsk/W2P1sqBORpdohhSizBF8lZFK4ez7sQLJlnkTZztZvHYEV
CaISmx5kW+aJOg3SbFF7YyOlR8yHHUhW5wkxpcFSa001MpsoD7f1eSIcjoqp
pzaOng87kKzMk6IKtVaV053mw622Mk9QbEw/9cK29XwMk410UrThE4Nt60y2
qJP0ihMjlNDpbF4n+olh1Ez7nxns0OZ0guaKXxsY9+e23zqR1CNs/fTW64RD
++nz4VbrdAJE/4QHt6KTPNN8uNVMJ/+QB7fH/f9wb3sEhm/g5xYg
   "], {{0, 0}, {199, 34}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 37},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->14980]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->8007]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Region Specifications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(11)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->10633],

Cell[TextData[{
 "By default, ",
 Cell[BoxData[
  ButtonBox["CellularAutomaton",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CellularAutomaton"]], "InlineFormula"],
 " automatically cuts off the region not covered by the pattern: "
}], "ExampleText",
 CellID->14359],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"110", ",", " ", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", " ", "0"}], "}"}], ",", " ", "50"}], 
   "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->16286],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJyVlNFtwzAMRMlu0hm6SUfIAtn/rw0k8h5lSZYDJI5gk8d70vn79f59fZnZ
T//+f/zzaReLP59rv2uxiN/hkfJsu229o2ldHlGfdidVu0pfp5hlNw/BNmxM
rnWKRTd5S68OT65uadKzeQwbj/aGqg6TaA4Ouc5JcyyQsMJB0FkdJgUOHAAd
e5kme2fscYXO7fYgAYYDJta4rDcNHANiKjqdWkzHY0BMRafPb6h0GhhUJJzN
o2vu56giYVXK+QXyIHzNzwQy/eQomGkGmX6m+ZlApp9pfiaQ6Weanwlk+hnz
s4JMP2N+VpDpp+ZnDZl+an7WkOmH+cHAG52aHwy80an5wcBbP8zPDjL9IAW2
g0w/jnOzg0w/nhu0h0w/Xl8f9zWa8RaydC6vjwMdi1NzpsL8HECWTp6awxLk
5wCydPLUHJYgP8clmZ9zFeXnQcnd62Klcw5ZOueQpfNQ5SFk6TwteWq/1fwB
rAQJLg==
   "], {{0, 0}, {51, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{118, 114},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{114., Automatic}},
 CellLabel->"Out[1]=",
 CellID->7655]
}, Open  ]],

Cell["An equivalent form:", "ExampleText",
 CellID->14628],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"110", ",", " ", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", " ", "0"}], "}"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"50", ",", "Automatic"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->18197],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJyVlNFtwzAMRMlu0hm6SUfIAtn/rw0k8h5lSZYDJI5gk8d70vn79f59fZnZ
T//+f/zzaReLP59rv2uxiN/hkfJsu229o2ldHlGfdidVu0pfp5hlNw/BNmxM
rnWKRTd5S68OT65uadKzeQwbj/aGqg6TaA4Ouc5JcyyQsMJB0FkdJgUOHAAd
e5kme2fscYXO7fYgAYYDJta4rDcNHANiKjqdWkzHY0BMRafPb6h0GhhUJJzN
o2vu56giYVXK+QXyIHzNzwQy/eQomGkGmX6m+ZlApp9pfiaQ6Weanwlk+hnz
s4JMP2N+VpDpp+ZnDZl+an7WkOmH+cHAG52aHwy80an5wcBbP8zPDjL9IAW2
g0w/jnOzg0w/nhu0h0w/Xl8f9zWa8RaydC6vjwMdi1NzpsL8HECWTp6awxLk
5wCydPLUHJYgP8clmZ9zFeXnQcnd62Klcw5ZOueQpfNQ5SFk6TwteWq/1fwB
rAQJLg==
   "], {{0, 0}, {51, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{118, 114},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{114., Automatic}},
 CellLabel->"Out[2]=",
 CellID->14456]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->9245],

Cell["\<\
Include all cells that could possibly be affected given the structure of the \
rule:\
\>", "ExampleText",
 CellID->11633],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"110", ",", " ", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", " ", "0"}], "}"}], ",", " ", 
    RowBox[{"{", 
     RowBox[{"50", ",", "All"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->18846],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy9llGOwjAMRG1usmfYm+wRkPjm/n/Qxp5xNuSjiU0lVALCL2NPM/zcn3/3
m4j8vl8POS89rnYTf3Pc7VvxhXJ99eqLdtWPgraJhuD6OiUUFS/VRFGZKeGm
rms5y1qzrJS4Sijt1wtamiDfsWJGwvbIv/WCFlU0Cx5QRQu9+OLgXYuwWcF0
GhyA9TpF0azgAekcsG5hajGY+uCdpd7PDQtTy3lT84DQb/bxloWpBePFaPA5
N7IFQTuwYfqND8teu5TtgAA3RHhYtgbvJWwOElgaB7GthO2AAN85nsh9JWwH
WZx5goVBmeVLioVBAS1aeDiVdymzfEmxMCizfEmxMCizfEmxMCg65kuehUH5
kC95FgZlyJdMC4My5EumhUFpN/V8MXSmkg/5YuhUyJgvhk6FjPmSa2FQNOZL
PJVTKf2Zn2xhUJgvDkyHdPki+RYGBTQpsDAoPLsKLAyK50uJhUERnC1VSmK+
lFgYFALKICFf6pTEfCmEIF8qlTBfSiH5fyQmlFILg6KVFgalvF1abmFQvgCp
H3yjXP/JC/0mE1Y=
   "], {{0, 0}, {101, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->13774]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->13194],

Cell["Include only the region that differs from the background:", \
"ExampleText",
 CellID->9760],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"45", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "50"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->13041],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJyllNtVxDAMRCU6oQba4IsStoHt/w+wo5mRH7H2bGBhj+PY16MbfT6eP48P
M/v6+3xbv9y9/et/2k/7bTf6aB+K6W2G3HQ+G0tc96+FrC1mae0+Pfa+RtKt
vo7hfkD0BWVtdxf8trnhi2MdvxZOEB634qsR/zpxbIk/AR6HjC0jxWCWUfdI
1hNjgBo2MC2Dg5mjVxJBg80Iajw1yoBKuY4GPTN1ZirBsfRIMcYQjCSLBxDM
7BhSJHNwRy2IrbVZOWZ4Cmas/AUy6r5yTMXWgiPZHCQqNjsmYvOY4q8UX7mW
jiVpiO/cNqYkrtkxFVvM4HSaNHJNjlEaPbtWzGVK4podE6NgBj2K+flKbohj
NArCOEc8UQ5cs2PwiY1hajobrsExviyaAtPwMat05OwYXpa0GZ+Ysspnlpyd
7VBmSu5zVrkU6hjaYZrK3FdUaX1xLF7rYabW9JYrOcYuRze0pieu5BhfTHyw
yJpqOIU6FtVivwHn7hoWpmMu0Ykou6ymoyfHJDrtqluq8ex9wSBji9GuWuRS
xxjd0FWrXOFYim7sqkUu9iXnQceuWuXi64yDzl21yEV/onpTV61zuQYeb2Ab
P1EtImDzRK+wg1dbLohkCOnk1ZaLB4yvR6/2XHQMcZWo1m1bpLLdpCKXurSd
U+QSx+77VYVLHCt5dceljtWpdln0Bate3XKpY29zqWPvcqljb3OV+lWR6yWv
TlyvZvV//QLu3Amp
   "], {{0, 0}, {76, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{159, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{155., Automatic}},
 CellLabel->"Out[1]=",
 CellID->9970]
}, Open  ]],

Cell["Include all cells that could possibly be affected:", "ExampleText",
 CellID->25348],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"45", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"50", ",", "All"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->28732],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy1lUFWwzAMRCVuwhm4CUfoe6y5/46SWDNjJ3bkuC0E2sZW/CV9+/Px+/34
MLOv5/XzvLz3sv3m9m//s/1sv9uNMsLi7f5hD2iYXm51Xi6xy8TyBNueYtVD
9+FYm8VnDXCOUu4bJsay9yfJQ8twQ/Dy2TGmjxKRETXW5nEr3hqBS4ZiLT6q
CcpRMsVqRAUM8Vy+dXekyKoJ/aqUBceScIMVdhnu0QsBoavoVyUyJa2DKEKJ
b4OXdRuTkNmjbaseICVJ0Qim0UckHVdQKVIGaTSC12kakfRcMYRDI58JythD
kp4rqrSBVAStizUmOXdFlGbGRFBpyRTJqSuiNL0UQSl7juToiirNNFJQafw0
ycEVNr+mUfvIZUiO5OiKmIFGpg8xvglyjVK7QjPQ385v3M+CXpEcXYEX3HYP
e/0sSeOKbN6SUCbW23qkSBpXsH+YroKhTiNekzSuWF1r43aOstwgqVzBuVXF
YNHP4yVI1JXYG5sQ2mn3SCpXeByxlbXTbpNUrnATw4Xoo/kZlPiHR+HCiH6s
FEnlikt5pK979UiT1K5IefRcHE3PonBDpP31ubhKoq6wPM25uEwSrlTlac/F
VRKeE86ctefiMgn3ROTseC6uktCD6KnDufgCEtdqx261fZ/LQhqFe6U7c5aY
OoOCR5SZV37MkzBX8fbSjxskdAUlSW99UygxAx6+nESdmJo8QyKujM+PJRJx
JeXHLRJ1ZaovZ1FUx/eQqCvvI1FX3kairryPJHV+rJJM+XGbZLYe/68/rp0M
TA==
   "], {{0, 0}, {101, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->10046]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->15717],

Cell["\<\
By default, different rules give regions of different widths:\
\>", "ExampleText",
 CellID->989],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"ArrayPlot", "[", 
    RowBox[{
     RowBox[{"CellularAutomaton", "[", 
      RowBox[{"n", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "40"}], "]"}], ",", 
     
     RowBox[{"ImageSize", "->", 
      RowBox[{"{", 
       RowBox[{"Automatic", ",", "60"}], "}"}]}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "57", ",", "60"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->9822],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{313, 62},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->180824983]
}, Open  ]],

Cell["\<\
Force all rules of the same type to give regions of the same width:\
\>", "ExampleText",
 CellID->11940],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"ArrayPlot", "[", 
    RowBox[{
     RowBox[{"CellularAutomaton", "[", 
      RowBox[{"n", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"40", ",", "All"}], "}"}]}], "]"}], ",", 
     RowBox[{"ImageSize", "->", 
      RowBox[{"{", 
       RowBox[{"Automatic", ",", "60"}], "}"}]}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "57", ",", "60"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->11368],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   GraphicsBox[RasterBox[CompressedData["
1:eJytz8sNhTAMRFFDJ29JOVACDbz+dyCEEJ9xMrYncnb5nPtb/8s6mtm073nf
A1gGx8Cgy2DhJ9GL8DJJJIHeL1wyLSSTeSGZHBBSyREhlRwSEskxIZEcFHaT
o8JucljYSY4LO8kJYTM5I2wmp4SN5JywkZwUuslZoZucFjrJeaGTXBDC5IoQ
JpeEILkmBMlF4Se5Kvwkl4Wv5LrwlSwQPpIVwkeyRHhL1ghvySLhlawSXsky
4ZmsE57JQuGRrBQafZD9lwYOG68PCMc=
     "], {{0, 0}, {81, 41}}, {0, 1}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->{Automatic, 60}], ",", 
   GraphicsBox[RasterBox[CompressedData["
1:eJzNzLkRgDAUA1GLTggpB0pwA/Sf4djj4x8bsBql76zvU49SytV+txf9fcI9
VhQsChYFi4JFwWIf7WXFUbSXEWfRXlRcRXsRcRfteUVLtOcRrdGeVfREexbR
G+3txEi0txKj0d5MzER7IzEb7fUiEe0J94R7wj3hnuzeBxjlC78=
     "], {{0, 0}, {81, 41}}, {0, 1}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->{Automatic, 60}], ",", 
   GraphicsBox[RasterBox[CompressedData["
1:eJytzrERgDAMBEFDJ4SUAyW4AfrPcAFIej3nGWV/4z3mc899jHGuu9Zt/vuO
h/++Wxr4QxikNNAXRiUNtIVhSANdYdzRQFOYZDTQE2YVDbSEaUQDHWHe0EBD
WCQ0sC+sChrYFpYBDewK6z0NbAqFOQ3sCZU1DWwJpTEN7Ai1LQ1sCMUpDdSF
6pIGykJ9qH6sDl9l8Aiz
     "], {{0, 0}, {81, 41}}, {0, 1}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->{Automatic, 60}], ",", 
   GraphicsBox[RasterBox[CompressedData["
1:eJytlMENwzAIRUk36bHjtCNkge5/a2QqWzbkJ3xwhJBF8vKEEM/9+9kfIvI6
4n3EdvPIkqN1+8H/sHXHQB+27hhosHXHAPfnqu4Y4P+H+qcG+P1Q/9QgV3cM
cnXHAJ8or8/Z2RMHCg7CEAdhqB5nmTAUmAlDnAlDnOPAZd7o+euAZd7o+RtG
c7/o+RtGc7/o+RtGs0+qf2ok8B4HCr7HgRf3ODC7/wwwu/+sYXL/GeASeWB2
/xlgdv8ZYHb/GeCS88Cq+evAqvnrwKr568CtsH8NWNm/Bqz1K+5fA9598QcI
VA3v
     "], {{0, 0}, {81, 41}}, {0, 1}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->{Automatic, 60}]}], "}"}]], "Output",
 ImageSize->{504, 62},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->8481]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->25048],

Cell["\<\
The region that can possibly be affected depends on the range of the rule:\
\>", "ExampleText",
 CellID->18500],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"10008", ",", "2", ",", 
      RowBox[{"3", "/", "2"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"50", ",", "All"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19386],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJzNl9FxAzEIRCGdpIZ0khLcQMpJu7HOsCxn8ikEiceek8Z6t+D1+vPx8/34
EJGv5+P3+dBdJepvLtffel6XrquxtF5jq9D/LqyLRpxCHAzH2xJTvOg2aqVO
Awo+UEyTpBw27YOyI8U0gXCun1/UUM4u7YXS6JmQcIKFYParsnWmFEfaCxIO
LC6d3YPv2wtFw5SFC/3wQRDv5G6llO7bGviSivXDkw3UdqXU791652dzx0Ie
4bXdWGQAcbZANt8Ss99C5c3jbkXLsCWsvQcrnJKGGc7ps9TgU4lKycK9cZAt
ZqnBpxJWsnDh7vm8UWvbqNgW2A9Ay63twyJbSB5qi9LmU4mKOsd+EIttPpWw
uHPsoUHXrZQDZAtvzVMlUmQVxIbWPFVjRVZBfsKgt/sUqBSDji7ywgmlNNoG
g2jPUxUUDTpMszdPlVj4zUXfyb15qqLCINF3cm+eKrHoE6hn8lRFpeZO8dOm
O0+VWEJDfiZPVVR5yE/kqRIrDfmRPFVU5LtjearEyg51fKZedXeoCUrpm0MN
gXLvHOBTVOGdQ2bqKvbOMVCYch2klCq71aQ6nKf+qUk+tUrubjWjZuSpe83I
U28l03xqlYzzqavm+dSqeTN11UCltEmpP6HeHd8=
   "], {{0, 0}, {151, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 66},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->19413]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->24574],

Cell["\<\
Show only the region from cell 0 (the position of the initial 1) to cell 40:\
\>", "ExampleText",
 CellID->2625],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"30", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"50", ",", 
      RowBox[{"{", 
       RowBox[{"0", ",", "40"}], "}"}]}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->10926],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJytVEESgkAMS/2JR8/+xCf4Af9/U6RNUtwFDjKgzGw2TdOw1+fr8bwAuH+e
2+cJRMTyi+Xle33flpV8CEAtrVeA8PVB8XApb4Ov/+SrpaqUXFmzVCFyl1Wi
kKzG11ysSiu41axu4Q1nae+BBOVMUKxsQnQsVDGoJFWTvRSZWO0y9lIENAjK
t9wacG9aKxFtq+1pEFpat/YYJOSIuSLZFRGOgS7blGm9rFb9GhRlVxSYnppa
pafNWTF2fVZQFVx+G2+fMzRUwbqnnid0vj4OJqb755MzcjOiIdunBhmxwekD
sM6BLR91hjpXsDeVlRWo0x8+GLYi/KMvu27fBS0cIYNW+8E0QPLk4o4JoWIx
8K+JJGDin3HyY5jxKcqV1CkfD6+Jfw2rqO2geJTs6PNepv55YT8/9hgxzd+G
cJy/AaEUHNc+5PPqh3wn9CXdGb444V/jPMX4X76IN3KdB50=
   "], {{0, 0}, {41, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{104, 123},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{100, Automatic}},
 CellLabel->"Out[1]=",
 CellID->12664]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->21469],

Cell["Negative positions are on the left:", "ExampleText",
 CellID->15252],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"30", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"50", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "20"}], ",", "40"}], "}"}]}], "}"}]}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->32300],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJydlNtZwzAMhSU2YQaeWYIRukD3fyOtfS5y7DQJtIUvtnT0S0f9fjz/Hl8R
8bO9f7d3ZOT2E+2z//f++/p9P+X563J/tws4j8C/vIKnVGiJoz3u2XEapsLr
/RWlyFBhUUrAcb+Ok0hC9KcqUroJXmsBKIP1ovR2SjzDSauyJWC9LB1tdkIr
hYPoFOgJ2JJ4YKgtQBhHFsbGEsiQhiF4prV07F5PioP+sjBkYstYkJJqHHjE
dmGGZgidsW6n8HalmDLEHXa3UFgYR84F4Gp43aRQWLKTZizzLT9odq0JrUMX
yCywWXG1CcAeqp2N1bxpQrbG/a5btIy41UT2iX5nCN2H0jhvZXRfc8PN9Zyu
zdu97eM0v5py5S7eLuOkYwdh48Z0zI2YSv3CkS/Ca9Q6yBYaQJbwyl0yquPe
Eo+u3ObK8I5HTHUHbl22Ra7ztZlVbnOlWj/MVw2LgdtcSZqY8aZVySFZKn4t
THRtncRtruQaLnRjz20Lr3yTWDPRzJUofqrr2jtXKulC17XdlR601p1z16Cj
2D03fLLb31Pc3NRD3hU36/io69pq8HR/P2hzmw7nu9Iujj0ZO3BnnuQdtenw
C7GF+wLvqH1uvivt67qufSfWuO9G39Rt0fdjL+j+A7g4CgI=
   "], {{0, 0}, {61, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{154, 126},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{150, Automatic}},
 CellLabel->"Out[1]=",
 CellID->21986]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->2614],

Cell["Give the region consisting just of cell 0 at each step:", "ExampleText",
 CellID->11523],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"30", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"15", ",", 
     RowBox[{"{", "0", "}"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->28330],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWcsOgjAQXKgY1Pg8efRr9ORHEGOiB6NR/h9bfBzWJixuq60uCQ1bp52d
DbOgrotytz0U5X5TLFbn4rTbby6L5fGsp1QCkHYBYDwHfV3pq/p8HLk+ATpm
VM/5l7jX8Hn/z/CDwPIRvW7xygy3sUVVDD7VIWaLFYfnQ8vvX3Tg+y6tKHen
QWXQ3NtzR+tMPibMPsTnex3uEe/qi6Uu1DxjrYtvPporKbHpCU1PKOq6Oqvs
Gdc52rW4ZPW9DndMnsq4atQu21+olH/WVx+nDSjcRXA8ZOKVVbNrFtf4kRMV
YWqnZRVTBfywYKck9b7c5+KYiaf1EC6La/zEiYowtdOyiqkCflj8vGeGqVX8
JH4SP/mrqPhJ/CR+Ej99Q7v4icZi9xMFZX79wN/Gpgycsqjh7MfBzRj5fUuH
jTckHZz98LrbP6bV/chbRpBcAXrL/+I=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{518, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->239449859]
}, Open  ]],

Cell["Include only the value of cell 0, not in a list:", "ExampleText",
 CellID->4566],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"30", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"15", ",", 
     RowBox[{"{", 
      RowBox[{"{", "0", "}"}], "}"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->7741],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "1", ",", "1", ",", "0", ",", "1", ",", "1", ",", "1", ",", "0", ",", "0", 
   ",", "1", ",", "1", ",", "0", ",", "0", ",", "0", ",", "1", ",", "0", ",", 
   "1"}], "}"}]], "Output",
 ImageSize->{298, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->14578]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->629],

Cell["Show every other cell in time and space:", "ExampleText",
 CellID->31014],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"45", ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"0", ",", "100", ",", "2"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "100"}], ",", "100", ",", "2"}], "}"}]}], "}"}]}], "]"}],
   "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19494],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy9VsFxwzAMI7tJZ+gmHSF3fXf/X30xCYK26CiWVKXJ1ZYsCAQJ+vPx+/34
EJGv7fuzfbUaoiIxL88b24/dEJt+Xsv+J8+b9qwt2fcpQWyRbbQ/y9cMbzvb
yVRsWpTPVVAxJDum07ENggZICbHZ12CfCyoeGyWkmBOejN3jEE7nEqShight
f0Zm6h68F6OlCofHVaUrDxImXmG0VTFZxGXVuPRFJj4ov6LSVAXC4yRgAf4Q
p4NKs1YMP4QPap4GFoOu0a6VOLlG9PS4dy9GWSuew5QQRENiopNKSxXUXEAj
VixPN5V2rXgKIIVSjOiRTipFrXgCe9CQAJ31kanQvlErZF/CkJwBb1EpHIz9
KpaS7bxFpd1XjonqJnPdP2oqJ1XoBo4ORncwClVQeqRAT/+oqbRVIc8nIe5h
lLUinmDh6+/mbqJyUCWsXnA9iNFQBVYP8ZXCd5fKQRU3QiqQUYyTKkLSqH/u
5S6jnFRJXgynHwNxJMXmCjHI5QdRWJUwDxZ/GOOgSg4Np8AgSqjCVZgb/zCI
I7nusFx2r2EUNneh/0brI4EkH4ycvdc/SpSkCrrgVIyQGGk70j9KFDR35zOp
PhIIx0qQW5MH3n+Sq8wGsR92sPkj/FDm1kcCUU/eHXIFRtT4lP5Rgii1lUUY
/N6zDkNR7wsxlBriShDFa/BKlFX1kUDW5S6j/APGnVj9AXDaEpQ=
   "], {{0, 0}, {101, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->12184]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->10973],

Cell["\<\
Cell 0 is always the leftmost cell in the explicit part of the initial \
condition:\
\>", "ExampleText",
 CellID->22384],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{
   RowBox[{"CellularAutomaton", "[", 
    RowBox[{"30", ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1", ",", "1", ",", "0", ",", "1"}], "}"}], ",", "0"}], "}"}],
      ",", 
     RowBox[{"{", 
      RowBox[{"10", ",", 
       RowBox[{"{", 
        RowBox[{"0", ",", "15"}], "}"}]}], "}"}]}], "]"}], ",", 
   RowBox[{"Mesh", "->", "True"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2306],

Cell[BoxData[
 GraphicsBox[{
   RasterBox[{{0, 1, 0, 1, 1, 1, 1, 0, 1, 0, 0, 0, 1, 0, 1, 1}, {0, 1, 0, 0, 
    0, 0, 1, 0, 0, 1, 1, 0, 0, 1, 1, 1}, {0, 0, 1, 1, 0, 1, 1, 1, 0, 0, 1, 0, 
    1, 1, 1, 1}, {1, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 1}, {0, 1, 1, 
    0, 0, 0, 0, 0, 1, 0, 1, 1, 1, 1, 1, 1}, {1, 1, 1, 1, 0, 1, 1, 0, 0, 1, 1, 
    1, 1, 1, 1, 1}, {0, 0, 0, 1, 0, 0, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1}, {1, 0, 
    1, 1, 1, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1}, {1, 0, 0, 0, 1, 0, 1, 1, 1, 1, 
    1, 1, 1, 1, 1, 1}, {0, 1, 1, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1}, {0, 
    0, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1}}, {{0, 0}, {16, 11}}, {0, 
    1}], {{
     {GrayLevel[-1 + GoldenRatio]}, 
     LineBox[{{{0, 11}, {16, 11}}, {{0, 10}, {16, 10}}, {{0, 9}, {16, 9}}, {{
      0, 8}, {16, 8}}, {{0, 7}, {16, 7}}, {{0, 6}, {16, 6}}, {{0, 5}, {16, 
      5}}, {{0, 4}, {16, 4}}, {{0, 3}, {16, 3}}, {{0, 2}, {16, 2}}, {{0, 1}, {
      16, 1}}, {{0, 0}, {16, 0}}}]}, {
     {GrayLevel[-1 + GoldenRatio]}, 
     LineBox[{{{0, 0}, {0, 11}}, {{1, 0}, {1, 11}}, {{2, 0}, {2, 11}}, {{3, 
      0}, {3, 11}}, {{4, 0}, {4, 11}}, {{5, 0}, {5, 11}}, {{6, 0}, {6, 
      11}}, {{7, 0}, {7, 11}}, {{8, 0}, {8, 11}}, {{9, 0}, {9, 11}}, {{10, 
      0}, {10, 11}}, {{11, 0}, {11, 11}}, {{12, 0}, {12, 11}}, {{13, 0}, {13, 
      11}}, {{14, 0}, {14, 11}}, {{15, 0}, {15, 11}}, {{16, 0}, {16, 
      11}}}]}}},
  Frame->False,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{154, 105},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{150, Automatic}},
 CellLabel->"Out[1]=",
 CellID->22299]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->27215],

Cell["\<\
Repeat a finite block to fill the region of initial conditions from positions \
-40 to 40: \
\>", "ExampleText",
 CellID->1643],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"30", ",", 
    RowBox[{"{", 
     RowBox[{
     "1", ",", "1", ",", "0", ",", "1", ",", "0", ",", "0", ",", "1"}], "}"}],
     ",", 
    RowBox[{"{", 
     RowBox[{"30", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"-", "40"}], ",", "40"}], "}"}]}], "}"}]}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1634],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJzt0sEJwCAQRNGZNBJSUlKCDaT/WzYqgjch35NBln8TV96R7ittkvaYM8b5
wDEcsYl70SiXTL7aYFx+E4vq5ljK5uby0/4e1LTbF4K0O5REhsEOBzTdXriU
6Rm0SdP1gSxt0vQk2qDpVWmjpt88dzAGwA==
   "], {{0, 0}, {81, 31}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None},
  ImageMargins->0.]], "Output",
 ImageSize->{229, 92},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{225, Automatic}},
 CellLabel->"Out[1]=",
 CellID->18761]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Multidimensional Rules",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(10)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->14626],

Cell["\<\
Evolve range-1 2D (9-neighbor) totalistic code 14 for 2 steps:\
\>", "ExampleText",
 CellID->32030],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"14", ",", " ", 
     RowBox[{"{", 
      RowBox[{"2", ",", " ", "1"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"1", ",", " ", "1"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"{", "1", "}"}], "}"}], ",", "0"}], "}"}], ",", "2"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->32516],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztW8FSwjAQDS2I4qB48+T0G/wIPfkRDOMMHBwd5P+xKbQyy7bJNgmh7WMG
ym7S3WfyGpJ948dyt/78Wu42q2X2vl3+rDer3+zte5u70pFSyYtS6vVZ5d/3
+bfiXb6y/P1vHqx7/ZlWvdSY2HfEfnS0afyZId8C+bzmo/dL89N81E71xxkK
DmWSm4ua6xMTVftp3NlJ/z7GWdTEo3FMs5JYzYruNclN6r8J7Kf2rSUu3U+b
E+KfCv3AFRdXXbypsL8trnZPB2frZ5GusD79Lv2K0Z9U/uJvVFXr1KEVSK8f
aR3LOJa7stL8bJmi0BXGdZdJ480N7Q/s3MwFXiBwR+C6N6XxTNwcFbil677v
E5Cp3f+8AIEcQehzk589Cbg5RATgJtcef16AANzk2+PPCxDE4qap4q57NdVG
yyvd7dKaYBlH+9PjaNFrl+PU1Ubrxseucr0/vrJgFhSXYeWD4tKlOG0VlzLu
2Go2dO+UWaOklXip31Shb8LloxIPXPFxcX5fypCf3XnRK0iVvW0rRctVxptR
+6i7A2M3MDazLIyGZHoK9d2HnYJJi0nZHJJzmcRr0gh4NNI9FHK75+ZjuZ/i
rX7bWv5mhGKtqZLAo/Eze8gtyR2KtaY1t1hlwVrkbpUbrHUdwaEyB6wFa5Fb
kjsua23UIG1yasnpld7HqSapalaVuhzn8SSezfiwVfxqArNgllQzhBrUr3yX
UYNS4V1Qh+zi+FGHpLPTNbXIl5pwKVxS9WUo43UdapH0aemjehTu/07ioZYr
MxhleebYalNi0fts/3Ll6pP72TA8GrsTc39HJa5a5ec367pqU6Hm1ycaF9b3
YVTi1rLAerAerAfrwXqwHqwPgwasj8l6qXoTQr3zHSdh4l4aT1z1To3+AJei
r5s=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{542, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->70310847]
}, Open  ]],

Cell["Give only the result after 2 steps:", "ExampleText",
 CellID->9389],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"14", ",", " ", 
     RowBox[{"{", 
      RowBox[{"2", ",", " ", "1"}], "}"}], ",", " ", 
     RowBox[{"{", 
      RowBox[{"1", ",", " ", "1"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"{", "1", "}"}], "}"}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"{", "2", "}"}], "}"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->30870],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWNEOwTAUvdvMkGA8efQ1PPmIRSQ8CGH/Tzu2yNmqbVoMd8na3dPr9Kz3
trqusny72Wf5bp3Nl6fsuN2tz/PF4SSgKCAKUyIaz0g8X8RTcZfXVNxEHVlG
FV6z+2CPHW3kH2j6S7k/r/3h7237x/4iWdxKjapQmKminjSwShzfZvDg/4s8
qYIPeXB8Q6MoSK+Y6nO++2Ic7Z6hLuknzRjwxBJnXZ/VpeJLLP1NdZnNhiZb
zjVcMX3iLn7FaMcVXrwjVa2JQysrbb9SVZbhP4mPrKzPpVDjhSuI6y4R+Yaa
9lFjLIYWKCtwV+C6t0Q+zLKg0Gm7rvv+YtG1+48DK7BX4Ps7x88eg3PxHxVw
LrYjDqyAc7EtcWAF78pFE69nZ5NljbtTPJMreSQe3UcH62/mUZ1Nqsan+aT4
cr+mlhYFVzIyYOo=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{528, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->466909047]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->7994],

Cell["Show the result after 30 steps:", "ExampleText",
 CellID->16999],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"14", ",", " ", 
      RowBox[{"{", 
       RowBox[{"2", ",", " ", "1"}], "}"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"1", ",", " ", "1"}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"{", "1", "}"}], "}"}], ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{"{", "30", "}"}], "}"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->12926],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJzFltERgzAIhkM36SBdoiN0ge7/1p4GAgSQBr1STTXJFwI/et5f7+fr1lp7
9LNmsG7/hhej3drVXI0mvQaA4OpwKnqH+55Zz306h4GGImz/kzCtF+q+D/dF
Zng7fBjAhffrGJZzeQcTzNjwGJ9gMahyj/kQ1SBhwB6jYnk/WLC88m9MzxS9
3sKI5UqYJ3Yq+SGWB1NarcuDbfeEKriRggFM7zSpFHXqvCtNQngSzfUM9Mt6
7rWIzxgKwEZCuA1N8cCOPKw8J2Hs4DGzYkvBMOx3eM1zKeZKtks6FyrsrNo2
4Nhz6XmuvUmkJmqrRzBqCkbL5lwDs+SaNyasXXETWzA9Uw0A0BRcdmRerIeR
ykGQpha3YFTSs2ku64DKNwmJ4cI8HwOGyneYzm9kOH1qjhaAGVEKJhY4D16w
bKSe/1DXI/snXLEPsIcIrA==
   "], {{0, 0}, {61, 61}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None},
  ImageMargins->0.]], "Output",
 ImageSize->{154, 150},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{150, Automatic}},
 CellLabel->"Out[1]=",
 CellID->5075]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->24504],

Cell["Show the mean color of each cell:", "ExampleText",
 CellID->24141],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Mean", "[", 
   RowBox[{"CellularAutomaton", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"14", ",", " ", 
       RowBox[{"{", 
        RowBox[{"2", ",", " ", "1"}], "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"1", ",", " ", "1"}], "}"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", "1", "}"}], "}"}], ",", "0"}], "}"}], ",", "30"}], "]"}],
    "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->26096],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJzFl1l6gzAMhEGLZUIu0YP0Ej1CLtD7v3XGkPULS9FDTNhs/bI0dsB8XX5/
LtJ13fe8nz9YxvN4pICb8EPwDT39t1zxJMwb7MO+Mlvf4ZFk3VfIj48w2BNr
txw0k9PU+wN8msB1ejKB7QuMhiFqjTV6NsD2BLeaISJKrMBojRhm+hlGQw1G
tZgv8Gb1BiaJtqW8h6mx+X8Hw28LfbJ6CjhqCzlm7y8w1ChUo7gb7Wq5F8Za
zL1QskJNX+Ha3MLUKRrOdi3EIZbj3IKrb2CGEw4bdFGqe8xsuMOXt5aYBVmA
AYabAXVGihzcpgo6WIHrUGAb4BTBqjJ3nswUHhCVlavhGxi5om8TGGNCqLQN
V3AmCAOS1hWYKptjKyqIFQgCFp3r7kO4DCuCduVRetC4ch43YYxIRSeFLBB6
mC4hGzSP1bCRNHQOQlBKWCgY7lHrt5SXBEMXpZBQ6VVMtEfK8INJw+Ffh9GB
BkMVJs1R4hXUx882YcZLwXoN6byTwFUTjE2bsBUOkHBo1dqx3beWTRgRAhWx
Ti1MO4No8MA5ugeWphUn9dimOFWTvXCi51TOGbUz45yZYam5nflXpf7PqSdJ
5hmWeXqmntuJN0bmXZV5S6bez8mVQWpNcng1lFqHJVeAibVnbtV7fL39qc+E
3NdN7rvqM+UPKc56Vg==
   "], {{0, 0}, {61, 61}}, {0, 13}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{166, 162},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{162., Automatic}},
 CellLabel->"Out[1]=",
 CellID->1618]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->31483],

Cell["Show a cube at the position of each 1 cell:", "ExampleText",
 CellID->26756],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{"Cuboid", "/@", 
    RowBox[{"Position", "[", 
     RowBox[{
      RowBox[{"CellularAutomaton", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"14", ",", " ", 
          RowBox[{"{", 
           RowBox[{"2", ",", " ", "1"}], "}"}], ",", " ", 
          RowBox[{"{", 
           RowBox[{"1", ",", " ", "1"}], "}"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"{", "1", "}"}], "}"}], ",", "0"}], "}"}], ",", "10"}], 
       "]"}], ",", "1"}], "]"}]}], ",", 
   RowBox[{"ViewVertical", "->", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "1"}], ",", "0", ",", "0"}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->5771],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{199, 145},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 Graphics3DBoxOptions->{ImageSize->{199., Automatic}},
 CellLabel->"Out[1]=",
 CellID->79060237]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"Opacity", "[", ".2", "]"}], ",", 
     RowBox[{"EdgeForm", "[", "]"}], ",", 
     RowBox[{"Cuboid", "/@", 
      RowBox[{"Position", "[", 
       RowBox[{
        RowBox[{"CellularAutomaton", "[", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"14", ",", " ", 
            RowBox[{"{", 
             RowBox[{"2", ",", " ", "1"}], "}"}], ",", " ", 
            RowBox[{"{", 
             RowBox[{"1", ",", " ", "1"}], "}"}]}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{
            RowBox[{"{", 
             RowBox[{"{", "1", "}"}], "}"}], ",", "0"}], "}"}], ",", "10"}], 
         "]"}], ",", "1"}], "]"}]}]}], "}"}], ",", 
   RowBox[{"ViewVertical", "->", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "1"}], ",", "0", ",", "0"}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->15031],

Cell[BoxData[
 Graphics3DBox[
  {Opacity[0.2], EdgeForm[None], CuboidBox[{1, 11, 11}], 
   CuboidBox[{2, 10, 10}], CuboidBox[{2, 10, 11}], CuboidBox[{2, 10, 12}], 
   CuboidBox[{2, 11, 10}], CuboidBox[{2, 11, 11}], CuboidBox[{2, 11, 12}], 
   CuboidBox[{2, 12, 10}], CuboidBox[{2, 12, 11}], CuboidBox[{2, 12, 12}], 
   CuboidBox[{3, 9, 9}], CuboidBox[{3, 9, 10}], CuboidBox[{3, 9, 11}], 
   CuboidBox[{3, 9, 12}], CuboidBox[{3, 9, 13}], CuboidBox[{3, 10, 9}], 
   CuboidBox[{3, 10, 13}], CuboidBox[{3, 11, 9}], CuboidBox[{3, 11, 13}], 
   CuboidBox[{3, 12, 9}], CuboidBox[{3, 12, 13}], CuboidBox[{3, 13, 9}], 
   CuboidBox[{3, 13, 10}], CuboidBox[{3, 13, 11}], CuboidBox[{3, 13, 12}], 
   CuboidBox[{3, 13, 13}], CuboidBox[{4, 8, 8}], CuboidBox[{4, 8, 9}], 
   CuboidBox[{4, 8, 10}], CuboidBox[{4, 8, 11}], CuboidBox[{4, 8, 12}], 
   CuboidBox[{4, 8, 13}], CuboidBox[{4, 8, 14}], CuboidBox[{4, 9, 8}], 
   CuboidBox[{4, 9, 9}], CuboidBox[{4, 9, 11}], CuboidBox[{4, 9, 13}], 
   CuboidBox[{4, 9, 14}], CuboidBox[{4, 10, 8}], CuboidBox[{4, 10, 11}], 
   CuboidBox[{4, 10, 14}], CuboidBox[{4, 11, 8}], CuboidBox[{4, 11, 9}], 
   CuboidBox[{4, 11, 10}], CuboidBox[{4, 11, 12}], CuboidBox[{4, 11, 13}], 
   CuboidBox[{4, 11, 14}], CuboidBox[{4, 12, 8}], CuboidBox[{4, 12, 11}], 
   CuboidBox[{4, 12, 14}], CuboidBox[{4, 13, 8}], CuboidBox[{4, 13, 9}], 
   CuboidBox[{4, 13, 11}], CuboidBox[{4, 13, 13}], CuboidBox[{4, 13, 14}], 
   CuboidBox[{4, 14, 8}], CuboidBox[{4, 14, 9}], CuboidBox[{4, 14, 10}], 
   CuboidBox[{4, 14, 11}], CuboidBox[{4, 14, 12}], CuboidBox[{4, 14, 13}], 
   CuboidBox[{4, 14, 14}], CuboidBox[{5, 7, 7}], CuboidBox[{5, 7, 8}], 
   CuboidBox[{5, 7, 9}], CuboidBox[{5, 7, 10}], CuboidBox[{5, 7, 11}], 
   CuboidBox[{5, 7, 12}], CuboidBox[{5, 7, 13}], CuboidBox[{5, 7, 14}], 
   CuboidBox[{5, 7, 15}], CuboidBox[{5, 8, 7}], CuboidBox[{5, 8, 15}], 
   CuboidBox[{5, 9, 7}], CuboidBox[{5, 9, 15}], CuboidBox[{5, 10, 7}], 
   CuboidBox[{5, 10, 15}], CuboidBox[{5, 11, 7}], CuboidBox[{5, 11, 15}], 
   CuboidBox[{5, 12, 7}], CuboidBox[{5, 12, 15}], CuboidBox[{5, 13, 7}], 
   CuboidBox[{5, 13, 15}], CuboidBox[{5, 14, 7}], CuboidBox[{5, 14, 15}], 
   CuboidBox[{5, 15, 7}], CuboidBox[{5, 15, 8}], CuboidBox[{5, 15, 9}], 
   CuboidBox[{5, 15, 10}], CuboidBox[{5, 15, 11}], CuboidBox[{5, 15, 12}], 
   CuboidBox[{5, 15, 13}], CuboidBox[{5, 15, 14}], CuboidBox[{5, 15, 15}], 
   CuboidBox[{6, 6, 6}], CuboidBox[{6, 6, 7}], CuboidBox[{6, 6, 8}], 
   CuboidBox[{6, 6, 9}], CuboidBox[{6, 6, 10}], CuboidBox[{6, 6, 11}], 
   CuboidBox[{6, 6, 12}], CuboidBox[{6, 6, 13}], CuboidBox[{6, 6, 14}], 
   CuboidBox[{6, 6, 15}], CuboidBox[{6, 6, 16}], CuboidBox[{6, 7, 6}], 
   CuboidBox[{6, 7, 7}], CuboidBox[{6, 7, 9}], CuboidBox[{6, 7, 10}], 
   CuboidBox[{6, 7, 11}], CuboidBox[{6, 7, 12}], CuboidBox[{6, 7, 13}], 
   CuboidBox[{6, 7, 15}], CuboidBox[{6, 7, 16}], CuboidBox[{6, 8, 6}], 
   CuboidBox[{6, 8, 9}], CuboidBox[{6, 8, 10}], CuboidBox[{6, 8, 11}], 
   CuboidBox[{6, 8, 12}], CuboidBox[{6, 8, 13}], CuboidBox[{6, 8, 16}], 
   CuboidBox[{6, 9, 6}], CuboidBox[{6, 9, 7}], CuboidBox[{6, 9, 8}], 
   CuboidBox[{6, 9, 14}], CuboidBox[{6, 9, 15}], CuboidBox[{6, 9, 16}], 
   CuboidBox[{6, 10, 6}], CuboidBox[{6, 10, 7}], CuboidBox[{6, 10, 8}], 
   CuboidBox[{6, 10, 14}], CuboidBox[{6, 10, 15}], CuboidBox[{6, 10, 16}], 
   CuboidBox[{6, 11, 6}], CuboidBox[{6, 11, 7}], CuboidBox[{6, 11, 8}], 
   CuboidBox[{6, 11, 14}], CuboidBox[{6, 11, 15}], CuboidBox[{6, 11, 16}], 
   CuboidBox[{6, 12, 6}], CuboidBox[{6, 12, 7}], CuboidBox[{6, 12, 8}], 
   CuboidBox[{6, 12, 14}], CuboidBox[{6, 12, 15}], CuboidBox[{6, 12, 16}], 
   CuboidBox[{6, 13, 6}], CuboidBox[{6, 13, 7}], CuboidBox[{6, 13, 8}], 
   CuboidBox[{6, 13, 14}], CuboidBox[{6, 13, 15}], CuboidBox[{6, 13, 16}], 
   CuboidBox[{6, 14, 6}], CuboidBox[{6, 14, 9}], CuboidBox[{6, 14, 10}], 
   CuboidBox[{6, 14, 11}], CuboidBox[{6, 14, 12}], CuboidBox[{6, 14, 13}], 
   CuboidBox[{6, 14, 16}], CuboidBox[{6, 15, 6}], CuboidBox[{6, 15, 7}], 
   CuboidBox[{6, 15, 9}], CuboidBox[{6, 15, 10}], CuboidBox[{6, 15, 11}], 
   CuboidBox[{6, 15, 12}], CuboidBox[{6, 15, 13}], CuboidBox[{6, 15, 15}], 
   CuboidBox[{6, 15, 16}], CuboidBox[{6, 16, 6}], CuboidBox[{6, 16, 7}], 
   CuboidBox[{6, 16, 8}], CuboidBox[{6, 16, 9}], CuboidBox[{6, 16, 10}], 
   CuboidBox[{6, 16, 11}], CuboidBox[{6, 16, 12}], CuboidBox[{6, 16, 13}], 
   CuboidBox[{6, 16, 14}], CuboidBox[{6, 16, 15}], CuboidBox[{6, 16, 16}], 
   CuboidBox[{7, 5, 5}], CuboidBox[{7, 5, 6}], CuboidBox[{7, 5, 7}], 
   CuboidBox[{7, 5, 8}], CuboidBox[{7, 5, 9}], CuboidBox[{7, 5, 10}], 
   CuboidBox[{7, 5, 11}], CuboidBox[{7, 5, 12}], CuboidBox[{7, 5, 13}], 
   CuboidBox[{7, 5, 14}], CuboidBox[{7, 5, 15}], CuboidBox[{7, 5, 16}], 
   CuboidBox[{7, 5, 17}], CuboidBox[{7, 6, 5}], CuboidBox[{7, 6, 17}], 
   CuboidBox[{7, 7, 5}], CuboidBox[{7, 7, 17}], CuboidBox[{7, 8, 5}], 
   CuboidBox[{7, 8, 17}], CuboidBox[{7, 9, 5}], CuboidBox[{7, 9, 10}], 
   CuboidBox[{7, 9, 11}], CuboidBox[{7, 9, 12}], CuboidBox[{7, 9, 17}], 
   CuboidBox[{7, 10, 5}], CuboidBox[{7, 10, 9}], CuboidBox[{7, 10, 13}], 
   CuboidBox[{7, 10, 17}], CuboidBox[{7, 11, 5}], CuboidBox[{7, 11, 9}], 
   CuboidBox[{7, 11, 13}], CuboidBox[{7, 11, 17}], CuboidBox[{7, 12, 5}], 
   CuboidBox[{7, 12, 9}], CuboidBox[{7, 12, 13}], CuboidBox[{7, 12, 17}], 
   CuboidBox[{7, 13, 5}], CuboidBox[{7, 13, 10}], CuboidBox[{7, 13, 11}], 
   CuboidBox[{7, 13, 12}], CuboidBox[{7, 13, 17}], CuboidBox[{7, 14, 5}], 
   CuboidBox[{7, 14, 17}], CuboidBox[{7, 15, 5}], CuboidBox[{7, 15, 17}], 
   CuboidBox[{7, 16, 5}], CuboidBox[{7, 16, 17}], CuboidBox[{7, 17, 5}], 
   CuboidBox[{7, 17, 6}], CuboidBox[{7, 17, 7}], CuboidBox[{7, 17, 8}], 
   CuboidBox[{7, 17, 9}], CuboidBox[{7, 17, 10}], CuboidBox[{7, 17, 11}], 
   CuboidBox[{7, 17, 12}], CuboidBox[{7, 17, 13}], CuboidBox[{7, 17, 14}], 
   CuboidBox[{7, 17, 15}], CuboidBox[{7, 17, 16}], CuboidBox[{7, 17, 17}], 
   CuboidBox[{8, 4, 4}], CuboidBox[{8, 4, 5}], CuboidBox[{8, 4, 6}], 
   CuboidBox[{8, 4, 7}], CuboidBox[{8, 4, 8}], CuboidBox[{8, 4, 9}], 
   CuboidBox[{8, 4, 10}], CuboidBox[{8, 4, 11}], CuboidBox[{8, 4, 12}], 
   CuboidBox[{8, 4, 13}], CuboidBox[{8, 4, 14}], CuboidBox[{8, 4, 15}], 
   CuboidBox[{8, 4, 16}], CuboidBox[{8, 4, 17}], CuboidBox[{8, 4, 18}], 
   CuboidBox[{8, 5, 4}], CuboidBox[{8, 5, 5}], CuboidBox[{8, 5, 7}], 
   CuboidBox[{8, 5, 8}], CuboidBox[{8, 5, 9}], CuboidBox[{8, 5, 10}], 
   CuboidBox[{8, 5, 11}], CuboidBox[{8, 5, 12}], CuboidBox[{8, 5, 13}], 
   CuboidBox[{8, 5, 14}], CuboidBox[{8, 5, 15}], CuboidBox[{8, 5, 17}], 
   CuboidBox[{8, 5, 18}], CuboidBox[{8, 6, 4}], CuboidBox[{8, 6, 7}], 
   CuboidBox[{8, 6, 8}], CuboidBox[{8, 6, 9}], CuboidBox[{8, 6, 10}], 
   CuboidBox[{8, 6, 11}], CuboidBox[{8, 6, 12}], CuboidBox[{8, 6, 13}], 
   CuboidBox[{8, 6, 14}], CuboidBox[{8, 6, 15}], CuboidBox[{8, 6, 18}], 
   CuboidBox[{8, 7, 4}], CuboidBox[{8, 7, 5}], CuboidBox[{8, 7, 6}], 
   CuboidBox[{8, 7, 16}], CuboidBox[{8, 7, 17}], CuboidBox[{8, 7, 18}], 
   CuboidBox[{8, 8, 4}], CuboidBox[{8, 8, 5}], CuboidBox[{8, 8, 6}], 
   CuboidBox[{8, 8, 9}], CuboidBox[{8, 8, 10}], CuboidBox[{8, 8, 11}], 
   CuboidBox[{8, 8, 12}], CuboidBox[{8, 8, 13}], CuboidBox[{8, 8, 16}], 
   CuboidBox[{8, 8, 17}], CuboidBox[{8, 8, 18}], CuboidBox[{8, 9, 4}], 
   CuboidBox[{8, 9, 5}], CuboidBox[{8, 9, 6}], CuboidBox[{8, 9, 8}], 
   CuboidBox[{8, 9, 9}], CuboidBox[{8, 9, 10}], CuboidBox[{8, 9, 11}], 
   CuboidBox[{8, 9, 12}], CuboidBox[{8, 9, 13}], CuboidBox[{8, 9, 14}], 
   CuboidBox[{8, 9, 16}], CuboidBox[{8, 9, 17}], CuboidBox[{8, 9, 18}], 
   CuboidBox[{8, 10, 4}], CuboidBox[{8, 10, 5}], CuboidBox[{8, 10, 6}], 
   CuboidBox[{8, 10, 8}], CuboidBox[{8, 10, 9}], CuboidBox[{8, 10, 11}], 
   CuboidBox[{8, 10, 13}], CuboidBox[{8, 10, 14}], CuboidBox[{8, 10, 16}], 
   CuboidBox[{8, 10, 17}], CuboidBox[{8, 10, 18}], CuboidBox[{8, 11, 4}], 
   CuboidBox[{8, 11, 5}], CuboidBox[{8, 11, 6}], CuboidBox[{8, 11, 8}], 
   CuboidBox[{8, 11, 9}], CuboidBox[{8, 11, 10}], CuboidBox[{8, 11, 12}], 
   CuboidBox[{8, 11, 13}], CuboidBox[{8, 11, 14}], CuboidBox[{8, 11, 16}], 
   CuboidBox[{8, 11, 17}], CuboidBox[{8, 11, 18}], CuboidBox[{8, 12, 4}], 
   CuboidBox[{8, 12, 5}], CuboidBox[{8, 12, 6}], CuboidBox[{8, 12, 8}], 
   CuboidBox[{8, 12, 9}], CuboidBox[{8, 12, 11}], CuboidBox[{8, 12, 13}], 
   CuboidBox[{8, 12, 14}], CuboidBox[{8, 12, 16}], CuboidBox[{8, 12, 17}], 
   CuboidBox[{8, 12, 18}], CuboidBox[{8, 13, 4}], CuboidBox[{8, 13, 5}], 
   CuboidBox[{8, 13, 6}], CuboidBox[{8, 13, 8}], CuboidBox[{8, 13, 9}], 
   CuboidBox[{8, 13, 10}], CuboidBox[{8, 13, 11}], CuboidBox[{8, 13, 12}], 
   CuboidBox[{8, 13, 13}], CuboidBox[{8, 13, 14}], CuboidBox[{8, 13, 16}], 
   CuboidBox[{8, 13, 17}], CuboidBox[{8, 13, 18}], CuboidBox[{8, 14, 4}], 
   CuboidBox[{8, 14, 5}], CuboidBox[{8, 14, 6}], CuboidBox[{8, 14, 9}], 
   CuboidBox[{8, 14, 10}], CuboidBox[{8, 14, 11}], CuboidBox[{8, 14, 12}], 
   CuboidBox[{8, 14, 13}], CuboidBox[{8, 14, 16}], CuboidBox[{8, 14, 17}], 
   CuboidBox[{8, 14, 18}], CuboidBox[{8, 15, 4}], CuboidBox[{8, 15, 5}], 
   CuboidBox[{8, 15, 6}], CuboidBox[{8, 15, 16}], CuboidBox[{8, 15, 17}], 
   CuboidBox[{8, 15, 18}], CuboidBox[{8, 16, 4}], CuboidBox[{8, 16, 7}], 
   CuboidBox[{8, 16, 8}], CuboidBox[{8, 16, 9}], CuboidBox[{8, 16, 10}], 
   CuboidBox[{8, 16, 11}], CuboidBox[{8, 16, 12}], CuboidBox[{8, 16, 13}], 
   CuboidBox[{8, 16, 14}], CuboidBox[{8, 16, 15}], CuboidBox[{8, 16, 18}], 
   CuboidBox[{8, 17, 4}], CuboidBox[{8, 17, 5}], CuboidBox[{8, 17, 7}], 
   CuboidBox[{8, 17, 8}], CuboidBox[{8, 17, 9}], CuboidBox[{8, 17, 10}], 
   CuboidBox[{8, 17, 11}], CuboidBox[{8, 17, 12}], CuboidBox[{8, 17, 13}], 
   CuboidBox[{8, 17, 14}], CuboidBox[{8, 17, 15}], CuboidBox[{8, 17, 17}], 
   CuboidBox[{8, 17, 18}], CuboidBox[{8, 18, 4}], CuboidBox[{8, 18, 5}], 
   CuboidBox[{8, 18, 6}], CuboidBox[{8, 18, 7}], CuboidBox[{8, 18, 8}], 
   CuboidBox[{8, 18, 9}], CuboidBox[{8, 18, 10}], CuboidBox[{8, 18, 11}], 
   CuboidBox[{8, 18, 12}], CuboidBox[{8, 18, 13}], CuboidBox[{8, 18, 14}], 
   CuboidBox[{8, 18, 15}], CuboidBox[{8, 18, 16}], CuboidBox[{8, 18, 17}], 
   CuboidBox[{8, 18, 18}], CuboidBox[{9, 3, 3}], CuboidBox[{9, 3, 4}], 
   CuboidBox[{9, 3, 5}], CuboidBox[{9, 3, 6}], CuboidBox[{9, 3, 7}], 
   CuboidBox[{9, 3, 8}], CuboidBox[{9, 3, 9}], CuboidBox[{9, 3, 10}], 
   CuboidBox[{9, 3, 11}], CuboidBox[{9, 3, 12}], CuboidBox[{9, 3, 13}], 
   CuboidBox[{9, 3, 14}], CuboidBox[{9, 3, 15}], CuboidBox[{9, 3, 16}], 
   CuboidBox[{9, 3, 17}], CuboidBox[{9, 3, 18}], CuboidBox[{9, 3, 19}], 
   CuboidBox[{9, 4, 3}], CuboidBox[{9, 4, 19}], CuboidBox[{9, 5, 3}], 
   CuboidBox[{9, 5, 19}], CuboidBox[{9, 6, 3}], CuboidBox[{9, 6, 19}], 
   CuboidBox[{9, 7, 3}], CuboidBox[{9, 7, 19}], CuboidBox[{9, 8, 3}], 
   CuboidBox[{9, 8, 8}], CuboidBox[{9, 8, 14}], CuboidBox[{9, 8, 19}], 
   CuboidBox[{9, 9, 3}], CuboidBox[{9, 9, 19}], CuboidBox[{9, 10, 3}], 
   CuboidBox[{9, 10, 19}], CuboidBox[{9, 11, 3}], CuboidBox[{9, 11, 19}], 
   CuboidBox[{9, 12, 3}], CuboidBox[{9, 12, 19}], CuboidBox[{9, 13, 3}], 
   CuboidBox[{9, 13, 19}], CuboidBox[{9, 14, 3}], CuboidBox[{9, 14, 8}], 
   CuboidBox[{9, 14, 14}], CuboidBox[{9, 14, 19}], CuboidBox[{9, 15, 3}], 
   CuboidBox[{9, 15, 19}], CuboidBox[{9, 16, 3}], CuboidBox[{9, 16, 19}], 
   CuboidBox[{9, 17, 3}], CuboidBox[{9, 17, 19}], CuboidBox[{9, 18, 3}], 
   CuboidBox[{9, 18, 19}], CuboidBox[{9, 19, 3}], CuboidBox[{9, 19, 4}], 
   CuboidBox[{9, 19, 5}], CuboidBox[{9, 19, 6}], CuboidBox[{9, 19, 7}], 
   CuboidBox[{9, 19, 8}], CuboidBox[{9, 19, 9}], CuboidBox[{9, 19, 10}], 
   CuboidBox[{9, 19, 11}], CuboidBox[{9, 19, 12}], CuboidBox[{9, 19, 13}], 
   CuboidBox[{9, 19, 14}], CuboidBox[{9, 19, 15}], CuboidBox[{9, 19, 16}], 
   CuboidBox[{9, 19, 17}], CuboidBox[{9, 19, 18}], CuboidBox[{9, 19, 19}], 
   CuboidBox[{10, 2, 2}], CuboidBox[{10, 2, 3}], CuboidBox[{10, 2, 4}], 
   CuboidBox[{10, 2, 5}], CuboidBox[{10, 2, 6}], CuboidBox[{10, 2, 7}], 
   CuboidBox[{10, 2, 8}], CuboidBox[{10, 2, 9}], CuboidBox[{10, 2, 10}], 
   CuboidBox[{10, 2, 11}], CuboidBox[{10, 2, 12}], CuboidBox[{10, 2, 13}], 
   CuboidBox[{10, 2, 14}], CuboidBox[{10, 2, 15}], CuboidBox[{10, 2, 16}], 
   CuboidBox[{10, 2, 17}], CuboidBox[{10, 2, 18}], CuboidBox[{10, 2, 19}], 
   CuboidBox[{10, 2, 20}], CuboidBox[{10, 3, 2}], CuboidBox[{10, 3, 3}], 
   CuboidBox[{10, 3, 5}], CuboidBox[{10, 3, 6}], CuboidBox[{10, 3, 7}], 
   CuboidBox[{10, 3, 8}], CuboidBox[{10, 3, 9}], CuboidBox[{10, 3, 10}], 
   CuboidBox[{10, 3, 11}], CuboidBox[{10, 3, 12}], CuboidBox[{10, 3, 13}], 
   CuboidBox[{10, 3, 14}], CuboidBox[{10, 3, 15}], CuboidBox[{10, 3, 16}], 
   CuboidBox[{10, 3, 17}], CuboidBox[{10, 3, 19}], CuboidBox[{10, 3, 20}], 
   CuboidBox[{10, 4, 2}], CuboidBox[{10, 4, 5}], CuboidBox[{10, 4, 6}], 
   CuboidBox[{10, 4, 7}], CuboidBox[{10, 4, 8}], CuboidBox[{10, 4, 9}], 
   CuboidBox[{10, 4, 10}], CuboidBox[{10, 4, 11}], CuboidBox[{10, 4, 12}], 
   CuboidBox[{10, 4, 13}], CuboidBox[{10, 4, 14}], CuboidBox[{10, 4, 15}], 
   CuboidBox[{10, 4, 16}], CuboidBox[{10, 4, 17}], CuboidBox[{10, 4, 20}], 
   CuboidBox[{10, 5, 2}], CuboidBox[{10, 5, 3}], CuboidBox[{10, 5, 4}], 
   CuboidBox[{10, 5, 18}], CuboidBox[{10, 5, 19}], CuboidBox[{10, 5, 20}], 
   CuboidBox[{10, 6, 2}], CuboidBox[{10, 6, 3}], CuboidBox[{10, 6, 4}], 
   CuboidBox[{10, 6, 18}], CuboidBox[{10, 6, 19}], CuboidBox[{10, 6, 20}], 
   CuboidBox[{10, 7, 2}], CuboidBox[{10, 7, 3}], CuboidBox[{10, 7, 4}], 
   CuboidBox[{10, 7, 7}], CuboidBox[{10, 7, 8}], CuboidBox[{10, 7, 9}], 
   CuboidBox[{10, 7, 13}], CuboidBox[{10, 7, 14}], CuboidBox[{10, 7, 15}], 
   CuboidBox[{10, 7, 18}], CuboidBox[{10, 7, 19}], CuboidBox[{10, 7, 20}], 
   CuboidBox[{10, 8, 2}], CuboidBox[{10, 8, 3}], CuboidBox[{10, 8, 4}], 
   CuboidBox[{10, 8, 7}], CuboidBox[{10, 8, 8}], CuboidBox[{10, 8, 9}], 
   CuboidBox[{10, 8, 13}], CuboidBox[{10, 8, 14}], CuboidBox[{10, 8, 15}], 
   CuboidBox[{10, 8, 18}], CuboidBox[{10, 8, 19}], CuboidBox[{10, 8, 20}], 
   CuboidBox[{10, 9, 2}], CuboidBox[{10, 9, 3}], CuboidBox[{10, 9, 4}], 
   CuboidBox[{10, 9, 7}], CuboidBox[{10, 9, 8}], CuboidBox[{10, 9, 9}], 
   CuboidBox[{10, 9, 13}], CuboidBox[{10, 9, 14}], CuboidBox[{10, 9, 15}], 
   CuboidBox[{10, 9, 18}], CuboidBox[{10, 9, 19}], CuboidBox[{10, 9, 20}], 
   CuboidBox[{10, 10, 2}], CuboidBox[{10, 10, 3}], CuboidBox[{10, 10, 4}], 
   CuboidBox[{10, 10, 18}], CuboidBox[{10, 10, 19}], CuboidBox[{10, 10, 20}], 
   CuboidBox[{10, 11, 2}], CuboidBox[{10, 11, 3}], CuboidBox[{10, 11, 4}], 
   CuboidBox[{10, 11, 18}], CuboidBox[{10, 11, 19}], CuboidBox[{10, 11, 20}], 
   CuboidBox[{10, 12, 2}], CuboidBox[{10, 12, 3}], CuboidBox[{10, 12, 4}], 
   CuboidBox[{10, 12, 18}], CuboidBox[{10, 12, 19}], CuboidBox[{10, 12, 20}], 
   CuboidBox[{10, 13, 2}], CuboidBox[{10, 13, 3}], CuboidBox[{10, 13, 4}], 
   CuboidBox[{10, 13, 7}], CuboidBox[{10, 13, 8}], CuboidBox[{10, 13, 9}], 
   CuboidBox[{10, 13, 13}], CuboidBox[{10, 13, 14}], CuboidBox[{10, 13, 15}], 
   CuboidBox[{10, 13, 18}], CuboidBox[{10, 13, 19}], CuboidBox[{10, 13, 20}], 
   CuboidBox[{10, 14, 2}], CuboidBox[{10, 14, 3}], CuboidBox[{10, 14, 4}], 
   CuboidBox[{10, 14, 7}], CuboidBox[{10, 14, 8}], CuboidBox[{10, 14, 9}], 
   CuboidBox[{10, 14, 13}], CuboidBox[{10, 14, 14}], CuboidBox[{10, 14, 15}], 
   CuboidBox[{10, 14, 18}], CuboidBox[{10, 14, 19}], CuboidBox[{10, 14, 20}], 
   CuboidBox[{10, 15, 2}], CuboidBox[{10, 15, 3}], CuboidBox[{10, 15, 4}], 
   CuboidBox[{10, 15, 7}], CuboidBox[{10, 15, 8}], CuboidBox[{10, 15, 9}], 
   CuboidBox[{10, 15, 13}], CuboidBox[{10, 15, 14}], CuboidBox[{10, 15, 15}], 
   CuboidBox[{10, 15, 18}], CuboidBox[{10, 15, 19}], CuboidBox[{10, 15, 20}], 
   CuboidBox[{10, 16, 2}], CuboidBox[{10, 16, 3}], CuboidBox[{10, 16, 4}], 
   CuboidBox[{10, 16, 18}], CuboidBox[{10, 16, 19}], CuboidBox[{10, 16, 20}], 
   CuboidBox[{10, 17, 2}], CuboidBox[{10, 17, 3}], CuboidBox[{10, 17, 4}], 
   CuboidBox[{10, 17, 18}], CuboidBox[{10, 17, 19}], CuboidBox[{10, 17, 20}], 
   CuboidBox[{10, 18, 2}], CuboidBox[{10, 18, 5}], CuboidBox[{10, 18, 6}], 
   CuboidBox[{10, 18, 7}], CuboidBox[{10, 18, 8}], CuboidBox[{10, 18, 9}], 
   CuboidBox[{10, 18, 10}], CuboidBox[{10, 18, 11}], CuboidBox[{10, 18, 12}], 
   CuboidBox[{10, 18, 13}], CuboidBox[{10, 18, 14}], CuboidBox[{10, 18, 15}], 
   CuboidBox[{10, 18, 16}], CuboidBox[{10, 18, 17}], CuboidBox[{10, 18, 20}], 
   CuboidBox[{10, 19, 2}], CuboidBox[{10, 19, 3}], CuboidBox[{10, 19, 5}], 
   CuboidBox[{10, 19, 6}], CuboidBox[{10, 19, 7}], CuboidBox[{10, 19, 8}], 
   CuboidBox[{10, 19, 9}], CuboidBox[{10, 19, 10}], CuboidBox[{10, 19, 11}], 
   CuboidBox[{10, 19, 12}], CuboidBox[{10, 19, 13}], CuboidBox[{10, 19, 14}], 
   CuboidBox[{10, 19, 15}], CuboidBox[{10, 19, 16}], CuboidBox[{10, 19, 17}], 
   CuboidBox[{10, 19, 19}], CuboidBox[{10, 19, 20}], CuboidBox[{10, 20, 2}], 
   CuboidBox[{10, 20, 3}], CuboidBox[{10, 20, 4}], CuboidBox[{10, 20, 5}], 
   CuboidBox[{10, 20, 6}], CuboidBox[{10, 20, 7}], CuboidBox[{10, 20, 8}], 
   CuboidBox[{10, 20, 9}], CuboidBox[{10, 20, 10}], CuboidBox[{10, 20, 11}], 
   CuboidBox[{10, 20, 12}], CuboidBox[{10, 20, 13}], CuboidBox[{10, 20, 14}], 
   CuboidBox[{10, 20, 15}], CuboidBox[{10, 20, 16}], CuboidBox[{10, 20, 17}], 
   CuboidBox[{10, 20, 18}], CuboidBox[{10, 20, 19}], CuboidBox[{10, 20, 20}], 
   CuboidBox[{11, 1, 1}], CuboidBox[{11, 1, 2}], CuboidBox[{11, 1, 3}], 
   CuboidBox[{11, 1, 4}], CuboidBox[{11, 1, 5}], CuboidBox[{11, 1, 6}], 
   CuboidBox[{11, 1, 7}], CuboidBox[{11, 1, 8}], CuboidBox[{11, 1, 9}], 
   CuboidBox[{11, 1, 10}], CuboidBox[{11, 1, 11}], CuboidBox[{11, 1, 12}], 
   CuboidBox[{11, 1, 13}], CuboidBox[{11, 1, 14}], CuboidBox[{11, 1, 15}], 
   CuboidBox[{11, 1, 16}], CuboidBox[{11, 1, 17}], CuboidBox[{11, 1, 18}], 
   CuboidBox[{11, 1, 19}], CuboidBox[{11, 1, 20}], CuboidBox[{11, 1, 21}], 
   CuboidBox[{11, 2, 1}], CuboidBox[{11, 2, 21}], CuboidBox[{11, 3, 1}], 
   CuboidBox[{11, 3, 21}], CuboidBox[{11, 4, 1}], CuboidBox[{11, 4, 21}], 
   CuboidBox[{11, 5, 1}], CuboidBox[{11, 5, 6}], CuboidBox[{11, 5, 7}], 
   CuboidBox[{11, 5, 8}], CuboidBox[{11, 5, 9}], CuboidBox[{11, 5, 10}], 
   CuboidBox[{11, 5, 11}], CuboidBox[{11, 5, 12}], CuboidBox[{11, 5, 13}], 
   CuboidBox[{11, 5, 14}], CuboidBox[{11, 5, 15}], CuboidBox[{11, 5, 16}], 
   CuboidBox[{11, 5, 21}], CuboidBox[{11, 6, 1}], CuboidBox[{11, 6, 5}], 
   CuboidBox[{11, 6, 6}], CuboidBox[{11, 6, 7}], CuboidBox[{11, 6, 8}], 
   CuboidBox[{11, 6, 9}], CuboidBox[{11, 6, 10}], CuboidBox[{11, 6, 12}], 
   CuboidBox[{11, 6, 13}], CuboidBox[{11, 6, 14}], CuboidBox[{11, 6, 15}], 
   CuboidBox[{11, 6, 16}], CuboidBox[{11, 6, 17}], CuboidBox[{11, 6, 21}], 
   CuboidBox[{11, 7, 1}], CuboidBox[{11, 7, 5}], CuboidBox[{11, 7, 6}], 
   CuboidBox[{11, 7, 10}], CuboidBox[{11, 7, 12}], CuboidBox[{11, 7, 16}], 
   CuboidBox[{11, 7, 17}], CuboidBox[{11, 7, 21}], CuboidBox[{11, 8, 1}], 
   CuboidBox[{11, 8, 5}], CuboidBox[{11, 8, 6}], CuboidBox[{11, 8, 10}], 
   CuboidBox[{11, 8, 12}], CuboidBox[{11, 8, 16}], CuboidBox[{11, 8, 17}], 
   CuboidBox[{11, 8, 21}], CuboidBox[{11, 9, 1}], CuboidBox[{11, 9, 5}], 
   CuboidBox[{11, 9, 6}], CuboidBox[{11, 9, 10}], CuboidBox[{11, 9, 12}], 
   CuboidBox[{11, 9, 16}], CuboidBox[{11, 9, 17}], CuboidBox[{11, 9, 21}], 
   CuboidBox[{11, 10, 1}], CuboidBox[{11, 10, 5}], CuboidBox[{11, 10, 6}], 
   CuboidBox[{11, 10, 7}], CuboidBox[{11, 10, 8}], CuboidBox[{11, 10, 9}], 
   CuboidBox[{11, 10, 10}], CuboidBox[{11, 10, 12}], CuboidBox[{11, 10, 13}], 
   CuboidBox[{11, 10, 14}], CuboidBox[{11, 10, 15}], CuboidBox[{11, 10, 16}], 
   CuboidBox[{11, 10, 17}], CuboidBox[{11, 10, 21}], CuboidBox[{11, 11, 1}], 
   CuboidBox[{11, 11, 5}], CuboidBox[{11, 11, 17}], CuboidBox[{11, 11, 21}], 
   CuboidBox[{11, 12, 1}], CuboidBox[{11, 12, 5}], CuboidBox[{11, 12, 6}], 
   CuboidBox[{11, 12, 7}], CuboidBox[{11, 12, 8}], CuboidBox[{11, 12, 9}], 
   CuboidBox[{11, 12, 10}], CuboidBox[{11, 12, 12}], CuboidBox[{11, 12, 13}], 
   CuboidBox[{11, 12, 14}], CuboidBox[{11, 12, 15}], CuboidBox[{11, 12, 16}], 
   CuboidBox[{11, 12, 17}], CuboidBox[{11, 12, 21}], CuboidBox[{11, 13, 1}], 
   CuboidBox[{11, 13, 5}], CuboidBox[{11, 13, 6}], CuboidBox[{11, 13, 10}], 
   CuboidBox[{11, 13, 12}], CuboidBox[{11, 13, 16}], CuboidBox[{11, 13, 17}], 
   CuboidBox[{11, 13, 21}], CuboidBox[{11, 14, 1}], CuboidBox[{11, 14, 5}], 
   CuboidBox[{11, 14, 6}], CuboidBox[{11, 14, 10}], CuboidBox[{11, 14, 12}], 
   CuboidBox[{11, 14, 16}], CuboidBox[{11, 14, 17}], CuboidBox[{11, 14, 21}], 
   CuboidBox[{11, 15, 1}], CuboidBox[{11, 15, 5}], CuboidBox[{11, 15, 6}], 
   CuboidBox[{11, 15, 10}], CuboidBox[{11, 15, 12}], CuboidBox[{11, 15, 16}], 
   CuboidBox[{11, 15, 17}], CuboidBox[{11, 15, 21}], CuboidBox[{11, 16, 1}], 
   CuboidBox[{11, 16, 5}], CuboidBox[{11, 16, 6}], CuboidBox[{11, 16, 7}], 
   CuboidBox[{11, 16, 8}], CuboidBox[{11, 16, 9}], CuboidBox[{11, 16, 10}], 
   CuboidBox[{11, 16, 12}], CuboidBox[{11, 16, 13}], CuboidBox[{11, 16, 14}], 
   CuboidBox[{11, 16, 15}], CuboidBox[{11, 16, 16}], CuboidBox[{11, 16, 17}], 
   CuboidBox[{11, 16, 21}], CuboidBox[{11, 17, 1}], CuboidBox[{11, 17, 6}], 
   CuboidBox[{11, 17, 7}], CuboidBox[{11, 17, 8}], CuboidBox[{11, 17, 9}], 
   CuboidBox[{11, 17, 10}], CuboidBox[{11, 17, 11}], CuboidBox[{11, 17, 12}], 
   CuboidBox[{11, 17, 13}], CuboidBox[{11, 17, 14}], CuboidBox[{11, 17, 15}], 
   CuboidBox[{11, 17, 16}], CuboidBox[{11, 17, 21}], CuboidBox[{11, 18, 1}], 
   CuboidBox[{11, 18, 21}], CuboidBox[{11, 19, 1}], CuboidBox[{11, 19, 21}], 
   CuboidBox[{11, 20, 1}], CuboidBox[{11, 20, 21}], CuboidBox[{11, 21, 1}], 
   CuboidBox[{11, 21, 2}], CuboidBox[{11, 21, 3}], CuboidBox[{11, 21, 4}], 
   CuboidBox[{11, 21, 5}], CuboidBox[{11, 21, 6}], CuboidBox[{11, 21, 7}], 
   CuboidBox[{11, 21, 8}], CuboidBox[{11, 21, 9}], CuboidBox[{11, 21, 10}], 
   CuboidBox[{11, 21, 11}], CuboidBox[{11, 21, 12}], CuboidBox[{11, 21, 13}], 
   CuboidBox[{11, 21, 14}], CuboidBox[{11, 21, 15}], CuboidBox[{11, 21, 16}], 
   CuboidBox[{11, 21, 17}], CuboidBox[{11, 21, 18}], CuboidBox[{11, 21, 19}], 
   CuboidBox[{11, 21, 20}], CuboidBox[{11, 21, 21}]},
  ImageMargins->0.,
  ViewAngle->Automatic,
  ViewCenter->Automatic,
  ViewPoint->{1.3, -2.4, 2.},
  ViewVertical->{-1, 0, 0}]], "Output",
 ImageSize->{203, 145},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 Graphics3DBoxOptions->{ImageSize->{199., Automatic}},
 CellLabel->"Out[2]=",
 CellID->26176,
 ImageCache->GraphicsData["CompressedBitmap", "\<\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\
\>"]]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->19910],

Cell[TextData[{
 "A spacetime slice for 50 steps across all ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " values at ",
 Cell[BoxData[
  FormBox["y", TraditionalForm]], "InlineMath"],
 " offset 0:"
}], "ExampleText",
 CellID->28743],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"14", ",", " ", 
      RowBox[{"{", 
       RowBox[{"2", ",", " ", "1"}], "}"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"1", ",", " ", "1"}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"{", "1", "}"}], "}"}], ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"50", ",", "All", ",", 
      RowBox[{"{", 
       RowBox[{"{", "0", "}"}], "}"}]}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->30524],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy1V0FywzAIhP6kb+hP+oTM9Nz/35qJDLuLaGLHcmYS2xJaWBbw5PP2+337
MLOv+/fn/jV3t/Hx8XR/9nGxcT82bNhtOybrYZxnN7TA983eDei+ucSxgEkD
hEAhZRC5MSILI5MIFHkcIhfhplIOTBAu3DQCdwqqJISWkxGdkIx44ZYZ9CJB
zTJd5DRjsh6ICrrgXGFKGcMTeTLlAD0ow5IPM3LpSAOnEMtwwwiyy9Gq16BN
58z4ppoogvJAoJptZIaIoaYokRJrowfv137RzDSfqepqf3QHar8Mz605h8bW
CdIf6vrliTmH5pqBTg8+UPslbiK6MW9MVp04/K8Hc6n9wl7EOxfe0/7oyU/V
ko7Fexbeq/7ouDTVgkE5PxTMZ3oIatMvjvmc9VRi26EH23f90s2uifFLPdje
pVo84pZBEIXG1ei+h0dw0WrhWWL6UKpxHw+EieawWNBILPcO6cFcKEKjXw2F
TXbrIQDUalpL3eoRPXBGysfa49XkEI+IyfLjVEvRJDabHPQhukT2ucENmryj
B3OJVCNuBABN3tGDuUhH1PHM22/xCCjSRIctrb6nB0Fhostkt60dz+jBXHSi
T98TejAXiM3C5/1JHgMjBcB/EbxVTuqRXkIXhotEndYjvXgglpf/Ej0YcILb
FtbwCECrcMv0IMD2fbKOxwaYc9gxgxfyGA7qK2WlHsCchV/tY87/eh4P1BfP
i7zI6/caH/MEvshL/Xd0jZdyvcjLpXqkF7+ax8PL5TweXo4f+QP4FhKY
   "], {{0, 0}, {101, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None},
  ImageMargins->0.]], "Output",
 ImageSize->{229, 118},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{225, Automatic}},
 CellLabel->"Out[1]=",
 CellID->27108]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->17027],

Cell[TextData[{
 "Mean colors of all cells with particular ",
 Cell[BoxData[
  FormBox["x", TraditionalForm]], "InlineMath"],
 " positions:"
}], "ExampleText",
 CellID->4430],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Mean", "/@", 
   RowBox[{"CellularAutomaton", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"14", ",", " ", 
       RowBox[{"{", 
        RowBox[{"2", ",", " ", "1"}], "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{"1", ",", " ", "1"}], "}"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"{", "1", "}"}], "}"}], ",", "0"}], "}"}], ",", "30"}], 
    "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7077],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJztmj9oVEEQxg+xsFBQ0CKFSAoFLSwsLBR9hYWFhaAWFhZBFLQQSWFhYWOR
wsIijUWKCBY2FiksFCwixOKKiIoo8W+8xESNJkIKhVh4d+9+e9z3bphNSLL+
uUD42N3Z3Zlv3szsvnfdpy8eO7umVCptq/4fKDX/Tu1ZO3NufjZT3P/qwdHe
8nB28vDAj5277zfaTwKWN965NbrwvDFezqavXDvS1zOa/dy5u+vqpckse7T9
0ND1qaw22nX1Y7bh4b7qzA/Zptq0zS9l/lj2a/ON9VWRgNXJe+eOv8ne3b53
Zt2LD1lt1buX32eDlf7uk4dfN/Z7m52fP1jdaTzghXxCmM96jOv6yOs6rM9+
7I8+uj6IPdiHvdgPH/ADX/AHn8xX3lv9MVzwG1hq+fvekL/ZwKGwzpa62l8D
0t/Xs7VKzadsV93wz8EO2oxP1OgZmA79IDzWWBusVAr26nNCG/4eL9QI/RL6
4Q/5E3VDJwpt9LLGabMe67Mf+1v6aT/2Ya/yAD+xfHp+wX/4E/8qImc9H61x
/qRhx7OCnvCidiGHftiZP7eTAZHP42wqxI/aB+pzBc8g44oqp7xb+6EP+qGv
2sF6qqfyonwhB7/wHeeXIdO/ipoXcr1Hgr600a+33jFj8on+yCkiB2/wvqOe
GCfMeYrKk/JpYayeiuiHvujPerF6KiIHvxb/Wldj/ds+vu38z3PWWlfGAlLf
tE7RD2r9Io/Wy9+pSmE+dZA2z31ttSpToY6iB/WVebQZR575rGfthz7oh756
/rDsZD797XkbdeMY/yzVv4o8P3re0nxGv8YHPLbPS5VCvuWcos+5ymue1/yr
+UGRcd1f8zDj7K966bnKkvf48HhtPZ+NLJt/2+fzslsnNI68+g3f6ketg7oO
vCKn9RH+PdR5rKd+1/qCnPpV67ZVfzWvxPFaXjH/KmpeIc/x/NFebP3O7WjW
zdoq/d1FeVDXV36tedb+zNf6aM1DP0v/2Ppr8ad1cbX8q+jVD55T6lIuPx6Q
5586xb1E7znEDfIg/azHPOt+yrh1v2ec9bx99V7Weg9+G+TVbvjw4hh+U/lX
kedPz0taZ+jXOOLcat1PiDPiR/Op1lmdD7ae64rjWte1PrA/49Z9UO3BXo8X
+IPP1H61ED3j7nnNeLHqd25/8z6r8ar+9ea35326UD/V3xrv3nzGNd94vMBf
aj/GouYlzhs8v7Rj6zdxlN9vvph1HN7V/yD+0n7kmW/VXfb3zg1afy37ta6l
9ttSMa7+jId7huZBeNc66uVZzd9ar611vDqg66Cf6o09XhzDT2o/LRfqdwd4
0DpFv+ZZ4kfjyIpL7df3mda4NV/zisY7455d+l0ktV9WCrEzrk5Nmvdm4og8
Clp12nrfadVlXZf9rHs0+np2YX9qP6wWal7T7xS0vfpt5V31o4fWdyTW9+qv
pb/WpdS8p0KvfsEz9xTOQ6D6S/lfLFrzdV/08eIY+1Lz/Kegvo+AR61z9Fv3
sNjvTB5a9yRPL32PkprXPxXhKe49b/Oeo/dl695ltZHX+7De4zy90D81j38L
al7kvEP80Lbyrb7/1u/Kmges9+W0rf21rqTm7W9Fr/4RZ953fcbvV7Pp06lv
0fJeHKNfap7+FbTeV2mdtO5P1u9/YuWt922peflXEZ69OkncenKLlWf/1Dz8
L0h86XmM+NPzE23i1Rq35rNfarv/V/TqZ/7+6VuIV0XGvfqf2s4O5kj85e+N
Zxq/ny/WYwuRZz7rpbarg+0RP8XWZUXmp7ajg3FIfJKX89/rzob4pc048qn1
7uDS0Ku/jKfWs4PLg8SvYmq9OrgymH8fnEuux2rjbwNFi2g=
   "], {{0, 0}, {61, 31}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->22123]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->7912],

Cell["5-neighbor totalistic rule:", "ExampleText",
 CellID->2171],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    StyleBox[
     RowBox[{"{", 
      RowBox[{"26", ",", 
       RowBox[{"{", 
        RowBox[{"2", ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"0", ",", "1", ",", "0"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"1", ",", "1", ",", "1"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"0", ",", "1", ",", "0"}], "}"}]}], "}"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", "1"}], "}"}]}], "}"}],
     GridElementStyleOptions -> {ColumnAlignments -> Left}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"{", "1", "}"}], "}"}], ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{"{", "100", "}"}], "}"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->14931],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJztmQt248oNRBvZSbaUJbwNZJ3ZVc6ziapboDT+SSLppOfYFsn+oIAqANT8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   "], {{0, 0}, {201, 201}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{178, 174},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{174., Automatic}},
 CellLabel->"Out[1]=",
 CellID->29190]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->22634],

Cell["5-neighbor outer totalistic rule:", "ExampleText",
 CellID->23862],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    StyleBox[
     RowBox[{"{", 
      RowBox[{"110", ",", 
       RowBox[{"{", 
        RowBox[{"2", ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"0", ",", "2", ",", "0"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"2", ",", "1", ",", "2"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"0", ",", "2", ",", "0"}], "}"}]}], "}"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", "1"}], "}"}]}], "}"}],
     GridElementStyleOptions -> {ColumnAlignments -> Left}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"{", "1", "}"}], "}"}], ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{"{", "100", "}"}], "}"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17818],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJztWYlx7DYMBdJJWkoJaSB1pquM/xLAw0GRe3wb1AQaj1ciReDhJvXn3//8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   "], {{0, 0}, {201, 201}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None},
  ImageMargins->0.]], "Output",
 ImageSize->{179, 175},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{175., Automatic}},
 CellLabel->"Out[1]=",
 CellID->3078]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->20010],

Cell["\<\
Use an initial condition with two black cells, specified in a sparse array:\
\>", "ExampleText",
 CellID->19184],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    StyleBox[
     RowBox[{"{", 
      RowBox[{"110", ",", 
       RowBox[{"{", 
        RowBox[{"2", ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"{", 
            RowBox[{"0", ",", "2", ",", "0"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"2", ",", "1", ",", "2"}], "}"}], ",", 
           RowBox[{"{", 
            RowBox[{"0", ",", "2", ",", "0"}], "}"}]}], "}"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", "1"}], "}"}]}], "}"}],
     GridElementStyleOptions -> {ColumnAlignments -> Left}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"SparseArray", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"1", ",", "1"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"1", ",", "100"}], "}"}]}], "}"}], "->", "1"}], "]"}], ",",
       "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{"{", "100", "}"}], "}"}], "}"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->16569],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJztWV2W9agNFNlJ3rOaLGE2kHVmV8l8RlJJAowvWE33aXlm+trYlKr0A/b8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   "], {{0, 0}, {300, 201}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None},
  ImageMargins->0.]], "Output",
 ImageSize->{258, 174},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{254., Automatic}},
 CellLabel->"Out[1]=",
 CellID->14845]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->8581],

Cell["3D nearest-neighbor totalistic cellular automaton:", "ExampleText",
 CellID->1892],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{"Cuboid", "/@", 
   RowBox[{"Position", "[", 
    RowBox[{
     RowBox[{"CellularAutomaton", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"14", ",", " ", 
         RowBox[{"{", 
          RowBox[{"2", ",", " ", "1"}], "}"}], ",", " ", 
         RowBox[{"{", 
          RowBox[{"1", ",", " ", "1", ",", "1"}], "}"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", 
           RowBox[{"{", "1", "}"}], "}"}], "}"}], ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"{", "10", "}"}], "}"}], "}"}]}], "]"}], ",", "1"}], "]"}]}],
   "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->32016],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{195, 212},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 Graphics3DBoxOptions->{ImageSize->{195., Automatic}},
 CellLabel->"Out[1]=",
 CellID->131610993]
}, Open  ]],

Cell[BoxData[""], "Input",
 CellLabel->"In[2]:=",
 CellID->15901]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Higher-Order Rules",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->14221],

Cell["Rule 30 written out explicitly as a \"first-order rule\":", \
"ExampleText",
 CellID->30960],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"30", ",", "2", ",", "1", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"1", ",", "0", ",", "1", ",", "1"}], "}"}], "}"}], ",", "4"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->15583],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "1", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "1", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "1", ",", "0"}], "}"}]}], "}"}]], "Output",
 ImageSize->{441, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->25913]
}, Open  ]],

Cell["\<\
A second-order analog of rule 30, involving two steps of initial conditions:\
\>", "ExampleText",
 CellID->10856],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"30", ",", "2", ",", "1", ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "0", ",", "1", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "1", ",", "0", ",", "1"}], "}"}]}], "}"}], ",", "4"}],
   "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->4122],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "0"}], "}"}]}], "}"}]], "Output",
 ImageSize->{441, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->12153]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->20332],

Cell["Include both initial-condition steps in the output:", "ExampleText",
 CellID->9371],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"30", ",", "2", ",", "1", ",", "2"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "0", ",", "1", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "1", ",", "0", ",", "1"}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "1"}], ",", "4"}], "}"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->24008],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "1", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0", ",", "0"}], "}"}]}], "}"}]], "Output",
 ImageSize->{528, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->8008]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->3173],

Cell["\<\
Second-order rule 1008, starting with a single 1 in both initial condition \
steps:\
\>", "ExampleText",
 CellID->1575],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1008", ",", "2", ",", "1", ",", "2"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", "1", "}"}], ",", 
        RowBox[{"{", "1", "}"}]}], "}"}], ",", "0"}], "}"}], ",", "40"}], 
   "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->5058],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy9k9sNgCAMRS9u4qcbOIcjsID7//lKDNS+ozaBNJXbwxUY67rUAcC0j3kf
BeWIa24Dj7mr48xYGRoZuDoje9ZpztIUSIbWuQ3JpN3ZASbzAfkVprIrhX6p
2Ftn0sb+S0M7+W8p/Qy5bpybx6fdWPNGVhRBKcrs96Yz9ZwH6nAvzfCmyN54
b1FgrLfL2700dm460wuMeWsJAW9E9uW5ScAMJyP7y1sL/IuWl+UjDfQIN/65
Clw=
   "], {{0, 0}, {56, 41}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{154, 112},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{150, Automatic}},
 CellLabel->"Out[1]=",
 CellID->32699]
}, Open  ]],

Cell["Include both steps in the initial conditions:", "ExampleText",
 CellID->11559],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1008", ",", "2", ",", "1", ",", "2"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", "1", "}"}], ",", 
        RowBox[{"{", "1", "}"}]}], "}"}], ",", "0"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", "1"}], ",", "10"}], "}"}], "}"}]}], "]"}], "]"}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->460],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzzTSzJSM1NLMlMTlRwL0osyMhMLlZwyy8CCjEzMjAwpABxjgQDkP0fyAJj
kBAQMIPYYMCQACQYiRQNBRKcQC7TCJAFMTmAWBvIBtFCUJp7VCWNVYIUC0Dj
RRtKC4+qwquKAy1UBaGqZUZVDwLVxlBVqqOySCkaFma6o6rwqlKChqbZqCxQ
VhIaVnajsiS23gaD6OBMUwMlq/UfX74flR3esogcAu2VMQIAwdtwZQ==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{100, 108},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{100, Automatic}},
 CellLabel->"Out[2]=",
 CellID->94197990]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->17881],

Cell["Second-order totalistic rule 10 with 2 colors and range 1:", \
"ExampleText",
 CellID->31406],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"10", ",", 
      RowBox[{"{", 
       RowBox[{"2", ",", "1"}], "}"}], ",", "1", ",", "2"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", "1", "}"}], ",", 
        RowBox[{"{", "1", "}"}]}], "}"}], ",", "0"}], "}"}], ",", "50"}], 
   "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->12755],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy9WOt5wyAMFN2kM3STjpAO0P3/NY4RSMdBeJbPxFiATgdc5OTz8fv9+BCR
r2f9eVYJ4arXx+u62uF1F2fPowK149w06poQzha5q0hsiYaX22qP4Svvwo5z
k11xst1hWz/pidtxrvWtPDQ4/dAQzDrn3tKOcw0Hi8XiIetK7TiX+BW7UsoV
9+WOm9pxrmAkdn1zC9dZYDRGFJwXwpXqw3hJ607tRA9YavrAfande/zV9IH9
FTvVA8ct9YH93M71wHmUZzztTzxOsZGulh44DxsT76/phfnhPNhpTCug58f1
mbFNDsGMRn0YLRql9emB8yj1kfLH7beZH96VjhiW+uOYVv4onnv0wOOo5Y/y
uUcPnEepj+RYSN8AB4yFcWlxG8Co5Q93ak3fEIccD8sfuY7qgfMo9aErP6MH
zuPc+DiH6UNMe3offFyoD7sH8/vgeYAGwpoeCIZrY2XjJjBQH2LaS/vg48P1
D866hkD0Iet6IBhDz5MY7vfEDj0QDOt3ix4IRtZH+pbaxyFiDNknMXQ/ZPc+
GIxgs1K2bcVw71H7EXy+tvfNGO496kTZ9d36BkPi3yaHEHiuOIFxFsHn7mMY
chrhX8o4iT8wLBLu
   "], {{0, 0}, {99, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 97},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->22438]
}, Open  ]],

Cell[TextData[{
 "A ",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["12", "th"], TraditionalForm]]],
 " order version of the same rule:"
}], "ExampleText",
 CellID->24539],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"10", ",", 
      RowBox[{"{", 
       RowBox[{"2", ",", "1"}], "}"}], ",", "1", ",", "12"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{"{", "1", "}"}], ",", 
        RowBox[{"{", "12", "}"}]}], "]"}], ",", "0"}], "}"}], ",", "50"}], 
   "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->2012],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJztl1sOwiAURAd34hrcie6gG3D/fzYkxir3MXPhz96EFFpyOKHpkF635327
ALjt7bE3tEOhwey33gc1sw99zC80n4dh8Wh11jGeYEMzxw+Z9kwdVU/GUfSk
HBVP1lHwpB1ZjOKYeuJwJR1zT8iOqae4j4wnHRusZxQbIXFBbJizfZuKI2eg
OQ6rZrFBESdiI2S6ViXH2KTmOKzuxYZELMSG7tn6iT/haHnO7KNpYRz4laJj
Q2LqT3ImERsyk7/LM5PYKDHzOzpzzbv+ZvqjOtOJjSnm2Jtnov8prCO+7VYS
jdhYwcRq4lln/U3xH88Lnv4RkA==
   "], {{0, 0}, {83, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 114},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1255]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->18897],

Cell["\<\
Rule 150R\[LongDash]the second-order reversible mod-2 rule:\
\>", "ExampleText",
 CellID->13672],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"Mod", "[", 
        RowBox[{
         RowBox[{"Total", "[", 
          RowBox[{"Flatten", "[", "#", "]"}], "]"}], ",", "2"}], "]"}], "&"}],
       ",", 
      RowBox[{"{", "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"-", "1"}], ",", "0"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", 
          RowBox[{"-", "1"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", "1"}], "}"}]}], "}"}], ",", "2"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", "1", "}"}], ",", 
        RowBox[{"{", "1", "}"}]}], "}"}], ",", "0"}], "}"}], ",", "127"}], 
   "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->14417],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJzlWA165CYMhd6kV9oj7AV6hd68zWYQFs+IJ4E9HjsvXwaD/ieyhPL3739+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   "], {{0, 0}, {255, 128}}, {0, 1}],
  Frame->Automatic,
  FrameLabel->{None, None},
  FrameTicks->{None, None},
  PlotRangeClipping->True]], "Output",
 ImageSize->{184, 94},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->162793075]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->651026136],

Cell["\<\
A spacetime slice of a second-order totalistic rule with 2 colors and range \
1:\
\>", "ExampleText",
 CellID->302524911],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"First", "/@", 
   RowBox[{"CellularAutomaton", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"97852", ",", 
       RowBox[{"{", 
        RowBox[{"2", ",", "1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", "1"}], "}"}], ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", "1", "}"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"{", "1", "}"}], "}"}]}], "}"}], ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"50", ",", 
       RowBox[{"{", "0", "}"}], ",", "All"}], "}"}]}], "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->298338114],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy1l11ywyAMhNnepGfoTXqEzPS593+rayNYiT9BwBknxgEtux+2k8/X7/fr
I4Twde0/1x4Q7nfIJzWew//GtaPcwvOV7i5FqOb9Mp2jThrp3OKQJAApfKvF
agjltNSIalnornQ2lY2uYx40ul7HxlXMQNdIPkQltLk4txYPkEqHizuuOg9W
4drrcdV4GC9VLu64+LDwMeLiiivLVHiMufji6vMYcnHH1eMx5OKJi8c0fbS5
cMVKXLof2jyMF8PFkRQb6/poc2nzQZV+X6PNpdV9jofxAsVlkJSfh/GiuHTi
muRhvGguKJ+NAZSkm4fxMnk1enkYL8hckiNVFXkq8PMwXkxt+cpqxtaUD/JS
/U2hG1jgYbxIeT314twsD+MFZSyFJuZ5GC+FiaK1xMN4sfexREDOrfEwXmrB
LNyvBl5yaZ7/8vXR8GIfgRr/Wz7Ii5AAz/99HsZLJpxMYQMP40WOclzYwYNV
IhcVF7bwYJWYT14D2MWDVcrnyy4erBKZh4RjGw9WAbEJW3mwiqxb+tyqQVyQ
Ftd2DX297OfBKmmBHfEhKs/8T/kQlXRbOaSR19g5H6KCoz5E5ayPpHJYg3+m
HlUJ5zXK/ymO7Q+gqg/L
   "], {{0, 0}, {101, 51}}, {0, 1}],
  Frame->Automatic,
  FrameLabel->{None, None},
  FrameTicks->{{None, None}, {None, None}},
  PlotRangeClipping->True]], "Output",
 ImageSize->{184, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->89265084]
}, Open  ]],

Cell["Step 50 of the same rule:", "ExampleText",
 CellID->93111161],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"First", "@", 
   RowBox[{"CellularAutomaton", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"97852", ",", 
       RowBox[{"{", 
        RowBox[{"2", ",", "1"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"1", ",", "1"}], "}"}], ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", "1", "}"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"{", "1", "}"}], "}"}]}], "}"}], ",", "0"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", "50", "}"}], ",", "All", ",", "All"}], "}"}]}], "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->258039322],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy9l1FupjAMhOO9yV6pR6i0z73/20p/iT0eO0BCDBWiQPDkizNO/r/fP1/f
f1pr/45TZo+Vb+Y1pFylfUhqVT4axSqHRqmKahSqHPlolXnpHFLIYjmvyz4A
lLH03vvLXhXKRwmLcqjIfhbmqGAJHAUsyCFHdNnM4jjcuY/F5QMSvzMvxGE+
2cji5xVkQ8M/V/EcnQYoNrCk/hDxU/khC/nDqq+7e8aScWTHE5bgj0aO7wIP
WLzPRRNu/bdnqyzIoeYAFL0RWWaJ+ZBEBN/Oq5A/elQSEcvUAsvQH7hoOc15
loTD9tjxZomF6hUG+bw9TujEfB1L6pUVE2uFBUZmWQIHhPEibmMxxRJ9juGw
oROZ8j773DTdzx/4XxN32/vR58EjGMqQWp8C1yyDuosjdPn4iuXUH8d7DSyc
oHt+yTm4jb9mMGcstNINGuXavpNjlrP1PKDQUDHNqJu8v8K8Xk5I13TMEn0u
vn8jCccsLgKrhHplGT7VCKVfxnXMcWACRWvsiYZYU/ieWbzPMYXY3REGDVwP
QSw8r/L5laiY/bK55Vg0zRiZZYL9/OOWfmuxmz/8MqLdHIuQCn/vH+sI++5M
itjgYsyB5GaSV3Lyzux6xyfvOP6d2vVOFX5nPXlnZSSWvIn30iDAmGPEwtEa
XVn7nCOwiASRHfuuEUvzoS4en3MwizqNvdaag4RWNziYpWpXH/wCQ8MJgvHX
bt3hYBaoqFwlmxc58/k1CwYnERa7zZGyWAgR0ROm+ixHwoJTiUatv5rliCxO
JL2b5xiwUFS7XePIWcAVXcS9ndfIvJ+IdJAlDmYR/cMc2LM1Dmax4gQo2oVV
DmY5O9Y5AkvT4aK7Jxwjlt8XuzgCi4hNgqV6dY+l1ymY1o85IotZ5vm8GrJA
7drHkbDAuYsDVCwblpltGnFvuccfAxWqXXs1EhbZzQEqtnvZzgEq4i97NZBF
ijhAxUQKNEJeSjR0m61Lcs0RNi+1KoUaUppzUinWOCpwsYb/lXL3+A9awh2g

   "], {{0, 0}, {101, 101}}, {0, 1}],
  Frame->Automatic,
  FrameLabel->{None, None},
  FrameTicks->{{None, None}, {None, None}},
  PlotRangeClipping->True]], "Output",
 ImageSize->{184, 180},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->512306136]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Functional Versions",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSubsection"],
  $Line = 0; Null]], "ExampleSubsection",
 CellID->17148],

Cell["Give the result of one step of rule 30 evolution:", "ExampleText",
 CellID->12105],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"30", ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "0"}], "}"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19556],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", "1", ",", "1", ",", "1", ",", "0"}], "}"}]], "Output",
 ImageSize->{98, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->27000]
}, Open  ]],

Cell["An alternative form:", "ExampleText",
 CellID->9665],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"CellularAutomaton", "[", "30", "]"}], "[", 
  RowBox[{"{", 
   RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "0"}], "}"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->9293],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0", ",", "1", ",", "1", ",", "1", ",", "0"}], "}"}]], "Output",
 ImageSize->{98, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->1185]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->11562],

Cell["Iterate a single step 3 times:", "ExampleText",
 CellID->32429],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NestList", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"CellularAutomaton", "[", 
     RowBox[{"30", ",", "#"}], "]"}], "&"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "0"}], "}"}], ",", "3"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->29116],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "1", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "0", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "1", ",", "1"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{427, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->18170]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NestList", "[", 
  RowBox[{
   RowBox[{"CellularAutomaton", "[", "30", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "0"}], "}"}], ",", "3"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->19572],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "1", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "0", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "1", ",", "1"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{427, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->15130]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->1498],

Cell["Give the result of 3 steps of evolution:", "ExampleText",
 CellID->26033],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"30", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"{", 
     RowBox[{"{", "3", "}"}], "}"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->14680],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"1", ",", "1", ",", "0", ",", "1", ",", "1", ",", "1", ",", "1"}], 
  "}"}]], "Output",
 ImageSize->{135, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2699]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->364],

Cell["\<\
Give the result as a center region surrounded by repeating background:\
\>", "ExampleText",
 CellID->12425],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"30", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->11879],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "1", ",", "1"}], "}"}], ",", 
   RowBox[{"{", "0", "}"}]}], "}"}]], "Output",
 ImageSize->{108, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->32738]
}, Open  ]],

Cell["Rule 45 gives a background of 1s after one step:", "ExampleText",
 CellID->9246],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"45", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->16272],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", "1", "}"}]}], "}"}]], "Output",
 ImageSize->{108, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->7742]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->21204],

Cell["Iterate a single step of rule 45:", "ExampleText",
 CellID->22562],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NestList", "[", 
  RowBox[{
   RowBox[{"CellularAutomaton", "[", "45", "]"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "3"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->24636],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0", ",", "1", ",", "0"}], "}"}], ",", 
     RowBox[{"{", "1", "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0", ",", "1", ",", "1", ",", "1", ",", "1"}], "}"}], ",", 
     RowBox[{"{", "0", "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
      "1", ",", "0", ",", "1", ",", "0", ",", "0", ",", "0", ",", "0"}], 
      "}"}], ",", 
     RowBox[{"{", "1", "}"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{525, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->15441]
}, Open  ]]
}, Closed]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(22)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->22713],

Cell["Make a color picture of rule 30:", "ExampleText",
 CellID->743],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{
   RowBox[{"CellularAutomaton", "[", 
    RowBox[{"30", ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "40"}], "]"}], ",", 
   RowBox[{"ColorRules", "->", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"1", "->", "Red"}], ",", 
      RowBox[{"0", "->", "Yellow"}]}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->29416],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWm1LG0EQvtx7LklFglgRUUREREQQEREREayIFRG/lyAF/VBa1B/WD/1/
2127T/Nk3LumeHm5w0Cc7O3uzDM7s8/OxnzuPd9//dZ7frjrrXx67P24f7h7
Wjn7/qgfBQ3P837p95ePnv6s9KeXt3mkX4H5/PLyfuo/jbKehn+f6leoLYa6
39cy0TK27aaWsZW+7Tftju2P7Xjf9odWRlZCT4f0RzSupWVmn4cVxMZ6m1ZP
SHgia8fv6/EC206FXeBJaLxPfofkV0T6pF9YB5/8+4Ozvj6EpNtgS8l2QhjZ
10D4FIrxjBHzWuSjaQdirWKxNk3S37dTXx9ikUs++ZRQTBHr1PbL3OF9zNh4
P8c5a5Ban2Krpw5Y2TZihNim1lZKuhEz5FBL7F2ZW+y7L7DG9DwhvXXAmhHn
gMuYg7CHArEHOGfgS0vszRblUKIGcwM5xTlj9GU1wdqmOYiBK0aSe9gXng9O
CsW4hManlDusDznE3NquKfYZwoRYA4urHmrS+JjGM4bY4WtIOcT6Onb8zDvG
kWOcpbzIq41xvnA9hf0t2+BZrhdYzpKsEqYurfmw9S32N/YlS1dsIbtadiuC
ZU7Eqklz8+pNYHHlrrxLcR3JsY2Eb3MVxzgvYjpMHcn1jTxX+A6KeklyimnP
VwzTgiquC2OBVXIuxx5Ym2owB/hcXagIlkXVv4cV1XWoB1w2mGPlWcLcIusV
M36xotiW1HB1mosj87gXHIAznbEiV5YqhmlZDd4b/lXLuLgWOWDapu7HfcFw
yfKU2l4VOl21iOv+Y2xlIgawbeTqu021RjpdeYtzBm2+A+H84POV49zWcm1K
ba8X5Kn8rs11D2BsrrujxLZeESwbIiZ59Yas6xNh0ydMBvPGlNncVMV1A9fl
vLaxsL05YRtblPM415ln8+pkYMGab03YxrZ6fd6C2/LqV9YNTFjr7SmxtaOK
z0183yB1pjYndiake1dwEs5pWf9BV9u2d2ugc4/inqrXNS3frTLb/0HLvTHr
3Be+y3ODeZ/v2JD7E9J9QL67+J5rbL6THoxZ5yHpYD6L1OC+8qnf+H44Zp1H
wnfJz/x/F9N/NCZdxw7eAc+Cr7j/eEy6Tgp4kznkZMQ6TnPmZDb2p1M498zB
Y6bWPit5znkO/6B9PqK5Fw4eMfvzouQ5lw4eMNguS55zJbBdlTT2Wuy365LG
3thcvHnjmFu7Rrf/P+bNv8iyv/dq/AYEL6qv\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 95},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->21067856]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->30082],

Cell["Make pictures of the first 32 elementary cellular automata:", \
"ExampleText",
 CellID->23142],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"ArrayPlot", "[", 
    RowBox[{
     RowBox[{"CellularAutomaton", "[", 
      RowBox[{"n", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"20", ",", "All"}], "}"}]}], "]"}], ",", 
     RowBox[{"ImageSize", "->", "50"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "0", ",", "31"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->18466],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{493, 119},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->213879773]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->5423],

Cell["\<\
Show the 2-color range-2 totalistic code 20 cellular automaton from a \
sequence of initial conditions:\
\>", "ExampleText",
 CellID->17873],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"ArrayPlot", "[", 
    RowBox[{
     RowBox[{"CellularAutomaton", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"20", ",", 
         RowBox[{"{", 
          RowBox[{"2", ",", "1"}], "}"}], ",", "2"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"IntegerDigits", "[", 
          RowBox[{"i", ",", "2"}], "]"}], ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"30", ",", 
         RowBox[{"{", 
          RowBox[{
           RowBox[{"-", "15"}], ",", "15"}], "}"}]}], "}"}]}], "]"}], ",", 
     RowBox[{"ImageSize", "->", "40"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"i", ",", "150", ",", "180"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->12238],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{566, 130},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->69066640]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->9869],

Cell["\<\
Show results for 3-color range-3 totalistic code 1599 for different initial \
conditions:\
\>", "ExampleText",
 CellID->14479],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"ArrayPlot", "[", 
    RowBox[{
     RowBox[{"CellularAutomaton", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"1599", ",", 
         RowBox[{"{", 
          RowBox[{"3", ",", "1"}], "}"}]}], "}"}], ",", " ", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"Table", "[", 
          RowBox[{"1", ",", 
           RowBox[{"{", "n", "}"}]}], "]"}], ",", " ", "0"}], "}"}], ",", " ",
        "80"}], "]"}], ",", 
     RowBox[{"ImageSize", "->", 
      RowBox[{"{", 
       RowBox[{"Automatic", ",", "100"}], "}"}]}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->20549],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{487, 102},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->71939180]
}, Open  ]],

Cell["Pad with three 0s at the side of each pattern:", "ExampleText",
 CellID->19700],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"ArrayPlot", "[", 
    RowBox[{
     RowBox[{
      RowBox[{
       RowBox[{"Flatten", "[", 
        RowBox[{"{", 
         RowBox[{
         "0", ",", "0", ",", "0", ",", "#", ",", "0", ",", "0", ",", "0"}], 
         "}"}], "]"}], "&"}], "/@", 
      RowBox[{"CellularAutomaton", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1599", ",", 
          RowBox[{"{", 
           RowBox[{"3", ",", "1"}], "}"}]}], "}"}], ",", " ", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Table", "[", 
           RowBox[{"1", ",", 
            RowBox[{"{", "n", "}"}]}], "]"}], ",", " ", "0"}], "}"}], ",", 
        " ", "80"}], "]"}]}], ",", 
     RowBox[{"ImageSize", "->", 
      RowBox[{"{", 
       RowBox[{"Automatic", ",", "100"}], "}"}]}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->31086],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{561, 102},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->605770273]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->8661],

Cell["Show a single evolution in multiple \"panels\":", "ExampleText",
 CellID->7893],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Row", "[", 
  RowBox[{"Table", "[", 
   RowBox[{
    RowBox[{"ArrayPlot", "[", 
     RowBox[{
      RowBox[{"CellularAutomaton", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"1599", ",", 
          RowBox[{"{", 
           RowBox[{"3", ",", "1"}], "}"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"{", 
          RowBox[{"t", ",", 
           RowBox[{"t", "+", "200"}]}], "}"}], "}"}]}], "]"}], ",", 
      RowBox[{"ImageSize", "->", 
       RowBox[{"{", 
        RowBox[{"Automatic", ",", "200"}], "}"}]}]}], "]"}], ",", 
    RowBox[{"{", 
     RowBox[{"t", ",", "0", ",", "600", ",", "200"}], "}"}]}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->14055],

Cell[BoxData[
 InterpretationBox[
  RowBox[{
   GraphicsBox[RasterBox[CompressedData["
1:eJylWoGB5DYIBDpJMWkgJXwDqTNd5VbMIGRrBfbt/368xhZjpGGwyF9//v3n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     "], {{0, 0}, {69, 201}}, {0, 2}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->{Automatic, 200}], "\[InvisibleSpace]", 
   GraphicsBox[RasterBox[CompressedData["
1:eJytXO159LgNBNBJWkoJ10B+pMp0FS8xMwApSitL1uPXtyuJFPE1A4Dy/euf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     "], {{0, 0}, {115, 201}}, {0, 2}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->{Automatic, 200}], "\[InvisibleSpace]", 
   GraphicsBox[RasterBox[CompressedData["
1:eJylW4uB5LiOI5nJpXQhbAIXxwvtZXXdIgBC/lSXvbU7PdVVlihRJADSnv/5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     "], {{0, 0}, {137, 201}}, {0, 2}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->{Automatic, 200}], "\[InvisibleSpace]", 
   GraphicsBox[RasterBox[CompressedData["
1:eJytW4GZczmKBDK5lDaETWADuoguq5sWVYAQz0/d3v7m77afJYQQFAXy/M+/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     "], {{0, 0}, {155, 201}}, {0, 2}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->{Automatic, 200}]}],
  Row[{
    Graphics[
     Raster[CompressedData["
1:eJylWoGB5DYIBDpJMWkgJXwDqTNd5VbMIGRrBfbt/368xhZjpGGwyF9//v3n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      "], {{0, 0}, {69, 201}}, {0, 2}], Frame -> Automatic, 
     FrameTicks -> {None, None}, ImageSize -> {Automatic, 200}], 
    Graphics[
     Raster[CompressedData["
1:eJytXO159LgNBNBJWkoJ10B+pMp0FS8xMwApSitL1uPXtyuJFPE1A4Dy/euf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      "], {{0, 0}, {115, 201}}, {0, 2}], Frame -> Automatic, 
     FrameTicks -> {None, None}, ImageSize -> {Automatic, 200}], 
    Graphics[
     Raster[CompressedData["
1:eJylW4uB5LiOI5nJpXQhbAIXxwvtZXXdIgBC/lSXvbU7PdVVlihRJADSnv/5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      "], {{0, 0}, {137, 201}}, {0, 2}], Frame -> Automatic, 
     FrameTicks -> {None, None}, ImageSize -> {Automatic, 200}], 
    Graphics[
     Raster[CompressedData["
1:eJytW4GZczmKBDK5lDaETWADuoguq5sWVYAQz0/d3v7m77afJYQQFAXy/M+/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      "], {{0, 0}, {155, 201}}, {0, 2}], Frame -> Automatic, 
     FrameTicks -> {None, None}, ImageSize -> {Automatic, 200}]}]]], "Output",\

 ImageSize->{487, 202},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->61625747]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->17367],

Cell["\<\
Show 200 steps of rule 30, with exactly 1 pixel representing each cell:\
\>", "ExampleText",
 CellID->12947],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{
   RowBox[{"CellularAutomaton", "[", 
    RowBox[{"30", ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "200"}], "]"}], ",", 
   RowBox[{"PixelConstrained", "->", "1"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->12452],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJztmYth3MiORQubyaa0IbwENpYX1ma1T27inoNqmZLHklof1nhkdrM+wP0A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   "], {{0, 0}, Offset[{401, 201}, {0, 0}]}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None},
  PlotRange->{{0, 401}, {0, 201}},
  PlotRangeClipping->True]], "Output",
 ImageSize->{405, 201},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{401, 201}},
 CellLabel->"Out[1]=",
 CellID->13144]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->24951],

Cell["\<\
Make a \"random\" walk from the values of cells on the center column of rule \
30:\
\>", "ExampleText",
 CellID->28052],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ListLinePlot", "[", 
  RowBox[{"Accumulate", "[", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{"-", "1"}], ")"}], "^", 
    RowBox[{"CellularAutomaton", "[", 
     RowBox[{"30", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"500", ",", 
        RowBox[{"{", 
         RowBox[{"{", "0", "}"}], "}"}]}], "}"}]}], "]"}]}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->6719],

Cell[BoxData[
 GraphicsBox[{{}, LineBox[CompressedData["
1:eJxdmEmyFFUUhq+OGOaAAQMGKUG8IAiCKPrGhhSwA4GiURBBUhAbUCjBBhTw
LqGWkEuoHXiXUEvIJdTEuWHk+TDu9yYVHx//Oec2VVnvvXX38ZX7b6aU/nkj
pf9ep5/NqXj9O/6hi9cyvWzrat/Ibw/eFrwjuAneKd8qv0t+t/Jb8nuU36v5
9snvl5/JH1D9g/KHlD8sf0T+KD5+jskfV/6E+p+Uf1v+Hfl35d8LZv9OBW8P
7mqfYfLvy8PkTwfvCH9a/oz8Gfmz8mflP5CHd4b/UB4m/5Hmh1nfx/Iw+U/k
YfLn5M/Jn5c/L/+pPEz/C/IX5C/KX5S/JH9Jfh4c+9fBsb8Zbicu8K7IXw7e
HfnLtc8weZjzu6L+V2qfYfIw/a+q/9XaZ5g8TP9rte+u1T7LF/n0mfIw/eGY
r8inz5WH6Q/H/hSY87te++567TMc51/k043adzdqn2H6y6cvlIfjfmf5ApO/
qflv1j7D5GHeP1+qP0xevsinW+p/q/ZZvsCc/23lb9c+yxeY/l9pfpj9g+Pz
vcA8P+7gp5/uTu2zfHnt4znf46fXtq9999rHf3/to/5rP803wDF/kR/l09fB
sf4Wjv3p5Hv5LD/A9JcflU93g+N82ru172D6w5yP/KB8Uf1RPt1T/3u17+R7
mPsFx+fHIF+UH1U/fVPP38Kxvk6+h9l/+UH5Ij/Kp/vBcT9aOO5XJ9/D7L/8
oHyRH5VP39bztTD7L9/D9JcflC8w+w/z+fFdcJxfC8fzoZPvYc4fjvoDTH+Y
/YdZ//e1b2HWD3P/YPrD3D845i9wPP9G+fRD7Vt4K/rL9zDPV/lBvsiPqp8e
qP+D2nfwnugP743+8oPyRfVHmPU/DOb8H9a+k+/lM8z65YvyI8z9+1H9YfrL
9/IZpr98UX6E6f+T+sP0l+9h7h/M/Zcvyo+qnx6p/6PadzD3Tz7LD/BW9Jcf
lU+Pa98+rn0n38tn+QGmPxz3c4Tj/qZF8L6JG3h/zAPPJp7Jd/Jz+EDMK7+Q
z/JL+QE+OPFKvsiv4UMTj/Ib+fRz8OGJG/jIxC18dOKZfCc/h49N3MPHJ17I
Z+WX8oP8Sr6o/ho+MfEov5FPT4JPTtw8qX0LR34GR/9Ofi7fwzH/Ao79y3Ds
/xKO8xlgzh/m/OG4P2s47ucIx/3eyKendb55WvtWfgZH/w6O+ebyvfIL+Sy/
lB9UfyVflF/Lj/Ib+fRLXb+BY/9bOM5nJt8pP5fv5Rcw65dfKj/Ir5QvMOuH
OX/5jfLp17p+A7N+mPXLd8rPYc4f5vzhmC/DcT+X8oP8Sr6o/lp+lN9ovvRb
7RuY+y8/g7n/MOuHOX/5hfJZfik/qP9K8xU49mctP8pvYJ5/v9e+gVk/zPph
1g+zfvle+YXqZ/VfwjHfIL9Svsiv5Uf5jeqnZ7VvntW+lZ/BPP/l58r38gv5
DMf3k6X8IL+SL6q/1nyj/EY+Pa9987z2Lcz5y3fKz2HWD8f8C3hr4iy/VH6Q
X8kX+bXqj/Ib+fRHcHx/bOD4ftnKz+Q7+bl8D8f364V8ll/KD6q/gmP+Ir+W
H+U38unP4NifBub+w9x/mPOXn8v3MO9/+az8Uv0H+ZV80fxrONY3wqwf5veP
F8Gx/82L2rcw5w9H/Q6O/nP5Xn4hn+WX8oP6rzRfkV8rP6r+Bubz72Xtm5e1
b+VncNTv5OfyvfxC9bP8Un6QX6l+gVm//Kj8Rj69qn3zqvYtzPnDvP/l58r3
qr+AWb/8UvlB9VfqX+TX8iMc93sD8/eHv4Lj7wfNX7Vv4cjPYNYPs/7//b+z
btQS
    "]]},
  AspectRatio->GoldenRatio^(-1),
  Axes->True,
  PlotRange->{Automatic, Automatic},
  PlotRangeClipping->True]], "Output",
 ImageSize->{184, 99},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->9195]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->4048],

Cell["\<\
Find repetition periods of rule 90 in cyclic regions of successive widths:\
\>", "ExampleText",
 CellID->18839],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"Length", "[", 
    RowBox[{"NestWhileList", "[", 
     RowBox[{
      RowBox[{"CellularAutomaton", "[", "90", "]"}], ",", 
      RowBox[{"SparseArray", "[", 
       RowBox[{
        RowBox[{"1", "->", "1"}], ",", "n"}], "]"}], ",", "Unequal", ",", 
      "All"}], "]"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "20"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->6989],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "3", ",", "3", ",", "3", ",", "4", ",", "5", ",", "4", ",", "9", ",", "6", 
   ",", "9", ",", "8", ",", "33", ",", "7", ",", "65", ",", "16", ",", "17", 
   ",", "10", ",", "17", ",", "16", ",", "513", ",", "15"}], "}"}]], "Output",\

 ImageSize->{441, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->700]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->27054],

Cell["\<\
Draw the state transition graph for rule 110 in a region of size 7:\
\>", "ExampleText",
 CellID->1096],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"GraphPlot", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"#", "->", 
     RowBox[{"CellularAutomaton", "[", 
      RowBox[{"110", ",", "#"}], "]"}]}], "&"}], "/@", 
   RowBox[{"Tuples", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"0", ",", "1"}], "}"}], ",", "7"}], "]"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->14321],

Cell[BoxData[
 GraphicsBox[
  TagBox[GraphicsComplexBox[CompressedData["
1:eJwtVWs4FAgXtqRcZmzGIJdhjDEzLsPcjGFpDiqxpG2JjY2GJRUhSYWUEikR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    "], {
     {RGBColor[0.5, 0., 0.], 
      LineBox[{{1, 129, 130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140,
         141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 1}, {2, 3}, {
        3, 6}, {4, 5}, {5, 11}, {6, 12}, {7, 8}, {8, 21}, {9, 10}, {10, 23}, {
        11, 22}, {12, 20}, {13, 14}, {14, 40}, {15, 16}, {16, 35}, {17, 18}, {
        18, 44}, {19, 20}, {20, 45}, {21, 42}, {22, 37}, {23, 41}, {24, 25}, {
        25, 77}, {26, 27}, {27, 80}, {28, 29}, {29, 67}, {30, 31}, {31, 82}, {
        32, 33}, {33, 87}, {34, 35}, {35, 89}, {36, 37}, {37, 88}, {38, 39}, {
        39, 53}, {40, 83}, {41, 57}, {42, 71}, {43, 31}, {44, 79}, {45, 78}, {
        46, 47}, {47, 89}, {48, 49}, {49, 58}, {50, 51}, {51, 57}, {52, 53}, {
        53, 70}, {54, 55}, {55, 82}, {56, 57}, {57, 40}, {58, 59}, {59, 65}, {
        60, 61}, {61, 35}, {62, 63}, {63, 103}, {64, 65}, {65, 11}, {66, 
        67}, {67, 92}, {68, 69}, {69, 1}, {70, 71}, {71, 98}, {72, 69}, {73, 
        74}, {74, 99}, {75, 76}, {76, 19}, {77, 114}, {78, 36}, {79, 106}, {
        80, 21}, {81, 69}, {82, 6}, {83, 76}, {84, 85}, {85, 37}, {86, 59}, {
        87, 61}, {88, 53}, {89, 52}, {90, 91}, {91, 92}, {92, 95}, {93, 94}, {
        94, 100}, {95, 99}, {96, 97}, {97, 106}, {98, 99}, {99, 111}, {100, 
        109}, {101, 102}, {102, 65}, {103, 104}, {104, 67}, {105, 106}, {106, 
        77}, {107, 108}, {108, 78}, {109, 80}, {110, 104}, {111, 76}, {112, 
        113}, {113, 30}, {114, 78}, {115, 80}, {116, 79}, {117, 82}, {118, 
        69}, {119, 69}, {120, 71}, {121, 69}, {122, 123}, {123, 20}, {124, 
        75}, {125, 41}, {126, 69}, {127, 120}, {128, 109}}]}, 
     {RGBColor[0, 0, 0.7], 
      TagBox[
       TooltipBox[PointBox[1],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 0, 0, 0, 0, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[2],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 0, 0, 0, 0, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[3],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 0, 0, 0, 0, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[4],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "0", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 0, 0, 0, 0, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[5],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "1", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 0, 0, 0, 1, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[6],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "1", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 0, 0, 0, 1, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[7],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "1", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 0, 0, 0, 1, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[8],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "0", ",", "1", ",", "1", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 0, 0, 1, 1, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[9],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "0", ",", "0", ",", "1", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 0, 0, 0, 1, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[10],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "0", ",", "1", ",", "1", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 0, 0, 1, 1, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[11],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "0", ",", "1", ",", "1", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 0, 0, 1, 1, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[12],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "0", ",", "1", ",", "1", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 0, 0, 1, 1, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[13],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "0", ",", "1", ",", "0", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 0, 0, 1, 0, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[14],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "1", ",", "1", ",", "0", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 0, 1, 1, 0, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[15],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "0", ",", "1", ",", "0", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 0, 0, 1, 0, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[16],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "1", ",", "1", ",", "0", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 0, 1, 1, 0, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[17],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "0", ",", "1", ",", "0", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 0, 0, 1, 0, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[18],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "1", ",", "1", ",", "1", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 0, 1, 1, 1, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[19],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "0", ",", "1", ",", "0", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 0, 0, 1, 0, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[20],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "1", ",", "1", ",", "1", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 0, 1, 1, 1, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[21],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "1", ",", "1", ",", "1", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 0, 1, 1, 1, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[22],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "1", ",", "1", ",", "0", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 0, 1, 1, 0, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[23],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "1", ",", "1", ",", "0", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 0, 1, 1, 0, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[24],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "0", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 0, 1, 0, 0, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[25],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "1", ",", "0", ",", "0", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 1, 1, 0, 0, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[26],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "0", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 0, 1, 0, 0, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[27],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "1", ",", "0", ",", "0", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 1, 1, 0, 0, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[28],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "0", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 0, 1, 0, 0, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[29],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "1", ",", "0", ",", "1", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 1, 1, 0, 1, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[30],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "0", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 0, 1, 0, 0, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[31],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "1", ",", "0", ",", "1", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 1, 1, 0, 1, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[32],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "1", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 0, 1, 0, 1, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[33],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "1", ",", "1", ",", "1", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 1, 1, 1, 1, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[34],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "1", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 0, 1, 0, 1, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[35],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "1", ",", "1", ",", "1", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 1, 1, 1, 1, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[36],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "1", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 0, 1, 0, 1, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[37],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "1", ",", "1", ",", "1", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 1, 1, 1, 1, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[38],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "1", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 0, 1, 0, 1, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[39],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "1", ",", "1", ",", "1", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 1, 1, 1, 1, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[40],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "1", ",", "1", ",", "0", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 1, 1, 1, 0, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[41],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "1", ",", "1", ",", "0", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 1, 1, 1, 0, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[42],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "1", ",", "0", ",", "1", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 1, 1, 0, 1, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[43],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "0", ",", "1", ",", "1", ",", "1", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 0, 1, 1, 1, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[44],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "1", ",", "0", ",", "0", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 1, 1, 0, 0, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[45],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "1", ",", "0", ",", "0", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 1, 1, 0, 0, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[46],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "0", ",", "0", ",", "0", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 1, 0, 0, 0, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[47],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "0", ",", "0", ",", "0", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 1, 0, 0, 0, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[48],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "0", ",", "0", ",", "0", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 1, 0, 0, 0, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[49],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "0", ",", "0", ",", "0", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 1, 0, 0, 0, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[50],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "0", ",", "0", ",", "0", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 1, 0, 0, 0, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[51],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "0", ",", "0", ",", "1", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 1, 0, 0, 1, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[52],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "0", ",", "0", ",", "0", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 1, 0, 0, 0, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[53],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "0", ",", "0", ",", "1", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 1, 0, 0, 1, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[54],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "0", ",", "0", ",", "1", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 1, 0, 0, 1, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[55],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "0", ",", "1", ",", "1", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 1, 0, 1, 1, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[56],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "0", ",", "0", ",", "1", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 1, 0, 0, 1, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[57],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "0", ",", "1", ",", "1", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 1, 0, 1, 1, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[58],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "0", ",", "0", ",", "1", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 1, 0, 0, 1, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[59],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "0", ",", "1", ",", "1", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 1, 0, 1, 1, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[60],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "0", ",", "0", ",", "1", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 1, 0, 0, 1, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[61],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "0", ",", "1", ",", "1", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 1, 0, 1, 1, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[62],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "0", ",", "1", ",", "0", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 1, 0, 1, 0, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[63],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "1", ",", "1", ",", "0", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 1, 1, 1, 0, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[64],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "0", ",", "1", ",", "0", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 1, 0, 1, 0, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[65],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "1", ",", "1", ",", "0", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 1, 1, 1, 0, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[66],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "0", ",", "1", ",", "0", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 1, 0, 1, 0, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[67],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "1", ",", "1", ",", "1", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 1, 1, 1, 1, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[68],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "0", ",", "1", ",", "0", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 1, 0, 1, 0, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[69],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "1", ",", "1", ",", "1", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 1, 1, 1, 1, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[70],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "0", ",", "1", ",", "1", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 1, 0, 1, 1, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[71],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "1", ",", "1", ",", "1", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 1, 1, 1, 1, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[72],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "0", ",", "1", ",", "1", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 1, 0, 1, 1, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[73],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "0", ",", "1", ",", "1", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 1, 0, 1, 1, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[74],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "1", ",", "1", ",", "0", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 1, 1, 1, 0, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[75],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "0", ",", "1", ",", "1", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 1, 0, 1, 1, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[76],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "1", ",", "1", ",", "0", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 1, 1, 1, 0, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[77],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "1", ",", "0", ",", "0", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 1, 1, 0, 0, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[78],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "1", ",", "0", ",", "0", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 1, 1, 0, 0, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[79],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "1", ",", "0", ",", "1", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 1, 1, 0, 1, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[80],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "1", ",", "0", ",", "1", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 1, 1, 0, 1, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[81],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "1", ",", "0", ",", "1", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 1, 1, 0, 1, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[82],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "1", ",", "1", ",", "1", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 1, 1, 1, 1, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[83],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "0", ",", "1", ",", "0", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 1, 0, 1, 0, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[84],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "1", ",", "1", ",", "0", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {0, 1, 1, 1, 0, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[85],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "0", ",", "1", ",", "0", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 1, 0, 1, 0, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[86],
        RowBox[{"{", 
          
          RowBox[{"0", ",", "1", ",", "1", ",", "1", ",", "0", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {0, 1, 1, 1, 0, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[87],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "0", ",", "0", ",", "1", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 1, 0, 0, 1, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[88],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "0", ",", "0", ",", "0", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 1, 0, 0, 0, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[89],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "0", ",", "0", ",", "0", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 1, 0, 0, 0, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[90],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 0, 0, 0, 0, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[91],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "0", ",", "0", ",", "0", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 0, 0, 0, 0, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[92],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "0", ",", "0", ",", "0", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 0, 0, 0, 0, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[93],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "0", ",", "0", ",", "0", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 0, 0, 0, 0, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[94],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "0", ",", "0", ",", "1", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 0, 0, 0, 1, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[95],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "0", ",", "0", ",", "1", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 0, 0, 0, 1, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[96],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "0", ",", "0", ",", "1", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 0, 0, 0, 1, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[97],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "0", ",", "1", ",", "1", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 0, 0, 1, 1, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[98],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "0", ",", "0", ",", "1", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 0, 0, 0, 1, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[99],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "0", ",", "1", ",", "1", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 0, 0, 1, 1, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[100],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "0", ",", "1", ",", "1", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 0, 0, 1, 1, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[101],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "0", ",", "1", ",", "0", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 0, 0, 1, 0, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[102],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "1", ",", "1", ",", "0", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 0, 1, 1, 0, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[103],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "0", ",", "1", ",", "0", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 0, 0, 1, 0, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[104],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "1", ",", "1", ",", "0", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 0, 1, 1, 0, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[105],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "0", ",", "1", ",", "0", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 0, 0, 1, 0, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[106],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "1", ",", "1", ",", "1", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 0, 1, 1, 1, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[107],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "0", ",", "1", ",", "0", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 0, 0, 1, 0, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[108],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "1", ",", "1", ",", "1", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 0, 1, 1, 1, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[109],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "1", ",", "1", ",", "1", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 0, 1, 1, 1, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[110],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "0", ",", "1", ",", "1", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 0, 0, 1, 1, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[111],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "1", ",", "1", ",", "0", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 0, 1, 1, 0, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[112],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "1", ",", "0", ",", "0", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 0, 1, 0, 0, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[113],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "1", ",", "0", ",", "0", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 1, 1, 0, 0, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[114],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "1", ",", "0", ",", "0", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 0, 1, 0, 0, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[115],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "1", ",", "0", ",", "0", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 0, 1, 0, 0, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[116],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "1", ",", "0", ",", "0", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 0, 1, 0, 0, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[117],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "1", ",", "0", ",", "1", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 0, 1, 0, 1, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[118],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "1", ",", "0", ",", "1", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 0, 1, 0, 1, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[119],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "1", ",", "0", ",", "1", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 0, 1, 0, 1, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[120],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "1", ",", "0", ",", "1", ",", "1", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 0, 1, 0, 1, 1, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[121],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "1", ",", "1", ",", "0", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 0, 1, 1, 0, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[122],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "0", ",", "1", ",", "1", ",", "1", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 0, 1, 1, 1, 0, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[123],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "1", ",", "0", ",", "1", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 1, 1, 0, 1, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[124],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "0", ",", "0", ",", "1", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 1, 0, 0, 1, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[125],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "0", ",", "1", ",", "0", ",", "0", ",", 
            "1"}], "}"}]],
       Annotation[#, {1, 1, 0, 1, 0, 0, 1}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[126],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "0", ",", "1", ",", "0", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 1, 0, 1, 0, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[127],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "1", ",", "0", ",", "0", ",", "1", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 1, 1, 0, 0, 1, 0}, "Tooltip"]& ], 
      TagBox[
       TooltipBox[PointBox[128],
        RowBox[{"{", 
          
          RowBox[{"1", ",", "1", ",", "1", ",", "0", ",", "1", ",", "0", ",", 
            "0"}], "}"}]],
       Annotation[#, {1, 1, 1, 0, 1, 0, 0}, "Tooltip"]& ]}}],
   Annotation[#, VertexCoordinateRules -> CompressedData["
1:eJwtlWs4FAgXgEVahDIGuTbm7jLMzcwQ7UFlWdL2ubS00bB0c41Jm1spkRKl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     "]]& ],
  AspectRatio->Automatic,
  FrameTicks->None,
  PlotRange->All,
  PlotRangePadding->Scaled[0.1]]], "Output",
 ImageSize->{166, 216},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->66732148]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->27921],

Cell["\<\
Generate a difference pattern for two cellular automaton with initial \
conditions differing by one bit:\
\>", "ExampleText",
 CellID->29367],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"With", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"u", "=", 
     RowBox[{"RandomInteger", "[", 
      RowBox[{"1", ",", "100"}], "]"}]}], "}"}], ",", 
   RowBox[{"ArrayPlot", "[", 
    RowBox[{"Sum", "[", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"(", 
         RowBox[{"-", "1"}], ")"}], "^", "i"}], " ", 
       RowBox[{"CellularAutomaton", "[", 
        RowBox[{"30", ",", 
         RowBox[{"ReplacePart", "[", 
          RowBox[{"u", ",", 
           RowBox[{"50", "->", "i"}]}], "]"}], ",", "50"}], "]"}]}], ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", "0", ",", "1"}], "}"}]}], "]"}], "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->25968],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy1lt1xwzAMg8lu0hmySUfI9b37vzW1SABKXF8kUemlsfPDTwREUp/3n6/7
h5ndHs/vx9Pff/x9O37QruLe4i++ktfH3fEYYHh83xg4A7a3goqg/d3bkPbi
CN7easzIh0HHs4jg+BUEaSIiK2bR8h1lQA2oJgjLnDK/CQRiUZzjIhdsqSH3
whgi99QpIgVjAvRqMAsoxOWScrJ5J7yQ6gAhHPHcWPDFbByB6nB3SUCMNXva
w+MIkberbe6tXr0pBFsDXXB4LIiQau5BwZ21yKQsi9Qn7AbD6YKUO8pDS3I2
B7zIkumxm348C8klPjd2aYwLdh8QNlX2Ju20soppBpcIXbQN5jZbQFAM6oNW
BSNKECk+y9zE9EUE4gKRU0iH0yoi95P2Eacby0IlBGUNg6XaV7Po25W43D6q
QfiZUBysC9XdMzAJMWHRVmoQ3ajQGTI99U7jy2HDmIQVIWTnyNEmt1rBpiXC
xQgVq8aLNBjFLu2jyG7H0KZY6X4lwlMU0+BlQrXIqAEeF/C/CoJwDF4oVI9A
A549DV5BPFutDJFaBBfOU1YtgpPclVWM0Em4esj5H5LHBN9hd0BwLvTKBvIC
yYPiDqFa5JhR+7LIMtyyaTtI3dS7gmxGFA7WC8ROuwWyGzFVF7+oKRLW
   "], {{0, 0}, {100, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 96},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->31545]
}, Open  ]],

Cell["Show the common evolution in gray:", "ExampleText",
 CellID->17716],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"With", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"u", "=", 
     RowBox[{"RandomInteger", "[", 
      RowBox[{"1", ",", "100"}], "]"}]}], "}"}], ",", 
   RowBox[{"ArrayPlot", "[", 
    RowBox[{"Sum", "[", 
     RowBox[{
      RowBox[{
       RowBox[{"(", 
        RowBox[{"0.1", "+", 
         RowBox[{
          RowBox[{"(", 
           RowBox[{"-", "1"}], ")"}], "^", "i"}]}], ")"}], " ", 
       RowBox[{"CellularAutomaton", "[", 
        RowBox[{"30", ",", 
         RowBox[{"ReplacePart", "[", 
          RowBox[{"u", ",", 
           RowBox[{"50", "->", "i"}]}], "]"}], ",", "50"}], "]"}]}], ",", 
      RowBox[{"{", 
       RowBox[{"i", ",", "0", ",", "1"}], "}"}]}], "]"}], "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1458],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy9m9uNI1UQhkdkQgwEMDkQAmKfiQdEQkiQCS9ksLtIfinp03+palsaWW53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   "], {{0, 0}, {100, 51}}, {0, 1.1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 96},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->26519]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->7096],

Cell["Show a \"glider\" in the \"Game of Life\":", "ExampleText",
 CellID->26135],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"ArrayPlot", "[", 
    RowBox[{"#", ",", 
     RowBox[{"ImageSize", "->", "40"}], ",", 
     RowBox[{"Mesh", "->", "True"}]}], "]"}], "&"}], "/@", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"224", ",", 
      RowBox[{"{", 
       RowBox[{"2", ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"2", ",", "2", ",", "2"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"2", ",", "1", ",", "2"}], "}"}], ",", 
          RowBox[{"{", 
           RowBox[{"2", ",", "2", ",", "2"}], "}"}]}], "}"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"1", ",", "1"}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"0", ",", "1", ",", "0"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"0", ",", "0", ",", "1"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"1", ",", "1", ",", "1"}], "}"}]}], "}"}], ",", "0"}], "}"}],
     ",", "8"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->13861],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{464, 42},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->201919402]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->27754],

Cell["\<\
Show the evolution of the \"puffer train\" in the \"Game of Life\":\
\>", "ExampleText",
 CellID->8670],

Cell[BoxData[
 RowBox[{
  RowBox[{"GameOfLife", "=", 
   RowBox[{"{", 
    RowBox[{"224", ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"2", ",", "2", ",", "2"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"2", ",", "1", ",", "2"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"2", ",", "2", ",", "2"}], "}"}]}], "}"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->18290],

Cell[BoxData[
 RowBox[{
  RowBox[{"Puffer", "=", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{"1", ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", "5"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"3", ",", "5"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"4", ",", "5"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"8", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"9", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"9", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"10", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"11", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"12", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"15", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"15", ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"16", ",", "5"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"17", ",", "1"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"17", ",", "5"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"18", ",", "2"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"18", ",", "3"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"18", ",", "4"}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"18", ",", "5"}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->24985],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Mean", "[", 
   RowBox[{"CellularAutomaton", "[", 
    RowBox[{"GameOfLife", ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"SparseArray", "[", 
        RowBox[{"Puffer", "->", "1"}], "]"}], ",", "0"}], "}"}], ",", "200"}],
     "]"}], "]"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->7278],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJztnWvIZVUZxwdLiOmDSRdFxDCkO3ihRLLcUh9yiIqkDKoPZkgGpt3UKJFE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   "], {{0, 0}, {118, 73}}, {0, 0.487562189054726}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 114},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->4671]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->14156],

Cell["\<\
The \"Game of Life\" from random initial conditions, averaged for 100 steps: \
\
\>", "ExampleText",
 CellID->2130],

Cell[BoxData[
 RowBox[{
  RowBox[{"GameOfLife", "=", 
   RowBox[{"{", 
    RowBox[{"224", ",", 
     RowBox[{"{", 
      RowBox[{"2", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"2", ",", "2", ",", "2"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"2", ",", "1", ",", "2"}], "}"}], ",", 
         RowBox[{"{", 
          RowBox[{"2", ",", "2", ",", "2"}], "}"}]}], "}"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{"1", ",", "1"}], "}"}]}], "}"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->6571],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Mean", "[", 
   RowBox[{"CellularAutomaton", "[", 
    RowBox[{"GameOfLife", ",", 
     RowBox[{"RandomInteger", "[", 
      RowBox[{"1", ",", 
       RowBox[{"{", 
        RowBox[{"100", ",", "100"}], "}"}]}], "]"}], ",", "100"}], "]"}], 
   "]"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->18834],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJyd3WuorltZxvGNEIIEUUQkRiRCRBKEEIUIi/pgJBJYfZAkEEkCCYIIzAgh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   "], {{0, 0}, {100, 100}}, {0, 0.97029702970297}],
  Frame->Automatic,
  FrameTicks->{None, None},
  ImageMargins->0.]], "Output",
 ImageSize->{229, 225},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{225, Automatic}},
 CellLabel->"Out[2]=",
 CellID->31354]
}, Open  ]],

Cell["Averaged spacetime slice:", "ExampleText",
 CellID->8343],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Mean", "/@", 
   RowBox[{"CellularAutomaton", "[", 
    RowBox[{"GameOfLife", ",", 
     RowBox[{"RandomInteger", "[", 
      RowBox[{"1", ",", 
       RowBox[{"{", 
        RowBox[{"10", ",", "200"}], "}"}]}], "]"}], ",", "100"}], "]"}]}], 
  "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->21429],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJztnc2N7TByhR+ciWNwAM7BIRgYbyceO5OXgRuwV726uwYGMDCAFxOCPaNh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   "], {{0, 0}, {200, 101}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None},
  ImageMargins->0.]], "Output",
 ImageSize->{378, 196},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{373.999999999999, Automatic}},
 CellLabel->"Out[3]=",
 CellID->6696]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->30423],

Cell["Patterns generated by a sequence of 2D 9-neighbor rules:", "ExampleText",
 CellID->8800],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"ArrayPlot", "[", 
    RowBox[{
     RowBox[{"CellularAutomaton", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"i", ",", " ", 
         RowBox[{"{", 
          RowBox[{"2", ",", " ", "1"}], "}"}], ",", " ", 
         RowBox[{"{", 
          RowBox[{"1", ",", " ", "1"}], "}"}]}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"{", "1", "}"}], "}"}], ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"{", 
         RowBox[{"{", "30", "}"}], "}"}], "}"}]}], "]"}], ",", 
     RowBox[{"ImageSize", "->", "Tiny"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"i", ",", "2", ",", "20", ",", "4"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->9716],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   GraphicsBox[RasterBox[CompressedData["
1:eJxTTMoPSmJiYGCwhWIGRkZGSvBQBKN+HvXzqJ9H/Tzq56EFRv086udRP4/6
edTPQwuM+nnUzwQxAJaED6w=
     "], {{0, 0}, {61, 61}}, {0, 1}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->Tiny], ",", 
   GraphicsBox[RasterBox[CompressedData["
1:eJylVtl1xDAIHKWTFJImUsI2sP3/ZV8siRlAhy0/WwdgbpC+X+/f1xeAn/oB
pUy+OfozfKb/N5sX6OMHjkvfY41uRjRo31faERrVKewdOINyNDr/DjBpF1vU
FWshhCZGpCEuPSGJ3/G9kaLGWYPNZhbvAAiu09rOVIHpTvoS5US10VJhwheD
nxF0q4XRIpNxK2x4i53LIxYGRQBzO4XSPNVG83ugdeBcgqORer4g8AWkf1um
xnpGsNYJS1mjqzL1ClkOB9htKpy6YGYrBpTRYiBHqICCS07M4w3wZGpxZSV0
F4EvHh/ViKZ6Dpqs9rFKtn8OPWw3Ukyc9y+f7V5SA1upjvUMKOljvTHMlQcJ
09R8IPnI5j498XYqaTfORxl2lNsCvVtVwv1uPR91kqMeZvsH3dMh7/Vt1vj2
iXF0VmXaWczaqNEePxZc5pf78uhmEGjv3EkGik+XmZ2Ob6FcH9zDJD21waxv
gHJUroPBxem9ykSIy4TQlONK3blvd/F4cNNvTuGzwyXuEN1d7ktNc3uAPnok
xHGeo5sBg2+O/gPujw2o
     "], {{0, 0}, {61, 61}}, {0, 1}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->Tiny], ",", 
   GraphicsBox[RasterBox[CompressedData["
1:eJzFV9ERwyAIhW7SQbJER8gC3f+vl7OpRXmSA3LkTiWKPF5ENM/9/dofRLR9
CzExM65XY0ftf+iv1eTVWAy5WegsSOm39LyoOj9U0DwPMl47jTOn8m5WpWy9
ZyB236/G9jjPi6yvY5a+ZQ3zvodvsyLlcd9a+zyCLNfz7Bs5j98knsNmT3Qu
OesqLc4xjGJb648g58hedD1ur8gR1FiJoldxzpA9yFWxPXtStZ/PvvtzWF3e
bhaqzqqa87nuTtI9kLL1noFYdfdE/Kw8nZu30T+FpedF7e1q3yK9CLIev6vY
/tUfCygPDA==
     "], {{0, 0}, {61, 61}}, {0, 1}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->Tiny], ",", 
   GraphicsBox[RasterBox[CompressedData["
1:eJzFltERgzAIhkM36SBdoiN0ge7/1p4GAgSQBr1STTXJFwI/et5f7+fr1lp7
9LNmsG7/hhej3drVXI0mvQaA4OpwKnqH+55Zz306h4GGImz/kzCtF+q+D/dF
Zng7fBjAhffrGJZzeQcTzNjwGJ9gMahyj/kQ1SBhwB6jYnk/WLC88m9MzxS9
3sKI5UqYJ3Yq+SGWB1NarcuDbfeEKriRggFM7zSpFHXqvCtNQngSzfUM9Mt6
7rWIzxgKwEZCuA1N8cCOPKw8J2Hs4DGzYkvBMOx3eM1zKeZKtks6FyrsrNo2
4Nhz6XmuvUmkJmqrRzBqCkbL5lwDs+SaNyasXXETWzA9Uw0A0BRcdmRerIeR
ykGQpha3YFTSs2ku64DKNwmJ4cI8HwOGyneYzm9kOH1qjhaAGVEKJhY4D16w
bKSe/1DXI/snXLEPsIcIrA==
     "], {{0, 0}, {61, 61}}, {0, 1}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->Tiny], ",", 
   GraphicsBox[RasterBox[CompressedData["
1:eJylV8FxwzAMA7tJB+kSHaELdP9fe5EIApRyl0jJOaYtkgZA0nY+f36/fz4A
fM0NEXG+4bH/t9D20fb7dZTf+BlH/kEtj8gMncs0mJ2RzO/+cvaRXdzITVM2
f7o9Amdw55fBIrGyIQ0B7ZYKZMqMA2SiUkNNOhRyUaUTXNQyJ1bLUg3p9gfi
hxKCqetXQNnJEhZDKskJIKsC+IkiFfymhKDo5YtSjuaMsG6j2AlyXI+qeB5h
ZWNAXMaqj4rqIWrCHOQIHqKNwKFTHZIfG1a7hg0rLd3J2Ao4Bja8q9nRdhMd
SvNrJ4RiaZpmKaHBVbxs15BI3Qthl92mThiB2B2pQw7xlKpGhCu4NziqsKsg
qqsMmJDJesuAlHf1QoNalEtwXYKslrjcJVZsokI6cM5TNl9CrcU+RF07ISTS
6HxIM28UXDtMUlo/aGLv9wo0nyfB71z5jvON2obxsM5nHXbX2zdTlQnO5/n0
TnJ3D7u5e67eauLZQtE9fmI0KBP+q88qBhvM156SDeD7z+dE9fabgfHDkqdH
cH6qheQlY2qZeeAnBOis89V7WGhOKoL9gfhZRQ7ePYuqT03pRJMOFs/abi7a
Ls+ae+cevOlrkdAJhuXf/MdoRWrAV83WokohfMS7MlU/q9bJP7rkdbT9AWN4
DFg=
     "], {{0, 0}, {61, 61}}, {0, 1}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->Tiny]}], "}"}]], "Output",
 ImageSize->{563, 102},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->26590]
}, Open  ]],

Cell["Mean cell values:", "ExampleText",
 CellID->24222],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"ArrayPlot", "[", 
    RowBox[{
     RowBox[{"Mean", "[", 
      RowBox[{"CellularAutomaton", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"i", ",", " ", 
          RowBox[{"{", 
           RowBox[{"2", ",", " ", "1"}], "}"}], ",", " ", 
          RowBox[{"{", 
           RowBox[{"1", ",", " ", "1"}], "}"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"{", "1", "}"}], "}"}], ",", "0"}], "}"}], ",", "30"}], 
       "]"}], "]"}], ",", 
     RowBox[{"ImageSize", "->", "Tiny"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"i", ",", "2", ",", "20", ",", "4"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->7572],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   GraphicsBox[RasterBox[CompressedData["
1:eJztW7FuFEEMPVFRUtKmpKRMgZAoKfMJoNQU+YWEijJ0tFBTUlJEJAVCUX4C
PgPliCmMHDPzxvPWu44UWbdzHo9nbD/bs3fw6s3R8YPdbvfs7v/00YuHB6fX
z98fvXzy8fzmH1rj6xqXz5pa/NZ8xb9sfnn+68PJj8Of37vlFP+y+bUd6O97
/u99v9UeS/5Y+d65Xz5+/e7ptyuTzxpH8aXkz5Vv2Qtqv6h9l/wY+Sh+lPxl
yvfsSeKFPBdq8Xv4osf1vDo+sf0pm/69+CByzvbPv5p69uILOn80PmbVvzV/
0/b26fx24MK0P0+e9g89n+cvs/PdrPqj8d9a3yx+dP1b0d/DC+/7rfaI+gO6
/q3pz8IvNv6W/mPs38sX0XyVTbeufxT+ZadZ9UftWfBIngu1+EfXp6PrW70+
rZfoy/JnFj4J39v9+GfznKPq0+j6Nlq/3vObnW9qe7/a0y9/P2v9PXvX8tH6
1ePX65L1Wvp48WpWvcGuH639mSU/+/p76+fofpDnDxLvZF6hVjzQ/o/6s16P
Pje9Lh2f0Xg0uh8Whb8ov+i5u/uzztnCPxR/vfm99UXvz2j8tuwFzRe9fFX7
i8TD//U3NJ/24oWsx4o3aL3g1StoPx+N/1EUxa/s8qP23/Nn1J7Y/sSOJ7P3
vxefUXxg4yM7P2Dtf2/+htoXK99l5/ut/jtq/9H4n73+zlL/ovyj8QT15+h+
FLsfN3v/e/EbHUfxB+0vs/vr7P2fRdH6Fr0vYt+XFY21K1b9v1Va/szZ79nj
hc+c/a98e9359ij8qvo5Z/1c/bA/862lHxaFn+j81d+ei/+WvdR91f3xZjTe
jt7/KPxAKfv+ly0/av89e6r3SXK9T9KLD2h9ycZHdn4Qvb/W+bbmb9re6n3P
tnzfk7/W3z+z69fs6++tn9H6t36PcX8/Do2HrefHwm92fxkdZ+uHnl9vPt6a
L7Lvi9Bx9L4MrVeWStn1N5tm1R/15+z2nE1/Fj4vBZ+2pv/sfHt2von651r0
Z9fPs+rHKP4s+rf2V1B7RP2hdb2t/rw2/aPwJ7o+jcbvrelv2QuaL6L5Kuq/
qH+vVX92/VfyY+R79sz2p5LfJr8Xn9j4WPL75Pfmb6x8t+T3yUfjf/Hn4Pfy
v1Z7LP5l8ffiR43nGv8NWRnmlA==
     "], {{0, 0}, {61, 61}}, {0, 0.290322580645161}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->Tiny], ",", 
   GraphicsBox[RasterBox[CompressedData["
1:eJztmyuPZVUQhTsoJBI7EolEEBIkciQSgkIg+AtDgkAODoEBRzIGgkRMYAQh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     "], {{0, 0}, {61, 61}}, {0, 0.354838709677419}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->Tiny], ",", 
   GraphicsBox[RasterBox[CompressedData["
1:eJztXL1ulEEMjKgoKWlTUlJSoEiUlHkEEDUFrxCoKENHCzUlJQWCFAghXgIe
AxHFkTIwN7b397vbSNHq2/Xa+3m9nvHeJcdPnp8+u3V0dPTw6vfszqPbx2c/
Tt6cPr737vznCT5ju2/yo+33lrdnbL3jqp1d/2j7vd/v99sX3x/8+kblcNwb
X7PqH22/t36lT/V74yuqNxvvUTuj7Y/yn8l9vfv09f0vF9f97JmtMzu/tN26
/VbrV/Fl8YKt973V/Nb4Nbv91v5j599aixdmx4sXrGX6S/HLu/7R9nv7LxqP
Kt9gv8ljy/SX4hfG++z2lR22n9HzzOSZP2z85WX7Sa4nKs/ssXj3xuNs9rPy
DJ/V/qp8pOLi/fnfjs/Xdu0Z/YlybD7Gl2px3tbt43zvuazFI1k+wfWxOPeO
o16Vzw7Nfu199PI2XD8+R/u9+Yjl363bx2fvOtR+RvFZ4Zf1G75YPz6z94ni
C8NN0z+rfXtmvIr5O4vPpXxb8Zko34zGsZKf3X6pf6P1gbc+s36LR4tbbG0c
9TBeUqu+VPpnsZ/136j6OXqekU+i31A/W79XXvFdJq/seuUVn299nr18GveL
4QPjj4iHJv/qsv3gvn/x2rd+02/97Jnl8ex9tL2P6cf9Zvyglv+9/FrFcZYv
snN0cdn/8Z985a0v1fkyebTjzQfKPuZfs+N9f2+94s0vUT6RxSNvnOA8tQ9Z
/GL6s/zDG0de/GT1c5YvZPFZjZfySfQn5lOs01S9is82z/RiPaPm475jv+ll
+ah2PZLdv2i9yOo7Vs9F84XpObr6QTy3Zxtn+87kUR/b16z96HljeKzOp7ee
b31f5sUHlq9s3PKhnRM8V0q/yaMeLx9Bu6aHxafiD1m/1b73bN1i/KJ/WJ5l
dSa7N2LypfPV+nrdT/fap2xc4jjjOV4+y3iA9Zs8tmqetx5gPDGLr7XP8yh8
VnzMxg3/GD/CcXbulT7cv1L7in+NwufRfFvdF7K4wvsIzKdevq3mM7+r+83Z
+Hbv+jmqH+UVP2L3Rt5xxvey6/Xyvd71c+v7MIwPVhepeyFWb+N+RvtZvcXs
s7qMvS/b31b3YV4+jHHP8JjxVeRbWI+iHzCfYv2F+lgezN5vj15/qf+9/Irt
G4vLUvxhfMjLt/GeTOlHf+M8zNeKbzP+V4u/lH5e5eXDrD/7+SnmW6x3rN/4
LPrZnm1c6WF5cZT9Uv+p/cnytXWet32eFz4fNj4vvr1ffHvVz/8fP5T6ed2H
3Zy/L/dh6377Jr8qta/2l/mn9f129vMPFVfeeFufV8X6t9Kif9BP6/PnOdrS
uMR8tL5P0vY8j8JnxRdNz/p+WF18Hs231/c9+/Dt3vUzm7e+v717Xu36ufV9
GOpbf4+x23+l92FePoz5heEx46tYPyHeMX9718NwZP191W5+xfaN7UcUf3C/
GQ/z5hcl771nY++jzo+SZ/VWLf9m8RntRfFL4ae6L4zikcLlWe0f2ufPUbyJ
ys9uv9d5ZvJsPay+w/71/0l22+/9/0my911Rvujt965f8eut2lf1ae37Mi8f
Z/Uz499qXN0PH7r92vvozfOqvlR8V/FR1So+vzX7ON+Ls1F8UfiM9wOKL0Tl
Ff4r/GL5dzb7WfksPpfybRVfUb4ZrQeZ37ZiX9kZVT+X1pfZ+k/hS3T9o+33
9l/0PGM8qVbNV/ajfCK6/tH2R/mP4XP2vjc6v3Y9sDX7rdbP8o+3P4pf3v7s
eYraGW2/tf+UPMOLUvyaRf9o+731Z+OrFn6N1j/afq/3Y8/M3r7Ij7bfS/4P
K4iAYw==
     "], {{0, 0}, {61, 61}}, {0, 0.290322580645161}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->Tiny], ",", 
   GraphicsBox[RasterBox[CompressedData["
1:eJztnL9u1jAUxT8xMTKydmRkZECVGBk7MoKYGBh4hQ4MjGVjYIENKQuIMQMq
HRBCCFW8ATwG4mtdKSc5OddJ6tjORaqOnD+O7/nZ106+hIPHz4+e3tjtdvcv
/05vP3l19/T3oaura3366fzRsw8fz3tl17yV8Yvl7pq3Ls31/cnRwzvvfrmu
rHO5O+cydGnO6jps/+v/cvLTNVKn+r00Z9zO2nt868HNg+MfrpGq+CsuS49n
1g7V3q33A6svyufUnFl7rPGu7XtqvrH9gPmdmjOLI5T/vnnx/d6fb714wvba
eU/1gZVTcba2B+ND/bq/3hmNvxZl8aKyfGjlnZoziwP3q3y1Np+545fFwfxg
fuXKWfFk8U2dz3JR5YPVD1VvrpxZ/7Xm9Vi/UuVhbIc1L6vxnCtnvL7ihOeH
4y7q/3KlF/tb6R/Ws7Sq/No9ru3FgfUw3hin8nktzioO5g8q+sP6+TiXdrYq
7mydhe1HteY55e/anFXeCfGgH6zMfAv14XFd/+L5Yvu6+aXfv1g7WVmts6x+
lsaZccTjkAfjOpwv+/mUzQ+4PbYdqh/Uyhnbh/6z/s74ne2P/0zzbNgfVM3v
6jy8TtjP+gPLV9jOuT7mypnNr+N5MvjbUJ/Zc4huf2jMZVbfeD9rev2RtW+u
b6VwVutl9IuVcRwGv1l+ZVxYHkb+U9un4t46Z8yjjAcex44P3F7u9S3lzepj
/Yddd+ucMW62fmbrKsWXcVH9IbZetm5j6/Olnu+Uxhl9YOtnnG/V+Bne30hV
6zB2fdZe9pxka5zZ/Qdy6/rEeSDXkJ+Hebe9+jCfs/qxPvacRt33b40zGwc4
Xmzzcb8f4Hyq7pdUfWy9z7az+XqrnG35l6/DcPx2fW4Od5f/wnmhHPYjL6zP
ug5TcTnnMV/4fRCOO9yOPFh5/Hh9/bh4nDPLh+NccV3VXo1XPB/zadjfHd/I
nV2vpfOLc47jjH6r9ZV6Th3isj4XR87D99+Nc/bx7Jx9ft4sZ19v183Z75/r
5uzPw+rmjLz9+XadnP33qro5++/P2/j92d8n2QZnls+CT8Oc+ve1OJ/7+2F5
cPb3PbvtrO19T39/e3h8M39q5ezfY5TJ2b+vGu6XbP7AenL/vsq/lxyeH7Ce
Ur+X9O+fhzmhlv79s/9/BtP6u9WvXDmr/KriKY1vbBxqPKt1Wi6cMU6Vl9fO
w6nyvDWvW/29bs6xvBlH6/qydJ3qg5VvKs5YtnKrne/UeNFH5ndqznh91t5a
5t+leCtflM/XzRm3O8c0/YD5f93jOXa8u9p0qt9zOVu5u6ZVxck516FzOcdy
d81Dp3J1zmUp4+fq6lqf/gMvzAEP
     "], {{0, 0}, {61, 61}}, {0, 0.419354838709677}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->Tiny], ",", 
   GraphicsBox[RasterBox[CompressedData["
1:eJztnC2PZUUQhicoJBK7EolEkE2QyJVICAqB4C+sRC4OC45kDAQ5YrKsIIQQ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     "], {{0, 0}, {61, 61}}, {0, 0.419354838709677}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->Tiny]}], "}"}]], "Output",
 ImageSize->{563, 102},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->14322]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->20708],

Cell["\<\
Render a 3D view of a 2D cellular automaton evolution using spheres:\
\>", "ExampleText",
 CellID->26748],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics3D", "[", 
  RowBox[{
   RowBox[{"Sphere", "/@", 
    RowBox[{"Position", "[", 
     RowBox[{
      RowBox[{"CellularAutomaton", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"10", ",", " ", 
          RowBox[{"{", 
           RowBox[{"2", ",", " ", "1"}], "}"}], ",", " ", 
          RowBox[{"{", 
           RowBox[{"1", ",", " ", "1"}], "}"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"{", "1", "}"}], "}"}], ",", "0"}], "}"}], ",", "20"}], 
       "]"}], ",", "1"}], "]"}]}], ",", 
   RowBox[{"ViewVertical", "\[Rule]", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"-", "1"}], ",", "0", ",", "0"}], "}"}]}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->2902],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{206, 150},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 Graphics3DBoxOptions->{ImageSize->{206., Automatic}},
 CellLabel->"Out[1]=",
 CellID->35726924]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->25446],

Cell["\<\
Show a sequence of steps in the evolution of a 3D cellular automaton:\
\>", "ExampleText",
 CellID->4728],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"Graphics3D", "[", 
    RowBox[{
     RowBox[{"Cuboid", "/@", 
      RowBox[{"Position", "[", 
       RowBox[{"#", ",", "1"}], "]"}]}], ",", 
     RowBox[{"ImageSize", "\[Rule]", "Tiny"}]}], "]"}], "&"}], "/@", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"14", ",", " ", 
      RowBox[{"{", 
       RowBox[{"2", ",", " ", "1"}], "}"}], ",", " ", 
      RowBox[{"{", 
       RowBox[{"1", ",", " ", "1", ",", "1"}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"{", 
        RowBox[{"{", "1", "}"}], "}"}], "}"}], ",", "0"}], "}"}], ",", "10"}],
    "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->24864],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{416, 318},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->158031399]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->31109],

Cell["\<\
Show time averages of slices at a sequence of heights in a 3D cellular \
automaton:\
\>", "ExampleText",
 CellID->28401],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"ArrayPlot", "[", 
    RowBox[{
     RowBox[{"Mean", "[", 
      RowBox[{"CellularAutomaton", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"14", ",", " ", 
          RowBox[{"{", 
           RowBox[{"2", ",", " ", "1"}], "}"}], ",", " ", 
          RowBox[{"{", 
           RowBox[{"1", ",", " ", "1", ",", "1"}], "}"}]}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{
          RowBox[{"{", 
           RowBox[{"{", 
            RowBox[{"{", "1", "}"}], "}"}], "}"}], ",", "0"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"20", ",", 
          RowBox[{"{", 
           RowBox[{"{", "#", "}"}], "}"}]}], "}"}]}], "]"}], "]"}], ",", 
     RowBox[{"ImageSize", "->", "Tiny"}]}], "]"}], "&"}], "/@", 
  RowBox[{"Range", "[", 
   RowBox[{"0", ",", "15", ",", "5"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->31603],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   GraphicsBox[RasterBox[CompressedData["
1:eJztmj1uGzEUhIWcIm2ANDlFarcBcgAHSJ06F8gBcgDXPoJqV65cGMgZco0I
ibfQIz/OzFIBJK8MGcQuueS8eTOP1M+7L98+fX2z2+0+vPz/fH9zeD1//Pz3
7xe2yzg1/rX0p7zQ89Sv5lXzqfbc5nf5qePufn8/vJ7sedx2mdfFmeqlXtc4
ZnlX/BC+Or7yoHS6jE/XXatHpYu1+ImP1PcuD27+a1wUn9vSfC6Otf5UeR3z
8oA4FX7Kf6pDpSdVLxTe4/sPsU4ynbd8vv1xe3g9Nteuv0/Vku4JH/Hn1gGl
Z4Wv8lRxKX7X1kXCla5H+F2/p7g8X7T568fR+qTG6ead+Kh1iOZX/nL9q3is
69W4+/nco9/JR/245vvVuhUvxaXOa2neVV5qfsk347rE5013fxvn+bHRXxpn
yhfhpnpDPlb1Z7nv1p/aP9ZbXa/1uRufe350nyN9kR/+3b9v+mldVXfUc318
97I+jetszqMar+o5tbuXP69utXy7ulbr9ddv9z9XX64eUn+Pdbng3zd4Kb5j
nG2++jzvJY7Uz64flP6UfpWOqV5R/ao8Ec80H9VdhZt05OqOeFU6HuuCz9vq
/OH6mvStfFLzszZ+pVO37rvrqfNH9aPSY+Wv6tnFk8ap/Kx4nl1H1fvKz5jH
lm9qZ3VCcc7yRfPWelPHKd2M99N2P6bnq84JR6rHS23X+u9S26sej3mg63Se
rdfH6349pxPF6/X8OHd+rPiVDrf+fkbplPghvVW++vvI639/rXTV779+3uOO
p3hr28//9j5/VM+R/vv92/k83L2mtu/37X4/Q/eVH+j8QXG5PM32e3X4dN8X
Kn9TS+eNOo+q5zXO1CfjunY+31/X+1R/3PMEjVO43Ly76xF+dz9W92mcyh/h
r/6Y5c3lVZ0bx3U6/32P59tn5G88/v99buzOl8W3/vdmbl5dv7v5oTqetjRf
qpO18dNz9Lyrc4pP4U11qZ6jvKX4lU5S/lV+3fwT/tQnys/p/qv8murYxZnq
JcV9LvOfmp9UL5fen/LyB0lJGLk=
     "], {{0, 0}, {41, 41}}, {0, 0.333333333333333}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->Tiny], ",", 
   GraphicsBox[RasterBox[CompressedData["
1:eJztmjFuG0EMRQWfIq0BNzmF67QGcoAESJ06F8gBcoDUOsLWW6VKIcBnyDUi
JB4D+jt/3p9ZrQM4a8gYrIbkkJ+f5K6k24+fHz7dHA6Ht0//3+7enV+n+/d/
/h7tWuRSeafn1tTe1vZ79VSOzh+NN7X3UnK9fqf4qNz3X1/Or5+xHYqr2Bv1
s/c8PWereAhHJ+/wULm1K+G8FncXJ8Xl4kx5TfiXfcLdyVHeyL6z484nud74
Ka91P2eMJ8UjxZfqheyN+lvipbyl9VTP5xLPsr75+uH8+vG8prg6eccPlSfc
1K82b+ZFftJ+k76fxpHhPsf8y/Ydf2b0k/JI+BBeWX0scVJ7dR4s48vqifX0
vHbf8/3KxZ/iSH2hns/JxDVbHB0+yo86X5b6dRxnY3+yOFH8hCPhSX2PcFM7
Ku/6WHt/RpxVnvroVvipP9qHXTwUv7NPOFL+3LzT84v/9f2cj4SbnjNaV9Rv
nJ7LE/WZtD5cvdR5Mn7/Q3MmrbPSjy7x0n66xLF9rX54+9RHaM44fFKepjxK
+0yJ7zKOSeKbbD2SvOKZ9r00XqrnlKduHjv+KF+IbxkPWa7Od29H/V2LE/XN
9pzm50fHN9WjtZ4nh1/+vEd153Ai3NVuOi/beVvyoPjxV/94f3j6K/vluuzr
/Kmvy/PT+4W0jxH/iJcah5s79XUyekeY+yfJ19HkUfulvw/SPBIPCcf0Pojy
7Oa38lXjueSf4qV2Ts/ydN/gzs3qpv95xskp/9L7IrVD/dDFq/rtulzWRW+/
1zhp7ux8fFk+Ort7f8zmjOP/Pq/b+Kb1vd8/Zvwj3Ov+7M8zhHObD/vzdVrv
VM9uTtZxeP2f97h+ofXs+oxeu/omXquezhM3x6iPurmv16rveJXWs9PT/Lk4
HV/Tealxu/cdXo5vLk/1/f37mbQ+0viuhR/VkZuPo3XleDTaV+r2t/u+MOs3
+/fXKY6Er/KD+rzDw+VZ38/2/93vKZx/Kpfh8v/+vsetVE9OLsVN80N9yPlH
9kb9zfpL/vkP8Zz8TfHq7U9kx51Pcr3xOz2Kq13v+XMo5TW9XrumcVG9pPiP
4nCtPtOLu8PjWvEQjr1+9vI8xXEruV6/r43PqF+9eV+bp7X2e/WK3G8p6tOy

     "], {{0, 0}, {41, 41}}, {0, 0.333333333333333}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->Tiny], ",", 
   GraphicsBox[RasterBox[CompressedData["
1:eJztmrFt3FAQRA+qwqkBJa7CsVJ3YAGOHbsBF+ACHF8JihU5UqAm1IYvMGHc
8A/f7JKEDYnCCR8kP/fvzs7sfvLu/f3XT19uTqfThz//P1++XT5PH3/c3l0+
z3ac5umYzif7tM5W9raKT+9L7Ti73ftSv6rX18a51X3vvn++fH7Fdgi/yV4a
r66vo/PPjXvFk/JK5zs8pvNdf/fmGeFBcVF+u/xI8Xf+LMf5GPNxmk92U3+6
8VMex/l4mOXRjcSfFN901HVd3Kn/U7zEx6q+1P7EB9KB8keP9XzKR7JDdsf5
fpzxpVpPUhydf2vxWMvDLv5ZvhnHbp9x/Hd+O90RHtQfHB/c+uN15/xz8ad1
lXDUvF1fP8/0Tn6nOq/Nm+d1rN+z7XNrcUzzP+bNvD6d5O8aRx3P5nx9vq7r
/Et5k+qZ6iPpRfPdHcc6U55on+uOaTz1favLS1o3XP6pX7j5y/5U949/9bFV
vMQ7ijfFcYyT6pD2ob5eKk+Vr7o+4Uj5cXilOLr8XJ/3caS4Of1276c8un5A
OHRxdPwY12XGUeNydUxxcPXU1dFlHH29qOq3x8sujvTcNT9eN9/tJxjHffTM
+4NrP3UfSc8NHneqY8t8dPvNM/KB6mVNv9yvHC5Uz5x/7nqmN+pH8/pdi4/f
ryiuDmc3jv1l3et8x+Nlfnt7qX80pvh0cdQ8uXy7/uHiSXngdOjWH+et/775
4OM+fDzq47r6SH0n7aOuH1If1bhonTFu/75f00h5J92Or7/+/aPDb8yj43mm
x8Pj+Xotjmm9fKvve5x+035FODp9VOtuWseoHu31/lFxJtyXefN234fTsdon
Pbj8v/bvZ6r6Xvbj+L6wWyed36QXml/Vh85361Neq3Ux1bfadXmjfKa4Vcfu
emkcaX2s4jjlz9Xr1G/SZxUPZ590OtZ7//c9ma6P35u59am/u3pFda6Kl/In
w+v/+f1jt8+4/FZ5T/whf7vrkH6qfSbFP9WXs5fikvIxxX2veFJ9pnzZm2fV
613err2vq/s03m7cKU+q9raKT+/7DeT9MC0=
     "], {{0, 0}, {41, 41}}, {0, 0.285714285714286}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->Tiny], ",", 
   GraphicsBox[RasterBox[CompressedData["
1:eJztmr1twzAQhYVMkTZAmkyR2m02SIDU2SIDZAqN4DqVx4qLqBDFd+8dRYui
fIaNAwySuvv0Hkn9PH18vX0+DMPw8v/7eT5dv7+vj9/v1+8FRtRu+r/3qNbL
2nnHP3pU67a5noNjJT5T+7yez3cSl3Uz3mv9f7S4dp7U2o0wDp18WB2l/JCv
U35qv71HTX9pvdzXKkfmbzXvbf2p5zPnonNkx0fnpzTfUt3U0ZuXJ+eonkfG
0a53LD7+2vzVfGzu9TnO52f/viHvm3p+T4+bP56+b5vqbcNxJPmq+1rmN9xe
43ZJ8m3PEenB1gfy1/rrsLwfl/nY+W/HUevPdcd8jriqXNg4aNytOWq+8q5P
+nrN6pzrX59X9sFR3a8s568yPeLxSvNtzVH1laY7vs7b42Dd9cIR1W/XjfyP
9Ljc3yGdqnXujaOtK/958HJgOu6Fo69u/3qM+tfJp92+x6sre/7Tozq+mm/o
sW89xvy4jR7TOtI4bxfrtZZf7B/jeqYFx7i+9nK0+8f9HsZR5ZLnGfcfkb/y
/on74WUc4/mMdt7jeeGtOObzvb/n16kPGEfUvjTfW8cyfv73KbwcVf57j0xP
Xo6IK+I812W8b1barpU/28wH/vujtj/jfVzm6zx3/3XG0WJtPsFR44P6sXny
aFzVelG7P9G1obU=
     "], {{0, 0}, {41, 41}}, {0, 0.19047619047619}],
    Frame->Automatic,
    FrameTicks->{None, None},
    ImageSize->Tiny]}], "}"}]], "Output",
 ImageSize->{452, 102},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2764]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->3215],

Cell["Construct Pascal\[CloseCurlyQuote]s triangle:", "ExampleText",
 CellID->32425],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"Total", "[", "#", "]"}], "&"}], ",", 
      RowBox[{"{", "}"}], ",", 
      RowBox[{"1", "/", "2"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "5"}], "]"}], "//", 
  "Grid"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->3134],

Cell[BoxData[
 TagBox[GridBox[{
    {"1", "0", "0", "0", "0", "0"},
    {"1", "1", "0", "0", "0", "0"},
    {"1", "2", "1", "0", "0", "0"},
    {"1", "3", "3", "1", "0", "0"},
    {"1", "4", "6", "4", "1", "0"},
    {"1", "5", "10", "10", "5", "1"}
   }],
  "Grid"]], "Output",
 ImageSize->{100, 87},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->18752]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{
        RowBox[{"#", "[", 
         RowBox[{"[", "1", "]"}], "]"}], "+", 
        RowBox[{"#", "[", 
         RowBox[{"[", "3", "]"}], "]"}]}], "&"}], ",", 
      RowBox[{"{", "}"}], ",", "1"}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "5"}], "]"}], "//", 
  "Grid"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->11435],

Cell[BoxData[
 TagBox[GridBox[{
    {"0", "0", "0", "0", "0", "1", "0", "0", "0", "0", "0"},
    {"0", "0", "0", "0", "1", "0", "1", "0", "0", "0", "0"},
    {"0", "0", "0", "1", "0", "2", "0", "1", "0", "0", "0"},
    {"0", "0", "1", "0", "3", "0", "3", "0", "1", "0", "0"},
    {"0", "1", "0", "4", "0", "6", "0", "4", "0", "1", "0"},
    {"1", "0", "5", "0", "10", "0", "10", "0", "5", "0", "1"}
   }],
  "Grid"]], "Output",
 ImageSize->{175, 87},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->4100]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->26208],

Cell["\<\
A cellular automaton based on multiplication of complex integers:\
\>", "ExampleText",
 CellID->9865],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"a_", ",", "b_"}], "}"}], "->", 
       RowBox[{"a", " ", "b"}]}], "}"}], ",", 
     RowBox[{"{", "}"}], ",", 
     RowBox[{"1", "/", "2"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "I", "}"}], ",", "1"}], "}"}], ",", "3"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->3847],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"\[ImaginaryI]", ",", "1", ",", "1", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\[ImaginaryI]", ",", "\[ImaginaryI]", ",", "1", ",", "1"}], 
    "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\[ImaginaryI]", ",", 
     RowBox[{"-", "1"}], ",", "\[ImaginaryI]", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"\[ImaginaryI]", ",", 
     RowBox[{"-", "\[ImaginaryI]"}], ",", 
     RowBox[{"-", "\[ImaginaryI]"}], ",", "\[ImaginaryI]"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{386, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->17041]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"Arg", "[", 
   RowBox[{"CellularAutomaton", "[", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", 
          RowBox[{"a_", ",", "b_"}], "}"}], "->", 
         RowBox[{"a", " ", "b"}]}], "}"}], ",", 
       RowBox[{"{", "}"}], ",", 
       RowBox[{"1", "/", "2"}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", "I", "}"}], ",", "1"}], "}"}], ",", "50"}], "]"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->26901],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJztWztOAzEQXdFxA9ocgIoDQMEJECdAoub8iIaCUKzx7vw9z2YsRU8bZzJ+
4zdjW04ubx8v7zfbtj1cX3f3z69fl8/H7dp+n3/w9onCvd3fZx7q/fsgKv8W
+by2pmnjkjMfdv+x/PXxsI3H7t93XtD4H+VNPLZ6G92y+VPz1R9nfP/+/fHP
2fy59Y2nJz/7/ufH9Wfz59a3/vjpfNPan8crvj+bPzdfqPnn6YGvD973x9ln
86fG1ffjVZ/5+xlZPOPPT9H8ZXrR12epf973x9ln8+fWR0qPUnup3kfbZ/OX
6cVr/y3Xu7TNzp8a156n1/7bvj5Qz7Pz5+qlr8c4fVD6j8kPHP68cXnvv+3r
Q+x85PPn6mW0Plp9jskPHP7UuHLq56j1A5e/TC/j/OfkBw5/3rjG+8+dj3z+
Mr38N8yOv7a+rt6Pitb9++z2qGjdv89uj4rH9bfVH0+PUn1m+0fF7PuHbP+o
eKyvvj759nP4R8VjPufx8Krf2f5RUb//p+o/1az+MeIXNy/n9Tnu919W/2uj
fv/f/zy/3+p/bdTv/8/jPer3W6uivn63+ublh5//tRH1/x/ZcclGvT51+eHn
f23Ux8M2H3b/a2Pdf2Bi3X9gYt1/YGLdf2Bi3X9gYt1/YGLdf2Bi3X9gYp0/
MLHOH5hY5w9MrPMHJtb6gYm1fmBi5Qcm1nyg4Tf7CDxz
   "], {{0, 0}, {51, 51}}, {0, 3.1416}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{154, 150},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{150, Automatic}},
 CellLabel->"Out[2]=",
 CellID->22927]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->9660],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NestList", "[", 
  RowBox[{
   RowBox[{"CellularAutomaton", "[", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{
       RowBox[{"f", "[", "#", "]"}], "&"}], ",", 
      RowBox[{"{", "}"}], ",", 
      RowBox[{"1", "/", "2"}]}], "}"}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "a", "}"}], ",", "b"}], "}"}], ",", "2"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->19282],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztW81u1DAQdpPd7sKy6c+pcOoTIMRDwIlngKiq1B4QVdvHBg7APcTehEbD
JOOfcWynjlRrZ+LYM5/H9njG/VQ/3lx/rR9vr+rLj/f13c3t1cPlh2/3Las8
EqJ4K4R4fyHa3037S/31z7v274k8UOeyLP/VEi/bomjJY03+C0CfAHrVfbcG
/FPm9qb4U7QsaoXEGXizI76k3veIwXZhvVKznm573LQsvqAIFW2xbkmqhcl6
q6GFrpERXI/wqV6Pg/TqD+0SkZCyK5P6hao0xEV06Emr24y+pbSTX28jkcXP
SJ0SNaG2cGRM65eo/nuUS7VVBezb9L3OiHxu4pk7OCqUFtSI+Ox7WauZXCsK
wN+N8CHNMSqc/Se2q0PNuv1WqKf6j0OhuZ+xr2XNgSENfWnoe7OsRA79+UMe
nnIkkqV48iigZNR7yetPMBtEit5Pd22D/5TTdA92FuShdM9FSzlhjvXDrc/Z
wvTRHR/Mu3aZG2oVVdpM1Vrq6V+3P2794Aq8NP10x4/CwVZfxSetOkdszOlJ
qYJjtsmYzWxn245PRXFcaeVNkzM6RwXdZfEha0w4Q48q45yCPW+94YzJfvCK
oaVQX+fINt4Wt2whcYTnuYxjSHusvOCI+Tix+h7Ly6qYyhYSR2o1yDjOaY9y
NFYMyEFa+QFRzP/nlMObw3459bFZCzKe/u3zVYOtFWHWiJxN5pNlLpmrjI8n
+5G5rJIBlXj366n9IbX7BrP4kw7yW/nizxgvV/t67YgPPmupTGTqd1Wotjnk
P09cfh383xjKDflqd+gIn/d/rGaBwSoH31O7Xgr3ioY0XA+45afG44csjLOb
UOqxexE93/X+TSz3a/r2obxc8uv2/9Nq1GR9nVy2bqYdZsCMvPRI8/39DQvX
fD+cr2GwoOjvjY0djWcysRVJP8uKZa9ZvMgkMsB8mXZsxY0FNx3ZD3uSaTwQ
z6BBPfFoGay1N+w7vcwiT4YWyhISF8qif6l2bPZMLOeilweivNn0c1LVCD4U
De95h0SCon83Npajt0tKiXWyBz4sKcZsxk4LtTmsa87clZ2F9bEIdPeajHFT
1pRGfP1kRH9T65hTWz+WAM+f0H+B0QQv+9JE+75jtReGeLHsPg766lmB65mz
31tgHFInilIK/zFOLMKOxbBM/5/TNv47pP+oEWi6xzQqKo7+Av1O514=\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{558, 50},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->9712325]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->19191],

Cell["Algebraically simplify a mod-3 additive rule:", "ExampleText",
 CellID->1397],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"FullSimplify", "[", 
   RowBox[{
    RowBox[{"CellularAutomaton", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{
        RowBox[{
         RowBox[{"Mod", "[", 
          RowBox[{
           RowBox[{
            RowBox[{"#", "[", 
             RowBox[{"[", "1", "]"}], "]"}], "+", 
            RowBox[{"#", "[", 
             RowBox[{"[", "2", "]"}], "]"}]}], ",", "3"}], "]"}], "&"}], ",", 
        
        RowBox[{"{", "}"}], ",", 
        RowBox[{"1", "/", "2"}]}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", "a", "}"}], ",", "0"}], "}"}], ",", "6"}], "]"}], ",", 
    RowBox[{
     RowBox[{"a", "\[Element]", "Integers"}], "&&", 
     RowBox[{"0", "<=", "a", "<=", "2"}]}]}], "]"}], "//", "Grid"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->16539],

Cell[BoxData[
 TagBox[GridBox[{
    {"a", "0", "0", "0", "0", "0", "0"},
    {"a", "a", "0", "0", "0", "0", "0"},
    {"a", 
     RowBox[{"Mod", "[", 
      RowBox[{
       RowBox[{"2", " ", "a"}], ",", "3"}], "]"}], "a", "0", "0", "0", "0"},
    {"a", "0", "0", "a", "0", "0", "0"},
    {"a", "a", "0", "a", "a", "0", "0"},
    {"a", 
     RowBox[{"Mod", "[", 
      RowBox[{
       RowBox[{"2", " ", "a"}], ",", "3"}], "]"}], "a", "a", 
     RowBox[{"Mod", "[", 
      RowBox[{
       RowBox[{"2", " ", "a"}], ",", "3"}], "]"}], "a", "0"},
    {"a", "0", "0", 
     RowBox[{"Mod", "[", 
      RowBox[{
       RowBox[{"2", " ", "a"}], ",", "3"}], "]"}], "0", "0", "a"}
   }],
  "Grid"]], "Output",
 ImageSize->{293, 103},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->9592]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->29025]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->9691],

Cell["Include mesh lines:", "ExampleText",
 CellID->26813],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{
   RowBox[{"CellularAutomaton", "[", 
    RowBox[{"30", ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "10"}], "]"}], ",", 
   RowBox[{"Mesh", "->", "True"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->17496],

Cell[BoxData[
 GraphicsBox[{
   RasterBox[{{0, 0, 1, 1, 0, 1, 1, 1, 1, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 1, 
    0}, {1, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 1, 1, 1, 0, 0, 0, 1}, {1, 
    1, 0, 0, 1, 1, 0, 1, 1, 1, 0, 0, 0, 1, 1, 1, 1, 1, 0, 1, 1}, {1, 1, 1, 0, 
    0, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 1, 1}, {1, 1, 1, 1, 0, 0, 1, 
    1, 0, 1, 1, 1, 1, 0, 1, 1, 0, 1, 1, 1, 1}, {1, 1, 1, 1, 1, 0, 0, 1, 0, 0, 
    0, 0, 1, 0, 0, 0, 1, 1, 1, 1, 1}, {1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 0, 1, 1, 
    1, 0, 1, 1, 1, 1, 1, 1}, {1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 0, 0, 0, 0, 1, 1, 
    1, 1, 1, 1, 1}, {1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 0, 1, 1, 1, 1, 1, 1, 
    1, 1}, {1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1}, {
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1}}, {{0, 
    0}, {21, 11}}, {0, 1}], {{
     {GrayLevel[-1 + GoldenRatio]}, 
     LineBox[{{{0, 11}, {21, 11}}, {{0, 10}, {21, 10}}, {{0, 9}, {21, 9}}, {{
      0, 8}, {21, 8}}, {{0, 7}, {21, 7}}, {{0, 6}, {21, 6}}, {{0, 5}, {21, 
      5}}, {{0, 4}, {21, 4}}, {{0, 3}, {21, 3}}, {{0, 2}, {21, 2}}, {{0, 1}, {
      21, 1}}, {{0, 0}, {21, 0}}}]}, {
     {GrayLevel[-1 + GoldenRatio]}, 
     LineBox[{{{0, 0}, {0, 11}}, {{1, 0}, {1, 11}}, {{2, 0}, {2, 11}}, {{3, 
      0}, {3, 11}}, {{4, 0}, {4, 11}}, {{5, 0}, {5, 11}}, {{6, 0}, {6, 
      11}}, {{7, 0}, {7, 11}}, {{8, 0}, {8, 11}}, {{9, 0}, {9, 11}}, {{10, 
      0}, {10, 11}}, {{11, 0}, {11, 11}}, {{12, 0}, {12, 11}}, {{13, 0}, {13, 
      11}}, {{14, 0}, {14, 11}}, {{15, 0}, {15, 11}}, {{16, 0}, {16, 11}}, {{
      17, 0}, {17, 11}}, {{18, 0}, {18, 11}}, {{19, 0}, {19, 11}}, {{20, 0}, {
      20, 11}}, {{21, 0}, {21, 11}}}]}}},
  Frame->False,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 98},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->13618]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->18000],

Cell["Label cells:", "ExampleText",
 CellID->23522],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{
   RowBox[{"CellularAutomaton", "[", 
    RowBox[{"30", ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "50"}], "]"}], ",", 
   RowBox[{"FrameTicks", "\[Rule]", "True"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->9370],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJy1lVFSwzAMRCVuwhm4CUdghm/u/0eLrd2VsWM3iUPJtImj9dNKyvvXz+fX
m5l9PP6/7Xm4Pz7PU/3+OD//vP76u1dXxMVyo1zFfSsRLBaXBYxZbnpdEI8b
49TdaKi4+vcLUjVK2WeNkjaHBVgXcTxwPQHWy+YiFYtlg7I5wwKsi1wiB1ge
YQ3wErMBTZsb+IIoVHGgCTyhkQpNAFm6voSd4gDRAE9opCInYOKLR0KQO0FD
4VnwMxU5ARNfkH4XeT5pjImNwtJWZdovJvLMHsPDisi8t8dKv6h8fCS8bJQ3
+ixH/aLykT0JHxXiodRVmfQL86XZU9vdRXakMukXXmb2JDxaZsQx9cU1X6bM
6I3I85hj6oucPDNHAbLYDjTm/ZKnFtqSbdjtjxHLqF8wPGBxJeMUnnAoy6Bf
MDyMsODwyPSKykG/eAyP+IKxiOcXNGb94t58Mczlw/4YsfR8MUkSTAq2JT9a
lo4voiylxIJb1jjyRZWl/Vhwr6n0fUnKrLD++2OVpfUlKTsrbK0/RiytLwzK
CgPfGZWOLzJNWGpl/QmNkS8yTZDFV/pjxKK+mEri9XWWQ1nUF5kmGO/nOZRF
fUHHsNMvcCiLjlyO/MC4ptHzhSO/il/kUBadJhr8OgdCuvqSg9+h8d+X6MfX
3h8LKsmX6JNr/dFRSb6A6zaOqqK+ROZu5Kgq6ks4c6tG68vZ98eCSop+P0dV
SdF3aCRfNvghKuHLJo6qgjG/79Axv1Vlox+ispmjqOzXOMXxC/ECEP0=
   "], {{0, 0}, {101, 51}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{{{19.5, 
      FormBox["20", StandardForm]}, {39.5, 
      FormBox["40", StandardForm]}, {59.5, 
      FormBox["60", StandardForm]}, {79.5, 
      FormBox["80", StandardForm]}, {99.5, 
      FormBox["100", StandardForm]}}, {{41.5, 
      FormBox["10", StandardForm]}, {31.5, 
      FormBox["20", StandardForm]}, {21.5, 
      FormBox["30", StandardForm]}, {11.5, 
      FormBox["40", StandardForm]}, {1.5, 
      FormBox["50", StandardForm]}}}]], "Output",
 ImageSize->{184, 109},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->16791]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->2784],

Cell["Highlight the center column of cells:", "ExampleText",
 CellID->32056],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"MapAt", "[", 
   RowBox[{
    RowBox[{
     RowBox[{"Blend", "[", 
      RowBox[{
       RowBox[{"{", 
        RowBox[{"#", ",", "Red"}], "}"}], ",", ".5"}], "]"}], "&"}], ",", 
    RowBox[{
     RowBox[{"CellularAutomaton", "[", 
      RowBox[{"30", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "10"}], "]"}], "/.",
      
     RowBox[{"{", 
      RowBox[{
       RowBox[{"0", "\[Rule]", "White"}], ",", 
       RowBox[{"1", "\[Rule]", "Blue"}]}], "}"}]}], ",", 
    RowBox[{"Table", "[", 
     RowBox[{
      RowBox[{"{", 
       RowBox[{"t", ",", "11"}], "}"}], ",", 
      RowBox[{"{", 
       RowBox[{"t", ",", "11"}], "}"}]}], "]"}]}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->28359],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJztlsEJgDAMRQsePTuAk7iDIwieXbkjOIJ6DY301x9bJYHyJMhPmpjacdnm
tQsh9OcaznU9p22f7v0oWTrsuFFQe5/lR/NEdVj1Z+WD6ljHlX2OL9WTpY/6
a+XTWj21vmtxrfNprT61vgdUpzSuPOdr5YP6v0LrfZXq5Pbdeu5YOl+htl/N
z9JBz3nrc8AZksauG+te5/P7DqWV1pl1r0PjOrmUltuXp/c6Z5uUVjrvzn/Q
+v/ePg9r5JBE
   "], {{0, 0}, {21, 11}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{184, 98},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->26311]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->25972],

Cell[TextData[{
 Cell[BoxData[
  FormBox["t", TraditionalForm]], "InlineMath"],
 " steps of cellular automaton evolution gives a list of length ",
 Cell[BoxData[
  FormBox[
   RowBox[{"t", "+", "1"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->3922],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"30", ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "1", ",", "1"}], "}"}], ",", "3"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->21045],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "1", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "1", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0", ",", "0"}], "}"}]}], "}"}]], "Output",
 ImageSize->{354, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->29064]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->8009],

Cell["\<\
Initial conditions given as explicit lists are assumed cyclic:\
\>", "ExampleText",
 CellID->26571],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{"170", ",", 
    RowBox[{"{", 
     RowBox[{
     "0", ",", "0", ",", "0", ",", "1", ",", "0", ",", "0", ",", "0", ",", 
      "0", ",", "0"}], "}"}], ",", "20"}], "]"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->31267],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzzTSzJSM1NLMlMTlRwL0osyMhMLlZwyy8CCjEzMjAwaABxlAQDkP0fyAJj
jf///0MkgIAZJAIGDExAggXIlQPSjHSUZYPKSg0qWS6orBidZfmABCuQK0hn
WWGoq3jpLCsOleWks6w0VJaVzrLyUFlGOsiCYluJDN5gKBUGZ5kxWiqMlgpD
vVQYvmXGSCgVoO02RgB1ludH\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{40, 90},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{40., Automatic}},
 CellLabel->"Out[1]=",
 CellID->428610088]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->5793],

Cell["\<\
The center cell starting from all possible tuples gives the digits in the \
rule number:\
\>", "ExampleText",
 CellID->1580],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{
    RowBox[{"CellularAutomaton", "[", 
     RowBox[{"30", ",", "#"}], "]"}], "[", 
    RowBox[{"[", "2", "]"}], "]"}], "&"}], "/@", 
  RowBox[{"Tuples", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"1", ",", "0"}], "}"}], ",", "3"}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->22789],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0", ",", "0", ",", "0", ",", "1", ",", "1", ",", "1", ",", "1", ",", "0"}],
   "}"}]], "Output",
 ImageSize->{153, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->31613]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"IntegerDigits", "[", 
  RowBox[{"30", ",", "2", ",", "8"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->1209],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
  "0", ",", "0", ",", "0", ",", "1", ",", "1", ",", "1", ",", "1", ",", "0"}],
   "}"}]], "Output",
 ImageSize->{153, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->4623]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(6)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->4143],

Cell["\<\
By default, the lists in each evolution are made only as long as they need to \
be:\
\>", "ExampleText",
 CellID->24570],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"90", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "2"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1157],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0", ",", "0", ",", "1"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{322, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->26955]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"170", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "2"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->22837],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0"}], "}"}]}], "}"}]], "Output",
 ImageSize->{213, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->11891]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"0", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "2"}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->182],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", "1", "}"}], ",", 
   RowBox[{"{", "0", "}"}], ",", 
   RowBox[{"{", "0", "}"}]}], "}"}]], "Output",
 ImageSize->{104, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->4390]
}, Open  ]],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    StyleBox["t", "TI"], ",", " ", 
    ButtonBox["All",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/All"]}], "}"}]], "InlineFormula"],
 " to get lists of the same length for all rules of a particular type:"
}], "ExampleText",
 CellID->4767],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"170", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"2", ",", "All"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->23687],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0", ",", "0", ",", "0"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{322, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->8135]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->3915],

Cell["\<\
If no cells ever differ from the background, the automatically selected \
region will always be empty:\
\>", "ExampleText",
 CellID->98],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"30", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "0", "}"}], ",", "0"}], "}"}], ",", "20"}], "]"}]], "Input",\

 CellLabel->"In[1]:=",
 CellID->27619],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWdsKgkAQHTWiC2VFSI9+TT31EYsE+hCF+f+2mxWxSrpZNKtnYYednSOe
48xewL3I4sNRZEkkwl0qznESXcLtKZVTnkPkronI35Ac53J0648WyE40UNZ7
zpd8tyY+6Rl+yowP9EJvG7ynTGHfoKqecqWrv81WnJ51bvyggxe/OlzlKWq8
ynR/1DP8mBkf6IXeNvjPdwG15zT5Wjbgmuyd0AEdJrjyKqu79w41f8Y8Pge/
TvPT69NRxvis8JnHF+DXaX7fufH8W4XtWQA/VLH9WQA/VLH9WQC/X1RxI9SL
v2QeX4Ffp/np9Vn8dcvvLTD0yLkCPXG/BA==\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{532, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->24982057]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->9974],

Cell["\<\
Rules with alternating backgrounds can give lists which contain only 0s:\
\>", "ExampleText",
 CellID->27296],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"105", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "1"}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->3764],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "1", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0"}], "}"}]}], "}"}]], "Output",
 ImageSize->{144, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->18223]
}, Open  ]],

Cell["\<\
If the evolution is continued for 2 steps, explicit 1s are included in the \
1-step result: \
\>", "ExampleText",
 CellID->6607],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"105", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "2"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->12074],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "1", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "0", ",", "0", ",", "1"}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{"1", ",", "0", ",", "1", ",", "0", ",", "1"}], "}"}]}], 
  "}"}]], "Output",
 ImageSize->{322, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->790]
}, Open  ]],

Cell["In this form, a specification of the background is included:", \
"ExampleText",
 CellID->22599],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CellularAutomaton", "[", 
  RowBox[{"105", ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", "0"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->32518],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{"0", ",", "0", ",", "0"}], "}"}], ",", 
   RowBox[{"{", "1", "}"}]}], "}"}]], "Output",
 ImageSize->{108, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->3140]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->17103],

Cell["\<\
By default, each evolution is made only as wide as it needs to be:\
\>", "ExampleText",
 CellID->7245],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"ArrayPlot", "[", 
    RowBox[{
     RowBox[{"CellularAutomaton", "[", 
      RowBox[{"n", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "20"}], "]"}], ",", 
     
     RowBox[{"ImageSize", "->", 
      RowBox[{"{", 
       RowBox[{"50", ",", "50"}], "}"}]}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "20", ",", "30"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->4257],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{370, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->447853197]
}, Open  ]],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  RowBox[{"{", 
   RowBox[{
    StyleBox["t", "TI"], ",", " ", 
    ButtonBox["All",
     BaseStyle->"Link",
     ButtonData->"paclet:ref/All"]}], "}"}]], "InlineFormula"],
 " to get consistent widths:"
}], "ExampleText",
 CellID->23357],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Table", "[", 
  RowBox[{
   RowBox[{"ArrayPlot", "[", 
    RowBox[{
     RowBox[{"CellularAutomaton", "[", 
      RowBox[{"n", ",", 
       RowBox[{"{", 
        RowBox[{
         RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", 
       RowBox[{"{", 
        RowBox[{"20", ",", "All"}], "}"}]}], "]"}], ",", 
     RowBox[{"ImageSize", "->", "50"}]}], "]"}], ",", 
   RowBox[{"{", 
    RowBox[{"n", ",", "20", ",", "30"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->18037],

Cell[GraphicsData["CompressedBitmap", "\<\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\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{370, 59},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->34612672]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->7373],

Cell[TextData[{
 "Cells and steps in ",
 Cell[BoxData[
  ButtonBox["CellularAutomaton",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CellularAutomaton"]], "InlineFormula"],
 " are not numbered according to their part numbers:"
}], "ExampleText",
 CellID->397],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{
   RowBox[{"CellularAutomaton", "[", 
    RowBox[{"30", ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "4"}], "]"}], ",", 
   RowBox[{"FrameTicks", "->", "True"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->7048],

Cell[GraphicsData["CompressedBitmap", "\<\
eJztWF1uwjAMDi0d41eDvewBbTyjHQDtANvTDlGhSTBp2sQ4ws7AdTu7jWnr
OG2KQCoFJOo4tb9+Tu0kzXu4XX18hdv1Mpy9bcKf1Xr5O3v93kCX31JK7eD/
+aCgHUEr/u+iCKTtegfXAMy6INsg+yA9kEOQPsie1vtax36FjSj+GffvY9QJ
s+oyHa1VO9VN3FvDomf0DKSnFFqMRXY9pku4gYBmcixnROM0EZn0S5iYT6T3
RbgjgQHdz3vKDEzG5rhnWSqmS1GYCC49Cb+x5j9gcsjiJok42Twe5MbbVgUY
9Z+uAmVka6ckPjHeCvZVdc6n0yDWU2hOoTECOdfSu2hP7MVMDg5AIF8/Rkog
MiWm2q6Ywd5Ge2IvLjgtlHEfx0KLJ4Y3BnljeB4vQn7XOT6GaOjx1a8yYgzB
p/4zyz7uiesLeqNcqGT/cDlevn6TfkVv8rNXShU09M9XEBU4/ogf7QaySKMM
/2DPp3xeEBk5zAnZyJxxLSNDT0jYewegkZ9ch/XLNqNqtRdV7uNFeVlXiRLv
ymughOeQ63xOoBrkeFPdz+NwmxvKohXmhpNFfJw3Uv/M4164Ys5BfW6UdXYN
kWbXnHf6pRYl31teDgOvN0X+RnYlSSrt31Al33kk78LSdYFXoEv83M61booj
cq8XG6/iOqlPXi2ipHpeGmUt7ams3rl6SHdWZRlpYyBnpG2FIZRFVLSynJ7L
4Wi2yrChFFdGfTLsPM9gzod1emZXfMp9Peu2vavrWXfxuDfprFu1/gEN5z83
\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{150, 106},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{150, Automatic}},
 CellLabel->"Out[1]=",
 CellID->459162640]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->360],

Cell["\<\
Rule numbers have different meanings when the offsets are specified in \
different orders:\
\>", "ExampleText",
 CellID->14126],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"30", ",", "2", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"-", "1"}], "}"}], ",", 
        RowBox[{"{", "0", "}"}], ",", 
        RowBox[{"{", "1", "}"}]}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "40"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->21640],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJytlNF5wyAMhHXdpI8ZJx0hC2T/tyZBJx0YbBzjL3EJhdPPSej38fx7/JjZ
7fW92/sBXh+Uwedd/nDg8+9fMP+Y/As+9VkSi8zny14r87E+pksQToEBwHXk
KYxgbHCz0Pg08iwpRTrqBGNzuIomrMizOADP5cEBYez6KC/UZ3HuMADUS8au
j/EKv7m58JnsV8Wxj+G62s+Uwqr9G8auj4TLAYmZKNR8hz6yGHJA4pBp9Q58
BNAM3BLjs9Hb99HkoGEqmTf+TfgoUSSlmfiu3p6PGkVKORM/Vuz7WEXJTKvF
+4ytj1UUZKa39TfrYwpkpoP7SLHjo9y4THlZf6A38lFuXDgxqr8Ro/poKh+t
doZPGdVHuXHRsub4lFF9jIrMGzLJp4zaWrKNEW9er+djtjEPdIJPGfXGqdA5
PmXUG6dCZ/W2PrK2x/1vlpE+sg7n62/ESB+D9ys+ZczEHvW/WcboMqfrb8QI
Pf0FvcZHfFN/I8a40Rf1Kh8v+ieK9HEBnytG61rzaOtaprjIP1FcyFcU1+qd
4PsHT8oLhg==
   "], {{0, 0}, {81, 41}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{154, 79},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{150, Automatic}},
 CellLabel->"Out[1]=",
 CellID->30119]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"ArrayPlot", "[", 
  RowBox[{"CellularAutomaton", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"30", ",", "2", ",", 
      RowBox[{"{", 
       RowBox[{
        RowBox[{"{", "0", "}"}], ",", 
        RowBox[{"{", "1", "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"-", "1"}], "}"}]}], "}"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{
      RowBox[{"{", "1", "}"}], ",", "0"}], "}"}], ",", "40"}], "]"}], 
  "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->18228],

Cell[BoxData[
 GraphicsBox[RasterBox[CompressedData["
1:eJytzrsNgDAMRdGYTSgZB0bIAuzfkbhBiHxs517JnfV09nxfeUspHeXOcklE
2Kub5NG+GusT3Eca31gfZfzG+gjjP9a3amzH+laM/Vhf1DiO9UWM81if12iL
9XmM9lif1eiL9VmM/ljfzBiL9Y2M8Vhfz7gW62sZ12N9uoj6dFFIny6iPl0U
0qeLqE8XrY8PW1cLvw==
   "], {{0, 0}, {81, 41}}, {0, 1}],
  Frame->Automatic,
  FrameTicks->{None, None}]], "Output",
 ImageSize->{154, 79},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 GraphicsBoxOptions->{ImageSize->{150, Automatic}},
 CellLabel->"Out[2]=",
 CellID->4741]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->17849],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["ListConvolve",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ListConvolve"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["TuringMachine",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/TuringMachine"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Partition",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Partition"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BitXor",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BitXor"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["BitShiftLeft",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/BitShiftLeft"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["ArrayPlot",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/ArrayPlot"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->25981]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["TUTORIALS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "TutorialsSection",
 CellID->4335],

Cell[TextData[ButtonBox["Cellular Automata",
 BaseStyle->"Link",
 ButtonData->"paclet:tutorial/CellularAutomata"]], "Tutorials",
 CellID->31587]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["RELATED LINKS",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "RelatedLinksSection",
 CellID->23714],

Cell[TextData[{
 ButtonBox["Demonstrations with CellularAutomaton",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/symbol.html?symbol=\
CellularAutomaton"], None}],
 " (",
 ButtonBox["Wolfram Demonstrations Project",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://demonstrations.wolfram.com/"], None}],
 ")"
}], "RelatedLinks"],

Cell[TextData[{
 ButtonBox["NKS|Online (",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/section-2.1"], None}],
 StyleBox[ButtonBox["A New Kind of Science",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/section-2.1"], None}],
  FontSlant->"Italic"],
 ButtonBox[")",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/section-2.1"], None}]
}], "RelatedLinks",
 CellID->28900],

Cell[TextData[ButtonBox["Implementation notes: Numerical and Related \
Functions",
 BaseStyle->"Hyperlink",
 ButtonData->{
   URL["http://reference.wolfram.com/mathematica/note/\
SomeNotesOnInternalImplementation.html#12526"], None}]], "RelatedLinks",
 CellID->6146],

Cell[TextData[{
 ButtonBox["NKS|Online",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/index/search.cgi?SearchIndex=\
CellularAutomaton"], None}],
 " (",
 StyleBox[ButtonBox["A New Kind of Science",
  BaseStyle->"Hyperlink",
  ButtonData->{
    URL["http://www.wolframscience.com/nksonline/"], None}],
  FontSlant->"Italic"],
 ")"
}], "RelatedLinks"]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->234708990],

Cell[TextData[ButtonBox["Bitwise Operations",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/BitwiseOperations"]], "MoreAbout",
 CellID->106726794],

Cell[TextData[ButtonBox["Computational Systems",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/ComputationalSystemsAndDiscovery"]], "MoreAbout",
 CellID->191401289],

Cell[TextData[ButtonBox["Cryptographic Number Theory",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/CryptographicNumberTheory"]], "MoreAbout",
 CellID->135827946],

Cell[TextData[ButtonBox["Discrete Mathematics",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/DiscreteMathematics"]], "MoreAbout",
 CellID->617717607],

Cell[TextData[ButtonBox["Image Processing",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/ImageProcessing"]], "MoreAbout",
 CellID->468196979],

Cell[TextData[ButtonBox["Precollege Education",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/PrecollegeEducation"]], "MoreAbout",
 CellID->36836740],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->267146821]
}, Open  ]],

Cell["New in 4.2  |  Last modified in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"CellularAutomaton - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 27, 50.0857750}", "context" -> "System`", 
    "keywords" -> {
     "A New Kind of Science", "additive cellular automata", "array automata", 
      "automata", "automorphisms of the shift", "block maps", "CA", 
      "cell automaton", "cell-automaton", "computational universe", 
      "continuous mappings of the shift", "coupled map lattices", 
      "discrete models", "distributed computation", 
      "dynamical systems theory", "ECA", "elementary cellular automata", 
      "iterated maps on the Cantor set", "iterative arrays", "line automata", 
      "models", "NKS", "parallel computation", "polyautomata", 
      "random programs", "rule numbers", "rule 110", "rule 30", "rule 90", 
      "shift map", "simple programs", "tesselation automata", 
      "tessellation automata", "totalistic cellular automata", "universality",
       "Wolfram automata", "Wolfram Science"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "CellularAutomaton[rule, init, t] generates a list representing the \
evolution of the cellular automaton with the specified rule from initial \
condition init for t steps. CellularAutomaton[rule, init] gives the result of \
evolving init for one step. CellularAutomaton[rule, init, {tspec, xspec, \
...}] gives only those parts of the evolution specified by tspec, xspec, etc. \
CellularAutomaton[rule, init, {t, All, ...}] includes at each step all cells \
that could be affected over the course of t steps. ", 
    "synonyms" -> {"cellular automaton"}, "title" -> "CellularAutomaton", 
    "type" -> "Symbol", "uri" -> "ref/CellularAutomaton"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[46966, 1475, 359, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->12012548]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 601390, 14550}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 3266, 73, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[3889, 100, 4201, 83, 70, "ObjectNameGrid"],
Cell[8093, 185, 2759, 83, 70, "Usage",
 CellID->24448]
}, Open  ]],
Cell[CellGroupData[{
Cell[10889, 273, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[11367, 290, 137, 6, 70, "Notes",
 CellID->9303],
Cell[11507, 298, 6330, 205, 70, "2ColumnTableMod",
 CellID->24715],
Cell[17840, 505, 925, 33, 70, "Notes",
 CellID->22959],
Cell[18768, 540, 80, 1, 70, "Notes",
 CellID->15156],
Cell[18851, 543, 2285, 66, 70, "2ColumnTableMod",
 CellID->27972],
Cell[21139, 611, 104, 3, 70, "Notes",
 CellID->23274],
Cell[21246, 616, 2460, 66, 70, "2ColumnTableMod",
 CellID->12134],
Cell[23709, 684, 284, 10, 70, "Notes",
 CellID->7892],
Cell[23996, 696, 336, 10, 70, "Notes",
 CellID->31317],
Cell[24335, 708, 429, 15, 70, "Notes",
 CellID->22815],
Cell[24767, 725, 428, 9, 70, "Notes",
 CellID->23759],
Cell[25198, 736, 974, 32, 70, "Notes",
 CellID->21964],
Cell[26175, 770, 361, 8, 70, "Notes",
 CellID->29209],
Cell[26539, 780, 262, 7, 70, "Notes",
 CellID->3347],
Cell[26804, 789, 601, 20, 70, "Notes",
 CellID->7030],
Cell[27408, 811, 165, 6, 70, "Notes",
 CellID->1413],
Cell[27576, 819, 5029, 165, 70, "2ColumnTableMod",
 CellID->11813],
Cell[32608, 986, 670, 21, 70, "Notes",
 CellID->19603],
Cell[33281, 1009, 450, 16, 70, "Notes",
 CellID->591],
Cell[33734, 1027, 477, 15, 70, "Notes",
 CellID->24629],
Cell[34214, 1044, 469, 16, 70, "Notes",
 CellID->29344],
Cell[34686, 1062, 3048, 95, 70, "2ColumnTableMod",
 CellID->31416],
Cell[37737, 1159, 85, 1, 70, "Notes",
 CellID->6485],
Cell[37825, 1162, 696, 24, 70, "Notes",
 CellID->31227],
Cell[38524, 1188, 285, 9, 70, "Notes",
 CellID->16133],
Cell[38812, 1199, 4594, 136, 70, "2ColumnTableMod",
 CellID->9639],
Cell[43409, 1337, 198, 6, 70, "Notes",
 CellID->20594],
Cell[43610, 1345, 458, 15, 70, "Notes",
 CellID->29739],
Cell[44071, 1362, 262, 8, 70, "Notes",
 CellID->7204],
Cell[44336, 1372, 581, 19, 70, "Notes",
 CellID->11387],
Cell[44920, 1393, 504, 19, 70, "Notes",
 CellID->23967],
Cell[45427, 1414, 247, 8, 70, "Notes",
 CellID->29415],
Cell[45677, 1424, 277, 9, 70, "Notes",
 CellID->2922],
Cell[45957, 1435, 368, 13, 70, "Notes",
 CellID->384],
Cell[46328, 1450, 601, 20, 70, "Notes",
 CellID->17996]
}, Closed]],
Cell[CellGroupData[{
Cell[46966, 1475, 359, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->12012548],
Cell[CellGroupData[{
Cell[47350, 1489, 148, 5, 70, "ExampleSection",
 CellID->524365467],
Cell[47501, 1496, 63, 1, 70, "ExampleText",
 CellID->30991],
Cell[CellGroupData[{
Cell[47589, 1501, 245, 7, 28, "Input",
 CellID->406],
Cell[47837, 1510, 522, 16, 36, "Output",
 CellID->21442]
}, Open  ]],
Cell[48374, 1529, 120, 3, 70, "ExampleDelimiter",
 CellID->1323],
Cell[48497, 1534, 94, 1, 70, "ExampleText",
 CellID->30617],
Cell[CellGroupData[{
Cell[48616, 1539, 262, 9, 70, "Input",
 CellID->20737],
Cell[48881, 1550, 1067, 22, 116, "Output",
 CellID->9375]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[49997, 1578, 216, 7, 70, "ExampleSection",
 CellID->9806],
Cell[CellGroupData[{
Cell[50238, 1589, 239, 7, 70, "ExampleSubsection",
 CellID->26724],
Cell[50480, 1598, 58, 1, 70, "ExampleText",
 CellID->10293],
Cell[CellGroupData[{
Cell[50563, 1603, 252, 9, 70, "Input",
 CellID->28596],
Cell[50818, 1614, 441, 13, 94, "Output",
 CellID->30927]
}, Open  ]],
Cell[CellGroupData[{
Cell[51296, 1632, 261, 9, 70, "Input",
 CellID->8548],
Cell[51560, 1643, 1036, 22, 116, "Output",
 CellID->29670]
}, Open  ]],
Cell[52611, 1668, 120, 3, 70, "ExampleDelimiter",
 CellID->4046],
Cell[52734, 1673, 59, 1, 70, "ExampleText",
 CellID->22842],
Cell[CellGroupData[{
Cell[52818, 1678, 338, 11, 70, "Input",
 CellID->24383],
Cell[53159, 1691, 1093, 23, 116, "Output",
 CellID->14663]
}, Open  ]],
Cell[CellGroupData[{
Cell[54289, 1719, 516, 17, 70, "Input",
 CellID->28182],
Cell[54808, 1738, 2131, 39, 70, "Output",
 Evaluatable->False,
 CellID->289680340]
}, Open  ]],
Cell[56954, 1780, 119, 3, 70, "ExampleDelimiter",
 CellID->770],
Cell[57076, 1785, 77, 1, 70, "ExampleText",
 CellID->14721],
Cell[CellGroupData[{
Cell[57178, 1790, 390, 13, 70, "Input",
 CellID->23071],
Cell[57571, 1805, 1450, 29, 116, "Output",
 CellID->6497]
}, Open  ]],
Cell[59036, 1837, 121, 3, 70, "ExampleDelimiter",
 CellID->28668],
Cell[59160, 1842, 246, 8, 70, "ExampleText",
 CellID->2040],
Cell[CellGroupData[{
Cell[59431, 1854, 363, 12, 70, "Input",
 CellID->8657],
Cell[59797, 1868, 808, 18, 87, "Output",
 CellID->20301]
}, Open  ]],
Cell[60620, 1889, 121, 3, 70, "ExampleDelimiter",
 CellID->13426],
Cell[60744, 1894, 169, 7, 70, "ExampleText",
 CellID->24618],
Cell[CellGroupData[{
Cell[60938, 1905, 339, 11, 70, "Input",
 CellID->8124],
Cell[61280, 1918, 1195, 25, 73, "Output",
 CellID->23622]
}, Open  ]],
Cell[62490, 1946, 121, 3, 70, "ExampleDelimiter",
 CellID->10900],
Cell[62614, 1951, 171, 6, 70, "ExampleText",
 CellID->4825],
Cell[CellGroupData[{
Cell[62810, 1961, 387, 13, 70, "Input",
 CellID->941],
Cell[63200, 1976, 1047, 22, 73, "Output",
 CellID->2744]
}, Open  ]],
Cell[64262, 2001, 120, 3, 70, "ExampleDelimiter",
 CellID->9648],
Cell[64385, 2006, 122, 3, 70, "ExampleText",
 CellID->20076],
Cell[CellGroupData[{
Cell[64532, 2013, 499, 17, 70, "Input",
 CellID->2431],
Cell[65034, 2032, 1067, 22, 116, "Output",
 CellID->4574]
}, Open  ]],
Cell[66116, 2057, 121, 3, 70, "ExampleDelimiter",
 CellID->32678],
Cell[66240, 2062, 82, 1, 70, "ExampleText",
 CellID->18067],
Cell[CellGroupData[{
Cell[66347, 2067, 499, 17, 70, "Input",
 CellID->4660],
Cell[66849, 2086, 966, 21, 87, "Output",
 CellID->6790]
}, Open  ]],
Cell[67830, 2110, 120, 3, 70, "ExampleDelimiter",
 CellID->6812],
Cell[67953, 2115, 63, 1, 70, "ExampleText",
 CellID->32554],
Cell[CellGroupData[{
Cell[68041, 2120, 460, 16, 70, "Input",
 CellID->21317],
Cell[68504, 2138, 692, 16, 87, "Output",
 CellID->277]
}, Open  ]],
Cell[69211, 2157, 121, 3, 70, "ExampleDelimiter",
 CellID->10994],
Cell[69335, 2162, 122, 3, 70, "ExampleText",
 CellID->15968],
Cell[CellGroupData[{
Cell[69482, 2169, 436, 15, 70, "Input",
 CellID->4149],
Cell[69921, 2186, 1068, 22, 116, "Output",
 CellID->30922]
}, Open  ]],
Cell[71004, 2211, 121, 3, 70, "ExampleDelimiter",
 CellID->23928],
Cell[71128, 2216, 65, 1, 70, "ExampleText",
 CellID->27160],
Cell[CellGroupData[{
Cell[71218, 2221, 390, 13, 70, "Input",
 CellID->4716],
Cell[71611, 2236, 1247, 25, 116, "Output",
 CellID->29767]
}, Open  ]],
Cell[72873, 2264, 114, 3, 70, "ExampleText",
 CellID->11743],
Cell[CellGroupData[{
Cell[73012, 2271, 460, 15, 70, "Input",
 CellID->21647],
Cell[73475, 2288, 1247, 25, 116, "Output",
 CellID->21315]
}, Open  ]],
Cell[74737, 2316, 121, 3, 70, "ExampleDelimiter",
 CellID->30541],
Cell[74861, 2321, 69, 1, 70, "ExampleText",
 CellID->366],
Cell[CellGroupData[{
Cell[74955, 2326, 440, 15, 70, "Input",
 CellID->15423],
Cell[75398, 2343, 1168, 24, 118, "Output",
 CellID->3164]
}, Open  ]],
Cell[76581, 2370, 121, 3, 70, "ExampleDelimiter",
 CellID->13767],
Cell[76705, 2375, 113, 3, 70, "ExampleText",
 CellID->23030],
Cell[CellGroupData[{
Cell[76843, 2382, 711, 23, 70, "Input",
 CellID->8756],
Cell[77557, 2407, 673, 16, 116, "Output",
 CellID->9226]
}, Open  ]],
Cell[78245, 2426, 121, 3, 70, "ExampleDelimiter",
 CellID->13576],
Cell[78369, 2431, 113, 3, 70, "ExampleText",
 CellID->24702],
Cell[CellGroupData[{
Cell[78507, 2438, 470, 16, 70, "Input",
 CellID->7419],
Cell[78980, 2456, 674, 16, 116, "Output",
 CellID->23643]
}, Open  ]],
Cell[79669, 2475, 121, 3, 70, "ExampleDelimiter",
 CellID->27493],
Cell[79793, 2480, 358, 13, 70, "ExampleText",
 CellID->5046],
Cell[CellGroupData[{
Cell[80176, 2497, 465, 16, 70, "Input",
 CellID->9123],
Cell[80644, 2515, 515, 16, 54, "Output",
 CellID->32177]
}, Open  ]],
Cell[CellGroupData[{
Cell[81196, 2536, 132, 4, 70, "Input",
 CellID->9398],
Cell[81331, 2542, 362, 11, 83, "Output",
 CellID->22532]
}, Open  ]],
Cell[81708, 2556, 121, 3, 70, "ExampleDelimiter",
 CellID->11575],
Cell[81832, 2561, 157, 4, 70, "ExampleText",
 CellID->26557],
Cell[CellGroupData[{
Cell[82014, 2569, 558, 19, 70, "Input",
 CellID->13182],
Cell[82575, 2590, 627, 18, 36, "Output",
 CellID->8496]
}, Open  ]],
Cell[83217, 2611, 121, 3, 70, "ExampleDelimiter",
 CellID->23662],
Cell[83341, 2616, 175, 5, 70, "ExampleText",
 CellID->30760],
Cell[CellGroupData[{
Cell[83541, 2625, 334, 11, 70, "Input",
 CellID->27980],
Cell[83878, 2638, 609, 21, 36, "Output",
 CellID->25303]
}, Open  ]],
Cell[84502, 2662, 121, 3, 70, "ExampleDelimiter",
 CellID->14720],
Cell[84626, 2667, 93, 1, 70, "ExampleText",
 CellID->11548],
Cell[CellGroupData[{
Cell[84744, 2672, 483, 17, 70, "Input",
 CellID->21301],
Cell[85230, 2691, 1196, 24, 70, "Output",
 Evaluatable->False,
 CellID->239097570]
}, Open  ]],
Cell[CellGroupData[{
Cell[86463, 2720, 96, 3, 70, "Input",
 CellID->15928],
Cell[86562, 2725, 1110, 22, 70, "Output",
 Evaluatable->False,
 CellID->168139396]
}, Open  ]],
Cell[87687, 2750, 120, 3, 70, "ExampleDelimiter",
 CellID->7809],
Cell[87810, 2755, 114, 3, 70, "ExampleText",
 CellID->30800],
Cell[CellGroupData[{
Cell[87949, 2762, 380, 13, 70, "Input",
 CellID->26698],
Cell[88332, 2777, 495, 15, 36, "Output",
 CellID->5017]
}, Open  ]],
Cell[88842, 2795, 120, 3, 70, "ExampleDelimiter",
 CellID->1445],
Cell[88965, 2800, 79, 1, 70, "ExampleText",
 CellID->25227],
Cell[CellGroupData[{
Cell[89069, 2805, 596, 20, 70, "Input",
 CellID->17896],
Cell[89668, 2827, 674, 16, 116, "Output",
 CellID->13334]
}, Open  ]],
Cell[90357, 2846, 121, 3, 70, "ExampleDelimiter",
 CellID->19826],
Cell[90481, 2851, 75, 1, 70, "ExampleText",
 CellID->3996],
Cell[CellGroupData[{
Cell[90581, 2856, 469, 16, 70, "Input",
 CellID->12432],
Cell[91053, 2874, 1177, 24, 116, "Output",
 CellID->8394]
}, Open  ]],
Cell[92245, 2901, 121, 3, 70, "ExampleDelimiter",
 CellID->31371],
Cell[92369, 2906, 99, 2, 70, "ExampleText",
 CellID->17127],
Cell[CellGroupData[{
Cell[92493, 2912, 364, 12, 70, "Input",
 CellID->10792],
Cell[92860, 2926, 701, 24, 36, "Output",
 CellID->88775419]
}, Open  ]],
Cell[93576, 2953, 121, 3, 70, "ExampleDelimiter",
 CellID->21819],
Cell[93700, 2958, 158, 5, 70, "ExampleText",
 CellID->16585],
Cell[CellGroupData[{
Cell[93883, 2967, 509, 18, 70, "Input",
 CellID->10752],
Cell[94395, 2987, 842, 20, 116, "Output",
 CellID->170013987]
}, Open  ]],
Cell[95252, 3010, 121, 3, 70, "ExampleDelimiter",
 CellID->22760],
Cell[95376, 3015, 71, 1, 70, "ExampleText",
 CellID->16613],
Cell[CellGroupData[{
Cell[95472, 3020, 508, 18, 70, "Input",
 CellID->15555],
Cell[95983, 3040, 7042, 120, 70, "Output",
 Evaluatable->False,
 CellID->1439720]
}, Open  ]],
Cell[CellGroupData[{
Cell[103062, 3165, 580, 19, 70, "Input",
 CellID->1022],
Cell[103645, 3186, 7731, 131, 70, "Output",
 Evaluatable->False,
 CellID->108011751]
}, Open  ]],
Cell[111391, 3320, 121, 3, 70, "ExampleDelimiter",
 CellID->14371]
}, Closed]],
Cell[CellGroupData[{
Cell[111549, 3328, 235, 7, 70, "ExampleSubsection",
 CellID->26673],
Cell[111787, 3337, 86, 1, 70, "ExampleText",
 CellID->30660],
Cell[CellGroupData[{
Cell[111898, 3342, 236, 7, 70, "Input",
 CellID->18312],
Cell[112137, 3351, 486, 16, 36, "Output",
 CellID->950]
}, Open  ]],
Cell[CellGroupData[{
Cell[112660, 3372, 266, 8, 70, "Input",
 CellID->6128],
Cell[112929, 3382, 293, 11, 64, "Output",
 CellID->24420]
}, Open  ]],
Cell[113237, 3396, 125, 4, 70, "ExampleText",
 CellID->8532],
Cell[CellGroupData[{
Cell[113387, 3404, 347, 12, 70, "Input",
 CellID->25744],
Cell[113737, 3418, 725, 24, 36, "Output",
 CellID->18918]
}, Open  ]],
Cell[114477, 3445, 121, 3, 70, "ExampleDelimiter",
 CellID->28213],
Cell[114601, 3450, 64, 1, 70, "ExampleText",
 CellID->8191],
Cell[CellGroupData[{
Cell[114690, 3455, 250, 7, 70, "Input",
 CellID->7022],
Cell[114943, 3464, 1512, 30, 117, "Output",
 CellID->28651]
}, Open  ]],
Cell[116470, 3497, 121, 3, 70, "ExampleDelimiter",
 CellID->29317],
Cell[116594, 3502, 116, 3, 70, "ExampleText",
 CellID->11573],
Cell[CellGroupData[{
Cell[116735, 3509, 325, 10, 70, "Input",
 CellID->11677],
Cell[117063, 3521, 752, 17, 70, "Output",
 Evaluatable->False,
 CellID->222513779]
}, Open  ]],
Cell[117830, 3541, 120, 3, 70, "ExampleDelimiter",
 CellID->1430],
Cell[117953, 3546, 78, 1, 70, "ExampleText",
 CellID->16061],
Cell[CellGroupData[{
Cell[118056, 3551, 263, 9, 70, "Input",
 CellID->13591],
Cell[118322, 3562, 713, 16, 70, "Output",
 Evaluatable->False,
 CellID->1699977]
}, Open  ]],
Cell[119050, 3581, 121, 3, 70, "ExampleDelimiter",
 CellID->31974],
Cell[119174, 3586, 91, 1, 70, "ExampleText",
 CellID->4779],
Cell[CellGroupData[{
Cell[119290, 3591, 355, 12, 70, "Input",
 CellID->31728],
Cell[119648, 3605, 731, 16, 70, "Output",
 Evaluatable->False,
 CellID->253880348]
}, Open  ]],
Cell[120394, 3624, 119, 3, 70, "ExampleDelimiter",
 CellID->179],
Cell[120516, 3629, 78, 1, 70, "ExampleText",
 CellID->31703],
Cell[CellGroupData[{
Cell[120619, 3634, 400, 13, 70, "Input",
 CellID->26352],
Cell[121022, 3649, 578, 15, 88, "Output",
 CellID->9545]
}, Open  ]],
Cell[121615, 3667, 121, 3, 70, "ExampleDelimiter",
 CellID->27583],
Cell[121739, 3672, 237, 8, 70, "ExampleText",
 CellID->21133],
Cell[CellGroupData[{
Cell[122001, 3684, 369, 12, 70, "Input",
 CellID->29570],
Cell[122373, 3698, 587, 15, 88, "Output",
 CellID->17591]
}, Open  ]],
Cell[122975, 3716, 121, 3, 70, "ExampleDelimiter",
 CellID->22609],
Cell[123099, 3721, 100, 2, 70, "ExampleText",
 CellID->23942],
Cell[CellGroupData[{
Cell[123224, 3727, 636, 21, 70, "Input",
 CellID->19660],
Cell[123863, 3750, 618, 15, 79, "Output",
 CellID->1396]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[124530, 3771, 227, 7, 70, "ExampleSubsection",
 CellID->32380],
Cell[124760, 3780, 62, 1, 70, "ExampleText",
 CellID->18199],
Cell[CellGroupData[{
Cell[124847, 3785, 218, 7, 70, "Input",
 CellID->15392],
Cell[125068, 3794, 450, 14, 36, "Output",
 CellID->29350]
}, Open  ]],
Cell[125533, 3811, 56, 1, 70, "ExampleText",
 CellID->18339],
Cell[CellGroupData[{
Cell[125614, 3816, 242, 8, 70, "Input",
 CellID->29353],
Cell[125859, 3826, 450, 14, 36, "Output",
 CellID->30867]
}, Open  ]],
Cell[126324, 3843, 119, 3, 70, "ExampleDelimiter",
 CellID->789],
Cell[126446, 3848, 78, 1, 70, "ExampleText",
 CellID->15165],
Cell[CellGroupData[{
Cell[126549, 3853, 267, 9, 70, "Input",
 CellID->18025],
Cell[126819, 3864, 256, 9, 36, "Output",
 CellID->6635]
}, Open  ]],
Cell[127090, 3876, 121, 3, 70, "ExampleDelimiter",
 CellID->20610],
Cell[127214, 3881, 70, 1, 70, "ExampleText",
 CellID->10454],
Cell[CellGroupData[{
Cell[127309, 3886, 293, 10, 70, "Input",
 CellID->15681],
Cell[127605, 3898, 229, 7, 36, "Output",
 CellID->12991]
}, Open  ]],
Cell[127849, 3908, 121, 3, 70, "ExampleDelimiter",
 CellID->13142],
Cell[127973, 3913, 59, 1, 70, "ExampleText",
 CellID->21766],
Cell[CellGroupData[{
Cell[128057, 3918, 338, 11, 70, "Input",
 CellID->21977],
Cell[128398, 3931, 1304, 26, 62, "Output",
 CellID->32747]
}, Open  ]],
Cell[129717, 3960, 120, 3, 70, "ExampleDelimiter",
 CellID->8582],
Cell[129840, 3965, 71, 1, 70, "ExampleText",
 CellID->11507],
Cell[CellGroupData[{
Cell[129936, 3970, 292, 10, 70, "Input",
 CellID->2783],
Cell[130231, 3982, 450, 14, 36, "Output",
 CellID->15797]
}, Open  ]],
Cell[130696, 3999, 120, 3, 70, "ExampleDelimiter",
 CellID->6096],
Cell[130819, 4004, 76, 1, 70, "ExampleText",
 CellID->26710],
Cell[CellGroupData[{
Cell[130920, 4009, 351, 12, 70, "Input",
 CellID->30259],
Cell[131274, 4023, 1247, 25, 58, "Output",
 CellID->14980]
}, Open  ]],
Cell[132536, 4051, 120, 3, 70, "ExampleDelimiter",
 CellID->8007]
}, Closed]],
Cell[CellGroupData[{
Cell[132693, 4059, 239, 7, 70, "ExampleSubsection",
 CellID->10633],
Cell[132935, 4068, 271, 8, 70, "ExampleText",
 CellID->14359],
Cell[CellGroupData[{
Cell[133231, 4080, 279, 9, 70, "Input",
 CellID->16286],
Cell[133513, 4091, 828, 19, 135, "Output",
 CellID->7655]
}, Open  ]],
Cell[134356, 4113, 58, 1, 70, "ExampleText",
 CellID->14628],
Cell[CellGroupData[{
Cell[134439, 4118, 334, 10, 70, "Input",
 CellID->18197],
Cell[134776, 4130, 829, 19, 135, "Output",
 CellID->14456]
}, Open  ]],
Cell[135620, 4152, 120, 3, 70, "ExampleDelimiter",
 CellID->9245],
Cell[135743, 4157, 131, 4, 70, "ExampleText",
 CellID->11633],
Cell[CellGroupData[{
Cell[135899, 4165, 328, 10, 70, "Input",
 CellID->18846],
Cell[136230, 4177, 845, 19, 116, "Output",
 CellID->13774]
}, Open  ]],
Cell[137090, 4199, 121, 3, 70, "ExampleDelimiter",
 CellID->13194],
Cell[137214, 4204, 97, 2, 70, "ExampleText",
 CellID->9760],
Cell[CellGroupData[{
Cell[137336, 4210, 262, 9, 70, "Input",
 CellID->13041],
Cell[137601, 4221, 1080, 23, 127, "Output",
 CellID->9970]
}, Open  ]],
Cell[138696, 4247, 89, 1, 70, "ExampleText",
 CellID->25348],
Cell[CellGroupData[{
Cell[138810, 4252, 312, 10, 70, "Input",
 CellID->28732],
Cell[139125, 4264, 1077, 23, 116, "Output",
 CellID->10046]
}, Open  ]],
Cell[140217, 4290, 121, 3, 70, "ExampleDelimiter",
 CellID->15717],
Cell[140341, 4295, 106, 3, 70, "ExampleText",
 CellID->989],
Cell[CellGroupData[{
Cell[140472, 4302, 519, 17, 70, "Input",
 CellID->9822],
Cell[140994, 4321, 2647, 47, 70, "Output",
 Evaluatable->False,
 CellID->180824983]
}, Open  ]],
Cell[143656, 4371, 114, 3, 70, "ExampleText",
 CellID->11940],
Cell[CellGroupData[{
Cell[143795, 4378, 573, 18, 70, "Input",
 CellID->11368],
Cell[144371, 4398, 1837, 45, 83, "Output",
 CellID->8481]
}, Open  ]],
Cell[146223, 4446, 121, 3, 70, "ExampleDelimiter",
 CellID->25048],
Cell[146347, 4451, 121, 3, 70, "ExampleText",
 CellID->18500],
Cell[CellGroupData[{
Cell[146493, 4458, 403, 13, 70, "Input",
 CellID->19386],
Cell[146899, 4473, 930, 20, 87, "Output",
 CellID->19413]
}, Open  ]],
Cell[147844, 4496, 121, 3, 70, "ExampleDelimiter",
 CellID->24574],
Cell[147968, 4501, 122, 3, 70, "ExampleText",
 CellID->2625],
Cell[CellGroupData[{
Cell[148115, 4508, 366, 12, 70, "Input",
 CellID->10926],
Cell[148484, 4522, 807, 18, 144, "Output",
 CellID->12664]
}, Open  ]],
Cell[149306, 4543, 121, 3, 70, "ExampleDelimiter",
 CellID->21469],
Cell[149430, 4548, 74, 1, 70, "ExampleText",
 CellID->15252],
Cell[CellGroupData[{
Cell[149529, 4553, 394, 14, 70, "Input",
 CellID->32300],
Cell[149926, 4569, 990, 21, 147, "Output",
 CellID->21986]
}, Open  ]],
Cell[150931, 4593, 120, 3, 70, "ExampleDelimiter",
 CellID->2614],
Cell[151054, 4598, 94, 1, 70, "ExampleText",
 CellID->11523],
Cell[CellGroupData[{
Cell[151173, 4603, 294, 10, 70, "Input",
 CellID->28330],
Cell[151470, 4615, 671, 15, 70, "Output",
 Evaluatable->False,
 CellID->239449859]
}, Open  ]],
Cell[152156, 4633, 86, 1, 70, "ExampleText",
 CellID->4566],
Cell[CellGroupData[{
Cell[152267, 4638, 320, 11, 70, "Input",
 CellID->7741],
Cell[152590, 4651, 351, 10, 36, "Output",
 CellID->14578]
}, Open  ]],
Cell[152956, 4664, 119, 3, 70, "ExampleDelimiter",
 CellID->629],
Cell[153078, 4669, 79, 1, 70, "ExampleText",
 CellID->31014],
Cell[CellGroupData[{
Cell[153182, 4674, 472, 16, 70, "Input",
 CellID->19494],
Cell[153657, 4692, 1003, 21, 116, "Output",
 CellID->12184]
}, Open  ]],
Cell[154675, 4716, 121, 3, 70, "ExampleDelimiter",
 CellID->10973],
Cell[154799, 4721, 130, 4, 70, "ExampleText",
 CellID->22384],
Cell[CellGroupData[{
Cell[154954, 4729, 483, 16, 70, "Input",
 CellID->2306],
Cell[155440, 4747, 1638, 30, 126, "Output",
 CellID->22299]
}, Open  ]],
Cell[157093, 4780, 121, 3, 70, "ExampleDelimiter",
 CellID->27215],
Cell[157217, 4785, 137, 4, 70, "ExampleText",
 CellID->1643],
Cell[CellGroupData[{
Cell[157379, 4793, 427, 15, 70, "Input",
 CellID->1634],
Cell[157809, 4810, 509, 14, 113, "Output",
 CellID->18761]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[158367, 4830, 240, 7, 70, "ExampleSubsection",
 CellID->14626],
Cell[158610, 4839, 109, 3, 70, "ExampleText",
 CellID->32030],
Cell[CellGroupData[{
Cell[158744, 4846, 433, 15, 70, "Input",
 CellID->32516],
Cell[159180, 4863, 1130, 23, 70, "Output",
 Evaluatable->False,
 CellID->70310847]
}, Open  ]],
Cell[160325, 4889, 73, 1, 70, "ExampleText",
 CellID->9389],
Cell[CellGroupData[{
Cell[160423, 4894, 505, 17, 70, "Input",
 CellID->30870],
Cell[160931, 4913, 655, 15, 70, "Output",
 Evaluatable->False,
 CellID->466909047]
}, Open  ]],
Cell[161601, 4931, 120, 3, 70, "ExampleDelimiter",
 CellID->7994],
Cell[161724, 4936, 70, 1, 70, "ExampleText",
 CellID->16999],
Cell[CellGroupData[{
Cell[161819, 4941, 556, 18, 70, "Input",
 CellID->12926],
Cell[162378, 4961, 802, 19, 171, "Output",
 CellID->5075]
}, Open  ]],
Cell[163195, 4983, 121, 3, 70, "ExampleDelimiter",
 CellID->24504],
Cell[163319, 4988, 72, 1, 70, "ExampleText",
 CellID->24141],
Cell[CellGroupData[{
Cell[163416, 4993, 525, 17, 70, "Input",
 CellID->26096],
Cell[163944, 5012, 1024, 22, 183, "Output",
 CellID->1618]
}, Open  ]],
Cell[164983, 5037, 121, 3, 70, "ExampleDelimiter",
 CellID->31483],
Cell[165107, 5042, 82, 1, 70, "ExampleText",
 CellID->26756],
Cell[CellGroupData[{
Cell[165214, 5047, 769, 24, 70, "Input",
 CellID->5771],
Cell[165986, 5073, 5610, 96, 70, "Output",
 Evaluatable->False,
 CellID->79060237]
}, Open  ]],
Cell[CellGroupData[{
Cell[171633, 5174, 932, 28, 70, "Input",
 CellID->15031],
Cell[172568, 5204, 45827, 693, 166, 22222, 301, "CachedBoxData", "BoxData", \
"Output",
 CellID->26176]
}, Open  ]],
Cell[218410, 5900, 121, 3, 70, "ExampleDelimiter",
 CellID->19910],
Cell[218534, 5905, 255, 9, 70, "ExampleText",
 CellID->28743],
Cell[CellGroupData[{
Cell[218814, 5918, 596, 19, 70, "Input",
 CellID->30524],
Cell[219413, 5939, 1141, 24, 139, "Output",
 CellID->27108]
}, Open  ]],
Cell[220569, 5966, 121, 3, 70, "ExampleDelimiter",
 CellID->17027],
Cell[220693, 5971, 174, 6, 70, "ExampleText",
 CellID->4430],
Cell[CellGroupData[{
Cell[220892, 5981, 521, 17, 70, "Input",
 CellID->7077],
Cell[221416, 6000, 2157, 40, 116, "Output",
 CellID->22123]
}, Open  ]],
Cell[223588, 6043, 120, 3, 70, "ExampleDelimiter",
 CellID->7912],
Cell[223711, 6048, 65, 1, 70, "ExampleText",
 CellID->2171],
Cell[CellGroupData[{
Cell[223801, 6053, 914, 28, 70, "Input",
 CellID->14931],
Cell[224718, 6083, 5235, 91, 195, "Output",
 CellID->29190]
}, Open  ]],
Cell[229968, 6177, 121, 3, 70, "ExampleDelimiter",
 CellID->22634],
Cell[230092, 6182, 72, 1, 70, "ExampleText",
 CellID->23862],
Cell[CellGroupData[{
Cell[230189, 6187, 915, 28, 70, "Input",
 CellID->17818],
Cell[231107, 6217, 3692, 66, 196, "Output",
 CellID->3078]
}, Open  ]],
Cell[234814, 6286, 121, 3, 70, "ExampleDelimiter",
 CellID->20010],
Cell[234938, 6291, 122, 3, 70, "ExampleText",
 CellID->19184],
Cell[CellGroupData[{
Cell[235085, 6298, 1125, 35, 70, "Input",
 CellID->16569],
Cell[236213, 6335, 5104, 89, 195, "Output",
 CellID->14845]
}, Open  ]],
Cell[241332, 6427, 120, 3, 70, "ExampleDelimiter",
 CellID->8581],
Cell[241455, 6432, 88, 1, 70, "ExampleText",
 CellID->1892],
Cell[CellGroupData[{
Cell[241568, 6437, 737, 23, 70, "Input",
 CellID->32016],
Cell[242308, 6462, 13167, 220, 70, "Output",
 Evaluatable->False,
 CellID->131610993]
}, Open  ]],
Cell[255490, 6685, 65, 2, 70, "Input",
 CellID->15901]
}, Closed]],
Cell[CellGroupData[{
Cell[255592, 6692, 235, 7, 70, "ExampleSubsection",
 CellID->14221],
Cell[255830, 6701, 98, 2, 70, "ExampleText",
 CellID->30960],
Cell[CellGroupData[{
Cell[255953, 6707, 310, 10, 70, "Input",
 CellID->15583],
Cell[256266, 6719, 575, 17, 36, "Output",
 CellID->25913]
}, Open  ]],
Cell[256856, 6739, 123, 3, 70, "ExampleText",
 CellID->10856],
Cell[CellGroupData[{
Cell[257004, 6746, 409, 13, 70, "Input",
 CellID->4122],
Cell[257416, 6761, 575, 17, 36, "Output",
 CellID->12153]
}, Open  ]],
Cell[258006, 6781, 121, 3, 70, "ExampleDelimiter",
 CellID->20332],
Cell[258130, 6786, 89, 1, 70, "ExampleText",
 CellID->9371],
Cell[CellGroupData[{
Cell[258244, 6791, 504, 16, 70, "Input",
 CellID->24008],
Cell[258751, 6809, 653, 19, 36, "Output",
 CellID->8008]
}, Open  ]],
Cell[259419, 6831, 120, 3, 70, "ExampleDelimiter",
 CellID->3173],
Cell[259542, 6836, 129, 4, 70, "ExampleText",
 CellID->1575],
Cell[CellGroupData[{
Cell[259696, 6844, 421, 14, 70, "Input",
 CellID->5058],
Cell[260120, 6860, 580, 15, 133, "Output",
 CellID->32699]
}, Open  ]],
Cell[260715, 6878, 84, 1, 70, "ExampleText",
 CellID->11559],
Cell[CellGroupData[{
Cell[260824, 6883, 519, 18, 70, "Input",
 CellID->460],
Cell[261346, 6903, 503, 12, 70, "Output",
 Evaluatable->False,
 CellID->94197990]
}, Open  ]],
Cell[261864, 6918, 121, 3, 70, "ExampleDelimiter",
 CellID->17881],
Cell[261988, 6923, 99, 2, 70, "ExampleText",
 CellID->31406],
Cell[CellGroupData[{
Cell[262112, 6929, 475, 16, 70, "Input",
 CellID->12755],
Cell[262590, 6947, 905, 20, 118, "Output",
 CellID->22438]
}, Open  ]],
Cell[263510, 6970, 170, 7, 70, "ExampleText",
 CellID->24539],
Cell[CellGroupData[{
Cell[263705, 6981, 485, 16, 70, "Input",
 CellID->2012],
Cell[264193, 6999, 612, 15, 135, "Output",
 CellID->1255]
}, Open  ]],
Cell[264820, 7017, 121, 3, 70, "ExampleDelimiter",
 CellID->18897],
Cell[264944, 7022, 106, 3, 70, "ExampleText",
 CellID->13672],
Cell[CellGroupData[{
Cell[265075, 7029, 979, 33, 70, "Input",
 CellID->14417],
Cell[266057, 7064, 2816, 52, 115, "Output",
 CellID->162793075]
}, Open  ]],
Cell[268888, 7119, 125, 3, 70, "ExampleDelimiter",
 CellID->651026136],
Cell[269016, 7124, 131, 4, 70, "ExampleText",
 CellID->302524911],
Cell[CellGroupData[{
Cell[269172, 7132, 733, 24, 70, "Input",
 CellID->298338114],
Cell[269908, 7158, 984, 22, 116, "Output",
 CellID->89265084]
}, Open  ]],
Cell[270907, 7183, 67, 1, 70, "ExampleText",
 CellID->93111161],
Cell[CellGroupData[{
Cell[270999, 7188, 734, 24, 70, "Input",
 CellID->258039322],
Cell[271736, 7214, 1394, 29, 201, "Output",
 CellID->512306136]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[273179, 7249, 236, 7, 70, "ExampleSubsection",
 CellID->17148],
Cell[273418, 7258, 88, 1, 70, "ExampleText",
 CellID->12105],
Cell[CellGroupData[{
Cell[273531, 7263, 215, 7, 70, "Input",
 CellID->19556],
Cell[273749, 7272, 229, 7, 36, "Output",
 CellID->27000]
}, Open  ]],
Cell[273993, 7282, 58, 1, 70, "ExampleText",
 CellID->9665],
Cell[CellGroupData[{
Cell[274076, 7287, 217, 7, 70, "Input",
 CellID->9293],
Cell[274296, 7296, 228, 7, 36, "Output",
 CellID->1185]
}, Open  ]],
Cell[274539, 7306, 121, 3, 70, "ExampleDelimiter",
 CellID->11562],
Cell[274663, 7311, 69, 1, 70, "ExampleText",
 CellID->32429],
Cell[CellGroupData[{
Cell[274757, 7316, 307, 10, 70, "Input",
 CellID->29116],
Cell[275067, 7328, 539, 16, 36, "Output",
 CellID->18170]
}, Open  ]],
Cell[CellGroupData[{
Cell[275643, 7349, 261, 8, 70, "Input",
 CellID->19572],
Cell[275907, 7359, 539, 16, 36, "Output",
 CellID->15130]
}, Open  ]],
Cell[276461, 7378, 120, 3, 70, "ExampleDelimiter",
 CellID->1498],
Cell[276584, 7383, 79, 1, 70, "ExampleText",
 CellID->26033],
Cell[CellGroupData[{
Cell[276688, 7388, 293, 10, 70, "Input",
 CellID->14680],
Cell[276984, 7400, 252, 8, 36, "Output",
 CellID->2699]
}, Open  ]],
Cell[277251, 7411, 119, 3, 70, "ExampleDelimiter",
 CellID->364],
Cell[277373, 7416, 117, 3, 70, "ExampleText",
 CellID->12425],
Cell[CellGroupData[{
Cell[277515, 7423, 208, 7, 70, "Input",
 CellID->11879],
Cell[277726, 7432, 283, 10, 36, "Output",
 CellID->32738]
}, Open  ]],
Cell[278024, 7445, 86, 1, 70, "ExampleText",
 CellID->9246],
Cell[CellGroupData[{
Cell[278135, 7450, 208, 7, 70, "Input",
 CellID->16272],
Cell[278346, 7459, 282, 10, 36, "Output",
 CellID->7742]
}, Open  ]],
Cell[278643, 7472, 121, 3, 70, "ExampleDelimiter",
 CellID->21204],
Cell[278767, 7477, 72, 1, 70, "ExampleText",
 CellID->22562],
Cell[CellGroupData[{
Cell[278864, 7482, 254, 8, 70, "Input",
 CellID->24636],
Cell[279121, 7492, 783, 27, 36, "Output",
 CellID->15441]
}, Open  ]]
}, Closed]]
}, Closed]],
Cell[CellGroupData[{
Cell[279965, 7526, 224, 7, 70, "ExampleSection",
 CellID->22713],
Cell[280192, 7535, 69, 1, 70, "ExampleText",
 CellID->743],
Cell[CellGroupData[{
Cell[280286, 7540, 435, 14, 70, "Input",
 CellID->29416],
Cell[280724, 7556, 1467, 28, 70, "Output",
 Evaluatable->False,
 CellID->21067856]
}, Open  ]],
Cell[282206, 7587, 121, 3, 70, "ExampleDelimiter",
 CellID->30082],
Cell[282330, 7592, 100, 2, 70, "ExampleText",
 CellID->23142],
Cell[CellGroupData[{
Cell[282455, 7598, 509, 16, 70, "Input",
 CellID->18466],
Cell[282967, 7616, 4568, 79, 70, "Output",
 Evaluatable->False,
 CellID->213879773]
}, Open  ]],
Cell[287550, 7698, 120, 3, 70, "ExampleDelimiter",
 CellID->5423],
Cell[287673, 7703, 150, 4, 70, "ExampleText",
 CellID->17873],
Cell[CellGroupData[{
Cell[287848, 7711, 777, 24, 70, "Input",
 CellID->12238],
Cell[288628, 7737, 5628, 96, 70, "Output",
 Evaluatable->False,
 CellID->69066640]
}, Open  ]],
Cell[294271, 7836, 120, 3, 70, "ExampleDelimiter",
 CellID->9869],
Cell[294394, 7841, 136, 4, 70, "ExampleText",
 CellID->14479],
Cell[CellGroupData[{
Cell[294555, 7849, 720, 23, 70, "Input",
 CellID->20549],
Cell[295278, 7874, 12098, 202, 70, "Output",
 Evaluatable->False,
 CellID->71939180]
}, Open  ]],
Cell[307391, 8079, 85, 1, 70, "ExampleText",
 CellID->19700],
Cell[CellGroupData[{
Cell[307501, 8084, 949, 30, 70, "Input",
 CellID->31086],
Cell[308453, 8116, 11811, 198, 70, "Output",
 Evaluatable->False,
 CellID->605770273]
}, Open  ]],
Cell[320279, 8317, 120, 3, 70, "ExampleDelimiter",
 CellID->8661],
Cell[320402, 8322, 85, 1, 70, "ExampleText",
 CellID->7893],
Cell[CellGroupData[{
Cell[320512, 8327, 821, 26, 70, "Input",
 CellID->14055],
Cell[321336, 8355, 45298, 763, 223, "Output",
 CellID->61625747]
}, Open  ]],
Cell[366649, 9121, 121, 3, 70, "ExampleDelimiter",
 CellID->17367],
Cell[366773, 9126, 118, 3, 70, "ExampleText",
 CellID->12947],
Cell[CellGroupData[{
Cell[366916, 9133, 328, 10, 70, "Input",
 CellID->12452],
Cell[367247, 9145, 10588, 179, 222, "Output",
 CellID->13144]
}, Open  ]],
Cell[377850, 9327, 121, 3, 70, "ExampleDelimiter",
 CellID->24951],
Cell[377974, 9332, 129, 4, 70, "ExampleText",
 CellID->28052],
Cell[CellGroupData[{
Cell[378128, 9340, 496, 17, 70, "Input",
 CellID->6719],
Cell[378627, 9359, 2152, 42, 120, "Output",
 CellID->9195]
}, Open  ]],
Cell[380794, 9404, 120, 3, 70, "ExampleDelimiter",
 CellID->4048],
Cell[380917, 9409, 121, 3, 70, "ExampleText",
 CellID->18839],
Cell[CellGroupData[{
Cell[381063, 9416, 455, 14, 70, "Input",
 CellID->6989],
Cell[381521, 9432, 401, 11, 36, "Output",
 CellID->700]
}, Open  ]],
Cell[381937, 9446, 121, 3, 70, "ExampleDelimiter",
 CellID->27054],
Cell[382061, 9451, 113, 3, 70, "ExampleText",
 CellID->1096],
Cell[CellGroupData[{
Cell[382199, 9458, 358, 12, 70, "Input",
 CellID->14321],
Cell[382560, 9472, 39355, 1031, 70, "Output",
 CellID->66732148]
}, Open  ]],
Cell[421930, 10506, 121, 3, 70, "ExampleDelimiter",
 CellID->27921],
Cell[422054, 10511, 151, 4, 70, "ExampleText",
 CellID->29367],
Cell[CellGroupData[{
Cell[422230, 10519, 712, 23, 70, "Input",
 CellID->25968],
Cell[422945, 10544, 946, 20, 117, "Output",
 CellID->31545]
}, Open  ]],
Cell[423906, 10567, 73, 1, 70, "ExampleText",
 CellID->17716],
Cell[CellGroupData[{
Cell[424004, 10572, 776, 25, 70, "Input",
 CellID->1458],
Cell[424783, 10599, 3214, 58, 117, "Output",
 CellID->26519]
}, Open  ]],
Cell[428012, 10660, 120, 3, 70, "ExampleDelimiter",
 CellID->7096],
Cell[428135, 10665, 81, 1, 70, "ExampleText",
 CellID->26135],
Cell[CellGroupData[{
Cell[428241, 10670, 1115, 35, 70, "Input",
 CellID->13861],
Cell[429359, 10707, 3343, 59, 70, "Output",
 Evaluatable->False,
 CellID->201919402]
}, Open  ]],
Cell[432717, 10769, 121, 3, 70, "ExampleDelimiter",
 CellID->27754],
Cell[432841, 10774, 113, 3, 70, "ExampleText",
 CellID->8670],
Cell[432957, 10779, 562, 18, 70, "Input",
 CellID->18290],
Cell[433522, 10799, 1601, 52, 70, "Input",
 CellID->24985],
Cell[CellGroupData[{
Cell[435148, 10855, 358, 11, 70, "Input",
 CellID->7278],
Cell[435509, 10868, 5082, 88, 135, "Output",
 CellID->4671]
}, Open  ]],
Cell[440606, 10959, 121, 3, 70, "ExampleDelimiter",
 CellID->14156],
Cell[440730, 10964, 125, 4, 70, "ExampleText",
 CellID->2130],
Cell[440858, 10970, 561, 18, 70, "Input",
 CellID->6571],
Cell[CellGroupData[{
Cell[441444, 10992, 358, 11, 70, "Input",
 CellID->18834],
Cell[441805, 11005, 15045, 252, 246, "Output",
 CellID->31354]
}, Open  ]],
Cell[456865, 11260, 63, 1, 70, "ExampleText",
 CellID->8343],
Cell[CellGroupData[{
Cell[456953, 11265, 352, 11, 70, "Input",
 CellID->21429],
Cell[457308, 11278, 10963, 185, 217, "Output",
 CellID->6696]
}, Open  ]],
Cell[468286, 11466, 121, 3, 70, "ExampleDelimiter",
 CellID->30423],
Cell[468410, 11471, 94, 1, 70, "ExampleText",
 CellID->8800],
Cell[CellGroupData[{
Cell[468529, 11476, 788, 24, 70, "Input",
 CellID->9716],
Cell[469320, 11502, 3093, 70, 123, "Output",
 CellID->26590]
}, Open  ]],
Cell[472428, 11575, 56, 1, 70, "ExampleText",
 CellID->24222],
Cell[CellGroupData[{
Cell[472509, 11580, 755, 23, 70, "Input",
 CellID->7572],
Cell[473267, 11605, 10342, 187, 123, "Output",
 CellID->14322]
}, Open  ]],
Cell[483624, 11795, 121, 3, 70, "ExampleDelimiter",
 CellID->20708],
Cell[483748, 11800, 115, 3, 70, "ExampleText",
 CellID->26748],
Cell[CellGroupData[{
Cell[483888, 11807, 774, 24, 70, "Input",
 CellID->2902],
Cell[484665, 11833, 29864, 494, 70, "Output",
 Evaluatable->False,
 CellID->35726924]
}, Open  ]],
Cell[514544, 12330, 121, 3, 70, "ExampleDelimiter",
 CellID->25446],
Cell[514668, 12335, 115, 3, 70, "ExampleText",
 CellID->4728],
Cell[CellGroupData[{
Cell[514808, 12342, 732, 24, 70, "Input",
 CellID->24864],
Cell[515543, 12368, 34190, 565, 70, "Output",
 Evaluatable->False,
 CellID->158031399]
}, Open  ]],
Cell[549748, 12936, 121, 3, 70, "ExampleDelimiter",
 CellID->31109],
Cell[549872, 12941, 130, 4, 70, "ExampleText",
 CellID->28401],
Cell[CellGroupData[{
Cell[550027, 12949, 898, 27, 70, "Input",
 CellID->31603],
Cell[550928, 12978, 4879, 95, 123, "Output",
 CellID->2764]
}, Open  ]],
Cell[555822, 13076, 120, 3, 70, "ExampleDelimiter",
 CellID->3215],
Cell[555945, 13081, 84, 1, 70, "ExampleText",
 CellID->32425],
Cell[CellGroupData[{
Cell[556054, 13086, 419, 15, 70, "Input",
 CellID->3134],
Cell[556476, 13103, 404, 14, 108, "Output",
 CellID->18752]
}, Open  ]],
Cell[CellGroupData[{
Cell[556917, 13122, 512, 18, 70, "Input",
 CellID->11435],
Cell[557432, 13142, 553, 14, 108, "Output",
 CellID->4100]
}, Open  ]],
Cell[558000, 13159, 121, 3, 70, "ExampleDelimiter",
 CellID->26208],
Cell[558124, 13164, 111, 3, 70, "ExampleText",
 CellID->9865],
Cell[CellGroupData[{
Cell[558260, 13171, 463, 16, 70, "Input",
 CellID->3847],
Cell[558726, 13189, 675, 20, 36, "Output",
 CellID->17041]
}, Open  ]],
Cell[CellGroupData[{
Cell[559438, 13214, 561, 19, 70, "Input",
 CellID->26901],
Cell[560002, 13235, 1052, 22, 171, "Output",
 CellID->22927]
}, Open  ]],
Cell[561069, 13260, 120, 3, 70, "ExampleDelimiter",
 CellID->9660],
Cell[CellGroupData[{
Cell[561214, 13267, 418, 14, 70, "Input",
 CellID->19282],
Cell[561635, 13283, 1425, 27, 70, "Output",
 Evaluatable->False,
 CellID->9712325]
}, Open  ]],
Cell[563075, 13313, 121, 3, 70, "ExampleDelimiter",
 CellID->19191],
Cell[563199, 13318, 83, 1, 70, "ExampleText",
 CellID->1397],
Cell[CellGroupData[{
Cell[563307, 13323, 824, 27, 70, "Input",
 CellID->16539],
Cell[564134, 13352, 825, 27, 124, "Output",
 CellID->9592]
}, Open  ]],
Cell[564974, 13382, 121, 3, 70, "ExampleDelimiter",
 CellID->29025]
}, Closed]],
Cell[CellGroupData[{
Cell[565132, 13390, 232, 7, 70, "ExampleSection",
 CellID->9691],
Cell[565367, 13399, 58, 1, 70, "ExampleText",
 CellID->26813],
Cell[CellGroupData[{
Cell[565450, 13404, 318, 10, 70, "Input",
 CellID->17496],
Cell[565771, 13416, 1872, 32, 119, "Output",
 CellID->13618]
}, Open  ]],
Cell[567658, 13451, 121, 3, 70, "ExampleDelimiter",
 CellID->18000],
Cell[567782, 13456, 51, 1, 70, "ExampleText",
 CellID->23522],
Cell[CellGroupData[{
Cell[567858, 13461, 328, 10, 70, "Input",
 CellID->9370],
Cell[568189, 13473, 1501, 32, 130, "Output",
 CellID->16791]
}, Open  ]],
Cell[569705, 13508, 120, 3, 70, "ExampleDelimiter",
 CellID->2784],
Cell[569828, 13513, 76, 1, 70, "ExampleText",
 CellID->32056],
Cell[CellGroupData[{
Cell[569929, 13518, 814, 27, 70, "Input",
 CellID->28359],
Cell[570746, 13547, 535, 14, 119, "Output",
 CellID->26311]
}, Open  ]],
Cell[571296, 13564, 121, 3, 70, "ExampleDelimiter",
 CellID->25972],
Cell[571420, 13569, 273, 9, 70, "ExampleText",
 CellID->3922],
Cell[CellGroupData[{
Cell[571718, 13582, 215, 7, 70, "Input",
 CellID->21045],
Cell[571936, 13591, 496, 15, 36, "Output",
 CellID->29064]
}, Open  ]],
Cell[572447, 13609, 120, 3, 70, "ExampleDelimiter",
 CellID->8009],
Cell[572570, 13614, 109, 3, 70, "ExampleText",
 CellID->26571],
Cell[CellGroupData[{
Cell[572704, 13621, 316, 9, 70, "Input",
 CellID->31267],
Cell[573023, 13632, 470, 12, 70, "Output",
 Evaluatable->False,
 CellID->428610088]
}, Open  ]],
Cell[573508, 13647, 120, 3, 70, "ExampleDelimiter",
 CellID->5793],
Cell[573631, 13652, 134, 4, 70, "ExampleText",
 CellID->1580],
Cell[CellGroupData[{
Cell[573790, 13660, 342, 12, 70, "Input",
 CellID->22789],
Cell[574135, 13674, 266, 9, 36, "Output",
 CellID->31613]
}, Open  ]],
Cell[CellGroupData[{
Cell[574438, 13688, 138, 4, 70, "Input",
 CellID->1209],
Cell[574579, 13694, 265, 9, 36, "Output",
 CellID->4623]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[574893, 13709, 225, 7, 70, "ExampleSection",
 CellID->4143],
Cell[575121, 13718, 130, 4, 70, "ExampleText",
 CellID->24570],
Cell[CellGroupData[{
Cell[575276, 13726, 217, 7, 70, "Input",
 CellID->1157],
Cell[575496, 13735, 450, 14, 36, "Output",
 CellID->26955]
}, Open  ]],
Cell[CellGroupData[{
Cell[575983, 13754, 219, 7, 70, "Input",
 CellID->22837],
Cell[576205, 13763, 387, 13, 36, "Output",
 CellID->11891]
}, Open  ]],
Cell[CellGroupData[{
Cell[576629, 13781, 215, 7, 70, "Input",
 CellID->182],
Cell[576847, 13790, 281, 10, 36, "Output",
 CellID->4390]
}, Open  ]],
Cell[577143, 13803, 318, 11, 70, "ExampleText",
 CellID->4767],
Cell[CellGroupData[{
Cell[577486, 13818, 270, 9, 70, "Input",
 CellID->23687],
Cell[577759, 13829, 449, 14, 36, "Output",
 CellID->8135]
}, Open  ]],
Cell[578223, 13846, 120, 3, 70, "ExampleDelimiter",
 CellID->3915],
Cell[578346, 13851, 146, 4, 70, "ExampleText",
 CellID->98],
Cell[CellGroupData[{
Cell[578517, 13859, 221, 8, 70, "Input",
 CellID->27619],
Cell[578741, 13869, 552, 13, 70, "Output",
 Evaluatable->False,
 CellID->24982057]
}, Open  ]],
Cell[579308, 13885, 120, 3, 70, "ExampleDelimiter",
 CellID->9974],
Cell[579431, 13890, 119, 3, 70, "ExampleText",
 CellID->27296],
Cell[CellGroupData[{
Cell[579575, 13897, 218, 7, 70, "Input",
 CellID->3764],
Cell[579796, 13906, 318, 11, 36, "Output",
 CellID->18223]
}, Open  ]],
Cell[580129, 13920, 138, 4, 70, "ExampleText",
 CellID->6607],
Cell[CellGroupData[{
Cell[580292, 13928, 219, 7, 70, "Input",
 CellID->12074],
Cell[580514, 13937, 448, 14, 36, "Output",
 CellID->790]
}, Open  ]],
Cell[580977, 13954, 101, 2, 70, "ExampleText",
 CellID->22599],
Cell[CellGroupData[{
Cell[581103, 13960, 209, 7, 70, "Input",
 CellID->32518],
Cell[581315, 13969, 282, 10, 36, "Output",
 CellID->3140]
}, Open  ]],
Cell[581612, 13982, 121, 3, 70, "ExampleDelimiter",
 CellID->17103],
Cell[581736, 13987, 112, 3, 70, "ExampleText",
 CellID->7245],
Cell[CellGroupData[{
Cell[581873, 13994, 512, 17, 70, "Input",
 CellID->4257],
Cell[582388, 14013, 2941, 52, 70, "Output",
 Evaluatable->False,
 CellID->447853197]
}, Open  ]],
Cell[585344, 14068, 277, 11, 70, "ExampleText",
 CellID->23357],
Cell[CellGroupData[{
Cell[585646, 14083, 510, 16, 70, "Input",
 CellID->18037],
Cell[586159, 14101, 2569, 46, 70, "Output",
 Evaluatable->False,
 CellID->34612672]
}, Open  ]],
Cell[588743, 14150, 120, 3, 70, "ExampleDelimiter",
 CellID->7373],
Cell[588866, 14155, 263, 8, 70, "ExampleText",
 CellID->397],
Cell[CellGroupData[{
Cell[589154, 14167, 322, 10, 70, "Input",
 CellID->7048],
Cell[589479, 14179, 1058, 22, 70, "Output",
 Evaluatable->False,
 CellID->459162640]
}, Open  ]],
Cell[590552, 14204, 119, 3, 70, "ExampleDelimiter",
 CellID->360],
Cell[590674, 14209, 137, 4, 70, "ExampleText",
 CellID->14126],
Cell[CellGroupData[{
Cell[590836, 14217, 500, 17, 70, "Input",
 CellID->21640],
Cell[591339, 14236, 892, 20, 100, "Output",
 CellID->30119]
}, Open  ]],
Cell[CellGroupData[{
Cell[592268, 14261, 500, 17, 70, "Input",
 CellID->18228],
Cell[592771, 14280, 533, 14, 100, "Output",
 CellID->4741]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[593365, 14301, 311, 9, 70, "SeeAlsoSection",
 CellID->17849],
Cell[593679, 14312, 1457, 53, 70, "SeeAlso",
 CellID->25981]
}, Open  ]],
Cell[CellGroupData[{
Cell[595173, 14370, 313, 9, 70, "TutorialsSection",
 CellID->4335],
Cell[595489, 14381, 144, 3, 70, "Tutorials",
 CellID->31587]
}, Open  ]],
Cell[CellGroupData[{
Cell[595670, 14389, 321, 9, 70, "RelatedLinksSection",
 CellID->23714],
Cell[595994, 14400, 376, 12, 70, "RelatedLinks"],
Cell[596373, 14414, 504, 15, 70, "RelatedLinks",
 CellID->28900],
Cell[596880, 14431, 266, 6, 70, "RelatedLinks",
 CellID->6146],
Cell[597149, 14439, 395, 13, 70, "RelatedLinks"]
}, Open  ]],
Cell[CellGroupData[{
Cell[597581, 14457, 319, 9, 70, "MoreAboutSection",
 CellID->234708990],
Cell[597903, 14468, 147, 3, 70, "MoreAbout",
 CellID->106726794],
Cell[598053, 14473, 165, 3, 70, "MoreAbout",
 CellID->191401289],
Cell[598221, 14478, 164, 3, 70, "MoreAbout",
 CellID->135827946],
Cell[598388, 14483, 151, 3, 70, "MoreAbout",
 CellID->617717607],
Cell[598542, 14488, 143, 3, 70, "MoreAbout",
 CellID->468196979],
Cell[598688, 14493, 150, 3, 70, "MoreAbout",
 CellID->36836740],
Cell[598841, 14498, 179, 3, 70, "MoreAbout",
 CellID->267146821]
}, Open  ]],
Cell[599035, 14504, 52, 0, 70, "History"],
Cell[599090, 14506, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

