(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='Mathematica 6.0' *)

(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       145,          7]
NotebookDataLength[     41174,       1559]
NotebookOptionsPosition[     32022,       1241]
NotebookOutlinePosition[     33420,       1280]
CellTagsIndexPosition[     33334,       1275]
WindowFrame->Normal
ContainsDynamic->True *)

(* Beginning of Notebook Content *)
Notebook[{
Cell[" ", "SymbolColorBar"],

Cell[BoxData[GridBox[{
   {Cell["BUILT-IN MATHEMATICA SYMBOL", "PacletNameCell"], Cell[TextData[{
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"RootIntervals" :> 
          Documentation`HelpLookup["paclet:ref/RootIntervals"], "Solve" :> 
          Documentation`HelpLookup["paclet:ref/Solve"], "Reduce" :> 
          Documentation`HelpLookup["paclet:ref/Reduce"], 
          "NumberFieldSignature" :> 
          Documentation`HelpLookup["paclet:ref/NumberFieldSignature"], 
          "Exponent" :> Documentation`HelpLookup["paclet:ref/Exponent"]}, #, 
         2]& , Evaluator -> Automatic], {1->"\<\"RootIntervals\"\>", 
       2->"\<\"Solve\"\>", 3->"\<\"Reduce\"\>", 
       4->"\<\"NumberFieldSignature\"\>", 
       5->"\<\"Exponent\"\>"}, "\<\"see also\"\>", 
       StyleBox["\<\"see also\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9],
     "\[ThickSpace]\[ThickSpace]\[ThickSpace]|\[ThickSpace]\[ThickSpace]\
\[ThickSpace]",
     Cell[BoxData[
      PopupMenuBox[
       Dynamic[{"ActionMenu", None}, 
        Part[{"Polynomial Equations" :> 
          Documentation`HelpLookup["paclet:guide/PolynomialEquations"], 
          "New in 6.0: Symbolic Computation" :> 
          Documentation`HelpLookup["paclet:guide/NewIn60AlgebraicComputing"], 
          "New in 6.0: Mathematics & Algorithms" :> 
          Documentation`HelpLookup[
           "paclet:guide/NewIn60MathematicsAndAlgorithms"]}, #, 2]& , 
        Evaluator -> Automatic], {1->"\<\"Polynomial Equations\"\>", 
       2->"\<\"New in 6.0: Symbolic Computation\"\>", 
       3->"\<\"New in 6.0: Mathematics & Algorithms\"\>"}, "\<\"more \
about\"\>", 
       StyleBox["\<\"more about\"\>",
        Background->Automatic],
       Appearance->Automatic,
       ImageSize->Automatic,
       MenuAppearance->"Menu"]],
      FontSize->9]
    }], "AnchorBar"]}
  }]], "AnchorBarGrid"],

Cell[CellGroupData[{

Cell[BoxData[GridBox[{
   {Cell["CountRoots", "ObjectName"], Cell[BoxData[
     TooltipBox[
      StyleBox[
       StyleBox[
        GraphicsBox[RasterBox[CompressedData["
1:eJztUltygzAMjGT5bYwJkHea9Cy9QY+QC/T+f5VMm5BpOpP/dmcMtrxarQSv
l4/3Cy4Wi4HXGy/4xxzWe2tk4xVAg1gPjvfUf1Nc4dsMoCUsOL6M4/Im0Vqs
7wY7JodSYgOAEaDH8kVBZ3VmlkKX5bxJSziM65lGldABuSp63nEpDB145/TE
iFbpBtkdUivnZRoA9vK4+pR4j5krB9fUNMxOe9vZiZCDFGKT3sMksQU4pFkr
tohHD+zGBookEtTkQBCnVoo4cNgAIVxdbOcSxEmRWd6wktIxmhYVRwiaUDsV
B8aGkPlaAvu03hxO6XiTUB465Dn1PMccteUumCr0livLPVqwDk0QWcE5jSmt
Zp9VOx0Cj6CTcZue00xg52RAx1pVEw+lRNC9DISxS7sN3CHYqVqEZ7E+b+8D
HuukW9RPa6zSbn40DuvvWdCrxwnG/hAfTi+r64/REkkboIiIfpEI5UF02D9p
+c/gE6ZrEwI=
          "], {{0, 0}, {67, 20}},
          ColorFunction->(Apply[RGBColor, 
            Part[CompressedData["
1:eJx90ssuAwEUxvGJeACJhIj7I9iJzXkHwcYtkVgTsfEkrLRPYFvRloWFhYho
CYL0olrtTKd1pzecz5yzcMIvab7Ff5ImM2d4YWl8sc1xnKHvX7sjqvT//qVK
72+sYFZ6OgVmpbearGxWeu2DFc1KP02CWemDsGdWejjEjs1KH4Ats9I31tmB
WenRHVaibjigDghrf35it2alX1+BWemNOnPNSn99YXdmpT8+sIxZ6bNQpmSC
ZWkZ0tovzsGs9NAm82kNijQK99pjUebSJJxRLxxqbzaYRyOQoX640R6PMS/4
rinqgUvt81DR5/rA0/7ZYn7wvy6NQUn7bpyVaRVytAJZ7XPg03aEFWgK8r/u
q6LP/dyPb+6/C/I0ATnt0+BS4gSoE/a1z4CndyHvUXq9xkrBezsK7jCi/Qvn
YdPd
             "], #]]& ),
          ColorFunctionScaling->False],
         Background->None,
         ImageSize->{67, 20},
         PlotRange->{{0, 67}, {0, 20}}],
        Deployed->True,
        StripOnInput->False], "NewIn6Graphic",
       StripOnInput->False],
      "New in 6",
      ActionDelay->0.3]],
     Magnification->1]}
  }]], "ObjectNameGrid"],

Cell[BoxData[GridBox[{
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["CountRoots",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/CountRoots"], "[", 
       RowBox[{
        StyleBox["poly", "TI"], ",", 
        StyleBox["x", "TI"]}], "]"}]], "InlineFormula"],
     "\[LineSeparator]gives the number of real roots of the polynomial ",
     Cell[BoxData[
      StyleBox["poly", "TI"]], "InlineFormula"],
     " in ",
     Cell[BoxData[
      StyleBox["x", "TI"]], "InlineFormula"],
     "."
    }]]},
   {"", Cell[TextData[{
     Cell[BoxData[
      RowBox[{
       ButtonBox["CountRoots",
        BaseStyle->"Link",
        ButtonData->"paclet:ref/CountRoots"], "[", 
       RowBox[{
        StyleBox["poly", "TI"], ",", 
        RowBox[{"{", 
         RowBox[{
          StyleBox["x", "TI"], ",", 
          StyleBox["a", "TI"], ",", 
          StyleBox["b", "TI"]}], "}"}]}], "]"}]], "InlineFormula"],
     " \[LineSeparator]gives the number of roots between ",
     Cell[BoxData[
      StyleBox["a", "TI"]], "InlineFormula"],
     " and ",
     Cell[BoxData[
      StyleBox["b", "TI"]], "InlineFormula"],
     ". "
    }]]}
  }]], "Usage",
 GridBoxOptions->{
 GridBoxBackground->{
  "Columns" -> {{None}}, "ColumnsIndexed" -> {}, "Rows" -> {None, {None}}, 
   "RowsIndexed" -> {}}},
 CellID->24241]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[Cell[BoxData[
 ButtonBox[
  FrameBox[
   StyleBox[
    RowBox[{"MORE", " ", "INFORMATION"}], "NotesFrameText"],
   StripOnInput->False],
  BaseStyle->None,
  ButtonFrame->"None",
  ButtonFunction:>(FrontEndExecute[{
     FrontEnd`SelectionMove[
      FrontEnd`SelectedNotebook[], All, ButtonCell], 
     FrontEndToken["OpenCloseGroup"]}]& ),
  Evaluator->None,
  Method->"Preemptive"]]]], "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],

Cell[TextData[{
 Cell[BoxData[
  ButtonBox["CountRoots",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/CountRoots"]], "InlineFormula"],
 " enumerates roots with multiplicities counted."
}], "Notes",
 CellID->11829],

Cell[TextData[{
 "Roots that lie exactly at ",
 Cell[BoxData[
  RowBox[{
   StyleBox["x", "TI"], "=", 
   StyleBox["a", "TI"]}]], "InlineFormula"],
 " or ",
 Cell[BoxData[
  RowBox[{
   StyleBox["x", "TI"], "=", 
   StyleBox["b", "TI"]}]], "InlineFormula"],
 " are counted."
}], "Notes",
 CellID->241480229],

Cell[TextData[{
 "The limits ",
 Cell[BoxData[
  StyleBox["a", "TI"]], "InlineFormula"],
 " and ",
 Cell[BoxData[
  StyleBox["b", "TI"]], "InlineFormula"],
 " can be complex, in which case the range is taken to be a closed rectangle \
in the complex plane. "
}], "Notes",
 CellID->12448]
}, Closed]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["EXAMPLES",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->188116678],

Cell[CellGroupData[{

Cell[TextData[{
 "Basic Examples",
 "\[NonBreakingSpace]\[NonBreakingSpace]",
 Cell["(1)", "ExampleCount"]
}], "ExampleSection",
 CellID->203930400],

Cell["Count the number of polynomial roots between 0 and 10:", "ExampleText",
 CellID->35951891],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CountRoots", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", "-", "1"}], ")"}], 
    RowBox[{"(", 
     RowBox[{"x", "-", "2"}], ")"}], 
    RowBox[{"(", 
     RowBox[{"x", "-", "3"}], ")"}], 
    RowBox[{"(", 
     RowBox[{"x", "-", "4"}], ")"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "10"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->34414617],

Cell[BoxData["4"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->13155228]
}, Open  ]],

Cell["Count roots of a polynomial in a closed rectangle:", "ExampleText",
 CellID->321282813],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CountRoots", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "17"}], "-", "1"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", 
     RowBox[{"1", "+", "I"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->346913294],

Cell[BoxData["5"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->161278401]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Scope",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(8)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->10629601],

Cell["Count roots is a real interval:", "ExampleText",
 CellID->965944092],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CountRoots", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "-", "2"}], ")"}], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "-", "5"}], ")"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "2"}], ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->284117812],

Cell[BoxData["2"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->328631799]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->531253269],

Cell["Find the number of the real roots:", "ExampleText",
 CellID->93304425],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CountRoots", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "-", "2"}], ")"}], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "-", "5"}], ")"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "Infinity"}], ",", "Infinity"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->173702359],

Cell[BoxData["4"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->106625040]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->611600201],

Cell["Count roots is a closed rectangle:", "ExampleText",
 CellID->32486528],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CountRoots", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "4"}], "-", "2"}], ")"}], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "4"}], "-", "20"}], ")"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{
      RowBox[{"-", "2"}], "-", 
      RowBox[{"2", "I"}]}], ",", 
     RowBox[{"2", "+", 
      RowBox[{"2", "I"}]}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->56600433],

Cell[BoxData["4"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->9521349]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->247479273],

Cell["Count roots in a vertical line segment:", "ExampleText",
 CellID->21598941],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CountRoots", "[", 
  RowBox[{
   RowBox[{"1", "+", 
    RowBox[{"x", "^", "2"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{
      RowBox[{"-", "2"}], "I"}], ",", 
     RowBox[{"2", "I"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->831276582],

Cell[BoxData["2"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->409835208]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->103188296],

Cell["Count roots in a horizontal line segment:", "ExampleText",
 CellID->56268299],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CountRoots", "[", 
  RowBox[{
   RowBox[{"1", "+", 
    RowBox[{"x", "^", "2"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{
      RowBox[{"-", "1"}], "+", "I"}], ",", 
     RowBox[{"1", "+", "I"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->567155249],

Cell[BoxData["1"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->503411892]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->211247560],

Cell["Multiple roots are counted with their multiplicities:", "ExampleText",
 CellID->979433896],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CountRoots", "[", 
  RowBox[{
   RowBox[{"x", "^", "10"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "1"}], ",", "1"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->313704712],

Cell[BoxData["10"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->914547919]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->220817433],

Cell["Roots at the end points of the interval are included:", "ExampleText",
 CellID->334397247],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CountRoots", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", "-", "1"}], ")"}], 
    RowBox[{"(", 
     RowBox[{"x", "-", "2"}], ")"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->63341386],

Cell[BoxData["2"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->2787295]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->839688405],

Cell["Roots on the boundary of the rectangle are included:", "ExampleText",
 CellID->823297536],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CountRoots", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"x", "^", "4"}], "-", "1"}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", 
     RowBox[{"1", "+", "I"}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->65321142],

Cell[BoxData["2"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->108289873]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Applications",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->482274386],

Cell[TextData[{
 "The number of 17",
 Cell[BoxData[
  FormBox[
   SuperscriptBox["\[Null]", "th"], TraditionalForm]]],
 " roots of unity in the closed unit square in the first quadrant:"
}], "ExampleText",
 CellID->23024],

Cell[CellGroupData[{

Cell[BoxData[{
 RowBox[{
  RowBox[{"poly", "=", 
   RowBox[{
    RowBox[{"x", "^", "17"}], "-", "1"}]}], ";"}], "\n", 
 RowBox[{"CountRoots", "[", 
  RowBox[{"poly", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", 
     RowBox[{"1", "+", "I"}]}], "}"}]}], "]"}]}], "Input",
 CellLabel->"In[1]:=",
 CellID->147106605],

Cell[BoxData["5"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->925059560]
}, Open  ]],

Cell["Roots on the boundary are counted:", "ExampleText",
 CellID->310164848],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Graphics", "[", 
  RowBox[{"{", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{
      RowBox[{"GrayLevel", "[", "0.7", "]"}], ",", 
      RowBox[{"Rectangle", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{"0", ",", "0"}], "}"}], ",", 
        RowBox[{"{", 
         RowBox[{"1", ",", "1"}], "}"}]}], "]"}]}], "}"}], ",", 
    RowBox[{"{", 
     RowBox[{"Red", ",", 
      RowBox[{"PointSize", "[", "Medium", "]"}], ",", 
      RowBox[{"Point", "[", 
       RowBox[{
        RowBox[{"{", 
         RowBox[{
          RowBox[{"Re", "[", "x", "]"}], ",", 
          RowBox[{"Im", "[", "x", "]"}]}], "}"}], "/.", 
        RowBox[{"NSolve", "[", 
         RowBox[{
          RowBox[{"poly", "==", "0"}], ",", "x"}], "]"}]}], "]"}]}], "}"}]}], 
   "}"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->361938877],

Cell[GraphicsData["CompressedBitmap", "\<\
eJzNl78vQ1EcxR/Pr4UJIQwSjAjd2050IH5V6FQNEgbxO7FQWxdiZGZQQy0l
/AkWm0TEbBGJ0Xjd2yvt/Z4eIhrRRK/2fc7rO+d+T5r3RhKbiwvLic2luUTH
0HpidXFpbqNjcGVdH/IrPM+71K+rFk+/V/pd7nWplP5v17hZq9X7u0qlVDis
trfV25s35rLDQ5U70fylUpLNzBRYJPLJQvQ72+zR6+vCGWdn8gy4Urs56psz
kkmjHXe1cOVVl8E1LAvDt63lju5QrwMuA1eCgQvBwIVlu+Ai4MyieO2k7vZd
Bu4EA3eCgTvLusDdwbfu7HpEPba4DDwKBh4FA4+WHYPH1h94LF7rqesTl4Fr
wcC1YODasgZwffor1+6apglqXQYJBIMEgkECy84hQV3JCSppgozLIIFgkEAw
SGCZDwkuSk5g1wzNUekyyGFZTK/RbDbrVT8HgyIL8tu9PZGnwKcM9zV/jMUM
8fNkWpOqsvvEZz7BNLBnVAMdoBrogtRgJybLaK+KP43m2nCVyTzE4689PU/R
6E067flEcz8/n0+txVTjtu4lEKAat3l3W1ugGXPaZ2hVmewV/3UfYhpoGdVA
y6gGWiY1+Osf+fcmBWmTGokGm8Q02CSmwSZJTQia1PSnO8Dvq/qZBvpBNdAP
qoF+SA3edw2UkLKLTraXaHCyTIOTZRqcrNR0w2T7vvAeojNpYhqYCdXATKgG
ZmI0NXT3hvNn4lNEc/4c3M1hkbOG7mZBE6eO7NPRLFzTPge5dyBexQfIpMcV
\
\>"], "Output",
 Evaluatable->False,
 ImageSize->{180, 181},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->408286004]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Properties & Relations",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(5)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->530322960],

Cell["\<\
The number of complex roots of a polynomial is equal to its degree:\
\>", "ExampleText",
 CellID->207877269],

Cell[BoxData[
 RowBox[{
  RowBox[{"poly", "=", 
   RowBox[{
    RowBox[{"x", "^", "10"}], "-", 
    RowBox[{"17", 
     RowBox[{"x", "^", "7"}]}], "+", 
    RowBox[{"21", 
     RowBox[{"x", "^", "3"}]}], "-", 
    RowBox[{"55", 
     RowBox[{"x", "^", "2"}]}], "+", 
    RowBox[{"22", "x"}], "+", "2"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->1098306],

Cell["\<\
This gives a bound on absolute values of roots of a polynomial:\
\>", "ExampleText",
 CellID->416249027],

Cell[BoxData[
 RowBox[{
  RowBox[{"CauchyBound", "[", 
   RowBox[{"f_", ",", "x_"}], "]"}], ":=", 
  RowBox[{"Module", "[", 
   RowBox[{
    RowBox[{"{", 
     RowBox[{"cfl", ",", "n"}], "}"}], ",", "\[IndentingNewLine]", 
    RowBox[{
     RowBox[{"cfl", "=", 
      RowBox[{"CoefficientList", "[", 
       RowBox[{"f", ",", "x"}], "]"}]}], ";", "\[IndentingNewLine]", 
     RowBox[{"n", "=", 
      RowBox[{
       RowBox[{"Length", "[", "cfl", "]"}], "-", "1"}]}], ";", 
     "\[IndentingNewLine]", 
     RowBox[{"Max", "[", 
      RowBox[{"Table", "[", 
       RowBox[{
        RowBox[{
         RowBox[{"(", 
          RowBox[{"n", " ", 
           RowBox[{"Abs", "[", 
            RowBox[{
             RowBox[{"cfl", "[", 
              RowBox[{"[", 
               RowBox[{"n", "+", "1", "-", "k"}], "]"}], "]"}], "/", 
             RowBox[{"cfl", "[", 
              RowBox[{"[", 
               RowBox[{"n", "+", "1"}], "]"}], "]"}]}], "]"}]}], ")"}], "^", 
         RowBox[{"(", 
          RowBox[{"1", "/", "k"}], ")"}]}], ",", 
        RowBox[{"{", 
         RowBox[{"k", ",", "n"}], "}"}]}], "]"}], "]"}]}]}], "]"}]}]], "Input",\

 CellLabel->"In[2]:=",
 CellID->476469004],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"b", "=", 
  RowBox[{"Ceiling", "[", 
   RowBox[{"CauchyBound", "[", 
    RowBox[{"poly", ",", "x"}], "]"}], "]"}]}]], "Input",
 CellLabel->"In[3]:=",
 CellID->64825081],

Cell[BoxData["6"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->203863120]
}, Open  ]],

Cell["\<\
The polynomial indeed has 10 roots within the Cauchy bounded region:\
\>", "ExampleText",
 CellID->151811354],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CountRoots", "[", 
  RowBox[{"poly", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{
      RowBox[{"-", "b"}], "-", 
      RowBox[{"b", " ", "I"}]}], ",", 
     RowBox[{"b", "+", 
      RowBox[{"b", " ", "I"}]}]}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[4]:=",
 CellID->7513659],

Cell[BoxData["10"], "Output",
 ImageSize->{18, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[4]=",
 CellID->23551179]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->96476135],

Cell[TextData[{
 "The number of real roots of a polynomial with ",
 Cell[BoxData[
  FormBox["k", TraditionalForm]], "InlineMath"],
 " nonzero terms is at most ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{"2", " ", "k"}], "-", "1"}], TraditionalForm]], "InlineMath"],
 ":"
}], "ExampleText",
 CellID->510290089],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"poly", "=", 
  RowBox[{"Expand", "[", 
   RowBox[{"x", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "4"}], "-", "1"}], ")"}], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "4"}], "-", "2"}], ")"}], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "4"}], "-", "3"}], ")"}]}], "]"}]}]], "Input",
 CellLabel->"In[1]:=",
 CellID->192157722],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"-", "6"}], " ", "x"}], "+", 
  RowBox[{"11", " ", 
   SuperscriptBox["x", "5"]}], "-", 
  RowBox[{"6", " ", 
   SuperscriptBox["x", "9"]}], "+", 
  SuperscriptBox["x", "13"]}]], "Output",
 ImageSize->{137, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->61971326]
}, Open  ]],

Cell[TextData[{
 "This polynomial has the maximal possible number of ",
 Cell[BoxData[
  FormBox[
   RowBox[{
    RowBox[{
     RowBox[{"2", "\[Times]", "4"}], "-", "1"}], "=", "7"}], 
   TraditionalForm]], "InlineMath"],
 " real roots:"
}], "ExampleText",
 CellID->276847544],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CountRoots", "[", 
  RowBox[{"poly", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "Infinity"}], ",", "Infinity"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->497277506],

Cell[BoxData["7"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->18913014]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->44742430],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["Reduce",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/Reduce"]], "InlineFormula"],
 " to find polynomial roots:"
}], "ExampleText",
 CellID->139477591],

Cell[BoxData[
 RowBox[{
  RowBox[{"poly", "=", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "2"}], "-", "2"}], ")"}], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "3"}], "-", 
      RowBox[{"5", "x"}], "+", "1"}], ")"}]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->59478566],

Cell["Count roots:", "ExampleText",
 CellID->182845853],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CountRoots", "[", 
  RowBox[{"poly", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "3"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->901892],

Cell[BoxData["3"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->55160521]
}, Open  ]],

Cell["Find roots:", "ExampleText",
 CellID->76337438],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"Reduce", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"poly", "\[Equal]", "0"}], "&&", 
    RowBox[{"0", "\[LessEqual]", "x", "\[LessEqual]", "3"}]}], ",", "x"}], 
  "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->187579913],

Cell[BoxData[
 RowBox[{
  RowBox[{"x", "\[Equal]", 
   SqrtBox["2"]}], "||", 
  RowBox[{"x", "\[Equal]", 
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"1", "-", 
       RowBox[{"5", " ", "#1"}], "+", 
       SuperscriptBox["#1", "3"]}], "&"}], ",", "2"}], "]"}]}], "||", 
  RowBox[{"x", "\[Equal]", 
   RowBox[{"Root", "[", 
    RowBox[{
     RowBox[{
      RowBox[{"1", "-", 
       RowBox[{"5", " ", "#1"}], "+", 
       SuperscriptBox["#1", "3"]}], "&"}], ",", "3"}], "]"}]}]}]], "Output",
 ImageSize->{432, 20},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->761478700]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->241036],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["RootIntervals",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/RootIntervals"]], "InlineFormula"],
 " to find isolating intervals for polynomial roots:"
}], "ExampleText",
 CellID->121197046],

Cell[BoxData[
 RowBox[{
  RowBox[{"poly", "=", 
   RowBox[{
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "4"}], "-", "2"}], ")"}], 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"x", "^", "8"}], "-", "3"}], ")"}]}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->387282702],

Cell["Count the real roots:", "ExampleText",
 CellID->484141508],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CountRoots", "[", 
  RowBox[{"poly", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "Infinity"}], ",", "Infinity"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->91147939],

Cell[BoxData["4"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->38203427]
}, Open  ]],

Cell["Isolate the real roots:", "ExampleText",
 CellID->536739186],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"RootIntervals", "[", "poly", "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->64124442],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", 
        FractionBox["5", "4"]}], ",", 
       RowBox[{"-", 
        FractionBox["19", "16"]}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       RowBox[{"-", 
        FractionBox["19", "16"]}], ",", 
       RowBox[{"-", 
        FractionBox["9", "8"]}]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       FractionBox["9", "8"], ",", 
       FractionBox["19", "16"]}], "}"}], ",", 
     RowBox[{"{", 
      RowBox[{
       FractionBox["19", "16"], ",", 
       FractionBox["5", "4"]}], "}"}]}], "}"}], ",", 
   RowBox[{"{", 
    RowBox[{
     RowBox[{"{", "1", "}"}], ",", 
     RowBox[{"{", "1", "}"}], ",", 
     RowBox[{"{", "1", "}"}], ",", 
     RowBox[{"{", "1", "}"}]}], "}"}]}], "}"}]], "Output",
 ImageSize->{457, 30},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->83701030]
}, Open  ]],

Cell[BoxData[
 InterpretationBox[Cell["\t", "ExampleDelimiter"],
  $Line = 0; Null]], "ExampleDelimiter",
 CellID->95695241],

Cell[TextData[{
 "Use ",
 Cell[BoxData[
  ButtonBox["NumberFieldSignature",
   BaseStyle->"Link",
   ButtonData->"paclet:ref/NumberFieldSignature"]], "InlineFormula"],
 " to count the real roots and the pairs of complex roots:"
}], "ExampleText",
 CellID->713935552],

Cell[BoxData[
 RowBox[{
  RowBox[{"poly", "=", 
   RowBox[{
    RowBox[{"x", "^", "8"}], "-", "2"}]}], ";"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->54789058],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"NumberFieldSignature", "[", 
  RowBox[{"poly", ",", "x"}], "]"}]], "Input",
 CellLabel->"In[2]:=",
 CellID->4298987],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"2", ",", "3"}], "}"}]], "Output",
 ImageSize->{44, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[2]=",
 CellID->138244449]
}, Open  ]],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CountRoots", "[", 
  RowBox[{"poly", ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", 
     RowBox[{"-", "Infinity"}], ",", "Infinity"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[3]:=",
 CellID->515818050],

Cell[BoxData["2"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[3]=",
 CellID->199673087]
}, Open  ]]
}, Closed]],

Cell[CellGroupData[{

Cell[BoxData[
 InterpretationBox[Cell[TextData[{
   "Possible Issues",
   "\[NonBreakingSpace]\[NonBreakingSpace]",
   Cell["(1)", "ExampleCount"]
  }], "ExampleSection"],
  $Line = 0; Null]], "ExampleSection",
 CellID->10891],

Cell["Roots at the end points of the interval are included:", "ExampleText",
 CellID->43546263],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{"CountRoots", "[", 
  RowBox[{
   RowBox[{
    RowBox[{"(", 
     RowBox[{"x", "-", "1"}], ")"}], 
    RowBox[{"(", 
     RowBox[{"x", "-", "2"}], ")"}]}], ",", 
   RowBox[{"{", 
    RowBox[{"x", ",", "0", ",", "2"}], "}"}]}], "]"}]], "Input",
 CellLabel->"In[1]:=",
 CellID->785057400],

Cell[BoxData["2"], "Output",
 ImageSize->{11, 15},
 ImageMargins->{{0, 0}, {0, 0}},
 ImageRegion->{{0, 1}, {0, 1}},
 CellLabel->"Out[1]=",
 CellID->702373385]
}, Open  ]]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["SEE ALSO",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "SeeAlsoSection",
 CellID->12274],

Cell[TextData[{
 Cell[BoxData[
  StyleBox[
   ButtonBox["RootIntervals",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/RootIntervals"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Solve",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Solve"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Reduce",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Reduce"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["NumberFieldSignature",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/NumberFieldSignature"],
   FontFamily->"Verdana"]], "InlineFormula"],
 "\[NonBreakingSpace]",
 StyleBox["\[FilledSmallSquare]", "InlineSeparator"],
 " ",
 Cell[BoxData[
  StyleBox[
   ButtonBox["Exponent",
    BaseStyle->"Link",
    ButtonData->"paclet:ref/Exponent"],
   FontFamily->"Verdana"]], "InlineFormula"]
}], "SeeAlso",
 CellID->18133]
}, Open  ]],

Cell[CellGroupData[{

Cell[TextData[ButtonBox["MORE ABOUT",
 BaseStyle->None,
 Evaluator->None,
 Method->"Preemptive",
 ButtonFunction:>(FrontEndExecute[{
    FrontEnd`SelectionMove[
     FrontEnd`SelectedNotebook[], All, ButtonCell], 
    FrontEndToken["OpenCloseGroup"]}]& ),
 ButtonFrame->"None"]], "MoreAboutSection",
 CellID->533002409],

Cell[TextData[ButtonBox["Polynomial Equations",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/PolynomialEquations"]], "MoreAbout",
 CellID->24801857],

Cell[TextData[ButtonBox["New in 6.0: Symbolic Computation",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60AlgebraicComputing"]], "MoreAbout",
 CellID->131860968],

Cell[TextData[ButtonBox["New in 6.0: Mathematics & Algorithms",
 BaseStyle->"Link",
 ButtonData->"paclet:guide/NewIn60MathematicsAndAlgorithms"]], "MoreAbout",
 CellID->34820731]
}, Open  ]],

Cell["New in 6", "History"],

Cell[" ", "FooterCell"]
},
Saveable->False,
ScreenStyleEnvironment->"Working",
WindowSize->{725, 750},
WindowMargins->{{0, Automatic}, {Automatic, 0}},
WindowTitle->"CountRoots - Wolfram Mathematica",
TaggingRules->{
 "ModificationHighlight" -> False, 
  "Metadata" -> {
   "built" -> "{2007, 4, 20, 19, 35, 38.8946512}", "context" -> "System`", 
    "keywords" -> {
     "complex root counts", "exact root counts", "real root counts", 
      "root counts in region"}, "index" -> True, "label" -> 
    "Built-in Mathematica Symbol", "language" -> "en", "paclet" -> 
    "Mathematica", "status" -> "None", "summary" -> 
    "CountRoots[poly, x] gives the number of real roots of the polynomial \
poly in x. CountRoots[poly, {x, a, b}] gives the number of roots between a \
and b. ", "synonyms" -> {"count roots"}, "title" -> "CountRoots", "type" -> 
    "Symbol", "uri" -> "ref/CountRoots"}},
FrontEndVersion->"6.0 for Microsoft Windows (32-bit) (April 17, 2007)",
StyleDefinitions->FrontEnd`FileName[{"Wolfram"}, "Reference.nb", 
  CharacterEncoding -> "WindowsANSI"]
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{
 "PrimaryExamplesSection"->{
  Cell[6909, 216, 360, 10, 70, "PrimaryExamplesSection",
   CellTags->"PrimaryExamplesSection",
   CellID->188116678]}
 }
*)
(*CellTagsIndex
CellTagsIndex->{
 {"PrimaryExamplesSection", 33191, 1268}
 }
*)
(*NotebookFileOutline
Notebook[{
Cell[568, 21, 27, 0, 8, "SymbolColorBar"],
Cell[598, 23, 1997, 45, 70, "AnchorBarGrid"],
Cell[CellGroupData[{
Cell[2620, 72, 1575, 37, 70, "ObjectNameGrid"],
Cell[4198, 111, 1342, 44, 70, "Usage",
 CellID->24241]
}, Open  ]],
Cell[CellGroupData[{
Cell[5577, 160, 475, 15, 70, "NotesSection",
 CellGroupingRules->{"SectionGrouping", 50},
 CellID->1],
Cell[6055, 177, 217, 7, 70, "Notes",
 CellID->11829],
Cell[6275, 186, 307, 13, 70, "Notes",
 CellID->241480229],
Cell[6585, 201, 287, 10, 70, "Notes",
 CellID->12448]
}, Closed]],
Cell[CellGroupData[{
Cell[6909, 216, 360, 10, 70, "PrimaryExamplesSection",
 CellTags->"PrimaryExamplesSection",
 CellID->188116678],
Cell[CellGroupData[{
Cell[7294, 230, 148, 5, 70, "ExampleSection",
 CellID->203930400],
Cell[7445, 237, 96, 1, 70, "ExampleText",
 CellID->35951891],
Cell[CellGroupData[{
Cell[7566, 242, 421, 15, 28, "Input",
 CellID->34414617],
Cell[7990, 259, 157, 5, 36, "Output",
 CellID->13155228]
}, Open  ]],
Cell[8162, 267, 93, 1, 70, "ExampleText",
 CellID->321282813],
Cell[CellGroupData[{
Cell[8280, 272, 264, 9, 28, "Input",
 CellID->346913294],
Cell[8547, 283, 158, 5, 36, "Output",
 CellID->161278401]
}, Open  ]]
}, Open  ]],
Cell[CellGroupData[{
Cell[8754, 294, 219, 7, 70, "ExampleSection",
 CellID->10629601],
Cell[8976, 303, 74, 1, 70, "ExampleText",
 CellID->965944092],
Cell[CellGroupData[{
Cell[9075, 308, 384, 14, 70, "Input",
 CellID->284117812],
Cell[9462, 324, 158, 5, 36, "Output",
 CellID->328631799]
}, Open  ]],
Cell[9635, 332, 125, 3, 70, "ExampleDelimiter",
 CellID->531253269],
Cell[9763, 337, 76, 1, 70, "ExampleText",
 CellID->93304425],
Cell[CellGroupData[{
Cell[9864, 342, 398, 14, 70, "Input",
 CellID->173702359],
Cell[10265, 358, 158, 5, 36, "Output",
 CellID->106625040]
}, Open  ]],
Cell[10438, 366, 125, 3, 70, "ExampleDelimiter",
 CellID->611600201],
Cell[10566, 371, 76, 1, 70, "ExampleText",
 CellID->32486528],
Cell[CellGroupData[{
Cell[10667, 376, 481, 18, 70, "Input",
 CellID->56600433],
Cell[11151, 396, 156, 5, 36, "Output",
 CellID->9521349]
}, Open  ]],
Cell[11322, 404, 125, 3, 70, "ExampleDelimiter",
 CellID->247479273],
Cell[11450, 409, 81, 1, 70, "ExampleText",
 CellID->21598941],
Cell[CellGroupData[{
Cell[11556, 414, 301, 11, 70, "Input",
 CellID->831276582],
Cell[11860, 427, 158, 5, 36, "Output",
 CellID->409835208]
}, Open  ]],
Cell[12033, 435, 125, 3, 70, "ExampleDelimiter",
 CellID->103188296],
Cell[12161, 440, 83, 1, 70, "ExampleText",
 CellID->56268299],
Cell[CellGroupData[{
Cell[12269, 445, 311, 11, 70, "Input",
 CellID->567155249],
Cell[12583, 458, 158, 5, 36, "Output",
 CellID->503411892]
}, Open  ]],
Cell[12756, 466, 125, 3, 70, "ExampleDelimiter",
 CellID->211247560],
Cell[12884, 471, 96, 1, 70, "ExampleText",
 CellID->979433896],
Cell[CellGroupData[{
Cell[13005, 476, 234, 8, 70, "Input",
 CellID->313704712],
Cell[13242, 486, 159, 5, 36, "Output",
 CellID->914547919]
}, Open  ]],
Cell[13416, 494, 125, 3, 70, "ExampleDelimiter",
 CellID->220817433],
Cell[13544, 499, 96, 1, 70, "ExampleText",
 CellID->334397247],
Cell[CellGroupData[{
Cell[13665, 504, 308, 11, 70, "Input",
 CellID->63341386],
Cell[13976, 517, 156, 5, 36, "Output",
 CellID->2787295]
}, Open  ]],
Cell[14147, 525, 125, 3, 70, "ExampleDelimiter",
 CellID->839688405],
Cell[14275, 530, 95, 1, 70, "ExampleText",
 CellID->823297536],
Cell[CellGroupData[{
Cell[14395, 535, 262, 9, 70, "Input",
 CellID->65321142],
Cell[14660, 546, 158, 5, 36, "Output",
 CellID->108289873]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[14867, 557, 227, 7, 70, "ExampleSection",
 CellID->482274386],
Cell[15097, 566, 221, 7, 70, "ExampleText",
 CellID->23024],
Cell[CellGroupData[{
Cell[15343, 577, 323, 11, 70, "Input",
 CellID->147106605],
Cell[15669, 590, 158, 5, 36, "Output",
 CellID->925059560]
}, Open  ]],
Cell[15842, 598, 77, 1, 70, "ExampleText",
 CellID->310164848],
Cell[CellGroupData[{
Cell[15944, 603, 844, 27, 70, "Input",
 CellID->361938877],
Cell[16791, 632, 1006, 21, 70, "Output",
 Evaluatable->False,
 CellID->408286004]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[17846, 659, 237, 7, 70, "ExampleSection",
 CellID->530322960],
Cell[18086, 668, 118, 3, 70, "ExampleText",
 CellID->207877269],
Cell[18207, 673, 363, 13, 70, "Input",
 CellID->1098306],
Cell[18573, 688, 114, 3, 70, "ExampleText",
 CellID->416249027],
Cell[18690, 693, 1187, 36, 70, "Input",
 CellID->476469004],
Cell[CellGroupData[{
Cell[19902, 733, 192, 6, 70, "Input",
 CellID->64825081],
Cell[20097, 741, 158, 5, 36, "Output",
 CellID->203863120]
}, Open  ]],
Cell[20270, 749, 119, 3, 70, "ExampleText",
 CellID->151811354],
Cell[CellGroupData[{
Cell[20414, 756, 317, 11, 70, "Input",
 CellID->7513659],
Cell[20734, 769, 158, 5, 36, "Output",
 CellID->23551179]
}, Open  ]],
Cell[20907, 777, 124, 3, 70, "ExampleDelimiter",
 CellID->96476135],
Cell[21034, 782, 318, 11, 70, "ExampleText",
 CellID->510290089],
Cell[CellGroupData[{
Cell[21377, 797, 391, 14, 70, "Input",
 CellID->192157722],
Cell[21771, 813, 364, 13, 36, "Output",
 CellID->61971326]
}, Open  ]],
Cell[22150, 829, 276, 10, 70, "ExampleText",
 CellID->276847544],
Cell[CellGroupData[{
Cell[22451, 843, 226, 7, 70, "Input",
 CellID->497277506],
Cell[22680, 852, 157, 5, 36, "Output",
 CellID->18913014]
}, Open  ]],
Cell[22852, 860, 124, 3, 70, "ExampleDelimiter",
 CellID->44742430],
Cell[22979, 865, 208, 8, 70, "ExampleText",
 CellID->139477591],
Cell[23190, 875, 319, 12, 70, "Input",
 CellID->59478566],
Cell[23512, 889, 55, 1, 70, "ExampleText",
 CellID->182845853],
Cell[CellGroupData[{
Cell[23592, 894, 188, 6, 70, "Input",
 CellID->901892],
Cell[23783, 902, 157, 5, 36, "Output",
 CellID->55160521]
}, Open  ]],
Cell[23955, 910, 53, 1, 70, "ExampleText",
 CellID->76337438],
Cell[CellGroupData[{
Cell[24033, 915, 245, 8, 70, "Input",
 CellID->187579913],
Cell[24281, 925, 644, 22, 41, "Output",
 CellID->761478700]
}, Open  ]],
Cell[24940, 950, 122, 3, 70, "ExampleDelimiter",
 CellID->241036],
Cell[25065, 955, 246, 8, 70, "ExampleText",
 CellID->121197046],
Cell[25314, 965, 288, 11, 70, "Input",
 CellID->387282702],
Cell[25605, 978, 64, 1, 70, "ExampleText",
 CellID->484141508],
Cell[CellGroupData[{
Cell[25694, 983, 225, 7, 70, "Input",
 CellID->91147939],
Cell[25922, 992, 157, 5, 36, "Output",
 CellID->38203427]
}, Open  ]],
Cell[26094, 1000, 66, 1, 70, "ExampleText",
 CellID->536739186],
Cell[CellGroupData[{
Cell[26185, 1005, 111, 3, 70, "Input",
 CellID->64124442],
Cell[26299, 1010, 969, 35, 51, "Output",
 CellID->83701030]
}, Open  ]],
Cell[27283, 1048, 124, 3, 70, "ExampleDelimiter",
 CellID->95695241],
Cell[27410, 1053, 266, 8, 70, "ExampleText",
 CellID->713935552],
Cell[27679, 1063, 161, 6, 70, "Input",
 CellID->54789058],
Cell[CellGroupData[{
Cell[27865, 1073, 140, 4, 70, "Input",
 CellID->4298987],
Cell[28008, 1079, 203, 7, 36, "Output",
 CellID->138244449]
}, Open  ]],
Cell[CellGroupData[{
Cell[28248, 1091, 226, 7, 70, "Input",
 CellID->515818050],
Cell[28477, 1100, 158, 5, 36, "Output",
 CellID->199673087]
}, Open  ]]
}, Closed]],
Cell[CellGroupData[{
Cell[28684, 1111, 226, 7, 70, "ExampleSection",
 CellID->10891],
Cell[28913, 1120, 95, 1, 70, "ExampleText",
 CellID->43546263],
Cell[CellGroupData[{
Cell[29033, 1125, 309, 11, 70, "Input",
 CellID->785057400],
Cell[29345, 1138, 158, 5, 36, "Output",
 CellID->702373385]
}, Open  ]]
}, Closed]]
}, Open  ]],
Cell[CellGroupData[{
Cell[29564, 1150, 311, 9, 70, "SeeAlsoSection",
 CellID->12274],
Cell[29878, 1161, 1210, 44, 70, "SeeAlso",
 CellID->18133]
}, Open  ]],
Cell[CellGroupData[{
Cell[31125, 1210, 319, 9, 70, "MoreAboutSection",
 CellID->533002409],
Cell[31447, 1221, 150, 3, 70, "MoreAbout",
 CellID->24801857],
Cell[31600, 1226, 169, 3, 70, "MoreAbout",
 CellID->131860968],
Cell[31772, 1231, 178, 3, 70, "MoreAbout",
 CellID->34820731]
}, Open  ]],
Cell[31965, 1237, 27, 0, 70, "History"],
Cell[31995, 1239, 23, 0, 70, "FooterCell"]
}
]
*)

(* End of internal cache information *)

